#include <dirent.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __linux__
#include <asm/param.h> /* get definition of HZ */
#endif

#ifdef __sun__
#include <sys/swap.h>
#include <sys/sysinfo.h>
#include <sys/mkdev.h>
#include <limits.h>
#include <fcntl.h>
#include <procfs.h>
#endif

#ifdef __FreeBSD__
#include <fcntl.h>
#include <sys/param.h>
#include <sys/proc.h>
#include <sys/sysctl.h>
#include <sys/user.h>
#include <kvm.h>
#endif


#include "procinfo.h"


#define PROC_BASE    "/proc"

ProcinfoMeter::ProcinfoMeter()
{
#ifdef __sun__
  getCpuinfo(cpuinfo);
#endif
}


void ProcinfoMeter::unmarkProcinfoInternalList()
{
  for (std::list < ProcinfoInternal > ::iterator it = procinfoInternalList.begin();
       it != procinfoInternalList.end();
       it++)
  {
    it->updated = false;
  }
}


bool operator<(const ProcinfoMeter::ProcinfoInternal& a, const ProcinfoMeter::ProcinfoInternal& b)
{
  return (a.procinfo.cpupercent > b.procinfo.cpupercent);
}


std::list < ProcinfoMeter::ProcinfoInternal > ::iterator ProcinfoMeter::getProcinfoInternalList(int pid)
{
  std::list < ProcinfoInternal > ::iterator pit;
  bool found = false;
  for (pit = procinfoInternalList.begin();
       pit != procinfoInternalList.end();
       pit++)
  {
    if (pit->procinfo.pid == pid)
    {
      found = true;
      break;
    }
  }
  if (!found)
  {
    ProcinfoInternal newentry;
    pit = procinfoInternalList.insert(pit, newentry);
    memset(&*pit, 0, sizeof(ProcinfoInternal));
    pit->procinfo.pid = pid;
    pit->uid = -1;
  }
  pit->updated = true;

  return pit;
}


#ifdef __linux__
// Ergebnis in Sekunden!
float ProcinfoMeter::get_elapsed_time(void)
{
  struct timeval time;
  struct timezone timez;
  gettimeofday(&time, &timez);

  float elapsed_time = (time.tv_sec - oldtime.tv_sec)
                       + (float) (time.tv_usec - oldtime.tv_usec) / 1000000.0;
  oldtime.tv_sec = time.tv_sec;
  oldtime.tv_usec = time.tv_usec;
  return (elapsed_time);
}


bool ProcinfoMeter::readProcinfo(ProcinfoMeter::ProcinfoInternal & pii)
{
  bool retval = true;

  char path[PATH_MAX + 1];
  snprintf(path, PATH_MAX + 1, "%s/%d/stat", PROC_BASE, pii.procinfo.pid);
  FILE *file;
  if ((file = fopen(path, "r")))
  {
    if (pii.uid < 0)
    {
      char pathuid[PATH_MAX + 1];
      snprintf(pathuid, PATH_MAX + 1, "%s/%d", PROC_BASE, pii.procinfo.pid);
      struct stat st;
      if (stat(pathuid, &st) < 0)
      {
        perror(path);
        retval = false;
      }
      pii.uid = st.st_uid;
    }

    if (fscanf(file, "%*d (%[^)]) %c %*d %*d %*d %*d %*d %*u "
               "%*u %*u %*u %*u %d %d %*d %*d %*d"
               "%d",
               &pii.procinfo.command[0],
               &pii.procinfo.state,

               &pii.stat_utime,
               &pii.stat_stime,

               &pii.procinfo.priority
              ) != 5)
    {
      retval = false;
      fprintf(stderr, "badly formated /proc/#/stat\n");
    }

    fclose(file);
  }
  else
  {
    retval = false;
  }

  return retval;
}


void ProcinfoMeter::updateProcinfoInternalList()
{
  float elapsed_time = get_elapsed_time();

  DIR *dir;
  if (!(dir = opendir(PROC_BASE)))
  {
    perror(PROC_BASE);
  }
  else
  {
    struct dirent *de;
    while ((de = readdir(dir)))
    {
      pid_t pid;
      if ((pid = atoi(de->d_name)))
      {
        std::list < ProcinfoInternal > ::iterator pit = getProcinfoInternalList(pid);

        pit->last_stat_utime = pit->stat_utime;
        pit->last_stat_stime = pit->stat_stime;

        readProcinfo(*pit);

        if (pit->last_stat_utime > 0)
        {
          int diff_stat_utime = pit->stat_utime - pit->last_stat_utime;
          int diff_stat_stime = pit->stat_stime - pit->last_stat_stime;
          pit->procinfo.cpupercent = float(diff_stat_utime + diff_stat_stime)
                                     / HZ / elapsed_time * 100.;
        }
      }
    }
    closedir(dir);
  }
}
#endif


#ifdef __sun__
bool ProcinfoMeter::readProcinfo(ProcinfoMeter::ProcinfoInternal & pii)
{
  char path[PATH_MAX + 1];
  snprintf(path, PATH_MAX + 1, "%s/%d/psinfo", PROC_BASE, pii.procinfo.pid);
  psinfo_t psi;

  int fd;
  if ((fd = open(path, O_RDONLY)) < 0)
    return false;
  int r;
  if (read(fd, (void *)&psi, sizeof(psi)) < (int)sizeof(psi))
  {
    close(fd);
    return false;
  }
  close(fd);

  pii.uid = psi.pr_uid;
  strncpy(pii.procinfo.command, psi.pr_fname, COMMSIZE);
  pii.procinfo.command[COMMSIZE - 1] = 0;
  pii.procinfo.state = psi.pr_lwp.pr_sname;
  pii.procinfo.priority = psi.pr_lwp.pr_nice;
  pii.procinfo.cpupercent = ((float)psi.pr_pctcpu) / 0x8000 * cpuinfo.cpus * 100.;

  return true;
}


void ProcinfoMeter::updateProcinfoInternalList()
{
  DIR *dir;
  if (!(dir = opendir(PROC_BASE)))
  {
    perror(PROC_BASE);
  }
  else
  {
    struct dirent *de;
    while (de = readdir(dir))
    {
      pid_t pid;
      if (pid = atoi(de->d_name))
      {
        std::list < ProcinfoInternal > ::iterator pit = getProcinfoInternalList(pid);
        readProcinfo(*pit);
      }
    }
    closedir(dir);
  }
}
#endif


#ifdef __FreeBSD__
void ProcinfoMeter::updateProcinfoInternalList()
{
  kvm_t *kd;
  if ((kd = kvm_open(NULL, NULL, NULL, O_RDONLY, "kvm_open")) == NULL)
  {
    fprintf(stderr, "kvm_open failed\n");
  }
  else
  {
    int nentries;
    struct kinfo_proc *kp;    /* defines in include/sys/user.h */
    if ((kp = kvm_getprocs(kd, KERN_PROC_ALL, 0, &nentries)) == 0)
    {
      fprintf(stderr, "kvm_getprocs failed\n");
    }
    else
    {
      for (int i = nentries; --i >= 0; ++kp)
      {
        /* struct proc is defined in include/sys/proc.h */


#ifdef HAVE_STRUCT_KINFO_PROC_KI_PID
        if ((kp->ki_flag & P_CONTROLT ) != 0)
        {
          std::list < ProcinfoInternal > ::iterator pit = getProcinfoInternalList(kp->ki_pid);

          strncpy(pit->procinfo.command, kp->ki_comm, COMMSIZE);
          pit->procinfo.command[COMMSIZE - 1] = 0;
          pit->procinfo.priority = kp->ki_nice;
          pit->uid = kp->ki_ruid;
          pit->procinfo.cpupercent = 100.0 * double(kp->ki_pctcpu) / double(FSCALE);
          switch (kp->ki_stat)
#else
        if ((kp->kp_proc.p_flag & P_CONTROLT ) != 0)
        {
          std::list < ProcinfoInternal > ::iterator pit = getProcinfoInternalList(kp->kp_proc.p_pid);

          strncpy(pit->procinfo.command, kp->kp_proc.p_comm, COMMSIZE);
          pit->procinfo.command[COMMSIZE - 1] = 0;
          pit->procinfo.priority = kp->kp_proc.p_nice;
          pit->uid = kp->kp_eproc.e_pcred.p_ruid;
          pit->procinfo.cpupercent = 100.0 * double(kp->kp_proc.p_pctcpu) / double(FSCALE);
          switch (kp->kp_proc.p_stat)
#endif

          {
          case SSTOP:
            pit->procinfo.state = 'T';
            break;
          case SSLEEP:
            pit->procinfo.state = 'D';
            break;
          case SRUN:
          case SIDL:
            pit->procinfo.state = 'R';
            break;
          case SZOMB:
            pit->procinfo.state = 'Z';
            break;
          default:
            pit->procinfo.state = '?';
          }
        }
      }
    }
    kvm_close(kd);
  }
}
#endif


void ProcinfoMeter::cleanupProcinfoInternalList()
{
  for (std::list < ProcinfoInternal > ::iterator pit = procinfoInternalList.begin();
       pit != procinfoInternalList.end();
      )
  {
    if (pit->updated)
    {

      /*
            cout << "pid:" << pit->procinfo.pid

                  << " cmd:" << pit->procinfo.command
      	    << " cpupercent:" << pit->procinfo.cpupercent
      	    << endl;
       */

      pit++;
    }
    else
    {
      pit = procinfoInternalList.erase(pit);
    }
  }
}


bool ProcinfoMeter::getTopList(int nr, Procinfo procinfo[])
{
  unmarkProcinfoInternalList();

  updateProcinfoInternalList();

  procinfoInternalList.sort();

  cleanupProcinfoInternalList();

  int i = 0;
  for (std::list < ProcinfoInternal > ::iterator pit = procinfoInternalList.begin();
       ((pit != procinfoInternalList.end()) && (i < nr));
       pit++)
  {

#ifdef SIMPLE_USER_CACHE
    if (0 == pit->procinfo.username[0])
    {
      // do some kind of name-caching to prevent (NIS/NIS+) name-lookups
      for (std::list < ProcinfoInternal > ::iterator ppit = procinfoInternalList.begin();
           ppit != procinfoInternalList.end();
           ppit++)
      {
        if ((pit->uid == ppit->uid)
            && (ppit->procinfo.username[0] != 0 ))
        {
          strncpy(pit->procinfo.username, ppit->procinfo.username, USERNAMEMAX);
          pit->procinfo.username[USERNAMEMAX - 1] = 0;
          break;
        }
      }
    }
#endif

    if (0 == pit->procinfo.username[0])
    {
      struct passwd * passwdent = getpwuid(pit->uid);
      if (passwdent != 0)
      {
        strncpy(pit->procinfo.username, passwdent->pw_name, USERNAMEMAX);
        pit->procinfo.username[USERNAMEMAX - 1] = 0;
      }
      else
      {
        snprintf(pit->procinfo.username, USERNAMEMAX, "%d", pit->uid);
      }
    }
    procinfo[i] = pit->procinfo;
    i++;
  }
  if (i == nr)
    return true;

  return false;
}
