/*
**  Sinek (Video Player)
**  Copyright (c) 2001 - 2002 the Sinek Team, see the AUTHORS file.
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  preferences
*/

#include "common.h"

#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#endif

static GtkWidget *notebook;
static GtkTooltips *tips;
static GList *page_list = NULL;

struct cfgitem
{
	char *label;
	cfg_entry_t *entry;
	GtkWidget *widget;
	GList *subitems;
};

static void print_usage(void);
static GtkWidget *build_page(GList *items);
static GtkWidget *build_page_list(void);
static void cb_ok(GtkWidget *w, gpointer data);
static void cb_apply(GtkWidget *w, gpointer data);

static char *shortopts = "V:A:u:a:fgrshv";
#ifdef HAVE_GETOPT_LONG
static struct option longopts[] =
{
	{ "video-driver", required_argument, 0, 'V' },
	{ "audio-driver", required_argument, 0, 'A' },
	{ "spu-channel", required_argument, 0, 'u' },
	{ "audio-channel", required_argument, 0, 'a' },
	{ "fullscreen", 0, 0, 'f' },
	{ "hide-gui", 0, 0, 'g' },
	{ "repeat", 0, 0, 'r' },
	{ "script", required_argument, 0, 's' },
	{ "help", 0, 0, 'h' },
	{ "version", 0, 0, 'v' },
	{ 0, 0, 0, 0 }
};
#endif


void prf_init(int argc, char *argv[])
{
	char *home, *tmp;
	int c, i;

#ifdef HAVE_GETOPT_LONG
	while((c = getopt_long(argc, argv, shortopts, longopts, &i)) != -1)
#else
	while((c = getopt(argc, argv, shortopts)) != -1)
#endif
	{
		switch(c)
		{
			case 'A':
				sinek.audio_id = g_strdup(optarg);
				break;

			case 'V':
				sinek.video_id = g_strdup(optarg);
				break;

			case 'u':
				sinek.spu_channel = atoi(optarg);
				break;

			case 'a':
				sinek.audio_channel = atoi(optarg);
				break;

			case 'f':
				sinek.fs_req = 1;
				break;

			case 'g':
				sinek.hide_gui = 1;
				break;

			case 'r':
				sinek.repeat_one = -1;
				break;

			case 's':
				sinek.script = g_strdup(optarg);
				break;

			case 'h':
				print_usage();

			case 'v':
				print_version();
				exit(0);
		}
	}
	sinek.start_mrl = NULL;
	if(optind < argc) sinek.start_mrl = strdup(argv[optind]);

	home = g_get_home_dir();
	tmp = g_strconcat(home, "/.xine/config", NULL);
	sinek.conf = config_file_init(tmp);
	g_free(tmp);

	print_version();
}


void print_version(void)
{
	printf(_("Sinek v%s (Video Player)\n"), VERSION);
	printf(_("built with xine library v%d.%d.%d, using v%d.%d.%d.\n"),
		XINE_MAJOR_VERSION, XINE_MINOR_VERSION, XINE_SUB_VERSION,
		xine_get_major_version(), xine_get_minor_version(), xine_get_sub_version());
}


static void print_usage(void)
{
	puts(_("Usage: Sinek [OPTIONS] [MRL]"));
	puts(_("  -V, --video-driver <id>    Select video driver."));
	puts(_("  -A, --audio-driver <id>    Select audio driver."));
	puts(_("  -u, --spu-channel <no>     Select SPU subtitle channel."));
	puts(_("  -a, --audio-channel <no>   Select Audio channel."));
	puts(_("  -f, --fullscreen           Start in fullscreen mode."));
	puts(_("  -g, --hide-gui             Hide control panel."));
	puts(_("  -r, --repeat               Play mrl continously."));
#ifdef USE_GUILE
	puts(_("  -s, --script <file>        Execute scheme script."));
#endif
	puts(_("  -h, --help                 Print this help text and exit."));
	puts(_("  -v, --version              Print version number and exit."));
#ifndef HAVE_GETOPT_LONG
	puts(_("(long options are not supported on your system)"));
#endif
	puts(_("Report bugs to <madcat@users.sourceforge.net>."));
	exit(0);
}


void prf_show(void)
{
	GtkWidget *vb, *hb, *bb, *b;
	GList *page;

	if(sinek.prefs_win) goto out;

	sinek.prefs_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(sinek.prefs_win), _("Sinek Settings"));
	gtk_window_set_wmclass(GTK_WINDOW(sinek.prefs_win), "Sinek Settings", "Sinek");

	tips = gtk_tooltips_new();

	vb = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vb);
	gtk_container_add(GTK_CONTAINER(sinek.prefs_win), vb);

	hb = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(hb), build_page_list(), FALSE, TRUE, 0);

	notebook = gtk_notebook_new();
	gtk_widget_show(notebook);
	gtk_box_pack_start(GTK_BOX(hb), notebook, TRUE, TRUE, 0);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), FALSE);
	page = g_list_first(page_list);
	while(page)
	{
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), build_page(page), NULL);
		page = g_list_next(page);
	}

	bb = gtk_hbutton_box_new();
	gtk_widget_show(bb);
	gtk_box_pack_start(GTK_BOX(vb), bb, FALSE, FALSE, 0);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bb), 10);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bb), GTK_BUTTONBOX_END);
	gtk_container_set_border_width(GTK_CONTAINER(bb), 5);

	b = gtk_button_new_with_label(_("OK"));
	gtk_widget_show(b);
	gtk_container_add(GTK_CONTAINER(bb), b);
	gtk_signal_connect(GTK_OBJECT(b), "clicked", GTK_SIGNAL_FUNC(cb_ok), NULL);

	b = gtk_button_new_with_label(_("Apply"));
	gtk_widget_show(b);
	gtk_container_add(GTK_CONTAINER(bb), b);
	gtk_signal_connect(GTK_OBJECT(b), "clicked", GTK_SIGNAL_FUNC(cb_apply), NULL);

	b = gtk_button_new_with_label(_("Close"));
	gtk_widget_show(b);
	gtk_container_add(GTK_CONTAINER(bb), b);
	gtk_signal_connect_object(GTK_OBJECT(b), "clicked", gtk_widget_hide, GTK_OBJECT(sinek.prefs_win));

	wm_manage(sinek.prefs_win, WM_NORMAL);
out:
	gtk_widget_show(sinek.prefs_win);
}


static void prf_apply_changes(void)
{
	GList *page, *item;
	struct cfgitem *citem;
	char *str;
	int val = 0;

	page = g_list_first(page_list);
	while(page)
	{
		item = g_list_first(((struct cfgitem *)page->data)->subitems);
		while(item)
		{
			citem = (struct cfgitem *)item->data;
			switch(citem->entry->type)
			{
				case CONFIG_TYPE_STRING:
					str = gtk_entry_get_text(GTK_ENTRY(citem->widget));
					sinek.conf->update_string(sinek.conf, citem->entry->key, str);
					break;
				case CONFIG_TYPE_BOOL:
					val = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(citem->widget));
					sinek.conf->update_num(sinek.conf, citem->entry->key, val);
					break;
				case CONFIG_TYPE_ENUM:
					val = (int)citem->widget;
					sinek.conf->update_num(sinek.conf, citem->entry->key, val);
					break;
				case CONFIG_TYPE_NUM:
					val = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(citem->widget));
					sinek.conf->update_num(sinek.conf, citem->entry->key, val);
					break;
				case CONFIG_TYPE_RANGE:
					val = ((GtkAdjustment *)citem->widget)->value;
					sinek.conf->update_num(sinek.conf, citem->entry->key, val);
					break;
			}
			item = g_list_next(item);
		}
		page = g_list_next(page);
	}
}


static void cb_ok(GtkWidget *w, gpointer data)
{
	prf_apply_changes();
	sinek.conf->save(sinek.conf);
	gtk_widget_hide(sinek.prefs_win);
}


static void cb_apply(GtkWidget *w, gpointer data)
{
	prf_apply_changes();
}


static GtkWidget *build_string(struct cfgitem *citem)
{
	GtkWidget *box, *t;

	box = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 2);
	t = gtk_label_new(citem->label);
	gtk_widget_show(t);
	gtk_box_pack_start(GTK_BOX(box), t, FALSE, FALSE, 0);
	t = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(t), citem->entry->str_value);
	gtk_widget_show(t);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), t, citem->entry->description, NULL);
	gtk_box_pack_start(GTK_BOX(box), t, TRUE, TRUE, 0);
	citem->widget = t;

	return box;
}


static GtkWidget *build_checkbox(struct cfgitem *citem)
{
	GtkWidget *t;

	t = gtk_check_button_new_with_label(citem->label);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(t), citem->entry->num_value);
	gtk_widget_show(t);
	citem->widget = t;
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), t, citem->entry->description, NULL);

	return t;
}


static void cb_option(GtkWidget *w, int i)
{
	struct cfgitem *citem;

	citem = gtk_object_get_data(GTK_OBJECT(w), "cfgitem");
	citem->widget = (GtkWidget *)i;
}


static GtkWidget *build_option(struct cfgitem *citem)
{
	GtkWidget *box, *opt, *m, *mi;
	char **enums;
	int i = 0;

	box = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 2);
	m = gtk_label_new(citem->label);
	gtk_widget_show(m);
	gtk_box_pack_start(GTK_BOX(box), m, FALSE, FALSE, 0);

	enums = citem->entry->enum_values;
	opt = gtk_option_menu_new();
	gtk_widget_show(opt);
	m = gtk_menu_new();
	while(enums[i])
	{
		mi = gtk_menu_item_new_with_label(enums[i]);
		gtk_widget_show(mi);
		gtk_signal_connect(GTK_OBJECT(mi), "activate", GTK_SIGNAL_FUNC(cb_option), (gpointer)i);
		gtk_object_set_data(GTK_OBJECT(mi), "cfgitem", (gpointer)citem);
		gtk_menu_append(GTK_MENU(m), mi);
		i++;
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(opt), m);
	gtk_option_menu_set_history(GTK_OPTION_MENU(opt), citem->entry->num_value);
	gtk_box_pack_start(GTK_BOX(box), opt, TRUE, TRUE, 0);
	citem->widget = (GtkWidget *)citem->entry->num_value;
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), opt, citem->entry->description, NULL);

	return box;
}


static GtkWidget *build_numeric(struct cfgitem *citem)
{
	GtkObject *adj;
	GtkWidget *box, *lab, *spin;

	box = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 2);
	lab = gtk_label_new(citem->label);
	gtk_widget_show(lab);
	gtk_box_pack_start(GTK_BOX(box), lab, FALSE, FALSE, 0);

	adj = gtk_adjustment_new(citem->entry->num_value, INT_MIN, INT_MAX, 1, 0, 0);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
	gtk_widget_show(spin);
	gtk_box_pack_start(GTK_BOX(box), spin, TRUE, TRUE, 0);
	citem->widget = spin;
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), spin, citem->entry->description, NULL);

	return box;
}


static GtkWidget *build_scale(struct cfgitem *citem)
{
	GtkObject *adj;
	GtkWidget *box, *lab, *scale;

	box = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 2);
	lab = gtk_label_new(citem->label);
	gtk_widget_show(lab);
	gtk_box_pack_start(GTK_BOX(box), lab, FALSE, FALSE, 0);

	adj = gtk_adjustment_new(citem->entry->num_value, citem->entry->range_min, citem->entry->range_max, 1, 0, 0);
	scale = gtk_hscale_new(GTK_ADJUSTMENT(adj));
	gtk_widget_show(scale);
	gtk_scale_set_digits(GTK_SCALE(scale), 0);
	gtk_box_pack_start(GTK_BOX(box), scale, TRUE, TRUE, 0);
	citem->widget = (GtkWidget *)adj;
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), scale, citem->entry->description, NULL);

	return box;
}


static GtkWidget *build_page(GList *page)
{
	GtkWidget *sw, *box, *w;
	GList *item;
	struct cfgitem *citem;
	cfg_entry_t *entry;

	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	box = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(box);
	gtk_container_set_border_width(GTK_CONTAINER(box), 10);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw), box);

	item = g_list_first(((struct cfgitem *)page->data)->subitems);
	while(item)
	{
		citem = (struct cfgitem *)item->data;
		entry = citem->entry;
		if(entry->description)
		{
			switch(entry->type)
			{
				case CONFIG_TYPE_STRING: w = build_string(citem); break;
				case CONFIG_TYPE_BOOL: w = build_checkbox(citem); break;
				case CONFIG_TYPE_ENUM: w = build_option(citem); break;
				case CONFIG_TYPE_NUM: w = build_numeric(citem); break;
				case CONFIG_TYPE_RANGE: w = build_scale(citem); break;
				default: w = NULL;
			}
			if(w) gtk_box_pack_start(GTK_BOX(box), w, FALSE, FALSE, 0);
		}
		item = g_list_next(item);
	}

	return sw;
}


static void cb_page(GtkWidget *w, gint row, gint col, GdkEventButton *ev, gpointer data)
{
	gtk_notebook_set_page(GTK_NOTEBOOK(notebook), row);
}


static GtkWidget *build_page_list(void)
{
	GtkWidget *list;
	GList *item;
	struct cfgitem *cpage, *citem;
	cfg_entry_t *entry;
	gchar *p, *tmp[1];
	int len;

	list = gtk_clist_new(1);
	gtk_widget_show(list);
	gtk_container_set_border_width(GTK_CONTAINER(list), 2);

	entry = sinek.conf->first;
	while(entry)
	{
		p = strchr(entry->key, '.');
		if(p && entry->description)
		{
			len = p - entry->key;
			cpage = NULL;
			item = g_list_first(page_list);
			while(item)
			{
				if(strncmp(((struct cfgitem *)item->data)->label, entry->key, len) == 0)
				{
					cpage = (struct cfgitem *)item->data;
					break;
				}
				item = g_list_next(item);
			}
			if(!cpage)
			{
				cpage = g_malloc0(sizeof(struct cfgitem));
				cpage->label = g_strndup(entry->key, len);
				tmp[0] = cpage->label;
				gtk_clist_append(GTK_CLIST(list), tmp);
				page_list = g_list_append(page_list, (gpointer)cpage);
			}
			citem = g_malloc0(sizeof(struct cfgitem));
			citem->label = g_strdup(entry->key + len + 1);
			citem->entry = entry;
			cpage->subitems = g_list_append(cpage->subitems, (gpointer)citem);
		}
		entry = entry->next;
	}
	gtk_clist_columns_autosize(GTK_CLIST(list));
	gtk_signal_connect(GTK_OBJECT(list), "select_row", GTK_SIGNAL_FUNC(cb_page), NULL);

	return list;
}
