/*
**  Sinek (Video Player)
**  Copyright (c) 2001 - 2002 the Sinek Team, see the AUTHORS file.
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  key bindings
*/

#include "common.h"

#define MOD_SHIFT 1
#define MOD_CONTROL 4
#define MOD_ALT 8

struct action_struct
{
	char *action;
	char *name;
	cmd_type cmd;
	char *key;
	int modifier;
};

struct action_struct actions[] =
{
	{ "play", N_("Play"), CMD_PLAY, "Return", 0 },
	{ "pause", N_("Pause"), CMD_PAUSE, "space", 0 },
	{ "stop", N_("Stop"), CMD_STOP, "s", 0 },
	{ "next", N_("Next"), CMD_NEXT, "Next", 0 },
	{ "previous", N_("Previous"), CMD_PREVIOUS, "Prior", 0 },
	{ "speed-fast", N_("Increase playback speed"), CMD_SPEED_FAST, "Up", 0 },
	{ "speed-slow", N_("Decrease playback speed"), CMD_SPEED_SLOW, "Down", 0 },
	{ "av-offset-down", N_("Speed up video (if it lags behind)"), CMD_AV_OFFSET_DOWN, "n", 0},
	{ "av-offset-up", N_("Slow down video (if audio lags behind)"), CMD_AV_OFFSET_UP, "m", 0},
	{ "toggle-fullscreen", N_("Toggle fullscreen mode"), CMD_TOGGLE_FULLSCREEN, "f", 0 },
	{ "zoom-out", N_("Zoom out"), CMD_ZOOM_OUT, "z", 0 },
	{ "zoom-in", N_("Zoom in"), CMD_ZOOM_IN, "z", MOD_SHIFT },
	{ "zoom-reset", N_("Reset zoom"), CMD_ZOOM_RESET, "x", 0 },
	{ "toggle-aspect", N_("Change video aspect"), CMD_TOGGLE_ASPECT, "a", 0 },
	{ "toggle-interleave", N_("Toggle interleave"), CMD_TOGGLE_INTERLEAVE, "i", 0 },
	{ "window-reduce", N_("Reduce video window"), CMD_WINDOW_REDUCE, "less", 0 },
	{ "window-enlarge", N_("Enlarge video window"), CMD_WINDOW_ENLARGE, "greater", 0 },
	{ "audio-mute", N_("Toggle audio muting"), CMD_TOGGLE_MUTE, "BackSpace", 0 },
	{ "audio-next", N_("Next audio channel"), CMD_AUDIO_NEXT, "plus", 0 },
	{ "audio-previous", N_("Previous audio channel"), CMD_AUDIO_PREVIOUS, "minus", 0 },
	{ "show-playlist", N_("Show playlist window"), CMD_SHOW_PLAYLIST, "p", 0 },
	{ "show-tuner", N_("Show tuner window"), CMD_SHOW_TUNER, "t", 0 },
	{ "show-about", N_("Show about window"), CMD_SHOW_ABOUT, "question", 0 },
	{ "show-controls", N_("Show controls window"), CMD_SHOW_CONTROLS, "c", 0 },
	{ "show-video", N_("Show video window"), CMD_SHOW_VIDEO, "v", 0 },
	{ "toggle-gui", N_("Toggle all windows"), CMD_TOGGLE_GUI, "g", 0 },
	{ "quit", N_("Quit"), CMD_QUIT, "Escape", 0 },
	{ "event-menu1", N_("Menu 1"), CMD_EVENT_MENU1, "F1", 0 },
	{ "event-menu2", N_("Menu 2"), CMD_EVENT_MENU2, "F2", 0 },
	{ "event-menu3", N_("Menu 3"), CMD_EVENT_MENU3, "F3", 0 },
	{ "event-up", N_("Menu up"), CMD_EVENT_UP, "KP_Up", 0 },
	{ "event-down", N_("Menu down"), CMD_EVENT_DOWN, "KP_Down", 0 },
	{ "event-left", N_("Menu left"), CMD_EVENT_LEFT, "KP_Left", 0 },
	{ "event-right", N_("Menu right"), CMD_EVENT_RIGHT, "KP_Right", 0 },
	{ "event-next", N_("Menu next"), CMD_EVENT_NEXT, "KP_Next", 0 },
	{ "event-prior", N_("Menu previous"), CMD_EVENT_PRIOR, "KP_Prior", 0 },
	{ "event-angle-next", N_("Menu next angle"), CMD_EVENT_ANGLE_NEXT, "KP_1", 0 },
	{ "event-angle-prior", N_("Menu previous angle"), CMD_EVENT_ANGLE_PRIOR, "KP_7", 0 },
	{ "event-select", N_("Menu select"), CMD_EVENT_SELECT, "KP_Enter", 0 },
	{ "spu-next", N_("Next subtitle channel"), CMD_SPU_NEXT, "period", 0 },
	{ "spu-previous", N_("Previous subtitle channel"), CMD_SPU_PREVIOUS, "comma", 0 },
	{ "mark-goto-0", N_("Go to mark 0"), CMD_GOTO_MARK_0, "0", 0 },
	{ "mark-goto-1", N_("Go to mark 1"), CMD_GOTO_MARK_1, "1", 0 },
	{ "mark-goto-2", N_("Go to mark 2"), CMD_GOTO_MARK_2, "2", 0 },
	{ "mark-goto-3", N_("Go to mark 3"), CMD_GOTO_MARK_3, "3", 0 },
	{ "mark-goto-4", N_("Go to mark 4"), CMD_GOTO_MARK_4, "4", 0 },
	{ "mark-goto-5", N_("Go to mark 5"), CMD_GOTO_MARK_5, "5", 0 },
	{ "mark-goto-6", N_("Go to mark 6"), CMD_GOTO_MARK_6, "6", 0 },
	{ "mark-goto-7", N_("Go to mark 7"), CMD_GOTO_MARK_7, "7", 0 },
	{ "mark-goto-8", N_("Go to mark 8"), CMD_GOTO_MARK_8, "8", 0 },
	{ "mark-goto-9", N_("Go to mark 9"), CMD_GOTO_MARK_9, "9", 0 },
	{ "mark-set-1", N_("Set mark 1"), CMD_SET_MARK_1, "1", MOD_SHIFT },
	{ "mark-set-2", N_("Set mark 2"), CMD_SET_MARK_2, "2", MOD_SHIFT },
	{ "mark-set-3", N_("Set mark 3"), CMD_SET_MARK_3, "3", MOD_SHIFT },
	{ "mark-set-4", N_("Set mark 4"), CMD_SET_MARK_4, "4", MOD_SHIFT },
	{ "mark-set-5", N_("Set mark 5"), CMD_SET_MARK_5, "5", MOD_SHIFT },
	{ "mark-set-6", N_("Set mark 6"), CMD_SET_MARK_6, "6", MOD_SHIFT },
	{ "mark-set-7", N_("Set mark 7"), CMD_SET_MARK_7, "7", MOD_SHIFT },
	{ "mark-set-8", N_("Set mark 8"), CMD_SET_MARK_8, "8", MOD_SHIFT },
	{ "mark-set-9", N_("Set mark 9"), CMD_SET_MARK_9, "9", MOD_SHIFT },
	{ NULL, NULL, CMD_NONE, NULL, 0 }
};

static char *modifiers[] =
{
	N_("Shift"),
	N_("Control"),
	N_("Alt"),
	N_("Control shift"),
	N_("Alt shift"),
	N_("Control alt"),
	N_("Control alt shift"),
	NULL
};

static GtkWidget *key_win = NULL;
static GtkWidget *action_wid, *key_wid, *shift_wid, *control_wid, *alt_wid, *clist_wid;
static int key_row = 0, key_updating = 0;


static struct action_struct *find_action(char *action)
{
	int i;

	for(i = 0; actions[i].action; i++)
	{
		if(strcasecmp(action, actions[i].action) == 0)
			return &actions[i];
	}
	return NULL;
}


void key_init(void)
{
	struct action_struct *action;
	char *fname, buf[2048], *t;
	FILE *fd;
	int i;

	for(i = 0; modifiers[i]; i++)
	{
		modifiers[i] = gettext(modifiers[i]);
	}

	for(i = 0; actions[i].action; i++)
	{
		actions[i].name = gettext(actions[i].name);
	}

	fname = g_strconcat(g_get_home_dir(), "/.xine/sinek.keys", NULL);
	fd = fopen(fname, "r");
	if(fd)
	{
		while(fgets(buf, 2040, fd))
		{
			strtok(buf, " \t");
			action = find_action(buf);
			if(action)
			{
				action->key = NULL;
				action->modifier = 0;
				t = strtok(NULL, " \t\r\n");
				if(t)
				{
					action->key = strdup(t);
					while(t)
					{
						t = strtok(NULL, " \t\r\n");
						if(t)
						{
							if(strcasecmp(t, "shift") == 0)
								action->modifier += MOD_SHIFT;
							else if(strcasecmp(t, "control") == 0)
								action->modifier += MOD_CONTROL;
							else if(strcasecmp(t, "alt") == 0)
								action->modifier += MOD_ALT;
						}
					}
				}
			}
		}
		fclose(fd);
	}
	g_free(fname);
}


static cmd_type map_event(char *key, int modifier)
{
	int i, j = -1;

	for(i = 0; actions[i].action; i++)
	{
		if(strcmp(actions[i].key, key) == 0)
		{
			if(actions[i].modifier == modifier) return(actions[i].cmd);
			j = i;
		}
	}
	if(j != -1 && actions[j].modifier == 0) return(actions[j].cmd);
	return CMD_NONE;
}


void key_handle(unsigned long keyval, int modifier)
{
	char *key;
	cmd_type cmd;

	modifier &= (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK);
	key = gdk_keyval_name(gdk_keyval_to_lower(keyval));
	if(!key) return;
	cmd = map_event(key, modifier);
	if(cmd != CMD_NONE) execute_cmd(cmd);
}


static char *key_modifier(int modifier)
{
	char *t = "";

	switch(modifier)
	{
		case 0: t = ""; break;
		case MOD_SHIFT: t = modifiers[0]; break;
		case MOD_CONTROL: t = modifiers[1]; break;
		case MOD_ALT: t = modifiers[2]; break;
		case MOD_SHIFT | MOD_CONTROL: t = modifiers[3]; break;
		case MOD_SHIFT | MOD_ALT: t = modifiers[4]; break;
		case MOD_CONTROL | MOD_ALT: t = modifiers[5]; break;
		case MOD_SHIFT | MOD_CONTROL | MOD_ALT: t = modifiers[6]; break;
	}
	return t;
}


static void cb_key_press(GtkWidget *w, GdkEventKey *ev, gpointer data)
{
	char *keyname;

	keyname = gdk_keyval_name(gdk_keyval_to_lower(ev->keyval));
	gtk_entry_set_text(GTK_ENTRY(w), keyname);
	actions[key_row].key = keyname;
	gtk_clist_set_text(GTK_CLIST(clist_wid), key_row, 1, keyname);
	gtk_signal_emit_stop_by_name(GTK_OBJECT(w), "key_press_event");
}


static void cb_modifier(GtkWidget *w, gpointer data)
{
	int mod = 0;

	if(key_updating) return;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(control_wid))) mod += MOD_CONTROL;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(shift_wid))) mod += MOD_SHIFT;
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alt_wid))) mod += MOD_ALT;
	actions[key_row].modifier = mod;
	gtk_clist_set_text(GTK_CLIST(clist_wid), key_row, 2, key_modifier(mod));
}


static void cb_save(GtkWidget *w, gpointer data)
{
	char *fname;
	FILE *fd;
	int i;

	fname = g_strconcat(g_get_home_dir(), "/.xine/sinek.keys", NULL);
	fd = fopen(fname, "w");
	if(fd)
	{
		for(i = 0; actions[i].action; i++)
		{
			fprintf(fd, "%s", actions[i].action);
			if(actions[i].key) fprintf(fd, " %s", actions[i].key);
			if(actions[i].modifier & MOD_SHIFT) fprintf(fd, " shift");
			if(actions[i].modifier & MOD_CONTROL) fprintf(fd, " control");
			if(actions[i].modifier & MOD_ALT) fprintf(fd, " alt");
			fprintf(fd, "\n");
		}
		fclose(fd);
	}
}


static void cb_change_key(GtkWidget *w, gint row, gint col, GdkEventButton *ev, gpointer data)
{
	gboolean t;

	key_row = row;
	key_updating = 1;
	gtk_entry_set_text(GTK_ENTRY(action_wid), actions[row].name);
	gtk_entry_set_text(GTK_ENTRY(key_wid), actions[row].key);
	if(actions[row].modifier & MOD_CONTROL) t = TRUE; else t = FALSE;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(control_wid), t);
	if(actions[row].modifier & MOD_SHIFT) t = TRUE; else t = FALSE;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(shift_wid), t);
	if(actions[row].modifier & MOD_ALT) t = TRUE; else t = FALSE;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alt_wid), t);
	key_updating = 0;
}


void key_show(void)
{
	GtkWidget *vb, *hb, *sw, *t, *clist;
	char *titles[] = { _("Action"), _("Key"), _("Modifiers") };
	char *buf[3];
	int i;

	if(key_win) goto out;

	key_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(key_win), _("Sinek Key Bindings"));
	gtk_window_set_wmclass(GTK_WINDOW(key_win), "Sinek Key Bindings", "Sinek");

	vb = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vb);
	gtk_container_add(GTK_CONTAINER(key_win), vb);
	gtk_container_set_border_width(GTK_CONTAINER(vb), 4);

	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(vb), sw, TRUE, TRUE, 0);

	clist = gtk_clist_new_with_titles(3, titles);
	clist_wid = clist;
	gtk_widget_show(clist);
	gtk_container_add(GTK_CONTAINER(sw), clist);
	gtk_container_set_border_width(GTK_CONTAINER(clist), 2);
	gtk_clist_freeze(GTK_CLIST(clist));
	for(i = 0; actions[i].action; i++)
	{
		buf[0] = actions[i].name;
		buf[1] = actions[i].key;
		buf[2] = key_modifier(actions[i].modifier);
		gtk_clist_append(GTK_CLIST(clist), buf);
	}
	gtk_clist_columns_autosize(GTK_CLIST(clist));
	gtk_clist_thaw(GTK_CLIST(clist));
	gtk_signal_connect(GTK_OBJECT(clist), "select_row", GTK_SIGNAL_FUNC(cb_change_key), NULL);

	hb = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hb), 5);

	t = gtk_label_new(_("Action"));
	gtk_widget_show(t);
	gtk_box_pack_start(GTK_BOX(hb), t, FALSE, FALSE, 0);
	action_wid = gtk_entry_new();
	gtk_widget_show(action_wid);
	gtk_box_pack_start(GTK_BOX(hb), action_wid, TRUE, TRUE, 0);
	gtk_entry_set_editable(GTK_ENTRY(action_wid), FALSE);
	t = gtk_label_new(_("Key"));
	gtk_widget_show(t);
	gtk_box_pack_start(GTK_BOX(hb), t, FALSE, FALSE, 0);
	key_wid = gtk_entry_new();
	gtk_widget_show(key_wid);
	gtk_box_pack_start(GTK_BOX(hb), key_wid, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(key_wid), "key_press_event", GTK_SIGNAL_FUNC(cb_key_press), NULL);

	control_wid = gtk_check_button_new_with_label(_("Control"));
	gtk_widget_show(control_wid);
	gtk_box_pack_start(GTK_BOX(hb), control_wid, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(control_wid), "toggled", GTK_SIGNAL_FUNC(cb_modifier), 0);
	shift_wid = gtk_check_button_new_with_label(_("Shift"));
	gtk_widget_show(shift_wid);
	gtk_box_pack_start(GTK_BOX(hb), shift_wid, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(shift_wid), "toggled", GTK_SIGNAL_FUNC(cb_modifier), 0);
	alt_wid = gtk_check_button_new_with_label(_("Alt"));
	gtk_widget_show(alt_wid);
	gtk_box_pack_start(GTK_BOX(hb), alt_wid, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(alt_wid), "toggled", GTK_SIGNAL_FUNC(cb_modifier), 0);

	hb = gtk_hbutton_box_new();
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, FALSE, FALSE, 0);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hb), 10);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hb), GTK_BUTTONBOX_END);
	gtk_container_set_border_width(GTK_CONTAINER(hb), 5);

	t = gtk_button_new_with_label(_("Save"));
	gtk_widget_show(t);
	gtk_container_add(GTK_CONTAINER(hb), t);
	gtk_signal_connect(GTK_OBJECT(t), "clicked", GTK_SIGNAL_FUNC(cb_save), NULL);

	gtk_clist_select_row(GTK_CLIST(clist), 0, 0);

	gtk_signal_connect_object(GTK_OBJECT(key_win), "delete_event", gtk_widget_hide, GTK_OBJECT(key_win));

	wm_manage(key_win, WM_NORMAL);
out:
	gtk_widget_show(key_win);
}


cmd_type find_cmd(char *command)
{
	int i;

	for(i = 0; actions[i].action; i++)
	{
		if(strcasecmp(command, actions[i].action) == 0)
			return actions[i].cmd;
	}
	return CMD_NONE;
}


void execute_cmd(cmd_type cmd)
{
	xine_event_t xinev;
	int tmp;

	switch(cmd)
	{
		case CMD_NONE:
			break;

		case CMD_QUIT:
			sinek.playing = 0;
			enable_blanking();
			xine_stop(sinek.xine);
			xine_exit(sinek.xine);
			gtk_main_quit();
			break;

		case CMD_PLAY:
			if(xine_get_status(sinek.xine) != XINE_PLAY)
				video_play(sinek.mrl);
			else
				xine_set_speed(sinek.xine, SPEED_NORMAL);
			sinek.pause = 0;
			break;

		case CMD_PAUSE:
			if(sinek.pause)
				xine_set_speed(sinek.xine, SPEED_NORMAL);
			else
				xine_set_speed(sinek.xine, SPEED_PAUSE);
			sinek.pause = ~sinek.pause;
			break;

		case CMD_STOP:
			sinek.playing = 0;
			xine_stop(sinek.xine);
			enable_blanking();
			break;

		case CMD_NEXT:
			pl_next();
			break;

		case CMD_PREVIOUS:
			pl_prev();
			break;

		case CMD_SPEED_FAST:
			tmp = xine_get_speed(sinek.xine);
			if(tmp != SPEED_PAUSE && tmp < SPEED_FAST_4)
				xine_set_speed(sinek.xine, tmp * 2);
			break;

		case CMD_SPEED_SLOW:
			tmp = xine_get_speed(sinek.xine);
			if(tmp > SPEED_SLOW_4)
				xine_set_speed(sinek.xine, tmp / 2);
			break;

		case CMD_AV_OFFSET_DOWN:
			tmp = xine_get_av_offset(sinek.xine);
			tmp = tmp - 3600;
			xine_set_av_offset(sinek.xine, tmp);
			break;

		case CMD_AV_OFFSET_UP:
			tmp = xine_get_av_offset(sinek.xine);
			tmp = tmp + 3600;
			xine_set_av_offset(sinek.xine, tmp);
			break;

		case CMD_SHOW_ABOUT:
			about_show();
			break;

		case CMD_SHOW_TUNER:
			tune_show();
			break;

		case CMD_SHOW_KEYS:
			key_show();
			break;

		case CMD_SHOW_PLAYLIST:
			if(GTK_WIDGET_VISIBLE(sinek.playlist_win))
				gtk_widget_hide(sinek.playlist_win);
			else
				gtk_widget_show(sinek.playlist_win);
			break;

		case CMD_SHOW_VIDEO:
			video_show();
			break;

		case CMD_SHOW_CONTROLS:
			gtk_widget_show(sinek.control_win);
			break;

		case CMD_SHOW_PREFS:
			prf_show();
			break;

		case CMD_TOGGLE_GUI:
			if(sinek.hide)
				wm_show_all();
			else
				wm_hide_all();
			sinek.hide = ~sinek.hide;
			break;

		case CMD_TOGGLE_FULLSCREEN:
			video_toggle_fullscreen();
			break;

		case CMD_TOGGLE_ASPECT:
			tmp = sinek.vd->get_property(sinek.vd, VO_PROP_ASPECT_RATIO);
			sinek.vd->set_property(sinek.vd, VO_PROP_ASPECT_RATIO, tmp + 1);
			break;

		case CMD_TOGGLE_INTERLEAVE:
			tmp = sinek.vd->get_property(sinek.vd, VO_PROP_INTERLACED);
			sinek.vd->set_property(sinek.vd, VO_PROP_INTERLACED, 1 - tmp);
			break;

		case CMD_TOGGLE_SUBTITLES:
			if(sinek.osd_subtitles)
				osd_hide();
			else
				osd_show();
			break;

		case CMD_TOGGLE_REPEAT:
			sinek.repeat_one = ~sinek.repeat_one;
			GTK_CHECK_MENU_ITEM(sinek.repeat_chk)->active = sinek.repeat_one;
			gtk_widget_queue_draw(sinek.repeat_chk);
			GTK_TOGGLE_BUTTON(sinek.repeat_tog)->active = sinek.repeat_one;
			update_pix_toggle(sinek.repeat_tog);
			break;

		case CMD_TOGGLE_MUTE:
			audio_toggle_mute();
			GTK_TOGGLE_BUTTON(sinek.mute_tog)->active = sinek.mute;
			update_pix_toggle(sinek.mute_tog);
			break;

		case CMD_SPU_NEXT:
			tmp = xine_get_spu_channel(sinek.xine);
			xine_select_spu_channel(sinek.xine, tmp + 1);
			break;

		case CMD_SPU_PREVIOUS:
			tmp = xine_get_spu_channel(sinek.xine);
			xine_select_spu_channel(sinek.xine, tmp - 1);
			break;

		case CMD_AUDIO_NEXT:
			tmp = xine_get_audio_selection(sinek.xine);
			xine_select_audio_channel(sinek.xine, tmp + 1);
			break;

		case CMD_AUDIO_PREVIOUS:
			tmp = xine_get_audio_selection(sinek.xine);
			xine_select_audio_channel(sinek.xine, tmp - 1);
			break;

		case CMD_ZOOM_IN:
			video_zoom(1,1);
			break;

		case CMD_ZOOM_OUT:
			video_zoom(-1,-1);
			break;

		case CMD_ZOOM_RESET:
			video_zoom(0,0);
			break;

		case CMD_WINDOW_REDUCE:
			video_scale(0.8);
			break;

		case CMD_WINDOW_ENLARGE:
			video_scale(1.2);
			break;

		case CMD_EVENT_MENU1:
			xinev.type = XINE_EVENT_INPUT_MENU1;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_MENU2:
			xinev.type = XINE_EVENT_INPUT_MENU2;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_MENU3:
			xinev.type = XINE_EVENT_INPUT_MENU3;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_UP:
			xinev.type = XINE_EVENT_INPUT_UP;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_DOWN:
			xinev.type = XINE_EVENT_INPUT_DOWN;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_LEFT:
			xinev.type = XINE_EVENT_INPUT_LEFT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_RIGHT:
			xinev.type = XINE_EVENT_INPUT_RIGHT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_NEXT:
			xinev.type = XINE_EVENT_INPUT_NEXT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_PRIOR:
			xinev.type = XINE_EVENT_INPUT_PREVIOUS;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_ANGLE_NEXT:
			xinev.type = XINE_EVENT_INPUT_ANGLE_NEXT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_ANGLE_PRIOR:
			xinev.type = XINE_EVENT_INPUT_ANGLE_PREVIOUS;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_EVENT_SELECT:
			xinev.type = XINE_EVENT_INPUT_SELECT;
			xine_send_event(sinek.xine, &xinev);
			break;

		case CMD_GOTO_MARK_0:
		case CMD_GOTO_MARK_1:
		case CMD_GOTO_MARK_2:
		case CMD_GOTO_MARK_3:
		case CMD_GOTO_MARK_4:
		case CMD_GOTO_MARK_5:
		case CMD_GOTO_MARK_6:
		case CMD_GOTO_MARK_7:
		case CMD_GOTO_MARK_8:
		case CMD_GOTO_MARK_9:
			video_seek(sinek.marks[cmd - CMD_GOTO_MARK_0]);
			sinek.marks[0] = xine_get_current_time(sinek.xine);
			break;

		case CMD_SET_MARK_1:
		case CMD_SET_MARK_2:
		case CMD_SET_MARK_3:
		case CMD_SET_MARK_4:
		case CMD_SET_MARK_5:
		case CMD_SET_MARK_6:
		case CMD_SET_MARK_7:
		case CMD_SET_MARK_8:
		case CMD_SET_MARK_9:
			sinek.marks[cmd - CMD_SET_MARK_1 + 1] = xine_get_current_time(sinek.xine);
			break;
	}
}
