
/*
drop.c

A program which handles drops from several drag and drop protocols.
To compile standalone, something like:

cc -DTESTDROP drop.c -o drop -L/usr/X11R6/lib -lXt -lX11
*/

#include <stdlib.h>
#include <stdio.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include "../common/common.h"
#include "drop.h"
#include <Mowitz/Mowitz.h>

static Atom DndProtocol, DndSelection, WM_STATE;
static Widget shell;
static Display *dpy;

#ifdef TESTDROP
#define Printf printf
#else
#define Printf xdnd_printf
static int xdnd_printf(char *fmt, ...)
{
	return 0;
}
#endif

static Atom intern_atom(char *n, Bool b)
{
	Atom a = XInternAtom(dpy, n, b);
	Printf("XInternAtom(dpy, %s, %d) => %ld\n", n, (int)b, a);
	return a;
}

/* OffiX DND */
/* Based on code by Cesar Crusius */

#define DndNotDnd       -1
#define DndUnknown      0
#define DndRawData      1
#define DndFile         2
#define DndFiles        3
#define DndText         4
#define DndDir          5
#define DndLink         6
#define DndExe          7
#define DndURL          8
#define DndMIME         9
#define DndEND          10

/* We only accept the new protocol - the old one is broken and deprecated */
static int DndIsDropMessage(XEvent *event)
{
	if (event->xany.type == ClientMessage) {
Printf("ClientMessage, type = %s\n", XGetAtomName(dpy, event->xclient.message_type));
		return (event->xclient.message_type == DndProtocol);
	}
Printf("Event is not ClientMessage\n");
	return 0;
}

static long DndProtocolVersion(XEvent *event)
{
	return DndIsDropMessage(event)?(event->xclient.data.l[4]):-1;
}

static int DndDataType(XEvent *event)
{
	int type = event->xclient.data.l[0];
	if (type >= DndEND) type = DndUnknown;
	return type;
}

static void DndGetData(unsigned char **data, unsigned long *size)
{
	Window root = DefaultRootWindow(dpy);

	Atom ActualType, ActualDndSelection;
	int ActualFormat;
	unsigned long RemainingBytes;

	ActualDndSelection = DndSelection;
	XGetWindowProperty(dpy, root, ActualDndSelection,
			0L, 1000000L,
			FALSE, AnyPropertyType,
			&ActualType, &ActualFormat,
			size, &RemainingBytes,
			data);
}

static void offix_init(Widget topLevel)
{
	DndProtocol = intern_atom("_DND_PROTOCOL", FALSE);
	DndSelection = intern_atom("_DND_SELECTION", FALSE);
	WM_STATE = intern_atom("WM_STATE", True);
}

/* OffiX-specific drop handler */
static void offix_handler(XEvent *event)
{
	unsigned char *data, *fn, cmd[1000];
	unsigned long size;
	int dt = DndDataType(event);

	DndGetData(&data, &size);
	Printf("OffiX drop; protocol version %ld\n",
		DndProtocolVersion(event));
	dt = DndDataType(event);
	DndGetData(&data, &size);
	switch (dt) {
	case DndRawData:
		Printf("Raw data\n");
		break;
	case DndFile:
		fn = data;
		Printf("File %s\n", fn);
		sprintf(cmd, "egon %s", fn);
		MwSpawn(cmd);
		break;
	case DndFiles:
		fn = data;
		while (fn[0] != '\0') {
			Printf("File %s\n", fn);
			sprintf(cmd, "egon %s", fn);
			MwSpawn(cmd);
			fn += strlen(fn)+1;
		}
		break;
	case DndText:
		Printf("Text\n");
		MwSpawn(cmd);
		break;
	case DndDir:
		fn = data;
		Printf("Directory %s\n", fn);
		if (!fork()) {
			close(2);
			chdir(fn);
			execlp("egon", "Egon", (char *)0);
			exit(1);
		}
		break;
	case DndLink:
		fn = data;
		Printf("Link %s\n", fn);
		break;
	case DndExe:
		fn = data;
		Printf("Program %s\n", fn);
		MwSpawn(fn);
		break;
	case DndURL:
		fn = data;
		Printf("URL %s\n", fn);
		sprintf(cmd, "siagrun help %s", fn);
		MwSpawn(fn);
		break;
	case DndMIME:
		fn = data;
		Printf("Mime %s\n", fn);
		break;
	case DndUnknown:
		Printf("Unknown\n");
		break;
	default:
		Printf("Something's wrong\n");
	}
}


/* XDND */
/* Based on code by Paul Sheer */

#define XDND_VERSION 3
 
 
/* XdndEnter */
#define XDND_THREE 3
#define XDND_ENTER_SOURCE_WIN(e)        ((e)->xclient.data.l[0])
#define XDND_ENTER_THREE_TYPES(e)       (((e)->xclient.data.l[1] & 0x1UL) == 0)
#define XDND_ENTER_THREE_TYPES_SET(e,b) (e)->xclient.data.l[1] = ((e)->xclient.data.l[1] & ~0x1UL) | (((b) == 0) ? 0 : 0x1UL)
#define XDND_ENTER_VERSION(e)           ((e)->xclient.data.l[1] >> 24)
#define XDND_ENTER_VERSION_SET(e,v)     (e)->xclient.data.l[1] = ((e)->xclient.data.l[1] & ~(0xFF << 24)) | ((v) << 24)
#define XDND_ENTER_TYPE(e,i)            ((e)->xclient.data.l[2 + i])    /* i => (0, 1, 2) */

/* XdndPosition */
#define XDND_POSITION_SOURCE_WIN(e)     ((e)->xclient.data.l[0])
#define XDND_POSITION_ROOT_X(e)         ((e)->xclient.data.l[2] >> 16)
#define XDND_POSITION_ROOT_Y(e)         ((e)->xclient.data.l[2] & 0xFFFFUL)
#define XDND_POSITION_ROOT_SET(e,x,y)   (e)->xclient.data.l[2]  = ((x) << 16) | ((y) & 0xFFFFUL)
#define XDND_POSITION_TIME(e)           ((e)->xclient.data.l[3])
#define XDND_POSITION_ACTION(e)         ((e)->xclient.data.l[4])

/* XdndStatus */
#define XDND_STATUS_TARGET_WIN(e)       ((e)->xclient.data.l[0])
#define XDND_STATUS_WILL_ACCEPT(e)      ((e)->xclient.data.l[1] & 0x1L)
#define XDND_STATUS_WILL_ACCEPT_SET(e,b) (e)->xclient.data.l[1] = ((e)->xclient.data.l[1] & ~0x1UL) | (((b) == 0) ? 0 : 0x1UL)
#define XDND_STATUS_WANT_POSITION(e)    ((e)->xclient.data.l[1] & 0x2UL)
#define XDND_STATUS_WANT_POSITION_SET(e,b) (e)->xclient.data.l[1] = ((e)->xclient.data.l[1] & ~0x2UL) | (((b) == 0) ? 0 : 0x2UL)
#define XDND_STATUS_RECT_X(e)           ((e)->xclient.data.l[2] >> 16)
#define XDND_STATUS_RECT_Y(e)           ((e)->xclient.data.l[2] & 0xFFFFL)
#define XDND_STATUS_RECT_WIDTH(e)       ((e)->xclient.data.l[3] >> 16)
#define XDND_STATUS_RECT_HEIGHT(e)      ((e)->xclient.data.l[3] & 0xFFFFL)
#define XDND_STATUS_RECT_SET(e,x,y,w,h) {(e)->xclient.data.l[2] = ((x) << 16) | ((y) & 0xFFFFUL); (e)->xclient.data.l[3] = ((w) << 16) | ((h) & 0xFFFFUL); }
#define XDND_STATUS_ACTION(e)           ((e)->xclient.data.l[4])

/* XdndLeave */
#define XDND_LEAVE_SOURCE_WIN(e)        ((e)->xclient.data.l[0])
 
/* XdndDrop */
#define XDND_DROP_SOURCE_WIN(e)         ((e)->xclient.data.l[0])
#define XDND_DROP_TIME(e)               ((e)->xclient.data.l[2])
 
/* XdndFinished */
#define XDND_FINISHED_TARGET_WIN(e)     ((e)->xclient.data.l[0])

static long xdnd_version = XDND_VERSION;

static Atom XdndAware, XdndSelection, XdndEnter, XdndLeave,
	XdndPosition, XdndDrop, XdndFinished, XdndStatus,
	XdndActionCopy, XdndActionMove, XdndActionLink,
	XdndActionPrivate, XdndTypeList, XdndActionList,
	XdndActionDescription, text_plain;

static int XdndIsEnter(XEvent *event)
{
	return (event->xany.type == ClientMessage
		&& event->xclient.message_type == XdndEnter);
}

/* get the protocol version, supported data types and the timestamp */
static void xdnd_enter(XEvent *event)
{
	Printf("xdnd_enter()\n");
	Printf("version = %d\n", (int)XDND_ENTER_VERSION(event));
	if (XDND_ENTER_THREE_TYPES(event)) {
		int i;
		for (i = 0; i < 3; i++) {
			char *n;
			long a = XDND_ENTER_TYPE(event, i);
			if (a == 0) continue;
			n = XGetAtomName(dpy, a);
			Printf("type %d = %ld (%s)\n", i, a, n);
		}
	} else {
		Printf("More than 3 types\n");
	}
}

static int XdndIsPosition(XEvent *event)
{
	return (event->xany.type == ClientMessage
		&& event->xclient.message_type == XdndPosition);
}

static void xdnd_position(XEvent *event)
{
	XEvent reply;
	int x = 0, y = 0, w = 0, h = 0;
	Window source = XDND_POSITION_SOURCE_WIN(event);

	Printf("xdnd_position()\n");
	Printf("position = (%ld,%ld)\n",
		XDND_POSITION_ROOT_X(event), XDND_POSITION_ROOT_Y(event));
	Printf("time = %ld\n", XDND_POSITION_TIME(event));
	Printf("requested action = %ld\n", XDND_POSITION_ACTION(event));

	/* reply with a status message */
	memset(&reply, 0, sizeof reply);
	reply.xany.type = ClientMessage;
	reply.xany.display = dpy;
	reply.xclient.window = source;
	reply.xclient.message_type = XdndStatus;
	reply.xclient.format = 32;
	XDND_STATUS_TARGET_WIN(&reply) = XtWindow(shell);
	XDND_STATUS_WILL_ACCEPT_SET(&reply, 1);
	XDND_STATUS_WANT_POSITION_SET(&reply, 1);
	XDND_STATUS_RECT_SET(&reply, x, y, w, h);
	XDND_STATUS_ACTION(&reply) = XdndActionCopy;
	XSendEvent(dpy, source, 0, 0, &reply);
}

static int XdndIsLeave(XEvent *event)
{
	return (event->xany.type == ClientMessage
		&& event->xclient.message_type == XdndLeave);
}

static void xdnd_leave(XEvent *event)
{
	Printf("xdnd_leave()\n");
}

/* get the data that was dropped */
static void xdnd_requestor_callback(Widget w, XtPointer client_data,
	Atom *selection, Atom *type, XtPointer value,
	unsigned long *length, int *format)
{
	char *p = value;
	int i;

	Printf("xdnd_requestor_callback()\n");

	if ((p == NULL) && (*length == 0)) {
		Printf("No selection or timed out\n");
		return;
	}

	for (i = 0; i < *length; i++)
		putchar(p[i]);
	putchar('\n');
	XtFree(value);
}

static int XdndIsDrop(XEvent *event)
{
	return (event->xany.type == ClientMessage
		&& event->xclient.message_type == XdndDrop);
}

static void xdnd_drop(XEvent *event)
{
	XEvent reply;
	Window source = XDND_POSITION_SOURCE_WIN(event);

	Printf("xdnd_drop()\n");
	XtGetSelectionValue(shell, XdndSelection, text_plain,
		xdnd_requestor_callback, event, CurrentTime);

	/* now tell the source we are done */
	memset(&reply, 0, sizeof reply);
	reply.xany.type = ClientMessage;
	reply.xany.display = dpy;
	reply.xclient.window = source;
	reply.xclient.message_type = XdndFinished;
	reply.xclient.format = 32;
	XDND_FINISHED_TARGET_WIN(&reply) = XtWindow(shell);
	XSendEvent(dpy, source, 0, 0, &reply);
}

static void xdnd_init(Widget topLevel)
{
	XdndAware = intern_atom("XdndAware", False);
	XdndSelection = intern_atom("XdndSelection", False);
	XdndEnter = intern_atom("XdndEnter", False);
	XdndLeave = intern_atom("XdndLeave", False);
	XdndPosition = intern_atom("XdndPosition", False);
	XdndDrop = intern_atom("XdndDrop", False);
	XdndFinished = intern_atom("XdndFinished", False);
	XdndStatus = intern_atom("XdndStatus", False);
	XdndActionCopy = intern_atom("XdndActionCopy", False);
	XdndActionMove = intern_atom("XdndActionMove", False);
	XdndActionLink = intern_atom("XdndActionLink", False);
	XdndActionPrivate = intern_atom("XdndActionPrivate", False);
	XdndTypeList = intern_atom("XdndTypeList", False);
	XdndActionList = intern_atom("XdndActionList", False);
	XdndActionDescription = intern_atom("XdndActionDescription", False);
	Printf("XdndActionCopy = %ld\n", XdndActionCopy);
	text_plain = intern_atom("text/plain", False);

	XChangeProperty(dpy, XtWindow(topLevel), XdndAware,
		XA_ATOM, 32, PropModeReplace,
		(unsigned char *)&xdnd_version, 1);
}


/* Motif */
/* So far I've been unable to find useful info on the Motif protocol.
   What I have is either incomplete or on a too high level, i.e. uses
   the Motif library */

static void motif_init(Widget w)
{
	Printf("motif_init()\n");
}


/* Generic stuff for all known protocols */

/* Deal with client events for the toplevel widget */
static void drop_event(Widget w, XtPointer data, XEvent *event, Boolean *p)
{
	Printf("drop_event()\n");

	if (DndIsDropMessage(event)) {
		offix_handler(event);
		return;
	} else if (XdndIsEnter(event)) {
		xdnd_enter(event);
	} else if (XdndIsPosition(event)) {
		xdnd_position(event);
	} else if (XdndIsLeave(event)) {
		xdnd_leave(event);
	} else if (XdndIsDrop(event)) {
		xdnd_drop(event);
	}
}


/* Initialization */
void drop_init(Widget w)
{
	dpy = XtDisplay(w);
	shell = w;

   	offix_init(w);
	xdnd_init(w);
	motif_init(w);
	XtAddEventHandler(w, NoEventMask, True, drop_event, NULL);
}


#ifdef TESTDROP
/* A tiny application for demo purposes */
static XtAppContext ac;

static String fallback_resources[] = {
	"*geometry: 100x100",
	NULL
};

/* All drag and drop protocols under X have potential for disaster */
static int xerror_handler(Display *dpy, XErrorEvent *event)
{
	char b[1000];

	XGetErrorText(dpy, event->error_code, b, 999);
	puts(b);
	return 0;
}

int main(int argc, char **argv)
{
	Widget topLevel = XtVaAppInitialize(&ac, "Xdnd", NULL, 0, &argc, argv,
		fallback_resources, NULL);
	XSetErrorHandler(xerror_handler);
	XtRealizeWidget(topLevel);
	drop_init(topLevel, NULL, NULL, NULL);
	XtAppMainLoop(ac);
	return 0;
}
#endif	/* TESTDROP */
