#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include "sg_include.h"
#include "sg_err.h"

/* This program is modeled on the example code in the SCSI Programming
   HOWTO V1.5 by Heiko Eissfeldt dated 7 May 1996.
*
*  Copyright (C) 1999 D. Gilbert
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your option)
*  any later version.

   Since this code has been used in the past to form the backbone of
   some Linux apps based on the "sg" device driver, it has been
   strengthened.

   Version 0.32 (990728)
*/


#define SCSI_OFF sizeof(struct sg_header)
static unsigned char cmd[SCSI_OFF + 18];      /* SCSI command buffer */
int fd;                               /* SCSI device/file descriptor */

/* process a complete SCSI cmd. Use the generic SCSI interface. */
static int handle_SCSI_cmd(int cmd_len,       /* SCSI command length */
                           int in_size,       /* sg_hd + cmd [+ in_data] */
                           unsigned char * i_buff, 
                           int out_size,      /* sg_hd [+ out_data] */
                           unsigned char * o_buff  /* if == 0 use i_buff */
                           )
{
    int status = 0;
    struct sg_header * sg_hd;

    /* safety checks */
    if (cmd_len < 6) return -1;            /* need a cmd_len != 0 */
    if (! i_buff) return -1;             /* need an input buffer != NULL */

    if (!o_buff) out_size = 0;      /* no output buffer, no output size */

    /* generic SCSI device header construction */
    sg_hd = (struct sg_header *)i_buff;
    sg_hd->reply_len   = SCSI_OFF + out_size;
    sg_hd->twelve_byte = cmd_len == 12;
    sg_hd->result = 0;
    sg_hd->pack_len = SCSI_OFF + cmd_len + in_size; /* not necessary */
    sg_hd->pack_id = 0;     /* not used internally, but passed back */
    sg_hd->other_flags = 0; /* not used */

    /* send command */
    status = write( fd, i_buff, SCSI_OFF + cmd_len + in_size );
    if ( status < 0 || status != SCSI_OFF + cmd_len + in_size || 
                       sg_hd->result ) {
        /* some error happened */
        fprintf( stderr, "write(generic) result = 0x%x cmd = 0x%x\n",
                    sg_hd->result, i_buff[SCSI_OFF] );
        perror("");
        return status;
    }
    
    if (!o_buff) o_buff = i_buff;       /* buffer pointer check */

    /* retrieve result */
    status = read( fd, o_buff, SCSI_OFF + out_size);
    if ( status < 0 || status != SCSI_OFF + out_size || sg_hd->result ) {
        /* some error happened */
        fprintf( stderr, "read(generic) status = 0x%x, result = 0x%x, "
                         "cmd = 0x%x\n", 
                         status, sg_hd->result, o_buff[SCSI_OFF] );
        fprintf( stderr, "read(generic) sense "
                "%x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x\n", 
                sg_hd->sense_buffer[0],         sg_hd->sense_buffer[1],
                sg_hd->sense_buffer[2],         sg_hd->sense_buffer[3],
                sg_hd->sense_buffer[4],         sg_hd->sense_buffer[5],
                sg_hd->sense_buffer[6],         sg_hd->sense_buffer[7],
                sg_hd->sense_buffer[8],         sg_hd->sense_buffer[9],
                sg_hd->sense_buffer[10],        sg_hd->sense_buffer[11],
                sg_hd->sense_buffer[12],        sg_hd->sense_buffer[13],
                sg_hd->sense_buffer[14],        sg_hd->sense_buffer[15]);
        if (status < 0)
            perror("");
    }
    /* Look if we got what we expected to get */
    if (status == SCSI_OFF + out_size) status = 0; /* got them all */

    return status;  /* 0 means no error */
}

#define INQUIRY_CMD     0x12
#define INQUIRY_CMDLEN  6
#define INQUIRY_REPLY_LEN 96
#define INQUIRY_VENDOR  8       /* Offset in reply data to vendor name */


static unsigned char Inqbuffer[ SCSI_OFF + INQUIRY_REPLY_LEN ];


/* request vendor brand and model */
static unsigned char *Inquiry ( void )
{
  unsigned char cmdblk [ INQUIRY_CMDLEN ] = 
      { INQUIRY_CMD,  /* command */
                  0,  /* lun/reserved */
                  0,  /* page code */
                  0,  /* reserved */
  INQUIRY_REPLY_LEN,  /* allocation length */
                  0 };/* reserved/flag/link */

  memcpy( cmd + SCSI_OFF, cmdblk, sizeof(cmdblk) );

  /*
   * +------------------+
   * | struct sg_header | <- cmd
   * +------------------+
   * | copy of cmdblk   | <- cmd + SCSI_OFF
   * +------------------+
   */

  if (handle_SCSI_cmd(sizeof(cmdblk), 0, cmd, 
                      sizeof(Inqbuffer) - SCSI_OFF, Inqbuffer )) {
      fprintf( stderr, "Inquiry failed\n" );
      exit(2);
  }
  return (Inqbuffer + SCSI_OFF);
}

int main(int argc, char * argv[])
{
  if (2 != argc) {
    printf("Usage:  sg_inquiry <sg_device>     eg: sg_inquiry /dev/sgb\n");
    return 1;
  }
  fd = open(argv[1], O_RDWR);
  if (fd < 0) {
    fprintf(stderr, "Error trying to open %s\n", argv[1]);
    perror("");
    return 1;
  }
  if (ioctl(fd, SG_GET_TIMEOUT, 0) < 0) {
    fprintf( stderr, "Given file not a SCSI generic device\n" );
    close(fd);
    return 1;
  }

  /* print some fields of the Inquiry result */
  printf( "%s\n", Inquiry() + INQUIRY_VENDOR );
  return 0;
}
