/***************************************************************************
 *  Module:      $Id: queue.c,v 2.20 1997/12/31 05:18:32 nemesis Exp nemesis $
 *  Description: queueing and support functions for sendpage
 *  Author:      maf, cjc
 *
 * Copyright (c) 1995 Mark Fullmer and The Ohio State University
 * Copyright (c) 1997 Cornelius Cook and Counterpoint Networking, Inc.
 * http://www.cpoint.net/projects/sendpage
 ***************************************************************************/
/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
$Log: queue.c,v $
Revision 2.20  1997/12/31 05:18:32  nemesis
CLOCAL for all, GCC fixes, WAIT_WORD finished

Revision 2.19  1997/12/29 02:45:13  nemesis
include strengthening.  works with db 2.0+1.85 compats.  compiles/runs on HPUX

Revision 2.18  1997/12/28 23:20:45  nemesis
include cleanups, configure additions/corrections

Revision 2.17  1997/12/27 20:53:04  nemesis
db.1.85 stuff, maxTransactions implemented

Revision 2.16  1997/12/26 02:37:31  nemesis
code clean up, b* -> mem* funcs, finding NL -> # bug

Revision 2.15  1997/12/25 08:30:52  nemesis
code cleanups

Revision 2.14  1997/12/25 06:44:34  nemesis
time_t declarations and "make install" fix

Revision 2.13  1997/12/24 21:50:04  nemesis
mailing list updates

Revision 2.12  1997/12/24 21:42:54  nemesis
0.8a released.

Revision 2.11  1997/12/24 21:02:02  nemesis
more changes

Revision 2.10  1997/12/24 20:56:03  nemesis
gearing up for 0.8a more

Revision 2.9  1997/12/24 20:45:35  nemesis
trying to make 0.8a release

Revision 2.8  1997/12/24 20:29:08  nemesis
fixed up autoconf modifications, cleaned up signal stuff

Revision 2.7  1997/12/24 20:15:13  nemesis
sendpage.h now mostly contained within 'configure'

Revision 2.6  1997/12/24 19:52:30  nemesis
synching version numbers

Revision 2.5  1997/12/24 19:52:04  nemesis
fixing posix checking

Revision 2.4  1997/12/24 19:41:49  nemesis
posix additions, syslog autoconf'd

Revision 2.3  1997/12/24 19:33:03  nemesis
check for POSIX

Revision 2.2  1997/12/17 09:37:47  nemesis
versioning

Revision 1.3  1997/12/17 09:37:27  nemesis
more autoconf changes... mostly strerror.o

Revision 1.2  1997/12/15 17:37:26  nemesis
Added patches by Steve Kennedy <steve@demon.net>
See http://www.gbnet.net/public/paging/sendpage-multi/

Revision 1.1  1997/12/15 15:56:44  nemesis
Initial revision

 * Revision 1.19  1996/02/14  09:49:23  maf
 * ARSYSTEM_NOTIFIER
 *
 * Revision 1.18  1996/02/14  07:09:57  maf
 * GLOBAL_FLAGS_MONTH_FIRST implementation
 *
 * Revision 1.17  1996/02/13  11:17:55  maf
 * PrintQueue works again
 *
 * Revision 1.16  1996/02/13  04:38:35  maf
 * MSGQ_FLAGS to replace STATUS flags
 *
 * Revision 1.15  1996/02/12  03:51:47  maf
 * emailCC, profile fixes
 *
 * Revision 1.14  1996/02/11  06:04:39  maf
 * missing null terminator when splitting messages
 *
 * Revision 1.13  1995/12/18  00:02:19  maf
 * date off by one fix
 * DEFAULT_QRUN_MODE implementation
 * EMAIL_ERRORS_ONLY implementation
 * fix for unknown recipient coredump
 * mode flag to AddMessageQueue() for less confusing logs
 *
 * Revision 1.12  1995/11/13  07:08:25  maf
 * implemented message splitting
 *
 * Revision 1.11  1995/11/13  04:44:06  maf
 * *** empty log message ***
 *
 * Revision 1.10  1995/11/13  03:28:57  maf
 * *** empty log message ***
 *
 * Revision 1.9  1995/10/09  01:40:21  maf
 * *** empty log message ***
 *
 * Revision 1.8  1995/08/28  02:35:47  maf
 * *** empty log message ***
 *
 * Revision 1.7  1995/08/28  02:00:07  maf
 * stdio bug, comments, lastEntry bug
 *
 * Revision 1.6  1995/05/24  02:08:57  maf
 * fixed previous queue file bug again (or maybe for real!)
 *
 * Revision 1.5  1995/05/24  00:51:43  maf
 * all unknown string sizes use %.512s for report()
 * fixed race condition in ExpandQueue()
 *  - entries could get marked for processing twice, but they still would
 *  - only get done once since the queue file was removed, which caused
 *  - a strange warning message.
 *
 * Revision 1.4  1995/03/17  04:34:35  maf
 * deferstring not initialized correctly.
 *
 * Revision 1.3  1995/03/17  04:13:14  maf
 * entry->len not set correctly in AddMessagegeQueue()
 * abandon packet == page rejected, not retry later
 * fixed logic to send replies
 *
 * Revision 1.2  1995/03/15  04:40:55  maf
 * *** empty log message ***
 *
 * Revision 1.1  1995/01/10  01:43:50  maf
 * Initial revision
 *
*/


#include "sendpage.h"
#include "report.h"
#include "queue.h"
#include "cf.h"
#include "io.h"
#include "deliver.h"

/*********************************************************************
 * Function: AddMessageQueue
 *	add a message to the disk based queue
 *
 *	Returns:	0	good
 *		       !0	bad
 *
 * args:
 *		entry		pointer to struct messagequeue.
 *				caller should initialize set all entries 
 *                              to appropiate values
 *				ie senderlen to strlen(sender), 
 *                                 queuetime to current time, etc.
 *		sender		pointer to sender
 *		recipient	pointer to recipient (
 *		recipient2	pointer to recipient2 
 *                                (this is recipient that has gone
 *				   through alias expansion)
 *		message		pager message
 *		qfname2		set by function to file name of queue message.
 *
 *	The queue is written with a single wrivtev to prevent race conditions
 *		(ie client is writing queue wile daemon is processing it)
 *		0 length queue files are just skipped by the daemon.
 *
 *	errors are reported via report()
 *
 *********************************************************************/
int AddMessageQueue(struct messagequeue *entry, char *sender, char *recipient,
		    char *recipient2, char *message, char *emailCC,
		    char **qfname2, int mode) {
  extern int errno;
  int fd;
  int err;
  struct iovec vec[11];
  char nullbyte;
  /* 6 bytes for XXXXXX 10 bytes for max size of unsigned long in ascii */
  /* note sizeof also returns the null byte so the trailing 0 is included */
  static char qfname[sizeof QUEUE_HEAD + 16];
  
  fd = -1;
  err = 1; /* bad */
  nullbyte = 0;
  *qfname2 = (char*)qfname;
  
  /* make sure entry->*len are set correctly */
  entry->recipientlen = strlen(recipient);
  entry->recipient2len = strlen(recipient2);
  entry->senderlen = strlen(sender);
  entry->messagelen = strlen(message);
  entry->emailCCLen = strlen(emailCC);
  
  /* Remove trailing whitespace (including NL) */
  while(entry->messagelen>1 && ( message[entry->messagelen-1]==' '
				 || message[entry->messagelen-1]=='\n')) {
    message[--entry->messagelen]='\0';
  }
  
  /* construct prototype temp file */
  sprintf(qfname, "%s%luXXXXXX", QUEUE_HEAD, (long unsigned int)entry->queuetime);
  
  /* create and open the file */
  if ((fd = mkstemp(qfname)) == -1) {
    report (LOG_ERR, "mkstemp(%s%luXXXXXX): failed", QUEUE_HEAD,
	    (long unsigned int)entry->queuetime);
    perror("mkstemp");
    goto AddMessageQueueout;
  }
  
  /* write out the queue entry.  This needs to be atomic otherwise there 
     is a race condition with the daemon reading part of the file */
  /* the string fields are also null terminated here */
  vec[0].iov_base = (caddr_t)entry;
  vec[0].iov_len = sizeof(struct messagequeue);
  
  vec[1].iov_base = (caddr_t)recipient;
  vec[1].iov_len = entry->recipientlen;
  vec[2].iov_base = (caddr_t)&nullbyte;
  vec[2].iov_len = 1;
  
  vec[3].iov_base = (caddr_t)recipient2;
  vec[3].iov_len = entry->recipient2len;
  vec[4].iov_base = (caddr_t)&nullbyte;
  vec[4].iov_len = 1;
  
  vec[5].iov_base = (caddr_t)sender;
  vec[5].iov_len = entry->senderlen;
  vec[6].iov_base = (caddr_t)&nullbyte;
  vec[6].iov_len = 1;
  
  vec[7].iov_base = (caddr_t)message;
  vec[7].iov_len = entry->messagelen;
  vec[8].iov_base = (caddr_t)&nullbyte;
  vec[8].iov_len = 1;
  
  vec[9].iov_base = (caddr_t)emailCC;
  vec[9].iov_len = entry->emailCCLen;
  vec[10].iov_base = (caddr_t)&nullbyte;
  vec[10].iov_len = 1;
  
  if (writev (fd, vec, 11) == -1) {
    report (LOG_ERR, "writev(): %s", strerror(errno));
    goto AddMessageQueueout;
  }
  
  err = 0; /* good */
  
  report (LOG_INFO,
	  "%.3s: %.512s: from=%.512s, size=%d, to=%.512s, status=%x, msg=%.512s",
	  (mode == 0) ? "Que" : "Run",
	  qfname, sender, entry->messagelen, recipient, entry->status, message);
  
AddMessageQueueout:
  
  if (fd != -1)
    err |= close (fd);
  
  return err;
  
} /* AddMessageQueue */

/*********************************************************************
 * Function: PrintMessageQueue
 *	print the contents of the message queue 
 *
 *	Returns	 0	good
 *		!0	bad
 *
 *	The disk based message queue is printed to stdout.  If a message is
 *	being delivered, a * precedes it
 *
 *	errors are reported to stderr
 *
 *********************************************************************/
int PrintMessageQueue() {
  extern int errno;
  int err, fd, n;
  struct dirent *dir_entry;
  DIR *dirp;
  int entries;
  struct stat statbuf;
  struct messagequeue *entry;
  char *queuebuf;
  int queuebufsize;
  struct tm *ptm, tm;
  char *sender, *recipient, *recipient2;
  char *tmpp;
  FILE *FP;
  unsigned int pid;
  
  err = 1; /* bad */
  entries = 0;
  dirp = (DIR*)NULL;
  fd = -1; /* bad */
  queuebuf = (char*)NULL;
  pid = 0;
  
  /* alert the user of the daemon is not running */
  if ((FP = fopen(PATH_SENDPAGE_PIDFILE, "r"))) {
    
    fscanf(FP, "%u", &pid);
    
    if (kill ((int)pid, 0))
      fprintf(stderr, "warning, daemon does not appear to be running.\n");
    fclose(FP);
  } else {
    /* no pidfile to read... */
    fprintf(stderr,
	    "warning, can't read daemon pidfile %s\n", PATH_SENDPAGE_PIDFILE);
  }
  
  
  /* allocate memory for contents of queue'd entry (may be expanded later) */
  if (!(queuebuf = (char*)malloc((queuebufsize = 1024)))) {
    fprintf(stderr, "malloc(1024) failed\n");
    goto PrintMessageQueueout;
  }
  
  /* allready did a chdir to PATH_QUEUE... */
  if (!(dirp = opendir("."))) {
    fprintf(stderr, "Can't opendir() %s\n", PATH_QUEUE);
    goto PrintMessageQueueout;
  }
  
  for (dir_entry = readdir(dirp); dir_entry; dir_entry = readdir(dirp)) {
    
    /* skip . and .. */
    if (dir_entry->d_name[0] == '.')
      if (!dir_entry->d_name[1])
	continue;
    if (dir_entry->d_name[1] == '.')
      if (!dir_entry->d_name[2])
	continue;
    
    /* skip anything that doesn't begin with QUEUE_HEAD */
    if (memcmp(dir_entry->d_name, QUEUE_HEAD, sizeof (QUEUE_HEAD)-1))
      continue;
    
    ++entries;
    
    /* ignore open errors - there's a race condition that means
       an open error here is probably because the daemon deleted
       it
    */
    if ((fd = open(dir_entry->d_name, O_RDONLY, 0)) == -1) {
      continue;
    }
    
    /* it's opened, so if was deleted, its really not */
    if (fstat(fd, &statbuf) == -1) {
      perror(dir_entry->d_name);
      goto PrintMessageQueueout;
    }
    
    /* sanity check thing could overflow below if  */
    if (statbuf.st_size > 32767)
      goto PrintMessageQueueout;
    
    /* make sure there's enough buffer space available to read it in */
    if (statbuf.st_size > queuebufsize) {
      
      queuebufsize += statbuf.st_size;
      
      tmpp = queuebuf;
      if (!(queuebuf = (char*)realloc(queuebuf, queuebufsize))) {
	queuebuf = tmpp;
	fprintf(stderr, "realloc(%d) failed\n", queuebufsize);
	goto PrintMessageQueueout;
      }
    }
    
    if ((n = read(fd, queuebuf, (int)statbuf.st_size)) == -1) {
      perror(dir_entry->d_name);
      goto PrintMessageQueueout;
    }
    
    if (close(fd))
      perror(dir_entry->d_name);
    fd = -1;
    
    if ((n != statbuf.st_size) || (n < sizeof (struct messagequeue))) {
      fprintf(stderr, "short queue file -- %s, skipping\n",
	      dir_entry->d_name);
      continue;
    }
    
    /* struct comes first */
    entry = (struct messagequeue*)queuebuf;
    recipient = queuebuf+sizeof(struct messagequeue);
    recipient2 = recipient + entry->recipientlen + 1;
    sender = recipient2 + entry->recipient2len + 1;
    
    /* make sure this looks valid */
    n = sizeof(struct messagequeue) + entry->recipientlen +
      entry->recipient2len + entry->senderlen + entry->messagelen + 
      entry->emailCCLen + 5;
    if (n != statbuf.st_size) {
      fprintf(stderr, "corrupt queue file -- %s, skipping\n",
	      dir_entry->d_name);
      continue;
    }
    
    if (entries == 1)  {
      printf("Sendpage queue entries\n----------------------\n\n");
      
      printf ("#/Retries From                     To                       Size      When\n");
      printf ("--------------------------------------------------------------------------------\n");
    } /* if */
    
    if (!(ptm = localtime(&(entry->queuetime)))) {
      fprintf(stderr, "locatime() failed\n");
      goto PrintMessageQueueout;
    } else
      memcpy(&tm, ptm, sizeof(tm));
    
    printf("%c%-3d/%3d  %-24.24s %-24.24s %-4d %d/%d %d:%d:%d\n",
	   entry->status&STATUS_DELIVERING ? (char)'*' : (char)' ', entries,
	   entry->retries, sender, recipient, entry->messagelen,
	   tm.tm_mon+1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);
    
  } /* for dirent */
  
  if (!entries)
    printf("Pager queue is empty\n");
  
  
PrintMessageQueueout:
  
  if (fd != -1)
    err |= close(fd);
  
  if (dirp)
    err |= closedir(dirp);
  
  if (queuebuf)
    free (queuebuf);
  
  return err;
  
} /* PrintMessageQueue */

/*********************************************************************
 * Function: QueueExpand
 *	Alias expands queue'd messages, and requeue's them so each queue
 *	entry has a single recipient (aliases can expand to multiple recipnts)
 *
 *	Returns	 0	good
 *		!0	bad
 *
 *	QueueExpand goes through the disk based message queue, and tries to
 *	do an alias expansion on each recipient.  If the alias expansion
 *	doesn't result in anything, the message is requeue'd with recipient2
 *	equal to recipient, and the STATUS_EXPANDED bit set in the status var.
 *
 *	If the alias does expand, each entry in the list of the alias expansion
 *	is queue'd as the original, but with recipient 2 is set to the list
 *      entry and the STATUS_EXPANDED bit set.
 *
 *	If the STATUS_EXPANDED bit is set, QueueExpand doesn't touch the entry.
 *
 *	Furthermore, dlist is (which is allocated and free()'d by the caller)
 *	will point to a list of queue entries for each paging central type.
 *	This effectively sorts the queue so multiple messages to the same
 *	paging central can be delivered in the same phone call.  The PET
 *	protocol supports this, maybe others won't...
 *
 *	errors are reported via report()
 *
 *********************************************************************/
int QueueExpand(DB* aliasdb, DB* pcdb, DB* profdb, DB* trdb, u_int ncentral,
		struct dynstring *dlist) {
  extern int errno;
  struct dynstring aliaslist;
  DBT dbkey;
  struct dirent *dir_entry;
  DIR *dirp;
  struct stat statbuf;
  struct messagequeue *entry, tmpentry;
  char *sender, *recipient, *recipient2, *message, *newMsg2;
  char *tmpp, nullbyte, *qfname, *p, *qfilename, *queuebuf, *newMsg;
  int err, len=0, fd, n, queuebufsize, maxMsgSize, nlen, nSplitMsg;
  struct dynstring dirstring;
  struct profile userProfile;
  struct pcinfo pc, temp_pc;
  int temp_flags ;
  char *splitMsg[MAX_MSG_SPLIT], *emailCC;
  extern struct global globalOptions;
  
  err = 1; /* bad */
  queuebuf = (char*)NULL;
  fd = -1;
  dirp = (DIR*)NULL;
  bzero(&aliaslist, sizeof(aliaslist));
  bzero(&dirstring, sizeof(dirstring));
  nullbyte = 0;
  newMsg = (char*)0;
  bzero(splitMsg, MAX_MSG_SPLIT * sizeof (char*));
  

  /* The delivery list is an array of pointers to dynstrings.
     Each dynstring is a list of queue filenames corresponding to the
     paging central definition to use.
     
     index 0 contains the filenames that are invalid - the caller of this f
             will discard these, maybe notifying the sender
     index 1 is the first paging central definition in sendpage.cf
     index 2 is the second paging central definition in sendpage.cf

     The indexes are stored while reading in the config file, and are
     the first entry in the pcdb.data list
     
     caller sets up dlist

  */

  /* allocate memory for contents of queue'd entry (may be expanded later) */
  if (!(queuebuf = (char*)malloc((queuebufsize = 1024)))) {
    report (LOG_ERR, "malloc(1024) failed");
    goto QueueExpandout;
  }
  
  if (!(dirp = opendir("."))) {
    fprintf(stderr, "Can't opendir() %s\n", PATH_QUEUE);
    goto QueueExpandout;
  }
  
  /*
   * read the directory contents into a dynstring --
   * since files will be added and removed from this directory
   * its necessary to read it first, else there is a race
   * condition.
   */

  for (dir_entry = readdir(dirp); dir_entry; dir_entry = readdir(dirp)) {
    
    /* skip . and .. */
    if (dir_entry->d_name[0] == '.')
      if (!dir_entry->d_name[1])
	continue;
    if (dir_entry->d_name[1] == '.')
      if (!dir_entry->d_name[2])
	continue;
    
    /* skip anything that doesn't begin with QUEUE_HEAD */
    if (memcmp(dir_entry->d_name, QUEUE_HEAD, sizeof (QUEUE_HEAD)-1)) 
      continue;
    
    if (AddToDynString(&dirstring, dir_entry->d_name, NAMLEN(dir_entry))) {
      report(LOG_ERR, "can't read directory into buf");
      goto QueueExpandout;
    }
  } /* for dirent */
  
  /* add trailing null byte */
  if (AddToDynString(&dirstring, &nullbyte, 0)) {
    report(LOG_ERR, "can't AddToDynstring nullbyte");
    goto QueueExpandout;
  }
  
  for (qfilename = dirstring.buf; *qfilename;
       qfilename += strlen(qfilename)+1) {
    
    if ((fd = open(qfilename, O_RDONLY, 0)) == -1) {
      report (LOG_ERR,
	      "open(%.512s): %s", qfilename, strerror(errno));
      goto QueueExpandout;
    }
    
    /* it's opened, so if was deleted, its really not */
    if (fstat(fd, &statbuf) == -1) {
      report (LOG_ERR, "fstat(%.512s): %s",
	      qfilename, strerror(errno));
      goto QueueExpandout;
    }
    
    /* sanity check thing could overflow below if  */
    if (statbuf.st_size > 32767)
      goto QueueExpandout;
    
    /* make sure there's enough buffer space available to read it in */
    if (statbuf.st_size > queuebufsize) {
      queuebufsize += statbuf.st_size;
      
      tmpp = queuebuf;
      if (!(queuebuf = (char*)realloc(queuebuf, queuebufsize))) {
	queuebuf = tmpp;
	report(LOG_ERR, "realloc(%d) failed", queuebufsize);
	goto QueueExpandout;
      }
    }
    
    if ((n = read(fd, queuebuf, (int)statbuf.st_size)) == -1) {
      report(LOG_ERR, "read(%.512s): %s",
	     qfilename, strerror(errno));
      goto QueueExpandout;
    }
    
    if (close(fd))
      report(LOG_ERR, "close(%.512s): %s",
	     qfilename, strerror(errno));
    fd = -1;
    
    if ((n != statbuf.st_size) || (n < sizeof (struct messagequeue))) {
      report(LOG_WARNING, "short queue file -- %.512s, skipping",
	     qfilename);
      continue;
    }
    
    /* struct comes first */
    entry = (struct messagequeue*)queuebuf;
    
    recipient = queuebuf+sizeof(struct messagequeue);
    recipient2 = recipient + entry->recipientlen + 1;
    sender = recipient2 + entry->recipient2len + 1;
    message = sender + entry->senderlen + 1;
    emailCC = message + entry->messagelen + 1;
    newMsg2 = message;
    
    /* if this entry has allready been expanded, then skip it */
    if (entry->status & STATUS_EXPANDED) {
      
      if (AddToDelivery(dlist, pcdb, ncentral, qfilename,
			recipient2)) {
	report (LOG_ERR, "AddToDelivery() failed for %.512s",
		qfilename);
	goto QueueExpandout;
      }
      
      continue;
    }
    
    /*
     * if this entry has a message size greater than the
     * global max message size, truncate it
     */
    if (entry->messagelen > globalOptions.maxMsgSize) {
      entry->messagelen = globalOptions.maxMsgSize;
      entry->flags |= MSGQ_FLAGS_SIZETRUNCATE1;
    }
    
    dbkey.data = recipient;
    dbkey.size = entry->recipientlen;
    
    /*
     * Determine the user profile:
     *	lookup this recipient in the profile database.  If it doesn't
     *	exist try using "default", else use the hard coded defaults
     *	to fill in userProfile
     */
    
    if (GetUserProfile(recipient, &userProfile, profdb)) {
      report (LOG_ERR, "GetUserProfile() failed");
      goto QueueExpandout;
    }
    
    aliaslist = AliasExpand(aliasdb, dbkey);
    
    if (aliaslist.bufused == -1) {
      report (LOG_ERR, "AliasExpand() returned an error condition");
      goto QueueExpandout;
    }
    
    /* if no expansion, just requeue this entry with the STATUS_EXPANDED
       bit set and recipient2 equal to recipient */
    
    if (!aliaslist.bufused) {
      /* setup a dummy list */
      
      entry->status |= STATUS_EXPANDED;
      memcpy(&tmpentry, entry, sizeof tmpentry);
      
      len = strlen(recipient);
      
      if (!(aliaslist.buf = (char*)malloc(len+2))) {
	report (LOG_ERR, "malloc() failed");
	goto QueueExpandout;
      }
      
      memcpy(aliaslist.buf, recipient, len);
      aliaslist.buf[len] = 0;
      aliaslist.buf[len+1] = 0; /* this is a list...*/
    } else {
      
      tmpentry.queuetime = entry->queuetime;
      tmpentry.retries = 0;
      tmpentry.status = entry->status | STATUS_EXPANDED;
      tmpentry.senderlen = entry->senderlen;
      tmpentry.messagelen = entry->messagelen;
      tmpentry.recipientlen = entry->recipientlen;
      tmpentry.recipient2len = len;
      tmpentry.emailCCLen = entry->emailCCLen;
      tmpentry.flags = entry->flags;
    }
    
    p = aliaslist.buf;
    
    /*
     * foreach rhs of the alias expansion
     */
    for (; (len = strlen(p)); p += len+1) {
      
      /*
       * if the user profile has the PROF_FLAGS_MAIL_GOOD
       * bit set, set this option in the message queue entry
       */
      if (userProfile.flags & PROF_FLAGS_MAIL_GOOD)
	tmpentry.flags |= MSGQ_FLAGS_MAIL_GOOD;
      
      /*
       * if the user profile has the PROF_FLAGS_MAIL_BAD bit
       * set, set this option in the message queue entry
       */
      if (userProfile.flags & PROF_FLAGS_MAIL_BAD)
	tmpentry.flags |= MSGQ_FLAGS_MAIL_BAD;
      
      /*
       * if tmpentry.flags has the PROF_FLAGS_MAIL_NO
       * bit set, then disable all mail replies --
       * MAIL_NO is set from the client to override 
       * run time configurations
       */
      if (tmpentry.flags & MSGQ_FLAGS_MAIL_NONE) {
	tmpentry.flags &= ~MSGQ_FLAGS_MAIL_NONE;
	tmpentry.flags &= ~MSGQ_FLAGS_MAIL_GOOD;
	tmpentry.flags &= ~MSGQ_FLAGS_MAIL_BAD;
      }
      
      /*
       * if the user profile has emailCC defined as a non
       * empty string, copy this info to the message q
       */
      tmpentry.emailCCLen = strlen(userProfile.emailCC);
      emailCC = userProfile.emailCC;
      /*
       * Determine paging central this belongs to
       */
      if (GetPagingCentral(p, len, &pc, pcdb)) {
	report (LOG_ERR, "Paging central error %s", p);
	goto skip1QueueExpand; /* deal with this later */
      }
      
      report (LOG_ERR, "Paging central %s", p);
      
      /*
       * lets frig the paging central in case it's a dopple SK
       */
      
      if (pc.pc_alias) {
	if (GetPagingCentral(pc.alias_name, len, &temp_pc, pcdb)) {
	  report (LOG_ERR, "Paging (alias) central %s", pc.alias_name);
	  goto skip1QueueExpand; /* deal with this later */
	}
	
	if (pc.dopple_set) {
	  if (pc.dopple_set & DOPPLE_TRMAP)
	    strcpy(temp_pc.trMap, pc.trMap);
	  if (pc.dopple_set & DOPPLE_MAXMSGSIZE)
	    temp_pc.maxMsgSize = pc.maxMsgSize;
	  if (pc.dopple_set & DOPPLE_MANGLE)
	    temp_pc.mangle = pc.mangle;
	}
	
	if (pc.pc_set) {
	  temp_pc.flags = 
	    ((~pc.pc_set & temp_pc.flags) |
	     (pc.pc_set & pc.flags));
	}
	
	/*
	 * Now copy the revised temp paging central into the orig
	 */
	memcpy(&pc, &temp_pc, sizeof(pc));
      }
      /* end dopple fix */

      
      /* maxMsgSize is lowest(userProfile.maxMsgSize, pc.maxMsgSize) */
      /* Gregc: Unless user profile is 0. */
      maxMsgSize = ((userProfile.maxMsgSize < pc.maxMsgSize)
		    && (userProfile.maxMsgSize > 0)) ?
	userProfile.maxMsgSize : pc.maxMsgSize;
      if (maxMsgSize == 0)
	maxMsgSize = DEFAULT_MAX_PAGELEN;
      
      /*
       * run message through translation maps
       */
      if (userProfile.trMap[0])
	if (MessageTranslate(userProfile.trMap, message, trdb)) {
	  report (LOG_ERR, "MessageTranslate() failed");
	  goto QueueExpandout;
	}
      
      if (pc.trMap[0])
	if (MessageTranslate(userProfile.trMap, message, trdb)) {
	  report (LOG_ERR, "MessageTranslate() failed");
	  goto QueueExpandout;
	}
      
      /*
       * prepend any interesting info like sender or date/time
       * Added pc.flags section SK
       */
      temp_flags = 0 ;
      
      if (pc.pc_set) {
	if (pc.pc_set & PROF_FLAGS_INC_SENDER) {
	  temp_flags |= (pc.flags & PROF_FLAGS_INC_SENDER);
	} else {
	  temp_flags |= (userProfile.flags & PROF_FLAGS_INC_SENDER);
	} 
	if (pc.pc_set & PROF_FLAGS_INC_DATE) {
	  temp_flags |= (pc.flags & PROF_FLAGS_INC_DATE);
	} else {
	  temp_flags |= (userProfile.flags & PROF_FLAGS_INC_DATE);
	} 
	if (pc.pc_set & PROF_FLAGS_INC_TIME) {
	  temp_flags |= (pc.flags & PROF_FLAGS_INC_TIME);
	} else {
	  temp_flags |= (userProfile.flags & PROF_FLAGS_INC_TIME);
	}
      } else
	temp_flags = userProfile.flags;
      
      if (MessagePrepender(message, tmpentry.messagelen,
			   &newMsg, &nlen, sender, tmpentry.senderlen,
			   tmpentry.queuetime, temp_flags)) {
	report (LOG_ERR, "MessagePrepender() failed");
	goto QueueExpandout;
      }

      /* If MessagePrepender() did no work, newMsg is not allocated */
      if (nlen) {
	tmpentry.messagelen = nlen;
	newMsg2 = newMsg;
      } else
	newMsg2 = message;
      
      /*
       * Do any necessary message splitting
       */
      
      if (MessageSplit(newMsg2, tmpentry.messagelen, splitMsg,
		       maxMsgSize, &nSplitMsg)) {
	report (LOG_ERR, "MessageSplit() failed");
	goto QueueExpandout;
      }
      
      if (nSplitMsg) { /* did the split */
	
	for (n = 0; n < nSplitMsg; ++n) {
	  
	  if (AddMessageQueue(&tmpentry, sender, recipient, p,
			      splitMsg[n], emailCC, &qfname, 1)) {
	    report (LOG_ERR, "addMessageQueue() failed");
	    goto QueueExpandout;
	  } 
	  
	  if (AddToDelivery(dlist, pcdb, ncentral, qfname, p)) {
	    report (LOG_ERR, "AddToDelivery() failed for %.512s",
		    qfname);
	    goto QueueExpandout;
	  }
	  
	} /* for */
	
	for (n = 0; n < MAX_MSG_SPLIT; ++n) {
	  if (splitMsg[n]) {
	    free (splitMsg[n]);
	    splitMsg[n] = (char*)0L;
	  } else {
	    break; /* splitMsg is allocated contiguously */
	  }
	}
	
      } else {
	
skip1QueueExpand:
	
	if (AddMessageQueue(&tmpentry, sender, recipient, p, newMsg2,
			    emailCC, &qfname, 1)) {
	  report (LOG_ERR, "addMessageQueue() failed");
	  goto QueueExpandout;
	} 
	
	if (AddToDelivery(dlist, pcdb, ncentral, qfname, p)) {
	  report (LOG_ERR, "AddToDelivery() failed for %.512s",
		  qfname);
	  goto QueueExpandout;
	}
      }
      
    } /* for p */
    
    /* and remove the old one */
    if (unlink(qfilename) == -1)  {
      report (LOG_ERR, "unlink(%.512s): %s", qfilename,
	      strerror(errno));
      goto QueueExpandout;
    } 
    
  } /* for each in dirstring */
  
  /* mark the end of each delivery list */
  for (n = 0; n < ncentral + 1; ++n) 
    if (dlist[n].buf) 
      AddToDynString(&dlist[n], &nullbyte, 0);
  
  err = 0;
  
QueueExpandout:
  
  if (fd != -1)
    err |= close(fd);
  
  if (dirp)
    err |= closedir(dirp);
  
  if (queuebuf)
    free (queuebuf);
  
  if (aliaslist.buf)
    free (aliaslist.buf);
  
  if (dirstring.buf)
    free (dirstring.buf);
  
  if (newMsg)
    free (newMsg);
  
  for (n = 0; n < MAX_MSG_SPLIT; ++n) 
    if (splitMsg[n])
      free (splitMsg[n]);
    else
      break; /* quick out, splitMsg is allocated contiguously */
  
  return err;
  
} /* QueueExpand */

/*********************************************************************
 * Function: AddToDelivery
 *	Adds a recipient to the delivery list 
 *
 *	Returns	 0	good
 *     		!0	bad
 *
 *	Takes a recipient that should of previously been alias expanded, 
 *	and determintes which delivery list to add it to.
 *	dlist[0] is the error list	CF_UNKNOWN == 0
 *	
 *	The delivery list is determined by looking in the PAGINGCENTRAL
 *	part of the recipient (ID.PAGINGCENTRAL).  If there is no PAGINGCENTRAL
 *	part, or the PAGINGCENTRAL part could not be looked up, the entry
 *	is added to the error list.
 *	If the PAGINGCENTRAL part was looked up, the appropiate delivery list
 *	is determined from the first string returned from the pcdb->get value.
 *
 *	dlist needs to be an array of ncentral+1 before calling, initially
 *	each entry in dlist needs to be bzero()'d.
 *
 *	AddToDynString() is called in here, so the elements of dlist need
 *	to be free()'d by the caller.
 *	
 *	reports errors via report()
 *********************************************************************/
int AddToDelivery(struct dynstring *dlist, DB* pcdb, u_int ncentral, 
		  char * fname, char * recipient) {
  char *pc;
  int lenfname;
  int err, ret;
  extern int debug;
  DBT val, key;
  int n;
  struct pcinfo pcinfo;
  
  err = 1; /* bad */
  
  /*
    The recipient is assumed to allready have gone through alias expansion 
    The PAGINGCENTRAL part should be in the pcdb database, else error.
    The recipient should be in the form ID.PAGINGCENTRAL ie 1111.usamobile1,
    else error.
  */
  
  lenfname = strlen(fname);
  
  pc = recipient + strlen(recipient);
  
  for (; pc != recipient; --pc)
    if (*pc == '.') {
      ++pc;
      break;
    }
  report (LOG_ERR, "pc %s", pc);
  
  /* if didn't find a ., store it to the error delivery list */
  if (pc == recipient) {
    
#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO, "Adding \"%.512s\" to error list (%.512s)", fname,
	      recipient);
#endif /* DEBUG */
    
    if (AddToDynString(&dlist[0], fname, lenfname)) 
      goto AddToDeliveryout;
    
  } else { 
    
    /* lookup the PAGINGCENTRAL part */
    key.data = pc;
    key.size = strlen(pc);
    
    if ((ret = pcdb->get(pcdb, &key, &val, (u_int)0)) == -1) {
      report (LOG_ERR, "AddToDelivery() db->get failed, errno=%d",
	      errno);
      goto AddToDeliveryout;
    }
    
    /* lookup failed? */
    if (ret == 1) { /* yes, add it to the error delivery list */
      
#ifdef DEBUG
      if (debug > 2)
	report (LOG_INFO, "Adding \"%.512s\" to error list (%.512s)", fname,
		recipient);
#endif /* DEBUG */
      
      if (AddToDynString(&dlist[0], fname, lenfname)) 
	goto AddToDeliveryout;
      
    } else { /* no, add it to the delivery list */
      
      /* first string in the val is the index to the delivery list */
      memcpy(&pcinfo, val.data, val.size);
      n = pcinfo.id + 1;
      
      /* SK XXX: was pc an alias? if so, look up real pc and queue for that */
      report (LOG_ERR, "pcinfo %s", pcinfo.name);
      if (pcinfo.pc_alias) {
	/* lookup the PAGINGCENTRAL part */
	key.data = pcinfo.alias_name;
	key.size = strlen(pcinfo.alias_name);
	
	report (LOG_ERR, "looking up alias %s", pcinfo.alias_name);
	
	if ((ret = pcdb->get(pcdb, &key, &val, (u_int)0)) == -1) {
	  report (LOG_ERR, "AddToDelivery() db->get failed, errno=%d looking up alias %s",
		  errno, pcinfo.alias_name);
	  goto AddToDeliveryout;
	}
	
	/* lookup failed? */
	if (ret == 1) { /* yes, add it to the error delivery list */
	  
#ifdef DEBUG
	  if (debug > 2)
	    report (LOG_INFO, "Adding \"%.512s\" to error list (%.512s)",
		    fname, recipient);
#endif /* DEBUG */
	  
	  if (AddToDynString(&dlist[0], fname, lenfname)) 
	    goto AddToDeliveryout;
	  
	} else { /* no, add it to the delivery list */
  			        /* first string in the val is the index to the delivery list */
	  memcpy(&pcinfo, val.data, val.size);
	  n = pcinfo.id + 1;
	  
	} 
      } 
      
#ifdef DEBUG
      if (debug > 2)
	report (LOG_INFO, "Adding \"%.512s\" to delivery list (%.512s %d)",
		fname, recipient, n);
#endif /* DEBUG */
      
      if (AddToDynString(&dlist[n], fname, lenfname)) 
	goto AddToDeliveryout;
      
    } /* else lookup failed */
    
  } /* else *pc == . */
  
  err = 0; /* good */	
  
AddToDeliveryout:

  return err;
} /* AddToDelivery */

/*********************************************************************
 * Function: QueueRun
 *	Perform a delivery run on the queue
 *
 *	Returns	0	good
 *	       !0	bad
 *
 *	if the queue needs to be run again (there are undelivered msgs)
 *	rerun is set to 1
 *
 * errors are reported via report()
 *
 *********************************************************************/
int QueueRun(DB* aliasdb, DB* pcdb, DB* profdb, DB* trdb, u_int ncentral,
	     int *rerun) {
  int err, n, fd, i=0, queuebufsize=0, mfd;
  extern int debug;
  struct dynstring *dlist; /* delivery list */
  char *queuebuf, *tmpp, *p=NULL, *q;
  char *sender, *recipient, *recipient2, *message, *emailCC;
  struct stat statbuf;
  struct messagequeue *entry;
  DBT dbval, dbkey;
  char recipientid[255];
  struct cbuf cbuf;
  int ret;
  struct pcinfo pcinfo;
  struct pcinfo temp_pc;
  int didProtStartup=0, abandonList=0, sendReply;
  int lastEntry=0, qfAction=0;
  struct dynstring deferstring;
  char nullbyte;
  int numTrans=0;
  unsigned char *field_vec[2];
  extern struct global globalOptions;
#ifdef ARSYSTEM_NOTIFIER
  NTStatusList remedyStatus;
#endif /* ARSYSTEM_NOTIFIER */
  
  err = 1; /* bad */
  dlist = (struct dynstring*)NULL;
  *rerun = 0; /* don't need to run again */
  fd = -1; /* fd invalid */
  mfd = -1; /* mfd invalid */
  queuebuf = NULL;
  bzero(&cbuf, sizeof(cbuf));
  bzero(&deferstring, sizeof(deferstring));
  nullbyte = 0;
  
#ifdef DEBUG
  if (debug > 5)
    report (LOG_INFO, "begin QueueRun");
#endif /* DEBUG */
  
  /* allocate memory for delivery list */
  if (!(dlist = (struct dynstring*)malloc(sizeof (struct dynstring) *
					  (ncentral + 1)))) {
    report (LOG_ERR, "malloc() for delivery list failed");
    *rerun = 1;
    goto skip2;
  }
  
  /* allocate memory for contents of queue'd entry (may be expanded later) */
  if (!(queuebuf = (char*)malloc((queuebufsize = 1024)))) {
    report (LOG_ERR, "malloc(1024) failed");
    *rerun = 1;
    goto skip2;
  }
  
  /* zero out the entries */
  for (n = 0; n < ncentral + 1; ++n)
    bzero(&dlist[n], sizeof(struct dynstring));
  
  
  /* expand and resolve aliases, construct a delivery list */
  if (QueueExpand(aliasdb, pcdb, profdb, trdb, ncentral, dlist)) {
    report (LOG_ERR, "QueueExpand() failed");
    *rerun = 1;
    goto QueueRunout;
  }
  
  /* for each delivery list
   * each entry contains a list of queue file names for a particular
   * destination, with i==0 being the special error list */
  for (i = 0; i < (ncentral + 1); ++i) {
    
    /* if no entries for this destination, then skip */
    if (!dlist[i].buf)
      continue;
    
    didProtStartup = 0;	/* startup code called? */
    abandonList = 0;	/* give up on the entire list? */
    numTrans = 1;       /* count number of transactions */
    mfd = -1;		/* invalidate modem fd */
    
    
    /* for each entry in that list */
    /* ie. each queue file name */
    for (p = dlist[i].buf; 
	 *p;
         p += strlen(p)+1, numTrans++) {
      
      sendReply = 0;	/* force mail reply */
      fd = -1;		/* invalidate fd of queue file */
      lastEntry = 0;	/* this is Not the last entry for this pc */
      
#ifdef DEBUG
      if (debug > 5) 
	report(LOG_INFO, "i=%d, p=%.512s", i, p);
#endif /* DEBUG */
      
      /*
       * if this is the last entry in the list, extra code
       * needs to be executed to cleanup 
       */
      if (!*(p + strlen(p)+1))
	lastEntry = 1;
      
      /* read in the queue file */
      if ((fd = open(p, O_RDWR, 0)) == -1) {
	report (LOG_WARNING, "Can't open queue file %.512s", p);
	*rerun = 1;
	goto skip2;
      }
      
      if (fstat(fd, &statbuf) == -1) {
	report (LOG_WARNING, "Can't fstat queue file %.512s", p);
	*rerun = 1;
	goto skip2;
      }
      
      /* sanity check */
      if (statbuf.st_size > 32767) {
	report (LOG_WARNING, "oversized queue file! -- %.512s", p);
	*rerun = 1;
	goto skip2;
      }
      
      /* make sure there's enough buffer space available to read it in */
      if (statbuf.st_size > queuebufsize) {
	
	queuebufsize += statbuf.st_size;
	
	tmpp = queuebuf;
	if (!(queuebuf = (char*)realloc(queuebuf, queuebufsize))) {
	  queuebuf = tmpp;
	  report (LOG_WARNING, "realloc(%d) failed", queuebufsize);
	  *rerun = 1;
	  goto skip2;
	}
      }
      
      entry = (struct messagequeue*)queuebuf;
      
      if ((n = read(fd, queuebuf, (int)statbuf.st_size)) == -1) {
	report (LOG_WARNING, "read(%.512s): ", p, strerror(errno));
	*rerun = 1;
	goto skip2;
      }
      
      /* increment retries */
      entry->retries ++;
      
      /* update the queue file for use with sendpage -bp */
      entry->status = STATUS_DELIVERING | STATUS_EXPANDED;
      
      /* unset other bits that could of been set previously */
      entry->status &= ~STATUS_TMPFAIL;
      
      
      if (UpdateQueueFile(p, fd, queuebuf, (int)statbuf.st_size)) {
	*rerun = 1;
	goto skip2;
      }
      
      /* calculate pointers to text fields */
      recipient = queuebuf+sizeof(struct messagequeue);
      recipient2 = recipient + entry->recipientlen + 1;
      sender = recipient2 + entry->recipient2len + 1;
      message = sender + entry->senderlen + 1;
      emailCC = message + entry->messagelen + 1;
      
      /*
       * if this is the special error list, mark this as an
       * error and skip processing any further -
       * -- this must be done now, code further down assumes
       * the recipient is formatted correctly and the paging
       * central part exists
       */
      if (i == 0) {

	entry->status = STATUS_NODELIVER1;
	strcpy(pcinfo.emailFrom, globalOptions.emailFrom);
	
	qfAction = QF_DELETE;
	sendReply = 1;
	goto skip1;
      }
      
#ifdef FOO
      /* if the STATUS_MSGTRUNCATE bit was set, drop it now */
      if (entry->status & STATUS_MSGTRUNCATE) {
	
	qfAction = QF_DELETE;
	sendReply = 1;
	goto skip1;
      }
#endif /* FOO */
      
      /*
       *   get the "paging central" part so can lookup options
       *   for the destination 
       */
      q = recipient2 + entry->recipient2len;
      
      for (; q != recipient2; --q)
	if (*q == '.') {
	  ++q;
	  break;
	}
      
      dbkey.data = q;
      dbkey.size = strlen(q);
      
      if (pcdb->get(pcdb, &dbkey, &dbval, (u_int)0)) {
	report (LOG_ERR,
		"%.512s: unexpected pcdb->get failure", p);
	entry->status |= STATUS_NODELIVER2;
	qfAction = QF_UPDATE;
	sendReply = 1;
	*rerun = 1;
	goto skip1;
	
      }
      
      memcpy(&pcinfo, dbval.data, dbval.size);
      
      /*
       * lets frig the paging central if it's a dopple SK
       */
      if (pcinfo.pc_alias) {
	
	dbkey.data = pcinfo.alias_name;
	dbkey.size = strlen(pcinfo.alias_name);
	
	if (pcdb->get(pcdb, &dbkey, &dbval, (u_int)0)) {
	  report (LOG_ERR,
		  "Unexpected paging (alias) failure %s", pcinfo.alias_name);
	  entry->status |= STATUS_NODELIVER2;
	  qfAction = QF_UPDATE;
	  sendReply = 1;
	  *rerun = 1;
	  goto skip1;
	  
	}
	
	memcpy(&temp_pc, dbval.data, dbval.size);
	
	if (pcinfo.dopple_set) {
	  if (pcinfo.dopple_set & DOPPLE_TRMAP)
	    strcpy(temp_pc.trMap, pcinfo.trMap);
	  if (pcinfo.dopple_set & DOPPLE_MAXMSGSIZE)
	    temp_pc.maxMsgSize = pcinfo.maxMsgSize;
	  if (pcinfo.dopple_set & DOPPLE_MANGLE)
	    temp_pc.mangle = pcinfo.mangle;
	}
	
	if (pcinfo.pc_set) {
	  temp_pc.flags = 
	    ((~pcinfo.pc_set & temp_pc.flags) |
	     (pcinfo.pc_set & pcinfo.flags));
	}
	
	/*
	 * Now copy the revised temp paging central into the orig
	 */
	memcpy(&pcinfo, &temp_pc, sizeof(pcinfo));
      }
      /* end dopple fix */
      
      
      /* 
       * if this pc has a max retries set, and the retries
       * is greater than this number, drop it now 
       */
      if ((pcinfo.msgRetries) &&
	  (entry->retries > pcinfo.msgRetries )) {
	
	entry->status = STATUS_EXPIRED;
	
	qfAction = QF_DELETE;
	sendReply = 1;
	goto skip1;
      }

      if (pcinfo.maxTransactions && pcinfo.maxTransactions==numTrans) {
	lastEntry=1;
	report (LOG_INFO, "max transactions reached with next entry");
      }

      
#ifdef FOO
      /*
       * if this pc has a max message size set and the 
       * messagelen is greater than this, then truncate the
       * message size, and set STATUS_SIZETRUNCATE
       */
      
      if ((pcinfo.maxMsgSize) &&
	  (entry->messagelen > pcinfo.maxMsgSize )) {
	
	statbuf.st_size -= (entry->messagelen - pcinfo.maxMsgSize);
	entry->messagelen = pcinfo.maxMsgSize;
	entry->status |= STATUS_SIZETRUNCATE;
	message[pcinfo.maxMsgSize] = 0;
      }
#endif /* FOO */
      
      /*
       * get the "recipient" part for later use 
       */
      
      if (entry->recipient2len >= sizeof(recipientid))
	strcpy(recipientid, "OVERFLOW");
      else
	strcpy(recipientid, recipient2);
      
      tmpp = recipientid + entry->recipient2len;
      for (; tmpp != recipientid; --tmpp)
	if (*tmpp == '.') {
	  break;
	}
      *tmpp = 0;
      
      /*
       * call the appropriate protocol startup code 
       */
      
      if (!didProtStartup)  {
	
	switch (pcinfo.protocol) {
	  
	case PC_NULL:
#ifdef ARSYSTEM_NOTIFIER
	case PC_REMEDY:
#endif /* ARSYSTEM_NOTIFIER */
	  didProtStartup = 1;
	  break;
	  
	case PC_PET1:
	case PC_PET3:
	  
	  if (PetStartup(&pcinfo, &mfd, &cbuf, p))
	    didProtStartup = 0;
	  else
	    didProtStartup = 1;
	  
	  break;
	default:
	  
	  report (LOG_ERR, "fatal, unknown protocol");
	  entry->status = STATUS_NODELIVER2;
	  qfAction = QF_DELETE;
	  goto skip1;
	  
	  break;
	  
	} /* switch */
	
      } /* need to call protocol startup */
      
      /*
       * if the protocol startup code failed, then
       * assume it will fail next time too -- 
       * defer this entire list 
       */
      
      if (!didProtStartup) {
	
	*rerun = 1;
	
	abandonList = 1;
	
	entry->status |= STATUS_TMPFAIL;
	
	qfAction = QF_UPDATE;
	goto skip1;
	
      }
      
      
      /*
       * at this point, the protocol has started up, and is
       * ready to process transactions
       */
      switch (pcinfo.protocol) {
      case PC_NULL:
	entry->status = STATUS_NONE;
	
	entry->delivertime = time((time_t*)0L);
	qfAction = QF_UPDATE|QF_DEFER;
	goto skip1;
	
	break;
	
#ifdef ARSYSTEM_NOTIFIER
      case PC_REMEDY:
	
	ret = NTNotificationServer(pcinfo.phone, recipientid,
				   message, 1, (char*)0L, &remedyStatus);
	
	for (x = 0; x < remedyStatus.numItems; ++x) {
	  report (LOG_WARNING, "%u:%lu: %s",
		  remedyStatus.statusList[x].messageType,
		  remedyStatus.statusList[x].messageNum,
		  remedyStatus.statusList[x].messageText);
	}
	
	FreeNTStatusList(&remedyStatus, (NTBoolean)FALSE);
	
	if ((ret == NT_RETURN_OK) || (ret == NT_RETURN_WARNING)) {
	  entry->status = STATUS_NONE;
	  entry->delivertime = time((time_t*)0L);
	  qfAction = QF_DELETE;
	  goto skip1;
	} else {
	  entry->status = STATUS_TMPFAIL;
	  qfAction = QF_UPDATE;
	  goto skip1;
	}
	
	
	break;
#endif /* ARSYSTEM_NOTIFIER */
	
	
      case PC_PET1:
      case PC_PET3:
	
	/*	if ((ret = SendPETTransaction(mfd, &cbuf, recipientid,
			message, (int)pcinfo.protocol))) {
	*/
	field_vec[0] = recipientid;
	field_vec[1] = message;
	if ((ret = send_pet_transaction(mfd, &cbuf, field_vec, 2,
					(int)pcinfo.protocol))) {
	  report (LOG_ERR, "%.512s: Page not accepted", p);
	  
	  if ((ret == 2) || (ret == 3)) {
	    entry->status = STATUS_NODELIVER3;
	    qfAction = QF_DELETE;
	    sendReply = 1;
	    goto skip1;
	  } else { /* whatever else is tmp error */
	    entry->status = STATUS_TMPFAIL;
	    qfAction = QF_UPDATE;
	    goto skip1;
	  } 
	  
	} else  {
	  entry->status = STATUS_NONE;
	  
	  entry->delivertime = time((time_t*)0L);
	  qfAction = QF_UPDATE|QF_DEFER;
	  goto skip1;
	}
	
	break;
	
      default: /* unreached */
	
	break;
	
      } /* switch */
      
skip1:

      /* if abandonList is set, thats the same as lastEntry below */
      if (abandonList) 
	lastEntry = 1, *rerun = 1;
      
      entry->status &= ~STATUS_DELIVERING;
      
      /*
       * the fate of the queue file is decided by qfAction 
       */
      
      /*
       * if QF_DEFER is set, then the actual confirmation 
       * that this page really was sent can't be known yet
       * (ie the protocol needs to shutdown gracefullty first
       * so, add this queue entry to a list that will be gone
       * through a second time
       */
      if (qfAction & QF_DEFER) 
	if (AddToDynString(&deferstring, p, strlen(p))) {
	  report (LOG_ERR, "AddToDynString failed for %.512s", p);
	  entry->status |= STATUS_NODELIVER2;
	  sendReply = 1;
	  qfAction = QF_DELETE;
	}
      
      if (qfAction & QF_UPDATE) {
	
	entry->status |= STATUS_EXPANDED;
	
	if (UpdateQueueFile(p, fd, queuebuf, (int)statbuf.st_size)) 
	  report (LOG_WARNING, "UpdateQueueFile failed for %.512s",
		  p);
	
      } else if (qfAction & QF_DELETE) {
	
	if (unlink(p) == -1) {
	  report (LOG_WARNING, "%.512s: unlink(): %s",  p,
		  strerror(errno));
	}
      } else {
	report (LOG_ERR, "Internal error, qfAction unset");
      }
      
      /* 
       * send an e-mail reply?
       */

      /*
       * (confusing) logic:
       *
       *  if QF_DEFER is set, don't send mail - it will be
       *   sent later -- done.
       *
       *  if sendReply is not set or pcinfo.retryNotify != retries
       *   don't send mail.
       *
       *  if MSGQ_FLAGS_MAIL_GOOD is set and this is not an error
       *   message, send mail.
       *
       *	MSGQ_FLAGS_MAIL_BAD is set and this is an error message,
       *   send mail.
       *  
       */
      if (
	  (!(qfAction & QF_DEFER)) && 
	  (sendReply || (pcinfo.retryNotify &&
			 (pcinfo.retryNotify == entry->retries))) &&
	  
	  (((entry->flags & MSGQ_FLAGS_MAIL_GOOD) &&
	    (!CHECKSTATUS(entry->status))) || 
	   ((entry->flags & MSGQ_FLAGS_MAIL_BAD) &&
	    (CHECKSTATUS(entry->status)))) ) {
	
	if (DeliverMAIL(pcinfo.emailFrom, sender, p, queuebuf)) {
	  report (LOG_WARNING,
		  "%.512s: DeliverMail() failed to %.512s", p, sender);
	}
      }
      
      
      /* cleanup for this loop run */
skip2:			

      
      /* close the open queue file */
      if (fd != -1)
	if (close (fd) == -1) 
	  report (LOG_WARNING, "%.512s: close(): %s", p,
		  strerror(errno));
      
      /*
       * if this is the last entry in the list
       * need to do some cleanup
       */
      if (lastEntry && didProtStartup) {
	
	
	switch (pcinfo.protocol) {
	  
	case PC_PET1:
	case PC_PET3:
	  
	  report (LOG_INFO, "Ending PET delivery run");
	  if 	((ret = EndPETTransaction(mfd, &cbuf))) {
	    report (LOG_ERR,
		    "%.512s: EndPETTransaction() failed", p);
	    if (ret == 2)
	      abandonList = 1;
	    *rerun = 1;
	  }
	  
	  HangUpModem(mfd, &cbuf);
	  
	  if (close (mfd) == -1)
	    report (LOG_WARNING, "%.512s: close(): %s", "modem",
		    strerror(errno));
	  
	  break;
	  
	  
	case PC_NULL:
	  
	  break;
	  
	default: /* unreached */
	  
	  break;
	  
	} /* switch */
	
	/*
	 * take care of the deferred messages 
	 */
	
	/* mark the end of the list */
	if (AddToDynString(&deferstring, &nullbyte, 0)) {
	  report (LOG_ERR, "AddToDynString() failed nullbyte");
	  /* this should never happen, so this is good enough */
	  deferstring.buf = (char*)NULL;
	}
	
	/* for each entry in that list */
	for (q = deferstring.buf; q && *q; q += strlen(q)+1) {
	  
#ifdef DEBUG
	  if (debug > 5)
	    report (LOG_INFO, "defer: q=%.512s", q);
#endif /* DEBUG */
	  
	  fd = -1;
	  
	  /* read in the queue file */
	  if ((fd = open(q, O_RDWR, 0)) == -1) {
	    report (LOG_WARNING, "Can't open queue file %.512s", p);
	    *rerun = 1;
	    goto skip3;
	  }
	  
	  if (fstat(fd, &statbuf) == -1) {
	    report (LOG_WARNING, "Can't fstat queue file %.512s",
		    p);
	    *rerun = 1;
	    goto skip3;
	  }
	  
	  entry = (struct messagequeue*)queuebuf;
	  
	  if ((n = read(fd, queuebuf, (int)statbuf.st_size)) == -1) {
	    report (LOG_WARNING, "read(%.512s): ", p,
		    strerror(errno));
	    *rerun = 1;
	    goto skip3;
	  }
	  
	  /* 
	   * if abandonList is set, these really didn't get
	   * delivered
	   */
	  
	  if (abandonList) {
	    entry->status &= ~STATUS_DELIVERING;
	    entry->status &= ~STATUS_DELIVERED;
	    entry->status |= STATUS_TMPFAIL;
	    
	    if (UpdateQueueFile(p, fd, queuebuf,
				(int)statbuf.st_size)) {
	      *rerun = 1;
	      goto skip3;
	    }
	    
	  } /* else they did, and the queue file is not needed */
	  
	  if (unlink(q) == -1) {
	    report (LOG_WARNING, "%.512s: unlink(): %s",  q,
		    strerror(errno));
	  }
	  
	  
	  /* calculate pointers to text fields */
	  recipient = queuebuf+sizeof(struct messagequeue);
	  recipient2 = recipient + entry->recipientlen + 1;
	  sender = recipient2 + entry->recipient2len + 1;
	  message = sender + entry->senderlen + 1;
	  emailCC = message + entry->messagelen + 1;
	  
	  entry->status |= STATUS_DELIVERED;
	  
	  
	  /*
	   * if the e-mail flag bit is on for this entry
	   * and (the e-mail errors only bit is off or
	   * the emil errors bit is on and this is an error)
	   * then send mail
	   */
	  
	  if ( (((entry->flags & MSGQ_FLAGS_MAIL_GOOD) &&
		 (!CHECKSTATUS(entry->status))) || 
		((entry->flags & MSGQ_FLAGS_MAIL_BAD) &&
		 (CHECKSTATUS(entry->status)))) ) {
	    
	    if (DeliverMAIL(pcinfo.emailFrom, sender, q,
			    queuebuf))  {
	      report (LOG_WARNING,
		      "%.512s: DeliverMail() failed", q);
	    }
	  }
	  
	  
skip3:
	  if (fd != -1)
	    close (fd);
	  
	} /* for each entry in the deferred list */
	
      } /* lastEntry */
      
      /*
       * lastEntry may have been set by an error condition, not
       * necessarly because this is the last entry...
       */
      
      if (lastEntry)
	break;
      
    } /* for each qfile name (entry) in the delivery list for this pc */
    
    /* clear ou the defer list between each pc */
    deferstring.bufused = 0;
    
    if (deferstring.buf)
      deferstring.buf[0] = 0;
    
  } /* foreach pc */

QueueRunout:

  if (queuebuf)
    free (queuebuf);
  
  if (dlist) {
    for (n = 0; n < ncentral + 1; ++n)
      if (dlist[n].buf)
	free (dlist[n].buf);
    free (dlist);
  }
  
  if (deferstring.buf)
    free (deferstring.buf);
  
  return 0;
} /* QueueRun */

/*********************************************************************
 * Function: UpdateQueueFile
 *	updates a queue file
 *
 *	Returns	0	good
 * 	       !0	bad
 *
 *	errors are reported via report()
 *
 *********************************************************************/
int UpdateQueueFile(char * qfname, int fd, char * queuebuf, int size) {
  if (lseek(fd, (off_t)0, SEEK_SET) == -1) {
    report (LOG_WARNING, "lseek(%.512s): %s", qfname, strerror(errno));
    return 1;
  }
  
  if (write(fd, queuebuf, (int)size) == -1) {
    report (LOG_WARNING, "write(%.512s): %s", qfname, strerror(errno));
    return 1;
  }
  
  return 0;
} /* UpdateQueueFile */


/*********************************************************************
 * Function: PetStartup
 *	starts up the pet protocol
 *
 *	Returns	 0	good
 *		!0	bad
 *
 *	errors are reported via report()
 *
 *********************************************************************/
int PetStartup(struct pcinfo * pcinfo, int * mfd, struct cbuf *cbuf,
	       char *p) {
  int x, err;
#ifdef UUCP_LOCKING
  int gotuucplock = 0;
#endif /* UUCP_LOCKING */
  
  err = 1; /* assume bad */
  
  /* assume modem could not be opened */
  *mfd = -1;
  
  report (LOG_INFO, "Starting up PET protocol");
  
#ifdef UUCP_LOCKING
  /* try to lock the device */
  if (LockTTY(pcinfo->modemDev)) {
    report (LOG_ERR, "%.512s: can't lock modem device", p);
    goto pet_startup_fail;
  }
  gotuucplock = 1;
#endif /* UUCP_LOCKING */
  
  /* open and initialize the modem device */
  if (OpenModem(pcinfo, mfd)) {
    *mfd = -1;
    report (LOG_ERR, "%.512s: Can't open modem", p);
    goto pet_startup_fail;
  }
  
  /* reset modem to known state */
  if (ResetModem(*mfd, cbuf, pcinfo)) {
    report (LOG_ERR, "%.512s: Can't reset modem", p);
    goto pet_startup_fail;
  }
  
  /* dial out */
  x = DialModem (*mfd, cbuf, pcinfo);
  
  if ((x != MODEM_RES_C300) && (x != MODEM_RES_C1200) &&
      (x != MODEM_RES_C2400) && (x != MODEM_RES_C4800) &&
      (x != MODEM_RES_C9600) && (x != MODEM_RES_C14400) &&
      (x != MODEM_RES_C38400)) {
    report (LOG_ERR, "%.512s: Not connected - reason %d", p, x);
    goto pet_startup_fail;
  }
  
  /* start the PET protocol */
  if (StartPET(*mfd, cbuf, pcinfo)) {
    report (LOG_ERR, "%.512s: PET protocol didn't startup", p);
    goto pet_startup_fail;
  }
  
  err = 0;
  goto PetStartupout;

pet_startup_fail:

  /*
   * handle startup failure
   */
  
  /* send modem hangup command if device opened */
  if (*mfd != -1)
    HangUpModem(*mfd, cbuf);
  
  /* close the modem device if its opened */
  if (*mfd != -1)
    close (*mfd);
  
PetStartupout:
  
#ifdef UUCP_LOCKING
  if (gotuucplock)
    if (UnLockTTY(pcinfo->modemDev))
      report (LOG_ERR, "%.512s: unlock of modem device failed", p);
#endif /* UUCP_LOCKING */
  
  return err;
  
} /* PetStartup */

/*********************************************************************
 * Function: GetUserProfile()
 *	return the user profile of recipient.  If that profile doesn't
 *	exist, try using "default", if that doesn't exist return
 *	the compiled in defaults.
 *
 *	Returns	0	good
 *	       !0	bad
 *
 *	errors are reported via report()
 *********************************************************************/
int GetUserProfile(char *key, struct profile *profile, DB *profdb) {
  extern struct profile defaultProfile;
  int ret, err;
  DBT dbkey, dbval;
  
  err = 1; /* bad */
  
  dbkey.data = (void*)key;
  dbkey.size = strlen(key);
  
  if ((ret = profdb->get(profdb, &dbkey, &dbval, (u_int)0)) == -1) {
    report (LOG_ERR, "profdb->get() failed, errno=%d", errno);
    goto outGetUserProfile;
  }
  
  if (ret) { /* lookup failed */
    
    dbkey.data = (void*)"default";
    dbkey.size = 7;
    
    if ((ret = profdb->get(profdb, &dbkey, &dbval, (u_int)0)) == -1) {
      report (LOG_ERR, "profdb->get() failed, errno=%d", errno);
      goto outGetUserProfile;
    }
    
    if (ret) /* lookup failed */
      memcpy(profile, &defaultProfile, sizeof (struct profile));
    else
      memcpy(profile, dbval.data, sizeof (struct profile));
    
  } else 
    
    memcpy(profile, dbval.data, sizeof (struct profile));
  
  err = 0; /* good */
  
outGetUserProfile:
  
  return err;
} /* GetUserProfile */


/*********************************************************************
 * Function: GetPagingCentral()
 *	return the paging central of recipient.
 *
 *	Returns	 0	good
 *		!0	bad
 *
 *	errors are reported via report()
 *********************************************************************/
int GetPagingCentral(char *key, int keyLen, struct pcinfo *pc, DB *pcdb) {
  int ret, err;
  DBT dbkey, dbval;
  char *c;
  
  err = 1; /* bad */
  
  c = key+keyLen;
  
  for (; c != key; --c)
    if (*c == '.') {
      ++c;
      break;
    }
  
  dbkey.data = (void*)c;
  dbkey.size = strlen(c);
  
  if ((ret = pcdb->get(pcdb, &dbkey, &dbval, (u_int)0)) == -1) {
    report (LOG_ERR, "pcdb->get() failed, errno=%d", errno);
    goto outGetPagingCentral;
  }
  
  if (!ret) { /* lookup success */
    memcpy(pc, dbval.data, sizeof (struct pcinfo));
    err = 0; /* good */
  }
  
outGetPagingCentral:
  
  return err;

} /* GetPagingCentral */

/*********************************************************************
 * Function: MessageTranslate()
 *	if "key" exists in the trdb database, run it over the message.
 *
 *	Returns	0	good
 *	       !0	bad
 *
 *	errors are reported via report()
 *********************************************************************/
int MessageTranslate(char *key, char *msg, DB* trdb) {
  int ret;
  DBT dbkey, dbval;
  char *c;
  char trData[256];
  
  /* key should be in id.paging central form */
  dbkey.data = (void*)key;
  dbkey.size = strlen(key);
  
  if ((ret = trdb->get(trdb, &dbkey, &dbval, (u_int)0)) == -1) {
    report (LOG_ERR, "trdb->get() failed, errno=%d", errno);
    return 1;
  }
  
  if (!ret) { /* lookup success */
    /* 		trData = dbval.data; *//* Old */
    /* Inefficient, but I have yet to figure out why Sun won't
     * let me access the memory area pointed to by dbval.data,
     * even when doing explicit casts.  Alignment problems?
     */
    memcpy(trData,dbval.data,256*sizeof(int));
    
    for (c = msg; *c; ++c) 
      *c = trData[(int)*c];
    
  }

  return 0;
} /* MessageTranslate */

/*********************************************************************
 * Function: MessagePrepender()
 *  prepend date, time, and/or sender to message
 *
 * Sender Date Time format:
 *
 * sender mm/dd hh:mm!
 *       0123456789ABC
 *
 *  Returns  0  good
 *          !0  bad
 *
 *	Caller is responsible for free()ing newMsg
 *
 *  errors are reported via report()
 *********************************************************************/
int MessagePrepender(char * fullMsg, int flen, char **newMsg, int *nlen,
		     char *sender, int slen, time_t time, int profile) {
  int x, slen2;
  struct tm *tm=NULL;
  extern struct global globalOptions;
  
  x = 0;
  *nlen = 0;
  
  *newMsg = (char*)0;
  
  if ((profile & PROF_FLAGS_INC_SENDER) || (profile & PROF_FLAGS_INC_DATE)
      || (profile & PROF_FLAGS_INC_TIME)) 
    
    /* allocate memory for new message */
    /* -- fullMessage + null byte + sender + max(date/time string) + > */
    if (!(*newMsg = (char*)malloc(flen + slen + 14 + 1)))
      return 1; /* bad */
  
  if (profile & PROF_FLAGS_INC_SENDER) {
    
    /* only copy upto 12 characters, a '@' or a '%' whichever is first */
    for (slen2 = 0; (slen2 != slen) && (slen2 != 12); ++slen2)
      if ((sender[slen2] == '@') || sender[slen2] == '%')
	break;
    memcpy(*newMsg, sender, (x = slen2));
  }
  
  if ((profile & PROF_FLAGS_INC_DATE) || (profile & PROF_FLAGS_INC_TIME))
    tm = localtime(&time);
  
  if (profile & PROF_FLAGS_INC_DATE) {
    
    *(*newMsg + x++) = ' ';
    
    if (globalOptions.flags & GLOBAL_FLAGS_MONTH_FIRST) {
      
      if ((tm->tm_mon+1) >= 10)
	*(*newMsg + x++) = ((tm->tm_mon+1) / 10) + '0'; 
      *(*newMsg + x++) = ((tm->tm_mon+1) % 10) + '0';
      
      *(*newMsg + x++) = '/';
      
      if (tm->tm_mday >= 10)
	*(*newMsg + x++) = (tm->tm_mday / 10) + '0';
      *(*newMsg + x++) = (tm->tm_mday % 10) + '0';
      
    } else {
      
      if (tm->tm_mday >= 10)
	*(*newMsg + x++) = (tm->tm_mday / 10) + '0';
      *(*newMsg + x++) = (tm->tm_mday % 10) + '0';
      
      *(*newMsg + x++) = '/';
      
      if ((tm->tm_mon+1) >= 10)
	*(*newMsg + x++) = ((tm->tm_mon+1) / 10) + '0'; 
      *(*newMsg + x++) = ((tm->tm_mon+1) % 10) + '0';
      
    }
  }
  
  if (profile & PROF_FLAGS_INC_TIME) {
    
    *(*newMsg + x++) = ' ';
    
    if (tm->tm_hour >= 10)
      *(*newMsg + x++) = (tm->tm_hour / 10) + '0';
    *(*newMsg + x++) = (tm->tm_hour % 10) + '0';
    
    *(*newMsg + x++) = ':';
    
    *(*newMsg + x++) = (tm->tm_min / 10) + '0';
    *(*newMsg + x++) = (tm->tm_min % 10) + '0';
    
  }
  
  /* add the !, null terminator, and rest of message */
  if (x) {
    *(*newMsg + x++) = '>';
    memcpy(*newMsg+x, fullMsg, flen);
    *(*newMsg + x + flen) = 0;
    *nlen = x + flen;
  }
  
  
  return 0;
} /* MessagePrepender */

/*********************************************************************
 * Function: MessageSplit()
 *  split up a message into smaller parts.
 *
 *  Returns 0   good
 *         !0  bad
 *
 *	fullMsg:	full message
 *	flen:		full message length
 *	splitMsg:	array of null terminated strings
 *	splitSize:	size to split at
 *	nSplitMsg:	# of splitMsg's returned.
 *
 *	if nsplitMsg is 0, then splitMsg isn't filled in (the message didn't
 *	need to be split)
 *
 *	Caller is responsible for free()ing splitMsg on good return.
 *
 *********************************************************************/
int MessageSplit(char *fullMsg, int flen, char*splitMsg[], int splitSize,
		 int *nSplitMsg) {
  int n;	/* partial message # */
  int err, done;
  int x, p, p2, p3, p4;
  char c;
  
  *nSplitMsg = 0; /* no splits yet */
  
  /* be sane about it */
  if (splitSize == 0)
    splitSize = DEFAULT_MAX_PAGELEN;
  
  /* quick out */
  if (flen <= splitSize)
    return 0; /* good */
  
  err = 1; /* bad */
  
  /* skip leading blanks */
  for (x = 0; x < flen; ++x)
    if (*(fullMsg+x) != ' ')
      break;
  
  /* compress out multiple blanks */
  for (p = 0, c = '-'; x < flen; ++x) {
    
    *(fullMsg+p) = *(fullMsg+x);
    
    if ((*(fullMsg+x) != ' ') || (c != ' '))
      ++p;
    
    c = *(fullMsg+x);
    
  } /* for */
  
  *(fullMsg+p) = 0;
  
  /* another quick out */
  if (p <= splitSize)
    return 0; /* good */
  
  flen = p;
  
  
  done = 0; /* not! */
  
  /* initialize */
  for (n = 0; n < MAX_MSG_SPLIT; ++n)
    splitMsg[n] = (char*)0L;
  
  p = 0; /* index into message */
  
  for (n = 0; n < MAX_MSG_SPLIT; ++n) {
    
    /* allocate a message slot */
    if(!(splitMsg[n] = (char*)malloc(splitSize+1))) 
      goto outMessageSplit;
    
    x = 0; /* index into this partial message */
    
    /* prepend the split # */
    if (n > 9)
      splitMsg[n][x++] = (n / 10) + '0';
    
    splitMsg[n][x++] = (n % 10) + '0';
    splitMsg[n][x++] = ':';
    
    /*
     * set p2 = the absolute optimal last byte of the fullMsg that
     * could fit into this splitMsg, being careful not to go past
     * the end of the fullMsg
     */
    
    p2 = p + splitSize;
    
    /* take into account fragment counter prepending */
    if (n > 9)
      p2 -= 3; /* nn: */
    else
      p2 -= 2; /* n: */
    
    /* just to make sure */
    if (p2 < 0)
      p2 = 0;
    
    /* stopping condition */
    if (p2 >= flen) {
      
      memcpy(&splitMsg[n][x], fullMsg+p, (flen-p));
      splitMsg[n][x+flen-p] = 0; /* null terminate */
      
      goto loopOut;
    }
    
    /*
     * try to break on word boundries by working backwards
     * from the optimal point back towards a ' ', '-',
     * end of previous split, begining of message,
     * or 6 characters, whatever comes first.
     */
    
    p3 = p2;		/* p3 points at optimal split */
    p4 = p2 - 6;	/* p4 point to maximum backup point */
    if (p4 < p)		/* don't back up too far... */
      p4 = p;
    
    for (; p3 != p4; --p3) 
      if ((*(fullMsg+p3) == '-') || (*(fullMsg+p3) == ' ')) {
	p2 = p3; /* split point is here */
	break;
      }
    
    p4 = p2 - p;	/* p4 is actualy length now */
    
    /* copy */
    memcpy(&splitMsg[n][x], fullMsg+p, p4);
    
    /* null terminate */
    splitMsg[n][x+p4] = 0;
    
    /* what was really copied out */
    p += p4;
    
    /* forward over any blanks */
    while (1) {
      
      if (p == flen) { /* hit end */
	
	done = 1; 
	
	break;
      }
      
      if (*(fullMsg+p) == ' ')
	++p;
      else
	break;
    } /* while */
    
    if (done)
      goto loopOut;
    
  } /* for */
  
  /*
   * loop terminated because got to MAX_MSG_SPLIT
   * indicate this by changing : to * ie 10: becomes 10*
   */
  
  --n;
  
  if (n <= 9)
    splitMsg[n][1] = '*';
  else 
    splitMsg[n][2] = '*';
  
  
loopOut:

  *nSplitMsg = n + 1;
  
  err = 0;

outMessageSplit:

  /* if error, cleanup allocated memory */
  if (err) {
    for (n = 0; n < MAX_MSG_SPLIT; ++n) 
      if (splitMsg[n]) 
	free(splitMsg[n]);
      else
	break;
  }
  
  return err;
} /* MessageSplit */


