/*
 * udp_client
 * 
 * Create a client socket, send a message and receive the echo.
 *
 * Permit testing of socket_secure, connect_secure, getsockname_secure,  
 * recvfrom_secure and sendto_secure.
 * 
 * Calls socket_secure if '-s socket_context' is specified.
 * Calls connect prior to sending if '-c' is specified.
 * Calls connect_secure if '-p peer_context' and '-c' is specified.
 * Calls sendto_secure if '-p peer_context' and '-c' is not specified.
 * Calls sendto_secure if '-m message_context' is specified.
 *
 * Usage:  udp_client [-c] [-s socket_context] [-p peer_context] [-m message_context] local_port peer_address peer_port
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <getopt.h>
#include <stdio.h>

#include <socket_secure.h>
#include <ss.h>


void usage(char *progname) 
{
	printf("usage:  %s [-c] [-s socket_context] [-p peer_context] [-m message_context] local_port peer_address peer_port\n", progname);
	exit(1);
}

int main(int argc, char **argv)
{
	struct sockaddr_in sin;
	char buf[1024], context[255], context2[255], useconnect = 0;
	int s, sinlen, ret, buflen, c, contextlen;
	security_id_t so_sid = 0, peer_sid = 0, msg_sid = 0, tmp_sid = 0; 


	while ( (c = getopt(argc, argv, "s:p:m:c")) != EOF) {
		switch (c) {
		case 's':
		case 'p':
		case 'm':
			ret = security_context_to_sid(optarg,strlen(optarg)+1,&tmp_sid);
			if (ret) {
				perror(optarg);
				exit(ret);
			}
			if (c == 's')
				so_sid = tmp_sid;
			else if (c == 'p')
				peer_sid = tmp_sid;
			else if (c == 'm')
				msg_sid = tmp_sid;
			break;
		case 'c':
			useconnect = 1;
			break;
		default:
			usage(argv[0]);
		}
	}

	if (optind != (argc - 3)) 
		usage(argv[0]);

	/* Create the client socket */
	if (so_sid) 
		s = socket_secure(AF_INET, SOCK_DGRAM, 0, so_sid);
	else
		s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s == -1) {
		perror("socket");
		exit(1);
	}
	/* Obtain the security context of the server socket */
	sin.sin_family = AF_INET;
	sinlen = sizeof(struct sockaddr_in);
	ret = getsockname_secure(s, &sin, &sinlen, &tmp_sid);
	if (ret) {
		perror("getsockname_secure");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(tmp_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}
	
	printf("Created socket with context %s\n", context);

	/* Bind to the specified port */
	sin.sin_family = AF_INET;
	sin.sin_port = atoi(argv[optind++]);
	sin.sin_port = htons(sin.sin_port);
	sin.sin_addr.s_addr = INADDR_ANY;
	sinlen = sizeof(struct sockaddr_in);
	ret = bind(s, (struct sockaddr *)&sin, sinlen);
	if (ret == -1) {
		perror("bind");
		exit(1);
	}

	sin.sin_addr.s_addr = inet_addr(argv[optind++]);
	sin.sin_port = atoi(argv[optind++]);
	sin.sin_port = htons(sin.sin_port);

	if (useconnect) {
		/* Set the peer */
		if (peer_sid) 
			/* Specify the peer SID */
			ret = connect_secure(s, (struct sockaddr *)&sin, sinlen, peer_sid);
		else
			ret = connect(s, (struct sockaddr *)&sin, sinlen);
	}

	/* Set up the message */
	strcpy(buf, "hello world");
	buflen = strlen(buf)+1;

	/* Send the message */
	if (useconnect) {
		if (msg_sid)
			/* Specify the message SID */
			ret = sendto_secure(s, buf, buflen, 0, 0, 0, 0,
					    msg_sid);
		else
			ret = send(s, buf, buflen, 0);
	} else {
		if (peer_sid || msg_sid)
			/* Specify the peer SID and/or message SID */
			ret = sendto_secure(s, buf, buflen, 0, 
					    (struct sockaddr*)&sin, sinlen, 
					    peer_sid, msg_sid);
		else
			ret = sendto(s, buf, buflen, 0, 
				     (struct sockaddr *)&sin, sinlen);
	}
	if (ret != buflen) {
		perror("sendto");
		exit(1);
	}

	/* Receive the echo from the server and obtain the peer socket context and message context */
	buf[0] = 0;
	ret = recvfrom_secure(s, buf, sizeof(buf), 0, 
			      (struct sockaddr*)&sin, &sinlen,
			      &peer_sid, &msg_sid);
	if (ret == -1) {
		perror("recvfrom");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(peer_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}

	contextlen = sizeof(context2);
	ret = security_sid_to_context(msg_sid, context2, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}

	printf("Received %s from peer (%s,%d) with socket context %s, message context %s\n", 
	       buf, inet_ntoa(sin.sin_addr), ntohs(sin.sin_port), context, context2);
}
