/*  Screem:  pageUI.c,
 *  a tidied up page.c, separating ui from backend, this is the GUI file
 * 
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gtk/gtkinvisible.h>

#include "editor.h"
#include "editMenu.h"
#include "fileops.h"
#include "htmlfuncs.h"
#include "page.h"
#include "pageUI.h"
#include "preferences.h"
#include "preview.h"
#include "site.h"
#include "siteUI.h"
#include "siteTreeUI.h"
#include "support.h"

extern GtkWidget *app;
extern Site *current_site;
extern Preferences *cfg;

extern GList *loaded_sites;

static void recent_proxy( GtkWidget *widget, gchar *filename );
static void screem_page_rename_tab_do( gchar *filename, GladeXML *xml );

/** 
 * screem_page_insert:
 * @page: the page to insert into the editor
 *
 * inserts the page into the editor
 *
 * return values: boolean
 */
gboolean screem_page_insert( Page *page )
{
	const gchar *page_path;
	gchar *pname;

	GtkWidget *notebook = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( app ), "notebook" );
	GtkWidget *pagebook = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );
	GtkWidget *label;
	GtkWidget *junk;
	GtkWidget *widget;
	GtkTooltips *tips;

	gint num;

	struct stat s;

	g_return_val_if_fail( page != NULL, FALSE );

	if( ! screem_page_get_data( page ) ) {
		if( stat( screem_page_get_pathname( page ), &s ) < 0 )
			return FALSE;
		screem_page_set_mod_time( page, s.st_mtime );
	}

	if( ! screem_page_load( page ) ) {
		g_print( "load failed\n" );
		return FALSE;
	}

	/* if we are working on a site then we don't want a page open
	   twice */

	num = screem_page_get_tab( page );

 	if( num != -1 ) {
		/* already open, switch to it */
		gtk_notebook_set_page( GTK_NOTEBOOK( pagebook ), num );
		return TRUE;
	}

	/* display in the editor */
	screem_editor_display_page( page );

   	page_path = screem_page_get_pathname( page );

	switch_to( notebook, "0" );

	if( screem_site_get_fake_flag( current_site ) ) {
		cfg->recent_pages = add_recent( cfg->recent_pages, page_path );
		build_recent_pages_menu();
	}

	/* create the new page tab */
	pname = g_basename( page_path );
	label = gtk_label_new( pname );
	junk = gtk_invisible_new();
	gtk_widget_show( junk );
	gtk_widget_show( label );
	widget = gtk_event_box_new();
	gtk_container_add( GTK_CONTAINER( widget ), label );
	gtk_signal_connect( GTK_OBJECT( widget ), "button_press_event",
			    GTK_SIGNAL_FUNC( screem_page_tab_clicked ),
			    0 );
	gtk_notebook_append_page( GTK_NOTEBOOK( pagebook ), junk, widget );
	gtk_widget_set_usize( junk, 0, 0 );

	tips = gtk_tooltips_new();
	gtk_tooltips_set_tip( tips, widget, page_path, NULL );

	screem_site_inc_open_pages( current_site );
	num = screem_site_get_open_pages( current_site );
	screem_page_set_tab( page, num );

	gtk_notebook_set_page( GTK_NOTEBOOK( pagebook ), num );

	return TRUE;
}

/**
 * screem_page_open:
 *
 * opens the fileselector to select a page to open
 *
 * return values: boolean
 */
gboolean screem_page_open()
{
	gchar *filename;
	const gchar *name;
	Site *site;
	GtkWidget *combo = (GtkWidget*)
		gtk_object_get_data(GTK_OBJECT( app ), "sitecombo");

	/* select the page to open */
	filename = file_select( _("Select page to open") );

	/* we didn't select one */
	if( ! filename )
		return FALSE;

	site = (Site*)loaded_sites->data;
	name = screem_site_get_name( site );

	gtk_entry_set_text( GTK_ENTRY( GTK_COMBO( combo )->entry ), name );
		
	return screem_page_open_with_filename( filename );
}


/**
 * screem_page_open_with_filename:
 * @filename: the filename of the page to open
 *
 * opens a page with the given filename
 *
 * return values: boolean
 */
gboolean screem_page_open_with_filename( const gchar *filename )
{
	Page *page;
	gchar *path;
	gchar cwd[ 16384 ];

	g_return_val_if_fail( filename != NULL, FALSE );

	/* is the filename absolute? */
	if( filename[ 0 ] != G_DIR_SEPARATOR ) {
		/* no */
		getcwd( cwd, 16384 );
		path = g_strconcat( cwd, G_DIR_SEPARATOR_S, filename, NULL );
	} else {
		path = g_strdup( filename );
	}

	page = screem_site_locate_page( current_site, (gchar*)path );

	if( screem_site_get_fake_flag( current_site ) && ! page ) {
		/* fake site, add to list of pages */
		screem_site_add_page(current_site, (gchar*)path);
		page = screem_site_locate_page( current_site, (gchar*)path );
	}

	/* set mod time */
	screem_page_set_mod_time( page, time( 0 ) );

	g_free( path );

	return screem_page_insert( page );
}

/**
 * screem_page_save_as:
 *
 * opens the fileselector to select the filename to save as
 *
 * return values: boolean
 */
gboolean screem_page_save_as()
{
	gchar *filename;
	Page *page;
	gboolean hold;

	page = screem_site_get_current_page( current_site );
	
	g_return_val_if_fail( page, FALSE );

	filename = file_select( _("Select filename to save as") );

	/* we didn't select one */
	if( ! filename )
		return FALSE;

	if( ! overwrite( filename ) ) {
		g_free( filename );
		return FALSE;
	}

	screem_page_set_pathname( page, filename );

	screem_page_set_tab_label( page );

	/* allow save by setting page as changed */
	screem_page_set_changed( page, TRUE );
	hold = screem_page_save( page );
	
	refresh_file_tree();
	return hold;
}

gint screem_page_save_confirmation( Page *page )
{
	GtkWidget *box;
        gint button;

	gchar *message;

	if( ! screem_page_get_data( page ) )
		return 1;

	/* hasn't changed, no need to confirm */
	if( ! screem_page_get_changed( page ) )
		return 1;

	message = g_strdup_printf( _("Save %s"),
				   screem_page_get_pathname( page ) );

        box = gnome_message_box_new( message,
                                     GNOME_MESSAGE_BOX_QUESTION,
                                     GNOME_STOCK_BUTTON_YES,
                                     GNOME_STOCK_BUTTON_NO, 
				     GNOME_STOCK_BUTTON_CANCEL,
				     NULL );
	g_free( message );
        gnome_dialog_set_default( GNOME_DIALOG( box ), 0 );

        button = gnome_dialog_run_and_close( GNOME_DIALOG( box ) );

        if( button == 0 ) {
		if( screem_site_get_current_page( current_site ) == page )
			screem_editor_buffer_text();
                screem_page_save( page );
	}

	return button;
}

/**
 * screem_page_save_proxy:
 *
 * handles the menu selection of save page
 *
 * return values: boolean
 */
void screem_page_save_proxy()
{
	Page *page;

	page = screem_site_get_current_page( current_site );
	
	g_return_if_fail( page != NULL );

	screem_editor_buffer_text();

	screem_page_save( page );
}

/**
 * screem_page_revert_proxy:
 *
 * handles the menu selection of revert page
 *
 * return values: boolean
 */
void screem_page_revert_proxy()
{
	Page *page;

	page = screem_site_get_current_page( current_site );
	
	g_return_if_fail( page != NULL );

  	if( screem_page_revert( page ) )
		screem_page_insert( page );
}

void screem_page_revert_page( Page *page )
{
	g_return_if_fail( page != NULL );

	if( page == screem_site_get_current_page( current_site ) )
		screem_page_revert_proxy();
	else
		screem_page_revert( page );
}

/**
 * screem_page_create_page:
 *
 * creates a new page
 *
 * return values: none
 */
void screem_page_create_page()
{
	Page *page;

	page = screem_page_new();
	screem_page_druid( page );
}

void screem_page_create_blank_page()
{
	Page *page;
	gchar *filename;

	filename = file_select( _("Select filename for new page") );

	if( ! filename )
		return;
	
	if( ! overwrite( filename ) ) {
		g_free( filename );
		return;
	}

	/* is there already a page with that name in the site? */
	if( ! ( page = screem_site_locate_page( current_site, filename ) ) ) {
		screem_site_add_page( current_site, filename );
		page = screem_site_locate_page( current_site, filename );
	} else if( page == screem_site_get_current_page( current_site ) ) {
		screem_editor_clear();
	}

	screem_page_set_data( page, "" );
	screem_page_save( page );
	refresh_file_tree();

	screem_page_insert( page );
}

void screem_page_close_page( Page *page )
{
	gint num;
	Page *current;
	GtkWidget *pagebook;
	GList *list;

	g_return_if_fail( page != NULL );

	current = screem_site_get_current_page( current_site );

	if( current == page ) {
		screem_page_close_current();
		return;
	}

	/* request save confirmation */
	if( screem_page_save_confirmation( page ) == 2 )
		return;

	num = screem_page_get_tab( page );
	pagebook = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );

	/* remove the notebook tab */
	gtk_notebook_remove_page( GTK_NOTEBOOK( pagebook ), num );
	/* update other tab numbers */
	list = screem_site_get_pages( current_site );
	screem_page_update_tab_nums( list, num );

	screem_site_dec_open_pages( current_site );

}

void screem_page_close_current()
{
	Page *page;
	GList *list = NULL;
	const gchar *pathname;

	gint num;
	GtkWidget *pagebook;

	pagebook = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );

	num = gtk_notebook_get_current_page( GTK_NOTEBOOK( pagebook ) );
	
	g_return_if_fail( num > -1 );

	list = screem_site_get_pages( current_site );

	page = screem_page_from_tab_num( list, num );

	g_return_if_fail( page != NULL );

	/* request save confirmation */
	if( screem_page_save_confirmation( page ) == 2 )
		return;

	screem_editor_display_page( NULL );
	screem_site_set_current_page( current_site, NULL );
       
	/* clear the pages undo redo lists */
	screem_page_clear_undo_list( page );
	screem_page_clear_redo_list( page );

	/* destroy */
	if( screem_site_get_fake_flag( current_site ) ) {
		pathname = screem_page_get_pathname( page );
		page = screem_site_remove_page( current_site, pathname );
		screem_page_destroy( page );
	}
	/* remove the notebook tab */
	gtk_notebook_remove_page( GTK_NOTEBOOK( pagebook ), num );
	/* update other tab numbers */
	list = screem_site_get_pages( current_site );
	screem_page_update_tab_nums( list, num );

	screem_site_dec_open_pages( current_site );
}

void preview_current_page()
{
	Page *page;
	const gchar *page_data;

	GString *data = NULL;

	gchar cwd[ 16384 ];

	/* we get called during initialisation where current site is still
	   NULL */
	if( ! current_site )
		return;

	page = screem_site_get_current_page( current_site );

	if( ! page )
		return;

	getcwd( cwd, 16384 );

	screem_editor_buffer_text();

	page_data = screem_page_get_data( page );

	/* we want to be able to use gnome-db2html2 here */

	if( ! data )
		data = g_string_new( page_data );

	screem_page_chdir( page );
      	screem_preview_update( data->str );

	g_string_free( data, TRUE );

	chdir( cwd );
}


void build_recent_pages_menu()
{
	GnomeUIInfo *menu;
	gchar *path = _("_File/_Recent Pages/");
        int i;
     
	GList *list = cfg->recent_pages;
	GList *prev = NULL;

	if( ( i = g_list_length( list ) ) ) {
                gnome_app_remove_menu_range( GNOME_APP( app ), path,  0, i );
        } else {
		return;
	}

	list = g_list_last( list );
	for( i = 3; list && i > 0; i --, list = list->prev )
		prev = list;
 
	if( ! list )
		list = prev;

	cfg->recent_pages = list;
	list->prev = NULL;

	for( list = cfg->recent_pages; list; list = list->next ) {
		menu = (GnomeUIInfo*)g_malloc0 (2 * sizeof (GnomeUIInfo));
		menu->label = escape_path( (gchar*)list->data );
		
		menu->type = GNOME_APP_UI_ITEM;
		menu->hint = NULL;
		
		menu->moreinfo = (gpointer)recent_proxy;
		
		menu->user_data = list->data;
		menu->unused_data = NULL;
		menu->pixmap_type = (GnomeUIPixmapType)0;
		menu->pixmap_info = NULL;
		menu->accelerator_key = 0;
		
		(menu + 1)->type = GNOME_APP_UI_ENDOFINFO;
		
		gnome_app_insert_menus( GNOME_APP(app), path, menu );

		g_free( menu->label );
		g_free( menu );
	}
	
	save_preferences();
}

static void recent_proxy( GtkWidget *widget, gchar *filename )
{
	Site *site;
	const gchar *name;
	GtkWidget *combo = (GtkWidget*)
		gtk_object_get_data(GTK_OBJECT( app ), "sitecombo");

	site = (Site*)loaded_sites->data;
	name = screem_site_get_name( site );

	gtk_entry_set_text( GTK_ENTRY( GTK_COMBO( combo )->entry ), name );
	screem_page_open_with_filename( filename );
}

void page_book_switch( GtkNotebook *notebook, GtkNotebookPage *p,
		       guint page_num )
{
	GList *list = NULL;
	Page *page;
  
	GtkWidget *edbook = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( app ), "notebook" );
	
	if( gtk_object_get_data( GTK_OBJECT( notebook ), "switching" ) )
		return;

	list = screem_site_get_pages( current_site );
	
	page = screem_page_from_tab_num( list, page_num );

	if( page ) {
		/* display in the editor */
		screem_editor_display_page( page );

		switch_to( edbook, "0" );
	}
	
}

gboolean screem_page_save_check( Page *page )
{
	const gchar *path;
	struct stat s;
	gchar *message;
	gint button = 0;
	GtkWidget *prompt;

	path = screem_page_get_pathname( page );

	/* if stat fails, and the file exists then 
	   make save check fail */
	if( stat( path, &s ) < 0 )
		return ( errno == ENOENT );

	/* stat suceeded, check last modified against when screem
	   thinks it was last modified */
	if( s.st_mtime - screem_page_get_mod_time( page ) > 0 ) {
		/* the file has been changed outside of screem! */
		message = g_strconcat( path, _(": has changed on disk"),NULL );
                prompt = gnome_message_box_new( message,
                                                GNOME_MESSAGE_BOX_WARNING,
						_("Save"), _("Revert"),
						NULL );
                g_free( message );
                button = gnome_dialog_run_and_close( GNOME_DIALOG( prompt ) );

		if( button == 1 )
			screem_page_revert_page( page );
	}

	return (button != 1);
}

void switch_to( GtkWidget *widget, gchar *which )
{
	GtkNotebook *notebook;
	GtkWidget *button;
	GladeXML *xml;
	gint page;
	static gboolean no = FALSE;
	const gchar *buttons[] = {
		"editor_button", "preview_button", "link_button", 
		"help_button", NULL
	};

 	if( no )
		return;
       
	page = atoi( which );
	notebook = (GtkNotebook*)
		gtk_object_get_data( GTK_OBJECT( app ), "notebook" );

	if( cfg->mpage->viewbar ) {
		button = (GtkWidget*)
			gtk_object_get_data( GTK_OBJECT( notebook ), 
					     "sidebar" );
		xml = glade_get_widget_tree( button );
		button = glade_xml_get_widget( xml, buttons[ page ] );
		
		if( widget != button ) {
			no = TRUE;
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
						     TRUE);
			no = FALSE;
		}
	}
	gtk_notebook_set_page( GTK_NOTEBOOK( notebook ), page );
	if( page != 2 )
		screem_link_view_stop_check();

}

gboolean screem_page_tab_clicked( GtkWidget *label, GdkEventButton *event )
{
	Page *page;
	GList *list;
	gint num;
	GtkWidget *notebook;

	GladeXML *xml;
	GtkWidget *widget;

	notebook = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );

     	num = screem_page_get_tab_num( label );
      
	/* this should't happen but we had better check */
	if( num == -1 )
		return TRUE;

	if( event->button == 1 || event->button == 2 ) {
		gtk_notebook_set_page( GTK_NOTEBOOK( notebook ), num );
		return TRUE;
	}

	list = screem_site_get_pages( current_site );
	page = screem_page_from_tab_num( list, num );

	xml = glade_xml_new( cfg->glade_path, "tab_popup" );
	glade_xml_signal_autoconnect( xml );
	widget = glade_xml_get_widget( xml, "tab_popup" );
	gtk_object_set_data( GTK_OBJECT( widget ), "page", page );
	gnome_popup_menu_do_popup_modal( widget, 0, 0, event, 0 );
	gtk_widget_destroy( widget );

	gtk_signal_emit_stop_by_name( GTK_OBJECT( label ),
				      "button_press_event" );

	return TRUE;
}

gint screem_page_get_tab_num( GtkWidget *label )
{
	GtkWidget *notebook;
	GtkWidget *widget;
	gint i;

	notebook = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );

	for( i = 0; TRUE ; i ++ ) {
		widget = gtk_notebook_get_nth_page( GTK_NOTEBOOK( notebook ),
						    i );
		if( ! widget ) {
			i = -1;
			break;
		}

		widget = gtk_notebook_get_tab_label( GTK_NOTEBOOK( notebook ),
						     widget );
	
		if( label == widget )
			break;
	}

	return i;
}

void screem_page_save_tab( GtkWidget *item )
{
	GladeXML *xml;
	GtkWidget *widget;
	Page *page;
	Page *current;

	xml = glade_get_widget_tree( item );
	widget = glade_xml_get_widget( xml, "tab_popup" );
	page = (Page*)gtk_object_get_data( GTK_OBJECT( widget ), "page" );

	current = screem_site_get_current_page( current_site );

	if( current == page )
		screem_editor_buffer_text();

	screem_page_save( page );
}

void screem_page_save_as_tab( GtkWidget *item )
{
	GladeXML *xml;
	GtkWidget *widget;
	Page *page;
	Page *current;

	xml = glade_get_widget_tree( item );
	widget = glade_xml_get_widget( xml, "tab_popup" );
	page = (Page*)gtk_object_get_data( GTK_OBJECT( widget ), "page" );

	current = screem_site_get_current_page( current_site );

	if( current == page )
		screem_editor_buffer_text();

	/* bit of a hack this */
	screem_site_set_current_page( current_site, page );
	screem_page_save_as();
	screem_site_set_current_page( current_site, current );
}

void screem_page_close_tab( GtkWidget *item )
{
	GladeXML *xml;
	GtkWidget *widget;
	Page *page;
	Page *current;

	xml = glade_get_widget_tree( item );
	widget = glade_xml_get_widget( xml, "tab_popup" );
	page = (Page*)gtk_object_get_data( GTK_OBJECT( widget ), "page" );

	current = screem_site_get_current_page( current_site );

	if( current == page )
		screem_editor_buffer_text();

	screem_page_close_page( page );
}

void screem_page_rename_tab( GtkWidget *item )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( item );

	gnome_app_request_string( GNOME_APP( app ),
				  "Enter the new filename:",
				  (GnomeStringCallback)
				  screem_page_rename_tab_do, xml );
}

static void screem_page_rename_tab_do( gchar *filename, GladeXML *xml )
{
	GtkWidget *widget;
	Page *page;
	Page *current;

	const gchar *pathname;
	gchar *temp;
	gchar *temp2;

	if( ! filename ) {
		screem_show_warning( _( "No filename given." ) );
		return;
	}

	widget = glade_xml_get_widget( xml, "tab_popup" );
	page = (Page*)gtk_object_get_data( GTK_OBJECT( widget ), "page" );

	current = screem_site_get_current_page( current_site );

	if( current == page )
		screem_editor_buffer_text();

	pathname = screem_page_get_pathname( page );

	if( move_file( pathname, filename ) ) {
		/* successful */
		temp = g_dirname( pathname );
		temp2 = g_strconcat( temp, G_DIR_SEPARATOR_S, filename, NULL );
		screem_page_set_pathname( page, temp2 );
		screem_page_set_mod_time( page, time( NULL ) );
		screem_page_set_tab_label( page );
		g_free( temp );
		g_free( temp2 );
		refresh_file_tree();
	} else
		/* failed */
		screem_show_warning( _( "Unable to rename file" ) );
}

void screem_page_set_tab_label( Page *page )
{
	GtkWidget *pagebook;
	GtkWidget *widget;
	GtkWidget *label;
	const gchar *filename;
	gchar *pname;
	GtkTooltipsData *ttd;

	pagebook = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );
	filename = screem_page_get_pathname( page );

	widget = gtk_notebook_get_nth_page( GTK_NOTEBOOK( pagebook ),
					    screem_page_get_tab( page ) );
	pname = g_basename( filename );

	label = gtk_notebook_get_tab_label( GTK_NOTEBOOK( pagebook ), widget );
	ttd = gtk_tooltips_data_get( label );
	gtk_tooltips_set_tip( ttd->tooltips, ttd->widget, filename, "" );
	label = GTK_BIN( label )->child;
	gtk_label_set_text( GTK_LABEL( label ), pname );
}
