/*  Screem:  editor.c,
 *  The main editor, handles insertion of text, context dependant popup menu,
 *  special keypresses etc
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  TODO:
 *  stop building the parse tree on each keypress.  It should be built
 *  once and then modified with changes made to the document.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <ctype.h>
#include <gnome.h>
#include <gmodule.h>
#include <libgnome/gnome-regex.h>
#include <glade/glade.h>

#include <time.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "editor.h"
#include "editMenu.h"
#include "fileops.h"
#include "helpers.h"
#include "htmlfuncs.h"
#include "menus.h"
#include "page.h"
#include "pageUI.h"
#include "plugin.h"
#include "preferences.h"
#include "preview.h"
#include "site.h"
#include "support.h"
#include "structure.h"
#include "guile.h"

#include "dtd.h"

GList *highlight_modes = NULL;  /* a list of ScreemEditorMode structs */

extern Site *current_site;
extern GtkWidget *app;
extern Preferences *cfg;

static GtkWidget *editor; /* the current editor in use */
static GtkWidget *attr_menu = NULL;
static gboolean attr_up = FALSE;

static void screem_editor_insert_file_cb( void );
static gboolean screem_editor_cursor_moved( void );
static gboolean rebuild_attr_list( gint *handle );

static GtkWidget *screem_create_editor( void );
static gboolean screem_editor_keypress(GtkWidget *widget, GdkEventKey *event);
gboolean html_key_press( GdkEventKey *event, KeyCombo *combo );
static void screem_editor_drop( GtkWidget *widget, GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *selectionData,
				guint info, guint time );
static gboolean screem_editor_motion( GtkWidget *widget, 
				      GdkDragContext *context, 
				      gint x, gint y, guint time, 
				      gpointer data );

static gboolean tag_attributes_menu( gchar *name );
static void no_attribute( GtkWidget *widget, GdkEventKey *event );
static gint xy_to_cursor_pos( gint x, gint y );

static GtkWidget *perl_menu( void );
static GtkWidget *java_menu( void );
static void java_compile( GtkWidget *widget );


static void screem_editor_leave( void );
static void screem_editor_motionev( GtkWidget *widget, GdkEventMotion *event );
static gboolean screem_editor_tooltip( void );
static void goto_line( gchar *string );
static gint hi_trigger( void );

static void screem_editor_update_parse_tree( void );


static ScreemDTD* screem_editor_get_doctype( void );


static gboolean undo = FALSE;
static gboolean hilighting = FALSE;
static gboolean changed = FALSE;
static gboolean popup_open = FALSE;

#define DELAY 500
static guint timeout_handle;

static guint tooltip_timer;
static gboolean active = FALSE;
static GdkEventMotion *tip_event = NULL;

static gboolean inserted;

GNode *editor_parse_tree = NULL;

static gboolean press_handled;

/* popup menu */
static GnomeUIInfo editor_file_menu[] = {
        GNOMEUIINFO_MENU_NEW_ITEM( N_("_New Page..."), 
                                   N_("Add a new page to the site"), 
                                   0, NULL ),
	GNOMEUIINFO_MENU_NEW_ITEM( N_("_New Blank Page"),
				   N_("Create a blank page (add if working on a site)" ),
				   GTK_SIGNAL_FUNC( screem_page_create_blank_page ),
				   NULL ),
        GNOMEUIINFO_MENU_REVERT_ITEM(GTK_SIGNAL_FUNC(screem_page_revert_proxy),
				     0 ),
        GNOMEUIINFO_MENU_SAVE_ITEM( GTK_SIGNAL_FUNC( screem_page_save_proxy ),
				    0 ),
        GNOMEUIINFO_MENU_SAVE_AS_ITEM( GTK_SIGNAL_FUNC( screem_page_save_as ),
				       0 ),
        { GNOME_APP_UI_ITEM, N_("_Insert file..."), N_("Insert file"),
          GTK_SIGNAL_FUNC( screem_editor_insert_file_cb ), NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_END
};

static GnomeUIInfo editor_main_menu[] = {
        GNOMEUIINFO_MENU_FILE_TREE( editor_file_menu ),
        GNOMEUIINFO_MENU_EDIT_TREE( edit_menu ),
	GNOMEUIINFO_SUBTREE( N_("CVS"), cvs_menu ),
	/*        GNOMEUIINFO_SUBTREE( N_("Move" ), editorMoveMenu ),*/
	GNOMEUIINFO_SEPARATOR,
       	GNOMEUIINFO_END
};

/* DnD menu */
static GnomeUIInfo editor_dnd_menu[] = {
        { GNOME_APP_UI_ITEM, N_("Insert relative filename"),
          N_("Insert relative filename"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert complete filename"),
          N_("Insert complete filename"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert tag"), N_("Insert tag"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert tag attibute"),
          N_("If the drop is into a tag then the data is converted into\
 an attribute for that tag"), 0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert inline"),
          N_("Insert the text from the file into the page"), 0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_SEPARATOR,
        { GNOME_APP_UI_ITEM, N_("Cancel drag"), N_("Cancel drag"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_END
};

typedef enum _Drop_actions {
        RELATIVE_PATH,
        FULL_PATH,
        TAG,
        ATTRIBUTE,
        INLINE     /* inserts the file contents (text files only) */
} Drop_actions;

typedef enum _Drop_types {
	TARGET_URI_LIST,
	TARGET_URL,
	TARGET_COLOUR
} Drop_types;

static const GtkTargetEntry drop_types[] = {
        { "text/uri-list", 0, TARGET_URI_LIST },
	{ "text/x-moz-url", 0, TARGET_URL },
        { "_NETSCAPE_URL", 0, TARGET_URL },
        { "x-url/http", 0, TARGET_URL },
        { "x-url/ftp", 0, TARGET_URL },
        { "application/x-color", 0, TARGET_COLOUR }
};
static const gint num_drop_types = sizeof(drop_types) / sizeof(drop_types [0]);

typedef gboolean (*Tipcb)( gint pos, gchar *text );

/**
 * screem_editor_new:
 *
 * create the editor area, returning the scrolled window that contains it,
 * the text widget itself is the "text_widget" data for the scrolled window
 *
 * return values: a GtkWidget
 */
GtkWidget *screem_editor_new()
{
	GtkWidget *sw;

	sw = screem_create_editor();

	gtk_widget_set_usize( sw, cfg->editor_width, cfg->editor_height );

	return sw;
}

static gboolean screem_editor_cursor_moved()
{
	Page *page;
	static gint handle = -1;

	if( hilighting )
		return FALSE;

	page = screem_site_get_current_page( current_site );
	if( ! page )
		return FALSE;

	/* add a timeout to build a new attribute list, this is to
	   avoid constantly building an attribute list if the cursor is
	   being moved quickly, and avoids scrolling being slowed down */
	if( handle != -1 )
		gtk_timeout_remove( handle );

	handle = gtk_timeout_add( 250, (GtkFunction)rebuild_attr_list,
				  &handle );

	return FALSE;
}

static gboolean rebuild_attr_list( gint *handle )
{
       	*handle = -1;
	build_attribute_list();

	return FALSE;
}

static GtkWidget *screem_create_editor()
{
	GtkWidget *sw;
	GtkWidget *text;
	GtkStyle *style;
	GdkColor *cb;
	GdkColor *cf;

	text = gtk_text_new( NULL, NULL );

	GTK_TEXT( text )->default_tab_width = 8;

	sw = gtk_scrolled_window_new( GTK_TEXT( text )->hadj,
				      GTK_TEXT( text )->vadj );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ), 
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );
	gtk_container_add( GTK_CONTAINER( sw ), text );

	gtk_signal_connect( GTK_OBJECT( text ), "button_press_event",
			    GTK_SIGNAL_FUNC( screem_editor_popup ), 0 );

	gtk_signal_connect_after( GTK_OBJECT( text ), "button_press_event",
				  GTK_SIGNAL_FUNC( screem_editor_cursor_moved),
				  0 );

	gtk_signal_connect( GTK_OBJECT( text ), "key_press_event",
			    GTK_SIGNAL_FUNC( screem_editor_keypress ), 0 );

	gtk_drag_dest_set( text,
                           GTK_DEST_DEFAULT_HIGHLIGHT |
                           GTK_DEST_DEFAULT_DROP,
                           drop_types, num_drop_types,
                           GDK_ACTION_COPY | GDK_ACTION_ASK );

        gtk_signal_connect( GTK_OBJECT( text ), "drag_data_received",
                            GTK_SIGNAL_FUNC( screem_editor_drop ), NULL );
        gtk_signal_connect( GTK_OBJECT( text ), "drag_motion",
                            GTK_SIGNAL_FUNC( screem_editor_motion ), NULL );
	
	/* set its style */
	style = gtk_style_copy( gtk_widget_get_style( text ) );
	cb = &style->base[ 0 ];
        cb->red = cfg->cpage->back[ 0 ];
        cb->green = cfg->cpage->back[ 1 ];
        cb->blue = cfg->cpage->back[ 2 ];

	cf = &style->text[ 0 ];

        cf->red = cfg->cpage->fore[ 0 ];
        cf->green = cfg->cpage->fore[ 1 ];
        cf->blue = cfg->cpage->fore[ 2 ];

	style->font = cfg->mpage->font;
        gtk_widget_set_style( text, style );

	screem_editor_build_highlight_tables();

	gtk_text_set_word_wrap( GTK_TEXT( text ), TRUE );
        gtk_text_set_line_wrap( GTK_TEXT( text ), TRUE );
	timeout_handle = gtk_timeout_add( DELAY, (GtkFunction)hi_trigger, 0 );

	gtk_signal_connect( GTK_OBJECT( text ), "enter_notify_event",
			    GTK_SIGNAL_FUNC( screem_editor_normal_cursor ),
			    NULL );
	gtk_signal_connect( GTK_OBJECT( text ), "leave_notify_event",
			    GTK_SIGNAL_FUNC( screem_editor_leave ),
			    NULL );
	gtk_signal_connect( GTK_OBJECT( text ), "motion_notify_event",
			    GTK_SIGNAL_FUNC( screem_editor_motionev ),
			    NULL );
	gtk_signal_connect( GTK_OBJECT( text ), "insert_text",
			    GTK_SIGNAL_FUNC( screem_editor_text_inserted ), 
			    NULL );
	gtk_signal_connect( GTK_OBJECT( text ), "delete_text",
			    GTK_SIGNAL_FUNC( screem_editor_text_removed ),
			    NULL );
	gtk_signal_connect_after( GTK_OBJECT( text ), "insert_text",
				  GTK_SIGNAL_FUNC( screem_editor_update_parse_tree ),
				  NULL );
	gtk_signal_connect_after( GTK_OBJECT( text ), "delete_text",
				  GTK_SIGNAL_FUNC( screem_editor_update_parse_tree ),
				  NULL );

	gtk_signal_connect( GTK_OBJECT( GTK_TEXT( text )->vadj ),
			    "value_changed",
			    GTK_SIGNAL_FUNC( screem_editor_cursor_moved ), 0 );
	gtk_signal_connect( GTK_OBJECT( GTK_TEXT( text )->hadj ),
			    "value_changed",
			    GTK_SIGNAL_FUNC( screem_editor_cursor_moved ), 0 );

	gtk_widget_add_events( text, GDK_POINTER_MOTION_MASK );

	editor = text;

	return sw;
}

void screem_editor_normal_cursor()
{
	/* change cursor */
	static GdkCursor *cursor = NULL;

	if( ! cursor )
		cursor = gdk_cursor_new( GDK_XTERM );

	gdk_window_set_cursor( GTK_TEXT( editor )->text_area, cursor );
}

static void screem_editor_leave()
{
	if( active ) {
		gtk_timeout_remove( tooltip_timer );
		active = FALSE;
		gdk_event_free( (GdkEvent*)tip_event );
		tip_event = NULL;
	}
}

static void screem_editor_motionev( GtkWidget *widget, GdkEventMotion *event )
{
	/* reset the tooltip timer */
	if( active ) {
		gtk_timeout_remove( tooltip_timer );
	}

	if( ! screem_site_get_current_page( current_site ) )
		active = FALSE;
	else {
		tip_event =(GdkEventMotion*)gdk_event_copy( (GdkEvent*)event );
		tooltip_timer = gtk_timeout_add( 100, 
						 (GtkFunction)
						 screem_editor_tooltip, 
						 NULL );
		active = TRUE;
	}
}

static gboolean screem_editor_tooltip()
{
	gint pos;
	gchar *text;
	Page *page;
	const gchar *mime_type;
	ScreemEditorMode *mode;
	GModule *self;
	Tipcb cb;
	GList *list;

    	active = FALSE;

	page = screem_site_get_current_page( current_site );
	mime_type = screem_page_get_mime_type( page );
	mode = screem_editor_get_mode( mime_type );

	if( ! mode )
		return FALSE;

	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	pos = xy_to_cursor_pos( (int)(tip_event->x + 0.5), 
				(int)(tip_event->y + 0.5) );
	text = screem_editor_get_text( 0, -1 );

	for( list = mode->tips; list; list = list->next ) {
		g_module_symbol( self, (gchar*)list->data, (gpointer*)(&cb) );
		if( cb && text[ 0 ] != '\0' && cb( pos, text ) )
			break;
	}

	g_free( text );
	gdk_event_free( (GdkEvent*)tip_event );
	tip_event = NULL;

	return FALSE;
}

gboolean html_editor_tip( gint pos, gchar *text );
gboolean html_editor_tip( gint pos, gchar *text )
{
	ScreemDTD *dtd;
	gchar *tag;
	gchar *name;
	ScreemDTDElement *element;
	gchar *message;

	if( ( pos = in_tag( text, pos ) ) ) {
		pos --;
		tag = next_tag( text + pos, &pos, &name );
		/* so we can tooltip PHP/ASP etc functions */
		if( tag[ 1 ] == '?' || tag[ 1 ] == '!' ) {
			g_free( name );
			return FALSE;
		}
		/* we have the tag, look it up in the DTD */
		dtd = screem_get_doctype( text );
		element = screem_dtd_check_element( dtd, name );
		if( ! element ) {
			message = g_strdup_printf( _("%s is not a valid element of this documents DTD" ), name );
			screem_show_message( message );
		} else {
			/* its valid, but is it allowed here? */
			message = g_strdup( "" );
		}

		g_free( name );
		g_free( tag );
		g_free( message );
		return TRUE;
	}

	return FALSE;
}

/**
 * screem_editor_popup:
 *
 * displays the editor menu
 *
 * return values: none
 */
gboolean screem_editor_popup( GtkWidget *widget, GdkEventButton *event )
{
	gint cpos;
	gint pos2;
    	gint pos;
	gint tag_pos;
	gint tag_len;
	gchar *text;
	gchar *tag;
	gchar *tag2;
	gchar *title;
	GtkWidget *menu;
	GtkWidget *menu2;
	GtkWidget *menu_item;

	GtkWidget *helper_menu;

	gboolean tag_known;

	Page *page;
	ScreemDTD *dtd;

	const gchar *pathname;
	const gchar *mime_type;

	gint start;
	gint len;
	gboolean sel;
	
	page = screem_site_get_current_page( current_site );

	if( ! page )
		return FALSE;

	pathname = screem_page_get_pathname( page );

	if( event->button != 3 )
		return FALSE;

	popup_open = TRUE;
	gtk_signal_emit_stop_by_name( GTK_OBJECT( widget ),
				      "button_press_event" );
	
#ifndef HAVE_GNOME_VFS
	mime_type = gnome_mime_type( pathname );
#else
	mime_type = gnome_vfs_mime_type_from_name( pathname );
#endif

	screem_editor_buffer_text();
	
	cpos = screem_editor_get_pos();

	/* get position of click */
	pos2 = pos = xy_to_cursor_pos( (int)(event->x + 0.5),
				       (int)(event->y + 0.5) );
	sel = screem_editor_has_selection( &start, &len );
	screem_editor_set_pos( pos );

	if( sel )
		screem_editor_select_region( start, len );
	
	/* create the menu */
	menu = gnome_popup_menu_new( editor_main_menu );
	
	helper_menu = screem_helper_menu();
	gtk_menu_insert( GTK_MENU( menu ), helper_menu, 3 );
	
	menu2 = screem_guile_get_menu();
	if( menu2 )
		gtk_menu_append( GTK_MENU( menu ), menu2 );

	/* id the tag we are inside so we can offer attribute editing */
	
	text = screem_editor_get_text( 0, -1 );
	if( ( ! strcmp( "text/html", mime_type ) ||
	      ! strcmp( "text/sgml", mime_type ) ||
	      ! strcmp( "text/xml", mime_type ) ||
	      ! strcmp( "application/x-php", mime_type ) ||
	      ! strcmp( "application/x-asp", mime_type ) ) &&
	    ( pos = in_tag( text, pos ) ) ) {
		pos --;
		tag_pos = pos;
		tag = next_tag( text + pos, &pos, &tag2 );
		tag_len = pos + 1;

		dtd = screem_get_doctype( text );
		menu_item = screem_dtd_build_attribute_menu( dtd, tag2, TRUE);
		if( menu_item )
			gtk_menu_prepend( GTK_MENU( menu ), menu_item );
		tag_known = (gboolean)menu_item;

		g_free( tag );
		/* add the wizard option */
		if( (menu_item = tag_plugin_wizard( tag2, tag_pos, tag_len )) )
			gtk_menu_append( GTK_MENU( menu ), menu_item );
				
		/* add the help option */
		if( tag_known && ! strcmp( "text/html", mime_type ) ) {
			title = g_strdup_printf( "<%s> Help", tag2 );
			menu_item = gtk_menu_item_new_with_label( title );    
			gtk_widget_show( menu_item );
			gtk_menu_append( GTK_MENU( menu ), menu_item );
			gtk_signal_connect_full( GTK_OBJECT( menu_item ), 
						 "activate", 
						 GTK_SIGNAL_FUNC( screem_help_show ), 
						 NULL, 
						 (gpointer) g_strdup( tag2 ),
						 g_free, 1, 0 );
		}
		g_free( tag2 );
	} else if( ( ! strcmp( "text/html", mime_type ) ||
		     ! strcmp( "text/sgml", mime_type ) ||
		     ! strcmp( "text/xml", mime_type ) ||
		     ! strcmp( "application/x-php", mime_type ) ||
		     ! strcmp( "application/x-asp", mime_type ) ) ) {
		/* add the insert option */
		GNode *node;
		if( cfg->mpage->parse )
			node = screem_html_parse_tree_find_pos( editor_parse_tree, 
								pos2,
								FALSE );
		else
			node = NULL;
		tag2 = NULL;
		if( node && node->data )
			tag2 = ((Node*)node->data)->name;
		dtd = screem_get_doctype( text );

		if( ( menu_item = screem_dtd_allowed_tags_menu( dtd, tag2 ) ) )
			gtk_menu_append( GTK_MENU( menu ), menu_item );
	} else if( ! strcmp( "text/x-perl", mime_type ) ) {
		/* create perl menu */
		menu_item = perl_menu();
		gtk_menu_prepend( GTK_MENU( menu ), menu_item );
		
	} else if( ! strcmp( "text/x-java", mime_type ) ) {
		/* create java menu */
		menu_item = java_menu();
		gtk_menu_prepend( GTK_MENU( menu ), menu_item );
	}
	g_free( text );
	
	/* display menu */
	inserted = FALSE;
	gnome_popup_menu_do_popup_modal( menu, 0, 0, event, 0 );
	gtk_widget_destroy( menu );
	if( ! inserted )
		screem_editor_set_pos( cpos );
	
	if( sel )
		screem_editor_select_region( start, len );

	popup_open = FALSE;
		
	return TRUE;
}

/**
 * screem_editor_enable:
 *
 * enables editing.
 *
 * return values: none
 */
void screem_editor_enable()
{
	gtk_editable_set_editable( GTK_EDITABLE( editor ), TRUE );
}

/**
 * screem_editor_disable:
 *
 * disables editing.
 *
 * return values: none
 */
void screem_editor_disable()
{
	gtk_editable_set_editable( GTK_EDITABLE( editor ), FALSE );
}

/**
 * screem_editor_clear:
 *
 * clears the editor
 *
 * return values: none
 */
void screem_editor_clear()
{
	screem_editor_delete_forward( 0, -1 );
}


GtkStyle *screem_editor_get_style()
{
	return gtk_style_copy( gtk_widget_get_style( editor ) );
}

void screem_editor_set_style( GtkStyle *style )
{
	g_return_if_fail( style != NULL );

	gtk_widget_set_style( editor, style );
}

gint screem_editor_get_width()
{
	if( ! editor )
		return cfg->editor_width;

	return editor->allocation.width;
}

gint screem_editor_get_height()
{

	if( ! editor )
		return cfg->editor_height;

	return editor->allocation.height;
}

gint screem_editor_get_length()
{
	gint len;

	len = gtk_text_get_length( GTK_TEXT( editor ) );

	return len;
}

gchar *screem_editor_get_text( gint from, gint len )
{
	gint end = len;

      	if( end == -1 )
		end = screem_editor_get_length();
	else
		end += from;

	return gtk_editable_get_chars( GTK_EDITABLE( editor ), from, end );
}

/**
 * screem_editor_has_selection:
 * @start:  store the start pos here;
 * @end: store the end pos here
 *
 * identify if we have selected some text in the editor
 *
 * return values: boolean
 */
gboolean screem_editor_has_selection( gint *start, gint *end )
{
	gint swap;
	gboolean ret = GTK_EDITABLE( editor )->has_selection;

	if( ret && start )
		*start = GTK_EDITABLE( editor )->selection_start_pos;
	if( ret && end )
                *end = GTK_EDITABLE( editor )->selection_end_pos;

	if( ret && start && end ) {
		if( *start > *end ) {
			swap = *end;
			*end = *start;
			*start = swap;
		}
	}

	return ret;
}

void screem_editor_select_region( gint start, gint len )
{
    	gtk_editable_select_region( GTK_EDITABLE( editor ), start, len );
	gtk_editable_claim_selection( GTK_EDITABLE( editor ), TRUE, time(0));
}

void screem_editor_cut()
{
	gtk_editable_cut_clipboard( GTK_EDITABLE( editor ) );
}

void screem_editor_copy()
{
	gtk_editable_copy_clipboard( GTK_EDITABLE( editor ) );
}

void screem_editor_clear_selection()
{
	gtk_editable_delete_selection( GTK_EDITABLE( editor ) );
}

void screem_editor_paste()
{
	gtk_editable_paste_clipboard( GTK_EDITABLE( editor ) );
}

void screem_editor_paste_encoded()
{
	gchar *text;

	/* FIXME: this shouldn't be done, the fix is probably to write
	   custom clipboard functions rather than using the gtk editable
	   functions */
	text = GTK_EDITABLE( editor )->clipboard_text;
	text = screem_html_encode_text( text );

	screem_editor_insert( -1, text );
}

void screem_editor_set_pos( gint pos )
{
	gtk_editable_set_position( GTK_EDITABLE( editor ), pos );
	screem_editor_cursor_moved();
}

gint screem_editor_get_pos()
{
	return gtk_editable_get_position( GTK_EDITABLE( editor ) );
}

/**
 * screem_editor_delete_forward:
 * @pos: the position to start at
 * @len: the number of chars to delete, -1 to delete everything
 *
 * return values: none
 */
void screem_editor_delete_forward( gint pos, gint len )
{
	if( len == -1 )
		len = ( screem_editor_get_length() - pos );

	gtk_text_freeze( GTK_TEXT( editor ) );

	gtk_editable_delete_text( GTK_EDITABLE( editor ), pos, pos + len );

	gtk_text_thaw( GTK_TEXT( editor ) );
}


/**
 * screem_editor_buffer_text:
 *
 * takes the text in the editor and places it in the buffer for the
 * page being editted
 *
 * return values: none
 */
void screem_editor_buffer_text()
{
	Page *page;
	gchar *text;

	page = screem_site_get_current_page( current_site );

	if( ! page )
		return;

	text = screem_editor_get_text( 0, -1 );

	screem_page_set_data( page, text );
	g_free( text );
}

/**
 * screem_editor_insert_markup:
 * @open_element:  the opening element
 * @close_element: the closing element
 *
 * handles inserting markup, including wrapping around selected text
 *
 * return values: none
 */
void screem_editor_insert_markup( gchar *open_element, gchar *close_element )
{
	gboolean has_selection;
	guint start;
	guint end;
	gint pos;
	gint ipos;
	Page *page;

	page = screem_site_get_current_page( current_site );
	
	if( ! page )
		return;

	pos = screem_editor_get_pos();
	ipos = pos;

	has_selection = screem_editor_has_selection( (gint*)(&start),
						     (gint*)(&end) );

	if( has_selection ) {
		if( open_element ) {
			screem_editor_insert( start, open_element );
                        end += strlen( open_element );
		}
		pos = end;
		if( close_element ) {
                        screem_editor_insert( end, close_element );
                        pos += strlen( close_element );
                }
	} else {
		if( open_element ) {
			screem_editor_insert( pos, open_element );
			pos += strlen( open_element );
		}
		if( close_element ) {
			screem_editor_insert( pos, close_element );
			pos += strlen( close_element );
		}

		if( open_element )
			pos = ipos + strlen( open_element );
		else if( close_element )
			pos = ipos + strlen( close_element );
	}
	screem_editor_set_pos( pos );

	if( popup_open )
		inserted = TRUE;
}


/**
 * screem_editor_insert_attribute:
 * @attribute: the attribute text to insert
 *
 * inserts an attribute into an element
 *
 * return values: none
 */
void screem_editor_insert_attribute( gchar *attribute )
{
	gint pos;
	gint cpos;
        gint start;
        gchar *insert;
        gchar *text;
        gchar *tag;
        gchar *attr;
	Page *page;

	gchar *temp;

	page = screem_site_get_current_page( current_site );
	
	if( ! page || ! attribute )
		return;

        cpos = pos = screem_editor_get_pos();
        text = screem_editor_get_text( 0, -1 );

        if( ! ( pos = in_tag( text, pos ) ) ) {
		g_free( text );
		return;
        }
        pos --;
        start = pos;
        /* pos == the start of the tag */
        tag = next_tag( text + pos, &pos, NULL );
        g_free( text );

  	/* does the tag already have this attribute?
           = match:  [A-Za-z]*=\"[^"]*\"
           other: [A-Za-z]*
        */
      	temp = g_strconcat( " ", attribute, NULL );
        if( ( attr = find_text( temp, "=", NULL, NULL ) ) ) {
                attr = g_strndup( temp, attr - temp );
                insert = g_strdup_printf( "%s=\\\"[^\\\"]*\\\"", attr );
                g_free( attr );
        } else {
                insert = g_strdup( temp );
        }

        /* now we try and replace it */
        if( ! ( attr = find_text( tag, insert, temp, NULL ) ) ) {
                /* we didn't find it so we just insert it */
                g_free( insert );
		insert = g_strdup_printf( " %s", attribute );
                screem_editor_insert( pos, insert );
		pos += strlen( attribute ) + 1;
	} else {
                /* erase all the tag's attributes */
		g_free( tag );
                tag = attr;
		screem_editor_delete_forward( start, pos - start + 1 );
                insert = g_strdup( tag );
                screem_editor_insert( start, insert );
		pos = start + strlen( insert );
        }

	g_free( tag );
	g_free( insert );
	g_free( temp );

	inserted = TRUE;

	screem_editor_set_pos( pos );
}

/**
 * screem_editor_insert_file:
 * @filename: the name of the file to insert
 *
 * inserts the contents of a text file into the editor at the
 * current cursor position
 *
 * return values: none
 */
void screem_editor_insert_file( gchar *filename )
{
	FILE *in;
	gint size;
	gint pos;
	gchar buffer[ 1025 ];

	pos = screem_editor_get_pos();

	if( ! ( in = fopen( filename, "r" ) ) )
		return;

	while( ( size = fread( buffer, 1, 1024, in ) ) ) {
		buffer[ size ] = 0;
		screem_editor_insert( pos, buffer );
		pos += size;
	}

	fclose( in );
}

static void screem_editor_insert_file_cb()
{
	gchar *filename;

	filename = file_select( _( "Select file to insert" ) );

	if( filename )
		screem_editor_insert_file( filename );

	g_free( filename );
}

void screem_editor_remove_syntax_menus( Page *page )
{
	const gchar *mime_type;
	ScreemEditorMode *mode;
	ScreemEditorMenu *menu;
	GList *list;
	GtkWidget *bar;
	GtkWidget *shell;

	g_return_if_fail( page != NULL );
	if( page != screem_site_get_current_page(current_site) )
		return;

	bar = GNOME_APP( app )->menubar;
	shell = (GtkWidget*)(& GTK_MENU_BAR( bar )->menu_shell);

	/* remove old menu items */
	mime_type = screem_page_get_mime_type( page );
	mode = screem_editor_get_mode( mime_type );
	if( mode )
		list = mode->menus;
	else
		list = NULL;
	for( ; list; list = list->next ) {
		menu = (ScreemEditorMenu*)list->data;
		gtk_container_remove( GTK_CONTAINER( shell ), menu->menu );
	}
}


void screem_editor_display_page( gpointer page )
{
	Page* cpage;
	GList *list;
	const gchar *mime_type;
	ScreemEditorMode *mode;
	ScreemEditorMenu *menu;
	GtkWidget *bar;
	GtkWidget *shell;
	gint pos;
	ScreemDTD *dtd;

	cpage = screem_site_get_current_page( current_site );
	if( cpage && cpage != page )
		screem_editor_buffer_text();

	if( editor_parse_tree )
		screem_html_destroy_parse_tree( editor_parse_tree );
	editor_parse_tree = NULL;

	bar = GNOME_APP( app )->menubar;
	shell = (GtkWidget*)(& GTK_MENU_BAR( bar )->menu_shell);

	if( cpage )
		screem_editor_remove_syntax_menus( cpage );
	
	undo = TRUE;
	if( ! page ) {
		screem_editor_clear();
		screem_editor_disable();
	} else {
		mime_type = screem_page_get_mime_type( (Page*)page );
		mode = screem_editor_get_mode( mime_type );
	
		/* custom menus go after the insert menu */
		shell = gnome_app_find_menu_pos( shell,
						 _("_Insert"), &pos );

		/* display new menus */
		if( mode )
			list = mode->menus;
		else
			list = NULL;

		for( ; list; list = list->next ) {
			menu = (ScreemEditorMenu*)list->data;
			
			gtk_menu_shell_insert( GTK_MENU_SHELL( shell ),
					       menu->menu, pos );
		}

		screem_site_set_current_page( current_site, (Page*)page );
		screem_editor_clear();
		screem_editor_insert( 0, screem_page_get_data( (Page*)page ) );
		screem_editor_enable();

		/* hmm, why do I need to do this here, the insertion should
		   cause it to be done automatically but it isn't */
		if( cfg->mpage->parse ) {
			dtd = screem_editor_get_doctype();
			editor_parse_tree = screem_html_build_parse_tree( dtd, screem_page_get_data( (Page*)page ), 0 );
		}
	}
	undo = FALSE;
}

/**
 * screem_editor_insert:
 * @pos:   the postion to insert the text at
 * @text: the text to insert
 *
 * inserts text into the editor at the position requested.
 * if pos > total text length in the editor then the text will be
 * appended to the current text in the editor
 *
 * return values: none
 */
void screem_editor_insert( gint pos, const gchar *text )
{
	gint len;
	gint start;
	const gchar *pathname;
	const gchar *mime_type;

	Page *page;

	page = screem_site_get_current_page( current_site );
	
	if( ! page || ! text )
		return;

	pathname = screem_page_get_pathname( page );
#ifndef HAVE_GNOME_VFS
	mime_type = gnome_mime_type( pathname );
#else
	mime_type = gnome_vfs_mime_type_from_name( pathname );
#endif

	len = screem_editor_get_length();

	if( pos == -1 )
		pos = screem_editor_get_pos();

	if( pos > len )
		pos = len;

	start = pos;

	len = strlen( text );

	gtk_text_freeze( GTK_TEXT( editor ) );

	gtk_editable_insert_text( GTK_EDITABLE( editor ), text, len, &pos );
	
	gtk_text_thaw( GTK_TEXT( editor ) );
	gtk_widget_grab_focus( editor );
}

static gboolean screem_editor_keypress( GtkWidget *widget, GdkEventKey *event )
{
	Page *page;

	const gchar *pathname;
	const gchar *mime_type;

	ScreemEditorMode *mode;
	GList *list;
	KeysCallback cb;
	GModule *self;

	gchar *string;
	KeyCombo *combo;
	gint pos;

	page = screem_site_get_current_page( current_site );
	
	press_handled = FALSE;

	if( ! page )
		return FALSE;

	pathname = screem_page_get_pathname( page );
#ifndef HAVE_GNOME_VFS
	mime_type = gnome_mime_type( pathname );
#else
	mime_type = gnome_vfs_mime_type_from_name( pathname );
#endif

	mode = screem_editor_get_mode( mime_type );
	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	/* does string match any of the editor keys set by the user? */
	string = convert_keysym_state_to_string( event->keyval, event->state );
	combo = NULL;

	for( list = cfg->kpage->keys; list; list = list->next ) {
		combo = (KeyCombo*)list->data;
		if( ! strcmp( string, combo->key_name ) ) {
			gtk_signal_emit_stop_by_name( GTK_OBJECT( editor ),
						      "key_press_event" );
			if( combo->action == TEXT ) {
				pos = screem_editor_get_pos();
				screem_editor_insert( pos, combo->text );
				pos += strlen( combo->text );
				screem_editor_set_pos( pos );
				g_free( string );
				return TRUE;
			}
			break;
		} else
			combo = NULL;
	}
	g_free( string );

	list = NULL;
	if( mode ) {
		for( list = mode->keys; list; list = list->next ) {
			g_module_symbol( self, (gchar*)list->data, 
					 (gpointer*)(&cb) );
			if( cb && cb( event, combo ) )
				break;
		}
	}

	press_handled = (gboolean)list;

	if( ! press_handled && 
	    ( ! strcmp( "text/sgml", mime_type ) || 
	      ! strcmp( "text/xml", mime_type ) ) ) {
		press_handled = html_key_press( event, NULL );
	}

	screem_editor_cursor_moved();

	return press_handled;
}

gboolean html_key_press( GdkEventKey *event, KeyCombo *combo )
{
	gchar *text;
	gchar *tag = NULL;
	gchar *tag2;
	gchar *format;
	gint pos;
	gint start;
	gint len;
	gint len2;
	gboolean empty = FALSE;
	Page *page;
	const gchar *charset;

	GNode *node;
	Node *n;

	pos = screem_editor_get_pos();
	page = screem_site_get_current_page( current_site );
	charset = screem_page_get_charset( page );

	text = screem_editor_get_text( 0, -1 );

	/* Are we in an inline for scripting languages, 
	   ie  <? ?>  <% %> <?php ?> <!-- --> */
	if( ( start = in_tag( text, pos ) ) &&
	    ( text[ start ] == '?' || text[ start ] == '%' ||
	      text[ start ] == '!' ) ) {
		g_free( text );
		return FALSE;
	}

	/* act upon key combos, the user must want them otherwise
	   they wouldn't have pressed the combo so the check
	   is made before the script check */
	if( combo ) {
		switch( combo->action ) {
		case ADD_CLOSE:
			/* FIXME: place code in sensible place,
			 *        check tag is allowed to be closed
			 */

			if( start ) {
				start --;
				tag = next_tag( &text[ start ], &start, &tag2);
				g_free( tag );
			} else
				tag2 = NULL;
			if( tag2 && tag2[ 0 ] != '/' &&
			    !screem_html_next_tag_close( text, tag2, start ) ){
				tag = g_strconcat( "</", tag2, ">", NULL );
				start ++;
				screem_editor_insert( start, tag );
				screem_editor_set_pos( start );
				g_free( tag );
				g_free( tag2 );
				g_free( text );
				gtk_signal_emit_stop_by_name(GTK_OBJECT(editor), "key_press_event");
				return TRUE;
			} else{
				g_free( tag2 );
			}
			break;
		default:
			/* we don't recognise it, this shouldn't happen */
			break;
		}
	}


	/* ensure we are in a context where the user would want
	   the features */
	if( cfg->mpage->parse )
		node = screem_html_parse_tree_find_pos( editor_parse_tree, pos, 
							FALSE );
	else
		node = NULL;

	while( node ) {
		n = (Node*)node->data;
		if( n && n->name && ! strcasecmp( "script", n->name ) ) {
			/* FIXME: should id what language the script is
			   and support key presses for that language,
			   eg bracket expansion etc */
			g_free( text );
			return FALSE;
		}
		/* we need to check its parents as well, incase the
		   script is inserting HTML */
		node = node->parent;
	}

	switch( event->keyval ) {
	case GDK_space:
		if( ( attr_up ) || ( ! cfg->mpage->inline_tagging ) ||
		    ( in_attribute( text, pos ) ) || ( ! start ) ) {
			g_free( text );
			return FALSE;
		}

		len = start;
		start --;
		tag = next_tag( &text[ start ], &start, &tag2 );
		len2 = strlen( tag2 );
		g_free( text );
	
		/* we must know about the tag and also not be positioned
		   inside its name in the editor */
		if( ( pos - len ) < len2 )  {
			g_free( tag2 );
			g_free( tag );
			return FALSE;
		} else if( tag_attributes_menu( tag2 ) ) {
			gtk_signal_emit_stop_by_name( GTK_OBJECT( editor ),
						      "key_press_event" );
			g_free( tag2 );
			g_free( tag );
			return TRUE;
		}
		g_free( tag2 );
		g_free( tag );
		break;
	case GDK_slash:
		if( ! cfg->mpage->intelliclose || ! start ||
		    ( start && text[ start ] != '>' ) ) {
			g_free( text );
			return FALSE;
		}
		tag2 = screem_html_autoclose( text, start - 2 );

		if( tag2 && 
		    ! screem_html_next_tag_close( text, tag2, start ) ) {
			/* we found a tag that needed closing */
			screem_editor_insert( pos, "/" );
			screem_editor_insert( pos + 1, tag2 );
			gtk_signal_emit_stop_by_name(GTK_OBJECT(editor),
						     "key_press_event");
			screem_editor_set_pos( pos + strlen( tag2 ) + 2 );
			g_free( tag2 );
		} else {
			if( tag2 )
				tag =g_strdup_printf(_("%s is already closed"),
						     tag2 );
			else
				tag=g_strdup_printf(_("no tag needs closing"));
			screem_show_message( tag );
			g_free( tag );
			g_free( tag2 );
			tag2 = NULL;
		}

		g_free( text );
	   		
		return (gboolean)tag2;

		break;
	default:
		if( ! ( tag = screem_html_key_to_ent( event->keyval ) ) ) {
			g_free( text );
			return FALSE;
		}
		
		/* we're about to insert a character with a special HTML
		   meaning (tag or entity). To make sure that this can
		   be done we have to check if we're already inside another
		   entity. Look for a semicolon after the current position
		   AND an ampersand before the current position.
		   This search is aborted as soon as we find a character that
		   is part of another HTML construct */
		if( pos != 0 )
			empty = ( text[ pos - 1 ] == '&' && 
				  text[ pos ] == ';' );
		else
			empty = FALSE;
		
		if( ! empty && in_entity( text, pos ) ) {
			/* let the widget handle insertion as normal as
			   the entity
			   already exists and is not empty */
			g_free( text );
			return FALSE;
		}

		if( empty ) 
			format = "%s";
		else if( text[ pos ] == ';' )
			format = "&%s";
		else if( pos > 0 && text[ pos - 1 ] == '&' )
			format = "%s;";
		else
			format = "&%s;";

		tag = g_strdup_printf( format, tag );

		/* special cases */
		if( event->keyval == GDK_ampersand && ! empty ) {
			g_free( tag );
			tag = g_strdup_printf( format , "" ) ;
		} else if( event->keyval == GDK_quotedbl &&
			   ( ! ( empty |= in_attribute( text, pos ) ) ) &&
			   ( ! ( empty = ! in_tag( text, pos ) ) ) ) {
			/* if pos - 1 == "=" then insert two, place between */
			g_free( tag );
			if( text[ pos - 1 ] == '=' &&
			   ( text[ pos ] == ' ' || text[ pos ] == '>' ) )
				tag = g_strdup_printf( "\"\"" );
			else {
				g_free( text );
				return FALSE;
			}
		} else if( event->keyval == GDK_Tab && ! empty ) {
			g_free( tag );
			if( ! cfg->mpage->auto_indent ) {
				/* insert a normal tab as not in an entity,
				   and auto indent is off */
				g_free( text );
				return FALSE;
			}
			screem_editor_indent( text, pos );
			g_free( text );
			gtk_signal_emit_stop_by_name( GTK_OBJECT( editor ), 
						      "key_press_event" );
			return TRUE;
		} else if( event->keyval == GDK_less && ! empty ) {
			/* We add this manually along with a
			   GDK_greater, unless we are found to be
			   inside a tag */
			g_free( tag );
			if( in_tag( text, pos ) ) {
				/* in tag so let widget do it */
				g_free( text );
				return FALSE;
			}
			tag = g_strdup_printf( "<>" );
		} else if( event->keyval == GDK_greater && ! empty ) {
			/* to keep consistant with GDK_less we
			   do a normal insert if we aren't inside
			   and empty entity */
			g_free( tag );
			g_free( text );
			return FALSE;
		}
		
		g_free( text );

		/* are we performing entity insertion? */
		if( cfg->mpage->ent != ON ) {
			switch( event->keyval ) {
			case GDK_quotedbl:
			case GDK_Tab:
			case GDK_less:
			case GDK_greater:
				if( empty && cfg->mpage->ent == OFF ) {
					/* we only don't do these for off */
					g_free( tag );
					return FALSE;
				}
				break;
			case GDK_ampersand:
				if( cfg->mpage->ent == BY_SET )
					break;
			default:
				if( cfg->mpage->ent == BY_SET &&
				    ! g_strcasecmp( "ISO-8859-1", charset ) )
					break;
				g_free( tag );
				return FALSE;
			}
		}

		screem_editor_insert( pos, tag );
		if( empty && event->keyval != GDK_quotedbl ) 
			pos += ( strlen(tag) + 1 );
		else if( event->keyval == GDK_less ||
			 event->keyval == GDK_ampersand ||
			 (event->keyval == GDK_quotedbl 
			  && text[ pos - 1] == '=') ) 
			pos ++;
		else 
			pos += strlen( tag );
		
		g_free( tag );
		screem_editor_set_pos( pos );
		gtk_signal_emit_stop_by_name( GTK_OBJECT( editor ), 
					      "key_press_event" );
		return TRUE;
		break;
	}
	return FALSE;
}

static void screem_editor_drop( GtkWidget *widget, GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *selectionData,
				guint info, guint time )
{
	Page *page;
	GtkWidget *popup;
	gint item;
	gint pos;
	guint16 *colours;
	gchar *text = NULL;
	gchar *dir;
	gchar *temp;
	Drop_actions action;

	gboolean inline_;

	const gchar *pathname;
	const gchar *mime_type;
	const gchar *spath;

	gchar cwd[ 16384 ];

	getcwd( cwd, 16384 );

	page = screem_site_get_current_page( current_site );
	spath = screem_site_get_pathname( current_site );
	
	if( ! page )
		return;

	pathname = screem_page_get_pathname( page );

	/* if we are doing a middle drag then we need to ask the user
	   what to do */
	if( context->action == GDK_ACTION_ASK ) {
                popup = gnome_popup_menu_new( editor_dnd_menu );
                item = gnome_popup_menu_do_popup_modal( popup, 0, 0, 0, 0 );
                switch( item ) {
                case 0: /* insert relative filename */
                        action = RELATIVE_PATH;
                        break;
                case 1: /* insert complete filename */
                        action = FULL_PATH;
                        break;
                case 2: /* insert tag */
                        action = TAG;
                        break;
                case 3: /* insert tag attribute */
                        action = ATTRIBUTE;
                        break;
                case 4: /* insert text inline */
                        action = INLINE;
                        break;
                default:
                        return;
                        break;
                }
        } else {
                action = RELATIVE_PATH;
	}

	/* position the cursor at the drop location */
	pos = xy_to_cursor_pos( x, y );
	screem_editor_set_pos( pos );

	/* handle the different drop types */
	switch( info ) {
	case TARGET_URI_LIST:
		text = selectionData->data + strlen( "file:" );
		text[ strlen( text ) - 2 ] = 0;
#ifndef HAVE_GNOME_VFS
		mime_type = gnome_mime_type_or_default( text, "unknown" );
#else
		mime_type = gnome_vfs_mime_type_from_name_or_default( pathname,
								      "unknown" );
#endif

		/* are we doing an inline insert? */
		if( action == INLINE ) {
			inline_ = screem_page_is_mime_type_page( mime_type );
			if( inline_ ) {
				screem_editor_insert_file( text );
				return;
			}
		}

		/* insert using relative path? */
		if( action != FULL_PATH ) {
			dir = g_dirname( pathname );
			chdir( dir );
			g_free( dir );
			text = relative_path( text, spath );
			chdir( cwd );
		} else {
			text = g_strdup( text );
		}

		/* inserting a tag? */
		if( action == TAG ) {
			temp = text;
			if( ! strncmp( mime_type, "image/", 6 ) )
				text = g_strdup_printf( "<img src=\"%s\" alt=\"\">", temp );
			else
				text = g_strdup_printf( "<a href=\"%s\"> </a>",
							temp );
			g_free( temp );
		}
		break;
	case TARGET_URL:
		if( action == TAG )
			text = g_strdup_printf( "<a href=\"%s\"> </a>",
						selectionData->data );
		else
			text = g_strdup( selectionData->data );
		break;
	case TARGET_COLOUR:
		colours = (guint16*)selectionData->data;
                temp = g_strdup_printf( "\"#%.2x%.2x%.2x\"",colours[ 0 ] >> 8, 
                                        colours[ 1 ] >> 8, colours[ 2 ] >> 8 );
		switch( action ) {
		case TAG:
                        text = g_strdup_printf( "<font color=%s> </font>",
                                                temp );
                        g_free( temp );
                        break;
		case ATTRIBUTE:
			screem_editor_get_text( 0, -1 );
                        if( ( pos = in_tag( text, pos ) ) ) {
                                /* we are inside a tag */
                                g_free( text );
                                text = g_strdup_printf( " color=%s", temp );
                                screem_editor_insert_attribute( text );
                                g_free( text );
                                return;
                        } else {
                                /* not in a tag, but we will insert it anyway*/
                                g_free( text );
                                text = g_strdup_printf( " color=%s", temp );
                                g_free( temp );
                        }
                        break;
		default:
			text = temp;
			break;
		}
		break;
	default:
		/* we don't know about this type, and as such
		   this case should never execute */
		screem_show_warning( _("Error: invalid drop type for editor occured\n") );
		g_free( text );
		text = NULL;
		break;
	}

	/* insert the text */
	if( text )
		screem_editor_insert( pos, text );

	g_free( text );
}

static gboolean screem_editor_motion( GtkWidget *widget, 
				      GdkDragContext *context, 
				      gint x, gint y, guint time, 
				      gpointer data )
{
	GdkDragAction action;
	
        if( context->suggested_action != GDK_ACTION_ASK )
                action = GDK_ACTION_COPY;
        else
                action = GDK_ACTION_ASK;

        gdk_drag_status( context, action, time );

        return TRUE;
}

static gint hi_trigger()
{
	Page *page;
	gchar *text;
	GtkAdjustment *adj;
	gfloat value;
	gint pos;

	gint spos;
	gchar *start_chunk;
	gchar *end_chunk;
	gchar *end;
	gint len_chunk;
	gchar *str;

	gint pat_len = 0;
	ScreemEditorMode *mode;
	ScreemHighlightTable *table;
	ScreemHighlightEntry *entry;
	GList *entries;
	const gchar *mime_type;

	GdkColor *f;
	GdkColor *b;
	ColourGroup *cg;

	size_t nmatch = 255;
	regmatch_t pmatch[ 255 ];

	gint top;

	page = screem_site_get_current_page( current_site );

	if( (! cfg->mpage->highlighting) || popup_open  || (! page) )
		return FALSE;
	
	mime_type = screem_page_get_mime_type( page );
	mode = screem_editor_get_mode( mime_type );

	if( ! mode )
		return FALSE;
       
	table = mode->table;
	hilighting = TRUE;
	adj = GTK_TEXT( editor )->vadj;
	value = adj->value;

	spos = screem_editor_get_pos();
	text = screem_editor_get_text( 0, -1 );

	top = GTK_TEXT( editor )->first_line_start_index;

	gtk_text_freeze( GTK_TEXT( editor ) );

	end = text + strlen( text );
	end_chunk = text;
	while( end_chunk < end ) {
		start_chunk = end_chunk;
		pos = start_chunk - text;

		if( regexec( table->all, end_chunk, nmatch, pmatch, 0 ) ) {
			len_chunk = end - start_chunk;
			screem_editor_delete_forward( pos, len_chunk );
			gtk_text_insert( GTK_TEXT( editor ), NULL,
					 NULL, NULL, start_chunk, len_chunk );
			break;
		}

		end_chunk += pmatch[ 0 ].rm_so;
		pat_len = pmatch[ 0 ].rm_eo - pmatch[ 0 ].rm_so;
		len_chunk = end_chunk - start_chunk;

		if( ( end_chunk - text + pat_len ) < top ) {
			end_chunk += pat_len;
			continue;
		}
		
		/* we have a match for one of the expressions */
		str = g_strndup( end_chunk, pat_len );

		/* match str against the correct regexp */
		entries = table->entries;
		cg = NULL;
		for(entries = table->entries;entries;entries = entries->next) {
			entry = (ScreemHighlightEntry*)entries->data;
			if( (!regexec(entry->rstart, str, nmatch, pmatch, 0 ) )
			    && ( pmatch[ 0 ].rm_so == 0 ) ) {
				/* is this one in use? */
				cg = (ColourGroup*)entry->cg;
				if( cg->fore[ 0 ] || cg->back[ 0 ] )
					break;
			}
		}
		pat_len = pmatch[ 0 ].rm_eo - pmatch[ 0 ].rm_so;

		if( end_chunk > start_chunk ) {
			screem_editor_delete_forward( pos, len_chunk );
			gtk_text_insert( GTK_TEXT( editor ), NULL,
					 NULL, NULL, start_chunk, len_chunk );
		}

		pos = end_chunk - text;
		screem_editor_delete_forward( pos, pat_len );
		
		f = b = NULL;
		if( cg->fore[ 0 ] )
			f = &cg->fg;
		if( cg->back[ 0 ] )
			b = &cg->bg;
		gtk_text_insert( GTK_TEXT( editor ), cfg->mpage->font,
				 f, b, str, pat_len );
		g_free( str );
		end_chunk += pat_len;
	}
	gtk_adjustment_set_value( adj, value );
	gtk_text_thaw( GTK_TEXT( editor ) );
	gtk_adjustment_set_value( adj, value );
	screem_editor_set_pos( spos ); 

	g_free( text );

	timeout_handle = -1;

	hilighting = FALSE;
	return FALSE;
}

static gboolean tag_attributes_menu( gchar *name )
{

	/* passing NULL here as the first param causes the default DTD to
	   be used */
	attr_menu = screem_dtd_build_attribute_menu( NULL, name, FALSE );

	if( ! attr_menu )
		return FALSE;

	/* attach a callback for a key press, that way the user
	   can ignore the popup and keep typing */
	gtk_signal_connect( GTK_OBJECT( attr_menu ), "key_press_event",
			    GTK_SIGNAL_FUNC( no_attribute ), 0 );
	gnome_popup_menu_do_popup( attr_menu, 0, 0, 0, 0 );

	return TRUE;
}

static void no_attribute( GtkWidget *widget, GdkEventKey *event )
{
        gint pos;
        gboolean ret;

        if( ! isascii( event->keyval ) && event->keyval != GDK_Escape )
                return;

        gtk_signal_emit_stop_by_name( GTK_OBJECT( widget ),"key_press_event" );

        /* if attrMenu is NULL then we must have come from the
           editor popup menu, which doesn't want this functionality */
        if( ! attr_menu )
                return;

        /* insert initial press of space */
	pos = screem_editor_get_pos();
        screem_editor_insert( pos, " " );
	screem_editor_set_pos( ++ pos );
        /* a key was pressed while the attribute menu was open,
           so emit a key_press_event signal on the editor */

        attr_up = TRUE;
	if( event->keyval != ' ' )
		gtk_signal_emit_by_name( GTK_OBJECT( editor ),
					 "key_press_event", event, &ret );
        /* kill the menu */
        gtk_widget_destroy( attr_menu );
        attr_menu = NULL;
        attr_up = FALSE;
}

static gint xy_to_cursor_pos( gint x, gint y )
{
	gint i = 0;
	gchar c;
	gint lineWidth = 0;
        gint editorWidth;
        gint height;
        gint start = 0;
        gint charWidth;
        GdkFont *font = cfg->mpage->font;
        GtkText *e = GTK_TEXT( editor );
	const int tab_width = 8;
	const int line_wrap_room = 8; /* 8 == LINE_WRAP_ROOM in gtktext.c */

	gboolean wrapped = FALSE;
	gint previ;

	const gint tab_size = gdk_char_width( font, ' ' ) * tab_width;

        gdk_window_get_size( e->text_area, &editorWidth, NULL );
        editorWidth -= line_wrap_room; 

        height = font->ascent + font->descent;
        y += e->first_cut_pixels;

        for( i = e->first_line_start_index; i < screem_editor_get_length(); 
	     i ++ ) {
		c = GTK_TEXT_INDEX( e, i );
                if( c != '\n' ) {
                        charWidth = gdk_char_width( font, c );
                        if( c == '\t' )
				charWidth = tab_size;
			lineWidth += charWidth;
                }
                if( ( c == '\n' ) || ( lineWidth > editorWidth ) ) {
			lineWidth = 0;
                        if( ( y -= height ) <= 0 )
                                break;
			wrapped = ( c != '\n' );
			previ = i;
			while( ! isspace( GTK_TEXT_INDEX( e, i ) ) )
					i --;
			if( i == start && wrapped )
				i = previ;
			start = i;
		}
        }


	/* wasn't inside the text area */
        if( y <= 0 ) {
                for( ; x > 0; start ++ ) {
			c = GTK_TEXT_INDEX( e, start );
                        if( c == '\t' )
				charWidth = tab_size;
			else
				charWidth = gdk_char_width( font, c );
			x -= charWidth;
                }
                if( ( start = ( i - start ) ) > 0 )
			i -= start;
        }

        return i;
}


static GtkWidget *perl_menu()
{
	GtkWidget *menu_item;
	GtkWidget *menu;

	GtkWidget *menu_item2;

	menu_item = gtk_menu_item_new_with_label( _("Perl Menu") );
	gtk_widget_show( menu_item );

	menu = gtk_menu_new();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( menu_item ), menu );

	menu_item2 = gtk_menu_item_new_with_label( _("") );
	gtk_widget_show( menu_item2 );
	gtk_menu_append( GTK_MENU( menu ), menu_item2 );

	return menu_item;
}

static GtkWidget *java_menu()
{
	GtkWidget *menu_item;
	GtkWidget *menu;

	GtkWidget *menu_item2;

	menu_item = gtk_menu_item_new_with_label( _("Java Menu") );
	gtk_widget_show( menu_item );

	menu = gtk_menu_new();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( menu_item ), menu );

	menu_item2 = gtk_menu_item_new_with_label( _("Compile...") );
      	gtk_widget_show( menu_item2 );
	gtk_signal_connect( GTK_OBJECT( menu_item2 ), "activate",
			    GTK_SIGNAL_FUNC( java_compile ), 0 );
	gtk_menu_append( GTK_MENU( menu ), menu_item2 );

	return menu_item;
}

static void java_compile( GtkWidget *widget )
{
	gchar *command;
	const gchar *pathname;
	Page *page;

	page = screem_site_get_current_page( current_site );

	pathname = screem_page_get_pathname( page );

	command = g_strconcat( cfg->jpage->compiler, " ", pathname, 
			       " 2>&1", NULL );

	execute_command( command );

	g_free( command );
}

void screem_editor_build_highlight_tables()
{
	GList *list;
	GList *list2;
	GList *entries;

     	ColourClass *cc;
	ColourGroup *cg;

	ScreemEditorMode *mode;
	ScreemHighlightTable *table;
	ScreemHighlightEntry *entry;
	GList *types;

	/* destroy all current modes */
	for( list = highlight_modes; list; list = list->next ) {
		mode = (ScreemEditorMode*)list->data;

		table = mode->table;
		for( entries=table->entries; entries; entries=entries->next ) {
			entry = (ScreemHighlightEntry*)entries->data;
			g_free( entry->start );
			g_free( entry->name );
			g_free( entry );
		}
		g_free( table->all );
		g_free( table );
		g_list_free( mode->mime_types );
		g_free( mode );
	}

	/* build new modes */
	for( list = cfg->cpage->colour_classes; list; list = list->next ) {
		cc = (ColourClass*)list->data;
		entries = NULL;
		for( list2 = cc->groups; list2; list2 = list2->next ) {
			cg = (ColourGroup*)list2->data;
			entries = screem_highlight_entry_new( entries, cg );
		}
		table = screem_highlight_table_new( entries );
		types = cc->types;
		mode = screem_highlight_mode_new( table, types );
		highlight_modes = g_list_append( highlight_modes, mode );
	}
}

void screem_editor_changed()
{
	Page *page;

	changed = TRUE;
	page = screem_site_get_current_page( current_site );
	screem_page_set_changed( page, TRUE );
	screem_editor_set_charset( page );
}

void screem_editor_text_inserted( GtkEditable *editable, const gchar *text,
				  gint length, gint *pos )
{
	Page *page;
	gchar *ins;

	page = screem_site_get_current_page( current_site );

	/* should never occur as text can't be inserted if we don't
	   have a page open */
	if( ! page || hilighting )
		return;

	/* reset idle timer */
	if( timeout_handle != -1 )
		gtk_timeout_remove( timeout_handle );
	timeout_handle = gtk_timeout_add( DELAY, (GtkFunction)hi_trigger, 0 );

	screem_editor_changed();
	screem_editor_cursor_moved();

	if( ! undo ) {
		screem_page_clear_redo_list( page );
		ins = g_strndup( text, length );
		screem_page_set_undo_last_action( page, TRUE, *pos, ins );
		g_free( ins );
	}
}

void screem_editor_text_removed( GtkEditable *editable, gint start, gint end )
{
	Page *page;
	gchar *text;

	page = screem_site_get_current_page( current_site );

	if( ! page || hilighting )
		return;
	
	if( ! undo ) {
		text = gtk_editable_get_chars( editable, start, end );
		screem_page_clear_redo_list( page );
		screem_page_set_undo_last_action( page, FALSE, start, text );
		g_free( text );
	}

	/* reset idle timer */
	if( timeout_handle != -1 )
		gtk_timeout_remove( timeout_handle );
	timeout_handle = gtk_timeout_add( DELAY, (GtkFunction)hi_trigger, 0 );
	
	screem_editor_changed();
	screem_editor_cursor_moved();
}

static void screem_editor_update_parse_tree()
{
	gchar *t;
	Page *page;
	ScreemDTD *dtd;

	page = screem_site_get_current_page( current_site );

	if( ! page || hilighting )
		return;

	if( ! cfg->mpage->parse )
		return;

	t = screem_editor_get_text( 0, -1 );
	if( editor_parse_tree )
		screem_html_destroy_parse_tree( editor_parse_tree );
	dtd = screem_editor_get_doctype();
	
	editor_parse_tree = screem_html_build_parse_tree( dtd, t, 0 );
	
	g_free( t );
	return;
}

void screem_editor_undo()
{
	Page *page;
	Undo *u;

	page = screem_site_get_current_page( current_site );

	if( ! page )
		return;

	u = screem_page_get_undo_last_action( page, TRUE );
	if( ! u )
		return;

	undo = TRUE;
	if( u->insert )
		screem_editor_delete_forward( u->pos, strlen( u->text ) );
	else
		screem_editor_insert( u->pos, u->text );
     
	screem_page_set_redo_last_action( page, u );

	undo = FALSE;
}

void screem_editor_redo()
{
	Page *page;
	Undo *u;

	page = screem_site_get_current_page( current_site );

	if( ! page )
		return;

	u = screem_page_get_redo_last_action( page, TRUE );
	if( ! u )
		return;

	undo = TRUE;
	if( ! u->insert ) {
		screem_editor_delete_forward( u->pos, strlen( u->text ) );
		screem_editor_set_pos( u->pos );
	} else {
		screem_editor_insert( u->pos, u->text );
		screem_editor_set_pos( u->pos + strlen( u->text ) );
	}
	
	/* add back into the undo list */
	screem_page_add_undo_action( page, u );

	undo = FALSE;
}

void screem_editor_select_context()
{
	gchar *text;
	gint start;
	gint end;
	GNode *node;
	Node *n;

	Page *page;
	const gchar *mime_type;

	page = screem_site_get_current_page( current_site );

	if( ! page )
		return;

	if( ! cfg->mpage->parse )
		return;

	mime_type = screem_page_get_mime_type( page );

	/* we must be in an sgml/xml based document to do this */
	if( strcmp( "text/html", mime_type ) &&
	    strcmp( "application/x-php", mime_type ) &&
	    strcmp( "application/x-asp", mime_type ) )
		return;

	text = screem_editor_get_text( 0, -1 );
	start = screem_editor_get_pos();

	node = screem_html_parse_tree_find_pos( editor_parse_tree, start, FALSE );

	if( node ) {
		n = (Node*)node->data;
		
		start = n->pos;
		end = n->cpos;

		g_print( "start: %i\tend: %i\n", start, end );
		
		while( text[ start ] != '<' )
			start --;
		while( text[ ++end ] != '>' );
	} else {
		start = 0;
		end = screem_editor_get_length();
	}

	screem_editor_select_region( start, end + 1 );
	inserted = TRUE;

	g_free( text );
}

regex_t* screem_highlight_compile_regex( const gchar *pattern, gint flags )
{
	regex_t *regex;
	gint ret;

	regex = g_new0( regex_t, 1 );
	ret = regcomp( regex, pattern, flags );

	return regex;
}


GList *screem_highlight_entry_new( GList *list, ColourGroup *cg )
{
	ScreemHighlightEntry *entry;
	GList *list2;
   	GString *temp;

	entry = g_new0( ScreemHighlightEntry, 1 );

	entry->name = g_strdup( cg->name );

	temp = g_string_new( "" );
	for( list2 = cg->patterns; list2; list2 = list2->next ) {
		g_string_append( temp, (gchar*)list2->data );
		if( list2->next )
			g_string_append_c( temp, '|' );
	}

	cg->all_patterns = g_strdup( temp->str );
	g_string_free( temp, TRUE );

	entry->start = g_strdup( cg->all_patterns );
	if( cg->end )
		entry->end = g_strdup( cg->end );
      	entry->font = cfg->mpage->font;

	entry->rstart = screem_highlight_compile_regex( cg->all_patterns,
							REG_EXTENDED );
	if( cg->end )
		entry->rend = screem_highlight_compile_regex( cg->end,
							      REG_EXTENDED );

	entry->cg = cg;  /* so we know if this entry is in use */

	entry->sub_entries = NULL;

	return g_list_append( list, entry );
}

ScreemHighlightTable* screem_highlight_table_new( GList *entries )
{
	ScreemHighlightTable *table;
	ScreemHighlightEntry *entry;
	GString *regexps;
	GList *list;

	table = g_new( ScreemHighlightTable, 1 );

	table->entries = entries;
	
	regexps = g_string_new( "" );

	for( list = entries; list; list = list->next ) {
		entry = (ScreemHighlightEntry*)list->data;
		g_string_append( regexps, entry->start );

		if( list->next )
			g_string_append_c( regexps, '|' );
	}

	table->all = screem_highlight_compile_regex( regexps->str, 
						     REG_EXTENDED );

	g_string_free( regexps, TRUE );

	return table;
}

ScreemEditorMode* screem_highlight_mode_new( ScreemHighlightTable *table,
					     GList *types )
{
	ScreemEditorMode *mode;
	ScreemHighlightEntry *entry;

	mode = g_new( ScreemEditorMode, 1 );

	mode->table = table;
	mode->mime_types = types;

	entry = (ScreemHighlightEntry*)table->entries->data;

	mode->menus = entry->cg->parent->menus;
	mode->keys = entry->cg->parent->keys;
	mode->tips = entry->cg->parent->tips;

	return mode;
}

ScreemEditorMode* screem_editor_get_mode( const gchar *mime_type )
{
	ScreemEditorMode *mode;
	GList *list;
	GList *types;

	for( list = highlight_modes; list; list = list->next ) {
		mode = (ScreemEditorMode*)list->data;
		for( types = mode->mime_types; types; types = types->next ) {
			if( ! strcmp( mime_type, (gchar*)types->data ) )
				return mode;
		}
	}

	return NULL;
}

void screem_editor_indent( gchar *text, gint pos )
{
	gint depth;
	GNode *node;
	Node *n;
	gint len;
	gchar *tag;

	while( pos > 0 && text[ --pos ] != '\n' );
	pos ++;

	if( ! cfg->mpage->parse )
		return;

	node = screem_html_parse_tree_find_pos( editor_parse_tree, pos, FALSE );

	if( ! node )
		depth = 0;
	else {
		n = (Node*)node->data;
		depth = g_node_depth( node );
		depth --;
	}
	
	len = 0;
	while( isspace( text[ pos ] ) && text[ pos ] != '\n' ) {
		pos ++;
		len ++;
	}
	
	/* erase the white space */
	screem_editor_delete_forward( pos - len, len );

	/* if the line is empty, don't do anything else */
	if( text[ pos ] == '\n' )
		return;
	
	if( text[ pos ] == '<' && text[ pos + 1 ] == '/' )
		depth --;
	
	/* depth found, indent line depth number of tabs */
	if( depth > 0 ) {
		tag = g_strnfill( depth, '\t' );
		screem_editor_insert( pos - len, tag );
		g_free( tag );
	}
}

void screem_editor_goto_line()
{
	Page *page;
	page = screem_site_get_current_page( current_site );

	if( ! page )
		return;

	gnome_app_request_string( GNOME_APP( app ),
				  "Goto Line: ",
				  (GnomeStringCallback)goto_line,
				  NULL );
}

static void goto_line( gchar *string )
{
	gint line;
	gint i;
	gchar *text;
	
	line = atoi( string );
	text = screem_editor_get_text( 0, -1 );
	i = 0;
	line --;
	while( i < strlen( text ) && line ) {
		line -= ( text[ i++ ] == '\n' );
	}
	screem_editor_set_pos( i );
	gtk_widget_grab_focus( editor );
}

/**
 * screem_editor_encode_text:
 *
 * Encodes the selected or all the text in the editor
 *
 * returns: none
 */
void screem_editor_encode_text()
{
	gchar *text;
	gchar *ret;
	gint start = 0;
	gint len = -1;
	Page *page;

	page = screem_site_get_current_page( current_site );

	if( ! page )
		return;

	if( screem_editor_has_selection( &start, &len ) )
		len = len - start;

	text = screem_editor_get_text( start, len );

	ret = screem_html_encode_text( text );

	screem_editor_delete_forward( start, len );
	screem_editor_insert( start, ret );

	g_free( ret );
	g_free( text );
}

gchar* screem_editor_get_word( gint pos, gchar *text )
{
	gint start;
	gint end;
	gchar *word;

	for( start = pos; start; start -- ) {
		if( ( ! isalnum( text[ start ] ) ) && text[ start ] != '_' )
			break;
	}
	for( end = pos; text[ end ]; end ++ ) {
		if( ( ! isalnum( text[ end ] ) ) && text[ end ] != '_' )
			break;
	}

	if( start != pos )
		start ++;

	if( end != pos )
		end --;

    	if( start == end )
		word = NULL;
	else
		word = g_strndup( &text[ start ], end - start + 1 );

	return word;
}

void screem_editor_set_charset( Page *page )
{
	GNode *node;
	Node *data;
	GList *list;
	gchar *temp;
	GString *charset;

	if( ! page )
		return;

	for( node = editor_parse_tree, charset = NULL; node; ) {
		if(!(node = screem_html_parse_tree_find_node( node, "meta" ) ))
			break;
		data = (Node*)node->data;
		node = node->next; /* next as meta's don't have children */
		for( list = data->attributes; list; list = list->next ) {
			if( g_strcasecmp( "Content-Type", 
					  (gchar*)list->data ) ) {
				list = list->next;
				continue;
			}
			/* list->data possibly holds charset */
			temp = find_text( (const gchar*)list->data, "charset=",
					  NULL, NULL );
			if( temp ) {
				/* found it */
				temp += strlen( "charset=" );
				/* temp now points at the
				   character set */
				charset = g_string_new( "" );
				while( *temp != '\0' && *temp !=  ' ' ) {
					g_string_append_c( charset, *temp );
					temp ++;
				}
				break;
			}
			list = list->next;
		}
	}
	if( charset ) {
		/* we found it */
		screem_page_set_charset( page, charset->str );
		g_string_free( charset, TRUE );
	} else {
		screem_page_set_charset( page, cfg->mpage->default_charset );
	}
}

ScreemDTD * screem_get_doctype( const gchar *text )
{
	ScreemDTD *dtd;
	gint len;
	gchar *doctype;
	GString *tag;

	doctype = find_text( text, "<!DOCTYPE", NULL, &len );
	if( ! doctype )
		doctype = find_text( text, "<!doctype", NULL, &len );
	if( ! doctype )
		return NULL;
	tag = g_string_new( "" );
	while( *doctype != '\0' && *doctype != '>' ) {
		g_string_append_c( tag, *doctype );
		doctype ++;
	}

	dtd = screem_dtd_load_doctype( tag->str );
	if( ! dtd ) {
		/* use default */
		dtd = screem_dtd_load_doctype( cfg->mpage->default_dtd );
	}
	g_string_free( tag, TRUE );

	return dtd;
}

static ScreemDTD* screem_editor_get_doctype()
{
	gchar *text;
	ScreemDTD *dtd;

	/* can't use parse tree as it may not exist, and relies upon
	   knowing the doctype to be correct as well */

	text = screem_editor_get_text( 0, -1 );

	dtd = screem_get_doctype( text );

	g_free( text );
	return dtd;
}
