/*  Screem:  screem-tag_inspector.c
 *
 *  Copyright (C) 2004  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <glib/gi18n.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <gtk/gtk.h>

#include <string.h>

#include <libcroco/libcroco.h>

#include "fileops.h"
#include "pageUI.h"
#include "support.h"
#include "screem-tag-inspector.h"

#include "screem-markup.h"

#include "screemmarshal.h"

#ifdef DASHBOARD_BUILD
#include "dashboard-frontend.c"
#endif

enum {
	PROP_0,
	PROP_WINDOW
};

enum {
	REPLACE_ATTRIBUTE_SIGNAL,
	LAST_SIGNAL
};

enum {
	ATTR_MODEL_NAME,
	ATTR_MODEL_DISPLAY_NAME,
	ATTR_MODEL_VALUES,
	ATTR_MODEL_VALUE,
	ATTR_MODEL_ROWS
};

enum {
	STYLE_MODEL_NAME,
	STYLE_MODEL_FILE,
	STYLE_MODEL_ROWS
};

static guint screem_tag_inspector_signals[ LAST_SIGNAL ] = { 0 };

static void dashboard_send_cluepacket( ScreemTagInspector *inspector );
static void build_attribute_entries( ScreemTagInspector *inspector, 
			ScreemDTD *dtd, const gchar *name,
			GSList *attr );
static void attr_edited( GtkCellRendererText *rend,
		const gchar *path_str, const gchar *txt,
		ScreemTagInspector *inspector );

static void combo_change( GtkComboBox *combo, gpointer data );
static void switch_page( GtkNotebook *notebook,
			GtkNotebookPage *page,
			guint page_num,
			gpointer data );


static void screem_tag_inspector_class_init( ScreemTagInspectorClass *klass );
static void screem_tag_inspector_init( ScreemTagInspector *tag_inspector );
static void screem_tag_inspector_finalize( GObject *tag_inspector );
static void screem_tag_inspector_set_prop( GObject *object, 
				guint property_id,
				const GValue *value, 
				GParamSpec *pspec );
static void screem_tag_inspector_get_prop( GObject *object, 
				guint property_id,
				GValue *value, GParamSpec *pspec );
static void screem_tag_inspector_size_request( GtkWidget *widget,
                           	GtkRequisition *requisition );
static void screem_tag_inspector_size_allocate( GtkWidget *widget,
                              	GtkAllocation *allocation );

struct ScreemTagInspectorPrivate {
	ScreemWindow *window;
	
	GtkWidget *label;
	GtkWidget *book;
	
	GtkWidget *attrs;
	GtkWidget *styles;
	GtkWidget *desc;

	GtkWidget *desc_label;

	ScreemPage *page;
	guint pos;

	gchar *tag;
	gchar *name;

	GtkTreeModel *attr_model;
	GtkTreeModel *style_model;
};

ScreemTagInspector *screem_tag_inspector_new( ScreemWindow *window )
{
	ScreemTagInspector *i;
	
	i = SCREEM_TAG_INSPECTOR( g_object_new( SCREEM_TYPE_TAG_INSPECTOR, 
				"window", window, NULL ) );

	return i;
}

gboolean screem_tag_inspector_inspect( ScreemTagInspector *inspector,
			ScreemPage *page, guint pos, guint ppos )
{
	ScreemTagInspectorPrivate *priv;
	GSList *set_attr;
	gchar *name;

	gchar *text;
	gchar *tag;
	ScreemDTD *dtd;
	gboolean istag;

	GtkTextIter it;
	guint line;
	guint col;
	gchar *tmp;

	g_return_val_if_fail( SCREEM_IS_TAG_INSPECTOR( inspector ), 
				FALSE );
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	priv = inspector->priv;
	priv->page = page;
	priv->pos = pos;

	if( pos == ppos ) {
		return FALSE;
	}

	text = NULL;
	istag = FALSE;
		
	if( page && screem_page_is_markup( page ) ) {
		/* id which tag we are in */
		text = screem_page_get_data( page );
		istag = screem_markup_is_tag( text, pos, &pos, NULL );
	}
	
	/* clear old attribute contents */
	gtk_list_store_clear( GTK_LIST_STORE( priv->attr_model ) );
		
	if( istag ) {
		/* get current tag */
		tag = screem_markup_next_tag( text, pos, NULL, 
						&pos, &name );
		if( tag && name ) {
			priv->tag = tag;
			priv->name = name;
			dashboard_send_cluepacket( inspector );
			set_attr = NULL;
			if( ( *name != '!' && *name != '?' ) || 
				! strncmp( "?xml", name,
					strlen( "?xml" ) ) ) {
				set_attr = screem_markup_build_attributes_list( tag, NULL );
				set_attr = g_slist_reverse( set_attr );
			}
		} else {
			name = NULL;
			set_attr = NULL;
		}
		
		g_free( tag );
		
		/* if we have a tag build the inspector pages */
		if( name ) {
			gtk_text_buffer_get_iter_at_offset( GTK_TEXT_BUFFER( page ), &it, pos );
			line = gtk_text_iter_get_line( &it );
			col = gtk_text_iter_get_line_offset( &it );
			tmp = g_strdup_printf( _("  Ln %d, Col. %d"), line + 1, col + 1 );
			
			tag = g_strdup_printf( "<span size=\"xx-large\">&lt;%s&gt;</span>%s",
									name, tmp );
			g_free( tmp );
			gtk_label_set_markup( GTK_LABEL( priv->label ), tag );
			gtk_label_set_use_markup( GTK_LABEL( priv->label ), TRUE );
			g_free( tag );
	
			dtd = screem_page_get_dtd( page );

			/* attribute page */
			build_attribute_entries( inspector, dtd, name,
						set_attr );
			
			/* destroy all of set_attr */
			g_slist_foreach( set_attr, (GFunc)g_free, NULL );
			g_slist_free( set_attr );

			g_free( name );
		}
	} else {
		/* not in a tag at all */
		gtk_label_set_markup( GTK_LABEL( priv->label ), "" );
	}
	g_free( text );

	gtk_widget_set_sensitive( GTK_WIDGET( inspector ), istag );

	if( gtk_notebook_get_current_page( GTK_NOTEBOOK( priv->book ) ) == 1 ) {
		switch_page( GTK_NOTEBOOK( priv->book ), NULL,
				1, inspector );
	}
	
	return istag;
}

/* static stuff */
static void dashboard_send_cluepacket( ScreemTagInspector *inspector )
{
#ifdef DASHBOARD_BUILD
	ScreemTagInspectorPrivate *priv;
	gchar *cluepacket;
	gchar *context;
	ScreemPage *page;
	ScreemDTD *dtd;
	const gchar *publicid;
	
	priv = inspector->priv;
	page = priv->page;
	dtd = screem_page_get_dtd( page );
	publicid = screem_dtd_get_public_id( dtd );
	
	context = g_strdup_printf( "&lt;%s^gt;", priv->name );
	
	cluepacket = dashboard_build_cluepacket_then_free_clues(
			"Screem", TRUE, context, 
			dashboard_build_clue( priv->tag,
				"htmlblock", 8 ),
			dashboard_build_clue( priv->name,
				"keyword", 8 ),
			dashboard_build_clue( "HTML",
				"keyword", 10 ),
			dashboard_build_clue( "tag",
				"keyword", 10 ),
			dashboard_build_clue( "element",
				"keyword", 10 ),
			dashboard_build_clue( "structure",
				"keyword", 10 ),
			dashboard_build_clue( publicid,
				"keyword", 9 ),
			NULL );

	dashboard_send_raw_cluepacket( cluepacket );

	g_free( cluepacket );
	g_free( context );
#endif
}

static void build_attribute_entries( ScreemTagInspector *inspector, 
				ScreemDTD *dtd, const gchar *name,
				GSList *attr )
{
	ScreemTagInspectorPrivate *priv;
	GtkTreeIter it;
	GSList *list;

	const ScreemDTDElement *element;
	const GSList *attrs;
	GtkWidget *label;
	const gchar *description;
	GCompareFunc compare;
	gint i;
	gchar *tmp;

	if( ! dtd ) {
		return;
	}
	
	element = screem_dtd_valid_element( dtd, name );
	if( ! element ) {
		return;
	}

	priv = inspector->priv;
	
	g_object_get( G_OBJECT( dtd ), "compare", &compare, NULL );

	label = inspector->priv->desc_label;
	description = screem_dtd_element_get_description( element );
	if( ! description ) {
		description = "";
	}
	if( label ) {
		tmp = screem_strip_extra_space( description );
		if( tmp ) {
			gtk_label_set_text( GTK_LABEL( label ), tmp );
			g_free( tmp );
		} else {
			g_print( "failed to strip space from desc\n" );
		}
	}
	
	attrs = screem_dtd_element_get_attrs( element );
	
	for( i = 0; attrs; attrs = attrs->next, ++ i ) {
		ScreemDTDAttribute *att;
		const gchar *attr_name;
		const GSList *values;
		gchar *tmp;
		GtkTooltips *tip;
		GtkListStore *store;
		gint j;
		
		att = (ScreemDTDAttribute*)attrs->data;
		attr_name = screem_dtd_attribute_get_name( att );
		description = screem_dtd_attribute_get_description( att );
		
		values = screem_dtd_attribute_get_values( att );

		/* add entrys, FIXME: escape attr_name */
		if( screem_dtd_attribute_get_required( att ) ) {
			tmp = g_strconcat( "<b>", attr_name, "</b>", 
					NULL );
		} else {
			tmp = g_strdup( attr_name );
		}
		store = gtk_list_store_new( 1, G_TYPE_STRING );
		for( j = 0; values; values = values->next, ++ j ) {
			gtk_list_store_insert_with_values( store, &it, j,
					0, values->data, -1 );
		}
		gtk_list_store_insert_with_values( GTK_LIST_STORE( priv->attr_model ),
				&it, i,
				ATTR_MODEL_NAME, attr_name,
				ATTR_MODEL_DISPLAY_NAME, tmp,
				ATTR_MODEL_VALUES, store,
				-1 );
		g_free( tmp );
		g_object_unref( store );
		
		if( description ) {
			tip = gtk_tooltips_new();
		}
		
		/* 
		 * set the entry value to whatever value the
		 * attribute currently has in the tag, or make it
		 * empty if the attribute isn't used
		 */

		/* find name in the attr list */
		for( list = attr; list; list = list->next ) {
			if( ! compare( attr_name, (gchar*)list->data ) )
				break;
			list = list->next;
		}
		if( ! list ) {
			tmp = "";
		} else if( ! list->next ) {
			tmp = (gchar*)attr_name;
		} else {
			tmp = list->next->data;
		}
		gtk_list_store_set( GTK_LIST_STORE( priv->attr_model ),
				&it,
				ATTR_MODEL_VALUE, tmp,
				-1 );
	}
}

static void attr_edited( GtkCellRendererText *rend,
		const gchar *path_str, const gchar *txt,
		ScreemTagInspector *inspector )
{
	ScreemTagInspectorPrivate *priv;
	GtkTreePath *path;
	GtkTreeIter it;
	gchar *attr_name;
	gchar *attr;
	
	priv = inspector->priv;

	path = gtk_tree_path_new_from_string( path_str );
	gtk_tree_model_get_iter( GTK_TREE_MODEL( priv->attr_model ),
			&it, path );
	gtk_tree_model_get( GTK_TREE_MODEL( priv->attr_model ),
			&it, ATTR_MODEL_NAME, &attr_name, -1 );

	/* let this get updated normally rather than setting here,
	 * as the insertion will cause the tag inspector to rescan
	 * the whole tag
	 
	 gtk_list_store_set( GTK_LIST_STORE( priv->attr_model ),
			&it,
			ATTR_MODEL_VALUE, txt,
			-1 );
	*/	
	gtk_tree_path_free( path );

	attr = g_strconcat( attr_name, "=\"", txt, "\"", NULL );

	g_signal_emit( G_OBJECT( inspector ),
		screem_tag_inspector_signals[ REPLACE_ATTRIBUTE_SIGNAL],
		0, attr );

	g_free( attr );
	g_free( attr_name );
}

static void ruleset_to_ui( CRStatement *s, GtkTreeModel *model,
		const gchar *file )
{
	GtkTreeIter it;
	guchar *txt;

	txt = cr_selector_to_string( s->kind.ruleset->sel_list );
	
	/* use large pos to ensure append */
	gtk_list_store_insert_with_values( GTK_LIST_STORE( model ), &it,
			G_MAXINT32,
			STYLE_MODEL_NAME, (gchar*)txt,
			STYLE_MODEL_FILE, file,
			-1 );
	g_free( txt );
}

static void statement_to_ui( CRStatement *s, GtkTreeModel *model,
		const gchar *file )
{
	switch( s->type ) {
		case RULESET_STMT:
			ruleset_to_ui( s, model, file );	
			break;
		case AT_FONT_FACE_RULE_STMT:
			break;
		case AT_CHARSET_RULE_STMT:
			break;
		case AT_MEDIA_RULE_STMT:
			break;
		case AT_IMPORT_RULE_STMT:
			break;
		default:
			break;
	}
}

static void calc_styles( ScreemTagInspector *inspector,
		xmlNodePtr node, ScreemPage *page,
		const gchar *sheet, gboolean isdata )
{
	ScreemTagInspectorPrivate *priv;
	const gchar *pathname;
	gchar *url;
	gchar *temp;
	
	enum CRStatus status;
	CROMParser *parser;
	CRStyleSheet *stylesheet;
	CRSelEng *sel_eng;
	
	CRStatement **statements;
	gulong num_statements;
	gulong i;
	
	ScreemSite *site;
	ScreemPage *cpage;
	gchar *data;
	
	priv = inspector->priv;

	url = NULL;
	cpage = NULL;
	if( ! isdata ) {
		pathname = screem_page_get_pathname( page );
		temp = g_path_get_dirname( pathname );
		url = relative_to_full( sheet, temp );
		g_free( temp );
		site = screem_window_get_current( priv->window );
		cpage = screem_site_locate_page( site, url );
	}

	parser = cr_om_parser_new( NULL );
	stylesheet = NULL;
	
	if( isdata ) {
		/* use strlen, we want byte length, not char */
		status = cr_om_parser_parse_buf( parser, 
				(const guchar*)sheet,
				strlen( sheet ),
				CR_UTF_8, &stylesheet );
		sheet = "";
	} else if( ! cpage ) {
		data = load_file( url, NULL, NULL, NULL );

		status = cr_om_parser_parse_buf( parser, 
				(const guchar*)data,
				strlen( data ),
				CR_UTF_8, &stylesheet );
		
		g_free( data );
	} else if( screem_page_load( cpage, NULL ) ){
		data = screem_page_get_data( cpage );
		
		/* use strlen, we want byte length, not char */
		status = cr_om_parser_parse_buf( parser, 
				(const guchar*)data,
				strlen( data ),
				CR_UTF_8, &stylesheet );
		
		g_free( data );
	} else {
		status = CR_ERROR;
		/* FIXME: report error loading stylesheet */

	}
	
	sel_eng = NULL;
	num_statements = 0;
	if( status == CR_OK ) {
		sel_eng = cr_sel_eng_new();
		statements = NULL;

		status = cr_sel_eng_get_matched_rulesets( sel_eng, stylesheet,
				node, &statements, &num_statements );
		if( status == CR_OK ) {
			for( i = 0; i < num_statements; ++ i ) {
				CRStatement *s;
				
				s = statements[ i ];

				statement_to_ui( s, priv->style_model,
						sheet );
			}
		}
		if( statements ) {
			g_free( statements );
		}
	}
	if( parser ) {
		cr_om_parser_destroy( parser );
	}
	if( stylesheet ) {
		cr_stylesheet_destroy( stylesheet );
	}
	if( sel_eng ) {
		cr_sel_eng_destroy( sel_eng );
	}
	
	g_free( url );
}

static void style_entries_build( ScreemPage *page, 
		ScreemTagInspector *inspector )
{
	ScreemTagInspectorPrivate *priv;
	xmlDocPtr doc;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter it;
	gboolean gotit;
	xmlNodePtr node;
	gchar *rel;
	gchar *sheet;
	gchar *temp;
	ScreemDTD *dtd;
	GCompareFunc compare;

	xmlNodePtr enode;	
	priv = inspector->priv;

	g_signal_handlers_disconnect_matched(G_OBJECT( page ),
				     G_SIGNAL_MATCH_DATA,
				     0, 0, NULL, 
				     NULL,
				     inspector );
	dtd = screem_page_get_dtd( page );
	if( dtd ) {
		g_object_get( G_OBJECT( dtd ), 
				"compare", &compare, NULL );
	} else {
		g_warning( "ScreemPage without ScreemDTD\n" );
		compare = (GCompareFunc)strcmp;
	}
		
	doc = NULL;
	if( priv->page == page ) {

		doc = g_object_get_data( G_OBJECT( page ),
				"xmlDoc" );
	}
	sheet = NULL;
	gotit = FALSE;
	model = NULL; /* shutup gcc */
	if( doc ) {
		temp = screem_page_query_context( page, priv->pos, 
				FALSE, FALSE, NULL, NULL, NULL, 
				&enode );
		g_free( temp );
		
		model = screem_page_get_model( page );
		path = gtk_tree_path_new_first();
		gtk_tree_path_next( path );
		gtk_tree_path_next( path );
		gtk_tree_path_down( path );
		gotit = gtk_tree_model_get_iter( model, &it, path );
		gtk_tree_path_free( path );
	}
	/* check all <link> */
	if( gotit ) {
		gotit = FALSE;
		do {
			gtk_tree_model_get( model, &it,
					SCREEM_PAGE_MODEL_NODE,
					&node, -1 );
			if( node && ! compare( "link", node->name ) ) {
				gotit = TRUE;
				rel = (gchar*)xmlGetProp( node, (const xmlChar*)"rel" );
				if( ! rel ) {
					rel = (gchar*)xmlGetProp( node,
							(const xmlChar*)"REL" );
				}
				if( rel &&
				 ( ! g_strcasecmp( "stylesheet", rel ) ||
				   ! g_strcasecmp( "alternate stylesheet",
					   	   rel ) ) ) {
					sheet = (gchar*)xmlGetProp( node, 
							(const xmlChar*)"href" );
					if( ! sheet ) {
						sheet = (gchar*)xmlGetProp( node, (const xmlChar*)"HREF" );
					}
					if( sheet && enode ) {
						calc_styles( inspector,
							enode, page, 
							sheet, FALSE );
					}
					g_free( sheet );
				}
				g_free( rel );
			} else if( gotit ) {
				break;
			}
		} while( gtk_tree_model_iter_next( model, &it ) );
	}
	/* check all <style> */
	gotit = FALSE;
	if( doc ) {
		model = screem_page_get_model( page );
		path = gtk_tree_path_new_first();
		gtk_tree_path_next( path );
		gtk_tree_path_next( path );
		gtk_tree_path_next( path );
		gtk_tree_path_down( path );
		gotit = gtk_tree_model_get_iter( model, &it, path );
		gtk_tree_path_free( path );
	}
	if( gotit ) {
		do {
			gtk_tree_model_get( model, &it,
					SCREEM_PAGE_MODEL_NODE,
					&node, -1 );
			if( node && node->children ) {
				sheet = (gchar*)xmlNodeGetContent( node->children );
				if( sheet ) {
					calc_styles( inspector,
						enode, page,
						sheet, TRUE );
					g_free( sheet );
				}
			}

		} while( gtk_tree_model_iter_next( model, &it ) );
	}
}

static void build_style_entries( ScreemTagInspector *inspector )
{
	ScreemTagInspectorPrivate *priv;
	ScreemPage *page;
	
	priv = inspector->priv;
	page = priv->page;

	gtk_list_store_clear( GTK_LIST_STORE( priv->style_model ) );
	
	if( screem_page_is_markup( page ) ) {
		g_signal_connect( G_OBJECT( page ), "built",
				G_CALLBACK( style_entries_build ),
				inspector );
		gdk_threads_leave();
		screem_page_build_model( page );
		gdk_threads_enter();
		screem_page_ensure_model_built( page );
	}
}

static void combo_change( GtkComboBox *combo, gpointer data )
{
	ScreemTagInspector *inspector;
	ScreemTagInspectorPrivate *priv;
	gint item;
	
	inspector = SCREEM_TAG_INSPECTOR( data );
	priv = inspector->priv;
	
	item = gtk_combo_box_get_active( combo );
	gtk_notebook_set_current_page( GTK_NOTEBOOK( priv->book ), item );
}

static void switch_page( GtkNotebook *notebook,
			GtkNotebookPage *page,
			guint page_num,
			gpointer data )
{
	ScreemTagInspector *inspector;
	ScreemTagInspectorPrivate *priv;
	GtkWidget *child;
	
	inspector = SCREEM_TAG_INSPECTOR( data );
	priv = inspector->priv;
	
	child = gtk_notebook_get_nth_page( notebook, page_num );
	
	/* have we switched to the styles page? */
	if( child == priv->styles ) {
		build_style_entries( inspector );
	} 

}

static void style_row_activated( GtkTreeView *view, 
			    GtkTreePath *path,
			    GtkTreeViewColumn *column,
			    gpointer data )
{
	ScreemTagInspector *inspector;
	ScreemTagInspectorPrivate *priv;
	GtkTreeModel *model;
	GtkTreeIter it;
	const gchar *pathname;
	gchar *name;
	gchar *file;
	gchar *url;
	gchar *tmp;
	ScreemSite *site;

	inspector = SCREEM_TAG_INSPECTOR( data );
	priv = inspector->priv;

	model = gtk_tree_view_get_model( view );
	gtk_tree_model_get_iter( GTK_TREE_MODEL( model ), &it, path );

	gtk_tree_model_get( model, &it, 
			STYLE_MODEL_NAME, &name,
			STYLE_MODEL_FILE, &file, -1 );

	if( file && *file ) {
		pathname = screem_page_get_pathname( priv->page );
		tmp = g_path_get_dirname( pathname );
		url = relative_to_full( file, tmp );
		g_free( tmp );
	
		/* open url */
		site = screem_window_get_current( priv->window );
		screem_page_open_with_filename( site, priv->window, url );

		g_free( url );
	}
	g_free( name );
	g_free( file );
}


/* G Object stuff */

G_DEFINE_TYPE( ScreemTagInspector, screem_tag_inspector, GTK_TYPE_BIN )

static void screem_tag_inspector_class_init( ScreemTagInspectorClass *klass )
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;
	GParamSpec *pspec;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;

	object_class->finalize = screem_tag_inspector_finalize;
	object_class->get_property = screem_tag_inspector_get_prop;
	object_class->set_property = screem_tag_inspector_set_prop;

	widget_class->size_request = screem_tag_inspector_size_request;
	widget_class->size_allocate = screem_tag_inspector_size_allocate;
	pspec = g_param_spec_pointer( "window", "window",
				      "window",
				      G_PARAM_READABLE |
				      G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_WINDOW,
					 pspec );

	screem_tag_inspector_signals[ REPLACE_ATTRIBUTE_SIGNAL ] = 
		g_signal_new( "replace_attribute",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemTagInspectorClass, 
					       replace_attribute ),
			      NULL, NULL,
			      screem_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING );		
}

static void screem_tag_inspector_init( ScreemTagInspector *tag_inspector )
{
	ScreemTagInspectorPrivate *priv;
	GtkWidget *sw;
	GtkWidget *box;
	
	GtkWidget *taglabel;
	GtkWidget *combo;
	GtkWidget *notebook;

	GtkWidget *label;
	GtkWidget *child;

	GtkWidget *view;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;
	
	tag_inspector->priv = priv = g_new0( ScreemTagInspectorPrivate, 1 );
	
	box = gtk_vbox_new( 0, FALSE );
	gtk_container_set_border_width( GTK_CONTAINER( box ), 6 );
	gtk_box_set_spacing( GTK_BOX( box ), 6 );
	
	/* top label */
	taglabel = gtk_label_new( "" );
	gtk_misc_set_alignment( GTK_MISC( taglabel ), 0.0, 0.0 );
	gtk_box_pack_start( GTK_BOX( box ), taglabel, FALSE, FALSE, 0 );
	
	/* notebook combo */
	combo = gtk_combo_box_new_text();
	g_signal_connect( G_OBJECT( combo ), "changed",
					G_CALLBACK( combo_change ),
					tag_inspector );
	gtk_box_pack_start( GTK_BOX( box ), combo, FALSE, FALSE, 0 );
	
	/* inspector notebook */
	notebook = gtk_notebook_new();
	gtk_notebook_set_show_border( GTK_NOTEBOOK( notebook ), FALSE );
	gtk_notebook_set_show_tabs( GTK_NOTEBOOK( notebook ), FALSE );
	g_signal_connect( G_OBJECT( notebook ), "switch_page",
			G_CALLBACK( switch_page ), tag_inspector );
	gtk_box_pack_start_defaults( GTK_BOX( box ), notebook );
		
	/* attributes page */
	label = gtk_label_new( _( "Attributes" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( combo ), _("Attributes" ) );
	priv->attrs = child = gtk_vbox_new( FALSE, 0 );
	gtk_box_set_spacing( GTK_BOX( child ), 6 );

	priv->attr_model = GTK_TREE_MODEL( gtk_list_store_new( ATTR_MODEL_ROWS,
			G_TYPE_STRING,
			G_TYPE_STRING,
			GTK_TYPE_LIST_STORE,
			G_TYPE_STRING ) );
	
	view = gtk_tree_view_new();
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
	   				GTK_POLICY_NEVER, 
	       				GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
					GTK_SHADOW_IN );
	gtk_container_add( GTK_CONTAINER( sw ), view );
	gtk_box_pack_start( GTK_BOX( child ), sw, TRUE, TRUE, 0 );

	rend = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, _( "Name" ) );
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_attributes( col, rend,
					     "markup", 
					     ATTR_MODEL_DISPLAY_NAME,
					     NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( view ), col );

	rend = gtk_cell_renderer_combo_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, _( "Value" ) );
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	g_object_set( G_OBJECT( rend ), 
			"text_column", 0, 
			"editable", TRUE, 
			"ellipsize", PANGO_ELLIPSIZE_END,
			"mode", GTK_CELL_RENDERER_MODE_EDITABLE,
			NULL );
	gtk_tree_view_column_set_attributes( col, rend,
			"text", ATTR_MODEL_VALUE,
			"model", ATTR_MODEL_VALUES,
			NULL );
	g_signal_connect( G_OBJECT( rend ), "edited",
			G_CALLBACK( attr_edited ), tag_inspector );

	gtk_tree_view_append_column( GTK_TREE_VIEW( view ), col );
	
	gtk_tree_view_set_model( GTK_TREE_VIEW( view ),
			priv->attr_model );
	gtk_notebook_append_page( GTK_NOTEBOOK( notebook ),
				child, label );

	/* styles page */
	label = gtk_label_new( _( "Applied Styles" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( combo ), _( "Applied Styles" ) );
	priv->styles = child = gtk_vbox_new( FALSE, 0 );
	gtk_box_set_spacing( GTK_BOX( child ), 6 );

	priv->style_model = GTK_TREE_MODEL( gtk_list_store_new( STYLE_MODEL_ROWS,
			G_TYPE_STRING, G_TYPE_STRING ) );
	view = gtk_tree_view_new();
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
	   				GTK_POLICY_AUTOMATIC, 
	       				GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
					GTK_SHADOW_IN );
	gtk_container_add( GTK_CONTAINER( sw ), view );
	gtk_box_pack_start( GTK_BOX( child ), sw, TRUE, TRUE, 0 );

	rend = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, _( "Rules" ) );
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 
					     STYLE_MODEL_NAME,
					     NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( view ), col );
	
	rend = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, _( "File" ) );
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 
					     STYLE_MODEL_FILE,
					     NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( view ), col );

	gtk_tree_view_set_model( GTK_TREE_VIEW( view ),
			priv->style_model );
	g_signal_connect( G_OBJECT( view ), "row_activated",
			G_CALLBACK( style_row_activated ),
			tag_inspector );
	
	gtk_notebook_append_page( GTK_NOTEBOOK( notebook ),
				child, label );
	
	/* description */
	label = gtk_label_new( _( "Description" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( combo ), _( "Description" ) );
	priv->desc = child = gtk_vbox_new( FALSE, 0 );
	gtk_container_set_border_width( GTK_CONTAINER( child ), 6 );
	gtk_box_set_spacing( GTK_BOX( child ), 6 );
	gtk_notebook_append_page( GTK_NOTEBOOK( notebook ),
				child, label );
	priv->desc_label = label = gtk_label_new( "" );
	gtk_label_set_line_wrap( GTK_LABEL( label ), TRUE );
	gtk_misc_set_alignment( GTK_MISC( label ), 0.0, 0.0 );
	gtk_box_pack_start( GTK_BOX( child ), label, TRUE, TRUE, 0 );

	gtk_container_add( GTK_CONTAINER( tag_inspector ), box );
	gtk_widget_show_all( box );
	
	priv->label = taglabel;	
	priv->book = notebook;
	
	gtk_combo_box_set_active( GTK_COMBO_BOX( combo ), 0 );
}

static void screem_tag_inspector_finalize( GObject *screem_tag_inspector )
{
	ScreemTagInspector *tag_inspector;

	tag_inspector = SCREEM_TAG_INSPECTOR( screem_tag_inspector );

	g_free( tag_inspector->priv );

	G_OBJECT_CLASS( screem_tag_inspector_parent_class )->finalize( screem_tag_inspector );
}

static void screem_tag_inspector_set_prop( GObject *object, 
		guint property_id,
		const GValue *value, GParamSpec *pspec )
{
	ScreemTagInspector *inspector;
	ScreemTagInspectorPrivate *priv;
	
	inspector = SCREEM_TAG_INSPECTOR( object );
	priv = inspector->priv;
	
	switch( property_id ) {
		case PROP_WINDOW:
			priv->window = SCREEM_WINDOW( g_value_get_pointer( value ) );
			break;
		default:
			break;
	}
}

static void screem_tag_inspector_get_prop( GObject *object, 
		guint property_id,
		GValue *value, GParamSpec *pspec)
{
	ScreemTagInspector *inspector;
	ScreemTagInspectorPrivate *priv;
	
	inspector = SCREEM_TAG_INSPECTOR( object );
	priv = inspector->priv;

	switch( property_id ) {
		case PROP_WINDOW:
			g_value_set_pointer( value, priv->window );
		default:
			break;
	}
}

static void screem_tag_inspector_size_request( GtkWidget *widget,
                			GtkRequisition *requisition )
{
	GtkBin *bin;

	bin = GTK_BIN (widget);

	requisition->width = GTK_CONTAINER( widget )->border_width * 2;
	requisition->height = GTK_CONTAINER( widget )->border_width * 2;

	if( bin->child && GTK_WIDGET_VISIBLE( bin->child ) ) {
		GtkRequisition child_requisition;

		gtk_widget_size_request( bin->child, &child_requisition );
		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static void screem_tag_inspector_size_allocate( GtkWidget *widget,
                            		GtkAllocation *allocation )
{
	GtkBin *bin;
	GtkAllocation child_allocation;

	bin = GTK_BIN( widget );
	widget->allocation = *allocation;
	
	if( bin->child ) {
	child_allocation.x = allocation->x + 
				GTK_CONTAINER( widget )->border_width; 
	child_allocation.y = allocation->y + 
				GTK_CONTAINER (widget)->border_width;
	child_allocation.width = MAX( allocation->width - 
				      GTK_CONTAINER( widget )->border_width * 2,
					0);
	child_allocation.height = MAX( allocation->height - 
				       GTK_CONTAINER (widget)->border_width * 2,
					0);
	gtk_widget_size_allocate( bin->child, &child_allocation );
	}
}

