/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2012 - Scilab Enterprises - Calixte Denizet
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.g2d.motor;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.geom.Path2D;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.scilab.forge.scirenderer.texture.Texture;
import org.scilab.forge.scirenderer.tranformations.Vector3d;
import org.scilab.forge.scirenderer.tranformations.Vector4d;

/**
 * @author Calixte DENIZET
 */
public class Triangle extends ConvexObject {

    private static final Color[] COLORS = new Color[]{Color.BLACK, Color.BLACK, Color.BLACK};

    private SpritedRectangle sprite;
    private BufferedImage image;
    private Vector3d[] textureCoords;
    private Texture.Filter filter;

    protected Set<Segment> segments;

    public Triangle(Vector3d[] vertices, Color[] colors, Vector3d normal) throws InvalidPolygonException {
        super(vertices, colors);
        if (vertices.length != 3) {
            throw new InvalidPolygonException("Invalid triangle: must have 3 vertices.");
        }
    }

    public Triangle(Vector3d[] vertices, Color[] colors) throws InvalidPolygonException {
        this(vertices, colors, null);
    }

    public Triangle(Vector3d[] vertices, Vector3d[] textureCoords, BufferedImage image, Texture.Filter filter) throws InvalidPolygonException {
        this(vertices, COLORS, null);
        this.textureCoords = textureCoords;
        this.image = image;
        this.filter = filter;
    }

    @Override
    public int isBehind(ConvexObject o) {
        if (o instanceof Segment && isSegmentAcross((Segment) o)) {
            return -1;
        }

        return super.isBehind(o);
    }

    public boolean addSegment(Segment s) {
        if (isSegmentInside(s)) {
            if (segments == null) {
                segments = new TreeSet<Segment>();
            }
            segments.add(s);
            s.addConvexObject(this);
            return true;
        }

        return false;
    }

    public boolean pointOnVertices(Vector3d p) {
        return p.equals(vertices[0]) || p.equals(vertices[1]) || p.equals(vertices[2]);
    }

    public void removeSegment(Segment s) {
        if (segments != null) {
            segments.remove(s);
        }
    }

    public void replaceSegment(Segment s, List<Segment> segs) {
        segments.remove(s);
        for (Segment ss : segs) {
            segments.add(ss);
            ss.addConvexObject(this);
        }
    }

    @Override
    public List<ConvexObject> breakObject(ConvexObject o) {
        if (o instanceof Triangle) {
            return breakObject((Triangle) o);
        } else if (o instanceof Segment) {
            return breakObject((Segment) o);
        } else if (o instanceof SpritedRectangle) {
            return ((SpritedRectangle) o).breakObject(this);
        }

        return null;
    }

    public List<ConvexObject> breakObject(Triangle o) {
        Vector3d n = Vector3d.product(this.v0v1, o.v0v1);
        n = n.times(1 / n.getNorm2());
        Vector3d u = Vector3d.product(o.v0v1, n);
        Vector3d v = Vector3d.product(n, this.v0v1);
        Vector3d p = Vector3d.getBarycenter(u, v, this.v0v1.scalar(this.vertices[0]), o.v0v1.scalar(o.vertices[0]));

        List<ConvexObject> list1 = breakTriangleOnLine(o, p, this.v0v1);
        List<ConvexObject> list2 = breakTriangleOnLine(this, p, o.v0v1);

        list1.addAll(list2);

        return list1;
    }

    public List<ConvexObject> breakObject(Segment o) {
        double c = this.getSegmentIntersection(o);
        if (Double.isNaN(c)) {
            return null;
        }

        List<ConvexObject> list = new ArrayList<ConvexObject>();
        Vector3d p = Vector3d.getBarycenter(o.vertices[0], o.vertices[1], c, 1 - c);
        Color cc = getColorsBarycenter(o.colors[0], o.colors[1], c, 1 - c);

        try {
            list.add(new Segment(new Vector3d[]{o.vertices[0], p}, new Color[]{o.colors[0], cc}, o.stroke));
            list.add(new Segment(new Vector3d[]{p, o.vertices[1]}, new Color[]{cc, o.colors[1]}, o.stroke));
        } catch (InvalidPolygonException e) { }

        List<ConvexObject> list1 = breakTriangleOnLine(this, p, Vector3d.product(v0v1, vertices[0].minus(p)));
        list.addAll(list1);

        return list;
    }

    protected void setSprite(SpritedRectangle sprite) {
        this.sprite = sprite;
    }

    protected SpritedRectangle getSprite() {
        return sprite;
    }

    @Override
    public void draw(Graphics2D g2d) {
        if (sprite != null) {
            Shape oldClip = g2d.getClip();
            g2d.clip(getProjectedContour());
            sprite.draw(g2d);
            g2d.setClip(oldClip);
        } else if (image != null) {
            Object key = filter == Texture.Filter.LINEAR ? RenderingHints.VALUE_INTERPOLATION_BILINEAR : RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR;
            DrawTools.drawTriangleTexture(g2d, image, new double[]{textureCoords[0].getX(), textureCoords[1].getX(), textureCoords[2].getX()}, new double[]{textureCoords[0].getY(), textureCoords[1].getY(), textureCoords[2].getY()}, new double[]{vertices[0].getX(), vertices[1].getX(), vertices[2].getX()}, new double[]{vertices[0].getY(), vertices[1].getY(), vertices[2].getY()}, key);
        } else if (colors[0].equals(colors[1]) && colors[1].equals(colors[2])) {
            Path2D contour = getProjectedContour();
            g2d.setColor(colors[0]);
            g2d.fill(contour);
            //g2d.draw(contour);
        } else {
            DrawTools.fillGouraud(g2d, this);
        }

        if (segments != null) {
            for (Segment s : segments) {
                s.removeConvexObject(this);
                s.draw(g2d);
            }
        }

        drawAreas(g2d);
    }

    @Override
    public List<ConvexObject> breakObject(Vector4d v) {
        double[] vv = v.getData();
        Vector3d np = new Vector3d(vv);
        int ret = isBehind(np, vv[3]);
        if (ret == 1) {
            List<ConvexObject> list = new ArrayList<ConvexObject>();
            list.add(this);
            return list;
        } else if (ret == -1) {
            return null;
        }

        Vector3d n = Vector3d.product(this.v0v1, np);
        n = n.times(1 / n.getNorm2());
        Vector3d u = Vector3d.product(np, n);
        Vector3d w = Vector3d.product(n, this.v0v1);
        Vector3d p = Vector3d.getBarycenter(u, w, this.v0v1.scalar(this.vertices[0]), -vv[3]);

        List<ConvexObject> list1 = breakTriangleOnLine(this, p, np);
        List<ConvexObject> list = new ArrayList<ConvexObject>();
        for (ConvexObject o : list1) {
            if (o.isBehind(np, vv[3]) == 1) {
                list.add(o);
            }
        }

        return list;
    }

    protected boolean isPointInside(final Vector3d v) {
        return isPointInside(v, true);
    }

    protected boolean isPointInside(final Vector3d v, final boolean checkCoplanarity) {
        Vector3d v2 = v.minus(vertices[0]);
        if (checkCoplanarity && !isNull(v2.scalar(v0v1))) {
            return false;
        }

        Vector3d v0v1v2 = Vector3d.product(v0v1, v2);
        double a = v0v1v2.scalar(v1);
        if (a < 0 && !isNull(a)) {
            return false;
        }

        double b = -v0v1v2.scalar(v0);
        if (b < 0 && !isNull(b)) {
            return false;
        }

        return a + b < nv0v1 || isEqual(a + b, nv0v1);
    }

    protected boolean isCoplanar(final Segment s) {
        double sc = vertices[0].scalar(v0v1);
        return isEqual(sc, s.vertices[0].scalar(v0v1)) && isEqual(sc, s.vertices[1].scalar(v0v1));
    }

    protected boolean isCoplanar(final Triangle t) {
        double sc = vertices[0].scalar(v0v1);
        return isEqual(sc, t.vertices[0].scalar(v0v1)) && isEqual(sc, t.vertices[1].scalar(v0v1)) && isEqual(sc, t.vertices[2].scalar(v0v1));
    }

    protected boolean isSegmentAcross(final Segment s) {
        if (!isCoplanar(s)) {
            // the segment and the triangle are not coplanar
            return false;
        }

        return check2DTrueIntersection(s);
    }

    protected boolean isSegmentInside(final Segment s) {
        if (!isCoplanar(s)) {
            // the segment and the triangle are not coplanar
            return false;
        }

        // Since there is a good probability that a segment is triangle border we check that
        if ((s.vertices[0].equals(vertices[0]) || s.vertices[0].equals(vertices[1]) || s.vertices[0].equals(vertices[2])) &&
            (s.vertices[1].equals(vertices[0]) || s.vertices[1].equals(vertices[1]) || s.vertices[1].equals(vertices[2]))) {
            return true;
        }

        return isPointInside(s.vertices[0], false) && isPointInside(s.vertices[1], false);
    }

    protected boolean isSegmentIntersects(final Segment s) {
        Vector3d v3 = s.vertices[0].minus(vertices[0]);
        Vector3d v4 = s.vertices[1].minus(vertices[0]);
        double c = v3.scalar(v0v1);

        if (Math.signum(c) == Math.signum(v4.scalar(v0v1))) {
            return false;
        }

        Vector3d v2 = s.vertices[0].minus(s.vertices[1]);
        Vector3d v1v2 = Vector3d.product(v1, v2);
        double a = v3.scalar(v1v2);
        double b = Vector3d.det(v3, v2, v0);
        double detv0v1v2 = v0.scalar(v1v2);
        double sign = Math.signum(detv0v1v2);

        return Math.signum(a) == sign && Math.signum(b) == sign && Math.signum(c) == sign && a + b <= detv0v1v2 && c <= detv0v1v2;
    }

    protected double getSegmentIntersection(final Segment s) {
        Vector3d v = s.vertices[1].minus(vertices[0]);
        double c = v.scalar(v0v1) / s.v0.scalar(v0v1);

        if (isNegativeOrNull(c) || isGreaterOrEqual(c, 1)) {
            return Double.NaN;
        }

        Vector3d p = Vector3d.getBarycenter(s.vertices[0], s.vertices[1], c, 1 - c);
        if (isPointInside(p, false)) {
            return c;
        }

        return Double.NaN;
    }

    protected static List<ConvexObject> breakSegmentOnTriangle(final Triangle t, final Segment s) {
        double c = t.getSegmentIntersection(s);
        if (Double.isNaN(c)) {
            return null;
        }

        List<ConvexObject> list = new ArrayList<ConvexObject>();
        Vector3d p = Vector3d.getBarycenter(s.vertices[0], s.vertices[1], c, 1 - c);
        Color cc = getColorsBarycenter(s.colors[0], s.colors[1], c, 1 - c);

        try {
            list.add(new Segment(new Vector3d[]{s.vertices[0], p}, new Color[]{s.colors[0], cc}, s.stroke));
            list.add(new Segment(new Vector3d[]{p, s.vertices[1]}, new Color[]{cc, s.colors[1]}, s.stroke));
        } catch (InvalidPolygonException e) { }


        return list;
    }

    /**
     * Break a triangle according to its intersection with a line containing p in the plane of the triangle and orthogonal to n
     * The triangle and the line are supposed to be coplanar.
     * @param t the triangle to break
     * @param p a point of the line
     * @param n a vector
     * @return a list of triangles
     */
    protected static List<ConvexObject> breakTriangleOnLine(Triangle t, Vector3d p, Vector3d n) {
        // aP0+(1-a)P1
        double a = t.vertices[1].minus(p).scalar(n) / t.v0.scalar(n);
        // bP0+(1-b)P2
        double b = t.vertices[2].minus(p).scalar(n) / t.v1.scalar(n);
        // cP1+(1-c)P2
        Vector3d v2 = t.vertices[2].minus(t.vertices[1]);
        double c = t.vertices[2].minus(p).scalar(n) / v2.scalar(n);

        List<ConvexObject> list = new ArrayList<ConvexObject>();
        int i = -1;
        int j = -1;
        int k = -1;
        double weight = -1;
        if (isNull(a)) {
            // We are on P1
            i = 1;
            j = 2;
            k = 0;
            weight = b;
        }

        if (isNull(a - 1)) {
            // We are on P0
            if (weight != -1) {
                list.add(t);
                return list;
            } else {
                i = 0;
                j = 1;
                k = 2;
                weight = c;
            }
        }

        if (isNull(b)) {
            // We are on P2
            if (weight != -1) {
                list.add(t);
                return list;
            } else {
                i = 2;
                j = 0;
                k = 1;
                weight = a;
            }
        }

        if (i != -1) {
            if (weight >= 0 && weight <= 1) {
                // We break into two triangles
                Vector3d vb = Vector3d.getBarycenter(t.vertices[j], t.vertices[k], weight, 1 - weight);
                Color cb = getColorsBarycenter(t.colors[j], t.colors[k], weight, 1 - weight);
                Vector3d[] vertices1 = new Vector3d[]{t.vertices[i], t.vertices[j], vb};
                Vector3d[] vertices2 = new Vector3d[]{t.vertices[i], vb, t.vertices[k]};
                Color[] colors1 = new Color[]{t.colors[i], t.colors[j], cb};
                Color[] colors2 = new Color[]{t.colors[i], cb, t.colors[k]};

                try {
                    Triangle t1 = new Triangle(vertices1, colors1, null);
                    t1.setSprite(t.getSprite());
                    list.add(t1);
                    Triangle t2 = new Triangle(vertices2, colors2, null);
                    t2.setSprite(t.getSprite());
                    list.add(t2);
                } catch (InvalidPolygonException e) { }

                addSegments(list, t, p, Vector3d.product(t.v0v1, n), n);

                return list;
            } else {
                list.add(t);
                return list;
            }
        }

        Color cu, cv;
        Vector3d u, v;
        // t has not been broken, so continue...
        if (a < 0 || a > 1) {
            i = 2;
            j = 0;
            k = 1;
            u = Vector3d.getBarycenter(t.vertices[1], t.vertices[2], c, 1 - c);
            v = Vector3d.getBarycenter(t.vertices[0], t.vertices[2], b, 1 - b);
            cu = getColorsBarycenter(t.colors[1], t.colors[2], c, 1 - c);
            cv = getColorsBarycenter(t.colors[0], t.colors[2], b, 1 - b);
        } else if (b < 0 || b > 1) {
            i = 1;
            j = 2;
            k = 0;
            u = Vector3d.getBarycenter(t.vertices[0], t.vertices[1], a, 1 - a);
            v = Vector3d.getBarycenter(t.vertices[1], t.vertices[2], c, 1 - c);
            cu = getColorsBarycenter(t.colors[0], t.colors[1], a, 1 - a);
            cv = getColorsBarycenter(t.colors[1], t.colors[2], c, 1 - c);
        } else {
            i = 0;
            j = 1;
            k = 2;
            u = Vector3d.getBarycenter(t.vertices[0], t.vertices[2], b, 1 - b);
            v = Vector3d.getBarycenter(t.vertices[0], t.vertices[1], a, 1 - a);
            cu = getColorsBarycenter(t.colors[0], t.colors[2], b, 1 - b);
            cv = getColorsBarycenter(t.colors[0], t.colors[1], a, 1 - a);
        }

        Vector3d[] vertices1 = new Vector3d[]{u, t.vertices[i], v};
        Color[] colors1 = new Color[]{cu, t.colors[i], cv};
        Vector3d[] vertices2 = new Vector3d[]{u, v, t.vertices[j]};
        Color[] colors2 = new Color[]{cu, cv, t.colors[j]};
        Vector3d[] vertices3 = new Vector3d[]{u, t.vertices[j], t.vertices[k]};
        Color[] colors3 = new Color[]{cu, t.colors[j], t.colors[k]};

        try {
            Triangle t1 = new Triangle(vertices1, colors1, null);
            t1.setSprite(t.getSprite());
            list.add(t1);
            Triangle t2 = new Triangle(vertices2, colors2, null);
            t2.setSprite(t.getSprite());
            list.add(t2);
            Triangle t3 = new Triangle(vertices3, colors3, null);
            t3.setSprite(t.getSprite());
            list.add(t3);
        } catch (InvalidPolygonException e) { }

        addSegments(list, t, p, Vector3d.product(t.v0v1, n), n);

        return list;
    }

    private static final void addSegments(List<ConvexObject> list, Triangle t, Vector3d p, Vector3d u, Vector3d n) {
        if (t.segments != null) {
            boolean bb = false;
            List<Segment> segs = new ArrayList<Segment>();
            for (Segment s : t.segments) {
                s.removeConvexObject(t);
                List<Segment> l = s.breakObject(p, u, n);
                if (l != null && !l.isEmpty()) {
                    segs.addAll(l);
                    s.replaceSegment(l);
                }
            }

            for (Segment s : segs) {
                for (ConvexObject tri : list) {
                    ((Triangle) tri).addSegment(s);
                }
            }
        }
    }

    /**
     * Get the broken triangles in following the intersection of the planes containing t1 and t2.
     * The planes containing t1 and t2 are supposed to be secant.
     * @param t1 the first triangle
     * @param t2 the second triangle
     * @return an array of length 2 containing the resulting triangles for t1 and t2.
     */
    protected static List<ConvexObject> breakIntersectingTriangles(Triangle t1, Triangle t2) {
        Vector3d n = Vector3d.product(t1.v0v1, t2.v0v1);
        n = n.times(1 / n.getNorm2());
        Vector3d u = Vector3d.product(t2.v0v1, n);
        Vector3d v = Vector3d.product(n, t1.v0v1);
        Vector3d p = Vector3d.getBarycenter(u, v, t1.v0v1.scalar(t1.vertices[0]), t2.v0v1.scalar(t2.vertices[0]));

        List<ConvexObject> list1 = breakTriangleOnLine(t1, p, t2.v0v1);
        List<ConvexObject> list2 = breakTriangleOnLine(t2, p, t1.v0v1);
        list1.addAll(list2);

        return list1;
    }

    public String toString() {
        return "Triangle: " + vertices[0].toString() + " " + vertices[1].toString() + " " + vertices[2].toString() + "\nColor: " + colors[0] + "\nPrecedence: " + precedence;
    }
}
