/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2012 - Scilab Enterprises - Calixte Denizet
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.g2d.motor;

import java.awt.Color;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Path2D;
import java.awt.image.BufferedImage;

/**
 * @author Calixte DENIZET
 */
public final class DrawTools {

    private static final Color TRANSLUCENT_BLACK = new Color(0, 0, 0, 0);

    /**
     * Fill a triangle in using a Gouraud shading
     * Only two gradient are used rather than three.
     * @param g2d the Graphics2D where to draw
     * @param t the Triangle to fill
     */
    public static final void fillGouraud(final Graphics2D g2d, final Triangle t) {
        Path2D contour = t.getProjectedContour();
        double[] v0 = new double[]{t.vertices[0].getX(), t.vertices[0].getY()};
        double[] v1 = new double[]{t.vertices[1].getX(), t.vertices[1].getY()};
        double[] v2 = new double[]{t.vertices[2].getX(), t.vertices[2].getY()};
        double[] pv0 = get2DProjection(v0[0], v0[1], v1[0], v1[1], v2[0], v1[1]);
        double[] pv2 = get2DProjection(v2[0], v2[1], v0[0], v0[1], v1[0], v0[0]);

        Paint oldPaint = g2d.getPaint();

        // When the colours are totally opaque, the combination of the two GradientPaint are equivalent to a Gouraud shading
        // If there is transparency, that could lead to a different rendering between JOGL and this implementation.
        GradientPaint gp = new GradientPaint((float) v0[0], (float) v0[1], t.colors[0], (float) pv0[0], (float) pv0[1], t.colors[1]);
        g2d.setPaint(gp);
        g2d.fill(contour);

        gp = new GradientPaint((float) v2[0], (float) v2[1], t.colors[2], (float) pv2[0], (float) pv2[1], TRANSLUCENT_BLACK);
        g2d.setPaint(gp);
        g2d.fill(contour);

        g2d.setPaint(oldPaint);
    }

    /**
     * Draw a texture (ie a BufferedImage) in a triangle
     * @param g2d the Graphics2D where to draw
     * @param image the texture to apply
     * @param ximg the x-coordinates of the triangle to use in the texture
     * @param yimg the y-coordinates of the triangle to use in the texture
     * @param xdest the x-coordinates of the destination triangle
     * @param ydest the y-coordinates of the destination triangle
     * @param key the rendering hint to use for interpolation
     */
    public static final void drawTriangleTexture(final Graphics2D g2d, final BufferedImage image, final double[] ximg, final double[] yimg, final double[] xdest, final double[] ydest, Object key) {
        try {
            Object oldKey = g2d.getRenderingHint(RenderingHints.KEY_INTERPOLATION);
            Shape oldClip = g2d.getClip();

            double w = image.getWidth();
            double h = image.getHeight();

            for (int i = 0; i < xdest.length; i++) {
                xdest[i] = Math.round(xdest[i]);
                ydest[i] = Math.round(ydest[i]);
            }

            // if we have a 1D texture we must slighlty modified the coordinates to use the algorithm below.
            checkSourceCoordinates(ximg, yimg);

            AffineTransform translationDest = AffineTransform.getTranslateInstance(xdest[0], ydest[0]);
            AffineTransform translationImg = AffineTransform.getTranslateInstance(-w * ximg[0], -h * yimg[0]);

            AffineTransform toDest = new AffineTransform(xdest[1] - xdest[0], ydest[1] - ydest[0], xdest[2] - xdest[0], ydest[2] - ydest[0], 0, 0);
            AffineTransform fromImg = new AffineTransform(w * (ximg[1] - ximg[0]), h * (yimg[1] - yimg[0]), w * (ximg[2] - ximg[0]), h * (yimg[2] - yimg[0]), 0, 0).createInverse();

            AffineTransform transformation = new AffineTransform();
            transformation.concatenate(translationDest);
            transformation.concatenate(toDest);
            transformation.concatenate(fromImg);
            transformation.concatenate(translationImg);

            Path2D.Double path = new Path2D.Double();
            path.moveTo(xdest[0], ydest[0]);
            path.lineTo(xdest[1], ydest[1]);
            path.lineTo(xdest[2], ydest[2]);
            path.closePath();

            g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, key);
            g2d.clip(new java.awt.geom.Area(path));
            g2d.drawImage(image, transformation, null);

            if (oldKey != null) {
                g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, oldKey);
            }

            g2d.setClip(oldClip);
        } catch (NoninvertibleTransformException e) { }
    }

    /**
     * Draw a texture (ie a BufferedImage) in a parallelogram
     * @param g2d the Graphics2D where to draw
     * @param image the texture to apply
     * @param ximg the x-coordinates of the parallelogram to use in the texture
     * @param yimg the y-coordinates of the parallelogram to use in the texture
     * @param xdest the x-coordinates of the destination parallelogram
     * @param ydest the y-coordinates of the destination parallelogram
     * @param key the rendering hint to use for interpolation
     */
    public static final void drawParallelogramTexture(final Graphics2D g2d, final BufferedImage image, final double[] ximg, final double[] yimg, final double[] xdest, final double[] ydest, Object key) {
        try {
            Object oldKey = g2d.getRenderingHint(RenderingHints.KEY_INTERPOLATION);

            double w = image.getWidth();
            double h = image.getHeight();

            AffineTransform translationDest = AffineTransform.getTranslateInstance(xdest[0], ydest[0]);
            AffineTransform translationImg = AffineTransform.getTranslateInstance(-w * ximg[0], -h * yimg[0]);

            AffineTransform toDest = new AffineTransform(xdest[1] - xdest[0], ydest[1] - ydest[0], xdest[2] - xdest[0], ydest[2] - ydest[0], 0, 0);
            AffineTransform fromImg = new AffineTransform(w * (ximg[1] - ximg[0]), h * (yimg[1] - yimg[0]), w * (ximg[2] - ximg[0]), h * (yimg[2] - yimg[0]), 0, 0).createInverse();

            AffineTransform transformation = new AffineTransform();
            transformation.concatenate(translationDest);
            transformation.concatenate(toDest);
            transformation.concatenate(fromImg);
            transformation.concatenate(translationImg);

            g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, key);
            g2d.drawImage(image, transformation, null);

            if (oldKey != null) {
                g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, oldKey);
            }
        } catch (NoninvertibleTransformException e) { }
    }

    /**
     * Check and modify the coordinates if we have a 1D texture (i.e. all the y-coords are zero)
     * @param x the x-coordinates
     * @param y the y-coordinates
     */
    private static final void checkSourceCoordinates(double[] x, double[] y) {
        boolean is1D = y[0] == 0 && y[1] == 0 && y[2] == 0;
        if (is1D) {
            if (x[0] != x[1] && x[1] != x[2] && x[2] != x[0]) {
                y[0] = 1;
            } else if (x[0] == x[1] && x[1] != x[2]) {
                y[0] = 1;
            } else if (x[0] == x[2] && x[2] != x[1]) {
                y[2] = 1;
            } else if (x[1] == x[2] && x[2] != x[0]) {
                y[2] = 1;
            } else {
                x[1] += 1e-9d;
                x[2] += 2e-9d;
                y[0] = 1;
            }
        }
    }

    /**
     * Get the projection in 2D of point A on line (BC)
     * @param A the point to project
     * @param B a point of the line
     * @param C an other point of the line (different of B)
     * @return the projected point
     */
    private static final double[] get2DProjection(final double xA, final double yA, final double xB, final double yB, final double xC, final double yC) {
        final double xBC = xC - xB;
        final double yBC = yC - yB;
        final double n = xBC * xBC + yBC * yBC;
        final double s = (xBC * (xA - xB) + yBC * (yA - yB)) / n;

        return new double[]{xB + s * xBC, yB + s * yBC};
    }
}
