/** @file scim_hangul_imengine.h
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@tsinghua.org.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_hangul_imengine.h 29 2004-08-03 02:55:30Z suzhe $
 */

#if !defined (__SCIM_HANGUL_IMENGINE_H)
#define __SCIM_HANGUL_IMENGINE_H

using namespace scim;

enum HangulComposerType
{
    HANGUL_COMPOSER_2,
    HANGUL_COMPOSER_3
};

enum HangulInputMode
{
    INPUT_MODE_DIRECT,
    INPUT_MODE_HANGUL,
    INPUT_MODE_HANJA
};

enum HangulOutputMode
{
    OUTPUT_MODE_SYLLABLE = 0,
    OUTPUT_MODE_JAMO     = 1 << 1,
    OUTPUT_MODE_JAMO_EXT = 1 << 2
};

struct HangulCombination 
{
  uint32  key;
  ucs4_t  code;
};

struct HangulFactoryData
{
    const char              *uuid;
    const char              *name;

    HangulComposerType       composer_type;
    const ucs4_t            *keyboard_table;
    int                      compose_table_size;
    const HangulCombination *compose_table;
    bool                     always_use_jamo;
};

class HangulFactory : public IMEngineFactoryBase
{
    String                   m_uuid;
    String                   m_name;

    HangulComposerType       m_composer_type;
    const ucs4_t            *m_keyboard_table;
    int                      m_compose_table_size;
    const HangulCombination *m_compose_table;

    bool                     m_always_use_jamo;

    bool                     m_use_capslock;
    bool                     m_use_dvorak;
    bool                     m_show_candidate_comment;

    KeyEventList             m_trigger_keys;
    KeyEventList             m_hangul_hanja_keys;
    KeyEventList             m_manual_mode_keys;

    friend class HangulInstance;

public:
    HangulFactory (const ConfigPointer &config, const HangulFactoryData &data);

    virtual ~HangulFactory ();

    virtual WideString  get_name () const;
    virtual WideString  get_authors () const;
    virtual WideString  get_credits () const;
    virtual WideString  get_help () const;
    virtual String      get_uuid () const;
    virtual String      get_icon_file () const;

    virtual IMEngineInstancePointer create_instance (const String& encoding, int id = -1);
};

class HangulInstance : public IMEngineInstanceBase
{
    HangulFactory       *m_factory;

    CommonLookupTable    m_lookup_table;
    std::vector<String>  m_candidate_comments;

    KeyEvent             m_prev_key;

    int                  m_index;         /* stack index */
    ucs4_t               m_stack     [12];

    int                  m_lindex;        /* leading consonant index */
    int                  m_vindex;        /* vowel index */
    int                  m_tindex;        /* trailing consonant index */
    ucs4_t               m_choseong  [4];
    ucs4_t               m_jungseong [4];
    ucs4_t               m_jongseong [4];

    int                  m_input_mode;
    int                  m_output_mode;

    bool (HangulInstance::*m_hangul_composer) (const KeyEvent &key);

public:
    HangulInstance (HangulFactory *factory,
                    const String  &encoding,
                    int            id = -1);

    virtual ~HangulInstance ();

    virtual bool process_key_event (const KeyEvent& key);
    virtual void move_preedit_caret (unsigned int pos);
    virtual void select_candidate (unsigned int item);
    virtual void update_lookup_table_page_size (unsigned int page_size);
    virtual void lookup_table_page_up ();
    virtual void lookup_table_page_down ();
    virtual void reset ();
    virtual void focus_in ();
    virtual void focus_out ();
    virtual void trigger_property (const String &property);

private:
    /* asistant function for hangul composer */
    bool im_hangul_is_choseong  (ucs4_t ch) const {
        return (ch >= 0x1100 && ch <= 0x1159);
    }

    bool im_hangul_is_jungseong (ucs4_t ch) const {
        return (ch >= 0x1161 && ch <= 0x11A2);
    }

    bool im_hangul_is_jongseong (ucs4_t ch) const {
        return (ch >= 0x11A7 && ch <= 0x11F9);
    }

    bool im_hangul_is_empty () const {
        return (m_choseong[0]  == 0 && m_jungseong[0] == 0 && m_jongseong[0] == 0);
    }

    bool im_hangul_is_modifier_key  (const KeyEvent &key) const {
        return ((key.mask & SCIM_KEY_ControlMask) || (key.mask & SCIM_KEY_AltMask));
    }

    bool im_hangul_is_backspace_key (const KeyEvent &key) const {
        return (key.code == SCIM_KEY_BackSpace);
    }

    bool im_hangul_is_trigger_key (const KeyEvent &key) const {
        return match_key_event (m_factory->m_trigger_keys, key);
    }

    bool im_hangul_is_hangul_hanja_key (const KeyEvent &key) const {
        return match_key_event (m_factory->m_hangul_hanja_keys, key);
    }

    bool im_hangul_is_manual_mode_key (const KeyEvent &key) const {
        return match_key_event (m_factory->m_manual_mode_keys, key);
    }

    ucs4_t im_hangul_compose (ucs4_t first, ucs4_t last) const;
    ucs4_t im_hangul_mapping (int code, int mask) const;

    bool   im_hangul_add_choseong  (ucs4_t ch);
    bool   im_hangul_add_jungseong (ucs4_t ch);
    bool   im_hangul_add_jongseong (ucs4_t ch);
    bool   im_hangul_sub_choseong  ();
    bool   im_hangul_sub_jungseong ();
    bool   im_hangul_sub_jongseong ();

    bool im_hangul_composer_2 (const KeyEvent &key);
    bool im_hangul_composer_3 (const KeyEvent &key);

    /* stack functions */
    ucs4_t im_hangul_pop  ();
    ucs4_t im_hangul_peek ();
    void   im_hangul_push (ucs4_t ch);

    /* commit functions */
    WideString im_hangul_get_hangul_string ();

    void   im_hangul_clear_buf ();
    void   im_hangul_commit ();
    void   im_hangul_commit_unicode (ucs4_t ch);

    /* candidate functions */
    void   im_hangul_update_candidates ();
    void   im_hangul_delete_candidates ();

    /* aux string */
    void   im_hangul_update_aux_string ();

    /* preedit string */
    void   im_hangul_update_preedit_string ();

    /* properties */
    void   im_hangul_register_properties ();
    void   im_hangul_update_input_mode_property ();
    void   im_hangul_update_output_mode_property ();

    /* candidate keys */
    bool   candidate_key_event (const KeyEvent &key);

    /* match key event */
    bool   match_key_event (const KeyEventList &keys, const KeyEvent &key) const;
};
#endif

/*
vi:ts=4:nowrap:ai:expandtab
*/
