//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [param2,cov2]=CL_dsp_covCoord(param1,cov1,conv1to2,mu)
// Conversion of type of orbital elements in a covariance matrix
//
// Calling Sequence
// [param2,cov2]=CL_dsp_covCoord(param1,cov1,conv1to2 [,mu])
//
// Description
// <itemizedlist><listitem>
// <p>Performs a change of type of orbital elements in a covariance matrix.</p> 
// <p>Allowed conversions are those handled by the CL_oe_xxx functions of CelestLab ('cir2car','car2cirEqua','car2kep', 'cirequa2kep' and so on).</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// param1: State vector (with components of type 1). (6xN)
// cov1: Covariance matrix associated with param1. (6x6xN)
// conv1to2: (string) Type of conversion (from 1 to 2). It is the name of the CelestLab function that performs the same conversion without the prefix 'CL_oe_'.
// mu : (optional) Gravitational constant [m^3/s^2]. (default value is %CL_mu)
// param2: State vector (with components of type 2). (6xN)
// cov2: Covariance matrix associated with param2.  (6x6xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_cor2cov
// CL_stat
//
// Examples
// // Change correlation matrix: cartesian to circular-adapted
// bul = [-1877901 -3909428 -5026025 7428.157 -1541.857 -1576.532]'; 
// cor = [1 -0.467016 -0.447601 0.960396 0.987145 0.995826;...
//      0 1 -0.088751 -0.359696 -0.412472 -0.540655;...
//      0 0 1 -0.248472 -0.582834 -0.431908;...
//      0 0 0 1 0.915197 0.943178;...
//      0 0 0 0 1 0.980679;...
//      0 0 0 0 0 1]; //upper triangle of correlation matrix
// cor = cor+cor'-eye(cor); // correlation matrix (symmetric)
// // standard  deviation : 
// sd = [15939.681;2912.099;3079.494;6.819104;9.500176;12.146244];
// // Covariance matrix in cartesian parameters
// cov_car = CL_cor2cov(cor,sd); 
// [bul_cir,cov_cir] = CL_dsp_covCoord(bul,cov_car,'car2cir');
// [cor_cir,sd_cir] = CL_cov2cor(cov_cir);
//

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end

select conv1to2
  case 'car2cir'
    [param2,jacob] = CL_oe_car2cir(param1(1:3,:),param1(4:6,:),mu);
  case 'car2cirEqua'
    [param2,jacob] = CL_oe_car2cirEqua(param1(1:3,:),param1(4:6,:),mu);
  case 'car2kep'
    [param2,jacob] = CL_oe_car2kep(param1(1:3,:),param1(4:6,:),mu);
  case 'cir2car'
    [pos,vel,jacob] = CL_oe_cir2car(param1,mu);
    param2 = [pos;vel];
  case 'cirequa2car'
    [pos,vel,jacob] = CL_oe_cirEqua2car(param1,mu);
    param2 = [pos;vel];
  case 'cirequa2kep'
    [param2,jacob] = CL_oe_cirEqua2kep(param1);
  case 'kep2car'
    [pos,vel,jacob] = CL_oe_kep2car(param1,mu);
    param2 = [pos;vel];
  case 'kep2cirequa'
    [param2,jacob] = CL_oe_kep2cirEqua(param1);
  else
    CL__error('conversion '+conv1to2+' unknown');
end

cov2 = jacob*cov1*jacob';

endfunction
