// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <math.h>
#include <float.h>

#include <unistd.h>

#include <fstream>
#include "fdstream.hpp"

#include "tcl.h"
#include <Xlib.h>
#include <Xutil.h>

#include "framebase.h"
#include "fitsimage.h"
#include "util.h"
#include "mmap.h"

#include "fvcontour.h"
#include "grid.h"

// Frame Command Member Functions

void FrameBase::clearCmd()
{
  unloadAllFits();
  reset();
  update(MATRIX);
}

void FrameBase::colorScaleCmd(FrScale::ColorScaleType s)
{
  setColorScale(s);
  update(BASE);
}

void FrameBase::getBgColorCmd()
{
  Tcl_AppendResult(interp, bgColorName, NULL);
}

void FrameBase::getColorScaleCmd()
{
  switch (currentScale->colorScaleType()) {
  case FrScale::LINEARSCALE:
    Tcl_AppendResult(interp, "linear", NULL);
    break;
  case FrScale::LOGSCALE:
    Tcl_AppendResult(interp, "log", NULL);
    break;
  case FrScale::SQUAREDSCALE:
    Tcl_AppendResult(interp, "squared", NULL);
    break;
  case FrScale::SQRTSCALE:
    Tcl_AppendResult(interp, "sqrt", NULL);
    break;
  case FrScale::IISSCALE:
    Tcl_AppendResult(interp, "linear", NULL);
    break;
  case FrScale::HISTEQUSCALE:
    Tcl_AppendResult(interp, "histequ", NULL);
    break;
  }
}

void FrameBase::getCoordCmd(const Vector& v, InternalSystem in, 
			CoordSystem out, SkyFrame sky, SkyFormat format)
{
  switch (in) {
  case CANVAS:
    if (FitsImage* ptr=isInFits(v, &FitsImage::getCanvasToData, NULL))
      printFromRef(ptr, v * canvasToRef, out, sky, format, DEFAULT);
    break;
  case PANNER:
    if (FitsImage* ptr=isInFits(v, &FitsImage::getPannerToData, NULL))
      printFromRef(ptr, v * pannerToRef, out, sky, format, DEFAULT);
    break;
  }
}

void FrameBase::getCursorCmd(InternalSystem sys)
{
  // make sure we have up-to-date matrices
  updateNow(MATRIX);

  if (currentFits)
    switch (sys) {
    case CANVAS:
      printVector(cursor * userToCanvas, DEFAULT);
      break;
    case PANNER:
      break;
      printVector(cursor * userToPanner, DEFAULT);
    }
  else
    printVector(Vector(), DEFAULT);
} 

void FrameBase::getCursorCmd(CoordSystem sys, SkyFrame sky, SkyFormat format,
			 Precision p)
{
  // make sure we have up-to-date matrices
  updateNow(MATRIX);
  
  if (currentFits)
    printFromRef(currentFits, cursor * userToRef, sys, sky, format, p);
  else
    printVector(Vector(), DEFAULT);
} 

void FrameBase::getInfoCmd(char* var)
{
  if (currentFits) {
    Tcl_SetVar2(interp,var,"filename",
		(char*)currentFits->getRootBaseFileName(),0);
    Tcl_SetVar2(interp,var,"objectname",(char*)currentFits->getObjectName(),0);
    Tcl_SetVar2(interp,var,"min",(char*)currentFits->getMin(),0);
    Tcl_SetVar2(interp,var,"max",(char*)currentFits->getMax(),0);
    Tcl_SetVar2(interp,var,"low",(char*)currentFits->getLow(),0);
    Tcl_SetVar2(interp,var,"high",(char*)currentFits->getHigh(),0);
  }
  else {
    Tcl_SetVar2(interp,var,"objectname","",0);
    Tcl_SetVar2(interp,var,"filename","",0);
    Tcl_SetVar2(interp,var,"min","",0);
    Tcl_SetVar2(interp,var,"max","",0);
    Tcl_SetVar2(interp,var,"low","",0);
    Tcl_SetVar2(interp,var,"high","",0);
  }

  Tcl_SetVar2(interp,var,"value","",0);
  Tcl_SetVar2(interp,var,"xImage","",0);
  Tcl_SetVar2(interp,var,"yImage","",0);
  Tcl_SetVar2(interp,var,"xPhysical","",0);
  Tcl_SetVar2(interp,var,"yPhysical","",0);
  Tcl_SetVar2(interp,var,"xAmplifier","",0);
  Tcl_SetVar2(interp,var,"yAmplifier","",0);
  Tcl_SetVar2(interp,var,"xDetector","",0);
  Tcl_SetVar2(interp,var,"yDetector","",0);

  Tcl_SetVar2(interp,var,"xWCS,","",0);
  Tcl_SetVar2(interp,var,"yWCS,","",0);
  Tcl_SetVar2(interp,var,"sysWCS,","",0);
  Tcl_SetVar2(interp,var,"xUnitWCS,","",0);
  Tcl_SetVar2(interp,var,"yUnitWCS,","",0);
}

void FrameBase::getInfoCmd(const Vector& v, InternalSystem ref,
		       SkyFrame sky, SkyFormat format, char* var)
{
  switch (ref) {
  case CANVAS:
    getInfo(v, ref, &FitsImage::getCanvasToData, &FitsImage::getCanvasToImage,
	    sky, format, var);
    break;
  case PANNER:
    getInfo(v, ref, &FitsImage::getPannerToData, &FitsImage::getPannerToImage,
	    sky, format, var);
    break;
  }
}

void FrameBase::getInfo(const Vector& v, InternalSystem ref,
		    Matrix& (FitsImage::*matrixToData)(),
		    Matrix& (FitsImage::*matrixToImage)(),
		    SkyFrame sky, SkyFormat format, char* var)
{
  if (!isMosaic()) 
    getInfoSingle(v,ref,matrixToData,matrixToImage,sky,format,var);
  else
    getInfoMosaic(v,ref,matrixToData,matrixToImage,sky,format,var);
}

void FrameBase::getInfoSingle(const Vector& v, InternalSystem ref,
			      Matrix& (FitsImage::*matrixToData)(),
			      Matrix& (FitsImage::*matrixToImage)(),
			      SkyFrame sky, SkyFormat format, char* var)
{
  // clear RGB values
  Tcl_SetVar2(interp,var,"value,red","",0);
  Tcl_SetVar2(interp,var,"value,green","",0);
  Tcl_SetVar2(interp,var,"value,blue","",0);

  // the rest
  if (currentFits) {
    Vector img = v * (currentFits->*matrixToData)();
    int* params = currentFits->getDataParams(currentScale->scanMode());
    int& xmin = params[1];
    int& xmax = params[2];
    int& ymin = params[3];
    int& ymax = params[4];

    Tcl_SetVar2(interp,var,"filename",
		(char*)currentFits->getRootBaseFileName(),0);
    Tcl_SetVar2(interp,var,"objectname",(char*)currentFits->getObjectName(),0);
    Tcl_SetVar2(interp,var,"min",(char*)currentFits->getMin(),0);
    Tcl_SetVar2(interp,var,"max",(char*)currentFits->getMax(),0);
    Tcl_SetVar2(interp,var,"low",(char*)currentFits->getLow(),0);
    Tcl_SetVar2(interp,var,"high",(char*)currentFits->getHigh(),0);

    if (img[0]>=xmin && img[0]<xmax && img[1]>=ymin && img[1]<ymax) {
      Tcl_SetVar2(interp,var,"value",(char*)currentFits->getValue(img),0);

      coordToTclArray(currentFits, v, ref, IMAGE, FIXED, var, "Image");
      coordToTclArray(currentFits, v, ref, PHYSICAL, FIXED, var, "Physical");

      if (hasATMV())
	coordToTclArray(currentFits,v,ref, AMPLIFIER, FIXED, var, "Amplifier");
      else {
	Tcl_SetVar2(interp,var,"xAmplifier","",0);
	Tcl_SetVar2(interp,var,"yAmplifier","",0);
      }

      if (hasDTMV())
	coordToTclArray(currentFits, v, ref, DETECTOR, FIXED, var, "Detector");
      else {
	Tcl_SetVar2(interp,var,"xDetector","",0);
	Tcl_SetVar2(interp,var,"yDetector","",0);
      }

      for (int i=0; i<MULTWCS; i++) {
	char buf[64];
	char w = !i ? NULL : '@'+i;
	CoordSystem www = (CoordSystem)(WCS+i);

	if (hasWCS(www)) {
	  char buff[64];
	  currentFits->pix2wcs(v*(currentFits->*matrixToImage)(), www, sky, 
			       format, buff, 64);

	  int argc;
	  const char** argv;
	  Tcl_SplitList(interp, buff, &argc, &argv);

	  if (argc > 0 && argv && argv[0])
	    Tcl_SetVar2(interp,var,varcat(buf,"xWCS,",w),argv[0],0);
	  else
	    Tcl_SetVar2(interp,var,varcat(buf,"xWCS,",w),"",0);

	  if (argc > 1 && argv && argv[1])
	    Tcl_SetVar2(interp,var,varcat(buf,"yWCS,",w),argv[1],0);
	  else
	    Tcl_SetVar2(interp,var,varcat(buf,"yWCS,",w),"",0);

	  char* wcsname = (char*)currentFits->getWCSName(www);
	  if (wcsname)
	    Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),wcsname,0);
	  else if (argc > 2 && argv && argv[2])
	    Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),argv[2],0);
	  else
	    Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),"",0);
	    
	  if (argc > 3 && argv && argv[3])
	    Tcl_SetVar2(interp,var,varcat(buf,"xUnitWCS,",w),argv[3],0);
	  else
	    Tcl_SetVar2(interp,var,varcat(buf,"xUnitWCS,",w),"",0);

	  if (argc > 4 && argv && argv[4])
	    Tcl_SetVar2(interp,var,varcat(buf,"yUnitWCS,",w),argv[4],0);
	  else
	    Tcl_SetVar2(interp,var,varcat(buf,"yUnitWCS,",w),"",0);

	  Tcl_Free((char*)argv);
	}
	else {
	  Tcl_SetVar2(interp,var,varcat(buf,"xWCS,",w),"",0);
	  Tcl_SetVar2(interp,var,varcat(buf,"yWCS,",w),"",0);
	  Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),"",0);
	  Tcl_SetVar2(interp,var,varcat(buf,"xUnitWCS,",w),"",0);
	  Tcl_SetVar2(interp,var,varcat(buf,"yUnitWCS,",w),"",0);
	}
      }
    }
    else
      goto noImage;

    return;
  }

  // else, return blanks
  Tcl_SetVar2(interp,var,"objectname","",0);
  Tcl_SetVar2(interp,var,"filename","",0);
  Tcl_SetVar2(interp,var,"min","",0);
  Tcl_SetVar2(interp,var,"max","",0);
  Tcl_SetVar2(interp,var,"low","",0);
  Tcl_SetVar2(interp,var,"high","",0);

noImage:
  Tcl_SetVar2(interp,var,"value","",0);
  Tcl_SetVar2(interp,var,"xImage","",0);
  Tcl_SetVar2(interp,var,"yImage","",0);
  Tcl_SetVar2(interp,var,"xPhysical","",0);
  Tcl_SetVar2(interp,var,"yPhysical","",0);
  Tcl_SetVar2(interp,var,"xAmplifier","",0);
  Tcl_SetVar2(interp,var,"yAmplifier","",0);
  Tcl_SetVar2(interp,var,"xDetector","",0);
  Tcl_SetVar2(interp,var,"yDetector","",0);

  for (int i=0; i<MULTWCS; i++) {
    char buf[64];
    char w = !i ? NULL : '@'+i;
    Tcl_SetVar2(interp,var,varcat(buf,"xWCS,",w),"",0);
    Tcl_SetVar2(interp,var,varcat(buf,"yWCS,",w),"",0);
    Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),"",0);
    Tcl_SetVar2(interp,var,varcat(buf,"xUnitWCS,",w),"",0);
    Tcl_SetVar2(interp,var,varcat(buf,"yUnitWCS,",w),"",0);
  }
}

void FrameBase::getInfoMosaic(const Vector& v, InternalSystem ref,
			      Matrix& (FitsImage::*matrixToData)(),
			      Matrix& (FitsImage::*matrixToImage)(),
			      SkyFrame sky, SkyFormat format, char* var)
{
  // clear RGB values
  Tcl_SetVar2(interp,var,"value,red","",0);
  Tcl_SetVar2(interp,var,"value,green","",0);
  Tcl_SetVar2(interp,var,"value,blue","",0);

  // the rest
  if (*channelFits) {
    FitsImage* ptr = *channelFits;
    while (ptr) {
      Vector img = v * (ptr->*matrixToData)();
      int* params = ptr->getDataParams(currentScale->scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      if (img[0]>=xmin && img[0]<xmax && img[1]>=ymin && img[1]<ymax) {
	// advance to correct slice
	for (int ss=1; ss<slice; ss++)
	  if (ptr)
	    ptr = ptr->nextSlice();
	if (!ptr)
	  break;

	Tcl_SetVar2(interp,var,"filename",(char*)ptr->getRootBaseFileName(),0);
	Tcl_SetVar2(interp,var,"objectname",(char*)ptr->getObjectName(),0);
	Tcl_SetVar2(interp,var,"min",(char*)ptr->getMin(),0);
	Tcl_SetVar2(interp,var,"max",(char*)ptr->getMax(),0);
	Tcl_SetVar2(interp,var,"low",(char*)ptr->getLow(),0);
	Tcl_SetVar2(interp,var,"high",(char*)ptr->getHigh(),0);
	Tcl_SetVar2(interp,var,"value",(char*)ptr->getValue(img),0);

	coordToTclArray(ptr, v, ref, IMAGE, FIXED, var, "Image");
	coordToTclArray(ptr, v, ref, PHYSICAL, FIXED, var, "Physical");

	if (hasATMV())
	  coordToTclArray(ptr, v, ref, AMPLIFIER, FIXED, var, "Amplifier");
	else {
	  Tcl_SetVar2(interp,var,"xAmplifier","",0);
	  Tcl_SetVar2(interp,var,"yAmplifier","",0);
	}

	if (hasDTMV())
	  coordToTclArray(ptr, v, ref, DETECTOR, FIXED, var, "Detector");
	else {
	  Tcl_SetVar2(interp,var,"xDetector","",0);
	  Tcl_SetVar2(interp,var,"yDetector","",0);
	}

	for (int i=0; i<MULTWCS; i++) {
	  char buf[64];
	  char w = !i ? NULL : '@'+i;
	  CoordSystem www = (CoordSystem)(WCS+i);

	  if (hasWCS(www)) {
	    char buff[64];
	    ptr->pix2wcs(v*(ptr->*matrixToImage)(), www, sky, format, buff,64);

	    int argc;
	    const char** argv;
	    Tcl_SplitList(interp, buff, &argc, &argv);

	    if (argc > 0 && argv && argv[0])
	      Tcl_SetVar2(interp,var,varcat(buf,"xWCS,",w),argv[0],0);
	    else
	      Tcl_SetVar2(interp,var,varcat(buf,"xWCS,",w),"",0);

	    if (argc > 1 && argv && argv[1])
	      Tcl_SetVar2(interp,var,varcat(buf,"yWCS,",w),argv[1],0);
	    else
	      Tcl_SetVar2(interp,var,varcat(buf,"yWCS,",w),"",0);

	    char* wcsname = (char*)ptr->getWCSName(www);
	    if (wcsname)
	      Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),wcsname,0);
	    else if (argc > 2 && argv && argv[2])
	      Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),argv[2],0);
	    else
	      Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),"",0);
	    
	    if (argc > 3 && argv && argv[3])
	      Tcl_SetVar2(interp,var,varcat(buf,"xUnitWCS,",w),argv[3],0);
	    else
	      Tcl_SetVar2(interp,var,varcat(buf,"xUnitWCS,",w),"",0);

	    if (argc > 4 && argv && argv[4])
	      Tcl_SetVar2(interp,var,varcat(buf,"yUnitWCS,",w),argv[4],0);
	    else
	      Tcl_SetVar2(interp,var,varcat(buf,"yUnitWCS,",w),"",0);

	    Tcl_Free((char*)argv);
	  }
	  else {
	    Tcl_SetVar2(interp,var,varcat(buf,"xWCS,",w),"",0);
	    Tcl_SetVar2(interp,var,varcat(buf,"yWCS,",w),"",0);
	    Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),"",0);
	    Tcl_SetVar2(interp,var,varcat(buf,"xUnitWCS,",w),"",0);
	    Tcl_SetVar2(interp,var,varcat(buf,"yUnitWCS,",w),"",0);
	  }
	}

	return;
      }
      ptr = ptr->nextMosaic();
    }
  }

  // else, return blanks
  Tcl_SetVar2(interp,var,"objectname","",0);
  Tcl_SetVar2(interp,var,"filename","",0);
  Tcl_SetVar2(interp,var,"min","",0);
  Tcl_SetVar2(interp,var,"max","",0);
  Tcl_SetVar2(interp,var,"low","",0);
  Tcl_SetVar2(interp,var,"high","",0);
  Tcl_SetVar2(interp,var,"value","",0);
  Tcl_SetVar2(interp,var,"xImage","",0);
  Tcl_SetVar2(interp,var,"yImage","",0);
  Tcl_SetVar2(interp,var,"xPhysical","",0);
  Tcl_SetVar2(interp,var,"yPhysical","",0);
  Tcl_SetVar2(interp,var,"xAmplifier","",0);
  Tcl_SetVar2(interp,var,"yAmplifier","",0);
  Tcl_SetVar2(interp,var,"xDetector","",0);
  Tcl_SetVar2(interp,var,"yDetector","",0);

  for (int i=0; i<MULTWCS; i++) {
    char buf[64];
    char w = !i ? NULL : '@'+i;
    Tcl_SetVar2(interp,var,varcat(buf,"xWCS,",w),"",0);
    Tcl_SetVar2(interp,var,varcat(buf,"yWCS,",w),"",0);
    Tcl_SetVar2(interp,var,varcat(buf,"sysWCS,",w),"",0);
    Tcl_SetVar2(interp,var,varcat(buf,"xUnitWCS,",w),"",0);
    Tcl_SetVar2(interp,var,varcat(buf,"yUnitWCS,",w),"",0);
  }
}

void FrameBase::getNANColorCmd()
{
  Tcl_AppendResult(interp, nanColorName, NULL);
}

void FrameBase::getValueCmd(const Vector& v, InternalSystem sys)
{
  Vector r;

  switch (sys) {
  case CANVAS:
    if (FitsImage* ptr=isInFits(v, &FitsImage::getCanvasToData, &r))
      Tcl_AppendResult(interp, ptr->getValue(r), NULL);
    break;
  case PANNER:
    if (FitsImage* ptr=isInFits(v, &FitsImage::getPannerToData, &r))
      Tcl_AppendResult(interp, ptr->getValue(r), NULL);
    break;
  }
}

void FrameBase::getWidthCmd()
{
  if (currentFits)
    printInteger(currentFits->width());
}

void FrameBase::getHeightCmd()
{
  if (currentFits)
    printInteger(currentFits->height());
}

void FrameBase::getBitpixCmd()
{
  if (currentFits)
    printInteger(currentFits->bitpix());
}

void FrameBase::highliteCmd(int which)
{
  useHighlite = which ? 1 : 0;
  update(PIXMAP);
}

void FrameBase::mosaicFastCmd(int which)
{
  mosaicFast = which ? 1 : 0;
}

void FrameBase::nanColorCmd(const char* color)
{
  if (nanColorName)
    delete [] nanColorName;
  nanColorName = dupstr(color);
  nanColor = getXColor(nanColorName);
  update(BASE);
}

void FrameBase::warpCmd(const Vector& v)
{
  Vector a=v; // can't use a const here
  XWARPPOINTER(display, None, None, 0, 0, 0, 0, (int)a[0], (int)a[1]);
}

void FrameBase::warpToCmd(const Vector& v)
{
  Vector a =v*canvasToWindow;
  XWARPPOINTER(display, None, Tk_WindowId(tkwin), 0,0,0,0,(int)a[0],(int)a[1]);
}

// Bin Commands

void FrameBase::binCmd(const Vector& b, const Vector& v,
		       const char* x, const char* y, const char* filter)
{
  binFactor_ = b;
  binDepth_ = 1;

  if (*channelFits) {
    (*channelFits)->setBinToFactor(b);
    (*channelFits)->setBinDepth(1);
    (*channelFits)->setBinX(x);
    (*channelFits)->setBinY(y);
    (*channelFits)->setBinFilter(filter);
    updateBin((*channelFits)->updateHist((*channelFits)->mapToRef(v,PHYSICAL)*refToUser));
  }
}

void FrameBase::binCmd(const Vector& b, const char* x, const char* y, 
		       const char* filter)
{
  binFactor_ = b;
  binDepth_ = 1;

  if (*channelFits) {
    (*channelFits)->setBinToFactor(b);
    (*channelFits)->setBinDepth(1);
    (*channelFits)->setBinX(x);
    (*channelFits)->setBinY(y);
    (*channelFits)->setBinFilter(filter);
    updateBin((*channelFits)->updateHistCenter());
  }
}

void FrameBase::binCmd(const Vector& b, int d, const Vector& lim, 
		       const Vector& v,
		       const char* x, const char* y, const char* z,
		       const char* filter)
{
  binFactor_ = b;
  binDepth_ = d;

  if (*channelFits) {
    (*channelFits)->setBinToFactor(b);
    (*channelFits)->setBinDepth(d);
    (*channelFits)->setBinX(x);
    (*channelFits)->setBinY(y);
    (*channelFits)->setBinZ(z);
    (*channelFits)->setBinFilter(filter);
    (*channelFits)->setBinZlim(lim);
    updateBin((*channelFits)->updateHist((*channelFits)->mapToRef(v,PHYSICAL)*refToUser));
  }
}

void FrameBase::binCmd(const Vector& b, int d, const Vector& lim,
		       const char* x, const char* y, const char* z,
		       const char* filter)
{
  binFactor_ = b;
  binDepth_ = d;

  if (*channelFits) {
    (*channelFits)->setBinToFactor(b);
    (*channelFits)->setBinDepth(d);
    (*channelFits)->setBinX(x);
    (*channelFits)->setBinY(y);
    (*channelFits)->setBinZ(z);
    (*channelFits)->setBinFilter(filter);
    (*channelFits)->setBinZlim(lim);
    updateBin((*channelFits)->updateHistCenter());
  }
}

void FrameBase::binAboutCmd(const Vector& v)
{
  if (*channelFits) {
    updateBin((*channelFits)->updateHist((*channelFits)->mapToRef(v,PHYSICAL)*refToUser));
  }
}

void FrameBase::binColsCmd(const char* x, const char* y, const char* z)
{
  if (*channelFits) {
    (*channelFits)->setBinX(x);
    (*channelFits)->setBinY(y);
    (*channelFits)->setBinZ(z);
    updateBin(currentFits->updateHistCursor());
  }
}

void FrameBase::binDepthCmd(int d)
{
  binDepth_ = d;
  if (*channelFits) {
    (*channelFits)->setBinDepth(d);
    updateBin(currentFits->updateHistCursor());
  }
}

void FrameBase::binFactorCmd(const Vector& b)
{
  Vector bb = b;
  binFactor_[0] *= bb[0];
  binFactor_[1] *= bb[1];
  if (*channelFits) {
    (*channelFits)->setBinFactor(b);
    updateBin(currentFits->updateHistCursor());
  }
}

void FrameBase::binFactorAboutCmd(const Vector& b, const Vector& v)
{
  Vector bb = b;
  binFactor_[0] *= bb[0];
  binFactor_[1] *= bb[1];
  if (*channelFits) {
    (*channelFits)->setBinFactor(b);
    updateBin((*channelFits)->updateHist((*channelFits)->mapToRef(v,PHYSICAL)*refToUser));
  }
}

void FrameBase::binFactorToCmd(const Vector& b)
{
  Vector bb = b;
  binFactor_[0] = bb[0];
  binFactor_[1] = bb[1];
  if (*channelFits) {
    (*channelFits)->setBinToFactor(b);
    updateBin(currentFits->updateHistCursor());
  }
}

void FrameBase::binFactorToAboutCmd(const Vector& b, const Vector& v)
{
  Vector bb = b;
  binFactor_[0] = bb[0];
  binFactor_[1] = bb[1];
  if (*channelFits) {
    (*channelFits)->setBinToFactor(b);
    updateBin((*channelFits)->updateHist((*channelFits)->mapToRef(v,PHYSICAL)*refToUser));
  }
}

void FrameBase::binToFitCmd()
{
  if (*channelFits) {
    double bf = 1/calcZoom(Vector(options->width,options->height),
			   (*channelFits)->getHistDim());

    // round up to next power of 2
    if (bf < 1) {
      (*channelFits)->setBinToFactor(Vector(1,1));
      binFactor_ = Vector(1,1);
    }
    else {
      int p=1;
      while (p<bf)
	p*=2;
      (*channelFits)->setBinToFactor(Vector(p,p));
      binFactor_ = Vector(p,p);
    }
  
    updateBin(currentFits->updateHistCursor());
  }
}

void FrameBase::binFunctionCmd(FitsHist::Function func)
{
  binFunction_ = func;
  if (*channelFits) {
    (*channelFits)->setBinFunction(func);
    updateBin(currentFits->updateHistCursor());
  }
}

void FrameBase::binBufferSizeCmd(int size)
{
  binBufferSize_ = size;
  if (*channelFits) {
    (*channelFits)->setBinBufferSize(size);
    updateBin(currentFits->updateHistCursor());
  }
}

void FrameBase::binFilterCmd(const char* filter)
{
  if (*channelFits) {
    (*channelFits)->setBinFilter(filter);
    updateBin(currentFits->updateHistCursor());
  }
}

void FrameBase::getBinCursorCmd()
{
  if (*channelFits)
    printVector((*channelFits)->getHistCursor(), DEFAULT);
}

void FrameBase::getBinColsCmd()
{
  if (*channelFits && (*channelFits)->isHist())
    if ((*channelFits)->binDepth()>1)
      Tcl_AppendResult(interp, (*channelFits)->getHistX(), " ", 
		       (*channelFits)->getHistY(), " ",
		       (*channelFits)->getHistZ(), NULL);
    else
      Tcl_AppendResult(interp, (*channelFits)->getHistX(), " ", 
		       (*channelFits)->getHistY(), NULL);

  else
    Tcl_AppendResult(interp, "", NULL);
}

void FrameBase::getBinColsMinMaxCmd(const char* col)
{
  if (*channelFits)
    printVector((*channelFits)->getHistColMinMax(col), DEFAULT);
}

void FrameBase::getBinDepthCmd()
{
  if (*channelFits)
    binDepth_ = (*channelFits)->binDepth();
 
  printDouble(binDepth_, DEFAULT);
}

void FrameBase::getBinFactorCmd()
{
  if (*channelFits)
    binFactor_ = (*channelFits)->binFactor();

  printVector(binFactor_, DEFAULT);
}

void FrameBase::getBinFunctionCmd()
{
  if (*channelFits)
    binFunction_ = (*channelFits)->binFunction();

  switch (binFunction_) {
  case FitsHist::AVERAGE:
    Tcl_AppendResult(interp, "average", NULL);
    return;
  case FitsHist::SUM:
    Tcl_AppendResult(interp, "sum", NULL);
    return;
  }
}

void FrameBase::getBinBufferSizeCmd()
{
  if (*channelFits)
    binBufferSize_ = (*channelFits)->binBufferSize();

  printInteger(binBufferSize_);
}

void FrameBase::getBinFilterCmd()
{
  if (*channelFits && (*channelFits)->isHist())
    Tcl_AppendResult(interp, (*channelFits)->getHistFilter(), NULL);
  else
    Tcl_AppendResult(interp, "", NULL);
}

void FrameBase::getBinListCmd()
{
  if (*channelFits && (*channelFits)->isHist()) {
    char* cols = (*channelFits)->getHistList();
    Tcl_AppendResult(interp, cols, NULL);
    delete cols;
  }
  else
    Tcl_AppendResult(interp, "", NULL);
}

void FrameBase::hasBinCmd()
{
  if (*channelFits && (*channelFits)->isHist())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

// Clip Commands

void FrameBase::clipMinMaxModeCmd(FrScale::MinMaxMode m)
{
  currentScale->setMMMode(m);
  resetClip(1);
  update(BASE);
}

void FrameBase::clipMinMaxParamCmd(int i)
{
  currentScale->setMMIncr(i);
  resetClip(1);
  update(BASE);
}

void FrameBase::clipModeCmd(float per)
{
  if (per == 100)
    currentScale->setClipMode(FrScale::MINMAX);
  else {
    currentScale->setAutoCutPer(per);
    currentScale->setClipMode(FrScale::AUTOCUT);
  }
  updateClip(1);
  update(BASE);
}

void FrameBase::clipModeCmd(FrScale::ClipMode m)
{
  currentScale->setClipMode(m);
  updateClip(1);
  update(BASE);
}

void FrameBase::clipPreserveCmd(int r)
{
  currentScale->setPreserve(r);
}

void FrameBase::clipScopeCmd(FrScale::ClipScope s)
{
  currentScale->setClipScope(s);
  updateClip(1);
  update(BASE);
}

void FrameBase::clipUserCmd(double l, double h)
{
  currentScale->setULow(l);
  currentScale->setUHigh(h);
  updateClip(1);
  update(BASE);
}

void FrameBase::clipUserLowCmd(double l)
{
  currentScale->setULow(l);
  updateClip(1);
  update(BASE);
}

void FrameBase::clipUserHighCmd(double h)
{
  currentScale->setUHigh(h);
  updateClip(1);
  update(BASE);
}

void FrameBase::clipZScaleParamCmd(float c, int s, int l)
{
  currentScale->setZContrast(c);
  currentScale->setZSampleSize(s);
  currentScale->setZSampleLine(l);
  updateClip(1);
  update(BASE);
}

void FrameBase::getClipCmd()
{
  printVector(Vector(currentScale->low(), currentScale->high()), DEFAULT);
}

void FrameBase::getClipCmd(float per)
{
  FrScale::ClipMode cm = (per == 100) ? FrScale::MINMAX : FrScale::AUTOCUT;
  float ac = per;

  ostringstream str;
  str << getClip(cm, ac) << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void FrameBase::getClipCmd(FrScale::ClipMode cm)
{
  ostringstream str;
  str << getClip(cm, currentScale->autoCutPer()) << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void FrameBase::getClipMinMaxModeCmd()
{
  switch (currentScale->mmMode()) {
  case FrScale::AUTOSCAN:
    Tcl_AppendResult(interp, "auto", NULL);
    return;
  case FrScale::SCAN:
    Tcl_AppendResult(interp, "scan", NULL);
    return;
  case FrScale::SAMPLE:
    Tcl_AppendResult(interp, "sample", NULL);
    return;
  case FrScale::DATAMIN:
    Tcl_AppendResult(interp, "datamin", NULL);
    return;
  case FrScale::IRAFMIN:
    Tcl_AppendResult(interp, "irafmin", NULL);
    return;
  }
}

void FrameBase::getClipMinMaxSampleCmd()
{
  printInteger(currentScale->mmIncr());
}

void FrameBase::getClipModeCmd()
{
  switch (currentScale->clipMode()) {
  case FrScale::MINMAX:
    Tcl_AppendResult(interp, "minmax", NULL);
    break;
  case FrScale::ZSCALE:
    Tcl_AppendResult(interp, "zscale", NULL);
    break;
  case FrScale::ZMAX:
    Tcl_AppendResult(interp, "zmax", NULL);
    break;
  case FrScale::AUTOCUT:
    printDouble(currentScale->autoCutPer(), DEFAULT);
    break;
  case FrScale::USERCLIP:
    Tcl_AppendResult(interp, "user", NULL);
    break;
  }
}

void FrameBase::getClipPreserveCmd()
{
  if (currentScale->preserve())
      Tcl_AppendResult(interp, "1", NULL);
  else
      Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::getClipScopeCmd()
{
  switch (currentScale->clipScope()) {
  case FrScale::GLOBAL:
    Tcl_AppendResult(interp, "global", NULL);
    break;
  case FrScale::LOCAL:
    Tcl_AppendResult(interp, "local", NULL);
    break;
  }
}

void FrameBase::getClipUserCmd()
{
  printVector(Vector(currentScale->uLow(), currentScale->uHigh()), DEFAULT);
}

void FrameBase::getClipZScaleParamCmd()
{
  ostringstream str;
  str << currentScale->zContrast() << ' ' 
      << currentScale->zSampleSize() << ' ' 
      << currentScale->zSampleLine() << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void FrameBase::getHistogramCmd(char* xName, char* yName)
{
  bltHist(xName,yName);
  printDouble((currentScale->max()-currentScale->min())/256., DEFAULT);
}

// Min Max

void FrameBase::getMinMaxCmd()
{
  printVector(Vector(currentScale->min(), currentScale->max()), DEFAULT);
}

void FrameBase::hasDATAMINCmd()
{
  if (currentFits && currentFits->hasDATAMIN())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasIRAFMINCmd()
{
  if (currentFits && currentFits->hasIRAFMIN())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

// Crosshair Commands

void FrameBase::crosshairCmd(int which)
{
  useCrosshair = which ? 1 : 0;
  update(PIXMAP);
}

void FrameBase::crosshairCmd(const Vector& v, InternalSystem sys)
{
  useCrosshair = 1;
  crosshair = mapToRef(v, sys);
  update(PIXMAP);
}

void FrameBase::crosshairCmd(const Vector& v, CoordSystem sys, SkyFrame sky)
{
  useCrosshair = 1;
  if (currentFits)
    crosshair = currentFits->mapToRef(v, sys, sky);
  update(PIXMAP);
}

void FrameBase::crosshairWarpCmd(const Vector& v)
{
  useCrosshair = 1;

  Vector r = crosshair * refToCanvas;
  r += v;
  crosshair = r * canvasToRef;

  update(PIXMAP);
  updateMagnifier();
}

void FrameBase::crosshairBeginCmd(const Vector& v, InternalSystem sys)
{
  // This routine is optimalized for rendering an interactive cursor. First,
  // a cursorless pixmap is generated. Then as the cursor is moved about, the 
  // cursor is written directly to the display, and erase by XCopyArea from the
  // widget pixmap.

  // turn cursor off and generate pixmap
  useCrosshair = 0;
  updateNow(PIXMAP);

  // turn cursor on, and draw directly to display
  useCrosshair = 1;
  crosshair = mapToRef(v, sys);
  drawCrosshair();
}

void FrameBase::crosshairBeginCmd(const Vector& v, CoordSystem sys)
{
  // This routine is optimalized for rendering an interactive cursor. First,
  // a cursorless pixmap is generated. Then as the cursor is moved about, the 
  // cursor is written directly to the display, and erase by XCopyArea from the
  // widget pixmap.

  // turn cursor off and generate pixmap
  useCrosshair = 0;
  updateNow(PIXMAP);

  // turn cursor on, and draw directly to display
  useCrosshair = 1;
  if (currentFits)
    crosshair = currentFits->mapToRef(v, sys, FK5);
  drawCrosshair();
}

void FrameBase::crosshairMotionCmd(const Vector& v, InternalSystem sys)
{
  eraseCrosshair();
  crosshair = mapToRef(v, sys);
  drawCrosshair();
}

void FrameBase::crosshairMotionCmd(const Vector& v, CoordSystem sys)
{
  eraseCrosshair();
  if (currentFits)
    crosshair = currentFits->mapToRef(v, sys, FK5);
  drawCrosshair();
}

void FrameBase::getCrosshairCmd(InternalSystem sys)
{
  printVector(mapFromRef(crosshair, sys), DEFAULT);
}

void FrameBase::getCrosshairCmd(CoordSystem sys, SkyFrame sky, SkyFormat format,
			    Precision p)
{
  if (currentFits)
    printFromRef(currentFits, crosshair, sys, sky, format, p);
  else
    printVector(Vector(), DEFAULT);
}

void FrameBase::getCrosshairStatusCmd()
{
  if (useCrosshair)
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::getHorzCutCmd(char* x, char* y, const Vector& v, 
			      InternalSystem in)
{
  switch (in) {
  case CANVAS:
    bltCut(x, y, XX, v * canvasToWidget);
    break;
  case PANNER:
    bltCut(x, y, XX, v * pannerToWidget);
    break;
  }
}

void FrameBase::getVertCutCmd(char* x, char* y, const Vector& v,
			      InternalSystem in)
{
  switch (in) {
  case CANVAS:
    bltCut(x, y, YY, v * canvasToWidget);
    break;
  case PANNER:
    bltCut(x, y, YY, v * pannerToWidget);
    break;
  }
}

// DATASEC Commands

void FrameBase::DATASECCmd(int which)
{
  switch (currentScale->scanMode()) {
  case FrScale::NODATASEC:
  case FrScale::DATASEC:
  case FrScale::RESET:
    currentScale->setScanMode(which ? FrScale::DATASEC : FrScale::NODATASEC);
    break;
  case FrScale::UNODATASEC:
  case FrScale::UDATASEC:
    currentScale->setScanMode(which ? FrScale::UDATASEC : FrScale::UNODATASEC);
    break;
  }

  updateWCSbb();
  resetClip(1);

 // imageBBox may change
  update(MATRIX);
}

void FrameBase::getDATASECCmd()
{
  if (currentScale->scanMode())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasDATASECCmd()
{
  if (currentFits && currentFits->hasDATASEC())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

// Fits Commands

void FrameBase::getFitsExtCountCmd()
{
  printInteger(*currentMosaicCount);
}

void FrameBase::getFitsDepthCmd()
{
  if (isCube())
    printInteger(*currentSliceCount);
  else
    printInteger(1);
}

void FrameBase::getFitsHeaderCmd(int which)
{
  FitsImage* ptr = *channelFits;
  if (which > 0) {
    for (int i=0; i<(which-1); i++) {
      if (ptr)
	ptr = ptr->nextMosaic();
    }
  }
  else if (which < 0) {
    for (int i=0; i<((-which)-1); i++) {
      if (ptr)
	ptr = ptr->nextMosaic();
    }
  }

  if (ptr) {
    char* h = (which > 0) ? ptr->displayHeader() : ptr->displayPrimary();
    Tcl_AppendResult(interp, h, NULL);
    delete h;
  }
  else
    result = TCL_ERROR;
}

void FrameBase::getFitsHeaderKeywordCmd(int which, const char* key)
{
  FitsImage* ptr = *channelFits;
  for (int i=0; i<(which-1); i++)
    if (ptr)
      ptr = ptr->nextMosaic();

  if (ptr) {
    char* value = ptr->getKeyword(key);
    if (value) {
      Tcl_AppendResult(interp, value, NULL);
      delete [] value;
      return;
    }
  }

  result = TCL_ERROR;
}

void FrameBase::getFitsFileNameCmd(FileNameType type)
{
  if (currentFits) {
    switch (type) {
    case ROOTBASE:
      Tcl_AppendResult(interp, currentFits->getRootBaseFileName(), NULL);
      break;
    case FULLBASE:
      Tcl_AppendResult(interp, currentFits->getFullBaseFileName(), NULL);
      break;
    case ROOT:
      Tcl_AppendResult(interp, currentFits->getRootFileName(), NULL);
      break;
    case FULL:
      Tcl_AppendResult(interp, currentFits->getFullFileName(), NULL);
      break;
    }
  }
}

void FrameBase::getFitsFileNameCmd(int which, FileNameType type)
{
  if (which > 0) {
    FitsImage* ptr = *channelFits;
    for (int i=0; i<(which-1); i++) {
      if (ptr)
	ptr = ptr->nextMosaic();
    }

    if (ptr) {
      switch (type) {
      case ROOTBASE:
	Tcl_AppendResult(interp, ptr->getRootBaseFileName(), NULL);
	break;
      case FULLBASE:
	Tcl_AppendResult(interp, ptr->getFullBaseFileName(), NULL);
	break;
      case ROOT:
	Tcl_AppendResult(interp, ptr->getRootFileName(), NULL);
	break;
      case FULL:
	Tcl_AppendResult(interp, ptr->getFullFileName(), NULL);
	break;
      }
    }
  }
  else {
    Tcl_AppendResult(interp, " file index must be greater than zero.", NULL);
    result = TCL_ERROR;
  }
}

void FrameBase::getFitsFileNameCmd(const Vector& v, InternalSystem sys,
			       FileNameType type)
{
  switch (sys) {
  case CANVAS:
    if (FitsImage* ptr = isInFits(v, &FitsImage::getCanvasToData, NULL)) {
      switch (type) {
      case ROOTBASE:
	Tcl_AppendResult(interp, ptr->getRootBaseFileName(), NULL);
	break;
      case FULLBASE:
	Tcl_AppendResult(interp, ptr->getFullBaseFileName(), NULL);
	break;
      case ROOT:
	Tcl_AppendResult(interp, ptr->getRootFileName(), NULL);
	break;
      case FULL:
	Tcl_AppendResult(interp, ptr->getFullFileName(), NULL);
	break;
      }
    }
    break;
  case PANNER:
    if (FitsImage* ptr = isInFits(v, &FitsImage::getPannerToData, NULL)) {
      switch (type) {
      case ROOTBASE:
	Tcl_AppendResult(interp, ptr->getRootBaseFileName(), NULL);
	break;
      case FULLBASE:
	Tcl_AppendResult(interp, ptr->getFullBaseFileName(), NULL);
	break;
      case ROOT:
	Tcl_AppendResult(interp, ptr->getRootFileName(), NULL);
	break;
      case FULL:
	Tcl_AppendResult(interp, ptr->getFullFileName(), NULL);
	break;
      }
    }
    break;
  }
}


void FrameBase::getFitsObjectNameCmd(int which)
{
  if (which > 0) {
    FitsImage* ptr = *channelFits;
    for (int i=0; i<(which-1); i++) {
      if (ptr)
	ptr = ptr->nextMosaic();
    }

    if (ptr)
      Tcl_AppendResult(interp, ptr->getObjectName(), NULL);
    else {
      ostringstream str;
      str << which << ends;
      Tcl_AppendResult(interp, " file ",str.str().c_str()," not loaded.",NULL);
      result = TCL_ERROR;
    }
  }
  else {
    Tcl_AppendResult(interp, " file index must be greater than zero.", NULL);
    result = TCL_ERROR;
  }
}

void FrameBase::getFitsMasterCmd()
{
  if (currentFits) {
    ostringstream str;
    str << currentFits->getFullFileName() << ' ' 
	<< currentFits->fitsFile() << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
}

void FrameBase::getFitsSliceCmd()
{
  if (*channelFits)
    printInteger(slice);
  else
    printInteger(1);
}

void FrameBase::hasFitsCmd()
{
  if (*channelFits)
      Tcl_AppendResult(interp, "1", NULL);
  else
      Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasFitsCubeCmd()
{
  if (isCube())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasFitsExtCmd(int which)
{
  if (which > 0) {
    FitsImage* ptr = *channelFits;
    for (int i=0; i<(which-1); i++) {
      if (ptr)
	ptr = ptr->nextMosaic();
    }

    if (ptr && (ptr->ext() > 0)) {
      Tcl_AppendResult(interp, "1", NULL);
      return;
    }
  }

  Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasFitsImageCmd()
{
  if (currentFits && currentFits->isImage())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasFitsMosaicCmd()
{
  if (isMosaic())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasFitsTableCmd()
{
  if (currentFits && currentFits->isTable())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::loadIncrDataCmd(int which, int x0, int y0, int x1, int y1)
{
  FitsImage* ptr = *channelFits;
  if (which > 0) {
    for (int i=0; i<(which-1); i++) {
      if (ptr)
	ptr = ptr->nextMosaic();
    }
  }

  if (ptr)
    ptr->setDataParams(x0,y0,x1,y1);
}

void FrameBase::loadIncrMinMaxCmd(int which, int x0, int y0, int x1, int y1)
{
  FitsImage* ptr = *channelFits;
  if (which > 0) {
    for (int i=0; i<(which-1); i++) {
      if (ptr)
	ptr = ptr->nextMosaic();
    }
  }

  if (ptr)
    ptr->setMinMaxParams(x0,y0,x1,y1);

  resetClip(1);
}

void FrameBase::loadIncrEndCmd()
{
  switch (currentScale->scanMode()) {
  case FrScale::NODATASEC:
  case FrScale::UNODATASEC:
  case FrScale::RESET:
    currentScale->setScanMode(FrScale::NODATASEC);
    break;
  case FrScale::DATASEC:
  case FrScale::UDATASEC:
    currentScale->setScanMode(FrScale::DATASEC);
    break;
  }

  resetClip(1);
  update(MATRIX);
}

void FrameBase::updateFitsCmd(int now)
{
  if (now)
    updateNow(BASE);
  else
    update(BASE);
}

void FrameBase::updateFitsCmd(int which, BBox bb, int now)
{
  // Note: bb is in IMAGE coords
  FitsImage* ptr = *channelFits;
  if (which > 0) {
    for (int i=0; i<(which-1); i++) {
      if (ptr)
	ptr = ptr->nextMosaic();
    }
  }

  if (ptr) {
    Vector ll = bb.ll * ptr->getImageToCanvas();
    Vector lr = bb.lr() * ptr->getImageToCanvas();
    Vector ur = bb.ur * ptr->getImageToCanvas();
    Vector ul = bb.ul() * ptr->getImageToCanvas();
    BBox rr(ll);
    rr.bound(lr);
    rr.bound(ur);
    rr.bound(ul);

    if (now)
      updateNow(BASE, rr);
    else
      update(BASE, rr);
  }
}

void FrameBase::unloadFitsCmd()
{
  unloadAllFits();
  resetValues();
  update(MATRIX);
}

// Fitsy Commands

void FrameBase::fitsyHasExtCmd(const char* fn)
{
  // verify that we have an ext specified
  if (fn && (fn[strlen(fn)-1] != ']')) {
    Tcl_AppendResult(interp, "0", NULL);
    return;
  }

  FitsFile* ext = new FitsFitsMMap(fn, FitsFile::EXACT);
  if (ext->isValid())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);

  delete ext;
  return;
}

// IIS Commands

void FrameBase::iisMessageCmd(const char* ptr)
{
  if (!currentFits)
    return;

  // note: the file name is the second value passed
  char fn[PATH_MAX];
  string x(ptr);
  istringstream str(x);
  str >> fn >> fn;
  currentFits->setFileName(fn);
}

void FrameBase::iisSetCursorCmd(const Vector& v, InternalSystem sys)
{
  // assume canvas
  if (!currentFits)
    return;

  iisLastCursor = v * canvasToWidget;

  Vector r = iisLastCursor * widgetToWindow;
  XWARPPOINTER(display, None, Tk_WindowId(tkwin), 0,0,0,0,(int)r[0],(int)r[1]);
}

void FrameBase::iisSetCursorCmd(const Vector& v, CoordSystem sys)
{
  if (!currentFits)
    return;

  iisLastCursor = currentFits->mapToRef(v,sys) * refToWidget;

  Vector r = iisLastCursor * widgetToWindow;
  XWARPPOINTER(display, None, Tk_WindowId(tkwin), 0,0,0,0,(int)r[0],(int)r[1]);
}

void FrameBase::iisGetCursorCmd()
{
  printVector(iisLastCursor,DEFAULT);
}

void FrameBase::iisCursorModeCmd(int state)
{
  if (state) {
    // save cursor pos
    Window root, child;
    int rootx,rooty,winx,winy;
    unsigned int mask;
    XQueryPointer(display,Tk_WindowId(tkwin),&root,&child,
		  &rootx,&rooty,&winx,&winy,&mask);

    if (!iisInteractive) {
      iisSaveRoot = root;
      iisSaveCursor = Vector(rootx,rooty);
    }
    iisInteractive++;

    // set iisLastCursor if first time thru
    if (iisLastCursor[0] == 0 && iisLastCursor[1] == 0)
      iisLastCursor = Vector(options->width,options->height)/2;

    // and move to last cursor position
    Vector r = iisLastCursor * widgetToWindow;
    XWARPPOINTER(display, None,Tk_WindowId(tkwin),0,0,0,0,(int)r[0],(int)r[1]);
  }
  else {
    // restore cursor pos
    if (!(--iisInteractive)) {
      XWARPPOINTER (display, None, iisSaveRoot, 0, 0, 0, 0, 
		    (int)iisSaveCursor[0], (int)iisSaveCursor[1]);
      iisSaveRoot = 0;
      iisSaveCursor = Vector();
    }
  }
}

void FrameBase::hasIISCmd()
{
  if (isIIS())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::iisGetFileNameCmd()
{
  FitsImage* ptr = *channelFits;
  while (ptr && ptr->nextMosaic())
    ptr = ptr->nextMosaic();

  if (ptr)
    Tcl_AppendResult(interp, ptr->iisGetFileName(), NULL);
}

void FrameBase::iisGetFileNameCmd(int which)
{
  FitsImage* ptr = *channelFits;
  for (int i=0; i<(which-1); i++) {
    if (ptr)
      ptr = ptr->nextMosaic();
  }

  if (ptr)
    Tcl_AppendResult(interp, ptr->iisGetFileName(), NULL);
}

void FrameBase::iisGetFileNameCmd(const Vector& v)
{
  if (FitsImage* ptr = isInFits(v, &FitsImage::getCanvasToData, NULL))
    Tcl_AppendResult(interp, ptr->iisGetFileName(), NULL);
}

void FrameBase::iisSetFileNameCmd(const char* fn)
{
  FitsImage* ptr = *channelFits;
  while (ptr && ptr->nextMosaic())
    ptr = ptr->nextMosaic();

  if (ptr)
    ptr->iisSetFileName(fn);
}

void FrameBase::iisSetFileNameCmd(const char* fn, int which)
{
  FitsImage* ptr = *channelFits;
  for (int i=0; i<(which-1); i++) {
    if (ptr)
      ptr = ptr->nextMosaic();
  }

  if (ptr)
    ptr->iisSetFileName(fn);
}

// Coordinate Commands

void FrameBase::getWCSNameCmd(CoordSystem sys)
{
  if (currentFits && currentFits->hasWCS(sys)) {
    char* wcsname = (char*)currentFits->getWCSName(sys);
    if (wcsname) {
      Tcl_AppendResult(interp, wcsname, NULL);
      return;
    }
  }
  Tcl_AppendResult(interp, NULL);
}

void FrameBase::hasSystemCmd(CoordSystem sys)
{
  switch (sys) {
  case IMAGE:
    Tcl_AppendResult(interp, "1", NULL);
    return;
  case PHYSICAL:
    hasPhysicalCmd();
    return;
  case AMPLIFIER:
    hasAmplifierCmd();
    return;
  case DETECTOR:
    hasDetectorCmd();
    return;
  default:
    hasWCSCmd(sys);
    return;
  }
}

void FrameBase::hasPhysicalCmd()
{
  if (hasLTMV())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasDetectorCmd()
{
  if (hasDTMV())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasAmplifierCmd()
{
  if (hasATMV())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasWCSCmd(CoordSystem sys)
{
  if (hasWCS(sys))
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasWCSEquCmd(CoordSystem sys)
{
  if (hasWCSEqu(sys))
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::hasWCSLinearCmd(CoordSystem sys)
{
  if (hasWCSLinear(sys))
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

// Pan Zoom Commands

void FrameBase::centerCmd()
{
  centerImage();
  update(MATRIX);
}

void FrameBase::getOrientCmd()
{
  switch (orientation) {
  case NORMAL:
    Tcl_AppendResult(interp, "none", NULL);
    return;
  case XX:
    Tcl_AppendResult(interp, "x", NULL);
    return;
  case YY:
    Tcl_AppendResult(interp, "y", NULL);
    return;
  case XY:
    Tcl_AppendResult(interp, "xy", NULL);
    return;
  }
}

void FrameBase::getPanPreserveCmd()
{
  if (preservePan)
      Tcl_AppendResult(interp, "1", NULL);
  else
      Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::getRotateCmd(Precision p)
{
  printDouble(radToDeg(rotation), p);
}

void FrameBase::getWCSAlignCmd()
{
  Tcl_AppendResult(interp, (wcsAlignment ? "1" : "0"), " ", 
		   coordSystemStr_[wcsSystem-IMAGE], " ",
		   skyFrameStr_[wcsSky-FK4], NULL);
}

void FrameBase::getZoomCmd(Precision p)
{
  printVector(zoom_, p);
}

void FrameBase::orientCmd(Orientation which)
{
  orientation = which;

  switch (orientation) {
  case NORMAL:
    orientationMatrix.identity();
    break;
  case XX:
    orientationMatrix = FlipX();
    break;
  case YY:
    orientationMatrix = FlipY();
    break;
  case XY:
    orientationMatrix = FlipXY();
    break;
  }

  update(MATRIX);
}

void FrameBase::panCmd(const Vector& v1, const Vector& v2)
{
  if (!currentFits)
    return;

  Vector start = v1 * canvasToUser;
  Vector stop = v2 * canvasToUser;
  cursor -= stop - start;
  update(MATRIX);
}

void FrameBase::panCmd(const Vector& v, InternalSystem sys)
{
  if (!currentFits)
    return;

  switch (sys) {
  case CANVAS:
    {
      Vector u = cursor * userToCanvas;
      u += v;
      cursor = u * canvasToUser;
    }
    break;
  case PANNER: 
    {
      Vector u = cursor * userToPanner;
      u += v;
      cursor = u * pannerToUser;
    }
    break;
  }

  setBinCursor(cursor);
  update(MATRIX);
}

void FrameBase::panCmd(const Vector& v, CoordSystem sys, SkyFrame sky)
{
  if (!currentFits)
    return;

  Vector u = currentFits->mapFromRef(cursor * userToRef, sys, sky);
  u += v;
  cursor = currentFits->mapToRef(u, sys, sky) * refToUser;
  
  setBinCursor(cursor);
  update(MATRIX);
}

void FrameBase::panBBoxCmd(const Vector& v)
{
  if (!currentFits)
    return;

  // we want to round to nearest pixel center
  Vector r = v * currentFits->getPannerToData();
  cursor = (r.floor() + Vector(.5,.5)) * currentFits->getDataToUser();

  setBinCursor(cursor);
  update(MATRIX);
}

void FrameBase::panMotionBeginCmd(const Vector& v)
{
  if (!currentFits)
    return;

  // save the current cursor
  panCursor = v;

  // copy tmp pixmap
  panPM = Tk_GetPixmap(display, Tk_WindowId(tkwin),
		       options->width, options->height, depth);
  if (!panPM) {
    cerr << "Frame Internal Error: Unable to Create Pan Motion Pixmap" << endl;
    exit(1);
  }
  XCopyArea(display, pixmap, panPM, gc, 0, 0, options->width, 
	    options->height, 0,0);
}

void FrameBase::panMotionCmd(const Vector& v)
{
  if (!currentFits)
    return;

  if (visible) {

    // Clear
    Vector diff = ((v * canvasToWidget) - (panCursor * canvasToWidget));

    BBox hh,vv;
    if (diff[0]>0 && diff[1]>0) {
      hh = BBox(Vector(0,0), Vector(options->width, diff[1]));
      vv = BBox(Vector(0,0), Vector(diff[0], options->height));
    } else if (diff[0]>0 && diff[1]<0) {
      hh = BBox(Vector(options->width,options->height), 
		Vector(0,options->height+diff[1]));
      vv = BBox(Vector(0,0), Vector(diff[0], options->height));
    } else if (diff[0]<0 && diff[1]>0) {
      hh = BBox(Vector(0,0), Vector(options->width, diff[1]));
      vv = BBox(Vector(options->width,options->height), 
		Vector(options->width+diff[0], 0));
    } else if (diff[0]<0 && diff[1]<0) {
      hh = BBox(Vector(options->width,options->height), 
		Vector(0,options->height+diff[1]));
      vv = BBox(Vector(options->width,options->height), 
		Vector(options->width+diff[0], 0));
    }

    hh = hh * widgetToWindow;
    vv = vv * widgetToWindow;
    
    XSetForeground(display, gc, bgColor->pixel);

    Vector hs = hh.size();
    XFillRectangle(display, Tk_WindowId(tkwin), gc, 
		   (int)hh.ll[0], (int)hh.ll[1], (int)hs[0], (int)hs[1]);

    Vector vs = vv.size();
    XFillRectangle(display, Tk_WindowId(tkwin), gc, 
		   (int)vv.ll[0], (int)vv.ll[1], (int)vs[0], (int)vs[1]);

    // display tmp pixmap at new location
    Vector d = ((v * canvasToWidget) - (panCursor * canvasToWidget)) * 
      widgetToWindow;
    XCopyArea(display, panPM, Tk_WindowId(tkwin), panGCXOR, 
	      0, 0, options->width, options->height, (int)d[0], (int)d[1]);
  }
}

void FrameBase::panMotionEndCmd(const Vector& v)
{
  if (!currentFits)
    return;

  // delete tmp pixmap
  if (panPM)
    Tk_FreePixmap(display, panPM);
  panPM = 0;

  Vector start = panCursor * canvasToUser;
  Vector stop = v * canvasToUser;
  cursor -= stop - start;

  setBinCursor(cursor);
  update(MATRIX);
}

void FrameBase::panToCmd(const Vector& v, InternalSystem sys)
{
  if (!currentFits)
    return;

  switch (sys) {
  case CANVAS:
    cursor = v * canvasToUser;
    break;
  case PANNER: 
    cursor = v * pannerToUser;
    break;
  }

  setBinCursor(cursor);
  update(MATRIX);
}

void FrameBase::panToCmd(const Vector& v, CoordSystem sys, SkyFrame sky)
{
  if (!currentFits)
    return;

  cursor = currentFits->mapToRef(v, sys, sky) * refToUser;

  setBinCursor(cursor);
  update(MATRIX);
}

void FrameBase::rotateCmd(double r)
{
  rotation += r;
  update(MATRIX);
}

void FrameBase::rotateToCmd(double r)
{
  rotation = r;
  update(MATRIX);
}

void FrameBase::rotateMotionBeginCmd()
{
  // save the current rotation
  rotateRotation = rotation;

  // Create src XImage
  if (!(rotateSrcXM = XGETIMAGE(display, pixmap, 0, 0,
				options->width, options->height, 
				AllPlanes, ZPixmap))) {
    cerr << "Frame Internal Error: Unable to Create Rotate XImage" << endl;
    exit(1);
  }

  // Create dest XImage
  if (!(rotateDestXM = XGETIMAGE(display, pixmap, 0, 0,
				 options->width, options->height, 
				 AllPlanes, ZPixmap))) {
    cerr << "Frame Internal Error: Unable to Create Rotate XImage" << endl;
    exit(1);
  }

  // Create dest Pixmap
  rotatePM = Tk_GetPixmap(display, Tk_WindowId(tkwin), 
			  options->width, options->height, depth);
  if (!rotatePM) {
    cerr << "Frame Internal Error: Unable to Create Rotate Motion Pixmap" 
	 << endl;
    exit(1);
  }
}

void FrameBase::rotateMotionCmd(double angle)
{
  rotation = rotateRotation + angle;
  rotateMotion();
}

void FrameBase::rotateMotionEndCmd()
{
  // Clean up
  if (rotateSrcXM)
    XDestroyImage(rotateSrcXM);
 
  if (rotateDestXM)
    XDestroyImage(rotateDestXM);

  if (rotatePM)
    Tk_FreePixmap(display, rotatePM);

  update(MATRIX);
}

void FrameBase::wcsAlignCmd(int which, CoordSystem sys, SkyFrame sky)
{
  wcsAlignment = which;
  wcsSystem = sys;
  wcsSky = sky;

  align();
  update(MATRIX);
}

void FrameBase::zoomCmd(const Vector& z)
{
  zoom_ *= ((Vector&)z).abs();
  update(MATRIX);
}

void FrameBase::zoomAboutCmd(const Vector& z, const Vector& v, 
			     InternalSystem sys)
{
  // assume canvas coords
  zoom_ *= ((Vector&)z).abs();

  if (currentFits) {
    switch (sys) {
    case CANVAS:
      cursor = v * canvasToUser;
      break;
    case PANNER:
      cursor = v * pannerToUser;
      break;
    }
    setBinCursor(cursor);
  }

  update(MATRIX);
}

void FrameBase::zoomAboutCmd(const Vector& z, const Vector& v, 
			     CoordSystem sys, SkyFrame sky)
{
  zoom_ *= ((Vector&)z).abs();

  if (currentFits) {
    cursor = currentFits->mapToRef(v, sys, sky) * refToUser;
    setBinCursor(cursor);
  }

  update(MATRIX);
}

void FrameBase::zoomToCmd(const Vector& z)
{
  zoom_ = ((Vector&)z).abs();
  update(MATRIX);
}

void FrameBase::zoomToAboutCmd(const Vector& z, const Vector& v,
			       InternalSystem sys)
{
  // assume canvas coords
  zoom_ = ((Vector&)z).abs();

  if (!currentFits) {
    switch (sys) {
    case CANVAS:
      cursor = v * canvasToUser;
      break;
    case PANNER:
      cursor = v * pannerToUser;
      break;
    }
    setBinCursor(cursor);
  }

  update(MATRIX);
}

void FrameBase::zoomToAboutCmd(const Vector& z, const Vector& v, 
			       CoordSystem sys, SkyFrame sky)
{
  zoom_ = ((Vector&)z).abs();
  if (currentFits) {
    cursor = currentFits->mapToRef(v, sys, sky) * refToUser;
    setBinCursor(cursor);
  }

  update(MATRIX);
}

// Panner/Magnifer Commands

void FrameBase::magnifierCmd(char* n, int w, int h)
{
  strcpy(magnifierName,n);
  magnifierWidth = w;
  magnifierHeight = h;

  if (magnifierPixmap)
    Tk_FreePixmap(display, magnifierPixmap);
  magnifierPixmap = 0;

  if (magnifierXImage)
    XDestroyImage(magnifierXImage);
  magnifierXImage = NULL;

  if (magnifierWidth > 0 && magnifierHeight > 0) {
    magnifierPixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin),
			     magnifierWidth, magnifierHeight, depth);
    if (!magnifierPixmap) {
      cerr << "Frame Internal Error: Unable to Create Magnifier Pixmap" 
	   << endl;
      exit(1);
    }

    if (!magnifierXImage)
      if (!(magnifierXImage = XGETIMAGE(display, magnifierPixmap, 0, 0, 
					magnifierWidth, magnifierHeight,
					AllPlanes, ZPixmap))){
	cerr << "Frame Internal Error: Unable to Create Magnifier XImage" 
	     << endl;
	exit(1);
      }
  }

  if (!magnifierGC)
    magnifierGC = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);

  updateMagnifierMatrix();
}

void FrameBase::magnifierCmd(int s)
{
  useMagnifier = s;
}

void FrameBase::magnifierGraphicsCmd(int which)
{
  useMagnifierGraphics = which;
  updateMagnifier();
}

void FrameBase::magnifierCursorCmd(int which)
{
  useMagnifierCursor = which;
  updateMagnifier();
}

void FrameBase::magnifierZoomCmd(double z)
{
  magnifierZoom_ = fabs(z);
  updateMagnifierMatrix();
  updateMagnifier();
}

void FrameBase::pannerCmd(char* n, int w, int h)
{
  strcpy(pannerName,n);
  pannerWidth = w;
  pannerHeight = h;

  if (pannerPixmap)
    Tk_FreePixmap(display, pannerPixmap);
  pannerPixmap = 0;

  if (pannerXImage)
    XDestroyImage(pannerXImage);
  pannerXImage = NULL;

  if (pannerWidth > 0 && pannerHeight > 0) {
    if (!(pannerPixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin), 
				      pannerWidth, pannerHeight, depth))) {
      cerr << "Frame Internal Error: Unable to Create Panner Pixmap" << endl;
      exit(1);
    }

    if (!(pannerXImage = XGETIMAGE(display, pannerPixmap, 0, 0,
				   pannerWidth, pannerHeight,
				   AllPlanes, ZPixmap))){
      cerr << "Frame Internal Error: Unable to Create Panner XImage" << endl;
      exit(1);
    }
  }

  // update panner matrices
  update(MATRIX);
}

void FrameBase::pannerCmd(int s)
{
  usePanner = s;
  updatePanner();
}

void FrameBase::pannerCmd(CoordSystem sys, SkyFrame sky)
{
  pannerSystem = sys;
  pannerSky = sky;
  updatePanner();
}

void FrameBase::updatePannerCmd()
{
  pannerPush();
}

void FrameBase::updateMagnifierCmd()
{
  updateMagnifier();
}

void FrameBase::updateMagnifierCmd(const Vector& v)
{
  updateMagnifier(v);
}

void FrameBase::contourCreateCmd(const char* color, int w,
			     int num, float res, int al)
{
  if (DebugPerf)
    cerr << "contourCreate" << endl;

  // make sure we have up-to-date matrices
  updateMatrices();

  if (hasContour())
    delete *currentcontour;
  *currentcontour = NULL;

  if (!isMosaic()) {
    if (currentFits)
      *currentcontour = new FVContour(this, currentFits, color, w, res, al,
			      contourScale(num, currentScale->low(),
					   currentScale->high(),
					   currentScale->colorScaleType()));
  }
  else {
    if (*channelFits) {
      *currentcontour = new FVContour(this, *channelFits, color, w, res, al,
			      contourScale(num, currentScale->low(),
					   currentScale->high(),
					   currentScale->colorScaleType()));
      FitsImage* ptr = (*channelFits)->nextMosaic();
      while (ptr) {
	(*currentcontour)->append(ptr);
	ptr = ptr->nextMosaic();
      }
    }
  }

  update(BASE);
}

void FrameBase::contourCreateCmd(const char* color, int w, 
			     float res, int al, const char* lev)
{
  if (DebugPerf)
    cerr << "contourCreate" << endl;

  // make sure we have up-to-date matrices
  updateMatrices();

  if (hasContour())
    delete *currentcontour;
  *currentcontour = NULL;

  int cnt = 0;
  float levels[100];
  string x(lev);
  istringstream str(x);

  while ((cnt<100) && (str >> levels[cnt]))
    cnt++;

  if (!isMosaic()) {
    if (currentFits)
      *currentcontour = new FVContour(this, currentFits, color, w, res, al, 
			      new ContourScale(cnt, levels));
  }
  else {
    if (*channelFits) {
      *currentcontour = new FVContour(this, *channelFits, color, w, res, al, 
			      new ContourScale(cnt, levels));
      FitsImage* ptr = (*channelFits)->nextMosaic();
      while (ptr) {
	(*currentcontour)->append(ptr);
	ptr = ptr->nextMosaic();
      }
    }
  }

  update(BASE);
}

void FrameBase::contourLoadCmd(const char* color, int w, const char* fn,
			   CoordSystem sys, SkyFrame sky)
{
  if (!currentFits)
    return;

  ifstream str(fn);

  if (str) {
    List<Vertex> c;
 
    while (!str.eof()) {
      Vertex* vv;

      char buf[64];
      str.getline(buf,64,'\n');
      if (strlen(buf) > 0) {
	Vector v;
	string x(buf);
	istringstream sstr(x);

	sstr >> v[0] >> v[1];
	vv = new Vertex(currentFits->mapToRef(v, sys, sky));
      }
      else
	vv = new Vertex(Vector(DBL_MAX, DBL_MAX));

      c.append(vv);
    }

    auxcontours.append(new Contour(this, color, w, c));
  }

  update(BASE);
}

void FrameBase::contourDeleteCmd()
{
  // main contour
  if (hasContour())
    delete *currentcontour;
  *currentcontour = NULL;

  update(BASE);
}

void FrameBase::contourDeleteAllCmd()
{
  // delete all contours
  // main contour
  if (hasContour())
    delete *currentcontour;
  *currentcontour = NULL;

  // aux contours
  auxcontours.deleteAll();

  update(BASE);
}

void FrameBase::contourCopyCmd(CoordSystem sys, SkyFrame sky)
{
  if (currentFits && hasContour()) {
    List<Vertex>* v = new List<Vertex>((*currentcontour)->contours());
    if (v->head())
      do {
	Vector& w = v->current()->vector;
	if (w[0] != DBL_MAX)
	  w = currentFits->mapFromRef(w, sys, sky);
      } while (v->next());

    ostringstream str;
    str << hex << v << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
}

void FrameBase::contourPasteCmd(const char* c, int w, void* u, 
			    CoordSystem sys, SkyFrame sky)
{
  if (!currentFits)
    return;

  List<Vertex>* v = (List<Vertex>*)u;

  if (v->head())
      do {
	Vector& w = v->current()->vector;
	if (w[0] != DBL_MAX)
	  w = currentFits->mapToRef(w, sys, sky);
      } while (v->next());
  
  auxcontours.append(new Contour(this, c, w, *v));
  delete v;

  update(BASE);
}

void FrameBase::contourSaveCmd(const char* fn, CoordSystem sys, SkyFrame sky)
{
  if (currentFits && hasContour()) {
    ofstream str(fn);

    if (str) {
      List<Vertex>& c = (List<Vertex>&)((*currentcontour)->contours());

      if (c.head())
	do {
	  Vector v = (c.current())->vector;
	  if (v[0] != DBL_MAX && v[1] != DBL_MAX)
	    str << setiosflags(ios::scientific) 
		<< setprecision(8) 
		<< currentFits->mapFromRef(v, sys, sky) 
		<< endl;
	  else
	    str << endl;
	}
	while (c.next());
    }
  }
}

void FrameBase::contourSetColorCmd(const char* clr)
{
  if (hasContour()) {
    (*currentcontour)->setColor(clr);
    update(BASE);
  }
}

void FrameBase::contourSetLineWidthCmd(int w)
{
  if (hasContour()) {
    (*currentcontour)->setLineWidth(w);
    update(BASE);
  }
}

void FrameBase::getContourCmd(CoordSystem sys, SkyFrame sky)
{
  if (currentFits && hasContour()) {
    List<Vertex>& v = (List<Vertex>&)(*currentcontour)->contours();
    if (v.head())
      do {
	Vector w = v.current()->vector;
	if (w[0] != DBL_MAX)
	  printVector(currentFits->mapFromRef(w, sys, sky),DEFAULT);
	Tcl_AppendResult(interp, "\n", NULL);
      } while (v.next());
  }
}

void FrameBase::getContourLevelCmd()
{
  if (hasContour()) {
    ContourScale* scale = (*currentcontour)->getScale();
    if (scale) {
      ostringstream str;
      str << *scale << ends;
      Tcl_AppendResult(interp, str.str().c_str(), NULL);
    }
  }
}

void FrameBase::getContourLevelCmd(int count, float ll, float hh,
			       FrScale::ColorScaleType scaleType)
{
  ContourScale* scale;
  switch (scaleType) {
  case FrScale::LINEARSCALE:
  case FrScale::IISSCALE:
    scale = new LinearContourScale(count, ll, hh);
    break;
  case FrScale::LOGSCALE:
    scale = new LogContourScale(count, ll, hh);
    break;
  case FrScale::SQUAREDSCALE:
    scale = new SquaredContourScale(count, ll, hh);
    break;
  case FrScale::SQRTSCALE:
    scale = new SqrtContourScale(count, ll, hh);
    break;
  case FrScale::HISTEQUSCALE:
    calcHistEqu();
    scale = new HistEquContourScale(count, currentScale->min(), 
				    currentScale->max(), 
				    currentScale->histequ(), HISTEQUSIZE);
    break;
  }

  ostringstream str;
  str << *scale << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
 
  delete scale;
}

void FrameBase::getContourMethodCmd()
{
  if (hasContour())
    Tcl_AppendResult(interp, (*currentcontour)->getMethodName(), NULL);
}

void FrameBase::getContourColorNameCmd()
{
  if (hasContour())
    Tcl_AppendResult(interp, (*currentcontour)->getColorName(), NULL);
}

void FrameBase::getContourLineWidthCmd()
{
  if (hasContour()) {
    ostringstream str;
    str << (*currentcontour)->getLineWidth() << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
}

void FrameBase::hasContourCmd()
{
  if (hasContour())
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

// Grid Commands

void FrameBase::gridCmd(CoordSystem sys, SkyFrame sky, SkyFormat format, 
			Grid::GridType type, const char* ops)
{
  if (grid)
    delete grid;
  grid = new Grid(this, sys, sky, format, type, ops);

  update(BASE);
}

void FrameBase::gridDeleteCmd()
{
  if (grid)
    delete grid;
  grid = NULL;

  update(BASE);
}

void FrameBase::hasGridCmd()
{
  if (grid)
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::getGridCmd()
{
  if (grid) {
    Tcl_AppendElement(interp, coordSystemStr_[(grid->system())-IMAGE]);

    Tcl_AppendElement(interp, skyFrameStr_[(grid->sky())-FK4]);

    Tcl_AppendElement(interp, skyFormatStr_[grid->skyFormat()]);

    switch (grid->type()) {
    case Grid::ANALYSIS:
      Tcl_AppendElement(interp, "analysis");
      break;
    case Grid::PUBLICATION:
      Tcl_AppendElement(interp, "publication");
      break;
    }
  }
  else
    Tcl_AppendResult(interp, "", NULL);
}

void FrameBase::getGridOptionCmd()
{
  if (grid)
    Tcl_AppendResult(interp, grid->option(), NULL);
  else
    Tcl_AppendResult(interp, "", NULL);
}

// Smooth commands

void FrameBase::smoothCmd(SmoothFunction f, int r)
{
  dosmooth_ = 1;
  smoothFunction_ = f;
  smoothRadius_ = r;

  analysisFits();
  update(BASE);
}

void FrameBase::smoothDeleteCmd()
{
  dosmooth_ = 0;

  analysisFits();
  update(BASE);
}

void FrameBase::hasSmoothCmd()
{
  if (dosmooth_)
    Tcl_AppendResult(interp, "1", NULL);
  else
    Tcl_AppendResult(interp, "0", NULL);
}

void FrameBase::getSmoothFunctionCmd()
{
  switch (smoothFunction_) {
  case BOXCAR:
    Tcl_AppendResult(interp, "boxcar", NULL);
    return;
  case TOPHAT:
    Tcl_AppendResult(interp, "tophat", NULL);
    return;
  case GAUSSIAN:
    Tcl_AppendResult(interp, "gaussian", NULL);
    return;
  }
}

void FrameBase::getSmoothRadiusCmd()
{
  printInteger(smoothRadius_);
}

// Pixel Table commands

void FrameBase::getPixelTableCmd(const Vector& v, InternalSystem sys,
			     int size, char* var)
{
  switch (sys) {
  case CANVAS:
    getPixelTable(v, &FitsImage::getCanvasToData, size, var);
    break;
  case PANNER:
    getPixelTable(v, &FitsImage::getPannerToData, size, var);
    break;
  }
}

void FrameBase::getPixelTable(const Vector& v, 
			  Matrix& (FitsImage::*getMatrix)(),
			  int size, char* var)
{
  Vector r;
  if (FitsImage* ptr = isInFits(v, getMatrix, &r)) {
    getPixelTable(ptr, r, size, var);
    return;
  }

  // else return blank
  for (int i=0; i<=size; i++)
    for (int j=0; j<=size; j++) {
      ostringstream str;
      str << i << j << ends;
      Tcl_SetVar2(interp,var,str.str().c_str(),"",0);
    }
}

void FrameBase::getPixelTable(FitsImage* ptr, const Vector& img, 
			  int size, char* var)
{
  Vector half((size-1)/2,(size-1)/2);
  Vector ur = img+half;
  Vector ll = img-half;
  int i,j;

  int* params = ptr->getDataParams(currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  // x (columns)
  for (i=0,j=0; i<size; i++) {
    ostringstream str;

    str << i+1 << j << ends;

    if (ur[0]>=xmin && ll[0]<xmax && ur[1]>=ymin && ll[1]<ymax) {
      Vector pt = ((ll+Vector(i,j)) * dataToImage).round();
      if (pt[0]>xmin && pt[0]<=xmax) {
	ostringstream lstr;
	lstr << pt[0] << ends;
	Tcl_SetVar2(interp,var,str.str().c_str(),lstr.str().c_str(),0);
      }
      else
	Tcl_SetVar2(interp,var,str.str().c_str(),"",0);
    }
    else
      Tcl_SetVar2(interp,var,str.str().c_str(),"",0);
  }

  // y (rows)
  for (i=0,j=0; j<size; j++) {
    ostringstream str;
    str << i << j+1 << ends;

    if (ur[0]>=xmin && ll[0]<xmax && ur[1]>=ymin && ll[1]<ymax) {
      Vector pt = ((ll+Vector(i,j)) * dataToImage).round();
      if (pt[1]>ymin && pt[1]<=ymax) {
	ostringstream lstr;
	lstr << pt[1] << ends;
	Tcl_SetVar2(interp,var,str.str().c_str(),lstr.str().c_str(),0);
      }
      else
	Tcl_SetVar2(interp,var,str.str().c_str(),"",0);
    }
    else
      Tcl_SetVar2(interp,var,str.str().c_str(),"",0);
  }

  // body
  for (i=0; i<size; i++)
    for (j=0; j<size; j++) {
      Vector pt = ll+Vector(i,j);
      ostringstream str;      
      str << i+1 << j+1 << ends;

      if (pt[0]>=xmin && pt[0]<xmax && pt[1]>=ymin && pt[1]<ymax)
	Tcl_SetVar2(interp,var,str.str().c_str(),(char*)ptr->getValue(pt),0);
      else
	Tcl_SetVar2(interp,var,str.str().c_str(),"",0);
    }
}

// Data Values

void FrameBase::getDataValuesCmd(int which, const Vector& v, 
			     CoordSystem sys, SkyFrame sky,
			     const Vector& ddd, char* var)
{
  // clear an previous values
  Tcl_UnsetVar(interp,var,0);

  // find anchor point
  FitsImage* base = findFits(which);
  if (!base) {
    Tcl_SetVar(interp,var,"",0);
    result = TCL_ERROR;
    return;
  }
  Vector d = base->mapLenToRef(ddd,sys,DEGREES);
  Vector ll = base->mapToRef(v,sys,sky);

  for (int i=0; i<d[0]; i++)
    for (int j=0; j<d[1]; j++) {
      Vector rr = ll+Vector(i,j);

      // index is in terms of native coords
      Vector in = base->mapFromRef(rr,sys,sky);

      ostringstream str;
      str << setprecision(12) << in[0] << ',' << in[1] << ends;

      int found = 0;
      FitsImage* ptr = *channelFits;
      while (ptr) {
	Vector dd = rr * ptr->getRefToData();

	int* params = ptr->getDataParams(currentScale->scanMode());
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (dd[0]>=xmin && dd[0]<xmax && dd[1]>=ymin && dd[1]<ymax) {
	  Tcl_SetVar2(interp,var,str.str().c_str(),(char*)ptr->getValue(dd),0);
	  found = 1;
	  break;
	}
	ptr = ptr->nextMosaic();
      }

      if (!found)
	Tcl_SetVar2(interp,var,str.str().c_str(),"",0);
    }
}

void FrameBase::getDataValuesCmd(const Vector& v, InternalSystem sys,
				 const Vector& b)
{
  Vector d = b;
  FitsImage* ptr = NULL;
  Vector ll;

  // find anchor point
  switch (sys) {
  case CANVAS:
    if (ptr=isInFits(v, &FitsImage::getCanvasToData, NULL))
      ll = v*canvasToRef - Vector((d[0]-1)/2,(d[1]-1)/2);
    break;
  case PANNER:
    if (ptr=isInFits(v, &FitsImage::getPannerToData, NULL))
      ll = v*pannerToRef - Vector((d[0]-1)/2,(d[1]-1)/2);
    break;
  }
  if (!ptr)
    return;

  for (int j=0; j<d[1]; j++)
    for (int i=0; i<d[0]; i++) {
      Vector dd = (ll+Vector(i,j)) * ptr->getRefToData();

      int* params = ptr->getDataParams(currentScale->scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      if (dd[0]>=xmin && dd[0]<xmax && dd[1]>=ymin && dd[1]<ymax)
	Tcl_AppendResult(interp, (char*)ptr->getValue(dd), " ", NULL);
    }
}

// WCS commands

void FrameBase::wcsResetCmd()
{
  if (currentFits)
    currentFits->resetWCS();
}

void FrameBase::wcsReplaceCmd(int fd)
{
  if (!currentFits)
    return;

  boost::fdistream str(fd);
  if (!str) {
    Tcl_AppendResult(interp, " unable to read wcs infomation", NULL);
    result = TCL_ERROR;
    return;
  }

  FitsHead* r = parseWCS(str);
  currentFits->replaceWCS(r);
  delete r;
}

void FrameBase::wcsReplaceCmd(const char* fn)
{
  if (!currentFits)
    return;

  ifstream str(fn);
  if (!str) {
    Tcl_AppendResult(interp, " unable to load wcs file ", fn, NULL);
    result = TCL_ERROR;
    return;
  }

  FitsHead* r = parseWCS(str);
  currentFits->replaceWCS(r);
  delete r;
}

void FrameBase::wcsAppendCmd(int fd)
{
  if (!currentFits)
    return;

  boost::fdistream str(fd);
  if (!str) {
    Tcl_AppendResult(interp, " unable to read wcs infomation", NULL);
    result = TCL_ERROR;
    return;
  }

  FitsHead* r = parseWCS(str);
  currentFits->appendWCS(r);
  delete r;
}

void FrameBase::wcsAppendCmd(const char* fn)
{
  if (!currentFits)
    return;

  ifstream str(fn);
  if (!str) {
    Tcl_AppendResult(interp, " unable to load wcs file ", fn, NULL);
    result = TCL_ERROR;
    return;
  }

  FitsHead* r = parseWCS(str);
  currentFits->appendWCS(r);
  delete r;
}

FitsHead* FrameBase::parseWCS(istream& str)
{
  if (!currentFits)
    return NULL;

  FitsHead* head = currentFits->getHead();
  FitsHead* r = new FitsHead(head->naxis(1), head->naxis(2),
			     head->naxis(3), head->bitpix());

  while (!str.eof()) {
    char buf[256];
    str.getline(buf,256,'\n');

    if (strlen(buf) > 0) {
      string x(buf);
      istringstream sstr(x);

      char keyword[256];
      sstr >> keyword;

      if (strchr(buf,'=')) {
	char dummy;
	sstr >> dummy;
      }

      if (strchr(buf,'\'')) {
	char string[32];
	sstr >> string;

	// remove both '
	string[strlen(string)-1] = '\0';
	r->appendString(keyword, string+1, "");
      }
      else {
	double val;
	sstr >> val;
	r->appendReal(keyword, val, 9, "");
      }
    }
    else
      break;
  }

  return r;
}

// utility member functions for cmds

void FrameBase::printCoordSystem(CoordSystem sys)
{
  Tcl_AppendResult(interp, coordSystemStr_[sys-IMAGE], NULL);
}

void FrameBase::printSkyFrame(SkyFrame sky)
{
  Tcl_AppendResult(interp, skyFrameStr_[sky-FK4], NULL);
}

void FrameBase::printSkyFormat(SkyFormat format)
{
  Tcl_AppendResult(interp, skyFormatStr_[format], NULL);
}

void FrameBase::printInteger(int i)
{
  ostringstream str;
  str << i << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void FrameBase::printDouble(double d, Precision p)
{
  ostringstream str;
  switch (p) {
  case DEFAULT:
    str << d << ends;
    break;
  case FIXED:
    str << setiosflags(ios::fixed) << setw(9) << setprecision(3) << d << ends;
    break;
  case SCIENTIFIC:
    str << setiosflags(ios::scientific) << setprecision(8) << d  << ends;
    break;
  case INTEGER:
    str << (int)d << ends;
    break;
  }
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void FrameBase::printVector(const Vector& v, Precision p)
{
  ostringstream str;
  switch (p) {
  case DEFAULT:
    str << setprecision(8) << v << ends;
    break;
  case FIXED:
    str << setiosflags(ios::fixed) << setw(9) << setprecision(3) << v << ends;
    break;
  case SCIENTIFIC:
    str << setiosflags(ios::scientific) << setprecision(8) << v << ends;
    break;
  case INTEGER:
    {
      Vector z = v;
      str << z.round() << ends;
      break;
    }
  }
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void FrameBase::printFromRef(FitsImage* ptr, const Vector& v,
			 CoordSystem sys, SkyFrame sky, SkyFormat format,
			 Precision p)
{
  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    printVector(ptr->mapFromRef(v, sys, sky), p);
    return;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	    printVector(ptr->mapFromRef(v, sys, sky), p);
	  break;
	case SEXAGESIMAL:
	case HMS:
	  {
	    char buf[64];
	    buf[0] = '\0';
	    ptr->mapFromRef(v, sys, sky, format, buf, 64);

	    // grap only the first two items
	    char* ptr = buf;
	    while (*ptr)
	      ptr++;
	    while (*ptr != ' ' && ptr >= buf)
	      ptr--;
	    *ptr = '\0';

	    Tcl_AppendResult(interp, buf, " ", NULL);
	  }
	  break;
	}
      }
      else
	printVector(ptr->mapFromRef(v, sys), p);
    }
    else
      printVector(Vector(),p);

    return;
  }
}

void FrameBase::coordToTclArray(FitsImage* ptr, const Vector& v, 
			    InternalSystem in, CoordSystem out, 
			    Precision p, const char* var, const char* index)
{
  Vector r = ptr->mapFromRef(mapToRef(v,in), out);
  doubleToTclArray(r[0], p, var, index, "x");
  doubleToTclArray(r[1], p, var, index, "y");
}

void FrameBase::doubleToTclArray(double d, Precision p, const char* var, 
			     const char* base, const char* mod)
{
  char index[64];
  strcpy(index,mod);
  strcat(index,base);

  ostringstream str;
  switch (p) {
  case DEFAULT:
    str << d << ends;
    break;
  case FIXED:
    str << setiosflags(ios::fixed) << setw(9) << setprecision(3) 
	<< d << ends;
    break;
  case SCIENTIFIC:
    str << setiosflags(ios::scientific) << setprecision(8) << d << ends;
    break;
  case INTEGER:
    str << (int)d << ends;
    break;
  }

  Tcl_SetVar2(interp, (char*)var, index, str.str().c_str(), 0);
}
