/*
 *	Copyright (c) 1999-2003 Smithsonian Astrophysical Observatory
 */

#include <funtoolsP.h>

/*
 *
 * private routines
 *
 */

/* default parser structs for Text files */
typedef struct _ptyperec{
  char *delims;
  char *comchars;
  char *eot;
  int nullvalue;
  int whitespace;
  int units;
  int comeot;
} *PType, PTypeRec;

static PTypeRec ptype[] = {{" \t,\n", "#\n", NULL,   0, 0, 0, 1}, /* funtools */
			   {"\t\n",   "#\n", "\f\n", 1, 0, 0, 1}, /* rdb */
			   {"|\t;\n", "#\n", NULL,   1, 0, 1, 1}};/* VizieR */
		           
/*
 *
 * _FunNew -- allocate a new fun record
 *
 */
#ifdef ANSI_FUNC
static Fun
_FunNew(void)
#else
static Fun _FunNew()
#endif
{
  Fun fun;

  fun = (Fun)xcalloc(1, sizeof(FunRec));
  /* set the magic value */
  fun->magic = FUN_MAGIC;
  /* set table binning values to impossible values */
  fun->bin[0] = -1;
  fun->bin[1] = -1;
  /* reset header position flag */
  fun->headpos = (off_t)-1;
  /* no data yet */
  fun->datastart = (off_t)-1;
  /* init save fd's to impossible values */
  fun->lmem = -1;
  /* set binsiz values to default value of 1 */
  fun->binsiz1 = 1;
  fun->binsiz2 = 1;
  /* no reaon why we can't filter */
  fun->dofilt = 1;
  /* convert flags not yet set */
  fun->iconvert = -1;
  fun->oconvert = -1;
  return fun;
}


#ifdef ANSI_FUNC
static int _FunSpecialFile(char *fname, char *type,
			   char *name, char *tail, char *extn)
#else
static int _FunSpecialFile(fname, type, name, tail, extn)
     char *fname;
     char *type;
     char *name;
     char *tail;
     char *extn;
#endif
{
  int len;
  char *s, *t;
  char tbuf[SZ_LINE];

  /* start off pessimistically */
  *name = '\0';
  *tail = '\0';
  *extn = '\0';

  /* get type string */
  strcpy(tbuf, type);
  strcat(tbuf, "(");
  len = strlen(tbuf);

  /* look for [TYPE(...) */
  if( ((s=strchr(fname, '['))||(s=strchr(fname, ',')))    	&& 
      !strncasecmp(s+1, tbuf, len) && (t=strchr(s, ')'))  	&&
      ((*(t+1) == ')') || (*(t+1) == ',') || (*(t+1) == ']'))	){
    strncpy(name, fname, s-fname);
    name[s-fname] = '\0';
    while( *(s+len+1) == '(' ) s++;
    strncpy(extn, s+len+1, t-(s+len+1));
    extn[t-(s+len+1)] = '\0';
    /* if extn specification is blank, try to get it from the environment */
    if( !*extn  && getenv(type) )
      strcpy(extn, getenv(type));
    /* create tail, skipping empty section if necessary */
    while( *t == ')' ) t++;
    if( *t != ']' ){
      strcpy(tail, "[");
      strcat(tail, t);
    }
    else{
      strcpy(tail, ++t);
    }
    return 1;
  }
  else
    return 0;
}


/*
 *
 * _FunImageSkip -- caclulcate the number of bytes to skip to get
 * to specified image plane of an N-dimensional image
 *
 */
#ifdef ANSI_FUNC
static off_t _FunImageSkip(Fun fun, char *tail)
#else
static off_t _FunImageSkip(fun, tail)
     Fun fun;
     char *tail;
#endif
{
  int i,j,k;
  int naxes;
  int plane;
  int planes[8];
  int skip;
  off_t total;
  char tbuf[SZ_LINE];
  char *s, *t;

  /* nothing to skip yet */
  total = 0;

  /* nothing to do if its not an n-D image */
  if( (naxes=ft_naxes(fun->header)) <= 2 )
    return total;

  /* look for a bracket specification */
  if( !_FunKeyword(tail, "plane", NULL, tbuf, SZ_LINE) )
    return 0;
  else
    s = tbuf;

  /* skip open parens */
  if( *s == '(' )
    s++;
  planes[1] = ft_naxis(fun->header, 1);
  planes[2] = ft_naxis(fun->header, 2);
  /* see the other planes */
  for(i=3; i<=7; i++)
    planes[i] = 1;
  /* we look for a plane specification for each axis > 2 */
  for(i=3; i<=naxes; i++){
    plane = strtol(s, &t, 10);
    if( (s == t) )
      plane = 1;
    else if( (plane < 1) || (plane > ft_naxis(fun->header, i)) )
      return -1;
    /* save the image plane desired for this dimension */
    planes[i] = plane;
    /* point past delimiter */
    if( *t == ':' )
      t++;
    /* see if we ran out of running room */
    if( (*t == '\0') || (*t == ']') || (*t == ')') )
      break;
    /* this is where we look next */
    s = t;
  }
  /* skip n-dimensional cubes of various dimensions until we
     get to the desired 2d image */
  for(j=naxes; j>2; j--){
    for(skip=1,k=1; k<j; k++)
      skip *= ft_naxis(fun->header, k);
    skip *= (planes[j]-1) * (ABS(fun->dtype));
    total += skip;
  }
  return total;
}


/*
 *
 * _FunOpenCommon -- common code for different Fun open routines
 *
 */
#ifdef ANSI_FUNC
int
_FunOpenCommon(Fun fun)
#else
int _FunOpenCommon(fun)
     Fun fun;
#endif
{
  int tval;

  if( fun && fun->header ){
    /* current file position is datapos */
    fun->datastart = gtell(fun->gio);
    /* calculate binning values */
    if( fun->header->table && (fun->bin[0]>=0) )
      tval = tldim(fun->min1, fun->max1, fun->binsiz1,
		   fun->header->table->col[fun->bin[0]].type);
    else
      tval = tldim(fun->min1, fun->max1,  fun->binsiz1, 0);
    if( (fun->x0 < 1) || (fun->x0 > tval) ) fun->x0 = 1;
    if( (fun->x1 < 1) || (fun->x1 > tval) ) fun->x1 = tval;
    if( fun->header->table && (fun->bin[1]>=0) )
      tval = tldim(fun->min2,fun->max2, fun->binsiz2,
		   fun->header->table->col[fun->bin[1]].type);
    else
      tval = tldim(fun->min2, fun->max2,  fun->binsiz2, 0);
    if( (fun->y0 < 1) || (fun->y0 > tval) ) fun->y0 = 1;
    if( (fun->y1 < 1) || (fun->y1 > tval) ) fun->y1 = tval;
    /* make sure the block factor divides the dimension evenly */
    if( (fun->x1 - fun->x0 + 1)%fun->block ){
      gwarning(stderr,
	       "block factor does not divide dim1 evenly; decreasing dim1\n");
      while( (fun->x1 - fun->x0 + 1)%fun->block ) fun->x1--;
    }
    if( (fun->y1 - fun->y0 + 1)%fun->block ){
      gwarning(stderr,
	       "block factor does not divide dim2 evenly; decreasing dim2\n");
      while( (fun->y1 - fun->y0 + 1)%fun->block ) fun->y1--;
    }
    /* initialize output binning parameters */
    fun->odim1 = (fun->x1 - fun->x0 + 1) / fun->block;
    fun->odim2 = (fun->y1 - fun->y0 + 1) / fun->block;
    fun->obitpix = fun->bitpix;
    fun->odtype = fun->dtype;
    /* initialize WCS -- this is the proper WCS, converted to image coords */
    fun->wcs = _FunWCS(fun, 1);
    /* initialize WCS -- not converted to image coords */
    fun->wcs0 = _FunWCS(fun, 0);
    return 1;
  }
  else
    return 0;
}



/*
 *
 *  _FunRowNum -- get table/event row limits
 *
 */
#ifdef ANSI_FUNC
static int
_FunRowNum(Fun fun, char *tail, char *env)
#else
static int _FunRowNum(fun, tail, env)
     Fun fun;
     char *tail;
     char *env;
#endif
{
  char lobuf[SZ_LINE];
  char hibuf[SZ_LINE];
  char key[SZ_LINE];
  char *k;
  int ip=0;
  int lo, hi;
  int skip;

  /* first look in the extension for row range specification */
  if( _FunKeyword(tail, "row#", env, key, SZ_LINE) ){
    /* use a few command separators (as well as \n) */
    newdtable(",:)");
    k = key;
    /* point past first paren */
    if( *k == '(' )
      k++;
    /* process row specifiers */
    if( word(k, lobuf, &ip) && word(k, hibuf, &ip) ){
      if( strcmp(lobuf, "*") )
	lo = atoi(lobuf);
      else
	lo = 1;
      if( strcmp(hibuf, "*") )
	hi = atoi(hibuf);
      else
	hi = fun->total;
      lo = MAX(1,lo);
      hi = MIN(fun->total,hi);
      /* skip past initial rows we are not interested in */
      if( lo > 1 ){
	skip = (lo-1) * fun->rawsize;
	gskip(fun->gio, (off_t)skip);
	fun->bytes += skip;
      }
      fun->left -= (lo-1);
      fun->left -= (fun->total-hi);
    }
    freedtable();
    return 1;
  }
  else
    return 0;
}

/*
 *
 *  _FunTableBincols -- get table columns for binning
 *
 */
#ifdef ANSI_FUNC
static int
_FunTableBincols(Fun fun, char *tail, char *env)
#else
static int _FunTableBincols(fun, tail, env)
     Fun fun;
     char *tail;
     char *env;
#endif
{
  FITSCard card, card2, card3;
  int i, j;
  int got;
  int ip=0;
  int dims;
  int scaled;
  double tlmin, tlmax, binsiz, tscale, tzero;
  char *s;
  char *k;
  char key[SZ_LINE];
  char tbuf[SZ_LINE];
  char tbuf2[SZ_LINE];
  char binstr[2][SZ_LINE];
  static char *tstr[] = {"X", "Y"};
  static char *tstr2[] = {"x", "y"};

  /* set keys to impossible values */
  fun->bin[0] = -1;
  fun->bin[1] = -1;

  /* for backward compatibility, we look for binkey as well */
  if( !strcmp(env, "FITS_BINCOLS" ) )
    strcpy(tbuf2, "FITS_BINKEY");
  else if( !strcmp(env, "EVENTS_BINCOLS" ) )
    strcpy(tbuf2, "EVENTS_BINKEY");
  else if( !strcmp(env, "TEXT_BINCOLS" ) )
    strcpy(tbuf2, "TEXT_BINKEY");
  else
    *tbuf2 = '\0';

  /* first look in the extension for binning specification */
  if( _FunKeyword(tail, "bincols", env, tbuf, SZ_LINE) ){
    strcpy(key, tbuf);
  }
  /* backward compatibility */
  else if( _FunKeyword(tail, "binkey", tbuf2, tbuf, SZ_LINE) ){
    strcpy(key, tbuf);
  }
  /* look for default binning specifications */
  else{
    got = 0;
    *key = '\0';
    /* look for CHANDRA PREFX keyword */
    for(j=1; j<=2; j++){
      if( (s=ft_headgets(fun->header, "CPREF", j, NULL, &card)) && card){
	strcat(key, s);
	strcat(key, ",");
	got |= j;
	if( s ) xfree(s);
      }
      else if( (s=ft_headgets(fun->header, "PREFX", j, NULL, &card)) && card){
	strcat(key, s);
	strcat(key, ",");
	got |= j;
	if( s ) xfree(s);
      }
    }
    /* look specifically for X, Y */
    if( got != 3 ){
      got = 0;
      *key = '\0';
      for(j=0; j<=1; j++){
	/* first look for X,Y as the default */
	for(i=0; i<fun->header->table->tfields; i++){
	  if( !fun->header->table->col[i].name ) continue;
	  if( !strcasecmp(fun->header->table->col[i].name, tstr[j]) ){
	    strcat(key, tstr[j]);
	    strcat(key, ",");
	    got |= (j+1);
	    break;
	  }
	}
      }
    }
    /* if we did not get X (or Y), look for anything containing an X (or Y) */
    if( got != 3 ){
      got = 0;
      *key = '\0';
      for(j=0; j<=1; j++){
	/* don't look a second time for a bin value we already have */
	if( got & (j+1) )
	  continue;
	for(i=0; i<fun->header->table->tfields; i++){
	  if( !fun->header->table->col[i].name ) continue;
	  if( strstr(fun->header->table->col[i].name, tstr[j]) != NULL ){
	    strcat(key, fun->header->table->col[i].name);
	    strcat(key, ",");
	    break;
	  }
	  /* sigh ... its gotta be case insensitive */
	  else if( strstr(fun->header->table->col[i].name, tstr2[j]) != NULL ){
	    strcat(key, fun->header->table->col[i].name);
	    strcat(key, ",");
	    break;
	  }
	}
      }
    }
  }

  /* seed the dim values */
  fun->min1 =  1;
  fun->max1 = -1;
  fun->min2 =  1;
  fun->max2 = -1;

  /* prepare to parse the bincol specifier */
  newdtable(",)");
  k = key;
  /* point past first paren */
  while( (*k == '(') || (*k == '"') ) k++;
  /* separate x and y specifiers */
  if( !word(k, binstr[0], &ip) || !word(k, binstr[1], &ip) )
    goto error;
  freedtable();

  /* make sure columns are in the table, and check for 
     manually-specified image dimensions and binsize info */
  for(got=0, j=0; j<=1; j++){
    /* look for specifiers after the name */
    if( (s = strchr(binstr[j], ':')) ){
      strncpy(tbuf, binstr[j], s-binstr[j]);
      tbuf[s-binstr[j]] = '\0';
    }
    else{
      strcpy(tbuf, binstr[j]);
    }
    /* look among columns for this column name */
    for(i=0; i<fun->header->table->tfields; i++){
      if( !fun->header->table->col[i].name ) continue;
      if( !strcasecmp(fun->header->table->col[i].name, tbuf) ){
	/* save the column number */
	fun->bin[j] = i;
	/* if dim/binsize was specified, put it all back into the header */
	if( _FunColumnDims(s, fun->header->table->col[i].type,
			   &tlmin, &tlmax, &binsiz, &dims,
			   &tscale, &tzero, &scaled)){
	  switch(fun->header->table->col[i].type){
	  case 'D':
	  case 'E':
	    if( tlmin != tlmax ){
	      ft_headsetr(fun->header, "TLMIN", i+1, tlmin, 7,
			  "Min. axis value", 1);
	      ft_headsetr(fun->header, "TLMAX", i+1, tlmax, 7,
			  "Max. axis value", 1);
	    }
	    if( binsiz > 0 ){
	      ft_headsetr(fun->header, "TDBIN", i+1, binsiz, 7,
			  "Binsize", 1);
	    }
	    break;
	  default:
	    if( tlmin != tlmax ){
	      ft_headseti(fun->header, "TLMIN", i+1, (int)tlmin,
			  "Min. axis value", 1);
	      ft_headseti(fun->header, "TLMAX", i+1, (int)tlmax,
			  "Max. axis value", 1);
	    }
	    if( binsiz > 0 ){
	      ft_headsetr(fun->header, "TDBIN", i+1, binsiz, 7,
			  "Binsize", 1);
	    }
	    break;
	  }
	}
	/* add scaling parameters if necvessary */
	if( scaled ){
	  ft_headsetr(fun->header, "TSCAL", i+1, tscale, 7,
		      "phys_val= tzero + tscale * field_val", 1);
	  ft_headsetr(fun->header, "TZERO", i+1, tscale, 7,
		      "phys_val= tzero + tscale * field_val", 1);
	}
	got |= (j+1);
	break;
      }
    }
  }
  /* better have both columns */
  if( got != 3 )
    return 0;

  /* save bincols (string is used in FilterOpen, for example) */
  snprintf(tbuf, SZ_LINE, "bincols=(%s,%s)",
	   fun->header->table->col[fun->bin[0]].name,
	   fun->header->table->col[fun->bin[1]].name);
  fun->bincols=xstrdup(tbuf);

  /* make sure these keys have valid axis lengths associated with them */
  /* first look for TLMAX (and perhaps TLMIN) */
  fun->max1 = ft_headgetr(fun->header, "TLMAX", fun->bin[0]+1, 0.0, &card);
  fun->max2 = ft_headgetr(fun->header, "TLMAX", fun->bin[1]+1, 0.0, &card2);
  fun->binsiz1 = ft_headgetr(fun->header, "TDBIN", fun->bin[0]+1, 1.0, &card3);
  fun->binsiz2 = ft_headgetr(fun->header, "TDBIN", fun->bin[1]+1, 1.0, &card3);
  if( card && card2 ){
    fun->min1 = ft_headgetr(fun->header, "TLMIN", fun->bin[0]+1, 1.0, &card);
    fun->min2 = ft_headgetr(fun->header, "TLMIN", fun->bin[1]+1, 1.0, &card);
  }
  /* found nothing useful */
  else{
    return 0;
  }

  /* calculate dimension from tlmin and tlmax */
  fun->dim1 = tldim(fun->min1, fun->max1, fun->binsiz1,
		    fun->header->table->col[fun->bin[0]].type);
  fun->dim2 = tldim(fun->min2, fun->max2, fun->binsiz2,
		    fun->header->table->col[fun->bin[1]].type);

  return 1;

error:
  freedtable();
  return 0;
}

/*
 *
 * FunRawEvHeader -- read a string that details the structure
 * of the event record and generate a fitsy header
 *
 */
#ifdef ANSI_FUNC
static FITSHead
_FunRawEvHeader(Fun fun, char *iname, char *iext, char *eventdef)
#else
static FITSHead _FunRawEvHeader(fun, iname, iext, eventdef)
     Fun fun;
     char *iname;
     char *iext;
     char *eventdef;
#endif
{
  char tbuf[SZ_LINE];
  char *s;
  char *neventdef=NULL;
  char **names=NULL;
  char **types=NULL;
  char *extname="EVENTS";
  int extver=1;
  int width=0;
  int ncol=0;
  int got;
  int i, r;
  int total;
  int skip;
  int type;
  int dim, scaled;
  int xtype, xoff;
  int *dims=NULL;
  int *scaleds=NULL;
  int *widths=NULL;
  double tlmin, tlmax, binsiz, tscale, tzero;
  double *tlmins=NULL, *tlmaxs=NULL, *binsizs=NULL;
  double *tscales=NULL, *tzeros=NULL;
  FITSHead header=NULL;
  struct stat buf;

  /* make sure we have some columns */
  if( !eventdef || !*eventdef ){
    gerror(stderr, "Event definition is empty\n");
    goto done;
  }

  /* init the fitsy header */
  header = ft_headinit(NULL, 0);

  /* get rid of enclosing parens */
  if(*eventdef == '('){
    neventdef = xstrdup(eventdef+1);
    i = strlen(neventdef);
    if( neventdef[i-1] == ')' )
      neventdef[i-1] = '\0';
  }
  /* get rid of enclosing quotes */
  else if( (*eventdef == '"') || (*eventdef == '\'') ){
    neventdef = xstrdup(eventdef+1);
    i = strlen(neventdef);
    if( neventdef[i-1] == *eventdef )
      neventdef[i-1] = '\0';
  }
  else{
    neventdef = xstrdup(eventdef);
  }
  /* we only deal with upper case in FITS */
  cluc(neventdef);

  /* count the number of columns, separated by ',' */
  for(ncol=0, s=neventdef; *s; s++){
    if( *s == ',' ) ncol++;
  }
  /* last column had a null after it */
  ncol++;

  /* allocate string space for each column */
  names = (char **)xcalloc(ncol, sizeof(char *));
  types = (char **)xcalloc(ncol, sizeof(char *));
  dims = (int *)xcalloc(ncol, sizeof(int));
  scaleds = (int *)xcalloc(ncol, sizeof(int));
  tlmins = (double *)xcalloc(ncol, sizeof(double));
  tlmaxs = (double *)xcalloc(ncol, sizeof(double));
  binsizs = (double *)xcalloc(ncol, sizeof(double));
  tscales = (double *)xcalloc(ncol, sizeof(double));
  tzeros = (double *)xcalloc(ncol, sizeof(double));
  widths = (int *)xcalloc(ncol, sizeof(int));

  /* process the string and gather up the information on each column */
  s = neventdef;
  for(got=0; got<ncol; got++){
    /* get name */
    for(i=0; (i < SZ_LINE-1) && *s && (*s != ':'); i++, s++){
      tbuf[i] = *s;
      tbuf[i+1] = '\0';
    }
    names[got] = xstrdup(tbuf);
    /* skip delim */
    if( *s == ':' )   s++;
    /* make sure we have something */
    if( *s == '\0' )  break;
    
    /* get column specification up to next comma */
    for(i=0; (i < SZ_LINE-1) && *s && (*s != ','); i++, s++){
      tbuf[i] = *s;
      tbuf[i+1] = '\0';
    }

    /* parse values for various column parameters */
    _FunColumnType(tbuf, &type, &r, &tlmin, &tlmax, &binsiz, &dim,
		   &tscale, &tzero, &scaled, &xtype, &xoff);

    /* save info for this column */
    snprintf(tbuf, SZ_LINE, "%d%c", r, type);
    types[got] = xstrdup(tbuf);
    tlmins[got] = tlmin;
    tlmaxs[got] = tlmax;
    binsizs[got] = binsiz;
    dims[got] = dim;
    tscales[got] = tscale;
    tzeros[got] = tzero;
    scaleds[got] = scaled;
    widths[got] = ft_sizeof(type);
    /* add to total width */
    if( type == 'X' )
      width += ((widths[got]*r) + 7 ) / 8;
    else
      width += widths[got] * r;
    /* bump past the comma */
    if( *s == ',' ) s++;
  }
  /* make sure we have some columns */
  if( width == 0 ){
    gerror(stderr, "Table width is zero\n");
    goto done;
  }

  /* get the size of the file and from this, the number of events */
  if( !iname || !*iname || 
      !strcasecmp(iname, "stdin") || (stat(iname, &buf) <0) ){
    total = -1;
  }
  else{
    if( fun == NULL )
      skip = 0;
    else
      skip = fun->skip;
    total = (buf.st_size - skip)/width;
  }

  /* check for monotonically desceasing widths */
  if( _FunKeyword(iext, "align", "EVENTS_ALIGN", tbuf, SZ_LINE) && 
      istrue(tbuf) ){
    for(i=1; i<got; i++){
      if(widths[i] > widths[i-1]){
	gwarning(stderr, 
		 "event alignment warning: %s (%d) follows %s (%d)\n",
		 names[i], widths[i], names[i-1], widths[i-1]);
      }
    }
  }

  /* fake the header to a binary table */
  ft_headsets(header, "XTENSION", 0, "BINTABLE", "FITS BINARY TABLE", 1);
  ft_headseti(header, "BITPIX", 0, 8, "Binary data", 1);
  ft_headseti(header, "NAXIS", 0, 2, "Table is a matrix", 1);
  ft_headseti(header, "NAXIS", 1, width, "Width of table in bytes", 1);
  ft_headseti(header, "NAXIS", 2, total, "Number of entries in table", 1);
  ft_headseti(header, "PCOUNT", 0, 0, "Random parameter count", 1);
  ft_headseti(header, "GCOUNT", 0, 1, "Group count", 1);
  ft_headseti(header, "TFIELDS", 0, got, "Number of fields in each row", 1);
  ft_headsets(header, "EXTNAME", 0, extname, "Table name", 1);
  ft_headseti(header, "EXTVER", 0, extver, "Version number of table", 1);
  /* info for each column */
  for(i=0; i<got; i++){
    ft_headsets(header, "TFORM", i+1, types[i], "Data type for field", 1);
    ft_headsets(header, "TTYPE", i+1, names[i], "Label for field", 1);
    if( dims[i] ){
      if( (tlmins[i] == (int)tlmins[i]) && (tlmaxs[i] == (int)tlmaxs[i]) ){
	ft_headseti(header, "TLMIN", i+1, tlmins[i], "Min. axis value", 1);
        ft_headseti(header, "TLMAX", i+1, tlmaxs[i], "Max. axis value", 1);
      }
      else{
	ft_headsetr(header, "TLMIN", i+1, tlmins[i], 7, "Min. axis value", 1);
        ft_headsetr(header, "TLMAX", i+1, tlmaxs[i], 7, "Max. axis value", 1);
      }
    }
    if( binsizs[i] != 1.0 ){
      ft_headsetr(header, "TDBIN", i+1, binsizs[i], 7, "Bin size", 1);
    }
    if( scaleds[i] ){
      ft_headseti(header, "TSCAL", i+1, tscales[i],
		  "phys_val = tzero + tscale * field_val", 1);
      ft_headseti(header, "TZERO", i+1, tzeros[i],
		  "phys_val = tzero + tscale * field_val", 1);
    }
  }

  /* get size of record and number of records */
  fun->rawsize = width;
  fun->total = total;
  fun->left = fun->total;

  /* now turn this into a table header */
  ft_syncdata(header);

done:
  /* free up the column strings */
  for(i=0; i<ncol; i++){
    if( names[i] ) xfree(names[i]);
    if( types[i] ) xfree(types[i]);
  }
  if( names )     xfree(names);
  if( types )     xfree(types);
  if( dims )	  xfree(dims);
  if( tlmins )	  xfree(tlmins);
  if( tlmaxs )	  xfree(tlmaxs);
  if( binsizs )	  xfree(binsizs);
  if( scaleds )	  xfree(scaleds);
  if( tscales )	  xfree(tscales);
  if( tzeros )	  xfree(tzeros);
  if( widths )    xfree(widths);
  if( neventdef ) xfree(neventdef);

  return header;
}

/*
 *
 * _FunRawEvOpen -- open a raw events file, 
 *	      set up binning and filtering parameters
 *
 */
#ifdef ANSI_FUNC
static Fun
_FunRawEvOpen(char *fname, char *tail, char *mode, char *eventdef)
#else
static Fun _FunRawEvOpen(fname, tail, mode, eventdef)
     char *fname;
     char *tail;
     char *mode;
     char *eventdef;
#endif
{
  Fun fun;
  char tbuf[SZ_LINE];

  /* sanity check */
  if( !eventdef || !*eventdef )
    return NULL;

  /* allocate a fun record */
  if( !(fun = _FunNew()) )
    return NULL;
  /* try to open the file  */
  if( !(fun->gio=gopen(fname, mode)) )
    goto error;
  /* save filename and mode */
  fun->fname = xstrdup(fname);
  fun->mode = xstrdup(mode);

  /* create a fake table header from the event description */
  if( !(fun->header = _FunRawEvHeader(fun, fname, tail, eventdef)) )
    goto error;
  
  /* its a valid event file */
  fun->type = FUN_EVENTS;
  /* no blanks for events */
  fun->doblank = 0;

  /* look for indication of whether these data are bigendian */
  if( _FunKeyword(tail, "endian", "EVENTS_ENDIAN", tbuf, SZ_LINE) )
    fun->endian = ((*tbuf == 'b') || (*tbuf == 'B'));
  else if( _FunKeyword(tail, "bigendian", "EVENTS_BIGENDIAN", tbuf, SZ_LINE) )
    fun->endian = istrue(tbuf);
  /* otherwise assume we have native format */
  else
    fun->endian = is_bigendian();

  /* look for indication of bitpix */
  if( _FunKeyword(tail, "bitpix", "EVENTS_BITPIX", tbuf, SZ_LINE) )
    fun->bitpix = atoi(tbuf);
  /* else assume a safe value */
  else
    fun->bitpix = 32;
  if( _FunKeyword(tail, "skip", "EVENTS_SKIP", tbuf, SZ_LINE) )
    fun->skip = atoi(tbuf);

  /* determine whether we are only processing specific rows */
  _FunRowNum(fun, tail, NULL);
  /* get the key for binning */
  _FunTableBincols(fun, tail, "EVENTS_BINCOLS");
  /* calculate the image length including padding */
  _FunImageSize(fun);
  /* now parse the section specification */
  _FunParseSection(fun, tail,
		   &(fun->x0), &(fun->x1), &(fun->y0), &(fun->y1),
		   &(fun->block), &(fun->btype), tail, SZ_LINE);
  /* get maxbufsize for table access */
  _FunMaxBufSize(fun, tail);
  /* what's left in the tail is the filter */
  fun->filter = xstrdup(tail);
  /* fill in the default selected columns */
  FunColumnSelect(fun, 0, "copy=reference", NULL);

  /* skip events header, if necessary */
  if( fun->skip )
    gskip(fun->gio, (off_t)fun->skip);

  /* common code */
  _FunOpenCommon(fun);

  /* return completed struct */
  return fun;

error:
  _FunFree(fun, 1);
  return NULL;
}

/*
 *
 * _FunArrayOpen -- open a raw array file, 
 *	      set up binning and filtering parameters
 *
 */
#ifdef ANSI_FUNC
static Fun
_FunArrayOpen(char *fname, char *tail, char *mode, char *arraydef)
#else
static Fun _FunArrayOpen(fname, tail, mode, arraydef)
     char *fname;
     char *tail;
     char *mode;
     char *arraydef;
#endif
{
  Fun fun;
  char tbuf[SZ_LINE];

  /* sanity check */
  if( !arraydef || !*arraydef )
    return NULL;

  /* allocate a fun record */
  if( !(fun = _FunNew()) )
    return NULL;
  /* try to open the file  */
  if( !(fun->gio=gopen(fname, mode)) )
    goto error;
  /* save filename and mode */
  fun->fname = xstrdup(fname);
  fun->mode = xstrdup(mode);

  /* look for indication of whether these data are bigendian */
  if( _FunKeyword(tail, "endian", "ARRAY_ENDIAN", tbuf, SZ_LINE) )
    fun->endian = ((*tbuf == 'b') || (*tbuf == 'B'));
  else if( _FunKeyword(tail, "bigendian", "ARRAY_BIGENDIAN", tbuf, SZ_LINE) )
    fun->endian = istrue(tbuf);
  /* otherwise assume we have native format */
  else
    fun->endian = is_bigendian();

  /* parse the array spec for dimension info */
  if( ParseArraySpec(arraydef, &(fun->dim1), &(fun->dim2),
		     &(fun->bitpix), &(fun->skip), &(fun->endian)) ){
    fun->min1 = 1;
    fun->max1 = fun->dim1;
    fun->min2 = 1;
    fun->max2 = fun->dim2;
  }
  else
    goto error;

  /* we have a valid array */
  fun->type = FUN_ARRAY;
  /* no blanks for arrays */
  fun->doblank = 0;

  /* make a dummy FITS header for this array */
  fun->header = ft_headinit(NULL, 0);
  ft_headsetl(fun->header, "SIMPLE", 0, 1, "Is FITS of standard type?", 1);
  ft_headseti(fun->header, "BITPIX", 0, fun->bitpix, "bits/pixel", 1);
  ft_headseti(fun->header, "NAXIS", 0, 2, "number of axes", 1);
  ft_headseti(fun->header, "NAXIS", 1, fun->dim1, "x axis dimension", 1);
  ft_headseti(fun->header, "NAXIS", 2, fun->dim2, "y axis dimension", 1);
  ft_syncdata(fun->header);

  /* calculate the image length including padding */
  _FunImageSize(fun);
  /* now parse the section specification */
  _FunParseSection(fun, tail,
		   &(fun->x0), &(fun->x1), &(fun->y0), &(fun->y1),
		   &(fun->block), &(fun->btype), tail, SZ_LINE);

  /* get maxbufsize for table access */
  _FunMaxBufSize(fun, tail);
  /* what's left in the tail is the filter */
  fun->filter = xstrdup(tail);

  /* skip array header, if necessary */
  if( fun->skip )
    gskip(fun->gio, (off_t)fun->skip);

  /* common code */
  _FunOpenCommon(fun);

  /* return completed struct */
  return fun;

error:
  _FunFree(fun, 1);
  return NULL;
}

/*
 *
 * _FunTextParam -- parse a line, looking for a valid parameter
 *
 */
#ifdef ANSI_FUNC
int
_FunTextParam(char *pdelims, 
	      char *lbuf, char *kbuf, char *vbuf, char *cbuf, int maxlen)
#else
  int _FunTextParam(pdelims, lbuf, kbuf, vbuf, cbuf, maxlen)
     char *pdelims;
     char *lbuf, *kbuf, *vbuf, *cbuf;
     int maxlen;
#endif
{
  int i;
  int hstate=0;
  int got=0;
  int kgot=0;
  int len=0;
  int docom=0;
  int dtable[PARSE_TABLE_SIZE];
  char *s;
  char *kptr, *vptr, *cptr;
  char *tptr;
  char *tbuf;
  char quote='\0';

  kptr = kbuf;
  vptr = vbuf;
  cptr = cbuf;
  *kptr = '\0';
  *vptr = '\0';
  *cptr = '\0';

  /* set up delim table for removing enclosing chars from keyword strings */
  memset(dtable, 0, PARSE_TABLE_SIZE*sizeof(int));
  if( pdelims && *pdelims ){
    /* set the delim table */
    for(s=pdelims; s && *s; s++){
      if( (i=(int)*s) == '\\' ){
	s++;
	if( *s == 'n' ) i = '\n';
	else if( *s == 't' ) i = '\t';
	else if( *s == 'r' ) i = '\r';
	else if( *s == 'f' ) i = '\014';
      }
      dtable[i] = 1;
    }
  }
  else{
    dtable[(int)'='] = 1;
    dtable[(int)':'] = 1;
  }

  tbuf = xstrdup(&lbuf[1]);
  nocr(tbuf);
  nowhite(tbuf, tbuf);
  for(tptr=tbuf; *tptr;){
    switch(hstate){
    /* gather up keyword */
    case 0:
      if( (*tptr == ' ') || (*tptr == '\t') || dtable[(int)*tptr] ){
	/* check for FITS-style comment */
	if( *tptr == '=' ) docom=1;
	hstate = 1;
	len = 0;
      }
      else{
	*kptr++ = *tptr;
	*kptr = '\0';
	if( ++len >= maxlen ) goto done;
      }
      tptr++;
      break;
    /* process whitespace, including = and : */
    case 1:
      /* skip past whitespace before value */
      if( (*tptr == ' ') || (*tptr == '\t') )
	tptr++;
      else if( dtable[(int)*tptr] ){
	/* check for FITS-style comment */
	if( *tptr == '=' ) docom=1;
	/*only one = or : allowed */
	kgot++;
	if( kgot > 1 ){
	  hstate = 2;
	  len = 0;
	}
	else{
	  tptr++;
	}
      }
      else{
	hstate = 2;
	len = 0;
      }
      break;
    /* check for quoted string */
    case 2:
      if( *tptr == '"' ){
	quote = '"';
	docom = 1;
	hstate = 3;
	len = 0;
      }
      else if( *tptr == '\'' ){
	quote = '\'';
	docom = 1;
	hstate = 3;
	len = 0;
      }
      else{
	*vptr++ = *tptr;
	*vptr = '\0';;
	if( ++len >= maxlen ) goto done;
	hstate = 4;
	len = 0;
      }
      tptr++;
      break;
    /* gather up value in quoted string */
    case 3:
      if( *tptr == quote ){
	hstate = 5;
	len = 0;
      }
      else{
	*vptr++ = *tptr;
	*vptr = '\0';
	if( ++len >= maxlen ) goto done;
      }
      tptr++;
      break;
    /* gather value up to whitespace */
    case 4:
      /* gather up value */
      if( (*tptr == ' ') || (*tptr == '\t') ){
	hstate = 5;
	len = 0;
      }
      else{
	*vptr++ = *tptr;
	*vptr = '\0';
	if( ++len >= maxlen ) goto done;
      }
      tptr++;
      break;
    /* skip past whitespace before possible comment */
    case 5:
      if( docom ){
	if( (*tptr == ' ') || (*tptr == '\t') ){
	  tptr++;
	}
	else{
	  hstate = 6;
	  len = 0;
	}
      }
      /* if comments are not wanted, then everything is value */
      else{
	*vptr++ = *tptr;
	*vptr = '\0';
	tptr++;
	if( ++len >= maxlen ) goto done;
      }
      break;
    /* look for comment */
    case 6:
      if( (*tptr == '/') ){
	tptr++;
	hstate = 7;
	len = 0;
      }
      /* extra chars but no comment char */
      else{
	got = 4;
	goto done2;
      }
      break;
    /* skip past whitespace before possible comment */
    case 7:
      if( (*tptr == ' ') || (*tptr == '\t') ){
	tptr++;
      }
      else{
	hstate = 8;
	len = 0;
      }
      break;
    /* gather up comment to end of line */
    case 8:
      *cptr++ = *tptr++;
      *cptr = '\0';
      if( ++len >= maxlen ) goto done;
      break;
    default:
      gerror(stderr, "unknown state (%d) processing text header\n", hstate);
      break;
    }
  }

  /* result code depends on what we gathered up */
done:
  if( *cbuf )
    got = 3;
  else if( *vbuf )
    got = 2;
  else if( *kbuf )
    got = 1;
  else
    got = 0;
  
done2:
  if( tbuf ) xfree(tbuf);
  return got;
}

/*
 *
 * _FunTextEvOpen -- open a ascii text events file, 
 *	      	     set up binning and filtering parameters
 *
 */
#ifdef ANSI_FUNC
static Fun
_FunTextEvOpen(char *fname, char *tail, char *mode, char *extn, 
	       char *iline, GIO ifd)
#else
static Fun _FunTextEvOpen(fname, tail, mode, extn, iline, ifd)
     char *fname;
     char *tail;
     char *mode;
     char *extn;
     char *iline;
     GIO ifd;
#endif
{
  int i=0, p=0, q=0, t=0;
  int got;
  int state;
  int len;
  int alen=0;
  int indx=0;
  int pgot;
  int dtype;
  long ival;
  double dval;
  char *delim=NULL, *comchars=NULL, *eot=NULL, *eptr=NULL;
  char *lptr;
  char tdelims[SZ_LINE];
  char pdelims[SZ_LINE];
  char tcomchars[SZ_LINE];
  char teot[SZ_LINE];
  char tnull1[SZ_LINE];
  char lbuf[SZ_LINE];
  char tbuf[SZ_LINE];
  char tbuf2[SZ_LINE];
  char pmode[SZ_LINE];
  char key[SZ_LINE];
  char val[SZ_LINE];
  char com[SZ_LINE];
  char eventdef[SZ_LINE];
  Parse parser;
  Parse fakep=NULL;
  ParsedLine line=NULL, header=NULL, data1=NULL;
  FITSHead theader=NULL, tmerge=NULL;
  Fun fun;

  /* allocate a fun record */
  if( !(fun = _FunNew()) )
    return NULL;
  /* save filename and mode */
  fun->fname = xstrdup(fname);
  fun->mode = xstrdup(mode);

  /* use passed-in fd or try to open the file  */
  if( ifd )
    fun->gio = ifd;
  else if( !(fun->gio=gopen(fname, mode)) )
    goto error;

  /* intialize */
  *tdelims = '\0';
  *tcomchars = '\0';
  *teot = '\0';
  *tnull1 = '\0';
  *eventdef = '\0';
  theader = ft_headinit(NULL, 0);

  /* process section and keywords (but not mode keywords) */
  eptr = extn;
  _FunKeyword(eptr, "delims", "TEXT_DELIMS", tdelims, SZ_LINE);
  _FunKeyword(eptr, "pdelims", "TEXT_DELIMS", pdelims, SZ_LINE);
  _FunKeyword(eptr, "comchars", "TEXT_COMCHARS", tcomchars, SZ_LINE);
  _FunKeyword(eptr, "cols", "TEXT_COLUMNS", eventdef, SZ_LINE);
  _FunKeyword(eptr, "eot", "TEXT_EOT", teot, SZ_LINE);
  _FunKeyword(eptr, "null1", "TEXT_NULL1", tnull1, SZ_LINE);
  _FunKeyword(eptr, "alen", "TEXT_ALEN", tbuf, SZ_LINE);
  alen = atoi(tbuf);
  if( extn && *extn ){
    indx = (int)strtol(extn, &eptr, 0);
    if( (eptr != extn) && (*eptr == ',') ) eptr++;
    if( strlen(eptr) && !strchr(eptr, '=') ){
      /* the whole string is the column spec */
      strncpy(eventdef, eptr, SZ_LINE);
      eptr = NULL;
    }
  }

  /* make sure CR is in the delim table */
  if(*tdelims && !strstr(tdelims, "\\n")) strncat(tdelims, "\\n", SZ_LINE);

  /* make sure index is valid */
  if( indx < 0 ) indx = 0;

  /* create parsers */
  if( tdelims && *tdelims ){
    fun->nparser = 1;
    fun->parsers = (Parse *)xcalloc(fun->nparser, sizeof(Parse));
    delim = tdelims;
    comchars = (*tcomchars ? tcomchars : PARSE_DEFAULT_COMCHARS);
    eot = (*teot ? teot : NULL);
    *pmode = '\0';
    if( eptr && *eptr ){
      strncat(pmode, eptr, SZ_LINE);
      strncat(pmode, ",", SZ_LINE);
    }
    strncat(pmode, PARSE_DEFAULT_NULLVALUES, SZ_LINE);
    strncat(pmode, ",", SZ_LINE);
    strncat(pmode, PARSE_DEFAULT_WHITESPACE, SZ_LINE);
    strncat(pmode, ",", SZ_LINE);
    strncat(pmode, PARSE_DEFAULT_UNITS, SZ_LINE);
    strncat(pmode, ",", SZ_LINE);
    strncat(pmode, PARSE_DEFAULT_COMEOT, SZ_LINE);
    if( !(fun->parsers[0] = ParseNew(delim, comchars, eot, pmode)) ){
      gwarning(stderr, "could not create parser #%d (%s)\n", p, delim);
      return NULL;
    }
  }
  else{
    fun->nparser = sizeof(ptype)/sizeof(PTypeRec);
    fun->parsers = (Parse *)xcalloc(fun->nparser, sizeof(Parse));
    for(p=0; p<fun->nparser; p++){
      delim = ptype[p].delims;
      comchars = (*tcomchars ? tcomchars : ptype[p].comchars);
      eot = (*teot ? teot : ptype[p].eot);
      *pmode = '\0';
      if( eptr && *eptr ){
	strncat(pmode, eptr, SZ_LINE);
      }
      if( *pmode ) strncat(pmode, ",", SZ_LINE);
      if( ptype[p].nullvalue ){
       	strncat(pmode, "nullvalues=true", SZ_LINE);
      }
      else{
	strncat(pmode, "nullvalues=false", SZ_LINE);
      }
      if( *pmode ) strncat(pmode, ",", SZ_LINE);
      if( ptype[p].whitespace ){
	strncat(pmode, "whitespace=true", SZ_LINE);
      }
      else{
	strncat(pmode, "whitespace=false", SZ_LINE);
      }
      if( *pmode ) strncat(pmode, ",", SZ_LINE);
      if( ptype[p].units ){
	strncat(pmode, "units=true", SZ_LINE);
      }
      else{
	strncat(pmode, "units=false", SZ_LINE);
      }
      if( *pmode ) strncat(pmode, ",", SZ_LINE);
      snprintf(tbuf, SZ_LINE, "comeot=%d", ptype[p].comeot);
      strncat(pmode, tbuf, SZ_LINE);
      if( !(fun->parsers[p] = ParseNew(delim, comchars, eot, pmode)) ){
	gwarning(stderr, "could not create text parser #%d (%s)\n", p, delim);
	goto error;
      }
    }
  }

  /* skip to the specified table */
  for(i=0; i<indx; i++){
    /* get next line, error if none */
    while( 1 ){
      /* see with previously-read line data, if necessary */
      if( iline && iline[0] ){
	strncpy(lbuf, iline, SZ_LINE);
	lptr = &lbuf[strlen(iline)];
	iline[0] = '\0';
      }
      else{
	lptr = lbuf;
      }
      if( !ggets(fun->gio, lptr, SZ_LINE) ){
	gwarning(stderr,
		 "text parser failure looking for data (section %d)\n",
		 indx);
	goto error;
      }
      /* analyze line and make sure one parser succeeded (even if its EOT) */
      if( !ParseAnalyze(fun->parsers, fun->nparser, lbuf) ){
	gwarning(stderr, "text parser failure analyzing line:\n%s", lbuf);
	goto error;
      }
      /* look for that valid parser */
      for(got=0, parser=NULL, p=0; p<fun->nparser; p++){
	if( fun->parsers[p]->state & PARSE_STATE_BAD) continue;
	if( fun->parsers[p]->state & PARSE_STATE_EOT){
	  for(q=0; q<fun->nparser; q++){
	    if( fun->parsers[q]->state & PARSE_STATE_BAD) continue;
	    /* a parser finding eot is still valid */
	    if( fun->parsers[q]->state & PARSE_STATE_EOT) continue;
	    /* all other parsers missed this eot, so they are "bad" */
	    fun->parsers[q]->state = PARSE_STATE_BADMATCH;
	  }
	  got++;
	  parser = NULL;
	  break;
	}
	else{
	  parser = fun->parsers[p];
	}
      }
      /* found another EOT */
      if( !parser ){
	if( got ){
	  /* bad parsers are kept bad, but others are reset */
	  for(p=0; p<fun->nparser; p++){
	    if( fun->parsers[p]->state & PARSE_STATE_BAD) continue;
	    /* if we are read and analyzed into the next table, back up a bit */
	    if( fun->parsers[p]->state & PARSE_STATE_NEXTLINE ){
	      line = ParseLineDup(fun->parsers[p], fun->parsers[p]->cur);
	      state = fun->parsers[p]->state;
	      state &= ~(PARSE_STATE_NEXTLINE|PARSE_STATE_EOT);
	      ParseReset(fun->parsers[p], line, state);
	    }
	    /* for comments, we just reprocess the line */
	    else if( fun->parsers[p]->state & PARSE_STATE_REDOLINE ){
	      ParseReset(fun->parsers[p], NULL, 0);
	      strncpy(tbuf, lbuf, SZ_LINE);
	      iline = tbuf;
	    }
	    else{
	      ParseReset(fun->parsers[p], NULL, 0);
	    }
	  }
	  break;
	}
	else{
	  gwarning(stderr,
		   "text parser failure looking for data (section %d)\n",
		   indx);
	  goto error;
	}
      }
    }
  }

/* look for a header */
header:
  /* seed with previously-read partial line, if necessary */
  if( iline && iline[0] ){
    strncpy(lbuf, iline, SZ_LINE-1);
    lbuf[SZ_LINE-1] = '\0';
    iline[0] = '\0';
    len = strlen(lbuf);
    /* if we have a whole line, process it */
    if( lbuf[len-1] == '\n' )
      goto header2;
    /* otherwise prepare to read rest of line */
    else
      lptr = &lbuf[len];
  }
  else{
    lptr = lbuf;
  }
  /* get next line, error if none */
  if( !ggets(fun->gio, lptr, SZ_LINE) ){
    gwarning(stderr, "text parser failure looking for a header\n");
    goto error;
  }

header2:
  /* parse and analyze line */
  if( !ParseAnalyze(fun->parsers, fun->nparser, lbuf) ){
    gwarning(stderr, "text parser failure analyzing header\n");
    goto error;
  }

  /* see if all are in data state, meaning we have found the header */
  for(p=0; p<fun->nparser; p++){
   if( fun->parsers[p]->state & PARSE_STATE_BAD ) continue;
    /* process comment into a header parameter */
    if( fun->parsers[p]->types[0] == PARSE_COMMENT ){
      if( strlen(lbuf) ){
	pgot = _FunTextParam(pdelims, lbuf, key, val, com, SZ_LINE);
	switch(pgot){
	case -1:
	  gerror(stderr, "internal error processing text params\n");
	  break;
	case 0:
	  break;
	case 1:
	  ft_headapps(theader, "COMMENT", 0, key, NULL);
	  break;
	case 2:
	case 3:
	  dtype = ParseDataType(val, &dval, &ival);
	  switch( dtype ){
	  case PARSE_COMMENT:
	  case PARSE_DASH:
	  case PARSE_EOL:
	  case PARSE_NULL:
	  case PARSE_EOT:
	    break;
	  case PARSE_FLOAT:
	    ft_headsetr(theader, key, 0, dval, 7, com, 1);
	    break;
	  case PARSE_HEXINT:
	  case PARSE_INTEGER:
	    ft_headseti(theader, key, 0, (int)ival, com, 1);
	    break;
	  case PARSE_STRING:
	    ft_headsets(theader, key, 0, val, com, 1);
	    break;
	  }
	  break;
	case 4:
	  /* clean up line */
	  nocr(lbuf);
	  nowhite(&lbuf[1], lbuf);
	  for(lptr=lbuf; *lptr; lptr++)
	    if( *lptr == '\t' ) *lptr = ' ';
	  ft_headapps(theader, "COMMENT", 0, lbuf, NULL);
	  break;
	}
      }
    }
    if( !(fun->parsers[p]->state & PARSE_STATE_DATA) ) goto header;
  }

  /* first line of data tells data types */
  for(p=0; p<fun->nparser; p++){
    if( !(fun->parsers[p]->state & PARSE_STATE_BAD) ){
      if( fun->parsers[p]->data1 ){
	data1 = fun->parsers[p]->data1;
	break;
      }
    }
  }
  /* make sure we read enough to get data */
  if( !data1 ) goto header;

  /* if the eventdef was not specified, make a guess */
  if( !*eventdef ){
    /* a header tells column names (or else we make then up) */
    for(p=0; p<fun->nparser; p++){
      if( !(fun->parsers[p]->state & PARSE_STATE_BAD) ){
	if( fun->parsers[p]->header ){
	  header = fun->parsers[p]->header;
	  break;
	}
      }
    }
    /* fake header line, if necessary */
    if( !header ){
      *tbuf = '\0';
      *lbuf = '\0';
      for(t=0; t<data1->ntoken; t++){
	snprintf(tbuf, SZ_LINE, "col%d", t+1);
	if( *lbuf )
	  strncat(lbuf, " ", SZ_LINE);
	strncat(lbuf, tbuf, SZ_LINE);
      }
      if( !(fakep = ParseNew(" \n", NULL, NULL, NULL)) ) goto error;
      if( !ParseAnalyze(&fakep, 1, lbuf) ) goto error;
      header = fakep->cur;
    }
    
    /* make up the event definition */
    for(t=0; t<header->ntoken; t++){
      *tbuf = '\0';
      switch(data1->tokens[t].type){
      case PARSE_FLOAT:
	strncpy(tbuf, header->tokens[t].sval, SZ_LINE);
	if( !strchr(header->tokens[t].sval, ':') ) 
	  strncat(tbuf, ":1D", SZ_LINE);
	break;
      case PARSE_HEXINT:
	strncpy(tbuf, header->tokens[t].sval, SZ_LINE);
	if( !strchr(header->tokens[t].sval, ':') ) 
	  strncat(tbuf, ":32X", SZ_LINE);
	break;
      case PARSE_INTEGER:
	strncpy(tbuf, header->tokens[t].sval, SZ_LINE);
	if( !strchr(header->tokens[t].sval, ':') ) 
	  strncat(tbuf, ":1J", SZ_LINE);
	break;
      case PARSE_STRING:
	strncpy(tbuf, header->tokens[t].sval, SZ_LINE);
	if( !strchr(header->tokens[t].sval, ':') ){
	  if( alen <=0 )
	    alen = MAX(strlen(data1->tokens[t].sval), PARSE_DEFAULT_ALEN);
	  snprintf(tbuf2, SZ_LINE, ":%dA", alen);
	  strncat(tbuf, tbuf2, SZ_LINE);
	}
       	break;
      case PARSE_NULL:
	if( *tnull1 ){
	  snprintf(tbuf, SZ_LINE, "%s:%s", header->tokens[t].sval, tnull1);
	}
	else{
	  gwarning(stderr,
	  "a NULL in row 1 (col %d) requires null1='i|d' or cols='...'\n",
		   t+1);
	  goto error;
	}
	break;
      /* ignore everything else */
      case PARSE_DASH:
      case PARSE_COMMENT:
      case PARSE_EOL:
	break;
      default:
	gwarning(stderr, "internal error: invalid type in text header: '%c'\n", 
	       data1->tokens[t].type);
	goto error;
      }
      if( *tbuf ){
	if( *eventdef ) strncat(eventdef, ",", SZ_LINE);
	strncat(eventdef, tbuf, SZ_LINE);
      }
    }
  }
    
  /* create a fake table header from the event description */
  if( !(fun->header = _FunRawEvHeader(fun, NULL, tail, eventdef)) )
    goto error;
  
  /* its a valid event file */
  fun->type = FUN_EVENTS;
  /* no blanks for events */
  fun->doblank = 0;

  /* data will not need conversion */
  fun->endian = is_bigendian();

  /* look for indication of bitpix */
  if( _FunKeyword(tail, "bitpix", "TEXT_BITPIX", tbuf, SZ_LINE) )
    fun->bitpix = atoi(tbuf);
  /* else assume a safe value */
  else
    fun->bitpix = 32;

  /* no header on text files */
  fun->skip = 0;

  /* determine whether we are only processing specific rows */
  _FunRowNum(fun, tail, NULL);
  /* get the key for binning */
  _FunTableBincols(fun, tail, "TEXT_BINCOLS");
  /* calculate the image length including padding */
  _FunImageSize(fun);
  /* now parse the section specification */
  _FunParseSection(fun, tail,
		   &(fun->x0), &(fun->x1), &(fun->y0), &(fun->y1),
		   &(fun->block), &(fun->btype), tail, SZ_LINE);
  /* get maxbufsize for table access */
  _FunMaxBufSize(fun, tail);
  /* what's left in the tail is the filter */
  fun->filter = xstrdup(tail);
  /* fill in the default selected columns */
  FunColumnSelect(fun, 0, "copy=reference", NULL);

  /* skip events header, if necessary */
  if( fun->skip )
    gskip(fun->gio, (off_t)fun->skip);

  /* merge in comments as params */
  if( theader && fun->header ){
    tmerge = ft_headmerge(fun->header, theader, 0);
    ft_headfree(fun->header, 1);
    ft_headfree(theader, 1);
    fun->header = tmerge;
  }

  /* common code */
  _FunOpenCommon(fun);

  /* free up tem space */
  if( fakep ) ParseFree(fakep);
  /* return completed struct */
  return fun;
  
error:
  /* free up tem space */
  if( fakep ) ParseFree(fakep);
  if( theader ) ft_headfree(theader, 1);
  _FunFree(fun, 1);
  return NULL;
}

/*
 *
 * semi-public routines, used by other modules
 *
 */

#ifdef ANSI_FUNC
Fun
_FunValid(Fun fun)
#else
Fun _FunValid(fun)
     Fun fun;
#endif
{
  Fun current;
  if( !fun ) return NULL;
  /* if a current list member is defined, test that one */
  if( fun->current )
    current = fun->current;
  /* else test this one */
  else
    current = fun;
  if( FUN_VALID(current) )
    return current;
  else
    return NULL;
}

/*
 *
 * _FunFree -- free up a fun record
 *
 */
#ifdef ANSI_FUNC
void
_FunFree(Fun fun, int flag)
#else
void _FunFree(fun, flag)
     Fun fun;
     int flag;
#endif
{
  int i;
  SaveBuf cur, tcur;

  if( fun ){
    _FunColumnFree(fun);
    if( fun->filter ){
      xfree(fun->filter);
      fun->filter = NULL;
    }
    if( fun->rawbuf ){
      xfree(fun->rawbuf);
      fun->rawbuf  = NULL;
    }
    if( fun->bincols ){
      xfree(fun->bincols);
      fun->bincols = NULL;
    }
    if( fun->activate ){
      xfree(fun->activate);
      fun->activate = NULL;
    }
    if( fun->header ){
      ft_headfree(fun->header, 1);
      fun->header = NULL;
    }
    if( fun->theader ){
      ft_headfree(fun->theader, 1);
      fun->theader = NULL;
    }
    if( fun->primary ){
      ft_headfree(fun->primary, 1);
      fun->primary = NULL;
    }
    /* close "save buffer" fds */
    if( fun->lefp ){
      fclose(fun->lefp);
      fun->lefp = NULL;
    }
    if( fun->lrfp ){
      fclose(fun->lrfp);
      fun->lrfp = NULL;
    }
    /* free save buffers */
    for(cur=fun->save; cur; ){
      tcur = cur->next;
      if(cur->ebuf ) xfree(cur->ebuf);
      if(cur->rbuf ) xfree(cur->rbuf);
      if( cur ) xfree(cur);
      cur = tcur;
    }
    fun->save = NULL;

    /* close filters, free masks */
    if( fun->filt && (fun->filt != NOFILTER) ){
      FilterClose(fun->filt);
      fun->filt = NULL;
    }
    if( fun->masks ){
      xfree(fun->masks);
      fun->masks = NULL;
    }
    fun->nmask = 0;

    /* free up text parsers */
    if( fun->parsers ){
      for(i=0; i<fun->nparser; i++){
	if( fun->parsers[i] ) ParseFree(fun->parsers[i]);
      }
      xfree(fun->parsers);
      fun->parsers = NULL;
    }

    /* if flag is true, we free up all space
       otherwise, its assumed we will do some more writing */
    if( flag ){
      if( fun->fname ) xfree(fun->fname);
      if( fun->mode )  xfree(fun->mode);
      if( fun->wcs )   wcsfree(fun->wcs);
      if( fun->wcs0 )  wcsfree(fun->wcs0);
      /* clear the magic flag as a last resort */
      fun->magic = 0;
      xfree(fun);
    }
    else{
      fun->type = 0;
      fun->overflow = 0;
      fun->rowsize = 0;
      fun->rawsize = 0;
      fun->total = 0;
      fun->left = 0;
      fun->io = 0;
      fun->bytes = 0;
      fun->rawbufsize = 0;
      /* re-init header for temp variables for next extension */
      fun->theader = ft_headinit(NULL, 0);
    }
  }
}
 
/*
 *
 * _FunFITSOpen -- open a FITS-type file, scan to the appropriate extension,
 *	      set up binning and filtering parameters
 *
 */
#ifdef ANSI_FUNC
Fun
_FunFITSOpen(Fun ifun, char *fname, char *mode)
#else
Fun _FunFITSOpen(ifun, fname, mode)
     Fun ifun;
     char *fname;
     char *mode;
#endif
{
  int indx;
  char file[SZ_LINE];
  char ext[SZ_LINE];
  char tail[SZ_LINE];
  char tbuf[SZ_LINE];
  char iline[SZ_LINE];
  GIO ifd=NULL;
  Fun fun;

  /* initialize iline */
  memset(iline, 0, SZ_LINE);

  /* use old fun handle */
  if( ifun ){
    fun = ifun;
  }
  /* allocate a fun record */
  else{
    if( !(fun = _FunNew()) )
      return NULL;
    /* save filename and mode */
    fun->fname = xstrdup(fname);
    fun->mode = xstrdup(mode);
  }

  /* set up a default ext name for searching table extensions */
  if( getenv("FITS_EXTNAME") == NULL )
    putenv("FITS_EXTNAME=EVENTS STDEVT");

  /* try to open as FITS */
  if(!(fun->gio=ft_fitsheadopenfd(fname, &(fun->header), tail, SZ_LINE, mode,
				  fun->gio, iline, &ifd))){
    /* if its not FITS, try opening as ascii text */
    if( iline[0] && isascii(iline[0]) && ifd ){
      if( !ifun ) _FunFree(fun, 1);
      *ext = '\0'; *tail = '\0';
      ft_parsefilename(fname, file, ext, SZ_LINE, &indx, tail, SZ_LINE);
      /* ignore extn, but put index there instead */
      if( indx > 0 ) snprintf(ext, SZ_LINE, "%d", indx);
      return _FunTextEvOpen(fname, tail, mode, ext, iline, ifd);
    }
    else{
      goto error;
    }
  }

  /* if we are delaying the real file positioning, exit here */
  if( !fun->header )
    return(fun);

  /* we have a valid header -- process its specific type */
  if( fun->header ){
    /* its a FITS image */
    if( fun->header->image ){
      /* its an image */
      fun->type = FUN_IMAGE;
      /* if its FITS, its big-endian */
      fun->endian = 1;
      /* get info about the image */
      fun->min1 = 1;
      fun->max1 = ft_naxis(fun->header, 1);
      fun->dim1 = tldim(fun->min1, fun->max1,  fun->binsiz1, 0);
      fun->min2 = 1;
      fun->max2 = ft_naxis(fun->header, 2);
      fun->dim2 = tldim(fun->min2, fun->max2,  fun->binsiz2, 0);
      if( (fun->max1 <0) || (fun->max2 <0) )
	return 0;
      fun->bitpix = ft_bitpix(fun->header);
      fun->dtype = fun->bitpix / FT_WORDLEN;

      /* determine scaling and blank parameters */
      fun->bscale = ft_bscale(fun->header);
      fun->bzero = ft_bzero(fun->header);
      if( (ft_hasbscale(fun->header) && (fun->bscale != 1.0)) || 
	  (ft_hasbzero(fun->header)  && (fun->bzero  != 0.0)) )
	fun->scaled = FUN_SCALE_EXISTS;
      fun->doblank = ft_hasblank(fun->header);
      fun->blank = ft_blank(fun->header);

      /* if more than 2 axes, we might have to skip to an image plane ...
	 we assume that the first 2 dimensions are the image */
      if( ft_naxes(fun->header) > 2 ){
	if( (fun->skip = _FunImageSkip(fun, tail)) <0 )
	  return 0;
	/* skip, if necessary */
	if( fun->skip )
	  gskip(fun->gio, (off_t)fun->skip);
      }
      /* calculate the image length including padding */
      _FunImageSize(fun);
      /* now parse the section specification */
      _FunParseSection(fun, tail,
		       &(fun->x0), &(fun->x1), &(fun->y0), &(fun->y1),
		       &(fun->block), &(fun->btype), tail, SZ_LINE);
      /* get maxbufsize for table access */
      _FunMaxBufSize(fun, tail);
      /* what's left in the tail is the filter */
      fun->filter = xstrdup(tail);
      /* common code */
      _FunOpenCommon(fun);
      /* return completed struct */
      return fun;
    }

    /* its a FITS binary table */
    else if( fun->header->table ){
      /* start positive */
      fun->type = FUN_TABLE;
      /* if its FITS, its big-endian */
      fun->endian = 1;
      /* no blanks for tables */
      fun->doblank = 0;
      /* get size of record and number of records */
      if( fun->header->basic ){
	fun->rawsize = fun->header->basic->naxis[0];
	fun->total = fun->header->basic->naxis[1];
	fun->left = fun->total;
      }
      /* look for indication of bitpix */
      if( _FunKeyword(tail, "bitpix", "FITS_BITPIX", tbuf, SZ_LINE) )
	fun->bitpix = atoi(tbuf);
      /* else assume a safe value */
      else
	fun->bitpix = 32;
      /* should we use indices if we find them? */
      if( _FunKeyword(tail, "idx_activate", "FILTER_IDX_ACTIVATE", tbuf, SZ_LINE) ){
	if( istrue(tbuf) )
	  fun->idx = 1;
	else if( isfalse(tbuf) )
	  fun->idx = -1;
      }
      /* else assume yes */
      else
	fun->idx = 1;
      if( _FunKeyword(tail, "idx_debug", "FILTER_IDX_DEBUG", tbuf, SZ_LINE) ){
	if( istrue(tbuf) )
	  idxdebug(1);
	else if( isfalse(tbuf) )
	  idxdebug(0);
      }
      /* determine whether we are only processing specific rows */
      _FunRowNum(fun, tail, NULL);
      /* get the key for binning */
      _FunTableBincols(fun, tail, "FITS_BINCOLS");
      /* calculate the image length including padding */
      _FunImageSize(fun);
      /* now parse the section specification */
      _FunParseSection(fun, tail,
		       &(fun->x0), &(fun->x1), &(fun->y0), &(fun->y1),
		       &(fun->block), &(fun->btype), tail, SZ_LINE);
      /* get maxbufsize for table access */
      _FunMaxBufSize(fun, tail);
      /* what's left in the tail is the filter */
      fun->filter = xstrdup(tail);
      /* fill in the default selected columns */
      FunColumnSelect(fun, 0, "copy=reference", NULL);
      /* common code */
      _FunOpenCommon(fun);
      /* return completed struct */
      return fun;
    }
    else
      goto error;
  }
  else
    goto error;

error:
  if( !ifun )
    _FunFree(fun, 1);
  return NULL;
}

/*
 *
 * _FunImageSize -- calculate the size of the image, including padding
 *
 */
#ifdef ANSI_FUNC
int
_FunImageSize(Fun fun)
#else
int _FunImageSize(fun)
     Fun fun;
#endif
{
  /* get data type code (which is abs(length)) */
  fun->dtype = fun->bitpix / FT_WORDLEN;
  /* this is the length of the image, without padding */
  fun->dlen = fun->dim1 * fun->dim2 * ABS(fun->dtype);
  /* determine the padding */
  if( (fun->dpad = FT_BLOCK - (fun->dlen % FT_BLOCK)) == FT_BLOCK )
    fun->dpad = 0;
  return fun->dlen;
}

/*
 *
 * _FunImageSize -- calculate the size of the image, including padding
 *
 */
#ifdef ANSI_FUNC
int
_FunMaxBufSize(Fun fun, char *tail)
#else
int _FunMaxBufSize(fun, tail)
     Fun fun;
     char *tail;
#endif
{
  char tbuf[SZ_LINE];

  fun->maxbufsize = 0;
  if( _FunKeyword(tail, "maxbufsize", "FUN_MAXBUFSIZE", tbuf, SZ_LINE) )
    fun->maxbufsize = atoi(tbuf);
  if( fun->maxbufsize <= 0 )
    fun->maxbufsize = FUN_MAXBUFSIZE;
  return(fun->maxbufsize);
}

/*
 *
 * public routines
 *
 */

/*
 *
 * FunOpen -- open a FITS-type file, scan to the appropriate extension,
 *	      set up binning and filtering parameters
 *
 */
#ifdef ANSI_FUNC
Fun
FunOpen(char *fname, char *mode, Fun ref)
#else
Fun FunOpen(fname, mode, ref)
     char *fname;
     char *mode;
     Fun ref;
#endif
{
  int i;
  int ltype;
  int lmem;
  int ip=0;
  int got=0;
  char name[SZ_LINE];
  char extn[SZ_LINE];
  char tail[SZ_LINE];
  char tbuf[SZ_LINE];
  char tbuf2[SZ_LINE];
  Fun tfun, fun=NULL;
  Fun cur, last=NULL;

  /* better have a filename and a mode */
  if( !fname || !*fname || !mode || !*mode )
    goto error;

  /* handle non-read opens specially -- we have to create a file */
  if( !strchr(mode, 'r') ){
    /* allocate a fun record */
    if( !(fun = _FunNew()) )
      goto error;
    /* for 'A' (mimick of append), we open r+ and seek to end */
    if( strchr(mode, 'A') ){
      if( !(fun->gio=gopen(fname, "r+")) )
	goto error;
      gseek(fun->gio, 0, SEEK_END);
    }
    /* normal open */
    else{
      if( !(fun->gio=gopen(fname, mode)) )
	goto error;
    }
    /* save filename and mode */
    fun->fname = xstrdup(fname);
    fun->mode = xstrdup(mode);
    /* make a primary header  */
    fun->primary = ft_headinit(NULL, 0);
    /* make a header for temp variables */
    fun->theader = ft_headinit(NULL, 0);
    /* if a ref struct was specified, make a ref from the fun struct */
    if( ref && (*(short *)ref == FUN_MAGIC) ){
      fun->ifun = ref;
      /* if we have delayed opening the reference for a copy, do it now */
      if( !fun->ifun->header ){
	/* set copy flag */
	fun->icopy = 1;
	/* set internal fitsy output channel */
	ft_cfile(fun->gio);
	/* open the file Now unless delay is requested */
	if( !strchr(mode, 'd') ){
	  _FunFITSOpen(fun->ifun, fun->ifun->fname, "r");
	  /* reset output channel */
	  ft_cfile(NULL);
	}
      }
      /* enter this handle in the backlink list of the reference file */
      for(i=0; i<FUN_MAXBFUN; i++){
	if( !fun->ifun->bfun[i] ){
	  fun->ifun->bfun[i] = fun;
	  break;
	}
      }
    }
  }
  /* this is "read" mode */
  else{
    /* fname might actually be a space-delimited list of files */
    ltype = LIST_FILEORDER;
    lmem = -1;
    while( _FunFile(fname, tbuf, SZ_LINE, &ip) ){
      if( _FunKeyword(tbuf, "listorder", "FUN_LISTORDER", tbuf2, SZ_LINE) ){
	if( !strncasecmp(tbuf2, "s", 1) )
	  ltype = LIST_SORT;
	else if( !strncasecmp(tbuf2, "t", 1) )
	  ltype = LIST_TRYSORT;
	else if( !strncasecmp(tbuf2, "u", 1) )
	  ltype = LIST_UNSORT;
	else if( !strncasecmp(tbuf2, "f", 1) )
	  ltype = LIST_FILEORDER;
	continue;
      }
      if( _FunKeyword(tbuf, "listmem", "FUN_LISTMEM", tbuf2, SZ_LINE) ){
	lmem = (int)(atof(tbuf2)*1000000);
	continue;
      }
      /* look for non-FITS array file */
      if( _FunSpecialFile(tbuf, "ARRAY", name, tail, extn) )
	tfun = _FunArrayOpen(name, tail, mode, extn);
      /* look for non-FITS events file */
      else if( _FunSpecialFile(tbuf, "EVENTS", name, tail, extn) )
	tfun = _FunRawEvOpen(name, tail, mode, extn);
      else if( _FunSpecialFile(tbuf, "TEXT", name, tail, extn) )
	tfun = _FunTextEvOpen(name, tail, mode, extn, NULL, NULL);
      /* just a normal FITS file */
      else
	tfun = _FunFITSOpen(NULL, tbuf, mode);
      if( !tfun )
	goto error;
      if( !fun )
	fun = tfun;
      else
	last->next = tfun;
      /* here is the bad bit: we will want to select on the input file fd, but
	 gio "files" do not always have an fd and when they do, they might
	 have two of them.  I am hacking this for the time being */
      if( (tfun->gio->type == GIO_DISK) || (tfun->gio->type == GIO_STREAM) )
	tfun->ifd = fileno(tfun->gio->fp);
      else if((tfun->gio->type == GIO_SOCKET) || (tfun->gio->type == GIO_PIPE))
	tfun->ifd = tfun->gio->ifd;
      else
	tfun->ifd = -1;
      last = tfun;
      got++;
    }
    /* only set current if we actually have a list */
    if( got == 1 )
      fun->ltype = LIST_NONE;
    else{
      fun->ltype = ltype;
      fun->lmem = lmem;
      fun->current = fun;
      /* list pre-processing */
      for(cur=fun; cur; cur=cur->next){
	/* set head of list */
	cur->head = fun;
	cur->lefp = tmpfile();
	cur->lrfp = tmpfile();
      }
    }
  }
  return(fun);

error:
  if( fun ) _FunFree(fun, 1);
  return NULL;
}

#ifdef ANSI_FUNC
void
FunFlush(Fun fun, char *plist)
#else
void FunFlush(fun, plist)
     Fun fun;
     char *plist;
#endif
{
  int i;
  int pad;
  int got;
  int doref=0;
  int dorest=0;
  int doback=0;
  int skip;
  char *pbuf;
  char tbuf[SZ_LINE];
  Fun tfun=NULL;
  GIO gios[1];

  /* make sure we have something to do */
  if( !_FunValid(fun) )
    return;

  /* check plist for copy mode -- but only if we have a reference handle */
  if( fun->ifun ){
    pbuf = xstrdup(plist);
    if( _FunKeyword(pbuf, "copy", NULL, tbuf, SZ_LINE) ){
      if( !strncasecmp(tbuf, "ref", 3) )
	doref = 1;
      else if( fun->icopy && !strncasecmp(tbuf, "remain", 6) )
	dorest = 1;
    }
    if( pbuf ) xfree(pbuf);
  }
  else{
    pbuf = xstrdup(plist);
    if( _FunKeyword(pbuf, "copy", NULL, tbuf, SZ_LINE) ){
      if( !strncasecmp(tbuf, "ref", 3) ){
	doback = 1;
	/* perform ordinary flush on each backlink */
	for(i=0; i<FUN_MAXBFUN; i++){
	  if( fun->bfun[i]  ){
	    FunFlush(fun->bfun[i], NULL);
	  }
	}
      }
    }
    if( pbuf ) xfree(pbuf);
  }

  /* If we did not set a type, perhaps because this is an output file
     that we did not write to, we might be able to set the type now to
     be the reference type. This will allow us to flush the ref header
     to the output file. */
  if( !fun->type && fun->ifun )
    fun->type = fun->ifun->type;

  /* if we are writing out the full reference extension, we just do it */
  if( doref ){
    /* copy header */
    ft_headwrite(fun->gio, fun->ifun->header);
    /* try to get back to the data, in case we read any already */
    gseek(fun->ifun->gio, ft_data(fun->ifun->header), 0);
    /* skip data, copying header if necessary */
    gios[0] = fun->gio;
    ft_dataskip(fun->ifun->gio, fun->ifun->header, gios, 1);
  }
  else{
    /* for writing, we might have to output the FITS header, padding, etc. */
    if( strchr(fun->mode, 'w') || strchr(fun->mode, 'a') || 
	strchr(fun->mode, 'A') ){
      /* process extension-specific flush */
      switch(fun->type){
      case FUN_IMAGE:
	/* might have to write the header */
	_FunImagePutHeader(fun, fun->dim1, fun->dim2, fun->bitpix);
	/* pad to end of extension, if necessary */
	pad = FT_BLOCK - (fun->bytes % FT_BLOCK);
	if( pad && (pad != FT_BLOCK) ){
	  pbuf = (char *)xcalloc(pad, sizeof(char));
	  gwrite(fun->gio, pbuf, 1, pad);
	  xfree(pbuf);
	  fun->io = 0;
	}
	break;
      case FUN_TABLE:
	/* might have to write the header */
	_FunTablePutHeader(fun);
	/* try to fix the naxis2 param, and if so, write padding as well */
	if( _FunFixNaxis2(fun) ){
	  pad = FT_BLOCK - (fun->bytes % FT_BLOCK);
	  if( pad && (pad != FT_BLOCK) ){
	    pbuf = (char *)xcalloc(pad, sizeof(char));
	    gwrite(fun->gio, pbuf, 1, pad);
	    xfree(pbuf);
	    fun->io = 0;
	  }
	}
	break;
      default:
	break;
      }
    }
  }

  /* if this is final flush, might have to copy remaining input extensions */
  if( dorest || doback ){
    if( dorest )
      tfun = fun->ifun;
    else if( doback )
      tfun = fun;
    /* might have to skip to end of this extension */
    switch(tfun->type){
    case FUN_IMAGE:
    case FUN_ARRAY:
      /* calculate bytes we should have read to end of extn, but didn't */
      skip = ((tfun->dim2*tfun->dim1)*ABS(tfun->dtype)) +
	tfun->dpad - tfun->curpos;
      if( skip ){
	gskip(tfun->gio, (off_t)skip);
	fun->curpos += skip;
      }
      break;
    case FUN_TABLE:
    case FUN_EVENTS:
      /* calculate bytes we should have read to end of extn, but didn't */
      skip = ((tfun->rawsize*tfun->total)+ft_pcount(tfun->header))-tfun->bytes;
      /* if there are any ... */
      if( skip ){
	/* ... it also means we did not skip the padding */
	pad = FT_BLOCK - ((tfun->rawsize*tfun->total) % FT_BLOCK);
	if( pad == FT_BLOCK ) pad = 0;
	skip += pad;
	/* do it now */
	gskip(tfun->gio, (off_t)skip);
      }
      break;
    default:
      break;
    }
    /* write out the rest of the reference file */
    while( (got=gread(tfun->gio, tbuf, sizeof(char), SZ_LINE)) >0 ){
      if( dorest ){
	gwrite(fun->gio, tbuf, sizeof(char), got);
      }
      else if( doback ){
	for(i=0; i<FUN_MAXBFUN; i++){
	  if( fun->bfun[i] && fun->bfun[i]->gio ){
	    gwrite(fun->bfun[i]->gio, tbuf, sizeof(char), got);
	  }
	}
      }
    }
  }

  /* we can free up most structure, but leave enough to carry on */
  _FunFree(fun,0);
  /* since we just flushed, flag that we can start again */
  fun->ops = 0;
}

#ifdef ANSI_FUNC
void
FunClose(Fun fun)
#else
void FunClose(fun)
     Fun fun;
#endif
{
  int i;
  Fun tfun;
  if( !_FunValid(fun) )
    return;
  /* close all files (this might be a list) */
  while( fun ){
    tfun = fun->next;
    FunFlush(fun, "copy=remaining");
    gclose(fun->gio);
    fun->gio = NULL;
    /* remove this handle from the backlink list of the reference file */
    if( fun->ifun && (*(short *)fun->ifun == FUN_MAGIC) ){
      for(i=0; i<FUN_MAXBFUN; i++){
	if( fun->ifun->bfun[i] == fun ){
	  fun->ifun->bfun[i] = (Fun)NULL;
	  break;
	}
      }
    }
    /* if this is a ref file for other files, clear all of those ifun flags */
    for(i=0; i<FUN_MAXBFUN; i++){
      if( fun->bfun[i] && (*(short *)fun->bfun[i] == FUN_MAGIC) )
	fun->bfun[i]->ifun = (Fun)NULL;;
    }
    _FunFree(fun,1);
    fun = tfun;
  }
}

