// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "ruler.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Ruler::Ruler(const Ruler& a) : Marker(a)
{
  p1 = a.p1;
  p2 = a.p2;
  p3 = a.p3;
  coordSystem = a.coordSystem;
  skyFrame = a.skyFrame;
  distSystem = a.distSystem;
  distFormat = a.distFormat;
  dist = a.dist;
  df =  Tk_GetFont(parent->interp, parent->tkwin, Tk_NameOfFont(a.df));
}

Ruler::Ruler(FrameBase* p, const Vector& ptr1, const Vector& ptr2,
	     CoordSystem sys, SkyFrame sky, 
	     CoordSystem distsys, SkyFormat distfor,
	     const char* clr, int w, const char* f, 
	     const char* t, unsigned short prop, const char* c,
	     const List<Tag>& tag) 
  : Marker(p, ptr1, clr, w, f, t, prop, c, tag)
{
  df = Tk_GetFont(parent->interp, parent->tkwin, "helvetica 9 normal");
  coordSystem = sys;
  skyFrame = sky;
  distSystem = distsys;
  distFormat = distfor;
  dist = 0;

  p1 = ptr1;
  p2 = ptr2;
  strcpy(type,"ruler");
  handle = new Vector[2];
  numHandle = 2;

  updateBBox();
}

Ruler::~Ruler()
{
  if (df)
    Tk_FreeFont(df);
}

double Ruler::getAngle()
{
  return asin((p2-p3).length()/(p2-p1).length());
}

void Ruler::updateBBox()
{
  FitsImage* ptr = parent->findFits(center);
  center = (p2-p1)/2 + p1;

  // calcP3
  Vector a = ptr->mapFromRef(p1,coordSystem,skyFrame);
  Vector b = ptr->mapFromRef(p2,coordSystem,skyFrame);
  p3 = ptr->mapToRef(Vector(b[0],a[1]),coordSystem,skyFrame);

  // calcDist
  dist = ptr->mapDistFromRef(p2, p1, distSystem, distFormat);

  // bound marker
  bbox = BBox(p1 * parent->refToCanvas, p2 * parent->refToCanvas);
  bbox.bound(p3 * parent->refToCanvas);

  // generate handles in canvas coords
  handle[0] = p1 * parent->refToCanvas;
  handle[1] = p2 * parent->refToCanvas;

  // make room for handles/arrows/text
  bbox.expand(5);

  // calculate overall bbox
  calcAllBBox();
}

void Ruler::updateCoords(const Matrix& m)
{
  p1*=m;
  p2*=m;
  p3*=m;

  Marker::updateCoords(m);
}

void Ruler::move(const Vector& v)
{
  p1+=v;
  p2+=v;

  updateBBox();
  doCallBack(&moveCB);
}

void Ruler::moveTo(const Vector& v)
{
  // v is the new location of center

  Vector diff = v - center;
  p1+=diff;
  p2+=diff;

  updateBBox();
  doCallBack(&moveCB);
}

void Ruler::edit(const Vector& v, int h)
{
  switch (h) {
  case 1:
    p1 = v;
    break;
  case 2:
    p2 = v;
    break;
  }

  updateBBox();
  doCallBack(&editCB);
}

#if __GNUC__ >= 3
void Ruler::ps(int mode)
{
  Marker::ps(mode);

  Vector a = p1 * parent->refToCanvas;
  Vector b = p2 * parent->refToCanvas;
  Vector c = p3 * parent->refToCanvas;

  ostringstream str;
  str << "[] 0 setdash" << endl;

  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << b.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << "[8 3] 0 setdash" << endl;

  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << c.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << "[8 3] 0 setdash" << endl;

  str << "newpath " 
      << b.TkCanvasPs(parent->canvas) << "moveto"
      << c.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  psArrow(a, (b-a).normalize());
  psArrow(b, (a-b).normalize());

  psDist(mode, (b-a)/2 + a);
}
#else
void Ruler::ps(int mode)
{
  Marker::ps(mode);

  Vector a = p1 * parent->refToCanvas;
  Vector b = p2 * parent->refToCanvas;
  Vector c = p3 * parent->refToCanvas;

  char buf[1024];
  ostrstream str(buf,1024);
  str << "[] 0 setdash" << endl;

  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << b.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << "[8 3] 0 setdash" << endl;

  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << c.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << "[8 3] 0 setdash" << endl;

  str << "newpath " 
      << b.TkCanvasPs(parent->canvas) << "moveto"
      << c.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << ends;
  Tcl_AppendResult(parent->interp, buf, NULL);

  psArrow(a, (b-a).normalize());
  psArrow(b, (a-b).normalize());

  psDist(mode, (b-a)/2 + a);
}
#endif

int Ruler::isOn(const Vector& v, const Vector& v1, const Vector& v2)
{
  // v : canvas coords
  // v1: ref coords
  // v2: ref coords

  // do this in canvas coords, not ref coords

  Vector l1 = v1 * parent->refToCanvas;
  Vector l2 = v2 * parent->refToCanvas;
  double a = (l2-l1).angle();

  Matrix m = Translate(-l1) * Rotate(a);
  Vector end = l2*m;
  Vector vv = v*m;
  return (vv[0]>0 && vv[0]<end[0] && vv[1]>-3 && vv[1]<3);
}

int Ruler::isIn(const Vector& vv)
{
  // test to see if point is on edge, if so, considered inside

  if (isOn(vv,p1,p2) || isOn(vv,p1,p3) || isOn(vv,p2,p3))
    return 1;

  /*
    v[0]-- x value of point being tested
    v[1]-- y value of point being tested

    This algorithm is from "An Introduction to Ray Tracing", Academic Press,
    1989, edited by Andrew Glassner, pg 53
    -- a point lies in a polygon if a line is extended from the point to 
    infinite
    in any direction and the number of intersections with the polygon is odd.
    This is valid for both concave and convex polygons.
    Points on a vertex are considered inside.
    Points on a edge are considered inside.
  */

  // analysis in ref coords
  Vector v = vv * parent->canvasToRef;

  int crossings = 0;   // number of crossings
  int sign;

  // for all edges

  for (int i=0; i<3; i++) {
    Vector v1, v2;
    switch (i) {
    case 0:
      v1 = p1-v;
      v2 = p2-v;
      sign = ((v1[1])>=0) ? 1 : -1; // init sign
      break;
    case 1:
      v1 = p2-v;
      v2 = p3-v;
      break;
    case 2:
      v1 = p3-v;
      v2 = p1-v;
      break;
    }

    int nextSign = (v2[1]>=0) ? 1 : -1; // sign holder for p2

    if (sign != nextSign) {
      if (v1[0]>0 && v2[0]>0)
	crossings++;
      else if (v1[0]>0 || v2[0]>0) {
	if (v1[0]-(v1[1]*(v2[0]-v1[0])/(v2[1]-v1[1])) > 0)
	  crossings++;
      }
      sign = nextSign;
    }
  }

  return fmod(float(crossings),float(2)) ? 1 : 0; // if odd, point is inside
}

void Ruler::setPoints(const Vector& v1, const Vector& v2)
{
  p1 = v1;
  p2 = v2;

  updateBBox();
  doCallBack(&editCB);
}

void Ruler::setCoordSystem(CoordSystem sys, SkyFrame sky, 
			   CoordSystem d, SkyFormat f)
{
  coordSystem = sys;
  skyFrame = sky;
  distSystem = d;
  distFormat = f;
  updateBBox();
}

// Private

void Ruler::render(Drawable drawable, const Matrix& matrix, double zoom,
		   RenderMode mode)
{
  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    break;
  case XOR:
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    break;
  }

  Vector a = (p1 * matrix).round();
  Vector b = (p2 * matrix).round();
  Vector c = (p3 * matrix).round();

  setLineNoDash();
  XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], 
	    (int)b[0], (int)b[1]);
  renderArrow(drawable, a, (b-a).normalize());
  renderArrow(drawable, b, (a-b).normalize());

  setLineDash();
  XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], 
	    (int)c[0], (int)c[1]);
  XDRAWLINE(display, drawable, gc, (int)b[0], (int)b[1], 
	    (int)c[0], (int)c[1]);

  renderDist(drawable, (b-a)/2 + a);
}

#if __GNUC__ >= 3
void Ruler::renderDist(Drawable drawable, Vector v)
{
  FitsImage* ptr = parent->findFits(center);

  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(df, &metrics);

  ostringstream str;
  switch (distSystem) {
  case IMAGE:
    str << dist << " img";
    break;
  case PHYSICAL:
    str << dist << " phy";
    break;
  case AMPLIFIER:
    str << dist << " amp";
    break;
  case DETECTOR:
    str << dist << " det";
    break;
  default:
    if (ptr->hasWCS(distSystem)) {
      if (ptr->hasWCSEqu(distSystem)) {
	switch (distFormat) {
	case DEGREES:
	  str << dist << " deg";
	  break;
	case ARCMIN:
	  str << dist << '\'';
	  break;
	case ARCSEC:
	  str << dist << '"';
	  break;
	}
      }
      else
	str << dist << " lin";
    }
  }

  char* buf = (char*)str.str().c_str();
  int ll = str.str().length();

  int w = Tk_TextWidth(df, buf, ll)+ll;
  int h = metrics.descent + metrics.ascent+2;
  int x = (int)(v[0]-w/2.);
  int y = (int)v[1];
  XSetForeground(display, gc, parent->getYellowColor());
  XFillRectangle(display, drawable, gc, x, y-metrics.ascent, w, h);

  XSetForeground(display, gc, parent->getBlackColor());
  XSetFont(display, gc, Tk_FontId(df));
  XDrawString(display, drawable, gc, x+1, y+1, buf, ll);
}

void Ruler::psDist(int mode, Vector v)
{
  FitsImage* ptr = parent->findFits(center);

  ostringstream vstr;
  switch (distSystem) {
  case IMAGE:
    vstr << dist << " img" << ends;
    break;
  case PHYSICAL:
    vstr << dist << " phy" << ends;
    break;
  case AMPLIFIER:
    vstr << dist << " amp" << ends;
    break;
  case DETECTOR:
    vstr << dist << " det" << ends;
    break;
  default:
    if (ptr->hasWCS(distSystem)) {
      if (ptr->hasWCSEqu(distSystem)) {
	switch (distFormat) {
	case DEGREES:
	  vstr << dist << " deg" << ends;
	  break;
	case ARCMIN:
	  vstr << dist << '\'' << ends;
	  break;
	case ARCSEC:
	  vstr << dist << '"' << ends;
	  break;
	}
      }
      else
	vstr << dist << " lin" << ends;
    }
  }

  // define font

  ostringstream str;

  Tcl_DString psFont;
  Tcl_DStringInit(&psFont);
  int psSize = Tk_PostscriptFontName(df, &psFont);

  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(df, &metrics);

  str << '/' << Tcl_DStringValue(&psFont) 
      << " findfont " << psSize << " scalefont setfont" << endl;

  Tcl_DStringFree(&psFont);

  // do rect

  switch (mode) {
  case 1:
    psColorGray("yellow", str);
    str << " setgray";
    break;
  case 2:
    psColorRGB("yellow", str);
    str << " setrgbcolor";
    break;
  }
  str << endl;

  char* vbuf = (char*)vstr.str().c_str();
  int vll = vstr.str().length();

  int width = Tk_TextWidth(df, vbuf, vll)-vll; // it works...
  int height = metrics.ascent;
  str << (v-Vector(width, 0)/2).TkCanvasPs(parent->canvas) << ' '
      << width << ' ' << height << " rectfill" << endl;

  // do text

  switch (mode) {
  case 1:
    psColorGray("black", str);
    str << " setgray";
    break;
  case 2:
    psColorRGB("black", str);
    str << " setrgbcolor";
    break;
  }
  str << endl;

  str << v.TkCanvasPs(parent->canvas) << " moveto" << endl
      << '(' << vbuf << ')' << endl
      << "dup stringwidth pop 2 div 0 exch sub " 
      << metrics.descent << " rmoveto show" << endl;

  // done

  str << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}

#else

void Ruler::renderDist(Drawable drawable, Vector v)
{
  FitsImage* ptr = parent->findFits(center);

  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(df, &metrics);

  char buf[256];
  ostrstream str(buf,256);
  switch (distSystem) {
  case IMAGE:
    str << dist << " img" << ends;
    break;
  case PHYSICAL:
    str << dist << " phy" << ends;
    break;
  case AMPLIFIER:
    str << dist << " amp" << ends;
    break;
  case DETECTOR:
    str << dist << " det" << ends;
    break;
  default:
    if (ptr->hasWCS(distSystem)) {
      if (ptr->hasWCSEqu(distSystem)) {
	switch (distFormat) {
	case DEGREES:
	  str << dist << " deg" << ends;
	  break;
	case ARCMIN:
	  str << dist << '\'' << ends;
	  break;
	case ARCSEC:
	  str << dist << '"' << ends;
	  break;
	}
      }
      else
	str << dist << " lin" << ends;
    }
  }

  int w = Tk_TextWidth(df, buf, strlen(buf))+strlen(buf);
  int h = metrics.descent + metrics.ascent+2;
  int x = (int)(v[0]-w/2.);
  int y = (int)v[1];
  XSetForeground(display, gc, parent->getYellowColor());
  XFillRectangle(display, drawable, gc, x, y-metrics.ascent, w, h);

  XSetForeground(display, gc, parent->getBlackColor());
  XSetFont(display, gc, Tk_FontId(df));
  XDrawString(display, drawable, gc, x+1, y+1, buf, strlen(buf));
}

void Ruler::psDist(int mode, Vector v)
{
  FitsImage* ptr = parent->findFits(center);

  char vbuf[32];
  ostrstream vstr(vbuf,32);
  switch (distSystem) {
  case IMAGE:
    vstr << dist << " img" << ends;
    break;
  case PHYSICAL:
    vstr << dist << " phy" << ends;
    break;
  case AMPLIFIER:
    vstr << dist << " amp" << ends;
    break;
  case DETECTOR:
    vstr << dist << " det" << ends;
    break;
  default:
    if (ptr->hasWCS(distSystem)) {
      if (ptr->hasWCSEqu(distSystem)) {
	switch (distFormat) {
	case DEGREES:
	  vstr << dist << " deg" << ends;
	  break;
	case ARCMIN:
	  vstr << dist << '\'' << ends;
	  break;
	case ARCSEC:
	  vstr << dist << '"' << ends;
	  break;
	}
      }
      else
	vstr << dist << " lin" << ends;
    }
  }

  // define font

  char buf[512];
  ostrstream str(buf,512);

  Tcl_DString psFont;
  Tcl_DStringInit(&psFont);
  int psSize = Tk_PostscriptFontName(df, &psFont);

  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(df, &metrics);

  str << '/' << Tcl_DStringValue(&psFont) 
      << " findfont " << psSize << " scalefont setfont" << endl;

  Tcl_DStringFree(&psFont);

  // do rect

  switch (mode) {
  case 1:
    psColorGray("yellow", str);
    str << " setgray";
    break;
  case 2:
    psColorRGB("yellow", str);
    str << " setrgbcolor";
    break;
  }
  str << endl;

  int width = Tk_TextWidth(df, vbuf, strlen(vbuf))-strlen(vbuf); // it works...
  int height = metrics.ascent;
  str << (v-Vector(width, 0)/2).TkCanvasPs(parent->canvas) << ' '
      << width << ' ' << height << " rectfill" << endl;

  // do text

  switch (mode) {
  case 1:
    psColorGray("black", str);
    str << " setgray";
    break;
  case 2:
    psColorRGB("black", str);
    str << " setrgbcolor";
    break;
  }
  str << endl;

  str << v.TkCanvasPs(parent->canvas) << " moveto" << endl
      << '(' << vbuf << ')' << endl
      << "dup stringwidth pop 2 div 0 exch sub " 
      << metrics.descent << " rmoveto show" << endl;

  // done

  str << ends;
  Tcl_AppendResult(parent->interp, buf, NULL);
}

#endif

// list

void Ruler::list(ostream& str, CoordSystem sys, SkyFrame sky,
		 SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector v1 = ptr->mapFromRef(p1,sys);
      Vector v2 = ptr->mapFromRef(p2,sys);
      str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	  << v2[0] << ',' << v2[1] << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v1 = ptr->mapFromRef(p1,sys,sky);
	    Vector v2 = ptr->mapFromRef(p2,sys,sky);
	    str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
		<< ',' << v2[0] << ',' << v2[1] << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    char ra1[16], ra2[16];
	    char dec1[16], dec2[16];
	    {
	      ptr->mapFromRef(p1,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	      string x(buf);
	      istringstream wcs(x);
#else
	      istrstream wcs(buf,64);
#endif
	      wcs >> ra1 >> dec1;
	    }
	    {
	      ptr->mapFromRef(p2,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	      string x(buf);
	      istringstream wcs(x);
#else
	      istrstream wcs(buf,64);
#endif
	      wcs >> ra2 >> dec2;
	    }
	    str << type << '(' << ra1 << ',' << dec1 << ',' 
		<< ra2 << ',' << dec2 << ')';
	  }
	  break;
	}
      }
      else {
	Vector v1 = ptr->mapFromRef(p1,sys);
	Vector v2 = ptr->mapFromRef(p2,sys);
	str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
	    << ',' << v2[0] << ',' << v2[1] << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Ruler::listProperties(ostream& str, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  if (delim == ';') 
    return;

  str << " # ruler=";

  listCoordSystem(str, coordSystem, skyFrame, ptr);
  str << ' ';

  switch (distSystem) {
  case IMAGE:
    str << "image";
    break;
  case PHYSICAL:
    str << "physical";
    break;
  case DETECTOR:
    str << "detector";
    break;
  case AMPLIFIER:
    str << "amplifier";
    break;
  default:
    if (ptr->hasWCS(distSystem)) {
      if (ptr->hasWCSEqu(distSystem)) {
	switch (distFormat) {
	case DEGREES:
	  str << "degrees";
	  break;
	case ARCMIN:
	  str << "arcmin";
	  break;
	case ARCSEC:
	  str << "arcsec";
	  break;
	}
      }
      else
	str << "pixels";
    }
  }

  listProps(str);
}
