// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "../../tk8.4.6/generic/tkCanvas.h"

#include "framebase.h"
#include "fitsimage.h"
#include "contour.h"

// this is copied from tkCanvPs.c

typedef struct TkPostscriptInfo {
    int x, y, width, height;	/* Area to print, in canvas pixel
				 * coordinates. */
    int x2, y2;			/* x+width and y+height. */
    char *pageXString;		/* String value of "-pagex" option or NULL. */
    char *pageYString;		/* String value of "-pagey" option or NULL. */
    double pageX, pageY;	/* Postscript coordinates (in points)
				 * corresponding to pageXString and
				 * pageYString. Don't forget that y-values
				 * grow upwards for Postscript! */
    char *pageWidthString;	/* Printed width of output. */
    char *pageHeightString;	/* Printed height of output. */
    double scale;		/* Scale factor for conversion: each pixel
				 * maps into this many points. */
    Tk_Anchor pageAnchor;	/* How to anchor bbox on Postscript page. */
    int rotate;			/* Non-zero means output should be rotated
				 * on page (landscape mode). */
    char *fontVar;		/* If non-NULL, gives name of global variable
				 * containing font mapping information.
				 * Malloc'ed. */
    char *colorVar;		/* If non-NULL, give name of global variable
				 * containing color mapping information.
				 * Malloc'ed. */
    char *colorMode;		/* Mode for handling colors:  "monochrome",
				 * "gray", or "color".  Malloc'ed. */
    int colorLevel;		/* Numeric value corresponding to colorMode:
				 * 0 for mono, 1 for gray, 2 for color. */
    char *fileName;		/* Name of file in which to write Postscript;
				 * NULL means return Postscript info as
				 * result. Malloc'ed. */
    char *channelName;		/* If -channel is specified, the name of
                                 * the channel to use. */
    Tcl_Channel chan;		/* Open channel corresponding to fileName. */
    Tcl_HashTable fontTable;	/* Hash table containing names of all font
				 * families used in output.  The hash table
				 * values are not used. */
    int prepass;		/* Non-zero means that we're currently in
				 * the pre-pass that collects font information,
				 * so the Postscript generated isn't
				 * relevant. */
    int prolog;			/* Non-zero means output should contain
				   the file prolog.ps in the header. */
} TkPostscriptInfo;

int FrameBase::postscriptProc(int prepass)
{
  TkPostscriptInfo* psInfo = (TkPostscriptInfo*)(((TkCanvas*)canvas)->psInfo);

  if (!visible)
    return TCL_OK;

  if (prepass)
    return TCL_OK;

  Vector org = psOrigin();

  // Image

  if (channelFits) {

    // save graphics state

    Tcl_AppendResult(interp, "gsave\n", NULL);

    float scale = psInfo->scale * psResolution / 72.;
    int width = (int)(options->width * scale);
    int height = (int)(options->height * scale);

    // scale

    {
#if __GNUC__ >= 3
      ostringstream str;
      str << org[0] << ' ' << org[1] << " translate" << endl
	  << 1/scale << ' ' << 1/scale << " scale" << endl
	  << ends;
      Tcl_AppendResult(interp, str.str().c_str(), NULL);
#else
      char buf[256];
      ostrstream str(buf,256);
      str << org[0] << ' ' << org[1] << " translate" << endl
	  << 1/scale << ' ' << 1/scale << " scale" << endl
	  << ends;
      Tcl_AppendResult(interp, buf, NULL);
#endif
    }

    switch (psLevel) {
    case 1:
      switch (psColorSpace) {
      case BW:
      case GRAY:
	psLevel1(GRAY, width, height, scale);
	break;
      case RGB:
      case CMYK:
	psLevel1(RGB, width, height, scale);
	break;
      }
      break;
    case 2:
      psLevel2(psColorSpace, width, height, scale);
      break;
    }

    // restore gstate

    Tcl_AppendResult(interp, "grestore\n", NULL);
  }

  // Markers & Contours & Grids

  // clip path
  // I don't know why the height needs to be reduced by one, however, it works

  {
#if __GNUC__ >= 3
    ostringstream str;
    str << org[0] << ' ' << org[1] << ' ' 
	<< options->width << ' ' << options->height-1
	<< " rectclip" << endl << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
#else
    char buf[256];
    ostrstream str(buf,256);
    str << org[0] << ' ' << org[1] << ' ' 
	<< options->width << ' ' << options->height-1
	<< " rectclip" << endl << ends;
    Tcl_AppendResult(interp, buf, NULL);
#endif
  }

  switch (psLevel) {
  case 1:
    switch (psColorSpace) {
    case BW:
    case GRAY:
      psContours(GRAY);
      if (grid)
	grid->ps(GRAY);

      markerbg();
      psMarkers(GRAY);
      markerfg();
      psMarkers(GRAY);

      if (useCrosshair)
	psCrosshair(GRAY);
      break;
    case RGB:
    case CMYK:
      psContours(RGB);
      if (grid)
	grid->ps(RGB);

      markerbg();
      psMarkers(RGB);
      markerfg();
      psMarkers(RGB);

      if (useCrosshair)
	psCrosshair(RGB);
      break;
    }
    break;
  case 2:
    psContours(psColorSpace);
    if (grid)
      grid->ps(psColorSpace);

    markerbg();
    psMarkers(psColorSpace);
    markerfg();
    psMarkers(psColorSpace);

    if (useCrosshair)
      psCrosshair(psColorSpace);
    break;
  }

  return TCL_OK;
}

#if __GNUC__ >= 3
void FrameBase::psLevel1Head(PSColorSpace mode, int width, int height)
{
  ostringstream str;
  switch (mode) {
  case BW:
  case GRAY:
    str << "/picstr " << dec << width << " string def" << endl
	<< width << ' ' << height << " 8 matrix" << endl
	<< "{currentfile picstr readhexstring pop}" << endl
	<< "image" << endl;
    break;
  case RGB:
  case CMYK:
    str << "/picstr " << dec << width*3 << " string def" << endl
	<< width << ' ' << height << " 8 matrix" << endl
	<< "{currentfile picstr readhexstring pop}" << endl
	<< "false 3 colorimage" << endl;
    break;
  }

  str << ends;

  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void FrameBase::psLevel1Head(PSColorSpace mode, int width, int height)
{
  char buf[512];
  ostrstream str(buf,512);
  switch (mode) {
  case BW:
  case GRAY:
    str << "/picstr " << dec << width << " string def" << endl
	<< width << ' ' << height << " 8 matrix" << endl
	<< "{currentfile picstr readhexstring pop}" << endl
	<< "image" << endl;
    break;
  case RGB:
  case CMYK:
    str << "/picstr " << dec << width*3 << " string def" << endl
	<< width << ' ' << height << " 8 matrix" << endl
	<< "{currentfile picstr readhexstring pop}" << endl
	<< "false 3 colorimage" << endl;
    break;
  }

  str << ends;

  Tcl_AppendResult(interp, buf, NULL);
}
#endif

Matrix FrameBase::psMatrix(float psZoom, int width, int height)
{
  Matrix m;
  if (!isIIS()) {
    // Note: imageCenter() is in IMAGE coords
    Vector center = imageCenter() * imageToData;

    m = Translate(-center) *
      FlipY() *
      Translate(center);
  }

  return m *
    Translate(-cursor) *                // translate to cursor position
    Rotate(rotation) *                  // rotate about cursor position
    Rotate(wcsRotation) *               // rotate about center position
    orientationMatrix *                 // flip x/y axis about cursor position
    wcsOrientationMatrix *              // flip x/y axis about center
    Scale(zoom) *                       // scale about cursor position
    Scale(psZoom) *

    FlipY() *
    Translate(Vector(width,height)/2.);
}

Vector FrameBase::psOrigin()
{
  // compute coordinates of lower-left corner of the image
  // taking into account the anchor position for the image.
  // this mapping is different than in widget.c, due to postscript origin 
  // (y is flipped)

  double xx = options->x;
  double yy = Tk_CanvasPsY(canvas, options->y);
     
  switch (options->anchor) {
  case TK_ANCHOR_NW:
    yy -= options->height;
    break;
  case TK_ANCHOR_N:
    xx -= options->width/2.0;
    yy -= options->height;
    break;
  case TK_ANCHOR_NE:
    xx -= options->width;
    yy -= options->height;
    break;
  case TK_ANCHOR_E:
    xx -= options->width;
    yy -= options->height/2.0;
    break;
  case TK_ANCHOR_SE:
    xx -= options->width;
    break;
  case TK_ANCHOR_S:
    xx -= options->width/2.0;
    break;
  case TK_ANCHOR_SW:
    break;
  case TK_ANCHOR_W:
    yy -= options->height/2.0;
    break;
  case TK_ANCHOR_CENTER:
    xx -= options->width/2.0;
    yy -= options->height/2.0;
    break;
  }

  return Vector(xx,yy);
}

void FrameBase::psContours(PSColorSpace cs)
{
  // render back to front
  // aux contours
  Contour* ptr=auxcontours.tail();
  while (ptr) {
    ptr->ps(cs);
    ptr=ptr->previous();
  }

  // main contour
  if (*currentcontour)
    (*currentcontour)->ps(cs);
}

// this routine attemps a work around for a problems with level 2 postscript
// and other software such as dvips and assemble

#if __GNUC__ >= 3
void FrameBase::psFix(ostringstream& ostr)
{
  string str = ostr.str();
  const char* buf = str.c_str();
  int size = str.length();
  char* a = (char*)buf;

  while (*a && a<buf+size) {
    // dvips version 5.55. Basically, we can't have %% at the start of a line
    if (*a=='\n' && *(a+1)=='%' && *(a+2)=='%') {
      // reverse the order so that we don't have a %% at the begining of a line
      *a++ = '%';
      *a++ = '%';
      *a++ = '\n';
    }

    // assemble. Basically, we can't have % at the start of a line
    else if (*a=='\n' && *(a+1)=='%') {
      // reverse the order so that we don't have a % at the begining of a line
      *a++ = '%';
      *a++ = '\n';
    }

    a++;
  }

  ostr.str(str);
}
#else
void FrameBase::psFix(char* buf, int size)
{
  char* a = buf;

  while (*a && a<buf+size) {
    // dvips version 5.55. Basically, we can't have %% at the start of a line
    if (*a=='\n' && *(a+1)=='%' && *(a+2)=='%') {
      // reverse the order so that we don't have a %% at the begining of a line
      *a++ = '%';
      *a++ = '%';
      *a++ = '\n';
    }

    // assemble. Basically, we can't have % at the start of a line
    else if (*a=='\n' && *(a+1)=='%') {
      // reverse the order so that we don't have a % at the begining of a line
      *a++ = '%';
      *a++ = '\n';
    }

    a++;
  }
}
#endif
