// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "tcl.h"
#include <Xlib.h>
#include <Xutil.h>

#include "frametrue.h"
#include "fitsimage.h"
#include "nan.h"
#include "ps.h"
#include "util.h"

// FrameTrueColor Member Functions

FrameTrueColor::FrameTrueColor(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item)
  : Frame(i, c, item)
{
  skipUpdate = 0;

  colormapData = NULL;
  colormapPM = 0;
  colormapXM = NULL;
  colormapGCXOR = 0;

  memset(bgTrueColor_,255,4);
  memset(nanTrueColor_,255,4);
}

FrameTrueColor::~FrameTrueColor()
{
  if (colormapXM)
    XDestroyImage(colormapXM);

  if (colormapPM)
    Tk_FreePixmap(display, colormapPM);

  if (colormapGCXOR)
    XFreeGC(display, colormapGCXOR);

  if (colormapData)
    delete [] colormapData;
}

void FrameTrueColor::bgColorCmd(const char* color)
{
  if (bgColorName)
    delete [] bgColorName;
  bgColorName = dupstr(color);
  bgColor = getXColor(bgColorName);
  encodeTrueColor(bgColor, bgTrueColor_);
  update(BASE);
}

void FrameTrueColor::nanColorCmd(const char* color)
{
  if (nanColorName)
    delete [] nanColorName;
  nanColorName = dupstr(color);
  nanColor = getXColor(nanColorName);
  encodeTrueColor(nanColor, nanTrueColor_);
  update(BASE);
}

void FrameTrueColor::fillXImage(XImage* xmap,
				int x0, int y0, int x1, int y1,
				double* (FitsImage::*getMatrix)())
{
  // we need a colorScale before we can render

  if (!colorScale)
    return;

  if (!currentFits)
    return;

  int& height = xmap->height;
  int bytesPerPixel = xmap->bits_per_pixel/8;

  double* m = (currentFits->*getMatrix)();
  int* params = currentFits->getDataParams(frScale.scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale->size() - 1;
  int last = length * bytesPerPixel;
  const unsigned char* table = colorScale->colors();

  double ll = currentFits->getLowDouble();
  double hh = currentFits->getHighDouble();
  double diff = hh - ll;

  for (int j=y0; j<y1; j++) {
    // the line may be padded at the end
    char* dest = xmap->data + j*xmap->bytes_per_line + x0*bytesPerPixel;

    for (int i=x0; i<x1; i++, dest+=bytesPerPixel) {

      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = currentFits->getValueDouble(((int)y)*srcw + (int)x);
	
	if (!isNaNd(value)) {
	  if (value <= ll)
	    memcpy(dest, table, bytesPerPixel);
	  else if (value >= hh)
	    memcpy(dest, table+last, bytesPerPixel);
	  else
	    memcpy(dest, table+((int)(((value - ll)/diff * length) + .5))*
		   bytesPerPixel,bytesPerPixel);
	}
	else
	  memcpy(dest, nanTrueColor_, bytesPerPixel);
      }
      else
	memcpy(dest, bgTrueColor_, bytesPerPixel);
    }
  }
}

void FrameTrueColor::fillXImageMosaic(XImage* xmap, 
				      int x0, int y0, int x1, int y1,
				      double* (FitsImage::*getMatrix)())
{
  // we need a colorScale before we can render

  if (!colorScale)
    return;

  int& height = xmap->height;
  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->colors();
  int bytesPerPixel = xmap->bits_per_pixel/8;
  int last = length * bytesPerPixel;

  for (int j=y0; j<y1; j++) {

    // the line may be padded at the end
    char* dest = xmap->data + j*xmap->bytes_per_line + x0*bytesPerPixel;

    for (int i=x0; i<x1; i++, dest+=bytesPerPixel) {
      memcpy(dest, bgTrueColor_, bytesPerPixel);

      FitsImage* ptr = fits;
      while (ptr) {
	double* m = (ptr->*getMatrix)();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = ptr->getDataParams(frScale.scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  double ll = ptr->getLowDouble();
	  double hh = ptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      memcpy(dest, table, bytesPerPixel);
	    else if (value >= hh)
	      memcpy(dest, table+last, bytesPerPixel);
	    else
	      memcpy(dest,
		     table+((int)(((value-ll)/diff*length)+.5))*bytesPerPixel,
		     bytesPerPixel);
	  }
	  else 
	    memcpy(dest, nanTrueColor_, bytesPerPixel);

	  break;
	}
	else 
	  ptr = ptr->next();
      }
    }
  }
}

void FrameTrueColor::fillXImageMosaicFast(XImage* xmap, 
					  int x0, int y0, int x1, int y1,
					  double* (FitsImage::*getMatrix)())
{
  // we need a colorScale before we can render

  if (!colorScale)
    return;

  if (!fits)
    return;

  int& height = xmap->height;
  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->colors();
  int bytesPerPixel = xmap->bits_per_pixel/8;
  int last = length * bytesPerPixel;

  FitsImage* ptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  for (int j=y0; j<y1; j++) {

    // the line may be padded at the end
    char* dest = xmap->data + j*xmap->bytes_per_line + x0*bytesPerPixel;

    for (int i=x0; i<x1; i++, dest+=bytesPerPixel) {
      memcpy(dest, bgTrueColor_, bytesPerPixel);
      double x,y;

      if (ptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      memcpy(dest, table, bytesPerPixel);
	    else if (value >= hh)
	      memcpy(dest, table+last, bytesPerPixel);
	    else
	      memcpy(dest,
		     table+((int)(((value-ll)/diff*length)+.5))*bytesPerPixel,
		     bytesPerPixel);
	  }
	  else 
	    memcpy(dest, nanTrueColor_, bytesPerPixel);

	}
	else
	  ptr = NULL;
      }

      if (!ptr) {
	ptr = fits;
	while (ptr) {
	  m = (ptr->*getMatrix)();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = ptr->getDataParams(frScale.scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	    ll = ptr->getLowDouble();
	    hh = ptr->getHighDouble();
	    diff = hh - ll;

	    if (!isNaNd(value)) {
	      if (value <= ll)
		memcpy(dest, table, bytesPerPixel);
	      else if (value >= hh)
		memcpy(dest, table+last, bytesPerPixel);
	      else
		memcpy(dest,
		       table+((int)(((value-ll)/diff*length)+.5))*
		       bytesPerPixel,
		       bytesPerPixel);
	    }
	    else 
	      memcpy(dest, nanTrueColor_, bytesPerPixel);

	    break;
	  }
	  else 
	    ptr = ptr->next();
	}
      }
    }
  }
}

void FrameTrueColor::colormapCmd(int id, float b, float c, int i, 
				 unsigned short* index, unsigned char* cells, 
				 int cnt)
{
  // first check for change

  if (cmapID == id && bias == b && contrast == c && invert == i && colorCells)
    return;

  // we got a change

  cmapID = id;
  bias = b;
  contrast = c;
  invert = i;

  updateColorCells(index, cells, cnt);
  updateColorScale();

  if (fitsCount) {
    update(BASE); // always update
    skipUpdate = 0;
  }
  else {
    if (!skipUpdate) {
      update(BASE); // update once
      skipUpdate = 1;
    }      
  }
}

void FrameTrueColor::colormapBeginCmd()
{
  colormapBeginCmd(Vector(options->width,options->height)/2*widgetToCanvas);
}

void FrameTrueColor::colormapBeginCmd(const Vector& w)
{
  // we need some fits data
  // we assume the colorScale length will not change during motion calls

  if (!currentFits)
    return;

  int width = options->width;
  int height = options->height;
  cmapOffset = Vector();
  if (cmapArea) {
    width = height = cmapArea;

    switch (cmapMode) {
    case CENTER:
      cmapOffset = (Vector(options->width,options->height) - 
		    Vector(width,height))/2;
      break;
    case CLICK: 
      {
	Vector where = w * canvasToWidget;
	if (where[0] >= 0 && where[0] < options->width &&
	    where[1] >= 0 && where[1] < options->height)
	  cmapOffset = where - Vector(width,height)/2;
	else
	  cmapOffset = (Vector(options->width,options->height) - 
			Vector(width,height))/2;
      }
      break;
    }

    if (cmapOffset[0] < 0) {
      width += (int)cmapOffset[0];
      cmapOffset[0] = 0;
    }
    if (cmapOffset[0]+width >= options->width)
      width = options->width - (int)cmapOffset[0];

    if (cmapOffset[1] < 0) {
      height += (int)cmapOffset[1];
      cmapOffset[1] = 0;
    }

    if (cmapOffset[1]+height >= options->height)
      height = options->height - (int)cmapOffset[1];
  }

  // Create XImage

  if (!(colormapXM = XGETIMAGE(display, pixmap, 0, 0, 
			       width, height, AllPlanes, ZPixmap))) {
    cerr << "FrameTrueColor Internal Error: Unable to Create Colormap XImage" 
	 << endl;
    exit(1);
  }

  // Create Pixmap

  colormapPM = 
    Tk_GetPixmap(display, Tk_WindowId(tkwin), width, height, depth);
  if (!colormapPM) {
    cerr << "FrameTrueColor Internal Error: Unable to Create Colormap Pixmap" 
	 << endl;
    exit(1);
  }

  // colormapGCXOR

  colormapGCXOR = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);

  // Create table index array

  if (colormapData)
    delete [] colormapData;
  colormapData = new long[width*height];
  if (!colormapData) {
    cerr << "FrameTrueColor Internal Error: Unable to alloc tmp data array" 
	 << endl;
    exit(1);
  }

  // fill data array

  switch (*currentMode) {
  case SINGLE:
    colormapBeginSingle(width, height);
    break;
  case MOSAIC:
    if (mosaicFast && *currentMosaic == IRAF)
      colormapBeginMosaicFast(width, height);
    else
      colormapBeginMosaic(width, height);

    break;
  }
}

void FrameTrueColor::colormapBeginSingle(int width, int height)
{
  // we need a colorScale before we can render

  if (!colorScale)
    return;

  int xoff = (int)cmapOffset[0];
  int yoff = (int)cmapOffset[1];

  int bytesPerPixel = colormapXM->bits_per_pixel/8;

  double* mm = currentFits->getmWidgetToData();
  int* params = currentFits->getDataParams(frScale.scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale->size() - 1;
  int last = length * bytesPerPixel;

  double ll = currentFits->getLowDouble();
  double hh = currentFits->getHighDouble();
  double diff = hh - ll;

  long* dest = colormapData;
  for (int j=yoff; j<yoff+height; j++) {
    for (int i=xoff; i<xoff+width; i++, dest++) {
      *dest = -2; // initialize

      double x = i*mm[0] + j*mm[3] + mm[6];
      double y = i*mm[1] + j*mm[4] + mm[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = currentFits->getValueDouble(((int)y)*srcw + (int)x);
	
	if (!isNaNd(value)) {
	  if (value <= ll)
	    *dest = 0;
	  else if (value >= hh)
	    *dest = last;
	  else
	    *dest = (int)(((value - ll)/diff * length) + .5)*bytesPerPixel;
	}
	else
	  *dest = -1;
      }
    }
  }
}

void FrameTrueColor::colormapBeginMosaic(int width, int height)
{
  // we need a colorScale before we can render

  if (!colorScale)
    return;

  int xoff = (int)cmapOffset[0];
  int yoff = (int)cmapOffset[1];

  int bytesPerPixel = colormapXM->bits_per_pixel/8;

  int length = colorScale->size() - 1;
  int last = length * bytesPerPixel;

  // scan fits

  long* dest = colormapData;
  for (int j=yoff; j<yoff+height; j++) {
    for (int i=xoff; i<xoff+width; i++, dest++) {
      *dest = -2;

      FitsImage* ptr = fits;
      while (ptr) {
	double* m = ptr->getmWidgetToData();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = ptr->getDataParams(frScale.scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  double ll = ptr->getLowDouble();
	  double hh = ptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      *dest = 0;
	    else if (value >= hh)
	      *dest = last;
	    else
	      *dest = (int)(((value - ll)/diff * length) + .5)*bytesPerPixel;
	  }
	  else
	    *dest = -1;

	  break;
	}
	else 
	  ptr = ptr->next();
      }
    }
  }
}

void FrameTrueColor::colormapBeginMosaicFast(int width, int height)
{
  // we need a colorScale before we can render

  if (!colorScale)
    return;

  int xoff = (int)cmapOffset[0];
  int yoff = (int)cmapOffset[1];

  int bytesPerPixel = colormapXM->bits_per_pixel/8;

  int length = colorScale->size() - 1;
  int last = length * bytesPerPixel;

  FitsImage* ptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  // scan fits

  long* dest = colormapData;
  for (int j=yoff; j<yoff+height; j++) {
    for (int i=xoff; i<xoff+width; i++, dest++) {
      double x,y;
      *dest = -2; // initialize

      if (ptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      *dest = 0;
	    else if (value >= hh)
	      *dest = last;
	    else
	      *dest = (int)(((value - ll)/diff * length) + .5)*bytesPerPixel;
	  }
	  else
	    *dest = -1;
	}
	else
	  ptr = NULL;
      }

      if (!ptr) {
	ptr = fits;
	while (ptr) {
	  m = ptr->getmWidgetToData();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = ptr->getDataParams(frScale.scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	    ll = ptr->getLowDouble();
	    hh = ptr->getHighDouble();
	    diff = hh - ll;

	    if (!isNaNd(value)) {
	      if (value <= ll)
		*dest = 0;
	      else if (value >= hh)
		*dest = last;
	      else
		*dest = (int)(((value - ll)/diff * length) + .5)*bytesPerPixel;
	    }
	    else
	      *dest = -1;

	    break;
	  }
	  else 
	    ptr = ptr->next();
	}
      }
    }
  }
}

void FrameTrueColor::colormapMotionCmd(int id, float b, float c, int i, 
				       unsigned short* index, 
				       unsigned char* cells, int cnt)
{
  // we need a colorScale before we can render

  if (!colorScale)
    return;

  // first check for change

  if (cmapID == id && bias == b && contrast == c && invert == i && colorCells)
    return;

  // we got a change

  cmapID = id;
  bias = b;
  contrast = c;
  invert = i;

  updateColorCells(index, cells, cnt);
  updateColorScale();

  // if we have no data, stop now

  if (!currentFits) 
    return;

  // clear ximage

  int& width = colormapXM->width;
  int& height = colormapXM->height;

  int bytesPerPixel = colormapXM->bits_per_pixel/8;
  int& bytesPerLine = colormapXM->bytes_per_line;

  const unsigned char* table = colorScale->colors();

  long* src = colormapData;
  for (int j=0; j<height; j++) {
    // line may be padded at end
    char* dest = colormapXM->data + j*bytesPerLine;

    for (int i=0; i<width; i++, src++, dest+=bytesPerPixel)
      if (*src >=0)
	memcpy(dest, table+(*src), bytesPerPixel);
      else if (*src == -1)
	memcpy(dest, nanTrueColor_, bytesPerPixel);
      else
	memcpy(dest, bgTrueColor_, bytesPerPixel);
  }

  // XImage to Pixmap

  XPutImage(display, colormapPM, gc, colormapXM, 0, 0, 0, 0, width, height);

  // Display Pixmap

  Vector d = cmapOffset * widgetToWindow;
  XCopyArea(display, colormapPM, Tk_WindowId(tkwin), colormapGCXOR, 0, 0, 
	    width, height, (int)d[0], (int)d[1]);
}

void FrameTrueColor::colormapEndCmd(int id, float b, float c, int i, 
				    unsigned short* index, 
				    unsigned char* cells, int cnt)
{
  if (colormapXM) {
    XDestroyImage(colormapXM);
    colormapXM = NULL;
  }

  if (colormapPM) {
    Tk_FreePixmap(display, colormapPM);
    colormapPM = 0;
  }

  if (colormapGCXOR) {
    XFreeGC(display, colormapGCXOR);
    colormapGCXOR = 0;
  }

  if (colormapData) {
    delete [] colormapData;
    colormapData = NULL;
  }

  update(BASE); // always update
}

void FrameTrueColor::rotateMotion()
{
  // Rotate from src to dest

  Vector c = Vector(options->width,options->height)/2;
  Matrix m = (Translate(-c) * Rotate(rotation-rotateRotation) * 
	      Translate(c)).invert();
  double* mm = m.mm();

  int& width = options->width;
  int& height = options->height;
  char* src = rotateSrcXM->data;

  int bytesPerPixel = rotateDestXM->bits_per_pixel/8;

  for (int j=0; j<height; j++) {
    // the line may be padded at the end
    char* dest = rotateDestXM->data + j*rotateDestXM->bytes_per_line;

    for (int i=0; i<width; i++, dest+=bytesPerPixel) {
      double x = i*mm[0] + j*mm[3] + mm[6];
      double y = i*mm[1] + j*mm[4] + mm[7];

      if (x >= 0 && x < width && y >= 0 && y < height)
	memcpy(dest, 
	       src + ((int)y)*rotateDestXM->bytes_per_line+
	       ((int)x)*bytesPerPixel, 
	       bytesPerPixel);
      else
	memcpy(dest, bgTrueColor_, bytesPerPixel);
    }
  }

  // XImage to Pixmap

  XPutImage(display, rotatePM, gc, rotateDestXM,
	    0, 0, 0, 0, options->width, options->height);

  // Display Pixmap

  Vector d = Vector(0,0) * widgetToWindow;
  XCopyArea(display, rotatePM, Tk_WindowId(tkwin), rotateGCXOR, 0, 0, 
	    options->width, options->height, (int)d[0], (int)d[1]);
}

#if __GNUC__ >= 3
void FrameTrueColor::psLevel2Head(PSColorSpace mode, int width, int height)
{
  ostringstream str;

  switch (mode) {
  case BW:
  case GRAY:
    str << "/DeviceGray setcolorspace" << endl;
    break;
  case RGB:
    str << "/DeviceRGB setcolorspace" << endl;
    break;
  case CMYK:
    str << "/DeviceCMYK setcolorspace" << endl;
    break;
  }

  str << "<<" << endl
      << "/ImageType 1" << endl
      << "/Width " << dec << width << endl
      << "/Height " << dec << height << endl
      << "/BitsPerComponent 8" << endl;

  switch (mode) {
  case BW:
  case GRAY:
    str << "/Decode [0 1]" << endl;
    break;
  case RGB:
    str << "/Decode [0 1 0 1 0 1]" << endl;
    break;
  case CMYK:
    str << "/Decode [0 1 0 1 0 1 0 1]" << endl;
    break;
  }

  if (psInterpolate)
    str << "/Interpolate true" << endl;

  str << "/ImageMatrix matrix" << endl
      << "/DataSource currentfile" << endl
      << "/ASCII85Decode filter" << endl
      << "/RunLengthDecode filter" << endl
      << ">>" << endl
      << "image" << endl
      << ends;

  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void FrameTrueColor::psLevel2Head(PSColorSpace mode, int width, int height)
{
  char buf[4096];
  ostrstream str(buf,4096);

  switch (mode) {
  case BW:
  case GRAY:
    str << "/DeviceGray setcolorspace" << endl;
    break;
  case RGB:
    str << "/DeviceRGB setcolorspace" << endl;
    break;
  case CMYK:
    str << "/DeviceCMYK setcolorspace" << endl;
    break;
  }

  str << "<<" << endl
      << "/ImageType 1" << endl
      << "/Width " << dec << width << endl
      << "/Height " << dec << height << endl
      << "/BitsPerComponent 8" << endl;

  switch (mode) {
  case BW:
  case GRAY:
    str << "/Decode [0 1]" << endl;
    break;
  case RGB:
    str << "/Decode [0 1 0 1 0 1]" << endl;
    break;
  case CMYK:
    str << "/Decode [0 1 0 1 0 1 0 1]" << endl;
    break;
  }

  if (psInterpolate)
    str << "/Interpolate true" << endl;

  str << "/ImageMatrix matrix" << endl
      << "/DataSource currentfile" << endl
      << "/ASCII85Decode filter" << endl
      << "/RunLengthDecode filter" << endl
      << ">>" << endl
      << "image" << endl
      << ends;

  Tcl_AppendResult(interp, buf, NULL);
}
#endif

#if __GNUC__ >= 3
void FrameTrueColor::psLevel2Single(PSColorSpace mode,
				    int width, int height, float scale)
{
  if (!currentFits)
    return;

  RLEAscii85 filter;
  currentFits->updatePS(psMatrix(scale, width, height));
  double* m = currentFits->getmPSToData();
  int* params = currentFits->getDataParams(frScale.scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();

  double ll = currentFits->getLowDouble();
  double hh = currentFits->getHighDouble();
  double diff = hh - ll;

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = bgColor->red;
      unsigned char green = bgColor->green;
      unsigned char blue = bgColor->blue;

      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = currentFits->getValueDouble(((int)y)*srcw + (int)x);
	
	if (!isNaNd(value)) {
	  if (value <= ll) {
	    blue = table[0];
	    green = table[1];
	    red = table[2];
	  }
	  else if (value >= hh) {
	    blue = table[length*3];
	    green = table[length*3+1];
	    red = table[length*3+2];
	  }
	  else {
	    int l = (int)(((value - ll)/diff * length) + .5);
	    blue = table[l*3];
	    green = table[l*3+1];
	    red = table[l*3+2];
	  }
	}
	else {
	  blue = nanColor->blue;
	  green = nanColor->green;
	  red = nanColor->red;
	}
      }

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }
    str << ends;
    psFix(str);
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(str);
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void FrameTrueColor::psLevel2Single(PSColorSpace mode,
				    int width, int height, float scale)
{
  if (!currentFits)
    return;

  RLEAscii85 filter;

  // size = 2 char per byte * 4 bytes per pixel + estimate of endls + 
  // endl + ends

  int size = (int)(width*4*2*(1 + 1./80.) +.5) + 2;
  char* buf = new char[size];

  currentFits->updatePS(psMatrix(scale, width, height));

  double* m = currentFits->getmPSToData();
  int* params = currentFits->getDataParams(frScale.scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();

  double ll = currentFits->getLowDouble();
  double hh = currentFits->getHighDouble();
  double diff = hh - ll;

  for (int j=0; j<height; j++) {
    ostrstream str(buf,size);

    for (int i=0; i<width; i++) {
      unsigned char red = bgColor->red;
      unsigned char green = bgColor->green;
      unsigned char blue = bgColor->blue;

      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = currentFits->getValueDouble(((int)y)*srcw + (int)x);
	
	if (!isNaNd(value)) {
	  if (value <= ll) {
	    blue = table[0];
	    green = table[1];
	    red = table[2];
	  }
	  else if (value >= hh) {
	    blue = table[length*3];
	    green = table[length*3+1];
	    red = table[length*3+2];
	  }
	  else {
	    int l = (int)(((value - ll)/diff * length) + .5);
	    blue = table[l*3];
	    green = table[l*3+1];
	    red = table[l*3+2];
	  }
	}
	else {
	  blue = nanColor->blue;
	  green = nanColor->green;
	  red = nanColor->red;
	}
      }

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }
    str << ends;
    psFix(buf,size);
    Tcl_AppendResult(interp, buf, NULL);
  }

  ostrstream str(buf,size);
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(buf,size);
  Tcl_AppendResult(interp, buf, NULL);
  delete buf;
}
#endif

#if __GNUC__ >= 3
void FrameTrueColor::psLevel2Mosaic(PSColorSpace mode, 
				    int width, int height, float scale)
{
  if (!fits)
    return;

  RLEAscii85 filter;
  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();
  FitsImage* ptr = fits;
  Matrix mm = psMatrix(scale, width, height);
  while (ptr) {
    ptr->updatePS(mm);
    ptr = ptr->next();
  }

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = bgColor->red;
      unsigned char green = bgColor->green;
      unsigned char blue = bgColor->blue;

      ptr = fits;
      while (ptr) {
	double* m = ptr->getmPSToData();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = ptr->getDataParams(frScale.scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  double ll = ptr->getLowDouble();
	  double hh = ptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isNaNd(value)) {
	    if (value <= ll) {
	      blue = table[0];
	      green = table[1];
	      red = table[2];
	    }
	    else if (value >= hh) {
	      blue = table[length*3];
	      green = table[length*3+1];
	      red = table[length*3+2];
	    }
	    else {
	      int l = (int)(((value - ll)/diff * length) + .5);
	      blue = table[l*3];
	      green = table[l*3+1];
	      red = table[l*3+2];
	    }
	  }
	  else {
	    blue = nanColor->blue;
	    green = nanColor->green;
	    red = nanColor->red;
	  }

	  break;
	}
	else
	  ptr = ptr->next();
      }

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }
    str << ends;
    psFix(str);
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(str);
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void FrameTrueColor::psLevel2Mosaic(PSColorSpace mode, 
				    int width, int height, float scale)
{
  if (!fits)
    return;

  RLEAscii85 filter;

  // size = 2 char per byte * 4 bytes per pixel + estimate of endls + 
  // endl + ends

  int size = (int)(width*4*2*(1 + 1./80.) +.5) + 2;
  char* buf = new char[size];

  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();

  FitsImage* ptr = fits;
  Matrix mm = psMatrix(scale, width, height);
  while (ptr) {
    ptr->updatePS(mm);
    ptr = ptr->next();
  }

  for (int j=0; j<height; j++) {
    ostrstream str(buf,size);

    for (int i=0; i<width; i++) {
      unsigned char red = bgColor->red;
      unsigned char green = bgColor->green;
      unsigned char blue = bgColor->blue;

      ptr = fits;
      while (ptr) {
	double* m = ptr->getmPSToData();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = ptr->getDataParams(frScale.scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  double ll = ptr->getLowDouble();
	  double hh = ptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isNaNd(value)) {
	    if (value <= ll) {
	      blue = table[0];
	      green = table[1];
	      red = table[2];
	    }
	    else if (value >= hh) {
	      blue = table[length*3];
	      green = table[length*3+1];
	      red = table[length*3+2];
	    }
	    else {
	      int l = (int)(((value - ll)/diff * length) + .5);
	      blue = table[l*3];
	      green = table[l*3+1];
	      red = table[l*3+2];
	    }
	  }
	  else {
	    blue = nanColor->blue;
	    green = nanColor->green;
	    red = nanColor->red;
	  }

	  break;
	}
	else
	  ptr = ptr->next();
      }

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }
    str << ends;
    psFix(buf,size);
    Tcl_AppendResult(interp, buf, NULL);
  }

  ostrstream str(buf,size);
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(buf,size);
  Tcl_AppendResult(interp, buf, NULL);
  delete buf;
}
#endif

#if __GNUC__ >= 3
void FrameTrueColor::psLevel2MosaicFast(PSColorSpace mode, 
					int width, int height, float scale)
{
  if (!fits)
    return;

  RLEAscii85 filter;
  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();
  FitsImage* ptr = fits;
  Matrix mm = psMatrix(scale, width, height);
  while (ptr) {
    ptr->updatePS(mm);
    ptr = ptr->next();
  }

  ptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = bgColor->red;
      unsigned char green = bgColor->green;
      unsigned char blue = bgColor->blue;

      double x,y;

      if (ptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  if (!isNaNd(value)) {
	    if (value <= ll) {
	      blue = table[0];
	      green = table[1];
	      red = table[2];
	    }
	    else if (value >= hh) {
	      blue = table[length*3];
	      green = table[length*3+1];
	      red = table[length*3+2];
	    }
	    else {
	      int l = (int)(((value - ll)/diff * length) + .5);
	      blue = table[l*3];
	      green = table[l*3+1];
	      red = table[l*3+2];
	    }
	  }
	  else {
	    blue = nanColor->blue;
	    green = nanColor->green;
	    red = nanColor->red;
	  }

	}
	else 
	  ptr = NULL;
      }

      if (!ptr) {
	ptr = fits;
	while (ptr) {
	  m = ptr->getmPSToData();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = ptr->getDataParams(frScale.scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	    ll = ptr->getLowDouble();
	    hh = ptr->getHighDouble();
	    diff = hh - ll;

	    if (!isNaNd(value)) {
	      if (value <= ll) {
		blue = table[0];
		green = table[1];
		red = table[2];
	      }
	      else if (value >= hh) {
		blue = table[length*3];
		green = table[length*3+1];
		red = table[length*3+2];
	      }
	      else {
		int l = (int)(((value - ll)/diff * length) + .5);
		blue = table[l*3];
		green = table[l*3+1];
		red = table[l*3+2];
	      }
	    }
	    else {
	      blue = nanColor->blue;
	      green = nanColor->green;
	      red = nanColor->red;
	    }

	    break;
	  }
	  else
	    ptr = ptr->next();
	}
      }

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }
    str << ends;
    psFix(str);
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(str);
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void FrameTrueColor::psLevel2MosaicFast(PSColorSpace mode, 
					int width, int height, float scale)
{
  if (!fits)
    return;

  RLEAscii85 filter;

  // size = 2 char per byte * 4 bytes per pixel + estimate of endls + 
  // endl + ends

  int size = (int)(width*4*2*(1 + 1./80.) +.5) + 2;
  char* buf = new char[size];

  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();

  FitsImage* ptr = fits;
  Matrix mm = psMatrix(scale, width, height);
  while (ptr) {
    ptr->updatePS(mm);
    ptr = ptr->next();
  }

  ptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  for (int j=0; j<height; j++) {
    ostrstream str(buf,size);

    for (int i=0; i<width; i++) {
      unsigned char red = bgColor->red;
      unsigned char green = bgColor->green;
      unsigned char blue = bgColor->blue;

      double x,y;

      if (ptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  if (!isNaNd(value)) {
	    if (value <= ll) {
	      blue = table[0];
	      green = table[1];
	      red = table[2];
	    }
	    else if (value >= hh) {
	      blue = table[length*3];
	      green = table[length*3+1];
	      red = table[length*3+2];
	    }
	    else {
	      int l = (int)(((value - ll)/diff * length) + .5);
	      blue = table[l*3];
	      green = table[l*3+1];
	      red = table[l*3+2];
	    }
	  }
	  else {
	    blue = nanColor->blue;
	    green = nanColor->green;
	    red = nanColor->red;
	  }

	}
	else 
	  ptr = NULL;
      }

      if (!ptr) {
	ptr = fits;
	while (ptr) {
	  m = ptr->getmPSToData();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = ptr->getDataParams(frScale.scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	    ll = ptr->getLowDouble();
	    hh = ptr->getHighDouble();
	    diff = hh - ll;

	    if (!isNaNd(value)) {
	      if (value <= ll) {
		blue = table[0];
		green = table[1];
		red = table[2];
	      }
	      else if (value >= hh) {
		blue = table[length*3];
		green = table[length*3+1];
		red = table[length*3+2];
	      }
	      else {
		int l = (int)(((value - ll)/diff * length) + .5);
		blue = table[l*3];
		green = table[l*3+1];
		red = table[l*3+2];
	      }
	    }
	    else {
	      blue = nanColor->blue;
	      green = nanColor->green;
	      red = nanColor->red;
	    }

	    break;
	  }
	  else
	    ptr = ptr->next();
	}
      }

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }
    str << ends;
    psFix(buf,size);
    Tcl_AppendResult(interp, buf, NULL);
  }

  ostrstream str(buf,size);
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(buf,size);
  Tcl_AppendResult(interp, buf, NULL);
  delete buf;
}
#endif
