// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __fitscolumn_h__
#define __fitscolumn_h__

#include "vector.h"

class FitsHead;

class FitsColumn {
protected:
  int index_;        // col number 1 to n
  int width_;        // overall width of field in chars
  int offset_;       // offset in chars from start of row
  char type_;        // type

  char* tform_;
  char* ttype_;
  char* tunit_;
  float tscal_;
  float tzero_;

  int tnull_;
  int hastnull_;

  float tlmin_;
  float tlmax_;
  int hastlmin_;
  int hastlmax_;

  double min_;
  double max_;
  int validmm_;

  char buf_[128];

  char keybuf[9];
  char* keycat(const char*, int);

public:
  FitsColumn(FitsHead*, int, int);
  virtual ~FitsColumn();

  int width() {return width_;}
  int offset() {return offset_;}
  int index() {return index_;}

  const char* tform() {return tform_;}
  const char* ttype() {return ttype_;}
  const char* tunit() {return tunit_;}
  float tscal() {return tscal_;}
  float tzero() {return tzero_;}
  int tnull() {return tnull_;}
  float tlmin() {return tlmin_;}
  float tlmax() {return tlmax_;}

  int hasscaling() {return tscal_ != 1 || tzero_ != 0;}
  int hastnull() {return hastnull_;}

  virtual double value(const char*, int i =0) =0;
  virtual char* str(const char*, int i =0) =0;
  virtual Vector dimension() =0;

  void setMin(double m) {min_=m;}
  void setMax(double m) {max_=m;}

  double getMin() {return min_;}
  double getMax() {return max_;}

  int hasMinMax() {return min_ != -DBL_MAX ? 1 : 0;}
};

class FitsAsciiColumn : public FitsColumn {
private:
  int prec_;

public:
  FitsAsciiColumn(FitsHead*, int, int);

  double value(const char* ptr, int i =0);
  char* str(const char*, int i =0);
  Vector dimension();
};

class FitsBinColumn : public FitsColumn {
protected:
  char* tdisp_;
  int repeat_;           // repeat count
  
public:
  FitsBinColumn(FitsHead*, int, int);
  ~FitsBinColumn();

  virtual double value(const char*, int i =0) =0;
  virtual char* str(const char*, int i =0) =0;
  virtual Vector dimension() =0;
};

class FitsBinColumnA : public FitsBinColumn {
private:
  int Awidth_;           // String width
  int Aterm_;            // String terminator

public:
  FitsBinColumnA(FitsHead*, int, int);

  double value(const char*, int i =0);
  char* str(const char*, int i =0);
  Vector dimension();
};

class FitsBinColumnB : public FitsBinColumn {
protected:
  int byteswap;

public:
  FitsBinColumnB(FitsHead*, int, int);

  virtual double value(const char*, int i =0) =0;
  virtual char* str(const char*, int i =0) =0;
  virtual Vector dimension() =0;
  virtual int size() =0;
};

class FitsBinColumnBit : public FitsBinColumnB {
private:
  int size() {return repeat_;}

public:
  FitsBinColumnBit(FitsHead*, int, int);

  double value(const char*, int i =0);
  char* str(const char*, int i =0);
  Vector dimension();
};

template<class T>
class FitsBinColumnT : public FitsBinColumnB {
private:
  int size() {return sizeof(T);}
  T swap(T*);

public:
  FitsBinColumnT(FitsHead*, int, int);

  double value(const char*, int =0);
  char* str(const char*, int =0);
  Vector dimension();
};

#endif
