/**
  * \file squash.cc
  * \author Donovan Rebbechi, Ben Bly, Giorgio Grasso
  * Copyright Ben Bly
  * This file is released under the artistic license. See the file
  * COPYING for details.
  */

#include <rumba/arghandler.h>
#include <iostream>
#include <string>

#include "../matrixutils/mask.h"

using RUMBA::Argument;

void usage()
{
	std::cerr << "usage: squash -i infile -m mask -o outfile" << std::endl;
}

Argument myArgs [] = {
	Argument ( "mask", RUMBA::ALPHA, 'm', "", true ),
	Argument()
};


int main(int argc,char** argv)
{
	RUMBA::ArgHandler::setRequiredDefaultArg("infile");
	RUMBA::ArgHandler::setRequiredDefaultArg("outfile");
	RUMBA::intPoint dims;
	std::string infile,outfile,maskfile;

	try {
		RUMBA::ArgHandler argh ( argc , argv, myArgs );
		if ( argh.arg("help") )
		{
		usage();
		exit(0);
		}
		argh.arg ( "infile" , infile );
		argh.arg ( "outfile" , outfile);
		argh.arg ( "mask" , maskfile);



		RUMBA::ManifoldFile* fin = 
			RUMBA::ManifoldFile::construct(infile.c_str(),std::ios::in);
		if (!fin)
		{
			std::cerr << "Couldn't open input file " << infile << std::endl;
			exit(2);
		}
	
		RUMBA::ManifoldFile* fout;  
		RUMBA::Manifold<char> maskman(maskfile.c_str());
	
		dims = squashSize(maskman,fin);
		fout = RUMBA::ManifoldFile::construct(outfile.c_str(),
				fin->headerData()["normalized_datatype"].asString(),dims);
	
		if (!fout)
		{
			std::cerr << "Couldn't open output file " << outfile << std::endl;
			exit(2);
		}

		if (maskman.timepoints() > 1)
		{
			fin->setCacheStrategy(RUMBA::CACHE_VOL);
			fout->setCacheStrategy(RUMBA::CACHE_VOL);
		}
		else
		{
			fin->setCacheStrategy(RUMBA::CACHE_TIME);
			fout->setCacheStrategy(RUMBA::CACHE_TIME);
		}
	
		squash(maskman,fin,fout);
		delete fout;
		delete fin;
		return 0;
	
	}
	catch ( RUMBA::InvalidArgumentException& s)
	{
		std::cerr << "Invalid argument: " << s.error() << std::endl;
		usage();
		exit(1);
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		std::cerr << "Duplicate argument: " << s.error() << std::endl;
		usage();
		exit(1);
	}
	catch (RUMBA::MissingArgumentException& s)
	{
		std::cerr << "Missing argument: " << s.error() << std::endl;
		usage();
		exit(1);
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		std::cerr << "Error: " << s.error() << std::endl; 
		usage();
		exit(1);
	}
	catch (RUMBA::Exception& s)
	{
		std::cerr << "Exception:" << s.error() << std::endl;
		usage();
		exit(1);
	}


}

