;;;
;;;  Parse standard (POSIX) Regex notation
;;;

(define (parse-posix-regex re)
  (parse-re (string->list re)))

(define (push-item item r)
  (if (char? item)
      (cond
       ((null? r) (cons item '()))
       ((string? (car r)) (cons (string-append (car r) (string item))
                                (cdr r)))
       ((char? (car r)) (cons (string (car r) item) (cdr r)))
       (else (cons item r)))
      (cons item r)))

(define (parse-subexpr lst)
  (bind ((re rest (parse-re lst)))
    (if (and (pair? rest)
             (eq? (car rest) #\)))
        (values (list 'save re) (cdr rest))
        (error "unclosed subexpr"))))

(define (build-branch-section b)
  (if (null? b)
      '(seq)
      (if (null? (cdr b))
          (car b)
          (cons 'or (reverse b)))))

(define (parse-re lst)
  (let loop ((l lst)
             (branches '()))
    (if (null? l)
        (values (build-branch-section branches) l)
        (bind ((branch rest (parse-branch l)))
          (if (and (pair? rest)
                   (eq? (car rest) #\|))
              (loop (cdr rest) (cons branch branches))
              (values (build-branch-section (cons branch branches))
                      rest))))))

(define (parse-branch lst)
  (let loop ((l lst)
             (r '()))
    (if (null? l)
        (values (cons 'seq (reverse r)) '())
        (bind ((item rest (parse-piece l)))
          (if (memq item '($OR $CLOSE))
              (values (cons 'seq (reverse r)) rest)
              (loop rest (push-item item r)))))))

(define (parse-piece lst)
  (bind ((atom rest (parse-atom lst)))
    (if (pair? rest)
        (case (car rest)
          ((#\*) (values (list '* atom) (cdr rest)))
          ((#\+) (values (list '+ atom) (cdr rest)))
          ((#\?) (values (list '? atom) (cdr rest)))
          ((#\{) (if (and (pair? (cdr rest))
                          (char-numeric? (cadr rest)))
                     (bind ((bound rest (parse-bound (cdr rest))))
                       (values (list 'bound bound atom) rest))
                     (values atom rest)))
          (else (values atom rest)))
        (values atom rest))))

(define (parse-bound lst)
  (bind ((n rest (parse-uint lst)))
    (if (not (pair? rest)) 
        (error "end of string inside bound: ~s" lst))
    (if (eq? (car rest) #\})
        (values (list n) (cdr rest))
        (if (eq? (car rest) #\,)
            (if (and (pair? (cdr rest))
                     (char-numeric? (cadr rest)))
                (bind ((m rest (parse-uint (cdr rest))))
                  (if (not (pair? rest)) 
                      (error "end of string inside bound: ~s" lst))
                  (if (eq? (car rest) #\})
                      (values (list n m) (cdr rest))
                      (error "unclosed bound: ~s" lst)))
                (if (and (pair? (cdr rest))
                         (eq? (cadr rest) #\}))
                    (values (list n '*) (cddr rest))
                    (error "bad bound syntax: ~s" lst)))
            (error "bad bound syntax: ~s" lst)))))

(define (parse-uint lst)
  (let loop ((n 0)
             (l lst))
    (if (null? l)
        (values n l)
        (if (char-numeric? (car l))
            (loop (+ (* n 10) (string->number (string (car l))))
                  (cdr l))
            (values n l)))))

(define (parse-atom lst)
  (case (car lst)
    ((#\^) (values 'bos (cdr lst)))
    ((#\$) (values 'eos (cdr lst)))
    ((#\\) (values (cadr lst) (cddr lst)))
    ((#\.) (values 'any (cdr lst)))
    ((#\() (parse-subexpr (cdr lst)))
    ((#\)) (values '$CLOSE lst))
    ((#\|) (values '$OR lst))
    ((#\[) (parse-bracket (cdr lst)))
    (else
     (values (car lst) (cdr lst)))))

(define (parse-bracket lst)
  (if (eq? (car lst) #\^)
      (bind ((item rest (parse-bracket-after-c (cdr lst))))
        (values (list 'not item) rest))
      (parse-bracket-after-c lst)))

(define (push-with-ranging ch r ranging?)
  (if ranging?
      (cons (list 'range (car r) ch) (cdr r))
      (cons ch r)))

(define (parse-bracket-after-c lst)
  (let loop ((l lst)
             (r '())
             (ranging? #f))
    (case (car l)
      ((#\]) 
       (if (null? r)
           (loop (cdr l) '(#\]) #f)
           (values (cons 'or (reverse r)) (cdr l))))
      ((#\-)
       (if (or (null? r)
               (eq? (cadr l) #\]))
           (loop (cdr l) (cons #\- r) #f)
           (begin
             (if (not (char? (car r)))
                 (error "cannot range from a non-literal: ~s" lst))
             (loop (cdr l) r #t))))
      ((#\\)
       (let ((ch (cadr l))
             (rest (cddr l)))
         (case ch
           ((#\d)
            (if ranging? (error "cannot use range as range endpoint: ~s" lst))
            (loop rest (cons 'digit r) #f))
           (else
            (loop rest (push-with-ranging ch r ranging?) #f)))))
      ((#\[)
       (case (cadr l)
         ((#\:) 
          (if ranging? (error "cannot use cclass as range endpoint: ~s" lst))
          (bind ((ccname rest (parse-cclass-name (cddr l))))
            (if (and (eq? (car rest) #\:)
                     (eq? (cadr rest) #\]))
                (loop (cddr rest) (cons (string->symbol ccname) r) #f)
                (error "cclass syntax error: ~s" lst))))
         ((#\.)
          (error "collating element not supported: ~s" lst))
         ((#\=)
          (error "equivalence classes not supported: ~s" lst))
         (else
          (loop (cdr l) (push-with-ranging #\[ r ranging?) #f))))
      (else
       (loop (cdr l) (push-with-ranging (car l) r ranging?) #f)))))

(define (parse-cclass-name lst)
  (let loop ((s '())
             (l lst))
    (if (null? l)
        (values (list->string (reverse s)) l)
        (if (char-alphabetic? (car l))
            (loop (cons (car l) s) (cdr l))
            (values (list->string (reverse s)) l)))))

;;

(define (posix-reg-expr->proc re)
  (reg-expr->proc (list 'posix re)))
