#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <setjmp.h>
#include <errno.h>
#include <time.h>

#include "lsspriv.h"
#include "lssv3.h"

typedef UINT_32 UINT_W;


static char *line_base = NULL;
static char *line_limit = NULL;
static char *line_ptr;

static jmp_buf line_full;

static void lputs( char *str )
{
  while (*str)
    {
      if (line_ptr < line_limit)
	{
	  *line_ptr++ = *str++;
	}
      else
	{
	  /* back-fill the "..." */
	  line_ptr[-3] = '.';
	  line_ptr[-2] = '.';
	  line_ptr[-1] = '.';
	  /* we reserved space for this, plus the NUL, in linit() */
	  *line_ptr++ = '\n';
	  longjmp( line_full, 1 );
	}
    }
}

static void lputc( char ch )
{
  char temp[2];
  temp[0] = ch;
  temp[1] = 0;
  lputs( temp );
}

static void linit( int width )
{
  if ((line_limit - line_base) < (width + 2))
    {
      if (line_base)
	free( line_base );
      line_base = malloc( width + 2 );
      line_limit = line_base + width;
    }
  line_ptr = line_base;
}

static void lprintf( char *fmt, ... )
{
  va_list va;
  char temp[100];
  int i, n;

  va_start( va, fmt );
  n = vsprintf( temp, fmt, va );
  va_end( va );

  lputs( temp );
}

static void lprint_hex_data( UINT_8 *addr, unsigned len )
{
  int i;

  lputs( "<" );
  for (i=0; i<len; i++)
    {
      if (i > 0)
	lputs( " " );
      lprintf( "%02x", addr[i] );
    }
  lputs( ">" );
}

static void lprint_data( UINT_8 *addr, unsigned len )
{
  unsigned i;

  lprintf( " %u <", len );

  for (i=0; i<len; i++)
    {
      switch (addr[i])
	{
	case '\0':
	  lputs( "\\0" );
	  break;

	case '\n':
	  lputs( "\\n" );
	  break;

	case '\t':
	  lputs( "\\t" );
	  break;

	case '\r':
	  lputs( "\\r" );
	  break;

	default:
	  if (isprint(addr[i]))
	    {
	      lputc( addr[i] );
	    }
	  else
	    {
	      lprintf( "\\%03o", addr[i] );
	    }
	}
    }
  lprintf( ">" );
}

static void lprint_volh( void *data )
{
  struct LSSVolumeHeader *h = data;
  lprintf( " (volume ~d of ~d)", h->vol_number, h->num_vols );
}

static void lprint_commit( struct LSSRecordHeader *h, void *data )
{
  struct LSSCommitRecord c;
  struct tm *t;
  time_t tv;
  char buf[30];

  memcpy( ((char *)&c), h, sizeof( struct LSSRecordHeader ) );
  memcpy( ((char *)&c) + sizeof( struct LSSRecordHeader ), data, 
          sizeof( struct LSSCommitRecord ) 
          - sizeof( struct LSSRecordHeader ) );

  tv = c.commit_time_ms / 1000;
  t = localtime( &tv );
  strftime( buf, sizeof(buf), "%Y-%m-%d %H:%M:%S", t );
  lprintf( " gen %u <%s>", c.generation, buf );
}

static void lprint_words( void *data, unsigned len )
{
  UINT_W *array = data;
  int left = 49;
  unsigned i, count;

  count = len / sizeof( UINT_W );
  
  lprintf( " %u <", len );

  for (i=0; i<count; i++)
    {
      if (i)
	lputs( " " );

      lprintf( "%#lx", array[i] );
    }
  lputs( ">" );
}

static void ldump_line( int fd, struct LSSRecordHeader *h, long at )
{
  char type[5];
  UINT_8 *data;
  int space, n;

  type[0] = (h->magic >> 24) & 0xFF;
  type[1] = (h->magic >> 16) & 0xFF;
  type[2] = (h->magic >>  8) & 0xFF;
  type[3] = (h->magic >>  0) & 0xFF;
  type[4] = 0;

  space = GET_SPACE_BYTES( h->space_word );

  lprintf( "%#010lx: %s %6d ", at, type, space );

  /*  we don't want to count the header in the space,
   *  since we aren't going to read() it, and there is 
   *  no need to print it
   */
  space -= sizeof( struct LSSRecordHeader );

  if (space == 0)
    {
      lprintf( "\n" );
      return;
    }

  data = malloc( space );
  assert( data );

  n = read( fd, data, space );
  if (n != space)
    {
      if (n < 0)
	{
	  lprintf( "--!-- %s (%d)\n", strerror( errno ), errno );
	}
      else
	{
	  lprintf( "--!-- short read (only %d)\n", n );
	}
      free( data );
      return;
    }

  if (GET_WHICH_ZIP( h->space_word ))
    {
      if (h->magic == DATAREC_MAGIC)
	lprintf( " %lu", h->recnum );
      
      lprintf( " %lu (z%d) ", 
	      h->length, GET_WHICH_ZIP( h->space_word ) );
      lprint_hex_data( data, space );
    }
  else if (h->length > 0)
    {
      switch (h->magic)
	{
	case MINDEX_MAGIC:
	case INDEX_MAGIC:
	  lprint_words( data, h->length );
	  break;
	case DATAREC_MAGIC:
	  lprintf( " %lu", h->recnum );
	default:
	  lprint_data( data, h->length );
	  break;
	case LSS_MAGIC:
	  break;
	}
    } else {
      switch (h->magic) {
      case LSS_MAGIC:
        lprint_volh( data );
        break;
      case COMMIT_MAGIC:
        lprint_commit( h, data );
        break;
      }
    }
  free( data );
  lputc( '\n' );
}

static void dump_line( int fd, struct LSSRecordHeader *h, long at )
{
  linit( 79 );
  if (setjmp( line_full ) == 0)
    {
      ldump_line( fd, h, at );
    }
  fwrite( line_base, 1, line_ptr - line_base, stdout );
}

void print_hex_data( UINT_8 *addr, unsigned len )
{
  linit( 79 );
  if (setjmp( line_full ) == 0)
    {
      lprint_hex_data( addr, len );
    }
  fwrite( line_base, 1, line_ptr - line_base, stdout );
}

void print_record( unsigned long recno, 
		   unsigned char *addr, 
		   unsigned len )
{
  linit( 79 );
  if (setjmp( line_full ) == 0)
    {
      lprintf( "%lu: ", recno );
      lprint_data( addr, len );
      lputs( "\n" );
    }
  fwrite( line_base, 1, line_ptr - line_base, stdout );
}


void low_level_dump( const char *file )
{
  struct LSSRecordHeader h;
  int n;
  int fd;

  fd = open( file, O_RDONLY );
  if (fd < 0)
    {
      perror( file );
      exit(1);
    }

  while (1)
    {
      long at = lseek( fd, 0, SEEK_CUR );

      n = read( fd, &h, sizeof h );
      if (n != sizeof h)
	{
	  if (n < 0)
	    {
	      perror( file );
	    }
	  else if (n == 0)
	    {
	      printf( "%#010lx:\n", at );
	      close( fd );
	      return;
	    }
	  else
	    {
	      fprintf( stderr, "%s: short read (%d)\n", file, n );
	    }
	  exit(1);
	}
      dump_line( fd, &h, at );
      lseek( fd, at + GET_SPACE_BYTES(h.space_word), SEEK_SET );
    }
}
