/*				       	-*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: upstream.c,v 1.17 2000/08/13 10:40:32 mbp Exp $
 * 
 * Copyright (C) 1999, 2000 by Martin Pool <mbp@linuxcare.com>
 * Copyright (C) 1999 by Andrew Tridgell <tridge@linuxcare.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "sysheaders.h"

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "rproxy.h"

#include "cache.h"
#include "util.h"
#include "urllib.h"
#include "header.h"
#include "socklib.h"
#include "msgpage.h"
#include "proto.h"
#include "trace.h"
#include "myname.h"
#include "error.h"

/* 
 * TODO: Resolve upstream address just once, then remember it.  Can we not rely on
 * anybody else to do this?
 *
 * TODO: shutdown(2) socket after sending request?  In that case
 * perhaps we should set SO_LINGER so that we know if something goes wrong. 
 */


/* Returns true if we should send all requests to a single upstream host,
   rather than extracting the hostname from the request.

   This is true for two cases that look different to the user, but which work 
   out the same way in here: being an HTTP accelerator, and using an upstream 
   proxy. */
static int
use_single_upstream(void)
{
    return config.upstream_host != NULL;
}


/*
 * true if we should send just the path to the upstream, stripping it out of
 * the URI.  At the moment this is true only if there's no upstream and we're 
 * going direct.
 * 
 * If we're an accelerator, then the server we're accelerating will probably
 * want only a path, but it's the responsibility of the downstream to do
 * that.
 */
static bool_t
upstream_wants_path(request_t * UNUSED(req))
{
    return !use_single_upstream();
}


/* Work out where to send a request, and store in req->dest_hostname /
   dest_port.

   If an upstream proxy is configured, then we always use it. Otherwise, we
   extract the hostname and port from the request and use it.  If there is
   none, or it's not an HTTP request, then we fail: we rely on having an
   upstream proxy for them. */
static int
find_upstream_addr(request_t * req)
{
    if (use_single_upstream()) {
	req->dest_hostname = config.upstream_host;
	req->dest_port = config.upstream_port;
    } else {
	req->dest_hostname = req->url_hostname;
	req->dest_port = req->url_port;
    }

    trace(LOGAREA_PROTO,
	  __FUNCTION__ ": trying to open upstream socket to %s:%d",
	  req->dest_hostname, req->dest_port);

    return 0;
}


static void
upstream_send_request_line(request_t * req)
{
    if (upstream_wants_path(req)) {
	assert(req->method_name);
	assert(req->path);
	assert(req->http_version);
	fprintf(req->f_upstream,
		"%s %s %s\r\n", req->method_name, req->path,
                req->http_version);

	trace(LOGAREA_HTTP, "req> %s %s %s", req->method_name,
	      req->path, req->http_version);
    } else {
	/* too easy, send verbatim */
	fprintf(req->f_upstream, "%s\r\n", req->req_line);
	trace(LOGAREA_HTTP, "req> %s", req->req_line);
    }
}



/* If the request contains a message body, then send it upstream.

   rfc1945 says ``The presence of an entity body in a request is signaled by
   the inclusion of a Content-Length header field in the request message
   headers. HTTP/1.0 requests containing an entity body must include a valid
   Content-Length header field.''  So that's pretty simple. */
static void
send_body_upstream_maybe(request_t * req)
{
    int             content_length;

    content_length = header_ival(req->headers, "Content-Length");
    if (content_length != -1) {
	trace(LOGAREA_HTTP,
	      "request has an entity body, content-length %d",
	      content_length);
	stream_body(req->f_from_client, req->f_upstream, NULL, content_length);
    } else {
	trace(LOGAREA_HTTP, "request has no entity body");
    }
}



/*
 * Send a requst to the upstream proxy then return a FILE pointer for
 * the upstream socket.
 */
FILE           *
send_upstream(request_t * req)
{
    int fd;
    int ret;
    char const *nice_host;

    if (!(nice_host = req->dest_hostname))
        nice_host = "localhost";

    rp_process_name("destination host is %s:%d", nice_host, req->dest_port);
    trace(LOGAREA_PROTO, "connecting upstream to %s:%d",
	  nice_host, req->dest_port);

    if (find_upstream_addr(req)) {
	rp_request_failed(req, HTTP_BAD_GATEWAY,
                          "couldn't find upstream address: %s",
                          strerror(errno));
    }

    ret = rp_open_socket_out(req, req->dest_hostname, req->dest_port, &fd);
    if (ret != DONE) {
	rp_request_failed(req, HTTP_BAD_GATEWAY,
                          "couldn't connect to upstream server %s:%d: %s",
                          req->dest_hostname, req->dest_port,
                          strerror(errno));
    }

    req->f_upstream = xfdopen(fd, "r+");
    setbuffer(req->f_upstream, NULL, BUFFER_SIZE);

    /* Send the request. */
    rp_process_name("requesting %s", req->req_line);
    upstream_send_request_line(req);

    if (header_send(req->headers, req->f_upstream) < 0) {
	rp_request_failed(req, HTTP_BAD_GATEWAY,
		       "error writing request headers to upstream server "
		       "%s:%d: %s\n",
		       req->dest_hostname, req->dest_port, strerror(errno));
    }

    /* header_send also adds a blank line at the end, so we can go
     * straight into the body if we want to. */
    send_body_upstream_maybe(req);
    fflush(req->f_upstream);

    return req->f_upstream;
}
