/*				       	-*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: encreq.c,v 1.19 2000/08/16 10:08:58 mbp Exp $
 * 
 * Copyright (C) 1999, 2000 by Martin Pool <mbp@linuxcare.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "sysheaders.h"
#include "rproxy.h"

#include "util.h"
#include "msgpage.h"
#include "header.h"
#include "rsglue.h"
#include "trace.h"
#include "myname.h"

/* Called after seeing the response headers from the upstream; we decide
   whether we want to rsync-encode the message on the way down. */
int
will_encode_hsync(request_t const *req)
{
    char const *encoding;
    
    if (req->resp_rsync_encoded)
	return 0;
    if (req->status != HTTP_OK) {
	trace(LOGAREA_PROTO,
	      "can't encode because this is not a %d OK response",
	      HTTP_OK);
	return 0;
    }
    if (!req->client_does_hsync) {
	trace(LOGAREA_PROTO,
	      "can't encode because the client doesn't do hsync");
	return 0;
    }
    encoding = header_content(req->headers, "Content-Encoding");
    if (encoding && *encoding) {
        trace(LOGAREA_PROTO,
              "can't encode because this response has a Content-Encoding"
              " already and we can't chain them yet");
        return 0;
    }

    /* TODO: Also, if this was a HEAD request then there is no body
     * and we shouldn't try to encode it.  Are there other methods
     * where this also applies? */
    return 1;
}



/* Actually encode the body, per the diagram above.

   For the time being we have a *really* simple implementation of this: we
   suck in the whole file, encode it in one go, generate the signature, and
   then write out the whole thing as one big chunk. When this works OK, we'll 
   try pipelining things but that will make it a bit harder. */
static int
encode_body(request_t * req)
{
    hs_filebuf_t   *diff_fb;
    hs_ptrbuf_t    *sig_mb;
    hs_encode_job_t *job;
    hs_sumset_t    *sums;
    hs_result_t     result;

    diff_fb = hs_filebuf_from_fd(fileno(req->f_to_client));

    /* make an empty signature, until we're ready to do it for real. */
    if (req->req_signature) {
	sig_mb = hs_ptrbuf_on_buffer((char *) req->req_signature,
				     req->req_signature_len);
	sums = hs_read_sumset(hs_ptrbuf_read, sig_mb);
    } else {
	sums = NULL;
    }

    /* TODO: Choose the right block length.  How?  We could look for a
       Content-Length header in the response, I suppose. */

    job = hs_encode_begin(fileno(req->f_upstream),
			  hs_filebuf_write, diff_fb,
			  sums, &req->hs_stats, 1024);

    do {
	result = hs_encode_iter(job);
    } while (result == HS_AGAIN);

    fflush(req->f_to_client);
    if (sums)
	hs_free_sumset(sums);

    return !(result == HS_DONE);
}



/* Encode a request for a downstream client who'd like it in chunky-rsync
   form.  We do the rsync-encoding thang; but also run the output through the 
   rsync-signature algorithm and spit out the signature at the end.

   This file sets up all of the encoding stuff; we don't use rs_sig_encode
   anymore.  */
int
encode_hsync_request(request_t * req)
{
    req->action_str = "hsync-encode";

    rp_process_name("%s %s", req->action_str, req->url);
    
    req->status = HTTPX_DELTA;
    free(req->status_msg);
    req->status_msg = xstrdup("Delta");

    /* rfc2616s4.4: The Content-Length header MUST NOT be sent if the
     * entity and encoded lengths are different. */
    header_remove(req->headers, "Content-Length");
    header_add_list(req->headers, "Content-Encoding", HSYNC_ENCODING_TOKEN);

    req->state = sending_response_headers;
    trace(LOGAREA_HTTP, "sending hsync response back");
    rp_send_response_line(req);
    if (header_send(req->headers, req->f_to_client) < 0)
	return -1;

    if (req->req_signature)
	trace(LOGAREA_PROTO, "got request signature `%#08x', len=%d",
	      *((int const *) req->req_signature), req->req_signature_len);

    req->state = sending_body;

    encode_body(req);

    return 0;
}
