
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#ifndef __RARRAY_H__
#define __RARRAY_H__

typedef struct _RArrayChunkEntry RArrayChunkEntry;

struct _RArrayChunkEntry {
    RLIST_HEADER;
    char *chunk;
};

typedef struct _RArray RArray;

struct _RArray {
    RList chunks;
    RArrayChunkEntry *curentry;
    int curindex;
    unsigned int elements_per_chunk;
    unsigned int element_size;
};


/* Create a new array with elements of size element_size.  Specify the
 * number of elements to allocate at a time (per block), via the
 * elements_per_chunk argument */
RArray *
rarray_new (unsigned int element_size, unsigned int elements_per_chunk);

/* Return a pointer to the last entry in the array so you can use
 * the memory there for whatever you like */
void *
rarray_append (RArray *array);

/* rarray_push is a synonym for rarray_append.
 * rarray_push and rarray_pop can be utilized to 
 * make queues */
#define rarray_push(array) rarray_append(array)

/* Return a pointer to the last entry in the array
 * and then pull it off the array (it doesn't really
 * go anywhere, just the next push will overwrite it,
 * and the next pop will keep going back */
void *
rarray_pop (RArray *array);

#define rarray_peek(array) rarray_last(array)

/* Return the last entry in the array. */
void *
rarray_last (RArray *array);

/* Return a pointer to the nth element in the array */
void *
rarray_nth (RArray *array, unsigned int nth);

unsigned int
rarray_len (RArray *array);

void
rarray_set_len (RArray *array, unsigned int len);


/* Free the array structure, and all memory associated
 * with all elements. */
void
rarray_free (RArray *array);


#define RARRAY_FOREACH(array, entry) \
do { \
    RArrayChunkEntry *tmp_rarray_next_entry; \
        for (tmp_rarray_next_entry = (void *) rlist_first (&(array)->chunks); \
            tmp_rarray_next_entry; tmp_rarray_next_entry = rlist_next (tmp_rarray_next_entry)) \
            for (entry = (void *) tmp_rarray_next_entry->chunk; \
                (entry != (void *) ((array)->curentry->chunk + (((array)->curindex + 1) * (array)->element_size))) && \
                (entry != (void *) (tmp_rarray_next_entry->chunk + (array->elements_per_chunk * array->element_size))); \
                entry++) 

#endif	/* __RARRAY_H__ */



