//client.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2012
 *
 *  This file is part of RoarAudio PlayList Daemon,
 *  a playlist management daemon for RoarAudio.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "rpld.h"

static const struct rpld_proto g_proto[] = {
 {RPLD_PROTO_SIMPLE, "RPLD Simple",
  NULL, NULL, proto_simple_client_handle, NULL, NULL},
 {RPLD_PROTO_MPD, "MPD",
  proto_mpd_client_set_proto, NULL, proto_mpd_client_handle, NULL, NULL}
};

static struct rpld_client {
 enum rpld_client_state state;
 pli_t playlist;
 pli_t queue; // which playlist is the main queue for this client?
 struct rpld_playlist_pointer * tmpptr;
 struct roar_vio_calls con;
 enum rpld_client_acclev acclev;

 // for proto functions.
 const struct rpld_proto * proto;
 struct roar_buffer * obuffer;
 void * userdata;
} g_clients[RPLD_CLIENTS_MAX];

#define PROTO_RUN(id,func) ((g_clients[(id)].proto == NULL || g_clients[(id)].proto->func == NULL) ? -1 : \
                            g_clients[(id)].proto->func((id), &(g_clients[(id)].con), \
                                                        &(g_clients[(id)].obuffer), &(g_clients[(id)].userdata)))

void client_init(void) {
 int i;

 memset(g_clients, 0, sizeof(g_clients));

 for (i = 0; i < RPLD_CLIENTS_MAX; i++)
  g_clients[i].state = CSTATE_UNUSED;
}

int  client_new(struct roar_vio_calls ** vio) {
 int i;

 if ( vio == NULL )
  return -1;

 for (i = 0; i < RPLD_CLIENTS_MAX; i++) {
  if ( g_clients[i].state == CSTATE_UNUSED ) {
   g_clients[i].state     =  CSTATE_USED;
   g_clients[i].playlist  =  0; /* 0=main queue */
   g_clients[i].queue     =  0; /* 0=main queue */
   g_clients[i].tmpptr    =  NULL;
   g_clients[i].acclev    =  ACCLEV_ALL; // currently all clients have fill access rights
   g_clients[i].proto     =  NULL;
   g_clients[i].userdata  =  NULL;
   g_clients[i].obuffer   =  NULL;
   roar_vio_clear_calls(&(g_clients[i].con));
   *vio = &(g_clients[i].con);
   client_set_proto(i, RPLD_PROTO_SIMPLE);
   return i;
  }
 }

 return -1;
}

int  client_delete(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( g_clients[id].state == CSTATE_UNUSED )
  return 0;

 g_clients[id].state = CSTATE_UNUSED;

 PROTO_RUN(id, unset_proto);
 if ( g_clients[id].userdata != NULL ) {
  roar_mm_free(g_clients[id].userdata);
  g_clients[id].userdata = NULL;
 }

 g_clients[id].proto = NULL;

 if ( g_clients[id].tmpptr != NULL ) {
  rpld_plp_unref(g_clients[id].tmpptr);
  g_clients[id].tmpptr = NULL;
 }

 roar_vio_close(&(g_clients[id].con));

 return 0;
}

int  client_get_proto(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return RPLD_PROTO_UNUSED;

 if ( g_clients[id].state == CSTATE_UNUSED )
  return RPLD_PROTO_UNUSED;

 if ( g_clients[id].proto == NULL )
  return RPLD_PROTO_UNUSED;

 return g_clients[id].proto->proto;
}

int  client_set_proto(int id, int proto) {
 const struct rpld_proto * newproto = NULL;
 size_t i;
 int ret = 0;

 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( g_clients[id].state != CSTATE_USED )
  return -1;

 for (i = 0; i < (sizeof(g_proto)/sizeof(*g_proto)); i++) {
  if ( g_proto[i].proto == proto ) {
   newproto = &(g_proto[i]);
   break;
  }
 }

 if ( newproto == NULL )
  return -1;

 PROTO_RUN(id, unset_proto);

 if ( g_clients[id].userdata != NULL ) {
  roar_mm_free(g_clients[id].userdata);
  g_clients[id].userdata = NULL;
 }

 g_clients[id].proto = newproto;

 if ( newproto->set_proto != NULL )
  ret = PROTO_RUN(id, set_proto);

 return ret;
}

struct roar_vio_calls * client_get_vio(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return NULL;

 if ( g_clients[id].state == CSTATE_UNUSED )
  return NULL;

 return &(g_clients[id].con);
}

enum rpld_client_acclev  client_get_acclev(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return ACCLEV_ERR;

 return g_clients[id].acclev;
}

int  client_set_acclev(int id, enum rpld_client_acclev acclev, int setter) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( setter != -1 ) {
  if ( setter != id )
   if ( setter < 0 || setter >= RPLD_CLIENTS_MAX )
    return -1;

  if ( acclev > g_clients[setter].acclev )
   return -1;
 }

 g_clients[id].acclev = acclev;

 return 0;
}

int  client_inc_acclev(int id, int inc, int setter) {
 int  newlev;
 enum rpld_client_acclev acclev;

 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( setter != id )
  if ( setter < 0 || setter >= RPLD_CLIENTS_MAX )
   return -1;

 newlev = (int)g_clients[id].acclev + inc;

 if ( newlev < 0 ) {
  acclev = ACCLEV_NONE;
 } else {
  acclev = newlev;
 }

 if ( acclev > g_clients[setter].acclev )
  return -1;

 g_clients[id].acclev = acclev;

 return 0;
}

int  client_set_playlist(int id, pli_t pl) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 g_clients[id].playlist = pl;

 return 0;
}

pli_t client_get_playlist(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 return g_clients[id].playlist;
}

int  client_set_queue(int id, pli_t pl) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 // TODO: check if the playlist is a queue.
 g_clients[id].queue = pl;

 return 0;
}

pli_t client_get_queue(int id) {
 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 return g_clients[id].queue;
}

int  client_set_pointer(int id, int pointer, struct rpld_playlist_pointer * plp) {

 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( pointer != POINTER_TEMP ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 if ( g_clients[id].tmpptr != NULL )
  rpld_plp_unref(g_clients[id].tmpptr);

 g_clients[id].tmpptr = plp;

 if ( g_clients[id].tmpptr != NULL )
  rpld_plp_ref(g_clients[id].tmpptr);

 return 0;
}

struct rpld_playlist_pointer * client_get_pointer(int id, int pointer) {
 struct rpld_playlist_pointer * ret;

 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return NULL;

 if ( pointer != POINTER_TEMP ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return NULL;
 }

 ret = g_clients[id].tmpptr;
 if ( ret == NULL ) {
  roar_err_set(ROAR_ERROR_NOENT);
 } else {
  rpld_plp_ref(ret);
 }

 return ret;
}

int  client_handle(int id) {
 int ret = -1;

 ROAR_DBG("client_handle(id=%i) = ?", id);

 if ( id < 0 || id >= RPLD_CLIENTS_MAX )
  return -1;

 if ( g_clients[id].state == CSTATE_UNUSED )
  return -1;

 roar_err_set(ROAR_ERROR_NONE);

 ret = PROTO_RUN(id, handle);

 ROAR_DBG("client_handle(id=%i) = %i", id, ret);
 return ret;
}

int  client_str2proto(const char * str) {
 if ( str == NULL || *str == 0 )
  return -1;

 if ( !strcasecmp(str, "simple") )
  return RPLD_PROTO_SIMPLE;

 if ( !strcasecmp(str, "mpd") )
  return RPLD_PROTO_MPD;

 return -1;
}

int  client_str2acclev(const char * str) {
 if ( !strcasecmp(str, "none") )
  return ACCLEV_NONE;

 if ( !strcasecmp(str, "conctl") )
  return ACCLEV_CONCTL;

 if ( !strcasecmp(str, "guest") )
  return ACCLEV_GUEST;

 if ( !strcasecmp(str, "user") )
  return ACCLEV_USER;

 if ( !strcasecmp(str, "all") )
  return ACCLEV_ALL;

 return -1;
}

//ll
