///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
//
/*Prog:field
NAME: @code{basis} -- view a basis 
@pindex basis
@cindex plotting

SYNOPSIS:
  @example
  	basis @var{name} @var{element} @var{options}
  @end example
DESCRIPTION:
  @noindent
  View a finite element polynomial basis on a given
  reference element (@pxref{reference_element iclass}).
  Also, could view the node set used during interpolation.

EXAMPLE: 
@example
	basis P5 t
	basis P5[warburton] t
	basis P5[warburton,monomial] t
	basis P3 t -node
        basis P5 -node-side 0
	basis P3 T -node
	basis B7 t
	basis S7 t
	basis RT0 t
	basis RT3 t -node
@end example

BASIS AND ELEMENT ARGUMENTS:
  The basis @var{name} is specified as for the @code{space}
  and @code{basis} constructor argument
  (@pxref{space class} and @ref{basis class}).
  The reference @var{element} is one of 
  @code{e}, @code{t}, @code{q}, @code{T}, @code{P}, @code{H}
  (@pxref{reference_element iclass}, 
  @ref{edge iclass} @ref{triangle iclass}, @ref{quadrangle iclass},
  @ref{tetrahedron iclass}, @ref{prism iclass}, @ref{hexahedron iclass}).
@toindex @code{gnuplot}
  The @code{gnuplot} render is used for vizualization.

RAW BASIS:
  @noindent
  The raw basis is used for computing the Vandermonde matrix and
  building the Lagrange basis.
  It can be also viewed by the @code{basis} unix command, suing the switches:
  @table @code
    @item -raw
    @item -fem
  @end table
  For instance:
  @example
	basis M4 t
	basis D5 e
	basis B5 e -raw
  @end example
@noindent
  Note that the @code{-raw} option is used to disambiguate  
  between the Bernstein basis as a FEM basis
  and the Bernstein basis as a raw basis 
  for building the Lagrange basis.

VISUALIZATION OPTION:
  The @code{basis} command supports several modes of visualization:
  @table @code
    @item -poly
	Represents the polynomial functions in evelation, for 1D and 2D
	elements (3D elements visualization not yet supported).
	All basis polynomials are showed in an animation.
    @item -node
        Represents the node location, with distinct colors for 
	each dimension associated to.
    @item -side-node
        Represents the node location, restricted on a specific side.
  @end table

OTHERS OPTIONS:
    @table @code
    @item -verbose
	print messages related to graphic files created and
       command system calls (this is the default).

    @item -noverbose
	does not print previous messages.

    @item -clean
	clear temporary graphic files (this is the default).
    
    @item -noclean
	does not clear temporary graphic files.

    @item -execute
	execute graphic command (this is the default).
    
    @item -noexecute
	does not execute graphic command. Generates only 
	graphic files. This is usefull in conjuction with the
	@code{-noclean} command.
    @end table

LIMITATIONS:
  Polynomial visualization in 3D are not yet supported:
  future development will use paraview with volume mode
  and animation for scaning all basis polynomials.

AUTHOR: Pierre.Saramito@imag.fr
DATE: 11 september 2017.
End:
*/
#include "rheolef/basis.h"
#include "rheolef/skit.h"
using namespace rheolef;
using namespace std;

void usage() {
      cerr << "basis: usage:" << endl
           << "basis "
           << "[approx=P1] "
           << "[element=t] "
           << "{-dubiner|-monomial} "
           << "{-fem|-raw} "
           << "{-poly|-node|-node-side int} "
           << "[-subdivide int] "
           << "[-[no]clean] [-[no]execute] [-[no]verbose] "
           << endl;
      exit (1);
}
int main(int argc, char**argv) {
  environment rheolef (argc,argv);
  // --------------------------------
  // scan command line
  // --------------------------------
  if (argc == 1) usage();
  typedef enum {
     show_poly         = 0,
     show_node         = 1,
     show_node_on_side = 2
  } show_type;
  show_type show = show_poly;
  string approx = "P1";
  char   t      = 't';
  reference_element hat_K;
  hat_K.set_name(t);
  bool   raw    = false;
  size_t nsub   = 0;
  side_information_type sid;
  for (int i = 1; i < argc; i++) {

    // general options:
         if (strcmp (argv[i], "-clean") == 0)     dout.os() << clean;
    else if (strcmp (argv[i], "-noclean") == 0)   dout.os() << noclean;
    else if (strcmp (argv[i], "-execute") == 0)   dout.os() << execute;
    else if (strcmp (argv[i], "-noexecute") == 0) dout.os() << noexecute;
    else if (strcmp (argv[i], "-verbose") == 0)   dout.os() << verbose;
    else if (strcmp (argv[i], "-noverbose") == 0) dout.os() << noverbose;

    // basis options:
    else if (strcmp (argv[i], "-fem") == 0)        raw = false;
    else if (strcmp (argv[i], "-raw") == 0)        raw = true;

    // view options:
    else if (strcmp (argv[i], "-poly")  == 0)      show = show_poly;
    else if (strcmp (argv[i], "-node")  == 0)      show = show_node;
    else if (strcmp (argv[i], "-node-side")  == 0) {
      show = show_node_on_side;
      if (i == argc-1) { cerr << "field -side: option argument missing" << endl; usage(); }
      sid.loc_isid = atoi(argv[++i]);
      warning_macro ("sid.loc_isid ="<<sid.loc_isid);
    } else if (strcmp (argv[i], "-subdivide") == 0) {
      if (i == argc-1) { cerr << "field -subdivide: option argument missing" << endl; usage(); }
      nsub = atoi(argv[++i]);
      dout.os() << setsubdivide (nsub);

    // basis spec:
    } else if (argv[i][0] != '-' && argv[i][0] >= 'A' && argv[i][0] <= 'Z' && strlen(argv[i]) >= 2) {
      approx = argv[i];
    } else if (argv[i][0] != '-' && strlen(argv[i]) == 1) {
      t = argv[i][0];
      hat_K.set_name(t);
    } else {
      cerr << "unexpected option `" << argv[i]<< endl;
      usage();
    }
  }
  if (approx[0] == 'M' || approx[0] == 'D') {
    raw = true;
  }
  if (nsub == 0) {
    nsub = (hat_K.dimension() == 1) ? 1000 : 40;
    dout.os() << setsubdivide (nsub);
  }
  // --------------------------------
  // show
  // --------------------------------
  if (raw) { // raw basis
    if (show != show_poly) {
      cerr << "basis: raw basis no not have nodes (HINT: use -poly option instead of -node)" << endl;
      exit (1);
    }
    basis_raw b (approx);
    b.put (cout, hat_K);
    return 0;
  }
  // fem basis
  basis b (approx);
  switch (show) {
    case show_poly:
      b.put (cout, hat_K);
      break;
    case show_node:
      b.put_hat_node (cout, hat_K);
      break;
    case show_node_on_side:
      b.put_hat_node_on_side (cout, hat_K, sid);
      break;
  }
}
