#include "RFMacroRecorder.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

namespace rfb {


#define MAX_BLOCK_SIZE 8192


RFMacroRecorder::RFMacroRecorder( char* _filename )
  : timer( 0 )
  , timestamp( 0 )
  , pos( 0 )
  , delay( false )
{
  fd = creat( _filename, S_IRUSR | S_IWUSR );
  buffer = (unsigned char*) malloc( MAX_BLOCK_SIZE );
  writeHeader();
}


RFMacroRecorder::RFMacroRecorder( int _fd )
  : fd( _fd )
  , timer( 0 )
  , timestamp( 0 )
  , pos( 0 )
  , delay( false )
{
  buffer = (unsigned char*) malloc( MAX_BLOCK_SIZE );
  writeHeader();
}

RFMacroRecorder::~RFMacroRecorder()
{
  char tmp[256];
  int oldTimestamp = timestamp;
  updateTimestamp();
  int delayTime = timestamp - oldTimestamp;
  if ( delay ) sprintf( tmp, " delay %ims\n#EOF", delayTime );
    else sprintf( tmp, "\n#EOF" );
  record( (unsigned char*) tmp, strlen( tmp ) );
  flush();
  free( buffer );
  close( fd );
}




void RFMacroRecorder::recordKeyDown( CARD32 keysym )
{
  char tmp[256];
  int oldTimestamp = timestamp;
  updateTimestamp();
  int delayTime = timestamp - oldTimestamp;
  if ( delay ) {
    sprintf( tmp, " delay %ims\nkey %#.6x down", delayTime, (int) keysym );
  } else {
    sprintf( tmp, "\nkey %#.6x down", (int) keysym );
  }
  record( (unsigned char*) tmp, strlen( tmp ) );
  delay = true;
}

void RFMacroRecorder::recordKeyUp( CARD32 keysym )
{
  char tmp[256];
  int oldTimestamp = timestamp;
  updateTimestamp();
  int delayTime = timestamp - oldTimestamp;
  if ( delay ) {
    sprintf( tmp, " delay %ims\nkey %#.6x up", delayTime, (int) keysym );
  } else {
    sprintf( tmp, "\nkey %#.6x up", (int) keysym );
  }
  record( (unsigned char*) tmp, strlen( tmp ) );
  delay = true;
}

void RFMacroRecorder::recordPointer( CARD16 x, CARD16 y )
{
  char tmp[256];
  int oldTimestamp = timestamp;
  updateTimestamp();
  int delayTime = timestamp - oldTimestamp;
  if ( delay ) {
    sprintf( tmp, " delay %ims\npointer %i %i", delayTime, (int) x, (int) y );
  } else {
    sprintf( tmp, "\npointer %i %i", (int) x, (int) y );
  }
  record( (unsigned char*) tmp, strlen( tmp ) );
  delay = true;
}

void RFMacroRecorder::recordButtonDown( int button )
{
  char tmp[256];
  int oldTimestamp = timestamp;
  updateTimestamp();
  int delayTime = timestamp - oldTimestamp;
  if ( delay ) {
    sprintf( tmp, " delay %ims\nbutton %i down", delayTime, button );
  } else {
    sprintf( tmp, "\nbutton %i down", button );
  }
  record( (unsigned char*) tmp, strlen( tmp ) );
  delay = true;
}

void RFMacroRecorder::recordButtonUp( int button )
{
  char tmp[256];
  int oldTimestamp = timestamp;
  updateTimestamp();
  int delayTime = timestamp - oldTimestamp;
  if ( delay ) {
    sprintf( tmp, " delay %ims\nbutton %i up", delayTime, button );
  } else {
    sprintf( tmp, "\nbutton %i up", button );
  }
  record( (unsigned char*) tmp, strlen( tmp ) );
  delay = true;
}



void RFMacroRecorder::record( unsigned char *_data, unsigned int _size )
{
  while ( _size ) {
    if ( pos == MAX_BLOCK_SIZE ) flush();
    unsigned int max = MAX_BLOCK_SIZE - pos;
    max = ( max < _size )? max : _size;
    memcpy( buffer + pos, _data, max );
    pos += max;
    _data += max;
    _size -= max;
  }
}


void RFMacroRecorder::flush()
{
  if (!pos) return;
  write( fd, buffer, pos );
  pos = 0;
}
  
  
void RFMacroRecorder::startTimer()
{
  if ( timer ) return;
  gettimeofday( &tvStart, NULL );
  timer = true;
}

void RFMacroRecorder::stopTimer()
{
  if ( !timer ) return;
  updateTimestamp();
  timer = false;
}

void RFMacroRecorder::updateTimestamp()
{
  if (!timer) return;
  struct timeval tv;
  gettimeofday( &tv, NULL );
  tv.tv_sec -= tvStart.tv_sec;
  tv.tv_usec -= tvStart.tv_usec;
  if ( tv.tv_usec < 0 ) {
    tv.tv_sec--;
    tv.tv_usec += 1000000;
  }
  unsigned int delta = tv.tv_sec * 1000 + tv.tv_usec / 1000;
  
  if ( delta ) {
    flush();
    timestamp = timestamp + delta;
    tvStart.tv_sec += tv.tv_sec;
    tvStart.tv_usec += tv.tv_usec;
    if ( tvStart.tv_usec > 1000000 ) {
      tv.tv_sec++;
      tv.tv_usec -= 1000000;
    }
  }
}


void RFMacroRecorder::writeHeader()
{
  char header[] = "RFM 001.000\nshared";
  write( fd, header, 18 );
}


} // namespace rfb
