/*
 *  Rexx/Tk
 *  Copyright (C) 2000  Mark Hessling  <M.Hessling@qut.edu.au>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "rexxtk.h"

char *RxPackageName = "rexxtktree";
char *ExtensionSource =
"# The following Tree widget based heavily on code from:\n"
"# drh@acm.org http://www.hwaci.com/drh/\n"
"# Copyright (C) 1997,1998 D. Richard Hipp\n"
"#\n"
"option add *highlightThickness 0\n"
"switch $tcl_platform(platform) {\n"
"   unix {\n"
"      set Tree(font) -adobe-helvetica-medium-r-normal-*-11-80-100-100-p-56-iso8859-1\n"
"   }\n"
"   windows {\n"
"      set Tree(font) -adobe-helvetica-medium-r-normal-*-14-100-100-100-p-76-iso8859-1\n"
"   }\n"
"}\n"
"#\n"
"# Bitmaps used to show which parts of the tree can be opened.\n"
"#\n"
"set maskdata \"#define solid_width 9\n#define solid_height 9\"\n"
"append maskdata {\n"
"  static unsigned char solid_bits[] = {\n"
"   0xff, 0x01, 0xff, 0x01, 0xff, 0x01, 0xff, 0x01, 0xff, 0x01, 0xff, 0x01,\n"
"   0xff, 0x01, 0xff, 0x01, 0xff, 0x01\n"
"  };\n"
"}\n"
"set data \"#define open_width 9\n#define open_height 9\"\n"
"append data {\n"
"  static unsigned char open_bits[] = {\n"
"   0xff, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x7d, 0x01, 0x01, 0x01,\n"
"   0x01, 0x01, 0x01, 0x01, 0xff, 0x01\n"
"  };\n"
"}\n"
"image create bitmap Tree:openbm -data $data -maskdata $maskdata -foreground black -background white\n"
"set data \"#define closed_width 9\n#define closed_height 9\"\n"
"append data {\n"
"  static unsigned char closed_bits[] = {\n"
"   0xff, 0x01, 0x01, 0x01, 0x11, 0x01, 0x11, 0x01, 0x7d, 0x01, 0x11, 0x01,\n"
"   0x11, 0x01, 0x01, 0x01, 0xff, 0x01\n"
"  };\n"
"}\n"
"image create bitmap Tree:closedbm -data $data -maskdata $maskdata -foreground black -background white\n"
"#\n"
"# Create a new tree widget.  $args become the configuration arguments to\n"
"# the canvas widget from which the tree is constructed.\n"
"#\n"
"proc Tree:create {w args} {\n"
"  global Tree\n"
"  set rc [eval canvas $w -bg white]\n"
"  bind $w <Destroy> \"Tree:delnode $w /\"\n"
"  Tree:dfltconfig $w /\n"
"  Tree:buildwhenidle $w\n"
"  set Tree($w:selection) {}\n"
"  set Tree($w:selidx) {}\n"
"  set Tree($w:opencommand) {}\n"
"  set Tree($w:closecommand) {}\n"
"  foreach {op arg} $args {\n"
"    switch -exact -- $op {\n"
"      -opencommand {set Tree($w:opencommand) $arg}\n"
"      -closecommand {set Tree($w:closecommand) $arg}\n"
"      default {eval $w config $op {$arg}}\n"
"    }\n"
"  }\n"
"  return $rc\n"
"}\n"
"#\n"
"# Pass configuration options to the tree widget\n"
"#\n"
"proc Tree:config {w args} {\n"
"  foreach {op arg} $args {\n"
"    switch -exact -- $op {\n"
"      -opencommand {set Tree($w:opencommand) $arg}\n"
"      -closecommand {set Tree($w:closecommand) $arg}\n"
"      default {eval $w config $op {$arg}}\n"
"    }\n"
"  }\n"
"}\n"
"#\n"
"# Insert a new element $v into the tree $w.\n"
"#\n"
"proc Tree:addnode {w v args} {\n"
"  global Tree\n"
"  set dir [file dirname $v]\n"
"  set n [file tail $v]\n"
"  if {![info exists Tree($w:$dir:open)]} {\n"
"    error \"Tree:parent item $dir is missing\"\n"
"  }\n"
"  set i [lsearch -exact $Tree($w:$dir:children) $n]\n"
"  if {$i>=0} {\n"
"    error \"Tree: item $v already exists\"\n"
"  }\n"
"  lappend Tree($w:$dir:children) $n\n"
"  set Tree($w:$dir:children) [lsort $Tree($w:$dir:children)]\n"
"  Tree:dfltconfig $w $v\n"
"  foreach {op arg} $args {\n"
"    switch -exact -- $op {\n"
"      -image {set Tree($w:$v:icon) $arg}\n"
"      -tags {set Tree($w:$v:tags) $arg}\n"
"    }\n"
"  }\n"
"  Tree:buildwhenidle $w\n"
"}\n"
"#\n"
"# Delete element $v from the tree $w.  If $v is /, then the widget is\n"
"# deleted.\n"
"#\n"
"proc Tree:delnode {w v} {\n"
"  global Tree\n"
"  if {![info exists Tree($w:$v:open)]} return\n"
"  if {[string compare $v /]==0} {\n"
"    # delete the whole widget\n"
"    catch {destroy $w}\n"
"    foreach c $Tree($w:$v:children) {\n"
"      catch {Tree:delnode $w $v/$c}\n"
"    }\n"
"    foreach t [array names Tree $w:*] {\n"
"      unset Tree($t)\n"
"    }\n"
"  } else {\n"
"    # just delete the specified widget\n"
"    foreach c $Tree($w:$v:children) {\n"
"      catch {Tree:delnode $w $v/$c}\n"
"    }\n"
"    unset Tree($w:$v:open)\n"
"    unset Tree($w:$v:children)\n"
"    unset Tree($w:$v:icon)\n"
"    set dir [file dirname $v]\n"
"    set n [file tail $v]\n"
"    set i [lsearch -exact $Tree($w:$dir:children) $n]\n"
"    if {$i>=0} {\n"
"      set Tree($w:$dir:children) [lreplace $Tree($w:$dir:children) $i $i]\n"
"    }\n"
"    Tree:buildwhenidle $w\n"
"  }\n"
"}\n"
"#\n"
"# Change the selection to the indicated item\n"
"#\n"
"proc Tree:setselection {w v} {\n"
"  global Tree\n"
"  set Tree($w:selection) $v\n"
"  Tree:drawselection $w\n"
"}\n"
"# \n"
"# Retrieve the current selection\n"
"#\n"
"proc Tree:getselection w {\n"
"  global Tree\n"
"  return $Tree($w:selection)\n"
"}\n"
"# Open a branch of a tree\n"
"#\n"
"proc Tree:open {w v} {\n"
"  global Tree\n"
"  if {[info exists Tree($w:$v:open)] && $Tree($w:$v:open)==0\n"
"      && [info exists Tree($w:$v:children)] \n"
"      && [string length $Tree($w:$v:children)]>0} {\n"
"    set Tree($w:$v:open) 1\n"
"    Tree:build $w\n"
"    if {[string length $Tree($w:opencommand)]} {\n"
"       eval setRexxtk $Tree($w:opencommand) $v\n"
"    }\n"
"  }\n"
"}\n"
"proc Tree:close {w v} {\n"
"  global Tree\n"
"  if {[info exists Tree($w:$v:open)] && $Tree($w:$v:open)==1} {\n"
"    set Tree($w:$v:open) 0\n"
"    Tree:build $w\n"
"    if {[string length $Tree($w:closecommand)]} {\n"
"       eval setRexxtk $Tree($w:closecommand) $v\n"
"    }\n"
"  }\n"
"}\n"
"#\n"
"# Return the full pathname of the label for widget $w that is located\n"
"# at real coordinates $x, $y\n"
"#\n"
"proc Tree:labelat {w x y} {\n"
"  set x [$w canvasx $x]\n"
"  set y [$w canvasy $y]\n"
"  global Tree\n"
"  foreach m [$w find overlapping $x $y $x $y] {\n"
"    if {[info exists Tree($w:tag:$m)]} {\n"
"      return $Tree($w:tag:$m)\n"
"    }\n"
"  }\n"
"  return ""\n"
"}\n"
"#\n"
"# INTERNAL FUNCTIONS\n"
"# Initialize a element of the tree.\n"
"# Internal use only\n"
"#\n"
"proc Tree:dfltconfig {w v} {\n"
"  global Tree\n"
"  set Tree($w:$v:children) {}\n"
"  set Tree($w:$v:open) 0\n"
"  set Tree($w:$v:icon) {}\n"
"  set Tree($w:$v:tags) {}\n"
"}\n"
"# Internal use only.\n"
"# Draw the tree on the canvas\n"
"proc Tree:build w {\n"
"  global Tree\n"
"  $w delete all\n"
"  catch {unset Tree($w:buildpending)}\n"
"  set Tree($w:y) 30\n"
"  Tree:buildlayer $w / 10\n"
"  $w config -scrollregion [$w bbox all]\n"
"  Tree:drawselection $w\n"
"}\n"
"# Internal use only.\n"
"# Build a single layer of the tree on the canvas.  Indent by $in pixels\n"
"proc Tree:buildlayer {w v in} {\n"
"  global Tree\n"
"  if {$v==\"/\"} {\n"
"    set vx {}\n"
"  } else {\n"
"    set vx $v\n"
"  }\n"
"  set start [expr $Tree($w:y)-10]\n"
"  set y 0\n"
"  foreach c $Tree($w:$v:children) {\n"
"    set y $Tree($w:y)\n"
"    incr Tree($w:y) 17\n"
"    $w create line $in $y [expr $in+10] $y -fill gray50 \n"
"    set icon $Tree($w:$vx/$c:icon)\n"
"    set taglist x\n"
"    foreach tag $Tree($w:$vx/$c:tags) {\n"
"      lappend taglist $tag\n"
"    }\n"
"    set x [expr $in+12]\n"
"    if {[string length $icon]>0} {\n"
"      set k [$w create image $x $y -image $icon -anchor w -tags $taglist]\n"
"      incr x 20\n"
"      set Tree($w:tag:$k) $vx/$c\n"
"    }\n"
"    set j [$w create text $x $y -text $c -font $Tree(font) -anchor w -tags $taglist]\n"
"    set Tree($w:tag:$j) $vx/$c\n"
"    set Tree($w:$vx/$c:tag) $j\n"
"    if {[string length $Tree($w:$vx/$c:children)]} {\n"
"      if {$Tree($w:$vx/$c:open)} {\n"
"         set j [$w create image $in $y -image Tree:openbm]\n"
"         if {[string length $Tree($w:closecommand)]} {\n"
"            $w bind $j <1> \"set Tree($w:$vx/$c:open) 0; Tree:build $w;setRexxtk $Tree($w:closecommand) $vx/$c\"\n"
"         } else {\n"
"            $w bind $j <1> \"set Tree($w:$vx/$c:open) 0; Tree:build $w\"\n"
"         }\n"
"         Tree:buildlayer $w $vx/$c [expr $in+18]\n"
"      } else {\n"
"         set j [$w create image $in $y -image Tree:closedbm]\n"
"         if {[string length $Tree($w:opencommand)]} {\n"
"            $w bind $j <1> \"set Tree($w:$vx/$c:open) 1; Tree:build $w;setRexxtk $Tree($w:opencommand) $vx/$c\"\n"
"         } else {\n"
"            $w bind $j <1> \"set Tree($w:$vx/$c:open) 1; Tree:build $w\"\n"
"         }\n"
"      }\n"
"    }\n"
"  }\n"
"  set j [$w create line $in $start $in [expr $y+1] -fill gray50 ]\n"
"  $w lower $j\n"
"}\n"
"# Internal use only.\n"
"# Draw the selection highlight\n"
"proc Tree:drawselection w {\n"
"  global Tree\n"
"  if {[string length $Tree($w:selidx)]} {\n"
"    $w delete $Tree($w:selidx)\n"
"  }\n"
"  set v $Tree($w:selection)\n"
"  if {[string length $v]==0} return\n"
"  if {![info exists Tree($w:$v:tag)]} return\n"
"  set bbox [$w bbox $Tree($w:$v:tag)]\n"
"  if {[llength $bbox]==4} {\n"
"    set i [eval $w create rectangle $bbox -fill skyblue -outline {{}}]\n"
"    set Tree($w:selidx) $i\n"
"    $w lower $i\n"
"  } else {\n"
"    set Tree($w:selidx) {}\n"
"  }\n"
"}\n"
"# Internal use only\n"
"# Call Tree:build then next time were idle\n"
"proc Tree:buildwhenidle w {\n"
"  global Tree\n"
"  if {![info exists Tree($w:buildpending)]} {\n"
"    set Tree($w:buildpending) 1\n"
"    after idle \"Tree:build $w\"\n"
"  }\n"
"}\n";

RexxFunctionHandler TkTree  ;
RexxFunctionHandler TkTreeAddNode  ;
RexxFunctionHandler TkTreeClose  ;
RexxFunctionHandler TkTreeDNode  ;
RexxFunctionHandler TkTreeGetLabel  ;
RexxFunctionHandler TkTreeGetSelection  ;
RexxFunctionHandler TkTreeOpen  ;
RexxFunctionHandler TkTreeSetSelection  ;
RexxFunctionHandler TkTreeLoadFuncs    ;
RexxFunctionHandler TkTreeDropFuncs    ;

/*-----------------------------------------------------------------------------
 * Table of TK Functions. Used to install/de-install functions.
 * If you change this table, don't forget to change the table at the end
 * of this file.
 *----------------------------------------------------------------------------*/
RexxFunction RxPackageFunctions[] = {
   { "TKTREEDROPFUNCS"       ,TkTreeDropFuncs       ,"TkTreeDropFuncs"       , 1 },
   { "TKTREELOADFUNCS"       ,TkTreeLoadFuncs       ,"TkTreeLoadFuncs"       , 0 }, /* don't load this from a DLL */
   { "TKTREE"                ,TkTree                ,"TkTree"                , 1 },
   { "TKTREEADDNODE"         ,TkTreeAddNode         ,"TkTreeAddNode"         , 1 },
   { "TKTREECLOSE"           ,TkTreeClose           ,"TkTreeClose"           , 1 },
   { "TKTREEDNODE"           ,TkTreeDNode           ,"TkTreeDNode"           , 1 },
   { "TKTREEGETLABEL"        ,TkTreeGetLabel        ,"TkTreeGetLabel"        , 1 },
   { "TKTREEGETSELECTION"    ,TkTreeGetSelection    ,"TkTreeGetSelection"    , 1 },
   { "TKTREEOPEN"            ,TkTreeOpen            ,"TkTreeOpen"            , 1 },
   { "TKTREESETSELECTION"    ,TkTreeSetSelection    ,"TkTreeSetSelection"    , 1 },
   { NULL, NULL, NULL, 0 }
};

static char czTclCommand[TCLCOMMANDLEN];
static REXXTKDATA *RexxTkData;
   
#ifdef WIN32
Tcl_Interp *RexxTk_TclCreateInterp(void)
{
   return RexxTkData->Dyn_TclCreateInterp();
}

int RexxTk_TclEval(Tcl_Interp *interp, char *string)
{
   return RexxTkData->Dyn_TclEval( interp, string );
}

int RexxTk_TclInit(Tcl_Interp *interp)
{
   return RexxTkData->Dyn_TclInit( interp );
}

int RexxTk_TkInit(Tcl_Interp *interp)
{
   return RexxTkData->Dyn_TkInit( interp );
}
#endif

/*
 * Rexx/Tk tree functions start here...
 */

/*
 * Tree:create pathName ?options?
 * TkTree(pathName [,options])
 */
RFH_RETURN_TYPE TkTree
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   return rtk_TypeA(RexxTkData,czTclCommand,name,"Tree:create", argc, argv, retstr);
}

/*
 * Tree:addnode pathName name ?option...?
 * TkTreeAddNode(pathName, name [,options])
 */
RFH_RETURN_TYPE TkTreeAddNode
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 0 ) )
      return 1;

   strcpy(czTclCommand, "Tree:addnode ");
   strncat(czTclCommand, argv[0].strptr, argv[0].strlength);
   strcat(czTclCommand, " ");
   strncat(czTclCommand, argv[1].strptr, argv[1].strlength);
   if (argc > 2)
   {
      if ( rtk_procOptArgs(name,czTclCommand,argc,argv,2) )
         return 1;
   }
   
   DEBUGDUMP(fprintf(stderr,"%s-%d: (TkTreeAddNode) command: %s\n",__FILE__,__LINE__,czTclCommand);)

   if (Tcl_Eval(RexxTkData->RexxTkInterp, czTclCommand) != TCL_OK)
   {
      return ReturnError(RexxTkData, retstr, -1, RexxTkData->RexxTkInterp->result );
   }
   return RxReturnString( retstr, RexxTkData->RexxTkInterp->result ) ;
}

/*
 * Tree:close pathName name
 * TkTreeClose(pathName, name)
 */
RFH_RETURN_TYPE TkTreeClose
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 2 ) )
      return 1;

   return rtk_TypeB(RexxTkData,czTclCommand,name,"Tree:close", argc, argv, retstr);
}

/*
 * Tree:delnode pathName name
 * TkTreeDNode(pathName, name)
 */
RFH_RETURN_TYPE TkTreeDNode
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 2 ) )
      return 1;

   return rtk_TypeB(RexxTkData,czTclCommand,name,"Tree:delnode", argc, argv, retstr);
}

/*
 * Tree:labelat pathName x y
 * TkTreeGetLabel(pathName, x, y)
 */
RFH_RETURN_TYPE TkTreeGetLabel
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 3, 3 ) )
      return 1;

   return rtk_TypeB(RexxTkData,czTclCommand,name,"Tree:labelat", argc, argv, retstr);
}

/*
 * Tree:getselection pathName
 * TkTreeGetSelection(pathName)
 */
RFH_RETURN_TYPE TkTreeGetSelection
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 1, 1 ) )
      return 1;

   return rtk_TypeB(RexxTkData,czTclCommand,name,"Tree:getselection", argc, argv, retstr);
}

/*
 * Tree:open pathName name
 * TkTreeOpen(pathName, name)
 */
RFH_RETURN_TYPE TkTreeOpen
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 2 ) )
      return 1;

   return rtk_TypeB(RexxTkData,czTclCommand,name,"Tree:open", argc, argv, retstr);
}

/*
 * Tree:setselection pathName label
 * TkTreeSetSelection(pathName, label)
 */
RFH_RETURN_TYPE TkTreeSetSelection
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 2 ) )
      return 1;

   return rtk_TypeB(RexxTkData,czTclCommand,name,"Tree:setselection", argc, argv, retstr);
}

RFH_RETURN_TYPE TkTreeDropFuncs
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
 ULONG rc=0;
 int unload=0;

 if ( my_checkparam(name, argc, 0, 1 ) )
    return( 1 );
 if ( argv[0].strlength == 6
 &&   memcmpi( argv[0].strptr, "UNLOAD", 6 ) == 0 )
    unload = 1;
 rc = DeregisterRxFunctions( unload );
 return RxReturnNumber( retstr, rc );
}


RFH_RETURN_TYPE TkTreeLoadFuncs
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   ULONG rc = 0L;

#if defined(DYNAMIC_LIBRARY)
   if ( !QueryRxFunction( "TKWAIT" ) )
   {
      fprintf(stderr,"The base Rexx/Tk function package must be loaded before this one\n");
      return RxReturnNumber( retstr, 1 );
   }
   /*
    * get the pointer to the tcl Interpreter and the base data from base Rexx/Tk
    * library
    */
   if ( argc == 0 )
   {
      fprintf(stderr,"You must pass the return value from TkGetBaseData() as the one and only argument.\n");
      return RxReturnNumber( retstr, 1 );
   }
   RexxTkData = (REXXTKDATA *)atol(argv[0].strptr);
   rc = InitRxPackage( NULL );
   /* 
    * Register all external functions
    */
   if ( !rc )
   {
      rc = RegisterRxFunctions( );
   }
#endif
   return RxReturnNumber( retstr, rc );
}
   
/*
 * The following functions are used in rxpackage.c
 */

/*-----------------------------------------------------------------------------
 * Execute any initialisation
 *----------------------------------------------------------------------------*/
int InitialisePackage

#ifdef HAVE_PROTO
   ( void )
#else
   ( )
#endif

{
   InternalTrace( "InitialisePackage", NULL );

   /*
    * Install the Tree widget
    */
   if (Tcl_Eval( RexxTkData->RexxTkInterp,ExtensionSource ) !=TCL_OK) {
      fprintf(stderr, "Tk_Eval for Tree widget failed miserably at line %d: %s\n", RexxTkData->RexxTkInterp->errorLine, RexxTkData->RexxTkInterp->result);
      return 1;
   }
   DEBUGDUMP(fprintf(stderr,"%s-%d: After Tcl_Eval()\n",__FILE__,__LINE__);)
   return 0;
}

/*-----------------------------------------------------------------------------
 * Execute any termination
 *----------------------------------------------------------------------------*/
int TerminatePackage

#ifdef HAVE_PROTO
   ( void )
#else
   ( )
#endif

{
   return 0;
}


#if defined(USE_REXX6000)
/*
 * This function is used as the entry point for the REXX/6000
 * Rexx Interpreter
 * If you change this table, don't forget to change the table at the
 * start of this file.
 */
USHORT InitFunc( RXFUNCBLOCK **FuncBlock )
{
   static RXFUNCBLOCK funcarray[] =
   {
      { "TKTREEDROPFUNCS"       ,TkTreeDropFuncs      ,NULL },
      { "TKTREELOADFUNCS"       ,TkTreeLoadFuncs      ,NULL },
      { "TKTREE"                ,TkTree               ,NULL },
      { "TKTREEADDNODE"         ,TkTreeAddNode        ,NULL },
      { "TKTREECLOSE"           ,TkTreeClose          ,NULL },
      { "TKTREEDNODE"           ,TkTreeDNode          ,NULL },
      { "TKTREEGETLABEL"        ,TkTreeGetLabel       ,NULL },
      { "TKTREEGETSELECTION"    ,TkTreeGetSelection   ,NULL },
      { "TKTREEOPEN"            ,TkTreeOpen           ,NULL },
      { "TKTREESETSELECTION"    ,TkTreeSetSelection   ,NULL },
      { NULL, NULL, NULL }
   } ;
   *FuncBlock = funcarray;
   return (USHORT)0;
}
#endif
