/*  RetroArch - A frontend for libretro.
 *  Copyright (C) 2011-2016 - Daniel De Matteis
 *
 *  RetroArch is free software: you can redistribute it and/or modify it under the terms
 *  of the GNU General Public License as published by the Free Software Found-
 *  ation, either version 3 of the License, or (at your option) any later version.
 *
 *  RetroArch is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 *  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along with RetroArch.
 *  If not, see <http://www.gnu.org/licenses/>.
 */

#include "../msg_hash.h"
#include <stdint.h>
#include <string.h>

#include <retro_assert.h>
#include <compat/strl.h>

#include "../../configuration.h"

 /* IMPORTANT:
  * For non-english characters to work without proper unicode support,
  * we need this file to be encoded in ISO 8859-1 (Latin1), not UTF-8.
  * If you save this file as UTF-8, you'll break non-english characters
  * (e.g. German "Umlauts" and Portugese diacritics).
 */
/* DO NOT REMOVE THIS. If it causes build failure, it's because you saved the file as UTF-8. Read the above comment. */
extern const char force_iso_8859_1[sizeof("")==12+1 ? 1 : -1];

int menu_hash_get_help_pt(uint32_t hash, char *s, size_t len)
{
   uint32_t driver_hash = 0;
   settings_t      *settings = config_get_ptr();

   /* If this one throws errors, stop sledgehammering square pegs into round holes and */
   /* READ THE COMMENTS at the top of the file. */
   (void)sizeof(force_iso_8859_1);

   switch (hash)
   {
      case MENU_LABEL_INPUT_DRIVER:
         if (settings)
            driver_hash = msg_hash_calculate(settings->input.driver);

         switch (driver_hash)
         {
            case MENU_LABEL_INPUT_DRIVER_UDEV:
               snprintf(s, len,
                     "Driver de entrada udev. \n"
                     " \n"
                     "Esse driver funciona sem o X. \n"
                     " \n"
                     "Ele usa a recente API de joypads \n"
                     "evdev para supporte a joystick. \n"
                     "Suporta Hot-Swap e force feedback \n"
                     "(se suportado pelo dispositivo). \n"
                     " \n"
                     "O driver l os eventos evdev para suporte a \n"
                     "teclado. Suporta tambm callback de teclado, \n"
                     "mouses e touchpads. \n"
                     " \n"
                     "Em geral, na maioria das distribuies, os ns \n"
                     "/dev/input so root-only (modo 600). Mas voc pode \n"
                     "definir uma regra udev para dar acesso a non-roots."
                     );
               break;
            case MENU_LABEL_INPUT_DRIVER_LINUXRAW:
               snprintf(s, len,
                     "Driver de Entrada linuxraw. \n"
                     " \n"
                     "Esse driver requer um TTY ativo. Eventos de \n"
                     "teclado so lidos diretamente do TTY, tornando-o \n"
                     "simples, mas no to flexvel quanto o udev. \n" "Mouses, etc, no so suportados. \n"
                     " \n"
                     "Esse driver usa a antiga API de joysticks \n"
                     "(/dev/input/js*).");
               break;
            default:
               snprintf(s, len,
                     "Driver de Entrada.\n"
                     " \n"
                     "Dependendo do driver de vdeo, pode ser necessrio \n"
                     "forar um driver de entrada diferente.");
               break;
         }
         break;
      case MENU_LABEL_LOAD_CONTENT:
         snprintf(s, len,
               "Carregar Contedo. \n"
               "Busca contedo. \n"
               " \n"
               "Para carregar contedo, voc precisa de \n"
               "um core libretro para usar, e um arquivo \n"
               "de contedo. \n"
               " \n"
               "Para controlar onde o menu comea a \n"
               "buscar contedo, defina o Diretrio \n"
               "de Navegao. Se no estiver definido, \n"
               "o Retroarch comear no diretrio raiz. \n"
               " \n"
               "O navegador vai filtrar pelas extenses \n"
               "do mais recente core definido em 'Core', \n"
               "e o usar quando o contedo estiver \n"
               "carregado."
               );
         break;
      case MENU_LABEL_CORE_LIST:
         snprintf(s, len,
               "Carregar Core. \n"
               " \n"
               "Busca uma implementao de um core \n"
               "libretro. Onde a busca inicia depende \n"
               "do caminho do seu Diretrio de Cores. \n"
               "Se no definido, comear no raiz. \n"
               " \n"
               "Se o Diretrio de Cores estiver definido, \n"
               "o menu o usar como pasta inicial. Se for um \n"
               "caminho completo, ele comear na pasta onde \n"
               "o arquivo estiver.");
         break;
      case MENU_LABEL_LOAD_CONTENT_HISTORY:
         snprintf(s, len,
               "Carregando contedo do histrico. \n"
               " \n"
               "Ao carregar contedos, suas combinaes com \n"
               "cores so salvas no histrico. \n"
               " \n"
               "O histrico  salvo em um arquivo no mesmo \n"
               "diretrio do arquivo de configurao. Se nenhuma \n"
               "configurao tiver sido carregada, o histrico \n"
               "no ser salvo ou carregado e no vai existir no \n"
               "menu principal."
               );
         break;
      case MENU_LABEL_VIDEO_DRIVER:
         if (settings)
            driver_hash = msg_hash_calculate(settings->video.driver);

         switch (driver_hash)
         {
            case MENU_LABEL_VIDEO_DRIVER_GL:
               snprintf(s, len,
                     "Driver de Vdeo OpenGL. \n"
                     " \n"
                     "Esse driver permite o uso de cores libretro GL  \n"
                     "em adio s implementaes de cores de \n"
                     "renderizao por software.\n"
                     " \n"
                     "O desempenho das implementaes dos cores de\n"
                     "renderizao por software e libretro GL \n"
                     "depende do driver GL instalado em sua \n"
                     "placa de vdeo.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_SDL2:
               snprintf(s, len,
                     "Driver de Vdeo SDL 2.\n"
                     " \n"
                     "Esse  um driver de vdeo SDL 2 de \n"
                     "renderizao por software.\n"
                     " \n"
                     "O desempenho das implementaes dos cores de \n"
                     "renderizao por software depende da \n"
                     "implementao SDL de sua plataforma.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_SDL1:
               snprintf(s, len,
                     "Driver de Vdeo SDL.\n"
                     " \n"
                     "Esse  um driver de vdeo SDL 1.2 de \n"
                     "renderizao por software.\n"
                     " \n"
                     "O desemprenho  considerado subtimo. \n"
                     "Considere seu uso apenas em ltimo caso.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_D3D:
               snprintf(s, len,
                     "Driver de Vdeo Direct3D. \n"
                     " \n"
                     "O desempenho das implementaes dos cores de\n"
                     "renderizao por software depende do driver \n"
                     "D3D instalado em sua placa de vdeo.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_EXYNOS:
               snprintf(s, len,
                     "Driver de Vdeo Exynos-G2D. \n"
                     " \n"
                     "Esse  um driver de vdeo Exynos de baixo nvel. \n"
                     "Usa o bloco G2D do SoC Samsung Exynos \n"
                     "para operaes de blit. \n"
                     " \n"
                     "O desempenho para cores de renderizao por \n"
                     "software deve ser timo.");
               break;
            case MENU_LABEL_VIDEO_DRIVER_SUNXI:
               snprintf(s, len,
                     "Driver de Vdeo Sunxi-G2D. \n"
                     " \n"
                     "Esse  um driver de vdeo Sunxi de baixo nvel. \n"
                     "Usa o bloco G2D dos SoCs Allwinner.");
               break;
            default:
               snprintf(s, len,
                     "Driver de Vdeo em uso.");
               break;
         }
         break;
      case MENU_LABEL_AUDIO_DSP_PLUGIN:
         snprintf(s, len,
               "Plugin de DSP de udio.\n"
               "Processa udio antes de ser enviado ao \n"
               "driver."
               );
         break;
      case MENU_LABEL_AUDIO_RESAMPLER_DRIVER:
         if (settings)
            driver_hash = msg_hash_calculate(settings->audio.resampler);

         switch (driver_hash)
         {
            case MENU_LABEL_AUDIO_RESAMPLER_DRIVER_SINC:
               snprintf(s, len,
                     "Implementao Windowed SINC.");
               break;
            case MENU_LABEL_AUDIO_RESAMPLER_DRIVER_CC:
               snprintf(s, len,
                     "Implementao Convoluted Cosine.");
               break;
         }
         break;
      case MENU_LABEL_VIDEO_SHADER_PRESET:
         snprintf(s, len,
               "Carregar Predefinies de Shader. \n"
               " \n"
               " Carregar predefinies em "
#ifdef HAVE_CG
               "Cg"
#endif
#ifdef HAVE_GLSL
#ifdef HAVE_CG
               "/"
#endif
               "GLSL"
#endif
#ifdef HAVE_HLSL
#if defined(HAVE_CG) || defined(HAVE_HLSL)
               "/"
#endif
               "HLSL"
#endif
               " diretamente. \n"
               "O menu de shaders  atualizado de acordo. \n"
               " \n"
               "Se o CGP usar mtodos de interpolao complexos, \n"
               "(fator de escala diferente para X e Y) o fator \n"
               "de escala mostrado no menu poder no ser \n"
               "correto."
               );
         break;
      case MENU_LABEL_VIDEO_SHADER_SCALE_PASS:
         snprintf(s, len,
               "Escala para este passo. \n"
               " \n"
               "O fator de escala se acumula, i.e. 2x \n"
               "para o primeiro passo e 2x para o segundo \n"
               "vai lhe fornecer uma escala total de 4x. \n"
               " \n"
               "Se houver um fator de escala no ltimo \n"
               "passo, o resultado ser esticado na tela \n"
               "com o filtro especificado em 'Filtro \n"
               "Padro'. \n"
               " \n"
               "Se 'Tanto faz' estiver definido, a escala \n"
               "de 1x ou o esticamento para tela cheia sero \n"
               "usados dependendo se o primeiro foi ou no \n"
               "definido no ltimo passo."
               );
         break;
      case MENU_LABEL_VIDEO_SHADER_NUM_PASSES:
         snprintf(s, len,
               "Nmero de Shaders. \n"
               " \n"
               "O RetroArch permite que voc combine vrios \n"
               "shaders com nmero arbitrrio de passos, filtros \n"
               "de hardware e fatores de escala personalizados. \n"
               " \n"
               "Essa opo especifica o nmero de passos a usar. \n"
               "Se for definido como 0 e usada a opo Aplicar \n"
               "Alteraes de Shaders, ser usado um shader vazio. \n"
               " \n"
               "A opo Filtro Padro ir afetar o filtro \n"
               "de esticamento.");
         break;
      case MENU_LABEL_VIDEO_SHADER_PARAMETERS:
         snprintf(s, len,
               "Parmetros de Shaders. \n"
               " \n"
               "Modifica o shader em uso diretamente. No ser \n"
               "salvo no arquivo de predefinies CGP/GLSLP.");
         break;
      case MENU_LABEL_VIDEO_SHADER_PRESET_PARAMETERS:
         snprintf(s, len,
               "Parmetros de Predefinies de Shader. \n"
               " \n"
               "Modifica as predefinies de shader em uso no menu."
               );
         break;
      case MENU_LABEL_VIDEO_SHADER_PASS:
         snprintf(s, len,
               "Caminho do shader. \n"
               " \n"
               "Todos os shaders devem ser do mesmo \n"
               "tipo (i.e. CG, GLSL ou HLSL). \n"
               " \n"
               "Defina o Diretrio de Shaders para indicar \n"
               "onde o buscador comea a procurar pelos \n"
               "shaders."
               );
         break;
      case MENU_LABEL_CONFIG_SAVE_ON_EXIT:
         snprintf(s, len,
               "Salva configurao ao sair. til para\n"
               "o menu, pois as definies podem ser\n"
               "modificadas. Sobrescreve a configurao.\n"
               " \n"
               "#includes e comentrios no so \n"
               "preservados. \n"
               " \n"
               "Por design, o arquivo de configurao \n"
               " considerado imutvel, pois ele  \n"
               "provavelmente mantido pelo usurio, \n"
               "e no deve ser sobrescrito sem o \n"
               "seu conhecimento."
#if defined(RARCH_CONSOLE) || defined(RARCH_MOBILE)
               "\nPorm, isso no funciona assim nos \n"
               "consoles, pois abrir o arquivo de \n"
               "configurao manualmente no  \n"
               "realmente uma opo."
#endif
               );
         break;
      case MENU_LABEL_VIDEO_SHADER_FILTER_PASS:
         snprintf(s, len,
               "Filtro de hardware para este passo. \n"
               " \n"
               "Se 'Tanto faz' estiver definido, o 'Filtro \n"
               "Padro' ser usado."
               );
         break;
      case MENU_LABEL_AUTOSAVE_INTERVAL:
         snprintf(s, len,
               "Salva automaticamente a SRAM \n"
               "em intervalos regulares.\n"
               " \n"
               "Est desativado por padro. O intervalo  \n"
               "medido em segundos. \n"
               " \n"
               "Um valor de 0 desativa o salvamento \n"
               "automtico.");
         break;
      case MENU_LABEL_INPUT_BIND_DEVICE_TYPE:
         snprintf(s, len,
               "Tipo de Dispositivo de Entrada. \n"
               " \n"
               "Escolhe o dispositivo a usar. Isso  \n"
               "relevante para o core libretro."
               );
         break;
      case MENU_LABEL_LIBRETRO_LOG_LEVEL:
         snprintf(s, len,
               "Define o nvel de registro para os cores \n"
               "(GET_LOG_INTERFACE). \n"
               " \n"
               " Se o nvel de registro de um core \n"
               " libretro estiver abaixo de libretro_log, \n"
               " esse ser ignorado.\n"
               " \n"
               " Registros DEBUG so sempre ignorados a menos \n"
               " que o modo prolixo seja ativado (--verbose).\n"
               " \n"
               " DEBUG = 0\n"
               " INFO  = 1\n"
               " WARN  = 2\n"
               " ERROR = 3"
               );
         break;
      case MENU_LABEL_STATE_SLOT_INCREASE:
      case MENU_LABEL_STATE_SLOT_DECREASE:
         snprintf(s, len,
               "Slot de Savestates.\n"
               " \n"
               " Com o slot definido em 0, o nome do Savestate \n"
               " ser *.state (ou o que estiver definido em commandline).\n"
               "Se diferente de 0, o nome ser (caminho)(d), \n"
               "em que (d)  o nmero do slot.");
         break;
      case MENU_LABEL_SHADER_APPLY_CHANGES:
         snprintf(s, len,
               "Aplicar Alteraes de Shaders. \n"
               " \n"
               "Aps alterar definies de shaders, use-o para \n"
               "aplicar as mudanas. \n"
               " \n"
               "Mudar definies de shaders  uma operao \n"
               "computacionalmente cara e deve ser \n"
               "realizada explicitamente. \n"
               " \n"
               "Quando se aplicam shaders, as definies do menu \n"
               "de shaders so salvas em um arquivo temporrio \n"
               "(menu.cgp ou menu.glslp) e carregadas. O arquivo \n"
               "persiste ao fechar o RetroArch. Ele  salvo \n"
               "no Diretrio de Shaders."
               );
         break;
      case MENU_LABEL_INPUT_BIND_DEVICE_ID:
         snprintf(s, len,
               "Dispositivo de Entrada. \n"
               " \n"
               "Escolhe um gamepad para o usurio N. \n"
               "O nome do pad est disponvel."
               );
         break;
      case MENU_LABEL_MENU_TOGGLE:
         snprintf(s, len,
               "Alterna menu.");
         break;
      case MENU_LABEL_GRAB_MOUSE_TOGGLE:
         snprintf(s, len,
               "Alterna uso de mouse.\n"
               " \n"
               "Quando o mouse  usado, RetroArch o esconde e \n"
               "mantm o seu ponteiro dentro da janela para \n"
               "permitir que a entrada relativa do mouse \n"
               "funcione melhor.");
         break;
      case MENU_LABEL_DISK_NEXT:
         snprintf(s, len,
               "Circula por imagens de discos. Usar \n"
               "aps ejetar. \n"
               " \n"
               " Finaliza ao usar ejetar novamente.");
         break;
      case MENU_LABEL_VIDEO_FILTER:
#ifdef HAVE_FILTERS_BUILTIN
         snprintf(s, len,
               "Filtro de vdeo baseado em CPU.");
#else
         snprintf(s, len,
               "Filtro de vdeo baseado em CPU.\n"
               " \n"
               "Caminho para uma biblioteca dinmica.");
#endif
         break;
      case MENU_LABEL_AUDIO_DEVICE:
         snprintf(s, len,
               "Sobrepe-se ao dispositivo de udio padro \n"
               "que est em uso.\n"
               " dependente do driver. \n"
#ifdef HAVE_ALSA
               " \n"
               "ALSA precisa de um dispositivo PCM."
#endif
#ifdef HAVE_OSS
               " \n"
               "OSS precisa de um caminho (ex.: /dev/dsp)."
#endif
#ifdef HAVE_JACK
               " \n"
               "JACK precisa de portas (ex.: system:playback1\n"
               ",system:playback_2)."
#endif
#ifdef HAVE_RSOUND
               " \n"
               "RSound precisa de um endereo IP para \n"
               "servidor RSound."
#endif
               );
         break;
      case MENU_LABEL_DISK_EJECT_TOGGLE:
         snprintf(s, len,
               "Alterna ejeo para discos.\n"
               " \n"
               "Usado para contedos multidiscos.");
         break;
      case MENU_LABEL_ENABLE_HOTKEY:
         snprintf(s, len,
               "Ativar outras hotkeys.\n"
               " \n"
               " Se esta hotkey  usada por teclado, joybutton \n"
               "ou joyaxis, todas as outras hotkeys sero \n"
               "desativadas a menos que esta hotkey esteja sendo \n"
               "usada ao mesmo tempo. \n"
               " \n"
               "Isso  til para implementaes RETRO_KEYBOARD que \n"
               "consultam uma grande rea do teclado, cujo caminho \n"
               "deve estar livre das hotkeys.");
         break;
      case MENU_LABEL_REWIND_ENABLE:
         snprintf(s, len,
               "Ativa retrocesso.\n"
               " \n"
               "Essa opo causa uma perda de desempenho, \n"
               "por isso est desativada por padro.");
         break;
      case MENU_LABEL_LIBRETRO_DIR_PATH:
         snprintf(s, len,
               "Diretrios de Cores. \n"
               " \n"
               "Um diretrio onde so buscadas as \n"
               "implementaes de cores libretro.");
         break;
      case MENU_LABEL_VIDEO_REFRESH_RATE_AUTO:
         {
            /* Work around C89 limitations */
            char u[501];
            char t[501];
            snprintf(u, sizeof(u),
                  "Taxa de Atualizao Automtica.\n"
                  " \n"
                  "A taxa de atualizao exata de nosso monitor (Hz).\n"
                  " usada para calcular a taxa de entrada de udio \n"
                  "com a frmula: \n"
                  " \n"
                  "audio_input_rate = game input rate * display \n"
                  "refresh rate / game refresh rate\n"
                  " \n");
            snprintf(t, sizeof(t),
                  "Se a implementao no informar valores, \n"
                  "valores NTSC sero assumidos por questo de \n"
                  "compatibilidade.\n"
                  " \n"
                  "Esse valor deve ficar prximo de 60Hz para \n"
                  "evitar grande mudanas de pitch. Se o monitor \n"
                  "no rodar a 60Hz, ou algo prximo a isso, desative\n"
                  "o VSync, e deixe-o com valores padro.");
            strlcat(s, u, len);
            strlcat(s, t, len);
         }
         break;
      case MENU_LABEL_VIDEO_ROTATION:
         snprintf(s, len,
               "Fora uma certa rotao da tela. \n"
               " \n"
               "A rotao  adicionada a outras definidas\n"
               "por conjuntos de cores (veja Permitir\n"
               "Rotao de Vdeo).");
         break;
      case MENU_LABEL_VIDEO_SCALE:
         snprintf(s, len,
               "Resoluo de tela cheia.\n"
               " \n"
               "Resoluo 0 usa a resoluo \n"
               "do ambiente.\n");
         break;
      case MENU_LABEL_FASTFORWARD_RATIO:
         snprintf(s, len,
               "Taxa de Avano Rpido."
               " \n"
               "A taxa mxima na qual o contedo ser\n"
               "executado ao se usar o Avano Rpido.\n"
               " \n"
               " (Ex.: 5.0 para contedo 60 fps => 300 fps \n"
               "mximo).\n"
               " \n"
               "RetroArch entra em modo sleep para assegurar \n"
               "que a taxa mxima no ser excedida.\n"
               "No confie que esse teto tenha exatido \n"
               "perfeita.");
         break;
      case MENU_LABEL_VIDEO_MONITOR_INDEX:
         snprintf(s, len,
               "Preferncia de monitor.\n"
               " \n"
               "0 (padro) significa nenhum monitor  \n"
               "preferido, 1 e demais (1  o primeiro \n"
               "monitor), sugere ao RetroArch usar esse \n"
               "monitor em particular.");
         break;
      case MENU_LABEL_VIDEO_CROP_OVERSCAN:
         snprintf(s, len,
               "Fora o descarte de quadros overscanned. \n"
               " \n"
               "O comportamento exato dessa opo  \n"
               "especfico da implementao do core.");
         break;
      case MENU_LABEL_VIDEO_SCALE_INTEGER:
         snprintf(s, len,
               "S interpola vdeo em escalas mltiplas \n"
               "inteiras da resoluo nativa.\n"
               " \n"
               "O tamanho base depende da geometria e da \n"
               "relao de aspecto informadas pelo sistema.\n"
               " \n"
               "Se Forar Aspecto no estiver definida, X/Y \n"
               "sero escalonados em inteiros independentemente.");
         break;
      case MENU_LABEL_AUDIO_VOLUME:
         snprintf(s, len,
               "Volume de udio, em dB.\n"
               " \n"
               " 0 dB  o volume normal. Nenhum ganho aplicado.\n"
               "O ganho pode ser controlado em execuo com \n"
               "Aumentar Volume / Baixar Volume.");
         break;
      case MENU_LABEL_AUDIO_RATE_CONTROL_DELTA:
         snprintf(s, len,
               "Controle de taxa de udio.\n"
               " \n"
               "Definindo como 0 desativa o controle de taxa.\n"
               "Outros valores controlam a variao da taxa \n"
               "de udio.\n"
               " \n"
               "Define quanto de taxa de entrada pode ser \n"
               "regulada dinamicamente.\n"
               " \n"
               " Taxa de entrada  definida como: \n"
               " input rate * (1.0 +/- (rate control delta))");
         break;
      case MENU_LABEL_AUDIO_MAX_TIMING_SKEW:
         snprintf(s, len,
               "Distoro de udio mxima.\n"
               " \n"
               "Define a mxima variao da taxa de entrada.\n"
               "Voc pode querer aument-la para obter grandes\n"
               "variaes no compasso, por exemplo, ao rodar\n"
               "cores PAL em telas NTSC, ao custo de um pitch\n"
               "de udio inexato.\n"
               " \n"
               " A taxa de entrada  definida como: \n"
               " input rate * (1.0 +/- (max timing skew))");
         break;
      case MENU_LABEL_OVERLAY_NEXT:
         snprintf(s, len,
               "Alterna para o prximo overlay.\n"
               " \n"
               "Navegao circular.");
         break;
      case MENU_LABEL_LOG_VERBOSITY:
         snprintf(s, len,
               "Ativa ou desativa nvel de prolixidade \n"
               "do frontend.");
         break;
      case MENU_LABEL_VOLUME_UP:
         snprintf(s, len,
               "Aumenta o volume de udio.");
         break;
      case MENU_LABEL_VOLUME_DOWN:
         snprintf(s, len,
               "Baixa o volume de udio.");
         break;
      case MENU_LABEL_VIDEO_DISABLE_COMPOSITION:
         snprintf(s, len,
               "Desativa composition  fora.\n"
               "Vlido somente para Windows Vista/7 atualmente.");
         break;
      case MENU_LABEL_PERFCNT_ENABLE:
         snprintf(s, len,
               "Ativa ou desativa contadores de desempenho \n"
               "do frontend.");
         break;
      case MENU_LABEL_SYSTEM_DIRECTORY:
         snprintf(s, len,
               "Diretrio system. \n"
               " \n"
               "Define o diretrio 'system'.\n"
               "Cores podem consultar esse diretrio\n"
               "para carregar BIOS, configuraes\n"
               "especficas de sistemas, etc.");
         break;
      case MENU_LABEL_SAVESTATE_AUTO_SAVE:
         snprintf(s, len,
               "Automaticamente salva um Savestate ao fechar \n"
               "o RetroArch.\n"
               " \n"
               "RetroArch carregar automaticamente qualquer\n"
               "Savestate com esse caminho ao iniciar se 'Carregar\n"
               "Savestate Automaticamente' estiver ativado.");
         break;
      case MENU_LABEL_VIDEO_THREADED:
         snprintf(s, len,
               "Usa driver de vdeo em thread.\n"
               " \n"
               "Usando isso pode melhorar o desempenho ao \n"
               "possvel custo de latncia e mais engasgos \n"
               "de vdeo.");
         break;
      case MENU_LABEL_VIDEO_VSYNC:
         snprintf(s, len,
               "Sincronismo Vertical de vdeo.\n");
         break;
      case MENU_LABEL_VIDEO_HARD_SYNC:
         snprintf(s, len,
               "Tenta sincronizar CPU com GPU via \n"
               "hardware.\n"
               " \n"
               "Pode reduzir a latncia ao custo de \n"
               "desempenho.");
         break;
      case MENU_LABEL_REWIND_GRANULARITY:
         snprintf(s, len,
               "Granularidade do retrocesso.\n"
               " \n"
               " Ao retroceder um nmero definido de \n"
               "quadros, voc pode retroceder vrios \n"
               "quadros por vez, aumentando a velocidade \n"
               "de retrocesso.");
         break;
      case MENU_LABEL_SCREENSHOT:
         snprintf(s, len,
               "Tira uma foto da tela.");
         break;
      case MENU_LABEL_VIDEO_FRAME_DELAY:
         snprintf(s, len,
               "Define quantos milissegundos retardar \n"
               "aps o VSync antes de executar o core.\n"
               "\n"
               "Pode reduzir a latncia ao custo de\n"
               "um maior risco de engasgo de vdeo.\n"
               " \n"
               "O valor mximo  15.");
         break;
      case MENU_LABEL_VIDEO_HARD_SYNC_FRAMES:
         snprintf(s, len,
               "Define quantos quadros a CPU pode rodar \n"
               "adiante da GPU com a opo 'Sincronismo \n"
               "de GPU via Hardware' ativada.\n"
               " \n"
               "O valor mximo  3.\n"
               " \n"
               " 0: Sincroniza com GPU de imediato.\n"
               " 1: Sincroniza com quadro anterior.\n"
               " 2: Etc ...");
         break;
      case MENU_LABEL_VIDEO_BLACK_FRAME_INSERTION:
         snprintf(s, len,
               "Insere um quadro preto entre quadros. \n"
               " \n"
               "til para monitores de 120 Hz ao rodar \n"
               "material de 60 Hz com eliminao do efeito \n"
               "'ghosting'.\n"
               " \n"
               "A taxa de atualizao de vdeo deve ainda \n"
               "ser configurada como se fosse um monitor de \n"
               "60 Hz (divida a taxa de atualizao por 2).");
         break;
      case MENU_LABEL_RGUI_SHOW_START_SCREEN:
         snprintf(s, len,
               "Mostra a tela inicial no menu.\n"
               " definida automaticamente como falso quando\n"
               "vista pela primeira vez.\n"
               " \n"
               " atualizada na configurao apenas quando a\n"
               "opo 'Salvar Configurao ao Sair' est ativada.\n");
         break;
      case MENU_LABEL_CORE_SPECIFIC_CONFIG:
         snprintf(s, len,
               "Carrega uma configurao especfica baseada \n"
               "no core que est sendo usado.\n");
         break;
      case MENU_LABEL_VIDEO_FULLSCREEN:
         snprintf(s, len, "Alterna tela cheia.");
         break;
      case MENU_LABEL_BLOCK_SRAM_OVERWRITE:
         snprintf(s, len,
               "Previne SRAM de ser sobrescrita ao \n"
               "carregar Savestates.\n"
               " \n"
               "Pode potencialmente levar a jogos bugados.");
         break;
      case MENU_LABEL_PAUSE_NONACTIVE:
         snprintf(s, len,
               "Pausa a jogatina quando o foco da janela \n"
               " perdido.");
         break;
      case MENU_LABEL_VIDEO_GPU_SCREENSHOT:
         snprintf(s, len,
               "Captura material grfico de sada da \n"
               "GPU se estiver disponvel.");
         break;
      case MENU_LABEL_SCREENSHOT_DIRECTORY:
         snprintf(s, len,
               "Diretrio de Capturas de Tela. \n"
               " \n"
               "Diretrio para guardar as capturas de tela."
               );
         break;
      case MENU_LABEL_VIDEO_SWAP_INTERVAL:
         snprintf(s, len,
               "Intervalo de Troca de VSync.\n"
               " \n"
               "Usa um intervalo de troca personalizado. \n"
               "Use-e para reduzir  metade a taxa de \n"
               "atualizao do monitor.");
         break;
      case MENU_LABEL_SAVEFILE_DIRECTORY:
         snprintf(s, len,
               "Diretrio de Saves. \n"
               " \n"
               "Salva todos os arquivos de save (*.srm) nesse \n"
               "diretrio. Isso inclui arquivos relacionados \n"
               "como .bsv, .rt, .psrm, etc...\n"
               " \n"
               "Pode ser sobreposto por opes explcitas de\n"
               "linha de comando.");
         break;
      case MENU_LABEL_SAVESTATE_DIRECTORY:
         snprintf(s, len,
               "Diretrio de Savestates. \n"
               " \n"
               "Salva todos os Savestates (*.state) nesse \n"
               "diretrio.\n"
               " \n"
               "Pode ser sobreposto por opes explcitas de\n"
               "linha de comando.");
         break;
      case MENU_LABEL_ASSETS_DIRECTORY:
         snprintf(s, len,
               "Diretrio de Recursos (Assets). \n"
               " \n"
               " Essa localizao  consultada quando se \n"
               "tenta buscar pelo menu recursos (assets) \n"
               "carregveis, etc.");
         break;
      case MENU_LABEL_DYNAMIC_WALLPAPERS_DIRECTORY:
         snprintf(s, len,
               "Diretrio de Papis de Parede Dinmicos. \n"
               " \n"
               " O lugar para armazenar papis de parede que \n"
               "sero carregados dinamicamente pelo menu \n"
               "dependendo do contexto.");
         break;
      case MENU_LABEL_SLOWMOTION_RATIO:
         snprintf(s, len,
               "Taxa de cmera lenta."
               " \n"
               "Quando ativado, o contedo rodar em velocidade\n"
               "reduzida por um fator.");
         break;
      case MENU_LABEL_INPUT_AXIS_THRESHOLD:
         snprintf(s, len,
               "Define o limiar de eixo.\n"
               " \n"
               "O quanto deve ser torcido um eixo para\n"
               "resultar em um boto pressionado.\n"
               " Valores possveis so [0.0, 1.0].");
         break;
      case MENU_LABEL_INPUT_TURBO_PERIOD:
         snprintf(s, len, 
               "Perodo de turbo.\n"
               " \n"
               "Descreve a velocidade na qual se alternam\n"
               "os botes com turbo ativado."
               );
         break;
      case MENU_LABEL_INPUT_AUTODETECT_ENABLE:
         snprintf(s, len,
               "Ativa autodeteco de entrada.\n"
               " \n"
               "Tentar autoconfigurar joypads \n"
               "em um estilo Plug-and-Play.");
         break;
      case MENU_LABEL_CAMERA_ALLOW:
         snprintf(s, len,
               "Autorizar ou desautorizar o acesso da cmera \n"
               "pelos cores.");
         break;
      case MENU_LABEL_LOCATION_ALLOW:
         snprintf(s, len,
               "Autorizar ou desautorizar o acesso de \n"
               "servios de localizao pelos cores.");
         break;
      case MENU_LABEL_TURBO:
         snprintf(s, len,
               "Ativar turbo.\n"
               " \n"
               "Segurando o turbo enquanto se pressiona outro \n"
               "boto permitir que o boto entre em modo \n"
               "turbo em que o seu estado ser modulado com \n"
               "um sinal peridico. \n"
               " \n"
               "A modulao pra quando o prprio boto \n"
               "(no  o boto de turbo)  solto.");
         break;
      case MENU_LABEL_OSK_ENABLE:
         snprintf(s, len,
               "Ativar/desativar teclado na tela.");
         break;
      case MENU_LABEL_AUDIO_MUTE:
         snprintf(s, len,
               "Ligar/desligar udio.");
         break;
      case MENU_LABEL_REWIND:
         snprintf(s, len,
               "Segure o boto para retroceder.\n"
               " \n"
               "Retrocesso deve estar ativado.");
         break;
      case MENU_LABEL_EXIT_EMULATOR:
         snprintf(s, len,
               "Tecla para sair corretamente do RetroArch."
#if !defined(RARCH_MOBILE) && !defined(RARCH_CONSOLE)
               "\nFechando-o de outra forma mais agressiva \n"
               "(SIGKILL, etc) sair sem salvar RAM, etc.\n"
               "Em sistemas baseados em Unix,\n"
               "SIGINT/SIGTERM permite um\n"
               "fechamento correto."
#endif
               );
         break;
      case MENU_LABEL_LOAD_STATE:
         snprintf(s, len,
               "Carrega Savestates.");
         break;
      case MENU_LABEL_SAVE_STATE:
         snprintf(s, len,
               "Salva Savestates.");
         break;
      case MENU_LABEL_NETPLAY_FLIP_PLAYERS:
         snprintf(s, len,
               "Netplay inverte usurios.");
         break;
      case MENU_LABEL_CHEAT_INDEX_PLUS:
         snprintf(s, len,
               "Incrementa o ndice de cheats.\n");
         break;
      case MENU_LABEL_CHEAT_INDEX_MINUS:
         snprintf(s, len,
               "Decrementa o ndice de cheats.\n");
         break;
      case MENU_LABEL_SHADER_PREV:
         snprintf(s, len,
               "Aplica o shader anterior no diretrio.");
         break;
      case MENU_LABEL_SHADER_NEXT:
         snprintf(s, len,
               "Aplica o prximo shader no diretrio.");
         break;
      case MENU_LABEL_RESET:
         snprintf(s, len,
               "Reinicia o contedo.\n");
         break;
      case MENU_LABEL_PAUSE_TOGGLE:
         snprintf(s, len,
               "Alterna estado de pausado e no pausado.");
         break;
      case MENU_LABEL_CHEAT_TOGGLE:
         snprintf(s, len,
               "Alterna ndice de cheats.\n");
         break;
      case MENU_LABEL_HOLD_FAST_FORWARD:
         snprintf(s, len,
               "Segure para avano rpido. Soltando o boto \n"
               "desativa o avano rpido.");
         break;
      case MENU_LABEL_SLOWMOTION:
         snprintf(s, len,
               "Segure para cmera lenta.");
         break;
      case MENU_LABEL_FRAME_ADVANCE:
         snprintf(s, len,
               "O quadro avana quando o contedo est pausado.");
         break;
      case MENU_LABEL_MOVIE_RECORD_TOGGLE:
         snprintf(s, len,
               "Alterna entre estar gravando ou no.");
         break;
      case MENU_LABEL_L_X_PLUS:
      case MENU_LABEL_L_X_MINUS:
      case MENU_LABEL_L_Y_PLUS:
      case MENU_LABEL_L_Y_MINUS:
      case MENU_LABEL_R_X_PLUS:
      case MENU_LABEL_R_X_MINUS:
      case MENU_LABEL_R_Y_PLUS:
      case MENU_LABEL_R_Y_MINUS:
         snprintf(s, len,
               "Eixo para o analgico (esquema DualShock).\n"
               " \n"
               "Associa normalmente, porm, se um analgico real \n"
               " associado, pode ser lido como um analgico\n"
               "verdadeiro. \n"
               " \n"
               "Eixo positivo X  para direita. \n"
               "Eixo positivo Y  para baixo.");
         break;
      default:
         if (s[0] == '\0')
            strlcpy(s, msg_hash_to_str(MENU_ENUM_LABEL_VALUE_NO_INFORMATION_AVAILABLE), len);
         return -1;
   }

   return 0;
}

const char *msg_hash_to_str_pt(enum msg_hash_enums msg)
{
   switch (msg)
   {
      case MSG_UNKNOWN:
         return "Desconhecido";
      case MSG_RECEIVED:
         return "recebido";
      case MSG_UNRECOGNIZED_COMMAND:
         return "Comando desconhecido";
      case MSG_SENDING_COMMAND:
         return "Enviando comando";
      case MSG_GOT_INVALID_DISK_INDEX:
         return "ndice de disco invlido.";
      case MSG_FAILED_TO_REMOVE_DISK_FROM_TRAY:
         return "Falha ao remover disco da bandeja.";
      case MSG_REMOVED_DISK_FROM_TRAY:
         return "Disco removido da bandeja.";
      case MSG_VIRTUAL_DISK_TRAY:
         return "bandeja de disco virtual.";
      case MSG_FAILED_TO:
         return "Falha ao";
      case MSG_TO:
         return "para";
      case MSG_SAVING_RAM_TYPE:
         return "Salvando tipo de RAM";
      case MSG_SAVING_STATE:
         return "Salvando estado";
      case MSG_LOADING_STATE:
         return "Carregando estado";
      case MSG_FAILED_TO_LOAD_MOVIE_FILE:
         return "Falha ao carregar vdeo";
      case MSG_FAILED_TO_LOAD_CONTENT:
         return "Falha ao carregar contedo";
      case MSG_COULD_NOT_READ_CONTENT_FILE:
         return "Incapaz de ler arquivo de contedo";
      case MSG_GRAB_MOUSE_STATE:
         return "Obter estado do mouse";
      case MSG_PAUSED:
         return "Pausado.";
      case MSG_UNPAUSED:
         return "Despausado.";
      case MSG_FAILED_TO_LOAD_OVERLAY:
         return "Falha ao carregar overlay.";
      case MSG_FAILED_TO_UNMUTE_AUDIO:
         return "Falha ao desativar mudo.";
      case MSG_AUDIO_MUTED:
         return "udio mudo.";
      case MSG_AUDIO_UNMUTED:
         return "udio normal.";
      case MSG_RESET:
         return "Reiniciar";
      case MSG_FAILED_TO_LOAD_STATE:
         return "Falha ao carregar estado de";
      case MSG_FAILED_TO_SAVE_STATE_TO:
         return "Falha ao salvar estado em";
      case MSG_FAILED_TO_SAVE_SRAM:
         return "Falha ao salvar SRAM";
      case MSG_STATE_SIZE:
         return "Tamanho do estado";
      case MSG_BLOCKING_SRAM_OVERWRITE:
         return "Bloqueando Sobrescrio de SRAM";
      case MSG_CORE_DOES_NOT_SUPPORT_SAVESTATES:
         return "O core no suporta savestates.";
      case MSG_SAVED_STATE_TO_SLOT:
         return "Estado salvo no slot";
      case MSG_SAVED_SUCCESSFULLY_TO:
         return "Salvo com sucesso em";
      case MSG_BYTES:
         return "bytes";
      case MSG_CONFIG_DIRECTORY_NOT_SET:
         return "Diretrio de configuraes no definido. Incapaz de salvar.";
      case MSG_SKIPPING_SRAM_LOAD:
         return "Ignorando carregamento de SRAM.";
      case MSG_APPENDED_DISK:
         return "Disco anexado";
      case MSG_STARTING_MOVIE_PLAYBACK:
         return "Iniciando reproduo de vdeo.";
      case MSG_FAILED_TO_REMOVE_TEMPORARY_FILE:
         return "Falha ao remover arquivo temporrio";
      case MSG_REMOVING_TEMPORARY_CONTENT_FILE:
         return "Removendo contedo temporrio";
      case MSG_LOADED_STATE_FROM_SLOT:
         return "Estado carregado do slot";
      case MSG_COULD_NOT_PROCESS_ZIP_FILE:
         return "Incapaz de processar arquivo ZIP.";
      case MSG_SCANNING_OF_DIRECTORY_FINISHED:
         return "Exame de diretrio concludo";
      case MSG_SCANNING:
         return "Examinando";
      case MSG_REDIRECTING_CHEATFILE_TO:
         return "Redirecionando cheat para";
      case MSG_REDIRECTING_SAVEFILE_TO:
         return "Redirecionando save para";
      case MSG_REDIRECTING_SAVESTATE_TO:
         return "Redirecionando savestate para";
      case MSG_SHADER:
         return "Shader";
      case MSG_APPLYING_SHADER:
         return "Aplicando shader";
      case MSG_FAILED_TO_APPLY_SHADER:
         return "Falha ao aplicar shader.";
      case MSG_STARTING_MOVIE_RECORD_TO:
         return "Iniciando gravao de vdeo em";
      case MSG_FAILED_TO_START_MOVIE_RECORD:
         return "Falha ao iniciar gravao de vdeo.";
      case MSG_STATE_SLOT:
         return "Slot de estado";
      case MSG_RESTARTING_RECORDING_DUE_TO_DRIVER_REINIT:
         return "Reiniciando gravao devido a reincio de driver.";
      case MSG_SLOW_MOTION:
         return "Cmera lenta.";
      case MSG_SLOW_MOTION_REWIND:
         return "Retrocesso em cmera lenta.";
      case MSG_REWINDING:
         return "Retrocedendo.";
      case MSG_REWIND_REACHED_END:
         return "Final do buffer de retrocesso atingido.";
      case MSG_TASK_FAILED:
         return "Falhou";
      case MSG_DOWNLOADING:
         return "Baixando";
      case MSG_EXTRACTING:
         return "Extraindo";
      case MENU_ENUM_LABEL_VALUE_START_NET_RETROPAD:
         return "Iniciar RetroPad Remoto";
      case MENU_ENUM_LABEL_VALUE_SCAN_THIS_DIRECTORY:
         return "<Escanear este Diretrio>";
      case MENU_ENUM_LABEL_VALUE_SCAN_FILE:
         return "Escanear Arquivo";
      case MENU_ENUM_LABEL_VALUE_SCAN_DIRECTORY:
         return "Escanear Diretrio";
      case MENU_ENUM_LABEL_VALUE_START_CORE:
         return "Iniciar Core";
      case MENU_ENUM_LABEL_VALUE_INFORMATION_LIST:
         return "Informao";
      case MENU_ENUM_LABEL_VALUE_USE_BUILTIN_PLAYER:
         return "Usar Player Interno"; /* TODO/FIXME */
      case MENU_ENUM_LABEL_VALUE_CONTENT_SETTINGS:
         return "Menu Rpido";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_CRC32:
         return "CRC32";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_MD5:
         return "MD5";
      case MENU_ENUM_LABEL_VALUE_LOAD_CONTENT_LIST:
         return "Carregar Contedo";
      case MENU_ENUM_LABEL_VALUE_LOAD_ARCHIVE:
         return "Carregar Arquivo";
      case MENU_ENUM_LABEL_VALUE_OPEN_ARCHIVE:
         return "Abrir Arquivo";
      case MENU_ENUM_LABEL_VALUE_ASK_ARCHIVE:
         return "Ask";
      case MENU_ENUM_LABEL_VALUE_PRIVACY_SETTINGS:
         return "Configuraes de Privacidade";
      case MENU_ENUM_LABEL_VALUE_HORIZONTAL_MENU:
         return "Horizontal Menu"; /* FIXME - don't edit this yet. */
      case MENU_ENUM_LABEL_VALUE_NO_SETTINGS_FOUND:
         return "Nenhuma definio encontrada.";
      case MENU_ENUM_LABEL_VALUE_NO_PERFORMANCE_COUNTERS:
         return "Nenhum medidor de desempenho.";
      case MENU_ENUM_LABEL_VALUE_DRIVER_SETTINGS:
         return "Drivers";
      case MENU_ENUM_LABEL_VALUE_CONFIGURATION_SETTINGS:
         return "Configuraes";
      case MENU_ENUM_LABEL_VALUE_CORE_SETTINGS:
         return "Core";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SETTINGS:
         return "Vdeo";
      case MENU_ENUM_LABEL_VALUE_LOGGING_SETTINGS:
         return "Registro de Dados";
      case MENU_ENUM_LABEL_VALUE_SAVING_SETTINGS:
         return "Saves";
      case MENU_ENUM_LABEL_VALUE_REWIND_SETTINGS:
         return "Retrocesso";
      case MENU_ENUM_LABEL_VALUE_SHADER:
         return "Shader";
      case MENU_ENUM_LABEL_VALUE_CHEAT:
         return "Cheat";
      case MENU_ENUM_LABEL_VALUE_USER:
         return "Usurio";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_BGM_ENABLE:
         return "Ativar Sistema BGM";
      case MENU_ENUM_LABEL_VALUE_RETROPAD:
         return "RetroPad";
      case MENU_ENUM_LABEL_VALUE_RETROKEYBOARD:
         return "RetroTeclado";
      case MENU_ENUM_LABEL_VALUE_AUDIO_BLOCK_FRAMES:
         return "Quadros de Blocos de udio";
      case MENU_ENUM_LABEL_VALUE_INPUT_DESCRIPTOR_LABEL_SHOW:
         return "Mostrar Rtulos de Entradas de Core";
      case MENU_ENUM_LABEL_VALUE_INPUT_DESCRIPTOR_HIDE_UNBOUND:
         return "Esconder Descritores de Entradas sem Uso";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FONT_ENABLE:
         return "Mostrar Mensagem de Tela";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FONT_PATH:
         return "Fonte da Mensagem de Tela";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FONT_SIZE:
         return "Tamanho da Mensagem de Tela";
      case MENU_ENUM_LABEL_VALUE_VIDEO_MESSAGE_POS_X:
         return "Posio X da Mensagem de Tela";
      case MENU_ENUM_LABEL_VALUE_VIDEO_MESSAGE_POS_Y:
         return "Posio Y da Mensagem de Tela";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SOFT_FILTER:
         return "Ativar Filtro de Suavizao";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FILTER_FLICKER:
         return "Filtro de Cintilao";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_CONTENT:
         return "<Diretrio de Contedo>";
      case MENU_ENUM_LABEL_VALUE_UNKNOWN:
         return "Desconhecido";
      case MENU_ENUM_LABEL_VALUE_DONT_CARE:
         return "Tanto faz";
      case MENU_ENUM_LABEL_VALUE_LINEAR:
         return "Linear";
      case MENU_ENUM_LABEL_VALUE_NEAREST:
         return "Nearest";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_DEFAULT:
         return "<Padro>";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_NONE:
         return "<Nenhum>";
      case MENU_ENUM_LABEL_VALUE_INPUT_REMAPPING_DIRECTORY:
         return "Remapeamentos de Controladores";
      case MENU_ENUM_LABEL_VALUE_JOYPAD_AUTOCONFIG_DIR:
         return "Autoconfiguraes de Dispositivos de Entrada";
      case MENU_ENUM_LABEL_VALUE_RECORDING_CONFIG_DIRECTORY:
         return "Configuraes de Gravaes";
      case MENU_ENUM_LABEL_VALUE_RECORDING_OUTPUT_DIRECTORY:
         return "Gravaes";
      case MENU_ENUM_LABEL_VALUE_SCREENSHOT_DIRECTORY:
         return "Capturas de Telas";
      case MENU_ENUM_LABEL_VALUE_PLAYLIST_DIRECTORY:
         return "Histricos";
      case MENU_ENUM_LABEL_VALUE_SAVEFILE_DIRECTORY:
         return "Saves";
      case MENU_ENUM_LABEL_VALUE_SAVESTATE_DIRECTORY:
         return "Savestates";
      case MENU_ENUM_LABEL_VALUE_STDIN_CMD_ENABLE:
         return "Comandos stdin";
      case MENU_ENUM_LABEL_VALUE_VIDEO_DRIVER:
         return "Driver de Vdeo";
      case MENU_ENUM_LABEL_VALUE_RECORD_ENABLE:
         return "Ativar Gravao";
      case MENU_ENUM_LABEL_VALUE_VIDEO_GPU_RECORD:
         return "Ativar Gravao por GPU";
      case MENU_ENUM_LABEL_VALUE_RECORD_PATH: /* FIXME/UPDATE */
         return "Caminho da Gravao";
      case MENU_ENUM_LABEL_VALUE_RECORD_USE_OUTPUT_DIRECTORY:
         return "Diretrio de Sada";
      case MENU_ENUM_LABEL_VALUE_RECORD_CONFIG:
         return "Configuraes de Gravao";
      case MENU_ENUM_LABEL_VALUE_VIDEO_POST_FILTER_RECORD:
         return "Ativar Filtro Ps-Gravao";
      case MENU_ENUM_LABEL_VALUE_CORE_ASSETS_DIRECTORY:
         return "Recursos (Assets) de Cores"; /* FIXME/UPDATE */
      case MENU_ENUM_LABEL_VALUE_ASSETS_DIRECTORY:
         return "Recursos (Assets)";
      case MENU_ENUM_LABEL_VALUE_DYNAMIC_WALLPAPERS_DIRECTORY:
         return "Papis de Parede Dinmicos";
      case MENU_ENUM_LABEL_VALUE_RGUI_BROWSER_DIRECTORY:
         return "Navegao";
      case MENU_ENUM_LABEL_VALUE_RGUI_CONFIG_DIRECTORY:
         return "Configuraes";
      case MENU_ENUM_LABEL_VALUE_LIBRETRO_INFO_PATH:
         return "Informaes de Cores";
      case MENU_ENUM_LABEL_VALUE_LIBRETRO_DIR_PATH:
         return "Cores";
      case MENU_ENUM_LABEL_VALUE_CURSOR_DIRECTORY:
         return "Cursores";
      case MENU_ENUM_LABEL_VALUE_CONTENT_DATABASE_DIRECTORY:
         return "Databases de Contedo";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_DIRECTORY:
         return "System/BIOS";
      case MENU_ENUM_LABEL_VALUE_CHEAT_DATABASE_PATH:
         return "Cheats";
      case MENU_ENUM_LABEL_VALUE_CACHE_DIRECTORY: /* UPDATE/FIXME */
         return "Descompactao";
      case MENU_ENUM_LABEL_VALUE_AUDIO_FILTER_DIR:
         return "Filtros de udio";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_DIR:
         return "Shaders";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FILTER_DIR:
         return "Filtros de Vdeo";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_DIRECTORY:
         return "Overlays";
      case MENU_ENUM_LABEL_VALUE_OSK_OVERLAY_DIRECTORY:
         return "Overlays de Teclado";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_CLIENT_SWAP_INPUT:
         return "Trocar Entradas de Netplay";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_SPECTATOR_MODE_ENABLE:
         return "Ativar Espectador de Netplay";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_IP_ADDRESS:
         return "Endereo IP";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_TCP_UDP_PORT:
         return "Portas TCP/UDP de Netplay";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_ENABLE:
         return "Ativar Netplay";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_DELAY_FRAMES:
         return "Quadros de Retardo de Netplay";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_MODE:
         return "Ativar Cliente de Netplay";
      case MENU_ENUM_LABEL_VALUE_RGUI_SHOW_START_SCREEN:
         return "Mostrar Tela de Incio";
      case MENU_ENUM_LABEL_VALUE_TITLE_COLOR:
         return "Cor do Menu Ttulo";
      case MENU_ENUM_LABEL_VALUE_ENTRY_HOVER_COLOR:
         return "Cor de Realce do Menu Inicial";
      case MENU_ENUM_LABEL_VALUE_TIMEDATE_ENABLE:
         return "Mostrar Hora / Data";
      case MENU_ENUM_LABEL_VALUE_THREADED_DATA_RUNLOOP_ENABLE:
         return "Ativar Runloop de Threads de Dados";
      case MENU_ENUM_LABEL_VALUE_ENTRY_NORMAL_COLOR:
         return "Cor do Menu Inicial";
      case MENU_ENUM_LABEL_VALUE_SHOW_ADVANCED_SETTINGS:
         return "Mostrar Configuraes Avanadas";
      case MENU_ENUM_LABEL_VALUE_MOUSE_ENABLE:
         return "Suporte a Mouse";
      case MENU_ENUM_LABEL_VALUE_POINTER_ENABLE:
         return "Suporte a Touch";
      case MENU_ENUM_LABEL_VALUE_CORE_ENABLE:
         return "Mostrar Nome dos Cores";
      case MENU_ENUM_LABEL_VALUE_DPI_OVERRIDE_ENABLE:
         return "Ativar Sobreposio de DPI";
      case MENU_ENUM_LABEL_VALUE_DPI_OVERRIDE_VALUE:
         return "Sobreposio de DPI";
      case MENU_ENUM_LABEL_VALUE_SUSPEND_SCREENSAVER_ENABLE:
         return "Suspender Proteo de Tela";
      case MENU_ENUM_LABEL_VALUE_VIDEO_DISABLE_COMPOSITION:
         return "Desativar Desktop Composition";
      case MENU_ENUM_LABEL_VALUE_PAUSE_NONACTIVE:
         return "No Rodar em Background";
      case MENU_ENUM_LABEL_VALUE_UI_COMPANION_START_ON_BOOT:
         return "Ativar UI Companion ao Iniciar";
      case MENU_ENUM_LABEL_VALUE_UI_MENUBAR_ENABLE:
         return "Barra de Menu (Dica)";
      case MENU_ENUM_LABEL_VALUE_ARCHIVE_MODE:
         return "Ao para Arquivos Compactados";
      case MENU_ENUM_LABEL_VALUE_NETWORK_CMD_ENABLE:
         return "Comandos de Rede";
      case MENU_ENUM_LABEL_VALUE_NETWORK_CMD_PORT:
         return "Porta para Comandos de Rede";
      case MENU_ENUM_LABEL_VALUE_HISTORY_LIST_ENABLE:
         return "Ativar Lista de Histrico";
      case MENU_ENUM_LABEL_VALUE_CONTENT_HISTORY_SIZE:
         return "Tamanho da Lista de Histrico";
      case MENU_ENUM_LABEL_VALUE_VIDEO_REFRESH_RATE_AUTO:
         return "Taxa de Atualizao de Quadros Estimada";
      case MENU_ENUM_LABEL_VALUE_DUMMY_ON_CORE_SHUTDOWN:
         return "Desligar Core Dummy On";
      case MENU_ENUM_LABEL_VALUE_CORE_SET_SUPPORTS_NO_CONTENT_ENABLE: /* TODO/FIXME */
         return "No Iniciar Cores Automaticamente";
      case MENU_ENUM_LABEL_VALUE_FRAME_THROTTLE_ENABLE:
         return "Limitar Velocidade Mxima de Execuo";
      case MENU_ENUM_LABEL_VALUE_FASTFORWARD_RATIO:
         return "Velocidade Mxima de Execuo";
      case MENU_ENUM_LABEL_VALUE_AUTO_REMAPS_ENABLE:
         return "Carregar Automaticamente Arquivos Remapeados";
      case MENU_ENUM_LABEL_VALUE_SLOWMOTION_RATIO:
         return "Taxa de Cmera Lenta";
      case MENU_ENUM_LABEL_VALUE_CORE_SPECIFIC_CONFIG:
         return "Configurao por Core";
      case MENU_ENUM_LABEL_VALUE_AUTO_OVERRIDES_ENABLE:
         return "Carregar Automaticamente Arquivos de Sobreposio";
      case MENU_ENUM_LABEL_VALUE_CONFIG_SAVE_ON_EXIT:
         return "Salvar Configurao ao Sair";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SMOOTH:
         return "Filtragem Bilinear por Hardware";
      case MENU_ENUM_LABEL_VALUE_VIDEO_GAMMA:
         return "Gamma de Vdeo";
      case MENU_ENUM_LABEL_VALUE_VIDEO_ALLOW_ROTATE:
         return "Permitir Rotao";
      case MENU_ENUM_LABEL_VALUE_VIDEO_HARD_SYNC:
         return "Sincronizar GPU com CPU";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SWAP_INTERVAL:
         return "Intervalo de Permuta do Sincronismo Vertical";
      case MENU_ENUM_LABEL_VALUE_VIDEO_VSYNC:
         return "Sincronismo Vertical";
      case MENU_ENUM_LABEL_VALUE_VIDEO_THREADED:
         return "Vdeo em Threads";
      case MENU_ENUM_LABEL_VALUE_VIDEO_ROTATION:
         return "Rotao";
      case MENU_ENUM_LABEL_VALUE_VIDEO_GPU_SCREENSHOT:
         return "Ativar Captura de Tela via GPU";
      case MENU_ENUM_LABEL_VALUE_VIDEO_CROP_OVERSCAN:
         return "Descartar Overscan (Recarregue)";
      case MENU_ENUM_LABEL_VALUE_VIDEO_ASPECT_RATIO_INDEX:
         return "ndice de Relaes de Aspecto";
      case MENU_ENUM_LABEL_VALUE_VIDEO_ASPECT_RATIO_AUTO:
         return "Relao de Aspecto Automtica";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FORCE_ASPECT:
         return "Forar Relao de Aspecto";
      case MENU_ENUM_LABEL_VALUE_VIDEO_REFRESH_RATE:
         return "Taxa de Atualizao de Tela";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FORCE_SRGB_DISABLE:
         return "Forcar Desativao de sRGB FBO";
      case MENU_ENUM_LABEL_VALUE_VIDEO_WINDOWED_FULLSCREEN:
         return "Modo Tela Cheia em Janela";
      case MENU_ENUM_LABEL_VALUE_PAL60_ENABLE:
         return "Usar Modo PAL60";
      case MENU_ENUM_LABEL_VALUE_VIDEO_VFILTER:
         return "Eliminar Cintilao";
      case MENU_ENUM_LABEL_VALUE_VIDEO_VI_WIDTH:
         return "Definir Largura de Tela VI";
      case MENU_ENUM_LABEL_VALUE_VIDEO_BLACK_FRAME_INSERTION:
         return "Insero de Quadro Negro";
      case MENU_ENUM_LABEL_VALUE_VIDEO_HARD_SYNC_FRAMES:
         return "Quadros de Sincronizao entre GPU e CPU";
      case MENU_ENUM_LABEL_VALUE_SORT_SAVEFILES_ENABLE:
         return "Ordenar Saves em Pastas";
      case MENU_ENUM_LABEL_VALUE_SORT_SAVESTATES_ENABLE:
         return "Ordenar Savestates em Pastas";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FULLSCREEN:
         return "Usar Modo Tela Cheia";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SCALE:
         return "Variar Escala em Janela";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SCALE_INTEGER:
         return "Escala em Degraus Inteiros";
      case MENU_ENUM_LABEL_VALUE_PERFCNT_ENABLE:
         return "Contadores de Desempenho";
      case MENU_ENUM_LABEL_VALUE_LIBRETRO_LOG_LEVEL:
         return "Nvel de Registro de Core";
      case MENU_ENUM_LABEL_VALUE_LOG_VERBOSITY:
         return "Detalhamento de Registro";
      case MENU_ENUM_LABEL_VALUE_SAVESTATE_AUTO_LOAD:
         return "Carregar Savestate Automaticamente";
      case MENU_ENUM_LABEL_VALUE_SAVESTATE_AUTO_INDEX:
         return "ndice Automtico de Savestates";
      case MENU_ENUM_LABEL_VALUE_SAVESTATE_AUTO_SAVE:
         return "Savestate Automtico";
      case MENU_ENUM_LABEL_VALUE_AUTOSAVE_INTERVAL:
         return "Intervalo de Gravao Automtica de SaveRAM";
      case MENU_ENUM_LABEL_VALUE_BLOCK_SRAM_OVERWRITE:
         return "No Sobrescrever SaveRAM ao Carregar Savestate";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHARED_CONTEXT:
         return "Ativar Contexto Compartilhado de Hardware";
      case MENU_ENUM_LABEL_VALUE_RESTART_RETROARCH:
         return "Reiniciar RetroArch";
      case MENU_ENUM_LABEL_VALUE_NETPLAY_NICKNAME:
         return "Nome de Usurio";
      case MENU_ENUM_LABEL_VALUE_USER_LANGUAGE:
         return "Idioma";
      case MENU_ENUM_LABEL_VALUE_CAMERA_ALLOW:
         return "Autorizar Cmera";
      case MENU_ENUM_LABEL_VALUE_LOCATION_ALLOW:
         return "Autorizar Localizao";
      case MENU_ENUM_LABEL_VALUE_PAUSE_LIBRETRO:
         return "Pausar Quando o Menu for Ativado";
      case MENU_ENUM_LABEL_VALUE_INPUT_OSK_OVERLAY_ENABLE:
         return "Mostrar Overlay de Teclado";
      case MENU_ENUM_LABEL_VALUE_INPUT_OVERLAY_ENABLE:
         return "Mostrar Overlay";
      case MENU_ENUM_LABEL_VALUE_VIDEO_MONITOR_INDEX:
         return "ndice de Monitores";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FRAME_DELAY:
         return "Retardo de Quadro";
      case MENU_ENUM_LABEL_VALUE_INPUT_DUTY_CYCLE:
         return "Ciclo de Trabalho";
      case MENU_ENUM_LABEL_VALUE_INPUT_TURBO_PERIOD:
         return "Perodo de Turbo";
      case MENU_ENUM_LABEL_VALUE_INPUT_AXIS_THRESHOLD:
         return "Limiar de Eixo do Controlador";
      case MENU_ENUM_LABEL_VALUE_INPUT_REMAP_BINDS_ENABLE:
         return "Ativar Remapeamentos";
      case MENU_ENUM_LABEL_VALUE_INPUT_MAX_USERS:
         return "Usurios Mximos";
      case MENU_ENUM_LABEL_VALUE_INPUT_AUTODETECT_ENABLE:
         return "Ativar Autoconfigurao";
      case MENU_ENUM_LABEL_VALUE_AUDIO_OUTPUT_RATE:
         return "Taxa de Amostragem de udio (KHz)";
      case MENU_ENUM_LABEL_VALUE_AUDIO_MAX_TIMING_SKEW:
         return "Distoro Mxima de Sincronizao de udio";
      case MENU_ENUM_LABEL_VALUE_CHEAT_NUM_PASSES:
         return "Cdigos de Cheat";
      case MENU_ENUM_LABEL_VALUE_REMAP_FILE_SAVE_CORE:
         return "Salvar Remapeamento de Core";
      case MENU_ENUM_LABEL_VALUE_REMAP_FILE_SAVE_GAME:
         return "Salvar Remapeamento de Jogo";
      case MENU_ENUM_LABEL_VALUE_CHEAT_APPLY_CHANGES:
         return "Aplicar Alteraes de Cheats";
      case MENU_ENUM_LABEL_VALUE_SHADER_APPLY_CHANGES:
         return "Aplicar Alteraes de Shaders";
      case MENU_ENUM_LABEL_VALUE_REWIND_ENABLE:
         return "Ativar Retrocesso";
      case MENU_ENUM_LABEL_VALUE_CONTENT_COLLECTION_LIST:
         return "Selecionar da Coleo";
      case MENU_ENUM_LABEL_VALUE_DETECT_CORE_LIST:
         return "Selecionar Arquivo e Detectar Core";
      case MENU_ENUM_LABEL_VALUE_LOAD_CONTENT_HISTORY:
         return "Selecionar do Histrico";
      case MENU_ENUM_LABEL_VALUE_AUDIO_ENABLE:
         return "Ativar udio";
      case MENU_ENUM_LABEL_VALUE_FPS_SHOW:
         return "Mostrar Taxa de Quadros";
      case MENU_ENUM_LABEL_VALUE_AUDIO_MUTE:
         return "Silenciar udio";
      case MENU_ENUM_LABEL_VALUE_AUDIO_VOLUME:
         return "Volume de udio (dB)";
      case MENU_ENUM_LABEL_VALUE_AUDIO_SYNC:
         return "Ativar Sincronismo de udio";
      case MENU_ENUM_LABEL_VALUE_AUDIO_RATE_CONTROL_DELTA:
         return "Variao Mxima de Taxa de udio";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_NUM_PASSES:
         return "Nmero de Shaders";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_SHA1:
         return "SHA1";
      case MENU_ENUM_LABEL_VALUE_CONFIGURATIONS:
         return "Carregar Configurao";
      case MENU_ENUM_LABEL_VALUE_REWIND_GRANULARITY:
         return "Granularidade do Retrocesso";
      case MENU_ENUM_LABEL_VALUE_REMAP_FILE_LOAD:
         return "Carregar Remapeamento";
      case MENU_ENUM_LABEL_VALUE_CUSTOM_RATIO:
         return "Relao de Aspecto Personalizada";
      case MENU_ENUM_LABEL_VALUE_USE_THIS_DIRECTORY:
         return "<Usar este diretrio>";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_START_CONTENT:
         return "Iniciar Contedo";
      case MENU_ENUM_LABEL_VALUE_DISK_OPTIONS:    /* UPDATE/FIXME */
         return "Opes de Disco do Core";
      case MENU_ENUM_LABEL_VALUE_CORE_OPTIONS:
         return "Opes";
      case MENU_ENUM_LABEL_VALUE_CORE_CHEAT_OPTIONS:
         return "Cheats";
      case MENU_ENUM_LABEL_VALUE_CHEAT_FILE_LOAD:
         return "Carregar Cheat";
      case MENU_ENUM_LABEL_VALUE_CHEAT_FILE_SAVE_AS:
         return "Salvar Cheat Como";
      case MENU_ENUM_LABEL_VALUE_CORE_COUNTERS:
         return "Contadores de Core";
      case MENU_ENUM_LABEL_VALUE_TAKE_SCREENSHOT:
         return "Capturar Tela";
      case MENU_ENUM_LABEL_VALUE_RESUME:
         return "Retomar";
      case MENU_ENUM_LABEL_VALUE_DISK_INDEX:
         return "ndice de Discos";
      case MENU_ENUM_LABEL_VALUE_FRONTEND_COUNTERS:
         return "Contadores do Frontend";
      case MENU_ENUM_LABEL_VALUE_DISK_IMAGE_APPEND:
         return "Anexar Imagem de Disco";
      case MENU_ENUM_LABEL_VALUE_DISK_CYCLE_TRAY_STATUS:
         return "Estado do Drive de Disco";
      case MENU_ENUM_LABEL_VALUE_NO_PLAYLIST_ENTRIES_AVAILABLE:
         return "Histrico vazio.";
      case MENU_ENUM_LABEL_VALUE_NO_CORE_INFORMATION_AVAILABLE:
         return "Nenhuma informao de core disponvel.";
      case MENU_ENUM_LABEL_VALUE_NO_CORE_OPTIONS_AVAILABLE:
         return "Nenhuma opo de core disponvel.";
      case MENU_ENUM_LABEL_VALUE_NO_CORES_AVAILABLE:
         return "Nenhum core disponvel.";
      case MENU_ENUM_LABEL_VALUE_NO_CORE:
         return "Nenhum Core";
      case MENU_ENUM_LABEL_VALUE_DATABASE_MANAGER:
         return "Gerenciador de Databases";
      case MENU_ENUM_LABEL_VALUE_CURSOR_MANAGER:
         return "Gerenciador de Cursores";
      case MENU_ENUM_LABEL_VALUE_MAIN_MENU: /* TODO/FIXME - translate */
         return "Main Menu";
      case MENU_ENUM_LABEL_VALUE_SETTINGS:
         return "Definies";
      case MENU_ENUM_LABEL_VALUE_QUIT_RETROARCH:
         return "Sair do RetroArch";
      case MENU_ENUM_LABEL_VALUE_SHUTDOWN:
         return "Desligar";
      case MENU_ENUM_LABEL_VALUE_REBOOT:
         return "Reiniciar";
      case MENU_ENUM_LABEL_VALUE_HELP:
         return "Ajuda";
      case MENU_ENUM_LABEL_VALUE_SAVE_NEW_CONFIG:
         return "Salvar Nova Configurao";
      case MENU_ENUM_LABEL_VALUE_RESTART_CONTENT:
         return "Reiniciar";
      case MENU_ENUM_LABEL_VALUE_CORE_UPDATER_LIST:
         return "Atualizao de Cores";
      case MENU_ENUM_LABEL_VALUE_CORE_UPDATER_BUILDBOT_URL:
         return "URL Buildbot de Cores";
      case MENU_ENUM_LABEL_VALUE_BUILDBOT_ASSETS_URL:
         return "URL Buildbot de Recursos (Assets)";
      case MENU_ENUM_LABEL_VALUE_NAVIGATION_WRAPAROUND:
         return "Navegao Circular";
      case MENU_ENUM_LABEL_VALUE_NAVIGATION_BROWSER_FILTER_SUPPORTED_EXTENSIONS_ENABLE:
         return "Filtrar por Extenses Suportadas";
      case MENU_ENUM_LABEL_VALUE_CORE_UPDATER_AUTO_EXTRACT_ARCHIVE:
         return "Autoextrair Arquivos Baixados";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFORMATION:
         return "Informao de Sistema";
      case MENU_ENUM_LABEL_VALUE_ONLINE_UPDATER:
         return "Atualizao Online";
      case MENU_ENUM_LABEL_VALUE_CORE_INFORMATION:
         return "Informao de Core";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_NOT_FOUND:
         return "Diretrio no encontrado.";
      case MENU_ENUM_LABEL_VALUE_NO_ITEMS:
         return "Nenhum item.";
      case MENU_ENUM_LABEL_VALUE_CORE_LIST:
         return "Carregar Core";
      case MENU_ENUM_LABEL_VALUE_LOAD_CONTENT:
         return "Selecionar Arquivo";
      case MENU_ENUM_LABEL_VALUE_CLOSE_CONTENT:
         return "Fechar";
      case MENU_ENUM_LABEL_VALUE_MANAGEMENT:
         return "Databases";
      case MENU_ENUM_LABEL_VALUE_SAVE_STATE:
         return "Salvar Savestate";
      case MENU_ENUM_LABEL_VALUE_LOAD_STATE:
         return "Carregar Savestate";
      case MENU_ENUM_LABEL_VALUE_RESUME_CONTENT:
         return "Retomar";
      case MENU_ENUM_LABEL_VALUE_INPUT_DRIVER:
         return "Driver de Controlador";
      case MENU_ENUM_LABEL_VALUE_AUDIO_DRIVER:
         return "Driver de udio";
      case MENU_ENUM_LABEL_VALUE_JOYPAD_DRIVER:
         return "Driver de Joypad";
      case MENU_ENUM_LABEL_VALUE_AUDIO_RESAMPLER_DRIVER:
         return "Driver do Amostrador de udio";
      case MENU_ENUM_LABEL_VALUE_RECORD_DRIVER:
         return "Driver de Gravao";
      case MENU_ENUM_LABEL_VALUE_MENU_DRIVER:
         return "Driver de Menu";
      case MENU_ENUM_LABEL_VALUE_CAMERA_DRIVER:
         return "Driver de Cmera";
      case MENU_ENUM_LABEL_VALUE_LOCATION_DRIVER:
         return "Driver de Localizao";
      case MENU_ENUM_LABEL_VALUE_UNABLE_TO_READ_COMPRESSED_FILE:
         return "Incapaz de ler arquivo compactado.";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_SCALE:
         return "Interpolao de Overlay";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_PRESET:
         return "Predefinies de Overlay";
      case MENU_ENUM_LABEL_VALUE_AUDIO_LATENCY:
         return "Latncia de udio (ms)";
      case MENU_ENUM_LABEL_VALUE_AUDIO_DEVICE:
         return "Dispositivo de udio";
      case MENU_ENUM_LABEL_VALUE_KEYBOARD_OVERLAY_PRESET:
         return "Predefinies de Overlay de Teclado";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_OPACITY:
         return "Opacidade de Overlay";
      case MENU_ENUM_LABEL_VALUE_MENU_WALLPAPER:
         return "Papel de Parede do Menu";
      case MENU_ENUM_LABEL_VALUE_DYNAMIC_WALLPAPER:
         return "Papel de Parede Dinmico";
      case MENU_ENUM_LABEL_VALUE_CORE_INPUT_REMAPPING_OPTIONS: /* UPDATE/FIXME */
         return "Opes de Remapeamento de Controlador de Core";
      case MENU_ENUM_LABEL_VALUE_SHADER_OPTIONS:
         return "Shaders";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_PARAMETERS:
         return "Parmetros de Shader em Uso";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_PRESET_PARAMETERS:
         return "Menu de Parmetros de Shader";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_PRESET_SAVE_AS:
         return "Salvar Predefinies de Shader Como";
      case MENU_ENUM_LABEL_VALUE_NO_SHADER_PARAMETERS:
         return "Nenhum parmetro de shader disponvel.";
      case MENU_ENUM_LABEL_VALUE_VIDEO_SHADER_PRESET:
         return "Carregar Predefinies de Shader";
      case MENU_ENUM_LABEL_VALUE_VIDEO_FILTER:
         return "Filtro de Vdeo";
      case MENU_ENUM_LABEL_VALUE_AUDIO_DSP_PLUGIN:
         return "Plugin de DSP de udio";
      case MENU_ENUM_LABEL_VALUE_STARTING_DOWNLOAD:
         return "Iniciando download: ";
      case MENU_ENUM_LABEL_VALUE_SECONDS:
         return "segundos";
      case MENU_ENUM_LABEL_VALUE_OFF:
         return "OFF";
      case MENU_ENUM_LABEL_VALUE_ON:
         return "ON";
      case MENU_ENUM_LABEL_VALUE_UPDATE_ASSETS:
         return "Atualizar Recursos (Assets)";
      case MENU_ENUM_LABEL_VALUE_UPDATE_CHEATS:
         return "Atualizar Cheats";
      case MENU_ENUM_LABEL_VALUE_UPDATE_AUTOCONFIG_PROFILES:
         return "Atualizar Perfis de Autoconfigurao";
      case MENU_ENUM_LABEL_VALUE_UPDATE_DATABASES:
         return "Atualizar Databases";
      case MENU_ENUM_LABEL_VALUE_UPDATE_OVERLAYS:
         return "Atualizar Overlays";
      case MENU_ENUM_LABEL_VALUE_UPDATE_CG_SHADERS:
         return "Atualizar Shaders Cg";
      case MENU_ENUM_LABEL_VALUE_UPDATE_GLSL_SHADERS:
         return "Atualizar Shaders GLSL";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_CORE_NAME:
         return "Nome do core";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_CORE_LABEL:
         return "Rtulo do core";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_SYSTEM_NAME:
         return "Nome do sistema";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_SYSTEM_MANUFACTURER:
         return "Fabricante do sistema";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_CATEGORIES:
         return "Categorias";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_AUTHORS:
         return "Autores";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_PERMISSIONS:
         return "Permisses";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_LICENSES:
         return "Licena(s)";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_SUPPORTED_EXTENSIONS:
         return "Extenses suportadas";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_FIRMWARE:
         return "Firmware";
      case MENU_ENUM_LABEL_VALUE_CORE_INFO_CORE_NOTES:
         return "Notas do core";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_BUILD_DATE:
         return "Data do build";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_GIT_VERSION:
         return "Verso do git";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_CPU_FEATURES:
         return "Atributos da CPU";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FRONTEND_IDENTIFIER:
         return "Indentificador do frontend";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FRONTEND_NAME:
         return "Nome do frontend";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FRONTEND_OS:
         return "OS do frontend";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_RETRORATING_LEVEL:
         return "Nvel de RetroRating";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE:
         return "Fonte de energia";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE_NO_SOURCE:
         return "Nenhuma fonte";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE_CHARGING:
         return "Carregando";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE_CHARGED:
         return "Carregado";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_POWER_SOURCE_DISCHARGING:
         return "Descarregando";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_VIDEO_CONTEXT_DRIVER:
         return "Driver de contexto de vdeo";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DISPLAY_METRIC_MM_WIDTH:
         return "Mostrar largura (mm)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DISPLAY_METRIC_MM_HEIGHT:
         return "Mostrar altura (mm)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DISPLAY_METRIC_DPI:
         return "Mostrar DPI";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_LIBRETRODB_SUPPORT:
         return "Suporte a LibretroDB";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OVERLAY_SUPPORT:
         return "Suporte a Overlay";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_COMMAND_IFACE_SUPPORT:
         return "Suporte a interface de comandos";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_NETWORK_COMMAND_IFACE_SUPPORT:
         return "Suporte a interface de comandos de rede";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_COCOA_SUPPORT:
         return "Suporte a Cocoa";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_RPNG_SUPPORT:
         return "Suporte a PNG (RPNG)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_SDL_SUPPORT:
         return "Suporte a SDL1.2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_SDL2_SUPPORT:
         return "Suporte a SDL2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENGL_SUPPORT:
         return "Suporte a OpenGL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENGLES_SUPPORT:
         return "Suporte a OpenGL ES";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_THREADING_SUPPORT:
         return "Suporte a Threading";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_KMS_SUPPORT:
         return "Suporte a KMS/EGL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_UDEV_SUPPORT:
         return "Suporte a Udev";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENVG_SUPPORT:
         return "Suporte a OpenVG";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_EGL_SUPPORT:
         return "Suporte a EGL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_X11_SUPPORT:
         return "Suporte a X11";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_WAYLAND_SUPPORT:
         return "Suporte a Wayland";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_XVIDEO_SUPPORT:
         return "Suporte a XVideo";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_ALSA_SUPPORT:
         return "Suporte a ALSA";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OSS_SUPPORT:
         return "Suporte a OSS";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENAL_SUPPORT:
         return "Suporte a OpenAL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_OPENSL_SUPPORT:
         return "Suporte a OpenSL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_RSOUND_SUPPORT:
         return "Suporte a RSound";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_ROARAUDIO_SUPPORT:
         return "Suporte a RoarAudio";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_JACK_SUPPORT:
         return "Suporte a JACK";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_PULSEAUDIO_SUPPORT:
         return "Suporte a PulseAudio";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DSOUND_SUPPORT:
         return "Suporte a DirectSound";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_XAUDIO2_SUPPORT:
         return "Suporte a XAudio2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_ZLIB_SUPPORT:
         return "Suporte a Zlib";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_7ZIP_SUPPORT:
         return "Suporte a 7zip";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_DYLIB_SUPPORT:
         return "Suporte a bibliotecas dinmicas";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_CG_SUPPORT:
         return "Suporte a Cg";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_GLSL_SUPPORT:
         return "Suporte a GLSL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_HLSL_SUPPORT:
         return "Suporte a HLSL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_LIBXML2_SUPPORT:
         return "Suporte a anlise XML libxml2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_SDL_IMAGE_SUPPORT:
         return "Suporte a imagem SDL";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FBO_SUPPORT:
         return "Suporte a OpenGL/Direct3D render-to-texture (multi-pass shaders)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FFMPEG_SUPPORT:
         return "Suporte a FFmpeg";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_CORETEXT_SUPPORT:
         return "Suporte a CoreText";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_FREETYPE_SUPPORT:
         return "Suporte a FreeType";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_NETPLAY_SUPPORT:
         return "Suporte a Netplay (peer-to-peer)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_PYTHON_SUPPORT:
         return "Suporte a Python (script em shaders)";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_V4L2_SUPPORT:
         return "Suporte a Video4Linux2";
      case MENU_ENUM_LABEL_VALUE_SYSTEM_INFO_LIBUSB_SUPPORT:
         return "Suporte a Libusb";
      case MENU_ENUM_LABEL_VALUE_YES:
         return "Sim";
      case MENU_ENUM_LABEL_VALUE_NO:
         return "No";
      case MENU_ENUM_LABEL_VALUE_BACK:
         return "VOLTAR";
      case MENU_ENUM_LABEL_VALUE_SCREEN_RESOLUTION:
         return "Resoluo de Tela";
      case MENU_ENUM_LABEL_VALUE_DISABLED:
         return "Desativado";
      case MENU_ENUM_LABEL_VALUE_PORT:
         return "Porta";
      case MENU_ENUM_LABEL_VALUE_NONE:
         return "Nenhum";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_DEVELOPER:
         return "Desenvolvedor";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_PUBLISHER:
         return "Editora";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_DESCRIPTION:
         return "Descrio";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_NAME:
         return "Nome";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_ORIGIN:
         return "Origem";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_FRANCHISE:
         return "Franquia";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_RELEASE_MONTH:
         return "Ms de Lanamento";
      case MENU_ENUM_LABEL_VALUE_RDB_ENTRY_RELEASE_YEAR:
         return "Ano de Lanamento";
      case MENU_ENUM_LABEL_VALUE_TRUE:
         return "Verdadeiro";
      case MENU_ENUM_LABEL_VALUE_FALSE:
         return "Falso";
      case MENU_ENUM_LABEL_VALUE_MISSING:
         return "Faltando";
      case MENU_ENUM_LABEL_VALUE_PRESENT:
         return "Presente";
      case MENU_ENUM_LABEL_VALUE_OPTIONAL:
         return "Opcional";
      case MENU_ENUM_LABEL_VALUE_REQUIRED:
         return "Obrigatrio";
      case MENU_ENUM_LABEL_VALUE_STATUS:
         return "Status";
      case MENU_ENUM_LABEL_VALUE_AUDIO_SETTINGS:
         return "udio";
      case MENU_ENUM_LABEL_VALUE_INPUT_SETTINGS:
         return "Entradas";
      case MENU_ENUM_LABEL_VALUE_ONSCREEN_DISPLAY_SETTINGS:
         return "Informaes de Tela";
      case MENU_ENUM_LABEL_VALUE_OVERLAY_SETTINGS:
         return "Overlay em Tela";
      case MENU_ENUM_LABEL_VALUE_MENU_SETTINGS:
         return "Menu";
      case MENU_ENUM_LABEL_VALUE_MULTIMEDIA_SETTINGS:
         return "Reproduo de Mdia"; /* UPDATE/FIXME */
      case MENU_ENUM_LABEL_VALUE_USER_INTERFACE_SETTINGS:
         return "Interface de Usurio";
      case MENU_ENUM_LABEL_VALUE_MENU_FILE_BROWSER_SETTINGS:
         return "Menu de Navegao";
      case MENU_ENUM_LABEL_VALUE_CORE_UPDATER_SETTINGS:
         return "Atualizao de Core"; /* UPDATE/FIXME */
      case MENU_ENUM_LABEL_VALUE_NETWORK_SETTINGS:
         return "Rede";
      case MENU_ENUM_LABEL_VALUE_PLAYLIST_SETTINGS:
         return "Histrico";
      case MENU_ENUM_LABEL_VALUE_USER_SETTINGS:
         return "Usurio";
      case MENU_ENUM_LABEL_VALUE_DIRECTORY_SETTINGS:
         return "Diretrios";
      case MENU_ENUM_LABEL_VALUE_RECORDING_SETTINGS:
         return "Gravao";
      case MENU_ENUM_LABEL_VALUE_NO_INFORMATION_AVAILABLE:
         return "Nenhuma informao disponvel.";
      case MENU_ENUM_LABEL_VALUE_INPUT_USER_BINDS:
         return "Usurio %u";
      case MENU_ENUM_LABEL_VALUE_LANG_ENGLISH:
         return "Ingls";
      case MENU_ENUM_LABEL_VALUE_LANG_JAPANESE:
         return "Japons";
      case MENU_ENUM_LABEL_VALUE_LANG_FRENCH:
         return "Francs";
      case MENU_ENUM_LABEL_VALUE_LANG_SPANISH:
         return "Espanhol";
      case MENU_ENUM_LABEL_VALUE_LANG_GERMAN:
         return "Alemo";
      case MENU_ENUM_LABEL_VALUE_LANG_ITALIAN:
         return "Italiano";
      case MENU_ENUM_LABEL_VALUE_LANG_DUTCH:
         return "Holands";
      case MENU_ENUM_LABEL_VALUE_LANG_PORTUGUESE:
         return "Portugus";
      case MENU_ENUM_LABEL_VALUE_LANG_RUSSIAN:
         return "Russo";
      case MENU_ENUM_LABEL_VALUE_LANG_KOREAN:
         return "Coreano";
      case MENU_ENUM_LABEL_VALUE_LANG_CHINESE_TRADITIONAL:
         return "Chins (Tradicional)";
      case MENU_ENUM_LABEL_VALUE_LANG_CHINESE_SIMPLIFIED:
         return "Chins (Simplificado)";
      case MENU_ENUM_LABEL_VALUE_LANG_ESPERANTO:
         return "Esperanto";
      case MENU_ENUM_LABEL_VALUE_LEFT_ANALOG:
         return "Analgico Esquerdo";
      case MENU_ENUM_LABEL_VALUE_RIGHT_ANALOG:
         return "Analgico Direito";
      case MENU_ENUM_LABEL_VALUE_INPUT_HOTKEY_BINDS:
         return "Associao de Teclas de Atalho";
      case MENU_ENUM_LABEL_VALUE_FRAME_THROTTLE_SETTINGS:
         return "Definies do Limitador de Quadros";
      case MENU_ENUM_LABEL_VALUE_SEARCH:
         return "Busca:";
      case MENU_ENUM_LABEL_VALUE_USE_BUILTIN_IMAGE_VIEWER:
         return "Usar Visualizador de Imagens Interno";
      default:
         break;
   }

   return "null";
}
