/*
 * Resource manager - grant access to device special files
 *
 * Copyright (C) 2001-2002, Olaf Kirch <okir@lst.de>
 */

#include <arpa/inet.h>
#include <arpa/nameser.h>

#include <sys/poll.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>

#include "resmgrd.h"

#define _PATH_RESMGR_PIDFILE	"/var/run/resmgr.pid"

#define MAXCONN			128
#define MAXCONN_USER		8
#define MAXDEVICES		32
#define MAX_USER_SESSIONS	2048

static void		handle_connect(struct svc *);
static void		handle_message(struct conn *);
static void		cleanup(void);

int			opt_debug;
static char *		opt_pidfile = _PATH_RESMGR_PIDFILE;
static char *		opt_sockpath = _PATH_RESMGR_SOCKET;
static char *		opt_confpath = NULL;

static struct conn *	connections;

int
main(int argc, char **argv)
{
	int		opt_kill = 0;
	struct svc	svc;
	int		c;

	while ((c = getopt(argc, argv, "df:p:ks:")) != -1) {
		switch (c) {
		case 'd':
			opt_debug++;
			break;

		case 'f':
			opt_confpath = optarg;
			break;

		case 'k':
			opt_kill = 1;
			break;

		case 'p':
			opt_pidfile = optarg;
			break;

		case 's':
			opt_sockpath = optarg;
			break;

		default:
			fatal("unknown option");
		}
	}

	if (opt_kill) {
		pid_t	pid = read_pidfile(opt_pidfile);

		if (pid <= 0) {
			fprintf(stderr, "resmgrd not running\n");
			return 1;
		}
		if (kill(pid, SIGTERM) < 0) {
			perror("cannot kill resmgrd");
			return 1;
		}
		return 0;
	}

	/* Install exit handlers */
	signal(SIGTERM, (void (*)(int)) exit);
	signal(SIGINT, (void (*)(int)) exit);
	atexit(cleanup);

	if (!make_pidfile(opt_pidfile, 0))
		fatal("unable to create %s", opt_pidfile);

	/* Initialize device list */
	if (opt_confpath)
		res_config_set_filename(opt_confpath);
	res_config_reload();

	/* Reload ACL file when receiving SIGHUP */
	signal(SIGHUP, (void (*)(int)) res_config_reload);

	/* Start service */
	svc_start(&svc, opt_sockpath);

	/* Become a daemon */
	if (!opt_debug) {
		pid_t	pid;

		if ((pid = fork()) < 0)
			fatal("unable to fork: %m");
		if (pid != 0) {
			/* Prevent the parent from removing the lockfile */
			opt_pidfile = NULL;
			return 0;
		}

		/* Recreate the pid file */
		make_pidfile(opt_pidfile, 1);
		close(0);
		close(1);
		close(2);
		setsid();
	}

	/* Intialize logging */
	log_open_syslog("resmgr", opt_debug);

	/* Don't die when writing to a disconnected client */
	signal(SIGPIPE, SIG_IGN);

	while (1) {
		struct pollfd	p[MAXCONN+1];
		struct conn	*conn, **cp;
		int		n = 2;

		memset(p, 0, sizeof(p));
		p[0].fd		= svc.fd;
		p[0].events	= POLLIN|POLLHUP;

		/* Axe all dead connections */
		for (cp = &connections; (conn = *cp) != NULL; ) {
			if (conn->fd < 0) {
				*cp = conn->next;
				rsm_close(conn);
			} else {
				cp = &conn->next;
			}
		}

		for (n = 1, conn = connections; conn; conn = conn->next, n++) {
			assert(n < MAXCONN + 1);
			p[n].fd		= conn->fd;
			p[n].events	= POLLIN|POLLHUP;
		}

		/* Wait for event */
		poll(p, n, -1);

		if (p[0].revents & POLLIN)
			handle_connect(&svc);

		for (n = 1, conn = connections; conn; conn = conn->next, n++) {
			if (p[n].revents & POLLIN) {
				handle_message(conn);
			}
		}

		for (n = 1, conn = connections; conn; conn = conn->next, n++) {
			if ((p[n].revents & POLLHUP)) {
				res_user_t	*user;

				log("disconnect from %s", conn->user);
				if ((user = conn->ruser) != NULL) {
					user->nconnections--;
					res_user_free(user);
				}
				rsm_close_stream(conn);
			}
		}
	}
}

/*
 * Handle a connection request
 */
static void
handle_connect(struct svc *svc)
{
	struct conn	*conn;
	res_user_t	*user;
	char		*reason;

	if (!(conn = svc_accept(svc)))
		return;

	reason = "unable to create user record";
	if (!(user = res_user_get(conn->user))
	 && !(user = res_user_create(conn->user)))
		goto reject;
	conn->ruser = user;
	user->refcnt++;

	reason = "too many connections by user";
	if (user->nconnections >= MAXCONN_USER)
		goto reject;
	user->nconnections++;

	log("accepted connection from user %s", conn->user);
	conn->next = connections;
	connections = conn;
	return;

reject:
	log("rejected connection from user %s: %s", conn->user, reason);
	if (conn->ruser)
		res_user_free(conn->ruser);
	rsm_close(conn);
}

/*
 * HELP command
 */
static void
cmd_help(struct conn *conn, int argc, char **argv)
{
	rsm_printf(conn, "%03d-help     this message", MSG_OKAY);
	rsm_printf(conn, "%03d-list     list devices", MSG_OKAY);
	rsm_printf(conn, "%03d-login    open user session", MSG_OKAY);
	rsm_printf(conn, "%03d-logout   close user session", MSG_OKAY);
	rsm_printf(conn, "%03d-grant    grant user acces", MSG_OKAY);
	rsm_printf(conn, "%03d-revoke   revoke user acces", MSG_OKAY);
	rsm_printf(conn, "%03d-open     open device", MSG_OKAY);
	rsm_printf(conn, "%03d-lock     lock device", MSG_OKAY);
	rsm_printf(conn, "%03d-unlock   unlock device", MSG_OKAY);
	rsm_printf(conn, "%03d .", MSG_OKAY);
	return;
}

static void
cmd_login(struct conn *conn, int argc, char **argv)
{
	res_user_t	*user;
	char		*id, *name;

	name = argv[1];
	id = argv[2];

	if (conn->cred.uid) {
		user = conn->ruser;
		if (strcmp(user->name, name)) {
			log("User %s attempted to log in as %s\n",
					user->name, name);
			respond(conn, MSG_DENIED,
					"you are not allowed to do this");
			return;
		}
	} else {
		if (!(user = res_user_get(name)))
			user = res_user_create(name);
	}

	user->refcnt++;

	if (user->nsessions > MAX_USER_SESSIONS) {
		respond(conn, MSG_ERROR,
				"too many sessions for user %s", name);
	} else
	if (!res_user_login(user, id)) {
		respond(conn, MSG_ERROR, "login failed for %s", name);
	} else {
		res_user_resolve_acls(user, id);
		respond(conn, MSG_OKAY, "success");
	}

	res_user_free(user);
}

static void
cmd_logout(struct conn *conn, int argc, char **argv)
{
	res_user_t *user;

	user = conn->cred.uid? conn->ruser : NULL;
	if (!res_user_logout(user, argv[1])) {
		respond(conn, MSG_ERROR, "you are not allowed to do this");
	} else {
		respond(conn, MSG_OKAY, "success");
	}
}

static void
cmd_sessions(struct conn *conn, int argc, char **argv)
{
	res_session_t	*sess, *prev = NULL;

	for (sess = res_sessions; sess; sess = sess->next) {
		if (conn->cred.uid && strcmp(conn->user, sess->user->name))
			continue;
		if (prev)
			rsm_printf(conn, "%03d-%s %s", MSG_OKAY,
				prev->id, prev->user->name);
		prev = sess;
	}
	if (prev)
		rsm_printf(conn, "%03d %s %s", MSG_OKAY,
			prev->id, prev->user->name);
	else
		respond(conn, MSG_NODEVS, "no sessions");
}

static void
cmd_list(struct conn *conn, int argc, char **argv)
{
	res_user_t	*user = conn->ruser;
	res_family_t	*fmly = NULL;
	res_class_t	*cl;
	res_device_t	*dev;
	char		buffer[128];
	unsigned int	j;

	fmly = &res_family_file;
	if (argc == 2 && !(fmly = res_family_get(argv[1]))) {
		respond(conn, MSG_SYNTAX, "unknown resource family");
		return;
	}

	buffer[0] = '\0';
	for (j = 0; j < user->nclasses; j++) {
		cl = user->classes[j];
		for (dev = cl->devices; dev; dev = dev->next) {
			res_name_t	*name;

			if ((fmly->access_flags & ~dev->flags)
			 || !(name = fmly->parse_name(dev->name)))
				continue;

			if (buffer[0])
				rsm_printf(conn, "%03d-%s", MSG_OKAY, buffer);

			snprintf(buffer, sizeof(buffer),
				"r%c%c%c %s",
				(dev->flags & DEV_FLAGS_RO)? '-' : 'w',
				(dev->flags & DEV_FLAGS_SCSI)? 's' : '-',
				(dev->flags & DEV_FLAGS_PARIDE)? 'p' : '-',
				res_name_print(name));
			res_name_free(name);
		}
	}

	if (buffer[0]) {
		rsm_printf(conn, "%03d %s", MSG_OKAY, buffer);
	} else {
		respond(conn, MSG_NODEVS, "no devices available");
	}
}

static void
cmd_grant(struct conn *conn, int argc, char **argv)
{
	res_class_t	*cls = NULL;
	res_user_t	*user;
	char		*name, *whom;

	whom = argv[1];
	name = argv[2];
	if (!(user = res_user_get(whom))) {
		respond(conn, MSG_ERROR, "no session for user %s", whom);
		return;
	}

	if (!(cls = res_class_get(name))) {
		respond(conn, MSG_ERROR, "no such class: %s", name);
		return;
	}

	res_user_grant(user, cls);
	respond(conn, MSG_OKAY, "success");
}

static void
cmd_revoke(struct conn *conn, int argc, char **argv)
{
	res_class_t	*cls = NULL;
	res_user_t	*user;
	char		*name, *whom;

	whom = argv[1];
	if (!(user = res_user_get(whom))) {
		/* No session, no problem */
		respond(conn, MSG_OKAY, "success");
		return;
	}

	if (argc == 3) {
		name = argv[2];
		if (!(cls = res_class_get(name))) {
			respond(conn, MSG_ERROR,
				"no such class: %s", name);
			return;
		}
	}

	if (cls)
		res_user_revoke(user, cls);
	else
		res_user_revoke_all(user);
	respond(conn, MSG_OKAY, "success");
}

static void
cmd_add(struct conn *conn, int argc, char **argv)
{
	char		*devname, *clsname;
	res_class_t	*cls;
	res_family_t	*family;
	int		n, flags;

	devname = argv[1];
	clsname = argv[2];
	if (!(cls = res_class_get(clsname))) {
		respond(conn, MSG_ERROR, "no such class: %s", clsname);
	} else if (!res_device_sane(devname)) {
		respond(conn, MSG_DENIED, "refusing to add %s", devname);
	} else {
		flags = 0;
		for (n = 3; n < argc; n++) {
			const char	*opt = argv[n];

			if (!strcmp(opt, "read-only")) {
				flags |= DEV_FLAGS_RO;
			} else
			if ((family = res_family_get(opt)) != NULL) {
				flags |= family->access_flags;
			} else {
				respond(conn, MSG_SYNTAX,
					"Invalid device flag %s", opt);
			}
		}
		res_class_add(cls, devname, flags);
		respond(conn, MSG_OKAY, "success");
	}
}

static void
cmd_remove(struct conn *conn, int argc, char **argv)
{
	res_class_t	*cls;
	char		*devname, *clsname;

	devname = argv[1];
	if (argc == 3) {
		clsname = argv[2];
		if ((cls = res_class_get(clsname)) != NULL)
			res_class_remove(cls, devname);
	} else {
		for (cls = res_classes; cls; cls = cls->next)
			res_class_remove(cls, devname);
	}
	respond(conn, MSG_OKAY, "success");
}

static void
cmd_open(struct conn *conn, int argc, char **argv)
{
	res_name_t	*name;
	res_user_t	*user = conn->ruser;
	char		*opt;
	int		n, fd, flags = 0;

	for (n = 1; n < argc - 1; n++) {
		opt = argv[n];

		if (!strcmp(opt, "-ro")) {
			flags |= DEV_FLAGS_RO;
		} else {
			respond(conn, MSG_SYNTAX, "invalid option");
			return;
		}
	}

	if (!user) {
		respond(conn, MSG_DENIED, "You do not have an open session");
		return;
	}

	if (!(name = res_name_parse(argv[n]))) {
		respond(conn, MSG_SYNTAX, "Invalid device name");
		return;
	}

	fd = res_user_open(user, name, flags);
	res_name_free(name);

	if (fd < 0) {
		if (errno == EACCES) {
			respond(conn, MSG_DENIED, "permission denied");
		} else {
			respond(conn, MSG_ERROR,
				"failed to open device [%d]: %s",
				errno, strerror(errno));
		}
		return;
	}

	/* Now send the file desc */
	rsm_queue_fd(conn, fd);
	respond(conn, MSG_OKAY, "success");
	close(fd);
}

static void
cmd_lock(struct conn *conn, int argc, char **argv)
{
	res_name_t	*name;
	res_device_t	*dev;
	int		res;

	if (!(name = res_name_parse(argv[1]))) {
		respond(conn, MSG_SYNTAX, "invalid device name");
		return;
	}

	dev = res_user_get_device(conn->ruser, name, 1);
	res_name_free(name);

	if (dev == NULL) {
		respond(conn, MSG_DENIED, "Access denied");
		return;
	}

	res = res_device_lock(dev, conn->cred.uid, conn->cred.pid);
	if (res < 0) {
		respond(conn, MSG_DENIED, "Access denied");
	} else if (res != 0) {
		respond(conn, MSG_STALELOCK, "Stale lock by pid %d", res);
	} else {
		respond(conn, MSG_OKAY, "success");
	}
}

static void
cmd_unlock(struct conn *conn, int argc, char **argv)
{
	res_name_t	*name;
	res_device_t	*dev;
	int		res;

	if (!(name = res_name_parse(argv[1]))) {
		respond(conn, MSG_SYNTAX, "invalid device name");
		return;
	}

	dev = res_user_get_device(conn->ruser, name, 1);
	res_name_free(name);

	if (dev == NULL) {
		respond(conn, MSG_DENIED, "Access denied");
		return;
	}

	res = res_device_unlock(dev, conn->cred.uid);
	if (res < 0) {
		respond(conn, MSG_DENIED, "Access denied");
	} else {
		respond(conn, MSG_OKAY, "success");
	}
}

static struct command {
	const char *	name;
	void		(*handler)(struct conn *, int, char **);
	unsigned int	min_args, max_args;
	unsigned int	privileged;
} resmgr_commands[] = {
      { "login",	cmd_login,		2,	2,	0	},
      { "logout",	cmd_logout,		1,	1,	0	},
      { "grant",	cmd_grant,		2,	2,	1	},
      { "revoke",	cmd_revoke,		1,	2,	1	},
      { "add",		cmd_add,		2,	99,	1	},
      { "remove",	cmd_remove,		1,	2,	1	},
      { "sessions",	cmd_sessions,		0,	0,	0	},
      { "list",		cmd_list,		0,	1,	0	},
      { "open",		cmd_open,		1,	2,	0	},
      { "lock",		cmd_lock,		1,	1,	0	},
      { "unlock",	cmd_unlock,		1,	1,	0	},
      { "help",		cmd_help,		0,	0,	0	},
      { NULL }
};

/*
 * Handle a message
 */
static void
handle_message(struct conn *conn)
{
	struct command	*cmd;
	res_user_t	*user;
	char		buffer[256];
	char		*argv[16];
	int		n, argc;

	n = rsm_recv(conn, buffer, sizeof(buffer));
	if (n < 0) {
		rsm_close_stream(conn);
		return;
	}

	/* Incomplete line */
	if (n == 0) {
		/* rsm_close_stream(conn); */
		return;
	}

	/* split line */
	if (!(argc = line_split(buffer, argv, 16)))
		return;

	if (!(user = conn->ruser)) {
		respond(conn, MSG_ERROR, "Error");
		return;
	}

	for (cmd = resmgr_commands; cmd->name; cmd++) {
		if (strcasecmp(cmd->name, argv[0]))
			continue;
		if (cmd->privileged && conn->cred.uid) {
			respond(conn, MSG_DENIED,
				"you are not allowed to do this");
			return;
		}
		if (argc-1 < cmd->min_args || argc-1 > cmd->max_args) {
			respond(conn, MSG_SYNTAX,
				"incorrect number of arguments");
			return;
		}
		cmd->handler(conn, argc, argv);
		return;
	}

	respond(conn, MSG_SYNTAX, "command not understood");
}

/*
 * Broadcast/respond to client(s)
 */
void
message(int num, const char *fmt, ...)
{
	static char	buffer[1024];
	struct conn	*conn;
	va_list		ap;

	assert(0 < num && num < 1000);

	sprintf(buffer, "%03d ", num);

	va_start(ap, fmt);
	vsnprintf(&buffer[4], sizeof(buffer)-5, fmt, ap);
	strcat(buffer, "\n");
	va_end(ap);

	for (conn = connections; conn; conn = conn->next) {
		if (conn->broadcast)
			rsm_send(conn, buffer, strlen(buffer));
	}

	if (opt_debug)
		printf("* %s", buffer);
}

/*
 * Broadcast/respond to client
 */
void
respond(struct conn *conn, int num, const char *fmt, ...)
{
	char	fmtbuf[1024];
	va_list	ap;
	int	res;

	assert(0 < num && num < 1000);

	va_start(ap, fmt);
	snprintf(fmtbuf, sizeof(fmtbuf), "%03d %s", num, fmt);
	res = rsm_vprintf(conn, fmtbuf, ap);
	va_end(ap);

	if (res < 0) {
		rsm_printf(conn, "%03d Communication error", MSG_ERROR);
		rsm_close_stream(conn);
	}
}

/*
 * Cleanup function
 */
void
cleanup(void)
{
	if (opt_pidfile)
		unlink(opt_pidfile);
}
