/* 
 * Copyright (c) 2001 Secure Software Solutions
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <ctype.h>
#include <string.h>
#include "report.h"
#include "version.h"

int     flags     = 0;
int     forcelang = 0;
char *  progname  = "a.out";

char *default_files[] =
{
    LIBDIR "/rats.xml",
    LIBDIR "/rats-c.xml",
    LIBDIR "/rats-python.xml",
    LIBDIR "/rats-perl.xml",
    LIBDIR "/rats-php.xml"
};

static
int string_compare(char *s1, char *s2) /* Case insensitive string compare */
{
    for( ; tolower(*s1) == tolower(*s2); s1++, s2++)
        if (!*s1)
            return 0;
    return tolower(*s1) - tolower(*s2);
}

static
int load_database(char *filename, int silent)
{
    int         result;
    FILE *      fd;
    char *      buf;
    struct stat st;

    if ((fd = fopen(filename, "r")) == (FILE *)NULL)
    {
        if (!silent)
        {
            fprintf(stderr, "Unable to open '%s' for reading.\n", filename);
        }
        return 0;
    }

    fstat(fileno(fd), &st);
    buf = (char *)malloc(st.st_size + 1);
    fread(buf, st.st_size, 1, fd);
    *(buf + st.st_size) = '\0';

    result = (ParseVulnDb(buf, &database) != NULL);
    defaultdb = HashGet(database, "default");
    free(buf);
    return result;
}

static
void usage(void)
{
    printf("RATS v%d.%d - Rough Auditing Tool for Security\n", VERSION_MAJOR, VERSION_MINOR);
    printf("Copyright 2001 by Secure Software Solutions\nhttp://www.securesw.com\n\n");
    printf("usage: %s [-adhilrwx] name1 name2 ... namen\n\n", progname);
    printf("    -a <fun>       report any occurance of function 'fun' in the source file(s)\n");
    printf("    -d <filename>  specify an alternate vulnerability database.\n");
    printf("    -h             display usage information (what you\'re reading)\n");
    printf("    -i             report functions that accept external input\n");
    printf("    -l <language>  force the specified langauge to be used\n");
    printf("    -r             include references that are not function calls\n");
    printf("    -w <1,2,3>     set warning level (default %d)\n", warning_level);
    printf("    -x             do not load default databases\n");
}

int main(int argc, char **argv)
{
    int     dbloaded = 0, i, load_default = 1;
    Vuln_t  *uservul = NULL;
    Hash    defh = NULL;

    progname = argv[0];
    while ((i = getopt(argc, argv, "a:d:hil:rw:x")) != -1)
    {
        switch (i)
        {
            case 'a':
                if (!database)
                    database = HashInit();
                if (!(defh = HashGet(database, "default")))
                {
                    defh = HashInit();
                    HashInsert(database, defh, "default");
                }
                uservul = malloc(sizeof(Vuln_t));
                InitVuln(uservul);
                uservul->Name = malloc(strlen(optarg)+1);
                strncpy(uservul->Name, optarg, strlen(optarg));
                uservul->Name[strlen(optarg)] = 0;
                uservul->Info = malloc(sizeof(Info_t));
                InitInfo(uservul->Info);
                uservul->Info->Severity = Medium;
                uservul->Info->Description = malloc(34);
                strcpy(uservul->Info->Description, "Reporting user specified function");
 
                HashInsert(defh, uservul, optarg);
                if (!defaultdb)
                    defaultdb = HashGet(database, "default");
                break;

            case 'd':
                if (load_database(optarg, 0))
                    dbloaded++;
                break;

            case 'h':
                usage();
                exit(1);
                return 1;

            case 'i':
                flags |= INPUT_MODE;
                break;

            case 'l':
                if(!string_compare(optarg, "python"))
                    forcelang = LANG_PYTHON;
                else if(!string_compare(optarg, "c"))
                    forcelang = LANG_C;
                else if (!string_compare(optarg, "perl"))
                    forcelang = LANG_PERL;
                else if (!string_compare(optarg, "php"))
                    forcelang = LANG_PHP;
                else
                    fprintf(stderr, "Language %s unknown, using filename extensions instead\n", optarg);
                break;
            case 'r':
                flags |= INCLUDE_ALL_REFERENCES;
                break;

            case 'w':
                warning_level = 4 - atoi(optarg);
                if (warning_level < 1)
                    warning_level = 1;
                if (warning_level > 3)
                    warning_level = 3;
                break;

            case 'x':
                load_default = 0;
                break;

            default:
                exit(1);
                return 1;
        }
    }

    if (load_default)
    {
        /* Load the vulnerability database into memory */
        int i;

        for (i = 0;  i < sizeof(default_files) / sizeof(char *);  i++)
        {
            if (load_database(default_files[i], 1))
                dbloaded++;
        }
    }

    if (!dbloaded)  
    {
        fprintf(stderr, "No database able to be loaded, exiting.\n");
        exit(1);
        return 1;
    }

    if (optind >= argc)
    {
        process_file("<stdin>", stdin, forcelang);
    }
    else
    {
        while (optind < argc)
        {
            char *  filename;
            FILE *  fd;

            filename = argv[optind++];
            if ((fd = fopen(filename, "r")) == (FILE *)NULL)
            {
                fprintf(stderr, "%s: unable to open '%s' for reading.\n", progname, filename);
                continue;
            }
            process_file(filename, fd, forcelang);
            fclose(fd);
        }
    }

    generate_report();

    exit(0);
    return 0;
}
