\name{poprofile}
\title{Produce Predicted Ordinal Time Profiles for Plotting}
\usage{
plot(poprofile(mu,pintercept,preg,pinitial=NULL,depend="independence",
               times=NULL,distribution="multinomial",
               curve.type="probability"),main="Predicted profile",
     xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,lty=NULL,add=F,axes=F,
     bty="n",at=NULL,touch=F,...)
}
\alias{poprofile}
\alias{poprofile.default}
\alias{plot.poprofile}
\description{
  \code{poprofile} is used for plotting predicted marginal ordinal
  profiles over time for models obtained from dynamic models. These are
  obtained from a function supplied by the user. It produces output for
  plotting highest probabilities and cumulative probabilities for
  predicted marginal ordinal profiles corresponding to a model fitted
  with \code{\link{kalord}}.

  See \code{\link{moprofile}} for plotting marginal ordinal profiles,
  and \code{\link{ioprofile}} for plotting individual ordinal profiles.
}
\arguments{
  \item{mu}{The location regression as a function of the parameters and
    the times for the desired covariate values. The regression function
    must not contain intercepts.}
  \item{pintercept}{Intercept coefficients for the regression model.}
  \item{preg}{Remaining coefficients for the regression model: one for
    each time-constant and time-varying covariate, or one for each
    unknown parameter in their order of appearance.}
  \item{pinitial}{A coefficient for the initial parameter, if it is
    \code{NULL} then this parameter will be fixed at zero.}
  \item{depend}{Type of dependence. Choices are \code{independence}
    (default), \code{Markov}, \code{serial}, and \code{frailty}.}
  \item{times}{Vector of time points at which profiles are to be
    plotted.}
  \item{distribution}{Specifies the parameterization of the logistic
    distribution used in the Pareto distribution. Choices are binary,
    multinomial, continuation-ratio, and proportional-odds.}
  \item{curve.type}{Specifies the type of curves to be plotted. Must
    either be "probability" for highest probabilities or "cumulative"
    for cumulative probabilities.}
  \item{main}{A main title for the plot.}
  \item{xlab}{A label for the x-axis.}
  \item{ylab}{A label for the y-axis.}
  \item{xlim}{The x limits (min,max) of the plot.}
  \item{ylim}{The y limits (min,max) of the plot.}
  \item{lty}{A vector of integers or character strings specifying the
    line type to be used as the default in plotting lines. For further
    information, see \code{\link{par}}.}
  \item{pch}{A vector of integers or single characters specifying
    symbols to be used as the default in plotting points. For further
    information, see \code{\link{par}}.}
  \item{add}{If TRUE, the graph is added to an existing plot.}
  \item{axes}{If FALSE, axes are not drawn around the plot.}
  \item{bty}{A character string which determined the type of box which
    is drawn about plots. For further information, see
    \code{\link{par}}.}
  \item{at}{The points at which tick-marks are to be drawn. For further
    information, see \code{\link{axis}}.}
  \item{touch}{If TRUE, the x-axis and y-axis will touch each other.}
}
\value{
  \code{poprofile} returns information ready for plotting by
  \code{plot.poprofile}.
}
\author{P.J. Lindsey}
\seealso{
  \code{\link{ioprofile}}, \code{\link{kalord}},
  \code{\link{moprofile}}, \code{\link{plot.ordinal}}.
}
\examples{
library(ordinal)

#
# Binary data
#
data(cardiac.indiv)

y <- restovec(cardiac.indiv[,1:4],type="ordinal")

cov <- tcctomat(as.matrix(cardiac.indiv[,5:10]))

w <- rmna(y,ccov=cov)

rm(cardiac.indiv,y,cov)

# Time-constant and time-varying covariate with a frailty dependence.
z <- kalord(w,distribution="binary",mu=~age+ren+cop+dia+sex+pmi+times,
            ptvc=c(4.43357,-0.03128,-0.62602,-0.37679,-0.32969,-0.17013,
                   -0.12209,-0.09095),pinit=0.1196,dep="frailty")
ren <- rep(0,4)
cop <- rep(0,4)
dia <- rep(0,4)
sex <- rep(0,4)
pmi <- rep(0,4)
times <- 1:4

# Predicted highest probability profiles.
par(mfrow=c(2,2))
for(i in c(1,25,50,80)) {
  age <- rep(i,4)
  mu <- finterp(attr(z$mu,"formula"),.intercep=F)
  prof <- poprofile(mu,pint=4.43362,preg=c(4.43362,-0.03128,-0.62602,-0.37679,
                                           -0.32969,-0.17013,-0.12209,-0.09095),
                    pinit=NULL,times=times,dist="binary",curve="prob")
  plot(prof,main=paste("Age: ",i,sep=""))
}
par(mfrow=c(1,1))

# Predicted cumulative probability profiles.
par(mfrow=c(2,2))
for(i in c(1,25,50,80)) {
  age <- rep(i,4)
  mu <- finterp(attr(z$mu,"formula"),.intercep=F)
  prof <- poprofile(mu,pint=4.43362,preg=c(-0.03128,-0.62602,-0.37679,-0.32969,
                                           -0.17013,-0.12209,-0.09095),
                    pinit=NULL,times=times,dist="binary",curve="cum")
  plot(prof,main=paste("Age: ",i,sep=""))
}
par(mfrow=c(1,1))

rm(w,z,ren,cop,dia,sex,pmi,times,mu,i,age,prof)

#
# Ordinal data
#
data(tmi2)

y <- restovec(tmi2[,1:4],type="ordinal")

cov <- tcctomat(tmi2[,5],name="distance")

w <- rmna(y,ccov=cov)

rm(tmi2,y,cov)

# Proportional-odds model with time-constant covariate with a Markov dependence.
z <- kalord(w,distribution="proportional-odds",ccov=~distance,
            preg=c(-1.89,11.652,-0.199),pinit=3.111,pdep=0.217,dep="Markov")

mu <- function(p) rep(p[1]*distance,4)

# Predicted highest probability profiles.
par(mfrow=c(1,2))
for(distance in 1:2)
  plot(poprofile(mu,pint=c(-1.89,11.655),preg=-0.199,pinit=NULL,times=1:4,
                 dist="prop",curve="prob"),main=paste("Distance: ",distance,sep=""))
par(mfrow=c(1,1))

# Predicted cumulative probability profiles.
par(mfrow=c(1,2))
for(distance in 1:2)
  plot(poprofile(mu,pint=c(-1.89,11.655),preg=-0.199,pinit=NULL,times=1:4,
                 dist="prop",curve="cum"),main=paste("Distance: ",distance,sep=""))
par(mfrow=c(1,1))

rm(w,z,mu,distance)
}
\keyword{hplot}
