/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtSimpleIMContactManager.h"

#include "ui_SimpleIMContactManager.h"

#include <presentation/qt/imaccount/QtIMAccountSettings.h>
#include <presentation/qt/QtQuteCom.h>

#include <control/CQuteCom.h>
#include <control/contactlist/CContactList.h>
#include <control/profile/CUserProfile.h>

#include <model/config/Config.h>
#include <model/config/ConfigManager.h>
#include <model/account/SipAccount.h>
#include <model/contactlist/ContactProfile.h>
#include <model/profile/UserProfile.h>

#include <imwrapper/IMAccount.h>

#include <util/Logger.h>
#include <util/SafeDelete.h>

#include <qtutil/MouseEventFilter.h>
#include <qtutil/SafeConnect.h>

#include <QtGui/QtGui>

static const QString IMCONTACT_TEXT_SEPARATOR = ";";

QtSimpleIMContactManager::QtSimpleIMContactManager(ContactProfile & contactProfile,
	CUserProfile & cUserProfile, QWidget * parent)
	: QtIMContactManagerInterface(contactProfile, cUserProfile, parent) {

	_imContactManagerWidget = new QWidget(NULL);

	_ui = new Ui::SimpleIMContactManager();
	_ui->setupUi(_imContactManagerWidget);

	_msnLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(msnLineEditClicked()));
	_ui->msnLineEdit->installEventFilter(_msnLineEditMouseEventFilter);
	_aimLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(aimLineEditClicked()));
	_ui->aimLineEdit->installEventFilter(_aimLineEditMouseEventFilter);
	_icqLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(icqLineEditClicked()));
	_ui->icqLineEdit->installEventFilter(_icqLineEditMouseEventFilter);
	_yahooLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(yahooLineEditClicked()));
	_ui->yahooLineEdit->installEventFilter(_yahooLineEditMouseEventFilter);
	_jabberLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(jabberLineEditClicked()));
	_ui->jabberLineEdit->installEventFilter(_jabberLineEditMouseEventFilter);
	_googleTalkLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(googleTalkLineEditClicked()));
	_ui->googleTalkLineEdit->installEventFilter(_googleTalkLineEditMouseEventFilter);

	_faceBookLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(faceBookLineEditClicked()));
	_ui->faceBookLineEdit->installEventFilter(_faceBookLineEditMouseEventFilter);
	_mySpaceLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(mySpaceLineEditClicked()));
	_ui->mySpaceLineEdit->installEventFilter(_mySpaceLineEditMouseEventFilter);
	_skypeLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(skypeLineEditClicked()));
	_ui->skypeLineEdit->installEventFilter(_skypeLineEditMouseEventFilter);
	_twitterLineEditMouseEventFilter = new MousePressEventFilter(this, SLOT(twitterLineEditClicked()));
	_ui->twitterLineEdit->installEventFilter(_twitterLineEditMouseEventFilter);

	SAFE_CONNECT(_ui->msnLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->aimLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->icqLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->yahooLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->jabberLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->googleTalkLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->qutecomLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->sipLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));

	SAFE_CONNECT(_ui->faceBookLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->mySpaceLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->skypeLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));
	SAFE_CONNECT(_ui->twitterLineEdit, SIGNAL(textChanged(const QString &)), SIGNAL(contentHasChanged()));

#ifndef CUSTOM_ACCOUNT
	_ui->qutecomLabel->hide();
	_ui->label_2->hide();
	_ui->qutecomLineEdit->hide();
#endif


#ifndef ENABLE_FACEBOOK
	_ui->faceBookLineEdit->hide();
	_ui->label_11->hide();
	_ui->faceBookLabel->hide();
#endif
#ifndef ENABLE_MYSPACE
	_ui->mySpaceLineEdit->hide();
	_ui->label_12->hide();
	_ui->mySpaceLabel->hide();
#endif
#ifndef ENABLE_SKYPE
	_ui->skypeLineEdit->hide();
	_ui->label_14->hide();
	_ui->skypeLabel->hide();
#endif
#ifndef ENABLE_TWITTER
	_ui->twitterLineEdit->hide();
	_ui->label_13->hide();
	_ui->twitterLabel->hide();
#endif
		
#ifndef MULTIIMWRAPPER
	_ui->msnLineEdit->hide();
	_ui->label_4->hide();
	_ui->msnLabel->hide();
		
	_ui->aimLineEdit->hide();
	_ui->label_7->hide();
	_ui->aimLabel->hide();
		
	_ui->icqLineEdit->hide();
	_ui->label_8->hide();
	_ui->icqLabel->hide();
		
	_ui->yahooLineEdit->hide();
	_ui->label_5->hide();
	_ui->yahooLabel->hide();
		
	_ui->googleTalkLineEdit->hide();
	_ui->label_9->hide();
	_ui->googleTalkLabel->hide();

	_ui->jabberLineEdit->hide();
	_ui->label_10->hide();
	_ui->jabberLabel->hide();
	
	_ui->faceBookLineEdit->hide();
	_ui->label_11->hide();
	_ui->faceBookLabel->hide();
		
	_ui->mySpaceLineEdit->hide();
	_ui->label_12->hide();
	_ui->mySpaceLabel->hide();
		
	_ui->skypeLineEdit->hide();
	_ui->label_14->hide();
	_ui->skypeLabel->hide();
		
	_ui->skypeLineEdit->hide();
	_ui->label_14->hide();
	_ui->skypeLabel->hide();
		
	_ui->twitterLineEdit->hide();
	_ui->label_13->hide();
	_ui->twitterLabel->hide();
#endif

	loadIMContacts();
}

QtSimpleIMContactManager::~QtSimpleIMContactManager() {
	OWSAFE_DELETE(_msnLineEditMouseEventFilter);
	OWSAFE_DELETE(_aimLineEditMouseEventFilter);
	OWSAFE_DELETE(_icqLineEditMouseEventFilter);
	OWSAFE_DELETE(_yahooLineEditMouseEventFilter);
	OWSAFE_DELETE(_jabberLineEditMouseEventFilter);
	OWSAFE_DELETE(_googleTalkLineEditMouseEventFilter);

	OWSAFE_DELETE(_faceBookLineEditMouseEventFilter);
	OWSAFE_DELETE(_mySpaceLineEditMouseEventFilter);
	OWSAFE_DELETE(_skypeLineEditMouseEventFilter);
	OWSAFE_DELETE(_twitterLineEditMouseEventFilter);

	OWSAFE_DELETE(_ui);
}

std::set<IMContact *> QtSimpleIMContactManager::findIMContactsOfProtocol(EnumIMProtocol::IMProtocol imProtocol) const {
	std::set<IMContact *> result;

	const IMContactSet & imContactSet = _contactProfile.getIMContactSet();
	for (IMContactSet::const_iterator it = imContactSet.begin(); it != imContactSet.end(); it++) {
		if ((*it).getProtocol() == imProtocol) {
			result.insert((IMContact *)(&(*it)));
		}
	}

	return result;
}

QString QtSimpleIMContactManager::getIMContactsOfProtocol(EnumIMProtocol::IMProtocol imProtocol) const {
	QString imContacts;

	std::set<IMContact *> imContactSet = findIMContactsOfProtocol(imProtocol);
	for (std::set<IMContact *>::const_iterator it = imContactSet.begin(); it != imContactSet.end(); it++) {
		IMContact * imContact = *it;
		imContacts += QString::fromUtf8(imContact->getDisplayContactId().c_str());
		imContacts += IMCONTACT_TEXT_SEPARATOR;
	}

	//Remove the last IMCONTACT_TEXT_SEPARATOR
	int index = imContacts.lastIndexOf(IMCONTACT_TEXT_SEPARATOR);
	imContacts.remove(index, 1);

	return imContacts;
}

void QtSimpleIMContactManager::changeIMProtocolPixmaps(EnumIMProtocol::IMProtocol imProtocol,
	QLabel * imProtocolLabel, const char * connectedPixmap, QLineEdit * imProtocolLineEdit) {

	IMAccountList imAccounts = _cUserProfile.getUserProfile().getIMAccountManager().getIMAccountsOfProtocol(imProtocol);
	if (!imAccounts.empty()) {
		imProtocolLineEdit->setEnabled(true);
		imProtocolLabel->setPixmap(QPixmap(connectedPixmap));
	}
}

void QtSimpleIMContactManager::loadIMContacts() {
	//Update presentation depending if IM accounts exist
	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolMSN,
		_ui->msnLabel, ":pics/protocols/msn.png", _ui->msnLineEdit);
	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolAIM,
		_ui->aimLabel, ":pics/protocols/aim.png", _ui->aimLineEdit);
	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolICQ,
		_ui->icqLabel, ":pics/protocols/icq.png", _ui->icqLineEdit);
	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolYahoo,
		_ui->yahooLabel, ":pics/protocols/yahoo.png", _ui->yahooLineEdit);
	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolJabber,
		_ui->jabberLabel, ":pics/protocols/jabber.png", _ui->jabberLineEdit);
	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolGoogleTalk,
		_ui->googleTalkLabel, ":pics/protocols/google-talk.png", _ui->googleTalkLineEdit);

	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolFaceBook,
		_ui->faceBookLabel, ":pics/protocols/facebook.png", _ui->faceBookLineEdit);
	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolMySpace,
		_ui->mySpaceLabel, ":pics/protocols/myspace.png", _ui->mySpaceLineEdit);
	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolSkype,
		_ui->skypeLabel, ":pics/protocols/skype.png", _ui->skypeLineEdit);
	changeIMProtocolPixmaps(EnumIMProtocol::IMProtocolTwitter,
		_ui->twitterLabel, ":pics/protocols/twitter.png", _ui->twitterLineEdit);

	//Loads all the IM contacts
	_ui->qutecomLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolQuteCom));
	if (_ui->msnLineEdit->isEnabled()) {
		_ui->msnLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolMSN));
	}
	if (_ui->aimLineEdit->isEnabled()) {
		_ui->aimLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolAIM));
	}
	if (_ui->icqLineEdit->isEnabled()) {
		_ui->icqLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolICQ));
	}
	if (_ui->yahooLineEdit->isEnabled()) {
		_ui->yahooLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolYahoo));
	}
	if (_ui->jabberLineEdit->isEnabled()) {
		_ui->jabberLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolJabber));
	}
	if (_ui->googleTalkLineEdit->isEnabled()) {
		_ui->googleTalkLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolGoogleTalk));
	}

	if (_ui->faceBookLineEdit->isEnabled()) {
		_ui->faceBookLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolFaceBook));
	}
	if (_ui->mySpaceLineEdit->isEnabled()) {
		_ui->mySpaceLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolMySpace));
	}
	if (_ui->skypeLineEdit->isEnabled()) {
		_ui->skypeLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolSkype));
	}
	if (_ui->twitterLineEdit->isEnabled()) {
		_ui->twitterLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolTwitter));
	}

	_ui->sipLineEdit->setText(getIMContactsOfProtocol(EnumIMProtocol::IMProtocolSIP));
}

void QtSimpleIMContactManager::addIMContactsOfProtocol(EnumIMProtocol::IMProtocol imProtocol, const QString & text) {
	QString imContactsBefore = getIMContactsOfProtocol(imProtocol);
	QStringList contactIdListBefore = imContactsBefore.split(IMCONTACT_TEXT_SEPARATOR);
	QString imContactsAfter = text;
	QStringList contactIdListAfter = imContactsAfter.split(IMCONTACT_TEXT_SEPARATOR);
	for (int i = 0; i < contactIdListAfter.size(); i++) {
		contactIdListAfter[i] = contactIdListAfter[i].trimmed();
	}

	//Remove IMContacts
	for (int i = 0; i < contactIdListBefore.size(); i++) {
		QString contactId = contactIdListBefore[i];

		if (!contactIdListAfter.contains(contactId)) {
			//Remove the imContact from the contactProfile
			IMContact imContact(imProtocol, std::string(contactId.toUtf8()));
			_contactProfile.removeIMContact(imContact);
		}
	}

	//Check if id is not empty
	if (text.isEmpty()) {
		return;
	}

	//Add IMContacts
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	QString qutecomSuffix = "@" + QString::fromUtf8( config.getQuteComRealm().c_str() );
	for (int i = 0; i < contactIdListAfter.size(); i++) {
		QString contactId = contactIdListAfter[i];

		if (!contactIdListBefore.contains(contactId)) {
			
			// sip contact should know their domain
			if (imProtocol == EnumIMProtocol::IMProtocolQuteCom) {
				contactId += qutecomSuffix;
			} else if (imProtocol == EnumIMProtocol::IMProtocolSIP) {
				if (!contactId.contains("@")) {
					SipAccount * sipaccount = _cUserProfile.getUserProfile().getSipAccount();
					if (sipaccount) {
						contactId += QString("@") + QString::fromUtf8(sipaccount->getRealm().c_str());
					}
				}
			}
			////
			
			//Add the imContact to the contactProfile
			IMContact imContact(imProtocol, std::string(contactId.toUtf8()));

			//IMAccount to associate with the IMContact
			IMAccountList imAccounts =
				_cUserProfile.getUserProfile().getIMAccountManager().getIMAccountsOfProtocol(imProtocol);
			IMAccountList::const_iterator it = imAccounts.begin();
			if (it != imAccounts.end()) {
				IMAccount * imAccount =
					_cUserProfile.getUserProfile().getIMAccountManager().getIMAccount((*it).getUUID());
				imContact.setIMAccount(imAccount);
				OWSAFE_DELETE(imAccount);
			}

			_contactProfile.addIMContact(imContact);
		}
	}
}

void QtSimpleIMContactManager::saveIMContacts() {
	addIMContactsOfProtocol(EnumIMProtocol::IMProtocolQuteCom, _ui->qutecomLineEdit->text());
	if (_ui->msnLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolMSN, _ui->msnLineEdit->text());
	}
	if (_ui->aimLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolAIM, _ui->aimLineEdit->text());
	}
	if (_ui->icqLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolICQ, _ui->icqLineEdit->text());
	}
	if (_ui->yahooLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolYahoo, _ui->yahooLineEdit->text());
	}
	if (_ui->jabberLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolJabber, _ui->jabberLineEdit->text());
	}
	if (_ui->googleTalkLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolGoogleTalk, _ui->googleTalkLineEdit->text());
	}

	if (_ui->faceBookLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolFaceBook, _ui->faceBookLineEdit->text());
	}
	if (_ui->mySpaceLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolMySpace, _ui->mySpaceLineEdit->text());
	}
	if (_ui->skypeLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolSkype, _ui->skypeLineEdit->text());
	}
	if (_ui->twitterLineEdit->isEnabled()) {
		addIMContactsOfProtocol(EnumIMProtocol::IMProtocolTwitter, _ui->twitterLineEdit->text());
	}

	addIMContactsOfProtocol(EnumIMProtocol::IMProtocolSIP, _ui->sipLineEdit->text().remove(" "));
}

void QtSimpleIMContactManager::msnLineEditClicked() {
	if (!_ui->msnLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
			QtEnumIMProtocol::IMProtocolMSN, _imContactManagerWidget);
		loadIMContacts();
	}
}

void QtSimpleIMContactManager::aimLineEditClicked() {
	if (!_ui->aimLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
			QtEnumIMProtocol::IMProtocolAIM, _imContactManagerWidget);
		loadIMContacts();
	}
}

void QtSimpleIMContactManager::icqLineEditClicked() {
	if (!_ui->icqLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
			QtEnumIMProtocol::IMProtocolICQ, _imContactManagerWidget);
		loadIMContacts();
	}
}

void QtSimpleIMContactManager::yahooLineEditClicked() {
	if (!_ui->yahooLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
			QtEnumIMProtocol::IMProtocolYahoo, _imContactManagerWidget);
		loadIMContacts();
	}
}

void QtSimpleIMContactManager::jabberLineEditClicked() {
	if (!_ui->jabberLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
			QtEnumIMProtocol::IMProtocolJabber, _imContactManagerWidget);
		loadIMContacts();
	}
}

void QtSimpleIMContactManager::faceBookLineEditClicked() {
	if (!_ui->faceBookLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
			QtEnumIMProtocol::IMProtocolFaceBook, _imContactManagerWidget);
		loadIMContacts();
	}
}
void QtSimpleIMContactManager::mySpaceLineEditClicked() {
	if (!_ui->mySpaceLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
			QtEnumIMProtocol::IMProtocolMySpace, _imContactManagerWidget);
		loadIMContacts();
	}
}
void QtSimpleIMContactManager::skypeLineEditClicked() {
	if (!_ui->skypeLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
			QtEnumIMProtocol::IMProtocolSkype, _imContactManagerWidget);
		loadIMContacts();
	}
}
void QtSimpleIMContactManager::twitterLineEditClicked() {
	if (!_ui->twitterLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
			QtEnumIMProtocol::IMProtocolTwitter, _imContactManagerWidget);
		loadIMContacts();
	}
}

void QtSimpleIMContactManager::googleTalkLineEditClicked() {
	if (!_ui->jabberLineEdit->isEnabled()) {
		QtIMAccountSettings(_cUserProfile.getUserProfile(),
							QtEnumIMProtocol::IMProtocolGoogleTalk, _imContactManagerWidget);
		loadIMContacts();
	}
}

void QtSimpleIMContactManager::setQuteComName(const QString & qutecomName) {
	_ui->qutecomLineEdit->setText(qutecomName);
	_ui->qutecomLineEdit->setEnabled(true);
}

void QtSimpleIMContactManager::setSipAddress(const QString & sipAddress) {
	_ui->sipLineEdit->setText(sipAddress);
	_ui->sipLineEdit->setEnabled(true);
}

bool QtSimpleIMContactManager::couldBeSaved() {
	return ( (/*(_ui->qutecomLineEdit->isEnabled()) &&*/ (!_ui->qutecomLineEdit->text().isEmpty())) ||
			 ((_ui->msnLineEdit->isEnabled()) && (!_ui->msnLineEdit->text().isEmpty())) ||
			 (( _ui->aimLineEdit->isEnabled()) && (! _ui->aimLineEdit->text().isEmpty())) ||
			 (( _ui->icqLineEdit->isEnabled()) && (! _ui->icqLineEdit->text().isEmpty())) ||
			 (( _ui->yahooLineEdit->isEnabled()) && (! _ui->yahooLineEdit->text().isEmpty())) ||
			 (( _ui->jabberLineEdit->isEnabled()) && (! _ui->jabberLineEdit->text().isEmpty())) ||

			 (( _ui->faceBookLineEdit->isEnabled()) && (! _ui->faceBookLineEdit->text().isEmpty())) ||
			 (( _ui->mySpaceLineEdit->isEnabled()) && (! _ui->mySpaceLineEdit->text().isEmpty())) ||
			 (( _ui->skypeLineEdit->isEnabled()) && (! _ui->skypeLineEdit->text().isEmpty())) ||
			 (( _ui->twitterLineEdit->isEnabled()) && (! _ui->twitterLineEdit->text().isEmpty())) ||

			 (/*( _ui->sipLineEdit->isEnabled()) &&*/ (!_ui->sipLineEdit->text().isEmpty()))
		   );
}
