/***************************************************************************
 *            edit.c
 *
 *  Fri Aug 25 14:51:36 2006
 *  Copyright  2006-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/** @file edit.c
	@brief Functions for edit - cut, copy and paste.
	@author Copyright 2006-2007  Neil Williams <linux@codehelp.co.uk>
	@author Copyright 1999  Robert Lissner
*/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include <string.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <gtkextra/gtksheet.h>
#include "types.h"
#include "dim_list_menu.h"
#include "edit.h"
#include "main.h"
#include "dialog_initial.h"

static void
dim_sum_menu (QlContext * qlc)
{
	dim_list_edit_menu (qlc);
	dim_list_sort_menu (qlc);
	dim_list_column_menu (qlc);
}

gboolean
activate_callback (GtkSheet G_GNUC_UNUSED * sheet, 
				   gint row, gint col, gpointer data)
{
	QlTabData * tab;
	gchar *textp;
	gboolean flag;
	gint colx, rowx;

	tab = (QlTabData*) data;
	if (!tab)
		return FALSE;
	if (row < 0 || col < 0)
		return TRUE;			/* row or column buttons */

	if (row >= tab->file->last_row)
	{
		flag = FALSE;
		if (tab->file->last_row < 7)
			flag = TRUE;
		else
			for (rowx = tab->file->last_row - 4;
				rowx <= tab->file->last_row && !flag; rowx++)
				for (colx = 0; colx <= tab->file->last_field; colx++)
					if (gtk_sheet_cell_get_text (tab->view->sheet,
							rowx, colx))
					{
						flag = TRUE;
						break;
					}
		if (flag)
		{
			big_draw_start (tab);
			add1row (tab);
			big_draw_end (tab);
		}
	}
	tab->view->sel_range.row0 = tab->view->sel_range.rowi = row;
	tab->view->sel_range.col0 = tab->view->sel_range.coli = col;
	tab->view->sel_type = SELECT_ENTRY;
	dim_sum_menu (tab->qlc);

	if (row == tab->view->activate_row && col == tab->view->activate_col)
		return (TRUE);

	check_if_changed (tab);

	if (tab->view->activate_text)
	{
		g_free (tab->view->activate_text);
		tab->view->activate_text = NULL;
	}

	/* save this activate */
	tab->view->activate_row = row;
	tab->view->activate_col = col;
	tab->view->activate_text = textp =
		gtk_sheet_cell_get_text (tab->view->sheet, row, col);
	if (textp)
		tab->view->activate_text = g_strdup (textp);
	return TRUE;
}

/** Paste one row from clipboard into sheet, at row and col */
static void
actual_paste (QlTabData * tab, gint sheet_row, gint cb_row, gint col)
{
	gchar *linebufp;
	gchar *prevbufp;
	gint colx;
	gint fieldx;
	prevbufp = linebufp = tab->view->cb[cb_row];
	colx = col;
	while (*linebufp != '\0')
	{
		if (*linebufp != '\t')
			linebufp++;
		else
		{
			if (colx > tab->file->last_field)
				return;
			*linebufp = '\0';	/* actually change the clipboard */
			fieldx = tab->file->col_to_field[colx];
			gtk_sheet_set_cell_text (tab->view->sheet, sheet_row,
				colx, prevbufp);
			*linebufp = '\t';	/* restore the clipboard */

			prevbufp = ++linebufp;
			colx++;
		}
	}
}	/* end of actual_paste */

static void
clear_the_range (QlTabData * tab)
{
	/*this clears, not for deleting */
	gint rowx;
	gint16 colx;
	big_draw_start (tab);
	for (rowx = tab->view->sel_range.row0;
		rowx <= tab->view->sel_range.rowi; rowx++)
	{

		if (row_is_visible (tab, rowx))
			for (colx = tab->view->sel_range.col0;
				colx <= tab->view->sel_range.coli; colx++)
				gtk_sheet_cell_clear (tab->view->sheet, rowx,
					colx);
	}
	big_draw_end (tab);
	front_is_changed (tab);
}	/* end of clear_the_range */

void
edit_ditto (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;
	gint row0;
	gint col0;
	gint rowx;
	gchar *text;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	if (tab->view->sel_type != SELECT_ENTRY)
		return;
	row0 = tab->view->sel_range.row0;
	col0 = tab->view->sel_range.col0;
	for (rowx = row0 - 1; rowx >= 0; rowx--)
	{
		if (row_is_visible (tab, rowx))
		{
			text = gtk_sheet_cell_get_text (tab->view->sheet,
				rowx, col0);
			if (text)
				gtk_sheet_set_cell_text (tab->view->sheet,
					row0, col0, text);
			else
				gtk_sheet_cell_clear (tab->view->sheet, row0,
					col0);
			break;
		}
	}
}	/* end of edit_ditto */

void
edit_clear (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	big_draw_start (tab);
	clear_the_range (tab);
	big_draw_end (tab);
	front_is_changed (tab);
}

void
edit_fill_down (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	gint colx, rowx;
	gchar *text;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	big_draw_start (tab);
	if (tab->view->sel_type && tab->view->sel_range.row0 != tab->view->sel_range.rowi)
		for (colx = tab->view->sel_range.col0;
			colx <= tab->view->sel_range.coli; colx++)
		{
			text = gtk_sheet_cell_get_text (tab->view->sheet,
				tab->view->sel_range.row0, colx);
			for (rowx = tab->view->sel_range.row0 + 1;
				rowx <= tab->view->sel_range.rowi; rowx++)
				if (row_is_visible (tab, rowx))
				{
					if (text)
						gtk_sheet_set_cell_text (tab->view->sheet,
							rowx, colx, text);
					else
						gtk_sheet_cell_clear (tab->view->sheet,
							rowx, colx);
				}
		}
	big_draw_end (tab);
	front_is_changed (tab);
}

void
edit_paste (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	gint rowx;
	gint start_row;
	gint start_col;
	gint cb_row;
	gint sheet_row;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
/*	if (!paste_is_ok ())
		return;*/
	start_row = tab->view->sel_range.row0;
	start_col = tab->view->sel_range.col0;
	big_draw_start (tab);

	/* paste rows */
	if (tab->view->cb_sel_type == SELECT_ROW)
	{
		gtk_sheet_insert_rows (tab->view->sheet,
			start_row, tab->view->cb_rows);
		/** \bug last_row shouldn't be changed here */
		tab->file->last_row += tab->view->cb_rows;
		for (rowx = 0; rowx < tab->view->cb_rows; rowx++)
		{
			gtk_sheet_row_button_add_label (tab->view->sheet,
				start_row + rowx, " ");
			actual_paste (tab, start_row + rowx, rowx, 0);
		}
		big_draw_end (tab);
		front_is_changed (tab);
		return;
	}	/* end of type R */

	/* paste the rest */
	cb_row = 0;
	sheet_row = start_row;
	while (cb_row < tab->view->cb_rows)
	{
		if (sheet_row >= tab->file->last_row)
			add1row (tab);
		if (row_is_visible (tab, sheet_row))
		{
			actual_paste (tab, sheet_row, cb_row, start_col);
			cb_row++;
		}
		sheet_row++;
	}
	big_draw_end (tab);
	front_is_changed (tab);
}	/* end of edit_paste */

void
edit_select_all (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	GtkSheetRange range;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	range.row0 = range.col0 = 0;
	range.rowi = tab->file->last_row - 1;
	range.coli = tab->file->last_field;
	gtk_sheet_select_range (tab->view->sheet, &range);
}

void
edit_insert_rows (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	if (tab->view->sel_type == SELECT_COLUMN)
		return;
	gtk_sheet_insert_rows (tab->view->sheet, tab->view->sel_range.row0,
		1);
	gtk_sheet_row_button_add_label (tab->view->sheet,
		tab->view->sel_range.row0, " ");
	front_is_changed (tab);
	/* adding a new row of data, ensure Data is updated. */
	tab->file->last_row++;
}

void
new_column_width (GtkSheet G_GNUC_UNUSED * sheet, gint col, 
				  gint width, gpointer data)
{
	QlTabData * tab;
	QlFieldInfo * field;

	tab = (QlTabData*)data;
	front_is_changed (tab);
	field = ql_get_fieldinfo (tab, tab->file->col_to_field[col]);
	field->width = width / 8;
}

/** \todo does QL really need to manage the clipboard itself? */
static void
put_range_on_clipboard (QlTabData * tab)
{
	gchar linebuf[4096];
	gchar *text;
	gchar *linebufp;
	gint rowx;
	gint16 colx;

	/* get rid of previous contents, if any */
	if (tab->view->cb_rows)
		for (rowx = 0; rowx < tab->view->cb_rows; rowx++)
			g_free (tab->view->cb[rowx]);
	tab->view->cb_sel_type = tab->view->sel_type;
	tab->view->cb_rows = 0;
	tab->view->cb_cols = tab->view->sel_range.coli - tab->view->sel_range.col0 + 1;

	for (rowx = tab->view->sel_range.row0;
		rowx <= tab->view->sel_range.rowi; rowx++)
	{
		if (row_is_visible (tab, rowx))
		{
			if (tab->view->cb_rows >= MAX_CLIPBOARD - 2)
				level1_error (tab, _("Clipboard is limited to 5,000 rows"));
			linebufp = linebuf;
			for (colx = tab->view->sel_range.col0;
				colx <= tab->view->sel_range.coli; colx++)
			{
				text = gtk_sheet_cell_get_text (tab->view->sheet,
					rowx, colx);
				if (text)
				{
					strcpy (linebufp, text);
					linebufp += strlen (text);
				}
				*linebufp++ = '\t';
			}
			*linebufp = '\0';
			tab->view->cb[tab->view->cb_rows] = g_strdup (linebuf);
			tab->view->cb_rows++;
		}
	}
}

void
edit_cut (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	gint rowx;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);

	big_draw_start (tab);
	put_range_on_clipboard (tab);
	if (tab->view->sel_type == SELECT_ROW)
	{							/* work backwards to keep rowx right */
		for (rowx = tab->view->sel_range.rowi; rowx >= tab->view->sel_range.row0;
			rowx--)
			if (rowx != tab->file->last_row)
			{
				if (row_is_visible (tab, rowx))
				{
					unselect (tab);
					gtk_sheet_delete_rows (tab->view->sheet, rowx,
						1);
					tab->file->last_row--;
				}
			}
		gtk_sheet_select_row (tab->view->sheet, tab->view->sel_range.row0);
	}
	else
		clear_the_range (tab);

	dim_sum_menu (qlc);
	big_draw_end (tab);
	front_is_changed (tab);
}

void
edit_copy (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	put_range_on_clipboard (tab);
}

gboolean
row_is_visible (QlTabData * tab, gint testrow)
{
	return (tab->view->sheet->row[testrow].is_visible);
}

void
select_range_callback (GtkSheet G_GNUC_UNUSED * sheet, GtkSheetRange * range,
					   gpointer data)
{
	QlTabData * tab;

	tab = (QlTabData*)data;
	tab->view->sel_range = *range;
	if (!tab->view->sel_range.row0 && tab->view->sel_range.rowi >= tab->file->last_row)
		tab->view->sel_type = SELECT_COLUMN;
	else if (!tab->view->sel_range.col0
		&& tab->view->sel_range.coli >= tab->file->last_field)
		tab->view->sel_type = SELECT_ROW;
	else
	{
		if (tab->view->sel_range.row0 == tab->view->sel_range.rowi &&
			tab->view->sel_range.col0 == tab->view->sel_range.coli)
			tab->view->sel_type = SELECT_ENTRY;
		else
			tab->view->sel_type = SELECT_BLOCK;
	}
	dim_sum_menu (tab->qlc);
}
