#!/usr/bin/python

"""
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
"""

__version__ = "$Revision: 1.14 $"
# $Source: /cvsroot/quantlib/QuantLib-Python/QuantLib/test/distributions.py,v $

import QuantLib
import unittest
from math import exp, sqrt, pi

# define a Gaussian
def gaussian(x, average, sigma):
    normFact = sigma * sqrt( 2 * pi )
    dx = x-average
    return exp( -dx*dx/(2.0*sigma*sigma) ) / normFact

def gaussianDerivative(x, average, sigma):
    normFact = sigma * sigma * sigma * sqrt( 2 * pi )
    dx = x-average
    return - dx * exp( -dx*dx/(2.0*sigma*sigma) ) / normFact

# define the norm of a discretized function
def norm(f,h):
    # squared values
    f2 = map(lambda x: x*x, f)
    # numeric integral of f^2
    I = h*(reduce(lambda x,y: x+y, f2)-0.5*f2[0]-0.5*f2[-1])
    return sqrt(I)


class DistributionTest(unittest.TestCase):
    def runTest(self):
        "Testing distributions"
        average = 0.0
        sigma = 1.0
        normal = QuantLib.NormalDistribution(average, sigma)
        cum = QuantLib.CumulativeNormalDistribution(average, sigma)
        invCum = QuantLib.InvCumulativeNormalDistribution(average, sigma)

        xMin = average - 4*sigma
        xMax = average + 4*sigma
        # odd in include average
        N = 10001
        h = (xMax-xMin)/(N-1)

        x = [0]*N        # creates a list of N elements
        for i in range(N):
            x[i] = xMin+h*i

        y = map(lambda x,average=average,sigma=sigma:
            gaussian(x,average,sigma), x)

        yIntegrated = map(cum, x)
        yTemp = map(normal, x)
        y2Temp = map(cum.derivative, x)
        xTemp = map(invCum, yIntegrated)
        yd = map(normal.derivative, x)
        ydTemp = map(lambda x,average=average,sigma=sigma:
            gaussianDerivative(x,average,sigma), x)

        #check norm=gaussian
        e = norm(map(lambda x,y:x-y,yTemp,y),h)
        if not (e <= 1.0e-16):
            self.fail("""
norm of C++ NormalDistribution minus analytic gaussian: %(e)5.2e
tolerance exceeded
                      """ % locals())

        #check invCum(cum) = Identity
        e = norm(map(lambda x,y:x-y,xTemp,x),h)
        if not (e <= 1.0e-3):
            self.fail("""
norm of C++ invCum(cum(.)) minus identity: %(e)5.2e
tolerance exceeded
                      """ % locals())

        #check cum.derivative=normal
        e = norm(map(lambda x,y:x-y,y2Temp,y),h)
        if not (e <= 1.0e-16):
            self.fail("""
norm of C++ Cumulative.derivative minus analytic gaussian: %(e)5.2e
tolerance exceeded
                      """ % locals())

        #check normal.derivative=gaussianDerivative
        e = norm(map(lambda x,y:x-y,ydTemp,yd),h)
        if not (e <= 1.0e-16):
            self.fail("""
norm of C++ NormalDist.derivative minus analytic gaussian.derivative: %(e)5.2e
tolerance exceeded
                      """ % locals())

        # ... and now let's toy with finite difference
        # define the first derivative operators
        D = QuantLib.DZero(N,h)
        D2 = QuantLib.DPlusDMinus(N,h)
        # and calculate the derivatives
        y3Temp  = D.applyTo(yIntegrated)
        yd2Temp = D2.applyTo(yIntegrated)

        #check finite difference first order derivative operator = gaussian
        e = norm(map(lambda x,y:x-y,y3Temp,y),h)
        if not (e <= 1.0e-6):
            self.fail("""
norm of C++ FD 1st deriv. of cum minus analytic gaussian: %(e)5.2e
tolerance exceeded
                      """ % locals())

        # check finite difference second order derivative operator =
        # normal.derivative
        e = norm(map(lambda x,y:x-y,yd2Temp,yd),h)
        if not (e <= 1.0e-4):
            self.fail("""
norm of C++ FD 2nd deriv. of cum minus analytic gaussian.derivative: %(e)5.2e
tolerance exceeded
                      """ % locals())


if __name__ == '__main__':
    print 'testing QuantLib', QuantLib.__version__, QuantLib.QuantLibc.__file__, QuantLib.__file__
    import sys
    suite = unittest.TestSuite()
    suite.addTest(DistributionTest())
    if sys.hexversion >= 0x020100f0:
        unittest.TextTestRunner(verbosity=2).run(suite)
    else:
        unittest.TextTestRunner().run(suite)
    raw_input('press any key to continue')

