/***************************************************************************
 *   Copyright (C) 2009 by Robert Keevil                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; under version 2 of the License.         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <iostream>
#include <atlbase.h>
#include "libscrobble.h"
#include "ConvertUTF.h"

std::string Scrobble::wstring_to_utf(const std::wstring& widestring)
{
    size_t widesize = widestring.length();

    if (sizeof(wchar_t) == 2)
    {
        size_t utf8size = 3 * widesize + 1;
        char* utf8stringnative = new char[utf8size];
        const UTF16* sourcestart = reinterpret_cast<const UTF16*>(widestring.c_str());
        const UTF16* sourceend = sourcestart + widesize;
        UTF8* targetstart = reinterpret_cast<UTF8*>(utf8stringnative);
        UTF8* targetend = targetstart + utf8size;
        ConversionResult res = ConvertUTF16toUTF8(&sourcestart, sourceend, &targetstart, targetend, strictConversion);
        if (res != conversionOK)
        {
            delete [] utf8stringnative;
            add_log(LOG_ERROR, "wstring_to_utf failed - wchar_t = 2");
            return "";
        }
        *targetstart = 0;
        std::string resultstring(utf8stringnative);
        delete [] utf8stringnative;
        return resultstring;
    }
    else if (sizeof(wchar_t) == 4)
    {
        size_t utf8size = 4 * widesize + 1;
        char* utf8stringnative = new char[utf8size];
        const UTF32* sourcestart = reinterpret_cast<const UTF32*>(widestring.c_str());
        const UTF32* sourceend = sourcestart + widesize;
        UTF8* targetstart = reinterpret_cast<UTF8*>(utf8stringnative);
        UTF8* targetend = targetstart + utf8size;
        ConversionResult res = ConvertUTF32toUTF8(&sourcestart, sourceend, &targetstart, targetend, strictConversion);
        if (res != conversionOK)
        {
            delete [] utf8stringnative;
            add_log(LOG_ERROR, "wstring_to_utf failed - wchar_t = 4");
            return "";
        }
        *targetstart = 0;
        std::string resultstring(utf8stringnative);
        delete [] utf8stringnative;
        return resultstring;
    }
    else
    {
        add_log(LOG_ERROR, "wstring_to_utf failed - wchar_t = unknown");
    }
    return "";
}


bool Scrobble::mtp_connect()
{
    if (mtp_connected)
        return true;

    mtp_pSacClient = new CSecureChannelClient;
    mtp_pIdvMgr = NULL;

    /* these are generic keys */
    BYTE abPVK[] = {0x00};
    BYTE abCert[] = {0x00};

    HRESULT hr = CoInitialize(NULL);
    if (FAILED(hr))
    {
        add_log(LOG_ERROR, "MTP: Failed to CoInitialize");
        return false;
    }

    IComponentAuthenticate* pICompAuth;

    // get an authentication interface
    hr = CoCreateInstance(CLSID_MediaDevMgr, NULL, CLSCTX_ALL ,IID_IComponentAuthenticate, (void **)&pICompAuth);
    if FAILED(hr)
    {
        add_log(LOG_ERROR, "MTP: Failed to create Authentication Interface");
        return false;
    }

    mtp_ICompAuth = pICompAuth;
    // create a secure channel client certificate
    hr = mtp_pSacClient->SetCertificate(SAC_CERT_V1, (BYTE*) abCert, sizeof(abCert), (BYTE*) abPVK, sizeof(abPVK));
    if FAILED(hr)
    {
        add_log(LOG_ERROR, "MTP: Failed to set SAC Certificate");
        return false;
    }

    // bind the authentication interface to the secure channel client
    mtp_pSacClient->SetInterface(pICompAuth);

    // trigger communication
    hr = mtp_pSacClient->Authenticate(SAC_PROTOCOL_V1);
    if FAILED(hr)
    {
        add_log(LOG_ERROR, "MTP: Failed to SAC Authenticate");
        return false;
    }

    // get main interface to media device manager
    IWMDeviceManager3* tmpComObj;
    hr = pICompAuth->QueryInterface(IID_IWMDeviceManager2, (void**)&tmpComObj);
    if FAILED(hr)
    {
        add_log(LOG_ERROR, "MTP: Failed to get main interface");
        return false;
    }
    mtp_pIdvMgr = tmpComObj;
    // we now have a media device manager interface...

    // enumerate devices...
    IWMDMEnumDevice *mtp_pIEnumDev;

    hr = ((IWMDeviceManager3*)mtp_pIdvMgr)->EnumDevices2(&mtp_pIEnumDev);
    if FAILED(hr)
    {
        add_log(LOG_ERROR, "MTP: Failed to enumerate devices");
        return false;
    }

    hr = mtp_pIEnumDev->Reset(); // Next will now return the first device
    if FAILED(hr)
    {
        add_log(LOG_ERROR, "MTP: Failed to reset device");
        return false;
    }

    IWMDMDevice* mtp_pIDevice;
    mtp_devices.clear();
    unsigned long ulNumFetched;

    hr = mtp_pIEnumDev->Next(1, (IWMDMDevice **)&mtp_pIDevice, &ulNumFetched);
    while (SUCCEEDED(hr) && (hr != S_FALSE))
    {
        // Check device protocol
        CComQIPtr<IWMDMDevice3>pIWMDMDevice3(mtp_pIDevice);
        if (pIWMDMDevice3 != NULL)
        {
            PROPVARIANT val;
            PropVariantInit(&val);
            hr = pIWMDMDevice3->GetProperty(g_wszWMDMDeviceProtocol, &val);

            if (hr == S_OK)
            {
                if (val.vt == VT_CLSID && *val.puuid == WMDM_DEVICE_PROTOCOL_MTP)
                {
                    mtp_devices.push_back(pIWMDMDevice3);
                }
                PropVariantClear(&val);
            }
        }

        // move to next device
        hr = mtp_pIEnumDev->Next(1, (IWMDMDevice **)&mtp_pIDevice, &ulNumFetched);
    }
    mtp_pIEnumDev->Release();

    add_log(LOG_INFO, "MTP: Found " + int2string(mtp_devices.size()) + " device(s)");

    return true;
}

void Scrobble::mtp_disconnect()
{
    if (mtp_connected)
    {
        mtp_pIdvMgr->Release();
        mtp_ICompAuth->Release();

        CoUninitialize();
        mtp_devices.clear();
    }
}

void Scrobble::mtp_get_tracks()
{
    entries.clear();
    mtp_do_storage(false);

    // No info on when the track was played with MTP devices
    recalc_now();

    if (entries.size() > 0)
        scrobble_method = SCROBBLE_MTP;
}

void Scrobble::mtp_clear_tracks()
{
    mtp_do_storage(true);
}

void Scrobble::mtp_do_storage(bool clear)
{
    size_t i;
    for (i = 0; i < mtp_devices.size(); i++)
    {
        IWMDMDevice3* mtp_pIDevice = mtp_devices.at(i);
        HRESULT hr = S_OK;

        mtp_device_name(mtp_pIDevice);

        // Get a root enumerator.
        IWMDMEnumStorage *pEnumStorage;
        hr = mtp_pIDevice->EnumStorage(&pEnumStorage);
        if (FAILED(hr))
        {
            add_log(LOG_ERROR, "MTP: Failed to EnumStorage from Device");
        }
        else
        {
            mtp_recurse_storage(pEnumStorage, clear);
        }
    }
}

void Scrobble::mtp_recurse_storage(IWMDMEnumStorage* pEnumStorage, bool clear)
{
    HRESULT hr = S_OK;
    IWMDMStorage* pStorage;
    ULONG numRetrieved = 0;
    // Loop through all storages in the current storage.
    while(pEnumStorage->Next(1, &pStorage, &numRetrieved) == S_OK && numRetrieved == 1)
    {
        // Get the name of the object. The first time this is called on a 
        // device, it will retrieve '\' as the root folder name.
        const UINT MAX_LEN = 255;
        WCHAR name[MAX_LEN];
        hr = pStorage->GetName((LPWSTR)&name, MAX_LEN);

        // Get metadata for the storage.
        if (SUCCEEDED(hr))
            mtp_get_metadata(pStorage, clear);

        // Find out something about the item.
        DWORD attributes = 0;
        _WAVEFORMATEX audioFormat;
        hr = pStorage->GetAttributes(&attributes, &audioFormat);
        if (FAILED(hr))
        {
            add_log(LOG_DEBUG, "Couldn't get storage attributes in RecursivelyExploreStorage.");
        }

        // If this is a folder, recurse into it.
        if (attributes & WMDM_FILE_ATTR_FOLDER)
        {
            IWMDMEnumStorage* pEnumSubStorage;
            hr = pStorage->EnumStorage(&pEnumSubStorage);
            if (FAILED(hr))
                add_log(LOG_INFO, "MTP: Failed to EnumStorage (folder)");
            else
                mtp_recurse_storage(pEnumSubStorage, clear);
        }

        pStorage->Release();
    }
}

// Function to print out all the metadata associated with a storage.
HRESULT Scrobble::mtp_get_metadata(IWMDMStorage *pStorage, bool clear)
{
    HRESULT hr = S_OK;

    // A dummy loop to handle unrecoverable errors. When we hit an error we
    // can't handle or don't like, we just use a 'break' statement.

    do
    {
        IWMDMStorage3* pStorage3;
        IWMDMMetaData* pMetadata;
        hr = pStorage->QueryInterface(__uuidof(IWMDMStorage3), (void**)&pStorage3);
        if (FAILED(hr))
        {
            add_log(LOG_ERROR, "Couldn't get an IWMDMStorage3 interface in mtp_get_metadata.");
            break;
        }

        hr = pStorage3->GetMetadata(&pMetadata);
        if (FAILED(hr))
        {
            add_log(LOG_ERROR, "Couldn't get an IWMDMMetaData interface in mtp_get_metadata.");
            break;
        }

        WMDM_TAG_DATATYPE type;
        BYTE *value;
        unsigned int len;
        scrob_entry tmp;
        int playcount = 0;
        tmp.length = tmp.when = 0;
        tmp.album = tmp.artist = tmp.mb_track_id = tmp.title = tmp.tracknum = "";
        tmp.played = '\0';

        hr = pMetadata->QueryByName(g_wszWMDMPlayCount, &type, &value, &len);
        if (SUCCEEDED(hr))
        {
            playcount = (DWORD)*value;
        }
        else
        {
            pMetadata->Release();
            break;  // playcount is required
        }

        hr = pMetadata->QueryByName(g_wszWMDMTitle, &type, &value, &len);
        if (SUCCEEDED(hr))
            tmp.title = wstring_to_utf((wchar_t*)value);
        else
        {
            pMetadata->Release();
            break; // title is required
        }

        hr = pMetadata->QueryByName(g_wszWMDMAuthor, &type, &value, &len);
        if (SUCCEEDED(hr))
            tmp.artist = wstring_to_utf((wchar_t*)value);
        else
        {
            pMetadata->Release();
            break; // artist is required
        }

        hr = pMetadata->QueryByName(g_wszWMDMAlbumTitle, &type, &value, &len);
        if (SUCCEEDED(hr))
            tmp.album = wstring_to_utf((wchar_t*)value);

        hr = pMetadata->QueryByName(g_wszWMDMTrack, &type, &value, &len);
        if (SUCCEEDED(hr))
        {
            int tracknum = (DWORD)*value;
            tmp.tracknum = int2string(tracknum);
        }

        hr = pMetadata->QueryByName(g_wszWMDMDuration, &type, &value, &len);
        if (SUCCEEDED(hr))
        {
            unsigned long long tracklen = 0;
            tracklen = value[7];
            tracklen = tracklen<<8;
            tracklen += value[6];
            tracklen = tracklen<<8;
            tracklen += value[5];
            tracklen = tracklen<<8;
            tracklen += value[4];
            tracklen = tracklen<<8;
            tracklen += value[3];
            tracklen = tracklen<<8;
            tracklen += value[2];
            tracklen = tracklen<<8;
            tracklen += value[1];
            tracklen = tracklen<<8;
            tracklen += value[0];
            tmp.length = tracklen/10000000;
        }
        else
        {
            pMetadata->Release();
            break; // duration is required
        }

        pMetadata->Release();

        if (playcount > 0 && tmp.artist.length() && tmp.title.length())
        {
            if (clear)
            {
                IWMDMMetaData* pnewMetadata;
                BYTE* newval = new BYTE[sizeof(WMDM_TYPE_DWORD)];
                newval[0] = newval[1] = newval[2] = newval[3] = 0;
                hr = pStorage3->CreateEmptyMetadataObject(&pnewMetadata);
                if (FAILED(hr))
                {
                    add_log(LOG_INFO, "Failed to create new metadata object");
                }

                hr = pnewMetadata->AddItem(WMDM_TYPE_DWORD, g_wszWMDMPlayCount, newval, sizeof(WMDM_TYPE_DWORD));
                if (FAILED(hr))
                {
                    add_log(LOG_INFO, "Failed to add to new metadata object");
                }

                hr = pStorage3->SetMetadata(pnewMetadata);
                if (FAILED(hr))
                {
                    add_log(LOG_INFO, "Failed to set new metadata object");
                }

                delete[] newval;
                pnewMetadata->Release();
            }
            else
            {
                tmp.played = 'L';
                for (int i = 0; i < playcount; i++)
                    entries.push_back(tmp);
            }
        }
    }
    while(FALSE); // End of dummy loop.

    return hr;
}

void Scrobble::mtp_device_name(IWMDMDevice3* mtp_pIDevice)
{
    // model is sometimes the "friendly" name, or just manufacturers model name
    wchar_t name[256];
    std::string make, model;
    make = model = "";
    HRESULT hr = mtp_pIDevice->GetManufacturer(name, 256);
    if (SUCCEEDED(hr))
        make = wstring_to_utf(name);
    hr = mtp_pIDevice->GetName(name, 256);
    if (SUCCEEDED(hr))
        model = wstring_to_utf(name);

    add_log(LOG_INFO, "MTP Reading: " + make + " \"" + model + "\"");
}
