/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de, Michael Ritzert
    michael.ritzert@googlemail.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/
#include "CGarminDBWpt.h"
#include "CToolViewWpt.h"
#include "CDlgSetupWpt.h"
#include "GarminIcons.h"
#include "CCentralResources.h"
#include "CGpx.h"
#include "icons.h"

#include <QtGui>
#include <QtNetwork/QHttp>
#include <QUrl>

CGarminDBWpt::CGarminDBWpt(QTabWidget * parent)
: QObject(parent)
, tab(parent)
, opencaching(0)
, rcvData(false)
{

    QSettings cfg;
    int size = cfg.beginReadArray("waypoints");
    for (int i = 0; i < size; ++i) {
        cfg.setArrayIndex(i);
        CGarminWpt * wpt = new CGarminWpt(this);
        connect(wpt,SIGNAL(sigSelected(CGarminWpt*)),this,SLOT(slotSelectWpt(CGarminWpt*)));

        wpt->ident   = cfg.value("ident",wpt->ident).toString();
        wpt->smbl    = cfg.value("smbl",wpt->smbl).value<quint16>();
        wpt->comment = cfg.value("comment",wpt->comment).toString();
        wpt->lon     = cfg.value("lon",wpt->lon).toDouble();
        wpt->lat     = cfg.value("lat",wpt->lat).toDouble();
        wpt->alt     = cfg.value("alt",wpt->alt).value<float>();
        wpt->dpth    = cfg.value("dpth",wpt->dpth).value<float>();
        wpt->dist    = cfg.value("dist",wpt->dist).value<float>();
        wpt->link    = cfg.value("link",wpt->link).toString();
        getWptIconById(wpt->smbl,wpt->icon);
        wpt->sticky  = true;
        wpt->key     = "sticky|" + wpt->ident;
        waypoints[wpt->key] = wpt;

    }
    cfg.endArray();

    toolview = new CToolViewWpt(this,0);
    toolview->show();
    parent->addTab(toolview,QPixmap(iconWaypoint16x16),"");
    parent->setTabToolTip(parent->indexOf(toolview), tr("Waypoint View"));

    // DELETE FROM `cache_type`;
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (1, 'Other', 'unbekannter Cachetyp', 'unknown cachetyp', 'cache/unknown.gif');
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (2, 'Trad.', 'normaler Cache', 'Traditional Cache', 'cache/traditional.gif');
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (3, 'Multi', 'Multicache', 'Multicache', 'cache/multi.gif');
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (4, 'Virt.', 'virtueller Cache', 'virtual Cache', 'cache/virtual.gif');
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (5, 'ICam.', 'Webcam Cache', 'Webcam Cache', 'cache/webcam.gif');
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (6, 'Event', 'Event Cache', 'Event Cache', 'cache/event.gif');
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (7, 'Quiz', 'Rtselcache', 'Quizcache', 'cache/mystery.gif');
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (8, 'Math', 'Mathe-/Physikcache', 'Math/Physics-Cache', 'cache/mathe.gif');
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (9, 'Moving', 'Beweglicher Cache', 'Moving Cache', 'cache/moving.gif');
    // INSERT INTO `cache_type` (`id`, `short`, `de`, `en`, `icon_large`) VALUES (10, 'Driv.', 'Drive-In', 'Drive-In', 'cache/drivein.gif');
    cfg.beginGroup("icons");
    cfg.beginGroup("oc.de");
    cacheIcons[1]   = type_t(tr("Other"),cfg.value("Other","Geocache").toString());
    cacheIcons[2]   = type_t(tr("Traditional"),cfg.value("Traditional","Geocache").toString());
    cacheIcons[3]   = type_t(tr("Multicache"),cfg.value("Multicache","Block, Blue").toString());
    cacheIcons[4]   = type_t(tr("Virtual"),cfg.value("Virtual","Block, Blue").toString());
    cacheIcons[5]   = type_t(tr("Webcam"),cfg.value("Webcam","Block, Blue").toString());
    cacheIcons[6]   = type_t(tr("Event"),cfg.value("Event","Block, Blue").toString());
    cacheIcons[7]   = type_t(tr("Quizcache"),cfg.value("Quizcache","Block, Red").toString());
    cacheIcons[8]   = type_t(tr("Math/Physics"),cfg.value("Math/Physics","Block, Blue").toString());
    cacheIcons[9]   = type_t(tr("Moving"),cfg.value("Moving","Block, Blue").toString());
    cacheIcons[10]  = type_t(tr("Drive-In"),cfg.value("Drive-In","Car").toString());
    cfg.endGroup();
    cfg.endGroup();
}


CGarminDBWpt::~CGarminDBWpt()
{
    int cnt = 0;
    QSettings cfg;
    cfg.remove("waypoints");
    cfg.beginWriteArray("waypoints");
    QMap<QString,CGarminWpt*>::const_iterator wpt = waypoints.begin();
    while(wpt != waypoints.end()) {
        if((*wpt)->sticky) {
            cfg.setArrayIndex(cnt);
            cfg.setValue("ident",(*wpt)->ident);
            cfg.setValue("smbl",(*wpt)->smbl);
            cfg.setValue("comment",(*wpt)->comment);
            cfg.setValue("lon",(*wpt)->lon);
            cfg.setValue("lat",(*wpt)->lat);
            cfg.setValue("alt",(*wpt)->alt);
            cfg.setValue("dpth",(*wpt)->dpth);
            cfg.setValue("dist",(*wpt)->dist);
            cfg.setValue("link",(*wpt)->link);
            ++cnt;
        }
        ++wpt;
    }
    cfg.endArray();

    cfg.beginGroup("icons");
    cfg.beginGroup("oc.de");
    cfg.setValue(cacheIcons[1].type,cacheIcons[1].icon);
    cfg.setValue(cacheIcons[2].type,cacheIcons[2].icon);
    cfg.setValue(cacheIcons[3].type,cacheIcons[3].icon);
    cfg.setValue(cacheIcons[4].type,cacheIcons[4].icon);
    cfg.setValue(cacheIcons[5].type,cacheIcons[5].icon);
    cfg.setValue(cacheIcons[6].type,cacheIcons[6].icon);
    cfg.setValue(cacheIcons[7].type,cacheIcons[7].icon);
    cfg.setValue(cacheIcons[8].type,cacheIcons[8].icon);
    cfg.setValue(cacheIcons[9].type,cacheIcons[9].icon);
    cfg.setValue(cacheIcons[10].type,cacheIcons[10].icon);
    cfg.endGroup();
    cfg.endGroup();

}


void CGarminDBWpt::gainFocus()
{
    if(tab) {
        if(tab->currentWidget() != toolview) {
            tab->setCurrentWidget(toolview);
        }
    }
}


void CGarminDBWpt::newWaypoint(const double lon, const double lat, const float alt, const QString& comment)
{
    CGarminWpt * wpt = new CGarminWpt(this);
    connect(wpt,SIGNAL(sigSelected(CGarminWpt*)),this,SLOT(slotSelectWpt(CGarminWpt*)));

    QDateTime time = QDateTime::currentDateTime();
    wpt->time = time.toUTC().toTime_t() - gpResources->getUTCOffset();

    if(!isnan(lon) && !isnan(lat)) {
        wpt->lon = lon;
        wpt->lat = lat;
        wpt->alt = alt;
        wpt->comment = comment;
    }

    CDlgSetupWpt dlg(*wpt,0);
    if(dlg.exec() == QDialog::Accepted) {
        if(wpt->sticky) {
            wpt->key = "sticky|" + wpt->ident;
            waypoints[wpt->key] = wpt;
        }
        else {
            wpt->key = "user|" + wpt->ident;
            waypoints[wpt->key] = wpt;
        }

        emit sigWaypointListChanged();
    }
    else {
        delete wpt;
    }
}


void CGarminDBWpt::editWaypoint(const QString& key)
{
    bool sticky = waypoints[key]->sticky;

    CDlgSetupWpt dlg(*waypoints[key],0);
    dlg.exec();

    if(sticky && !waypoints[key]->sticky) {
        CGarminWpt * wpt = waypoints[key];
        wpt->key = "user|" + wpt->ident;
        waypoints[wpt->key] = wpt;
        waypoints.remove(key);
    }
    else if(!sticky && waypoints[key]->sticky) {
        CGarminWpt * wpt = waypoints[key];
        wpt->key = "sticky|" + wpt->ident;
        waypoints[wpt->key] = wpt;
        waypoints.remove(key);
    }

    emit sigWaypointListChanged();
}


void CGarminDBWpt::removeWaypointBy()
{
    toolview->slotDeleteBy();
}


void CGarminDBWpt::removeWaypoint(const QString& key)
{
    if(!waypoints.contains(key)) return;
    if(waypoints[key]->sticky) {
        QString msg = tr("Do you really want to delete the sticky waypoint '%1'").arg(waypoints[key]->ident);
        if(QMessageBox::question(0,tr("Delete sticky waypoint ..."),msg, QMessageBox::Ok|QMessageBox::No, QMessageBox::No) == QMessageBox::No) {
            return;
        }
    }
    waypoints.remove(key);
    emit sigWaypointListChanged();
}


void CGarminDBWpt::removeWaypoint(const QStringList& keys)
{
    QString key;
    foreach(key,keys) {
        if(!waypoints.contains(key)) continue;
        if(waypoints[key]->sticky) {
            QString msg = tr("Do you really want to delete the sticky waypoint '%1'").arg(waypoints[key]->ident);
            if(QMessageBox::question(0,tr("Delete sticky waypoint ..."),msg, QMessageBox::Ok|QMessageBox::No, QMessageBox::No) == QMessageBox::No) {
                continue;
            }
        }
        waypoints.remove(key);
    }
    emit sigWaypointListChanged();
}


void CGarminDBWpt::downloadWaypoints()
{
    Garmin::IDevice * dev = 0;
    try
    {
        std::list<Garmin::Wpt_t> wpts;

        dev = gpResources->device();
        if(dev) {
            dev->downloadWaypoints(wpts);
        }

        std::list<Garmin::Wpt_t>::const_iterator wpt = wpts.begin();
        while(wpt != wpts.end()) {
            CGarminWpt * w = new CGarminWpt(*wpt,this);
            connect(w,SIGNAL(sigSelected(CGarminWpt*)),this,SLOT(slotSelectWpt(CGarminWpt*)));
            w->key = "device|" + w->ident;
            waypoints[w->key] = w;
            ++wpt;
        }

        emit sigWaypointListChanged();
    }
    catch(int e) {
        if(dev == 0) return;
        QMessageBox::warning(0,tr("Device Link Error"),dev->getLastError().c_str(),QMessageBox::Ok,QMessageBox::NoButton);
        if(e == Garmin::errSync) {
            gpResources->resetDevice();
        }
    }
}


void CGarminDBWpt::uploadWaypoints()
{
    std::list<Garmin::Wpt_t> wpts;
    QMap<QString,CGarminWpt*>::iterator wpt = waypoints.begin();

    while(wpt != waypoints.end()) {
        wpts.push_back((*wpt)->toGarminWpt());
        ++wpt;
    }

    // nothing to do?
    if(wpts.size() == 0) return;

    Garmin::IDevice * dev = 0;
    try
    {
        dev = gpResources->device();
        if(dev) {
            dev->uploadWaypoints(wpts);
        }
    }
    catch(int e) {
        if(dev == 0) return;
        QMessageBox::warning(0,tr("Device Link Error"),dev->getLastError().c_str(),QMessageBox::Ok,QMessageBox::NoButton);
        if(e == Garmin::errSync) {
            gpResources->resetDevice();
        }
    }
}


void CGarminDBWpt::loadGPX(CGpx& gpx)
{
    const QDomNodeList& wpts = gpx.elementsByTagName("wpt");
    uint N = wpts.count();
    for(uint n = 0; n < N; ++n) {
        const QDomNode& wpt = wpts.item(n);

        CGarminWpt * w = new CGarminWpt(this);
        connect(w,SIGNAL(sigSelected(CGarminWpt*)),this,SLOT(slotSelectWpt(CGarminWpt*)));

        const QDomNamedNodeMap& attr = wpt.attributes();
        w->lon = attr.namedItem("lon").nodeValue().toDouble();
        w->lat = attr.namedItem("lat").nodeValue().toDouble();
        if(wpt.namedItem("name").isElement()) {
            w->ident = wpt.namedItem("name").toElement().text();
        }
        if(wpt.namedItem("cmt").isElement()) {
            w->comment = wpt.namedItem("cmt").toElement().text();
        }
        if(wpt.namedItem("desc").isElement()) {
            w->comment = wpt.namedItem("desc").toElement().text();
        }
        if(wpt.namedItem("link").isElement()) {
            const QDomNode& link = wpt.namedItem("link");
            const QDomNamedNodeMap& attr = link.toElement().attributes();
            w->link = attr.namedItem("href").nodeValue();
        }
        if(wpt.namedItem("url").isElement()) {
            w->link = wpt.namedItem("url").toElement().text();
        }
        if(wpt.namedItem("sym").isElement()) {
            getWptIconByName(wpt.namedItem("sym").toElement().text(),w->smbl,w->icon);
        }
        if(wpt.namedItem("ele").isElement()) {
            double ele = wpt.namedItem("ele").toElement().text().toDouble();
            if(ele < 0) {
                w->dpth = -ele;
            }
            else {
                w->alt = ele;
            }
        }

        if(wpt.namedItem("time").isElement()) {
            QDateTime time = QDateTime::fromString(wpt.namedItem("time").toElement().text(),"yyyy-MM-dd'T'hh:mm:ss'Z'");
            time.setTimeSpec(Qt::UTC);
            w->time = time.toTime_t() - gpResources->getUTCOffset();
        }

        if(wpt.namedItem("extension").isElement()) {
            const QDomNode& ext = wpt.namedItem("extension");
            if(ext.namedItem("dist").isElement()) {
                w->dist = ext.namedItem("dist").toElement().text().toDouble();
            }
        }

        if(w->lat == 1000 || w->lon == 1000 || w->ident.isEmpty()) {
            delete w;
            continue;
        }
        w->key = "gpx|" + w->ident;
        waypoints[w->key] = w;
    }

    emit sigWaypointListChanged();
}


void CGarminDBWpt::saveGPX(CGpx& gpx)
{
    QDomElement root = gpx.documentElement();
    QMap<QString,CGarminWpt*>::const_iterator w = waypoints.begin();
    while(w != waypoints.end()) {
        if((*w)->sticky) {
            ++w;
            continue;
        }
        QDomElement wpt = gpx.createElement("wpt");
        root.appendChild(wpt);
        wpt.setAttribute("lat",QString::number((*w)->lat,'f',6));
        wpt.setAttribute("lon",QString::number((*w)->lon,'f',6));

        if((*w)->alt != 1e25f || (*w)->dpth != 1e25f) {
            QDomElement ele = gpx.createElement("ele");
            wpt.appendChild(ele);
            QDomText _ele_ = gpx.createTextNode(QString::number(((*w)->alt != 1e25f) ? (*w)->alt : -(*w)->dpth));
            ele.appendChild(_ele_);
        }

        if((*w)->time != 0x000000000 && (*w)->time != 0xFFFFFFFF) {
            QDateTime t = QDateTime::fromTime_t((*w)->time + gpResources->getUTCOffset()).toUTC();
            QDomElement time = gpx.createElement("time");
            wpt.appendChild(time);
            QDomText _time_ = gpx.createTextNode(t.toString("yyyy-MM-dd'T'hh:mm:ss'Z'"));
            time.appendChild(_time_);
        }

        QDomElement name = gpx.createElement("name");
        wpt.appendChild(name);
        QDomText _name_ = gpx.createTextNode((*w)->ident);
        name.appendChild(_name_);

        if(!(*w)->comment.isEmpty()) {
            QDomElement cmt = gpx.createElement("cmt");
            wpt.appendChild(cmt);
            QDomText _cmt_ = gpx.createTextNode((*w)->comment);
            cmt.appendChild(_cmt_);
        }

        if(!(*w)->link.isEmpty()) {
            QDomElement link = gpx.createElement("link");
            wpt.appendChild(link);
            link.setAttribute("href",(*w)->link);
            QDomElement text = gpx.createElement("text");
            link.appendChild(text);
            QDomText _text_ = gpx.createTextNode((*w)->ident);
            text.appendChild(_text_);
        }

        QPixmap icon;
        QString str;
        getWptIconById((*w)->smbl,icon,str);
        QDomElement sym = gpx.createElement("sym");
        wpt.appendChild(sym);
        QDomText _sym_ = gpx.createTextNode(str);
        sym.appendChild(_sym_);

        if((*w)->dist != 1e25f) {
            QDomElement extension = gpx.createElement("extension");
            wpt.appendChild(extension);

            if((*w)->dist != 1e25f) {
                QDomElement dist = gpx.createElement("dist");
                extension.appendChild(dist);
                QDomText _dist_ = gpx.createTextNode(QString::number((*w)->dist));
                dist.appendChild(_dist_);
            }
        }

        ++w;
    }
}


void CGarminDBWpt::clear()
{
    QList<QString> keys = waypoints.keys();
    QString key;
    foreach(key,keys) {
        if(waypoints[key]->sticky) continue;
        waypoints.remove(key);
    }

    emit sigWaypointListChanged();
}


void CGarminDBWpt::slotSelectWpt(CGarminWpt * wpt)
{
    CGarminWpt * w;
    foreach(w,waypoints) {
        w->selected = (wpt == w);
    }
    emit sigWaypointListChanged();
}


// ############ Network queries ############ //
void CGarminDBWpt::queryOpenChachingDe(double lon, double lat, double radius)
{
    QString proxy;
    quint16 port;
    bool enableProxy;

    rcvData = false;

    center.u = lon * DEG_TO_RAD;
    center.v = lat * DEG_TO_RAD;
    this->radius  = radius;

    QUrl url;
    url.setPath("/xml/ocxml11.php");
    url.addQueryItem("modifiedsince","20000101000000");
    url.addQueryItem("cache","1");
    url.addQueryItem("lat",QString::number(lat));
    url.addQueryItem("lon",QString::number(lon));
    url.addQueryItem("distance",QString::number(radius/1000.0));
    url.addQueryItem("zip","gzip");
    url.addQueryItem("session","0");

    delete progressDlg;

    progressDlg = new QProgressDialog(tr("Request waypoints from www.opencaching.de ..."),0,0,0,toolview);
    progressDlg->setAttribute(Qt::WA_DeleteOnClose,true);
    progressDlg->setWindowTitle("Query waypoints...");
    progressDlg->setAutoClose(false);
    progressDlg->show();

    opencaching = new QHttp(progressDlg);
    opencaching->setHost("www.opencaching.de");

    enableProxy = gpResources->getHttpProxy(proxy,port);
    if(enableProxy) {
        opencaching->setProxy(proxy,port);
    }
    connect(opencaching,SIGNAL(requestStarted(int)),this,SLOT(slotRequestStarted(int)));
    connect(opencaching,SIGNAL(requestFinished(int,bool)),this,SLOT(slotRequestFinished(int,bool)));
    connect(opencaching,SIGNAL(dataReadProgress (int,int)),this,SLOT(slotDataReadProgress(int,int)));

    opencaching->get(url.toEncoded( ));

}


void CGarminDBWpt::slotRequestStarted(int )
{

}


void CGarminDBWpt::slotDataReadProgress(int done, int total)
{
    progressDlg->setRange(0,total);
    progressDlg->setValue(done);
}


void CGarminDBWpt::slotRequestFinished(int , bool error)
{
    try
    {
        if(error) {
            throw opencaching->errorString();
        }

        QHttpResponseHeader resp =  opencaching->lastResponse();
        if(!resp.isValid()) return;

        if(resp.statusCode() == 302) {
            rcvData = true;
            progressDlg->setLabelText(tr("Download waypoints from www.opencaching.de ..."));
            opencaching->get(resp.value("Location"));
            return;
        }
        else if(resp.statusCode() == 200 && rcvData) {
            QDir tmp = QDir::temp();

            QFile::remove(tmp.absoluteFilePath("wpt"));

            QByteArray data = opencaching->readAll();

            QFile f1(tmp.absoluteFilePath("wpt.gz"));
            f1.open(QIODevice::WriteOnly);
            f1.write(data);
            f1.close();

            QString cmd("gunzip ");
            cmd += tmp.absoluteFilePath("wpt.gz");
            system(cmd.toLatin1());

            QFile f2(tmp.absoluteFilePath("wpt"));
            QDomDocument xml;

            if(!f2.open(QIODevice::ReadOnly)) {
                throw tr("Failed to read cache data (code 1)");
            }
            if(!xml.setContent(&f2)) {
                f2.close();
                throw tr("Failed to read cache data (code 2)");
            }
            f2.close();
            //QFile::remove(tmp.absoluteFilePath("wpt"));

            const  QDomElement& docElem = xml.documentElement();
            if(docElem.tagName() != "oc11xml") {
                throw tr("Failed to read cache data (code 3)");
            }
            const QDomNodeList& caches = xml.elementsByTagName("cache");

            uint N = caches.count();
            for(uint n = 0; n < N; ++n) {
                const QDomNode& cache = caches.item(n);

                CGarminWpt * wpt = new CGarminWpt(this);
                connect(wpt,SIGNAL(sigSelected(CGarminWpt*)),this,SLOT(slotSelectWpt(CGarminWpt*)));

                double lon = cache.namedItem("longitude").toElement().text().toDouble();
                double lat = cache.namedItem("latitude").toElement().text().toDouble();

                XY pt;  pt.u = lon * DEG_TO_RAD;  pt.v = lat * DEG_TO_RAD;
                double a1 = 0, a2 = 0;
                if(gpProj->distance(center,pt,a1,a2) > radius) continue;

                wpt->lon = lon; wpt->lat = lat;
                const QDomElement& wpts = cache.namedItem("waypoints").toElement();
                const QDomNamedNodeMap& attr = wpts.attributes();
                wpt->ident   = attr.namedItem("oc").nodeValue();
                wpt->comment = cache.namedItem("name").toElement().text();

                int typeId = cache.namedItem("type").toElement().attributes().namedItem("id").nodeValue().toInt();
                getWptIconByName(cacheIcons.value(typeId,type_t("Unknown","Flag, Blue")).icon,wpt->smbl,wpt->icon);
                //                 if(typeId == 2){ // traditional
                //                     getWptIconByName("Geocache",wpt->smbl,wpt->icon);
                //                 }
                //                 else if(typeId == 3){ // multi
                //                     getWptIconByName("Block, Blue",wpt->smbl,wpt->icon);
                //                 }
                //                 else if(typeId == 7){ // quest
                //                     getWptIconByName("Block, Red",wpt->smbl,wpt->icon);
                //                 }
                //                 else if(typeId == 10){ // drive-in
                //                     getWptIconByName("Car",wpt->smbl,wpt->icon);
                //                 }
                //                 else{ // other
                //                     getWptIconByName("Flag, Blue",wpt->smbl,wpt->icon);
                //                 }
                wpt->link = "http://www.opencaching.de/viewcache.php?wp=" + wpt->ident;

                wpt->key = "oc.de|" + wpt->ident;
                waypoints[wpt->key] = wpt;
            }

            progressDlg->deleteLater();
            emit sigWaypointListChanged();
        }
        else if(resp.statusCode() == 200) {
            QString msg = opencaching->readAll();
            throw msg;
        }
        else {
            QString msg = opencaching->readAll();
            if(!msg.isEmpty()) {
                throw msg;
            }
            else {
                msg = tr("Error: ") + QString::number(resp.statusCode());
                throw msg;
            }
        }
    }
    catch(const QString& msg) {
        opencaching->close();
        if(progressDlg && !msg.isEmpty()) {
            progressDlg->setLabelText(msg);

            QPushButton * cancel = new QPushButton(tr("Cancel"),progressDlg);
            connect(cancel,SIGNAL(pressed()),progressDlg,SLOT(close()));

            progressDlg->setCancelButton(cancel);
        }
        return;
    }

}


void CGarminDBWpt::setProxyDistance(const QStringList& keys, double dist)
{
    QString key;
    foreach(key,keys) {
        waypoints[key]->dist = dist;
    }

    emit sigWaypointListChanged();
}
