/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de, Michael Ritzert
    michael.ritzert@googlemail.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/

#include "CFunctionStateMachine.h"
#include "icons.h"
#include "CCentralResources.h"
#include "CCanvas.h"
#include "CGarminDBMap.h"
#include "CGarminDBWpt.h"
#include "CGarminDBTrack.h"
#include "CGarminDBRoute.h"
#include "CGarminRoute.h"
#include "CGarminLiveLog.h"

#include <QtGui>

class CLabel : public QLabel
{
    public:
        CLabel(QWidget * parent):QLabel(parent){setMouseTracking(true);}
        ~CLabel(){}

        void mouseMoveEvent(QMouseEvent * /*e*/) {
        setBackgroundRole(QPalette::Highlight);
        setForegroundRole(QPalette::HighlightedText);
}


void leaveEvent(QEvent *) {
    setBackgroundRole(QPalette::Window);
    setForegroundRole(QPalette::WindowText);
}


};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsMain[] =
{
    {0,QObject::tr("-"),0,tr("")}
    ,{iconMap16x16,QObject::tr("Map ..."),&CFunctionStateMachine::funcSwitchToMap,tr("Manage maps.")}
    ,{iconWaypoint16x16,QObject::tr("Waypoints ..."),&CFunctionStateMachine::funcSwitchToWaypoint,tr("Manage waypoints.")}
    ,{iconTrack16x16,QObject::tr("Tracks ..."),&CFunctionStateMachine::funcSwitchToTrack,tr("Manage tracks.")}
    ,{iconRoute16x16,QObject::tr("Routes ..."),&CFunctionStateMachine::funcSwitchToRoute,tr("Manage routes.")}
    ,{iconLiveLog16x16,QObject::tr("Live Log ..."),&CFunctionStateMachine::funcSwitchToLiveLog,tr("Record position data from the GPSr")}
    ,{iconRuler16x16,QObject::tr("Distance ..."),&CFunctionStateMachine::funcSwitchToDistance,tr("Measure the distance between points.")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{iconTrash16x16,QObject::tr("Clear all"),&CFunctionStateMachine::funcClearAll,tr("Delete all selected maps, waypoints and so on.")}
    ,{iconUpload16x16,QObject::tr("Upload All"),&CFunctionStateMachine::funcUpload,tr("Upload all data to device.")}
    ,{iconDownload16x16,QObject::tr("Download All"),&CFunctionStateMachine::funcDownload,tr("Download all data from device.")}
};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsMap[] =
{
    {iconBack16x16,QObject::tr("Back"),&CFunctionStateMachine::funcSwitchToMain,tr("Go back to main menu.")}
    ,{iconMove16x16,QObject::tr("Move Map"),&CFunctionStateMachine::funcMoveArea,tr("Move the map.\nPress down the left mouse button and move the mouse.")}
    ,{iconZoomArea16x16,QObject::tr("Zoom Area"),&CFunctionStateMachine::funcZoomArea,tr("Zoom selected area.\nPress down the left mouse button and select area.\nThis will switch back to 'Move' as soon as you have zoomed.")}
    ,{iconZoomWorld16x16,QObject::tr("Zoom World"),&CFunctionStateMachine::funcZoomWorld,tr("Zoom out to make the complete map fit to screen.")}
    ,{iconInfo16x16,QObject::tr("Show Info"),&CFunctionStateMachine::funcToggleInfo,tr("Show information about elements close to cursor.")}
    ,{iconSelect16x16,QObject::tr("Select Map"),&CFunctionStateMachine::funcSelectArea,tr("Select / deselect map tiles for upload.\nBy point-click-n-move you can select an area of map tiles.\nSelected tiles will be deselected and vice versa.")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{iconUpload16x16,QObject::tr("Upload"),&CFunctionStateMachine::funcUploadMaps,tr("Upload maps to device.")}
    ,{iconDownload16x16,QObject::tr("Download"),&CFunctionStateMachine::funcDownloadMaps,tr("Download map information from device.\nNOTE: This is not downloading the maps themselves.")}
};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsWaypoint[] =
{
    {iconBack16x16,QObject::tr("Back"),&CFunctionStateMachine::funcSwitchToMain,tr("Go back to main menu.")}
    ,{iconMove16x16,QObject::tr("Move Map"),&CFunctionStateMachine::funcMoveArea,tr("Move the map.\nPress down the left mouse button and move the mouse.")}
    ,{iconZoomArea16x16,QObject::tr("Zoom Area"),&CFunctionStateMachine::funcZoomArea,tr("Zoom selected area.\nPress down the left mouse button and select area.\nThis will switch back to 'Move' as soon as you have zoomed.")}
    ,{iconZoomWorld16x16,QObject::tr("Zoom World"),&CFunctionStateMachine::funcZoomWorld,tr("Zoom out to make the complete map fit to screen.")}
    ,{iconInfo16x16,QObject::tr("Show Info"),&CFunctionStateMachine::funcToggleInfo,tr("Show information about elements close to cursor.")}
    ,{iconNewWpt16x16,QObject::tr("New Waypoint"),&CFunctionStateMachine::funcNewWpt,tr("Create a new user waypoint.\nThe default position will be the\ncurrent cursor position.")}
    ,{iconEditWpt16x16,QObject::tr("Edit Waypoint"),&CFunctionStateMachine::funcEditWpt,tr("Edit existing waypoint.\nClick on the waypoint on the map.")}
    ,{iconMoveWpt16x16,QObject::tr("Move Waypoint"),&CFunctionStateMachine::funcMoveWpt,tr("Move existing waypoint.\nDrag-and-drop the waypoint on the map.")}
    ,{iconMore16x16,QObject::tr("More ..."),&CFunctionStateMachine::funcSwitchToWptMore,tr("Simply more stuff on waypoints.")}
    ,{iconUpload16x16,QObject::tr("Upload"),&CFunctionStateMachine::funcUploadWpt,tr("Upload waypoints to device.")}
    ,{iconDownload16x16,QObject::tr("Download"),&CFunctionStateMachine::funcDownloadWpt,tr("Download waypoints from device.")}
};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsWptMore[] =
{
    {iconBack16x16,QObject::tr("Back"),&CFunctionStateMachine::funcSwitchToWaypoint,tr("Go back to waypoint menu.")}
    ,{iconWWWSearch16x16,QObject::tr("Search oc.de"),&CFunctionStateMachine::funcSearchOCde,tr("Search opencaching.de for caches within a given radius. Simply define a circle with your mouse on the map.")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{iconEditWpt16x16,QObject::tr("Edit Waypoint"),&CFunctionStateMachine::funcEditWpt,tr("Edit existing waypoint.\nClick on the waypoint on the map.")}
    ,{iconTrash16x16,QObject::tr("Delete By ..."),&CFunctionStateMachine::funcDeleteByWpt,tr("Delete waypoints by type and/or source.")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsTrack[] =
{
    {iconBack16x16,QObject::tr("Back"),&CFunctionStateMachine::funcSwitchToMain,tr("Go back to main menu.")}
    ,{iconMove16x16,QObject::tr("Move Map"),&CFunctionStateMachine::funcMoveArea,tr("Move the map.\nPress down the left mouse button and move the mouse.")}
    ,{iconZoomArea16x16,QObject::tr("Zoom Area"),&CFunctionStateMachine::funcZoomArea,tr("Zoom selected area.\nPress down the left mouse button and select area.\nThis will switch back to 'Move' as soon as you have zoomed.")}
    ,{iconZoomWorld16x16,QObject::tr("Zoom World"),&CFunctionStateMachine::funcZoomWorld,tr("Zoom out to make the complete map fit to screen.")}
    ,{iconInfo16x16,QObject::tr("Show Info"),&CFunctionStateMachine::funcToggleInfo,tr("Show information about elements close to cursor.")}
    ,{iconInfoII16x16,QObject::tr("Track Info"),&CFunctionStateMachine::funcTrackInfo,tr("Show all information stored in the selected track log.")}
    ,{iconEdit16x16,QObject::tr("Combine Tracks"),&CFunctionStateMachine::funcTrackCombine,tr("Combine several selected tracks into a new one.")}
    ,{iconCut16x16,QObject::tr("Split Track"),&CFunctionStateMachine::funcTrackCut,tr("Split a selected track into two new tracks.")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,                            //,{iconUpload16x16,QObject::tr("Upload"),0,tr("Upload track data to device.")}
    {
        0,QObject::tr("-"),0,tr("")
    }
    ,{iconDownload16x16,QObject::tr("Download"),&CFunctionStateMachine::funcDownloadTrack,tr("Download track data from device.")}
};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsLiveLog[] =
{
    {iconBack16x16,QObject::tr("Back"),&CFunctionStateMachine::funcSwitchToMain,tr("Go back to main menu.")}
    ,{iconMove16x16,QObject::tr("Move Map"),&CFunctionStateMachine::funcMoveArea,tr("Move the map.\nPress down the left mouse button and move the mouse.")}
    ,{iconZoomArea16x16,QObject::tr("Zoom Area"),&CFunctionStateMachine::funcZoomArea,tr("Zoom selected area.\nPress down the left mouse button and select area.\nThis will switch back to 'Move' as soon as you have zoomed.")}
    ,{iconZoomWorld16x16,QObject::tr("Zoom World"),&CFunctionStateMachine::funcZoomWorld,tr("Zoom out to make the complete map fit to screen.")}
    ,{iconInfo16x16,QObject::tr("Show Info"),&CFunctionStateMachine::funcToggleInfo,tr("Show information about elements close to cursor.")}
    ,{iconRecord16x16,QObject::tr("Toggle Log"),&CFunctionStateMachine::funcToggleLog,tr("Switch live log on & off")}
    ,{iconLock16x16,QObject::tr("Toggle Lock-to-center"),&CFunctionStateMachine::funcToggleLock,tr("Toggle if map center is locked to actual position")}
    ,{iconNewWpt16x16,QObject::tr("Add Waypoint"),&CFunctionStateMachine::funcNewWptHere,tr("Create a new user waypoint.\nThe default position will be the\ncurrent live cursor position.")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsDistance[] =
{
    {iconBack16x16,QObject::tr("Back"),&CFunctionStateMachine::funcSwitchToMain,tr("Go back to main menu.")}
    ,{iconMove16x16,QObject::tr("Move Map"),&CFunctionStateMachine::funcMoveArea,tr("Move the map.\nPress down the left mouse button and move the mouse.")}
    ,{iconZoomArea16x16,QObject::tr("Zoom Area"),&CFunctionStateMachine::funcZoomArea,tr("Zoom selected area.\nPress down the left mouse button and select area.\nThis will switch back to 'Move' as soon as you have zoomed.")}
    ,{iconZoomWorld16x16,QObject::tr("Zoom World"),&CFunctionStateMachine::funcZoomWorld,tr("Zoom out to make the complete map fit to screen.")}
    ,{iconInfo16x16,QObject::tr("Show Info"),&CFunctionStateMachine::funcToggleInfo,tr("Show information about elements close to cursor.")}
    ,{iconNewWpt16x16,QObject::tr("New Polyline"),&CFunctionStateMachine::funcDistNew,tr("Add a new polygon for distance measurement.")}
    ,{iconEdit16x16,QObject::tr("Tune Polyline"),&CFunctionStateMachine::funcDistEdit,tr("Edit current selected distance measurement")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsDistEdit[] =
{
    {iconBack16x16,QObject::tr("Back"),&CFunctionStateMachine::funcSwitchToDistance,tr("Go back to distance menu.")}
    ,{iconMove16x16,QObject::tr("Move Map"),&CFunctionStateMachine::funcMoveArea,tr("Move the map.\nPress down the left mouse button and move the mouse.")}
    ,{iconZoomArea16x16,QObject::tr("Zoom Area"),&CFunctionStateMachine::funcZoomArea,tr("Zoom selected area.\nPress down the left mouse button and select area.\nThis will switch back to 'Move' as soon as you have zoomed.")}
    ,{iconZoomWorld16x16,QObject::tr("Zoom World"),&CFunctionStateMachine::funcZoomWorld,tr("Zoom out to make the complete map fit to screen.")}
    ,{iconInfo16x16,QObject::tr("Show Info"),&CFunctionStateMachine::funcToggleInfo,tr("Show information about elements close to cursor.")}
    ,{iconNewWpt16x16,QObject::tr("Add Points"),&CFunctionStateMachine::funcEditRouteAdd,tr("Append the current distance polyline by addition points.")}
    ,{iconMoveWpt16x16,QObject::tr("Move Points"),&CFunctionStateMachine::funcEditRouteMove,tr("Move selected distance points.")}
    ,{iconTrash16x16,QObject::tr("Delete Points"),&CFunctionStateMachine::funcEditRouteDel,tr("Delete selected distance points.")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsRoute[] =
{
    {iconBack16x16,QObject::tr("Back"),&CFunctionStateMachine::funcSwitchToMain,tr("Go back to main menu.")}
    ,{iconMove16x16,QObject::tr("Move Map"),&CFunctionStateMachine::funcMoveArea,tr("Move the map.\nPress down the left mouse button and move the mouse.")}
    ,{iconZoomArea16x16,QObject::tr("Zoom Area"),&CFunctionStateMachine::funcZoomArea,tr("Zoom selected area.\nPress down the left mouse button and select area.\nThis will switch back to 'Move' as soon as you have zoomed.")}
    ,{iconZoomWorld16x16,QObject::tr("Zoom World"),&CFunctionStateMachine::funcZoomWorld,tr("Zoom out to make the complete map fit to screen.")}
    ,{iconInfo16x16,QObject::tr("Show Info"),&CFunctionStateMachine::funcToggleInfo,tr("Show information about elements close to cursor.")}
    ,{iconNewWpt16x16,QObject::tr("New Route"),&CFunctionStateMachine::funcRouteNew,tr("Add a new route polyline.")}
    ,{iconEdit16x16,QObject::tr("Tune Route"),&CFunctionStateMachine::funcRouteEdit,tr("Edit current selected route.")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{iconUpload16x16,QObject::tr("Upload"),&CFunctionStateMachine::funcRouteUpload,tr("Send routes to device.")}
    ,                            //,{iconDownload16x16,QObject::tr("Download"),0,tr("No function yet")}
    {
        0,QObject::tr("-"),0,tr("")
    }
};

const CFunctionStateMachine::func_key_state_t CFunctionStateMachine::fsRouteEdit[] =
{
    {iconBack16x16,QObject::tr("Back"),&CFunctionStateMachine::funcSwitchToRoute,tr("Go back to route menu.")}
    ,{iconMove16x16,QObject::tr("Move Map"),&CFunctionStateMachine::funcMoveArea,tr("Move the map.\nPress down the left mouse button and move the mouse.")}
    ,{iconZoomArea16x16,QObject::tr("Zoom Area"),&CFunctionStateMachine::funcZoomArea,tr("Zoom selected area.\nPress down the left mouse button and select area.\nThis will switch back to 'Move' as soon as you have zoomed.")}
    ,{iconZoomWorld16x16,QObject::tr("Zoom World"),&CFunctionStateMachine::funcZoomWorld,tr("Zoom out to make the complete map fit to screen.")}
    ,{iconInfo16x16,QObject::tr("Show Info"),&CFunctionStateMachine::funcToggleInfo,tr("Show information about elements close to cursor.")}
    ,{iconNewWpt16x16,QObject::tr("Add Points"),&CFunctionStateMachine::funcEditRouteAdd,tr("Append the current rote by addition points.")}
    ,{iconMoveWpt16x16,QObject::tr("Move Points"),&CFunctionStateMachine::funcEditRouteMove,tr("Move selected route points.")}
    ,{iconTrash16x16,QObject::tr("Delete Points"),&CFunctionStateMachine::funcEditRouteDel,tr("Delete selected route points.")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
    ,{0,QObject::tr("-"),0,tr("")}
};

CFunctionStateMachine::CFunctionStateMachine(QWidget& parent)
: QWidget(&parent)
{
    unsigned i;

    setMouseTracking(true);

    QVBoxLayout * mainLayout = new QVBoxLayout(this);

    QHBoxLayout * titleLayout = new QHBoxLayout();
    mainLayout->addLayout(titleLayout);

    menuIcon = new QLabel(this);
    menuIcon->setGeometry(QRect(0,0,16,16));
    menuIcon->setSizePolicy(QSizePolicy::Maximum,QSizePolicy::Maximum);
    titleLayout->addWidget(menuIcon);

    menuTitle = new QLabel(this);
    menuTitle->setAlignment(Qt::AlignCenter);
    titleLayout->addWidget(menuTitle);

    layout = new QGridLayout();
    mainLayout->addLayout(layout);

    keyEsc   = new QLabel("Esc",this);
    layout->addWidget(keyEsc,0,0);
    keyF1   = new QLabel("F1",this);
    layout->addWidget(keyF1,1,0);
    keyF2 = new QLabel("F2",this);
    layout->addWidget(keyF2,2,0);
    keyF3 = new QLabel("F3",this);
    layout->addWidget(keyF3,3,0);
    keyF4 = new QLabel("F4",this);
    layout->addWidget(keyF4,4,0);
    keyF5 = new QLabel("F5",this);
    layout->addWidget(keyF5,5,0);
    keyF6 = new QLabel("F6",this);
    layout->addWidget(keyF6,6,0);
    keyF7 = new QLabel("F7",this);
    layout->addWidget(keyF7,7,0);
    keyF8 = new QLabel("F8",this);
    layout->addWidget(keyF8,8,0);
    keyF9 = new QLabel("F9",this);
    layout->addWidget(keyF9,9,0);
    keyF10 = new QLabel("F10",this);
    layout->addWidget(keyF10,10,0);

    for(i=0; i<11; ++i) {
        icons[i] = new QLabel(this);
        icons[i]->setGeometry(QRect(0,0,16,16));
        layout->addWidget(icons[i],i,1);
    }

    for(i=0; i<11; ++i) {
        names[i] = new CLabel(this);
        names[i]->setAutoFillBackground(true);
        names[i]->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
        layout->addWidget(names[i],i,2);
    }

    funcSwitchToMain();

}


CFunctionStateMachine::~CFunctionStateMachine()
{

}


CFunctionStateMachine::mode_e CFunctionStateMachine::getMode()
{

    if(current == fsMain)
        return eMain;
    if(current == fsMap)
        return eMap;
    if(current == fsWaypoint || current == fsWptMore)
        return eWaypoint;
    if(current == fsTrack)
        return eTrack;

    return eMain;
}


void CFunctionStateMachine::switchByKeyWord(const QString& key)
{
    if(key == "Maps" && current != fsMap) {
        funcSwitchToMap();
        funcMoveArea();
    }
    else if(key == "Waypoints" && current != fsWaypoint) {
        funcSwitchToWaypoint();
        funcMoveArea();
    }
    else if(key == "Tracks" && current != fsTrack) {
        funcSwitchToTrack();
        funcMoveArea();
    }
    else if(key == "Log" && current != fsLiveLog) {
        funcSwitchToLiveLog();
        funcMoveArea();
    }
    else if(key == "Distance" && current != fsDistance) {
        funcSwitchToDistance();
        funcMoveArea();
    }
    else if(key == "Route" && current != fsRoute) {
        funcSwitchToRoute();
        funcMoveArea();
    }
    else if(key == "Google" && current != fsMain) {
        funcSwitchToMain();
        funcMoveArea();
    }
}


void CFunctionStateMachine::switchState(const func_key_state_t statedef[11])
{
    unsigned i;

    for(i=0; i<11; ++i) {
        if(statedef[i].icon) {
            icons[i]->setPixmap(QPixmap(statedef[i].icon));
        }
        else {
            icons[i]->setPixmap(QPixmap());
        }
        names[i]->setText(statedef[i].name);
        names[i]->setToolTip(statedef[i].tooltip);
    }

    current = statedef;

    gpResources->canvas().update();
}


void CFunctionStateMachine::keyPressEvent(QKeyEvent * e)
{

    if((e->key() >= Qt::Key_F1) && (e->key() < Qt::Key_F11)) {
        unsigned i = e->key() - Qt::Key_F1 + 1;
        if(current[i].func) {
            (this->*current[i].func)();

        }
        return e->accept();
    }
    else if(e->key() == Qt::Key_Escape) {
        if(current[0].func) {
            (this->*current[0].func)();
        }
        return e->accept();
    }
    else if(e->key() == Qt::Key_Plus) {
        gpResources->canvas().zoom(true, gpResources->canvas().geometry().center());
        return e->accept();
    }
    else if(e->key() == Qt::Key_Minus) {
        gpResources->canvas().zoom(false, gpResources->canvas().geometry().center());
        return e->accept();
    }
    else if(e->key() == Qt::Key_Left) {
        gpResources->canvas().move(CCanvas::eMoveLeft);
        return e->accept();
    }
    else if(e->key() == Qt::Key_Right) {
        gpResources->canvas().move(CCanvas::eMoveRight);
        return e->accept();
    }
    else if(e->key() == Qt::Key_Up) {
        gpResources->canvas().move(CCanvas::eMoveUp);
        return e->accept();
    }
    else if(e->key() == Qt::Key_Down) {
        gpResources->canvas().move(CCanvas::eMoveDown);
        return e->accept();
    }
}


void CFunctionStateMachine::mousePressEvent(QMouseEvent * e)
{
    unsigned i;

    if(e->button() != Qt::LeftButton) return;

    for(i=0; i<11; ++i) {
        if(names[i]->geometry().contains(e->pos())) {
            if(current[i].func) {
                (this->*current[i].func)();
            }
            return;
        }
    }
}


void CFunctionStateMachine::funcSwitchToDistEdit()
{
    menuTitle->setText(tr("<b>Distance Edit...</b>"));
    menuIcon->setPixmap(iconRuler16x16);
    switchState(fsDistEdit);
    gpResources->routedb().gainFocus("Distance");
}


void CFunctionStateMachine::funcSwitchToRouteEdit()
{
    menuTitle->setText(tr("<b>Route Edit...</b>"));
    menuIcon->setPixmap(iconRoute16x16);
    switchState(fsRouteEdit);
    gpResources->routedb().gainFocus("Routes");
}


void CFunctionStateMachine::funcSwitchToMap()
{
    menuTitle->setText(tr("<b>Maps ...</b>"));
    menuIcon->setPixmap(iconMap16x16);
    switchState(fsMap);
    gpResources->mapdb().gainFocus();
    funcMoveArea();
}


void CFunctionStateMachine::funcSwitchToWaypoint()
{
    menuTitle->setText(tr("<b>Waypoints ...</b>"));
    menuIcon->setPixmap(iconWaypoint16x16);
    gpResources->wptdb().gainFocus();
    switchState(fsWaypoint);
    funcMoveArea();
}


void CFunctionStateMachine::funcSwitchToWptMore()
{
    menuTitle->setText(tr("<b>Waypoints More ...</b>"));
    menuIcon->setPixmap(iconWaypoint16x16);
    gpResources->wptdb().gainFocus();
    switchState(fsWptMore);
    funcMoveArea();
}


void CFunctionStateMachine::funcSwitchToTrack()
{
    menuTitle->setText(tr("<b>Tracks ...</b>"));
    menuIcon->setPixmap(iconTrack16x16);
    gpResources->trackdb().gainFocus();
    switchState(fsTrack);
    funcMoveArea();
}


void CFunctionStateMachine::funcSwitchToLiveLog()
{
    menuTitle->setText(tr("<b>Live Log ...</b>"));
    menuIcon->setPixmap(iconLiveLog16x16);
    gpResources->livelog().gainFocus();
    switchState(fsLiveLog);
    funcMoveArea();
}


void CFunctionStateMachine::funcSwitchToDistance()
{
    menuTitle->setText(tr("<b>Distance ...</b>"));
    menuIcon->setPixmap(iconRuler16x16);
    switchState(fsDistance);
    gpResources->routedb().gainFocus("Distance");
    funcMoveArea();
}


void CFunctionStateMachine::funcSwitchToRoute()
{
    menuTitle->setText(tr("<b>Routes ...</b>"));
    menuIcon->setPixmap(iconRoute16x16);
    switchState(fsRoute);
    gpResources->routedb().gainFocus("Route");
    funcMoveArea();
}


void CFunctionStateMachine::funcSwitchToMain()
{
    menuTitle->setText(tr("<b>Main ...</b>"));
    menuIcon->setPixmap(QPixmap(":/icons/QLandkarte16x16.png").scaled(16,16, Qt::IgnoreAspectRatio, Qt::SmoothTransformation));
    switchState(fsMain);
    funcMoveArea();
}


void CFunctionStateMachine::funcClearAll()
{
    int res = QMessageBox::question(0,tr("Delete all data?"),tr("Do you really want to delete all user data?"),QMessageBox::Yes|QMessageBox::No,QMessageBox::Yes);
    if(res == QMessageBox::Yes) {
        gpResources->mapdb().clear();
        gpResources->wptdb().clear();
        gpResources->trackdb().clear();
        gpResources->routedb().clear();
    }
}


void CFunctionStateMachine::funcUpload()
{
    gpResources->upload();
}


void CFunctionStateMachine::funcDownload()
{
    gpResources->download();
}


void CFunctionStateMachine::funcZoomArea()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseZoomArea);
}


void CFunctionStateMachine::funcMoveArea()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseMoveArea);
}


void CFunctionStateMachine::funcZoomWorld()
{
    gpResources->canvas().fitMapToScreen();
}


void CFunctionStateMachine::funcToggleInfo()
{
    gpResources->canvas().toggleInfo();
}


void CFunctionStateMachine::funcSelectArea()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseSelectArea);
}


void CFunctionStateMachine::funcUploadMaps()
{
    gpResources->mapdb().uploadSelectedTiles(false);
}


void CFunctionStateMachine::funcDownloadMaps()
{
    gpResources->mapdb().downloadMapInfo();
}


void CFunctionStateMachine::funcNewWpt()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseNewWpt);
}


void CFunctionStateMachine::funcNewWptHere()
{
    CGarminLiveLog::Pvt_t * pos = gpResources->livelog().getPosition();
    if(!pos) return;             // live log not active
    gpResources->wptdb().newWaypoint(pos-> lon, pos->lat, pos->alt + pos->msl_hght);
}


void CFunctionStateMachine::funcEditWpt()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseEditWpt);
}


void CFunctionStateMachine::funcMoveWpt()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseMoveWpt);
}


void CFunctionStateMachine::funcUploadWpt()
{
    gpResources->wptdb().uploadWaypoints();
}


void CFunctionStateMachine::funcDownloadWpt()
{
    gpResources->wptdb().downloadWaypoints();
}


void CFunctionStateMachine::funcSearchOCde()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseSearchOC);
}


void CFunctionStateMachine::funcDeleteByWpt()
{
    gpResources->wptdb().removeWaypointBy();
}


void CFunctionStateMachine::funcTrackInfo()
{
    gpResources->trackdb().showTrackInfo();
}


void CFunctionStateMachine::funcTrackCombine()
{
    gpResources->trackdb().combine();
    gpResources->canvas().setMouseMode(CCanvas::eMouseMoveArea);
}


void CFunctionStateMachine::funcTrackCut()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseCutTrack);
}


void CFunctionStateMachine::funcDownloadTrack()
{
    gpResources->trackdb().downloadTracks();
}


void CFunctionStateMachine::funcToggleLog()
{
    gpResources->livelog().toggle();
}


void CFunctionStateMachine::funcToggleLock()
{
    gpResources->livelog().toggleLockToCenter();
}


void CFunctionStateMachine::funcDistNew()
{
    gpResources->routedb().newDistance();
    funcSwitchToDistEdit();
    funcEditRouteAdd();
}


void CFunctionStateMachine::funcDistEdit()
{
    if(gpResources->routedb().getCurrentRoute()) {
        funcSwitchToDistEdit();
        funcMoveArea();
    }
    else {
        QMessageBox::information(0,tr("Edit distance ..."),tr("Please select a distance polyline from the list, first."));
    }
}


void CFunctionStateMachine::funcRouteNew()
{
    gpResources->routedb().newRoute();
    funcSwitchToRouteEdit();
    funcEditRouteAdd();

}


void CFunctionStateMachine::funcRouteEdit()
{
    if(gpResources->routedb().getCurrentRoute()) {
        funcSwitchToRouteEdit();
        funcMoveArea();
    }
    else {
        QMessageBox::information(0,tr("Edit route ..."),tr("Please select a route from the list, first."));
    }
}


void CFunctionStateMachine::funcRouteUpload()
{
    gpResources->routedb().uploadRoutes();
}


void CFunctionStateMachine::funcEditRouteAdd()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseEditRte);
}


void CFunctionStateMachine::funcEditRouteMove()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseMoveRte);
}


void CFunctionStateMachine::funcEditRouteDel()
{
    gpResources->canvas().setMouseMode(CCanvas::eMouseDelRte);
}
