#ifndef __LINUX_QUICKCAM_H
#define __LINUX_QUICKCAM_H

#include <linux/list.h>
#include <linux/usb.h>
#include <linux/videodev.h>

#define _QUICKCAM_ISOPIPE 	0x81
#define SHUTTER_VAL 		0x80
#define EXPO_VAL 		0xa0
#define RGAIN_DEF		0x80
#define BGAIN_DEF		0x80
#define GGAIN_DEF		0x80

// define debug levels
#define DEBUGBASE      		1
#define DEBUGLOGIC      	2
#define DEBUGDATA       	4
#define DEBUGREAD       	8
#define DEBUGIOCTL     		16
#define DEBUGINT		32

#define VID_HARDWARE_QCE 	50

// Control register of the STV0600 ASIC
#define STV_ISO_ENABLE 		0x1440
#define STV_SCAN_RATE  		0x1443

#define STV_ISO_SIZE 		0x15c1
#define STV_Y_CTRL   		0x15c3
#define STV_X_CTRL   		0x1680

#define STV_REG00 		0x1500
#define STV_REG01 		0x1501
#define STV_REG02 		0x1502
#define STV_REG03 		0x1503
#define STV_REG04 		0x1504

#define STV_REG23 		0x0423

#define STREAM_BUF_SIZE		(PAGE_SIZE * 4)

#define SCRATCH_BUF_SIZE	(STREAM_BUF_SIZE * 2)

#define FRAMES_PER_DESC		10
#define FRAME_SIZE_PER_DESC	1023	// Shouldn't be hardcoded JFC was 960
// 356*292 from VV6410 352*288 otherwise.
// max size of frame received from the camera
#define BAYER_FRAME_SIZE	103952

#define QUICKCAM_NUMFRAMES	2
#define QUICKCAM_NUMSBUF	2

/* scanstate */
enum {
	STATE_OK,		// Ok.
	STATE_ERROR,		// An error has been detected.
};

/* store state, that to know what to do with the frame */
enum {
	FRAME_STORE,		// Ok.
	FRAME_SKIP,		// we have skipped data the frame is partial.
};

/* grabstate (of the frame). */
enum {
	FRAME_UNUSED,		/* Unused (no MCAPTURE) */
	FRAME_READY,		/* Ready to start grabbing */
	FRAME_GRABBING,		/* In the process of being grabbed into */
	FRAME_DONE,		/* Finished grabbing, but not been synced yet */
	FRAME_ERROR,		/* Something bad happened while processing */
};


struct usb_device;

struct quickcam_sbuf {
	char *data;
	urb_t *urb;
};

struct palette_list {
     int num;
     char *name;
     int supported;
};

/*
 * Structure filled in for each of the types of sensor (HDCS, PB0100)
 */
struct sensorctrl {
  int (*init) (struct usb_device * dev, int mode,
               int *rgain, int *bgain, int *ggain,
               struct sensorctrl *sensor_ctrl);
  int (*set_shutter) (struct usb_device * dev, int sval, int xval);
  int (*set_gains) (struct usb_device * dev, int rgain, int bgain, int ggain);
  int (*set_window) (struct usb_device * dev, int x, int y, int w, int h,
                     struct sensorctrl *sensor_ctrl);
  int (*set_size) (struct usb_device * dev, int mode);
  int (*start) (struct usb_device * dev, struct sensorctrl *sensor_ctrl);
  int (*stop) (struct usb_device * dev, struct sensorctrl *sensor_ctrl);
  // size delivered by the sensor.
  int width;
  int height;
  // mode: 0 full; 1 half; 2 quater.
  int mode;
};

/**
 * Structure storing the I2C messages.
 * Do use it:
 * usb_quickcam_i2c_new reset the structure.
 * usb_quickcam_i2c_add add register and value.
 * usb_quickcam_i2c_send send it to the sensor. It call a usb_quickcam_i2c_new.
 */
struct quickcam_i2c {
	int length;
	unsigned char buff[35];
};

struct quickcam_frame {

	char *data;		/* Frame buffer */
	long scanlength;	/* uncompressed, raw data length of frame */

	char *storedata;	/* Bayer data */
	long storelength;	/* len of received data */

 	int width;		/* Width application is expecting */
	int height;		/* Height */

//	int hdrwidth;		/* Width the frame actually is */
//	int hdrheight;		/* Height */

	volatile int grabstate;	/* State of grabbing */

	long bytes_read;	/* amount of scanlength that has been read from *data */

	wait_queue_head_t wq;	/* Processes waiting */
};

struct usb_quickcam {
	struct video_device vdev;

        struct video_picture vpic;
	struct video_window vwin;

	/* Device structure */
	struct usb_device *dev;

        /* For /proc interface */
	struct proc_dir_entry *proc_entry;

	unsigned char iface;

	struct semaphore lock;
	int user;		/* user count for exclusive use */

	int streaming;		/* Are we streaming Isochronous? */
	int grabbing;		/* Are we grabbing? */
	int readframe;		/* the frame we are reading. */

	int scanstate;		/* state of the automaton */

	int storestate;		/* indicated that we have to wait for the end */

	int sizechanged;	/* indicated size changes */

	int yy;
	int x;

	int shutter_val;
	int gain; /* global gain */
	int val; /* global exposure */

	int blue; /* blue gain */
	int red;  /* red gain  */
	int green;/* green gain  */
	int brightness;/* Control brightness of image through V4L */

	char *sensor_name;	/* for /proc */
	unsigned char sensor_addr; /* hdcs and photobit have different addr */

	char *fbuf;		/* Videodev buffer area */

	int curframe;
	struct quickcam_frame frame[QUICKCAM_NUMFRAMES];	/* Double buffering */

	struct quickcam_sbuf sbuf[QUICKCAM_NUMSBUF];		/* Double buffering */

	/* Scratch space from the Isochronous pipe */
	unsigned char scratch[SCRATCH_BUF_SIZE];
	int scratchlen;

	struct semaphore busy_lock;     /* guard against SMP multithreading */

	struct sensorctrl sensor_ctrl;  /* specific routines for a sensor */

};

/* sensor informations (input about all supported sensors) */

struct sensor_data {
	char	*name;
	int	reg23;
	unsigned char	i2c_addr;
	int	id_reg;
	unsigned char	id;
	int	length_id;
	void	(*load)(struct sensorctrl *);
}; 

/* Add prototyping to prevent warnings */

/* Sensor initialisation */
void load_hdcs_mod(struct sensorctrl *sensor_ctrl);
void load_hdcs20_mod(struct sensorctrl *sensor_ctrl);
void load_pb0100_mod(struct sensorctrl *sensor_ctrl);
void load_vv6410_mod(struct sensorctrl *sensor_ctrl);

/* Sensor commmon routines */
void usb_quickcam_i2c_new(struct quickcam_i2c *i2cbuff);
void usb_quickcam_i2c_add(struct quickcam_i2c *i2cbuff,unsigned char reg,
        unsigned char value);
void usb_quickcam_i2c_add2(struct quickcam_i2c *i2cbuff,unsigned char reg,
        unsigned short value);
int usb_quickcam_i2c_send(struct usb_device *dev, struct quickcam_i2c *i2cbuff,
        unsigned char sensor_add);

/* USB interface chip control */
int usb_quickcam_set1(struct usb_device *dev, short reg, char val);
int usb_quickcam_set2(struct usb_device *dev, short reg, short val);

/* conversion routines */
int IsSupported(int format);
void quickcam_convert_image(struct quickcam_frame *frame, int format);
char *FormatName(int format);

#endif
