/***************************************************************************
  hops.h
  -------------------
  A hops class
  -------------------
  begin         October 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef HOPS_H
#define HOPS_H

#include <qmap.h>
#include <qlistview.h>
#include <qstring.h>

class Hops {
public:
    // constructor
    Hops(const QString &name="hop", const double &quantity=1.0, const QString &form="pellet",
        const double &alpha=5.0, const unsigned &time=60);
    // copy constructor
    Hops(const Hops &h);
    // assignment operator
    Hops operator=(const Hops &h);
    // comparison operator
    bool operator==(const Hops &h);
    // destructor
    ~Hops();

    // return name
    QString name();
    // return quantity (in ounces)
    double quantity();
    // return hops form
    QString form();
    // return hops alpha content
    double alpha();
    // return boiling time
    unsigned time();
    
    // return precalculated values for bitterness
    double HBU();
        
    // return quantity as string
    QString quantityString();
    // return alpha as string
    QString alphaString();
    // return time as string
    QString timeString();
    
    // return object as string, for serialization (deprecated)
    QString obsoleteSerializeOut();
    // set object to string, for serialization (deprecated)
    void obsoleteSerializeIn(const QString &name, const QString &value);

private:
    QString name_;
    double quantity_;
    QString form_;
    double alpha_;
    unsigned time_;
    double hbu_;
};

typedef QValueList<Hops> HopsList;
typedef QValueList<Hops>::Iterator HopIterator;

class HopsListItem : public QListViewItem {
public:
    // constructor
    HopsListItem(QListView * parent, QString, QString = QString::null, QString = QString::null,
            QString = QString::null, QString = QString::null, QString = QString::null,
            QString = QString::null, QString = QString::null );
    // destructor
    ~HopsListItem();
    // get iterator to Hop item
    HopIterator hopIterator();
    // set iterator to Hop item
    void setHopIterator(HopIterator &it);
    // set hop to match iterator
    void setToIterator();
private:
    HopIterator hit_;
};

//////////////////////////////////////////////////////////////////////////////
// Inlined Operators

inline QString Hops::name() { return name_; }

inline double Hops::quantity() { return quantity_; }

inline QString Hops::form() { return form_; }

inline double Hops::alpha() { return alpha_; }

inline unsigned Hops::time() { return time_; }

inline QString Hops::quantityString() { return QString::number(quantity_, 'f', 2); }

inline QString Hops::alphaString() { return (QString::number(alpha_) + '%'); }

inline QString Hops::timeString() {return QString::number(time_); }

inline double Hops::HBU() { return hbu_; }

inline HopsListItem::~HopsListItem() { ; }

inline HopIterator HopsListItem::hopIterator() { return hit_; }

inline void HopsListItem::setHopIterator(HopIterator &it) { hit_ = it; }

#endif // HOPS_H
