
##############################################################################
#
# Copyright (c) 2003-2016 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2016 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
basic tests for functions in util.py effecting the spatial distribution 

it is assumed that the domain is the usint square/cube 

not all these test will run for all domains. check the doc string for the assumptions of a particular test

:var __author__: name of author
:var __copyright__: copyrights
:var __license__: licence agreement
:var __url__: url entry point on documentation
:var __version__: version
:var __date__: date of the version
"""
__author__="Lutz Gross, l.gross@uq.edu.au"

import esys.escriptcore.utestselect as unittest
from esys.escript import *
from numpy import array
import numpy

from test_util_spatial_functions1 import Test_Util_SpatialFunctions_noGradOnBoundary_noContact

#begin

class Test_Util_SpatialFunctions_noGradOnBoundary(Test_Util_SpatialFunctions_noGradOnBoundary_noContact):
   RES_TOL=1.e-8

   def test_normal_onFunctionOnContactZero(self):
     """
     test getNormal() on contact side 0

     assumptions: FunctionOnContactZero(self.domain) exists
     """
     dim=self.domain.getDim()
     f=FunctionOnContactZero(self.domain)
     x=f.getX()
     ref=Vector(0.,what=f)
     if dim==3:
         ref+=whereZero(x[0]-0.5,tol=self.RES_TOL)*[1,0,0]
     else:
         ref+=whereZero(x[0]-0.5,tol=self.RES_TOL)*[1,0]

     res=f.getNormal()
     self.assertEqual(res.getShape(),(dim,),"wrong shape of result.")
     self.assertEqual(res.getFunctionSpace(),f,"wrong functionspace of result.")
     self.assertTrue(Lsup(length(res)-1)<=self.RES_TOL,"wrong length")
     self.assertTrue(Lsup(abs(inner(ref,res))-1)<=self.RES_TOL,"wrong direction")
   def test_normal_onReducedFunctionOnContactZero(self):
     """
     test getNormal() on contact side 0

     assumptions: FunctionOnContactZero(self.domain) exists
     """
     dim=self.domain.getDim()
     f=ReducedFunctionOnContactZero(self.domain)
     x=f.getX()
     ref=Vector(0.,what=f)
     if dim==3:
         ref+=whereZero(x[0]-0.5,tol=self.RES_TOL)*[1,0,0]
     else:
         ref+=whereZero(x[0]-0.5,tol=self.RES_TOL)*[1,0]

     res=f.getNormal()
     self.assertEqual(res.getShape(),(dim,),"wrong shape of result.")
     self.assertEqual(res.getFunctionSpace(),f,"wrong functionspace of result.")
     self.assertTrue(Lsup(length(res)-1)<=self.RES_TOL,"wrong length")
     self.assertTrue(Lsup(abs(inner(ref,res))-1)<=self.RES_TOL,"wrong direction")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_normal_onFunctionOnContactOne(self):
     """
     test getNormal() on contact side 1

     assumptions: FunctionOnContactOne(self.domain) exists
     """
     dim=self.domain.getDim()
     f=FunctionOnContactOne(self.domain)
     x=f.getX()
     ref=Vector(0.,what=f)
     if dim==3:
         ref+=whereZero(x[0]-0.5,tol=self.RES_TOL)*[-1,0,0]
     else:
         ref+=whereZero(x[0]-0.5,tol=self.RES_TOL)*[-1,0]

     res=f.getNormal()
     self.assertEqual(res.getShape(),(dim,),"wrong shape of result.")
     self.assertEqual(res.getFunctionSpace(),f,"wrong functionspace of result.")
     self.assertTrue(Lsup(length(res)-1)<=self.RES_TOL,"wrong length")
     self.assertTrue(Lsup(abs(inner(ref,res))-1)<=self.RES_TOL,"wrong direction")
   def test_normal_onReducedFunctionOnContactOne(self):
     """
     test getNormal() on contact side 1

     assumptions: FunctionOnContactOne(self.domain) exists
     """
     dim=self.domain.getDim()
     f=ReducedFunctionOnContactOne(self.domain)
     x=f.getX()
     ref=Vector(0.,what=f)
     if dim==3:
         ref+=whereZero(x[0]-0.5,tol=self.RES_TOL)*[-1,0,0]
     else:
         ref+=whereZero(x[0]-0.5,tol=self.RES_TOL)*[-1,0]

     res=f.getNormal()
     self.assertEqual(res.getShape(),(dim,),"wrong shape of result.")
     self.assertEqual(res.getFunctionSpace(),f,"wrong functionspace of result.")
     self.assertTrue(Lsup(length(res)-1)<=self.RES_TOL,"wrong length")
     self.assertTrue(Lsup(abs(inner(ref,res))-1)<=self.RES_TOL,"wrong direction")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.311774053155)*x[0]**o+(0.52292841627)*x[0]+(-0.518467126196)*x[1]**o+(-0.11329290159)*x[1]
        ref=(-0.518467126196)/(o+1.)+(0.20481775734)+(0.311774053155)*0.5**o
      else:
        arg=(0.233765319091)*x[0]**o+(-0.453551209762)*x[0]+(0.920140231713)*x[1]**o+(0.741695479084)*x[1]+(-0.255834461065)*x[2]**o+(-0.102580899053)*x[2]
        ref=(0.664305770648)/(o+1.)+(0.0927816851348)+(0.233765319091)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.651678780872)*x[0]**o+(-0.749709447394)*x[0]+(0.311088209082)*x[1]**o+(0.553081401193)*x[1]
        arg[1]=(0.657887728987)*x[0]**o+(0.052270143604)*x[0]+(-0.15507994277)*x[1]**o+(-0.30302150082)*x[1]
        ref[0]=(0.311088209082)/(o+1.)+(-0.0983140231004)+(-0.651678780872)*0.5**o
        ref[1]=(-0.15507994277)/(o+1.)+(-0.125375678608)+(0.657887728987)*0.5**o
      else:
        arg[0]=(0.844909347644)*x[0]**o+(-0.342697697483)*x[0]+(0.126242548186)*x[1]**o+(-0.854010732849)*x[1]+(-0.307627587461)*x[2]**o+(-0.998566290754)*x[2]
        arg[1]=(0.900073726701)*x[0]**o+(-0.703841701859)*x[0]+(-0.651690896499)*x[1]**o+(0.828362322181)*x[1]+(0.284410420895)*x[2]**o+(-0.394606348974)*x[2]
        ref[0]=(-0.181385039275)/(o+1.)+(-1.09763736054)+(0.844909347644)*0.5**o
        ref[1]=(-0.367280475604)/(o+1.)+(-0.135042864326)+(0.900073726701)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.263642681371)*x[0]**o+(0.668992571787)*x[0]+(0.873939846194)*x[1]**o+(0.784820355998)*x[1]
        arg[0,1]=(-0.389306088628)*x[0]**o+(-0.0549482389907)*x[0]+(-0.787101008457)*x[1]**o+(0.860345123812)*x[1]
        arg[0,2]=(-0.995721191961)*x[0]**o+(-0.587928136153)*x[0]+(-0.378312614832)*x[1]**o+(0.199765674281)*x[1]
        arg[0,3]=(-0.374679425145)*x[0]**o+(-0.503606183146)*x[0]+(0.23995738992)*x[1]**o+(-0.525359181744)*x[1]
        arg[0,4]=(-0.69923255665)*x[0]**o+(0.547634979941)*x[0]+(0.728792702739)*x[1]**o+(0.949888533828)*x[1]
        arg[1,0]=(-0.61856415528)*x[0]**o+(-0.0516239651093)*x[0]+(0.158516649979)*x[1]**o+(-0.0522288881011)*x[1]
        arg[1,1]=(0.0285524133857)*x[0]**o+(0.741357758307)*x[0]+(-0.0714860903847)*x[1]**o+(-0.539632750385)*x[1]
        arg[1,2]=(-0.291101401382)*x[0]**o+(-0.114900246437)*x[0]+(-0.982305031665)*x[1]**o+(0.0823478844055)*x[1]
        arg[1,3]=(-0.931305732525)*x[0]**o+(0.077332436417)*x[0]+(-0.880626154921)*x[1]**o+(-0.321222234507)*x[1]
        arg[1,4]=(-0.00778033208815)*x[0]**o+(-0.536626847957)*x[0]+(-0.599742714546)*x[1]**o+(0.454108299336)*x[1]
        arg[2,0]=(-0.575243090445)*x[0]**o+(-0.920247228564)*x[0]+(0.4569784462)*x[1]**o+(-0.105175798804)*x[1]
        arg[2,1]=(0.333985911748)*x[0]**o+(-0.017876213076)*x[0]+(0.475164778165)*x[1]**o+(0.561072178015)*x[1]
        arg[2,2]=(-0.559587786773)*x[0]**o+(-0.917134128672)*x[0]+(0.316654889327)*x[1]**o+(-0.640569595027)*x[1]
        arg[2,3]=(-0.140461324817)*x[0]**o+(0.259052969883)*x[0]+(0.397898509159)*x[1]**o+(0.932233331785)*x[1]
        arg[2,4]=(0.368626304985)*x[0]**o+(0.616310502384)*x[0]+(-0.756845293566)*x[1]**o+(-0.920088968132)*x[1]
        arg[3,0]=(-0.824398741087)*x[0]**o+(0.0498115894252)*x[0]+(-0.5256621655)*x[1]**o+(0.114806237108)*x[1]
        arg[3,1]=(-0.77671057316)*x[0]**o+(0.569249443162)*x[0]+(0.416628465801)*x[1]**o+(0.0603627487913)*x[1]
        arg[3,2]=(-0.247875220913)*x[0]**o+(0.19548828878)*x[0]+(0.643226101838)*x[1]**o+(-0.154558826499)*x[1]
        arg[3,3]=(-0.918937118963)*x[0]**o+(0.193777174417)*x[0]+(-0.906042347107)*x[1]**o+(0.872033230033)*x[1]
        arg[3,4]=(-0.628918336086)*x[0]**o+(0.823242191714)*x[0]+(-0.593691166712)*x[1]**o+(-0.661715840814)*x[1]
        ref[0,0]=(0.873939846194)/(o+1.)+(0.726906463892)+(-0.263642681371)*0.5**o
        ref[0,1]=(-0.787101008457)/(o+1.)+(0.402698442411)+(-0.389306088628)*0.5**o
        ref[0,2]=(-0.378312614832)/(o+1.)+(-0.194081230936)+(-0.995721191961)*0.5**o
        ref[0,3]=(0.23995738992)/(o+1.)+(-0.514482682445)+(-0.374679425145)*0.5**o
        ref[0,4]=(0.728792702739)/(o+1.)+(0.748761756885)+(-0.69923255665)*0.5**o
        ref[1,0]=(0.158516649979)/(o+1.)+(-0.0519264266052)+(-0.61856415528)*0.5**o
        ref[1,1]=(-0.0714860903847)/(o+1.)+(0.100862503961)+(0.0285524133857)*0.5**o
        ref[1,2]=(-0.982305031665)/(o+1.)+(-0.0162761810159)+(-0.291101401382)*0.5**o
        ref[1,3]=(-0.880626154921)/(o+1.)+(-0.121944899045)+(-0.931305732525)*0.5**o
        ref[1,4]=(-0.599742714546)/(o+1.)+(-0.0412592743102)+(-0.00778033208815)*0.5**o
        ref[2,0]=(0.4569784462)/(o+1.)+(-0.512711513684)+(-0.575243090445)*0.5**o
        ref[2,1]=(0.475164778165)/(o+1.)+(0.27159798247)+(0.333985911748)*0.5**o
        ref[2,2]=(0.316654889327)/(o+1.)+(-0.778851861849)+(-0.559587786773)*0.5**o
        ref[2,3]=(0.397898509159)/(o+1.)+(0.595643150834)+(-0.140461324817)*0.5**o
        ref[2,4]=(-0.756845293566)/(o+1.)+(-0.151889232874)+(0.368626304985)*0.5**o
        ref[3,0]=(-0.5256621655)/(o+1.)+(0.0823089132668)+(-0.824398741087)*0.5**o
        ref[3,1]=(0.416628465801)/(o+1.)+(0.314806095977)+(-0.77671057316)*0.5**o
        ref[3,2]=(0.643226101838)/(o+1.)+(0.0204647311405)+(-0.247875220913)*0.5**o
        ref[3,3]=(-0.906042347107)/(o+1.)+(0.532905202225)+(-0.918937118963)*0.5**o
        ref[3,4]=(-0.593691166712)/(o+1.)+(0.0807631754499)+(-0.628918336086)*0.5**o
      else:
        arg[0,0]=(0.748857269399)*x[0]**o+(-0.21982973469)*x[0]+(-0.765987921295)*x[1]**o+(0.556577019909)*x[1]+(-0.0438548079278)*x[2]**o+(0.543610874817)*x[2]
        arg[0,1]=(-0.784526004748)*x[0]**o+(-0.949423172035)*x[0]+(-0.29888188317)*x[1]**o+(-0.850638766848)*x[1]+(-0.853027558387)*x[2]**o+(-0.846378486588)*x[2]
        arg[0,2]=(0.53445882525)*x[0]**o+(-0.653655652989)*x[0]+(-0.968129875689)*x[1]**o+(-0.603598808323)*x[1]+(0.545841014272)*x[2]**o+(0.126131861082)*x[2]
        arg[0,3]=(0.181662498503)*x[0]**o+(-0.769571080865)*x[0]+(0.771109181078)*x[1]**o+(0.980677373303)*x[1]+(-0.625057940268)*x[2]**o+(0.0585672357346)*x[2]
        arg[0,4]=(-0.0320907540755)*x[0]**o+(-0.519517369917)*x[0]+(-0.365190231996)*x[1]**o+(-0.192392652661)*x[1]+(-0.693267159516)*x[2]**o+(0.106458640425)*x[2]
        arg[1,0]=(0.414286358161)*x[0]**o+(0.0223282279507)*x[0]+(0.215881279469)*x[1]**o+(0.68990977849)*x[1]+(-0.532075885581)*x[2]**o+(-0.269164904232)*x[2]
        arg[1,1]=(-0.0445879250589)*x[0]**o+(0.889985400599)*x[0]+(0.842050582947)*x[1]**o+(-0.0275307884642)*x[1]+(0.774650107671)*x[2]**o+(0.19446514971)*x[2]
        arg[1,2]=(0.913032880034)*x[0]**o+(-0.286537470008)*x[0]+(0.0902778267032)*x[1]**o+(0.548596933564)*x[1]+(-0.996100405074)*x[2]**o+(-0.509554019549)*x[2]
        arg[1,3]=(0.44308812765)*x[0]**o+(-0.0641227726924)*x[0]+(0.484249008075)*x[1]**o+(0.581496384894)*x[1]+(0.592538533183)*x[2]**o+(-0.821638541512)*x[2]
        arg[1,4]=(-0.898304735153)*x[0]**o+(0.333217967121)*x[0]+(-0.788218899371)*x[1]**o+(0.579723459039)*x[1]+(0.467006473159)*x[2]**o+(-0.277113860535)*x[2]
        arg[2,0]=(0.515308716564)*x[0]**o+(0.255947750577)*x[0]+(0.232578337945)*x[1]**o+(0.900611887592)*x[1]+(-0.247710965611)*x[2]**o+(-0.985523228221)*x[2]
        arg[2,1]=(0.611196756618)*x[0]**o+(0.146822999187)*x[0]+(-0.318793147278)*x[1]**o+(0.790772432028)*x[1]+(0.138036152668)*x[2]**o+(-0.800234207543)*x[2]
        arg[2,2]=(-0.293576196546)*x[0]**o+(-0.521059592737)*x[0]+(0.301178293957)*x[1]**o+(-0.141430900198)*x[1]+(-0.782381874467)*x[2]**o+(0.92221179729)*x[2]
        arg[2,3]=(0.673951778483)*x[0]**o+(-0.335068772752)*x[0]+(0.408793200723)*x[1]**o+(0.36906212176)*x[1]+(0.246292093855)*x[2]**o+(0.356147233675)*x[2]
        arg[2,4]=(-0.218899302236)*x[0]**o+(0.271285962886)*x[0]+(-0.16364707283)*x[1]**o+(0.177195607727)*x[1]+(-0.287331396699)*x[2]**o+(-0.465391460497)*x[2]
        arg[3,0]=(-0.369727856735)*x[0]**o+(-0.44351931666)*x[0]+(0.0375097686939)*x[1]**o+(0.276779121402)*x[1]+(0.662505672881)*x[2]**o+(-0.642709866174)*x[2]
        arg[3,1]=(0.444940917816)*x[0]**o+(-0.178926126454)*x[0]+(0.824660645994)*x[1]**o+(0.825996820355)*x[1]+(0.798260651848)*x[2]**o+(0.331744022239)*x[2]
        arg[3,2]=(-0.82489261483)*x[0]**o+(-0.340862079766)*x[0]+(-0.998375587383)*x[1]**o+(0.880222106116)*x[1]+(0.656096580473)*x[2]**o+(0.530854486127)*x[2]
        arg[3,3]=(0.732863411435)*x[0]**o+(-0.324252004147)*x[0]+(0.789472654884)*x[1]**o+(-0.207794796011)*x[1]+(0.50903180567)*x[2]**o+(0.724319112504)*x[2]
        arg[3,4]=(-0.980748139471)*x[0]**o+(0.0974331161906)*x[0]+(0.196275528452)*x[1]**o+(0.305231910003)*x[1]+(-0.0417825350947)*x[2]**o+(-0.2916834261)*x[2]
        ref[0,0]=(-0.809842729223)/(o+1.)+(0.440179080018)+(0.748857269399)*0.5**o
        ref[0,1]=(-1.15190944156)/(o+1.)+(-1.32322021274)+(-0.784526004748)*0.5**o
        ref[0,2]=(-0.422288861417)/(o+1.)+(-0.565561300115)+(0.53445882525)*0.5**o
        ref[0,3]=(0.146051240811)/(o+1.)+(0.134836764087)+(0.181662498503)*0.5**o
        ref[0,4]=(-1.05845739151)/(o+1.)+(-0.302725691077)+(-0.0320907540755)*0.5**o
        ref[1,0]=(-0.316194606113)/(o+1.)+(0.221536551104)+(0.414286358161)*0.5**o
        ref[1,1]=(1.61670069062)/(o+1.)+(0.528459880922)+(-0.0445879250589)*0.5**o
        ref[1,2]=(-0.905822578371)/(o+1.)+(-0.123747277996)+(0.913032880034)*0.5**o
        ref[1,3]=(1.07678754126)/(o+1.)+(-0.152132464655)+(0.44308812765)*0.5**o
        ref[1,4]=(-0.321212426212)/(o+1.)+(0.317913782812)+(-0.898304735153)*0.5**o
        ref[2,0]=(-0.015132627666)/(o+1.)+(0.085518204974)+(0.515308716564)*0.5**o
        ref[2,1]=(-0.18075699461)/(o+1.)+(0.0686806118359)+(0.611196756618)*0.5**o
        ref[2,2]=(-0.481203580509)/(o+1.)+(0.129860652178)+(-0.293576196546)*0.5**o
        ref[2,3]=(0.655085294578)/(o+1.)+(0.195070291341)+(0.673951778483)*0.5**o
        ref[2,4]=(-0.450978469529)/(o+1.)+(-0.00845494494169)+(-0.218899302236)*0.5**o
        ref[3,0]=(0.700015441575)/(o+1.)+(-0.404725030716)+(-0.369727856735)*0.5**o
        ref[3,1]=(1.62292129784)/(o+1.)+(0.48940735807)+(0.444940917816)*0.5**o
        ref[3,2]=(-0.342279006911)/(o+1.)+(0.535107256239)+(-0.82489261483)*0.5**o
        ref[3,3]=(1.29850446055)/(o+1.)+(0.0961361561734)+(0.732863411435)*0.5**o
        ref[3,4]=(0.154492993358)/(o+1.)+(0.055490800047)+(-0.980748139471)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.33788666753)*x[0]**o+(0.0147671449643)*x[0]+(0.14066054662)*x[1]**o+(-0.675231899756)*x[1]
        arg[0,0,1]=(0.0100961441611)*x[0]**o+(0.401304171984)*x[0]+(-0.594786282119)*x[1]**o+(-0.561543495067)*x[1]
        arg[0,1,0]=(-0.118311175162)*x[0]**o+(0.62454992688)*x[0]+(-0.8163725079)*x[1]**o+(-0.756355655277)*x[1]
        arg[0,1,1]=(0.586991709411)*x[0]**o+(-0.600757941586)*x[0]+(0.0953383109011)*x[1]**o+(0.317827629801)*x[1]
        arg[1,0,0]=(-0.251125236294)*x[0]**o+(0.166603616954)*x[0]+(0.978704943828)*x[1]**o+(0.589215389731)*x[1]
        arg[1,0,1]=(0.0543182786738)*x[0]**o+(-0.334756305272)*x[0]+(-0.919571042987)*x[1]**o+(-0.0425734727213)*x[1]
        arg[1,1,0]=(-0.867177621518)*x[0]**o+(-0.0750659593232)*x[0]+(-0.227277029921)*x[1]**o+(0.361120763296)*x[1]
        arg[1,1,1]=(0.85129514889)*x[0]**o+(-0.441962000795)*x[0]+(0.425335668395)*x[1]**o+(0.204091109081)*x[1]
        arg[2,0,0]=(-0.537930401585)*x[0]**o+(-0.206510941636)*x[0]+(-0.247374421196)*x[1]**o+(0.67043841725)*x[1]
        arg[2,0,1]=(-0.101273864474)*x[0]**o+(0.283769727506)*x[0]+(-0.0850059833654)*x[1]**o+(-0.71291661379)*x[1]
        arg[2,1,0]=(0.684430088582)*x[0]**o+(0.107733116086)*x[0]+(0.40398578341)*x[1]**o+(-0.525004991854)*x[1]
        arg[2,1,1]=(-0.00801759014554)*x[0]**o+(-0.214858014851)*x[0]+(-0.427822011518)*x[1]**o+(0.0168962506945)*x[1]
        arg[3,0,0]=(0.528563052182)*x[0]**o+(-0.630485796029)*x[0]+(0.0525765317703)*x[1]**o+(-0.469639006097)*x[1]
        arg[3,0,1]=(0.178622695427)*x[0]**o+(0.0952996500383)*x[0]+(-0.785851578496)*x[1]**o+(0.856665398338)*x[1]
        arg[3,1,0]=(0.547391926072)*x[0]**o+(-0.0179195091852)*x[0]+(-0.831368961862)*x[1]**o+(-0.437229107453)*x[1]
        arg[3,1,1]=(-0.657011566453)*x[0]**o+(0.530176142378)*x[0]+(0.649145894746)*x[1]**o+(-0.755185337598)*x[1]
        arg[4,0,0]=(-0.188305872652)*x[0]**o+(0.0207245930982)*x[0]+(-0.590757998421)*x[1]**o+(-0.793662033952)*x[1]
        arg[4,0,1]=(0.417712824349)*x[0]**o+(-0.138327612527)*x[0]+(-0.381595921766)*x[1]**o+(0.0287368364725)*x[1]
        arg[4,1,0]=(-0.83750540403)*x[0]**o+(0.247261273008)*x[0]+(-0.0769629479034)*x[1]**o+(0.0693501875686)*x[1]
        arg[4,1,1]=(0.697434044074)*x[0]**o+(0.396582403702)*x[0]+(0.736132107034)*x[1]**o+(0.482245183242)*x[1]
        arg[5,0,0]=(0.769936158184)*x[0]**o+(0.808871896459)*x[0]+(-0.559205314785)*x[1]**o+(-0.258014827146)*x[1]
        arg[5,0,1]=(0.667132553969)*x[0]**o+(0.618184492083)*x[0]+(0.475852440157)*x[1]**o+(0.084499293443)*x[1]
        arg[5,1,0]=(0.910882712823)*x[0]**o+(0.153897139268)*x[0]+(-0.61197280039)*x[1]**o+(0.441331985295)*x[1]
        arg[5,1,1]=(-0.290188623857)*x[0]**o+(0.318266320131)*x[0]+(0.9587856247)*x[1]**o+(0.311981486508)*x[1]
        ref[0,0,0]=(0.14066054662)/(o+1.)+(-0.330232377396)+(-0.33788666753)*0.5**o
        ref[0,0,1]=(-0.594786282119)/(o+1.)+(-0.0801196615418)+(0.0100961441611)*0.5**o
        ref[0,1,0]=(-0.8163725079)/(o+1.)+(-0.0659028641985)+(-0.118311175162)*0.5**o
        ref[0,1,1]=(0.0953383109011)/(o+1.)+(-0.141465155893)+(0.586991709411)*0.5**o
        ref[1,0,0]=(0.978704943828)/(o+1.)+(0.377909503343)+(-0.251125236294)*0.5**o
        ref[1,0,1]=(-0.919571042987)/(o+1.)+(-0.188664888997)+(0.0543182786738)*0.5**o
        ref[1,1,0]=(-0.227277029921)/(o+1.)+(0.143027401987)+(-0.867177621518)*0.5**o
        ref[1,1,1]=(0.425335668395)/(o+1.)+(-0.118935445857)+(0.85129514889)*0.5**o
        ref[2,0,0]=(-0.247374421196)/(o+1.)+(0.231963737807)+(-0.537930401585)*0.5**o
        ref[2,0,1]=(-0.0850059833654)/(o+1.)+(-0.214573443142)+(-0.101273864474)*0.5**o
        ref[2,1,0]=(0.40398578341)/(o+1.)+(-0.208635937884)+(0.684430088582)*0.5**o
        ref[2,1,1]=(-0.427822011518)/(o+1.)+(-0.0989808820785)+(-0.00801759014554)*0.5**o
        ref[3,0,0]=(0.0525765317703)/(o+1.)+(-0.550062401063)+(0.528563052182)*0.5**o
        ref[3,0,1]=(-0.785851578496)/(o+1.)+(0.475982524188)+(0.178622695427)*0.5**o
        ref[3,1,0]=(-0.831368961862)/(o+1.)+(-0.227574308319)+(0.547391926072)*0.5**o
        ref[3,1,1]=(0.649145894746)/(o+1.)+(-0.11250459761)+(-0.657011566453)*0.5**o
        ref[4,0,0]=(-0.590757998421)/(o+1.)+(-0.386468720427)+(-0.188305872652)*0.5**o
        ref[4,0,1]=(-0.381595921766)/(o+1.)+(-0.0547953880271)+(0.417712824349)*0.5**o
        ref[4,1,0]=(-0.0769629479034)/(o+1.)+(0.158305730288)+(-0.83750540403)*0.5**o
        ref[4,1,1]=(0.736132107034)/(o+1.)+(0.439413793472)+(0.697434044074)*0.5**o
        ref[5,0,0]=(-0.559205314785)/(o+1.)+(0.275428534657)+(0.769936158184)*0.5**o
        ref[5,0,1]=(0.475852440157)/(o+1.)+(0.351341892763)+(0.667132553969)*0.5**o
        ref[5,1,0]=(-0.61197280039)/(o+1.)+(0.297614562282)+(0.910882712823)*0.5**o
        ref[5,1,1]=(0.9587856247)/(o+1.)+(0.315123903319)+(-0.290188623857)*0.5**o
      else:
        arg[0,0,0]=(0.292346424086)*x[0]**o+(0.105058465241)*x[0]+(-0.281467283372)*x[1]**o+(0.828345267021)*x[1]+(0.741242558638)*x[2]**o+(0.0858780765977)*x[2]
        arg[0,0,1]=(-0.460556250835)*x[0]**o+(-0.845199338719)*x[0]+(-0.547000154821)*x[1]**o+(-0.972236450512)*x[1]+(0.823031420765)*x[2]**o+(0.278285294464)*x[2]
        arg[0,1,0]=(0.380027894487)*x[0]**o+(-0.579359210385)*x[0]+(-0.760849798908)*x[1]**o+(0.0065870459398)*x[1]+(-0.869735580799)*x[2]**o+(-0.937378284335)*x[2]
        arg[0,1,1]=(0.948682597897)*x[0]**o+(-0.882044080711)*x[0]+(-0.694398833899)*x[1]**o+(0.883635235205)*x[1]+(-0.473761903609)*x[2]**o+(0.961453193379)*x[2]
        arg[1,0,0]=(0.357218603696)*x[0]**o+(-0.694504294144)*x[0]+(-0.187119222895)*x[1]**o+(0.727463485088)*x[1]+(0.150410254875)*x[2]**o+(-0.373430202907)*x[2]
        arg[1,0,1]=(-0.776893233002)*x[0]**o+(0.565238637388)*x[0]+(-0.155674628662)*x[1]**o+(-0.938226259914)*x[1]+(0.575183633137)*x[2]**o+(0.319871583783)*x[2]
        arg[1,1,0]=(0.593512911569)*x[0]**o+(-0.0578762085927)*x[0]+(-0.913450847289)*x[1]**o+(0.7074653217)*x[1]+(0.991175636143)*x[2]**o+(-0.535632452281)*x[2]
        arg[1,1,1]=(-0.154542349616)*x[0]**o+(-0.841642050346)*x[0]+(0.878833704431)*x[1]**o+(0.85568524402)*x[1]+(-0.710374336281)*x[2]**o+(0.346540187252)*x[2]
        arg[2,0,0]=(-0.765160864305)*x[0]**o+(0.608017010487)*x[0]+(0.841265469171)*x[1]**o+(-0.409854579584)*x[1]+(0.728931324538)*x[2]**o+(0.121315209213)*x[2]
        arg[2,0,1]=(0.0397067301375)*x[0]**o+(-0.0451925117802)*x[0]+(-0.412724345414)*x[1]**o+(-0.447675002107)*x[1]+(0.141728105561)*x[2]**o+(0.923970189305)*x[2]
        arg[2,1,0]=(-0.237044096923)*x[0]**o+(-0.826168294669)*x[0]+(0.19833520121)*x[1]**o+(-0.796076984382)*x[1]+(0.367399642334)*x[2]**o+(0.71172848801)*x[2]
        arg[2,1,1]=(-0.3881830464)*x[0]**o+(0.988354722657)*x[0]+(0.762756805728)*x[1]**o+(0.551778683619)*x[1]+(0.975932105591)*x[2]**o+(-0.173192098292)*x[2]
        arg[3,0,0]=(-0.226163777371)*x[0]**o+(-0.301094659695)*x[0]+(0.0682567814743)*x[1]**o+(-0.154914902824)*x[1]+(0.984669238436)*x[2]**o+(-0.431011897738)*x[2]
        arg[3,0,1]=(-0.47230847571)*x[0]**o+(-0.579315893476)*x[0]+(-0.989289806396)*x[1]**o+(-0.673951334134)*x[1]+(0.98143767614)*x[2]**o+(-0.311074531383)*x[2]
        arg[3,1,0]=(-0.276309910109)*x[0]**o+(-0.70244304164)*x[0]+(-0.963669800079)*x[1]**o+(-0.947723474952)*x[1]+(0.513451083144)*x[2]**o+(-0.0651803692263)*x[2]
        arg[3,1,1]=(0.65175527234)*x[0]**o+(-0.294569414615)*x[0]+(-0.32846386125)*x[1]**o+(-0.868822318527)*x[1]+(0.268414480892)*x[2]**o+(-0.866835042217)*x[2]
        arg[4,0,0]=(0.304002845434)*x[0]**o+(0.938467246002)*x[0]+(-0.106648262772)*x[1]**o+(0.227863374571)*x[1]+(-0.789305204045)*x[2]**o+(-0.171526142133)*x[2]
        arg[4,0,1]=(0.70982305774)*x[0]**o+(0.404228697442)*x[0]+(0.190029230022)*x[1]**o+(-0.825276083634)*x[1]+(-0.739774636921)*x[2]**o+(-0.726529211281)*x[2]
        arg[4,1,0]=(-0.238487290076)*x[0]**o+(-0.4901234628)*x[0]+(-0.836670757945)*x[1]**o+(0.0841493712661)*x[1]+(-0.685144435198)*x[2]**o+(-0.304978244786)*x[2]
        arg[4,1,1]=(0.264176905985)*x[0]**o+(-0.410582158319)*x[0]+(-0.271915471538)*x[1]**o+(-0.996082427599)*x[1]+(-0.466654446945)*x[2]**o+(-0.723243781429)*x[2]
        arg[5,0,0]=(0.379390860481)*x[0]**o+(-0.78556086955)*x[0]+(0.189126836375)*x[1]**o+(0.235947881164)*x[1]+(0.149142076852)*x[2]**o+(-0.944419890441)*x[2]
        arg[5,0,1]=(0.768867280189)*x[0]**o+(-0.728364777249)*x[0]+(0.60414049451)*x[1]**o+(-0.538876171991)*x[1]+(-0.0947560131508)*x[2]**o+(0.474986163327)*x[2]
        arg[5,1,0]=(-0.228069986734)*x[0]**o+(-0.680841588894)*x[0]+(-0.663806510587)*x[1]**o+(-0.852775021416)*x[1]+(-0.59539582999)*x[2]**o+(0.52369159839)*x[2]
        arg[5,1,1]=(0.177872029425)*x[0]**o+(0.93423742256)*x[0]+(-0.0369475988192)*x[1]**o+(0.242202072343)*x[1]+(0.526144455402)*x[2]**o+(0.0217547142691)*x[2]
        ref[0,0,0]=(0.459775275267)/(o+1.)+(0.50964090443)+(0.292346424086)*0.5**o
        ref[0,0,1]=(0.276031265944)/(o+1.)+(-0.769575247383)+(-0.460556250835)*0.5**o
        ref[0,1,0]=(-1.63058537971)/(o+1.)+(-0.75507522439)+(0.380027894487)*0.5**o
        ref[0,1,1]=(-1.16816073751)/(o+1.)+(0.481522173937)+(0.948682597897)*0.5**o
        ref[1,0,0]=(-0.0367089680209)/(o+1.)+(-0.170235505982)+(0.357218603696)*0.5**o
        ref[1,0,1]=(0.419509004475)/(o+1.)+(-0.0265580193719)+(-0.776893233002)*0.5**o
        ref[1,1,0]=(0.0777247888533)/(o+1.)+(0.0569783304129)+(0.593512911569)*0.5**o
        ref[1,1,1]=(0.168459368149)/(o+1.)+(0.180291690463)+(-0.154542349616)*0.5**o
        ref[2,0,0]=(1.57019679371)/(o+1.)+(0.159738820058)+(-0.765160864305)*0.5**o
        ref[2,0,1]=(-0.270996239853)/(o+1.)+(0.215551337709)+(0.0397067301375)*0.5**o
        ref[2,1,0]=(0.565734843545)/(o+1.)+(-0.455258395521)+(-0.237044096923)*0.5**o
        ref[2,1,1]=(1.73868891132)/(o+1.)+(0.683470653992)+(-0.3881830464)*0.5**o
        ref[3,0,0]=(1.05292601991)/(o+1.)+(-0.443510730129)+(-0.226163777371)*0.5**o
        ref[3,0,1]=(-0.00785213025503)/(o+1.)+(-0.782170879496)+(-0.47230847571)*0.5**o
        ref[3,1,0]=(-0.450218716935)/(o+1.)+(-0.857673442909)+(-0.276309910109)*0.5**o
        ref[3,1,1]=(-0.0600493803575)/(o+1.)+(-1.01511338768)+(0.65175527234)*0.5**o
        ref[4,0,0]=(-0.895953466817)/(o+1.)+(0.49740223922)+(0.304002845434)*0.5**o
        ref[4,0,1]=(-0.549745406899)/(o+1.)+(-0.573788298736)+(0.70982305774)*0.5**o
        ref[4,1,0]=(-1.52181519314)/(o+1.)+(-0.35547616816)+(-0.238487290076)*0.5**o
        ref[4,1,1]=(-0.738569918483)/(o+1.)+(-1.06495418367)+(0.264176905985)*0.5**o
        ref[5,0,0]=(0.338268913228)/(o+1.)+(-0.747016439414)+(0.379390860481)*0.5**o
        ref[5,0,1]=(0.50938448136)/(o+1.)+(-0.396127392956)+(0.768867280189)*0.5**o
        ref[5,1,0]=(-1.25920234058)/(o+1.)+(-0.50496250596)+(-0.228069986734)*0.5**o
        ref[5,1,1]=(0.489196856583)/(o+1.)+(0.599097104586)+(0.177872029425)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.423100205741)*x[0]**o+(-0.681307394941)*x[0]+(-0.848175950825)*x[1]**o+(0.559338235783)*x[1]
        arg[0,0,0,1]=(-0.739687474039)*x[0]**o+(-0.202202358237)*x[0]+(-0.838397517739)*x[1]**o+(-0.377580621185)*x[1]
        arg[0,0,1,0]=(-0.423462958438)*x[0]**o+(-0.335929684088)*x[0]+(-0.616976579504)*x[1]**o+(-0.77803694581)*x[1]
        arg[0,0,1,1]=(-0.432533121428)*x[0]**o+(0.0723865130937)*x[0]+(-0.953014661461)*x[1]**o+(0.92761707252)*x[1]
        arg[0,0,2,0]=(-0.891613775753)*x[0]**o+(0.780592229933)*x[0]+(-0.156191993934)*x[1]**o+(0.985464289737)*x[1]
        arg[0,0,2,1]=(0.909573147888)*x[0]**o+(-0.225603851475)*x[0]+(-0.68287286477)*x[1]**o+(0.330105905794)*x[1]
        arg[0,1,0,0]=(-0.39808363424)*x[0]**o+(-0.290872560992)*x[0]+(-0.631624427826)*x[1]**o+(-0.389239289826)*x[1]
        arg[0,1,0,1]=(-0.748658247224)*x[0]**o+(0.22314629362)*x[0]+(-0.0750972377036)*x[1]**o+(0.908914722147)*x[1]
        arg[0,1,1,0]=(0.877627912148)*x[0]**o+(-0.988434656301)*x[0]+(-0.163646238293)*x[1]**o+(-0.169410711361)*x[1]
        arg[0,1,1,1]=(-0.592580581615)*x[0]**o+(0.997989912544)*x[0]+(0.0667564973731)*x[1]**o+(-0.121580259121)*x[1]
        arg[0,1,2,0]=(-0.505569292541)*x[0]**o+(-0.534428841308)*x[0]+(-0.899276958155)*x[1]**o+(-0.833972415825)*x[1]
        arg[0,1,2,1]=(-0.215752366701)*x[0]**o+(-0.249993262232)*x[0]+(0.912452712698)*x[1]**o+(0.0601524115958)*x[1]
        arg[0,2,0,0]=(0.684112319204)*x[0]**o+(0.343018234122)*x[0]+(0.267409202785)*x[1]**o+(0.7781133929)*x[1]
        arg[0,2,0,1]=(0.552438501301)*x[0]**o+(-0.101558113559)*x[0]+(-0.152472725029)*x[1]**o+(-0.8771654517)*x[1]
        arg[0,2,1,0]=(-0.31213422297)*x[0]**o+(0.597042319279)*x[0]+(-0.873134889407)*x[1]**o+(-0.68920786345)*x[1]
        arg[0,2,1,1]=(0.883048532408)*x[0]**o+(-0.653997880306)*x[0]+(-0.761310589424)*x[1]**o+(-0.698342623962)*x[1]
        arg[0,2,2,0]=(-0.324489237508)*x[0]**o+(-0.717724692716)*x[0]+(-0.936381308418)*x[1]**o+(0.712416971077)*x[1]
        arg[0,2,2,1]=(-0.971735193469)*x[0]**o+(0.0943629094751)*x[0]+(0.419811305063)*x[1]**o+(0.125623316569)*x[1]
        arg[0,3,0,0]=(-0.305553454905)*x[0]**o+(-0.936112994503)*x[0]+(0.707279059071)*x[1]**o+(-0.794754809337)*x[1]
        arg[0,3,0,1]=(0.803934620926)*x[0]**o+(-0.632020033455)*x[0]+(0.819264129099)*x[1]**o+(0.770744069672)*x[1]
        arg[0,3,1,0]=(0.996632326055)*x[0]**o+(-0.537850178404)*x[0]+(-0.60262522048)*x[1]**o+(-0.431667704232)*x[1]
        arg[0,3,1,1]=(0.402799911553)*x[0]**o+(0.155798565063)*x[0]+(0.722837842981)*x[1]**o+(0.462463978227)*x[1]
        arg[0,3,2,0]=(0.937966588808)*x[0]**o+(-0.797151899551)*x[0]+(0.62363567076)*x[1]**o+(-0.60458214868)*x[1]
        arg[0,3,2,1]=(-0.0389713032301)*x[0]**o+(0.497820326557)*x[0]+(-0.950660964237)*x[1]**o+(0.0247372614337)*x[1]
        arg[0,4,0,0]=(-0.589529640529)*x[0]**o+(-0.71761078076)*x[0]+(-0.872148906918)*x[1]**o+(-0.666618835503)*x[1]
        arg[0,4,0,1]=(0.78615039967)*x[0]**o+(0.99287672137)*x[0]+(0.329194160571)*x[1]**o+(0.309370990805)*x[1]
        arg[0,4,1,0]=(-0.801187683466)*x[0]**o+(0.476957978386)*x[0]+(0.107007555422)*x[1]**o+(-0.211909676249)*x[1]
        arg[0,4,1,1]=(0.817551079419)*x[0]**o+(0.958544826763)*x[0]+(-0.720479146591)*x[1]**o+(0.0769824586505)*x[1]
        arg[0,4,2,0]=(-0.245391293539)*x[0]**o+(0.240588876322)*x[0]+(0.766214937295)*x[1]**o+(0.625282150348)*x[1]
        arg[0,4,2,1]=(-0.780163545255)*x[0]**o+(-0.84424180556)*x[0]+(-0.993826355778)*x[1]**o+(-0.708432857746)*x[1]
        arg[1,0,0,0]=(0.0998104654851)*x[0]**o+(-0.933870759023)*x[0]+(-0.00667060999489)*x[1]**o+(-0.304328472241)*x[1]
        arg[1,0,0,1]=(0.716858223574)*x[0]**o+(0.0907083966301)*x[0]+(0.118004968786)*x[1]**o+(-0.989434356769)*x[1]
        arg[1,0,1,0]=(0.433505460082)*x[0]**o+(0.927792393402)*x[0]+(0.10117572965)*x[1]**o+(-0.719175235748)*x[1]
        arg[1,0,1,1]=(0.93394463477)*x[0]**o+(0.30554551436)*x[0]+(-0.221528653572)*x[1]**o+(-0.965153365935)*x[1]
        arg[1,0,2,0]=(0.217926136552)*x[0]**o+(0.75089182057)*x[0]+(0.798438119547)*x[1]**o+(-0.131885152617)*x[1]
        arg[1,0,2,1]=(0.489885370386)*x[0]**o+(-0.958785443386)*x[0]+(-0.934023631551)*x[1]**o+(-0.120727978484)*x[1]
        arg[1,1,0,0]=(-0.104659311537)*x[0]**o+(0.323817119133)*x[0]+(0.207178629335)*x[1]**o+(0.132677138305)*x[1]
        arg[1,1,0,1]=(0.855593354128)*x[0]**o+(-0.319099268106)*x[0]+(-0.75522659926)*x[1]**o+(0.398648791668)*x[1]
        arg[1,1,1,0]=(0.165499959353)*x[0]**o+(0.025416591231)*x[0]+(-0.454711374202)*x[1]**o+(-0.149643847094)*x[1]
        arg[1,1,1,1]=(0.668195333828)*x[0]**o+(-0.954315351102)*x[0]+(-0.15549992542)*x[1]**o+(0.960203842246)*x[1]
        arg[1,1,2,0]=(-0.0830862028828)*x[0]**o+(0.104413343709)*x[0]+(0.0666836881524)*x[1]**o+(-0.909918510114)*x[1]
        arg[1,1,2,1]=(-0.614061685098)*x[0]**o+(-0.696536572273)*x[0]+(0.674867602585)*x[1]**o+(0.742233774001)*x[1]
        arg[1,2,0,0]=(-0.271025278576)*x[0]**o+(0.194146585691)*x[0]+(0.369545867263)*x[1]**o+(-0.444209773003)*x[1]
        arg[1,2,0,1]=(-0.646570016511)*x[0]**o+(-0.1181935343)*x[0]+(-0.473468176784)*x[1]**o+(0.980639666617)*x[1]
        arg[1,2,1,0]=(-0.274558546949)*x[0]**o+(-0.994564294804)*x[0]+(-0.710677848448)*x[1]**o+(-0.471981673909)*x[1]
        arg[1,2,1,1]=(0.920326662139)*x[0]**o+(-0.503477706392)*x[0]+(-0.488075258363)*x[1]**o+(-0.390701293427)*x[1]
        arg[1,2,2,0]=(-0.734779609501)*x[0]**o+(-0.211522964885)*x[0]+(-0.209704158218)*x[1]**o+(0.602143419316)*x[1]
        arg[1,2,2,1]=(-0.71042927509)*x[0]**o+(-0.320801535296)*x[0]+(-0.307645214006)*x[1]**o+(0.902857741355)*x[1]
        arg[1,3,0,0]=(0.430044544461)*x[0]**o+(-0.321809854969)*x[0]+(-0.202705286059)*x[1]**o+(-0.136245903236)*x[1]
        arg[1,3,0,1]=(-0.0215232394148)*x[0]**o+(0.560515199075)*x[0]+(0.183422356152)*x[1]**o+(0.0500950744933)*x[1]
        arg[1,3,1,0]=(0.956072319541)*x[0]**o+(-0.910407504398)*x[0]+(0.710351406673)*x[1]**o+(-0.673149941902)*x[1]
        arg[1,3,1,1]=(0.512105745894)*x[0]**o+(0.617656019287)*x[0]+(0.178878380134)*x[1]**o+(-0.304972517759)*x[1]
        arg[1,3,2,0]=(0.505194196756)*x[0]**o+(-0.588278604691)*x[0]+(0.657312673212)*x[1]**o+(-0.198666173753)*x[1]
        arg[1,3,2,1]=(0.562199962955)*x[0]**o+(-0.855080919048)*x[0]+(0.798106457622)*x[1]**o+(-0.462368594285)*x[1]
        arg[1,4,0,0]=(-0.95103819151)*x[0]**o+(0.996179304819)*x[0]+(0.720809353558)*x[1]**o+(-0.357882371835)*x[1]
        arg[1,4,0,1]=(-0.337212448334)*x[0]**o+(-0.647939092937)*x[0]+(0.432152553594)*x[1]**o+(0.284393585377)*x[1]
        arg[1,4,1,0]=(0.286170290079)*x[0]**o+(0.813168170391)*x[0]+(-0.178814496497)*x[1]**o+(0.414687546663)*x[1]
        arg[1,4,1,1]=(0.00961378883254)*x[0]**o+(-0.946718643529)*x[0]+(0.828780485661)*x[1]**o+(0.54461217692)*x[1]
        arg[1,4,2,0]=(0.784403348421)*x[0]**o+(0.430078584466)*x[0]+(-0.542741131881)*x[1]**o+(0.0290828265624)*x[1]
        arg[1,4,2,1]=(-0.246343667616)*x[0]**o+(-0.27406067089)*x[0]+(-0.938704878102)*x[1]**o+(0.907921504301)*x[1]
        arg[2,0,0,0]=(-0.551056874357)*x[0]**o+(-0.697006980356)*x[0]+(-0.733447125487)*x[1]**o+(-0.241902075189)*x[1]
        arg[2,0,0,1]=(-0.75556701159)*x[0]**o+(0.155303331948)*x[0]+(0.480546881289)*x[1]**o+(-0.420954472057)*x[1]
        arg[2,0,1,0]=(0.12990139506)*x[0]**o+(-0.0990388974263)*x[0]+(-0.913089275462)*x[1]**o+(-0.766316334455)*x[1]
        arg[2,0,1,1]=(-0.96760564527)*x[0]**o+(0.94596820894)*x[0]+(-0.208507053422)*x[1]**o+(0.82642580152)*x[1]
        arg[2,0,2,0]=(-0.801660576339)*x[0]**o+(0.724091208253)*x[0]+(-0.416451086696)*x[1]**o+(-0.255959329575)*x[1]
        arg[2,0,2,1]=(0.380233654687)*x[0]**o+(-0.27467061721)*x[0]+(-0.626861863133)*x[1]**o+(0.498725464891)*x[1]
        arg[2,1,0,0]=(0.733227197081)*x[0]**o+(-0.111622822078)*x[0]+(-0.199363221943)*x[1]**o+(-0.449664634046)*x[1]
        arg[2,1,0,1]=(0.381774304841)*x[0]**o+(0.914208421879)*x[0]+(-0.545958908798)*x[1]**o+(-0.344721917598)*x[1]
        arg[2,1,1,0]=(-0.861412661903)*x[0]**o+(-0.202264578976)*x[0]+(-0.225404313047)*x[1]**o+(0.716189374265)*x[1]
        arg[2,1,1,1]=(-0.180088205315)*x[0]**o+(0.133136657032)*x[0]+(-0.341650807958)*x[1]**o+(-0.570348715256)*x[1]
        arg[2,1,2,0]=(-0.626863150737)*x[0]**o+(-0.236854925766)*x[0]+(-0.271186155613)*x[1]**o+(-0.750500826476)*x[1]
        arg[2,1,2,1]=(-0.138580929054)*x[0]**o+(-0.925057205334)*x[0]+(0.404961686669)*x[1]**o+(0.542298342705)*x[1]
        arg[2,2,0,0]=(0.453186536783)*x[0]**o+(0.368065177204)*x[0]+(0.545211472015)*x[1]**o+(0.334551805933)*x[1]
        arg[2,2,0,1]=(0.0639699038803)*x[0]**o+(0.154192875522)*x[0]+(0.101347900441)*x[1]**o+(-0.668313576443)*x[1]
        arg[2,2,1,0]=(0.465173445732)*x[0]**o+(-0.475009875122)*x[0]+(-0.748444748361)*x[1]**o+(0.620845514745)*x[1]
        arg[2,2,1,1]=(0.720975965615)*x[0]**o+(-0.266631032721)*x[0]+(0.0996389916136)*x[1]**o+(-0.96886977517)*x[1]
        arg[2,2,2,0]=(-0.396391719953)*x[0]**o+(-0.240091626001)*x[0]+(-0.42411685734)*x[1]**o+(-0.458446010754)*x[1]
        arg[2,2,2,1]=(-0.662584035264)*x[0]**o+(0.23813614536)*x[0]+(0.810893160816)*x[1]**o+(-0.130143663026)*x[1]
        arg[2,3,0,0]=(-0.651550480594)*x[0]**o+(0.370844612999)*x[0]+(-0.808002027275)*x[1]**o+(0.531083477622)*x[1]
        arg[2,3,0,1]=(-0.287184182382)*x[0]**o+(0.928877646221)*x[0]+(-0.271563784371)*x[1]**o+(-0.46908309398)*x[1]
        arg[2,3,1,0]=(0.342080138259)*x[0]**o+(0.458994887529)*x[0]+(-0.544475387926)*x[1]**o+(-0.203224249914)*x[1]
        arg[2,3,1,1]=(-0.0791009979206)*x[0]**o+(0.536480501773)*x[0]+(-0.624008015173)*x[1]**o+(-0.380504135174)*x[1]
        arg[2,3,2,0]=(-0.707089360149)*x[0]**o+(0.217716900812)*x[0]+(0.020441009853)*x[1]**o+(0.680214150153)*x[1]
        arg[2,3,2,1]=(0.0368635610611)*x[0]**o+(0.646697307691)*x[0]+(0.36118713723)*x[1]**o+(-0.535295583008)*x[1]
        arg[2,4,0,0]=(0.806938680905)*x[0]**o+(0.449096825134)*x[0]+(-0.43507826254)*x[1]**o+(-0.300956286252)*x[1]
        arg[2,4,0,1]=(-0.268005914326)*x[0]**o+(0.583132947291)*x[0]+(-0.344067495236)*x[1]**o+(-0.526011968282)*x[1]
        arg[2,4,1,0]=(-0.0953663458348)*x[0]**o+(0.834765967023)*x[0]+(-0.0410345097335)*x[1]**o+(0.493198054713)*x[1]
        arg[2,4,1,1]=(0.964777812988)*x[0]**o+(-0.791397009733)*x[0]+(-0.962108127047)*x[1]**o+(0.378083818799)*x[1]
        arg[2,4,2,0]=(0.0977715857241)*x[0]**o+(-0.118265910205)*x[0]+(-0.0181736456148)*x[1]**o+(-0.480499109439)*x[1]
        arg[2,4,2,1]=(0.396936262802)*x[0]**o+(0.979807961416)*x[0]+(0.383251286227)*x[1]**o+(-0.921599329372)*x[1]
        arg[3,0,0,0]=(-0.193869568904)*x[0]**o+(0.335533150532)*x[0]+(-0.542087879921)*x[1]**o+(0.916531699685)*x[1]
        arg[3,0,0,1]=(0.732192438113)*x[0]**o+(0.610866210623)*x[0]+(0.232455584069)*x[1]**o+(0.109577342902)*x[1]
        arg[3,0,1,0]=(0.0487640958479)*x[0]**o+(-0.250229114846)*x[0]+(-0.539343546695)*x[1]**o+(-0.456485153733)*x[1]
        arg[3,0,1,1]=(0.951844805407)*x[0]**o+(-0.811442383281)*x[0]+(-0.537581471096)*x[1]**o+(0.350217063609)*x[1]
        arg[3,0,2,0]=(0.495490307938)*x[0]**o+(0.529667417659)*x[0]+(-0.853447043865)*x[1]**o+(0.97002519322)*x[1]
        arg[3,0,2,1]=(0.38188303469)*x[0]**o+(-0.75911672537)*x[0]+(-0.67563531103)*x[1]**o+(0.384002769031)*x[1]
        arg[3,1,0,0]=(0.44471228922)*x[0]**o+(-0.958995287488)*x[0]+(-0.356730659314)*x[1]**o+(0.853749081247)*x[1]
        arg[3,1,0,1]=(-0.674677862576)*x[0]**o+(-0.651945231545)*x[0]+(-0.869871613483)*x[1]**o+(0.518493292422)*x[1]
        arg[3,1,1,0]=(-0.803923033456)*x[0]**o+(0.916638584514)*x[0]+(-0.410056071233)*x[1]**o+(-0.313811798732)*x[1]
        arg[3,1,1,1]=(-0.677586574312)*x[0]**o+(0.101317771211)*x[0]+(-0.848629733526)*x[1]**o+(0.956935407827)*x[1]
        arg[3,1,2,0]=(0.723956969878)*x[0]**o+(-0.677929268987)*x[0]+(0.168694427833)*x[1]**o+(0.675664564011)*x[1]
        arg[3,1,2,1]=(-0.0961257584415)*x[0]**o+(0.870273004118)*x[0]+(0.0979731186867)*x[1]**o+(-0.561786099655)*x[1]
        arg[3,2,0,0]=(-0.506857925404)*x[0]**o+(0.950429186893)*x[0]+(-0.765686972231)*x[1]**o+(0.300787357163)*x[1]
        arg[3,2,0,1]=(-0.479464705052)*x[0]**o+(-0.814881390688)*x[0]+(-0.549033177495)*x[1]**o+(0.875164713771)*x[1]
        arg[3,2,1,0]=(0.984579775952)*x[0]**o+(0.822464384935)*x[0]+(-0.352937841183)*x[1]**o+(-0.389516730371)*x[1]
        arg[3,2,1,1]=(-0.75230207503)*x[0]**o+(0.921759892886)*x[0]+(0.683120790039)*x[1]**o+(0.470250085193)*x[1]
        arg[3,2,2,0]=(-0.257935272452)*x[0]**o+(-0.437370511818)*x[0]+(-0.350363036315)*x[1]**o+(-0.232415372202)*x[1]
        arg[3,2,2,1]=(0.0784702766009)*x[0]**o+(0.460959683819)*x[0]+(0.758201028024)*x[1]**o+(0.101597092257)*x[1]
        arg[3,3,0,0]=(0.292597181224)*x[0]**o+(-0.322992033991)*x[0]+(-0.536413055541)*x[1]**o+(-0.95609054732)*x[1]
        arg[3,3,0,1]=(0.174134715952)*x[0]**o+(0.811729326606)*x[0]+(0.89040777319)*x[1]**o+(0.893572419966)*x[1]
        arg[3,3,1,0]=(0.768234215847)*x[0]**o+(0.36391719033)*x[0]+(0.598889528719)*x[1]**o+(0.152035960251)*x[1]
        arg[3,3,1,1]=(-0.77938361663)*x[0]**o+(-0.448945619013)*x[0]+(0.471242869055)*x[1]**o+(0.112888214554)*x[1]
        arg[3,3,2,0]=(-0.00617753872016)*x[0]**o+(0.0155218100192)*x[0]+(0.674100983537)*x[1]**o+(0.211406831421)*x[1]
        arg[3,3,2,1]=(0.124470359441)*x[0]**o+(0.117393617733)*x[0]+(-0.297791856337)*x[1]**o+(-0.210405798241)*x[1]
        arg[3,4,0,0]=(0.139920132608)*x[0]**o+(-0.0073381380034)*x[0]+(-0.77267392403)*x[1]**o+(0.792299687107)*x[1]
        arg[3,4,0,1]=(0.328592839797)*x[0]**o+(-0.758899359217)*x[0]+(0.730856047416)*x[1]**o+(-0.389548922357)*x[1]
        arg[3,4,1,0]=(-0.239362896774)*x[0]**o+(0.513205289336)*x[0]+(-0.393781550386)*x[1]**o+(0.229968513732)*x[1]
        arg[3,4,1,1]=(0.777985150139)*x[0]**o+(0.273278793895)*x[0]+(0.17782134987)*x[1]**o+(0.491127331923)*x[1]
        arg[3,4,2,0]=(0.348482781008)*x[0]**o+(0.458696185739)*x[0]+(-0.77882422986)*x[1]**o+(0.455739446004)*x[1]
        arg[3,4,2,1]=(-0.396751338134)*x[0]**o+(0.449026376364)*x[0]+(0.308553967581)*x[1]**o+(-0.777653415416)*x[1]
        ref[0,0,0,0]=(-0.848175950825)/(o+1.)+(-0.060984579579)+(0.423100205741)*0.5**o
        ref[0,0,0,1]=(-0.838397517739)/(o+1.)+(-0.289891489711)+(-0.739687474039)*0.5**o
        ref[0,0,1,0]=(-0.616976579504)/(o+1.)+(-0.556983314949)+(-0.423462958438)*0.5**o
        ref[0,0,1,1]=(-0.953014661461)/(o+1.)+(0.500001792807)+(-0.432533121428)*0.5**o
        ref[0,0,2,0]=(-0.156191993934)/(o+1.)+(0.883028259835)+(-0.891613775753)*0.5**o
        ref[0,0,2,1]=(-0.68287286477)/(o+1.)+(0.0522510271596)+(0.909573147888)*0.5**o
        ref[0,1,0,0]=(-0.631624427826)/(o+1.)+(-0.340055925409)+(-0.39808363424)*0.5**o
        ref[0,1,0,1]=(-0.0750972377036)/(o+1.)+(0.566030507884)+(-0.748658247224)*0.5**o
        ref[0,1,1,0]=(-0.163646238293)/(o+1.)+(-0.578922683831)+(0.877627912148)*0.5**o
        ref[0,1,1,1]=(0.0667564973731)/(o+1.)+(0.438204826712)+(-0.592580581615)*0.5**o
        ref[0,1,2,0]=(-0.899276958155)/(o+1.)+(-0.684200628566)+(-0.505569292541)*0.5**o
        ref[0,1,2,1]=(0.912452712698)/(o+1.)+(-0.0949204253183)+(-0.215752366701)*0.5**o
        ref[0,2,0,0]=(0.267409202785)/(o+1.)+(0.560565813511)+(0.684112319204)*0.5**o
        ref[0,2,0,1]=(-0.152472725029)/(o+1.)+(-0.48936178263)+(0.552438501301)*0.5**o
        ref[0,2,1,0]=(-0.873134889407)/(o+1.)+(-0.0460827720855)+(-0.31213422297)*0.5**o
        ref[0,2,1,1]=(-0.761310589424)/(o+1.)+(-0.676170252134)+(0.883048532408)*0.5**o
        ref[0,2,2,0]=(-0.936381308418)/(o+1.)+(-0.00265386081956)+(-0.324489237508)*0.5**o
        ref[0,2,2,1]=(0.419811305063)/(o+1.)+(0.109993113022)+(-0.971735193469)*0.5**o
        ref[0,3,0,0]=(0.707279059071)/(o+1.)+(-0.86543390192)+(-0.305553454905)*0.5**o
        ref[0,3,0,1]=(0.819264129099)/(o+1.)+(0.0693620181083)+(0.803934620926)*0.5**o
        ref[0,3,1,0]=(-0.60262522048)/(o+1.)+(-0.484758941318)+(0.996632326055)*0.5**o
        ref[0,3,1,1]=(0.722837842981)/(o+1.)+(0.309131271645)+(0.402799911553)*0.5**o
        ref[0,3,2,0]=(0.62363567076)/(o+1.)+(-0.700867024115)+(0.937966588808)*0.5**o
        ref[0,3,2,1]=(-0.950660964237)/(o+1.)+(0.261278793995)+(-0.0389713032301)*0.5**o
        ref[0,4,0,0]=(-0.872148906918)/(o+1.)+(-0.692114808131)+(-0.589529640529)*0.5**o
        ref[0,4,0,1]=(0.329194160571)/(o+1.)+(0.651123856087)+(0.78615039967)*0.5**o
        ref[0,4,1,0]=(0.107007555422)/(o+1.)+(0.132524151069)+(-0.801187683466)*0.5**o
        ref[0,4,1,1]=(-0.720479146591)/(o+1.)+(0.517763642707)+(0.817551079419)*0.5**o
        ref[0,4,2,0]=(0.766214937295)/(o+1.)+(0.432935513335)+(-0.245391293539)*0.5**o
        ref[0,4,2,1]=(-0.993826355778)/(o+1.)+(-0.776337331653)+(-0.780163545255)*0.5**o
        ref[1,0,0,0]=(-0.00667060999489)/(o+1.)+(-0.619099615632)+(0.0998104654851)*0.5**o
        ref[1,0,0,1]=(0.118004968786)/(o+1.)+(-0.449362980069)+(0.716858223574)*0.5**o
        ref[1,0,1,0]=(0.10117572965)/(o+1.)+(0.104308578827)+(0.433505460082)*0.5**o
        ref[1,0,1,1]=(-0.221528653572)/(o+1.)+(-0.329803925788)+(0.93394463477)*0.5**o
        ref[1,0,2,0]=(0.798438119547)/(o+1.)+(0.309503333977)+(0.217926136552)*0.5**o
        ref[1,0,2,1]=(-0.934023631551)/(o+1.)+(-0.539756710935)+(0.489885370386)*0.5**o
        ref[1,1,0,0]=(0.207178629335)/(o+1.)+(0.228247128719)+(-0.104659311537)*0.5**o
        ref[1,1,0,1]=(-0.75522659926)/(o+1.)+(0.0397747617809)+(0.855593354128)*0.5**o
        ref[1,1,1,0]=(-0.454711374202)/(o+1.)+(-0.0621136279316)+(0.165499959353)*0.5**o
        ref[1,1,1,1]=(-0.15549992542)/(o+1.)+(0.00294424557217)+(0.668195333828)*0.5**o
        ref[1,1,2,0]=(0.0666836881524)/(o+1.)+(-0.402752583203)+(-0.0830862028828)*0.5**o
        ref[1,1,2,1]=(0.674867602585)/(o+1.)+(0.0228486008642)+(-0.614061685098)*0.5**o
        ref[1,2,0,0]=(0.369545867263)/(o+1.)+(-0.125031593656)+(-0.271025278576)*0.5**o
        ref[1,2,0,1]=(-0.473468176784)/(o+1.)+(0.431223066159)+(-0.646570016511)*0.5**o
        ref[1,2,1,0]=(-0.710677848448)/(o+1.)+(-0.733272984356)+(-0.274558546949)*0.5**o
        ref[1,2,1,1]=(-0.488075258363)/(o+1.)+(-0.44708949991)+(0.920326662139)*0.5**o
        ref[1,2,2,0]=(-0.209704158218)/(o+1.)+(0.195310227215)+(-0.734779609501)*0.5**o
        ref[1,2,2,1]=(-0.307645214006)/(o+1.)+(0.29102810303)+(-0.71042927509)*0.5**o
        ref[1,3,0,0]=(-0.202705286059)/(o+1.)+(-0.229027879102)+(0.430044544461)*0.5**o
        ref[1,3,0,1]=(0.183422356152)/(o+1.)+(0.305305136784)+(-0.0215232394148)*0.5**o
        ref[1,3,1,0]=(0.710351406673)/(o+1.)+(-0.79177872315)+(0.956072319541)*0.5**o
        ref[1,3,1,1]=(0.178878380134)/(o+1.)+(0.156341750764)+(0.512105745894)*0.5**o
        ref[1,3,2,0]=(0.657312673212)/(o+1.)+(-0.393472389222)+(0.505194196756)*0.5**o
        ref[1,3,2,1]=(0.798106457622)/(o+1.)+(-0.658724756667)+(0.562199962955)*0.5**o
        ref[1,4,0,0]=(0.720809353558)/(o+1.)+(0.319148466492)+(-0.95103819151)*0.5**o
        ref[1,4,0,1]=(0.432152553594)/(o+1.)+(-0.18177275378)+(-0.337212448334)*0.5**o
        ref[1,4,1,0]=(-0.178814496497)/(o+1.)+(0.613927858527)+(0.286170290079)*0.5**o
        ref[1,4,1,1]=(0.828780485661)/(o+1.)+(-0.201053233304)+(0.00961378883254)*0.5**o
        ref[1,4,2,0]=(-0.542741131881)/(o+1.)+(0.229580705514)+(0.784403348421)*0.5**o
        ref[1,4,2,1]=(-0.938704878102)/(o+1.)+(0.316930416706)+(-0.246343667616)*0.5**o
        ref[2,0,0,0]=(-0.733447125487)/(o+1.)+(-0.469454527772)+(-0.551056874357)*0.5**o
        ref[2,0,0,1]=(0.480546881289)/(o+1.)+(-0.132825570055)+(-0.75556701159)*0.5**o
        ref[2,0,1,0]=(-0.913089275462)/(o+1.)+(-0.432677615941)+(0.12990139506)*0.5**o
        ref[2,0,1,1]=(-0.208507053422)/(o+1.)+(0.88619700523)+(-0.96760564527)*0.5**o
        ref[2,0,2,0]=(-0.416451086696)/(o+1.)+(0.234065939339)+(-0.801660576339)*0.5**o
        ref[2,0,2,1]=(-0.626861863133)/(o+1.)+(0.112027423841)+(0.380233654687)*0.5**o
        ref[2,1,0,0]=(-0.199363221943)/(o+1.)+(-0.280643728062)+(0.733227197081)*0.5**o
        ref[2,1,0,1]=(-0.545958908798)/(o+1.)+(0.284743252141)+(0.381774304841)*0.5**o
        ref[2,1,1,0]=(-0.225404313047)/(o+1.)+(0.256962397644)+(-0.861412661903)*0.5**o
        ref[2,1,1,1]=(-0.341650807958)/(o+1.)+(-0.218606029112)+(-0.180088205315)*0.5**o
        ref[2,1,2,0]=(-0.271186155613)/(o+1.)+(-0.493677876121)+(-0.626863150737)*0.5**o
        ref[2,1,2,1]=(0.404961686669)/(o+1.)+(-0.191379431315)+(-0.138580929054)*0.5**o
        ref[2,2,0,0]=(0.545211472015)/(o+1.)+(0.351308491569)+(0.453186536783)*0.5**o
        ref[2,2,0,1]=(0.101347900441)/(o+1.)+(-0.257060350461)+(0.0639699038803)*0.5**o
        ref[2,2,1,0]=(-0.748444748361)/(o+1.)+(0.0729178198114)+(0.465173445732)*0.5**o
        ref[2,2,1,1]=(0.0996389916136)/(o+1.)+(-0.617750403946)+(0.720975965615)*0.5**o
        ref[2,2,2,0]=(-0.42411685734)/(o+1.)+(-0.349268818378)+(-0.396391719953)*0.5**o
        ref[2,2,2,1]=(0.810893160816)/(o+1.)+(0.053996241167)+(-0.662584035264)*0.5**o
        ref[2,3,0,0]=(-0.808002027275)/(o+1.)+(0.45096404531)+(-0.651550480594)*0.5**o
        ref[2,3,0,1]=(-0.271563784371)/(o+1.)+(0.22989727612)+(-0.287184182382)*0.5**o
        ref[2,3,1,0]=(-0.544475387926)/(o+1.)+(0.127885318807)+(0.342080138259)*0.5**o
        ref[2,3,1,1]=(-0.624008015173)/(o+1.)+(0.0779881832997)+(-0.0791009979206)*0.5**o
        ref[2,3,2,0]=(0.020441009853)/(o+1.)+(0.448965525483)+(-0.707089360149)*0.5**o
        ref[2,3,2,1]=(0.36118713723)/(o+1.)+(0.0557008623413)+(0.0368635610611)*0.5**o
        ref[2,4,0,0]=(-0.43507826254)/(o+1.)+(0.0740702694408)+(0.806938680905)*0.5**o
        ref[2,4,0,1]=(-0.344067495236)/(o+1.)+(0.0285604895043)+(-0.268005914326)*0.5**o
        ref[2,4,1,0]=(-0.0410345097335)/(o+1.)+(0.663982010868)+(-0.0953663458348)*0.5**o
        ref[2,4,1,1]=(-0.962108127047)/(o+1.)+(-0.206656595467)+(0.964777812988)*0.5**o
        ref[2,4,2,0]=(-0.0181736456148)/(o+1.)+(-0.299382509822)+(0.0977715857241)*0.5**o
        ref[2,4,2,1]=(0.383251286227)/(o+1.)+(0.0291043160221)+(0.396936262802)*0.5**o
        ref[3,0,0,0]=(-0.542087879921)/(o+1.)+(0.626032425109)+(-0.193869568904)*0.5**o
        ref[3,0,0,1]=(0.232455584069)/(o+1.)+(0.360221776762)+(0.732192438113)*0.5**o
        ref[3,0,1,0]=(-0.539343546695)/(o+1.)+(-0.353357134289)+(0.0487640958479)*0.5**o
        ref[3,0,1,1]=(-0.537581471096)/(o+1.)+(-0.230612659836)+(0.951844805407)*0.5**o
        ref[3,0,2,0]=(-0.853447043865)/(o+1.)+(0.74984630544)+(0.495490307938)*0.5**o
        ref[3,0,2,1]=(-0.67563531103)/(o+1.)+(-0.187556978169)+(0.38188303469)*0.5**o
        ref[3,1,0,0]=(-0.356730659314)/(o+1.)+(-0.0526231031206)+(0.44471228922)*0.5**o
        ref[3,1,0,1]=(-0.869871613483)/(o+1.)+(-0.0667259695618)+(-0.674677862576)*0.5**o
        ref[3,1,1,0]=(-0.410056071233)/(o+1.)+(0.301413392891)+(-0.803923033456)*0.5**o
        ref[3,1,1,1]=(-0.848629733526)/(o+1.)+(0.529126589519)+(-0.677586574312)*0.5**o
        ref[3,1,2,0]=(0.168694427833)/(o+1.)+(-0.00113235248793)+(0.723956969878)*0.5**o
        ref[3,1,2,1]=(0.0979731186867)/(o+1.)+(0.154243452232)+(-0.0961257584415)*0.5**o
        ref[3,2,0,0]=(-0.765686972231)/(o+1.)+(0.625608272028)+(-0.506857925404)*0.5**o
        ref[3,2,0,1]=(-0.549033177495)/(o+1.)+(0.0301416615418)+(-0.479464705052)*0.5**o
        ref[3,2,1,0]=(-0.352937841183)/(o+1.)+(0.216473827282)+(0.984579775952)*0.5**o
        ref[3,2,1,1]=(0.683120790039)/(o+1.)+(0.696004989039)+(-0.75230207503)*0.5**o
        ref[3,2,2,0]=(-0.350363036315)/(o+1.)+(-0.33489294201)+(-0.257935272452)*0.5**o
        ref[3,2,2,1]=(0.758201028024)/(o+1.)+(0.281278388038)+(0.0784702766009)*0.5**o
        ref[3,3,0,0]=(-0.536413055541)/(o+1.)+(-0.639541290655)+(0.292597181224)*0.5**o
        ref[3,3,0,1]=(0.89040777319)/(o+1.)+(0.852650873286)+(0.174134715952)*0.5**o
        ref[3,3,1,0]=(0.598889528719)/(o+1.)+(0.257976575291)+(0.768234215847)*0.5**o
        ref[3,3,1,1]=(0.471242869055)/(o+1.)+(-0.168028702229)+(-0.77938361663)*0.5**o
        ref[3,3,2,0]=(0.674100983537)/(o+1.)+(0.11346432072)+(-0.00617753872016)*0.5**o
        ref[3,3,2,1]=(-0.297791856337)/(o+1.)+(-0.0465060902539)+(0.124470359441)*0.5**o
        ref[3,4,0,0]=(-0.77267392403)/(o+1.)+(0.392480774552)+(0.139920132608)*0.5**o
        ref[3,4,0,1]=(0.730856047416)/(o+1.)+(-0.574224140787)+(0.328592839797)*0.5**o
        ref[3,4,1,0]=(-0.393781550386)/(o+1.)+(0.371586901534)+(-0.239362896774)*0.5**o
        ref[3,4,1,1]=(0.17782134987)/(o+1.)+(0.382203062909)+(0.777985150139)*0.5**o
        ref[3,4,2,0]=(-0.77882422986)/(o+1.)+(0.457217815872)+(0.348482781008)*0.5**o
        ref[3,4,2,1]=(0.308553967581)/(o+1.)+(-0.164313519526)+(-0.396751338134)*0.5**o
      else:
        arg[0,0,0,0]=(0.663998508168)*x[0]**o+(0.797340164931)*x[0]+(0.285182001599)*x[1]**o+(0.99720606542)*x[1]+(0.127610260108)*x[2]**o+(0.0141819362674)*x[2]
        arg[0,0,0,1]=(-0.010130464476)*x[0]**o+(-0.694140348803)*x[0]+(-0.554528445238)*x[1]**o+(0.192639775817)*x[1]+(-0.520164891934)*x[2]**o+(0.316830081989)*x[2]
        arg[0,0,1,0]=(-0.176542289385)*x[0]**o+(-0.312451025451)*x[0]+(0.33951668081)*x[1]**o+(-0.66674551066)*x[1]+(-0.0515929546862)*x[2]**o+(0.688180995863)*x[2]
        arg[0,0,1,1]=(-0.98152572229)*x[0]**o+(0.156411605196)*x[0]+(0.588065240125)*x[1]**o+(0.665830732172)*x[1]+(-0.834625744106)*x[2]**o+(-0.138276534671)*x[2]
        arg[0,0,2,0]=(0.555897422173)*x[0]**o+(0.0269431977874)*x[0]+(0.671161746561)*x[1]**o+(0.0857445285374)*x[1]+(0.578027911522)*x[2]**o+(0.217805007393)*x[2]
        arg[0,0,2,1]=(0.23127040923)*x[0]**o+(0.624353504617)*x[0]+(-0.433915231286)*x[1]**o+(-0.860173971996)*x[1]+(-0.801472434932)*x[2]**o+(-0.779575445432)*x[2]
        arg[0,1,0,0]=(0.0202445177842)*x[0]**o+(-0.0654289514039)*x[0]+(-0.877581892208)*x[1]**o+(-0.231833216639)*x[1]+(0.609752401432)*x[2]**o+(-0.802673983603)*x[2]
        arg[0,1,0,1]=(0.99935098551)*x[0]**o+(0.665639246142)*x[0]+(0.593510187831)*x[1]**o+(0.213336090684)*x[1]+(0.232667772738)*x[2]**o+(-0.651386262633)*x[2]
        arg[0,1,1,0]=(0.151799117419)*x[0]**o+(-0.34631043288)*x[0]+(-0.264898731671)*x[1]**o+(0.697527693137)*x[1]+(0.389030412719)*x[2]**o+(0.613338381577)*x[2]
        arg[0,1,1,1]=(-0.51422532626)*x[0]**o+(0.228040429268)*x[0]+(-0.276760984751)*x[1]**o+(0.379484010198)*x[1]+(0.896837565192)*x[2]**o+(0.195866552401)*x[2]
        arg[0,1,2,0]=(0.908244888839)*x[0]**o+(0.217098236162)*x[0]+(0.798725721675)*x[1]**o+(0.216635630826)*x[1]+(0.960227216319)*x[2]**o+(-0.885381182879)*x[2]
        arg[0,1,2,1]=(-0.872307766634)*x[0]**o+(0.843197737136)*x[0]+(-0.895220198791)*x[1]**o+(0.866433237518)*x[1]+(0.560344834226)*x[2]**o+(0.895797915384)*x[2]
        arg[0,2,0,0]=(0.00173619592035)*x[0]**o+(-0.209651858019)*x[0]+(0.323249367733)*x[1]**o+(-0.848445788048)*x[1]+(-0.747443572813)*x[2]**o+(0.342929929066)*x[2]
        arg[0,2,0,1]=(0.998184336552)*x[0]**o+(-0.669882578313)*x[0]+(-0.632794054696)*x[1]**o+(-0.779832061924)*x[1]+(0.0588815237456)*x[2]**o+(0.419538634163)*x[2]
        arg[0,2,1,0]=(-0.921234331968)*x[0]**o+(0.319724478943)*x[0]+(0.867192523265)*x[1]**o+(-0.183047755109)*x[1]+(-0.753047661507)*x[2]**o+(-0.797179040607)*x[2]
        arg[0,2,1,1]=(0.153578015941)*x[0]**o+(0.887820807812)*x[0]+(-0.607944647299)*x[1]**o+(-0.0487413965527)*x[1]+(-0.812369692315)*x[2]**o+(0.323102876844)*x[2]
        arg[0,2,2,0]=(0.29601674794)*x[0]**o+(-0.108738759628)*x[0]+(0.242837345895)*x[1]**o+(-0.324487554664)*x[1]+(0.584399369463)*x[2]**o+(0.683877139789)*x[2]
        arg[0,2,2,1]=(0.0580007647099)*x[0]**o+(0.214488016057)*x[0]+(0.669087265419)*x[1]**o+(-0.311999860513)*x[1]+(0.678940598195)*x[2]**o+(0.658847203441)*x[2]
        arg[0,3,0,0]=(-0.337999890505)*x[0]**o+(-0.842479530409)*x[0]+(0.786376708623)*x[1]**o+(0.597298234359)*x[1]+(0.495442668997)*x[2]**o+(-0.183828960943)*x[2]
        arg[0,3,0,1]=(-0.825813329629)*x[0]**o+(0.197040670392)*x[0]+(0.997784261354)*x[1]**o+(-0.355290279278)*x[1]+(0.657909516664)*x[2]**o+(-0.568281670001)*x[2]
        arg[0,3,1,0]=(0.699313512079)*x[0]**o+(-0.916350147519)*x[0]+(-0.311605808179)*x[1]**o+(0.222618888987)*x[1]+(0.872150798976)*x[2]**o+(0.139773150125)*x[2]
        arg[0,3,1,1]=(0.140764124286)*x[0]**o+(0.994013777309)*x[0]+(-0.0792107151558)*x[1]**o+(-0.285357856751)*x[1]+(-0.800285107223)*x[2]**o+(0.413029089978)*x[2]
        arg[0,3,2,0]=(-0.68177752688)*x[0]**o+(0.0339522871939)*x[0]+(-0.641399841911)*x[1]**o+(0.263279621719)*x[1]+(-0.406115766497)*x[2]**o+(-0.774639092638)*x[2]
        arg[0,3,2,1]=(0.655685968811)*x[0]**o+(0.20085918281)*x[0]+(-0.753491228829)*x[1]**o+(0.393669658887)*x[1]+(0.339469131428)*x[2]**o+(0.424803724709)*x[2]
        arg[0,4,0,0]=(-0.18027010068)*x[0]**o+(-0.773007683353)*x[0]+(0.767133020895)*x[1]**o+(-0.810449138606)*x[1]+(0.295421955604)*x[2]**o+(-0.955926880008)*x[2]
        arg[0,4,0,1]=(-0.914768182433)*x[0]**o+(-0.67024319498)*x[0]+(0.533244666277)*x[1]**o+(-0.705719628704)*x[1]+(-0.963633923897)*x[2]**o+(-0.0818895784023)*x[2]
        arg[0,4,1,0]=(0.274727423835)*x[0]**o+(0.838218676826)*x[0]+(0.269989900726)*x[1]**o+(0.650600179293)*x[1]+(-0.855210211935)*x[2]**o+(-0.800261683904)*x[2]
        arg[0,4,1,1]=(0.257574026756)*x[0]**o+(-0.787643878696)*x[0]+(-0.619136916008)*x[1]**o+(-0.242580269843)*x[1]+(0.0229959873555)*x[2]**o+(-0.685725009107)*x[2]
        arg[0,4,2,0]=(0.703185665611)*x[0]**o+(0.37686548395)*x[0]+(-0.414901304236)*x[1]**o+(0.310335568741)*x[1]+(-0.740764649572)*x[2]**o+(-0.389356987349)*x[2]
        arg[0,4,2,1]=(-0.153792811673)*x[0]**o+(-0.179777301012)*x[0]+(0.694402537357)*x[1]**o+(-0.928918851291)*x[1]+(-0.439307255129)*x[2]**o+(0.148671231214)*x[2]
        arg[1,0,0,0]=(0.298044833507)*x[0]**o+(0.538000859279)*x[0]+(-0.405432041063)*x[1]**o+(-0.8912548734)*x[1]+(0.929554154255)*x[2]**o+(-0.515905504512)*x[2]
        arg[1,0,0,1]=(-0.575485929806)*x[0]**o+(0.757951009672)*x[0]+(-0.815024122535)*x[1]**o+(0.846519632117)*x[1]+(0.594901198403)*x[2]**o+(-0.672034020351)*x[2]
        arg[1,0,1,0]=(0.998358005707)*x[0]**o+(-0.144729504804)*x[0]+(0.463445062889)*x[1]**o+(0.428918148475)*x[1]+(-0.172684103233)*x[2]**o+(-0.394742387769)*x[2]
        arg[1,0,1,1]=(0.783593367143)*x[0]**o+(-0.269882783638)*x[0]+(-0.880336283806)*x[1]**o+(0.337087191943)*x[1]+(-0.819512478233)*x[2]**o+(0.582924902797)*x[2]
        arg[1,0,2,0]=(-0.42345692756)*x[0]**o+(-0.251910598711)*x[0]+(-0.882193473121)*x[1]**o+(0.535405977525)*x[1]+(-0.117109433242)*x[2]**o+(0.99565775159)*x[2]
        arg[1,0,2,1]=(-0.790917746681)*x[0]**o+(-0.291814540513)*x[0]+(-0.0828604085821)*x[1]**o+(-0.0715382319561)*x[1]+(-0.188453557262)*x[2]**o+(0.297596604241)*x[2]
        arg[1,1,0,0]=(0.944343622558)*x[0]**o+(0.537893843235)*x[0]+(-0.370697218599)*x[1]**o+(-0.043709074747)*x[1]+(-0.803318845377)*x[2]**o+(0.814392946479)*x[2]
        arg[1,1,0,1]=(-0.246711023019)*x[0]**o+(-0.847902406307)*x[0]+(0.062135604814)*x[1]**o+(0.581341495124)*x[1]+(0.561484848132)*x[2]**o+(0.06820017559)*x[2]
        arg[1,1,1,0]=(-0.530782063061)*x[0]**o+(-0.951246029994)*x[0]+(-0.809647094351)*x[1]**o+(0.198165508669)*x[1]+(0.591475451485)*x[2]**o+(-0.0567469904903)*x[2]
        arg[1,1,1,1]=(-0.710844761085)*x[0]**o+(0.742146961382)*x[0]+(0.482950602277)*x[1]**o+(-0.701576649521)*x[1]+(0.752305355358)*x[2]**o+(0.658478074845)*x[2]
        arg[1,1,2,0]=(0.574657505707)*x[0]**o+(-0.268094098242)*x[0]+(-0.79745549318)*x[1]**o+(0.994732385826)*x[1]+(0.36752110393)*x[2]**o+(0.0733521344978)*x[2]
        arg[1,1,2,1]=(-0.365373199615)*x[0]**o+(0.295532428151)*x[0]+(-0.916049089517)*x[1]**o+(0.562876895658)*x[1]+(0.698818392781)*x[2]**o+(0.413166153315)*x[2]
        arg[1,2,0,0]=(0.544776615353)*x[0]**o+(-0.647889580869)*x[0]+(-0.0569263298125)*x[1]**o+(0.145083519829)*x[1]+(-0.0456368749942)*x[2]**o+(0.125915481669)*x[2]
        arg[1,2,0,1]=(0.11684721915)*x[0]**o+(-0.793592672443)*x[0]+(-0.428585283805)*x[1]**o+(0.280358391066)*x[1]+(0.72731741847)*x[2]**o+(-0.320459136831)*x[2]
        arg[1,2,1,0]=(0.561736948232)*x[0]**o+(0.833411538285)*x[0]+(0.361519222525)*x[1]**o+(-0.785948355356)*x[1]+(-0.595938528895)*x[2]**o+(0.720247127573)*x[2]
        arg[1,2,1,1]=(0.578630452142)*x[0]**o+(0.218238639275)*x[0]+(0.792496779641)*x[1]**o+(-0.818431209813)*x[1]+(-0.252522807953)*x[2]**o+(0.648505893606)*x[2]
        arg[1,2,2,0]=(-0.0269203575755)*x[0]**o+(-0.586454813205)*x[0]+(-0.239885637578)*x[1]**o+(0.186263805313)*x[1]+(0.585909451795)*x[2]**o+(-0.630731036615)*x[2]
        arg[1,2,2,1]=(-0.534642633404)*x[0]**o+(0.921034930282)*x[0]+(-0.326672365556)*x[1]**o+(0.690509146367)*x[1]+(-0.802567575176)*x[2]**o+(-0.235903818924)*x[2]
        arg[1,3,0,0]=(0.892302802156)*x[0]**o+(-0.528836075071)*x[0]+(-0.806520835766)*x[1]**o+(0.771321197577)*x[1]+(-0.545545197249)*x[2]**o+(-0.633185919719)*x[2]
        arg[1,3,0,1]=(0.883702447985)*x[0]**o+(0.698176356191)*x[0]+(-0.348189466787)*x[1]**o+(-0.390058291315)*x[1]+(0.805736008481)*x[2]**o+(0.861787745203)*x[2]
        arg[1,3,1,0]=(0.492527250981)*x[0]**o+(0.635712008949)*x[0]+(-0.464397346193)*x[1]**o+(-0.820952575379)*x[1]+(0.235015001164)*x[2]**o+(-0.846160845231)*x[2]
        arg[1,3,1,1]=(0.372499842901)*x[0]**o+(-0.524824414377)*x[0]+(0.553450359486)*x[1]**o+(-0.187372498756)*x[1]+(-0.389983195918)*x[2]**o+(-0.310421627605)*x[2]
        arg[1,3,2,0]=(0.533098362924)*x[0]**o+(0.917820078197)*x[0]+(-0.557683070597)*x[1]**o+(-0.388498248383)*x[1]+(-0.551866333066)*x[2]**o+(0.696873468511)*x[2]
        arg[1,3,2,1]=(-0.909038123273)*x[0]**o+(0.19323884412)*x[0]+(-0.00134861813725)*x[1]**o+(-0.858193032021)*x[1]+(0.0436856504635)*x[2]**o+(-0.646994265986)*x[2]
        arg[1,4,0,0]=(-0.974474210185)*x[0]**o+(-0.39786371112)*x[0]+(-0.829024534519)*x[1]**o+(0.836151642189)*x[1]+(0.679654655991)*x[2]**o+(0.750995377779)*x[2]
        arg[1,4,0,1]=(0.964745405431)*x[0]**o+(0.269189443297)*x[0]+(0.531636623506)*x[1]**o+(-0.522687387383)*x[1]+(0.463500385567)*x[2]**o+(0.160556224359)*x[2]
        arg[1,4,1,0]=(-0.714725894011)*x[0]**o+(0.0478807291181)*x[0]+(0.591494552904)*x[1]**o+(-0.240293119817)*x[1]+(-0.0463360360003)*x[2]**o+(-0.932835908096)*x[2]
        arg[1,4,1,1]=(-0.809072545597)*x[0]**o+(-0.963335650229)*x[0]+(-0.781885522838)*x[1]**o+(0.822395174213)*x[1]+(-0.42590899181)*x[2]**o+(-0.237357227214)*x[2]
        arg[1,4,2,0]=(-0.21281876112)*x[0]**o+(-0.131640536998)*x[0]+(-0.648680371617)*x[1]**o+(0.555759774031)*x[1]+(-0.911030244394)*x[2]**o+(0.962859403252)*x[2]
        arg[1,4,2,1]=(-0.247930702516)*x[0]**o+(0.509829884687)*x[0]+(0.0387816417333)*x[1]**o+(-0.605945140743)*x[1]+(0.187880415018)*x[2]**o+(-0.813242364046)*x[2]
        arg[2,0,0,0]=(0.616352869937)*x[0]**o+(-0.930708209966)*x[0]+(-0.874778661809)*x[1]**o+(-0.330225154904)*x[1]+(0.632020336392)*x[2]**o+(0.205631598228)*x[2]
        arg[2,0,0,1]=(-0.85704374519)*x[0]**o+(0.99958496465)*x[0]+(0.807095767473)*x[1]**o+(0.63097388144)*x[1]+(0.000665614692828)*x[2]**o+(0.418070858864)*x[2]
        arg[2,0,1,0]=(0.326870102902)*x[0]**o+(0.565855791221)*x[0]+(-0.39418644908)*x[1]**o+(-0.96081735968)*x[1]+(0.937255293725)*x[2]**o+(0.533015538451)*x[2]
        arg[2,0,1,1]=(-0.699939842176)*x[0]**o+(0.808055402204)*x[0]+(0.448948790025)*x[1]**o+(-0.720122241439)*x[1]+(0.0532200858319)*x[2]**o+(-0.204851171261)*x[2]
        arg[2,0,2,0]=(-0.985996827188)*x[0]**o+(-0.604602591167)*x[0]+(0.445903925662)*x[1]**o+(-0.163135114049)*x[1]+(-0.264281316158)*x[2]**o+(-0.381814212863)*x[2]
        arg[2,0,2,1]=(-0.195357773436)*x[0]**o+(0.73285122682)*x[0]+(-0.914846292757)*x[1]**o+(-0.190524885095)*x[1]+(-0.103440307801)*x[2]**o+(-0.0309647945217)*x[2]
        arg[2,1,0,0]=(0.601652939329)*x[0]**o+(-0.757171386065)*x[0]+(-0.91147090556)*x[1]**o+(-0.665258414941)*x[1]+(-0.0504245123191)*x[2]**o+(-0.274928427711)*x[2]
        arg[2,1,0,1]=(0.534038022604)*x[0]**o+(-0.977365399298)*x[0]+(0.884656682614)*x[1]**o+(0.0866421048629)*x[1]+(-0.70687555879)*x[2]**o+(-0.99928465036)*x[2]
        arg[2,1,1,0]=(-0.361196978257)*x[0]**o+(0.192559020092)*x[0]+(-0.0767715201942)*x[1]**o+(0.222254953069)*x[1]+(-0.128362555436)*x[2]**o+(-0.667485829139)*x[2]
        arg[2,1,1,1]=(0.00930987921229)*x[0]**o+(0.269207633273)*x[0]+(0.943798095916)*x[1]**o+(0.483123564952)*x[1]+(-0.231669365037)*x[2]**o+(-0.877005245489)*x[2]
        arg[2,1,2,0]=(-0.181479246679)*x[0]**o+(0.619551810132)*x[0]+(0.499400298722)*x[1]**o+(-0.307864349855)*x[1]+(0.19303040809)*x[2]**o+(0.523640816928)*x[2]
        arg[2,1,2,1]=(-0.99665050175)*x[0]**o+(-0.647775674514)*x[0]+(-0.0015481673821)*x[1]**o+(-0.749198451256)*x[1]+(0.652528193579)*x[2]**o+(-0.0194127861341)*x[2]
        arg[2,2,0,0]=(0.870069083779)*x[0]**o+(0.179747121592)*x[0]+(0.387523112512)*x[1]**o+(0.625711925024)*x[1]+(0.840141938096)*x[2]**o+(0.629709976556)*x[2]
        arg[2,2,0,1]=(-0.841259995642)*x[0]**o+(0.220858862445)*x[0]+(0.397334296484)*x[1]**o+(-0.036654179636)*x[1]+(0.235704322138)*x[2]**o+(-0.6299228864)*x[2]
        arg[2,2,1,0]=(-0.561184923307)*x[0]**o+(-0.643644162376)*x[0]+(0.728784994685)*x[1]**o+(-0.697418070844)*x[1]+(-0.56166463788)*x[2]**o+(-0.668760084534)*x[2]
        arg[2,2,1,1]=(0.92243000401)*x[0]**o+(0.96165833182)*x[0]+(0.540379002453)*x[1]**o+(0.630245078647)*x[1]+(-0.375873656541)*x[2]**o+(0.71424898756)*x[2]
        arg[2,2,2,0]=(0.331131296272)*x[0]**o+(-0.136167378345)*x[0]+(-0.375407637025)*x[1]**o+(-0.0890691004072)*x[1]+(-0.787306129385)*x[2]**o+(-0.447474749986)*x[2]
        arg[2,2,2,1]=(-0.47892171854)*x[0]**o+(0.341448171328)*x[0]+(0.662403724393)*x[1]**o+(0.581128263913)*x[1]+(0.00987835902932)*x[2]**o+(-0.884847129103)*x[2]
        arg[2,3,0,0]=(0.0219406022052)*x[0]**o+(-0.370001982912)*x[0]+(0.948162807268)*x[1]**o+(-0.870857624404)*x[1]+(0.580053532448)*x[2]**o+(-0.488657040041)*x[2]
        arg[2,3,0,1]=(-0.114799511617)*x[0]**o+(0.547853919356)*x[0]+(-0.224367707174)*x[1]**o+(0.551491902831)*x[1]+(-0.101845907889)*x[2]**o+(-0.793527438395)*x[2]
        arg[2,3,1,0]=(-0.103518223905)*x[0]**o+(-0.319840011093)*x[0]+(-0.856612861717)*x[1]**o+(0.0208578084364)*x[1]+(0.132248308449)*x[2]**o+(-0.580663554499)*x[2]
        arg[2,3,1,1]=(-0.958142224242)*x[0]**o+(-0.401299403423)*x[0]+(-0.289034019185)*x[1]**o+(-0.0913958572584)*x[1]+(-0.818515945817)*x[2]**o+(-0.918366425737)*x[2]
        arg[2,3,2,0]=(0.730554594734)*x[0]**o+(0.209964841318)*x[0]+(-0.4157212696)*x[1]**o+(-0.673375403124)*x[1]+(-0.409899056244)*x[2]**o+(0.124418003637)*x[2]
        arg[2,3,2,1]=(-0.930974589897)*x[0]**o+(0.256452400333)*x[0]+(0.778606258675)*x[1]**o+(0.23173182074)*x[1]+(-0.846439682798)*x[2]**o+(-0.408860675089)*x[2]
        arg[2,4,0,0]=(-0.246046780287)*x[0]**o+(-0.653766430943)*x[0]+(0.725379461155)*x[1]**o+(-0.305490823497)*x[1]+(0.0837637468427)*x[2]**o+(-0.458770631133)*x[2]
        arg[2,4,0,1]=(-0.612751147889)*x[0]**o+(0.504418327803)*x[0]+(-0.835051783805)*x[1]**o+(-0.86410559166)*x[1]+(0.928919612861)*x[2]**o+(0.303164594248)*x[2]
        arg[2,4,1,0]=(-0.26092972064)*x[0]**o+(-0.990592356395)*x[0]+(-0.48716443965)*x[1]**o+(0.659447608304)*x[1]+(0.429865593656)*x[2]**o+(-0.661470437053)*x[2]
        arg[2,4,1,1]=(0.325840618456)*x[0]**o+(-0.261974904995)*x[0]+(0.476982730267)*x[1]**o+(-0.738119182885)*x[1]+(0.903618858571)*x[2]**o+(-0.287442022624)*x[2]
        arg[2,4,2,0]=(0.655590135983)*x[0]**o+(-0.135505327297)*x[0]+(-0.38079131346)*x[1]**o+(0.663850751769)*x[1]+(-0.658568109867)*x[2]**o+(-0.267875066063)*x[2]
        arg[2,4,2,1]=(0.570712162768)*x[0]**o+(0.390833274898)*x[0]+(-0.97326103687)*x[1]**o+(-0.419300688467)*x[1]+(0.55695479018)*x[2]**o+(0.547271407555)*x[2]
        arg[3,0,0,0]=(-0.486023722572)*x[0]**o+(-0.790544923201)*x[0]+(0.101237181961)*x[1]**o+(0.23267779263)*x[1]+(-0.458580701132)*x[2]**o+(-0.479449733139)*x[2]
        arg[3,0,0,1]=(-0.705291625776)*x[0]**o+(0.144166702405)*x[0]+(-0.210596631253)*x[1]**o+(0.843721258425)*x[1]+(-0.716420627484)*x[2]**o+(0.145427688875)*x[2]
        arg[3,0,1,0]=(-0.838974120992)*x[0]**o+(0.824423674933)*x[0]+(-0.211643366873)*x[1]**o+(0.603255736693)*x[1]+(-0.119484817371)*x[2]**o+(-0.357272543813)*x[2]
        arg[3,0,1,1]=(0.140027541071)*x[0]**o+(0.373598174368)*x[0]+(-0.661831753147)*x[1]**o+(-0.973724961653)*x[1]+(-0.0738899100301)*x[2]**o+(0.0128579666074)*x[2]
        arg[3,0,2,0]=(-0.39083471939)*x[0]**o+(-0.00821070494132)*x[0]+(-0.315936839906)*x[1]**o+(-0.967556350778)*x[1]+(0.136991762476)*x[2]**o+(0.374041520875)*x[2]
        arg[3,0,2,1]=(0.0734743009776)*x[0]**o+(-0.103186084491)*x[0]+(0.426933353985)*x[1]**o+(-0.631917740007)*x[1]+(0.755296770294)*x[2]**o+(0.492171872216)*x[2]
        arg[3,1,0,0]=(-0.414502599055)*x[0]**o+(0.512031410722)*x[0]+(0.610677049483)*x[1]**o+(-0.075275839509)*x[1]+(0.853407100545)*x[2]**o+(-0.569731924126)*x[2]
        arg[3,1,0,1]=(0.886115095396)*x[0]**o+(0.811807969433)*x[0]+(0.550623354147)*x[1]**o+(-0.703728894875)*x[1]+(0.784318903356)*x[2]**o+(0.45891829089)*x[2]
        arg[3,1,1,0]=(0.312577129879)*x[0]**o+(0.0127110992422)*x[0]+(-0.174614340667)*x[1]**o+(0.744535847679)*x[1]+(0.991125452344)*x[2]**o+(-0.405595301834)*x[2]
        arg[3,1,1,1]=(0.784369258288)*x[0]**o+(-0.709503832235)*x[0]+(0.032963558933)*x[1]**o+(-0.471036543959)*x[1]+(-0.138022967476)*x[2]**o+(-0.38452297541)*x[2]
        arg[3,1,2,0]=(-0.337332295082)*x[0]**o+(-0.460457867002)*x[0]+(-0.102806204559)*x[1]**o+(-0.502889108364)*x[1]+(-0.641488011668)*x[2]**o+(0.359501036527)*x[2]
        arg[3,1,2,1]=(0.791575892417)*x[0]**o+(0.126853712619)*x[0]+(-0.408911942637)*x[1]**o+(0.557039825987)*x[1]+(-0.260719710621)*x[2]**o+(-0.648326395935)*x[2]
        arg[3,2,0,0]=(-0.0909007497318)*x[0]**o+(-0.337085519884)*x[0]+(-0.651987922715)*x[1]**o+(-0.130225558042)*x[1]+(0.926567188664)*x[2]**o+(0.764369275478)*x[2]
        arg[3,2,0,1]=(-0.764137934896)*x[0]**o+(0.443892891832)*x[0]+(0.688197384487)*x[1]**o+(0.485210058592)*x[1]+(0.267932741224)*x[2]**o+(0.0366064885582)*x[2]
        arg[3,2,1,0]=(0.49435832562)*x[0]**o+(0.860649583676)*x[0]+(0.667415091305)*x[1]**o+(0.144069385904)*x[1]+(-0.933847080974)*x[2]**o+(-0.378123304101)*x[2]
        arg[3,2,1,1]=(-0.460752644946)*x[0]**o+(0.491079753273)*x[0]+(-0.536055633506)*x[1]**o+(0.786378615881)*x[1]+(-0.190647453518)*x[2]**o+(-0.192868019034)*x[2]
        arg[3,2,2,0]=(0.0946687985697)*x[0]**o+(0.978197346579)*x[0]+(0.972747948856)*x[1]**o+(-0.575670550665)*x[1]+(-0.096229657916)*x[2]**o+(-0.772567430449)*x[2]
        arg[3,2,2,1]=(0.177521713734)*x[0]**o+(0.0683685252148)*x[0]+(0.423995112682)*x[1]**o+(-0.489247981968)*x[1]+(-0.537130434099)*x[2]**o+(-0.0228879131449)*x[2]
        arg[3,3,0,0]=(0.592201472175)*x[0]**o+(0.674304504132)*x[0]+(0.428414872879)*x[1]**o+(0.956414209094)*x[1]+(-0.272853242491)*x[2]**o+(-0.639858910901)*x[2]
        arg[3,3,0,1]=(0.572972211447)*x[0]**o+(0.943096858489)*x[0]+(-0.990948797302)*x[1]**o+(-0.475383526007)*x[1]+(0.542076985363)*x[2]**o+(-0.884809421862)*x[2]
        arg[3,3,1,0]=(0.775163727164)*x[0]**o+(0.101808074039)*x[0]+(0.575490330857)*x[1]**o+(0.199751068512)*x[1]+(0.538742155685)*x[2]**o+(-0.725822956214)*x[2]
        arg[3,3,1,1]=(-0.244586606882)*x[0]**o+(0.983712254807)*x[0]+(0.187370538052)*x[1]**o+(0.332611385073)*x[1]+(0.536377667373)*x[2]**o+(0.255272558489)*x[2]
        arg[3,3,2,0]=(-0.676040282048)*x[0]**o+(-0.172059284021)*x[0]+(0.25100451419)*x[1]**o+(0.303226061816)*x[1]+(0.138976177556)*x[2]**o+(-0.330104849626)*x[2]
        arg[3,3,2,1]=(-0.136437682299)*x[0]**o+(0.938786085213)*x[0]+(0.853018944962)*x[1]**o+(-0.628078903025)*x[1]+(-0.903567849919)*x[2]**o+(0.837621994591)*x[2]
        arg[3,4,0,0]=(0.28632587806)*x[0]**o+(-0.480626263522)*x[0]+(-0.0537965100699)*x[1]**o+(0.423252432505)*x[1]+(0.200777153039)*x[2]**o+(0.629713098071)*x[2]
        arg[3,4,0,1]=(0.326448964148)*x[0]**o+(0.809991245673)*x[0]+(-0.527362539073)*x[1]**o+(-0.0727890632651)*x[1]+(0.210806278508)*x[2]**o+(0.736535063786)*x[2]
        arg[3,4,1,0]=(0.760474271531)*x[0]**o+(0.378083396435)*x[0]+(-0.991296179693)*x[1]**o+(0.431965391269)*x[1]+(-0.12597704605)*x[2]**o+(-0.128017966575)*x[2]
        arg[3,4,1,1]=(-0.21508452109)*x[0]**o+(-0.583147117492)*x[0]+(0.779535489322)*x[1]**o+(0.601144982341)*x[1]+(-0.870362610456)*x[2]**o+(-0.318535391517)*x[2]
        arg[3,4,2,0]=(-0.195181632154)*x[0]**o+(0.237570596161)*x[0]+(-0.794725125363)*x[1]**o+(0.862030507166)*x[1]+(-0.612206533195)*x[2]**o+(0.494078406766)*x[2]
        arg[3,4,2,1]=(0.270278273932)*x[0]**o+(-0.115833233113)*x[0]+(0.754464364285)*x[1]**o+(-0.186069521903)*x[1]+(-0.864426716229)*x[2]**o+(-0.425054438064)*x[2]
        ref[0,0,0,0]=(0.412792261707)/(o+1.)+(0.904364083309)+(0.663998508168)*0.5**o
        ref[0,0,0,1]=(-1.07469333717)/(o+1.)+(-0.0923352454984)+(-0.010130464476)*0.5**o
        ref[0,0,1,0]=(0.287923726124)/(o+1.)+(-0.145507770124)+(-0.176542289385)*0.5**o
        ref[0,0,1,1]=(-0.24656050398)/(o+1.)+(0.341982901349)+(-0.98152572229)*0.5**o
        ref[0,0,2,0]=(1.24918965808)/(o+1.)+(0.165246366859)+(0.555897422173)*0.5**o
        ref[0,0,2,1]=(-1.23538766622)/(o+1.)+(-0.507697956406)+(0.23127040923)*0.5**o
        ref[0,1,0,0]=(-0.267829490776)/(o+1.)+(-0.549968075823)+(0.0202445177842)*0.5**o
        ref[0,1,0,1]=(0.826177960569)/(o+1.)+(0.113794537096)+(0.99935098551)*0.5**o
        ref[0,1,1,0]=(0.124131681048)/(o+1.)+(0.482277820918)+(0.151799117419)*0.5**o
        ref[0,1,1,1]=(0.620076580442)/(o+1.)+(0.401695495934)+(-0.51422532626)*0.5**o
        ref[0,1,2,0]=(1.75895293799)/(o+1.)+(-0.225823657945)+(0.908244888839)*0.5**o
        ref[0,1,2,1]=(-0.334875364565)/(o+1.)+(1.30271444502)+(-0.872307766634)*0.5**o
        ref[0,2,0,0]=(-0.42419420508)/(o+1.)+(-0.3575838585)+(0.00173619592035)*0.5**o
        ref[0,2,0,1]=(-0.57391253095)/(o+1.)+(-0.515088003037)+(0.998184336552)*0.5**o
        ref[0,2,1,0]=(0.114144861758)/(o+1.)+(-0.330251158386)+(-0.921234331968)*0.5**o
        ref[0,2,1,1]=(-1.42031433961)/(o+1.)+(0.581091144051)+(0.153578015941)*0.5**o
        ref[0,2,2,0]=(0.827236715358)/(o+1.)+(0.125325412749)+(0.29601674794)*0.5**o
        ref[0,2,2,1]=(1.34802786361)/(o+1.)+(0.280667679493)+(0.0580007647099)*0.5**o
        ref[0,3,0,0]=(1.28181937762)/(o+1.)+(-0.214505128496)+(-0.337999890505)*0.5**o
        ref[0,3,0,1]=(1.65569377802)/(o+1.)+(-0.363265639443)+(-0.825813329629)*0.5**o
        ref[0,3,1,0]=(0.560544990797)/(o+1.)+(-0.276979054204)+(0.699313512079)*0.5**o
        ref[0,3,1,1]=(-0.879495822379)/(o+1.)+(0.560842505268)+(0.140764124286)*0.5**o
        ref[0,3,2,0]=(-1.04751560841)/(o+1.)+(-0.238703591863)+(-0.68177752688)*0.5**o
        ref[0,3,2,1]=(-0.414022097401)/(o+1.)+(0.509666283203)+(0.655685968811)*0.5**o
        ref[0,4,0,0]=(1.0625549765)/(o+1.)+(-1.26969185098)+(-0.18027010068)*0.5**o
        ref[0,4,0,1]=(-0.430389257621)/(o+1.)+(-0.728926201043)+(-0.914768182433)*0.5**o
        ref[0,4,1,0]=(-0.585220311209)/(o+1.)+(0.344278586108)+(0.274727423835)*0.5**o
        ref[0,4,1,1]=(-0.596140928652)/(o+1.)+(-0.857974578823)+(0.257574026756)*0.5**o
        ref[0,4,2,0]=(-1.15566595381)/(o+1.)+(0.148922032671)+(0.703185665611)*0.5**o
        ref[0,4,2,1]=(0.255095282229)/(o+1.)+(-0.480012460544)+(-0.153792811673)*0.5**o
        ref[1,0,0,0]=(0.524122113192)/(o+1.)+(-0.434579759316)+(0.298044833507)*0.5**o
        ref[1,0,0,1]=(-0.220122924131)/(o+1.)+(0.466218310719)+(-0.575485929806)*0.5**o
        ref[1,0,1,0]=(0.290760959656)/(o+1.)+(-0.0552768720487)+(0.998358005707)*0.5**o
        ref[1,0,1,1]=(-1.69984876204)/(o+1.)+(0.325064655551)+(0.783593367143)*0.5**o
        ref[1,0,2,0]=(-0.999302906363)/(o+1.)+(0.639576565202)+(-0.42345692756)*0.5**o
        ref[1,0,2,1]=(-0.271313965844)/(o+1.)+(-0.0328780841139)+(-0.790917746681)*0.5**o
        ref[1,1,0,0]=(-1.17401606398)/(o+1.)+(0.654288857484)+(0.944343622558)*0.5**o
        ref[1,1,0,1]=(0.623620452946)/(o+1.)+(-0.0991803677964)+(-0.246711023019)*0.5**o
        ref[1,1,1,0]=(-0.218171642866)/(o+1.)+(-0.404913755908)+(-0.530782063061)*0.5**o
        ref[1,1,1,1]=(1.23525595763)/(o+1.)+(0.349524193353)+(-0.710844761085)*0.5**o
        ref[1,1,2,0]=(-0.42993438925)/(o+1.)+(0.399995211041)+(0.574657505707)*0.5**o
        ref[1,1,2,1]=(-0.217230696736)/(o+1.)+(0.635787738562)+(-0.365373199615)*0.5**o
        ref[1,2,0,0]=(-0.102563204807)/(o+1.)+(-0.188445289686)+(0.544776615353)*0.5**o
        ref[1,2,0,1]=(0.298732134665)/(o+1.)+(-0.416846709104)+(0.11684721915)*0.5**o
        ref[1,2,1,0]=(-0.234419306371)/(o+1.)+(0.383855155251)+(0.561736948232)*0.5**o
        ref[1,2,1,1]=(0.539973971687)/(o+1.)+(0.0241566615339)+(0.578630452142)*0.5**o
        ref[1,2,2,0]=(0.346023814217)/(o+1.)+(-0.515461022254)+(-0.0269203575755)*0.5**o
        ref[1,2,2,1]=(-1.12923994073)/(o+1.)+(0.687820128862)+(-0.534642633404)*0.5**o
        ref[1,3,0,0]=(-1.35206603301)/(o+1.)+(-0.195350398606)+(0.892302802156)*0.5**o
        ref[1,3,0,1]=(0.457546541694)/(o+1.)+(0.584952905039)+(0.883702447985)*0.5**o
        ref[1,3,1,0]=(-0.229382345029)/(o+1.)+(-0.51570070583)+(0.492527250981)*0.5**o
        ref[1,3,1,1]=(0.163467163568)/(o+1.)+(-0.511309270369)+(0.372499842901)*0.5**o
        ref[1,3,2,0]=(-1.10954940366)/(o+1.)+(0.613097649163)+(0.533098362924)*0.5**o
        ref[1,3,2,1]=(0.0423370323263)/(o+1.)+(-0.655974226943)+(-0.909038123273)*0.5**o
        ref[1,4,0,0]=(-0.149369878527)/(o+1.)+(0.594641654424)+(-0.974474210185)*0.5**o
        ref[1,4,0,1]=(0.995137009073)/(o+1.)+(-0.0464708598635)+(0.964745405431)*0.5**o
        ref[1,4,1,0]=(0.545158516903)/(o+1.)+(-0.562624149397)+(-0.714725894011)*0.5**o
        ref[1,4,1,1]=(-1.20779451465)/(o+1.)+(-0.189148851615)+(-0.809072545597)*0.5**o
        ref[1,4,2,0]=(-1.55971061601)/(o+1.)+(0.693489320143)+(-0.21281876112)*0.5**o
        ref[1,4,2,1]=(0.226662056752)/(o+1.)+(-0.454678810051)+(-0.247930702516)*0.5**o
        ref[2,0,0,0]=(-0.242758325417)/(o+1.)+(-0.527650883321)+(0.616352869937)*0.5**o
        ref[2,0,0,1]=(0.807761382166)/(o+1.)+(1.02431485248)+(-0.85704374519)*0.5**o
        ref[2,0,1,0]=(0.543068844645)/(o+1.)+(0.0690269849966)+(0.326870102902)*0.5**o
        ref[2,0,1,1]=(0.502168875857)/(o+1.)+(-0.0584590052478)+(-0.699939842176)*0.5**o
        ref[2,0,2,0]=(0.181622609504)/(o+1.)+(-0.57477595904)+(-0.985996827188)*0.5**o
        ref[2,0,2,1]=(-1.01828660056)/(o+1.)+(0.255680773601)+(-0.195357773436)*0.5**o
        ref[2,1,0,0]=(-0.961895417879)/(o+1.)+(-0.848679114358)+(0.601652939329)*0.5**o
        ref[2,1,0,1]=(0.177781123824)/(o+1.)+(-0.945003972397)+(0.534038022604)*0.5**o
        ref[2,1,1,0]=(-0.20513407563)/(o+1.)+(-0.126335927989)+(-0.361196978257)*0.5**o
        ref[2,1,1,1]=(0.712128730879)/(o+1.)+(-0.0623370236322)+(0.00930987921229)*0.5**o
        ref[2,1,2,0]=(0.692430706812)/(o+1.)+(0.417664138602)+(-0.181479246679)*0.5**o
        ref[2,1,2,1]=(0.650980026197)/(o+1.)+(-0.708193455952)+(-0.99665050175)*0.5**o
        ref[2,2,0,0]=(1.22766505061)/(o+1.)+(0.717584511586)+(0.870069083779)*0.5**o
        ref[2,2,0,1]=(0.633038618622)/(o+1.)+(-0.222859101796)+(-0.841259995642)*0.5**o
        ref[2,2,1,0]=(0.167120356805)/(o+1.)+(-1.00491115888)+(-0.561184923307)*0.5**o
        ref[2,2,1,1]=(0.164505345911)/(o+1.)+(1.15307619901)+(0.92243000401)*0.5**o
        ref[2,2,2,0]=(-1.16271376641)/(o+1.)+(-0.336355614369)+(0.331131296272)*0.5**o
        ref[2,2,2,1]=(0.672282083422)/(o+1.)+(0.018864653069)+(-0.47892171854)*0.5**o
        ref[2,3,0,0]=(1.52821633972)/(o+1.)+(-0.864758323678)+(0.0219406022052)*0.5**o
        ref[2,3,0,1]=(-0.326213615062)/(o+1.)+(0.152909191896)+(-0.114799511617)*0.5**o
        ref[2,3,1,0]=(-0.724364553267)/(o+1.)+(-0.439822878578)+(-0.103518223905)*0.5**o
        ref[2,3,1,1]=(-1.107549965)/(o+1.)+(-0.705530843209)+(-0.958142224242)*0.5**o
        ref[2,3,2,0]=(-0.825620325844)/(o+1.)+(-0.169496279085)+(0.730554594734)*0.5**o
        ref[2,3,2,1]=(-0.0678334241233)/(o+1.)+(0.0396617729923)+(-0.930974589897)*0.5**o
        ref[2,4,0,0]=(0.809143207998)/(o+1.)+(-0.709013942786)+(-0.246046780287)*0.5**o
        ref[2,4,0,1]=(0.0938678290563)/(o+1.)+(-0.0282613348041)+(-0.612751147889)*0.5**o
        ref[2,4,1,0]=(-0.0572988459938)/(o+1.)+(-0.496307592572)+(-0.26092972064)*0.5**o
        ref[2,4,1,1]=(1.38060158884)/(o+1.)+(-0.643768055252)+(0.325840618456)*0.5**o
        ref[2,4,2,0]=(-1.03935942333)/(o+1.)+(0.130235179204)+(0.655590135983)*0.5**o
        ref[2,4,2,1]=(-0.41630624669)/(o+1.)+(0.259401996992)+(0.570712162768)*0.5**o
        ref[3,0,0,0]=(-0.357343519171)/(o+1.)+(-0.518658431855)+(-0.486023722572)*0.5**o
        ref[3,0,0,1]=(-0.927017258737)/(o+1.)+(0.566657824853)+(-0.705291625776)*0.5**o
        ref[3,0,1,0]=(-0.331128184244)/(o+1.)+(0.535203433907)+(-0.838974120992)*0.5**o
        ref[3,0,1,1]=(-0.735721663177)/(o+1.)+(-0.293634410339)+(0.140027541071)*0.5**o
        ref[3,0,2,0]=(-0.178945077431)/(o+1.)+(-0.300862767422)+(-0.39083471939)*0.5**o
        ref[3,0,2,1]=(1.18223012428)/(o+1.)+(-0.121465976141)+(0.0734743009776)*0.5**o
        ref[3,1,0,0]=(1.46408415003)/(o+1.)+(-0.0664881764563)+(-0.414502599055)*0.5**o
        ref[3,1,0,1]=(1.3349422575)/(o+1.)+(0.283498682724)+(0.886115095396)*0.5**o
        ref[3,1,1,0]=(0.816511111677)/(o+1.)+(0.175825822544)+(0.312577129879)*0.5**o
        ref[3,1,1,1]=(-0.105059408543)/(o+1.)+(-0.782531675802)+(0.784369258288)*0.5**o
        ref[3,1,2,0]=(-0.744294216228)/(o+1.)+(-0.301922969419)+(-0.337332295082)*0.5**o
        ref[3,1,2,1]=(-0.669631653258)/(o+1.)+(0.0177835713351)+(0.791575892417)*0.5**o
        ref[3,2,0,0]=(0.274579265949)/(o+1.)+(0.148529098776)+(-0.0909007497318)*0.5**o
        ref[3,2,0,1]=(0.956130125711)/(o+1.)+(0.482854719491)+(-0.764137934896)*0.5**o
        ref[3,2,1,0]=(-0.266431989669)/(o+1.)+(0.313297832739)+(0.49435832562)*0.5**o
        ref[3,2,1,1]=(-0.726703087024)/(o+1.)+(0.542295175061)+(-0.460752644946)*0.5**o
        ref[3,2,2,0]=(0.87651829094)/(o+1.)+(-0.185020317268)+(0.0946687985697)*0.5**o
        ref[3,2,2,1]=(-0.113135321417)/(o+1.)+(-0.221883684949)+(0.177521713734)*0.5**o
        ref[3,3,0,0]=(0.155561630388)/(o+1.)+(0.495429901163)+(0.592201472175)*0.5**o
        ref[3,3,0,1]=(-0.44887181194)/(o+1.)+(-0.20854804469)+(0.572972211447)*0.5**o
        ref[3,3,1,0]=(1.11423248654)/(o+1.)+(-0.212131906831)+(0.775163727164)*0.5**o
        ref[3,3,1,1]=(0.723748205425)/(o+1.)+(0.785798099184)+(-0.244586606882)*0.5**o
        ref[3,3,2,0]=(0.389980691746)/(o+1.)+(-0.0994690359155)+(-0.676040282048)*0.5**o
        ref[3,3,2,1]=(-0.0505489049572)/(o+1.)+(0.57416458839)+(-0.136437682299)*0.5**o
        ref[3,4,0,0]=(0.14698064297)/(o+1.)+(0.286169633527)+(0.28632587806)*0.5**o
        ref[3,4,0,1]=(-0.316556260565)/(o+1.)+(0.736868623097)+(0.326448964148)*0.5**o
        ref[3,4,1,0]=(-1.11727322574)/(o+1.)+(0.341015410564)+(0.760474271531)*0.5**o
        ref[3,4,1,1]=(-0.0908271211344)/(o+1.)+(-0.150268763334)+(-0.21508452109)*0.5**o
        ref[3,4,2,0]=(-1.40693165856)/(o+1.)+(0.796839755046)+(-0.195181632154)*0.5**o
        ref[3,4,2,1]=(-0.109962351944)/(o+1.)+(-0.36347859654)+(0.270278273932)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_Solution_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.0433006726989)*x[0]**o+(0.819109331838)*x[0]+(0.150292245587)*x[1]**o+(0.949397165008)*x[1]
        ref=(0.150292245587)/(o+1.)+(0.884253248423)+(0.0433006726989)*0.5**o
      else:
        arg=(-0.765858486678)*x[0]**o+(-0.829061249978)*x[0]+(-0.0255273602735)*x[1]**o+(-0.93097352763)*x[1]+(-0.775974817954)*x[2]**o+(0.0414580763278)*x[2]
        ref=(-0.801502178228)/(o+1.)+(-0.85928835064)+(-0.765858486678)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_Solution_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.708418284738)*x[0]**o+(-0.265416249865)*x[0]+(0.0412873279676)*x[1]**o+(0.18827690539)*x[1]
        arg[1]=(-0.0639578348425)*x[0]**o+(-0.435408788697)*x[0]+(0.429134354603)*x[1]**o+(0.0391558318157)*x[1]
        ref[0]=(0.0412873279676)/(o+1.)+(-0.0385696722374)+(0.708418284738)*0.5**o
        ref[1]=(0.429134354603)/(o+1.)+(-0.198126478441)+(-0.0639578348425)*0.5**o
      else:
        arg[0]=(-0.516066856372)*x[0]**o+(-0.509677909948)*x[0]+(0.189594948337)*x[1]**o+(0.743432553688)*x[1]+(0.687864109046)*x[2]**o+(0.257280277379)*x[2]
        arg[1]=(0.948762785699)*x[0]**o+(0.372861763607)*x[0]+(-0.166460043702)*x[1]**o+(0.0474018901531)*x[1]+(-0.485380626595)*x[2]**o+(-0.341701937267)*x[2]
        ref[0]=(0.877459057383)/(o+1.)+(0.245517460559)+(-0.516066856372)*0.5**o
        ref[1]=(-0.651840670298)/(o+1.)+(0.0392808582465)+(0.948762785699)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_Solution_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.0540280486222)*x[0]**o+(-0.936629447756)*x[0]+(0.530867101627)*x[1]**o+(-0.404522355122)*x[1]
        arg[0,1]=(0.585051034599)*x[0]**o+(-0.899683342239)*x[0]+(0.61865830472)*x[1]**o+(0.491768251743)*x[1]
        arg[0,2]=(0.924131379905)*x[0]**o+(-0.394367734246)*x[0]+(0.432122525065)*x[1]**o+(0.79923634141)*x[1]
        arg[0,3]=(-0.41435196904)*x[0]**o+(-0.22039395628)*x[0]+(-0.866551833459)*x[1]**o+(-0.0966534238435)*x[1]
        arg[0,4]=(0.855785701017)*x[0]**o+(0.0879754354535)*x[0]+(-0.0727782016239)*x[1]**o+(0.746878225435)*x[1]
        arg[1,0]=(-0.868450306659)*x[0]**o+(-0.00845792225901)*x[0]+(0.227399876353)*x[1]**o+(-0.522196483799)*x[1]
        arg[1,1]=(0.53076931297)*x[0]**o+(0.165744140226)*x[0]+(0.14586398181)*x[1]**o+(-0.694169551043)*x[1]
        arg[1,2]=(-0.936146144862)*x[0]**o+(-0.198367580291)*x[0]+(0.848822682163)*x[1]**o+(0.360521711263)*x[1]
        arg[1,3]=(-0.395889451325)*x[0]**o+(0.510713345487)*x[0]+(-0.62440033386)*x[1]**o+(0.29695714683)*x[1]
        arg[1,4]=(0.847685553395)*x[0]**o+(-0.291266469606)*x[0]+(-0.517114839541)*x[1]**o+(-0.279948585383)*x[1]
        arg[2,0]=(-0.622481093435)*x[0]**o+(0.0911922519622)*x[0]+(0.216648336495)*x[1]**o+(0.663562516802)*x[1]
        arg[2,1]=(0.567833843918)*x[0]**o+(0.18116887359)*x[0]+(-0.0433012798352)*x[1]**o+(0.998332142285)*x[1]
        arg[2,2]=(-0.81729786652)*x[0]**o+(-0.634680104287)*x[0]+(0.991513806139)*x[1]**o+(-0.0721700196003)*x[1]
        arg[2,3]=(0.121181286827)*x[0]**o+(-0.677803464822)*x[0]+(0.870139639102)*x[1]**o+(-0.134662002755)*x[1]
        arg[2,4]=(-0.167376807346)*x[0]**o+(-0.500556007898)*x[0]+(-0.839256017197)*x[1]**o+(-0.82147890878)*x[1]
        arg[3,0]=(0.576521064132)*x[0]**o+(-0.510276372588)*x[0]+(0.92888227487)*x[1]**o+(-0.88293512036)*x[1]
        arg[3,1]=(-0.159048471862)*x[0]**o+(-0.0479500210678)*x[0]+(-0.0216438779517)*x[1]**o+(-0.108252189548)*x[1]
        arg[3,2]=(-0.483779576334)*x[0]**o+(-0.871829147209)*x[0]+(-0.22556218283)*x[1]**o+(0.451007116911)*x[1]
        arg[3,3]=(-0.358888369902)*x[0]**o+(0.436554685191)*x[0]+(-0.0359187908139)*x[1]**o+(-0.0148930474244)*x[1]
        arg[3,4]=(-0.995411478882)*x[0]**o+(-0.457206328785)*x[0]+(-0.833823947273)*x[1]**o+(-0.0517068976093)*x[1]
        ref[0,0]=(0.530867101627)/(o+1.)+(-0.670575901439)+(-0.0540280486222)*0.5**o
        ref[0,1]=(0.61865830472)/(o+1.)+(-0.203957545248)+(0.585051034599)*0.5**o
        ref[0,2]=(0.432122525065)/(o+1.)+(0.202434303582)+(0.924131379905)*0.5**o
        ref[0,3]=(-0.866551833459)/(o+1.)+(-0.158523690062)+(-0.41435196904)*0.5**o
        ref[0,4]=(-0.0727782016239)/(o+1.)+(0.417426830444)+(0.855785701017)*0.5**o
        ref[1,0]=(0.227399876353)/(o+1.)+(-0.265327203029)+(-0.868450306659)*0.5**o
        ref[1,1]=(0.14586398181)/(o+1.)+(-0.264212705408)+(0.53076931297)*0.5**o
        ref[1,2]=(0.848822682163)/(o+1.)+(0.0810770654864)+(-0.936146144862)*0.5**o
        ref[1,3]=(-0.62440033386)/(o+1.)+(0.403835246159)+(-0.395889451325)*0.5**o
        ref[1,4]=(-0.517114839541)/(o+1.)+(-0.285607527495)+(0.847685553395)*0.5**o
        ref[2,0]=(0.216648336495)/(o+1.)+(0.377377384382)+(-0.622481093435)*0.5**o
        ref[2,1]=(-0.0433012798352)/(o+1.)+(0.589750507938)+(0.567833843918)*0.5**o
        ref[2,2]=(0.991513806139)/(o+1.)+(-0.353425061944)+(-0.81729786652)*0.5**o
        ref[2,3]=(0.870139639102)/(o+1.)+(-0.406232733789)+(0.121181286827)*0.5**o
        ref[2,4]=(-0.839256017197)/(o+1.)+(-0.661017458339)+(-0.167376807346)*0.5**o
        ref[3,0]=(0.92888227487)/(o+1.)+(-0.696605746474)+(0.576521064132)*0.5**o
        ref[3,1]=(-0.0216438779517)/(o+1.)+(-0.078101105308)+(-0.159048471862)*0.5**o
        ref[3,2]=(-0.22556218283)/(o+1.)+(-0.210411015149)+(-0.483779576334)*0.5**o
        ref[3,3]=(-0.0359187908139)/(o+1.)+(0.210830818883)+(-0.358888369902)*0.5**o
        ref[3,4]=(-0.833823947273)/(o+1.)+(-0.254456613197)+(-0.995411478882)*0.5**o
      else:
        arg[0,0]=(-0.937858877179)*x[0]**o+(-0.41590845461)*x[0]+(0.317385444442)*x[1]**o+(-0.166284606371)*x[1]+(-0.846106994172)*x[2]**o+(0.227876218054)*x[2]
        arg[0,1]=(0.497785867084)*x[0]**o+(0.883557362705)*x[0]+(-0.846602726567)*x[1]**o+(-0.859091910548)*x[1]+(-0.295742503292)*x[2]**o+(-0.853931233266)*x[2]
        arg[0,2]=(0.744501879094)*x[0]**o+(-0.0181115125382)*x[0]+(-0.856099357899)*x[1]**o+(0.767449074734)*x[1]+(-0.858552187616)*x[2]**o+(-0.580142057156)*x[2]
        arg[0,3]=(0.210031467359)*x[0]**o+(-0.994426053833)*x[0]+(0.648522478175)*x[1]**o+(0.0355805603417)*x[1]+(-0.274347447112)*x[2]**o+(-0.280165537012)*x[2]
        arg[0,4]=(-0.323650660168)*x[0]**o+(-0.279592941595)*x[0]+(-0.478811425713)*x[1]**o+(0.19753271141)*x[1]+(0.476202875551)*x[2]**o+(0.657358100659)*x[2]
        arg[1,0]=(0.890659868437)*x[0]**o+(0.100235405989)*x[0]+(0.325177871418)*x[1]**o+(0.675379114656)*x[1]+(0.458473828724)*x[2]**o+(-0.449822467434)*x[2]
        arg[1,1]=(-0.467895039242)*x[0]**o+(-0.551300000422)*x[0]+(0.976111922801)*x[1]**o+(0.809389941274)*x[1]+(-0.693066235685)*x[2]**o+(0.595719115505)*x[2]
        arg[1,2]=(-0.727304617172)*x[0]**o+(-0.270682330445)*x[0]+(-0.32926621635)*x[1]**o+(0.0389068111165)*x[1]+(0.300096425532)*x[2]**o+(-0.964910767632)*x[2]
        arg[1,3]=(0.195012106644)*x[0]**o+(0.18137839607)*x[0]+(-0.823129279257)*x[1]**o+(-0.0440106889304)*x[1]+(-0.298898659898)*x[2]**o+(0.801623925974)*x[2]
        arg[1,4]=(0.590200326369)*x[0]**o+(-0.800085209049)*x[0]+(-0.323836133834)*x[1]**o+(-0.953413527643)*x[1]+(0.0257128844735)*x[2]**o+(0.652469516082)*x[2]
        arg[2,0]=(-0.467537281458)*x[0]**o+(-0.05673369912)*x[0]+(-0.813542619659)*x[1]**o+(-0.778596374878)*x[1]+(0.752073310476)*x[2]**o+(0.187824163254)*x[2]
        arg[2,1]=(-0.75708112983)*x[0]**o+(0.866111788737)*x[0]+(0.715424327432)*x[1]**o+(-0.297967739435)*x[1]+(0.420241611178)*x[2]**o+(-0.616867390634)*x[2]
        arg[2,2]=(0.00628628803839)*x[0]**o+(0.700962301988)*x[0]+(0.712979263919)*x[1]**o+(-0.414401132518)*x[1]+(-0.975498928756)*x[2]**o+(0.858116391039)*x[2]
        arg[2,3]=(0.948164399923)*x[0]**o+(-0.88227825142)*x[0]+(0.30703295864)*x[1]**o+(0.510607674984)*x[1]+(0.362361719392)*x[2]**o+(-0.208696380629)*x[2]
        arg[2,4]=(0.661545494115)*x[0]**o+(-0.122960355184)*x[0]+(-0.801367604574)*x[1]**o+(0.395708459987)*x[1]+(0.0197225340457)*x[2]**o+(-0.377005073637)*x[2]
        arg[3,0]=(0.733597350529)*x[0]**o+(0.339318978843)*x[0]+(0.317608635437)*x[1]**o+(0.147746051469)*x[1]+(-0.33413935488)*x[2]**o+(-0.793337674933)*x[2]
        arg[3,1]=(0.0918904281646)*x[0]**o+(-0.530786152204)*x[0]+(-0.587542750401)*x[1]**o+(-0.810331029621)*x[1]+(-0.40221065987)*x[2]**o+(-0.250472092093)*x[2]
        arg[3,2]=(0.966442952042)*x[0]**o+(-0.756649866552)*x[0]+(-0.36466223222)*x[1]**o+(-0.48639005055)*x[1]+(-0.831374462742)*x[2]**o+(0.503451143644)*x[2]
        arg[3,3]=(-0.322275644664)*x[0]**o+(-0.254067149982)*x[0]+(-0.440264794128)*x[1]**o+(0.226546045529)*x[1]+(-0.761870916789)*x[2]**o+(0.70024066379)*x[2]
        arg[3,4]=(0.595459342613)*x[0]**o+(-0.439436200258)*x[0]+(0.0932538180176)*x[1]**o+(-0.975450806707)*x[1]+(0.0461423323623)*x[2]**o+(0.0294352472154)*x[2]
        ref[0,0]=(-0.52872154973)/(o+1.)+(-0.177158421464)+(-0.937858877179)*0.5**o
        ref[0,1]=(-1.14234522986)/(o+1.)+(-0.414732890554)+(0.497785867084)*0.5**o
        ref[0,2]=(-1.71465154551)/(o+1.)+(0.0845977525197)+(0.744501879094)*0.5**o
        ref[0,3]=(0.374175031062)/(o+1.)+(-0.619505515252)+(0.210031467359)*0.5**o
        ref[0,4]=(-0.00260855016132)/(o+1.)+(0.287648935237)+(-0.323650660168)*0.5**o
        ref[1,0]=(0.783651700142)/(o+1.)+(0.162896026605)+(0.890659868437)*0.5**o
        ref[1,1]=(0.283045687117)/(o+1.)+(0.426904528178)+(-0.467895039242)*0.5**o
        ref[1,2]=(-0.0291697908188)/(o+1.)+(-0.59834314348)+(-0.727304617172)*0.5**o
        ref[1,3]=(-1.12202793916)/(o+1.)+(0.469495816557)+(0.195012106644)*0.5**o
        ref[1,4]=(-0.29812324936)/(o+1.)+(-0.550514610305)+(0.590200326369)*0.5**o
        ref[2,0]=(-0.0614693091826)/(o+1.)+(-0.323752955372)+(-0.467537281458)*0.5**o
        ref[2,1]=(1.13566593861)/(o+1.)+(-0.024361670666)+(-0.75708112983)*0.5**o
        ref[2,2]=(-0.262519664837)/(o+1.)+(0.572338780255)+(0.00628628803839)*0.5**o
        ref[2,3]=(0.669394678032)/(o+1.)+(-0.290183478532)+(0.948164399923)*0.5**o
        ref[2,4]=(-0.781645070528)/(o+1.)+(-0.0521284844165)+(0.661545494115)*0.5**o
        ref[3,0]=(-0.0165307194439)/(o+1.)+(-0.15313632231)+(0.733597350529)*0.5**o
        ref[3,1]=(-0.989753410271)/(o+1.)+(-0.795794636959)+(0.0918904281646)*0.5**o
        ref[3,2]=(-1.19603669496)/(o+1.)+(-0.369794386729)+(0.966442952042)*0.5**o
        ref[3,3]=(-1.20213571092)/(o+1.)+(0.336359779668)+(-0.322275644664)*0.5**o
        ref[3,4]=(0.13939615038)/(o+1.)+(-0.692725879875)+(0.595459342613)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_Solution_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.12237588236)*x[0]**o+(-0.671341785528)*x[0]+(0.998910161099)*x[1]**o+(-0.552290246966)*x[1]
        arg[0,0,1]=(-0.90821257221)*x[0]**o+(-0.413391927764)*x[0]+(-0.428904099491)*x[1]**o+(-0.226766222189)*x[1]
        arg[0,1,0]=(-0.333819313127)*x[0]**o+(-0.458569899029)*x[0]+(-0.83776177332)*x[1]**o+(0.22951184631)*x[1]
        arg[0,1,1]=(-0.0571879315489)*x[0]**o+(-0.38056227895)*x[0]+(0.0784628125977)*x[1]**o+(-0.659353613638)*x[1]
        arg[1,0,0]=(0.645871468353)*x[0]**o+(-0.986005542006)*x[0]+(0.395704509276)*x[1]**o+(0.980666918726)*x[1]
        arg[1,0,1]=(0.337293036597)*x[0]**o+(-0.520809490378)*x[0]+(0.543657498632)*x[1]**o+(0.632777244118)*x[1]
        arg[1,1,0]=(-0.432349538121)*x[0]**o+(-0.579685247099)*x[0]+(0.331943072726)*x[1]**o+(0.25042188644)*x[1]
        arg[1,1,1]=(0.329861800715)*x[0]**o+(-0.48725090895)*x[0]+(-0.149357998414)*x[1]**o+(-0.580650134153)*x[1]
        arg[2,0,0]=(-0.75424658357)*x[0]**o+(-0.696352517414)*x[0]+(-0.00272313166917)*x[1]**o+(-0.488581343546)*x[1]
        arg[2,0,1]=(-0.136489901339)*x[0]**o+(-0.752587775699)*x[0]+(0.986413392791)*x[1]**o+(0.759344728162)*x[1]
        arg[2,1,0]=(-0.0832674877963)*x[0]**o+(0.765419402778)*x[0]+(0.958533611564)*x[1]**o+(-0.315461136294)*x[1]
        arg[2,1,1]=(0.362764663642)*x[0]**o+(-0.351519282456)*x[0]+(-0.115554433923)*x[1]**o+(-0.789365036398)*x[1]
        arg[3,0,0]=(0.0977752933224)*x[0]**o+(-0.838881001576)*x[0]+(0.275578737811)*x[1]**o+(-0.971208155806)*x[1]
        arg[3,0,1]=(0.0588243034654)*x[0]**o+(-0.141302234145)*x[0]+(-0.460044396206)*x[1]**o+(-0.776793718354)*x[1]
        arg[3,1,0]=(0.786922930047)*x[0]**o+(-0.187797924361)*x[0]+(0.971516347812)*x[1]**o+(-0.250547377602)*x[1]
        arg[3,1,1]=(-0.69240453335)*x[0]**o+(0.439425578473)*x[0]+(0.00864940781276)*x[1]**o+(-0.458898387231)*x[1]
        arg[4,0,0]=(0.0700104829681)*x[0]**o+(0.905031121846)*x[0]+(0.37676750477)*x[1]**o+(0.411190149331)*x[1]
        arg[4,0,1]=(-0.615737714754)*x[0]**o+(0.1945204765)*x[0]+(0.0889014902938)*x[1]**o+(0.650020022597)*x[1]
        arg[4,1,0]=(-0.883448519911)*x[0]**o+(0.00124188371434)*x[0]+(-0.501662684156)*x[1]**o+(0.290235036398)*x[1]
        arg[4,1,1]=(-0.723885968554)*x[0]**o+(-0.0112308623755)*x[0]+(0.679499427805)*x[1]**o+(-0.00261690404287)*x[1]
        arg[5,0,0]=(0.0834731365594)*x[0]**o+(-0.873857991293)*x[0]+(0.831158842487)*x[1]**o+(-0.585808323324)*x[1]
        arg[5,0,1]=(0.250830078938)*x[0]**o+(0.310347394372)*x[0]+(-0.412138455827)*x[1]**o+(0.974823718029)*x[1]
        arg[5,1,0]=(-0.865155065381)*x[0]**o+(0.72190541012)*x[0]+(0.72020941875)*x[1]**o+(-0.604997858133)*x[1]
        arg[5,1,1]=(0.57252234131)*x[0]**o+(-0.366230226771)*x[0]+(0.115400105646)*x[1]**o+(0.0485229323141)*x[1]
        ref[0,0,0]=(0.998910161099)/(o+1.)+(-0.611816016247)+(-0.12237588236)*0.5**o
        ref[0,0,1]=(-0.428904099491)/(o+1.)+(-0.320079074977)+(-0.90821257221)*0.5**o
        ref[0,1,0]=(-0.83776177332)/(o+1.)+(-0.11452902636)+(-0.333819313127)*0.5**o
        ref[0,1,1]=(0.0784628125977)/(o+1.)+(-0.519957946294)+(-0.0571879315489)*0.5**o
        ref[1,0,0]=(0.395704509276)/(o+1.)+(-0.00266931163989)+(0.645871468353)*0.5**o
        ref[1,0,1]=(0.543657498632)/(o+1.)+(0.0559838768697)+(0.337293036597)*0.5**o
        ref[1,1,0]=(0.331943072726)/(o+1.)+(-0.16463168033)+(-0.432349538121)*0.5**o
        ref[1,1,1]=(-0.149357998414)/(o+1.)+(-0.533950521552)+(0.329861800715)*0.5**o
        ref[2,0,0]=(-0.00272313166917)/(o+1.)+(-0.59246693048)+(-0.75424658357)*0.5**o
        ref[2,0,1]=(0.986413392791)/(o+1.)+(0.00337847623164)+(-0.136489901339)*0.5**o
        ref[2,1,0]=(0.958533611564)/(o+1.)+(0.224979133242)+(-0.0832674877963)*0.5**o
        ref[2,1,1]=(-0.115554433923)/(o+1.)+(-0.570442159427)+(0.362764663642)*0.5**o
        ref[3,0,0]=(0.275578737811)/(o+1.)+(-0.905044578691)+(0.0977752933224)*0.5**o
        ref[3,0,1]=(-0.460044396206)/(o+1.)+(-0.45904797625)+(0.0588243034654)*0.5**o
        ref[3,1,0]=(0.971516347812)/(o+1.)+(-0.219172650981)+(0.786922930047)*0.5**o
        ref[3,1,1]=(0.00864940781276)/(o+1.)+(-0.00973640437899)+(-0.69240453335)*0.5**o
        ref[4,0,0]=(0.37676750477)/(o+1.)+(0.658110635589)+(0.0700104829681)*0.5**o
        ref[4,0,1]=(0.0889014902938)/(o+1.)+(0.422270249549)+(-0.615737714754)*0.5**o
        ref[4,1,0]=(-0.501662684156)/(o+1.)+(0.145738460056)+(-0.883448519911)*0.5**o
        ref[4,1,1]=(0.679499427805)/(o+1.)+(-0.00692388320921)+(-0.723885968554)*0.5**o
        ref[5,0,0]=(0.831158842487)/(o+1.)+(-0.729833157309)+(0.0834731365594)*0.5**o
        ref[5,0,1]=(-0.412138455827)/(o+1.)+(0.642585556201)+(0.250830078938)*0.5**o
        ref[5,1,0]=(0.72020941875)/(o+1.)+(0.0584537759935)+(-0.865155065381)*0.5**o
        ref[5,1,1]=(0.115400105646)/(o+1.)+(-0.158853647228)+(0.57252234131)*0.5**o
      else:
        arg[0,0,0]=(0.962792187023)*x[0]**o+(0.799861778652)*x[0]+(-0.811367630134)*x[1]**o+(-0.553496861095)*x[1]+(-0.810089307714)*x[2]**o+(0.135449765364)*x[2]
        arg[0,0,1]=(0.644573004654)*x[0]**o+(-0.565823623798)*x[0]+(0.337026645127)*x[1]**o+(0.413612857799)*x[1]+(0.604366347987)*x[2]**o+(0.0867373815054)*x[2]
        arg[0,1,0]=(0.829914553754)*x[0]**o+(-0.462419132852)*x[0]+(0.889464473044)*x[1]**o+(-0.425594549042)*x[1]+(0.679251542389)*x[2]**o+(-0.501090535615)*x[2]
        arg[0,1,1]=(0.874166303558)*x[0]**o+(0.437595762221)*x[0]+(-0.836229998654)*x[1]**o+(-0.704020633222)*x[1]+(-0.279878325513)*x[2]**o+(0.697289463565)*x[2]
        arg[1,0,0]=(-0.963425479949)*x[0]**o+(0.399090631945)*x[0]+(-0.80371987894)*x[1]**o+(0.614102223184)*x[1]+(-0.821741166107)*x[2]**o+(-0.245209306654)*x[2]
        arg[1,0,1]=(-0.288791346674)*x[0]**o+(0.0125896799209)*x[0]+(0.0536290891541)*x[1]**o+(-0.127968165901)*x[1]+(-0.635591402779)*x[2]**o+(-0.723850350215)*x[2]
        arg[1,1,0]=(-0.554196270608)*x[0]**o+(-0.409293351346)*x[0]+(-0.759821124187)*x[1]**o+(-0.828913354317)*x[1]+(0.66900958614)*x[2]**o+(0.412403847239)*x[2]
        arg[1,1,1]=(-0.862735330269)*x[0]**o+(-0.406644646366)*x[0]+(-0.375399872184)*x[1]**o+(-0.10762837777)*x[1]+(0.540886844249)*x[2]**o+(0.97487281528)*x[2]
        arg[2,0,0]=(0.388899900771)*x[0]**o+(-0.692092360652)*x[0]+(0.794415524913)*x[1]**o+(0.369239170999)*x[1]+(0.482269641926)*x[2]**o+(0.58717167121)*x[2]
        arg[2,0,1]=(0.324795219208)*x[0]**o+(0.508631698719)*x[0]+(0.128456415737)*x[1]**o+(0.73678163559)*x[1]+(0.0527520660005)*x[2]**o+(-0.180289299372)*x[2]
        arg[2,1,0]=(-0.901972287405)*x[0]**o+(-0.635701120264)*x[0]+(0.298343193246)*x[1]**o+(-0.870902858159)*x[1]+(-0.979619200422)*x[2]**o+(0.708567755329)*x[2]
        arg[2,1,1]=(0.391663724321)*x[0]**o+(0.69891810611)*x[0]+(0.717069638885)*x[1]**o+(0.493487853359)*x[1]+(0.270536570073)*x[2]**o+(0.679115502666)*x[2]
        arg[3,0,0]=(0.830590943609)*x[0]**o+(-0.000320739222143)*x[0]+(-0.659146599349)*x[1]**o+(-0.858428982679)*x[1]+(-0.158070494634)*x[2]**o+(-0.364394913197)*x[2]
        arg[3,0,1]=(0.903615175916)*x[0]**o+(0.186562406505)*x[0]+(0.463860110418)*x[1]**o+(0.0382550628307)*x[1]+(0.835148180357)*x[2]**o+(-0.117637969912)*x[2]
        arg[3,1,0]=(-0.901874408714)*x[0]**o+(0.00493918499294)*x[0]+(0.180179475909)*x[1]**o+(-0.549428619271)*x[1]+(-0.392536708051)*x[2]**o+(0.68054148024)*x[2]
        arg[3,1,1]=(0.807693382033)*x[0]**o+(-0.47532617097)*x[0]+(-0.52499678021)*x[1]**o+(-0.0365395713918)*x[1]+(0.654881606829)*x[2]**o+(-0.63350736651)*x[2]
        arg[4,0,0]=(0.440830144989)*x[0]**o+(-0.454694325046)*x[0]+(0.403509667018)*x[1]**o+(0.360080283937)*x[1]+(0.164208775222)*x[2]**o+(0.203945755083)*x[2]
        arg[4,0,1]=(-0.836804887763)*x[0]**o+(-0.910979763109)*x[0]+(-0.629411293881)*x[1]**o+(0.0546020113997)*x[1]+(0.63435754491)*x[2]**o+(-0.323224122475)*x[2]
        arg[4,1,0]=(0.204216251019)*x[0]**o+(-0.372349067194)*x[0]+(-0.320768886225)*x[1]**o+(-0.139715856072)*x[1]+(0.62329130659)*x[2]**o+(0.626784461235)*x[2]
        arg[4,1,1]=(0.275074685897)*x[0]**o+(0.0664691742991)*x[0]+(0.949175504221)*x[1]**o+(-0.438858093561)*x[1]+(0.330489174462)*x[2]**o+(0.361586647545)*x[2]
        arg[5,0,0]=(0.370194889673)*x[0]**o+(-0.0549428893157)*x[0]+(0.0219932528442)*x[1]**o+(-0.699553487775)*x[1]+(0.0161212665659)*x[2]**o+(-0.268426682266)*x[2]
        arg[5,0,1]=(-0.527801267225)*x[0]**o+(0.151774276637)*x[0]+(-0.0623666836223)*x[1]**o+(-0.953572390482)*x[1]+(0.778912307756)*x[2]**o+(-0.851760269136)*x[2]
        arg[5,1,0]=(0.11418535486)*x[0]**o+(-0.299719975234)*x[0]+(-0.508303365848)*x[1]**o+(-0.856367302992)*x[1]+(0.311609370032)*x[2]**o+(0.4731881155)*x[2]
        arg[5,1,1]=(0.455947680647)*x[0]**o+(0.856811965126)*x[0]+(-0.176215162097)*x[1]**o+(0.581325693162)*x[1]+(-0.306514780798)*x[2]**o+(-0.005241787834)*x[2]
        ref[0,0,0]=(-1.62145693785)/(o+1.)+(0.19090734146)+(0.962792187023)*0.5**o
        ref[0,0,1]=(0.941392993115)/(o+1.)+(-0.0327366922471)+(0.644573004654)*0.5**o
        ref[0,1,0]=(1.56871601543)/(o+1.)+(-0.694552108754)+(0.829914553754)*0.5**o
        ref[0,1,1]=(-1.11610832417)/(o+1.)+(0.215432296282)+(0.874166303558)*0.5**o
        ref[1,0,0]=(-1.62546104505)/(o+1.)+(0.383991774237)+(-0.963425479949)*0.5**o
        ref[1,0,1]=(-0.581962313625)/(o+1.)+(-0.419614418097)+(-0.288791346674)*0.5**o
        ref[1,1,0]=(-0.0908115380472)/(o+1.)+(-0.412901429212)+(-0.554196270608)*0.5**o
        ref[1,1,1]=(0.165486972066)/(o+1.)+(0.230299895572)+(-0.862735330269)*0.5**o
        ref[2,0,0]=(1.27668516684)/(o+1.)+(0.132159240779)+(0.388899900771)*0.5**o
        ref[2,0,1]=(0.181208481738)/(o+1.)+(0.532562017469)+(0.324795219208)*0.5**o
        ref[2,1,0]=(-0.681276007176)/(o+1.)+(-0.399018111547)+(-0.901972287405)*0.5**o
        ref[2,1,1]=(0.987606208958)/(o+1.)+(0.935760731067)+(0.391663724321)*0.5**o
        ref[3,0,0]=(-0.817217093983)/(o+1.)+(-0.611572317549)+(0.830590943609)*0.5**o
        ref[3,0,1]=(1.29900829077)/(o+1.)+(0.0535897497116)+(0.903615175916)*0.5**o
        ref[3,1,0]=(-0.212357232141)/(o+1.)+(0.0680260229807)+(-0.901874408714)*0.5**o
        ref[3,1,1]=(0.12988482662)/(o+1.)+(-0.572686554436)+(0.807693382033)*0.5**o
        ref[4,0,0]=(0.567718442239)/(o+1.)+(0.0546658569873)+(0.440830144989)*0.5**o
        ref[4,0,1]=(0.00494625102884)/(o+1.)+(-0.589800937092)+(-0.836804887763)*0.5**o
        ref[4,1,0]=(0.302522420366)/(o+1.)+(0.0573597689848)+(0.204216251019)*0.5**o
        ref[4,1,1]=(1.27966467868)/(o+1.)+(-0.0054011358584)+(0.275074685897)*0.5**o
        ref[5,0,0]=(0.0381145194101)/(o+1.)+(-0.511461529678)+(0.370194889673)*0.5**o
        ref[5,0,1]=(0.716545624134)/(o+1.)+(-0.826779191491)+(-0.527801267225)*0.5**o
        ref[5,1,0]=(-0.196693995816)/(o+1.)+(-0.341449581363)+(0.11418535486)*0.5**o
        ref[5,1,1]=(-0.482729942895)/(o+1.)+(0.716447935227)+(0.455947680647)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_Solution_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.415003066171)*x[0]**o+(-0.47441250281)*x[0]+(0.765391656872)*x[1]**o+(-0.170257448926)*x[1]
        arg[0,0,0,1]=(-0.928539602254)*x[0]**o+(0.792240012243)*x[0]+(-0.152730990885)*x[1]**o+(-0.667500454847)*x[1]
        arg[0,0,1,0]=(-0.937831078575)*x[0]**o+(0.963649346531)*x[0]+(0.505558859525)*x[1]**o+(-0.921354612243)*x[1]
        arg[0,0,1,1]=(0.909415074451)*x[0]**o+(0.199360871726)*x[0]+(0.194087349442)*x[1]**o+(0.875065381425)*x[1]
        arg[0,0,2,0]=(0.935190448625)*x[0]**o+(-0.109984767478)*x[0]+(0.518460161897)*x[1]**o+(-0.597684157349)*x[1]
        arg[0,0,2,1]=(-0.94062714999)*x[0]**o+(-0.308960574286)*x[0]+(0.578391007321)*x[1]**o+(0.864501040998)*x[1]
        arg[0,1,0,0]=(-0.197219678026)*x[0]**o+(-0.982899373778)*x[0]+(-0.603956105825)*x[1]**o+(-0.242717148913)*x[1]
        arg[0,1,0,1]=(-0.590715606611)*x[0]**o+(0.461185487788)*x[0]+(-0.270612249495)*x[1]**o+(-0.644742980654)*x[1]
        arg[0,1,1,0]=(-0.365516638755)*x[0]**o+(-0.255307471195)*x[0]+(0.0875989587033)*x[1]**o+(0.923189427818)*x[1]
        arg[0,1,1,1]=(0.315218558548)*x[0]**o+(-0.373070897551)*x[0]+(0.625272530385)*x[1]**o+(-0.174054543645)*x[1]
        arg[0,1,2,0]=(-0.502497884169)*x[0]**o+(-0.933785405889)*x[0]+(0.115500534905)*x[1]**o+(-0.864454092408)*x[1]
        arg[0,1,2,1]=(0.615837581258)*x[0]**o+(0.832538142367)*x[0]+(-0.0715650027129)*x[1]**o+(-0.775173801322)*x[1]
        arg[0,2,0,0]=(-0.898904577889)*x[0]**o+(-0.430072579352)*x[0]+(-0.532213103734)*x[1]**o+(0.970885652167)*x[1]
        arg[0,2,0,1]=(-0.0056483770747)*x[0]**o+(-0.566550554106)*x[0]+(0.751663691868)*x[1]**o+(0.677878661708)*x[1]
        arg[0,2,1,0]=(0.271196923039)*x[0]**o+(-0.291608730386)*x[0]+(-0.932579716748)*x[1]**o+(-0.437823650176)*x[1]
        arg[0,2,1,1]=(-0.788206773125)*x[0]**o+(0.18675055747)*x[0]+(0.747625244884)*x[1]**o+(0.654933611758)*x[1]
        arg[0,2,2,0]=(-0.787603701957)*x[0]**o+(-0.789784850865)*x[0]+(0.197684766271)*x[1]**o+(-0.988404958072)*x[1]
        arg[0,2,2,1]=(0.417538087399)*x[0]**o+(0.745832250585)*x[0]+(-0.525306115268)*x[1]**o+(0.134086803871)*x[1]
        arg[0,3,0,0]=(-0.230658480436)*x[0]**o+(-0.072983085881)*x[0]+(-0.149386157207)*x[1]**o+(-0.506870604002)*x[1]
        arg[0,3,0,1]=(-0.815859549425)*x[0]**o+(-0.962632495033)*x[0]+(-0.542257139282)*x[1]**o+(0.93598141472)*x[1]
        arg[0,3,1,0]=(0.0508974791413)*x[0]**o+(-0.722669867743)*x[0]+(0.714019527678)*x[1]**o+(0.710015283752)*x[1]
        arg[0,3,1,1]=(0.119419200166)*x[0]**o+(0.365663074152)*x[0]+(0.352455072404)*x[1]**o+(-0.0788953701144)*x[1]
        arg[0,3,2,0]=(-0.82075904747)*x[0]**o+(-0.224534176808)*x[0]+(-0.585094684368)*x[1]**o+(0.242958171035)*x[1]
        arg[0,3,2,1]=(-0.748061395205)*x[0]**o+(0.676995891547)*x[0]+(0.686644239408)*x[1]**o+(0.260350484065)*x[1]
        arg[0,4,0,0]=(0.128293903485)*x[0]**o+(0.226748407809)*x[0]+(0.129373873322)*x[1]**o+(0.988775321025)*x[1]
        arg[0,4,0,1]=(0.639638672694)*x[0]**o+(-0.817791454136)*x[0]+(-0.170677596319)*x[1]**o+(0.404161643274)*x[1]
        arg[0,4,1,0]=(0.43743097727)*x[0]**o+(-0.408230274495)*x[0]+(0.282544628333)*x[1]**o+(-0.542352697967)*x[1]
        arg[0,4,1,1]=(-0.995599965283)*x[0]**o+(-0.0512373236039)*x[0]+(0.218383977926)*x[1]**o+(0.929355335423)*x[1]
        arg[0,4,2,0]=(0.922609534479)*x[0]**o+(-0.166291574079)*x[0]+(-0.449331655188)*x[1]**o+(-0.0279347569009)*x[1]
        arg[0,4,2,1]=(0.950186933361)*x[0]**o+(-0.426615341822)*x[0]+(-0.167639663907)*x[1]**o+(0.219020502174)*x[1]
        arg[1,0,0,0]=(0.0573632292788)*x[0]**o+(0.543681035865)*x[0]+(-0.622510875807)*x[1]**o+(-0.171567467655)*x[1]
        arg[1,0,0,1]=(0.368975600444)*x[0]**o+(-0.754217532725)*x[0]+(0.469744047929)*x[1]**o+(-0.922445394777)*x[1]
        arg[1,0,1,0]=(-0.810194964715)*x[0]**o+(0.0861796117934)*x[0]+(0.625367069433)*x[1]**o+(-0.950906871025)*x[1]
        arg[1,0,1,1]=(0.744923275143)*x[0]**o+(0.465744045178)*x[0]+(0.359324627118)*x[1]**o+(0.273934205663)*x[1]
        arg[1,0,2,0]=(0.891959004941)*x[0]**o+(-0.498241249484)*x[0]+(0.318198109895)*x[1]**o+(0.0674564169104)*x[1]
        arg[1,0,2,1]=(0.842125439023)*x[0]**o+(0.731884782944)*x[0]+(-0.674805144805)*x[1]**o+(0.431782566319)*x[1]
        arg[1,1,0,0]=(-0.786719774805)*x[0]**o+(-0.55929144961)*x[0]+(-0.740263211102)*x[1]**o+(0.777914467998)*x[1]
        arg[1,1,0,1]=(0.831245977993)*x[0]**o+(-0.758991188073)*x[0]+(0.714032957488)*x[1]**o+(-0.879935556766)*x[1]
        arg[1,1,1,0]=(0.862796574462)*x[0]**o+(0.746875042565)*x[0]+(0.455949140326)*x[1]**o+(0.585491946591)*x[1]
        arg[1,1,1,1]=(-0.500409398892)*x[0]**o+(0.292753163538)*x[0]+(-0.62474682088)*x[1]**o+(0.583368504524)*x[1]
        arg[1,1,2,0]=(-0.402886961189)*x[0]**o+(-0.288733078165)*x[0]+(-0.770315341866)*x[1]**o+(-0.462764773299)*x[1]
        arg[1,1,2,1]=(0.358161520677)*x[0]**o+(-0.524052652951)*x[0]+(0.654162491323)*x[1]**o+(0.902382016998)*x[1]
        arg[1,2,0,0]=(-0.15539934232)*x[0]**o+(0.693162750731)*x[0]+(-0.0538737948754)*x[1]**o+(-0.025534990068)*x[1]
        arg[1,2,0,1]=(0.371159212282)*x[0]**o+(-0.751292369171)*x[0]+(-0.377617770006)*x[1]**o+(0.707647483842)*x[1]
        arg[1,2,1,0]=(0.117790556271)*x[0]**o+(0.848017682445)*x[0]+(-0.360936107893)*x[1]**o+(-0.564060873236)*x[1]
        arg[1,2,1,1]=(-0.524828128971)*x[0]**o+(0.737876348154)*x[0]+(0.183316445987)*x[1]**o+(-0.821207746237)*x[1]
        arg[1,2,2,0]=(0.300516177535)*x[0]**o+(-0.204187510383)*x[0]+(0.126601242737)*x[1]**o+(0.0794834423875)*x[1]
        arg[1,2,2,1]=(-0.29046926624)*x[0]**o+(-0.310057692431)*x[0]+(-0.810227948574)*x[1]**o+(-0.464128160277)*x[1]
        arg[1,3,0,0]=(-0.134275834453)*x[0]**o+(0.448577162559)*x[0]+(0.757723452446)*x[1]**o+(0.139202834725)*x[1]
        arg[1,3,0,1]=(0.882915728842)*x[0]**o+(-0.0750591947315)*x[0]+(0.503482692135)*x[1]**o+(-0.176350465386)*x[1]
        arg[1,3,1,0]=(-0.954792910945)*x[0]**o+(0.772215353622)*x[0]+(-0.861747803918)*x[1]**o+(-0.932727438775)*x[1]
        arg[1,3,1,1]=(0.747255393187)*x[0]**o+(0.0392661297355)*x[0]+(-0.322274864209)*x[1]**o+(-0.491462177972)*x[1]
        arg[1,3,2,0]=(0.139287057836)*x[0]**o+(0.312258185231)*x[0]+(-0.488288412308)*x[1]**o+(-0.185131700928)*x[1]
        arg[1,3,2,1]=(-0.861625516304)*x[0]**o+(0.681454010645)*x[0]+(0.206644380777)*x[1]**o+(-0.0344138929268)*x[1]
        arg[1,4,0,0]=(-0.714198122614)*x[0]**o+(-0.413113909739)*x[0]+(-0.971680855409)*x[1]**o+(0.602598611545)*x[1]
        arg[1,4,0,1]=(0.293578109913)*x[0]**o+(-0.495635685351)*x[0]+(0.347408878778)*x[1]**o+(-0.923814185786)*x[1]
        arg[1,4,1,0]=(0.843760126513)*x[0]**o+(-0.671075074679)*x[0]+(-0.284086603407)*x[1]**o+(0.437492233496)*x[1]
        arg[1,4,1,1]=(-0.342911357374)*x[0]**o+(-0.440232224635)*x[0]+(0.537701123084)*x[1]**o+(-0.525203729082)*x[1]
        arg[1,4,2,0]=(-0.71201360314)*x[0]**o+(-0.706332525171)*x[0]+(0.831438049852)*x[1]**o+(0.819141670055)*x[1]
        arg[1,4,2,1]=(-0.616403541928)*x[0]**o+(0.635973739636)*x[0]+(-0.0369972206543)*x[1]**o+(-0.08595692743)*x[1]
        arg[2,0,0,0]=(-0.959754423491)*x[0]**o+(-0.332473557912)*x[0]+(-0.464334304695)*x[1]**o+(0.597214020843)*x[1]
        arg[2,0,0,1]=(0.231051530608)*x[0]**o+(0.368524293555)*x[0]+(0.173044245392)*x[1]**o+(0.505867390994)*x[1]
        arg[2,0,1,0]=(0.376577055194)*x[0]**o+(-0.739990982509)*x[0]+(-0.816201949845)*x[1]**o+(-0.71233193641)*x[1]
        arg[2,0,1,1]=(-0.4891837531)*x[0]**o+(-0.121562226208)*x[0]+(-0.876294548464)*x[1]**o+(0.377763924093)*x[1]
        arg[2,0,2,0]=(-0.759548081841)*x[0]**o+(-0.118749434442)*x[0]+(0.602375475355)*x[1]**o+(-0.877213577079)*x[1]
        arg[2,0,2,1]=(0.114018259802)*x[0]**o+(-0.627149204501)*x[0]+(0.813648191424)*x[1]**o+(-0.612548616323)*x[1]
        arg[2,1,0,0]=(-0.572699800518)*x[0]**o+(0.507715829662)*x[0]+(-0.12122456959)*x[1]**o+(0.75522878228)*x[1]
        arg[2,1,0,1]=(0.0655268452799)*x[0]**o+(0.211191572823)*x[0]+(0.539760655497)*x[1]**o+(0.657397992057)*x[1]
        arg[2,1,1,0]=(0.358281119336)*x[0]**o+(0.119364032507)*x[0]+(-0.0242249415295)*x[1]**o+(-0.645986730609)*x[1]
        arg[2,1,1,1]=(0.523751704889)*x[0]**o+(-0.237670103222)*x[0]+(-0.995787126907)*x[1]**o+(-0.504129088442)*x[1]
        arg[2,1,2,0]=(-0.185380508646)*x[0]**o+(-0.218341933583)*x[0]+(0.149400154478)*x[1]**o+(0.00252616040715)*x[1]
        arg[2,1,2,1]=(0.81086932122)*x[0]**o+(0.387271527481)*x[0]+(0.908525646396)*x[1]**o+(-0.75383394204)*x[1]
        arg[2,2,0,0]=(-0.463416100256)*x[0]**o+(-0.891654021409)*x[0]+(-0.0495088159461)*x[1]**o+(0.398915277911)*x[1]
        arg[2,2,0,1]=(-0.200698149193)*x[0]**o+(0.127899540753)*x[0]+(-0.895280577411)*x[1]**o+(-0.197025422242)*x[1]
        arg[2,2,1,0]=(0.888683772867)*x[0]**o+(-0.84436563509)*x[0]+(-0.375709672588)*x[1]**o+(0.591336312818)*x[1]
        arg[2,2,1,1]=(-0.0167949020901)*x[0]**o+(0.145017501865)*x[0]+(0.868194479789)*x[1]**o+(-0.62439676328)*x[1]
        arg[2,2,2,0]=(0.441981698578)*x[0]**o+(-0.481156166596)*x[0]+(0.267937553127)*x[1]**o+(-0.37032246733)*x[1]
        arg[2,2,2,1]=(0.876701424579)*x[0]**o+(0.473798212378)*x[0]+(0.697645775876)*x[1]**o+(-0.798143376838)*x[1]
        arg[2,3,0,0]=(0.111231509217)*x[0]**o+(0.188121806171)*x[0]+(-0.0706155476337)*x[1]**o+(0.779423336221)*x[1]
        arg[2,3,0,1]=(-0.858913402812)*x[0]**o+(0.655459361097)*x[0]+(0.326282512878)*x[1]**o+(0.863386952614)*x[1]
        arg[2,3,1,0]=(-0.524551221953)*x[0]**o+(-0.807242415277)*x[0]+(0.575866251831)*x[1]**o+(-0.74503615846)*x[1]
        arg[2,3,1,1]=(0.391514075129)*x[0]**o+(0.477257809825)*x[0]+(-0.488263373008)*x[1]**o+(-0.29101442481)*x[1]
        arg[2,3,2,0]=(0.26532129161)*x[0]**o+(0.22794872765)*x[0]+(0.85094548724)*x[1]**o+(-0.369690624817)*x[1]
        arg[2,3,2,1]=(0.2339994493)*x[0]**o+(-0.901432708956)*x[0]+(0.309349741876)*x[1]**o+(0.132010344317)*x[1]
        arg[2,4,0,0]=(-0.713553783036)*x[0]**o+(0.791484993301)*x[0]+(0.339247480459)*x[1]**o+(0.158842307171)*x[1]
        arg[2,4,0,1]=(-0.733490022564)*x[0]**o+(0.631284830844)*x[0]+(0.340009298488)*x[1]**o+(-0.483162310574)*x[1]
        arg[2,4,1,0]=(0.709413437683)*x[0]**o+(0.896590124659)*x[0]+(0.680862759763)*x[1]**o+(-0.23579845663)*x[1]
        arg[2,4,1,1]=(-0.130761551034)*x[0]**o+(0.269095282394)*x[0]+(0.687201138134)*x[1]**o+(0.905183436376)*x[1]
        arg[2,4,2,0]=(0.256894911335)*x[0]**o+(0.706744915172)*x[0]+(0.910504833397)*x[1]**o+(-0.285088297895)*x[1]
        arg[2,4,2,1]=(0.95137247728)*x[0]**o+(-0.924669436646)*x[0]+(-0.146356272875)*x[1]**o+(-0.673326194162)*x[1]
        arg[3,0,0,0]=(0.663180444534)*x[0]**o+(0.894728781279)*x[0]+(0.894383747285)*x[1]**o+(0.969779862148)*x[1]
        arg[3,0,0,1]=(0.424556283891)*x[0]**o+(-0.97849198275)*x[0]+(0.367908872519)*x[1]**o+(0.624098355956)*x[1]
        arg[3,0,1,0]=(0.908698580251)*x[0]**o+(-0.899102004477)*x[0]+(0.509310001763)*x[1]**o+(-0.8399697646)*x[1]
        arg[3,0,1,1]=(0.523444225814)*x[0]**o+(0.410650511976)*x[0]+(-0.986699126405)*x[1]**o+(0.081494682537)*x[1]
        arg[3,0,2,0]=(-0.163604985314)*x[0]**o+(0.959338619715)*x[0]+(-0.843900271478)*x[1]**o+(-0.0218386544947)*x[1]
        arg[3,0,2,1]=(-0.490804271432)*x[0]**o+(0.123332672938)*x[0]+(0.902072835174)*x[1]**o+(-0.202340709957)*x[1]
        arg[3,1,0,0]=(-0.949731382685)*x[0]**o+(0.533695608969)*x[0]+(0.430512336347)*x[1]**o+(0.977434372776)*x[1]
        arg[3,1,0,1]=(0.945102298088)*x[0]**o+(-0.117807763531)*x[0]+(-0.203921504027)*x[1]**o+(-0.261266079346)*x[1]
        arg[3,1,1,0]=(0.762442770795)*x[0]**o+(-0.318456087863)*x[0]+(-0.365021977395)*x[1]**o+(0.0516932816311)*x[1]
        arg[3,1,1,1]=(-0.829061607399)*x[0]**o+(0.854900781365)*x[0]+(-0.397139314214)*x[1]**o+(-0.549679351555)*x[1]
        arg[3,1,2,0]=(-0.780040060687)*x[0]**o+(0.719393460042)*x[0]+(-0.435377419129)*x[1]**o+(0.655962063649)*x[1]
        arg[3,1,2,1]=(-0.853144390228)*x[0]**o+(0.0940222442927)*x[0]+(-0.642258121264)*x[1]**o+(-0.615386923437)*x[1]
        arg[3,2,0,0]=(0.377161814597)*x[0]**o+(0.349238572634)*x[0]+(0.361300611554)*x[1]**o+(0.554382639639)*x[1]
        arg[3,2,0,1]=(-0.51174671413)*x[0]**o+(-0.0586248632463)*x[0]+(0.083224329707)*x[1]**o+(0.506614699443)*x[1]
        arg[3,2,1,0]=(0.612695079311)*x[0]**o+(0.82053876378)*x[0]+(-0.132608876197)*x[1]**o+(0.701720563108)*x[1]
        arg[3,2,1,1]=(0.0928734170244)*x[0]**o+(-0.127564289866)*x[0]+(0.68468196965)*x[1]**o+(0.0168958305852)*x[1]
        arg[3,2,2,0]=(-0.863542893122)*x[0]**o+(0.0533007397457)*x[0]+(-0.0602194449612)*x[1]**o+(0.391122261927)*x[1]
        arg[3,2,2,1]=(-0.0942537498633)*x[0]**o+(-0.421741120475)*x[0]+(-0.0746938765947)*x[1]**o+(0.988111802567)*x[1]
        arg[3,3,0,0]=(-0.0588913579205)*x[0]**o+(0.439330145374)*x[0]+(-0.541714868844)*x[1]**o+(0.845344272279)*x[1]
        arg[3,3,0,1]=(0.221784155922)*x[0]**o+(0.983539923382)*x[0]+(-0.32332836636)*x[1]**o+(-0.892579743718)*x[1]
        arg[3,3,1,0]=(0.804097639203)*x[0]**o+(-0.342072604164)*x[0]+(0.755681096012)*x[1]**o+(0.432897246925)*x[1]
        arg[3,3,1,1]=(0.825696602707)*x[0]**o+(0.536999112965)*x[0]+(0.745033643035)*x[1]**o+(0.156806358776)*x[1]
        arg[3,3,2,0]=(-0.372114391352)*x[0]**o+(0.670417556114)*x[0]+(-0.569395510546)*x[1]**o+(-0.523784408534)*x[1]
        arg[3,3,2,1]=(0.439723359721)*x[0]**o+(0.809733189174)*x[0]+(0.781753556415)*x[1]**o+(0.375991547044)*x[1]
        arg[3,4,0,0]=(-0.429896412297)*x[0]**o+(-0.908820042346)*x[0]+(0.328787484151)*x[1]**o+(-0.610488251755)*x[1]
        arg[3,4,0,1]=(0.803567111353)*x[0]**o+(-0.487219379817)*x[0]+(-0.541538468561)*x[1]**o+(-0.692964506777)*x[1]
        arg[3,4,1,0]=(-0.963266835039)*x[0]**o+(0.667510333677)*x[0]+(0.308312799035)*x[1]**o+(0.686605377458)*x[1]
        arg[3,4,1,1]=(0.651198604373)*x[0]**o+(0.306465142929)*x[0]+(-0.923691546947)*x[1]**o+(-0.376240035765)*x[1]
        arg[3,4,2,0]=(-0.339406539705)*x[0]**o+(0.524709759622)*x[0]+(-0.0326564953763)*x[1]**o+(-0.658296780229)*x[1]
        arg[3,4,2,1]=(0.471355209305)*x[0]**o+(-0.473756698531)*x[0]+(0.193890658409)*x[1]**o+(-0.266632530307)*x[1]
        ref[0,0,0,0]=(0.765391656872)/(o+1.)+(-0.322334975868)+(0.415003066171)*0.5**o
        ref[0,0,0,1]=(-0.152730990885)/(o+1.)+(0.0623697786981)+(-0.928539602254)*0.5**o
        ref[0,0,1,0]=(0.505558859525)/(o+1.)+(0.0211473671443)+(-0.937831078575)*0.5**o
        ref[0,0,1,1]=(0.194087349442)/(o+1.)+(0.537213126576)+(0.909415074451)*0.5**o
        ref[0,0,2,0]=(0.518460161897)/(o+1.)+(-0.353834462413)+(0.935190448625)*0.5**o
        ref[0,0,2,1]=(0.578391007321)/(o+1.)+(0.277770233356)+(-0.94062714999)*0.5**o
        ref[0,1,0,0]=(-0.603956105825)/(o+1.)+(-0.612808261345)+(-0.197219678026)*0.5**o
        ref[0,1,0,1]=(-0.270612249495)/(o+1.)+(-0.0917787464331)+(-0.590715606611)*0.5**o
        ref[0,1,1,0]=(0.0875989587033)/(o+1.)+(0.333940978312)+(-0.365516638755)*0.5**o
        ref[0,1,1,1]=(0.625272530385)/(o+1.)+(-0.273562720598)+(0.315218558548)*0.5**o
        ref[0,1,2,0]=(0.115500534905)/(o+1.)+(-0.899119749149)+(-0.502497884169)*0.5**o
        ref[0,1,2,1]=(-0.0715650027129)/(o+1.)+(0.0286821705227)+(0.615837581258)*0.5**o
        ref[0,2,0,0]=(-0.532213103734)/(o+1.)+(0.270406536408)+(-0.898904577889)*0.5**o
        ref[0,2,0,1]=(0.751663691868)/(o+1.)+(0.0556640538013)+(-0.0056483770747)*0.5**o
        ref[0,2,1,0]=(-0.932579716748)/(o+1.)+(-0.364716190281)+(0.271196923039)*0.5**o
        ref[0,2,1,1]=(0.747625244884)/(o+1.)+(0.420842084614)+(-0.788206773125)*0.5**o
        ref[0,2,2,0]=(0.197684766271)/(o+1.)+(-0.889094904468)+(-0.787603701957)*0.5**o
        ref[0,2,2,1]=(-0.525306115268)/(o+1.)+(0.439959527228)+(0.417538087399)*0.5**o
        ref[0,3,0,0]=(-0.149386157207)/(o+1.)+(-0.289926844941)+(-0.230658480436)*0.5**o
        ref[0,3,0,1]=(-0.542257139282)/(o+1.)+(-0.0133255401565)+(-0.815859549425)*0.5**o
        ref[0,3,1,0]=(0.714019527678)/(o+1.)+(-0.00632729199566)+(0.0508974791413)*0.5**o
        ref[0,3,1,1]=(0.352455072404)/(o+1.)+(0.143383852019)+(0.119419200166)*0.5**o
        ref[0,3,2,0]=(-0.585094684368)/(o+1.)+(0.00921199711359)+(-0.82075904747)*0.5**o
        ref[0,3,2,1]=(0.686644239408)/(o+1.)+(0.468673187806)+(-0.748061395205)*0.5**o
        ref[0,4,0,0]=(0.129373873322)/(o+1.)+(0.607761864417)+(0.128293903485)*0.5**o
        ref[0,4,0,1]=(-0.170677596319)/(o+1.)+(-0.206814905431)+(0.639638672694)*0.5**o
        ref[0,4,1,0]=(0.282544628333)/(o+1.)+(-0.475291486231)+(0.43743097727)*0.5**o
        ref[0,4,1,1]=(0.218383977926)/(o+1.)+(0.43905900591)+(-0.995599965283)*0.5**o
        ref[0,4,2,0]=(-0.449331655188)/(o+1.)+(-0.0971131654902)+(0.922609534479)*0.5**o
        ref[0,4,2,1]=(-0.167639663907)/(o+1.)+(-0.103797419824)+(0.950186933361)*0.5**o
        ref[1,0,0,0]=(-0.622510875807)/(o+1.)+(0.186056784105)+(0.0573632292788)*0.5**o
        ref[1,0,0,1]=(0.469744047929)/(o+1.)+(-0.838331463751)+(0.368975600444)*0.5**o
        ref[1,0,1,0]=(0.625367069433)/(o+1.)+(-0.432363629616)+(-0.810194964715)*0.5**o
        ref[1,0,1,1]=(0.359324627118)/(o+1.)+(0.36983912542)+(0.744923275143)*0.5**o
        ref[1,0,2,0]=(0.318198109895)/(o+1.)+(-0.215392416287)+(0.891959004941)*0.5**o
        ref[1,0,2,1]=(-0.674805144805)/(o+1.)+(0.581833674632)+(0.842125439023)*0.5**o
        ref[1,1,0,0]=(-0.740263211102)/(o+1.)+(0.109311509194)+(-0.786719774805)*0.5**o
        ref[1,1,0,1]=(0.714032957488)/(o+1.)+(-0.81946337242)+(0.831245977993)*0.5**o
        ref[1,1,1,0]=(0.455949140326)/(o+1.)+(0.666183494578)+(0.862796574462)*0.5**o
        ref[1,1,1,1]=(-0.62474682088)/(o+1.)+(0.438060834031)+(-0.500409398892)*0.5**o
        ref[1,1,2,0]=(-0.770315341866)/(o+1.)+(-0.375748925732)+(-0.402886961189)*0.5**o
        ref[1,1,2,1]=(0.654162491323)/(o+1.)+(0.189164682023)+(0.358161520677)*0.5**o
        ref[1,2,0,0]=(-0.0538737948754)/(o+1.)+(0.333813880332)+(-0.15539934232)*0.5**o
        ref[1,2,0,1]=(-0.377617770006)/(o+1.)+(-0.0218224426646)+(0.371159212282)*0.5**o
        ref[1,2,1,0]=(-0.360936107893)/(o+1.)+(0.141978404604)+(0.117790556271)*0.5**o
        ref[1,2,1,1]=(0.183316445987)/(o+1.)+(-0.0416656990415)+(-0.524828128971)*0.5**o
        ref[1,2,2,0]=(0.126601242737)/(o+1.)+(-0.0623520339976)+(0.300516177535)*0.5**o
        ref[1,2,2,1]=(-0.810227948574)/(o+1.)+(-0.387092926354)+(-0.29046926624)*0.5**o
        ref[1,3,0,0]=(0.757723452446)/(o+1.)+(0.293889998642)+(-0.134275834453)*0.5**o
        ref[1,3,0,1]=(0.503482692135)/(o+1.)+(-0.125704830059)+(0.882915728842)*0.5**o
        ref[1,3,1,0]=(-0.861747803918)/(o+1.)+(-0.0802560425765)+(-0.954792910945)*0.5**o
        ref[1,3,1,1]=(-0.322274864209)/(o+1.)+(-0.226098024118)+(0.747255393187)*0.5**o
        ref[1,3,2,0]=(-0.488288412308)/(o+1.)+(0.0635632421511)+(0.139287057836)*0.5**o
        ref[1,3,2,1]=(0.206644380777)/(o+1.)+(0.323520058859)+(-0.861625516304)*0.5**o
        ref[1,4,0,0]=(-0.971680855409)/(o+1.)+(0.0947423509032)+(-0.714198122614)*0.5**o
        ref[1,4,0,1]=(0.347408878778)/(o+1.)+(-0.709724935569)+(0.293578109913)*0.5**o
        ref[1,4,1,0]=(-0.284086603407)/(o+1.)+(-0.116791420591)+(0.843760126513)*0.5**o
        ref[1,4,1,1]=(0.537701123084)/(o+1.)+(-0.482717976859)+(-0.342911357374)*0.5**o
        ref[1,4,2,0]=(0.831438049852)/(o+1.)+(0.056404572442)+(-0.71201360314)*0.5**o
        ref[1,4,2,1]=(-0.0369972206543)/(o+1.)+(0.275008406103)+(-0.616403541928)*0.5**o
        ref[2,0,0,0]=(-0.464334304695)/(o+1.)+(0.132370231466)+(-0.959754423491)*0.5**o
        ref[2,0,0,1]=(0.173044245392)/(o+1.)+(0.437195842274)+(0.231051530608)*0.5**o
        ref[2,0,1,0]=(-0.816201949845)/(o+1.)+(-0.72616145946)+(0.376577055194)*0.5**o
        ref[2,0,1,1]=(-0.876294548464)/(o+1.)+(0.128100848943)+(-0.4891837531)*0.5**o
        ref[2,0,2,0]=(0.602375475355)/(o+1.)+(-0.497981505761)+(-0.759548081841)*0.5**o
        ref[2,0,2,1]=(0.813648191424)/(o+1.)+(-0.619848910412)+(0.114018259802)*0.5**o
        ref[2,1,0,0]=(-0.12122456959)/(o+1.)+(0.631472305971)+(-0.572699800518)*0.5**o
        ref[2,1,0,1]=(0.539760655497)/(o+1.)+(0.43429478244)+(0.0655268452799)*0.5**o
        ref[2,1,1,0]=(-0.0242249415295)/(o+1.)+(-0.263311349051)+(0.358281119336)*0.5**o
        ref[2,1,1,1]=(-0.995787126907)/(o+1.)+(-0.370899595832)+(0.523751704889)*0.5**o
        ref[2,1,2,0]=(0.149400154478)/(o+1.)+(-0.107907886588)+(-0.185380508646)*0.5**o
        ref[2,1,2,1]=(0.908525646396)/(o+1.)+(-0.18328120728)+(0.81086932122)*0.5**o
        ref[2,2,0,0]=(-0.0495088159461)/(o+1.)+(-0.246369371749)+(-0.463416100256)*0.5**o
        ref[2,2,0,1]=(-0.895280577411)/(o+1.)+(-0.0345629407446)+(-0.200698149193)*0.5**o
        ref[2,2,1,0]=(-0.375709672588)/(o+1.)+(-0.126514661136)+(0.888683772867)*0.5**o
        ref[2,2,1,1]=(0.868194479789)/(o+1.)+(-0.239689630708)+(-0.0167949020901)*0.5**o
        ref[2,2,2,0]=(0.267937553127)/(o+1.)+(-0.425739316963)+(0.441981698578)*0.5**o
        ref[2,2,2,1]=(0.697645775876)/(o+1.)+(-0.16217258223)+(0.876701424579)*0.5**o
        ref[2,3,0,0]=(-0.0706155476337)/(o+1.)+(0.483772571196)+(0.111231509217)*0.5**o
        ref[2,3,0,1]=(0.326282512878)/(o+1.)+(0.759423156855)+(-0.858913402812)*0.5**o
        ref[2,3,1,0]=(0.575866251831)/(o+1.)+(-0.776139286869)+(-0.524551221953)*0.5**o
        ref[2,3,1,1]=(-0.488263373008)/(o+1.)+(0.0931216925076)+(0.391514075129)*0.5**o
        ref[2,3,2,0]=(0.85094548724)/(o+1.)+(-0.0708709485834)+(0.26532129161)*0.5**o
        ref[2,3,2,1]=(0.309349741876)/(o+1.)+(-0.384711182319)+(0.2339994493)*0.5**o
        ref[2,4,0,0]=(0.339247480459)/(o+1.)+(0.475163650236)+(-0.713553783036)*0.5**o
        ref[2,4,0,1]=(0.340009298488)/(o+1.)+(0.0740612601349)+(-0.733490022564)*0.5**o
        ref[2,4,1,0]=(0.680862759763)/(o+1.)+(0.330395834014)+(0.709413437683)*0.5**o
        ref[2,4,1,1]=(0.687201138134)/(o+1.)+(0.587139359385)+(-0.130761551034)*0.5**o
        ref[2,4,2,0]=(0.910504833397)/(o+1.)+(0.210828308638)+(0.256894911335)*0.5**o
        ref[2,4,2,1]=(-0.146356272875)/(o+1.)+(-0.798997815404)+(0.95137247728)*0.5**o
        ref[3,0,0,0]=(0.894383747285)/(o+1.)+(0.932254321713)+(0.663180444534)*0.5**o
        ref[3,0,0,1]=(0.367908872519)/(o+1.)+(-0.177196813397)+(0.424556283891)*0.5**o
        ref[3,0,1,0]=(0.509310001763)/(o+1.)+(-0.869535884539)+(0.908698580251)*0.5**o
        ref[3,0,1,1]=(-0.986699126405)/(o+1.)+(0.246072597257)+(0.523444225814)*0.5**o
        ref[3,0,2,0]=(-0.843900271478)/(o+1.)+(0.46874998261)+(-0.163604985314)*0.5**o
        ref[3,0,2,1]=(0.902072835174)/(o+1.)+(-0.0395040185096)+(-0.490804271432)*0.5**o
        ref[3,1,0,0]=(0.430512336347)/(o+1.)+(0.755564990872)+(-0.949731382685)*0.5**o
        ref[3,1,0,1]=(-0.203921504027)/(o+1.)+(-0.189536921439)+(0.945102298088)*0.5**o
        ref[3,1,1,0]=(-0.365021977395)/(o+1.)+(-0.133381403116)+(0.762442770795)*0.5**o
        ref[3,1,1,1]=(-0.397139314214)/(o+1.)+(0.152610714905)+(-0.829061607399)*0.5**o
        ref[3,1,2,0]=(-0.435377419129)/(o+1.)+(0.687677761846)+(-0.780040060687)*0.5**o
        ref[3,1,2,1]=(-0.642258121264)/(o+1.)+(-0.260682339572)+(-0.853144390228)*0.5**o
        ref[3,2,0,0]=(0.361300611554)/(o+1.)+(0.451810606136)+(0.377161814597)*0.5**o
        ref[3,2,0,1]=(0.083224329707)/(o+1.)+(0.223994918098)+(-0.51174671413)*0.5**o
        ref[3,2,1,0]=(-0.132608876197)/(o+1.)+(0.761129663444)+(0.612695079311)*0.5**o
        ref[3,2,1,1]=(0.68468196965)/(o+1.)+(-0.0553342296404)+(0.0928734170244)*0.5**o
        ref[3,2,2,0]=(-0.0602194449612)/(o+1.)+(0.222211500837)+(-0.863542893122)*0.5**o
        ref[3,2,2,1]=(-0.0746938765947)/(o+1.)+(0.283185341046)+(-0.0942537498633)*0.5**o
        ref[3,3,0,0]=(-0.541714868844)/(o+1.)+(0.642337208826)+(-0.0588913579205)*0.5**o
        ref[3,3,0,1]=(-0.32332836636)/(o+1.)+(0.0454800898319)+(0.221784155922)*0.5**o
        ref[3,3,1,0]=(0.755681096012)/(o+1.)+(0.0454123213802)+(0.804097639203)*0.5**o
        ref[3,3,1,1]=(0.745033643035)/(o+1.)+(0.34690273587)+(0.825696602707)*0.5**o
        ref[3,3,2,0]=(-0.569395510546)/(o+1.)+(0.0733165737898)+(-0.372114391352)*0.5**o
        ref[3,3,2,1]=(0.781753556415)/(o+1.)+(0.592862368109)+(0.439723359721)*0.5**o
        ref[3,4,0,0]=(0.328787484151)/(o+1.)+(-0.75965414705)+(-0.429896412297)*0.5**o
        ref[3,4,0,1]=(-0.541538468561)/(o+1.)+(-0.590091943297)+(0.803567111353)*0.5**o
        ref[3,4,1,0]=(0.308312799035)/(o+1.)+(0.677057855567)+(-0.963266835039)*0.5**o
        ref[3,4,1,1]=(-0.923691546947)/(o+1.)+(-0.0348874464182)+(0.651198604373)*0.5**o
        ref[3,4,2,0]=(-0.0326564953763)/(o+1.)+(-0.0667935103036)+(-0.339406539705)*0.5**o
        ref[3,4,2,1]=(0.193890658409)/(o+1.)+(-0.370194614419)+(0.471355209305)*0.5**o
      else:
        arg[0,0,0,0]=(0.0737627367843)*x[0]**o+(-0.897508234376)*x[0]+(0.499361502035)*x[1]**o+(-0.830793706142)*x[1]+(0.967672023686)*x[2]**o+(-0.94209897356)*x[2]
        arg[0,0,0,1]=(0.353968533247)*x[0]**o+(-0.828575104081)*x[0]+(0.111197799977)*x[1]**o+(0.475627052455)*x[1]+(0.10233739786)*x[2]**o+(-0.297090205985)*x[2]
        arg[0,0,1,0]=(0.315704665816)*x[0]**o+(0.450577419574)*x[0]+(-0.0225047225996)*x[1]**o+(0.221436098936)*x[1]+(-0.210708820992)*x[2]**o+(0.22813686037)*x[2]
        arg[0,0,1,1]=(-0.453912049362)*x[0]**o+(0.443266972459)*x[0]+(-0.166003491301)*x[1]**o+(0.325157034345)*x[1]+(0.975435329063)*x[2]**o+(0.516477372955)*x[2]
        arg[0,0,2,0]=(0.705225432895)*x[0]**o+(0.406870556161)*x[0]+(0.967131048725)*x[1]**o+(-0.31111693058)*x[1]+(0.915125425876)*x[2]**o+(-0.417407852906)*x[2]
        arg[0,0,2,1]=(-0.397162258761)*x[0]**o+(-0.49079016751)*x[0]+(-0.444905720899)*x[1]**o+(0.258998080457)*x[1]+(-0.406172174923)*x[2]**o+(0.166620074269)*x[2]
        arg[0,1,0,0]=(0.00444300957662)*x[0]**o+(0.59448685984)*x[0]+(-0.755596203939)*x[1]**o+(-0.587866578467)*x[1]+(-0.994620629569)*x[2]**o+(0.558539124849)*x[2]
        arg[0,1,0,1]=(0.726535641072)*x[0]**o+(-0.515254556008)*x[0]+(-0.737095044797)*x[1]**o+(0.396218854768)*x[1]+(-0.873340093434)*x[2]**o+(0.538194586759)*x[2]
        arg[0,1,1,0]=(-0.554579794515)*x[0]**o+(-0.754117891877)*x[0]+(0.242579027102)*x[1]**o+(0.368307663104)*x[1]+(0.308372932825)*x[2]**o+(-0.521440130826)*x[2]
        arg[0,1,1,1]=(0.294744137189)*x[0]**o+(0.44996935134)*x[0]+(0.118263237453)*x[1]**o+(-0.143566244715)*x[1]+(-0.927385432821)*x[2]**o+(0.912735237305)*x[2]
        arg[0,1,2,0]=(-0.30261020867)*x[0]**o+(-0.994883662031)*x[0]+(0.514662053738)*x[1]**o+(0.327770859964)*x[1]+(0.862452368458)*x[2]**o+(-0.739026842429)*x[2]
        arg[0,1,2,1]=(0.494855964972)*x[0]**o+(-0.358461686787)*x[0]+(0.406489619959)*x[1]**o+(0.417722618206)*x[1]+(0.679808195501)*x[2]**o+(-0.00502769522311)*x[2]
        arg[0,2,0,0]=(-0.914585215232)*x[0]**o+(0.305890296017)*x[0]+(-0.867826544668)*x[1]**o+(0.304580882671)*x[1]+(-0.401401956053)*x[2]**o+(0.842276423411)*x[2]
        arg[0,2,0,1]=(-0.260590397572)*x[0]**o+(0.541867335986)*x[0]+(0.0279475511153)*x[1]**o+(0.0255320376447)*x[1]+(-0.584584718877)*x[2]**o+(-0.640388032783)*x[2]
        arg[0,2,1,0]=(0.0455797513404)*x[0]**o+(-0.764984318339)*x[0]+(-0.257174640639)*x[1]**o+(0.745888292962)*x[1]+(0.964524202534)*x[2]**o+(0.435680191282)*x[2]
        arg[0,2,1,1]=(-0.300900136798)*x[0]**o+(-0.899164082043)*x[0]+(-0.63273853331)*x[1]**o+(0.56522562781)*x[1]+(-0.846482371222)*x[2]**o+(-0.909985752269)*x[2]
        arg[0,2,2,0]=(0.745711773093)*x[0]**o+(0.0627291150681)*x[0]+(-0.090340386706)*x[1]**o+(-0.0912293084958)*x[1]+(0.108613205226)*x[2]**o+(-0.28985444593)*x[2]
        arg[0,2,2,1]=(0.291340077334)*x[0]**o+(0.592856577974)*x[0]+(0.141857446535)*x[1]**o+(0.830526360843)*x[1]+(0.956840406828)*x[2]**o+(-0.717465405888)*x[2]
        arg[0,3,0,0]=(0.89242099254)*x[0]**o+(0.529348654364)*x[0]+(-0.074382350183)*x[1]**o+(0.874670648345)*x[1]+(-0.728104560168)*x[2]**o+(0.0781153963606)*x[2]
        arg[0,3,0,1]=(0.095390666498)*x[0]**o+(0.0124610436496)*x[0]+(0.278014506594)*x[1]**o+(0.923928359216)*x[1]+(-0.0905667357972)*x[2]**o+(-0.894031439498)*x[2]
        arg[0,3,1,0]=(-0.317750910722)*x[0]**o+(0.677406387747)*x[0]+(0.550094997541)*x[1]**o+(-0.30098732949)*x[1]+(0.501753868155)*x[2]**o+(-0.42859127752)*x[2]
        arg[0,3,1,1]=(-0.204455448392)*x[0]**o+(0.445237113438)*x[0]+(0.853596204887)*x[1]**o+(0.315701645178)*x[1]+(0.307192864939)*x[2]**o+(-0.708325809173)*x[2]
        arg[0,3,2,0]=(0.38581656579)*x[0]**o+(-0.79289714243)*x[0]+(-0.412226202311)*x[1]**o+(0.655534112291)*x[1]+(0.31352728014)*x[2]**o+(0.502089385396)*x[2]
        arg[0,3,2,1]=(-0.225571343976)*x[0]**o+(0.687765563016)*x[0]+(0.0421378888728)*x[1]**o+(0.54466301835)*x[1]+(-0.430288255786)*x[2]**o+(-0.117571440224)*x[2]
        arg[0,4,0,0]=(0.834083057621)*x[0]**o+(0.0292441566815)*x[0]+(-0.947154369416)*x[1]**o+(0.597153808995)*x[1]+(0.323706009194)*x[2]**o+(0.664093182644)*x[2]
        arg[0,4,0,1]=(-0.613152578007)*x[0]**o+(0.253155875339)*x[0]+(0.297138228345)*x[1]**o+(-0.604393247773)*x[1]+(-0.936718722281)*x[2]**o+(-0.410140848312)*x[2]
        arg[0,4,1,0]=(-0.373668614153)*x[0]**o+(-0.401240526097)*x[0]+(-0.417458139032)*x[1]**o+(0.456033864594)*x[1]+(-0.158434012225)*x[2]**o+(-0.33938136039)*x[2]
        arg[0,4,1,1]=(0.00943802199128)*x[0]**o+(-0.800756064115)*x[0]+(0.899199671858)*x[1]**o+(0.376830611387)*x[1]+(-0.930822199046)*x[2]**o+(0.192235807799)*x[2]
        arg[0,4,2,0]=(0.157099024354)*x[0]**o+(0.195942080381)*x[0]+(-0.554519054021)*x[1]**o+(0.871823312361)*x[1]+(-0.34803033039)*x[2]**o+(-0.646933331859)*x[2]
        arg[0,4,2,1]=(0.933762543076)*x[0]**o+(-0.169843985744)*x[0]+(0.971089176971)*x[1]**o+(-0.605176382286)*x[1]+(0.254369209427)*x[2]**o+(0.0314820745541)*x[2]
        arg[1,0,0,0]=(0.0417476479889)*x[0]**o+(-0.144293810851)*x[0]+(-0.677039420744)*x[1]**o+(-0.137097132377)*x[1]+(0.721082639859)*x[2]**o+(-0.32510577691)*x[2]
        arg[1,0,0,1]=(0.450986977446)*x[0]**o+(-0.921371106219)*x[0]+(0.536898979198)*x[1]**o+(-0.814951108186)*x[1]+(-0.733774506956)*x[2]**o+(0.938941747764)*x[2]
        arg[1,0,1,0]=(-0.23712795738)*x[0]**o+(0.450883240541)*x[0]+(0.101102533189)*x[1]**o+(-0.998253664232)*x[1]+(-0.843977488601)*x[2]**o+(-0.807025450448)*x[2]
        arg[1,0,1,1]=(-0.308721170682)*x[0]**o+(-0.546268734305)*x[0]+(0.643628083343)*x[1]**o+(0.206934468523)*x[1]+(-0.257667927436)*x[2]**o+(0.468860780891)*x[2]
        arg[1,0,2,0]=(-0.267232319593)*x[0]**o+(0.221737693042)*x[0]+(0.202218228468)*x[1]**o+(-0.560941556406)*x[1]+(0.910244160614)*x[2]**o+(0.361584784536)*x[2]
        arg[1,0,2,1]=(0.405254374432)*x[0]**o+(-0.102574362257)*x[0]+(0.527874164745)*x[1]**o+(0.0318395137256)*x[1]+(-0.448050293943)*x[2]**o+(-0.6401160036)*x[2]
        arg[1,1,0,0]=(-0.877224972777)*x[0]**o+(-0.544175806369)*x[0]+(0.528037913494)*x[1]**o+(-0.477854554671)*x[1]+(-0.394875438861)*x[2]**o+(-0.704656012962)*x[2]
        arg[1,1,0,1]=(-0.873928172379)*x[0]**o+(0.273455449944)*x[0]+(0.681090480547)*x[1]**o+(0.883243766093)*x[1]+(0.557357114631)*x[2]**o+(0.586148966774)*x[2]
        arg[1,1,1,0]=(0.552826305913)*x[0]**o+(-0.277269605193)*x[0]+(-0.681872908424)*x[1]**o+(-0.84578699459)*x[1]+(0.6543624385)*x[2]**o+(-0.355051513153)*x[2]
        arg[1,1,1,1]=(-0.71233607947)*x[0]**o+(-0.779861555179)*x[0]+(0.11496674995)*x[1]**o+(0.634901982779)*x[1]+(0.833044137166)*x[2]**o+(0.0443514408009)*x[2]
        arg[1,1,2,0]=(-0.411667274338)*x[0]**o+(0.218709061898)*x[0]+(0.132573928273)*x[1]**o+(-0.450677430962)*x[1]+(0.88420873514)*x[2]**o+(-0.618513305411)*x[2]
        arg[1,1,2,1]=(0.533005373575)*x[0]**o+(0.846343252936)*x[0]+(0.697076801609)*x[1]**o+(0.342260840742)*x[1]+(0.923380136498)*x[2]**o+(-0.727450493778)*x[2]
        arg[1,2,0,0]=(-0.637366803484)*x[0]**o+(-0.654649539562)*x[0]+(-0.443035194144)*x[1]**o+(0.4928304386)*x[1]+(0.889290653473)*x[2]**o+(0.208513517091)*x[2]
        arg[1,2,0,1]=(-0.404870452835)*x[0]**o+(-0.556431486625)*x[0]+(-0.428885935216)*x[1]**o+(0.208771213187)*x[1]+(0.706109349268)*x[2]**o+(0.726168991522)*x[2]
        arg[1,2,1,0]=(0.85574176569)*x[0]**o+(-0.371989246254)*x[0]+(0.0376283091444)*x[1]**o+(-0.164811219077)*x[1]+(0.842513532253)*x[2]**o+(-0.963454790343)*x[2]
        arg[1,2,1,1]=(0.576825533647)*x[0]**o+(0.866570909412)*x[0]+(-0.286228055595)*x[1]**o+(-0.509258889664)*x[1]+(0.358087406384)*x[2]**o+(0.2558870556)*x[2]
        arg[1,2,2,0]=(0.741411326194)*x[0]**o+(0.887444210472)*x[0]+(0.176659455323)*x[1]**o+(-0.292731470152)*x[1]+(0.179836617557)*x[2]**o+(-0.10894424483)*x[2]
        arg[1,2,2,1]=(0.386515069261)*x[0]**o+(-0.423389396097)*x[0]+(0.300384913804)*x[1]**o+(-0.569308908813)*x[1]+(0.845849686169)*x[2]**o+(-0.802483942058)*x[2]
        arg[1,3,0,0]=(0.848952965744)*x[0]**o+(0.113555873929)*x[0]+(0.132168122705)*x[1]**o+(0.464116031047)*x[1]+(-0.928746352905)*x[2]**o+(-0.176374657603)*x[2]
        arg[1,3,0,1]=(0.357870838328)*x[0]**o+(-0.294290586373)*x[0]+(0.395859596543)*x[1]**o+(-0.265635871788)*x[1]+(0.283887401402)*x[2]**o+(0.500378090749)*x[2]
        arg[1,3,1,0]=(-0.509333127149)*x[0]**o+(0.328856435768)*x[0]+(-0.377387784476)*x[1]**o+(-0.878505533565)*x[1]+(0.448760304667)*x[2]**o+(0.996692047048)*x[2]
        arg[1,3,1,1]=(-0.837885550386)*x[0]**o+(0.0608579436757)*x[0]+(-0.627132023609)*x[1]**o+(0.902538475491)*x[1]+(-0.449596037062)*x[2]**o+(-0.158786724549)*x[2]
        arg[1,3,2,0]=(0.205889276236)*x[0]**o+(0.576561726109)*x[0]+(-0.00281354319689)*x[1]**o+(0.214968600999)*x[1]+(0.811758703832)*x[2]**o+(0.260164242506)*x[2]
        arg[1,3,2,1]=(-0.331660060588)*x[0]**o+(0.713879758554)*x[0]+(0.804549392751)*x[1]**o+(-0.346397636822)*x[1]+(-0.364073503513)*x[2]**o+(0.338612348994)*x[2]
        arg[1,4,0,0]=(0.394201404247)*x[0]**o+(-0.416853395415)*x[0]+(-0.0581465874206)*x[1]**o+(0.105267182588)*x[1]+(0.791799718009)*x[2]**o+(0.0840172853234)*x[2]
        arg[1,4,0,1]=(-0.346144268453)*x[0]**o+(-0.661228785728)*x[0]+(0.326588395606)*x[1]**o+(0.766137046896)*x[1]+(0.566377202267)*x[2]**o+(0.613565461425)*x[2]
        arg[1,4,1,0]=(-0.907566140749)*x[0]**o+(-0.391612881392)*x[0]+(0.263322351365)*x[1]**o+(-0.4373853914)*x[1]+(0.248855925063)*x[2]**o+(0.973515668505)*x[2]
        arg[1,4,1,1]=(0.726194147182)*x[0]**o+(0.497593423927)*x[0]+(-0.951344923018)*x[1]**o+(0.55327321528)*x[1]+(-0.266195110749)*x[2]**o+(-0.310893832826)*x[2]
        arg[1,4,2,0]=(-0.940824546553)*x[0]**o+(-0.47838524024)*x[0]+(0.408728941942)*x[1]**o+(0.627300786998)*x[1]+(-0.0547545650101)*x[2]**o+(0.433290388525)*x[2]
        arg[1,4,2,1]=(0.728499572603)*x[0]**o+(-0.232088507059)*x[0]+(0.0550057703801)*x[1]**o+(0.159356057226)*x[1]+(0.111191390097)*x[2]**o+(-0.64498018454)*x[2]
        arg[2,0,0,0]=(0.467025642879)*x[0]**o+(-0.695379603222)*x[0]+(0.591732172092)*x[1]**o+(0.84325648142)*x[1]+(0.965696472916)*x[2]**o+(0.626196181241)*x[2]
        arg[2,0,0,1]=(-0.211129222294)*x[0]**o+(-0.501208172412)*x[0]+(-0.0162146409175)*x[1]**o+(-0.672364205306)*x[1]+(-0.851073479662)*x[2]**o+(-0.817568385627)*x[2]
        arg[2,0,1,0]=(-0.668065295797)*x[0]**o+(-0.90359857855)*x[0]+(0.040519772083)*x[1]**o+(-0.295766307198)*x[1]+(0.0491234828524)*x[2]**o+(-0.589557178985)*x[2]
        arg[2,0,1,1]=(0.928751918298)*x[0]**o+(-0.773787731767)*x[0]+(-0.744325160449)*x[1]**o+(-0.426987905682)*x[1]+(0.503339956851)*x[2]**o+(0.386756648382)*x[2]
        arg[2,0,2,0]=(-0.531818105049)*x[0]**o+(-0.586811490681)*x[0]+(0.780379352762)*x[1]**o+(-0.5575328003)*x[1]+(0.0354867101462)*x[2]**o+(0.581226069105)*x[2]
        arg[2,0,2,1]=(0.0150988200159)*x[0]**o+(0.56415563975)*x[0]+(0.540723773613)*x[1]**o+(0.0653377679637)*x[1]+(-0.52633696371)*x[2]**o+(-0.122811009405)*x[2]
        arg[2,1,0,0]=(-0.0829154151121)*x[0]**o+(-0.416192198737)*x[0]+(0.944232067474)*x[1]**o+(-0.159852455491)*x[1]+(-0.198397797188)*x[2]**o+(-0.684211447305)*x[2]
        arg[2,1,0,1]=(-0.834502772339)*x[0]**o+(-0.856817328109)*x[0]+(0.213303574315)*x[1]**o+(-0.283140791695)*x[1]+(-0.727061494352)*x[2]**o+(0.12871372765)*x[2]
        arg[2,1,1,0]=(0.247439924718)*x[0]**o+(-0.912464825903)*x[0]+(0.791177764168)*x[1]**o+(0.160047921728)*x[1]+(0.591086305304)*x[2]**o+(-0.774025941232)*x[2]
        arg[2,1,1,1]=(-0.120169534728)*x[0]**o+(-0.747254680906)*x[0]+(-0.394003570933)*x[1]**o+(-0.938198960695)*x[1]+(-0.0142776288517)*x[2]**o+(0.0627959740849)*x[2]
        arg[2,1,2,0]=(0.703841647274)*x[0]**o+(-0.946039767174)*x[0]+(-0.836717609846)*x[1]**o+(0.776573515772)*x[1]+(0.250502751027)*x[2]**o+(0.403896552902)*x[2]
        arg[2,1,2,1]=(-0.19760048131)*x[0]**o+(-0.250991189554)*x[0]+(0.278810924634)*x[1]**o+(-0.580148097047)*x[1]+(-0.197518781697)*x[2]**o+(0.116835806412)*x[2]
        arg[2,2,0,0]=(-0.507872937196)*x[0]**o+(-0.771841217512)*x[0]+(0.852666314762)*x[1]**o+(-0.482947444136)*x[1]+(0.805824662389)*x[2]**o+(-0.299255374349)*x[2]
        arg[2,2,0,1]=(-0.481403986338)*x[0]**o+(0.904322169038)*x[0]+(0.937553651477)*x[1]**o+(0.727763542606)*x[1]+(-0.69485539356)*x[2]**o+(0.547755372933)*x[2]
        arg[2,2,1,0]=(0.241675645258)*x[0]**o+(0.957681326492)*x[0]+(-0.305747218513)*x[1]**o+(-0.0393749799535)*x[1]+(-0.448892234759)*x[2]**o+(0.707429367761)*x[2]
        arg[2,2,1,1]=(0.114772627763)*x[0]**o+(-0.411147710203)*x[0]+(0.13389320737)*x[1]**o+(0.806309117051)*x[1]+(-0.572620769665)*x[2]**o+(0.0389925355751)*x[2]
        arg[2,2,2,0]=(0.887023209688)*x[0]**o+(0.158261816385)*x[0]+(0.687399231822)*x[1]**o+(-0.858491519094)*x[1]+(-0.00348150506922)*x[2]**o+(-0.863496570874)*x[2]
        arg[2,2,2,1]=(0.836731324893)*x[0]**o+(0.975472534225)*x[0]+(0.407275572226)*x[1]**o+(0.565132026147)*x[1]+(0.79110912675)*x[2]**o+(-0.0816802575667)*x[2]
        arg[2,3,0,0]=(0.196161020545)*x[0]**o+(0.254301467318)*x[0]+(0.759363127106)*x[1]**o+(0.536320544038)*x[1]+(0.522705219849)*x[2]**o+(0.376531450811)*x[2]
        arg[2,3,0,1]=(-0.862326385376)*x[0]**o+(0.766714365146)*x[0]+(0.279327620722)*x[1]**o+(-0.44416461578)*x[1]+(0.115982232563)*x[2]**o+(0.824064889151)*x[2]
        arg[2,3,1,0]=(0.571263839862)*x[0]**o+(-0.679998793927)*x[0]+(0.63548515376)*x[1]**o+(0.464171242163)*x[1]+(0.712124752567)*x[2]**o+(0.913131755622)*x[2]
        arg[2,3,1,1]=(0.272559456412)*x[0]**o+(0.0577748172306)*x[0]+(-0.610464803438)*x[1]**o+(0.233987628538)*x[1]+(0.904147555826)*x[2]**o+(-0.131245442847)*x[2]
        arg[2,3,2,0]=(0.3409363126)*x[0]**o+(0.40060221588)*x[0]+(-0.603551487822)*x[1]**o+(-0.697430353793)*x[1]+(-0.867009193249)*x[2]**o+(-0.178991284979)*x[2]
        arg[2,3,2,1]=(0.949170673623)*x[0]**o+(-0.0217541208215)*x[0]+(-0.775307373317)*x[1]**o+(0.421565612874)*x[1]+(0.419254478365)*x[2]**o+(-0.732203964657)*x[2]
        arg[2,4,0,0]=(0.876774885889)*x[0]**o+(0.441155536177)*x[0]+(-0.292113312365)*x[1]**o+(0.880590008772)*x[1]+(0.0732363738743)*x[2]**o+(-0.590855067487)*x[2]
        arg[2,4,0,1]=(0.347441329383)*x[0]**o+(-0.96566344519)*x[0]+(-0.0870932691958)*x[1]**o+(-0.793898415692)*x[1]+(0.911757004621)*x[2]**o+(0.623070769788)*x[2]
        arg[2,4,1,0]=(0.732123350436)*x[0]**o+(-0.707615856771)*x[0]+(-0.57969408421)*x[1]**o+(0.0643464469617)*x[1]+(-0.648380091597)*x[2]**o+(0.567100447405)*x[2]
        arg[2,4,1,1]=(-0.128872747083)*x[0]**o+(-0.463336399393)*x[0]+(-0.1679086419)*x[1]**o+(0.602218708561)*x[1]+(0.62788576637)*x[2]**o+(0.223548351914)*x[2]
        arg[2,4,2,0]=(0.619034818105)*x[0]**o+(0.167994824318)*x[0]+(-0.244892750219)*x[1]**o+(0.471338499791)*x[1]+(0.526950200066)*x[2]**o+(0.345611623572)*x[2]
        arg[2,4,2,1]=(0.645979236759)*x[0]**o+(0.79896884484)*x[0]+(-0.587486435644)*x[1]**o+(-0.467838428827)*x[1]+(0.672707985612)*x[2]**o+(-0.169680966884)*x[2]
        arg[3,0,0,0]=(-0.265836015694)*x[0]**o+(-0.383762854285)*x[0]+(0.420375720992)*x[1]**o+(0.739662661302)*x[1]+(-0.593559713248)*x[2]**o+(0.0743208592009)*x[2]
        arg[3,0,0,1]=(0.306246512933)*x[0]**o+(0.0357065903179)*x[0]+(0.213209244461)*x[1]**o+(-0.069836605971)*x[1]+(0.668535341179)*x[2]**o+(0.59870885392)*x[2]
        arg[3,0,1,0]=(0.509420593773)*x[0]**o+(0.995859415429)*x[0]+(-0.07734119959)*x[1]**o+(0.269006992126)*x[1]+(-0.465576902019)*x[2]**o+(0.318766487761)*x[2]
        arg[3,0,1,1]=(-0.609626065482)*x[0]**o+(0.970976299205)*x[0]+(-0.314734168176)*x[1]**o+(-0.536438958228)*x[1]+(0.900096595598)*x[2]**o+(-0.893816431641)*x[2]
        arg[3,0,2,0]=(0.269212290711)*x[0]**o+(-0.119897130167)*x[0]+(-0.937350065698)*x[1]**o+(0.913313769348)*x[1]+(0.311695335026)*x[2]**o+(0.58822688023)*x[2]
        arg[3,0,2,1]=(-0.445116541697)*x[0]**o+(-0.64630078598)*x[0]+(0.811173945666)*x[1]**o+(-0.910503175962)*x[1]+(0.160819266078)*x[2]**o+(-0.807050548571)*x[2]
        arg[3,1,0,0]=(0.25585673506)*x[0]**o+(-0.174293490388)*x[0]+(-0.510750951271)*x[1]**o+(-0.215780338673)*x[1]+(-0.531739311594)*x[2]**o+(0.479569840128)*x[2]
        arg[3,1,0,1]=(-0.159301665223)*x[0]**o+(0.88558738006)*x[0]+(-0.933255168276)*x[1]**o+(0.739621529924)*x[1]+(-0.371501556668)*x[2]**o+(-0.224043643673)*x[2]
        arg[3,1,1,0]=(0.673375237931)*x[0]**o+(0.29315421593)*x[0]+(0.378983835599)*x[1]**o+(-0.187220470447)*x[1]+(-0.122366431969)*x[2]**o+(-0.178182488821)*x[2]
        arg[3,1,1,1]=(0.273992606955)*x[0]**o+(-0.42302685869)*x[0]+(0.490650718142)*x[1]**o+(0.067822229282)*x[1]+(0.758676637186)*x[2]**o+(-0.840062442278)*x[2]
        arg[3,1,2,0]=(-0.533129977534)*x[0]**o+(0.743362501638)*x[0]+(0.0397932337285)*x[1]**o+(0.134261172942)*x[1]+(0.076467475088)*x[2]**o+(0.750124646154)*x[2]
        arg[3,1,2,1]=(0.977824314677)*x[0]**o+(0.322104772509)*x[0]+(0.478366489119)*x[1]**o+(0.139966919282)*x[1]+(0.823631645543)*x[2]**o+(0.29259439301)*x[2]
        arg[3,2,0,0]=(0.969349250234)*x[0]**o+(0.158604629065)*x[0]+(0.517280925739)*x[1]**o+(-0.32151445591)*x[1]+(0.516829269337)*x[2]**o+(0.935139450142)*x[2]
        arg[3,2,0,1]=(-0.673721580725)*x[0]**o+(0.1932869215)*x[0]+(0.47375679317)*x[1]**o+(0.377941367578)*x[1]+(-0.359306451576)*x[2]**o+(0.00498822119998)*x[2]
        arg[3,2,1,0]=(-0.432873699676)*x[0]**o+(0.191782167044)*x[0]+(0.191395777927)*x[1]**o+(-0.540088847941)*x[1]+(0.608451872364)*x[2]**o+(0.151589512705)*x[2]
        arg[3,2,1,1]=(0.633726614581)*x[0]**o+(-0.824133266602)*x[0]+(-0.380289061608)*x[1]**o+(-0.722136850408)*x[1]+(-0.807853184729)*x[2]**o+(0.152552828168)*x[2]
        arg[3,2,2,0]=(-0.182700878265)*x[0]**o+(0.105776705469)*x[0]+(0.837527799957)*x[1]**o+(0.120438395343)*x[1]+(-0.211142277471)*x[2]**o+(-0.0499206664041)*x[2]
        arg[3,2,2,1]=(0.999790796581)*x[0]**o+(0.523238189614)*x[0]+(0.108118557801)*x[1]**o+(-0.899694790145)*x[1]+(0.165089392648)*x[2]**o+(0.38599870475)*x[2]
        arg[3,3,0,0]=(-0.959143897162)*x[0]**o+(-0.968197835338)*x[0]+(-0.521866033091)*x[1]**o+(-0.560799919169)*x[1]+(0.604562365858)*x[2]**o+(0.195593261393)*x[2]
        arg[3,3,0,1]=(0.385513188376)*x[0]**o+(0.918485762628)*x[0]+(0.581126961809)*x[1]**o+(-0.365882492483)*x[1]+(-0.260563773993)*x[2]**o+(0.997016165521)*x[2]
        arg[3,3,1,0]=(0.809269330969)*x[0]**o+(0.472424328796)*x[0]+(-0.920680159926)*x[1]**o+(0.756205962059)*x[1]+(-0.886275742364)*x[2]**o+(-0.800132407145)*x[2]
        arg[3,3,1,1]=(-0.983474390608)*x[0]**o+(-0.436192301851)*x[0]+(0.653210945793)*x[1]**o+(-0.465357783291)*x[1]+(-0.799261076035)*x[2]**o+(-0.349990963152)*x[2]
        arg[3,3,2,0]=(0.710820041642)*x[0]**o+(-0.973768204693)*x[0]+(0.0408403957292)*x[1]**o+(0.61546181161)*x[1]+(0.915939389579)*x[2]**o+(0.634659938301)*x[2]
        arg[3,3,2,1]=(0.633364771689)*x[0]**o+(0.887553239682)*x[0]+(-0.64913373177)*x[1]**o+(0.557465560353)*x[1]+(-0.744072153243)*x[2]**o+(-0.301528624747)*x[2]
        arg[3,4,0,0]=(0.44039719913)*x[0]**o+(-0.0339525806656)*x[0]+(0.574166395893)*x[1]**o+(-0.810441051899)*x[1]+(0.254523432152)*x[2]**o+(-0.798403721953)*x[2]
        arg[3,4,0,1]=(0.253149910502)*x[0]**o+(-0.909846068796)*x[0]+(0.855107984234)*x[1]**o+(-0.973196168972)*x[1]+(0.646832394105)*x[2]**o+(0.811892995874)*x[2]
        arg[3,4,1,0]=(0.13027869782)*x[0]**o+(-0.424094152568)*x[0]+(-0.197473699021)*x[1]**o+(0.0795845331856)*x[1]+(-0.511301398362)*x[2]**o+(-0.331330168767)*x[2]
        arg[3,4,1,1]=(0.476554240918)*x[0]**o+(0.18800431479)*x[0]+(0.259031626216)*x[1]**o+(-0.0532533771273)*x[1]+(0.914847196458)*x[2]**o+(0.333851370508)*x[2]
        arg[3,4,2,0]=(0.800058524963)*x[0]**o+(-0.402028928896)*x[0]+(0.913881556051)*x[1]**o+(0.895516442873)*x[1]+(-0.0227700921994)*x[2]**o+(-0.0502319834692)*x[2]
        arg[3,4,2,1]=(-0.533834741079)*x[0]**o+(0.557954226232)*x[0]+(-0.121676666641)*x[1]**o+(0.781782385789)*x[1]+(0.0682870158781)*x[2]**o+(-0.236432552686)*x[2]
        ref[0,0,0,0]=(1.46703352572)/(o+1.)+(-1.33520045704)+(0.0737627367843)*0.5**o
        ref[0,0,0,1]=(0.213535197837)/(o+1.)+(-0.325019128805)+(0.353968533247)*0.5**o
        ref[0,0,1,0]=(-0.233213543591)/(o+1.)+(0.450075189439)+(0.315704665816)*0.5**o
        ref[0,0,1,1]=(0.809431837761)/(o+1.)+(0.642450689879)+(-0.453912049362)*0.5**o
        ref[0,0,2,0]=(1.8822564746)/(o+1.)+(-0.160827113662)+(0.705225432895)*0.5**o
        ref[0,0,2,1]=(-0.851077895823)/(o+1.)+(-0.032586006392)+(-0.397162258761)*0.5**o
        ref[0,1,0,0]=(-1.75021683351)/(o+1.)+(0.282579703111)+(0.00444300957662)*0.5**o
        ref[0,1,0,1]=(-1.61043513823)/(o+1.)+(0.209579442759)+(0.726535641072)*0.5**o
        ref[0,1,1,0]=(0.550951959927)/(o+1.)+(-0.453625179799)+(-0.554579794515)*0.5**o
        ref[0,1,1,1]=(-0.809122195368)/(o+1.)+(0.609569171964)+(0.294744137189)*0.5**o
        ref[0,1,2,0]=(1.3771144222)/(o+1.)+(-0.703069822248)+(-0.30261020867)*0.5**o
        ref[0,1,2,1]=(1.08629781546)/(o+1.)+(0.0271166180983)+(0.494855964972)*0.5**o
        ref[0,2,0,0]=(-1.26922850072)/(o+1.)+(0.72637380105)+(-0.914585215232)*0.5**o
        ref[0,2,0,1]=(-0.556637167762)/(o+1.)+(-0.036494329576)+(-0.260590397572)*0.5**o
        ref[0,2,1,0]=(0.707349561895)/(o+1.)+(0.208292082952)+(0.0455797513404)*0.5**o
        ref[0,2,1,1]=(-1.47922090453)/(o+1.)+(-0.621962103251)+(-0.300900136798)*0.5**o
        ref[0,2,2,0]=(0.01827281852)/(o+1.)+(-0.159177319679)+(0.745711773093)*0.5**o
        ref[0,2,2,1]=(1.09869785336)/(o+1.)+(0.352958766464)+(0.291340077334)*0.5**o
        ref[0,3,0,0]=(-0.802486910351)/(o+1.)+(0.741067349535)+(0.89242099254)*0.5**o
        ref[0,3,0,1]=(0.187447770797)/(o+1.)+(0.0211789816838)+(0.095390666498)*0.5**o
        ref[0,3,1,0]=(1.0518488657)/(o+1.)+(-0.0260861096315)+(-0.317750910722)*0.5**o
        ref[0,3,1,1]=(1.16078906983)/(o+1.)+(0.0263064747213)+(-0.204455448392)*0.5**o
        ref[0,3,2,0]=(-0.0986989221708)/(o+1.)+(0.182363177629)+(0.38581656579)*0.5**o
        ref[0,3,2,1]=(-0.388150366913)/(o+1.)+(0.557428570571)+(-0.225571343976)*0.5**o
        ref[0,4,0,0]=(-0.623448360222)/(o+1.)+(0.64524557416)+(0.834083057621)*0.5**o
        ref[0,4,0,1]=(-0.639580493935)/(o+1.)+(-0.380689110373)+(-0.613152578007)*0.5**o
        ref[0,4,1,0]=(-0.575892151257)/(o+1.)+(-0.142294010946)+(-0.373668614153)*0.5**o
        ref[0,4,1,1]=(-0.0316225271882)/(o+1.)+(-0.115844822465)+(0.00943802199128)*0.5**o
        ref[0,4,2,0]=(-0.902549384411)/(o+1.)+(0.210416030442)+(0.157099024354)*0.5**o
        ref[0,4,2,1]=(1.2254583864)/(o+1.)+(-0.371769146738)+(0.933762543076)*0.5**o
        ref[1,0,0,0]=(0.0440432191149)/(o+1.)+(-0.303248360069)+(0.0417476479889)*0.5**o
        ref[1,0,0,1]=(-0.196875527758)/(o+1.)+(-0.39869023332)+(0.450986977446)*0.5**o
        ref[1,0,1,0]=(-0.742874955412)/(o+1.)+(-0.677197937069)+(-0.23712795738)*0.5**o
        ref[1,0,1,1]=(0.385960155907)/(o+1.)+(0.0647632575546)+(-0.308721170682)*0.5**o
        ref[1,0,2,0]=(1.11246238908)/(o+1.)+(0.0111904605859)+(-0.267232319593)*0.5**o
        ref[1,0,2,1]=(0.0798238708014)/(o+1.)+(-0.355425426066)+(0.405254374432)*0.5**o
        ref[1,1,0,0]=(0.133162474633)/(o+1.)+(-0.863343187001)+(-0.877224972777)*0.5**o
        ref[1,1,0,1]=(1.23844759518)/(o+1.)+(0.871424091405)+(-0.873928172379)*0.5**o
        ref[1,1,1,0]=(-0.0275104699246)/(o+1.)+(-0.739054056468)+(0.552826305913)*0.5**o
        ref[1,1,1,1]=(0.948010887116)/(o+1.)+(-0.0503040657995)+(-0.71233607947)*0.5**o
        ref[1,1,2,0]=(1.01678266341)/(o+1.)+(-0.425240837237)+(-0.411667274338)*0.5**o
        ref[1,1,2,1]=(1.62045693811)/(o+1.)+(0.23057679995)+(0.533005373575)*0.5**o
        ref[1,2,0,0]=(0.44625545933)/(o+1.)+(0.0233472080645)+(-0.637366803484)*0.5**o
        ref[1,2,0,1]=(0.277223414052)/(o+1.)+(0.189254359042)+(-0.404870452835)*0.5**o
        ref[1,2,1,0]=(0.880141841398)/(o+1.)+(-0.750127627837)+(0.85574176569)*0.5**o
        ref[1,2,1,1]=(0.0718593507895)/(o+1.)+(0.306599537674)+(0.576825533647)*0.5**o
        ref[1,2,2,0]=(0.35649607288)/(o+1.)+(0.242884247745)+(0.741411326194)*0.5**o
        ref[1,2,2,1]=(1.14623459997)/(o+1.)+(-0.897591123484)+(0.386515069261)*0.5**o
        ref[1,3,0,0]=(-0.796578230199)/(o+1.)+(0.200648623687)+(0.848952965744)*0.5**o
        ref[1,3,0,1]=(0.679746997945)/(o+1.)+(-0.0297741837061)+(0.357870838328)*0.5**o
        ref[1,3,1,0]=(0.0713725201916)/(o+1.)+(0.223521474626)+(-0.509333127149)*0.5**o
        ref[1,3,1,1]=(-1.07672806067)/(o+1.)+(0.402304847309)+(-0.837885550386)*0.5**o
        ref[1,3,2,0]=(0.808945160636)/(o+1.)+(0.525847284807)+(0.205889276236)*0.5**o
        ref[1,3,2,1]=(0.440475889238)/(o+1.)+(0.353047235363)+(-0.331660060588)*0.5**o
        ref[1,4,0,0]=(0.733653130588)/(o+1.)+(-0.113784463751)+(0.394201404247)*0.5**o
        ref[1,4,0,1]=(0.892965597874)/(o+1.)+(0.359236861297)+(-0.346144268453)*0.5**o
        ref[1,4,1,0]=(0.512178276429)/(o+1.)+(0.0722586978563)+(-0.907566140749)*0.5**o
        ref[1,4,1,1]=(-1.21754003377)/(o+1.)+(0.369986403191)+(0.726194147182)*0.5**o
        ref[1,4,2,0]=(0.353974376932)/(o+1.)+(0.291102967642)+(-0.940824546553)*0.5**o
        ref[1,4,2,1]=(0.166197160477)/(o+1.)+(-0.358856317187)+(0.728499572603)*0.5**o
        ref[2,0,0,0]=(1.55742864501)/(o+1.)+(0.38703652972)+(0.467025642879)*0.5**o
        ref[2,0,0,1]=(-0.867288120579)/(o+1.)+(-0.995570381672)+(-0.211129222294)*0.5**o
        ref[2,0,1,0]=(0.0896432549354)/(o+1.)+(-0.894461032367)+(-0.668065295797)*0.5**o
        ref[2,0,1,1]=(-0.240985203598)/(o+1.)+(-0.407009494534)+(0.928751918298)*0.5**o
        ref[2,0,2,0]=(0.815866062908)/(o+1.)+(-0.281559110938)+(-0.531818105049)*0.5**o
        ref[2,0,2,1]=(0.0143868099026)/(o+1.)+(0.253341199154)+(0.0150988200159)*0.5**o
        ref[2,1,0,0]=(0.745834270286)/(o+1.)+(-0.630128050767)+(-0.0829154151121)*0.5**o
        ref[2,1,0,1]=(-0.513757920038)/(o+1.)+(-0.505622196077)+(-0.834502772339)*0.5**o
        ref[2,1,1,0]=(1.38226406947)/(o+1.)+(-0.763221422703)+(0.247439924718)*0.5**o
        ref[2,1,1,1]=(-0.408281199784)/(o+1.)+(-0.811328833758)+(-0.120169534728)*0.5**o
        ref[2,1,2,0]=(-0.586214858819)/(o+1.)+(0.11721515075)+(0.703841647274)*0.5**o
        ref[2,1,2,1]=(0.0812921429368)/(o+1.)+(-0.357151740094)+(-0.19760048131)*0.5**o
        ref[2,2,0,0]=(1.65849097715)/(o+1.)+(-0.777022017999)+(-0.507872937196)*0.5**o
        ref[2,2,0,1]=(0.242698257917)/(o+1.)+(1.08992054229)+(-0.481403986338)*0.5**o
        ref[2,2,1,0]=(-0.754639453272)/(o+1.)+(0.81286785715)+(0.241675645258)*0.5**o
        ref[2,2,1,1]=(-0.438727562295)/(o+1.)+(0.217076971211)+(0.114772627763)*0.5**o
        ref[2,2,2,0]=(0.683917726753)/(o+1.)+(-0.781863136791)+(0.887023209688)*0.5**o
        ref[2,2,2,1]=(1.19838469898)/(o+1.)+(0.729462151403)+(0.836731324893)*0.5**o
        ref[2,3,0,0]=(1.28206834695)/(o+1.)+(0.583576731084)+(0.196161020545)*0.5**o
        ref[2,3,0,1]=(0.395309853285)/(o+1.)+(0.573307319258)+(-0.862326385376)*0.5**o
        ref[2,3,1,0]=(1.34760990633)/(o+1.)+(0.348652101929)+(0.571263839862)*0.5**o
        ref[2,3,1,1]=(0.293682752388)/(o+1.)+(0.080258501461)+(0.272559456412)*0.5**o
        ref[2,3,2,0]=(-1.47056068107)/(o+1.)+(-0.237909711446)+(0.3409363126)*0.5**o
        ref[2,3,2,1]=(-0.356052894952)/(o+1.)+(-0.166196236302)+(0.949170673623)*0.5**o
        ref[2,4,0,0]=(-0.21887693849)/(o+1.)+(0.365445238731)+(0.876774885889)*0.5**o
        ref[2,4,0,1]=(0.824663735425)/(o+1.)+(-0.568245545547)+(0.347441329383)*0.5**o
        ref[2,4,1,0]=(-1.22807417581)/(o+1.)+(-0.0380844812022)+(0.732123350436)*0.5**o
        ref[2,4,1,1]=(0.45997712447)/(o+1.)+(0.181215330541)+(-0.128872747083)*0.5**o
        ref[2,4,2,0]=(0.282057449847)/(o+1.)+(0.49247247384)+(0.619034818105)*0.5**o
        ref[2,4,2,1]=(0.0852215499679)/(o+1.)+(0.0807247245641)+(0.645979236759)*0.5**o
        ref[3,0,0,0]=(-0.173183992256)/(o+1.)+(0.215110333109)+(-0.265836015694)*0.5**o
        ref[3,0,0,1]=(0.88174458564)/(o+1.)+(0.282289419133)+(0.306246512933)*0.5**o
        ref[3,0,1,0]=(-0.542918101609)/(o+1.)+(0.791816447658)+(0.509420593773)*0.5**o
        ref[3,0,1,1]=(0.585362427423)/(o+1.)+(-0.229639545331)+(-0.609626065482)*0.5**o
        ref[3,0,2,0]=(-0.625654730672)/(o+1.)+(0.690821759706)+(0.269212290711)*0.5**o
        ref[3,0,2,1]=(0.971993211744)/(o+1.)+(-1.18192725526)+(-0.445116541697)*0.5**o
        ref[3,1,0,0]=(-1.04249026287)/(o+1.)+(0.0447480055334)+(0.25585673506)*0.5**o
        ref[3,1,0,1]=(-1.30475672494)/(o+1.)+(0.700582633156)+(-0.159301665223)*0.5**o
        ref[3,1,1,0]=(0.25661740363)/(o+1.)+(-0.036124371669)+(0.673375237931)*0.5**o
        ref[3,1,1,1]=(1.24932735533)/(o+1.)+(-0.597633535843)+(0.273992606955)*0.5**o
        ref[3,1,2,0]=(0.116260708817)/(o+1.)+(0.813874160367)+(-0.533129977534)*0.5**o
        ref[3,1,2,1]=(1.30199813466)/(o+1.)+(0.377333042401)+(0.977824314677)*0.5**o
        ref[3,2,0,0]=(1.03411019508)/(o+1.)+(0.386114811649)+(0.969349250234)*0.5**o
        ref[3,2,0,1]=(0.114450341594)/(o+1.)+(0.288108255139)+(-0.673721580725)*0.5**o
        ref[3,2,1,0]=(0.799847650292)/(o+1.)+(-0.0983585840958)+(-0.432873699676)*0.5**o
        ref[3,2,1,1]=(-1.18814224634)/(o+1.)+(-0.696858644421)+(0.633726614581)*0.5**o
        ref[3,2,2,0]=(0.626385522485)/(o+1.)+(0.0881472172038)+(-0.182700878265)*0.5**o
        ref[3,2,2,1]=(0.27320795045)/(o+1.)+(0.0047710521094)+(0.999790796581)*0.5**o
        ref[3,3,0,0]=(0.0826963327674)/(o+1.)+(-0.666702246557)+(-0.959143897162)*0.5**o
        ref[3,3,0,1]=(0.320563187816)/(o+1.)+(0.774809717833)+(0.385513188376)*0.5**o
        ref[3,3,1,0]=(-1.80695590229)/(o+1.)+(0.214248941855)+(0.809269330969)*0.5**o
        ref[3,3,1,1]=(-0.146050130242)/(o+1.)+(-0.625770524147)+(-0.983474390608)*0.5**o
        ref[3,3,2,0]=(0.956779785308)/(o+1.)+(0.138176772609)+(0.710820041642)*0.5**o
        ref[3,3,2,1]=(-1.39320588501)/(o+1.)+(0.571745087644)+(0.633364771689)*0.5**o
        ref[3,4,0,0]=(0.828689828045)/(o+1.)+(-0.821398677259)+(0.44039719913)*0.5**o
        ref[3,4,0,1]=(1.50194037834)/(o+1.)+(-0.535574620947)+(0.253149910502)*0.5**o
        ref[3,4,1,0]=(-0.708775097383)/(o+1.)+(-0.337919894074)+(0.13027869782)*0.5**o
        ref[3,4,1,1]=(1.17387882267)/(o+1.)+(0.234301154085)+(0.476554240918)*0.5**o
        ref[3,4,2,0]=(0.891111463851)/(o+1.)+(0.221627765254)+(0.800058524963)*0.5**o
        ref[3,4,2,1]=(-0.0533896507631)/(o+1.)+(0.551652029667)+(-0.533834741079)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedSolution_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.570707549273)*x[0]+(-0.726264032186)*x[1]
        ref=(-0.0423765208873)/(o+1.)+(-0.791542656673)+(0.328490252775)*0.5**o
      else:
        arg=(1.28222892931)*x[0]+(-0.798170675576)*x[1]+(1.5701208405)*x[2]
        ref=(0.721975372868)/(o+1.)+(0.344720905832)+(0.642761909705)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedSolution_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.575166906798)*x[0]+(-1.63698773418)*x[1]
        arg[1]=(-0.0535946584636)*x[0]+(0.105969101775)*x[1]
        ref[0]=(-0.751926426344)/(o+1.)+(-0.395800138888)+(0.481705876734)*0.5**o
        ref[1]=(0.698496252242)/(o+1.)+(-0.265927161866)+(-0.114267485199)*0.5**o
      else:
        arg[0]=(-0.17879254138)*x[0]+(-1.12752123792)*x[1]+(0.921297311605)*x[2]
        arg[1]=(1.33131824618)*x[0]+(-0.392175059176)*x[1]+(0.545754281465)*x[2]
        ref[0]=(-0.494870303061)/(o+1.)+(0.0697636237654)+(-0.0296734121631)*0.5**o
        ref[1]=(-0.394300983704)/(o+1.)+(0.722261312754)+(0.434675826665)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedSolution_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.338658191734)*x[0]+(1.46127609683)*x[1]
        arg[0,1]=(-0.629116867491)*x[0]+(1.40141569066)*x[1]
        arg[0,2]=(0.105569779347)*x[0]+(0.239936324967)*x[1]
        arg[0,3]=(-0.281718331718)*x[0]+(0.280323404866)*x[1]
        arg[0,4]=(0.467341113661)*x[0]+(-0.50452062075)*x[1]
        arg[1,0]=(-0.841295219186)*x[0]+(0.524038163723)*x[1]
        arg[1,1]=(1.40143704236)*x[0]+(-1.50467961637)*x[1]
        arg[1,2]=(1.21078790505)*x[0]+(-0.731453470292)*x[1]
        arg[1,3]=(0.469556922076)*x[0]+(0.0686293204391)*x[1]
        arg[1,4]=(-1.26394637585)*x[0]+(-0.0361395276585)*x[1]
        arg[2,0]=(-1.12850883148)*x[0]+(0.284296673326)*x[1]
        arg[2,1]=(-0.626362828813)*x[0]+(1.19459827573)*x[1]
        arg[2,2]=(-0.545457293083)*x[0]+(1.547337019)*x[1]
        arg[2,3]=(0.664461986629)*x[0]+(-1.69082198266)*x[1]
        arg[2,4]=(-0.100004763305)*x[0]+(0.152159044423)*x[1]
        arg[3,0]=(0.307574843061)*x[0]+(1.20209181301)*x[1]
        arg[3,1]=(-0.0663203356627)*x[0]+(-1.26669863527)*x[1]
        arg[3,2]=(-1.20436798894)*x[0]+(-0.0375693379661)*x[1]
        arg[3,3]=(0.753649087747)*x[0]+(0.69376223486)*x[1]
        arg[3,4]=(-0.0512732264834)*x[0]+(-0.658097606436)*x[1]
        ref[0,0]=(0.996821439952)/(o+1.)+(0.015088160779)+(0.0956201435879)*0.5**o
        ref[0,1]=(0.643092963726)/(o+1.)+(0.239390724844)+(-0.349575590244)*0.5**o
        ref[0,2]=(-0.198468008977)/(o+1.)+(0.00411201743342)+(0.535750078425)*0.5**o
        ref[0,3]=(0.642491201137)/(o+1.)+(-0.623413199652)+(0.602940271315)*0.5**o
        ref[0,4]=(-0.105706120549)/(o+1.)+(-0.353560054939)+(0.775646723338)*0.5**o
        ref[1,0]=(0.144540669221)/(o+1.)+(0.0228268215803)+(-0.507451367845)*0.5**o
        ref[1,1]=(-0.820379339897)/(o+1.)+(0.0951825678509)+(0.526771630185)*0.5**o
        ref[1,2]=(-0.21214585824)/(o+1.)+(-0.0889183062985)+(0.869316905591)*0.5**o
        ref[1,3]=(0.291228940674)/(o+1.)+(0.192623928703)+(-0.138290555565)*0.5**o
        ref[1,4]=(-0.0663718956434)/(o+1.)+(-0.309484335614)+(-0.614745336642)*0.5**o
        ref[2,0]=(0.484892962201)/(o+1.)+(-0.367129859793)+(-0.594845400763)*0.5**o
        ref[2,1]=(0.768457634672)/(o+1.)+(0.154447657562)+(-0.509117502876)*0.5**o
        ref[2,2]=(0.707320338363)/(o+1.)+(-0.0592149232183)+(0.412989233992)*0.5**o
        ref[2,3]=(-0.983684339171)/(o+1.)+(-0.495377645598)+(0.948079634337)*0.5**o
        ref[2,4]=(-0.0315947220465)/(o+1.)+(0.506833986759)+(-0.929918970353)*0.5**o
        ref[3,0]=(0.54141067098)/(o+1.)+(0.489049131493)+(-0.00984227789859)*0.5**o
        ref[3,1]=(-0.552335053051)/(o+1.)+(-0.383325638803)+(-0.0140326402734)*0.5**o
        ref[3,2]=(0.0821472412575)/(o+1.)+(-0.335827719718)+(-0.652429128733)*0.5**o
        ref[3,3]=(0.112917208924)/(o+1.)+(0.667772807338)+(-0.00105150099312)*0.5**o
        ref[3,4]=(-0.19331508157)/(o+1.)+(-0.102296385708)+(-0.311462979932)*0.5**o
      else:
        arg[0,0]=(0.069245963142)*x[0]+(-0.185727379494)*x[1]+(-0.887410889699)*x[2]
        arg[0,1]=(-0.816745668949)*x[0]+(-0.250435680662)*x[1]+(0.701607830582)*x[2]
        arg[0,2]=(-0.862433954308)*x[0]+(0.731874372527)*x[1]+(-0.115384982779)*x[2]
        arg[0,3]=(-0.265247631604)*x[0]+(-0.157059741958)*x[1]+(-0.368419335078)*x[2]
        arg[0,4]=(1.66720203567)*x[0]+(-0.759393708719)*x[1]+(1.36360850704)*x[2]
        arg[1,0]=(-0.192822813795)*x[0]+(0.0358776152527)*x[1]+(-0.722864520788)*x[2]
        arg[1,1]=(-0.480606215471)*x[0]+(1.13682774573)*x[1]+(0.17900467623)*x[2]
        arg[1,2]=(0.346760852584)*x[0]+(0.994833030281)*x[1]+(0.507055933913)*x[2]
        arg[1,3]=(-0.509961853478)*x[0]+(-0.852997575505)*x[1]+(0.586527030253)*x[2]
        arg[1,4]=(0.93957975846)*x[0]+(0.28878335152)*x[1]+(0.128070931738)*x[2]
        arg[2,0]=(0.338283306313)*x[0]+(0.0244500761608)*x[1]+(-0.606875554888)*x[2]
        arg[2,1]=(0.850782624552)*x[0]+(-0.836382038189)*x[1]+(-1.41808459947)*x[2]
        arg[2,2]=(-1.06176229613)*x[0]+(-0.259247046061)*x[1]+(-0.583455484089)*x[2]
        arg[2,3]=(1.17437923282)*x[0]+(1.31485601495)*x[1]+(1.3139743231)*x[2]
        arg[2,4]=(0.362851636275)*x[0]+(0.804049783156)*x[1]+(1.09205000214)*x[2]
        arg[3,0]=(0.343603837195)*x[0]+(-1.49082257981)*x[1]+(-0.628668674545)*x[2]
        arg[3,1]=(-0.676847711765)*x[0]+(1.32350560453)*x[1]+(1.07275185686)*x[2]
        arg[3,2]=(0.570784985469)*x[0]+(-1.01212360602)*x[1]+(-0.0808776667606)*x[2]
        arg[3,3]=(0.506392444236)*x[0]+(-1.00848318041)*x[1]+(0.558777901541)*x[2]
        arg[3,4]=(-0.60925738291)*x[0]+(-0.386755957809)*x[1]+(0.60930115434)*x[2]
        ref[0,0]=(-0.695230136419)/(o+1.)+(-0.388400729083)+(0.468139288533)*0.5**o
        ref[0,1]=(0.462098132187)/(o+1.)+(-0.215041893287)+(-0.397587864642)*0.5**o
        ref[0,2]=(-0.262411385363)/(o+1.)+(0.275729226874)+(-0.534991632946)*0.5**o
        ref[0,3]=(0.117822633378)/(o+1.)+(0.0286827897592)+(-0.965914921536)*0.5**o
        ref[0,4]=(0.293575681039)/(o+1.)+(0.584039692827)+(0.809761767299)*0.5**o
        ref[1,0]=(0.627951922463)/(o+1.)+(-0.994889162557)+(0.482016683321)*0.5**o
        ref[1,1]=(1.83482238662)/(o+1.)+(-0.205719558739)+(-0.588157062647)*0.5**o
        ref[1,2]=(1.13612020398)/(o+1.)+(0.0262824938134)+(0.659964625174)*0.5**o
        ref[1,3]=(0.325264487741)/(o+1.)+(-0.0752154813413)+(-0.951265923787)*0.5**o
        ref[1,4]=(-0.149919283302)/(o+1.)+(0.77701502585)+(-0.0476767266789)*0.5**o
        ref[2,0]=(0.491504686171)/(o+1.)+(-0.562534706926)+(0.389422555265)*0.5**o
        ref[2,1]=(-1.52704217417)/(o+1.)+(0.120525042098)+(-0.117691923134)*0.5**o
        ref[2,2]=(-0.779796772543)/(o+1.)+(-0.318864997267)+(-0.486938059201)*0.5**o
        ref[2,3]=(1.53343318682)/(o+1.)+(0.731149467308)+(0.807477449443)*0.5**o
        ref[2,4]=(1.15919207694)/(o+1.)+(0.113921448815)+(0.871916446996)*0.5**o
        ref[3,0]=(-1.75312920944)/(o+1.)+(-0.153798982308)+(0.284839756895)*0.5**o
        ref[3,1]=(1.30985814569)/(o+1.)+(0.438430874237)+(-0.467310144534)*0.5**o
        ref[3,2]=(-0.737480320972)/(o+1.)+(0.0604403094397)+(0.094383414777)*0.5**o
        ref[3,3]=(-0.688723266524)/(o+1.)+(0.43486027018)+(-0.124310108473)*0.5**o
        ref[3,4]=(0.128399262674)/(o+1.)+(-0.372445844122)+(0.229780239192)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedSolution_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.993339371954)*x[0]+(-1.35308128985)*x[1]
        arg[0,0,1]=(-1.07886149134)*x[0]+(-1.38999283206)*x[1]
        arg[0,1,0]=(0.246468026972)*x[0]+(-1.37217464882)*x[1]
        arg[0,1,1]=(0.645672648599)*x[0]+(1.8070624103)*x[1]
        arg[1,0,0]=(-1.02278989388)*x[0]+(0.151634023103)*x[1]
        arg[1,0,1]=(-0.667332860541)*x[0]+(0.312742149193)*x[1]
        arg[1,1,0]=(-0.848647385832)*x[0]+(0.466438038516)*x[1]
        arg[1,1,1]=(-0.185495911372)*x[0]+(-0.137118786891)*x[1]
        arg[2,0,0]=(-0.0250555870731)*x[0]+(0.905553573283)*x[1]
        arg[2,0,1]=(-1.03940852939)*x[0]+(-1.57326016394)*x[1]
        arg[2,1,0]=(1.48834718811)*x[0]+(-0.340889253246)*x[1]
        arg[2,1,1]=(0.875721514739)*x[0]+(-0.310857412212)*x[1]
        arg[3,0,0]=(1.25129508831)*x[0]+(-1.69312043525)*x[1]
        arg[3,0,1]=(1.50124675312)*x[0]+(0.244195748654)*x[1]
        arg[3,1,0]=(0.292060478909)*x[0]+(-0.660005422696)*x[1]
        arg[3,1,1]=(0.791335231308)*x[0]+(0.466481573591)*x[1]
        arg[4,0,0]=(-0.361241947236)*x[0]+(-1.43591648587)*x[1]
        arg[4,0,1]=(-1.10385456695)*x[0]+(-0.325314195825)*x[1]
        arg[4,1,0]=(1.01004173033)*x[0]+(0.426824552857)*x[1]
        arg[4,1,1]=(-0.631382479612)*x[0]+(0.0603792190966)*x[1]
        arg[5,0,0]=(-0.30404368786)*x[0]+(-0.0102498522932)*x[1]
        arg[5,0,1]=(0.109188848746)*x[0]+(-1.16846160105)*x[1]
        arg[5,1,0]=(-0.351916879547)*x[0]+(-0.320714795448)*x[1]
        arg[5,1,1]=(-0.534138562652)*x[0]+(-0.256013589844)*x[1]
        ref[0,0,0]=(-0.506159375456)/(o+1.)+(-0.522895228783)+(-0.794470828783)*0.5**o
        ref[0,0,1]=(-0.930984367068)/(o+1.)+(-0.636851258685)+(-0.264167438968)*0.5**o
        ref[0,1,0]=(-0.674222792172)/(o+1.)+(-0.52131379048)+(0.591143751282)*0.5**o
        ref[0,1,1]=(0.953370751283)/(o+1.)+(0.377773224645)+(0.743817858326)*0.5**o
        ref[1,0,0]=(0.700508215428)/(o+1.)+(-0.303343530867)+(-0.964977024471)*0.5**o
        ref[1,0,1]=(-0.408185116452)/(o+1.)+(0.0668983569743)+(-0.0802023088445)*0.5**o
        ref[1,1,0]=(0.80742345527)/(o+1.)+(-0.405780705683)+(-0.37807139122)*0.5**o
        ref[1,1,1]=(0.0184615568095)/(o+1.)+(0.221921562266)+(-0.784919379605)*0.5**o
        ref[2,0,0]=(0.571906256883)/(o+1.)+(-0.0415588382133)+(0.391709405753)*0.5**o
        ref[2,0,1]=(-0.778176009451)/(o+1.)+(-0.761680990566)+(-0.311130702752)*0.5**o
        ref[2,1,0]=(0.498214686321)/(o+1.)+(-0.146173705483)+(0.941590659509)*0.5**o
        ref[2,1,1]=(-0.273747071818)/(o+1.)+(-0.0319184747153)+(0.902448123776)*0.5**o
        ref[3,0,0]=(-0.730029471178)/(o+1.)+(-0.288672747933)+(0.865549620109)*0.5**o
        ref[3,0,1]=(-0.371409186475)/(o+1.)+(0.623169375401)+(0.870512937445)*0.5**o
        ref[3,1,0]=(0.0611454391867)/(o+1.)+(-0.627723874214)+(0.826357365454)*0.5**o
        ref[3,1,1]=(0.785553447935)/(o+1.)+(0.246940213405)+(-0.021617069845)*0.5**o
        ref[4,0,0]=(-0.936201955907)/(o+1.)+(-0.484823720021)+(0.108690962842)*0.5**o
        ref[4,0,1]=(-0.277830390815)/(o+1.)+(-0.151786547071)+(-0.847765277819)*0.5**o
        ref[4,1,0]=(0.12293099533)/(o+1.)+(0.295620382054)+(0.722694523749)*0.5**o
        ref[4,1,1]=(-0.5517019185)/(o+1.)+(0.0850748937088)+(-0.189451129433)*0.5**o
        ref[5,0,0]=(-0.942533148805)/(o+1.)+(0.425680504872)+(-0.223121401092)*0.5**o
        ref[5,0,1]=(-0.377134228332)/(o+1.)+(-0.721019409569)+(0.759900295162)*0.5**o
        ref[5,1,0]=(-0.643828391942)/(o+1.)+(0.399974456311)+(-0.828752195674)*0.5**o
        ref[5,1,1]=(-0.118293731439)/(o+1.)+(-0.355568544431)+(0.0392786678054)*0.5**o
      else:
        arg[0,0,0]=(0.205446621454)*x[0]+(0.10254878667)*x[1]+(1.27673728979)*x[2]
        arg[0,0,1]=(0.758846313193)*x[0]+(0.63320336713)*x[1]+(-0.42638505807)*x[2]
        arg[0,1,0]=(-0.0127116561603)*x[0]+(-0.360359420027)*x[1]+(0.540508591493)*x[2]
        arg[0,1,1]=(-0.278918629538)*x[0]+(-0.0291534835772)*x[1]+(0.708655731565)*x[2]
        arg[1,0,0]=(-0.612965210483)*x[0]+(0.787787556912)*x[1]+(0.222384448462)*x[2]
        arg[1,0,1]=(1.07409083742)*x[0]+(1.19429173706)*x[1]+(-0.484979976779)*x[2]
        arg[1,1,0]=(-1.25714999157)*x[0]+(-0.321346019622)*x[1]+(0.278329320837)*x[2]
        arg[1,1,1]=(-0.512262689707)*x[0]+(1.6891992053)*x[1]+(-1.02351830068)*x[2]
        arg[2,0,0]=(-1.57911907773)*x[0]+(0.096124258122)*x[1]+(0.336619253521)*x[2]
        arg[2,0,1]=(0.0901144254199)*x[0]+(-1.36818186255)*x[1]+(1.2969609239)*x[2]
        arg[2,1,0]=(0.158329511499)*x[0]+(0.696997185217)*x[1]+(-0.400535235927)*x[2]
        arg[2,1,1]=(-1.44570853201)*x[0]+(0.237359587195)*x[1]+(-1.17323024801)*x[2]
        arg[3,0,0]=(-1.4325812824)*x[0]+(-0.733085265902)*x[1]+(0.125883186484)*x[2]
        arg[3,0,1]=(-0.0529889021435)*x[0]+(0.335154600942)*x[1]+(0.839337375209)*x[2]
        arg[3,1,0]=(-0.423552182963)*x[0]+(1.31813289315)*x[1]+(0.13235854915)*x[2]
        arg[3,1,1]=(0.53278572217)*x[0]+(0.159434073071)*x[1]+(-0.313621199136)*x[2]
        arg[4,0,0]=(0.747779887673)*x[0]+(0.30977483911)*x[1]+(-0.0760295896739)*x[2]
        arg[4,0,1]=(0.00295118909194)*x[0]+(-0.687914867842)*x[1]+(0.0386268422915)*x[2]
        arg[4,1,0]=(-0.276363532754)*x[0]+(0.52102466785)*x[1]+(-0.684840191917)*x[2]
        arg[4,1,1]=(0.0283138898538)*x[0]+(0.116042458294)*x[1]+(0.485578401528)*x[2]
        arg[5,0,0]=(-0.118402060192)*x[0]+(-1.18409809944)*x[1]+(-0.304048852342)*x[2]
        arg[5,0,1]=(0.174393790186)*x[0]+(-0.524912689263)*x[1]+(-0.648708392355)*x[2]
        arg[5,1,0]=(-0.254958756344)*x[0]+(-0.772071348728)*x[1]+(-0.0572494064027)*x[2]
        arg[5,1,1]=(-0.74777072018)*x[0]+(0.19109939364)*x[1]+(0.498973987814)*x[2]
        ref[0,0,0]=(0.986457444152)/(o+1.)+(0.258994989412)+(0.0802852749346)*0.5**o
        ref[0,0,1]=(-0.0286717143415)/(o+1.)+(0.554867138611)+(-0.115397940627)*0.5**o
        ref[0,1,0]=(-0.886482131138)/(o+1.)+(0.464293129749)+(0.125333386946)*0.5**o
        ref[0,1,1]=(0.199214164457)/(o+1.)+(0.336882900218)+(-0.472396346443)*0.5**o
        ref[1,0,0]=(0.36021680954)/(o+1.)+(0.408903924703)+(-0.780817864055)*0.5**o
        ref[1,0,1]=(0.44795581074)/(o+1.)+(0.472159267742)+(0.391128251477)*0.5**o
        ref[1,1,0]=(1.02665734893)/(o+1.)+(-0.71418625728)+(-0.898451524719)*0.5**o
        ref[1,1,1]=(0.0322676607777)/(o+1.)+(0.105378958385)+(-0.0896073626306)*0.5**o
        ref[2,0,0]=(1.15723247637)/(o+1.)+(-0.719604538702)+(-0.86439896505)*0.5**o
        ref[2,0,1]=(-0.254100181576)/(o+1.)+(-0.0387997401585)+(0.35059314866)*0.5**o
        ref[2,1,0]=(-0.636439167982)/(o+1.)+(0.563390681981)+(-0.0355507351918)*0.5**o
        ref[2,1,1]=(-0.265265225142)/(o+1.)+(-0.635620798949)+(-0.845072369786)*0.5**o
        ref[3,0,0]=(-0.062162800164)/(o+1.)+(-0.554499928282)+(-0.868620705087)*0.5**o
        ref[3,0,1]=(0.784681961813)/(o+1.)+(0.355202655131)+(-0.373584198067)*0.5**o
        ref[3,1,0]=(1.20272043429)/(o+1.)+(0.352677694103)+(-0.881136563162)*0.5**o
        ref[3,1,1]=(-0.311152797156)/(o+1.)+(0.313736563953)+(0.0622782653549)*0.5**o
        ref[4,0,0]=(-0.365517620238)/(o+1.)+(0.733502174253)+(-0.119961591158)*0.5**o
        ref[4,0,1]=(0.0155845896619)/(o+1.)+(-0.654963543839)+(0.648005661558)*0.5**o
        ref[4,1,0]=(0.405697975915)/(o+1.)+(-0.637805133954)+(0.429733235172)*0.5**o
        ref[4,1,1]=(-0.0469738326054)/(o+1.)+(0.0590721074184)+(0.558764367445)*0.5**o
        ref[5,0,0]=(-0.333825007954)/(o+1.)+(-0.46448520882)+(-0.343753586377)*0.5**o
        ref[5,0,1]=(-0.469125954795)/(o+1.)+(-0.147192309255)+(-0.235716718127)*0.5**o
        ref[5,1,0]=(-0.573275891777)/(o+1.)+(-0.478328163121)+(0.445652706545)*0.5**o
        ref[5,1,1]=(0.0545515234726)/(o+1.)+(-0.153919681346)+(0.195590500493)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedSolution_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.226142137728)*x[0]+(0.85681895183)*x[1]
        arg[0,0,0,1]=(-0.824407181762)*x[0]+(-0.290023144892)*x[1]
        arg[0,0,1,0]=(-1.25987816017)*x[0]+(0.824340990595)*x[1]
        arg[0,0,1,1]=(-0.0917811606765)*x[0]+(0.660211932964)*x[1]
        arg[0,0,2,0]=(-1.31073149178)*x[0]+(0.482324984918)*x[1]
        arg[0,0,2,1]=(0.256995458227)*x[0]+(0.79244254852)*x[1]
        arg[0,1,0,0]=(-0.0682147475407)*x[0]+(0.243620052061)*x[1]
        arg[0,1,0,1]=(0.071099982687)*x[0]+(-0.62986754897)*x[1]
        arg[0,1,1,0]=(-0.0716793733131)*x[0]+(-1.15235525814)*x[1]
        arg[0,1,1,1]=(0.193107575216)*x[0]+(-0.596149957543)*x[1]
        arg[0,1,2,0]=(-0.160142629886)*x[0]+(1.36581707067)*x[1]
        arg[0,1,2,1]=(-1.21477648376)*x[0]+(-0.732000298369)*x[1]
        arg[0,2,0,0]=(-1.02468971059)*x[0]+(0.330721595217)*x[1]
        arg[0,2,0,1]=(1.11414907391)*x[0]+(1.29386127812)*x[1]
        arg[0,2,1,0]=(1.1485850154)*x[0]+(0.387585977495)*x[1]
        arg[0,2,1,1]=(0.237382583072)*x[0]+(1.56248221279)*x[1]
        arg[0,2,2,0]=(-0.949129049229)*x[0]+(0.578823379063)*x[1]
        arg[0,2,2,1]=(-1.0667182537)*x[0]+(0.462227485959)*x[1]
        arg[0,3,0,0]=(1.37954377554)*x[0]+(0.541856880139)*x[1]
        arg[0,3,0,1]=(0.401584700319)*x[0]+(0.577102327147)*x[1]
        arg[0,3,1,0]=(-0.289796408108)*x[0]+(0.807853785325)*x[1]
        arg[0,3,1,1]=(0.0038467861822)*x[0]+(-0.931311510771)*x[1]
        arg[0,3,2,0]=(0.859036254821)*x[0]+(0.198884218516)*x[1]
        arg[0,3,2,1]=(-0.00310256070968)*x[0]+(0.511838475968)*x[1]
        arg[0,4,0,0]=(0.539686886802)*x[0]+(-0.324612924267)*x[1]
        arg[0,4,0,1]=(-0.0458655898565)*x[0]+(0.211810380966)*x[1]
        arg[0,4,1,0]=(0.302333533003)*x[0]+(0.0663033316831)*x[1]
        arg[0,4,1,1]=(-0.981519161639)*x[0]+(-0.00207756909914)*x[1]
        arg[0,4,2,0]=(0.283659447004)*x[0]+(-0.231526592979)*x[1]
        arg[0,4,2,1]=(-1.17679167875)*x[0]+(0.285698985756)*x[1]
        arg[1,0,0,0]=(-0.162900059582)*x[0]+(-0.877164340783)*x[1]
        arg[1,0,0,1]=(-0.50468605065)*x[0]+(-0.160440073709)*x[1]
        arg[1,0,1,0]=(-0.100901676644)*x[0]+(-0.296111526029)*x[1]
        arg[1,0,1,1]=(0.176107421869)*x[0]+(1.74393708745)*x[1]
        arg[1,0,2,0]=(-0.979807721703)*x[0]+(-0.229400841067)*x[1]
        arg[1,0,2,1]=(0.217472795645)*x[0]+(0.0881503243323)*x[1]
        arg[1,1,0,0]=(-0.127980123625)*x[0]+(0.0932095456541)*x[1]
        arg[1,1,0,1]=(0.829674410931)*x[0]+(-0.796196041601)*x[1]
        arg[1,1,1,0]=(1.28656110233)*x[0]+(-1.27850471392)*x[1]
        arg[1,1,1,1]=(1.159269884)*x[0]+(-0.973135433631)*x[1]
        arg[1,1,2,0]=(1.46274837856)*x[0]+(-1.81554270863)*x[1]
        arg[1,1,2,1]=(0.0973472511688)*x[0]+(0.551451261495)*x[1]
        arg[1,2,0,0]=(0.510667671293)*x[0]+(-0.51075982099)*x[1]
        arg[1,2,0,1]=(-1.41612093451)*x[0]+(0.614417669629)*x[1]
        arg[1,2,1,0]=(1.46595125796)*x[0]+(0.60374279122)*x[1]
        arg[1,2,1,1]=(-0.317185218187)*x[0]+(-0.0785532882696)*x[1]
        arg[1,2,2,0]=(-0.388504041993)*x[0]+(-0.838003398775)*x[1]
        arg[1,2,2,1]=(0.0497207129012)*x[0]+(-0.622155692247)*x[1]
        arg[1,3,0,0]=(-0.0471400179563)*x[0]+(1.3378948207)*x[1]
        arg[1,3,0,1]=(-1.22250822195)*x[0]+(0.629714783138)*x[1]
        arg[1,3,1,0]=(-1.28801080408)*x[0]+(-1.54079710038)*x[1]
        arg[1,3,1,1]=(-0.426412448469)*x[0]+(-1.09410474021)*x[1]
        arg[1,3,2,0]=(0.022834167351)*x[0]+(-0.497423389788)*x[1]
        arg[1,3,2,1]=(-0.787492124301)*x[0]+(-0.245621168419)*x[1]
        arg[1,4,0,0]=(-1.37814880919)*x[0]+(-0.704805848451)*x[1]
        arg[1,4,0,1]=(-1.12494770055)*x[0]+(1.44920024305)*x[1]
        arg[1,4,1,0]=(-0.809532572761)*x[0]+(1.16638587657)*x[1]
        arg[1,4,1,1]=(0.931784486816)*x[0]+(1.34629889453)*x[1]
        arg[1,4,2,0]=(-0.172430942793)*x[0]+(-0.584736998321)*x[1]
        arg[1,4,2,1]=(0.00542948736389)*x[0]+(0.43917597284)*x[1]
        arg[2,0,0,0]=(-0.946122337406)*x[0]+(-0.161940168137)*x[1]
        arg[2,0,0,1]=(-0.7856653779)*x[0]+(-0.75287701788)*x[1]
        arg[2,0,1,0]=(-1.08858367592)*x[0]+(-0.530380818694)*x[1]
        arg[2,0,1,1]=(-1.0573555095)*x[0]+(-0.854148765785)*x[1]
        arg[2,0,2,0]=(0.723412825328)*x[0]+(0.326761950437)*x[1]
        arg[2,0,2,1]=(-0.0143142161225)*x[0]+(-0.752724311876)*x[1]
        arg[2,1,0,0]=(-0.942548092355)*x[0]+(0.700861094897)*x[1]
        arg[2,1,0,1]=(-0.712286567957)*x[0]+(0.230406545738)*x[1]
        arg[2,1,1,0]=(1.27414839774)*x[0]+(0.724784649037)*x[1]
        arg[2,1,1,1]=(0.637494163002)*x[0]+(0.442489599192)*x[1]
        arg[2,1,2,0]=(1.81617631726)*x[0]+(1.56870218809)*x[1]
        arg[2,1,2,1]=(1.16419900666)*x[0]+(-0.044314729141)*x[1]
        arg[2,2,0,0]=(0.880267125196)*x[0]+(0.77124406259)*x[1]
        arg[2,2,0,1]=(0.482402867542)*x[0]+(0.0746924305017)*x[1]
        arg[2,2,1,0]=(0.105861259696)*x[0]+(0.484292207267)*x[1]
        arg[2,2,1,1]=(-0.107785838196)*x[0]+(0.701132243209)*x[1]
        arg[2,2,2,0]=(0.559717405844)*x[0]+(-1.5585942717)*x[1]
        arg[2,2,2,1]=(-0.989130712605)*x[0]+(0.221361137556)*x[1]
        arg[2,3,0,0]=(-0.045471818095)*x[0]+(-0.437941959792)*x[1]
        arg[2,3,0,1]=(1.25840468379)*x[0]+(-1.18872011057)*x[1]
        arg[2,3,1,0]=(1.70665829408)*x[0]+(0.705273921003)*x[1]
        arg[2,3,1,1]=(-0.0928709253845)*x[0]+(0.785115449756)*x[1]
        arg[2,3,2,0]=(-0.887243418838)*x[0]+(1.06790660642)*x[1]
        arg[2,3,2,1]=(-0.572644686872)*x[0]+(-0.672831948394)*x[1]
        arg[2,4,0,0]=(1.28282566801)*x[0]+(0.651985199447)*x[1]
        arg[2,4,0,1]=(0.0965663229335)*x[0]+(1.33819998052)*x[1]
        arg[2,4,1,0]=(0.205569041098)*x[0]+(-0.873095358733)*x[1]
        arg[2,4,1,1]=(-1.04452025745)*x[0]+(-0.204917033907)*x[1]
        arg[2,4,2,0]=(-0.181163938972)*x[0]+(0.0341993409272)*x[1]
        arg[2,4,2,1]=(1.68327027059)*x[0]+(-0.289436355642)*x[1]
        arg[3,0,0,0]=(1.05712810731)*x[0]+(0.915711819735)*x[1]
        arg[3,0,0,1]=(-0.171072039482)*x[0]+(-0.160656943623)*x[1]
        arg[3,0,1,0]=(-1.27063799869)*x[0]+(-0.605131642813)*x[1]
        arg[3,0,1,1]=(0.940512990835)*x[0]+(-0.116675577531)*x[1]
        arg[3,0,2,0]=(-0.194213869952)*x[0]+(0.364066752025)*x[1]
        arg[3,0,2,1]=(-0.328780465397)*x[0]+(-0.114689413673)*x[1]
        arg[3,1,0,0]=(-0.825914305895)*x[0]+(-0.167010056731)*x[1]
        arg[3,1,0,1]=(0.657779056207)*x[0]+(0.0666252451638)*x[1]
        arg[3,1,1,0]=(-0.464004426036)*x[0]+(-0.340932918409)*x[1]
        arg[3,1,1,1]=(-1.6698652667)*x[0]+(0.731870373141)*x[1]
        arg[3,1,2,0]=(1.39626250779)*x[0]+(-0.879152855557)*x[1]
        arg[3,1,2,1]=(1.34973919508)*x[0]+(-0.129753332761)*x[1]
        arg[3,2,0,0]=(1.03443450706)*x[0]+(0.807333477811)*x[1]
        arg[3,2,0,1]=(0.366011302596)*x[0]+(0.543804800926)*x[1]
        arg[3,2,1,0]=(0.262446846036)*x[0]+(-0.247786850991)*x[1]
        arg[3,2,1,1]=(-1.04657091196)*x[0]+(-0.485443426155)*x[1]
        arg[3,2,2,0]=(0.342434327538)*x[0]+(-0.148503585588)*x[1]
        arg[3,2,2,1]=(-0.189833429716)*x[0]+(0.424391874819)*x[1]
        arg[3,3,0,0]=(-0.118348838118)*x[0]+(0.397628264341)*x[1]
        arg[3,3,0,1]=(0.481440510084)*x[0]+(-0.393576221424)*x[1]
        arg[3,3,1,0]=(0.907533552208)*x[0]+(-0.668646549376)*x[1]
        arg[3,3,1,1]=(-1.49731714213)*x[0]+(-1.18710678264)*x[1]
        arg[3,3,2,0]=(-0.460660079207)*x[0]+(1.48078266444)*x[1]
        arg[3,3,2,1]=(-0.161073556988)*x[0]+(-0.486827042754)*x[1]
        arg[3,4,0,0]=(-0.668755455623)*x[0]+(0.345804059434)*x[1]
        arg[3,4,0,1]=(1.00528889193)*x[0]+(0.978507928191)*x[1]
        arg[3,4,1,0]=(0.98955856331)*x[0]+(1.0421852144)*x[1]
        arg[3,4,1,1]=(0.711688260873)*x[0]+(1.39962222322)*x[1]
        arg[3,4,2,0]=(1.08512429209)*x[0]+(-0.555249097743)*x[1]
        arg[3,4,2,1]=(-1.10443784007)*x[0]+(-0.782274429169)*x[1]
        ref[0,0,0,0]=(0.151153165564)/(o+1.)+(0.112609656739)+(0.254304335059)*0.5**o
        ref[0,0,0,1]=(-0.530602583511)/(o+1.)+(0.1817855378)+(-0.947398818743)*0.5**o
        ref[0,0,1,0]=(0.308747437738)/(o+1.)+(-0.0338129255643)+(-0.676658756184)*0.5**o
        ref[0,0,1,1]=(0.321026553427)/(o+1.)+(0.0692705367618)+(0.108863145337)*0.5**o
        ref[0,0,2,0]=(-0.0908034666655)/(o+1.)+(-0.1100205155)+(-0.517562009199)*0.5**o
        ref[0,0,2,1]=(0.303456040903)/(o+1.)+(0.500090875336)+(-0.254199784829)*0.5**o
        ref[0,1,0,0]=(-0.440641221831)/(o+1.)+(0.456218946873)+(-0.296391367396)*0.5**o
        ref[0,1,0,1]=(0.222628561762)/(o+1.)+(-0.162064412854)+(-0.457267302337)*0.5**o
        ref[0,1,1,0]=(-0.913361887408)/(o+1.)+(-0.356168674239)+(0.401664604431)*0.5**o
        ref[0,1,1,1]=(-0.543116639248)/(o+1.)+(-0.120902439422)+(0.381879135767)*0.5**o
        ref[0,1,2,0]=(0.993649262831)/(o+1.)+(0.384695858206)+(-0.55736653846)*0.5**o
        ref[0,1,2,1]=(-0.968077797451)/(o+1.)+(-0.187765942637)+(-0.603167099406)*0.5**o
        ref[0,2,0,0]=(0.873662019918)/(o+1.)+(-0.710635360033)+(-0.146359415223)*0.5**o
        ref[0,2,0,1]=(0.466408230693)/(o+1.)+(0.77626565285)+(0.389070815631)*0.5**o
        ref[0,2,1,0]=(-0.475010813579)/(o+1.)+(0.525435880006)+(0.960310046461)*0.5**o
        ref[0,2,1,1]=(0.697127520484)/(o+1.)+(0.357204276552)+(0.388328722275)*0.5**o
        ref[0,2,2,0]=(-0.108434738487)/(o+1.)+(0.356443300466)+(-0.97475753261)*0.5**o
        ref[0,2,2,1]=(0.952274964304)/(o+1.)+(-0.497379508976)+(-0.562006714089)*0.5**o
        ref[0,3,0,0]=(-0.0368012761671)/(o+1.)+(0.594000625518)+(0.770200680808)*0.5**o
        ref[0,3,0,1]=(0.450361861836)/(o+1.)+(0.182031052848)+(0.164263059934)*0.5**o
        ref[0,3,1,0]=(0.500714741956)/(o+1.)+(0.0479711343168)+(-0.0785996333735)*0.5**o
        ref[0,3,1,1]=(-0.949280938645)/(o+1.)+(0.452441874504)+(-0.88306753495)*0.5**o
        ref[0,3,2,0]=(0.186931712837)/(o+1.)+(0.222335175903)+(0.426318408694)*0.5**o
        ref[0,3,2,1]=(-0.288015428918)/(o+1.)+(0.80948657963)+(-0.822221815085)*0.5**o
        ref[0,4,0,0]=(0.670151184695)/(o+1.)+(-0.415883847013)+(0.376690471866)*0.5**o
        ref[0,4,0,1]=(-0.107106843992)/(o+1.)+(0.555375298435)+(-0.837698961768)*0.5**o
        ref[0,4,1,0]=(-0.299999096802)/(o+1.)+(0.668996782774)+(-0.669357604061)*0.5**o
        ref[0,4,1,1]=(0.778006183401)/(o+1.)+(-0.614473529032)+(-0.532655856076)*0.5**o
        ref[0,4,2,0]=(-0.250068389818)/(o+1.)+(0.156274314197)+(-0.0103473845504)*0.5**o
        ref[0,4,2,1]=(0.902125322216)/(o+1.)+(-0.616424567683)+(-0.560368879846)*0.5**o
        ref[1,0,0,0]=(-0.133690258295)/(o+1.)+(-0.715949360249)+(0.525524578427)*0.5**o
        ref[1,0,0,1]=(-0.788097689216)/(o+1.)+(0.0998405659256)+(-0.076709566994)*0.5**o
        ref[1,0,1,0]=(0.144280886187)/(o+1.)+(0.0775377290314)+(-0.696369546924)*0.5**o
        ref[1,0,1,1]=(0.824172196605)/(o+1.)+(0.561804029444)+(-0.0277357461773)*0.5**o
        ref[1,0,2,0]=(0.422564421802)/(o+1.)+(-0.60719307099)+(-0.417386842591)*0.5**o
        ref[1,0,2,1]=(0.778908672762)/(o+1.)+(0.0163446340765)+(-0.505974820938)*0.5**o
        ref[1,1,0,0]=(0.0574071047387)/(o+1.)+(-0.350514214742)+(0.608850746775)*0.5**o
        ref[1,1,0,1]=(-0.991660961792)/(o+1.)+(0.322084279455)+(0.380970772212)*0.5**o
        ref[1,1,1,0]=(-0.823761120268)/(o+1.)+(-0.0495419092898)+(0.930901327253)*0.5**o
        ref[1,1,1,1]=(-0.384729980569)/(o+1.)+(-0.0392583789163)+(0.649381188769)*0.5**o
        ref[1,1,2,0]=(-0.840365304069)/(o+1.)+(-0.0150512662777)+(0.517673506558)*0.5**o
        ref[1,1,2,1]=(0.555976660846)/(o+1.)+(-0.307265886898)+(0.707353625614)*0.5**o
        ref[1,2,0,0]=(-0.423056705491)/(o+1.)+(0.0559404393793)+(0.311083677036)*0.5**o
        ref[1,2,0,1]=(0.15409797664)/(o+1.)+(-0.0906637067519)+(-0.774473828018)*0.5**o
        ref[1,2,1,0]=(0.796718644088)/(o+1.)+(0.248012077098)+(0.776951250898)*0.5**o
        ref[1,2,1,1]=(-0.342830889263)/(o+1.)+(-0.282162737841)+(0.511417858487)*0.5**o
        ref[1,2,2,0]=(-0.722403016671)/(o+1.)+(-0.367059457954)+(0.230014491812)*0.5**o
        ref[1,2,2,1]=(-0.773685058881)/(o+1.)+(0.412807900451)+(-0.624365721366)*0.5**o
        ref[1,3,0,0]=(0.527481434758)/(o+1.)+(0.36602961117)+(0.031214145649)*0.5**o
        ref[1,3,0,1]=(0.0630927531585)/(o+1.)+(-0.164989701995)+(-0.325906787975)*0.5**o
        ref[1,3,1,0]=(-0.612390590593)/(o+1.)+(-0.78855996274)+(-0.63929738839)*0.5**o
        ref[1,3,1,1]=(-0.962656044265)/(o+1.)+(-0.00576535037253)+(-0.546330443673)*0.5**o
        ref[1,3,2,0]=(-0.947280922008)/(o+1.)+(-0.137449579996)+(0.747590859563)*0.5**o
        ref[1,3,2,1]=(-0.946623026202)/(o+1.)+(0.0787536550213)+(-0.24399757656)*0.5**o
        ref[1,4,0,0]=(-0.174471016943)/(o+1.)+(-0.636500556039)+(-0.635482528623)*0.5**o
        ref[1,4,0,1]=(0.806379391595)/(o+1.)+(-0.00304106577093)+(-0.476044717545)*0.5**o
        ref[1,4,1,0]=(0.729649125134)/(o+1.)+(0.0151477051616)+(-0.403091231646)*0.5**o
        ref[1,4,1,1]=(0.559976454745)/(o+1.)+(0.479295896421)+(0.75951513376)*0.5**o
        ref[1,4,2,0]=(0.272199270818)/(o+1.)+(-0.0929072274332)+(-0.843552757065)*0.5**o
        ref[1,4,2,1]=(-0.552663678377)/(o+1.)+(0.613784745778)+(-0.230300352975)*0.5**o
        ref[2,0,0,0]=(-0.174247908098)/(o+1.)+(-0.108873783282)+(-0.716067030882)*0.5**o
        ref[2,0,0,1]=(-0.300584841532)/(o+1.)+(-0.485363665493)+(-0.267230223262)*0.5**o
        ref[2,0,1,0]=(0.316261805154)/(o+1.)+(-0.523474106451)+(-0.888278086869)*0.5**o
        ref[2,0,1,1]=(-0.699622562861)/(o+1.)+(-0.151089538248)+(-0.909702635929)*0.5**o
        ref[2,0,2,0]=(0.728545639829)/(o+1.)+(0.192083148877)+(-0.0625371618174)*0.5**o
        ref[2,0,2,1]=(-0.22042267239)/(o+1.)+(-0.611974599324)+(0.677333343041)*0.5**o
        ref[2,1,0,0]=(-0.0734883811113)/(o+1.)+(0.36100831022)+(-0.890215236787)*0.5**o
        ref[2,1,0,1]=(0.121656829057)/(o+1.)+(-0.130801635799)+(-0.341933579678)*0.5**o
        ref[2,1,1,0]=(0.221353168629)/(o+1.)+(0.618635673483)+(0.540308531179)*0.5**o
        ref[2,1,1,1]=(0.0152580531273)/(o+1.)+(0.252173415542)+(0.560378877984)*0.5**o
        ref[2,1,2,0]=(0.580319777731)/(o+1.)+(0.90566612137)+(0.993226484879)*0.5**o
        ref[2,1,2,1]=(-0.710250320949)/(o+1.)+(0.742177021871)+(0.345780554724)*0.5**o
        ref[2,2,0,0]=(0.770701563507)/(o+1.)+(0.240823471881)+(0.399162680517)*0.5**o
        ref[2,2,0,1]=(0.76121761761)/(o+1.)+(0.128382656738)+(-0.460887633042)*0.5**o
        ref[2,2,1,0]=(-0.269854962057)/(o+1.)+(0.690854898294)+(-0.521701367567)*0.5**o
        ref[2,2,1,1]=(0.719518595712)/(o+1.)+(0.029596430244)+(-0.185365051187)*0.5**o
        ref[2,2,2,0]=(-0.671157761165)/(o+1.)+(-0.142688285012)+(-0.0423425346644)*0.5**o
        ref[2,2,2,1]=(-0.541156097112)/(o+1.)+(0.0765180918929)+(-0.379649661722)*0.5**o
        ref[2,3,0,0]=(-0.0604347109212)/(o+1.)+(-0.19403702073)+(-0.034905025507)*0.5**o
        ref[2,3,0,1]=(-0.793398507071)/(o+1.)+(0.255681280626)+(0.351720519037)*0.5**o
        ref[2,3,1,0]=(-0.267058413731)/(o+1.)+(0.852211426049)+(0.974567776717)*0.5**o
        ref[2,3,1,1]=(0.228396641207)/(o+1.)+(0.322922896735)+(-0.181997910305)*0.5**o
        ref[2,3,2,0]=(0.480987773937)/(o+1.)+(-0.197709028569)+(0.0950934707835)*0.5**o
        ref[2,3,2,1]=(-0.648934475027)/(o+1.)+(-0.418383297875)+(0.240224435511)*0.5**o
        ref[2,4,0,0]=(0.430020261528)/(o+1.)+(0.53421018733)+(0.436370231269)*0.5**o
        ref[2,4,0,1]=(0.906390660223)/(o+1.)+(-0.056057545856)+(0.640490734948)*0.5**o
        ref[2,4,1,0]=(-0.732099770943)/(o+1.)+(-0.0485027617732)+(0.161578976854)*0.5**o
        ref[2,4,1,1]=(-0.827495680848)/(o+1.)+(0.285044728936)+(-0.992031068384)*0.5**o
        ref[2,4,2,0]=(-0.919905471787)/(o+1.)+(0.656971386264)+(-0.541001898785)*0.5**o
        ref[2,4,2,1]=(0.161975875302)/(o+1.)+(0.272665206557)+(0.686527626533)*0.5**o
        ref[3,0,0,0]=(0.584120963259)/(o+1.)+(0.207930729804)+(0.972857504177)*0.5**o
        ref[3,0,0,1]=(0.315823141491)/(o+1.)+(-0.217561208386)+(-0.212429707823)*0.5**o
        ref[3,0,1,0]=(-0.732294996139)/(o+1.)+(-0.278653143596)+(-0.586168358175)*0.5**o
        ref[3,0,1,1]=(0.481321847225)/(o+1.)+(-0.070080648792)+(0.482676863662)*0.5**o
        ref[3,0,2,0]=(0.616679859076)/(o+1.)+(-0.418784546677)+(0.390742116352)*0.5**o
        ref[3,0,2,1]=(0.866778417448)/(o+1.)+(-0.686585656643)+(0.0629230167692)*0.5**o
        ref[3,1,0,0]=(0.531890054167)/(o+1.)+(-0.516187607633)+(-0.492439201527)*0.5**o
        ref[3,1,0,1]=(-0.512361639413)/(o+1.)+(0.499557182545)+(0.237651575694)*0.5**o
        ref[3,1,1,0]=(-0.67542965405)/(o+1.)+(0.274414249559)+(-0.678336189514)*0.5**o
        ref[3,1,1,1]=(0.371936242507)/(o+1.)+(-0.1636482466)+(-0.982634642862)*0.5**o
        ref[3,1,2,0]=(-0.799408831501)/(o+1.)+(0.458633399549)+(0.399251684639)*0.5**o
        ref[3,1,2,1]=(-0.0145253903686)/(o+1.)+(0.238472345905)+(0.757566560882)*0.5**o
        ref[3,2,0,0]=(-0.130519473112)/(o+1.)+(0.695277604837)+(0.581732248306)*0.5**o
        ref[3,2,0,1]=(0.632010536438)/(o+1.)+(0.123066205529)+(0.0316731560259)*0.5**o
        ref[3,2,1,0]=(-0.709926210356)/(o+1.)+(0.0926699972302)+(0.53924621094)*0.5**o
        ref[3,2,1,1]=(0.428094002247)/(o+1.)+(-0.841997926131)+(-0.276112488096)*0.5**o
        ref[3,2,2,0]=(0.708481148086)/(o+1.)+(-0.50498161534)+(0.495412824545)*0.5**o
        ref[3,2,2,1]=(0.782824622703)/(o+1.)+(-0.151509698687)+(-0.245246780227)*0.5**o
        ref[3,3,0,0]=(0.848460868418)/(o+1.)+(-0.104639248461)+(-0.359902945273)*0.5**o
        ref[3,3,0,1]=(0.0569151307298)/(o+1.)+(-0.0674224535469)+(0.165794065024)*0.5**o
        ref[3,3,1,0]=(-0.165683739908)/(o+1.)+(-0.183388699236)+(0.771348141212)*0.5**o
        ref[3,3,1,1]=(-0.932550058069)/(o+1.)+(-0.47828109001)+(-0.795311686688)*0.5**o
        ref[3,3,2,0]=(0.821542940923)/(o+1.)+(0.166853194487)+(-0.13512674466)*0.5**o
        ref[3,3,2,1]=(-0.46375871628)/(o+1.)+(0.132701267732)+(-0.449544418926)*0.5**o
        ref[3,4,0,0]=(-0.074530542168)/(o+1.)+(-0.0567877963797)+(-0.134845261261)*0.5**o
        ref[3,4,0,1]=(0.238671520479)/(o+1.)+(0.50689253572)+(0.731340228204)*0.5**o
        ref[3,4,1,0]=(0.981601513335)/(o+1.)+(0.400717209822)+(0.248707844726)*0.5**o
        ref[3,4,1,1]=(0.592304390107)/(o+1.)+(0.342408637328)+(0.834188819333)*0.5**o
        ref[3,4,2,0]=(-0.339991330909)/(o+1.)+(-0.0145922376156)+(0.899051000485)*0.5**o
        ref[3,4,2,1]=(-0.626610688164)/(o+1.)+(-0.417008165115)+(-0.426085250846)*0.5**o
      else:
        arg[0,0,0,0]=(0.697666818785)*x[0]+(1.00202295791)*x[1]+(0.617700157184)*x[2]
        arg[0,0,0,1]=(-0.465577819457)*x[0]+(-1.40054693355)*x[1]+(1.16912653295)*x[2]
        arg[0,0,1,0]=(-1.26696500634)*x[0]+(-0.0469756169975)*x[1]+(-0.445591738991)*x[2]
        arg[0,0,1,1]=(-1.61654321709)*x[0]+(-1.69522962467)*x[1]+(-1.06089094851)*x[2]
        arg[0,0,2,0]=(0.229468378506)*x[0]+(0.716813596991)*x[1]+(-0.237703937029)*x[2]
        arg[0,0,2,1]=(-1.12052051905)*x[0]+(0.835000004618)*x[1]+(-0.853004812404)*x[2]
        arg[0,1,0,0]=(-1.75500745705)*x[0]+(0.655055836882)*x[1]+(1.05207750735)*x[2]
        arg[0,1,0,1]=(-0.0746024824253)*x[0]+(-0.421596816958)*x[1]+(-0.375715855989)*x[2]
        arg[0,1,1,0]=(-0.963269684396)*x[0]+(0.431993248836)*x[1]+(-1.63061928649)*x[2]
        arg[0,1,1,1]=(-0.418757265018)*x[0]+(-0.665626168583)*x[1]+(-0.335734432507)*x[2]
        arg[0,1,2,0]=(0.980346404307)*x[0]+(1.45442049802)*x[1]+(-0.833420308612)*x[2]
        arg[0,1,2,1]=(-1.52536017847)*x[0]+(0.99484028681)*x[1]+(0.109754321663)*x[2]
        arg[0,2,0,0]=(1.697179262)*x[0]+(-0.928732802147)*x[1]+(-0.653379513626)*x[2]
        arg[0,2,0,1]=(0.682698902948)*x[0]+(0.829307959583)*x[1]+(-1.0300112276)*x[2]
        arg[0,2,1,0]=(0.026084323266)*x[0]+(1.28865784608)*x[1]+(0.830446580645)*x[2]
        arg[0,2,1,1]=(0.486291506032)*x[0]+(-0.519637742913)*x[1]+(0.81671330629)*x[2]
        arg[0,2,2,0]=(-0.761591624634)*x[0]+(0.370951131011)*x[1]+(-0.256748479312)*x[2]
        arg[0,2,2,1]=(-0.249974326513)*x[0]+(0.267323484635)*x[1]+(-1.09642130905)*x[2]
        arg[0,3,0,0]=(0.752057563255)*x[0]+(-1.2772262175)*x[1]+(-0.264569668983)*x[2]
        arg[0,3,0,1]=(-0.173167796447)*x[0]+(0.31873208356)*x[1]+(-0.599780701112)*x[2]
        arg[0,3,1,0]=(-0.912464014941)*x[0]+(-1.56452517166)*x[1]+(1.37651660499)*x[2]
        arg[0,3,1,1]=(0.302009667854)*x[0]+(0.524213064769)*x[1]+(0.316301522683)*x[2]
        arg[0,3,2,0]=(-1.2940154673)*x[0]+(-0.900061396992)*x[1]+(0.3860029847)*x[2]
        arg[0,3,2,1]=(-0.97354131341)*x[0]+(1.08221883964)*x[1]+(-0.739323076416)*x[2]
        arg[0,4,0,0]=(1.86535317192)*x[0]+(-0.478359589546)*x[1]+(-0.798014592302)*x[2]
        arg[0,4,0,1]=(-1.08518693851)*x[0]+(-0.186048654536)*x[1]+(1.04915573267)*x[2]
        arg[0,4,1,0]=(-1.09692811055)*x[0]+(1.01042920935)*x[1]+(-0.73978089555)*x[2]
        arg[0,4,1,1]=(0.654656964663)*x[0]+(0.279204596126)*x[1]+(0.886217781361)*x[2]
        arg[0,4,2,0]=(-0.54009886291)*x[0]+(0.378037248739)*x[1]+(1.09785568886)*x[2]
        arg[0,4,2,1]=(-1.43995958913)*x[0]+(-0.720157090994)*x[1]+(0.0676362028136)*x[2]
        arg[1,0,0,0]=(1.47735412608)*x[0]+(0.342355546228)*x[1]+(-0.0134502415508)*x[2]
        arg[1,0,0,1]=(-0.419536590369)*x[0]+(-1.61108470511)*x[1]+(1.51281979355)*x[2]
        arg[1,0,1,0]=(1.31765698245)*x[0]+(-0.7499566294)*x[1]+(0.934537503478)*x[2]
        arg[1,0,1,1]=(0.688467914883)*x[0]+(-1.32681280673)*x[1]+(-0.387405705058)*x[2]
        arg[1,0,2,0]=(1.37069587556)*x[0]+(0.540830111773)*x[1]+(0.477311068263)*x[2]
        arg[1,0,2,1]=(-0.572066213761)*x[0]+(-1.11950925494)*x[1]+(1.07939217872)*x[2]
        arg[1,1,0,0]=(0.564997105326)*x[0]+(-0.73130546389)*x[1]+(0.435910937304)*x[2]
        arg[1,1,0,1]=(0.513931739674)*x[0]+(-0.944043404826)*x[1]+(0.014114227468)*x[2]
        arg[1,1,1,0]=(-0.77850597256)*x[0]+(-0.445835136484)*x[1]+(0.157478357479)*x[2]
        arg[1,1,1,1]=(0.752431176866)*x[0]+(0.659759664199)*x[1]+(-0.859515433521)*x[2]
        arg[1,1,2,0]=(0.0693158730911)*x[0]+(-0.820997827204)*x[1]+(-0.714694436829)*x[2]
        arg[1,1,2,1]=(0.604674152847)*x[0]+(0.647261183132)*x[1]+(-1.21981352028)*x[2]
        arg[1,2,0,0]=(0.0997504695984)*x[0]+(-0.271549839083)*x[1]+(-0.573467835942)*x[2]
        arg[1,2,0,1]=(0.01861845581)*x[0]+(0.157410502406)*x[1]+(0.939257868776)*x[2]
        arg[1,2,1,0]=(1.44909894631)*x[0]+(0.116038517547)*x[1]+(-0.851940489233)*x[2]
        arg[1,2,1,1]=(1.34598841007)*x[0]+(1.79163125841)*x[1]+(0.719171044245)*x[2]
        arg[1,2,2,0]=(-0.214737791448)*x[0]+(-1.85810839095)*x[1]+(-0.240776599927)*x[2]
        arg[1,2,2,1]=(-0.574707645928)*x[0]+(-0.664173831956)*x[1]+(0.310262106696)*x[2]
        arg[1,3,0,0]=(0.296213453592)*x[0]+(-0.831112482555)*x[1]+(0.130075739128)*x[2]
        arg[1,3,0,1]=(-0.369510604457)*x[0]+(0.0373909697998)*x[1]+(1.6689129783)*x[2]
        arg[1,3,1,0]=(0.0922435881419)*x[0]+(-0.280519827004)*x[1]+(0.285933839232)*x[2]
        arg[1,3,1,1]=(-0.649253282272)*x[0]+(-0.460065044123)*x[1]+(-0.0711164948063)*x[2]
        arg[1,3,2,0]=(0.793313307581)*x[0]+(-0.0587956351149)*x[1]+(0.810487259739)*x[2]
        arg[1,3,2,1]=(-1.38668597055)*x[0]+(1.51396934198)*x[1]+(-1.13918550255)*x[2]
        arg[1,4,0,0]=(1.40078167737)*x[0]+(-1.61915436345)*x[1]+(0.421710091214)*x[2]
        arg[1,4,0,1]=(-1.37982015877)*x[0]+(1.30607371738)*x[1]+(-0.608888848419)*x[2]
        arg[1,4,1,0]=(-0.558087802805)*x[0]+(-0.214169499512)*x[1]+(-0.766242546503)*x[2]
        arg[1,4,1,1]=(-1.75503436584)*x[0]+(0.183604705451)*x[1]+(-0.148071484576)*x[2]
        arg[1,4,2,0]=(0.537942069142)*x[0]+(-0.00640749928252)*x[1]+(-0.277544629657)*x[2]
        arg[1,4,2,1]=(-0.652645833371)*x[0]+(-1.33439077026)*x[1]+(-0.226308197201)*x[2]
        arg[2,0,0,0]=(1.45646289279)*x[0]+(1.40842127675)*x[1]+(1.75078281495)*x[2]
        arg[2,0,0,1]=(0.867988651995)*x[0]+(1.35061812241)*x[1]+(-1.56520776323)*x[2]
        arg[2,0,1,0]=(0.549686523522)*x[0]+(-0.528085465813)*x[1]+(-0.994978820267)*x[2]
        arg[2,0,1,1]=(0.383595996552)*x[0]+(0.612989859087)*x[1]+(-0.560401247231)*x[2]
        arg[2,0,2,0]=(-0.341259919499)*x[0]+(-0.125491675042)*x[1]+(0.545572340757)*x[2]
        arg[2,0,2,1]=(-0.414269395434)*x[0]+(-0.433969881898)*x[1]+(-1.01820199328)*x[2]
        arg[2,1,0,0]=(-0.113575280389)*x[0]+(0.326916039942)*x[1]+(-0.576072890249)*x[2]
        arg[2,1,0,1]=(-0.296473969661)*x[0]+(0.195675099431)*x[1]+(-1.72378150831)*x[2]
        arg[2,1,1,0]=(-1.31575932758)*x[0]+(-0.122557423522)*x[1]+(0.53435027695)*x[2]
        arg[2,1,1,1]=(0.299435754608)*x[0]+(0.727140413706)*x[1]+(0.848931358475)*x[2]
        arg[2,1,2,0]=(0.2618965198)*x[0]+(-0.461672573339)*x[1]+(0.923592803172)*x[2]
        arg[2,1,2,1]=(-0.326817258523)*x[0]+(-0.0363300805442)*x[1]+(1.82049875918)*x[2]
        arg[2,2,0,0]=(-0.435113320888)*x[0]+(0.0257984353786)*x[1]+(0.287846577657)*x[2]
        arg[2,2,0,1]=(1.39178068279)*x[0]+(0.136781967539)*x[1]+(-1.38161590351)*x[2]
        arg[2,2,1,0]=(-0.887874199356)*x[0]+(-0.476769153654)*x[1]+(-0.618760485829)*x[2]
        arg[2,2,1,1]=(1.58883355098)*x[0]+(-0.889469047555)*x[1]+(-0.678486755843)*x[2]
        arg[2,2,2,0]=(0.237147029365)*x[0]+(-0.509883462401)*x[1]+(0.533163151551)*x[2]
        arg[2,2,2,1]=(0.66513913749)*x[0]+(1.30317030657)*x[1]+(0.804083365617)*x[2]
        arg[2,3,0,0]=(1.01531518403)*x[0]+(1.3753053143)*x[1]+(0.000877284235964)*x[2]
        arg[2,3,0,1]=(-0.40853642543)*x[0]+(0.154794268783)*x[1]+(-1.34306671117)*x[2]
        arg[2,3,1,0]=(0.43065284359)*x[0]+(-0.28965146009)*x[1]+(0.660248828789)*x[2]
        arg[2,3,1,1]=(-1.4334922681)*x[0]+(-0.0348802858609)*x[1]+(0.502051304974)*x[2]
        arg[2,3,2,0]=(-1.17399537487)*x[0]+(0.865787043946)*x[1]+(-0.0105949677478)*x[2]
        arg[2,3,2,1]=(0.63104158794)*x[0]+(0.436196079951)*x[1]+(0.498606623061)*x[2]
        arg[2,4,0,0]=(-1.44410177354)*x[0]+(0.513406326011)*x[1]+(0.699127199447)*x[2]
        arg[2,4,0,1]=(-0.981727023077)*x[0]+(0.484182120695)*x[1]+(0.667175832045)*x[2]
        arg[2,4,1,0]=(-1.3352602588)*x[0]+(0.626269023364)*x[1]+(0.813537646695)*x[2]
        arg[2,4,1,1]=(-0.932486930114)*x[0]+(0.299448807849)*x[1]+(0.195862014797)*x[2]
        arg[2,4,2,0]=(-0.0621489644806)*x[0]+(0.131979664158)*x[1]+(0.442029269778)*x[2]
        arg[2,4,2,1]=(-0.507508608633)*x[0]+(-1.28613226533)*x[1]+(-0.697920444023)*x[2]
        arg[3,0,0,0]=(-0.984942677585)*x[0]+(-1.58913299616)*x[1]+(-0.373026784822)*x[2]
        arg[3,0,0,1]=(0.616666427113)*x[0]+(-0.625162800471)*x[1]+(-0.796307290839)*x[2]
        arg[3,0,1,0]=(-0.48469203683)*x[0]+(-0.0572105166553)*x[1]+(0.809572383581)*x[2]
        arg[3,0,1,1]=(-0.487160309892)*x[0]+(0.77646454341)*x[1]+(-0.41947544573)*x[2]
        arg[3,0,2,0]=(-1.17902121554)*x[0]+(-0.244939532033)*x[1]+(-0.282994359476)*x[2]
        arg[3,0,2,1]=(0.33515909881)*x[0]+(0.146866561705)*x[1]+(1.81486098364)*x[2]
        arg[3,1,0,0]=(0.379977901341)*x[0]+(0.834531701517)*x[1]+(-0.0242027062166)*x[2]
        arg[3,1,0,1]=(0.184318292766)*x[0]+(1.08367911805)*x[1]+(-0.164531741594)*x[2]
        arg[3,1,1,0]=(-1.00967115996)*x[0]+(0.542699640149)*x[1]+(0.707616701259)*x[2]
        arg[3,1,1,1]=(0.856948233811)*x[0]+(-1.2186550297)*x[1]+(-0.034268798159)*x[2]
        arg[3,1,2,0]=(0.259470684477)*x[0]+(-1.03586983732)*x[1]+(-0.157275979341)*x[2]
        arg[3,1,2,1]=(-0.161954890658)*x[0]+(1.65853209885)*x[1]+(0.381113456719)*x[2]
        arg[3,2,0,0]=(-0.437413004681)*x[0]+(-0.730340273929)*x[1]+(-0.687689272494)*x[2]
        arg[3,2,0,1]=(-0.336618164118)*x[0]+(-1.24388666895)*x[1]+(-0.2198689254)*x[2]
        arg[3,2,1,0]=(0.0872793087887)*x[0]+(-0.336554573215)*x[1]+(-0.0597803523262)*x[2]
        arg[3,2,1,1]=(0.192805432018)*x[0]+(0.0540495582064)*x[1]+(0.846658042071)*x[2]
        arg[3,2,2,0]=(0.00794192604414)*x[0]+(-0.33504559053)*x[1]+(0.615701149787)*x[2]
        arg[3,2,2,1]=(0.210870961503)*x[0]+(1.14164095846)*x[1]+(-0.192785742791)*x[2]
        arg[3,3,0,0]=(-0.25895162041)*x[0]+(-0.225800436488)*x[1]+(-0.198007079838)*x[2]
        arg[3,3,0,1]=(-0.150363703231)*x[0]+(1.23870429535)*x[1]+(0.28635881247)*x[2]
        arg[3,3,1,0]=(-0.865980718216)*x[0]+(-0.0849853244335)*x[1]+(-0.292164217189)*x[2]
        arg[3,3,1,1]=(-0.350465968722)*x[0]+(0.09853736936)*x[1]+(-0.476711524665)*x[2]
        arg[3,3,2,0]=(0.311396300263)*x[0]+(-0.92437612034)*x[1]+(-0.757139962928)*x[2]
        arg[3,3,2,1]=(-0.912311919837)*x[0]+(0.267793631099)*x[1]+(0.4917345401)*x[2]
        arg[3,4,0,0]=(0.613662541753)*x[0]+(0.782757341361)*x[1]+(1.04209808158)*x[2]
        arg[3,4,0,1]=(0.492828613997)*x[0]+(1.11979095664)*x[1]+(-0.843545072354)*x[2]
        arg[3,4,1,0]=(-0.51939120556)*x[0]+(0.476004484286)*x[1]+(-0.516311785432)*x[2]
        arg[3,4,1,1]=(1.00423313567)*x[0]+(0.90776890388)*x[1]+(1.34423309497)*x[2]
        arg[3,4,2,0]=(-1.44069600826)*x[0]+(-1.39515290909)*x[1]+(1.19851590247)*x[2]
        arg[3,4,2,1]=(0.391851619877)*x[0]+(0.265529120553)*x[1]+(0.996610410913)*x[2]
        ref[0,0,0,0]=(1.19330263935)/(o+1.)+(0.55394275251)+(0.0162017895041)*0.5**o
        ref[0,0,0,1]=(-0.672351783684)/(o+1.)+(-0.187889622039)+(0.351132807711)*0.5**o
        ref[0,0,1,0]=(-1.27021996264)/(o+1.)+(0.236286530101)+(-0.961885459894)*0.5**o
        ref[0,0,1,1]=(-1.43442517107)/(o+1.)+(-1.00113374859)+(-0.935971122022)*0.5**o
        ref[0,0,2,0]=(-0.434813695127)/(o+1.)+(0.606673209583)+(-0.0699546855712)*0.5**o
        ref[0,0,2,1]=(-0.0434901247948)/(o+1.)+(-0.180911877983)+(-0.733211446073)*0.5**o
        ref[0,1,0,0]=(1.41514302182)/(o+1.)+(-0.247977413325)+(-0.967062307989)*0.5**o
        ref[0,1,0,1]=(0.69128100142)/(o+1.)+(-0.621609882923)+(-0.319976390946)*0.5**o
        ref[0,1,1,0]=(-0.881870429191)/(o+1.)+(-0.654075774082)+(0.0281262553057)*0.5**o
        ref[0,1,1,1]=(-1.27239116086)/(o+1.)+(-0.0854066429492)+(0.0230865806541)*0.5**o
        ref[0,1,2,0]=(-0.222663945351)/(o+1.)+(0.685312598747)+(0.453385341573)*0.5**o
        ref[0,1,2,1]=(0.101409038706)/(o+1.)+(0.151367946744)+(-0.824910502191)*0.5**o
        ref[0,2,0,0]=(-1.19509443892)/(o+1.)+(0.225437856866)+(0.85928567141)*0.5**o
        ref[0,2,0,1]=(0.285764347754)/(o+1.)+(-0.383432127133)+(0.963095541446)*0.5**o
        ref[0,2,1,0]=(0.728010227911)/(o+1.)+(0.983790919896)+(-0.550403317717)*0.5**o
        ref[0,2,1,1]=(0.563748259036)/(o+1.)+(0.165164845927)+(-0.110710881481)*0.5**o
        ref[0,2,2,0]=(-0.361502606429)/(o+1.)+(0.0276145482611)+(-0.341115463028)*0.5**o
        ref[0,2,2,1]=(-0.293320207933)/(o+1.)+(0.0336757304385)+(-0.853103403872)*0.5**o
        ref[0,3,0,0]=(-0.410233956527)/(o+1.)+(-0.474281126245)+(0.569057885785)*0.5**o
        ref[0,3,0,1]=(-0.835861970115)/(o+1.)+(0.0173715483175)+(0.346902459481)*0.5**o
        ref[0,3,1,0]=(0.0830466456187)/(o+1.)+(-0.288204681495)+(-0.607109864241)*0.5**o
        ref[0,3,1,1]=(1.07074130823)/(o+1.)+(-0.0437983012867)+(0.159379549652)*0.5**o
        ref[0,3,2,0]=(0.172301049609)/(o+1.)+(-0.491724874152)+(-0.996925180895)*0.5**o
        ref[0,3,2,1]=(-0.24891941221)/(o+1.)+(-0.03935856269)+(-0.3030090126)*0.5**o
        ref[0,4,0,0]=(-0.778575122429)/(o+1.)+(0.213085222078)+(0.941383668349)*0.5**o
        ref[0,4,0,1]=(0.880801627174)/(o+1.)+(-0.454451723401)+(-0.193978040747)*0.5**o
        ref[0,4,1,0]=(-0.835604724882)/(o+1.)+(0.127240834083)+(-0.245156740036)*0.5**o
        ref[0,4,1,1]=(1.62336716324)/(o+1.)+(-0.101085918464)+(0.398884015839)*0.5**o
        ref[0,4,2,0]=(1.16805827702)/(o+1.)+(0.211232591688)+(-0.654729385702)*0.5**o
        ref[0,4,2,1]=(-0.305424138957)/(o+1.)+(-0.431076572693)+(-0.924903192969)*0.5**o
        ref[1,0,0,0]=(-0.481596247529)/(o+1.)+(0.769902632559)+(0.748050413171)*0.5**o
        ref[1,0,0,1]=(0.177388661252)/(o+1.)+(-0.118766442613)+(-0.457657277952)*0.5**o
        ref[1,0,1,0]=(0.709890860372)/(o+1.)+(0.0668010230727)+(0.658744950012)*0.5**o
        ref[1,0,1,1]=(-1.45685781381)/(o+1.)+(-0.178674426089)+(0.788456069079)*0.5**o
        ref[1,0,2,0]=(0.194388185612)/(o+1.)+(0.649177543063)+(0.896093783855)*0.5**o
        ref[1,0,2,1]=(0.181561279507)/(o+1.)+(-0.200618291902)+(-0.392507985678)*0.5**o
        ref[1,1,0,0]=(-0.884376814027)/(o+1.)+(0.280382736159)+(0.593213920451)*0.5**o
        ref[1,1,0,1]=(-0.250709602218)/(o+1.)+(-0.492104211211)+(0.818920586956)*0.5**o
        ref[1,1,1,0]=(0.0529723694625)/(o+1.)+(-0.502047650638)+(-0.115739819752)*0.5**o
        ref[1,1,1,1]=(0.215812202957)/(o+1.)+(-0.308796450292)+(0.954456105171)*0.5**o
        ref[1,1,2,0]=(-0.991481050902)/(o+1.)+(0.162626252838)+(-0.800147845717)*0.5**o
        ref[1,1,2,1]=(-0.433989756951)/(o+1.)+(0.362776336712)+(-0.259441100777)*0.5**o
        ref[1,2,0,0]=(-0.930287052624)/(o+1.)+(-0.347752763108)+(0.880525373413)*0.5**o
        ref[1,2,0,1]=(0.698674852861)/(o+1.)+(-0.15119613196)+(0.719004238051)*0.5**o
        ref[1,2,1,0]=(-0.777216821365)/(o+1.)+(0.436336076297)+(0.617741643399)*0.5**o
        ref[1,2,1,1]=(1.09595223135)/(o+1.)+(0.978251130983)+(0.804336219412)*0.5**o
        ref[1,2,2,0]=(-0.651012630849)/(o+1.)+(-1.04894606664)+(0.43528198181)*0.5**o
        ref[1,2,2,1]=(-1.5227353883)/(o+1.)+(0.670019307557)+(-0.745922598001)*0.5**o
        ref[1,3,0,0]=(-0.750195637997)/(o+1.)+(0.231864245348)+(-0.118356142533)*0.5**o
        ref[1,3,0,1]=(1.17136226596)/(o+1.)+(-0.114976421389)+(0.395383920468)*0.5**o
        ref[1,3,1,0]=(-0.108070636769)/(o+1.)+(0.279145429234)+(-0.352562621327)*0.5**o
        ref[1,3,1,1]=(-0.144223481714)/(o+1.)+(-0.472954352493)+(-0.0903026345013)*0.5**o
        ref[1,3,2,0]=(-0.487564307786)/(o+1.)+(0.636973351449)+(0.758622537094)*0.5**o
        ref[1,3,2,1]=(0.390860965954)/(o+1.)+(-0.36647678165)+(-0.66980953377)*0.5**o
        ref[1,4,0,0]=(-1.04824984837)/(o+1.)+(0.213932231728)+(0.823722790054)*0.5**o
        ref[1,4,0,1]=(-0.0984653601336)/(o+1.)+(0.07867877705)+(-0.741527483773)*0.5**o
        ref[1,4,1,0]=(-0.146287867383)/(o+1.)+(-0.908951868158)+(0.425691754878)*0.5**o
        ref[1,4,1,1]=(-0.94160416554)/(o+1.)+(0.0459877010016)+(-0.869872381431)*0.5**o
        ref[1,4,2,0]=(0.0115742858952)/(o+1.)+(-0.222488399041)+(0.687392452389)*0.5**o
        ref[1,4,2,1]=(-0.956561936575)/(o+1.)+(-0.59770879042)+(-0.0613652834159)*0.5**o
        ref[2,0,0,0]=(1.83987381729)/(o+1.)+(0.956579317963)+(0.862634531274)*0.5**o
        ref[2,0,0,1]=(0.105446127421)/(o+1.)+(0.180661881191)+(0.186629121375)*0.5**o
        ref[2,0,1,0]=(0.0212167440766)/(o+1.)+(-0.337825588888)+(-0.318943328859)*0.5**o
        ref[2,0,1,1]=(0.744834715631)/(o+1.)+(-0.350058416663)+(0.391466726104)*0.5**o
        ref[2,0,2,0]=(-0.788776613856)/(o+1.)+(0.233776806014)+(0.400043748043)*0.5**o
        ref[2,0,2,1]=(-0.95346976307)/(o+1.)+(0.0274998652324)+(-0.967971238003)*0.5**o
        ref[2,1,0,0]=(-0.536051408942)/(o+1.)+(-0.208498016994)+(0.590315312233)*0.5**o
        ref[2,1,0,1]=(-1.04158508788)/(o+1.)+(-0.397885793101)+(0.0127762955396)*0.5**o
        ref[2,1,1,0]=(0.529914207253)/(o+1.)+(-0.547503443987)+(-0.338873793431)*0.5**o
        ref[2,1,1,1]=(0.62298670274)/(o+1.)+(0.449860048786)+(0.352800726478)*0.5**o
        ref[2,1,2,0]=(0.965292650595)/(o+1.)+(0.00980498639)+(-0.261085873742)*0.5**o
        ref[2,1,2,1]=(0.403285360803)/(o+1.)+(0.369160592269)+(0.315744874774)*0.5**o
        ref[2,2,0,0]=(-0.457764884724)/(o+1.)+(0.563657776706)+(-0.791018976541)*0.5**o
        ref[2,2,0,1]=(0.188151274044)/(o+1.)+(-0.476470545911)+(0.911736564588)*0.5**o
        ref[2,2,1,0]=(-0.558515085739)/(o+1.)+(-0.445583796553)+(-0.533721159994)*0.5**o
        ref[2,2,1,1]=(-0.431992337265)/(o+1.)+(-0.0957726221961)+(0.64441532924)*0.5**o
        ref[2,2,2,0]=(-0.159056461947)/(o+1.)+(-0.0326864536156)+(0.484856087693)*0.5**o
        ref[2,2,2,1]=(0.533116456037)/(o+1.)+(1.10197257725)+(0.0353311991352)*0.5**o
        ref[2,3,0,0]=(0.0570625147023)/(o+1.)+(0.74170779854)+(0.851019670777)*0.5**o
        ref[2,3,0,1]=(-0.495059904832)/(o+1.)+(-0.169954912181)+(-0.761839138623)*0.5**o
        ref[2,3,1,0]=(0.482375173314)/(o+1.)+(-0.19374838651)+(0.706371811996)*0.5**o
        ref[2,3,1,1]=(0.12928100721)/(o+1.)+(-0.317327862062)+(-0.460946532077)*0.5**o
        ref[2,3,2,0]=(-0.234362084763)/(o+1.)+(0.423486279181)+(-0.93141377227)*0.5**o
        ref[2,3,2,1]=(1.25586693002)/(o+1.)+(-0.127551329993)+(0.56508002092)*0.5**o
        ref[2,4,0,0]=(0.45235297749)/(o+1.)+(-0.0217537364788)+(-0.640413752615)*0.5**o
        ref[2,4,0,1]=(0.936809207384)/(o+1.)+(-0.151519610695)+(-0.464139056331)*0.5**o
        ref[2,4,1,0]=(0.867693208863)/(o+1.)+(0.105683574314)+(-0.974513946233)*0.5**o
        ref[2,4,1,1]=(-1.27821575101)/(o+1.)+(0.463163847311)+(-0.0852880510814)*0.5**o
        ref[2,4,2,0]=(-0.846224501366)/(o+1.)+(0.432619461409)+(0.492845548002)*0.5**o
        ref[2,4,2,1]=(-1.48923229995)/(o+1.)+(-0.505747396375)+(0.00916577472118)*0.5**o
        ref[3,0,0,0]=(-0.959215731121)/(o+1.)+(-0.904609108587)+(-0.178668510268)*0.5**o
        ref[3,0,0,1]=(-1.53181231865)/(o+1.)+(-0.0758543610408)+(0.878717376536)*0.5**o
        ref[3,0,1,0]=(1.18643238831)/(o+1.)+(-0.288825159097)+(-0.341112240015)*0.5**o
        ref[3,0,1,1]=(-0.0675694917031)/(o+1.)+(-0.168955703616)+(0.275309686724)*0.5**o
        ref[3,0,2,0]=(-0.663269860133)/(o+1.)+(-0.235102421885)+(-0.573480403148)*0.5**o
        ref[3,0,2,1]=(1.0969289778)/(o+1.)+(0.46603172821)+(0.267894209932)*0.5**o
        ref[3,1,0,0]=(-0.716870513559)/(o+1.)+(0.712039409283)+(0.483098591635)*0.5**o
        ref[3,1,0,1]=(0.0328531999411)/(o+1.)+(0.379319450988)+(0.311973567308)*0.5**o
        ref[3,1,1,0]=(1.08648208064)/(o+1.)+(-0.196831959338)+(-0.452172980519)*0.5**o
        ref[3,1,1,1]=(0.0813181502612)/(o+1.)+(-0.227364064527)+(-0.0225656152548)*0.5**o
        ref[3,1,2,0]=(-0.271000454408)/(o+1.)+(-0.765711687155)+(0.868748696531)*0.5**o
        ref[3,1,2,1]=(1.60569646253)/(o+1.)+(0.137437739736)+(-0.00288127709197)*0.5**o
        ref[3,2,0,0]=(-1.25887089059)/(o+1.)+(-0.525842290945)+(0.455112921376)*0.5**o
        ref[3,2,0,1]=(-1.63787880725)/(o+1.)+(0.00214706725405)+(-0.166789085722)*0.5**o
        ref[3,2,1,0]=(-1.82111307894)/(o+1.)+(0.561806270366)+(0.388444921454)*0.5**o
        ref[3,2,1,1]=(-0.489259906414)/(o+1.)+(0.582548398381)+(0.417676141947)*0.5**o
        ref[3,2,2,0]=(-0.0674375340391)/(o+1.)+(-0.00345549227349)+(0.362946003887)*0.5**o
        ref[3,2,2,1]=(0.662780682001)/(o+1.)+(-0.15453848218)+(0.806022459534)*0.5**o
        ref[3,3,0,0]=(-1.15795406515)/(o+1.)+(0.0926725746467)+(0.289849779118)*0.5**o
        ref[3,3,0,1]=(1.27520269547)/(o+1.)+(-0.286529818361)+(0.672556345847)*0.5**o
        ref[3,3,1,0]=(-0.343290651469)/(o+1.)+(-0.0269962100904)+(-0.845847188189)*0.5**o
        ref[3,3,1,1]=(0.497715053371)/(o+1.)+(-0.158577177766)+(-0.909200821866)*0.5**o
        ref[3,3,2,0]=(-0.952431538662)/(o+1.)+(0.0816482063045)+(-0.580984656952)*0.5**o
        ref[3,3,2,1]=(-0.176227469799)/(o+1.)+(0.0456157853688)+(-0.0677878495767)*0.5**o
        ref[3,4,0,0]=(0.976460483802)/(o+1.)+(0.347595506011)+(0.766866468871)*0.5**o
        ref[3,4,0,1]=(0.0358170327662)/(o+1.)+(-0.0991456259778)+(0.931548717473)*0.5**o
        ref[3,4,1,0]=(0.114544318514)/(o+1.)+(-0.201014865792)+(-0.272213093635)*0.5**o
        ref[3,4,1,1]=(0.554509925385)/(o+1.)+(1.23219332557)+(0.237338557994)*0.5**o
        ref[3,4,2,0]=(0.0803465444931)/(o+1.)+(-0.518257569997)+(-0.681164419371)*0.5**o
        ref[3,4,2,1]=(-0.0443609570522)/(o+1.)+(0.572813368096)+(0.552725372202)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.570707549273)*x[0]+(-0.726264032186)*x[1]
        ref=(-0.0423765208873)/(o+1.)+(-0.791542656673)+(0.328490252775)*0.5**o
      else:
        arg=(1.28222892931)*x[0]+(-0.798170675576)*x[1]+(1.5701208405)*x[2]
        ref=(0.721975372868)/(o+1.)+(0.344720905832)+(0.642761909705)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.575166906798)*x[0]+(-1.63698773418)*x[1]
        arg[1]=(-0.0535946584636)*x[0]+(0.105969101775)*x[1]
        ref[0]=(-0.751926426344)/(o+1.)+(-0.395800138888)+(0.481705876734)*0.5**o
        ref[1]=(0.698496252242)/(o+1.)+(-0.265927161866)+(-0.114267485199)*0.5**o
      else:
        arg[0]=(-0.17879254138)*x[0]+(-1.12752123792)*x[1]+(0.921297311605)*x[2]
        arg[1]=(1.33131824618)*x[0]+(-0.392175059176)*x[1]+(0.545754281465)*x[2]
        ref[0]=(-0.494870303061)/(o+1.)+(0.0697636237654)+(-0.0296734121631)*0.5**o
        ref[1]=(-0.394300983704)/(o+1.)+(0.722261312754)+(0.434675826665)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.338658191734)*x[0]+(1.46127609683)*x[1]
        arg[0,1]=(-0.629116867491)*x[0]+(1.40141569066)*x[1]
        arg[0,2]=(0.105569779347)*x[0]+(0.239936324967)*x[1]
        arg[0,3]=(-0.281718331718)*x[0]+(0.280323404866)*x[1]
        arg[0,4]=(0.467341113661)*x[0]+(-0.50452062075)*x[1]
        arg[1,0]=(-0.841295219186)*x[0]+(0.524038163723)*x[1]
        arg[1,1]=(1.40143704236)*x[0]+(-1.50467961637)*x[1]
        arg[1,2]=(1.21078790505)*x[0]+(-0.731453470292)*x[1]
        arg[1,3]=(0.469556922076)*x[0]+(0.0686293204391)*x[1]
        arg[1,4]=(-1.26394637585)*x[0]+(-0.0361395276585)*x[1]
        arg[2,0]=(-1.12850883148)*x[0]+(0.284296673326)*x[1]
        arg[2,1]=(-0.626362828813)*x[0]+(1.19459827573)*x[1]
        arg[2,2]=(-0.545457293083)*x[0]+(1.547337019)*x[1]
        arg[2,3]=(0.664461986629)*x[0]+(-1.69082198266)*x[1]
        arg[2,4]=(-0.100004763305)*x[0]+(0.152159044423)*x[1]
        arg[3,0]=(0.307574843061)*x[0]+(1.20209181301)*x[1]
        arg[3,1]=(-0.0663203356627)*x[0]+(-1.26669863527)*x[1]
        arg[3,2]=(-1.20436798894)*x[0]+(-0.0375693379661)*x[1]
        arg[3,3]=(0.753649087747)*x[0]+(0.69376223486)*x[1]
        arg[3,4]=(-0.0512732264834)*x[0]+(-0.658097606436)*x[1]
        ref[0,0]=(0.996821439952)/(o+1.)+(0.015088160779)+(0.0956201435879)*0.5**o
        ref[0,1]=(0.643092963726)/(o+1.)+(0.239390724844)+(-0.349575590244)*0.5**o
        ref[0,2]=(-0.198468008977)/(o+1.)+(0.00411201743342)+(0.535750078425)*0.5**o
        ref[0,3]=(0.642491201137)/(o+1.)+(-0.623413199652)+(0.602940271315)*0.5**o
        ref[0,4]=(-0.105706120549)/(o+1.)+(-0.353560054939)+(0.775646723338)*0.5**o
        ref[1,0]=(0.144540669221)/(o+1.)+(0.0228268215803)+(-0.507451367845)*0.5**o
        ref[1,1]=(-0.820379339897)/(o+1.)+(0.0951825678509)+(0.526771630185)*0.5**o
        ref[1,2]=(-0.21214585824)/(o+1.)+(-0.0889183062985)+(0.869316905591)*0.5**o
        ref[1,3]=(0.291228940674)/(o+1.)+(0.192623928703)+(-0.138290555565)*0.5**o
        ref[1,4]=(-0.0663718956434)/(o+1.)+(-0.309484335614)+(-0.614745336642)*0.5**o
        ref[2,0]=(0.484892962201)/(o+1.)+(-0.367129859793)+(-0.594845400763)*0.5**o
        ref[2,1]=(0.768457634672)/(o+1.)+(0.154447657562)+(-0.509117502876)*0.5**o
        ref[2,2]=(0.707320338363)/(o+1.)+(-0.0592149232183)+(0.412989233992)*0.5**o
        ref[2,3]=(-0.983684339171)/(o+1.)+(-0.495377645598)+(0.948079634337)*0.5**o
        ref[2,4]=(-0.0315947220465)/(o+1.)+(0.506833986759)+(-0.929918970353)*0.5**o
        ref[3,0]=(0.54141067098)/(o+1.)+(0.489049131493)+(-0.00984227789859)*0.5**o
        ref[3,1]=(-0.552335053051)/(o+1.)+(-0.383325638803)+(-0.0140326402734)*0.5**o
        ref[3,2]=(0.0821472412575)/(o+1.)+(-0.335827719718)+(-0.652429128733)*0.5**o
        ref[3,3]=(0.112917208924)/(o+1.)+(0.667772807338)+(-0.00105150099312)*0.5**o
        ref[3,4]=(-0.19331508157)/(o+1.)+(-0.102296385708)+(-0.311462979932)*0.5**o
      else:
        arg[0,0]=(0.069245963142)*x[0]+(-0.185727379494)*x[1]+(-0.887410889699)*x[2]
        arg[0,1]=(-0.816745668949)*x[0]+(-0.250435680662)*x[1]+(0.701607830582)*x[2]
        arg[0,2]=(-0.862433954308)*x[0]+(0.731874372527)*x[1]+(-0.115384982779)*x[2]
        arg[0,3]=(-0.265247631604)*x[0]+(-0.157059741958)*x[1]+(-0.368419335078)*x[2]
        arg[0,4]=(1.66720203567)*x[0]+(-0.759393708719)*x[1]+(1.36360850704)*x[2]
        arg[1,0]=(-0.192822813795)*x[0]+(0.0358776152527)*x[1]+(-0.722864520788)*x[2]
        arg[1,1]=(-0.480606215471)*x[0]+(1.13682774573)*x[1]+(0.17900467623)*x[2]
        arg[1,2]=(0.346760852584)*x[0]+(0.994833030281)*x[1]+(0.507055933913)*x[2]
        arg[1,3]=(-0.509961853478)*x[0]+(-0.852997575505)*x[1]+(0.586527030253)*x[2]
        arg[1,4]=(0.93957975846)*x[0]+(0.28878335152)*x[1]+(0.128070931738)*x[2]
        arg[2,0]=(0.338283306313)*x[0]+(0.0244500761608)*x[1]+(-0.606875554888)*x[2]
        arg[2,1]=(0.850782624552)*x[0]+(-0.836382038189)*x[1]+(-1.41808459947)*x[2]
        arg[2,2]=(-1.06176229613)*x[0]+(-0.259247046061)*x[1]+(-0.583455484089)*x[2]
        arg[2,3]=(1.17437923282)*x[0]+(1.31485601495)*x[1]+(1.3139743231)*x[2]
        arg[2,4]=(0.362851636275)*x[0]+(0.804049783156)*x[1]+(1.09205000214)*x[2]
        arg[3,0]=(0.343603837195)*x[0]+(-1.49082257981)*x[1]+(-0.628668674545)*x[2]
        arg[3,1]=(-0.676847711765)*x[0]+(1.32350560453)*x[1]+(1.07275185686)*x[2]
        arg[3,2]=(0.570784985469)*x[0]+(-1.01212360602)*x[1]+(-0.0808776667606)*x[2]
        arg[3,3]=(0.506392444236)*x[0]+(-1.00848318041)*x[1]+(0.558777901541)*x[2]
        arg[3,4]=(-0.60925738291)*x[0]+(-0.386755957809)*x[1]+(0.60930115434)*x[2]
        ref[0,0]=(-0.695230136419)/(o+1.)+(-0.388400729083)+(0.468139288533)*0.5**o
        ref[0,1]=(0.462098132187)/(o+1.)+(-0.215041893287)+(-0.397587864642)*0.5**o
        ref[0,2]=(-0.262411385363)/(o+1.)+(0.275729226874)+(-0.534991632946)*0.5**o
        ref[0,3]=(0.117822633378)/(o+1.)+(0.0286827897592)+(-0.965914921536)*0.5**o
        ref[0,4]=(0.293575681039)/(o+1.)+(0.584039692827)+(0.809761767299)*0.5**o
        ref[1,0]=(0.627951922463)/(o+1.)+(-0.994889162557)+(0.482016683321)*0.5**o
        ref[1,1]=(1.83482238662)/(o+1.)+(-0.205719558739)+(-0.588157062647)*0.5**o
        ref[1,2]=(1.13612020398)/(o+1.)+(0.0262824938134)+(0.659964625174)*0.5**o
        ref[1,3]=(0.325264487741)/(o+1.)+(-0.0752154813413)+(-0.951265923787)*0.5**o
        ref[1,4]=(-0.149919283302)/(o+1.)+(0.77701502585)+(-0.0476767266789)*0.5**o
        ref[2,0]=(0.491504686171)/(o+1.)+(-0.562534706926)+(0.389422555265)*0.5**o
        ref[2,1]=(-1.52704217417)/(o+1.)+(0.120525042098)+(-0.117691923134)*0.5**o
        ref[2,2]=(-0.779796772543)/(o+1.)+(-0.318864997267)+(-0.486938059201)*0.5**o
        ref[2,3]=(1.53343318682)/(o+1.)+(0.731149467308)+(0.807477449443)*0.5**o
        ref[2,4]=(1.15919207694)/(o+1.)+(0.113921448815)+(0.871916446996)*0.5**o
        ref[3,0]=(-1.75312920944)/(o+1.)+(-0.153798982308)+(0.284839756895)*0.5**o
        ref[3,1]=(1.30985814569)/(o+1.)+(0.438430874237)+(-0.467310144534)*0.5**o
        ref[3,2]=(-0.737480320972)/(o+1.)+(0.0604403094397)+(0.094383414777)*0.5**o
        ref[3,3]=(-0.688723266524)/(o+1.)+(0.43486027018)+(-0.124310108473)*0.5**o
        ref[3,4]=(0.128399262674)/(o+1.)+(-0.372445844122)+(0.229780239192)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.993339371954)*x[0]+(-1.35308128985)*x[1]
        arg[0,0,1]=(-1.07886149134)*x[0]+(-1.38999283206)*x[1]
        arg[0,1,0]=(0.246468026972)*x[0]+(-1.37217464882)*x[1]
        arg[0,1,1]=(0.645672648599)*x[0]+(1.8070624103)*x[1]
        arg[1,0,0]=(-1.02278989388)*x[0]+(0.151634023103)*x[1]
        arg[1,0,1]=(-0.667332860541)*x[0]+(0.312742149193)*x[1]
        arg[1,1,0]=(-0.848647385832)*x[0]+(0.466438038516)*x[1]
        arg[1,1,1]=(-0.185495911372)*x[0]+(-0.137118786891)*x[1]
        arg[2,0,0]=(-0.0250555870731)*x[0]+(0.905553573283)*x[1]
        arg[2,0,1]=(-1.03940852939)*x[0]+(-1.57326016394)*x[1]
        arg[2,1,0]=(1.48834718811)*x[0]+(-0.340889253246)*x[1]
        arg[2,1,1]=(0.875721514739)*x[0]+(-0.310857412212)*x[1]
        arg[3,0,0]=(1.25129508831)*x[0]+(-1.69312043525)*x[1]
        arg[3,0,1]=(1.50124675312)*x[0]+(0.244195748654)*x[1]
        arg[3,1,0]=(0.292060478909)*x[0]+(-0.660005422696)*x[1]
        arg[3,1,1]=(0.791335231308)*x[0]+(0.466481573591)*x[1]
        arg[4,0,0]=(-0.361241947236)*x[0]+(-1.43591648587)*x[1]
        arg[4,0,1]=(-1.10385456695)*x[0]+(-0.325314195825)*x[1]
        arg[4,1,0]=(1.01004173033)*x[0]+(0.426824552857)*x[1]
        arg[4,1,1]=(-0.631382479612)*x[0]+(0.0603792190966)*x[1]
        arg[5,0,0]=(-0.30404368786)*x[0]+(-0.0102498522932)*x[1]
        arg[5,0,1]=(0.109188848746)*x[0]+(-1.16846160105)*x[1]
        arg[5,1,0]=(-0.351916879547)*x[0]+(-0.320714795448)*x[1]
        arg[5,1,1]=(-0.534138562652)*x[0]+(-0.256013589844)*x[1]
        ref[0,0,0]=(-0.506159375456)/(o+1.)+(-0.522895228783)+(-0.794470828783)*0.5**o
        ref[0,0,1]=(-0.930984367068)/(o+1.)+(-0.636851258685)+(-0.264167438968)*0.5**o
        ref[0,1,0]=(-0.674222792172)/(o+1.)+(-0.52131379048)+(0.591143751282)*0.5**o
        ref[0,1,1]=(0.953370751283)/(o+1.)+(0.377773224645)+(0.743817858326)*0.5**o
        ref[1,0,0]=(0.700508215428)/(o+1.)+(-0.303343530867)+(-0.964977024471)*0.5**o
        ref[1,0,1]=(-0.408185116452)/(o+1.)+(0.0668983569743)+(-0.0802023088445)*0.5**o
        ref[1,1,0]=(0.80742345527)/(o+1.)+(-0.405780705683)+(-0.37807139122)*0.5**o
        ref[1,1,1]=(0.0184615568095)/(o+1.)+(0.221921562266)+(-0.784919379605)*0.5**o
        ref[2,0,0]=(0.571906256883)/(o+1.)+(-0.0415588382133)+(0.391709405753)*0.5**o
        ref[2,0,1]=(-0.778176009451)/(o+1.)+(-0.761680990566)+(-0.311130702752)*0.5**o
        ref[2,1,0]=(0.498214686321)/(o+1.)+(-0.146173705483)+(0.941590659509)*0.5**o
        ref[2,1,1]=(-0.273747071818)/(o+1.)+(-0.0319184747153)+(0.902448123776)*0.5**o
        ref[3,0,0]=(-0.730029471178)/(o+1.)+(-0.288672747933)+(0.865549620109)*0.5**o
        ref[3,0,1]=(-0.371409186475)/(o+1.)+(0.623169375401)+(0.870512937445)*0.5**o
        ref[3,1,0]=(0.0611454391867)/(o+1.)+(-0.627723874214)+(0.826357365454)*0.5**o
        ref[3,1,1]=(0.785553447935)/(o+1.)+(0.246940213405)+(-0.021617069845)*0.5**o
        ref[4,0,0]=(-0.936201955907)/(o+1.)+(-0.484823720021)+(0.108690962842)*0.5**o
        ref[4,0,1]=(-0.277830390815)/(o+1.)+(-0.151786547071)+(-0.847765277819)*0.5**o
        ref[4,1,0]=(0.12293099533)/(o+1.)+(0.295620382054)+(0.722694523749)*0.5**o
        ref[4,1,1]=(-0.5517019185)/(o+1.)+(0.0850748937088)+(-0.189451129433)*0.5**o
        ref[5,0,0]=(-0.942533148805)/(o+1.)+(0.425680504872)+(-0.223121401092)*0.5**o
        ref[5,0,1]=(-0.377134228332)/(o+1.)+(-0.721019409569)+(0.759900295162)*0.5**o
        ref[5,1,0]=(-0.643828391942)/(o+1.)+(0.399974456311)+(-0.828752195674)*0.5**o
        ref[5,1,1]=(-0.118293731439)/(o+1.)+(-0.355568544431)+(0.0392786678054)*0.5**o
      else:
        arg[0,0,0]=(0.205446621454)*x[0]+(0.10254878667)*x[1]+(1.27673728979)*x[2]
        arg[0,0,1]=(0.758846313193)*x[0]+(0.63320336713)*x[1]+(-0.42638505807)*x[2]
        arg[0,1,0]=(-0.0127116561603)*x[0]+(-0.360359420027)*x[1]+(0.540508591493)*x[2]
        arg[0,1,1]=(-0.278918629538)*x[0]+(-0.0291534835772)*x[1]+(0.708655731565)*x[2]
        arg[1,0,0]=(-0.612965210483)*x[0]+(0.787787556912)*x[1]+(0.222384448462)*x[2]
        arg[1,0,1]=(1.07409083742)*x[0]+(1.19429173706)*x[1]+(-0.484979976779)*x[2]
        arg[1,1,0]=(-1.25714999157)*x[0]+(-0.321346019622)*x[1]+(0.278329320837)*x[2]
        arg[1,1,1]=(-0.512262689707)*x[0]+(1.6891992053)*x[1]+(-1.02351830068)*x[2]
        arg[2,0,0]=(-1.57911907773)*x[0]+(0.096124258122)*x[1]+(0.336619253521)*x[2]
        arg[2,0,1]=(0.0901144254199)*x[0]+(-1.36818186255)*x[1]+(1.2969609239)*x[2]
        arg[2,1,0]=(0.158329511499)*x[0]+(0.696997185217)*x[1]+(-0.400535235927)*x[2]
        arg[2,1,1]=(-1.44570853201)*x[0]+(0.237359587195)*x[1]+(-1.17323024801)*x[2]
        arg[3,0,0]=(-1.4325812824)*x[0]+(-0.733085265902)*x[1]+(0.125883186484)*x[2]
        arg[3,0,1]=(-0.0529889021435)*x[0]+(0.335154600942)*x[1]+(0.839337375209)*x[2]
        arg[3,1,0]=(-0.423552182963)*x[0]+(1.31813289315)*x[1]+(0.13235854915)*x[2]
        arg[3,1,1]=(0.53278572217)*x[0]+(0.159434073071)*x[1]+(-0.313621199136)*x[2]
        arg[4,0,0]=(0.747779887673)*x[0]+(0.30977483911)*x[1]+(-0.0760295896739)*x[2]
        arg[4,0,1]=(0.00295118909194)*x[0]+(-0.687914867842)*x[1]+(0.0386268422915)*x[2]
        arg[4,1,0]=(-0.276363532754)*x[0]+(0.52102466785)*x[1]+(-0.684840191917)*x[2]
        arg[4,1,1]=(0.0283138898538)*x[0]+(0.116042458294)*x[1]+(0.485578401528)*x[2]
        arg[5,0,0]=(-0.118402060192)*x[0]+(-1.18409809944)*x[1]+(-0.304048852342)*x[2]
        arg[5,0,1]=(0.174393790186)*x[0]+(-0.524912689263)*x[1]+(-0.648708392355)*x[2]
        arg[5,1,0]=(-0.254958756344)*x[0]+(-0.772071348728)*x[1]+(-0.0572494064027)*x[2]
        arg[5,1,1]=(-0.74777072018)*x[0]+(0.19109939364)*x[1]+(0.498973987814)*x[2]
        ref[0,0,0]=(0.986457444152)/(o+1.)+(0.258994989412)+(0.0802852749346)*0.5**o
        ref[0,0,1]=(-0.0286717143415)/(o+1.)+(0.554867138611)+(-0.115397940627)*0.5**o
        ref[0,1,0]=(-0.886482131138)/(o+1.)+(0.464293129749)+(0.125333386946)*0.5**o
        ref[0,1,1]=(0.199214164457)/(o+1.)+(0.336882900218)+(-0.472396346443)*0.5**o
        ref[1,0,0]=(0.36021680954)/(o+1.)+(0.408903924703)+(-0.780817864055)*0.5**o
        ref[1,0,1]=(0.44795581074)/(o+1.)+(0.472159267742)+(0.391128251477)*0.5**o
        ref[1,1,0]=(1.02665734893)/(o+1.)+(-0.71418625728)+(-0.898451524719)*0.5**o
        ref[1,1,1]=(0.0322676607777)/(o+1.)+(0.105378958385)+(-0.0896073626306)*0.5**o
        ref[2,0,0]=(1.15723247637)/(o+1.)+(-0.719604538702)+(-0.86439896505)*0.5**o
        ref[2,0,1]=(-0.254100181576)/(o+1.)+(-0.0387997401585)+(0.35059314866)*0.5**o
        ref[2,1,0]=(-0.636439167982)/(o+1.)+(0.563390681981)+(-0.0355507351918)*0.5**o
        ref[2,1,1]=(-0.265265225142)/(o+1.)+(-0.635620798949)+(-0.845072369786)*0.5**o
        ref[3,0,0]=(-0.062162800164)/(o+1.)+(-0.554499928282)+(-0.868620705087)*0.5**o
        ref[3,0,1]=(0.784681961813)/(o+1.)+(0.355202655131)+(-0.373584198067)*0.5**o
        ref[3,1,0]=(1.20272043429)/(o+1.)+(0.352677694103)+(-0.881136563162)*0.5**o
        ref[3,1,1]=(-0.311152797156)/(o+1.)+(0.313736563953)+(0.0622782653549)*0.5**o
        ref[4,0,0]=(-0.365517620238)/(o+1.)+(0.733502174253)+(-0.119961591158)*0.5**o
        ref[4,0,1]=(0.0155845896619)/(o+1.)+(-0.654963543839)+(0.648005661558)*0.5**o
        ref[4,1,0]=(0.405697975915)/(o+1.)+(-0.637805133954)+(0.429733235172)*0.5**o
        ref[4,1,1]=(-0.0469738326054)/(o+1.)+(0.0590721074184)+(0.558764367445)*0.5**o
        ref[5,0,0]=(-0.333825007954)/(o+1.)+(-0.46448520882)+(-0.343753586377)*0.5**o
        ref[5,0,1]=(-0.469125954795)/(o+1.)+(-0.147192309255)+(-0.235716718127)*0.5**o
        ref[5,1,0]=(-0.573275891777)/(o+1.)+(-0.478328163121)+(0.445652706545)*0.5**o
        ref[5,1,1]=(0.0545515234726)/(o+1.)+(-0.153919681346)+(0.195590500493)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_ReducedContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.226142137728)*x[0]+(0.85681895183)*x[1]
        arg[0,0,0,1]=(-0.824407181762)*x[0]+(-0.290023144892)*x[1]
        arg[0,0,1,0]=(-1.25987816017)*x[0]+(0.824340990595)*x[1]
        arg[0,0,1,1]=(-0.0917811606765)*x[0]+(0.660211932964)*x[1]
        arg[0,0,2,0]=(-1.31073149178)*x[0]+(0.482324984918)*x[1]
        arg[0,0,2,1]=(0.256995458227)*x[0]+(0.79244254852)*x[1]
        arg[0,1,0,0]=(-0.0682147475407)*x[0]+(0.243620052061)*x[1]
        arg[0,1,0,1]=(0.071099982687)*x[0]+(-0.62986754897)*x[1]
        arg[0,1,1,0]=(-0.0716793733131)*x[0]+(-1.15235525814)*x[1]
        arg[0,1,1,1]=(0.193107575216)*x[0]+(-0.596149957543)*x[1]
        arg[0,1,2,0]=(-0.160142629886)*x[0]+(1.36581707067)*x[1]
        arg[0,1,2,1]=(-1.21477648376)*x[0]+(-0.732000298369)*x[1]
        arg[0,2,0,0]=(-1.02468971059)*x[0]+(0.330721595217)*x[1]
        arg[0,2,0,1]=(1.11414907391)*x[0]+(1.29386127812)*x[1]
        arg[0,2,1,0]=(1.1485850154)*x[0]+(0.387585977495)*x[1]
        arg[0,2,1,1]=(0.237382583072)*x[0]+(1.56248221279)*x[1]
        arg[0,2,2,0]=(-0.949129049229)*x[0]+(0.578823379063)*x[1]
        arg[0,2,2,1]=(-1.0667182537)*x[0]+(0.462227485959)*x[1]
        arg[0,3,0,0]=(1.37954377554)*x[0]+(0.541856880139)*x[1]
        arg[0,3,0,1]=(0.401584700319)*x[0]+(0.577102327147)*x[1]
        arg[0,3,1,0]=(-0.289796408108)*x[0]+(0.807853785325)*x[1]
        arg[0,3,1,1]=(0.0038467861822)*x[0]+(-0.931311510771)*x[1]
        arg[0,3,2,0]=(0.859036254821)*x[0]+(0.198884218516)*x[1]
        arg[0,3,2,1]=(-0.00310256070968)*x[0]+(0.511838475968)*x[1]
        arg[0,4,0,0]=(0.539686886802)*x[0]+(-0.324612924267)*x[1]
        arg[0,4,0,1]=(-0.0458655898565)*x[0]+(0.211810380966)*x[1]
        arg[0,4,1,0]=(0.302333533003)*x[0]+(0.0663033316831)*x[1]
        arg[0,4,1,1]=(-0.981519161639)*x[0]+(-0.00207756909914)*x[1]
        arg[0,4,2,0]=(0.283659447004)*x[0]+(-0.231526592979)*x[1]
        arg[0,4,2,1]=(-1.17679167875)*x[0]+(0.285698985756)*x[1]
        arg[1,0,0,0]=(-0.162900059582)*x[0]+(-0.877164340783)*x[1]
        arg[1,0,0,1]=(-0.50468605065)*x[0]+(-0.160440073709)*x[1]
        arg[1,0,1,0]=(-0.100901676644)*x[0]+(-0.296111526029)*x[1]
        arg[1,0,1,1]=(0.176107421869)*x[0]+(1.74393708745)*x[1]
        arg[1,0,2,0]=(-0.979807721703)*x[0]+(-0.229400841067)*x[1]
        arg[1,0,2,1]=(0.217472795645)*x[0]+(0.0881503243323)*x[1]
        arg[1,1,0,0]=(-0.127980123625)*x[0]+(0.0932095456541)*x[1]
        arg[1,1,0,1]=(0.829674410931)*x[0]+(-0.796196041601)*x[1]
        arg[1,1,1,0]=(1.28656110233)*x[0]+(-1.27850471392)*x[1]
        arg[1,1,1,1]=(1.159269884)*x[0]+(-0.973135433631)*x[1]
        arg[1,1,2,0]=(1.46274837856)*x[0]+(-1.81554270863)*x[1]
        arg[1,1,2,1]=(0.0973472511688)*x[0]+(0.551451261495)*x[1]
        arg[1,2,0,0]=(0.510667671293)*x[0]+(-0.51075982099)*x[1]
        arg[1,2,0,1]=(-1.41612093451)*x[0]+(0.614417669629)*x[1]
        arg[1,2,1,0]=(1.46595125796)*x[0]+(0.60374279122)*x[1]
        arg[1,2,1,1]=(-0.317185218187)*x[0]+(-0.0785532882696)*x[1]
        arg[1,2,2,0]=(-0.388504041993)*x[0]+(-0.838003398775)*x[1]
        arg[1,2,2,1]=(0.0497207129012)*x[0]+(-0.622155692247)*x[1]
        arg[1,3,0,0]=(-0.0471400179563)*x[0]+(1.3378948207)*x[1]
        arg[1,3,0,1]=(-1.22250822195)*x[0]+(0.629714783138)*x[1]
        arg[1,3,1,0]=(-1.28801080408)*x[0]+(-1.54079710038)*x[1]
        arg[1,3,1,1]=(-0.426412448469)*x[0]+(-1.09410474021)*x[1]
        arg[1,3,2,0]=(0.022834167351)*x[0]+(-0.497423389788)*x[1]
        arg[1,3,2,1]=(-0.787492124301)*x[0]+(-0.245621168419)*x[1]
        arg[1,4,0,0]=(-1.37814880919)*x[0]+(-0.704805848451)*x[1]
        arg[1,4,0,1]=(-1.12494770055)*x[0]+(1.44920024305)*x[1]
        arg[1,4,1,0]=(-0.809532572761)*x[0]+(1.16638587657)*x[1]
        arg[1,4,1,1]=(0.931784486816)*x[0]+(1.34629889453)*x[1]
        arg[1,4,2,0]=(-0.172430942793)*x[0]+(-0.584736998321)*x[1]
        arg[1,4,2,1]=(0.00542948736389)*x[0]+(0.43917597284)*x[1]
        arg[2,0,0,0]=(-0.946122337406)*x[0]+(-0.161940168137)*x[1]
        arg[2,0,0,1]=(-0.7856653779)*x[0]+(-0.75287701788)*x[1]
        arg[2,0,1,0]=(-1.08858367592)*x[0]+(-0.530380818694)*x[1]
        arg[2,0,1,1]=(-1.0573555095)*x[0]+(-0.854148765785)*x[1]
        arg[2,0,2,0]=(0.723412825328)*x[0]+(0.326761950437)*x[1]
        arg[2,0,2,1]=(-0.0143142161225)*x[0]+(-0.752724311876)*x[1]
        arg[2,1,0,0]=(-0.942548092355)*x[0]+(0.700861094897)*x[1]
        arg[2,1,0,1]=(-0.712286567957)*x[0]+(0.230406545738)*x[1]
        arg[2,1,1,0]=(1.27414839774)*x[0]+(0.724784649037)*x[1]
        arg[2,1,1,1]=(0.637494163002)*x[0]+(0.442489599192)*x[1]
        arg[2,1,2,0]=(1.81617631726)*x[0]+(1.56870218809)*x[1]
        arg[2,1,2,1]=(1.16419900666)*x[0]+(-0.044314729141)*x[1]
        arg[2,2,0,0]=(0.880267125196)*x[0]+(0.77124406259)*x[1]
        arg[2,2,0,1]=(0.482402867542)*x[0]+(0.0746924305017)*x[1]
        arg[2,2,1,0]=(0.105861259696)*x[0]+(0.484292207267)*x[1]
        arg[2,2,1,1]=(-0.107785838196)*x[0]+(0.701132243209)*x[1]
        arg[2,2,2,0]=(0.559717405844)*x[0]+(-1.5585942717)*x[1]
        arg[2,2,2,1]=(-0.989130712605)*x[0]+(0.221361137556)*x[1]
        arg[2,3,0,0]=(-0.045471818095)*x[0]+(-0.437941959792)*x[1]
        arg[2,3,0,1]=(1.25840468379)*x[0]+(-1.18872011057)*x[1]
        arg[2,3,1,0]=(1.70665829408)*x[0]+(0.705273921003)*x[1]
        arg[2,3,1,1]=(-0.0928709253845)*x[0]+(0.785115449756)*x[1]
        arg[2,3,2,0]=(-0.887243418838)*x[0]+(1.06790660642)*x[1]
        arg[2,3,2,1]=(-0.572644686872)*x[0]+(-0.672831948394)*x[1]
        arg[2,4,0,0]=(1.28282566801)*x[0]+(0.651985199447)*x[1]
        arg[2,4,0,1]=(0.0965663229335)*x[0]+(1.33819998052)*x[1]
        arg[2,4,1,0]=(0.205569041098)*x[0]+(-0.873095358733)*x[1]
        arg[2,4,1,1]=(-1.04452025745)*x[0]+(-0.204917033907)*x[1]
        arg[2,4,2,0]=(-0.181163938972)*x[0]+(0.0341993409272)*x[1]
        arg[2,4,2,1]=(1.68327027059)*x[0]+(-0.289436355642)*x[1]
        arg[3,0,0,0]=(1.05712810731)*x[0]+(0.915711819735)*x[1]
        arg[3,0,0,1]=(-0.171072039482)*x[0]+(-0.160656943623)*x[1]
        arg[3,0,1,0]=(-1.27063799869)*x[0]+(-0.605131642813)*x[1]
        arg[3,0,1,1]=(0.940512990835)*x[0]+(-0.116675577531)*x[1]
        arg[3,0,2,0]=(-0.194213869952)*x[0]+(0.364066752025)*x[1]
        arg[3,0,2,1]=(-0.328780465397)*x[0]+(-0.114689413673)*x[1]
        arg[3,1,0,0]=(-0.825914305895)*x[0]+(-0.167010056731)*x[1]
        arg[3,1,0,1]=(0.657779056207)*x[0]+(0.0666252451638)*x[1]
        arg[3,1,1,0]=(-0.464004426036)*x[0]+(-0.340932918409)*x[1]
        arg[3,1,1,1]=(-1.6698652667)*x[0]+(0.731870373141)*x[1]
        arg[3,1,2,0]=(1.39626250779)*x[0]+(-0.879152855557)*x[1]
        arg[3,1,2,1]=(1.34973919508)*x[0]+(-0.129753332761)*x[1]
        arg[3,2,0,0]=(1.03443450706)*x[0]+(0.807333477811)*x[1]
        arg[3,2,0,1]=(0.366011302596)*x[0]+(0.543804800926)*x[1]
        arg[3,2,1,0]=(0.262446846036)*x[0]+(-0.247786850991)*x[1]
        arg[3,2,1,1]=(-1.04657091196)*x[0]+(-0.485443426155)*x[1]
        arg[3,2,2,0]=(0.342434327538)*x[0]+(-0.148503585588)*x[1]
        arg[3,2,2,1]=(-0.189833429716)*x[0]+(0.424391874819)*x[1]
        arg[3,3,0,0]=(-0.118348838118)*x[0]+(0.397628264341)*x[1]
        arg[3,3,0,1]=(0.481440510084)*x[0]+(-0.393576221424)*x[1]
        arg[3,3,1,0]=(0.907533552208)*x[0]+(-0.668646549376)*x[1]
        arg[3,3,1,1]=(-1.49731714213)*x[0]+(-1.18710678264)*x[1]
        arg[3,3,2,0]=(-0.460660079207)*x[0]+(1.48078266444)*x[1]
        arg[3,3,2,1]=(-0.161073556988)*x[0]+(-0.486827042754)*x[1]
        arg[3,4,0,0]=(-0.668755455623)*x[0]+(0.345804059434)*x[1]
        arg[3,4,0,1]=(1.00528889193)*x[0]+(0.978507928191)*x[1]
        arg[3,4,1,0]=(0.98955856331)*x[0]+(1.0421852144)*x[1]
        arg[3,4,1,1]=(0.711688260873)*x[0]+(1.39962222322)*x[1]
        arg[3,4,2,0]=(1.08512429209)*x[0]+(-0.555249097743)*x[1]
        arg[3,4,2,1]=(-1.10443784007)*x[0]+(-0.782274429169)*x[1]
        ref[0,0,0,0]=(0.151153165564)/(o+1.)+(0.112609656739)+(0.254304335059)*0.5**o
        ref[0,0,0,1]=(-0.530602583511)/(o+1.)+(0.1817855378)+(-0.947398818743)*0.5**o
        ref[0,0,1,0]=(0.308747437738)/(o+1.)+(-0.0338129255643)+(-0.676658756184)*0.5**o
        ref[0,0,1,1]=(0.321026553427)/(o+1.)+(0.0692705367618)+(0.108863145337)*0.5**o
        ref[0,0,2,0]=(-0.0908034666655)/(o+1.)+(-0.1100205155)+(-0.517562009199)*0.5**o
        ref[0,0,2,1]=(0.303456040903)/(o+1.)+(0.500090875336)+(-0.254199784829)*0.5**o
        ref[0,1,0,0]=(-0.440641221831)/(o+1.)+(0.456218946873)+(-0.296391367396)*0.5**o
        ref[0,1,0,1]=(0.222628561762)/(o+1.)+(-0.162064412854)+(-0.457267302337)*0.5**o
        ref[0,1,1,0]=(-0.913361887408)/(o+1.)+(-0.356168674239)+(0.401664604431)*0.5**o
        ref[0,1,1,1]=(-0.543116639248)/(o+1.)+(-0.120902439422)+(0.381879135767)*0.5**o
        ref[0,1,2,0]=(0.993649262831)/(o+1.)+(0.384695858206)+(-0.55736653846)*0.5**o
        ref[0,1,2,1]=(-0.968077797451)/(o+1.)+(-0.187765942637)+(-0.603167099406)*0.5**o
        ref[0,2,0,0]=(0.873662019918)/(o+1.)+(-0.710635360033)+(-0.146359415223)*0.5**o
        ref[0,2,0,1]=(0.466408230693)/(o+1.)+(0.77626565285)+(0.389070815631)*0.5**o
        ref[0,2,1,0]=(-0.475010813579)/(o+1.)+(0.525435880006)+(0.960310046461)*0.5**o
        ref[0,2,1,1]=(0.697127520484)/(o+1.)+(0.357204276552)+(0.388328722275)*0.5**o
        ref[0,2,2,0]=(-0.108434738487)/(o+1.)+(0.356443300466)+(-0.97475753261)*0.5**o
        ref[0,2,2,1]=(0.952274964304)/(o+1.)+(-0.497379508976)+(-0.562006714089)*0.5**o
        ref[0,3,0,0]=(-0.0368012761671)/(o+1.)+(0.594000625518)+(0.770200680808)*0.5**o
        ref[0,3,0,1]=(0.450361861836)/(o+1.)+(0.182031052848)+(0.164263059934)*0.5**o
        ref[0,3,1,0]=(0.500714741956)/(o+1.)+(0.0479711343168)+(-0.0785996333735)*0.5**o
        ref[0,3,1,1]=(-0.949280938645)/(o+1.)+(0.452441874504)+(-0.88306753495)*0.5**o
        ref[0,3,2,0]=(0.186931712837)/(o+1.)+(0.222335175903)+(0.426318408694)*0.5**o
        ref[0,3,2,1]=(-0.288015428918)/(o+1.)+(0.80948657963)+(-0.822221815085)*0.5**o
        ref[0,4,0,0]=(0.670151184695)/(o+1.)+(-0.415883847013)+(0.376690471866)*0.5**o
        ref[0,4,0,1]=(-0.107106843992)/(o+1.)+(0.555375298435)+(-0.837698961768)*0.5**o
        ref[0,4,1,0]=(-0.299999096802)/(o+1.)+(0.668996782774)+(-0.669357604061)*0.5**o
        ref[0,4,1,1]=(0.778006183401)/(o+1.)+(-0.614473529032)+(-0.532655856076)*0.5**o
        ref[0,4,2,0]=(-0.250068389818)/(o+1.)+(0.156274314197)+(-0.0103473845504)*0.5**o
        ref[0,4,2,1]=(0.902125322216)/(o+1.)+(-0.616424567683)+(-0.560368879846)*0.5**o
        ref[1,0,0,0]=(-0.133690258295)/(o+1.)+(-0.715949360249)+(0.525524578427)*0.5**o
        ref[1,0,0,1]=(-0.788097689216)/(o+1.)+(0.0998405659256)+(-0.076709566994)*0.5**o
        ref[1,0,1,0]=(0.144280886187)/(o+1.)+(0.0775377290314)+(-0.696369546924)*0.5**o
        ref[1,0,1,1]=(0.824172196605)/(o+1.)+(0.561804029444)+(-0.0277357461773)*0.5**o
        ref[1,0,2,0]=(0.422564421802)/(o+1.)+(-0.60719307099)+(-0.417386842591)*0.5**o
        ref[1,0,2,1]=(0.778908672762)/(o+1.)+(0.0163446340765)+(-0.505974820938)*0.5**o
        ref[1,1,0,0]=(0.0574071047387)/(o+1.)+(-0.350514214742)+(0.608850746775)*0.5**o
        ref[1,1,0,1]=(-0.991660961792)/(o+1.)+(0.322084279455)+(0.380970772212)*0.5**o
        ref[1,1,1,0]=(-0.823761120268)/(o+1.)+(-0.0495419092898)+(0.930901327253)*0.5**o
        ref[1,1,1,1]=(-0.384729980569)/(o+1.)+(-0.0392583789163)+(0.649381188769)*0.5**o
        ref[1,1,2,0]=(-0.840365304069)/(o+1.)+(-0.0150512662777)+(0.517673506558)*0.5**o
        ref[1,1,2,1]=(0.555976660846)/(o+1.)+(-0.307265886898)+(0.707353625614)*0.5**o
        ref[1,2,0,0]=(-0.423056705491)/(o+1.)+(0.0559404393793)+(0.311083677036)*0.5**o
        ref[1,2,0,1]=(0.15409797664)/(o+1.)+(-0.0906637067519)+(-0.774473828018)*0.5**o
        ref[1,2,1,0]=(0.796718644088)/(o+1.)+(0.248012077098)+(0.776951250898)*0.5**o
        ref[1,2,1,1]=(-0.342830889263)/(o+1.)+(-0.282162737841)+(0.511417858487)*0.5**o
        ref[1,2,2,0]=(-0.722403016671)/(o+1.)+(-0.367059457954)+(0.230014491812)*0.5**o
        ref[1,2,2,1]=(-0.773685058881)/(o+1.)+(0.412807900451)+(-0.624365721366)*0.5**o
        ref[1,3,0,0]=(0.527481434758)/(o+1.)+(0.36602961117)+(0.031214145649)*0.5**o
        ref[1,3,0,1]=(0.0630927531585)/(o+1.)+(-0.164989701995)+(-0.325906787975)*0.5**o
        ref[1,3,1,0]=(-0.612390590593)/(o+1.)+(-0.78855996274)+(-0.63929738839)*0.5**o
        ref[1,3,1,1]=(-0.962656044265)/(o+1.)+(-0.00576535037253)+(-0.546330443673)*0.5**o
        ref[1,3,2,0]=(-0.947280922008)/(o+1.)+(-0.137449579996)+(0.747590859563)*0.5**o
        ref[1,3,2,1]=(-0.946623026202)/(o+1.)+(0.0787536550213)+(-0.24399757656)*0.5**o
        ref[1,4,0,0]=(-0.174471016943)/(o+1.)+(-0.636500556039)+(-0.635482528623)*0.5**o
        ref[1,4,0,1]=(0.806379391595)/(o+1.)+(-0.00304106577093)+(-0.476044717545)*0.5**o
        ref[1,4,1,0]=(0.729649125134)/(o+1.)+(0.0151477051616)+(-0.403091231646)*0.5**o
        ref[1,4,1,1]=(0.559976454745)/(o+1.)+(0.479295896421)+(0.75951513376)*0.5**o
        ref[1,4,2,0]=(0.272199270818)/(o+1.)+(-0.0929072274332)+(-0.843552757065)*0.5**o
        ref[1,4,2,1]=(-0.552663678377)/(o+1.)+(0.613784745778)+(-0.230300352975)*0.5**o
        ref[2,0,0,0]=(-0.174247908098)/(o+1.)+(-0.108873783282)+(-0.716067030882)*0.5**o
        ref[2,0,0,1]=(-0.300584841532)/(o+1.)+(-0.485363665493)+(-0.267230223262)*0.5**o
        ref[2,0,1,0]=(0.316261805154)/(o+1.)+(-0.523474106451)+(-0.888278086869)*0.5**o
        ref[2,0,1,1]=(-0.699622562861)/(o+1.)+(-0.151089538248)+(-0.909702635929)*0.5**o
        ref[2,0,2,0]=(0.728545639829)/(o+1.)+(0.192083148877)+(-0.0625371618174)*0.5**o
        ref[2,0,2,1]=(-0.22042267239)/(o+1.)+(-0.611974599324)+(0.677333343041)*0.5**o
        ref[2,1,0,0]=(-0.0734883811113)/(o+1.)+(0.36100831022)+(-0.890215236787)*0.5**o
        ref[2,1,0,1]=(0.121656829057)/(o+1.)+(-0.130801635799)+(-0.341933579678)*0.5**o
        ref[2,1,1,0]=(0.221353168629)/(o+1.)+(0.618635673483)+(0.540308531179)*0.5**o
        ref[2,1,1,1]=(0.0152580531273)/(o+1.)+(0.252173415542)+(0.560378877984)*0.5**o
        ref[2,1,2,0]=(0.580319777731)/(o+1.)+(0.90566612137)+(0.993226484879)*0.5**o
        ref[2,1,2,1]=(-0.710250320949)/(o+1.)+(0.742177021871)+(0.345780554724)*0.5**o
        ref[2,2,0,0]=(0.770701563507)/(o+1.)+(0.240823471881)+(0.399162680517)*0.5**o
        ref[2,2,0,1]=(0.76121761761)/(o+1.)+(0.128382656738)+(-0.460887633042)*0.5**o
        ref[2,2,1,0]=(-0.269854962057)/(o+1.)+(0.690854898294)+(-0.521701367567)*0.5**o
        ref[2,2,1,1]=(0.719518595712)/(o+1.)+(0.029596430244)+(-0.185365051187)*0.5**o
        ref[2,2,2,0]=(-0.671157761165)/(o+1.)+(-0.142688285012)+(-0.0423425346644)*0.5**o
        ref[2,2,2,1]=(-0.541156097112)/(o+1.)+(0.0765180918929)+(-0.379649661722)*0.5**o
        ref[2,3,0,0]=(-0.0604347109212)/(o+1.)+(-0.19403702073)+(-0.034905025507)*0.5**o
        ref[2,3,0,1]=(-0.793398507071)/(o+1.)+(0.255681280626)+(0.351720519037)*0.5**o
        ref[2,3,1,0]=(-0.267058413731)/(o+1.)+(0.852211426049)+(0.974567776717)*0.5**o
        ref[2,3,1,1]=(0.228396641207)/(o+1.)+(0.322922896735)+(-0.181997910305)*0.5**o
        ref[2,3,2,0]=(0.480987773937)/(o+1.)+(-0.197709028569)+(0.0950934707835)*0.5**o
        ref[2,3,2,1]=(-0.648934475027)/(o+1.)+(-0.418383297875)+(0.240224435511)*0.5**o
        ref[2,4,0,0]=(0.430020261528)/(o+1.)+(0.53421018733)+(0.436370231269)*0.5**o
        ref[2,4,0,1]=(0.906390660223)/(o+1.)+(-0.056057545856)+(0.640490734948)*0.5**o
        ref[2,4,1,0]=(-0.732099770943)/(o+1.)+(-0.0485027617732)+(0.161578976854)*0.5**o
        ref[2,4,1,1]=(-0.827495680848)/(o+1.)+(0.285044728936)+(-0.992031068384)*0.5**o
        ref[2,4,2,0]=(-0.919905471787)/(o+1.)+(0.656971386264)+(-0.541001898785)*0.5**o
        ref[2,4,2,1]=(0.161975875302)/(o+1.)+(0.272665206557)+(0.686527626533)*0.5**o
        ref[3,0,0,0]=(0.584120963259)/(o+1.)+(0.207930729804)+(0.972857504177)*0.5**o
        ref[3,0,0,1]=(0.315823141491)/(o+1.)+(-0.217561208386)+(-0.212429707823)*0.5**o
        ref[3,0,1,0]=(-0.732294996139)/(o+1.)+(-0.278653143596)+(-0.586168358175)*0.5**o
        ref[3,0,1,1]=(0.481321847225)/(o+1.)+(-0.070080648792)+(0.482676863662)*0.5**o
        ref[3,0,2,0]=(0.616679859076)/(o+1.)+(-0.418784546677)+(0.390742116352)*0.5**o
        ref[3,0,2,1]=(0.866778417448)/(o+1.)+(-0.686585656643)+(0.0629230167692)*0.5**o
        ref[3,1,0,0]=(0.531890054167)/(o+1.)+(-0.516187607633)+(-0.492439201527)*0.5**o
        ref[3,1,0,1]=(-0.512361639413)/(o+1.)+(0.499557182545)+(0.237651575694)*0.5**o
        ref[3,1,1,0]=(-0.67542965405)/(o+1.)+(0.274414249559)+(-0.678336189514)*0.5**o
        ref[3,1,1,1]=(0.371936242507)/(o+1.)+(-0.1636482466)+(-0.982634642862)*0.5**o
        ref[3,1,2,0]=(-0.799408831501)/(o+1.)+(0.458633399549)+(0.399251684639)*0.5**o
        ref[3,1,2,1]=(-0.0145253903686)/(o+1.)+(0.238472345905)+(0.757566560882)*0.5**o
        ref[3,2,0,0]=(-0.130519473112)/(o+1.)+(0.695277604837)+(0.581732248306)*0.5**o
        ref[3,2,0,1]=(0.632010536438)/(o+1.)+(0.123066205529)+(0.0316731560259)*0.5**o
        ref[3,2,1,0]=(-0.709926210356)/(o+1.)+(0.0926699972302)+(0.53924621094)*0.5**o
        ref[3,2,1,1]=(0.428094002247)/(o+1.)+(-0.841997926131)+(-0.276112488096)*0.5**o
        ref[3,2,2,0]=(0.708481148086)/(o+1.)+(-0.50498161534)+(0.495412824545)*0.5**o
        ref[3,2,2,1]=(0.782824622703)/(o+1.)+(-0.151509698687)+(-0.245246780227)*0.5**o
        ref[3,3,0,0]=(0.848460868418)/(o+1.)+(-0.104639248461)+(-0.359902945273)*0.5**o
        ref[3,3,0,1]=(0.0569151307298)/(o+1.)+(-0.0674224535469)+(0.165794065024)*0.5**o
        ref[3,3,1,0]=(-0.165683739908)/(o+1.)+(-0.183388699236)+(0.771348141212)*0.5**o
        ref[3,3,1,1]=(-0.932550058069)/(o+1.)+(-0.47828109001)+(-0.795311686688)*0.5**o
        ref[3,3,2,0]=(0.821542940923)/(o+1.)+(0.166853194487)+(-0.13512674466)*0.5**o
        ref[3,3,2,1]=(-0.46375871628)/(o+1.)+(0.132701267732)+(-0.449544418926)*0.5**o
        ref[3,4,0,0]=(-0.074530542168)/(o+1.)+(-0.0567877963797)+(-0.134845261261)*0.5**o
        ref[3,4,0,1]=(0.238671520479)/(o+1.)+(0.50689253572)+(0.731340228204)*0.5**o
        ref[3,4,1,0]=(0.981601513335)/(o+1.)+(0.400717209822)+(0.248707844726)*0.5**o
        ref[3,4,1,1]=(0.592304390107)/(o+1.)+(0.342408637328)+(0.834188819333)*0.5**o
        ref[3,4,2,0]=(-0.339991330909)/(o+1.)+(-0.0145922376156)+(0.899051000485)*0.5**o
        ref[3,4,2,1]=(-0.626610688164)/(o+1.)+(-0.417008165115)+(-0.426085250846)*0.5**o
      else:
        arg[0,0,0,0]=(0.697666818785)*x[0]+(1.00202295791)*x[1]+(0.617700157184)*x[2]
        arg[0,0,0,1]=(-0.465577819457)*x[0]+(-1.40054693355)*x[1]+(1.16912653295)*x[2]
        arg[0,0,1,0]=(-1.26696500634)*x[0]+(-0.0469756169975)*x[1]+(-0.445591738991)*x[2]
        arg[0,0,1,1]=(-1.61654321709)*x[0]+(-1.69522962467)*x[1]+(-1.06089094851)*x[2]
        arg[0,0,2,0]=(0.229468378506)*x[0]+(0.716813596991)*x[1]+(-0.237703937029)*x[2]
        arg[0,0,2,1]=(-1.12052051905)*x[0]+(0.835000004618)*x[1]+(-0.853004812404)*x[2]
        arg[0,1,0,0]=(-1.75500745705)*x[0]+(0.655055836882)*x[1]+(1.05207750735)*x[2]
        arg[0,1,0,1]=(-0.0746024824253)*x[0]+(-0.421596816958)*x[1]+(-0.375715855989)*x[2]
        arg[0,1,1,0]=(-0.963269684396)*x[0]+(0.431993248836)*x[1]+(-1.63061928649)*x[2]
        arg[0,1,1,1]=(-0.418757265018)*x[0]+(-0.665626168583)*x[1]+(-0.335734432507)*x[2]
        arg[0,1,2,0]=(0.980346404307)*x[0]+(1.45442049802)*x[1]+(-0.833420308612)*x[2]
        arg[0,1,2,1]=(-1.52536017847)*x[0]+(0.99484028681)*x[1]+(0.109754321663)*x[2]
        arg[0,2,0,0]=(1.697179262)*x[0]+(-0.928732802147)*x[1]+(-0.653379513626)*x[2]
        arg[0,2,0,1]=(0.682698902948)*x[0]+(0.829307959583)*x[1]+(-1.0300112276)*x[2]
        arg[0,2,1,0]=(0.026084323266)*x[0]+(1.28865784608)*x[1]+(0.830446580645)*x[2]
        arg[0,2,1,1]=(0.486291506032)*x[0]+(-0.519637742913)*x[1]+(0.81671330629)*x[2]
        arg[0,2,2,0]=(-0.761591624634)*x[0]+(0.370951131011)*x[1]+(-0.256748479312)*x[2]
        arg[0,2,2,1]=(-0.249974326513)*x[0]+(0.267323484635)*x[1]+(-1.09642130905)*x[2]
        arg[0,3,0,0]=(0.752057563255)*x[0]+(-1.2772262175)*x[1]+(-0.264569668983)*x[2]
        arg[0,3,0,1]=(-0.173167796447)*x[0]+(0.31873208356)*x[1]+(-0.599780701112)*x[2]
        arg[0,3,1,0]=(-0.912464014941)*x[0]+(-1.56452517166)*x[1]+(1.37651660499)*x[2]
        arg[0,3,1,1]=(0.302009667854)*x[0]+(0.524213064769)*x[1]+(0.316301522683)*x[2]
        arg[0,3,2,0]=(-1.2940154673)*x[0]+(-0.900061396992)*x[1]+(0.3860029847)*x[2]
        arg[0,3,2,1]=(-0.97354131341)*x[0]+(1.08221883964)*x[1]+(-0.739323076416)*x[2]
        arg[0,4,0,0]=(1.86535317192)*x[0]+(-0.478359589546)*x[1]+(-0.798014592302)*x[2]
        arg[0,4,0,1]=(-1.08518693851)*x[0]+(-0.186048654536)*x[1]+(1.04915573267)*x[2]
        arg[0,4,1,0]=(-1.09692811055)*x[0]+(1.01042920935)*x[1]+(-0.73978089555)*x[2]
        arg[0,4,1,1]=(0.654656964663)*x[0]+(0.279204596126)*x[1]+(0.886217781361)*x[2]
        arg[0,4,2,0]=(-0.54009886291)*x[0]+(0.378037248739)*x[1]+(1.09785568886)*x[2]
        arg[0,4,2,1]=(-1.43995958913)*x[0]+(-0.720157090994)*x[1]+(0.0676362028136)*x[2]
        arg[1,0,0,0]=(1.47735412608)*x[0]+(0.342355546228)*x[1]+(-0.0134502415508)*x[2]
        arg[1,0,0,1]=(-0.419536590369)*x[0]+(-1.61108470511)*x[1]+(1.51281979355)*x[2]
        arg[1,0,1,0]=(1.31765698245)*x[0]+(-0.7499566294)*x[1]+(0.934537503478)*x[2]
        arg[1,0,1,1]=(0.688467914883)*x[0]+(-1.32681280673)*x[1]+(-0.387405705058)*x[2]
        arg[1,0,2,0]=(1.37069587556)*x[0]+(0.540830111773)*x[1]+(0.477311068263)*x[2]
        arg[1,0,2,1]=(-0.572066213761)*x[0]+(-1.11950925494)*x[1]+(1.07939217872)*x[2]
        arg[1,1,0,0]=(0.564997105326)*x[0]+(-0.73130546389)*x[1]+(0.435910937304)*x[2]
        arg[1,1,0,1]=(0.513931739674)*x[0]+(-0.944043404826)*x[1]+(0.014114227468)*x[2]
        arg[1,1,1,0]=(-0.77850597256)*x[0]+(-0.445835136484)*x[1]+(0.157478357479)*x[2]
        arg[1,1,1,1]=(0.752431176866)*x[0]+(0.659759664199)*x[1]+(-0.859515433521)*x[2]
        arg[1,1,2,0]=(0.0693158730911)*x[0]+(-0.820997827204)*x[1]+(-0.714694436829)*x[2]
        arg[1,1,2,1]=(0.604674152847)*x[0]+(0.647261183132)*x[1]+(-1.21981352028)*x[2]
        arg[1,2,0,0]=(0.0997504695984)*x[0]+(-0.271549839083)*x[1]+(-0.573467835942)*x[2]
        arg[1,2,0,1]=(0.01861845581)*x[0]+(0.157410502406)*x[1]+(0.939257868776)*x[2]
        arg[1,2,1,0]=(1.44909894631)*x[0]+(0.116038517547)*x[1]+(-0.851940489233)*x[2]
        arg[1,2,1,1]=(1.34598841007)*x[0]+(1.79163125841)*x[1]+(0.719171044245)*x[2]
        arg[1,2,2,0]=(-0.214737791448)*x[0]+(-1.85810839095)*x[1]+(-0.240776599927)*x[2]
        arg[1,2,2,1]=(-0.574707645928)*x[0]+(-0.664173831956)*x[1]+(0.310262106696)*x[2]
        arg[1,3,0,0]=(0.296213453592)*x[0]+(-0.831112482555)*x[1]+(0.130075739128)*x[2]
        arg[1,3,0,1]=(-0.369510604457)*x[0]+(0.0373909697998)*x[1]+(1.6689129783)*x[2]
        arg[1,3,1,0]=(0.0922435881419)*x[0]+(-0.280519827004)*x[1]+(0.285933839232)*x[2]
        arg[1,3,1,1]=(-0.649253282272)*x[0]+(-0.460065044123)*x[1]+(-0.0711164948063)*x[2]
        arg[1,3,2,0]=(0.793313307581)*x[0]+(-0.0587956351149)*x[1]+(0.810487259739)*x[2]
        arg[1,3,2,1]=(-1.38668597055)*x[0]+(1.51396934198)*x[1]+(-1.13918550255)*x[2]
        arg[1,4,0,0]=(1.40078167737)*x[0]+(-1.61915436345)*x[1]+(0.421710091214)*x[2]
        arg[1,4,0,1]=(-1.37982015877)*x[0]+(1.30607371738)*x[1]+(-0.608888848419)*x[2]
        arg[1,4,1,0]=(-0.558087802805)*x[0]+(-0.214169499512)*x[1]+(-0.766242546503)*x[2]
        arg[1,4,1,1]=(-1.75503436584)*x[0]+(0.183604705451)*x[1]+(-0.148071484576)*x[2]
        arg[1,4,2,0]=(0.537942069142)*x[0]+(-0.00640749928252)*x[1]+(-0.277544629657)*x[2]
        arg[1,4,2,1]=(-0.652645833371)*x[0]+(-1.33439077026)*x[1]+(-0.226308197201)*x[2]
        arg[2,0,0,0]=(1.45646289279)*x[0]+(1.40842127675)*x[1]+(1.75078281495)*x[2]
        arg[2,0,0,1]=(0.867988651995)*x[0]+(1.35061812241)*x[1]+(-1.56520776323)*x[2]
        arg[2,0,1,0]=(0.549686523522)*x[0]+(-0.528085465813)*x[1]+(-0.994978820267)*x[2]
        arg[2,0,1,1]=(0.383595996552)*x[0]+(0.612989859087)*x[1]+(-0.560401247231)*x[2]
        arg[2,0,2,0]=(-0.341259919499)*x[0]+(-0.125491675042)*x[1]+(0.545572340757)*x[2]
        arg[2,0,2,1]=(-0.414269395434)*x[0]+(-0.433969881898)*x[1]+(-1.01820199328)*x[2]
        arg[2,1,0,0]=(-0.113575280389)*x[0]+(0.326916039942)*x[1]+(-0.576072890249)*x[2]
        arg[2,1,0,1]=(-0.296473969661)*x[0]+(0.195675099431)*x[1]+(-1.72378150831)*x[2]
        arg[2,1,1,0]=(-1.31575932758)*x[0]+(-0.122557423522)*x[1]+(0.53435027695)*x[2]
        arg[2,1,1,1]=(0.299435754608)*x[0]+(0.727140413706)*x[1]+(0.848931358475)*x[2]
        arg[2,1,2,0]=(0.2618965198)*x[0]+(-0.461672573339)*x[1]+(0.923592803172)*x[2]
        arg[2,1,2,1]=(-0.326817258523)*x[0]+(-0.0363300805442)*x[1]+(1.82049875918)*x[2]
        arg[2,2,0,0]=(-0.435113320888)*x[0]+(0.0257984353786)*x[1]+(0.287846577657)*x[2]
        arg[2,2,0,1]=(1.39178068279)*x[0]+(0.136781967539)*x[1]+(-1.38161590351)*x[2]
        arg[2,2,1,0]=(-0.887874199356)*x[0]+(-0.476769153654)*x[1]+(-0.618760485829)*x[2]
        arg[2,2,1,1]=(1.58883355098)*x[0]+(-0.889469047555)*x[1]+(-0.678486755843)*x[2]
        arg[2,2,2,0]=(0.237147029365)*x[0]+(-0.509883462401)*x[1]+(0.533163151551)*x[2]
        arg[2,2,2,1]=(0.66513913749)*x[0]+(1.30317030657)*x[1]+(0.804083365617)*x[2]
        arg[2,3,0,0]=(1.01531518403)*x[0]+(1.3753053143)*x[1]+(0.000877284235964)*x[2]
        arg[2,3,0,1]=(-0.40853642543)*x[0]+(0.154794268783)*x[1]+(-1.34306671117)*x[2]
        arg[2,3,1,0]=(0.43065284359)*x[0]+(-0.28965146009)*x[1]+(0.660248828789)*x[2]
        arg[2,3,1,1]=(-1.4334922681)*x[0]+(-0.0348802858609)*x[1]+(0.502051304974)*x[2]
        arg[2,3,2,0]=(-1.17399537487)*x[0]+(0.865787043946)*x[1]+(-0.0105949677478)*x[2]
        arg[2,3,2,1]=(0.63104158794)*x[0]+(0.436196079951)*x[1]+(0.498606623061)*x[2]
        arg[2,4,0,0]=(-1.44410177354)*x[0]+(0.513406326011)*x[1]+(0.699127199447)*x[2]
        arg[2,4,0,1]=(-0.981727023077)*x[0]+(0.484182120695)*x[1]+(0.667175832045)*x[2]
        arg[2,4,1,0]=(-1.3352602588)*x[0]+(0.626269023364)*x[1]+(0.813537646695)*x[2]
        arg[2,4,1,1]=(-0.932486930114)*x[0]+(0.299448807849)*x[1]+(0.195862014797)*x[2]
        arg[2,4,2,0]=(-0.0621489644806)*x[0]+(0.131979664158)*x[1]+(0.442029269778)*x[2]
        arg[2,4,2,1]=(-0.507508608633)*x[0]+(-1.28613226533)*x[1]+(-0.697920444023)*x[2]
        arg[3,0,0,0]=(-0.984942677585)*x[0]+(-1.58913299616)*x[1]+(-0.373026784822)*x[2]
        arg[3,0,0,1]=(0.616666427113)*x[0]+(-0.625162800471)*x[1]+(-0.796307290839)*x[2]
        arg[3,0,1,0]=(-0.48469203683)*x[0]+(-0.0572105166553)*x[1]+(0.809572383581)*x[2]
        arg[3,0,1,1]=(-0.487160309892)*x[0]+(0.77646454341)*x[1]+(-0.41947544573)*x[2]
        arg[3,0,2,0]=(-1.17902121554)*x[0]+(-0.244939532033)*x[1]+(-0.282994359476)*x[2]
        arg[3,0,2,1]=(0.33515909881)*x[0]+(0.146866561705)*x[1]+(1.81486098364)*x[2]
        arg[3,1,0,0]=(0.379977901341)*x[0]+(0.834531701517)*x[1]+(-0.0242027062166)*x[2]
        arg[3,1,0,1]=(0.184318292766)*x[0]+(1.08367911805)*x[1]+(-0.164531741594)*x[2]
        arg[3,1,1,0]=(-1.00967115996)*x[0]+(0.542699640149)*x[1]+(0.707616701259)*x[2]
        arg[3,1,1,1]=(0.856948233811)*x[0]+(-1.2186550297)*x[1]+(-0.034268798159)*x[2]
        arg[3,1,2,0]=(0.259470684477)*x[0]+(-1.03586983732)*x[1]+(-0.157275979341)*x[2]
        arg[3,1,2,1]=(-0.161954890658)*x[0]+(1.65853209885)*x[1]+(0.381113456719)*x[2]
        arg[3,2,0,0]=(-0.437413004681)*x[0]+(-0.730340273929)*x[1]+(-0.687689272494)*x[2]
        arg[3,2,0,1]=(-0.336618164118)*x[0]+(-1.24388666895)*x[1]+(-0.2198689254)*x[2]
        arg[3,2,1,0]=(0.0872793087887)*x[0]+(-0.336554573215)*x[1]+(-0.0597803523262)*x[2]
        arg[3,2,1,1]=(0.192805432018)*x[0]+(0.0540495582064)*x[1]+(0.846658042071)*x[2]
        arg[3,2,2,0]=(0.00794192604414)*x[0]+(-0.33504559053)*x[1]+(0.615701149787)*x[2]
        arg[3,2,2,1]=(0.210870961503)*x[0]+(1.14164095846)*x[1]+(-0.192785742791)*x[2]
        arg[3,3,0,0]=(-0.25895162041)*x[0]+(-0.225800436488)*x[1]+(-0.198007079838)*x[2]
        arg[3,3,0,1]=(-0.150363703231)*x[0]+(1.23870429535)*x[1]+(0.28635881247)*x[2]
        arg[3,3,1,0]=(-0.865980718216)*x[0]+(-0.0849853244335)*x[1]+(-0.292164217189)*x[2]
        arg[3,3,1,1]=(-0.350465968722)*x[0]+(0.09853736936)*x[1]+(-0.476711524665)*x[2]
        arg[3,3,2,0]=(0.311396300263)*x[0]+(-0.92437612034)*x[1]+(-0.757139962928)*x[2]
        arg[3,3,2,1]=(-0.912311919837)*x[0]+(0.267793631099)*x[1]+(0.4917345401)*x[2]
        arg[3,4,0,0]=(0.613662541753)*x[0]+(0.782757341361)*x[1]+(1.04209808158)*x[2]
        arg[3,4,0,1]=(0.492828613997)*x[0]+(1.11979095664)*x[1]+(-0.843545072354)*x[2]
        arg[3,4,1,0]=(-0.51939120556)*x[0]+(0.476004484286)*x[1]+(-0.516311785432)*x[2]
        arg[3,4,1,1]=(1.00423313567)*x[0]+(0.90776890388)*x[1]+(1.34423309497)*x[2]
        arg[3,4,2,0]=(-1.44069600826)*x[0]+(-1.39515290909)*x[1]+(1.19851590247)*x[2]
        arg[3,4,2,1]=(0.391851619877)*x[0]+(0.265529120553)*x[1]+(0.996610410913)*x[2]
        ref[0,0,0,0]=(1.19330263935)/(o+1.)+(0.55394275251)+(0.0162017895041)*0.5**o
        ref[0,0,0,1]=(-0.672351783684)/(o+1.)+(-0.187889622039)+(0.351132807711)*0.5**o
        ref[0,0,1,0]=(-1.27021996264)/(o+1.)+(0.236286530101)+(-0.961885459894)*0.5**o
        ref[0,0,1,1]=(-1.43442517107)/(o+1.)+(-1.00113374859)+(-0.935971122022)*0.5**o
        ref[0,0,2,0]=(-0.434813695127)/(o+1.)+(0.606673209583)+(-0.0699546855712)*0.5**o
        ref[0,0,2,1]=(-0.0434901247948)/(o+1.)+(-0.180911877983)+(-0.733211446073)*0.5**o
        ref[0,1,0,0]=(1.41514302182)/(o+1.)+(-0.247977413325)+(-0.967062307989)*0.5**o
        ref[0,1,0,1]=(0.69128100142)/(o+1.)+(-0.621609882923)+(-0.319976390946)*0.5**o
        ref[0,1,1,0]=(-0.881870429191)/(o+1.)+(-0.654075774082)+(0.0281262553057)*0.5**o
        ref[0,1,1,1]=(-1.27239116086)/(o+1.)+(-0.0854066429492)+(0.0230865806541)*0.5**o
        ref[0,1,2,0]=(-0.222663945351)/(o+1.)+(0.685312598747)+(0.453385341573)*0.5**o
        ref[0,1,2,1]=(0.101409038706)/(o+1.)+(0.151367946744)+(-0.824910502191)*0.5**o
        ref[0,2,0,0]=(-1.19509443892)/(o+1.)+(0.225437856866)+(0.85928567141)*0.5**o
        ref[0,2,0,1]=(0.285764347754)/(o+1.)+(-0.383432127133)+(0.963095541446)*0.5**o
        ref[0,2,1,0]=(0.728010227911)/(o+1.)+(0.983790919896)+(-0.550403317717)*0.5**o
        ref[0,2,1,1]=(0.563748259036)/(o+1.)+(0.165164845927)+(-0.110710881481)*0.5**o
        ref[0,2,2,0]=(-0.361502606429)/(o+1.)+(0.0276145482611)+(-0.341115463028)*0.5**o
        ref[0,2,2,1]=(-0.293320207933)/(o+1.)+(0.0336757304385)+(-0.853103403872)*0.5**o
        ref[0,3,0,0]=(-0.410233956527)/(o+1.)+(-0.474281126245)+(0.569057885785)*0.5**o
        ref[0,3,0,1]=(-0.835861970115)/(o+1.)+(0.0173715483175)+(0.346902459481)*0.5**o
        ref[0,3,1,0]=(0.0830466456187)/(o+1.)+(-0.288204681495)+(-0.607109864241)*0.5**o
        ref[0,3,1,1]=(1.07074130823)/(o+1.)+(-0.0437983012867)+(0.159379549652)*0.5**o
        ref[0,3,2,0]=(0.172301049609)/(o+1.)+(-0.491724874152)+(-0.996925180895)*0.5**o
        ref[0,3,2,1]=(-0.24891941221)/(o+1.)+(-0.03935856269)+(-0.3030090126)*0.5**o
        ref[0,4,0,0]=(-0.778575122429)/(o+1.)+(0.213085222078)+(0.941383668349)*0.5**o
        ref[0,4,0,1]=(0.880801627174)/(o+1.)+(-0.454451723401)+(-0.193978040747)*0.5**o
        ref[0,4,1,0]=(-0.835604724882)/(o+1.)+(0.127240834083)+(-0.245156740036)*0.5**o
        ref[0,4,1,1]=(1.62336716324)/(o+1.)+(-0.101085918464)+(0.398884015839)*0.5**o
        ref[0,4,2,0]=(1.16805827702)/(o+1.)+(0.211232591688)+(-0.654729385702)*0.5**o
        ref[0,4,2,1]=(-0.305424138957)/(o+1.)+(-0.431076572693)+(-0.924903192969)*0.5**o
        ref[1,0,0,0]=(-0.481596247529)/(o+1.)+(0.769902632559)+(0.748050413171)*0.5**o
        ref[1,0,0,1]=(0.177388661252)/(o+1.)+(-0.118766442613)+(-0.457657277952)*0.5**o
        ref[1,0,1,0]=(0.709890860372)/(o+1.)+(0.0668010230727)+(0.658744950012)*0.5**o
        ref[1,0,1,1]=(-1.45685781381)/(o+1.)+(-0.178674426089)+(0.788456069079)*0.5**o
        ref[1,0,2,0]=(0.194388185612)/(o+1.)+(0.649177543063)+(0.896093783855)*0.5**o
        ref[1,0,2,1]=(0.181561279507)/(o+1.)+(-0.200618291902)+(-0.392507985678)*0.5**o
        ref[1,1,0,0]=(-0.884376814027)/(o+1.)+(0.280382736159)+(0.593213920451)*0.5**o
        ref[1,1,0,1]=(-0.250709602218)/(o+1.)+(-0.492104211211)+(0.818920586956)*0.5**o
        ref[1,1,1,0]=(0.0529723694625)/(o+1.)+(-0.502047650638)+(-0.115739819752)*0.5**o
        ref[1,1,1,1]=(0.215812202957)/(o+1.)+(-0.308796450292)+(0.954456105171)*0.5**o
        ref[1,1,2,0]=(-0.991481050902)/(o+1.)+(0.162626252838)+(-0.800147845717)*0.5**o
        ref[1,1,2,1]=(-0.433989756951)/(o+1.)+(0.362776336712)+(-0.259441100777)*0.5**o
        ref[1,2,0,0]=(-0.930287052624)/(o+1.)+(-0.347752763108)+(0.880525373413)*0.5**o
        ref[1,2,0,1]=(0.698674852861)/(o+1.)+(-0.15119613196)+(0.719004238051)*0.5**o
        ref[1,2,1,0]=(-0.777216821365)/(o+1.)+(0.436336076297)+(0.617741643399)*0.5**o
        ref[1,2,1,1]=(1.09595223135)/(o+1.)+(0.978251130983)+(0.804336219412)*0.5**o
        ref[1,2,2,0]=(-0.651012630849)/(o+1.)+(-1.04894606664)+(0.43528198181)*0.5**o
        ref[1,2,2,1]=(-1.5227353883)/(o+1.)+(0.670019307557)+(-0.745922598001)*0.5**o
        ref[1,3,0,0]=(-0.750195637997)/(o+1.)+(0.231864245348)+(-0.118356142533)*0.5**o
        ref[1,3,0,1]=(1.17136226596)/(o+1.)+(-0.114976421389)+(0.395383920468)*0.5**o
        ref[1,3,1,0]=(-0.108070636769)/(o+1.)+(0.279145429234)+(-0.352562621327)*0.5**o
        ref[1,3,1,1]=(-0.144223481714)/(o+1.)+(-0.472954352493)+(-0.0903026345013)*0.5**o
        ref[1,3,2,0]=(-0.487564307786)/(o+1.)+(0.636973351449)+(0.758622537094)*0.5**o
        ref[1,3,2,1]=(0.390860965954)/(o+1.)+(-0.36647678165)+(-0.66980953377)*0.5**o
        ref[1,4,0,0]=(-1.04824984837)/(o+1.)+(0.213932231728)+(0.823722790054)*0.5**o
        ref[1,4,0,1]=(-0.0984653601336)/(o+1.)+(0.07867877705)+(-0.741527483773)*0.5**o
        ref[1,4,1,0]=(-0.146287867383)/(o+1.)+(-0.908951868158)+(0.425691754878)*0.5**o
        ref[1,4,1,1]=(-0.94160416554)/(o+1.)+(0.0459877010016)+(-0.869872381431)*0.5**o
        ref[1,4,2,0]=(0.0115742858952)/(o+1.)+(-0.222488399041)+(0.687392452389)*0.5**o
        ref[1,4,2,1]=(-0.956561936575)/(o+1.)+(-0.59770879042)+(-0.0613652834159)*0.5**o
        ref[2,0,0,0]=(1.83987381729)/(o+1.)+(0.956579317963)+(0.862634531274)*0.5**o
        ref[2,0,0,1]=(0.105446127421)/(o+1.)+(0.180661881191)+(0.186629121375)*0.5**o
        ref[2,0,1,0]=(0.0212167440766)/(o+1.)+(-0.337825588888)+(-0.318943328859)*0.5**o
        ref[2,0,1,1]=(0.744834715631)/(o+1.)+(-0.350058416663)+(0.391466726104)*0.5**o
        ref[2,0,2,0]=(-0.788776613856)/(o+1.)+(0.233776806014)+(0.400043748043)*0.5**o
        ref[2,0,2,1]=(-0.95346976307)/(o+1.)+(0.0274998652324)+(-0.967971238003)*0.5**o
        ref[2,1,0,0]=(-0.536051408942)/(o+1.)+(-0.208498016994)+(0.590315312233)*0.5**o
        ref[2,1,0,1]=(-1.04158508788)/(o+1.)+(-0.397885793101)+(0.0127762955396)*0.5**o
        ref[2,1,1,0]=(0.529914207253)/(o+1.)+(-0.547503443987)+(-0.338873793431)*0.5**o
        ref[2,1,1,1]=(0.62298670274)/(o+1.)+(0.449860048786)+(0.352800726478)*0.5**o
        ref[2,1,2,0]=(0.965292650595)/(o+1.)+(0.00980498639)+(-0.261085873742)*0.5**o
        ref[2,1,2,1]=(0.403285360803)/(o+1.)+(0.369160592269)+(0.315744874774)*0.5**o
        ref[2,2,0,0]=(-0.457764884724)/(o+1.)+(0.563657776706)+(-0.791018976541)*0.5**o
        ref[2,2,0,1]=(0.188151274044)/(o+1.)+(-0.476470545911)+(0.911736564588)*0.5**o
        ref[2,2,1,0]=(-0.558515085739)/(o+1.)+(-0.445583796553)+(-0.533721159994)*0.5**o
        ref[2,2,1,1]=(-0.431992337265)/(o+1.)+(-0.0957726221961)+(0.64441532924)*0.5**o
        ref[2,2,2,0]=(-0.159056461947)/(o+1.)+(-0.0326864536156)+(0.484856087693)*0.5**o
        ref[2,2,2,1]=(0.533116456037)/(o+1.)+(1.10197257725)+(0.0353311991352)*0.5**o
        ref[2,3,0,0]=(0.0570625147023)/(o+1.)+(0.74170779854)+(0.851019670777)*0.5**o
        ref[2,3,0,1]=(-0.495059904832)/(o+1.)+(-0.169954912181)+(-0.761839138623)*0.5**o
        ref[2,3,1,0]=(0.482375173314)/(o+1.)+(-0.19374838651)+(0.706371811996)*0.5**o
        ref[2,3,1,1]=(0.12928100721)/(o+1.)+(-0.317327862062)+(-0.460946532077)*0.5**o
        ref[2,3,2,0]=(-0.234362084763)/(o+1.)+(0.423486279181)+(-0.93141377227)*0.5**o
        ref[2,3,2,1]=(1.25586693002)/(o+1.)+(-0.127551329993)+(0.56508002092)*0.5**o
        ref[2,4,0,0]=(0.45235297749)/(o+1.)+(-0.0217537364788)+(-0.640413752615)*0.5**o
        ref[2,4,0,1]=(0.936809207384)/(o+1.)+(-0.151519610695)+(-0.464139056331)*0.5**o
        ref[2,4,1,0]=(0.867693208863)/(o+1.)+(0.105683574314)+(-0.974513946233)*0.5**o
        ref[2,4,1,1]=(-1.27821575101)/(o+1.)+(0.463163847311)+(-0.0852880510814)*0.5**o
        ref[2,4,2,0]=(-0.846224501366)/(o+1.)+(0.432619461409)+(0.492845548002)*0.5**o
        ref[2,4,2,1]=(-1.48923229995)/(o+1.)+(-0.505747396375)+(0.00916577472118)*0.5**o
        ref[3,0,0,0]=(-0.959215731121)/(o+1.)+(-0.904609108587)+(-0.178668510268)*0.5**o
        ref[3,0,0,1]=(-1.53181231865)/(o+1.)+(-0.0758543610408)+(0.878717376536)*0.5**o
        ref[3,0,1,0]=(1.18643238831)/(o+1.)+(-0.288825159097)+(-0.341112240015)*0.5**o
        ref[3,0,1,1]=(-0.0675694917031)/(o+1.)+(-0.168955703616)+(0.275309686724)*0.5**o
        ref[3,0,2,0]=(-0.663269860133)/(o+1.)+(-0.235102421885)+(-0.573480403148)*0.5**o
        ref[3,0,2,1]=(1.0969289778)/(o+1.)+(0.46603172821)+(0.267894209932)*0.5**o
        ref[3,1,0,0]=(-0.716870513559)/(o+1.)+(0.712039409283)+(0.483098591635)*0.5**o
        ref[3,1,0,1]=(0.0328531999411)/(o+1.)+(0.379319450988)+(0.311973567308)*0.5**o
        ref[3,1,1,0]=(1.08648208064)/(o+1.)+(-0.196831959338)+(-0.452172980519)*0.5**o
        ref[3,1,1,1]=(0.0813181502612)/(o+1.)+(-0.227364064527)+(-0.0225656152548)*0.5**o
        ref[3,1,2,0]=(-0.271000454408)/(o+1.)+(-0.765711687155)+(0.868748696531)*0.5**o
        ref[3,1,2,1]=(1.60569646253)/(o+1.)+(0.137437739736)+(-0.00288127709197)*0.5**o
        ref[3,2,0,0]=(-1.25887089059)/(o+1.)+(-0.525842290945)+(0.455112921376)*0.5**o
        ref[3,2,0,1]=(-1.63787880725)/(o+1.)+(0.00214706725405)+(-0.166789085722)*0.5**o
        ref[3,2,1,0]=(-1.82111307894)/(o+1.)+(0.561806270366)+(0.388444921454)*0.5**o
        ref[3,2,1,1]=(-0.489259906414)/(o+1.)+(0.582548398381)+(0.417676141947)*0.5**o
        ref[3,2,2,0]=(-0.0674375340391)/(o+1.)+(-0.00345549227349)+(0.362946003887)*0.5**o
        ref[3,2,2,1]=(0.662780682001)/(o+1.)+(-0.15453848218)+(0.806022459534)*0.5**o
        ref[3,3,0,0]=(-1.15795406515)/(o+1.)+(0.0926725746467)+(0.289849779118)*0.5**o
        ref[3,3,0,1]=(1.27520269547)/(o+1.)+(-0.286529818361)+(0.672556345847)*0.5**o
        ref[3,3,1,0]=(-0.343290651469)/(o+1.)+(-0.0269962100904)+(-0.845847188189)*0.5**o
        ref[3,3,1,1]=(0.497715053371)/(o+1.)+(-0.158577177766)+(-0.909200821866)*0.5**o
        ref[3,3,2,0]=(-0.952431538662)/(o+1.)+(0.0816482063045)+(-0.580984656952)*0.5**o
        ref[3,3,2,1]=(-0.176227469799)/(o+1.)+(0.0456157853688)+(-0.0677878495767)*0.5**o
        ref[3,4,0,0]=(0.976460483802)/(o+1.)+(0.347595506011)+(0.766866468871)*0.5**o
        ref[3,4,0,1]=(0.0358170327662)/(o+1.)+(-0.0991456259778)+(0.931548717473)*0.5**o
        ref[3,4,1,0]=(0.114544318514)/(o+1.)+(-0.201014865792)+(-0.272213093635)*0.5**o
        ref[3,4,1,1]=(0.554509925385)/(o+1.)+(1.23219332557)+(0.237338557994)*0.5**o
        ref[3,4,2,0]=(0.0803465444931)/(o+1.)+(-0.518257569997)+(-0.681164419371)*0.5**o
        ref[3,4,2,1]=(-0.0443609570522)/(o+1.)+(0.572813368096)+(0.552725372202)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_FunctionOnContactZero_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactZero

      assumptions: FunctionOnContactZero(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=FunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.114918917602)*x[0]**o+(0.304834778607)*x[0]+(0.704057176139)*x[1]**o+(-0.0664624375434)*x[1]
        ref=(0.704057176139)/(o+1.)+(0.119186170532)+(-0.114918917602)*0.5**o
      else:
        arg=(-0.134961573162)*x[0]**o+(0.585433861814)*x[0]+(0.209596676649)*x[1]**o+(0.870867416311)*x[1]+(-0.902854033973)*x[2]**o+(-0.812135783993)*x[2]
        ref=(-0.693257357324)/(o+1.)+(0.322082747066)+(-0.134961573162)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_FunctionOnContactZero_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactZero

      assumptions: FunctionOnContactZero(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=FunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.441677901994)*x[0]**o+(0.0234076994082)*x[0]+(-0.520465368981)*x[1]**o+(-0.668717137452)*x[1]
        arg[1]=(0.0637451453664)*x[0]**o+(0.610543665309)*x[0]+(0.639008465664)*x[1]**o+(-0.460500756968)*x[1]
        ref[0]=(-0.520465368981)/(o+1.)+(-0.322654719022)+(0.441677901994)*0.5**o
        ref[1]=(0.639008465664)/(o+1.)+(0.0750214541708)+(0.0637451453664)*0.5**o
      else:
        arg[0]=(0.207042421638)*x[0]**o+(0.586067682859)*x[0]+(-0.731189752251)*x[1]**o+(0.439755444977)*x[1]+(-0.215027146515)*x[2]**o+(-0.22109699198)*x[2]
        arg[1]=(-0.482897746923)*x[0]**o+(-0.262848758384)*x[0]+(-0.130231246709)*x[1]**o+(0.494143899685)*x[1]+(0.871226664446)*x[2]**o+(-0.45136577521)*x[2]
        ref[0]=(-0.946216898767)/(o+1.)+(0.402363067928)+(0.207042421638)*0.5**o
        ref[1]=(0.740995417737)/(o+1.)+(-0.110035316955)+(-0.482897746923)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_FunctionOnContactZero_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactZero

      assumptions: FunctionOnContactZero(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=FunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(0.418217706031)*x[0]**o+(0.817101274679)*x[0]+(0.359227724634)*x[1]**o+(0.106952446872)*x[1]
        arg[0,1]=(0.735477258623)*x[0]**o+(-0.181692404492)*x[0]+(0.118248218763)*x[1]**o+(0.910033225874)*x[1]
        arg[0,2]=(-0.370301174151)*x[0]**o+(0.421589410528)*x[0]+(-0.91974297037)*x[1]**o+(0.676441845707)*x[1]
        arg[0,3]=(0.170535646217)*x[0]**o+(0.650458506701)*x[0]+(-0.431898202846)*x[1]**o+(-0.89133664224)*x[1]
        arg[0,4]=(-0.288793192482)*x[0]**o+(-0.752066662422)*x[0]+(0.279565184651)*x[1]**o+(-0.20569858505)*x[1]
        arg[1,0]=(-0.831478902729)*x[0]**o+(-0.545347589281)*x[0]+(-0.315738709444)*x[1]**o+(-0.407156023524)*x[1]
        arg[1,1]=(0.119396512736)*x[0]**o+(0.56655186098)*x[0]+(0.798206239231)*x[1]**o+(0.0342467197178)*x[1]
        arg[1,2]=(-0.264721761214)*x[0]**o+(-0.7583989175)*x[0]+(-0.350411882147)*x[1]**o+(-0.556879396863)*x[1]
        arg[1,3]=(0.976029513102)*x[0]**o+(-0.117830794452)*x[0]+(0.318067419468)*x[1]**o+(-0.496129688948)*x[1]
        arg[1,4]=(0.462797018845)*x[0]**o+(-0.214502973992)*x[0]+(0.155052684044)*x[1]**o+(0.264794267973)*x[1]
        arg[2,0]=(0.193892132778)*x[0]**o+(-0.440406597181)*x[0]+(0.527330868842)*x[1]**o+(0.856602716155)*x[1]
        arg[2,1]=(0.0418968976051)*x[0]**o+(0.944283628708)*x[0]+(0.923548919008)*x[1]**o+(0.958011594347)*x[1]
        arg[2,2]=(0.744026460715)*x[0]**o+(0.238896690247)*x[0]+(0.968891312128)*x[1]**o+(0.0636831438113)*x[1]
        arg[2,3]=(-0.289048983574)*x[0]**o+(0.897469203226)*x[0]+(0.948131139828)*x[1]**o+(0.821749060686)*x[1]
        arg[2,4]=(-0.903766144138)*x[0]**o+(-0.453139974489)*x[0]+(-0.715818671181)*x[1]**o+(-0.712367492971)*x[1]
        arg[3,0]=(0.315277263598)*x[0]**o+(-0.676815102073)*x[0]+(0.428704415911)*x[1]**o+(0.54703485039)*x[1]
        arg[3,1]=(-0.301882901745)*x[0]**o+(0.888822293672)*x[0]+(-0.496204235393)*x[1]**o+(0.210598259748)*x[1]
        arg[3,2]=(0.568071178492)*x[0]**o+(-0.552863936415)*x[0]+(0.758619958752)*x[1]**o+(0.520428540448)*x[1]
        arg[3,3]=(0.477425351692)*x[0]**o+(-0.427405588029)*x[0]+(0.197959279245)*x[1]**o+(0.328047998649)*x[1]
        arg[3,4]=(-0.869704203328)*x[0]**o+(-0.0825164966999)*x[0]+(0.779325592334)*x[1]**o+(0.419916607608)*x[1]
        ref[0,0]=(0.359227724634)/(o+1.)+(0.462026860776)+(0.418217706031)*0.5**o
        ref[0,1]=(0.118248218763)/(o+1.)+(0.364170410691)+(0.735477258623)*0.5**o
        ref[0,2]=(-0.91974297037)/(o+1.)+(0.549015628118)+(-0.370301174151)*0.5**o
        ref[0,3]=(-0.431898202846)/(o+1.)+(-0.120439067769)+(0.170535646217)*0.5**o
        ref[0,4]=(0.279565184651)/(o+1.)+(-0.478882623736)+(-0.288793192482)*0.5**o
        ref[1,0]=(-0.315738709444)/(o+1.)+(-0.476251806402)+(-0.831478902729)*0.5**o
        ref[1,1]=(0.798206239231)/(o+1.)+(0.300399290349)+(0.119396512736)*0.5**o
        ref[1,2]=(-0.350411882147)/(o+1.)+(-0.657639157182)+(-0.264721761214)*0.5**o
        ref[1,3]=(0.318067419468)/(o+1.)+(-0.3069802417)+(0.976029513102)*0.5**o
        ref[1,4]=(0.155052684044)/(o+1.)+(0.02514564699)+(0.462797018845)*0.5**o
        ref[2,0]=(0.527330868842)/(o+1.)+(0.208098059487)+(0.193892132778)*0.5**o
        ref[2,1]=(0.923548919008)/(o+1.)+(0.951147611528)+(0.0418968976051)*0.5**o
        ref[2,2]=(0.968891312128)/(o+1.)+(0.151289917029)+(0.744026460715)*0.5**o
        ref[2,3]=(0.948131139828)/(o+1.)+(0.859609131956)+(-0.289048983574)*0.5**o
        ref[2,4]=(-0.715818671181)/(o+1.)+(-0.58275373373)+(-0.903766144138)*0.5**o
        ref[3,0]=(0.428704415911)/(o+1.)+(-0.0648901258416)+(0.315277263598)*0.5**o
        ref[3,1]=(-0.496204235393)/(o+1.)+(0.54971027671)+(-0.301882901745)*0.5**o
        ref[3,2]=(0.758619958752)/(o+1.)+(-0.0162176979831)+(0.568071178492)*0.5**o
        ref[3,3]=(0.197959279245)/(o+1.)+(-0.0496787946897)+(0.477425351692)*0.5**o
        ref[3,4]=(0.779325592334)/(o+1.)+(0.168700055454)+(-0.869704203328)*0.5**o
      else:
        arg[0,0]=(0.204452789147)*x[0]**o+(0.411843111144)*x[0]+(-0.0502866367412)*x[1]**o+(0.957356261522)*x[1]+(0.173522462943)*x[2]**o+(-0.0194908382525)*x[2]
        arg[0,1]=(-0.49641528149)*x[0]**o+(0.158625922243)*x[0]+(0.670932733725)*x[1]**o+(-0.611829064862)*x[1]+(0.758373881782)*x[2]**o+(0.151236368375)*x[2]
        arg[0,2]=(0.577504765611)*x[0]**o+(0.872996499874)*x[0]+(-0.734000014722)*x[1]**o+(-0.890824538872)*x[1]+(-0.796786080271)*x[2]**o+(-0.389005562901)*x[2]
        arg[0,3]=(-0.933166619447)*x[0]**o+(-0.528627639194)*x[0]+(-0.828111326322)*x[1]**o+(0.528345547301)*x[1]+(0.99894455165)*x[2]**o+(0.472982583662)*x[2]
        arg[0,4]=(-0.495503045054)*x[0]**o+(0.629909217292)*x[0]+(0.901515784751)*x[1]**o+(-0.863668288904)*x[1]+(0.76892679355)*x[2]**o+(0.34730747389)*x[2]
        arg[1,0]=(0.303189803506)*x[0]**o+(-0.17543401741)*x[0]+(-0.283157585975)*x[1]**o+(0.950171529024)*x[1]+(0.0483247572497)*x[2]**o+(-0.604332298954)*x[2]
        arg[1,1]=(-0.225480407643)*x[0]**o+(-0.0521067696425)*x[0]+(-0.10259234894)*x[1]**o+(0.373551917005)*x[1]+(0.89047195372)*x[2]**o+(-0.292740890417)*x[2]
        arg[1,2]=(-0.476041156832)*x[0]**o+(0.916078681628)*x[0]+(-0.393887610676)*x[1]**o+(-0.651852504525)*x[1]+(0.611404175606)*x[2]**o+(0.336594647903)*x[2]
        arg[1,3]=(-0.415178048882)*x[0]**o+(0.145394549559)*x[0]+(0.680115054186)*x[1]**o+(0.261517386893)*x[1]+(0.781566117396)*x[2]**o+(-0.659634539391)*x[2]
        arg[1,4]=(0.874821341857)*x[0]**o+(0.317418381126)*x[0]+(0.051455923305)*x[1]**o+(0.670170143071)*x[1]+(-0.211902435336)*x[2]**o+(-0.175299130233)*x[2]
        arg[2,0]=(-0.372961546517)*x[0]**o+(0.199493150648)*x[0]+(-0.630239499285)*x[1]**o+(0.657412283442)*x[1]+(-0.82211915324)*x[2]**o+(-0.468612977158)*x[2]
        arg[2,1]=(-0.503168527887)*x[0]**o+(0.904406483201)*x[0]+(-0.206179628633)*x[1]**o+(0.0787026584997)*x[1]+(0.899590466467)*x[2]**o+(-0.893401768413)*x[2]
        arg[2,2]=(-0.860556408954)*x[0]**o+(0.39021140432)*x[0]+(-0.66425383352)*x[1]**o+(0.287711780417)*x[1]+(0.776884760126)*x[2]**o+(-0.115905192334)*x[2]
        arg[2,3]=(0.461359695993)*x[0]**o+(0.997877835019)*x[0]+(0.0513514112685)*x[1]**o+(-0.904812988908)*x[1]+(-0.903222331838)*x[2]**o+(-0.470185679041)*x[2]
        arg[2,4]=(-0.646560730528)*x[0]**o+(0.10038324067)*x[0]+(0.426753245517)*x[1]**o+(-0.109889624344)*x[1]+(-0.20158193753)*x[2]**o+(0.014258531224)*x[2]
        arg[3,0]=(0.364407704271)*x[0]**o+(0.840519040103)*x[0]+(0.528721642544)*x[1]**o+(-0.853679853614)*x[1]+(0.079836122575)*x[2]**o+(-0.439039178308)*x[2]
        arg[3,1]=(-0.0726491927955)*x[0]**o+(0.217804431494)*x[0]+(-0.454354285545)*x[1]**o+(0.390539936003)*x[1]+(-0.989667912688)*x[2]**o+(-0.0674079606774)*x[2]
        arg[3,2]=(0.62239422825)*x[0]**o+(0.998733716844)*x[0]+(0.796258161536)*x[1]**o+(0.618545466026)*x[1]+(-0.243188896911)*x[2]**o+(-0.677687584224)*x[2]
        arg[3,3]=(0.444073180258)*x[0]**o+(-0.764729326792)*x[0]+(-0.823491453567)*x[1]**o+(0.985393963131)*x[1]+(0.623800470977)*x[2]**o+(0.947328422513)*x[2]
        arg[3,4]=(0.686438920325)*x[0]**o+(0.789760134176)*x[0]+(-0.26011082782)*x[1]**o+(0.0692377043339)*x[1]+(-0.3712894236)*x[2]**o+(-0.542974191103)*x[2]
        ref[0,0]=(0.123235826202)/(o+1.)+(0.674854267207)+(0.204452789147)*0.5**o
        ref[0,1]=(1.42930661551)/(o+1.)+(-0.150983387122)+(-0.49641528149)*0.5**o
        ref[0,2]=(-1.53078609499)/(o+1.)+(-0.203416800949)+(0.577504765611)*0.5**o
        ref[0,3]=(0.170833225328)/(o+1.)+(0.236350245885)+(-0.933166619447)*0.5**o
        ref[0,4]=(1.6704425783)/(o+1.)+(0.0567742011389)+(-0.495503045054)*0.5**o
        ref[1,0]=(-0.234832828725)/(o+1.)+(0.0852026063299)+(0.303189803506)*0.5**o
        ref[1,1]=(0.787879604779)/(o+1.)+(0.0143521284725)+(-0.225480407643)*0.5**o
        ref[1,2]=(0.21751656493)/(o+1.)+(0.300410412503)+(-0.476041156832)*0.5**o
        ref[1,3]=(1.46168117158)/(o+1.)+(-0.126361301469)+(-0.415178048882)*0.5**o
        ref[1,4]=(-0.160446512031)/(o+1.)+(0.406144696982)+(0.874821341857)*0.5**o
        ref[2,0]=(-1.45235865253)/(o+1.)+(0.194146228466)+(-0.372961546517)*0.5**o
        ref[2,1]=(0.693410837835)/(o+1.)+(0.0448536866438)+(-0.503168527887)*0.5**o
        ref[2,2]=(0.112630926606)/(o+1.)+(0.281008996202)+(-0.860556408954)*0.5**o
        ref[2,3]=(-0.85187092057)/(o+1.)+(-0.188560416464)+(0.461359695993)*0.5**o
        ref[2,4]=(0.225171307987)/(o+1.)+(0.00237607377473)+(-0.646560730528)*0.5**o
        ref[3,0]=(0.608557765119)/(o+1.)+(-0.22609999591)+(0.364407704271)*0.5**o
        ref[3,1]=(-1.44402219823)/(o+1.)+(0.27046820341)+(-0.0726491927955)*0.5**o
        ref[3,2]=(0.553069264625)/(o+1.)+(0.469795799323)+(0.62239422825)*0.5**o
        ref[3,3]=(-0.19969098259)/(o+1.)+(0.583996529426)+(0.444073180258)*0.5**o
        ref[3,4]=(-0.631400251421)/(o+1.)+(0.158011823704)+(0.686438920325)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_FunctionOnContactZero_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnContactZero

      assumptions: FunctionOnContactZero(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=FunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.963014866001)*x[0]**o+(-0.592091539519)*x[0]+(0.397897510985)*x[1]**o+(0.485612885034)*x[1]
        arg[0,0,1]=(-0.764849027936)*x[0]**o+(0.936217923087)*x[0]+(-0.6256860991)*x[1]**o+(0.162359755936)*x[1]
        arg[0,1,0]=(-0.269607090458)*x[0]**o+(0.197100067821)*x[0]+(-0.216513043667)*x[1]**o+(0.143038940436)*x[1]
        arg[0,1,1]=(-0.432206914977)*x[0]**o+(0.177851208963)*x[0]+(0.0477652910843)*x[1]**o+(0.278333389583)*x[1]
        arg[1,0,0]=(-0.123940629628)*x[0]**o+(0.818553665062)*x[0]+(-0.168322144522)*x[1]**o+(0.485324994762)*x[1]
        arg[1,0,1]=(0.593609516922)*x[0]**o+(0.530508390905)*x[0]+(-0.938421636915)*x[1]**o+(-0.686459217917)*x[1]
        arg[1,1,0]=(-0.874022523162)*x[0]**o+(0.694500749056)*x[0]+(-0.650951516798)*x[1]**o+(0.45826621855)*x[1]
        arg[1,1,1]=(-0.259058106634)*x[0]**o+(0.0837282545694)*x[0]+(0.465734884341)*x[1]**o+(0.185558060279)*x[1]
        arg[2,0,0]=(0.346292866419)*x[0]**o+(0.982902010078)*x[0]+(0.0297606445334)*x[1]**o+(0.339043518987)*x[1]
        arg[2,0,1]=(0.749684718966)*x[0]**o+(-0.855799436867)*x[0]+(-0.186333913876)*x[1]**o+(-0.548968316245)*x[1]
        arg[2,1,0]=(-0.255759466052)*x[0]**o+(0.441882703647)*x[0]+(-0.0239807362322)*x[1]**o+(-0.236417762222)*x[1]
        arg[2,1,1]=(-0.0586694299212)*x[0]**o+(-0.912208422332)*x[0]+(-0.11610871626)*x[1]**o+(-0.512805786403)*x[1]
        arg[3,0,0]=(0.826027179564)*x[0]**o+(-0.589876767386)*x[0]+(-0.654858828463)*x[1]**o+(0.601656609023)*x[1]
        arg[3,0,1]=(-0.148714999203)*x[0]**o+(-0.409138795316)*x[0]+(0.290726299307)*x[1]**o+(-0.94009130345)*x[1]
        arg[3,1,0]=(0.63431415979)*x[0]**o+(-0.184553980231)*x[0]+(0.4087395755)*x[1]**o+(-0.694376406449)*x[1]
        arg[3,1,1]=(-0.000928514830108)*x[0]**o+(-0.782814501236)*x[0]+(0.177390244103)*x[1]**o+(0.698562703841)*x[1]
        arg[4,0,0]=(-0.939264907756)*x[0]**o+(0.191360117367)*x[0]+(-0.318750748728)*x[1]**o+(0.910713477162)*x[1]
        arg[4,0,1]=(-0.105186863713)*x[0]**o+(0.0319652211881)*x[0]+(0.118896230389)*x[1]**o+(-0.212677289177)*x[1]
        arg[4,1,0]=(-0.702838100428)*x[0]**o+(0.881695294345)*x[0]+(0.728231539571)*x[1]**o+(-0.192311604244)*x[1]
        arg[4,1,1]=(0.25197450886)*x[0]**o+(0.464857988125)*x[0]+(0.851996888259)*x[1]**o+(-0.408525083481)*x[1]
        arg[5,0,0]=(0.665627182446)*x[0]**o+(0.5367740452)*x[0]+(0.295800616406)*x[1]**o+(0.0620589956995)*x[1]
        arg[5,0,1]=(-0.37320168697)*x[0]**o+(0.072855481498)*x[0]+(0.539806161326)*x[1]**o+(-0.999091813027)*x[1]
        arg[5,1,0]=(-0.060845772709)*x[0]**o+(-0.931869215952)*x[0]+(-0.393477853475)*x[1]**o+(0.968288081996)*x[1]
        arg[5,1,1]=(-0.0768045781739)*x[0]**o+(-0.257318730887)*x[0]+(0.781473167319)*x[1]**o+(-0.458870794544)*x[1]
        ref[0,0,0]=(0.397897510985)/(o+1.)+(-0.0532393272425)+(-0.963014866001)*0.5**o
        ref[0,0,1]=(-0.6256860991)/(o+1.)+(0.549288839511)+(-0.764849027936)*0.5**o
        ref[0,1,0]=(-0.216513043667)/(o+1.)+(0.170069504128)+(-0.269607090458)*0.5**o
        ref[0,1,1]=(0.0477652910843)/(o+1.)+(0.228092299273)+(-0.432206914977)*0.5**o
        ref[1,0,0]=(-0.168322144522)/(o+1.)+(0.651939329912)+(-0.123940629628)*0.5**o
        ref[1,0,1]=(-0.938421636915)/(o+1.)+(-0.0779754135063)+(0.593609516922)*0.5**o
        ref[1,1,0]=(-0.650951516798)/(o+1.)+(0.576383483803)+(-0.874022523162)*0.5**o
        ref[1,1,1]=(0.465734884341)/(o+1.)+(0.134643157424)+(-0.259058106634)*0.5**o
        ref[2,0,0]=(0.0297606445334)/(o+1.)+(0.660972764533)+(0.346292866419)*0.5**o
        ref[2,0,1]=(-0.186333913876)/(o+1.)+(-0.702383876556)+(0.749684718966)*0.5**o
        ref[2,1,0]=(-0.0239807362322)/(o+1.)+(0.102732470713)+(-0.255759466052)*0.5**o
        ref[2,1,1]=(-0.11610871626)/(o+1.)+(-0.712507104367)+(-0.0586694299212)*0.5**o
        ref[3,0,0]=(-0.654858828463)/(o+1.)+(0.0058899208184)+(0.826027179564)*0.5**o
        ref[3,0,1]=(0.290726299307)/(o+1.)+(-0.674615049383)+(-0.148714999203)*0.5**o
        ref[3,1,0]=(0.4087395755)/(o+1.)+(-0.43946519334)+(0.63431415979)*0.5**o
        ref[3,1,1]=(0.177390244103)/(o+1.)+(-0.0421258986973)+(-0.000928514830108)*0.5**o
        ref[4,0,0]=(-0.318750748728)/(o+1.)+(0.551036797264)+(-0.939264907756)*0.5**o
        ref[4,0,1]=(0.118896230389)/(o+1.)+(-0.0903560339946)+(-0.105186863713)*0.5**o
        ref[4,1,0]=(0.728231539571)/(o+1.)+(0.344691845051)+(-0.702838100428)*0.5**o
        ref[4,1,1]=(0.851996888259)/(o+1.)+(0.0281664523221)+(0.25197450886)*0.5**o
        ref[5,0,0]=(0.295800616406)/(o+1.)+(0.29941652045)+(0.665627182446)*0.5**o
        ref[5,0,1]=(0.539806161326)/(o+1.)+(-0.463118165764)+(-0.37320168697)*0.5**o
        ref[5,1,0]=(-0.393477853475)/(o+1.)+(0.0182094330217)+(-0.060845772709)*0.5**o
        ref[5,1,1]=(0.781473167319)/(o+1.)+(-0.358094762716)+(-0.0768045781739)*0.5**o
      else:
        arg[0,0,0]=(0.638389367885)*x[0]**o+(-0.77703387527)*x[0]+(0.951496212227)*x[1]**o+(-0.484305156823)*x[1]+(-0.582437996275)*x[2]**o+(-0.460130065965)*x[2]
        arg[0,0,1]=(0.859011160063)*x[0]**o+(0.999372374168)*x[0]+(-0.946049724069)*x[1]**o+(0.522184535496)*x[1]+(0.00820405492191)*x[2]**o+(0.769305042211)*x[2]
        arg[0,1,0]=(-0.998681640062)*x[0]**o+(0.916063974594)*x[0]+(0.0801858148899)*x[1]**o+(0.475476401258)*x[1]+(-0.805247247843)*x[2]**o+(-0.809331278141)*x[2]
        arg[0,1,1]=(0.1340261052)*x[0]**o+(0.843204972218)*x[0]+(-0.809433597166)*x[1]**o+(-0.947798215446)*x[1]+(0.268796347717)*x[2]**o+(-0.896620567629)*x[2]
        arg[1,0,0]=(0.251201824008)*x[0]**o+(0.762438451805)*x[0]+(0.892722429031)*x[1]**o+(0.146944927124)*x[1]+(0.00953973207152)*x[2]**o+(0.423262196414)*x[2]
        arg[1,0,1]=(-0.258843593921)*x[0]**o+(-0.950574260099)*x[0]+(-0.49096256134)*x[1]**o+(0.446238902975)*x[1]+(-0.880706204083)*x[2]**o+(-0.400075002454)*x[2]
        arg[1,1,0]=(-0.854529145157)*x[0]**o+(0.118165533114)*x[0]+(-0.913218271084)*x[1]**o+(0.991977570414)*x[1]+(0.626241160161)*x[2]**o+(0.667843914102)*x[2]
        arg[1,1,1]=(0.636818650335)*x[0]**o+(-0.174996100829)*x[0]+(0.77317621688)*x[1]**o+(0.369280208456)*x[1]+(-0.0130124322187)*x[2]**o+(0.857098751007)*x[2]
        arg[2,0,0]=(-0.415847479141)*x[0]**o+(0.675986357495)*x[0]+(0.704926614111)*x[1]**o+(-0.166544295134)*x[1]+(-0.00917226654826)*x[2]**o+(-0.896432190739)*x[2]
        arg[2,0,1]=(-0.882383192316)*x[0]**o+(0.11122730635)*x[0]+(-0.693587089789)*x[1]**o+(-0.0905457091132)*x[1]+(-0.610526000157)*x[2]**o+(-0.992542379681)*x[2]
        arg[2,1,0]=(-0.476987072283)*x[0]**o+(-0.450296923812)*x[0]+(0.905270434608)*x[1]**o+(0.976270820516)*x[1]+(-0.233422717373)*x[2]**o+(-0.641253735386)*x[2]
        arg[2,1,1]=(0.873832514958)*x[0]**o+(-0.478059419217)*x[0]+(-0.941650455606)*x[1]**o+(0.307250152442)*x[1]+(0.865160620819)*x[2]**o+(0.961340749451)*x[2]
        arg[3,0,0]=(-0.443837945036)*x[0]**o+(-0.688773058425)*x[0]+(-0.654988618952)*x[1]**o+(-0.259511611372)*x[1]+(-0.931332326494)*x[2]**o+(0.939379631125)*x[2]
        arg[3,0,1]=(-0.181860312567)*x[0]**o+(-0.229066064127)*x[0]+(-0.359408642857)*x[1]**o+(-0.47945907585)*x[1]+(0.312345402782)*x[2]**o+(0.559298232285)*x[2]
        arg[3,1,0]=(-0.372379819015)*x[0]**o+(0.308179383085)*x[0]+(0.123613160006)*x[1]**o+(0.191944822984)*x[1]+(0.950869988808)*x[2]**o+(0.229360419984)*x[2]
        arg[3,1,1]=(-0.803126271269)*x[0]**o+(0.833324260042)*x[0]+(-0.917001746509)*x[1]**o+(-0.649776906898)*x[1]+(0.49624769362)*x[2]**o+(0.414080802594)*x[2]
        arg[4,0,0]=(-0.951747333891)*x[0]**o+(0.5919442178)*x[0]+(0.121310623341)*x[1]**o+(-0.321939845724)*x[1]+(0.358051737083)*x[2]**o+(-0.222801347833)*x[2]
        arg[4,0,1]=(-0.59462534105)*x[0]**o+(0.943860617698)*x[0]+(0.787822231831)*x[1]**o+(-0.0913435772425)*x[1]+(0.831651946729)*x[2]**o+(-0.903604465879)*x[2]
        arg[4,1,0]=(0.68655463207)*x[0]**o+(-0.98918689213)*x[0]+(0.251300089098)*x[1]**o+(-0.317828131349)*x[1]+(-0.105934216764)*x[2]**o+(-0.874506361945)*x[2]
        arg[4,1,1]=(-0.80544616293)*x[0]**o+(-0.321523684723)*x[0]+(0.874455533992)*x[1]**o+(0.200912811442)*x[1]+(-0.646776981862)*x[2]**o+(-0.988728118812)*x[2]
        arg[5,0,0]=(0.413387141328)*x[0]**o+(0.349210032211)*x[0]+(0.80870689028)*x[1]**o+(-0.633296311752)*x[1]+(0.24621029112)*x[2]**o+(0.853398479355)*x[2]
        arg[5,0,1]=(0.398787817372)*x[0]**o+(0.662875239272)*x[0]+(-0.719962467558)*x[1]**o+(0.49562860294)*x[1]+(-0.694859272211)*x[2]**o+(-0.792616661463)*x[2]
        arg[5,1,0]=(-0.298838035128)*x[0]**o+(0.121578680468)*x[0]+(-0.931062588633)*x[1]**o+(-0.0575832984853)*x[1]+(-0.696140085453)*x[2]**o+(0.836738246058)*x[2]
        arg[5,1,1]=(-0.949626960081)*x[0]**o+(-0.449843145228)*x[0]+(-0.304260073311)*x[1]**o+(-0.258880519466)*x[1]+(0.880466679751)*x[2]**o+(0.97505579364)*x[2]
        ref[0,0,0]=(0.369058215952)/(o+1.)+(-0.860734549029)+(0.638389367885)*0.5**o
        ref[0,0,1]=(-0.937845669147)/(o+1.)+(1.14543097594)+(0.859011160063)*0.5**o
        ref[0,1,0]=(-0.725061432954)/(o+1.)+(0.291104548855)+(-0.998681640062)*0.5**o
        ref[0,1,1]=(-0.540637249448)/(o+1.)+(-0.500606905428)+(0.1340261052)*0.5**o
        ref[1,0,0]=(0.902262161102)/(o+1.)+(0.666322787672)+(0.251201824008)*0.5**o
        ref[1,0,1]=(-1.37166876542)/(o+1.)+(-0.452205179789)+(-0.258843593921)*0.5**o
        ref[1,1,0]=(-0.286977110923)/(o+1.)+(0.888993508815)+(-0.854529145157)*0.5**o
        ref[1,1,1]=(0.760163784662)/(o+1.)+(0.525691429317)+(0.636818650335)*0.5**o
        ref[2,0,0]=(0.695754347563)/(o+1.)+(-0.193495064189)+(-0.415847479141)*0.5**o
        ref[2,0,1]=(-1.30411308995)/(o+1.)+(-0.485930391222)+(-0.882383192316)*0.5**o
        ref[2,1,0]=(0.671847717235)/(o+1.)+(-0.0576399193408)+(-0.476987072283)*0.5**o
        ref[2,1,1]=(-0.0764898347868)/(o+1.)+(0.395265741338)+(0.873832514958)*0.5**o
        ref[3,0,0]=(-1.58632094545)/(o+1.)+(-0.00445251933618)+(-0.443837945036)*0.5**o
        ref[3,0,1]=(-0.0470632400747)/(o+1.)+(-0.0746134538458)+(-0.181860312567)*0.5**o
        ref[3,1,0]=(1.07448314881)/(o+1.)+(0.364742313027)+(-0.372379819015)*0.5**o
        ref[3,1,1]=(-0.420754052889)/(o+1.)+(0.298814077869)+(-0.803126271269)*0.5**o
        ref[4,0,0]=(0.479362360424)/(o+1.)+(0.0236015121217)+(-0.951747333891)*0.5**o
        ref[4,0,1]=(1.61947417856)/(o+1.)+(-0.0255437127116)+(-0.59462534105)*0.5**o
        ref[4,1,0]=(0.145365872334)/(o+1.)+(-1.09076069271)+(0.68655463207)*0.5**o
        ref[4,1,1]=(0.22767855213)/(o+1.)+(-0.554669496046)+(-0.80544616293)*0.5**o
        ref[5,0,0]=(1.0549171814)/(o+1.)+(0.284656099907)+(0.413387141328)*0.5**o
        ref[5,0,1]=(-1.41482173977)/(o+1.)+(0.182943590374)+(0.398787817372)*0.5**o
        ref[5,1,0]=(-1.62720267409)/(o+1.)+(0.450366814021)+(-0.298838035128)*0.5**o
        ref[5,1,1]=(0.576206606439)/(o+1.)+(0.133166064473)+(-0.949626960081)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactZero_fromData_FunctionOnContactZero_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnContactZero

      assumptions: FunctionOnContactZero(self.domain) exists
                   self.domain supports integral on FunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactZero(self.domain)
      w=FunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.392628958003)*x[0]**o+(0.186084293395)*x[0]+(0.965300679335)*x[1]**o+(0.0986282244747)*x[1]
        arg[0,0,0,1]=(-0.503312425492)*x[0]**o+(-0.205791466802)*x[0]+(0.384782163488)*x[1]**o+(-0.608371097114)*x[1]
        arg[0,0,1,0]=(0.660162511699)*x[0]**o+(-0.263141686561)*x[0]+(-0.765235012795)*x[1]**o+(-0.371233144611)*x[1]
        arg[0,0,1,1]=(-0.986935412732)*x[0]**o+(0.645433351743)*x[0]+(-0.451665346257)*x[1]**o+(-0.713815759872)*x[1]
        arg[0,0,2,0]=(0.632875178502)*x[0]**o+(-0.257542208821)*x[0]+(-0.695553902133)*x[1]**o+(-0.677920876645)*x[1]
        arg[0,0,2,1]=(0.102625648731)*x[0]**o+(0.511896902175)*x[0]+(0.345909088853)*x[1]**o+(-0.0609413722414)*x[1]
        arg[0,1,0,0]=(-0.0108106931425)*x[0]**o+(-0.595497517075)*x[0]+(0.332480141972)*x[1]**o+(-0.18131109924)*x[1]
        arg[0,1,0,1]=(0.220826593491)*x[0]**o+(-0.202422451222)*x[0]+(0.328480565578)*x[1]**o+(0.851432750557)*x[1]
        arg[0,1,1,0]=(0.22735402466)*x[0]**o+(0.51437836535)*x[0]+(0.292927451477)*x[1]**o+(-0.148121856023)*x[1]
        arg[0,1,1,1]=(0.495079875966)*x[0]**o+(0.582755117761)*x[0]+(-0.322061131008)*x[1]**o+(0.953437449986)*x[1]
        arg[0,1,2,0]=(0.968583889554)*x[0]**o+(-0.593253483731)*x[0]+(0.286037351205)*x[1]**o+(0.997314411263)*x[1]
        arg[0,1,2,1]=(0.361498191707)*x[0]**o+(0.633275777697)*x[0]+(0.891305263219)*x[1]**o+(-0.194468605638)*x[1]
        arg[0,2,0,0]=(0.185647566516)*x[0]**o+(0.101448861247)*x[0]+(0.0870608077911)*x[1]**o+(-0.86477908948)*x[1]
        arg[0,2,0,1]=(0.227262327826)*x[0]**o+(0.812586702875)*x[0]+(-0.0229924234496)*x[1]**o+(-0.323581477982)*x[1]
        arg[0,2,1,0]=(0.0405753322301)*x[0]**o+(0.891289727227)*x[0]+(0.957377329048)*x[1]**o+(-0.982119724379)*x[1]
        arg[0,2,1,1]=(-0.546046301701)*x[0]**o+(0.628511534929)*x[0]+(-0.628262263897)*x[1]**o+(-0.454315708999)*x[1]
        arg[0,2,2,0]=(-0.828455793175)*x[0]**o+(0.190147126828)*x[0]+(0.499595322279)*x[1]**o+(-0.0882297087937)*x[1]
        arg[0,2,2,1]=(0.926138085222)*x[0]**o+(0.402659001583)*x[0]+(-0.678432418122)*x[1]**o+(0.236054704896)*x[1]
        arg[0,3,0,0]=(-0.473066058658)*x[0]**o+(0.57622727181)*x[0]+(0.62665109546)*x[1]**o+(0.948409659533)*x[1]
        arg[0,3,0,1]=(0.776790978673)*x[0]**o+(-0.0244752180934)*x[0]+(-0.459252843816)*x[1]**o+(0.616203849528)*x[1]
        arg[0,3,1,0]=(0.34783214571)*x[0]**o+(0.806528120142)*x[0]+(0.445864469556)*x[1]**o+(-0.221961922099)*x[1]
        arg[0,3,1,1]=(0.0717966037463)*x[0]**o+(-0.434117388366)*x[0]+(0.577708844669)*x[1]**o+(-0.548519157145)*x[1]
        arg[0,3,2,0]=(-0.548573971799)*x[0]**o+(-0.926704595492)*x[0]+(-0.33936340707)*x[1]**o+(0.0687290934591)*x[1]
        arg[0,3,2,1]=(-0.28309240461)*x[0]**o+(0.12914675768)*x[0]+(-0.833323497891)*x[1]**o+(-0.577727985559)*x[1]
        arg[0,4,0,0]=(-0.105416428922)*x[0]**o+(0.41564239907)*x[0]+(0.146657173319)*x[1]**o+(-0.0141417271845)*x[1]
        arg[0,4,0,1]=(-0.876111094069)*x[0]**o+(0.491957669628)*x[0]+(0.534941970665)*x[1]**o+(-0.995507289274)*x[1]
        arg[0,4,1,0]=(-0.678996394303)*x[0]**o+(0.183159947613)*x[0]+(-0.739368558349)*x[1]**o+(0.274319401135)*x[1]
        arg[0,4,1,1]=(0.0472145581152)*x[0]**o+(-0.669737608207)*x[0]+(-0.651647023142)*x[1]**o+(0.351208329111)*x[1]
        arg[0,4,2,0]=(0.114936857967)*x[0]**o+(0.564688790898)*x[0]+(-0.524234284672)*x[1]**o+(-0.127615425724)*x[1]
        arg[0,4,2,1]=(0.568834117773)*x[0]**o+(-0.936513147599)*x[0]+(0.133122220811)*x[1]**o+(0.31510090512)*x[1]
        arg[1,0,0,0]=(-0.821241886235)*x[0]**o+(0.779734235239)*x[0]+(-0.132449009841)*x[1]**o+(-0.0179700618848)*x[1]
        arg[1,0,0,1]=(0.624372422235)*x[0]**o+(-0.752831611955)*x[0]+(-0.573597121344)*x[1]**o+(0.376463665377)*x[1]
        arg[1,0,1,0]=(-0.568749808355)*x[0]**o+(-0.697521100346)*x[0]+(-0.249786553024)*x[1]**o+(-0.624593568423)*x[1]
        arg[1,0,1,1]=(0.0765557360835)*x[0]**o+(-0.744005823966)*x[0]+(0.59246455477)*x[1]**o+(0.231310687774)*x[1]
        arg[1,0,2,0]=(0.909614864187)*x[0]**o+(0.350543076144)*x[0]+(0.0689432766231)*x[1]**o+(-0.88465698758)*x[1]
        arg[1,0,2,1]=(0.0610206203468)*x[0]**o+(0.620401328094)*x[0]+(0.378273922837)*x[1]**o+(0.91988744934)*x[1]
        arg[1,1,0,0]=(0.917249169051)*x[0]**o+(0.756656473206)*x[0]+(0.728786835113)*x[1]**o+(0.116994159781)*x[1]
        arg[1,1,0,1]=(0.331129708654)*x[0]**o+(-0.14342302905)*x[0]+(-0.418402258918)*x[1]**o+(-0.808676850609)*x[1]
        arg[1,1,1,0]=(-0.527408405279)*x[0]**o+(-0.671440239997)*x[0]+(0.989603613532)*x[1]**o+(0.364280392731)*x[1]
        arg[1,1,1,1]=(-0.189543145217)*x[0]**o+(0.14762392216)*x[0]+(0.11459047938)*x[1]**o+(-0.760477267267)*x[1]
        arg[1,1,2,0]=(0.385274935809)*x[0]**o+(0.527744200995)*x[0]+(-0.862654310338)*x[1]**o+(0.904005467598)*x[1]
        arg[1,1,2,1]=(-0.380421555378)*x[0]**o+(0.598300323394)*x[0]+(-0.541819816686)*x[1]**o+(0.446934572528)*x[1]
        arg[1,2,0,0]=(0.446781075767)*x[0]**o+(0.906830344178)*x[0]+(0.158734254777)*x[1]**o+(-0.722982586936)*x[1]
        arg[1,2,0,1]=(0.927811104869)*x[0]**o+(-0.446795135388)*x[0]+(-0.90301726295)*x[1]**o+(0.289530807979)*x[1]
        arg[1,2,1,0]=(0.741527109588)*x[0]**o+(-0.370029728344)*x[0]+(0.157695458386)*x[1]**o+(-0.620804465293)*x[1]
        arg[1,2,1,1]=(0.193242816889)*x[0]**o+(0.455789913444)*x[0]+(0.0491001091271)*x[1]**o+(-0.942127264151)*x[1]
        arg[1,2,2,0]=(0.220709813709)*x[0]**o+(-0.713472965425)*x[0]+(-0.736262418061)*x[1]**o+(0.341347605023)*x[1]
        arg[1,2,2,1]=(0.0034506429721)*x[0]**o+(0.397289753453)*x[0]+(-0.167013881353)*x[1]**o+(0.390364445974)*x[1]
        arg[1,3,0,0]=(0.321471168172)*x[0]**o+(-0.237896433618)*x[0]+(-0.199843421009)*x[1]**o+(0.811319091471)*x[1]
        arg[1,3,0,1]=(-0.0799622908037)*x[0]**o+(-0.66155883346)*x[0]+(0.0303112997369)*x[1]**o+(-0.199256497975)*x[1]
        arg[1,3,1,0]=(-0.12051943489)*x[0]**o+(0.674374603779)*x[0]+(0.335933608833)*x[1]**o+(0.366671959553)*x[1]
        arg[1,3,1,1]=(-0.587242845986)*x[0]**o+(-0.495652656487)*x[0]+(0.10609614346)*x[1]**o+(0.177210588935)*x[1]
        arg[1,3,2,0]=(0.947782336638)*x[0]**o+(-0.327283099142)*x[0]+(-0.0280648836948)*x[1]**o+(0.133334543825)*x[1]
        arg[1,3,2,1]=(-0.0887414134508)*x[0]**o+(0.640694455701)*x[0]+(0.266916366237)*x[1]**o+(0.345886606493)*x[1]
        arg[1,4,0,0]=(-0.138184872938)*x[0]**o+(0.658635897645)*x[0]+(-0.00526240442626)*x[1]**o+(0.800443113371)*x[1]
        arg[1,4,0,1]=(-0.549831917928)*x[0]**o+(-0.666733703837)*x[0]+(0.400982699808)*x[1]**o+(0.739099662817)*x[1]
        arg[1,4,1,0]=(-0.137046722563)*x[0]**o+(-0.818450480933)*x[0]+(-0.827665726293)*x[1]**o+(-0.286585607624)*x[1]
        arg[1,4,1,1]=(-0.474156202572)*x[0]**o+(-0.855381707793)*x[0]+(0.113318564035)*x[1]**o+(0.218410511108)*x[1]
        arg[1,4,2,0]=(0.18476112439)*x[0]**o+(-0.964533442641)*x[0]+(-0.968611931598)*x[1]**o+(0.0973026144203)*x[1]
        arg[1,4,2,1]=(-0.372995418552)*x[0]**o+(0.998322146553)*x[0]+(-0.384672070744)*x[1]**o+(0.47536977268)*x[1]
        arg[2,0,0,0]=(0.882400857222)*x[0]**o+(0.455324640128)*x[0]+(0.764297272234)*x[1]**o+(0.498669000791)*x[1]
        arg[2,0,0,1]=(0.894017590794)*x[0]**o+(0.217984119634)*x[0]+(-0.0934000967188)*x[1]**o+(-0.886960376026)*x[1]
        arg[2,0,1,0]=(-0.382158057826)*x[0]**o+(-0.897639065996)*x[0]+(0.819002513335)*x[1]**o+(-0.0739374877518)*x[1]
        arg[2,0,1,1]=(-0.259771272131)*x[0]**o+(-0.533540086977)*x[0]+(0.366917477923)*x[1]**o+(-0.751904525722)*x[1]
        arg[2,0,2,0]=(-0.33946164111)*x[0]**o+(-0.317441178182)*x[0]+(-0.857625945147)*x[1]**o+(-0.84647892719)*x[1]
        arg[2,0,2,1]=(0.153353093148)*x[0]**o+(0.832883645984)*x[0]+(-0.359476461654)*x[1]**o+(0.332390899654)*x[1]
        arg[2,1,0,0]=(0.780980225775)*x[0]**o+(0.619330950566)*x[0]+(0.194312746335)*x[1]**o+(0.595311976655)*x[1]
        arg[2,1,0,1]=(-0.334149973283)*x[0]**o+(-0.670178805858)*x[0]+(-0.313038325156)*x[1]**o+(0.793149618246)*x[1]
        arg[2,1,1,0]=(-0.832852528577)*x[0]**o+(-0.606634692926)*x[0]+(-0.0643816133251)*x[1]**o+(-0.663765987285)*x[1]
        arg[2,1,1,1]=(0.300975732963)*x[0]**o+(0.965431406497)*x[0]+(-0.618456119184)*x[1]**o+(0.459530421885)*x[1]
        arg[2,1,2,0]=(0.122326105776)*x[0]**o+(0.0940802696378)*x[0]+(-0.986852501828)*x[1]**o+(0.370428043236)*x[1]
        arg[2,1,2,1]=(0.447916985436)*x[0]**o+(-0.161926311978)*x[0]+(-0.0933563424166)*x[1]**o+(0.836290263421)*x[1]
        arg[2,2,0,0]=(0.032076347277)*x[0]**o+(-0.447517143643)*x[0]+(-0.053769157756)*x[1]**o+(-0.0242981207801)*x[1]
        arg[2,2,0,1]=(-0.403935703163)*x[0]**o+(-0.960750093184)*x[0]+(-0.287956277998)*x[1]**o+(-0.168614701082)*x[1]
        arg[2,2,1,0]=(-0.205077009492)*x[0]**o+(-0.409536289697)*x[0]+(0.359071620541)*x[1]**o+(-0.466455759025)*x[1]
        arg[2,2,1,1]=(0.443658057141)*x[0]**o+(-0.748681939527)*x[0]+(0.2153167916)*x[1]**o+(-0.475899799738)*x[1]
        arg[2,2,2,0]=(-0.463879681251)*x[0]**o+(0.758317222041)*x[0]+(0.608221916958)*x[1]**o+(-0.237798399085)*x[1]
        arg[2,2,2,1]=(0.533981754168)*x[0]**o+(0.590125912931)*x[0]+(0.8957267808)*x[1]**o+(0.294612478767)*x[1]
        arg[2,3,0,0]=(0.499387701403)*x[0]**o+(0.0379132293822)*x[0]+(0.717384215206)*x[1]**o+(0.754497698503)*x[1]
        arg[2,3,0,1]=(0.866500398882)*x[0]**o+(0.23678788012)*x[0]+(0.395132533716)*x[1]**o+(-0.451619930593)*x[1]
        arg[2,3,1,0]=(-0.879663354176)*x[0]**o+(-0.328937922978)*x[0]+(0.688771917581)*x[1]**o+(-0.87392051489)*x[1]
        arg[2,3,1,1]=(0.291214284356)*x[0]**o+(-0.783159133488)*x[0]+(0.240960084233)*x[1]**o+(0.339043900275)*x[1]
        arg[2,3,2,0]=(0.47064755192)*x[0]**o+(-0.42618815064)*x[0]+(-0.505342655776)*x[1]**o+(0.408218280069)*x[1]
        arg[2,3,2,1]=(0.775731405735)*x[0]**o+(-0.768750178591)*x[0]+(-0.896333275063)*x[1]**o+(0.286558848957)*x[1]
        arg[2,4,0,0]=(-0.7221248688)*x[0]**o+(0.612509823967)*x[0]+(0.436744528143)*x[1]**o+(-0.761502758764)*x[1]
        arg[2,4,0,1]=(-0.742652222385)*x[0]**o+(0.214811997988)*x[0]+(-0.031320962337)*x[1]**o+(-0.320134664789)*x[1]
        arg[2,4,1,0]=(-0.529575527815)*x[0]**o+(0.623770482091)*x[0]+(-0.50790148262)*x[1]**o+(0.629874274271)*x[1]
        arg[2,4,1,1]=(0.535631957883)*x[0]**o+(0.439578618993)*x[0]+(-0.770747718564)*x[1]**o+(0.0798696116058)*x[1]
        arg[2,4,2,0]=(0.81688202721)*x[0]**o+(-0.0301054197201)*x[0]+(0.613495031968)*x[1]**o+(-0.153619455295)*x[1]
        arg[2,4,2,1]=(0.493471708612)*x[0]**o+(0.423626757568)*x[0]+(0.093174141001)*x[1]**o+(0.307452108722)*x[1]
        arg[3,0,0,0]=(-0.0668630243284)*x[0]**o+(-0.577709692979)*x[0]+(0.984307685082)*x[1]**o+(-0.171487030152)*x[1]
        arg[3,0,0,1]=(-0.210452484293)*x[0]**o+(-0.130476372055)*x[0]+(-0.693596086089)*x[1]**o+(-0.692143724329)*x[1]
        arg[3,0,1,0]=(-0.34853285523)*x[0]**o+(-0.369649709966)*x[0]+(0.824648362105)*x[1]**o+(0.733046240399)*x[1]
        arg[3,0,1,1]=(0.861903045965)*x[0]**o+(0.609535692452)*x[0]+(0.681514895836)*x[1]**o+(0.66773275418)*x[1]
        arg[3,0,2,0]=(0.0197670342844)*x[0]**o+(-0.251645539228)*x[0]+(-0.0390293468405)*x[1]**o+(-0.00955605638373)*x[1]
        arg[3,0,2,1]=(-0.499093940395)*x[0]**o+(0.660356623112)*x[0]+(0.969057635866)*x[1]**o+(-0.821377317197)*x[1]
        arg[3,1,0,0]=(-0.955568800095)*x[0]**o+(0.557877698917)*x[0]+(0.829599262959)*x[1]**o+(0.933736802562)*x[1]
        arg[3,1,0,1]=(-0.910307638058)*x[0]**o+(-0.3287646946)*x[0]+(-0.777548579252)*x[1]**o+(-0.971744353969)*x[1]
        arg[3,1,1,0]=(0.22353820834)*x[0]**o+(0.904479243377)*x[0]+(-0.108768166204)*x[1]**o+(0.823767232517)*x[1]
        arg[3,1,1,1]=(0.371995768061)*x[0]**o+(0.747630377273)*x[0]+(0.931152588498)*x[1]**o+(-0.0154330268188)*x[1]
        arg[3,1,2,0]=(0.4567417369)*x[0]**o+(0.424951293892)*x[0]+(-0.14032259215)*x[1]**o+(0.603329614924)*x[1]
        arg[3,1,2,1]=(-0.855100213731)*x[0]**o+(-0.667832882495)*x[0]+(0.942530922378)*x[1]**o+(0.55861156259)*x[1]
        arg[3,2,0,0]=(0.17190936572)*x[0]**o+(-0.904098615762)*x[0]+(0.991497679624)*x[1]**o+(0.373020358381)*x[1]
        arg[3,2,0,1]=(-0.683587206983)*x[0]**o+(0.718257814718)*x[0]+(-0.730243933765)*x[1]**o+(-0.583892189979)*x[1]
        arg[3,2,1,0]=(0.051672736629)*x[0]**o+(-0.991172563807)*x[0]+(0.36605487466)*x[1]**o+(0.680719877794)*x[1]
        arg[3,2,1,1]=(0.438014662973)*x[0]**o+(0.172969450125)*x[0]+(-0.424515284177)*x[1]**o+(0.615315132547)*x[1]
        arg[3,2,2,0]=(-0.709499532674)*x[0]**o+(0.985097212621)*x[0]+(0.326610298255)*x[1]**o+(0.994701669258)*x[1]
        arg[3,2,2,1]=(-0.657647707844)*x[0]**o+(-0.683954483768)*x[0]+(-0.846657945161)*x[1]**o+(0.852308904063)*x[1]
        arg[3,3,0,0]=(0.21346320453)*x[0]**o+(-0.181665119768)*x[0]+(0.549866100727)*x[1]**o+(-0.0858045316871)*x[1]
        arg[3,3,0,1]=(0.249957470685)*x[0]**o+(0.537464773638)*x[0]+(0.0224330511636)*x[1]**o+(0.114623595638)*x[1]
        arg[3,3,1,0]=(0.166507029533)*x[0]**o+(0.232150195873)*x[0]+(0.119453667263)*x[1]**o+(-0.261110960642)*x[1]
        arg[3,3,1,1]=(0.358248720077)*x[0]**o+(-0.871911261059)*x[0]+(0.57743453064)*x[1]**o+(0.950322066335)*x[1]
        arg[3,3,2,0]=(-0.896899724447)*x[0]**o+(-0.898884767716)*x[0]+(0.63832012742)*x[1]**o+(-0.833800375634)*x[1]
        arg[3,3,2,1]=(0.425194254583)*x[0]**o+(0.161328467295)*x[0]+(0.496219405916)*x[1]**o+(0.380935896093)*x[1]
        arg[3,4,0,0]=(0.793902581592)*x[0]**o+(-0.33428511164)*x[0]+(-0.283510438468)*x[1]**o+(0.228913046074)*x[1]
        arg[3,4,0,1]=(0.372028980539)*x[0]**o+(-0.388909011671)*x[0]+(-0.259885631206)*x[1]**o+(-0.950910172897)*x[1]
        arg[3,4,1,0]=(-0.80489673786)*x[0]**o+(-0.828175420038)*x[0]+(0.895070489116)*x[1]**o+(-0.742272132103)*x[1]
        arg[3,4,1,1]=(0.272535126101)*x[0]**o+(0.902668974891)*x[0]+(0.399816667015)*x[1]**o+(-0.953186908789)*x[1]
        arg[3,4,2,0]=(0.0341775288582)*x[0]**o+(0.962241492639)*x[0]+(0.906020184876)*x[1]**o+(-0.400536255726)*x[1]
        arg[3,4,2,1]=(-0.745016464485)*x[0]**o+(-0.85241829747)*x[0]+(-0.648936544722)*x[1]**o+(-0.997030074218)*x[1]
        ref[0,0,0,0]=(0.965300679335)/(o+1.)+(0.142356258935)+(0.392628958003)*0.5**o
        ref[0,0,0,1]=(0.384782163488)/(o+1.)+(-0.407081281958)+(-0.503312425492)*0.5**o
        ref[0,0,1,0]=(-0.765235012795)/(o+1.)+(-0.317187415586)+(0.660162511699)*0.5**o
        ref[0,0,1,1]=(-0.451665346257)/(o+1.)+(-0.0341912040643)+(-0.986935412732)*0.5**o
        ref[0,0,2,0]=(-0.695553902133)/(o+1.)+(-0.467731542733)+(0.632875178502)*0.5**o
        ref[0,0,2,1]=(0.345909088853)/(o+1.)+(0.225477764967)+(0.102625648731)*0.5**o
        ref[0,1,0,0]=(0.332480141972)/(o+1.)+(-0.388404308158)+(-0.0108106931425)*0.5**o
        ref[0,1,0,1]=(0.328480565578)/(o+1.)+(0.324505149668)+(0.220826593491)*0.5**o
        ref[0,1,1,0]=(0.292927451477)/(o+1.)+(0.183128254664)+(0.22735402466)*0.5**o
        ref[0,1,1,1]=(-0.322061131008)/(o+1.)+(0.768096283874)+(0.495079875966)*0.5**o
        ref[0,1,2,0]=(0.286037351205)/(o+1.)+(0.202030463766)+(0.968583889554)*0.5**o
        ref[0,1,2,1]=(0.891305263219)/(o+1.)+(0.219403586029)+(0.361498191707)*0.5**o
        ref[0,2,0,0]=(0.0870608077911)/(o+1.)+(-0.381665114117)+(0.185647566516)*0.5**o
        ref[0,2,0,1]=(-0.0229924234496)/(o+1.)+(0.244502612446)+(0.227262327826)*0.5**o
        ref[0,2,1,0]=(0.957377329048)/(o+1.)+(-0.0454149985758)+(0.0405753322301)*0.5**o
        ref[0,2,1,1]=(-0.628262263897)/(o+1.)+(0.0870979129647)+(-0.546046301701)*0.5**o
        ref[0,2,2,0]=(0.499595322279)/(o+1.)+(0.0509587090171)+(-0.828455793175)*0.5**o
        ref[0,2,2,1]=(-0.678432418122)/(o+1.)+(0.319356853239)+(0.926138085222)*0.5**o
        ref[0,3,0,0]=(0.62665109546)/(o+1.)+(0.762318465671)+(-0.473066058658)*0.5**o
        ref[0,3,0,1]=(-0.459252843816)/(o+1.)+(0.295864315717)+(0.776790978673)*0.5**o
        ref[0,3,1,0]=(0.445864469556)/(o+1.)+(0.292283099022)+(0.34783214571)*0.5**o
        ref[0,3,1,1]=(0.577708844669)/(o+1.)+(-0.491318272755)+(0.0717966037463)*0.5**o
        ref[0,3,2,0]=(-0.33936340707)/(o+1.)+(-0.428987751016)+(-0.548573971799)*0.5**o
        ref[0,3,2,1]=(-0.833323497891)/(o+1.)+(-0.224290613939)+(-0.28309240461)*0.5**o
        ref[0,4,0,0]=(0.146657173319)/(o+1.)+(0.200750335943)+(-0.105416428922)*0.5**o
        ref[0,4,0,1]=(0.534941970665)/(o+1.)+(-0.251774809823)+(-0.876111094069)*0.5**o
        ref[0,4,1,0]=(-0.739368558349)/(o+1.)+(0.228739674374)+(-0.678996394303)*0.5**o
        ref[0,4,1,1]=(-0.651647023142)/(o+1.)+(-0.159264639548)+(0.0472145581152)*0.5**o
        ref[0,4,2,0]=(-0.524234284672)/(o+1.)+(0.218536682587)+(0.114936857967)*0.5**o
        ref[0,4,2,1]=(0.133122220811)/(o+1.)+(-0.31070612124)+(0.568834117773)*0.5**o
        ref[1,0,0,0]=(-0.132449009841)/(o+1.)+(0.380882086677)+(-0.821241886235)*0.5**o
        ref[1,0,0,1]=(-0.573597121344)/(o+1.)+(-0.188183973289)+(0.624372422235)*0.5**o
        ref[1,0,1,0]=(-0.249786553024)/(o+1.)+(-0.661057334385)+(-0.568749808355)*0.5**o
        ref[1,0,1,1]=(0.59246455477)/(o+1.)+(-0.256347568096)+(0.0765557360835)*0.5**o
        ref[1,0,2,0]=(0.0689432766231)/(o+1.)+(-0.267056955718)+(0.909614864187)*0.5**o
        ref[1,0,2,1]=(0.378273922837)/(o+1.)+(0.770144388717)+(0.0610206203468)*0.5**o
        ref[1,1,0,0]=(0.728786835113)/(o+1.)+(0.436825316493)+(0.917249169051)*0.5**o
        ref[1,1,0,1]=(-0.418402258918)/(o+1.)+(-0.47604993983)+(0.331129708654)*0.5**o
        ref[1,1,1,0]=(0.989603613532)/(o+1.)+(-0.153579923633)+(-0.527408405279)*0.5**o
        ref[1,1,1,1]=(0.11459047938)/(o+1.)+(-0.306426672553)+(-0.189543145217)*0.5**o
        ref[1,1,2,0]=(-0.862654310338)/(o+1.)+(0.715874834296)+(0.385274935809)*0.5**o
        ref[1,1,2,1]=(-0.541819816686)/(o+1.)+(0.522617447961)+(-0.380421555378)*0.5**o
        ref[1,2,0,0]=(0.158734254777)/(o+1.)+(0.0919238786215)+(0.446781075767)*0.5**o
        ref[1,2,0,1]=(-0.90301726295)/(o+1.)+(-0.0786321637044)+(0.927811104869)*0.5**o
        ref[1,2,1,0]=(0.157695458386)/(o+1.)+(-0.495417096819)+(0.741527109588)*0.5**o
        ref[1,2,1,1]=(0.0491001091271)/(o+1.)+(-0.243168675353)+(0.193242816889)*0.5**o
        ref[1,2,2,0]=(-0.736262418061)/(o+1.)+(-0.186062680201)+(0.220709813709)*0.5**o
        ref[1,2,2,1]=(-0.167013881353)/(o+1.)+(0.393827099713)+(0.0034506429721)*0.5**o
        ref[1,3,0,0]=(-0.199843421009)/(o+1.)+(0.286711328927)+(0.321471168172)*0.5**o
        ref[1,3,0,1]=(0.0303112997369)/(o+1.)+(-0.430407665717)+(-0.0799622908037)*0.5**o
        ref[1,3,1,0]=(0.335933608833)/(o+1.)+(0.520523281666)+(-0.12051943489)*0.5**o
        ref[1,3,1,1]=(0.10609614346)/(o+1.)+(-0.159221033776)+(-0.587242845986)*0.5**o
        ref[1,3,2,0]=(-0.0280648836948)/(o+1.)+(-0.0969742776586)+(0.947782336638)*0.5**o
        ref[1,3,2,1]=(0.266916366237)/(o+1.)+(0.493290531097)+(-0.0887414134508)*0.5**o
        ref[1,4,0,0]=(-0.00526240442626)/(o+1.)+(0.729539505508)+(-0.138184872938)*0.5**o
        ref[1,4,0,1]=(0.400982699808)/(o+1.)+(0.0361829794896)+(-0.549831917928)*0.5**o
        ref[1,4,1,0]=(-0.827665726293)/(o+1.)+(-0.552518044279)+(-0.137046722563)*0.5**o
        ref[1,4,1,1]=(0.113318564035)/(o+1.)+(-0.318485598343)+(-0.474156202572)*0.5**o
        ref[1,4,2,0]=(-0.968611931598)/(o+1.)+(-0.433615414111)+(0.18476112439)*0.5**o
        ref[1,4,2,1]=(-0.384672070744)/(o+1.)+(0.736845959617)+(-0.372995418552)*0.5**o
        ref[2,0,0,0]=(0.764297272234)/(o+1.)+(0.476996820459)+(0.882400857222)*0.5**o
        ref[2,0,0,1]=(-0.0934000967188)/(o+1.)+(-0.334488128196)+(0.894017590794)*0.5**o
        ref[2,0,1,0]=(0.819002513335)/(o+1.)+(-0.485788276874)+(-0.382158057826)*0.5**o
        ref[2,0,1,1]=(0.366917477923)/(o+1.)+(-0.64272230635)+(-0.259771272131)*0.5**o
        ref[2,0,2,0]=(-0.857625945147)/(o+1.)+(-0.581960052686)+(-0.33946164111)*0.5**o
        ref[2,0,2,1]=(-0.359476461654)/(o+1.)+(0.582637272819)+(0.153353093148)*0.5**o
        ref[2,1,0,0]=(0.194312746335)/(o+1.)+(0.607321463611)+(0.780980225775)*0.5**o
        ref[2,1,0,1]=(-0.313038325156)/(o+1.)+(0.0614854061938)+(-0.334149973283)*0.5**o
        ref[2,1,1,0]=(-0.0643816133251)/(o+1.)+(-0.635200340106)+(-0.832852528577)*0.5**o
        ref[2,1,1,1]=(-0.618456119184)/(o+1.)+(0.712480914191)+(0.300975732963)*0.5**o
        ref[2,1,2,0]=(-0.986852501828)/(o+1.)+(0.232254156437)+(0.122326105776)*0.5**o
        ref[2,1,2,1]=(-0.0933563424166)/(o+1.)+(0.337181975722)+(0.447916985436)*0.5**o
        ref[2,2,0,0]=(-0.053769157756)/(o+1.)+(-0.235907632211)+(0.032076347277)*0.5**o
        ref[2,2,0,1]=(-0.287956277998)/(o+1.)+(-0.564682397133)+(-0.403935703163)*0.5**o
        ref[2,2,1,0]=(0.359071620541)/(o+1.)+(-0.437996024361)+(-0.205077009492)*0.5**o
        ref[2,2,1,1]=(0.2153167916)/(o+1.)+(-0.612290869632)+(0.443658057141)*0.5**o
        ref[2,2,2,0]=(0.608221916958)/(o+1.)+(0.260259411478)+(-0.463879681251)*0.5**o
        ref[2,2,2,1]=(0.8957267808)/(o+1.)+(0.442369195849)+(0.533981754168)*0.5**o
        ref[2,3,0,0]=(0.717384215206)/(o+1.)+(0.396205463942)+(0.499387701403)*0.5**o
        ref[2,3,0,1]=(0.395132533716)/(o+1.)+(-0.107416025237)+(0.866500398882)*0.5**o
        ref[2,3,1,0]=(0.688771917581)/(o+1.)+(-0.601429218934)+(-0.879663354176)*0.5**o
        ref[2,3,1,1]=(0.240960084233)/(o+1.)+(-0.222057616607)+(0.291214284356)*0.5**o
        ref[2,3,2,0]=(-0.505342655776)/(o+1.)+(-0.00898493528542)+(0.47064755192)*0.5**o
        ref[2,3,2,1]=(-0.896333275063)/(o+1.)+(-0.241095664817)+(0.775731405735)*0.5**o
        ref[2,4,0,0]=(0.436744528143)/(o+1.)+(-0.0744964673986)+(-0.7221248688)*0.5**o
        ref[2,4,0,1]=(-0.031320962337)/(o+1.)+(-0.0526613334003)+(-0.742652222385)*0.5**o
        ref[2,4,1,0]=(-0.50790148262)/(o+1.)+(0.626822378181)+(-0.529575527815)*0.5**o
        ref[2,4,1,1]=(-0.770747718564)/(o+1.)+(0.2597241153)+(0.535631957883)*0.5**o
        ref[2,4,2,0]=(0.613495031968)/(o+1.)+(-0.0918624375077)+(0.81688202721)*0.5**o
        ref[2,4,2,1]=(0.093174141001)/(o+1.)+(0.365539433145)+(0.493471708612)*0.5**o
        ref[3,0,0,0]=(0.984307685082)/(o+1.)+(-0.374598361566)+(-0.0668630243284)*0.5**o
        ref[3,0,0,1]=(-0.693596086089)/(o+1.)+(-0.411310048192)+(-0.210452484293)*0.5**o
        ref[3,0,1,0]=(0.824648362105)/(o+1.)+(0.181698265217)+(-0.34853285523)*0.5**o
        ref[3,0,1,1]=(0.681514895836)/(o+1.)+(0.638634223316)+(0.861903045965)*0.5**o
        ref[3,0,2,0]=(-0.0390293468405)/(o+1.)+(-0.130600797806)+(0.0197670342844)*0.5**o
        ref[3,0,2,1]=(0.969057635866)/(o+1.)+(-0.0805103470423)+(-0.499093940395)*0.5**o
        ref[3,1,0,0]=(0.829599262959)/(o+1.)+(0.745807250739)+(-0.955568800095)*0.5**o
        ref[3,1,0,1]=(-0.777548579252)/(o+1.)+(-0.650254524284)+(-0.910307638058)*0.5**o
        ref[3,1,1,0]=(-0.108768166204)/(o+1.)+(0.864123237947)+(0.22353820834)*0.5**o
        ref[3,1,1,1]=(0.931152588498)/(o+1.)+(0.366098675227)+(0.371995768061)*0.5**o
        ref[3,1,2,0]=(-0.14032259215)/(o+1.)+(0.514140454408)+(0.4567417369)*0.5**o
        ref[3,1,2,1]=(0.942530922378)/(o+1.)+(-0.0546106599524)+(-0.855100213731)*0.5**o
        ref[3,2,0,0]=(0.991497679624)/(o+1.)+(-0.26553912869)+(0.17190936572)*0.5**o
        ref[3,2,0,1]=(-0.730243933765)/(o+1.)+(0.0671828123695)+(-0.683587206983)*0.5**o
        ref[3,2,1,0]=(0.36605487466)/(o+1.)+(-0.155226343006)+(0.051672736629)*0.5**o
        ref[3,2,1,1]=(-0.424515284177)/(o+1.)+(0.394142291336)+(0.438014662973)*0.5**o
        ref[3,2,2,0]=(0.326610298255)/(o+1.)+(0.98989944094)+(-0.709499532674)*0.5**o
        ref[3,2,2,1]=(-0.846657945161)/(o+1.)+(0.0841772101477)+(-0.657647707844)*0.5**o
        ref[3,3,0,0]=(0.549866100727)/(o+1.)+(-0.133734825728)+(0.21346320453)*0.5**o
        ref[3,3,0,1]=(0.0224330511636)/(o+1.)+(0.326044184638)+(0.249957470685)*0.5**o
        ref[3,3,1,0]=(0.119453667263)/(o+1.)+(-0.0144803823844)+(0.166507029533)*0.5**o
        ref[3,3,1,1]=(0.57743453064)/(o+1.)+(0.0392054026384)+(0.358248720077)*0.5**o
        ref[3,3,2,0]=(0.63832012742)/(o+1.)+(-0.866342571675)+(-0.896899724447)*0.5**o
        ref[3,3,2,1]=(0.496219405916)/(o+1.)+(0.271132181694)+(0.425194254583)*0.5**o
        ref[3,4,0,0]=(-0.283510438468)/(o+1.)+(-0.0526860327828)+(0.793902581592)*0.5**o
        ref[3,4,0,1]=(-0.259885631206)/(o+1.)+(-0.669909592284)+(0.372028980539)*0.5**o
        ref[3,4,1,0]=(0.895070489116)/(o+1.)+(-0.78522377607)+(-0.80489673786)*0.5**o
        ref[3,4,1,1]=(0.399816667015)/(o+1.)+(-0.0252589669489)+(0.272535126101)*0.5**o
        ref[3,4,2,0]=(0.906020184876)/(o+1.)+(0.280852618457)+(0.0341775288582)*0.5**o
        ref[3,4,2,1]=(-0.648936544722)/(o+1.)+(-0.924724185844)+(-0.745016464485)*0.5**o
      else:
        arg[0,0,0,0]=(-0.891975745754)*x[0]**o+(0.678331209591)*x[0]+(-0.450022305538)*x[1]**o+(0.521272586024)*x[1]+(-0.28868708769)*x[2]**o+(-0.631617544339)*x[2]
        arg[0,0,0,1]=(0.428698930349)*x[0]**o+(-0.856251438077)*x[0]+(0.915766367729)*x[1]**o+(-0.35019463199)*x[1]+(-0.0029340113102)*x[2]**o+(-0.394458505835)*x[2]
        arg[0,0,1,0]=(0.974069118653)*x[0]**o+(-0.674028781962)*x[0]+(0.489718068973)*x[1]**o+(-0.471323892144)*x[1]+(0.11469271678)*x[2]**o+(-0.264584383354)*x[2]
        arg[0,0,1,1]=(0.813412823502)*x[0]**o+(0.739276532652)*x[0]+(-0.863216628203)*x[1]**o+(0.486539683681)*x[1]+(-0.771448803712)*x[2]**o+(-0.621526889056)*x[2]
        arg[0,0,2,0]=(-0.874502916672)*x[0]**o+(0.0708047915752)*x[0]+(0.917576817333)*x[1]**o+(-0.257626031306)*x[1]+(-0.161822453369)*x[2]**o+(0.488866912786)*x[2]
        arg[0,0,2,1]=(-0.258524075807)*x[0]**o+(0.206589504672)*x[0]+(0.232971263871)*x[1]**o+(-0.917991566808)*x[1]+(0.725943193794)*x[2]**o+(-0.170341108992)*x[2]
        arg[0,1,0,0]=(0.489505903018)*x[0]**o+(0.386376068728)*x[0]+(0.510797276159)*x[1]**o+(0.45356046467)*x[1]+(-0.664963144251)*x[2]**o+(0.128579947316)*x[2]
        arg[0,1,0,1]=(0.441567332096)*x[0]**o+(0.658686083904)*x[0]+(0.46601056348)*x[1]**o+(-0.130759928108)*x[1]+(0.324576589)*x[2]**o+(0.844700852997)*x[2]
        arg[0,1,1,0]=(0.388576628492)*x[0]**o+(0.41731399682)*x[0]+(-0.37037879729)*x[1]**o+(-0.00306376860454)*x[1]+(-0.464420131682)*x[2]**o+(0.570470055678)*x[2]
        arg[0,1,1,1]=(-0.0537780037583)*x[0]**o+(0.641854502644)*x[0]+(-0.475355608842)*x[1]**o+(-0.155501886306)*x[1]+(0.159833591997)*x[2]**o+(0.889951997355)*x[2]
        arg[0,1,2,0]=(0.58262140787)*x[0]**o+(0.487380368384)*x[0]+(-0.783228845976)*x[1]**o+(-0.683221249176)*x[1]+(0.98628901689)*x[2]**o+(0.591884616904)*x[2]
        arg[0,1,2,1]=(0.438472233911)*x[0]**o+(0.989088416794)*x[0]+(0.221688966923)*x[1]**o+(-0.119266918098)*x[1]+(-0.887421255415)*x[2]**o+(0.728082459156)*x[2]
        arg[0,2,0,0]=(-0.334110476968)*x[0]**o+(-0.877094545216)*x[0]+(-0.293135209996)*x[1]**o+(0.578110480224)*x[1]+(-0.898686340104)*x[2]**o+(0.528072783107)*x[2]
        arg[0,2,0,1]=(0.33041258354)*x[0]**o+(-0.70349630629)*x[0]+(0.349551161792)*x[1]**o+(-0.296687014146)*x[1]+(-0.272377440466)*x[2]**o+(-0.42221458274)*x[2]
        arg[0,2,1,0]=(-0.80021736912)*x[0]**o+(0.271727608951)*x[0]+(0.689351606243)*x[1]**o+(-0.95063104158)*x[1]+(0.447293221661)*x[2]**o+(0.962435109303)*x[2]
        arg[0,2,1,1]=(-0.773331228218)*x[0]**o+(0.569225340051)*x[0]+(-0.0330042729183)*x[1]**o+(-0.732867386826)*x[1]+(-0.399631128084)*x[2]**o+(0.235907094874)*x[2]
        arg[0,2,2,0]=(-0.943072868965)*x[0]**o+(-0.959364749946)*x[0]+(0.922934300445)*x[1]**o+(0.27257203243)*x[1]+(0.450453864748)*x[2]**o+(0.521284310775)*x[2]
        arg[0,2,2,1]=(-0.821201079286)*x[0]**o+(-0.801011107627)*x[0]+(0.554575277011)*x[1]**o+(-0.411361733513)*x[1]+(-0.858195570474)*x[2]**o+(0.420660103079)*x[2]
        arg[0,3,0,0]=(-0.0564164524098)*x[0]**o+(-0.865516143298)*x[0]+(-0.85754022189)*x[1]**o+(-0.0159800361584)*x[1]+(-0.891158469429)*x[2]**o+(-0.809909518115)*x[2]
        arg[0,3,0,1]=(-0.553858251068)*x[0]**o+(0.827558838148)*x[0]+(0.690081127613)*x[1]**o+(0.3475090336)*x[1]+(-0.190426828891)*x[2]**o+(-0.99589316535)*x[2]
        arg[0,3,1,0]=(-0.992317453163)*x[0]**o+(-0.992812531491)*x[0]+(-0.859683310869)*x[1]**o+(0.441779587739)*x[1]+(0.582457020078)*x[2]**o+(-0.959111461424)*x[2]
        arg[0,3,1,1]=(-0.713753922213)*x[0]**o+(0.504885148776)*x[0]+(-0.076988545634)*x[1]**o+(0.530828889101)*x[1]+(-0.335616242708)*x[2]**o+(-0.0421293420747)*x[2]
        arg[0,3,2,0]=(0.536024118289)*x[0]**o+(-0.527038052847)*x[0]+(-0.558392588656)*x[1]**o+(0.928618057856)*x[1]+(0.427507510033)*x[2]**o+(0.223570914141)*x[2]
        arg[0,3,2,1]=(-0.197968756117)*x[0]**o+(-0.58157606173)*x[0]+(-0.878630012825)*x[1]**o+(-0.380635552726)*x[1]+(-0.477570149125)*x[2]**o+(-0.642904015434)*x[2]
        arg[0,4,0,0]=(0.688763655127)*x[0]**o+(0.286011523933)*x[0]+(0.0933014250929)*x[1]**o+(0.355550563233)*x[1]+(0.121445291305)*x[2]**o+(-0.814538082974)*x[2]
        arg[0,4,0,1]=(-0.402717742839)*x[0]**o+(0.403946649835)*x[0]+(0.286978800579)*x[1]**o+(-0.363358056216)*x[1]+(-0.185233594102)*x[2]**o+(-0.902660250807)*x[2]
        arg[0,4,1,0]=(0.39352838659)*x[0]**o+(-0.430755011337)*x[0]+(-0.912518631162)*x[1]**o+(-0.634118795867)*x[1]+(-0.802445843416)*x[2]**o+(0.643395158048)*x[2]
        arg[0,4,1,1]=(-0.375507250175)*x[0]**o+(-0.609576942939)*x[0]+(0.35060243693)*x[1]**o+(0.384630130603)*x[1]+(0.165576820334)*x[2]**o+(-0.319012357341)*x[2]
        arg[0,4,2,0]=(0.0603590263818)*x[0]**o+(-0.435954775883)*x[0]+(-0.895329092913)*x[1]**o+(0.364809539103)*x[1]+(0.912864788565)*x[2]**o+(0.582044419963)*x[2]
        arg[0,4,2,1]=(-0.111089454685)*x[0]**o+(-0.247527778755)*x[0]+(0.897263901657)*x[1]**o+(0.168097655548)*x[1]+(-0.532273075067)*x[2]**o+(-0.158529268933)*x[2]
        arg[1,0,0,0]=(-0.940865155622)*x[0]**o+(0.639652431549)*x[0]+(0.204433172026)*x[1]**o+(-0.333435100662)*x[1]+(0.922286591304)*x[2]**o+(0.245091036189)*x[2]
        arg[1,0,0,1]=(-0.843207880309)*x[0]**o+(0.496567621809)*x[0]+(-0.747925907677)*x[1]**o+(-0.666195656311)*x[1]+(-0.836848401917)*x[2]**o+(-0.77301298659)*x[2]
        arg[1,0,1,0]=(-0.484987528212)*x[0]**o+(-0.208798022576)*x[0]+(-0.516295195217)*x[1]**o+(0.273733676249)*x[1]+(0.642865886081)*x[2]**o+(0.94556566928)*x[2]
        arg[1,0,1,1]=(0.285548625366)*x[0]**o+(-0.316870037241)*x[0]+(0.166055429845)*x[1]**o+(-0.907240066063)*x[1]+(-0.973602611383)*x[2]**o+(-0.232655535454)*x[2]
        arg[1,0,2,0]=(0.96797757731)*x[0]**o+(-0.0920831342923)*x[0]+(0.0367163548581)*x[1]**o+(0.372503137376)*x[1]+(0.114902128151)*x[2]**o+(0.147390259838)*x[2]
        arg[1,0,2,1]=(-0.486190283922)*x[0]**o+(-0.713417720041)*x[0]+(0.0212905465522)*x[1]**o+(-0.358636168795)*x[1]+(0.719370965272)*x[2]**o+(-0.506502687359)*x[2]
        arg[1,1,0,0]=(-0.216544472879)*x[0]**o+(-0.690540499698)*x[0]+(0.190155427044)*x[1]**o+(0.832206912182)*x[1]+(-0.108713936252)*x[2]**o+(-0.221373087337)*x[2]
        arg[1,1,0,1]=(0.982934604971)*x[0]**o+(0.604369826835)*x[0]+(-0.34365978148)*x[1]**o+(-0.167486779981)*x[1]+(-0.753609370582)*x[2]**o+(-0.220160888977)*x[2]
        arg[1,1,1,0]=(0.516177509446)*x[0]**o+(-0.474445723273)*x[0]+(0.813647087566)*x[1]**o+(-0.281201093802)*x[1]+(0.941465944108)*x[2]**o+(0.352728360059)*x[2]
        arg[1,1,1,1]=(-0.914732890633)*x[0]**o+(0.214962090901)*x[0]+(-0.710485058274)*x[1]**o+(-0.070756754056)*x[1]+(-0.449571590211)*x[2]**o+(0.353655711451)*x[2]
        arg[1,1,2,0]=(-0.96655555619)*x[0]**o+(-0.47460534732)*x[0]+(0.0610695053327)*x[1]**o+(0.815390116709)*x[1]+(-0.844916163552)*x[2]**o+(-0.433015304307)*x[2]
        arg[1,1,2,1]=(-0.611879453709)*x[0]**o+(0.849092848509)*x[0]+(0.0181322361566)*x[1]**o+(0.264173017827)*x[1]+(0.0452154806937)*x[2]**o+(0.0395300648465)*x[2]
        arg[1,2,0,0]=(0.558820946357)*x[0]**o+(-0.471519705598)*x[0]+(-0.200404864145)*x[1]**o+(0.452044160691)*x[1]+(-0.507559485963)*x[2]**o+(-0.993437841726)*x[2]
        arg[1,2,0,1]=(-0.944738664192)*x[0]**o+(-0.497536772117)*x[0]+(0.111399525675)*x[1]**o+(-0.93450294271)*x[1]+(-0.463702615737)*x[2]**o+(-0.330112672884)*x[2]
        arg[1,2,1,0]=(0.238225961058)*x[0]**o+(-0.471520173419)*x[0]+(0.483767084892)*x[1]**o+(-0.0359407284537)*x[1]+(-0.767379409753)*x[2]**o+(-0.283596371915)*x[2]
        arg[1,2,1,1]=(0.188699181238)*x[0]**o+(0.829593121989)*x[0]+(0.603377767593)*x[1]**o+(0.163334775468)*x[1]+(-0.695141594131)*x[2]**o+(-0.335208640181)*x[2]
        arg[1,2,2,0]=(-0.0953160757881)*x[0]**o+(0.179594592176)*x[0]+(0.95983902191)*x[1]**o+(-0.689622683527)*x[1]+(0.418442085936)*x[2]**o+(-0.271399861129)*x[2]
        arg[1,2,2,1]=(0.598691886555)*x[0]**o+(0.334120340646)*x[0]+(0.580763872443)*x[1]**o+(0.769016736836)*x[1]+(-0.442121483951)*x[2]**o+(0.0685264980499)*x[2]
        arg[1,3,0,0]=(-0.482944276372)*x[0]**o+(-0.0518057205519)*x[0]+(-0.878100108766)*x[1]**o+(-0.130957239599)*x[1]+(0.71270341547)*x[2]**o+(-0.313822116756)*x[2]
        arg[1,3,0,1]=(0.565128725562)*x[0]**o+(0.706149004661)*x[0]+(0.837747652328)*x[1]**o+(-0.429417376176)*x[1]+(0.179437627997)*x[2]**o+(-0.841831183522)*x[2]
        arg[1,3,1,0]=(0.608648887224)*x[0]**o+(0.118220851451)*x[0]+(0.492890967377)*x[1]**o+(-0.162344084552)*x[1]+(0.141203210671)*x[2]**o+(0.883457779687)*x[2]
        arg[1,3,1,1]=(0.0322521291544)*x[0]**o+(0.839748977403)*x[0]+(0.75094786544)*x[1]**o+(0.954833145672)*x[1]+(0.461457599013)*x[2]**o+(-0.761973385747)*x[2]
        arg[1,3,2,0]=(-0.757022609619)*x[0]**o+(-0.250893790099)*x[0]+(-0.973631035961)*x[1]**o+(0.506574565217)*x[1]+(0.103557430896)*x[2]**o+(0.881026337134)*x[2]
        arg[1,3,2,1]=(-0.395945302352)*x[0]**o+(-0.125428842065)*x[0]+(-0.339663004568)*x[1]**o+(-0.0272044592179)*x[1]+(0.75200506467)*x[2]**o+(-0.632922582454)*x[2]
        arg[1,4,0,0]=(0.0462127928202)*x[0]**o+(-0.745613488751)*x[0]+(0.357338297422)*x[1]**o+(0.739034094804)*x[1]+(-0.451247272253)*x[2]**o+(0.418030110847)*x[2]
        arg[1,4,0,1]=(0.914131238526)*x[0]**o+(-0.0280436066624)*x[0]+(0.861048515118)*x[1]**o+(0.148206807819)*x[1]+(-0.990856990354)*x[2]**o+(0.903615279172)*x[2]
        arg[1,4,1,0]=(0.189809824038)*x[0]**o+(-0.749477832258)*x[0]+(-0.802713794942)*x[1]**o+(0.0610727255572)*x[1]+(0.292940815327)*x[2]**o+(0.213443422205)*x[2]
        arg[1,4,1,1]=(0.792313292544)*x[0]**o+(0.321344675677)*x[0]+(-0.417516596097)*x[1]**o+(-0.63797359819)*x[1]+(-0.426112319872)*x[2]**o+(0.69281310288)*x[2]
        arg[1,4,2,0]=(-0.11017764073)*x[0]**o+(0.703037562344)*x[0]+(0.739591950873)*x[1]**o+(0.445612268331)*x[1]+(-0.071251825024)*x[2]**o+(0.00577350654185)*x[2]
        arg[1,4,2,1]=(-0.793584849832)*x[0]**o+(-0.546439256231)*x[0]+(-0.947194030803)*x[1]**o+(-0.0455700937677)*x[1]+(0.535368301918)*x[2]**o+(0.814435535521)*x[2]
        arg[2,0,0,0]=(0.249661949438)*x[0]**o+(-0.581533619092)*x[0]+(-0.674373505747)*x[1]**o+(0.464175602196)*x[1]+(-0.528086662834)*x[2]**o+(-0.212416176351)*x[2]
        arg[2,0,0,1]=(-0.588282108798)*x[0]**o+(0.761293399494)*x[0]+(0.99918874903)*x[1]**o+(0.542499992928)*x[1]+(0.433925746483)*x[2]**o+(0.972864491889)*x[2]
        arg[2,0,1,0]=(0.0572335623442)*x[0]**o+(0.444906443533)*x[0]+(-0.305538151235)*x[1]**o+(0.244530068532)*x[1]+(-0.20198975858)*x[2]**o+(-0.00490928477351)*x[2]
        arg[2,0,1,1]=(0.832496085695)*x[0]**o+(-0.797246004595)*x[0]+(0.873816520001)*x[1]**o+(0.628293733929)*x[1]+(-0.110922710624)*x[2]**o+(-0.0718221914062)*x[2]
        arg[2,0,2,0]=(0.140970192196)*x[0]**o+(0.582230323368)*x[0]+(-0.757764751267)*x[1]**o+(0.206061262231)*x[1]+(0.657237474788)*x[2]**o+(0.598632779121)*x[2]
        arg[2,0,2,1]=(-0.686218790734)*x[0]**o+(-0.157012470162)*x[0]+(0.448565721848)*x[1]**o+(0.0539087954076)*x[1]+(0.619821805007)*x[2]**o+(0.289144379283)*x[2]
        arg[2,1,0,0]=(0.787423884494)*x[0]**o+(-0.985959420171)*x[0]+(0.155584360983)*x[1]**o+(0.223292293778)*x[1]+(-0.437349719833)*x[2]**o+(0.989077189275)*x[2]
        arg[2,1,0,1]=(-0.597301631508)*x[0]**o+(-0.0414813766671)*x[0]+(-0.64883048443)*x[1]**o+(-0.18731615522)*x[1]+(0.735189823623)*x[2]**o+(-0.746633725819)*x[2]
        arg[2,1,1,0]=(-0.393486075452)*x[0]**o+(-0.480110060647)*x[0]+(0.559811870081)*x[1]**o+(0.0814282056993)*x[1]+(0.244899906416)*x[2]**o+(-0.884478480652)*x[2]
        arg[2,1,1,1]=(0.547747813175)*x[0]**o+(0.317890585903)*x[0]+(-0.70925099068)*x[1]**o+(-0.543227445702)*x[1]+(0.749693240223)*x[2]**o+(0.79984501038)*x[2]
        arg[2,1,2,0]=(-0.289583636666)*x[0]**o+(0.513926622265)*x[0]+(-0.00283544918865)*x[1]**o+(0.530724065811)*x[1]+(0.900479623243)*x[2]**o+(-0.146332301083)*x[2]
        arg[2,1,2,1]=(-0.487223003118)*x[0]**o+(0.161546483594)*x[0]+(-0.685211654772)*x[1]**o+(0.4803179816)*x[1]+(0.676120396917)*x[2]**o+(-0.174953085068)*x[2]
        arg[2,2,0,0]=(0.31746495834)*x[0]**o+(-0.513225182059)*x[0]+(0.33220677322)*x[1]**o+(-0.302371334854)*x[1]+(0.0964988251907)*x[2]**o+(-0.259532745762)*x[2]
        arg[2,2,0,1]=(0.313462846078)*x[0]**o+(0.079720902627)*x[0]+(-0.487932457563)*x[1]**o+(-0.786522786773)*x[1]+(0.357868214554)*x[2]**o+(0.251856050362)*x[2]
        arg[2,2,1,0]=(-0.180472378556)*x[0]**o+(0.345084802362)*x[0]+(0.641108557953)*x[1]**o+(0.198717742483)*x[1]+(0.0374114682326)*x[2]**o+(-0.915249907773)*x[2]
        arg[2,2,1,1]=(0.303787742688)*x[0]**o+(-0.941822487745)*x[0]+(-0.262567025842)*x[1]**o+(-0.197051445307)*x[1]+(0.381561965565)*x[2]**o+(-0.198549777232)*x[2]
        arg[2,2,2,0]=(-0.850284190461)*x[0]**o+(0.985224868544)*x[0]+(0.633868153422)*x[1]**o+(-0.57559927346)*x[1]+(0.68753726951)*x[2]**o+(-0.923696997656)*x[2]
        arg[2,2,2,1]=(0.559527478503)*x[0]**o+(-0.619764636475)*x[0]+(-0.0352163693945)*x[1]**o+(-0.37351931995)*x[1]+(0.0772333960938)*x[2]**o+(-0.353057755509)*x[2]
        arg[2,3,0,0]=(-0.274260374984)*x[0]**o+(0.365180349506)*x[0]+(-0.879564869545)*x[1]**o+(-0.0908162765131)*x[1]+(0.813491263836)*x[2]**o+(-0.396739585204)*x[2]
        arg[2,3,0,1]=(-0.793218219979)*x[0]**o+(0.227474218201)*x[0]+(0.97716610661)*x[1]**o+(0.317074032722)*x[1]+(0.076776446926)*x[2]**o+(0.069448617786)*x[2]
        arg[2,3,1,0]=(0.411400569812)*x[0]**o+(0.517743454926)*x[0]+(0.646697523033)*x[1]**o+(-0.973369313024)*x[1]+(-0.599741793926)*x[2]**o+(-0.743687020283)*x[2]
        arg[2,3,1,1]=(-0.491445354697)*x[0]**o+(-0.549840110396)*x[0]+(-0.260012915614)*x[1]**o+(0.671054626537)*x[1]+(0.0394966796969)*x[2]**o+(-0.701243931411)*x[2]
        arg[2,3,2,0]=(-0.806859723517)*x[0]**o+(-0.753940066715)*x[0]+(-0.664229274865)*x[1]**o+(0.0595484609931)*x[1]+(-0.921892262667)*x[2]**o+(-0.701770672395)*x[2]
        arg[2,3,2,1]=(-0.313979805844)*x[0]**o+(-0.542583864222)*x[0]+(-0.985990886553)*x[1]**o+(0.664926443979)*x[1]+(-0.99778334615)*x[2]**o+(0.268046810979)*x[2]
        arg[2,4,0,0]=(-0.41348240167)*x[0]**o+(0.473517720534)*x[0]+(-0.290880345559)*x[1]**o+(-0.607611624107)*x[1]+(-0.25537697471)*x[2]**o+(-0.472402477362)*x[2]
        arg[2,4,0,1]=(0.0222534371058)*x[0]**o+(-0.0739354104963)*x[0]+(-0.960057765718)*x[1]**o+(-0.772161403037)*x[1]+(-0.0356127352146)*x[2]**o+(-0.623605628308)*x[2]
        arg[2,4,1,0]=(-0.750995516053)*x[0]**o+(-0.289601799215)*x[0]+(0.294370180152)*x[1]**o+(0.956012147239)*x[1]+(-0.27958662106)*x[2]**o+(-0.370609593537)*x[2]
        arg[2,4,1,1]=(0.321767371741)*x[0]**o+(0.999223434136)*x[0]+(0.665865428241)*x[1]**o+(0.427942082269)*x[1]+(-0.135020552387)*x[2]**o+(0.692025479514)*x[2]
        arg[2,4,2,0]=(-0.519508249649)*x[0]**o+(0.0779688946471)*x[0]+(0.493489242872)*x[1]**o+(-0.0961136530093)*x[1]+(-0.0750243559261)*x[2]**o+(0.860970740104)*x[2]
        arg[2,4,2,1]=(0.807909049346)*x[0]**o+(0.388180877227)*x[0]+(-0.9044578433)*x[1]**o+(0.86765389355)*x[1]+(-0.196105779343)*x[2]**o+(0.448661151455)*x[2]
        arg[3,0,0,0]=(-0.780895567041)*x[0]**o+(-0.571253266808)*x[0]+(-0.904183399932)*x[1]**o+(-0.883296586722)*x[1]+(0.855945559487)*x[2]**o+(0.618533553098)*x[2]
        arg[3,0,0,1]=(-0.831076704037)*x[0]**o+(0.898347446798)*x[0]+(-0.716442620396)*x[1]**o+(0.761028369421)*x[1]+(-0.486505604444)*x[2]**o+(0.25474220145)*x[2]
        arg[3,0,1,0]=(-0.0127541044953)*x[0]**o+(0.759482315308)*x[0]+(-0.725655006866)*x[1]**o+(0.667310781285)*x[1]+(0.266075927439)*x[2]**o+(0.423300182252)*x[2]
        arg[3,0,1,1]=(-0.594972266882)*x[0]**o+(-0.842501149639)*x[0]+(-0.347686790781)*x[1]**o+(0.686403640474)*x[1]+(0.3452971966)*x[2]**o+(0.722258585706)*x[2]
        arg[3,0,2,0]=(-0.715711146058)*x[0]**o+(-0.835739963371)*x[0]+(0.966177361301)*x[1]**o+(-0.909111754995)*x[1]+(-0.175184576557)*x[2]**o+(0.695058239921)*x[2]
        arg[3,0,2,1]=(0.306251544915)*x[0]**o+(0.944775715803)*x[0]+(-0.740601754617)*x[1]**o+(0.434394890125)*x[1]+(-0.876527474971)*x[2]**o+(0.209321129689)*x[2]
        arg[3,1,0,0]=(-0.86788221871)*x[0]**o+(-0.833030657309)*x[0]+(0.84576621947)*x[1]**o+(-0.469186887277)*x[1]+(-0.527611740694)*x[2]**o+(-0.312054431001)*x[2]
        arg[3,1,0,1]=(-0.353678394652)*x[0]**o+(0.591287444611)*x[0]+(-0.213012375931)*x[1]**o+(-0.917259099251)*x[1]+(0.739016360622)*x[2]**o+(0.718754459378)*x[2]
        arg[3,1,1,0]=(-0.92646825221)*x[0]**o+(-0.870564973175)*x[0]+(-0.781805219404)*x[1]**o+(0.078530434642)*x[1]+(0.386018501087)*x[2]**o+(-0.120490492288)*x[2]
        arg[3,1,1,1]=(-0.824273737615)*x[0]**o+(0.401211543421)*x[0]+(0.360077381624)*x[1]**o+(-0.0599234883152)*x[1]+(0.246820480884)*x[2]**o+(-0.175313542134)*x[2]
        arg[3,1,2,0]=(0.807756928769)*x[0]**o+(0.889534807508)*x[0]+(0.633296189954)*x[1]**o+(-0.471821951252)*x[1]+(0.396011249874)*x[2]**o+(0.386800809575)*x[2]
        arg[3,1,2,1]=(-0.494145142543)*x[0]**o+(-0.514793985721)*x[0]+(0.825379848403)*x[1]**o+(-0.457860259786)*x[1]+(-0.599557868275)*x[2]**o+(-0.952704784672)*x[2]
        arg[3,2,0,0]=(0.970345295013)*x[0]**o+(-0.838468036663)*x[0]+(0.265634348738)*x[1]**o+(-0.873847124748)*x[1]+(0.416242648913)*x[2]**o+(0.59908438222)*x[2]
        arg[3,2,0,1]=(0.692377432429)*x[0]**o+(-0.338749041227)*x[0]+(-0.449383750398)*x[1]**o+(0.841669527559)*x[1]+(0.902482059741)*x[2]**o+(-0.655719647704)*x[2]
        arg[3,2,1,0]=(0.152417917522)*x[0]**o+(-0.78860144)*x[0]+(-0.881276170743)*x[1]**o+(0.692901550061)*x[1]+(-0.985377819322)*x[2]**o+(0.362747973601)*x[2]
        arg[3,2,1,1]=(0.238845792119)*x[0]**o+(-0.65838373309)*x[0]+(-0.659676514296)*x[1]**o+(0.870742876778)*x[1]+(-0.367345415947)*x[2]**o+(0.322353666541)*x[2]
        arg[3,2,2,0]=(0.456386394533)*x[0]**o+(-0.135575574979)*x[0]+(-0.731714323922)*x[1]**o+(0.936807859479)*x[1]+(0.511595594967)*x[2]**o+(0.570282464669)*x[2]
        arg[3,2,2,1]=(0.53299212624)*x[0]**o+(0.800998166434)*x[0]+(-0.431740087552)*x[1]**o+(-0.880092532661)*x[1]+(-0.418657988674)*x[2]**o+(-0.181008149338)*x[2]
        arg[3,3,0,0]=(0.60015312936)*x[0]**o+(-0.163125583081)*x[0]+(0.299668981953)*x[1]**o+(-0.761494422747)*x[1]+(0.404839953231)*x[2]**o+(-0.114039836538)*x[2]
        arg[3,3,0,1]=(-0.597481072133)*x[0]**o+(-0.506468582896)*x[0]+(0.406822348574)*x[1]**o+(0.247431348859)*x[1]+(-0.137941563908)*x[2]**o+(0.0503979259383)*x[2]
        arg[3,3,1,0]=(0.0921069092744)*x[0]**o+(-0.113960490794)*x[0]+(0.750059773909)*x[1]**o+(0.894945048248)*x[1]+(-0.154873449786)*x[2]**o+(-0.520229467775)*x[2]
        arg[3,3,1,1]=(0.349078696154)*x[0]**o+(0.141298430827)*x[0]+(0.249161007832)*x[1]**o+(0.0240310278591)*x[1]+(-0.18294527838)*x[2]**o+(-0.666698832632)*x[2]
        arg[3,3,2,0]=(-0.476684936147)*x[0]**o+(0.334093402615)*x[0]+(-0.326434684009)*x[1]**o+(0.91336431975)*x[1]+(0.483080537399)*x[2]**o+(0.439096732335)*x[2]
        arg[3,3,2,1]=(0.214713426286)*x[0]**o+(-0.973239849984)*x[0]+(0.984290637567)*x[1]**o+(-0.644668525299)*x[1]+(0.00985524319943)*x[2]**o+(0.68777403533)*x[2]
        arg[3,4,0,0]=(-0.88715423933)*x[0]**o+(0.642412222188)*x[0]+(-0.119667464132)*x[1]**o+(-0.201583116481)*x[1]+(-0.870921042381)*x[2]**o+(-0.00379039429884)*x[2]
        arg[3,4,0,1]=(0.511114986801)*x[0]**o+(-0.497617928997)*x[0]+(-0.189435233543)*x[1]**o+(-0.29288141421)*x[1]+(-0.476297664539)*x[2]**o+(-0.202915859137)*x[2]
        arg[3,4,1,0]=(0.0739878219028)*x[0]**o+(0.675050675951)*x[0]+(-0.45950101521)*x[1]**o+(0.306538598103)*x[1]+(-0.726069135031)*x[2]**o+(0.193290422951)*x[2]
        arg[3,4,1,1]=(-0.141324267565)*x[0]**o+(0.377343882109)*x[0]+(-0.16075604872)*x[1]**o+(0.263332274591)*x[1]+(0.886416182992)*x[2]**o+(0.931905996631)*x[2]
        arg[3,4,2,0]=(-0.76912732001)*x[0]**o+(-0.255011103121)*x[0]+(-0.372810348364)*x[1]**o+(0.108137766687)*x[1]+(-0.0761382180171)*x[2]**o+(0.46825331282)*x[2]
        arg[3,4,2,1]=(-0.515945940217)*x[0]**o+(-0.630017609554)*x[0]+(-0.815236764508)*x[1]**o+(0.636572471005)*x[1]+(0.244112465789)*x[2]**o+(-0.471661017589)*x[2]
        ref[0,0,0,0]=(-0.738709393227)/(o+1.)+(0.283993125638)+(-0.891975745754)*0.5**o
        ref[0,0,0,1]=(0.912832356419)/(o+1.)+(-0.800452287951)+(0.428698930349)*0.5**o
        ref[0,0,1,0]=(0.604410785754)/(o+1.)+(-0.70496852873)+(0.974069118653)*0.5**o
        ref[0,0,1,1]=(-1.63466543191)/(o+1.)+(0.302144663639)+(0.813412823502)*0.5**o
        ref[0,0,2,0]=(0.755754363963)/(o+1.)+(0.151022836528)+(-0.874502916672)*0.5**o
        ref[0,0,2,1]=(0.958914457666)/(o+1.)+(-0.440871585564)+(-0.258524075807)*0.5**o
        ref[0,1,0,0]=(-0.154165868092)/(o+1.)+(0.484258240357)+(0.489505903018)*0.5**o
        ref[0,1,0,1]=(0.79058715248)/(o+1.)+(0.686313504396)+(0.441567332096)*0.5**o
        ref[0,1,1,0]=(-0.834798928973)/(o+1.)+(0.492360141946)+(0.388576628492)*0.5**o
        ref[0,1,1,1]=(-0.315522016845)/(o+1.)+(0.688152306846)+(-0.0537780037583)*0.5**o
        ref[0,1,2,0]=(0.203060170914)/(o+1.)+(0.198021868056)+(0.58262140787)*0.5**o
        ref[0,1,2,1]=(-0.665732288492)/(o+1.)+(0.798951978927)+(0.438472233911)*0.5**o
        ref[0,2,0,0]=(-1.1918215501)/(o+1.)+(0.114544359058)+(-0.334110476968)*0.5**o
        ref[0,2,0,1]=(0.0771737213265)/(o+1.)+(-0.711198951588)+(0.33041258354)*0.5**o
        ref[0,2,1,0]=(1.1366448279)/(o+1.)+(0.141765838337)+(-0.80021736912)*0.5**o
        ref[0,2,1,1]=(-0.432635401002)/(o+1.)+(0.0361325240497)+(-0.773331228218)*0.5**o
        ref[0,2,2,0]=(1.37338816519)/(o+1.)+(-0.0827542033711)+(-0.943072868965)*0.5**o
        ref[0,2,2,1]=(-0.303620293463)/(o+1.)+(-0.39585636903)+(-0.821201079286)*0.5**o
        ref[0,3,0,0]=(-1.74869869132)/(o+1.)+(-0.845702848785)+(-0.0564164524098)*0.5**o
        ref[0,3,0,1]=(0.499654298722)/(o+1.)+(0.0895873531986)+(-0.553858251068)*0.5**o
        ref[0,3,1,0]=(-0.277226290791)/(o+1.)+(-0.755072202588)+(-0.992317453163)*0.5**o
        ref[0,3,1,1]=(-0.412604788342)/(o+1.)+(0.496792347901)+(-0.713753922213)*0.5**o
        ref[0,3,2,0]=(-0.130885078624)/(o+1.)+(0.312575459575)+(0.536024118289)*0.5**o
        ref[0,3,2,1]=(-1.35620016195)/(o+1.)+(-0.802557814945)+(-0.197968756117)*0.5**o
        ref[0,4,0,0]=(0.214746716397)/(o+1.)+(-0.0864879979042)+(0.688763655127)*0.5**o
        ref[0,4,0,1]=(0.101745206477)/(o+1.)+(-0.431035828594)+(-0.402717742839)*0.5**o
        ref[0,4,1,0]=(-1.71496447458)/(o+1.)+(-0.210739324578)+(0.39352838659)*0.5**o
        ref[0,4,1,1]=(0.516179257264)/(o+1.)+(-0.271979584839)+(-0.375507250175)*0.5**o
        ref[0,4,2,0]=(0.0175356956523)/(o+1.)+(0.255449591592)+(0.0603590263818)*0.5**o
        ref[0,4,2,1]=(0.36499082659)/(o+1.)+(-0.11897969607)+(-0.111089454685)*0.5**o
        ref[1,0,0,0]=(1.12671976333)/(o+1.)+(0.275654183538)+(-0.940865155622)*0.5**o
        ref[1,0,0,1]=(-1.58477430959)/(o+1.)+(-0.471320510546)+(-0.843207880309)*0.5**o
        ref[1,0,1,0]=(0.126570690864)/(o+1.)+(0.505250661477)+(-0.484987528212)*0.5**o
        ref[1,0,1,1]=(-0.807547181539)/(o+1.)+(-0.728382819379)+(0.285548625366)*0.5**o
        ref[1,0,2,0]=(0.151618483009)/(o+1.)+(0.213905131461)+(0.96797757731)*0.5**o
        ref[1,0,2,1]=(0.740661511824)/(o+1.)+(-0.789278288097)+(-0.486190283922)*0.5**o
        ref[1,1,0,0]=(0.0814414907926)/(o+1.)+(-0.0398533374263)+(-0.216544472879)*0.5**o
        ref[1,1,0,1]=(-1.09726915206)/(o+1.)+(0.108361078938)+(0.982934604971)*0.5**o
        ref[1,1,1,0]=(1.75511303167)/(o+1.)+(-0.201459228508)+(0.516177509446)*0.5**o
        ref[1,1,1,1]=(-1.16005664849)/(o+1.)+(0.248930524148)+(-0.914732890633)*0.5**o
        ref[1,1,2,0]=(-0.78384665822)/(o+1.)+(-0.0461152674588)+(-0.96655555619)*0.5**o
        ref[1,1,2,1]=(0.0633477168503)/(o+1.)+(0.576397965591)+(-0.611879453709)*0.5**o
        ref[1,2,0,0]=(-0.707964350108)/(o+1.)+(-0.506456693317)+(0.558820946357)*0.5**o
        ref[1,2,0,1]=(-0.352303090063)/(o+1.)+(-0.881076193856)+(-0.944738664192)*0.5**o
        ref[1,2,1,0]=(-0.283612324861)/(o+1.)+(-0.395528636894)+(0.238225961058)*0.5**o
        ref[1,2,1,1]=(-0.0917638265383)/(o+1.)+(0.328859628638)+(0.188699181238)*0.5**o
        ref[1,2,2,0]=(1.37828110785)/(o+1.)+(-0.39071397624)+(-0.0953160757881)*0.5**o
        ref[1,2,2,1]=(0.138642388492)/(o+1.)+(0.585831787766)+(0.598691886555)*0.5**o
        ref[1,3,0,0]=(-0.165396693296)/(o+1.)+(-0.248292538454)+(-0.482944276372)*0.5**o
        ref[1,3,0,1]=(1.01718528032)/(o+1.)+(-0.282549777519)+(0.565128725562)*0.5**o
        ref[1,3,1,0]=(0.634094178049)/(o+1.)+(0.419667273293)+(0.608648887224)*0.5**o
        ref[1,3,1,1]=(1.21240546445)/(o+1.)+(0.516304368664)+(0.0322521291544)*0.5**o
        ref[1,3,2,0]=(-0.870073605065)/(o+1.)+(0.568353556126)+(-0.757022609619)*0.5**o
        ref[1,3,2,1]=(0.412342060102)/(o+1.)+(-0.392777941868)+(-0.395945302352)*0.5**o
        ref[1,4,0,0]=(-0.0939089748304)/(o+1.)+(0.20572535845)+(0.0462127928202)*0.5**o
        ref[1,4,0,1]=(-0.129808475236)/(o+1.)+(0.511889240164)+(0.914131238526)*0.5**o
        ref[1,4,1,0]=(-0.509772979615)/(o+1.)+(-0.237480842248)+(0.189809824038)*0.5**o
        ref[1,4,1,1]=(-0.843628915969)/(o+1.)+(0.188092090183)+(0.792313292544)*0.5**o
        ref[1,4,2,0]=(0.668340125849)/(o+1.)+(0.577211668608)+(-0.11017764073)*0.5**o
        ref[1,4,2,1]=(-0.411825728886)/(o+1.)+(0.111213092761)+(-0.793584849832)*0.5**o
        ref[2,0,0,0]=(-1.20246016858)/(o+1.)+(-0.164887096623)+(0.249661949438)*0.5**o
        ref[2,0,0,1]=(1.43311449551)/(o+1.)+(1.13832894216)+(-0.588282108798)*0.5**o
        ref[2,0,1,0]=(-0.507527909815)/(o+1.)+(0.342263613646)+(0.0572335623442)*0.5**o
        ref[2,0,1,1]=(0.762893809377)/(o+1.)+(-0.120387231036)+(0.832496085695)*0.5**o
        ref[2,0,2,0]=(-0.100527276479)/(o+1.)+(0.69346218236)+(0.140970192196)*0.5**o
        ref[2,0,2,1]=(1.06838752686)/(o+1.)+(0.0930203522642)+(-0.686218790734)*0.5**o
        ref[2,1,0,0]=(-0.28176535885)/(o+1.)+(0.113205031441)+(0.787423884494)*0.5**o
        ref[2,1,0,1]=(0.0863593391922)/(o+1.)+(-0.487715628853)+(-0.597301631508)*0.5**o
        ref[2,1,1,0]=(0.804711776497)/(o+1.)+(-0.6415801678)+(-0.393486075452)*0.5**o
        ref[2,1,1,1]=(0.0404422495423)/(o+1.)+(0.28725407529)+(0.547747813175)*0.5**o
        ref[2,1,2,0]=(0.897644174054)/(o+1.)+(0.449159193497)+(-0.289583636666)*0.5**o
        ref[2,1,2,1]=(-0.00909125785473)/(o+1.)+(0.233455690063)+(-0.487223003118)*0.5**o
        ref[2,2,0,0]=(0.428705598411)/(o+1.)+(-0.537564631337)+(0.31746495834)*0.5**o
        ref[2,2,0,1]=(-0.13006424301)/(o+1.)+(-0.227472916892)+(0.313462846078)*0.5**o
        ref[2,2,1,0]=(0.678520026185)/(o+1.)+(-0.185723681464)+(-0.180472378556)*0.5**o
        ref[2,2,1,1]=(0.118994939723)/(o+1.)+(-0.668711855142)+(0.303787742688)*0.5**o
        ref[2,2,2,0]=(1.32140542293)/(o+1.)+(-0.257035701286)+(-0.850284190461)*0.5**o
        ref[2,2,2,1]=(0.0420170266993)/(o+1.)+(-0.673170855967)+(0.559527478503)*0.5**o
        ref[2,3,0,0]=(-0.0660736057092)/(o+1.)+(-0.0611877561056)+(-0.274260374984)*0.5**o
        ref[2,3,0,1]=(1.05394255354)/(o+1.)+(0.306998434354)+(-0.793218219979)*0.5**o
        ref[2,3,1,0]=(0.046955729107)/(o+1.)+(-0.599656439191)+(0.411400569812)*0.5**o
        ref[2,3,1,1]=(-0.220516235917)/(o+1.)+(-0.290014707635)+(-0.491445354697)*0.5**o
        ref[2,3,2,0]=(-1.58612153753)/(o+1.)+(-0.698081139059)+(-0.806859723517)*0.5**o
        ref[2,3,2,1]=(-1.9837742327)/(o+1.)+(0.195194695368)+(-0.313979805844)*0.5**o
        ref[2,4,0,0]=(-0.546257320269)/(o+1.)+(-0.303248190467)+(-0.41348240167)*0.5**o
        ref[2,4,0,1]=(-0.995670500932)/(o+1.)+(-0.734851220921)+(0.0222534371058)*0.5**o
        ref[2,4,1,0]=(0.0147835590923)/(o+1.)+(0.147900377243)+(-0.750995516053)*0.5**o
        ref[2,4,1,1]=(0.530844875854)/(o+1.)+(1.05959549796)+(0.321767371741)*0.5**o
        ref[2,4,2,0]=(0.418464886946)/(o+1.)+(0.421412990871)+(-0.519508249649)*0.5**o
        ref[2,4,2,1]=(-1.10056362264)/(o+1.)+(0.852247961116)+(0.807909049346)*0.5**o
        ref[3,0,0,0]=(-0.0482378404454)/(o+1.)+(-0.418008150216)+(-0.780895567041)*0.5**o
        ref[3,0,0,1]=(-1.20294822484)/(o+1.)+(0.957059008835)+(-0.831076704037)*0.5**o
        ref[3,0,1,0]=(-0.459579079427)/(o+1.)+(0.925046639422)+(-0.0127541044953)*0.5**o
        ref[3,0,1,1]=(-0.00238959418047)/(o+1.)+(0.28308053827)+(-0.594972266882)*0.5**o
        ref[3,0,2,0]=(0.790992784744)/(o+1.)+(-0.524896739222)+(-0.715711146058)*0.5**o
        ref[3,0,2,1]=(-1.61712922959)/(o+1.)+(0.794245867809)+(0.306251544915)*0.5**o
        ref[3,1,0,0]=(0.318154478777)/(o+1.)+(-0.807135987793)+(-0.86788221871)*0.5**o
        ref[3,1,0,1]=(0.526003984691)/(o+1.)+(0.196391402369)+(-0.353678394652)*0.5**o
        ref[3,1,1,0]=(-0.395786718317)/(o+1.)+(-0.456262515411)+(-0.92646825221)*0.5**o
        ref[3,1,1,1]=(0.606897862508)/(o+1.)+(0.082987256486)+(-0.824273737615)*0.5**o
        ref[3,1,2,0]=(1.02930743983)/(o+1.)+(0.402256832916)+(0.807756928769)*0.5**o
        ref[3,1,2,1]=(0.225821980128)/(o+1.)+(-0.96267951509)+(-0.494145142543)*0.5**o
        ref[3,2,0,0]=(0.681876997651)/(o+1.)+(-0.556615389596)+(0.970345295013)*0.5**o
        ref[3,2,0,1]=(0.453098309344)/(o+1.)+(-0.0763995806862)+(0.692377432429)*0.5**o
        ref[3,2,1,0]=(-1.86665399006)/(o+1.)+(0.133524041831)+(0.152417917522)*0.5**o
        ref[3,2,1,1]=(-1.02702193024)/(o+1.)+(0.267356405114)+(0.238845792119)*0.5**o
        ref[3,2,2,0]=(-0.220118728954)/(o+1.)+(0.685757374584)+(0.456386394533)*0.5**o
        ref[3,2,2,1]=(-0.850398076225)/(o+1.)+(-0.130051257782)+(0.53299212624)*0.5**o
        ref[3,3,0,0]=(0.704508935183)/(o+1.)+(-0.519329921183)+(0.60015312936)*0.5**o
        ref[3,3,0,1]=(0.268880784666)/(o+1.)+(-0.10431965405)+(-0.597481072133)*0.5**o
        ref[3,3,1,0]=(0.595186324123)/(o+1.)+(0.13037754484)+(0.0921069092744)*0.5**o
        ref[3,3,1,1]=(0.0662157294518)/(o+1.)+(-0.250684686973)+(0.349078696154)*0.5**o
        ref[3,3,2,0]=(0.156645853389)/(o+1.)+(0.84327722735)+(-0.476684936147)*0.5**o
        ref[3,3,2,1]=(0.994145880767)/(o+1.)+(-0.465067169976)+(0.214713426286)*0.5**o
        ref[3,4,0,0]=(-0.990588506513)/(o+1.)+(0.218519355704)+(-0.88715423933)*0.5**o
        ref[3,4,0,1]=(-0.665732898082)/(o+1.)+(-0.496707601172)+(0.511114986801)*0.5**o
        ref[3,4,1,0]=(-1.18557015024)/(o+1.)+(0.587439848502)+(0.0739878219028)*0.5**o
        ref[3,4,1,1]=(0.725660134271)/(o+1.)+(0.786291076666)+(-0.141324267565)*0.5**o
        ref[3,4,2,0]=(-0.448948566381)/(o+1.)+(0.160689988193)+(-0.76912732001)*0.5**o
        ref[3,4,2,1]=(-0.571124298719)/(o+1.)+(-0.232553078069)+(-0.515945940217)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.311774053155)*x[0]**o+(0.52292841627)*x[0]+(-0.518467126196)*x[1]**o+(-0.11329290159)*x[1]
        ref=(-0.518467126196)/(o+1.)+(0.20481775734)+(0.311774053155)*0.5**o
      else:
        arg=(0.233765319091)*x[0]**o+(-0.453551209762)*x[0]+(0.920140231713)*x[1]**o+(0.741695479084)*x[1]+(-0.255834461065)*x[2]**o+(-0.102580899053)*x[2]
        ref=(0.664305770648)/(o+1.)+(0.0927816851348)+(0.233765319091)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(-0.651678780872)*x[0]**o+(-0.749709447394)*x[0]+(0.311088209082)*x[1]**o+(0.553081401193)*x[1]
        arg[1]=(0.657887728987)*x[0]**o+(0.052270143604)*x[0]+(-0.15507994277)*x[1]**o+(-0.30302150082)*x[1]
        ref[0]=(0.311088209082)/(o+1.)+(-0.0983140231004)+(-0.651678780872)*0.5**o
        ref[1]=(-0.15507994277)/(o+1.)+(-0.125375678608)+(0.657887728987)*0.5**o
      else:
        arg[0]=(0.844909347644)*x[0]**o+(-0.342697697483)*x[0]+(0.126242548186)*x[1]**o+(-0.854010732849)*x[1]+(-0.307627587461)*x[2]**o+(-0.998566290754)*x[2]
        arg[1]=(0.900073726701)*x[0]**o+(-0.703841701859)*x[0]+(-0.651690896499)*x[1]**o+(0.828362322181)*x[1]+(0.284410420895)*x[2]**o+(-0.394606348974)*x[2]
        ref[0]=(-0.181385039275)/(o+1.)+(-1.09763736054)+(0.844909347644)*0.5**o
        ref[1]=(-0.367280475604)/(o+1.)+(-0.135042864326)+(0.900073726701)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.263642681371)*x[0]**o+(0.668992571787)*x[0]+(0.873939846194)*x[1]**o+(0.784820355998)*x[1]
        arg[0,1]=(-0.389306088628)*x[0]**o+(-0.0549482389907)*x[0]+(-0.787101008457)*x[1]**o+(0.860345123812)*x[1]
        arg[0,2]=(-0.995721191961)*x[0]**o+(-0.587928136153)*x[0]+(-0.378312614832)*x[1]**o+(0.199765674281)*x[1]
        arg[0,3]=(-0.374679425145)*x[0]**o+(-0.503606183146)*x[0]+(0.23995738992)*x[1]**o+(-0.525359181744)*x[1]
        arg[0,4]=(-0.69923255665)*x[0]**o+(0.547634979941)*x[0]+(0.728792702739)*x[1]**o+(0.949888533828)*x[1]
        arg[1,0]=(-0.61856415528)*x[0]**o+(-0.0516239651093)*x[0]+(0.158516649979)*x[1]**o+(-0.0522288881011)*x[1]
        arg[1,1]=(0.0285524133857)*x[0]**o+(0.741357758307)*x[0]+(-0.0714860903847)*x[1]**o+(-0.539632750385)*x[1]
        arg[1,2]=(-0.291101401382)*x[0]**o+(-0.114900246437)*x[0]+(-0.982305031665)*x[1]**o+(0.0823478844055)*x[1]
        arg[1,3]=(-0.931305732525)*x[0]**o+(0.077332436417)*x[0]+(-0.880626154921)*x[1]**o+(-0.321222234507)*x[1]
        arg[1,4]=(-0.00778033208815)*x[0]**o+(-0.536626847957)*x[0]+(-0.599742714546)*x[1]**o+(0.454108299336)*x[1]
        arg[2,0]=(-0.575243090445)*x[0]**o+(-0.920247228564)*x[0]+(0.4569784462)*x[1]**o+(-0.105175798804)*x[1]
        arg[2,1]=(0.333985911748)*x[0]**o+(-0.017876213076)*x[0]+(0.475164778165)*x[1]**o+(0.561072178015)*x[1]
        arg[2,2]=(-0.559587786773)*x[0]**o+(-0.917134128672)*x[0]+(0.316654889327)*x[1]**o+(-0.640569595027)*x[1]
        arg[2,3]=(-0.140461324817)*x[0]**o+(0.259052969883)*x[0]+(0.397898509159)*x[1]**o+(0.932233331785)*x[1]
        arg[2,4]=(0.368626304985)*x[0]**o+(0.616310502384)*x[0]+(-0.756845293566)*x[1]**o+(-0.920088968132)*x[1]
        arg[3,0]=(-0.824398741087)*x[0]**o+(0.0498115894252)*x[0]+(-0.5256621655)*x[1]**o+(0.114806237108)*x[1]
        arg[3,1]=(-0.77671057316)*x[0]**o+(0.569249443162)*x[0]+(0.416628465801)*x[1]**o+(0.0603627487913)*x[1]
        arg[3,2]=(-0.247875220913)*x[0]**o+(0.19548828878)*x[0]+(0.643226101838)*x[1]**o+(-0.154558826499)*x[1]
        arg[3,3]=(-0.918937118963)*x[0]**o+(0.193777174417)*x[0]+(-0.906042347107)*x[1]**o+(0.872033230033)*x[1]
        arg[3,4]=(-0.628918336086)*x[0]**o+(0.823242191714)*x[0]+(-0.593691166712)*x[1]**o+(-0.661715840814)*x[1]
        ref[0,0]=(0.873939846194)/(o+1.)+(0.726906463892)+(-0.263642681371)*0.5**o
        ref[0,1]=(-0.787101008457)/(o+1.)+(0.402698442411)+(-0.389306088628)*0.5**o
        ref[0,2]=(-0.378312614832)/(o+1.)+(-0.194081230936)+(-0.995721191961)*0.5**o
        ref[0,3]=(0.23995738992)/(o+1.)+(-0.514482682445)+(-0.374679425145)*0.5**o
        ref[0,4]=(0.728792702739)/(o+1.)+(0.748761756885)+(-0.69923255665)*0.5**o
        ref[1,0]=(0.158516649979)/(o+1.)+(-0.0519264266052)+(-0.61856415528)*0.5**o
        ref[1,1]=(-0.0714860903847)/(o+1.)+(0.100862503961)+(0.0285524133857)*0.5**o
        ref[1,2]=(-0.982305031665)/(o+1.)+(-0.0162761810159)+(-0.291101401382)*0.5**o
        ref[1,3]=(-0.880626154921)/(o+1.)+(-0.121944899045)+(-0.931305732525)*0.5**o
        ref[1,4]=(-0.599742714546)/(o+1.)+(-0.0412592743102)+(-0.00778033208815)*0.5**o
        ref[2,0]=(0.4569784462)/(o+1.)+(-0.512711513684)+(-0.575243090445)*0.5**o
        ref[2,1]=(0.475164778165)/(o+1.)+(0.27159798247)+(0.333985911748)*0.5**o
        ref[2,2]=(0.316654889327)/(o+1.)+(-0.778851861849)+(-0.559587786773)*0.5**o
        ref[2,3]=(0.397898509159)/(o+1.)+(0.595643150834)+(-0.140461324817)*0.5**o
        ref[2,4]=(-0.756845293566)/(o+1.)+(-0.151889232874)+(0.368626304985)*0.5**o
        ref[3,0]=(-0.5256621655)/(o+1.)+(0.0823089132668)+(-0.824398741087)*0.5**o
        ref[3,1]=(0.416628465801)/(o+1.)+(0.314806095977)+(-0.77671057316)*0.5**o
        ref[3,2]=(0.643226101838)/(o+1.)+(0.0204647311405)+(-0.247875220913)*0.5**o
        ref[3,3]=(-0.906042347107)/(o+1.)+(0.532905202225)+(-0.918937118963)*0.5**o
        ref[3,4]=(-0.593691166712)/(o+1.)+(0.0807631754499)+(-0.628918336086)*0.5**o
      else:
        arg[0,0]=(0.748857269399)*x[0]**o+(-0.21982973469)*x[0]+(-0.765987921295)*x[1]**o+(0.556577019909)*x[1]+(-0.0438548079278)*x[2]**o+(0.543610874817)*x[2]
        arg[0,1]=(-0.784526004748)*x[0]**o+(-0.949423172035)*x[0]+(-0.29888188317)*x[1]**o+(-0.850638766848)*x[1]+(-0.853027558387)*x[2]**o+(-0.846378486588)*x[2]
        arg[0,2]=(0.53445882525)*x[0]**o+(-0.653655652989)*x[0]+(-0.968129875689)*x[1]**o+(-0.603598808323)*x[1]+(0.545841014272)*x[2]**o+(0.126131861082)*x[2]
        arg[0,3]=(0.181662498503)*x[0]**o+(-0.769571080865)*x[0]+(0.771109181078)*x[1]**o+(0.980677373303)*x[1]+(-0.625057940268)*x[2]**o+(0.0585672357346)*x[2]
        arg[0,4]=(-0.0320907540755)*x[0]**o+(-0.519517369917)*x[0]+(-0.365190231996)*x[1]**o+(-0.192392652661)*x[1]+(-0.693267159516)*x[2]**o+(0.106458640425)*x[2]
        arg[1,0]=(0.414286358161)*x[0]**o+(0.0223282279507)*x[0]+(0.215881279469)*x[1]**o+(0.68990977849)*x[1]+(-0.532075885581)*x[2]**o+(-0.269164904232)*x[2]
        arg[1,1]=(-0.0445879250589)*x[0]**o+(0.889985400599)*x[0]+(0.842050582947)*x[1]**o+(-0.0275307884642)*x[1]+(0.774650107671)*x[2]**o+(0.19446514971)*x[2]
        arg[1,2]=(0.913032880034)*x[0]**o+(-0.286537470008)*x[0]+(0.0902778267032)*x[1]**o+(0.548596933564)*x[1]+(-0.996100405074)*x[2]**o+(-0.509554019549)*x[2]
        arg[1,3]=(0.44308812765)*x[0]**o+(-0.0641227726924)*x[0]+(0.484249008075)*x[1]**o+(0.581496384894)*x[1]+(0.592538533183)*x[2]**o+(-0.821638541512)*x[2]
        arg[1,4]=(-0.898304735153)*x[0]**o+(0.333217967121)*x[0]+(-0.788218899371)*x[1]**o+(0.579723459039)*x[1]+(0.467006473159)*x[2]**o+(-0.277113860535)*x[2]
        arg[2,0]=(0.515308716564)*x[0]**o+(0.255947750577)*x[0]+(0.232578337945)*x[1]**o+(0.900611887592)*x[1]+(-0.247710965611)*x[2]**o+(-0.985523228221)*x[2]
        arg[2,1]=(0.611196756618)*x[0]**o+(0.146822999187)*x[0]+(-0.318793147278)*x[1]**o+(0.790772432028)*x[1]+(0.138036152668)*x[2]**o+(-0.800234207543)*x[2]
        arg[2,2]=(-0.293576196546)*x[0]**o+(-0.521059592737)*x[0]+(0.301178293957)*x[1]**o+(-0.141430900198)*x[1]+(-0.782381874467)*x[2]**o+(0.92221179729)*x[2]
        arg[2,3]=(0.673951778483)*x[0]**o+(-0.335068772752)*x[0]+(0.408793200723)*x[1]**o+(0.36906212176)*x[1]+(0.246292093855)*x[2]**o+(0.356147233675)*x[2]
        arg[2,4]=(-0.218899302236)*x[0]**o+(0.271285962886)*x[0]+(-0.16364707283)*x[1]**o+(0.177195607727)*x[1]+(-0.287331396699)*x[2]**o+(-0.465391460497)*x[2]
        arg[3,0]=(-0.369727856735)*x[0]**o+(-0.44351931666)*x[0]+(0.0375097686939)*x[1]**o+(0.276779121402)*x[1]+(0.662505672881)*x[2]**o+(-0.642709866174)*x[2]
        arg[3,1]=(0.444940917816)*x[0]**o+(-0.178926126454)*x[0]+(0.824660645994)*x[1]**o+(0.825996820355)*x[1]+(0.798260651848)*x[2]**o+(0.331744022239)*x[2]
        arg[3,2]=(-0.82489261483)*x[0]**o+(-0.340862079766)*x[0]+(-0.998375587383)*x[1]**o+(0.880222106116)*x[1]+(0.656096580473)*x[2]**o+(0.530854486127)*x[2]
        arg[3,3]=(0.732863411435)*x[0]**o+(-0.324252004147)*x[0]+(0.789472654884)*x[1]**o+(-0.207794796011)*x[1]+(0.50903180567)*x[2]**o+(0.724319112504)*x[2]
        arg[3,4]=(-0.980748139471)*x[0]**o+(0.0974331161906)*x[0]+(0.196275528452)*x[1]**o+(0.305231910003)*x[1]+(-0.0417825350947)*x[2]**o+(-0.2916834261)*x[2]
        ref[0,0]=(-0.809842729223)/(o+1.)+(0.440179080018)+(0.748857269399)*0.5**o
        ref[0,1]=(-1.15190944156)/(o+1.)+(-1.32322021274)+(-0.784526004748)*0.5**o
        ref[0,2]=(-0.422288861417)/(o+1.)+(-0.565561300115)+(0.53445882525)*0.5**o
        ref[0,3]=(0.146051240811)/(o+1.)+(0.134836764087)+(0.181662498503)*0.5**o
        ref[0,4]=(-1.05845739151)/(o+1.)+(-0.302725691077)+(-0.0320907540755)*0.5**o
        ref[1,0]=(-0.316194606113)/(o+1.)+(0.221536551104)+(0.414286358161)*0.5**o
        ref[1,1]=(1.61670069062)/(o+1.)+(0.528459880922)+(-0.0445879250589)*0.5**o
        ref[1,2]=(-0.905822578371)/(o+1.)+(-0.123747277996)+(0.913032880034)*0.5**o
        ref[1,3]=(1.07678754126)/(o+1.)+(-0.152132464655)+(0.44308812765)*0.5**o
        ref[1,4]=(-0.321212426212)/(o+1.)+(0.317913782812)+(-0.898304735153)*0.5**o
        ref[2,0]=(-0.015132627666)/(o+1.)+(0.085518204974)+(0.515308716564)*0.5**o
        ref[2,1]=(-0.18075699461)/(o+1.)+(0.0686806118359)+(0.611196756618)*0.5**o
        ref[2,2]=(-0.481203580509)/(o+1.)+(0.129860652178)+(-0.293576196546)*0.5**o
        ref[2,3]=(0.655085294578)/(o+1.)+(0.195070291341)+(0.673951778483)*0.5**o
        ref[2,4]=(-0.450978469529)/(o+1.)+(-0.00845494494169)+(-0.218899302236)*0.5**o
        ref[3,0]=(0.700015441575)/(o+1.)+(-0.404725030716)+(-0.369727856735)*0.5**o
        ref[3,1]=(1.62292129784)/(o+1.)+(0.48940735807)+(0.444940917816)*0.5**o
        ref[3,2]=(-0.342279006911)/(o+1.)+(0.535107256239)+(-0.82489261483)*0.5**o
        ref[3,3]=(1.29850446055)/(o+1.)+(0.0961361561734)+(0.732863411435)*0.5**o
        ref[3,4]=(0.154492993358)/(o+1.)+(0.055490800047)+(-0.980748139471)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.33788666753)*x[0]**o+(0.0147671449643)*x[0]+(0.14066054662)*x[1]**o+(-0.675231899756)*x[1]
        arg[0,0,1]=(0.0100961441611)*x[0]**o+(0.401304171984)*x[0]+(-0.594786282119)*x[1]**o+(-0.561543495067)*x[1]
        arg[0,1,0]=(-0.118311175162)*x[0]**o+(0.62454992688)*x[0]+(-0.8163725079)*x[1]**o+(-0.756355655277)*x[1]
        arg[0,1,1]=(0.586991709411)*x[0]**o+(-0.600757941586)*x[0]+(0.0953383109011)*x[1]**o+(0.317827629801)*x[1]
        arg[1,0,0]=(-0.251125236294)*x[0]**o+(0.166603616954)*x[0]+(0.978704943828)*x[1]**o+(0.589215389731)*x[1]
        arg[1,0,1]=(0.0543182786738)*x[0]**o+(-0.334756305272)*x[0]+(-0.919571042987)*x[1]**o+(-0.0425734727213)*x[1]
        arg[1,1,0]=(-0.867177621518)*x[0]**o+(-0.0750659593232)*x[0]+(-0.227277029921)*x[1]**o+(0.361120763296)*x[1]
        arg[1,1,1]=(0.85129514889)*x[0]**o+(-0.441962000795)*x[0]+(0.425335668395)*x[1]**o+(0.204091109081)*x[1]
        arg[2,0,0]=(-0.537930401585)*x[0]**o+(-0.206510941636)*x[0]+(-0.247374421196)*x[1]**o+(0.67043841725)*x[1]
        arg[2,0,1]=(-0.101273864474)*x[0]**o+(0.283769727506)*x[0]+(-0.0850059833654)*x[1]**o+(-0.71291661379)*x[1]
        arg[2,1,0]=(0.684430088582)*x[0]**o+(0.107733116086)*x[0]+(0.40398578341)*x[1]**o+(-0.525004991854)*x[1]
        arg[2,1,1]=(-0.00801759014554)*x[0]**o+(-0.214858014851)*x[0]+(-0.427822011518)*x[1]**o+(0.0168962506945)*x[1]
        arg[3,0,0]=(0.528563052182)*x[0]**o+(-0.630485796029)*x[0]+(0.0525765317703)*x[1]**o+(-0.469639006097)*x[1]
        arg[3,0,1]=(0.178622695427)*x[0]**o+(0.0952996500383)*x[0]+(-0.785851578496)*x[1]**o+(0.856665398338)*x[1]
        arg[3,1,0]=(0.547391926072)*x[0]**o+(-0.0179195091852)*x[0]+(-0.831368961862)*x[1]**o+(-0.437229107453)*x[1]
        arg[3,1,1]=(-0.657011566453)*x[0]**o+(0.530176142378)*x[0]+(0.649145894746)*x[1]**o+(-0.755185337598)*x[1]
        arg[4,0,0]=(-0.188305872652)*x[0]**o+(0.0207245930982)*x[0]+(-0.590757998421)*x[1]**o+(-0.793662033952)*x[1]
        arg[4,0,1]=(0.417712824349)*x[0]**o+(-0.138327612527)*x[0]+(-0.381595921766)*x[1]**o+(0.0287368364725)*x[1]
        arg[4,1,0]=(-0.83750540403)*x[0]**o+(0.247261273008)*x[0]+(-0.0769629479034)*x[1]**o+(0.0693501875686)*x[1]
        arg[4,1,1]=(0.697434044074)*x[0]**o+(0.396582403702)*x[0]+(0.736132107034)*x[1]**o+(0.482245183242)*x[1]
        arg[5,0,0]=(0.769936158184)*x[0]**o+(0.808871896459)*x[0]+(-0.559205314785)*x[1]**o+(-0.258014827146)*x[1]
        arg[5,0,1]=(0.667132553969)*x[0]**o+(0.618184492083)*x[0]+(0.475852440157)*x[1]**o+(0.084499293443)*x[1]
        arg[5,1,0]=(0.910882712823)*x[0]**o+(0.153897139268)*x[0]+(-0.61197280039)*x[1]**o+(0.441331985295)*x[1]
        arg[5,1,1]=(-0.290188623857)*x[0]**o+(0.318266320131)*x[0]+(0.9587856247)*x[1]**o+(0.311981486508)*x[1]
        ref[0,0,0]=(0.14066054662)/(o+1.)+(-0.330232377396)+(-0.33788666753)*0.5**o
        ref[0,0,1]=(-0.594786282119)/(o+1.)+(-0.0801196615418)+(0.0100961441611)*0.5**o
        ref[0,1,0]=(-0.8163725079)/(o+1.)+(-0.0659028641985)+(-0.118311175162)*0.5**o
        ref[0,1,1]=(0.0953383109011)/(o+1.)+(-0.141465155893)+(0.586991709411)*0.5**o
        ref[1,0,0]=(0.978704943828)/(o+1.)+(0.377909503343)+(-0.251125236294)*0.5**o
        ref[1,0,1]=(-0.919571042987)/(o+1.)+(-0.188664888997)+(0.0543182786738)*0.5**o
        ref[1,1,0]=(-0.227277029921)/(o+1.)+(0.143027401987)+(-0.867177621518)*0.5**o
        ref[1,1,1]=(0.425335668395)/(o+1.)+(-0.118935445857)+(0.85129514889)*0.5**o
        ref[2,0,0]=(-0.247374421196)/(o+1.)+(0.231963737807)+(-0.537930401585)*0.5**o
        ref[2,0,1]=(-0.0850059833654)/(o+1.)+(-0.214573443142)+(-0.101273864474)*0.5**o
        ref[2,1,0]=(0.40398578341)/(o+1.)+(-0.208635937884)+(0.684430088582)*0.5**o
        ref[2,1,1]=(-0.427822011518)/(o+1.)+(-0.0989808820785)+(-0.00801759014554)*0.5**o
        ref[3,0,0]=(0.0525765317703)/(o+1.)+(-0.550062401063)+(0.528563052182)*0.5**o
        ref[3,0,1]=(-0.785851578496)/(o+1.)+(0.475982524188)+(0.178622695427)*0.5**o
        ref[3,1,0]=(-0.831368961862)/(o+1.)+(-0.227574308319)+(0.547391926072)*0.5**o
        ref[3,1,1]=(0.649145894746)/(o+1.)+(-0.11250459761)+(-0.657011566453)*0.5**o
        ref[4,0,0]=(-0.590757998421)/(o+1.)+(-0.386468720427)+(-0.188305872652)*0.5**o
        ref[4,0,1]=(-0.381595921766)/(o+1.)+(-0.0547953880271)+(0.417712824349)*0.5**o
        ref[4,1,0]=(-0.0769629479034)/(o+1.)+(0.158305730288)+(-0.83750540403)*0.5**o
        ref[4,1,1]=(0.736132107034)/(o+1.)+(0.439413793472)+(0.697434044074)*0.5**o
        ref[5,0,0]=(-0.559205314785)/(o+1.)+(0.275428534657)+(0.769936158184)*0.5**o
        ref[5,0,1]=(0.475852440157)/(o+1.)+(0.351341892763)+(0.667132553969)*0.5**o
        ref[5,1,0]=(-0.61197280039)/(o+1.)+(0.297614562282)+(0.910882712823)*0.5**o
        ref[5,1,1]=(0.9587856247)/(o+1.)+(0.315123903319)+(-0.290188623857)*0.5**o
      else:
        arg[0,0,0]=(0.292346424086)*x[0]**o+(0.105058465241)*x[0]+(-0.281467283372)*x[1]**o+(0.828345267021)*x[1]+(0.741242558638)*x[2]**o+(0.0858780765977)*x[2]
        arg[0,0,1]=(-0.460556250835)*x[0]**o+(-0.845199338719)*x[0]+(-0.547000154821)*x[1]**o+(-0.972236450512)*x[1]+(0.823031420765)*x[2]**o+(0.278285294464)*x[2]
        arg[0,1,0]=(0.380027894487)*x[0]**o+(-0.579359210385)*x[0]+(-0.760849798908)*x[1]**o+(0.0065870459398)*x[1]+(-0.869735580799)*x[2]**o+(-0.937378284335)*x[2]
        arg[0,1,1]=(0.948682597897)*x[0]**o+(-0.882044080711)*x[0]+(-0.694398833899)*x[1]**o+(0.883635235205)*x[1]+(-0.473761903609)*x[2]**o+(0.961453193379)*x[2]
        arg[1,0,0]=(0.357218603696)*x[0]**o+(-0.694504294144)*x[0]+(-0.187119222895)*x[1]**o+(0.727463485088)*x[1]+(0.150410254875)*x[2]**o+(-0.373430202907)*x[2]
        arg[1,0,1]=(-0.776893233002)*x[0]**o+(0.565238637388)*x[0]+(-0.155674628662)*x[1]**o+(-0.938226259914)*x[1]+(0.575183633137)*x[2]**o+(0.319871583783)*x[2]
        arg[1,1,0]=(0.593512911569)*x[0]**o+(-0.0578762085927)*x[0]+(-0.913450847289)*x[1]**o+(0.7074653217)*x[1]+(0.991175636143)*x[2]**o+(-0.535632452281)*x[2]
        arg[1,1,1]=(-0.154542349616)*x[0]**o+(-0.841642050346)*x[0]+(0.878833704431)*x[1]**o+(0.85568524402)*x[1]+(-0.710374336281)*x[2]**o+(0.346540187252)*x[2]
        arg[2,0,0]=(-0.765160864305)*x[0]**o+(0.608017010487)*x[0]+(0.841265469171)*x[1]**o+(-0.409854579584)*x[1]+(0.728931324538)*x[2]**o+(0.121315209213)*x[2]
        arg[2,0,1]=(0.0397067301375)*x[0]**o+(-0.0451925117802)*x[0]+(-0.412724345414)*x[1]**o+(-0.447675002107)*x[1]+(0.141728105561)*x[2]**o+(0.923970189305)*x[2]
        arg[2,1,0]=(-0.237044096923)*x[0]**o+(-0.826168294669)*x[0]+(0.19833520121)*x[1]**o+(-0.796076984382)*x[1]+(0.367399642334)*x[2]**o+(0.71172848801)*x[2]
        arg[2,1,1]=(-0.3881830464)*x[0]**o+(0.988354722657)*x[0]+(0.762756805728)*x[1]**o+(0.551778683619)*x[1]+(0.975932105591)*x[2]**o+(-0.173192098292)*x[2]
        arg[3,0,0]=(-0.226163777371)*x[0]**o+(-0.301094659695)*x[0]+(0.0682567814743)*x[1]**o+(-0.154914902824)*x[1]+(0.984669238436)*x[2]**o+(-0.431011897738)*x[2]
        arg[3,0,1]=(-0.47230847571)*x[0]**o+(-0.579315893476)*x[0]+(-0.989289806396)*x[1]**o+(-0.673951334134)*x[1]+(0.98143767614)*x[2]**o+(-0.311074531383)*x[2]
        arg[3,1,0]=(-0.276309910109)*x[0]**o+(-0.70244304164)*x[0]+(-0.963669800079)*x[1]**o+(-0.947723474952)*x[1]+(0.513451083144)*x[2]**o+(-0.0651803692263)*x[2]
        arg[3,1,1]=(0.65175527234)*x[0]**o+(-0.294569414615)*x[0]+(-0.32846386125)*x[1]**o+(-0.868822318527)*x[1]+(0.268414480892)*x[2]**o+(-0.866835042217)*x[2]
        arg[4,0,0]=(0.304002845434)*x[0]**o+(0.938467246002)*x[0]+(-0.106648262772)*x[1]**o+(0.227863374571)*x[1]+(-0.789305204045)*x[2]**o+(-0.171526142133)*x[2]
        arg[4,0,1]=(0.70982305774)*x[0]**o+(0.404228697442)*x[0]+(0.190029230022)*x[1]**o+(-0.825276083634)*x[1]+(-0.739774636921)*x[2]**o+(-0.726529211281)*x[2]
        arg[4,1,0]=(-0.238487290076)*x[0]**o+(-0.4901234628)*x[0]+(-0.836670757945)*x[1]**o+(0.0841493712661)*x[1]+(-0.685144435198)*x[2]**o+(-0.304978244786)*x[2]
        arg[4,1,1]=(0.264176905985)*x[0]**o+(-0.410582158319)*x[0]+(-0.271915471538)*x[1]**o+(-0.996082427599)*x[1]+(-0.466654446945)*x[2]**o+(-0.723243781429)*x[2]
        arg[5,0,0]=(0.379390860481)*x[0]**o+(-0.78556086955)*x[0]+(0.189126836375)*x[1]**o+(0.235947881164)*x[1]+(0.149142076852)*x[2]**o+(-0.944419890441)*x[2]
        arg[5,0,1]=(0.768867280189)*x[0]**o+(-0.728364777249)*x[0]+(0.60414049451)*x[1]**o+(-0.538876171991)*x[1]+(-0.0947560131508)*x[2]**o+(0.474986163327)*x[2]
        arg[5,1,0]=(-0.228069986734)*x[0]**o+(-0.680841588894)*x[0]+(-0.663806510587)*x[1]**o+(-0.852775021416)*x[1]+(-0.59539582999)*x[2]**o+(0.52369159839)*x[2]
        arg[5,1,1]=(0.177872029425)*x[0]**o+(0.93423742256)*x[0]+(-0.0369475988192)*x[1]**o+(0.242202072343)*x[1]+(0.526144455402)*x[2]**o+(0.0217547142691)*x[2]
        ref[0,0,0]=(0.459775275267)/(o+1.)+(0.50964090443)+(0.292346424086)*0.5**o
        ref[0,0,1]=(0.276031265944)/(o+1.)+(-0.769575247383)+(-0.460556250835)*0.5**o
        ref[0,1,0]=(-1.63058537971)/(o+1.)+(-0.75507522439)+(0.380027894487)*0.5**o
        ref[0,1,1]=(-1.16816073751)/(o+1.)+(0.481522173937)+(0.948682597897)*0.5**o
        ref[1,0,0]=(-0.0367089680209)/(o+1.)+(-0.170235505982)+(0.357218603696)*0.5**o
        ref[1,0,1]=(0.419509004475)/(o+1.)+(-0.0265580193719)+(-0.776893233002)*0.5**o
        ref[1,1,0]=(0.0777247888533)/(o+1.)+(0.0569783304129)+(0.593512911569)*0.5**o
        ref[1,1,1]=(0.168459368149)/(o+1.)+(0.180291690463)+(-0.154542349616)*0.5**o
        ref[2,0,0]=(1.57019679371)/(o+1.)+(0.159738820058)+(-0.765160864305)*0.5**o
        ref[2,0,1]=(-0.270996239853)/(o+1.)+(0.215551337709)+(0.0397067301375)*0.5**o
        ref[2,1,0]=(0.565734843545)/(o+1.)+(-0.455258395521)+(-0.237044096923)*0.5**o
        ref[2,1,1]=(1.73868891132)/(o+1.)+(0.683470653992)+(-0.3881830464)*0.5**o
        ref[3,0,0]=(1.05292601991)/(o+1.)+(-0.443510730129)+(-0.226163777371)*0.5**o
        ref[3,0,1]=(-0.00785213025503)/(o+1.)+(-0.782170879496)+(-0.47230847571)*0.5**o
        ref[3,1,0]=(-0.450218716935)/(o+1.)+(-0.857673442909)+(-0.276309910109)*0.5**o
        ref[3,1,1]=(-0.0600493803575)/(o+1.)+(-1.01511338768)+(0.65175527234)*0.5**o
        ref[4,0,0]=(-0.895953466817)/(o+1.)+(0.49740223922)+(0.304002845434)*0.5**o
        ref[4,0,1]=(-0.549745406899)/(o+1.)+(-0.573788298736)+(0.70982305774)*0.5**o
        ref[4,1,0]=(-1.52181519314)/(o+1.)+(-0.35547616816)+(-0.238487290076)*0.5**o
        ref[4,1,1]=(-0.738569918483)/(o+1.)+(-1.06495418367)+(0.264176905985)*0.5**o
        ref[5,0,0]=(0.338268913228)/(o+1.)+(-0.747016439414)+(0.379390860481)*0.5**o
        ref[5,0,1]=(0.50938448136)/(o+1.)+(-0.396127392956)+(0.768867280189)*0.5**o
        ref[5,1,0]=(-1.25920234058)/(o+1.)+(-0.50496250596)+(-0.228069986734)*0.5**o
        ref[5,1,1]=(0.489196856583)/(o+1.)+(0.599097104586)+(0.177872029425)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunctionOnContactZero

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.423100205741)*x[0]**o+(-0.681307394941)*x[0]+(-0.848175950825)*x[1]**o+(0.559338235783)*x[1]
        arg[0,0,0,1]=(-0.739687474039)*x[0]**o+(-0.202202358237)*x[0]+(-0.838397517739)*x[1]**o+(-0.377580621185)*x[1]
        arg[0,0,1,0]=(-0.423462958438)*x[0]**o+(-0.335929684088)*x[0]+(-0.616976579504)*x[1]**o+(-0.77803694581)*x[1]
        arg[0,0,1,1]=(-0.432533121428)*x[0]**o+(0.0723865130937)*x[0]+(-0.953014661461)*x[1]**o+(0.92761707252)*x[1]
        arg[0,0,2,0]=(-0.891613775753)*x[0]**o+(0.780592229933)*x[0]+(-0.156191993934)*x[1]**o+(0.985464289737)*x[1]
        arg[0,0,2,1]=(0.909573147888)*x[0]**o+(-0.225603851475)*x[0]+(-0.68287286477)*x[1]**o+(0.330105905794)*x[1]
        arg[0,1,0,0]=(-0.39808363424)*x[0]**o+(-0.290872560992)*x[0]+(-0.631624427826)*x[1]**o+(-0.389239289826)*x[1]
        arg[0,1,0,1]=(-0.748658247224)*x[0]**o+(0.22314629362)*x[0]+(-0.0750972377036)*x[1]**o+(0.908914722147)*x[1]
        arg[0,1,1,0]=(0.877627912148)*x[0]**o+(-0.988434656301)*x[0]+(-0.163646238293)*x[1]**o+(-0.169410711361)*x[1]
        arg[0,1,1,1]=(-0.592580581615)*x[0]**o+(0.997989912544)*x[0]+(0.0667564973731)*x[1]**o+(-0.121580259121)*x[1]
        arg[0,1,2,0]=(-0.505569292541)*x[0]**o+(-0.534428841308)*x[0]+(-0.899276958155)*x[1]**o+(-0.833972415825)*x[1]
        arg[0,1,2,1]=(-0.215752366701)*x[0]**o+(-0.249993262232)*x[0]+(0.912452712698)*x[1]**o+(0.0601524115958)*x[1]
        arg[0,2,0,0]=(0.684112319204)*x[0]**o+(0.343018234122)*x[0]+(0.267409202785)*x[1]**o+(0.7781133929)*x[1]
        arg[0,2,0,1]=(0.552438501301)*x[0]**o+(-0.101558113559)*x[0]+(-0.152472725029)*x[1]**o+(-0.8771654517)*x[1]
        arg[0,2,1,0]=(-0.31213422297)*x[0]**o+(0.597042319279)*x[0]+(-0.873134889407)*x[1]**o+(-0.68920786345)*x[1]
        arg[0,2,1,1]=(0.883048532408)*x[0]**o+(-0.653997880306)*x[0]+(-0.761310589424)*x[1]**o+(-0.698342623962)*x[1]
        arg[0,2,2,0]=(-0.324489237508)*x[0]**o+(-0.717724692716)*x[0]+(-0.936381308418)*x[1]**o+(0.712416971077)*x[1]
        arg[0,2,2,1]=(-0.971735193469)*x[0]**o+(0.0943629094751)*x[0]+(0.419811305063)*x[1]**o+(0.125623316569)*x[1]
        arg[0,3,0,0]=(-0.305553454905)*x[0]**o+(-0.936112994503)*x[0]+(0.707279059071)*x[1]**o+(-0.794754809337)*x[1]
        arg[0,3,0,1]=(0.803934620926)*x[0]**o+(-0.632020033455)*x[0]+(0.819264129099)*x[1]**o+(0.770744069672)*x[1]
        arg[0,3,1,0]=(0.996632326055)*x[0]**o+(-0.537850178404)*x[0]+(-0.60262522048)*x[1]**o+(-0.431667704232)*x[1]
        arg[0,3,1,1]=(0.402799911553)*x[0]**o+(0.155798565063)*x[0]+(0.722837842981)*x[1]**o+(0.462463978227)*x[1]
        arg[0,3,2,0]=(0.937966588808)*x[0]**o+(-0.797151899551)*x[0]+(0.62363567076)*x[1]**o+(-0.60458214868)*x[1]
        arg[0,3,2,1]=(-0.0389713032301)*x[0]**o+(0.497820326557)*x[0]+(-0.950660964237)*x[1]**o+(0.0247372614337)*x[1]
        arg[0,4,0,0]=(-0.589529640529)*x[0]**o+(-0.71761078076)*x[0]+(-0.872148906918)*x[1]**o+(-0.666618835503)*x[1]
        arg[0,4,0,1]=(0.78615039967)*x[0]**o+(0.99287672137)*x[0]+(0.329194160571)*x[1]**o+(0.309370990805)*x[1]
        arg[0,4,1,0]=(-0.801187683466)*x[0]**o+(0.476957978386)*x[0]+(0.107007555422)*x[1]**o+(-0.211909676249)*x[1]
        arg[0,4,1,1]=(0.817551079419)*x[0]**o+(0.958544826763)*x[0]+(-0.720479146591)*x[1]**o+(0.0769824586505)*x[1]
        arg[0,4,2,0]=(-0.245391293539)*x[0]**o+(0.240588876322)*x[0]+(0.766214937295)*x[1]**o+(0.625282150348)*x[1]
        arg[0,4,2,1]=(-0.780163545255)*x[0]**o+(-0.84424180556)*x[0]+(-0.993826355778)*x[1]**o+(-0.708432857746)*x[1]
        arg[1,0,0,0]=(0.0998104654851)*x[0]**o+(-0.933870759023)*x[0]+(-0.00667060999489)*x[1]**o+(-0.304328472241)*x[1]
        arg[1,0,0,1]=(0.716858223574)*x[0]**o+(0.0907083966301)*x[0]+(0.118004968786)*x[1]**o+(-0.989434356769)*x[1]
        arg[1,0,1,0]=(0.433505460082)*x[0]**o+(0.927792393402)*x[0]+(0.10117572965)*x[1]**o+(-0.719175235748)*x[1]
        arg[1,0,1,1]=(0.93394463477)*x[0]**o+(0.30554551436)*x[0]+(-0.221528653572)*x[1]**o+(-0.965153365935)*x[1]
        arg[1,0,2,0]=(0.217926136552)*x[0]**o+(0.75089182057)*x[0]+(0.798438119547)*x[1]**o+(-0.131885152617)*x[1]
        arg[1,0,2,1]=(0.489885370386)*x[0]**o+(-0.958785443386)*x[0]+(-0.934023631551)*x[1]**o+(-0.120727978484)*x[1]
        arg[1,1,0,0]=(-0.104659311537)*x[0]**o+(0.323817119133)*x[0]+(0.207178629335)*x[1]**o+(0.132677138305)*x[1]
        arg[1,1,0,1]=(0.855593354128)*x[0]**o+(-0.319099268106)*x[0]+(-0.75522659926)*x[1]**o+(0.398648791668)*x[1]
        arg[1,1,1,0]=(0.165499959353)*x[0]**o+(0.025416591231)*x[0]+(-0.454711374202)*x[1]**o+(-0.149643847094)*x[1]
        arg[1,1,1,1]=(0.668195333828)*x[0]**o+(-0.954315351102)*x[0]+(-0.15549992542)*x[1]**o+(0.960203842246)*x[1]
        arg[1,1,2,0]=(-0.0830862028828)*x[0]**o+(0.104413343709)*x[0]+(0.0666836881524)*x[1]**o+(-0.909918510114)*x[1]
        arg[1,1,2,1]=(-0.614061685098)*x[0]**o+(-0.696536572273)*x[0]+(0.674867602585)*x[1]**o+(0.742233774001)*x[1]
        arg[1,2,0,0]=(-0.271025278576)*x[0]**o+(0.194146585691)*x[0]+(0.369545867263)*x[1]**o+(-0.444209773003)*x[1]
        arg[1,2,0,1]=(-0.646570016511)*x[0]**o+(-0.1181935343)*x[0]+(-0.473468176784)*x[1]**o+(0.980639666617)*x[1]
        arg[1,2,1,0]=(-0.274558546949)*x[0]**o+(-0.994564294804)*x[0]+(-0.710677848448)*x[1]**o+(-0.471981673909)*x[1]
        arg[1,2,1,1]=(0.920326662139)*x[0]**o+(-0.503477706392)*x[0]+(-0.488075258363)*x[1]**o+(-0.390701293427)*x[1]
        arg[1,2,2,0]=(-0.734779609501)*x[0]**o+(-0.211522964885)*x[0]+(-0.209704158218)*x[1]**o+(0.602143419316)*x[1]
        arg[1,2,2,1]=(-0.71042927509)*x[0]**o+(-0.320801535296)*x[0]+(-0.307645214006)*x[1]**o+(0.902857741355)*x[1]
        arg[1,3,0,0]=(0.430044544461)*x[0]**o+(-0.321809854969)*x[0]+(-0.202705286059)*x[1]**o+(-0.136245903236)*x[1]
        arg[1,3,0,1]=(-0.0215232394148)*x[0]**o+(0.560515199075)*x[0]+(0.183422356152)*x[1]**o+(0.0500950744933)*x[1]
        arg[1,3,1,0]=(0.956072319541)*x[0]**o+(-0.910407504398)*x[0]+(0.710351406673)*x[1]**o+(-0.673149941902)*x[1]
        arg[1,3,1,1]=(0.512105745894)*x[0]**o+(0.617656019287)*x[0]+(0.178878380134)*x[1]**o+(-0.304972517759)*x[1]
        arg[1,3,2,0]=(0.505194196756)*x[0]**o+(-0.588278604691)*x[0]+(0.657312673212)*x[1]**o+(-0.198666173753)*x[1]
        arg[1,3,2,1]=(0.562199962955)*x[0]**o+(-0.855080919048)*x[0]+(0.798106457622)*x[1]**o+(-0.462368594285)*x[1]
        arg[1,4,0,0]=(-0.95103819151)*x[0]**o+(0.996179304819)*x[0]+(0.720809353558)*x[1]**o+(-0.357882371835)*x[1]
        arg[1,4,0,1]=(-0.337212448334)*x[0]**o+(-0.647939092937)*x[0]+(0.432152553594)*x[1]**o+(0.284393585377)*x[1]
        arg[1,4,1,0]=(0.286170290079)*x[0]**o+(0.813168170391)*x[0]+(-0.178814496497)*x[1]**o+(0.414687546663)*x[1]
        arg[1,4,1,1]=(0.00961378883254)*x[0]**o+(-0.946718643529)*x[0]+(0.828780485661)*x[1]**o+(0.54461217692)*x[1]
        arg[1,4,2,0]=(0.784403348421)*x[0]**o+(0.430078584466)*x[0]+(-0.542741131881)*x[1]**o+(0.0290828265624)*x[1]
        arg[1,4,2,1]=(-0.246343667616)*x[0]**o+(-0.27406067089)*x[0]+(-0.938704878102)*x[1]**o+(0.907921504301)*x[1]
        arg[2,0,0,0]=(-0.551056874357)*x[0]**o+(-0.697006980356)*x[0]+(-0.733447125487)*x[1]**o+(-0.241902075189)*x[1]
        arg[2,0,0,1]=(-0.75556701159)*x[0]**o+(0.155303331948)*x[0]+(0.480546881289)*x[1]**o+(-0.420954472057)*x[1]
        arg[2,0,1,0]=(0.12990139506)*x[0]**o+(-0.0990388974263)*x[0]+(-0.913089275462)*x[1]**o+(-0.766316334455)*x[1]
        arg[2,0,1,1]=(-0.96760564527)*x[0]**o+(0.94596820894)*x[0]+(-0.208507053422)*x[1]**o+(0.82642580152)*x[1]
        arg[2,0,2,0]=(-0.801660576339)*x[0]**o+(0.724091208253)*x[0]+(-0.416451086696)*x[1]**o+(-0.255959329575)*x[1]
        arg[2,0,2,1]=(0.380233654687)*x[0]**o+(-0.27467061721)*x[0]+(-0.626861863133)*x[1]**o+(0.498725464891)*x[1]
        arg[2,1,0,0]=(0.733227197081)*x[0]**o+(-0.111622822078)*x[0]+(-0.199363221943)*x[1]**o+(-0.449664634046)*x[1]
        arg[2,1,0,1]=(0.381774304841)*x[0]**o+(0.914208421879)*x[0]+(-0.545958908798)*x[1]**o+(-0.344721917598)*x[1]
        arg[2,1,1,0]=(-0.861412661903)*x[0]**o+(-0.202264578976)*x[0]+(-0.225404313047)*x[1]**o+(0.716189374265)*x[1]
        arg[2,1,1,1]=(-0.180088205315)*x[0]**o+(0.133136657032)*x[0]+(-0.341650807958)*x[1]**o+(-0.570348715256)*x[1]
        arg[2,1,2,0]=(-0.626863150737)*x[0]**o+(-0.236854925766)*x[0]+(-0.271186155613)*x[1]**o+(-0.750500826476)*x[1]
        arg[2,1,2,1]=(-0.138580929054)*x[0]**o+(-0.925057205334)*x[0]+(0.404961686669)*x[1]**o+(0.542298342705)*x[1]
        arg[2,2,0,0]=(0.453186536783)*x[0]**o+(0.368065177204)*x[0]+(0.545211472015)*x[1]**o+(0.334551805933)*x[1]
        arg[2,2,0,1]=(0.0639699038803)*x[0]**o+(0.154192875522)*x[0]+(0.101347900441)*x[1]**o+(-0.668313576443)*x[1]
        arg[2,2,1,0]=(0.465173445732)*x[0]**o+(-0.475009875122)*x[0]+(-0.748444748361)*x[1]**o+(0.620845514745)*x[1]
        arg[2,2,1,1]=(0.720975965615)*x[0]**o+(-0.266631032721)*x[0]+(0.0996389916136)*x[1]**o+(-0.96886977517)*x[1]
        arg[2,2,2,0]=(-0.396391719953)*x[0]**o+(-0.240091626001)*x[0]+(-0.42411685734)*x[1]**o+(-0.458446010754)*x[1]
        arg[2,2,2,1]=(-0.662584035264)*x[0]**o+(0.23813614536)*x[0]+(0.810893160816)*x[1]**o+(-0.130143663026)*x[1]
        arg[2,3,0,0]=(-0.651550480594)*x[0]**o+(0.370844612999)*x[0]+(-0.808002027275)*x[1]**o+(0.531083477622)*x[1]
        arg[2,3,0,1]=(-0.287184182382)*x[0]**o+(0.928877646221)*x[0]+(-0.271563784371)*x[1]**o+(-0.46908309398)*x[1]
        arg[2,3,1,0]=(0.342080138259)*x[0]**o+(0.458994887529)*x[0]+(-0.544475387926)*x[1]**o+(-0.203224249914)*x[1]
        arg[2,3,1,1]=(-0.0791009979206)*x[0]**o+(0.536480501773)*x[0]+(-0.624008015173)*x[1]**o+(-0.380504135174)*x[1]
        arg[2,3,2,0]=(-0.707089360149)*x[0]**o+(0.217716900812)*x[0]+(0.020441009853)*x[1]**o+(0.680214150153)*x[1]
        arg[2,3,2,1]=(0.0368635610611)*x[0]**o+(0.646697307691)*x[0]+(0.36118713723)*x[1]**o+(-0.535295583008)*x[1]
        arg[2,4,0,0]=(0.806938680905)*x[0]**o+(0.449096825134)*x[0]+(-0.43507826254)*x[1]**o+(-0.300956286252)*x[1]
        arg[2,4,0,1]=(-0.268005914326)*x[0]**o+(0.583132947291)*x[0]+(-0.344067495236)*x[1]**o+(-0.526011968282)*x[1]
        arg[2,4,1,0]=(-0.0953663458348)*x[0]**o+(0.834765967023)*x[0]+(-0.0410345097335)*x[1]**o+(0.493198054713)*x[1]
        arg[2,4,1,1]=(0.964777812988)*x[0]**o+(-0.791397009733)*x[0]+(-0.962108127047)*x[1]**o+(0.378083818799)*x[1]
        arg[2,4,2,0]=(0.0977715857241)*x[0]**o+(-0.118265910205)*x[0]+(-0.0181736456148)*x[1]**o+(-0.480499109439)*x[1]
        arg[2,4,2,1]=(0.396936262802)*x[0]**o+(0.979807961416)*x[0]+(0.383251286227)*x[1]**o+(-0.921599329372)*x[1]
        arg[3,0,0,0]=(-0.193869568904)*x[0]**o+(0.335533150532)*x[0]+(-0.542087879921)*x[1]**o+(0.916531699685)*x[1]
        arg[3,0,0,1]=(0.732192438113)*x[0]**o+(0.610866210623)*x[0]+(0.232455584069)*x[1]**o+(0.109577342902)*x[1]
        arg[3,0,1,0]=(0.0487640958479)*x[0]**o+(-0.250229114846)*x[0]+(-0.539343546695)*x[1]**o+(-0.456485153733)*x[1]
        arg[3,0,1,1]=(0.951844805407)*x[0]**o+(-0.811442383281)*x[0]+(-0.537581471096)*x[1]**o+(0.350217063609)*x[1]
        arg[3,0,2,0]=(0.495490307938)*x[0]**o+(0.529667417659)*x[0]+(-0.853447043865)*x[1]**o+(0.97002519322)*x[1]
        arg[3,0,2,1]=(0.38188303469)*x[0]**o+(-0.75911672537)*x[0]+(-0.67563531103)*x[1]**o+(0.384002769031)*x[1]
        arg[3,1,0,0]=(0.44471228922)*x[0]**o+(-0.958995287488)*x[0]+(-0.356730659314)*x[1]**o+(0.853749081247)*x[1]
        arg[3,1,0,1]=(-0.674677862576)*x[0]**o+(-0.651945231545)*x[0]+(-0.869871613483)*x[1]**o+(0.518493292422)*x[1]
        arg[3,1,1,0]=(-0.803923033456)*x[0]**o+(0.916638584514)*x[0]+(-0.410056071233)*x[1]**o+(-0.313811798732)*x[1]
        arg[3,1,1,1]=(-0.677586574312)*x[0]**o+(0.101317771211)*x[0]+(-0.848629733526)*x[1]**o+(0.956935407827)*x[1]
        arg[3,1,2,0]=(0.723956969878)*x[0]**o+(-0.677929268987)*x[0]+(0.168694427833)*x[1]**o+(0.675664564011)*x[1]
        arg[3,1,2,1]=(-0.0961257584415)*x[0]**o+(0.870273004118)*x[0]+(0.0979731186867)*x[1]**o+(-0.561786099655)*x[1]
        arg[3,2,0,0]=(-0.506857925404)*x[0]**o+(0.950429186893)*x[0]+(-0.765686972231)*x[1]**o+(0.300787357163)*x[1]
        arg[3,2,0,1]=(-0.479464705052)*x[0]**o+(-0.814881390688)*x[0]+(-0.549033177495)*x[1]**o+(0.875164713771)*x[1]
        arg[3,2,1,0]=(0.984579775952)*x[0]**o+(0.822464384935)*x[0]+(-0.352937841183)*x[1]**o+(-0.389516730371)*x[1]
        arg[3,2,1,1]=(-0.75230207503)*x[0]**o+(0.921759892886)*x[0]+(0.683120790039)*x[1]**o+(0.470250085193)*x[1]
        arg[3,2,2,0]=(-0.257935272452)*x[0]**o+(-0.437370511818)*x[0]+(-0.350363036315)*x[1]**o+(-0.232415372202)*x[1]
        arg[3,2,2,1]=(0.0784702766009)*x[0]**o+(0.460959683819)*x[0]+(0.758201028024)*x[1]**o+(0.101597092257)*x[1]
        arg[3,3,0,0]=(0.292597181224)*x[0]**o+(-0.322992033991)*x[0]+(-0.536413055541)*x[1]**o+(-0.95609054732)*x[1]
        arg[3,3,0,1]=(0.174134715952)*x[0]**o+(0.811729326606)*x[0]+(0.89040777319)*x[1]**o+(0.893572419966)*x[1]
        arg[3,3,1,0]=(0.768234215847)*x[0]**o+(0.36391719033)*x[0]+(0.598889528719)*x[1]**o+(0.152035960251)*x[1]
        arg[3,3,1,1]=(-0.77938361663)*x[0]**o+(-0.448945619013)*x[0]+(0.471242869055)*x[1]**o+(0.112888214554)*x[1]
        arg[3,3,2,0]=(-0.00617753872016)*x[0]**o+(0.0155218100192)*x[0]+(0.674100983537)*x[1]**o+(0.211406831421)*x[1]
        arg[3,3,2,1]=(0.124470359441)*x[0]**o+(0.117393617733)*x[0]+(-0.297791856337)*x[1]**o+(-0.210405798241)*x[1]
        arg[3,4,0,0]=(0.139920132608)*x[0]**o+(-0.0073381380034)*x[0]+(-0.77267392403)*x[1]**o+(0.792299687107)*x[1]
        arg[3,4,0,1]=(0.328592839797)*x[0]**o+(-0.758899359217)*x[0]+(0.730856047416)*x[1]**o+(-0.389548922357)*x[1]
        arg[3,4,1,0]=(-0.239362896774)*x[0]**o+(0.513205289336)*x[0]+(-0.393781550386)*x[1]**o+(0.229968513732)*x[1]
        arg[3,4,1,1]=(0.777985150139)*x[0]**o+(0.273278793895)*x[0]+(0.17782134987)*x[1]**o+(0.491127331923)*x[1]
        arg[3,4,2,0]=(0.348482781008)*x[0]**o+(0.458696185739)*x[0]+(-0.77882422986)*x[1]**o+(0.455739446004)*x[1]
        arg[3,4,2,1]=(-0.396751338134)*x[0]**o+(0.449026376364)*x[0]+(0.308553967581)*x[1]**o+(-0.777653415416)*x[1]
        ref[0,0,0,0]=(-0.848175950825)/(o+1.)+(-0.060984579579)+(0.423100205741)*0.5**o
        ref[0,0,0,1]=(-0.838397517739)/(o+1.)+(-0.289891489711)+(-0.739687474039)*0.5**o
        ref[0,0,1,0]=(-0.616976579504)/(o+1.)+(-0.556983314949)+(-0.423462958438)*0.5**o
        ref[0,0,1,1]=(-0.953014661461)/(o+1.)+(0.500001792807)+(-0.432533121428)*0.5**o
        ref[0,0,2,0]=(-0.156191993934)/(o+1.)+(0.883028259835)+(-0.891613775753)*0.5**o
        ref[0,0,2,1]=(-0.68287286477)/(o+1.)+(0.0522510271596)+(0.909573147888)*0.5**o
        ref[0,1,0,0]=(-0.631624427826)/(o+1.)+(-0.340055925409)+(-0.39808363424)*0.5**o
        ref[0,1,0,1]=(-0.0750972377036)/(o+1.)+(0.566030507884)+(-0.748658247224)*0.5**o
        ref[0,1,1,0]=(-0.163646238293)/(o+1.)+(-0.578922683831)+(0.877627912148)*0.5**o
        ref[0,1,1,1]=(0.0667564973731)/(o+1.)+(0.438204826712)+(-0.592580581615)*0.5**o
        ref[0,1,2,0]=(-0.899276958155)/(o+1.)+(-0.684200628566)+(-0.505569292541)*0.5**o
        ref[0,1,2,1]=(0.912452712698)/(o+1.)+(-0.0949204253183)+(-0.215752366701)*0.5**o
        ref[0,2,0,0]=(0.267409202785)/(o+1.)+(0.560565813511)+(0.684112319204)*0.5**o
        ref[0,2,0,1]=(-0.152472725029)/(o+1.)+(-0.48936178263)+(0.552438501301)*0.5**o
        ref[0,2,1,0]=(-0.873134889407)/(o+1.)+(-0.0460827720855)+(-0.31213422297)*0.5**o
        ref[0,2,1,1]=(-0.761310589424)/(o+1.)+(-0.676170252134)+(0.883048532408)*0.5**o
        ref[0,2,2,0]=(-0.936381308418)/(o+1.)+(-0.00265386081956)+(-0.324489237508)*0.5**o
        ref[0,2,2,1]=(0.419811305063)/(o+1.)+(0.109993113022)+(-0.971735193469)*0.5**o
        ref[0,3,0,0]=(0.707279059071)/(o+1.)+(-0.86543390192)+(-0.305553454905)*0.5**o
        ref[0,3,0,1]=(0.819264129099)/(o+1.)+(0.0693620181083)+(0.803934620926)*0.5**o
        ref[0,3,1,0]=(-0.60262522048)/(o+1.)+(-0.484758941318)+(0.996632326055)*0.5**o
        ref[0,3,1,1]=(0.722837842981)/(o+1.)+(0.309131271645)+(0.402799911553)*0.5**o
        ref[0,3,2,0]=(0.62363567076)/(o+1.)+(-0.700867024115)+(0.937966588808)*0.5**o
        ref[0,3,2,1]=(-0.950660964237)/(o+1.)+(0.261278793995)+(-0.0389713032301)*0.5**o
        ref[0,4,0,0]=(-0.872148906918)/(o+1.)+(-0.692114808131)+(-0.589529640529)*0.5**o
        ref[0,4,0,1]=(0.329194160571)/(o+1.)+(0.651123856087)+(0.78615039967)*0.5**o
        ref[0,4,1,0]=(0.107007555422)/(o+1.)+(0.132524151069)+(-0.801187683466)*0.5**o
        ref[0,4,1,1]=(-0.720479146591)/(o+1.)+(0.517763642707)+(0.817551079419)*0.5**o
        ref[0,4,2,0]=(0.766214937295)/(o+1.)+(0.432935513335)+(-0.245391293539)*0.5**o
        ref[0,4,2,1]=(-0.993826355778)/(o+1.)+(-0.776337331653)+(-0.780163545255)*0.5**o
        ref[1,0,0,0]=(-0.00667060999489)/(o+1.)+(-0.619099615632)+(0.0998104654851)*0.5**o
        ref[1,0,0,1]=(0.118004968786)/(o+1.)+(-0.449362980069)+(0.716858223574)*0.5**o
        ref[1,0,1,0]=(0.10117572965)/(o+1.)+(0.104308578827)+(0.433505460082)*0.5**o
        ref[1,0,1,1]=(-0.221528653572)/(o+1.)+(-0.329803925788)+(0.93394463477)*0.5**o
        ref[1,0,2,0]=(0.798438119547)/(o+1.)+(0.309503333977)+(0.217926136552)*0.5**o
        ref[1,0,2,1]=(-0.934023631551)/(o+1.)+(-0.539756710935)+(0.489885370386)*0.5**o
        ref[1,1,0,0]=(0.207178629335)/(o+1.)+(0.228247128719)+(-0.104659311537)*0.5**o
        ref[1,1,0,1]=(-0.75522659926)/(o+1.)+(0.0397747617809)+(0.855593354128)*0.5**o
        ref[1,1,1,0]=(-0.454711374202)/(o+1.)+(-0.0621136279316)+(0.165499959353)*0.5**o
        ref[1,1,1,1]=(-0.15549992542)/(o+1.)+(0.00294424557217)+(0.668195333828)*0.5**o
        ref[1,1,2,0]=(0.0666836881524)/(o+1.)+(-0.402752583203)+(-0.0830862028828)*0.5**o
        ref[1,1,2,1]=(0.674867602585)/(o+1.)+(0.0228486008642)+(-0.614061685098)*0.5**o
        ref[1,2,0,0]=(0.369545867263)/(o+1.)+(-0.125031593656)+(-0.271025278576)*0.5**o
        ref[1,2,0,1]=(-0.473468176784)/(o+1.)+(0.431223066159)+(-0.646570016511)*0.5**o
        ref[1,2,1,0]=(-0.710677848448)/(o+1.)+(-0.733272984356)+(-0.274558546949)*0.5**o
        ref[1,2,1,1]=(-0.488075258363)/(o+1.)+(-0.44708949991)+(0.920326662139)*0.5**o
        ref[1,2,2,0]=(-0.209704158218)/(o+1.)+(0.195310227215)+(-0.734779609501)*0.5**o
        ref[1,2,2,1]=(-0.307645214006)/(o+1.)+(0.29102810303)+(-0.71042927509)*0.5**o
        ref[1,3,0,0]=(-0.202705286059)/(o+1.)+(-0.229027879102)+(0.430044544461)*0.5**o
        ref[1,3,0,1]=(0.183422356152)/(o+1.)+(0.305305136784)+(-0.0215232394148)*0.5**o
        ref[1,3,1,0]=(0.710351406673)/(o+1.)+(-0.79177872315)+(0.956072319541)*0.5**o
        ref[1,3,1,1]=(0.178878380134)/(o+1.)+(0.156341750764)+(0.512105745894)*0.5**o
        ref[1,3,2,0]=(0.657312673212)/(o+1.)+(-0.393472389222)+(0.505194196756)*0.5**o
        ref[1,3,2,1]=(0.798106457622)/(o+1.)+(-0.658724756667)+(0.562199962955)*0.5**o
        ref[1,4,0,0]=(0.720809353558)/(o+1.)+(0.319148466492)+(-0.95103819151)*0.5**o
        ref[1,4,0,1]=(0.432152553594)/(o+1.)+(-0.18177275378)+(-0.337212448334)*0.5**o
        ref[1,4,1,0]=(-0.178814496497)/(o+1.)+(0.613927858527)+(0.286170290079)*0.5**o
        ref[1,4,1,1]=(0.828780485661)/(o+1.)+(-0.201053233304)+(0.00961378883254)*0.5**o
        ref[1,4,2,0]=(-0.542741131881)/(o+1.)+(0.229580705514)+(0.784403348421)*0.5**o
        ref[1,4,2,1]=(-0.938704878102)/(o+1.)+(0.316930416706)+(-0.246343667616)*0.5**o
        ref[2,0,0,0]=(-0.733447125487)/(o+1.)+(-0.469454527772)+(-0.551056874357)*0.5**o
        ref[2,0,0,1]=(0.480546881289)/(o+1.)+(-0.132825570055)+(-0.75556701159)*0.5**o
        ref[2,0,1,0]=(-0.913089275462)/(o+1.)+(-0.432677615941)+(0.12990139506)*0.5**o
        ref[2,0,1,1]=(-0.208507053422)/(o+1.)+(0.88619700523)+(-0.96760564527)*0.5**o
        ref[2,0,2,0]=(-0.416451086696)/(o+1.)+(0.234065939339)+(-0.801660576339)*0.5**o
        ref[2,0,2,1]=(-0.626861863133)/(o+1.)+(0.112027423841)+(0.380233654687)*0.5**o
        ref[2,1,0,0]=(-0.199363221943)/(o+1.)+(-0.280643728062)+(0.733227197081)*0.5**o
        ref[2,1,0,1]=(-0.545958908798)/(o+1.)+(0.284743252141)+(0.381774304841)*0.5**o
        ref[2,1,1,0]=(-0.225404313047)/(o+1.)+(0.256962397644)+(-0.861412661903)*0.5**o
        ref[2,1,1,1]=(-0.341650807958)/(o+1.)+(-0.218606029112)+(-0.180088205315)*0.5**o
        ref[2,1,2,0]=(-0.271186155613)/(o+1.)+(-0.493677876121)+(-0.626863150737)*0.5**o
        ref[2,1,2,1]=(0.404961686669)/(o+1.)+(-0.191379431315)+(-0.138580929054)*0.5**o
        ref[2,2,0,0]=(0.545211472015)/(o+1.)+(0.351308491569)+(0.453186536783)*0.5**o
        ref[2,2,0,1]=(0.101347900441)/(o+1.)+(-0.257060350461)+(0.0639699038803)*0.5**o
        ref[2,2,1,0]=(-0.748444748361)/(o+1.)+(0.0729178198114)+(0.465173445732)*0.5**o
        ref[2,2,1,1]=(0.0996389916136)/(o+1.)+(-0.617750403946)+(0.720975965615)*0.5**o
        ref[2,2,2,0]=(-0.42411685734)/(o+1.)+(-0.349268818378)+(-0.396391719953)*0.5**o
        ref[2,2,2,1]=(0.810893160816)/(o+1.)+(0.053996241167)+(-0.662584035264)*0.5**o
        ref[2,3,0,0]=(-0.808002027275)/(o+1.)+(0.45096404531)+(-0.651550480594)*0.5**o
        ref[2,3,0,1]=(-0.271563784371)/(o+1.)+(0.22989727612)+(-0.287184182382)*0.5**o
        ref[2,3,1,0]=(-0.544475387926)/(o+1.)+(0.127885318807)+(0.342080138259)*0.5**o
        ref[2,3,1,1]=(-0.624008015173)/(o+1.)+(0.0779881832997)+(-0.0791009979206)*0.5**o
        ref[2,3,2,0]=(0.020441009853)/(o+1.)+(0.448965525483)+(-0.707089360149)*0.5**o
        ref[2,3,2,1]=(0.36118713723)/(o+1.)+(0.0557008623413)+(0.0368635610611)*0.5**o
        ref[2,4,0,0]=(-0.43507826254)/(o+1.)+(0.0740702694408)+(0.806938680905)*0.5**o
        ref[2,4,0,1]=(-0.344067495236)/(o+1.)+(0.0285604895043)+(-0.268005914326)*0.5**o
        ref[2,4,1,0]=(-0.0410345097335)/(o+1.)+(0.663982010868)+(-0.0953663458348)*0.5**o
        ref[2,4,1,1]=(-0.962108127047)/(o+1.)+(-0.206656595467)+(0.964777812988)*0.5**o
        ref[2,4,2,0]=(-0.0181736456148)/(o+1.)+(-0.299382509822)+(0.0977715857241)*0.5**o
        ref[2,4,2,1]=(0.383251286227)/(o+1.)+(0.0291043160221)+(0.396936262802)*0.5**o
        ref[3,0,0,0]=(-0.542087879921)/(o+1.)+(0.626032425109)+(-0.193869568904)*0.5**o
        ref[3,0,0,1]=(0.232455584069)/(o+1.)+(0.360221776762)+(0.732192438113)*0.5**o
        ref[3,0,1,0]=(-0.539343546695)/(o+1.)+(-0.353357134289)+(0.0487640958479)*0.5**o
        ref[3,0,1,1]=(-0.537581471096)/(o+1.)+(-0.230612659836)+(0.951844805407)*0.5**o
        ref[3,0,2,0]=(-0.853447043865)/(o+1.)+(0.74984630544)+(0.495490307938)*0.5**o
        ref[3,0,2,1]=(-0.67563531103)/(o+1.)+(-0.187556978169)+(0.38188303469)*0.5**o
        ref[3,1,0,0]=(-0.356730659314)/(o+1.)+(-0.0526231031206)+(0.44471228922)*0.5**o
        ref[3,1,0,1]=(-0.869871613483)/(o+1.)+(-0.0667259695618)+(-0.674677862576)*0.5**o
        ref[3,1,1,0]=(-0.410056071233)/(o+1.)+(0.301413392891)+(-0.803923033456)*0.5**o
        ref[3,1,1,1]=(-0.848629733526)/(o+1.)+(0.529126589519)+(-0.677586574312)*0.5**o
        ref[3,1,2,0]=(0.168694427833)/(o+1.)+(-0.00113235248793)+(0.723956969878)*0.5**o
        ref[3,1,2,1]=(0.0979731186867)/(o+1.)+(0.154243452232)+(-0.0961257584415)*0.5**o
        ref[3,2,0,0]=(-0.765686972231)/(o+1.)+(0.625608272028)+(-0.506857925404)*0.5**o
        ref[3,2,0,1]=(-0.549033177495)/(o+1.)+(0.0301416615418)+(-0.479464705052)*0.5**o
        ref[3,2,1,0]=(-0.352937841183)/(o+1.)+(0.216473827282)+(0.984579775952)*0.5**o
        ref[3,2,1,1]=(0.683120790039)/(o+1.)+(0.696004989039)+(-0.75230207503)*0.5**o
        ref[3,2,2,0]=(-0.350363036315)/(o+1.)+(-0.33489294201)+(-0.257935272452)*0.5**o
        ref[3,2,2,1]=(0.758201028024)/(o+1.)+(0.281278388038)+(0.0784702766009)*0.5**o
        ref[3,3,0,0]=(-0.536413055541)/(o+1.)+(-0.639541290655)+(0.292597181224)*0.5**o
        ref[3,3,0,1]=(0.89040777319)/(o+1.)+(0.852650873286)+(0.174134715952)*0.5**o
        ref[3,3,1,0]=(0.598889528719)/(o+1.)+(0.257976575291)+(0.768234215847)*0.5**o
        ref[3,3,1,1]=(0.471242869055)/(o+1.)+(-0.168028702229)+(-0.77938361663)*0.5**o
        ref[3,3,2,0]=(0.674100983537)/(o+1.)+(0.11346432072)+(-0.00617753872016)*0.5**o
        ref[3,3,2,1]=(-0.297791856337)/(o+1.)+(-0.0465060902539)+(0.124470359441)*0.5**o
        ref[3,4,0,0]=(-0.77267392403)/(o+1.)+(0.392480774552)+(0.139920132608)*0.5**o
        ref[3,4,0,1]=(0.730856047416)/(o+1.)+(-0.574224140787)+(0.328592839797)*0.5**o
        ref[3,4,1,0]=(-0.393781550386)/(o+1.)+(0.371586901534)+(-0.239362896774)*0.5**o
        ref[3,4,1,1]=(0.17782134987)/(o+1.)+(0.382203062909)+(0.777985150139)*0.5**o
        ref[3,4,2,0]=(-0.77882422986)/(o+1.)+(0.457217815872)+(0.348482781008)*0.5**o
        ref[3,4,2,1]=(0.308553967581)/(o+1.)+(-0.164313519526)+(-0.396751338134)*0.5**o
      else:
        arg[0,0,0,0]=(0.663998508168)*x[0]**o+(0.797340164931)*x[0]+(0.285182001599)*x[1]**o+(0.99720606542)*x[1]+(0.127610260108)*x[2]**o+(0.0141819362674)*x[2]
        arg[0,0,0,1]=(-0.010130464476)*x[0]**o+(-0.694140348803)*x[0]+(-0.554528445238)*x[1]**o+(0.192639775817)*x[1]+(-0.520164891934)*x[2]**o+(0.316830081989)*x[2]
        arg[0,0,1,0]=(-0.176542289385)*x[0]**o+(-0.312451025451)*x[0]+(0.33951668081)*x[1]**o+(-0.66674551066)*x[1]+(-0.0515929546862)*x[2]**o+(0.688180995863)*x[2]
        arg[0,0,1,1]=(-0.98152572229)*x[0]**o+(0.156411605196)*x[0]+(0.588065240125)*x[1]**o+(0.665830732172)*x[1]+(-0.834625744106)*x[2]**o+(-0.138276534671)*x[2]
        arg[0,0,2,0]=(0.555897422173)*x[0]**o+(0.0269431977874)*x[0]+(0.671161746561)*x[1]**o+(0.0857445285374)*x[1]+(0.578027911522)*x[2]**o+(0.217805007393)*x[2]
        arg[0,0,2,1]=(0.23127040923)*x[0]**o+(0.624353504617)*x[0]+(-0.433915231286)*x[1]**o+(-0.860173971996)*x[1]+(-0.801472434932)*x[2]**o+(-0.779575445432)*x[2]
        arg[0,1,0,0]=(0.0202445177842)*x[0]**o+(-0.0654289514039)*x[0]+(-0.877581892208)*x[1]**o+(-0.231833216639)*x[1]+(0.609752401432)*x[2]**o+(-0.802673983603)*x[2]
        arg[0,1,0,1]=(0.99935098551)*x[0]**o+(0.665639246142)*x[0]+(0.593510187831)*x[1]**o+(0.213336090684)*x[1]+(0.232667772738)*x[2]**o+(-0.651386262633)*x[2]
        arg[0,1,1,0]=(0.151799117419)*x[0]**o+(-0.34631043288)*x[0]+(-0.264898731671)*x[1]**o+(0.697527693137)*x[1]+(0.389030412719)*x[2]**o+(0.613338381577)*x[2]
        arg[0,1,1,1]=(-0.51422532626)*x[0]**o+(0.228040429268)*x[0]+(-0.276760984751)*x[1]**o+(0.379484010198)*x[1]+(0.896837565192)*x[2]**o+(0.195866552401)*x[2]
        arg[0,1,2,0]=(0.908244888839)*x[0]**o+(0.217098236162)*x[0]+(0.798725721675)*x[1]**o+(0.216635630826)*x[1]+(0.960227216319)*x[2]**o+(-0.885381182879)*x[2]
        arg[0,1,2,1]=(-0.872307766634)*x[0]**o+(0.843197737136)*x[0]+(-0.895220198791)*x[1]**o+(0.866433237518)*x[1]+(0.560344834226)*x[2]**o+(0.895797915384)*x[2]
        arg[0,2,0,0]=(0.00173619592035)*x[0]**o+(-0.209651858019)*x[0]+(0.323249367733)*x[1]**o+(-0.848445788048)*x[1]+(-0.747443572813)*x[2]**o+(0.342929929066)*x[2]
        arg[0,2,0,1]=(0.998184336552)*x[0]**o+(-0.669882578313)*x[0]+(-0.632794054696)*x[1]**o+(-0.779832061924)*x[1]+(0.0588815237456)*x[2]**o+(0.419538634163)*x[2]
        arg[0,2,1,0]=(-0.921234331968)*x[0]**o+(0.319724478943)*x[0]+(0.867192523265)*x[1]**o+(-0.183047755109)*x[1]+(-0.753047661507)*x[2]**o+(-0.797179040607)*x[2]
        arg[0,2,1,1]=(0.153578015941)*x[0]**o+(0.887820807812)*x[0]+(-0.607944647299)*x[1]**o+(-0.0487413965527)*x[1]+(-0.812369692315)*x[2]**o+(0.323102876844)*x[2]
        arg[0,2,2,0]=(0.29601674794)*x[0]**o+(-0.108738759628)*x[0]+(0.242837345895)*x[1]**o+(-0.324487554664)*x[1]+(0.584399369463)*x[2]**o+(0.683877139789)*x[2]
        arg[0,2,2,1]=(0.0580007647099)*x[0]**o+(0.214488016057)*x[0]+(0.669087265419)*x[1]**o+(-0.311999860513)*x[1]+(0.678940598195)*x[2]**o+(0.658847203441)*x[2]
        arg[0,3,0,0]=(-0.337999890505)*x[0]**o+(-0.842479530409)*x[0]+(0.786376708623)*x[1]**o+(0.597298234359)*x[1]+(0.495442668997)*x[2]**o+(-0.183828960943)*x[2]
        arg[0,3,0,1]=(-0.825813329629)*x[0]**o+(0.197040670392)*x[0]+(0.997784261354)*x[1]**o+(-0.355290279278)*x[1]+(0.657909516664)*x[2]**o+(-0.568281670001)*x[2]
        arg[0,3,1,0]=(0.699313512079)*x[0]**o+(-0.916350147519)*x[0]+(-0.311605808179)*x[1]**o+(0.222618888987)*x[1]+(0.872150798976)*x[2]**o+(0.139773150125)*x[2]
        arg[0,3,1,1]=(0.140764124286)*x[0]**o+(0.994013777309)*x[0]+(-0.0792107151558)*x[1]**o+(-0.285357856751)*x[1]+(-0.800285107223)*x[2]**o+(0.413029089978)*x[2]
        arg[0,3,2,0]=(-0.68177752688)*x[0]**o+(0.0339522871939)*x[0]+(-0.641399841911)*x[1]**o+(0.263279621719)*x[1]+(-0.406115766497)*x[2]**o+(-0.774639092638)*x[2]
        arg[0,3,2,1]=(0.655685968811)*x[0]**o+(0.20085918281)*x[0]+(-0.753491228829)*x[1]**o+(0.393669658887)*x[1]+(0.339469131428)*x[2]**o+(0.424803724709)*x[2]
        arg[0,4,0,0]=(-0.18027010068)*x[0]**o+(-0.773007683353)*x[0]+(0.767133020895)*x[1]**o+(-0.810449138606)*x[1]+(0.295421955604)*x[2]**o+(-0.955926880008)*x[2]
        arg[0,4,0,1]=(-0.914768182433)*x[0]**o+(-0.67024319498)*x[0]+(0.533244666277)*x[1]**o+(-0.705719628704)*x[1]+(-0.963633923897)*x[2]**o+(-0.0818895784023)*x[2]
        arg[0,4,1,0]=(0.274727423835)*x[0]**o+(0.838218676826)*x[0]+(0.269989900726)*x[1]**o+(0.650600179293)*x[1]+(-0.855210211935)*x[2]**o+(-0.800261683904)*x[2]
        arg[0,4,1,1]=(0.257574026756)*x[0]**o+(-0.787643878696)*x[0]+(-0.619136916008)*x[1]**o+(-0.242580269843)*x[1]+(0.0229959873555)*x[2]**o+(-0.685725009107)*x[2]
        arg[0,4,2,0]=(0.703185665611)*x[0]**o+(0.37686548395)*x[0]+(-0.414901304236)*x[1]**o+(0.310335568741)*x[1]+(-0.740764649572)*x[2]**o+(-0.389356987349)*x[2]
        arg[0,4,2,1]=(-0.153792811673)*x[0]**o+(-0.179777301012)*x[0]+(0.694402537357)*x[1]**o+(-0.928918851291)*x[1]+(-0.439307255129)*x[2]**o+(0.148671231214)*x[2]
        arg[1,0,0,0]=(0.298044833507)*x[0]**o+(0.538000859279)*x[0]+(-0.405432041063)*x[1]**o+(-0.8912548734)*x[1]+(0.929554154255)*x[2]**o+(-0.515905504512)*x[2]
        arg[1,0,0,1]=(-0.575485929806)*x[0]**o+(0.757951009672)*x[0]+(-0.815024122535)*x[1]**o+(0.846519632117)*x[1]+(0.594901198403)*x[2]**o+(-0.672034020351)*x[2]
        arg[1,0,1,0]=(0.998358005707)*x[0]**o+(-0.144729504804)*x[0]+(0.463445062889)*x[1]**o+(0.428918148475)*x[1]+(-0.172684103233)*x[2]**o+(-0.394742387769)*x[2]
        arg[1,0,1,1]=(0.783593367143)*x[0]**o+(-0.269882783638)*x[0]+(-0.880336283806)*x[1]**o+(0.337087191943)*x[1]+(-0.819512478233)*x[2]**o+(0.582924902797)*x[2]
        arg[1,0,2,0]=(-0.42345692756)*x[0]**o+(-0.251910598711)*x[0]+(-0.882193473121)*x[1]**o+(0.535405977525)*x[1]+(-0.117109433242)*x[2]**o+(0.99565775159)*x[2]
        arg[1,0,2,1]=(-0.790917746681)*x[0]**o+(-0.291814540513)*x[0]+(-0.0828604085821)*x[1]**o+(-0.0715382319561)*x[1]+(-0.188453557262)*x[2]**o+(0.297596604241)*x[2]
        arg[1,1,0,0]=(0.944343622558)*x[0]**o+(0.537893843235)*x[0]+(-0.370697218599)*x[1]**o+(-0.043709074747)*x[1]+(-0.803318845377)*x[2]**o+(0.814392946479)*x[2]
        arg[1,1,0,1]=(-0.246711023019)*x[0]**o+(-0.847902406307)*x[0]+(0.062135604814)*x[1]**o+(0.581341495124)*x[1]+(0.561484848132)*x[2]**o+(0.06820017559)*x[2]
        arg[1,1,1,0]=(-0.530782063061)*x[0]**o+(-0.951246029994)*x[0]+(-0.809647094351)*x[1]**o+(0.198165508669)*x[1]+(0.591475451485)*x[2]**o+(-0.0567469904903)*x[2]
        arg[1,1,1,1]=(-0.710844761085)*x[0]**o+(0.742146961382)*x[0]+(0.482950602277)*x[1]**o+(-0.701576649521)*x[1]+(0.752305355358)*x[2]**o+(0.658478074845)*x[2]
        arg[1,1,2,0]=(0.574657505707)*x[0]**o+(-0.268094098242)*x[0]+(-0.79745549318)*x[1]**o+(0.994732385826)*x[1]+(0.36752110393)*x[2]**o+(0.0733521344978)*x[2]
        arg[1,1,2,1]=(-0.365373199615)*x[0]**o+(0.295532428151)*x[0]+(-0.916049089517)*x[1]**o+(0.562876895658)*x[1]+(0.698818392781)*x[2]**o+(0.413166153315)*x[2]
        arg[1,2,0,0]=(0.544776615353)*x[0]**o+(-0.647889580869)*x[0]+(-0.0569263298125)*x[1]**o+(0.145083519829)*x[1]+(-0.0456368749942)*x[2]**o+(0.125915481669)*x[2]
        arg[1,2,0,1]=(0.11684721915)*x[0]**o+(-0.793592672443)*x[0]+(-0.428585283805)*x[1]**o+(0.280358391066)*x[1]+(0.72731741847)*x[2]**o+(-0.320459136831)*x[2]
        arg[1,2,1,0]=(0.561736948232)*x[0]**o+(0.833411538285)*x[0]+(0.361519222525)*x[1]**o+(-0.785948355356)*x[1]+(-0.595938528895)*x[2]**o+(0.720247127573)*x[2]
        arg[1,2,1,1]=(0.578630452142)*x[0]**o+(0.218238639275)*x[0]+(0.792496779641)*x[1]**o+(-0.818431209813)*x[1]+(-0.252522807953)*x[2]**o+(0.648505893606)*x[2]
        arg[1,2,2,0]=(-0.0269203575755)*x[0]**o+(-0.586454813205)*x[0]+(-0.239885637578)*x[1]**o+(0.186263805313)*x[1]+(0.585909451795)*x[2]**o+(-0.630731036615)*x[2]
        arg[1,2,2,1]=(-0.534642633404)*x[0]**o+(0.921034930282)*x[0]+(-0.326672365556)*x[1]**o+(0.690509146367)*x[1]+(-0.802567575176)*x[2]**o+(-0.235903818924)*x[2]
        arg[1,3,0,0]=(0.892302802156)*x[0]**o+(-0.528836075071)*x[0]+(-0.806520835766)*x[1]**o+(0.771321197577)*x[1]+(-0.545545197249)*x[2]**o+(-0.633185919719)*x[2]
        arg[1,3,0,1]=(0.883702447985)*x[0]**o+(0.698176356191)*x[0]+(-0.348189466787)*x[1]**o+(-0.390058291315)*x[1]+(0.805736008481)*x[2]**o+(0.861787745203)*x[2]
        arg[1,3,1,0]=(0.492527250981)*x[0]**o+(0.635712008949)*x[0]+(-0.464397346193)*x[1]**o+(-0.820952575379)*x[1]+(0.235015001164)*x[2]**o+(-0.846160845231)*x[2]
        arg[1,3,1,1]=(0.372499842901)*x[0]**o+(-0.524824414377)*x[0]+(0.553450359486)*x[1]**o+(-0.187372498756)*x[1]+(-0.389983195918)*x[2]**o+(-0.310421627605)*x[2]
        arg[1,3,2,0]=(0.533098362924)*x[0]**o+(0.917820078197)*x[0]+(-0.557683070597)*x[1]**o+(-0.388498248383)*x[1]+(-0.551866333066)*x[2]**o+(0.696873468511)*x[2]
        arg[1,3,2,1]=(-0.909038123273)*x[0]**o+(0.19323884412)*x[0]+(-0.00134861813725)*x[1]**o+(-0.858193032021)*x[1]+(0.0436856504635)*x[2]**o+(-0.646994265986)*x[2]
        arg[1,4,0,0]=(-0.974474210185)*x[0]**o+(-0.39786371112)*x[0]+(-0.829024534519)*x[1]**o+(0.836151642189)*x[1]+(0.679654655991)*x[2]**o+(0.750995377779)*x[2]
        arg[1,4,0,1]=(0.964745405431)*x[0]**o+(0.269189443297)*x[0]+(0.531636623506)*x[1]**o+(-0.522687387383)*x[1]+(0.463500385567)*x[2]**o+(0.160556224359)*x[2]
        arg[1,4,1,0]=(-0.714725894011)*x[0]**o+(0.0478807291181)*x[0]+(0.591494552904)*x[1]**o+(-0.240293119817)*x[1]+(-0.0463360360003)*x[2]**o+(-0.932835908096)*x[2]
        arg[1,4,1,1]=(-0.809072545597)*x[0]**o+(-0.963335650229)*x[0]+(-0.781885522838)*x[1]**o+(0.822395174213)*x[1]+(-0.42590899181)*x[2]**o+(-0.237357227214)*x[2]
        arg[1,4,2,0]=(-0.21281876112)*x[0]**o+(-0.131640536998)*x[0]+(-0.648680371617)*x[1]**o+(0.555759774031)*x[1]+(-0.911030244394)*x[2]**o+(0.962859403252)*x[2]
        arg[1,4,2,1]=(-0.247930702516)*x[0]**o+(0.509829884687)*x[0]+(0.0387816417333)*x[1]**o+(-0.605945140743)*x[1]+(0.187880415018)*x[2]**o+(-0.813242364046)*x[2]
        arg[2,0,0,0]=(0.616352869937)*x[0]**o+(-0.930708209966)*x[0]+(-0.874778661809)*x[1]**o+(-0.330225154904)*x[1]+(0.632020336392)*x[2]**o+(0.205631598228)*x[2]
        arg[2,0,0,1]=(-0.85704374519)*x[0]**o+(0.99958496465)*x[0]+(0.807095767473)*x[1]**o+(0.63097388144)*x[1]+(0.000665614692828)*x[2]**o+(0.418070858864)*x[2]
        arg[2,0,1,0]=(0.326870102902)*x[0]**o+(0.565855791221)*x[0]+(-0.39418644908)*x[1]**o+(-0.96081735968)*x[1]+(0.937255293725)*x[2]**o+(0.533015538451)*x[2]
        arg[2,0,1,1]=(-0.699939842176)*x[0]**o+(0.808055402204)*x[0]+(0.448948790025)*x[1]**o+(-0.720122241439)*x[1]+(0.0532200858319)*x[2]**o+(-0.204851171261)*x[2]
        arg[2,0,2,0]=(-0.985996827188)*x[0]**o+(-0.604602591167)*x[0]+(0.445903925662)*x[1]**o+(-0.163135114049)*x[1]+(-0.264281316158)*x[2]**o+(-0.381814212863)*x[2]
        arg[2,0,2,1]=(-0.195357773436)*x[0]**o+(0.73285122682)*x[0]+(-0.914846292757)*x[1]**o+(-0.190524885095)*x[1]+(-0.103440307801)*x[2]**o+(-0.0309647945217)*x[2]
        arg[2,1,0,0]=(0.601652939329)*x[0]**o+(-0.757171386065)*x[0]+(-0.91147090556)*x[1]**o+(-0.665258414941)*x[1]+(-0.0504245123191)*x[2]**o+(-0.274928427711)*x[2]
        arg[2,1,0,1]=(0.534038022604)*x[0]**o+(-0.977365399298)*x[0]+(0.884656682614)*x[1]**o+(0.0866421048629)*x[1]+(-0.70687555879)*x[2]**o+(-0.99928465036)*x[2]
        arg[2,1,1,0]=(-0.361196978257)*x[0]**o+(0.192559020092)*x[0]+(-0.0767715201942)*x[1]**o+(0.222254953069)*x[1]+(-0.128362555436)*x[2]**o+(-0.667485829139)*x[2]
        arg[2,1,1,1]=(0.00930987921229)*x[0]**o+(0.269207633273)*x[0]+(0.943798095916)*x[1]**o+(0.483123564952)*x[1]+(-0.231669365037)*x[2]**o+(-0.877005245489)*x[2]
        arg[2,1,2,0]=(-0.181479246679)*x[0]**o+(0.619551810132)*x[0]+(0.499400298722)*x[1]**o+(-0.307864349855)*x[1]+(0.19303040809)*x[2]**o+(0.523640816928)*x[2]
        arg[2,1,2,1]=(-0.99665050175)*x[0]**o+(-0.647775674514)*x[0]+(-0.0015481673821)*x[1]**o+(-0.749198451256)*x[1]+(0.652528193579)*x[2]**o+(-0.0194127861341)*x[2]
        arg[2,2,0,0]=(0.870069083779)*x[0]**o+(0.179747121592)*x[0]+(0.387523112512)*x[1]**o+(0.625711925024)*x[1]+(0.840141938096)*x[2]**o+(0.629709976556)*x[2]
        arg[2,2,0,1]=(-0.841259995642)*x[0]**o+(0.220858862445)*x[0]+(0.397334296484)*x[1]**o+(-0.036654179636)*x[1]+(0.235704322138)*x[2]**o+(-0.6299228864)*x[2]
        arg[2,2,1,0]=(-0.561184923307)*x[0]**o+(-0.643644162376)*x[0]+(0.728784994685)*x[1]**o+(-0.697418070844)*x[1]+(-0.56166463788)*x[2]**o+(-0.668760084534)*x[2]
        arg[2,2,1,1]=(0.92243000401)*x[0]**o+(0.96165833182)*x[0]+(0.540379002453)*x[1]**o+(0.630245078647)*x[1]+(-0.375873656541)*x[2]**o+(0.71424898756)*x[2]
        arg[2,2,2,0]=(0.331131296272)*x[0]**o+(-0.136167378345)*x[0]+(-0.375407637025)*x[1]**o+(-0.0890691004072)*x[1]+(-0.787306129385)*x[2]**o+(-0.447474749986)*x[2]
        arg[2,2,2,1]=(-0.47892171854)*x[0]**o+(0.341448171328)*x[0]+(0.662403724393)*x[1]**o+(0.581128263913)*x[1]+(0.00987835902932)*x[2]**o+(-0.884847129103)*x[2]
        arg[2,3,0,0]=(0.0219406022052)*x[0]**o+(-0.370001982912)*x[0]+(0.948162807268)*x[1]**o+(-0.870857624404)*x[1]+(0.580053532448)*x[2]**o+(-0.488657040041)*x[2]
        arg[2,3,0,1]=(-0.114799511617)*x[0]**o+(0.547853919356)*x[0]+(-0.224367707174)*x[1]**o+(0.551491902831)*x[1]+(-0.101845907889)*x[2]**o+(-0.793527438395)*x[2]
        arg[2,3,1,0]=(-0.103518223905)*x[0]**o+(-0.319840011093)*x[0]+(-0.856612861717)*x[1]**o+(0.0208578084364)*x[1]+(0.132248308449)*x[2]**o+(-0.580663554499)*x[2]
        arg[2,3,1,1]=(-0.958142224242)*x[0]**o+(-0.401299403423)*x[0]+(-0.289034019185)*x[1]**o+(-0.0913958572584)*x[1]+(-0.818515945817)*x[2]**o+(-0.918366425737)*x[2]
        arg[2,3,2,0]=(0.730554594734)*x[0]**o+(0.209964841318)*x[0]+(-0.4157212696)*x[1]**o+(-0.673375403124)*x[1]+(-0.409899056244)*x[2]**o+(0.124418003637)*x[2]
        arg[2,3,2,1]=(-0.930974589897)*x[0]**o+(0.256452400333)*x[0]+(0.778606258675)*x[1]**o+(0.23173182074)*x[1]+(-0.846439682798)*x[2]**o+(-0.408860675089)*x[2]
        arg[2,4,0,0]=(-0.246046780287)*x[0]**o+(-0.653766430943)*x[0]+(0.725379461155)*x[1]**o+(-0.305490823497)*x[1]+(0.0837637468427)*x[2]**o+(-0.458770631133)*x[2]
        arg[2,4,0,1]=(-0.612751147889)*x[0]**o+(0.504418327803)*x[0]+(-0.835051783805)*x[1]**o+(-0.86410559166)*x[1]+(0.928919612861)*x[2]**o+(0.303164594248)*x[2]
        arg[2,4,1,0]=(-0.26092972064)*x[0]**o+(-0.990592356395)*x[0]+(-0.48716443965)*x[1]**o+(0.659447608304)*x[1]+(0.429865593656)*x[2]**o+(-0.661470437053)*x[2]
        arg[2,4,1,1]=(0.325840618456)*x[0]**o+(-0.261974904995)*x[0]+(0.476982730267)*x[1]**o+(-0.738119182885)*x[1]+(0.903618858571)*x[2]**o+(-0.287442022624)*x[2]
        arg[2,4,2,0]=(0.655590135983)*x[0]**o+(-0.135505327297)*x[0]+(-0.38079131346)*x[1]**o+(0.663850751769)*x[1]+(-0.658568109867)*x[2]**o+(-0.267875066063)*x[2]
        arg[2,4,2,1]=(0.570712162768)*x[0]**o+(0.390833274898)*x[0]+(-0.97326103687)*x[1]**o+(-0.419300688467)*x[1]+(0.55695479018)*x[2]**o+(0.547271407555)*x[2]
        arg[3,0,0,0]=(-0.486023722572)*x[0]**o+(-0.790544923201)*x[0]+(0.101237181961)*x[1]**o+(0.23267779263)*x[1]+(-0.458580701132)*x[2]**o+(-0.479449733139)*x[2]
        arg[3,0,0,1]=(-0.705291625776)*x[0]**o+(0.144166702405)*x[0]+(-0.210596631253)*x[1]**o+(0.843721258425)*x[1]+(-0.716420627484)*x[2]**o+(0.145427688875)*x[2]
        arg[3,0,1,0]=(-0.838974120992)*x[0]**o+(0.824423674933)*x[0]+(-0.211643366873)*x[1]**o+(0.603255736693)*x[1]+(-0.119484817371)*x[2]**o+(-0.357272543813)*x[2]
        arg[3,0,1,1]=(0.140027541071)*x[0]**o+(0.373598174368)*x[0]+(-0.661831753147)*x[1]**o+(-0.973724961653)*x[1]+(-0.0738899100301)*x[2]**o+(0.0128579666074)*x[2]
        arg[3,0,2,0]=(-0.39083471939)*x[0]**o+(-0.00821070494132)*x[0]+(-0.315936839906)*x[1]**o+(-0.967556350778)*x[1]+(0.136991762476)*x[2]**o+(0.374041520875)*x[2]
        arg[3,0,2,1]=(0.0734743009776)*x[0]**o+(-0.103186084491)*x[0]+(0.426933353985)*x[1]**o+(-0.631917740007)*x[1]+(0.755296770294)*x[2]**o+(0.492171872216)*x[2]
        arg[3,1,0,0]=(-0.414502599055)*x[0]**o+(0.512031410722)*x[0]+(0.610677049483)*x[1]**o+(-0.075275839509)*x[1]+(0.853407100545)*x[2]**o+(-0.569731924126)*x[2]
        arg[3,1,0,1]=(0.886115095396)*x[0]**o+(0.811807969433)*x[0]+(0.550623354147)*x[1]**o+(-0.703728894875)*x[1]+(0.784318903356)*x[2]**o+(0.45891829089)*x[2]
        arg[3,1,1,0]=(0.312577129879)*x[0]**o+(0.0127110992422)*x[0]+(-0.174614340667)*x[1]**o+(0.744535847679)*x[1]+(0.991125452344)*x[2]**o+(-0.405595301834)*x[2]
        arg[3,1,1,1]=(0.784369258288)*x[0]**o+(-0.709503832235)*x[0]+(0.032963558933)*x[1]**o+(-0.471036543959)*x[1]+(-0.138022967476)*x[2]**o+(-0.38452297541)*x[2]
        arg[3,1,2,0]=(-0.337332295082)*x[0]**o+(-0.460457867002)*x[0]+(-0.102806204559)*x[1]**o+(-0.502889108364)*x[1]+(-0.641488011668)*x[2]**o+(0.359501036527)*x[2]
        arg[3,1,2,1]=(0.791575892417)*x[0]**o+(0.126853712619)*x[0]+(-0.408911942637)*x[1]**o+(0.557039825987)*x[1]+(-0.260719710621)*x[2]**o+(-0.648326395935)*x[2]
        arg[3,2,0,0]=(-0.0909007497318)*x[0]**o+(-0.337085519884)*x[0]+(-0.651987922715)*x[1]**o+(-0.130225558042)*x[1]+(0.926567188664)*x[2]**o+(0.764369275478)*x[2]
        arg[3,2,0,1]=(-0.764137934896)*x[0]**o+(0.443892891832)*x[0]+(0.688197384487)*x[1]**o+(0.485210058592)*x[1]+(0.267932741224)*x[2]**o+(0.0366064885582)*x[2]
        arg[3,2,1,0]=(0.49435832562)*x[0]**o+(0.860649583676)*x[0]+(0.667415091305)*x[1]**o+(0.144069385904)*x[1]+(-0.933847080974)*x[2]**o+(-0.378123304101)*x[2]
        arg[3,2,1,1]=(-0.460752644946)*x[0]**o+(0.491079753273)*x[0]+(-0.536055633506)*x[1]**o+(0.786378615881)*x[1]+(-0.190647453518)*x[2]**o+(-0.192868019034)*x[2]
        arg[3,2,2,0]=(0.0946687985697)*x[0]**o+(0.978197346579)*x[0]+(0.972747948856)*x[1]**o+(-0.575670550665)*x[1]+(-0.096229657916)*x[2]**o+(-0.772567430449)*x[2]
        arg[3,2,2,1]=(0.177521713734)*x[0]**o+(0.0683685252148)*x[0]+(0.423995112682)*x[1]**o+(-0.489247981968)*x[1]+(-0.537130434099)*x[2]**o+(-0.0228879131449)*x[2]
        arg[3,3,0,0]=(0.592201472175)*x[0]**o+(0.674304504132)*x[0]+(0.428414872879)*x[1]**o+(0.956414209094)*x[1]+(-0.272853242491)*x[2]**o+(-0.639858910901)*x[2]
        arg[3,3,0,1]=(0.572972211447)*x[0]**o+(0.943096858489)*x[0]+(-0.990948797302)*x[1]**o+(-0.475383526007)*x[1]+(0.542076985363)*x[2]**o+(-0.884809421862)*x[2]
        arg[3,3,1,0]=(0.775163727164)*x[0]**o+(0.101808074039)*x[0]+(0.575490330857)*x[1]**o+(0.199751068512)*x[1]+(0.538742155685)*x[2]**o+(-0.725822956214)*x[2]
        arg[3,3,1,1]=(-0.244586606882)*x[0]**o+(0.983712254807)*x[0]+(0.187370538052)*x[1]**o+(0.332611385073)*x[1]+(0.536377667373)*x[2]**o+(0.255272558489)*x[2]
        arg[3,3,2,0]=(-0.676040282048)*x[0]**o+(-0.172059284021)*x[0]+(0.25100451419)*x[1]**o+(0.303226061816)*x[1]+(0.138976177556)*x[2]**o+(-0.330104849626)*x[2]
        arg[3,3,2,1]=(-0.136437682299)*x[0]**o+(0.938786085213)*x[0]+(0.853018944962)*x[1]**o+(-0.628078903025)*x[1]+(-0.903567849919)*x[2]**o+(0.837621994591)*x[2]
        arg[3,4,0,0]=(0.28632587806)*x[0]**o+(-0.480626263522)*x[0]+(-0.0537965100699)*x[1]**o+(0.423252432505)*x[1]+(0.200777153039)*x[2]**o+(0.629713098071)*x[2]
        arg[3,4,0,1]=(0.326448964148)*x[0]**o+(0.809991245673)*x[0]+(-0.527362539073)*x[1]**o+(-0.0727890632651)*x[1]+(0.210806278508)*x[2]**o+(0.736535063786)*x[2]
        arg[3,4,1,0]=(0.760474271531)*x[0]**o+(0.378083396435)*x[0]+(-0.991296179693)*x[1]**o+(0.431965391269)*x[1]+(-0.12597704605)*x[2]**o+(-0.128017966575)*x[2]
        arg[3,4,1,1]=(-0.21508452109)*x[0]**o+(-0.583147117492)*x[0]+(0.779535489322)*x[1]**o+(0.601144982341)*x[1]+(-0.870362610456)*x[2]**o+(-0.318535391517)*x[2]
        arg[3,4,2,0]=(-0.195181632154)*x[0]**o+(0.237570596161)*x[0]+(-0.794725125363)*x[1]**o+(0.862030507166)*x[1]+(-0.612206533195)*x[2]**o+(0.494078406766)*x[2]
        arg[3,4,2,1]=(0.270278273932)*x[0]**o+(-0.115833233113)*x[0]+(0.754464364285)*x[1]**o+(-0.186069521903)*x[1]+(-0.864426716229)*x[2]**o+(-0.425054438064)*x[2]
        ref[0,0,0,0]=(0.412792261707)/(o+1.)+(0.904364083309)+(0.663998508168)*0.5**o
        ref[0,0,0,1]=(-1.07469333717)/(o+1.)+(-0.0923352454984)+(-0.010130464476)*0.5**o
        ref[0,0,1,0]=(0.287923726124)/(o+1.)+(-0.145507770124)+(-0.176542289385)*0.5**o
        ref[0,0,1,1]=(-0.24656050398)/(o+1.)+(0.341982901349)+(-0.98152572229)*0.5**o
        ref[0,0,2,0]=(1.24918965808)/(o+1.)+(0.165246366859)+(0.555897422173)*0.5**o
        ref[0,0,2,1]=(-1.23538766622)/(o+1.)+(-0.507697956406)+(0.23127040923)*0.5**o
        ref[0,1,0,0]=(-0.267829490776)/(o+1.)+(-0.549968075823)+(0.0202445177842)*0.5**o
        ref[0,1,0,1]=(0.826177960569)/(o+1.)+(0.113794537096)+(0.99935098551)*0.5**o
        ref[0,1,1,0]=(0.124131681048)/(o+1.)+(0.482277820918)+(0.151799117419)*0.5**o
        ref[0,1,1,1]=(0.620076580442)/(o+1.)+(0.401695495934)+(-0.51422532626)*0.5**o
        ref[0,1,2,0]=(1.75895293799)/(o+1.)+(-0.225823657945)+(0.908244888839)*0.5**o
        ref[0,1,2,1]=(-0.334875364565)/(o+1.)+(1.30271444502)+(-0.872307766634)*0.5**o
        ref[0,2,0,0]=(-0.42419420508)/(o+1.)+(-0.3575838585)+(0.00173619592035)*0.5**o
        ref[0,2,0,1]=(-0.57391253095)/(o+1.)+(-0.515088003037)+(0.998184336552)*0.5**o
        ref[0,2,1,0]=(0.114144861758)/(o+1.)+(-0.330251158386)+(-0.921234331968)*0.5**o
        ref[0,2,1,1]=(-1.42031433961)/(o+1.)+(0.581091144051)+(0.153578015941)*0.5**o
        ref[0,2,2,0]=(0.827236715358)/(o+1.)+(0.125325412749)+(0.29601674794)*0.5**o
        ref[0,2,2,1]=(1.34802786361)/(o+1.)+(0.280667679493)+(0.0580007647099)*0.5**o
        ref[0,3,0,0]=(1.28181937762)/(o+1.)+(-0.214505128496)+(-0.337999890505)*0.5**o
        ref[0,3,0,1]=(1.65569377802)/(o+1.)+(-0.363265639443)+(-0.825813329629)*0.5**o
        ref[0,3,1,0]=(0.560544990797)/(o+1.)+(-0.276979054204)+(0.699313512079)*0.5**o
        ref[0,3,1,1]=(-0.879495822379)/(o+1.)+(0.560842505268)+(0.140764124286)*0.5**o
        ref[0,3,2,0]=(-1.04751560841)/(o+1.)+(-0.238703591863)+(-0.68177752688)*0.5**o
        ref[0,3,2,1]=(-0.414022097401)/(o+1.)+(0.509666283203)+(0.655685968811)*0.5**o
        ref[0,4,0,0]=(1.0625549765)/(o+1.)+(-1.26969185098)+(-0.18027010068)*0.5**o
        ref[0,4,0,1]=(-0.430389257621)/(o+1.)+(-0.728926201043)+(-0.914768182433)*0.5**o
        ref[0,4,1,0]=(-0.585220311209)/(o+1.)+(0.344278586108)+(0.274727423835)*0.5**o
        ref[0,4,1,1]=(-0.596140928652)/(o+1.)+(-0.857974578823)+(0.257574026756)*0.5**o
        ref[0,4,2,0]=(-1.15566595381)/(o+1.)+(0.148922032671)+(0.703185665611)*0.5**o
        ref[0,4,2,1]=(0.255095282229)/(o+1.)+(-0.480012460544)+(-0.153792811673)*0.5**o
        ref[1,0,0,0]=(0.524122113192)/(o+1.)+(-0.434579759316)+(0.298044833507)*0.5**o
        ref[1,0,0,1]=(-0.220122924131)/(o+1.)+(0.466218310719)+(-0.575485929806)*0.5**o
        ref[1,0,1,0]=(0.290760959656)/(o+1.)+(-0.0552768720487)+(0.998358005707)*0.5**o
        ref[1,0,1,1]=(-1.69984876204)/(o+1.)+(0.325064655551)+(0.783593367143)*0.5**o
        ref[1,0,2,0]=(-0.999302906363)/(o+1.)+(0.639576565202)+(-0.42345692756)*0.5**o
        ref[1,0,2,1]=(-0.271313965844)/(o+1.)+(-0.0328780841139)+(-0.790917746681)*0.5**o
        ref[1,1,0,0]=(-1.17401606398)/(o+1.)+(0.654288857484)+(0.944343622558)*0.5**o
        ref[1,1,0,1]=(0.623620452946)/(o+1.)+(-0.0991803677964)+(-0.246711023019)*0.5**o
        ref[1,1,1,0]=(-0.218171642866)/(o+1.)+(-0.404913755908)+(-0.530782063061)*0.5**o
        ref[1,1,1,1]=(1.23525595763)/(o+1.)+(0.349524193353)+(-0.710844761085)*0.5**o
        ref[1,1,2,0]=(-0.42993438925)/(o+1.)+(0.399995211041)+(0.574657505707)*0.5**o
        ref[1,1,2,1]=(-0.217230696736)/(o+1.)+(0.635787738562)+(-0.365373199615)*0.5**o
        ref[1,2,0,0]=(-0.102563204807)/(o+1.)+(-0.188445289686)+(0.544776615353)*0.5**o
        ref[1,2,0,1]=(0.298732134665)/(o+1.)+(-0.416846709104)+(0.11684721915)*0.5**o
        ref[1,2,1,0]=(-0.234419306371)/(o+1.)+(0.383855155251)+(0.561736948232)*0.5**o
        ref[1,2,1,1]=(0.539973971687)/(o+1.)+(0.0241566615339)+(0.578630452142)*0.5**o
        ref[1,2,2,0]=(0.346023814217)/(o+1.)+(-0.515461022254)+(-0.0269203575755)*0.5**o
        ref[1,2,2,1]=(-1.12923994073)/(o+1.)+(0.687820128862)+(-0.534642633404)*0.5**o
        ref[1,3,0,0]=(-1.35206603301)/(o+1.)+(-0.195350398606)+(0.892302802156)*0.5**o
        ref[1,3,0,1]=(0.457546541694)/(o+1.)+(0.584952905039)+(0.883702447985)*0.5**o
        ref[1,3,1,0]=(-0.229382345029)/(o+1.)+(-0.51570070583)+(0.492527250981)*0.5**o
        ref[1,3,1,1]=(0.163467163568)/(o+1.)+(-0.511309270369)+(0.372499842901)*0.5**o
        ref[1,3,2,0]=(-1.10954940366)/(o+1.)+(0.613097649163)+(0.533098362924)*0.5**o
        ref[1,3,2,1]=(0.0423370323263)/(o+1.)+(-0.655974226943)+(-0.909038123273)*0.5**o
        ref[1,4,0,0]=(-0.149369878527)/(o+1.)+(0.594641654424)+(-0.974474210185)*0.5**o
        ref[1,4,0,1]=(0.995137009073)/(o+1.)+(-0.0464708598635)+(0.964745405431)*0.5**o
        ref[1,4,1,0]=(0.545158516903)/(o+1.)+(-0.562624149397)+(-0.714725894011)*0.5**o
        ref[1,4,1,1]=(-1.20779451465)/(o+1.)+(-0.189148851615)+(-0.809072545597)*0.5**o
        ref[1,4,2,0]=(-1.55971061601)/(o+1.)+(0.693489320143)+(-0.21281876112)*0.5**o
        ref[1,4,2,1]=(0.226662056752)/(o+1.)+(-0.454678810051)+(-0.247930702516)*0.5**o
        ref[2,0,0,0]=(-0.242758325417)/(o+1.)+(-0.527650883321)+(0.616352869937)*0.5**o
        ref[2,0,0,1]=(0.807761382166)/(o+1.)+(1.02431485248)+(-0.85704374519)*0.5**o
        ref[2,0,1,0]=(0.543068844645)/(o+1.)+(0.0690269849966)+(0.326870102902)*0.5**o
        ref[2,0,1,1]=(0.502168875857)/(o+1.)+(-0.0584590052478)+(-0.699939842176)*0.5**o
        ref[2,0,2,0]=(0.181622609504)/(o+1.)+(-0.57477595904)+(-0.985996827188)*0.5**o
        ref[2,0,2,1]=(-1.01828660056)/(o+1.)+(0.255680773601)+(-0.195357773436)*0.5**o
        ref[2,1,0,0]=(-0.961895417879)/(o+1.)+(-0.848679114358)+(0.601652939329)*0.5**o
        ref[2,1,0,1]=(0.177781123824)/(o+1.)+(-0.945003972397)+(0.534038022604)*0.5**o
        ref[2,1,1,0]=(-0.20513407563)/(o+1.)+(-0.126335927989)+(-0.361196978257)*0.5**o
        ref[2,1,1,1]=(0.712128730879)/(o+1.)+(-0.0623370236322)+(0.00930987921229)*0.5**o
        ref[2,1,2,0]=(0.692430706812)/(o+1.)+(0.417664138602)+(-0.181479246679)*0.5**o
        ref[2,1,2,1]=(0.650980026197)/(o+1.)+(-0.708193455952)+(-0.99665050175)*0.5**o
        ref[2,2,0,0]=(1.22766505061)/(o+1.)+(0.717584511586)+(0.870069083779)*0.5**o
        ref[2,2,0,1]=(0.633038618622)/(o+1.)+(-0.222859101796)+(-0.841259995642)*0.5**o
        ref[2,2,1,0]=(0.167120356805)/(o+1.)+(-1.00491115888)+(-0.561184923307)*0.5**o
        ref[2,2,1,1]=(0.164505345911)/(o+1.)+(1.15307619901)+(0.92243000401)*0.5**o
        ref[2,2,2,0]=(-1.16271376641)/(o+1.)+(-0.336355614369)+(0.331131296272)*0.5**o
        ref[2,2,2,1]=(0.672282083422)/(o+1.)+(0.018864653069)+(-0.47892171854)*0.5**o
        ref[2,3,0,0]=(1.52821633972)/(o+1.)+(-0.864758323678)+(0.0219406022052)*0.5**o
        ref[2,3,0,1]=(-0.326213615062)/(o+1.)+(0.152909191896)+(-0.114799511617)*0.5**o
        ref[2,3,1,0]=(-0.724364553267)/(o+1.)+(-0.439822878578)+(-0.103518223905)*0.5**o
        ref[2,3,1,1]=(-1.107549965)/(o+1.)+(-0.705530843209)+(-0.958142224242)*0.5**o
        ref[2,3,2,0]=(-0.825620325844)/(o+1.)+(-0.169496279085)+(0.730554594734)*0.5**o
        ref[2,3,2,1]=(-0.0678334241233)/(o+1.)+(0.0396617729923)+(-0.930974589897)*0.5**o
        ref[2,4,0,0]=(0.809143207998)/(o+1.)+(-0.709013942786)+(-0.246046780287)*0.5**o
        ref[2,4,0,1]=(0.0938678290563)/(o+1.)+(-0.0282613348041)+(-0.612751147889)*0.5**o
        ref[2,4,1,0]=(-0.0572988459938)/(o+1.)+(-0.496307592572)+(-0.26092972064)*0.5**o
        ref[2,4,1,1]=(1.38060158884)/(o+1.)+(-0.643768055252)+(0.325840618456)*0.5**o
        ref[2,4,2,0]=(-1.03935942333)/(o+1.)+(0.130235179204)+(0.655590135983)*0.5**o
        ref[2,4,2,1]=(-0.41630624669)/(o+1.)+(0.259401996992)+(0.570712162768)*0.5**o
        ref[3,0,0,0]=(-0.357343519171)/(o+1.)+(-0.518658431855)+(-0.486023722572)*0.5**o
        ref[3,0,0,1]=(-0.927017258737)/(o+1.)+(0.566657824853)+(-0.705291625776)*0.5**o
        ref[3,0,1,0]=(-0.331128184244)/(o+1.)+(0.535203433907)+(-0.838974120992)*0.5**o
        ref[3,0,1,1]=(-0.735721663177)/(o+1.)+(-0.293634410339)+(0.140027541071)*0.5**o
        ref[3,0,2,0]=(-0.178945077431)/(o+1.)+(-0.300862767422)+(-0.39083471939)*0.5**o
        ref[3,0,2,1]=(1.18223012428)/(o+1.)+(-0.121465976141)+(0.0734743009776)*0.5**o
        ref[3,1,0,0]=(1.46408415003)/(o+1.)+(-0.0664881764563)+(-0.414502599055)*0.5**o
        ref[3,1,0,1]=(1.3349422575)/(o+1.)+(0.283498682724)+(0.886115095396)*0.5**o
        ref[3,1,1,0]=(0.816511111677)/(o+1.)+(0.175825822544)+(0.312577129879)*0.5**o
        ref[3,1,1,1]=(-0.105059408543)/(o+1.)+(-0.782531675802)+(0.784369258288)*0.5**o
        ref[3,1,2,0]=(-0.744294216228)/(o+1.)+(-0.301922969419)+(-0.337332295082)*0.5**o
        ref[3,1,2,1]=(-0.669631653258)/(o+1.)+(0.0177835713351)+(0.791575892417)*0.5**o
        ref[3,2,0,0]=(0.274579265949)/(o+1.)+(0.148529098776)+(-0.0909007497318)*0.5**o
        ref[3,2,0,1]=(0.956130125711)/(o+1.)+(0.482854719491)+(-0.764137934896)*0.5**o
        ref[3,2,1,0]=(-0.266431989669)/(o+1.)+(0.313297832739)+(0.49435832562)*0.5**o
        ref[3,2,1,1]=(-0.726703087024)/(o+1.)+(0.542295175061)+(-0.460752644946)*0.5**o
        ref[3,2,2,0]=(0.87651829094)/(o+1.)+(-0.185020317268)+(0.0946687985697)*0.5**o
        ref[3,2,2,1]=(-0.113135321417)/(o+1.)+(-0.221883684949)+(0.177521713734)*0.5**o
        ref[3,3,0,0]=(0.155561630388)/(o+1.)+(0.495429901163)+(0.592201472175)*0.5**o
        ref[3,3,0,1]=(-0.44887181194)/(o+1.)+(-0.20854804469)+(0.572972211447)*0.5**o
        ref[3,3,1,0]=(1.11423248654)/(o+1.)+(-0.212131906831)+(0.775163727164)*0.5**o
        ref[3,3,1,1]=(0.723748205425)/(o+1.)+(0.785798099184)+(-0.244586606882)*0.5**o
        ref[3,3,2,0]=(0.389980691746)/(o+1.)+(-0.0994690359155)+(-0.676040282048)*0.5**o
        ref[3,3,2,1]=(-0.0505489049572)/(o+1.)+(0.57416458839)+(-0.136437682299)*0.5**o
        ref[3,4,0,0]=(0.14698064297)/(o+1.)+(0.286169633527)+(0.28632587806)*0.5**o
        ref[3,4,0,1]=(-0.316556260565)/(o+1.)+(0.736868623097)+(0.326448964148)*0.5**o
        ref[3,4,1,0]=(-1.11727322574)/(o+1.)+(0.341015410564)+(0.760474271531)*0.5**o
        ref[3,4,1,1]=(-0.0908271211344)/(o+1.)+(-0.150268763334)+(-0.21508452109)*0.5**o
        ref[3,4,2,0]=(-1.40693165856)/(o+1.)+(0.796839755046)+(-0.195181632154)*0.5**o
        ref[3,4,2,1]=(-0.109962351944)/(o+1.)+(-0.36347859654)+(0.270278273932)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_Solution_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.0433006726989)*x[0]**o+(0.819109331838)*x[0]+(0.150292245587)*x[1]**o+(0.949397165008)*x[1]
        ref=(0.150292245587)/(o+1.)+(0.884253248423)+(0.0433006726989)*0.5**o
      else:
        arg=(-0.765858486678)*x[0]**o+(-0.829061249978)*x[0]+(-0.0255273602735)*x[1]**o+(-0.93097352763)*x[1]+(-0.775974817954)*x[2]**o+(0.0414580763278)*x[2]
        ref=(-0.801502178228)/(o+1.)+(-0.85928835064)+(-0.765858486678)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_Solution_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.708418284738)*x[0]**o+(-0.265416249865)*x[0]+(0.0412873279676)*x[1]**o+(0.18827690539)*x[1]
        arg[1]=(-0.0639578348425)*x[0]**o+(-0.435408788697)*x[0]+(0.429134354603)*x[1]**o+(0.0391558318157)*x[1]
        ref[0]=(0.0412873279676)/(o+1.)+(-0.0385696722374)+(0.708418284738)*0.5**o
        ref[1]=(0.429134354603)/(o+1.)+(-0.198126478441)+(-0.0639578348425)*0.5**o
      else:
        arg[0]=(-0.516066856372)*x[0]**o+(-0.509677909948)*x[0]+(0.189594948337)*x[1]**o+(0.743432553688)*x[1]+(0.687864109046)*x[2]**o+(0.257280277379)*x[2]
        arg[1]=(0.948762785699)*x[0]**o+(0.372861763607)*x[0]+(-0.166460043702)*x[1]**o+(0.0474018901531)*x[1]+(-0.485380626595)*x[2]**o+(-0.341701937267)*x[2]
        ref[0]=(0.877459057383)/(o+1.)+(0.245517460559)+(-0.516066856372)*0.5**o
        ref[1]=(-0.651840670298)/(o+1.)+(0.0392808582465)+(0.948762785699)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_Solution_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.0540280486222)*x[0]**o+(-0.936629447756)*x[0]+(0.530867101627)*x[1]**o+(-0.404522355122)*x[1]
        arg[0,1]=(0.585051034599)*x[0]**o+(-0.899683342239)*x[0]+(0.61865830472)*x[1]**o+(0.491768251743)*x[1]
        arg[0,2]=(0.924131379905)*x[0]**o+(-0.394367734246)*x[0]+(0.432122525065)*x[1]**o+(0.79923634141)*x[1]
        arg[0,3]=(-0.41435196904)*x[0]**o+(-0.22039395628)*x[0]+(-0.866551833459)*x[1]**o+(-0.0966534238435)*x[1]
        arg[0,4]=(0.855785701017)*x[0]**o+(0.0879754354535)*x[0]+(-0.0727782016239)*x[1]**o+(0.746878225435)*x[1]
        arg[1,0]=(-0.868450306659)*x[0]**o+(-0.00845792225901)*x[0]+(0.227399876353)*x[1]**o+(-0.522196483799)*x[1]
        arg[1,1]=(0.53076931297)*x[0]**o+(0.165744140226)*x[0]+(0.14586398181)*x[1]**o+(-0.694169551043)*x[1]
        arg[1,2]=(-0.936146144862)*x[0]**o+(-0.198367580291)*x[0]+(0.848822682163)*x[1]**o+(0.360521711263)*x[1]
        arg[1,3]=(-0.395889451325)*x[0]**o+(0.510713345487)*x[0]+(-0.62440033386)*x[1]**o+(0.29695714683)*x[1]
        arg[1,4]=(0.847685553395)*x[0]**o+(-0.291266469606)*x[0]+(-0.517114839541)*x[1]**o+(-0.279948585383)*x[1]
        arg[2,0]=(-0.622481093435)*x[0]**o+(0.0911922519622)*x[0]+(0.216648336495)*x[1]**o+(0.663562516802)*x[1]
        arg[2,1]=(0.567833843918)*x[0]**o+(0.18116887359)*x[0]+(-0.0433012798352)*x[1]**o+(0.998332142285)*x[1]
        arg[2,2]=(-0.81729786652)*x[0]**o+(-0.634680104287)*x[0]+(0.991513806139)*x[1]**o+(-0.0721700196003)*x[1]
        arg[2,3]=(0.121181286827)*x[0]**o+(-0.677803464822)*x[0]+(0.870139639102)*x[1]**o+(-0.134662002755)*x[1]
        arg[2,4]=(-0.167376807346)*x[0]**o+(-0.500556007898)*x[0]+(-0.839256017197)*x[1]**o+(-0.82147890878)*x[1]
        arg[3,0]=(0.576521064132)*x[0]**o+(-0.510276372588)*x[0]+(0.92888227487)*x[1]**o+(-0.88293512036)*x[1]
        arg[3,1]=(-0.159048471862)*x[0]**o+(-0.0479500210678)*x[0]+(-0.0216438779517)*x[1]**o+(-0.108252189548)*x[1]
        arg[3,2]=(-0.483779576334)*x[0]**o+(-0.871829147209)*x[0]+(-0.22556218283)*x[1]**o+(0.451007116911)*x[1]
        arg[3,3]=(-0.358888369902)*x[0]**o+(0.436554685191)*x[0]+(-0.0359187908139)*x[1]**o+(-0.0148930474244)*x[1]
        arg[3,4]=(-0.995411478882)*x[0]**o+(-0.457206328785)*x[0]+(-0.833823947273)*x[1]**o+(-0.0517068976093)*x[1]
        ref[0,0]=(0.530867101627)/(o+1.)+(-0.670575901439)+(-0.0540280486222)*0.5**o
        ref[0,1]=(0.61865830472)/(o+1.)+(-0.203957545248)+(0.585051034599)*0.5**o
        ref[0,2]=(0.432122525065)/(o+1.)+(0.202434303582)+(0.924131379905)*0.5**o
        ref[0,3]=(-0.866551833459)/(o+1.)+(-0.158523690062)+(-0.41435196904)*0.5**o
        ref[0,4]=(-0.0727782016239)/(o+1.)+(0.417426830444)+(0.855785701017)*0.5**o
        ref[1,0]=(0.227399876353)/(o+1.)+(-0.265327203029)+(-0.868450306659)*0.5**o
        ref[1,1]=(0.14586398181)/(o+1.)+(-0.264212705408)+(0.53076931297)*0.5**o
        ref[1,2]=(0.848822682163)/(o+1.)+(0.0810770654864)+(-0.936146144862)*0.5**o
        ref[1,3]=(-0.62440033386)/(o+1.)+(0.403835246159)+(-0.395889451325)*0.5**o
        ref[1,4]=(-0.517114839541)/(o+1.)+(-0.285607527495)+(0.847685553395)*0.5**o
        ref[2,0]=(0.216648336495)/(o+1.)+(0.377377384382)+(-0.622481093435)*0.5**o
        ref[2,1]=(-0.0433012798352)/(o+1.)+(0.589750507938)+(0.567833843918)*0.5**o
        ref[2,2]=(0.991513806139)/(o+1.)+(-0.353425061944)+(-0.81729786652)*0.5**o
        ref[2,3]=(0.870139639102)/(o+1.)+(-0.406232733789)+(0.121181286827)*0.5**o
        ref[2,4]=(-0.839256017197)/(o+1.)+(-0.661017458339)+(-0.167376807346)*0.5**o
        ref[3,0]=(0.92888227487)/(o+1.)+(-0.696605746474)+(0.576521064132)*0.5**o
        ref[3,1]=(-0.0216438779517)/(o+1.)+(-0.078101105308)+(-0.159048471862)*0.5**o
        ref[3,2]=(-0.22556218283)/(o+1.)+(-0.210411015149)+(-0.483779576334)*0.5**o
        ref[3,3]=(-0.0359187908139)/(o+1.)+(0.210830818883)+(-0.358888369902)*0.5**o
        ref[3,4]=(-0.833823947273)/(o+1.)+(-0.254456613197)+(-0.995411478882)*0.5**o
      else:
        arg[0,0]=(-0.937858877179)*x[0]**o+(-0.41590845461)*x[0]+(0.317385444442)*x[1]**o+(-0.166284606371)*x[1]+(-0.846106994172)*x[2]**o+(0.227876218054)*x[2]
        arg[0,1]=(0.497785867084)*x[0]**o+(0.883557362705)*x[0]+(-0.846602726567)*x[1]**o+(-0.859091910548)*x[1]+(-0.295742503292)*x[2]**o+(-0.853931233266)*x[2]
        arg[0,2]=(0.744501879094)*x[0]**o+(-0.0181115125382)*x[0]+(-0.856099357899)*x[1]**o+(0.767449074734)*x[1]+(-0.858552187616)*x[2]**o+(-0.580142057156)*x[2]
        arg[0,3]=(0.210031467359)*x[0]**o+(-0.994426053833)*x[0]+(0.648522478175)*x[1]**o+(0.0355805603417)*x[1]+(-0.274347447112)*x[2]**o+(-0.280165537012)*x[2]
        arg[0,4]=(-0.323650660168)*x[0]**o+(-0.279592941595)*x[0]+(-0.478811425713)*x[1]**o+(0.19753271141)*x[1]+(0.476202875551)*x[2]**o+(0.657358100659)*x[2]
        arg[1,0]=(0.890659868437)*x[0]**o+(0.100235405989)*x[0]+(0.325177871418)*x[1]**o+(0.675379114656)*x[1]+(0.458473828724)*x[2]**o+(-0.449822467434)*x[2]
        arg[1,1]=(-0.467895039242)*x[0]**o+(-0.551300000422)*x[0]+(0.976111922801)*x[1]**o+(0.809389941274)*x[1]+(-0.693066235685)*x[2]**o+(0.595719115505)*x[2]
        arg[1,2]=(-0.727304617172)*x[0]**o+(-0.270682330445)*x[0]+(-0.32926621635)*x[1]**o+(0.0389068111165)*x[1]+(0.300096425532)*x[2]**o+(-0.964910767632)*x[2]
        arg[1,3]=(0.195012106644)*x[0]**o+(0.18137839607)*x[0]+(-0.823129279257)*x[1]**o+(-0.0440106889304)*x[1]+(-0.298898659898)*x[2]**o+(0.801623925974)*x[2]
        arg[1,4]=(0.590200326369)*x[0]**o+(-0.800085209049)*x[0]+(-0.323836133834)*x[1]**o+(-0.953413527643)*x[1]+(0.0257128844735)*x[2]**o+(0.652469516082)*x[2]
        arg[2,0]=(-0.467537281458)*x[0]**o+(-0.05673369912)*x[0]+(-0.813542619659)*x[1]**o+(-0.778596374878)*x[1]+(0.752073310476)*x[2]**o+(0.187824163254)*x[2]
        arg[2,1]=(-0.75708112983)*x[0]**o+(0.866111788737)*x[0]+(0.715424327432)*x[1]**o+(-0.297967739435)*x[1]+(0.420241611178)*x[2]**o+(-0.616867390634)*x[2]
        arg[2,2]=(0.00628628803839)*x[0]**o+(0.700962301988)*x[0]+(0.712979263919)*x[1]**o+(-0.414401132518)*x[1]+(-0.975498928756)*x[2]**o+(0.858116391039)*x[2]
        arg[2,3]=(0.948164399923)*x[0]**o+(-0.88227825142)*x[0]+(0.30703295864)*x[1]**o+(0.510607674984)*x[1]+(0.362361719392)*x[2]**o+(-0.208696380629)*x[2]
        arg[2,4]=(0.661545494115)*x[0]**o+(-0.122960355184)*x[0]+(-0.801367604574)*x[1]**o+(0.395708459987)*x[1]+(0.0197225340457)*x[2]**o+(-0.377005073637)*x[2]
        arg[3,0]=(0.733597350529)*x[0]**o+(0.339318978843)*x[0]+(0.317608635437)*x[1]**o+(0.147746051469)*x[1]+(-0.33413935488)*x[2]**o+(-0.793337674933)*x[2]
        arg[3,1]=(0.0918904281646)*x[0]**o+(-0.530786152204)*x[0]+(-0.587542750401)*x[1]**o+(-0.810331029621)*x[1]+(-0.40221065987)*x[2]**o+(-0.250472092093)*x[2]
        arg[3,2]=(0.966442952042)*x[0]**o+(-0.756649866552)*x[0]+(-0.36466223222)*x[1]**o+(-0.48639005055)*x[1]+(-0.831374462742)*x[2]**o+(0.503451143644)*x[2]
        arg[3,3]=(-0.322275644664)*x[0]**o+(-0.254067149982)*x[0]+(-0.440264794128)*x[1]**o+(0.226546045529)*x[1]+(-0.761870916789)*x[2]**o+(0.70024066379)*x[2]
        arg[3,4]=(0.595459342613)*x[0]**o+(-0.439436200258)*x[0]+(0.0932538180176)*x[1]**o+(-0.975450806707)*x[1]+(0.0461423323623)*x[2]**o+(0.0294352472154)*x[2]
        ref[0,0]=(-0.52872154973)/(o+1.)+(-0.177158421464)+(-0.937858877179)*0.5**o
        ref[0,1]=(-1.14234522986)/(o+1.)+(-0.414732890554)+(0.497785867084)*0.5**o
        ref[0,2]=(-1.71465154551)/(o+1.)+(0.0845977525197)+(0.744501879094)*0.5**o
        ref[0,3]=(0.374175031062)/(o+1.)+(-0.619505515252)+(0.210031467359)*0.5**o
        ref[0,4]=(-0.00260855016132)/(o+1.)+(0.287648935237)+(-0.323650660168)*0.5**o
        ref[1,0]=(0.783651700142)/(o+1.)+(0.162896026605)+(0.890659868437)*0.5**o
        ref[1,1]=(0.283045687117)/(o+1.)+(0.426904528178)+(-0.467895039242)*0.5**o
        ref[1,2]=(-0.0291697908188)/(o+1.)+(-0.59834314348)+(-0.727304617172)*0.5**o
        ref[1,3]=(-1.12202793916)/(o+1.)+(0.469495816557)+(0.195012106644)*0.5**o
        ref[1,4]=(-0.29812324936)/(o+1.)+(-0.550514610305)+(0.590200326369)*0.5**o
        ref[2,0]=(-0.0614693091826)/(o+1.)+(-0.323752955372)+(-0.467537281458)*0.5**o
        ref[2,1]=(1.13566593861)/(o+1.)+(-0.024361670666)+(-0.75708112983)*0.5**o
        ref[2,2]=(-0.262519664837)/(o+1.)+(0.572338780255)+(0.00628628803839)*0.5**o
        ref[2,3]=(0.669394678032)/(o+1.)+(-0.290183478532)+(0.948164399923)*0.5**o
        ref[2,4]=(-0.781645070528)/(o+1.)+(-0.0521284844165)+(0.661545494115)*0.5**o
        ref[3,0]=(-0.0165307194439)/(o+1.)+(-0.15313632231)+(0.733597350529)*0.5**o
        ref[3,1]=(-0.989753410271)/(o+1.)+(-0.795794636959)+(0.0918904281646)*0.5**o
        ref[3,2]=(-1.19603669496)/(o+1.)+(-0.369794386729)+(0.966442952042)*0.5**o
        ref[3,3]=(-1.20213571092)/(o+1.)+(0.336359779668)+(-0.322275644664)*0.5**o
        ref[3,4]=(0.13939615038)/(o+1.)+(-0.692725879875)+(0.595459342613)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_Solution_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.12237588236)*x[0]**o+(-0.671341785528)*x[0]+(0.998910161099)*x[1]**o+(-0.552290246966)*x[1]
        arg[0,0,1]=(-0.90821257221)*x[0]**o+(-0.413391927764)*x[0]+(-0.428904099491)*x[1]**o+(-0.226766222189)*x[1]
        arg[0,1,0]=(-0.333819313127)*x[0]**o+(-0.458569899029)*x[0]+(-0.83776177332)*x[1]**o+(0.22951184631)*x[1]
        arg[0,1,1]=(-0.0571879315489)*x[0]**o+(-0.38056227895)*x[0]+(0.0784628125977)*x[1]**o+(-0.659353613638)*x[1]
        arg[1,0,0]=(0.645871468353)*x[0]**o+(-0.986005542006)*x[0]+(0.395704509276)*x[1]**o+(0.980666918726)*x[1]
        arg[1,0,1]=(0.337293036597)*x[0]**o+(-0.520809490378)*x[0]+(0.543657498632)*x[1]**o+(0.632777244118)*x[1]
        arg[1,1,0]=(-0.432349538121)*x[0]**o+(-0.579685247099)*x[0]+(0.331943072726)*x[1]**o+(0.25042188644)*x[1]
        arg[1,1,1]=(0.329861800715)*x[0]**o+(-0.48725090895)*x[0]+(-0.149357998414)*x[1]**o+(-0.580650134153)*x[1]
        arg[2,0,0]=(-0.75424658357)*x[0]**o+(-0.696352517414)*x[0]+(-0.00272313166917)*x[1]**o+(-0.488581343546)*x[1]
        arg[2,0,1]=(-0.136489901339)*x[0]**o+(-0.752587775699)*x[0]+(0.986413392791)*x[1]**o+(0.759344728162)*x[1]
        arg[2,1,0]=(-0.0832674877963)*x[0]**o+(0.765419402778)*x[0]+(0.958533611564)*x[1]**o+(-0.315461136294)*x[1]
        arg[2,1,1]=(0.362764663642)*x[0]**o+(-0.351519282456)*x[0]+(-0.115554433923)*x[1]**o+(-0.789365036398)*x[1]
        arg[3,0,0]=(0.0977752933224)*x[0]**o+(-0.838881001576)*x[0]+(0.275578737811)*x[1]**o+(-0.971208155806)*x[1]
        arg[3,0,1]=(0.0588243034654)*x[0]**o+(-0.141302234145)*x[0]+(-0.460044396206)*x[1]**o+(-0.776793718354)*x[1]
        arg[3,1,0]=(0.786922930047)*x[0]**o+(-0.187797924361)*x[0]+(0.971516347812)*x[1]**o+(-0.250547377602)*x[1]
        arg[3,1,1]=(-0.69240453335)*x[0]**o+(0.439425578473)*x[0]+(0.00864940781276)*x[1]**o+(-0.458898387231)*x[1]
        arg[4,0,0]=(0.0700104829681)*x[0]**o+(0.905031121846)*x[0]+(0.37676750477)*x[1]**o+(0.411190149331)*x[1]
        arg[4,0,1]=(-0.615737714754)*x[0]**o+(0.1945204765)*x[0]+(0.0889014902938)*x[1]**o+(0.650020022597)*x[1]
        arg[4,1,0]=(-0.883448519911)*x[0]**o+(0.00124188371434)*x[0]+(-0.501662684156)*x[1]**o+(0.290235036398)*x[1]
        arg[4,1,1]=(-0.723885968554)*x[0]**o+(-0.0112308623755)*x[0]+(0.679499427805)*x[1]**o+(-0.00261690404287)*x[1]
        arg[5,0,0]=(0.0834731365594)*x[0]**o+(-0.873857991293)*x[0]+(0.831158842487)*x[1]**o+(-0.585808323324)*x[1]
        arg[5,0,1]=(0.250830078938)*x[0]**o+(0.310347394372)*x[0]+(-0.412138455827)*x[1]**o+(0.974823718029)*x[1]
        arg[5,1,0]=(-0.865155065381)*x[0]**o+(0.72190541012)*x[0]+(0.72020941875)*x[1]**o+(-0.604997858133)*x[1]
        arg[5,1,1]=(0.57252234131)*x[0]**o+(-0.366230226771)*x[0]+(0.115400105646)*x[1]**o+(0.0485229323141)*x[1]
        ref[0,0,0]=(0.998910161099)/(o+1.)+(-0.611816016247)+(-0.12237588236)*0.5**o
        ref[0,0,1]=(-0.428904099491)/(o+1.)+(-0.320079074977)+(-0.90821257221)*0.5**o
        ref[0,1,0]=(-0.83776177332)/(o+1.)+(-0.11452902636)+(-0.333819313127)*0.5**o
        ref[0,1,1]=(0.0784628125977)/(o+1.)+(-0.519957946294)+(-0.0571879315489)*0.5**o
        ref[1,0,0]=(0.395704509276)/(o+1.)+(-0.00266931163989)+(0.645871468353)*0.5**o
        ref[1,0,1]=(0.543657498632)/(o+1.)+(0.0559838768697)+(0.337293036597)*0.5**o
        ref[1,1,0]=(0.331943072726)/(o+1.)+(-0.16463168033)+(-0.432349538121)*0.5**o
        ref[1,1,1]=(-0.149357998414)/(o+1.)+(-0.533950521552)+(0.329861800715)*0.5**o
        ref[2,0,0]=(-0.00272313166917)/(o+1.)+(-0.59246693048)+(-0.75424658357)*0.5**o
        ref[2,0,1]=(0.986413392791)/(o+1.)+(0.00337847623164)+(-0.136489901339)*0.5**o
        ref[2,1,0]=(0.958533611564)/(o+1.)+(0.224979133242)+(-0.0832674877963)*0.5**o
        ref[2,1,1]=(-0.115554433923)/(o+1.)+(-0.570442159427)+(0.362764663642)*0.5**o
        ref[3,0,0]=(0.275578737811)/(o+1.)+(-0.905044578691)+(0.0977752933224)*0.5**o
        ref[3,0,1]=(-0.460044396206)/(o+1.)+(-0.45904797625)+(0.0588243034654)*0.5**o
        ref[3,1,0]=(0.971516347812)/(o+1.)+(-0.219172650981)+(0.786922930047)*0.5**o
        ref[3,1,1]=(0.00864940781276)/(o+1.)+(-0.00973640437899)+(-0.69240453335)*0.5**o
        ref[4,0,0]=(0.37676750477)/(o+1.)+(0.658110635589)+(0.0700104829681)*0.5**o
        ref[4,0,1]=(0.0889014902938)/(o+1.)+(0.422270249549)+(-0.615737714754)*0.5**o
        ref[4,1,0]=(-0.501662684156)/(o+1.)+(0.145738460056)+(-0.883448519911)*0.5**o
        ref[4,1,1]=(0.679499427805)/(o+1.)+(-0.00692388320921)+(-0.723885968554)*0.5**o
        ref[5,0,0]=(0.831158842487)/(o+1.)+(-0.729833157309)+(0.0834731365594)*0.5**o
        ref[5,0,1]=(-0.412138455827)/(o+1.)+(0.642585556201)+(0.250830078938)*0.5**o
        ref[5,1,0]=(0.72020941875)/(o+1.)+(0.0584537759935)+(-0.865155065381)*0.5**o
        ref[5,1,1]=(0.115400105646)/(o+1.)+(-0.158853647228)+(0.57252234131)*0.5**o
      else:
        arg[0,0,0]=(0.962792187023)*x[0]**o+(0.799861778652)*x[0]+(-0.811367630134)*x[1]**o+(-0.553496861095)*x[1]+(-0.810089307714)*x[2]**o+(0.135449765364)*x[2]
        arg[0,0,1]=(0.644573004654)*x[0]**o+(-0.565823623798)*x[0]+(0.337026645127)*x[1]**o+(0.413612857799)*x[1]+(0.604366347987)*x[2]**o+(0.0867373815054)*x[2]
        arg[0,1,0]=(0.829914553754)*x[0]**o+(-0.462419132852)*x[0]+(0.889464473044)*x[1]**o+(-0.425594549042)*x[1]+(0.679251542389)*x[2]**o+(-0.501090535615)*x[2]
        arg[0,1,1]=(0.874166303558)*x[0]**o+(0.437595762221)*x[0]+(-0.836229998654)*x[1]**o+(-0.704020633222)*x[1]+(-0.279878325513)*x[2]**o+(0.697289463565)*x[2]
        arg[1,0,0]=(-0.963425479949)*x[0]**o+(0.399090631945)*x[0]+(-0.80371987894)*x[1]**o+(0.614102223184)*x[1]+(-0.821741166107)*x[2]**o+(-0.245209306654)*x[2]
        arg[1,0,1]=(-0.288791346674)*x[0]**o+(0.0125896799209)*x[0]+(0.0536290891541)*x[1]**o+(-0.127968165901)*x[1]+(-0.635591402779)*x[2]**o+(-0.723850350215)*x[2]
        arg[1,1,0]=(-0.554196270608)*x[0]**o+(-0.409293351346)*x[0]+(-0.759821124187)*x[1]**o+(-0.828913354317)*x[1]+(0.66900958614)*x[2]**o+(0.412403847239)*x[2]
        arg[1,1,1]=(-0.862735330269)*x[0]**o+(-0.406644646366)*x[0]+(-0.375399872184)*x[1]**o+(-0.10762837777)*x[1]+(0.540886844249)*x[2]**o+(0.97487281528)*x[2]
        arg[2,0,0]=(0.388899900771)*x[0]**o+(-0.692092360652)*x[0]+(0.794415524913)*x[1]**o+(0.369239170999)*x[1]+(0.482269641926)*x[2]**o+(0.58717167121)*x[2]
        arg[2,0,1]=(0.324795219208)*x[0]**o+(0.508631698719)*x[0]+(0.128456415737)*x[1]**o+(0.73678163559)*x[1]+(0.0527520660005)*x[2]**o+(-0.180289299372)*x[2]
        arg[2,1,0]=(-0.901972287405)*x[0]**o+(-0.635701120264)*x[0]+(0.298343193246)*x[1]**o+(-0.870902858159)*x[1]+(-0.979619200422)*x[2]**o+(0.708567755329)*x[2]
        arg[2,1,1]=(0.391663724321)*x[0]**o+(0.69891810611)*x[0]+(0.717069638885)*x[1]**o+(0.493487853359)*x[1]+(0.270536570073)*x[2]**o+(0.679115502666)*x[2]
        arg[3,0,0]=(0.830590943609)*x[0]**o+(-0.000320739222143)*x[0]+(-0.659146599349)*x[1]**o+(-0.858428982679)*x[1]+(-0.158070494634)*x[2]**o+(-0.364394913197)*x[2]
        arg[3,0,1]=(0.903615175916)*x[0]**o+(0.186562406505)*x[0]+(0.463860110418)*x[1]**o+(0.0382550628307)*x[1]+(0.835148180357)*x[2]**o+(-0.117637969912)*x[2]
        arg[3,1,0]=(-0.901874408714)*x[0]**o+(0.00493918499294)*x[0]+(0.180179475909)*x[1]**o+(-0.549428619271)*x[1]+(-0.392536708051)*x[2]**o+(0.68054148024)*x[2]
        arg[3,1,1]=(0.807693382033)*x[0]**o+(-0.47532617097)*x[0]+(-0.52499678021)*x[1]**o+(-0.0365395713918)*x[1]+(0.654881606829)*x[2]**o+(-0.63350736651)*x[2]
        arg[4,0,0]=(0.440830144989)*x[0]**o+(-0.454694325046)*x[0]+(0.403509667018)*x[1]**o+(0.360080283937)*x[1]+(0.164208775222)*x[2]**o+(0.203945755083)*x[2]
        arg[4,0,1]=(-0.836804887763)*x[0]**o+(-0.910979763109)*x[0]+(-0.629411293881)*x[1]**o+(0.0546020113997)*x[1]+(0.63435754491)*x[2]**o+(-0.323224122475)*x[2]
        arg[4,1,0]=(0.204216251019)*x[0]**o+(-0.372349067194)*x[0]+(-0.320768886225)*x[1]**o+(-0.139715856072)*x[1]+(0.62329130659)*x[2]**o+(0.626784461235)*x[2]
        arg[4,1,1]=(0.275074685897)*x[0]**o+(0.0664691742991)*x[0]+(0.949175504221)*x[1]**o+(-0.438858093561)*x[1]+(0.330489174462)*x[2]**o+(0.361586647545)*x[2]
        arg[5,0,0]=(0.370194889673)*x[0]**o+(-0.0549428893157)*x[0]+(0.0219932528442)*x[1]**o+(-0.699553487775)*x[1]+(0.0161212665659)*x[2]**o+(-0.268426682266)*x[2]
        arg[5,0,1]=(-0.527801267225)*x[0]**o+(0.151774276637)*x[0]+(-0.0623666836223)*x[1]**o+(-0.953572390482)*x[1]+(0.778912307756)*x[2]**o+(-0.851760269136)*x[2]
        arg[5,1,0]=(0.11418535486)*x[0]**o+(-0.299719975234)*x[0]+(-0.508303365848)*x[1]**o+(-0.856367302992)*x[1]+(0.311609370032)*x[2]**o+(0.4731881155)*x[2]
        arg[5,1,1]=(0.455947680647)*x[0]**o+(0.856811965126)*x[0]+(-0.176215162097)*x[1]**o+(0.581325693162)*x[1]+(-0.306514780798)*x[2]**o+(-0.005241787834)*x[2]
        ref[0,0,0]=(-1.62145693785)/(o+1.)+(0.19090734146)+(0.962792187023)*0.5**o
        ref[0,0,1]=(0.941392993115)/(o+1.)+(-0.0327366922471)+(0.644573004654)*0.5**o
        ref[0,1,0]=(1.56871601543)/(o+1.)+(-0.694552108754)+(0.829914553754)*0.5**o
        ref[0,1,1]=(-1.11610832417)/(o+1.)+(0.215432296282)+(0.874166303558)*0.5**o
        ref[1,0,0]=(-1.62546104505)/(o+1.)+(0.383991774237)+(-0.963425479949)*0.5**o
        ref[1,0,1]=(-0.581962313625)/(o+1.)+(-0.419614418097)+(-0.288791346674)*0.5**o
        ref[1,1,0]=(-0.0908115380472)/(o+1.)+(-0.412901429212)+(-0.554196270608)*0.5**o
        ref[1,1,1]=(0.165486972066)/(o+1.)+(0.230299895572)+(-0.862735330269)*0.5**o
        ref[2,0,0]=(1.27668516684)/(o+1.)+(0.132159240779)+(0.388899900771)*0.5**o
        ref[2,0,1]=(0.181208481738)/(o+1.)+(0.532562017469)+(0.324795219208)*0.5**o
        ref[2,1,0]=(-0.681276007176)/(o+1.)+(-0.399018111547)+(-0.901972287405)*0.5**o
        ref[2,1,1]=(0.987606208958)/(o+1.)+(0.935760731067)+(0.391663724321)*0.5**o
        ref[3,0,0]=(-0.817217093983)/(o+1.)+(-0.611572317549)+(0.830590943609)*0.5**o
        ref[3,0,1]=(1.29900829077)/(o+1.)+(0.0535897497116)+(0.903615175916)*0.5**o
        ref[3,1,0]=(-0.212357232141)/(o+1.)+(0.0680260229807)+(-0.901874408714)*0.5**o
        ref[3,1,1]=(0.12988482662)/(o+1.)+(-0.572686554436)+(0.807693382033)*0.5**o
        ref[4,0,0]=(0.567718442239)/(o+1.)+(0.0546658569873)+(0.440830144989)*0.5**o
        ref[4,0,1]=(0.00494625102884)/(o+1.)+(-0.589800937092)+(-0.836804887763)*0.5**o
        ref[4,1,0]=(0.302522420366)/(o+1.)+(0.0573597689848)+(0.204216251019)*0.5**o
        ref[4,1,1]=(1.27966467868)/(o+1.)+(-0.0054011358584)+(0.275074685897)*0.5**o
        ref[5,0,0]=(0.0381145194101)/(o+1.)+(-0.511461529678)+(0.370194889673)*0.5**o
        ref[5,0,1]=(0.716545624134)/(o+1.)+(-0.826779191491)+(-0.527801267225)*0.5**o
        ref[5,1,0]=(-0.196693995816)/(o+1.)+(-0.341449581363)+(0.11418535486)*0.5**o
        ref[5,1,1]=(-0.482729942895)/(o+1.)+(0.716447935227)+(0.455947680647)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_Solution_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunctionOnContactZero

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.415003066171)*x[0]**o+(-0.47441250281)*x[0]+(0.765391656872)*x[1]**o+(-0.170257448926)*x[1]
        arg[0,0,0,1]=(-0.928539602254)*x[0]**o+(0.792240012243)*x[0]+(-0.152730990885)*x[1]**o+(-0.667500454847)*x[1]
        arg[0,0,1,0]=(-0.937831078575)*x[0]**o+(0.963649346531)*x[0]+(0.505558859525)*x[1]**o+(-0.921354612243)*x[1]
        arg[0,0,1,1]=(0.909415074451)*x[0]**o+(0.199360871726)*x[0]+(0.194087349442)*x[1]**o+(0.875065381425)*x[1]
        arg[0,0,2,0]=(0.935190448625)*x[0]**o+(-0.109984767478)*x[0]+(0.518460161897)*x[1]**o+(-0.597684157349)*x[1]
        arg[0,0,2,1]=(-0.94062714999)*x[0]**o+(-0.308960574286)*x[0]+(0.578391007321)*x[1]**o+(0.864501040998)*x[1]
        arg[0,1,0,0]=(-0.197219678026)*x[0]**o+(-0.982899373778)*x[0]+(-0.603956105825)*x[1]**o+(-0.242717148913)*x[1]
        arg[0,1,0,1]=(-0.590715606611)*x[0]**o+(0.461185487788)*x[0]+(-0.270612249495)*x[1]**o+(-0.644742980654)*x[1]
        arg[0,1,1,0]=(-0.365516638755)*x[0]**o+(-0.255307471195)*x[0]+(0.0875989587033)*x[1]**o+(0.923189427818)*x[1]
        arg[0,1,1,1]=(0.315218558548)*x[0]**o+(-0.373070897551)*x[0]+(0.625272530385)*x[1]**o+(-0.174054543645)*x[1]
        arg[0,1,2,0]=(-0.502497884169)*x[0]**o+(-0.933785405889)*x[0]+(0.115500534905)*x[1]**o+(-0.864454092408)*x[1]
        arg[0,1,2,1]=(0.615837581258)*x[0]**o+(0.832538142367)*x[0]+(-0.0715650027129)*x[1]**o+(-0.775173801322)*x[1]
        arg[0,2,0,0]=(-0.898904577889)*x[0]**o+(-0.430072579352)*x[0]+(-0.532213103734)*x[1]**o+(0.970885652167)*x[1]
        arg[0,2,0,1]=(-0.0056483770747)*x[0]**o+(-0.566550554106)*x[0]+(0.751663691868)*x[1]**o+(0.677878661708)*x[1]
        arg[0,2,1,0]=(0.271196923039)*x[0]**o+(-0.291608730386)*x[0]+(-0.932579716748)*x[1]**o+(-0.437823650176)*x[1]
        arg[0,2,1,1]=(-0.788206773125)*x[0]**o+(0.18675055747)*x[0]+(0.747625244884)*x[1]**o+(0.654933611758)*x[1]
        arg[0,2,2,0]=(-0.787603701957)*x[0]**o+(-0.789784850865)*x[0]+(0.197684766271)*x[1]**o+(-0.988404958072)*x[1]
        arg[0,2,2,1]=(0.417538087399)*x[0]**o+(0.745832250585)*x[0]+(-0.525306115268)*x[1]**o+(0.134086803871)*x[1]
        arg[0,3,0,0]=(-0.230658480436)*x[0]**o+(-0.072983085881)*x[0]+(-0.149386157207)*x[1]**o+(-0.506870604002)*x[1]
        arg[0,3,0,1]=(-0.815859549425)*x[0]**o+(-0.962632495033)*x[0]+(-0.542257139282)*x[1]**o+(0.93598141472)*x[1]
        arg[0,3,1,0]=(0.0508974791413)*x[0]**o+(-0.722669867743)*x[0]+(0.714019527678)*x[1]**o+(0.710015283752)*x[1]
        arg[0,3,1,1]=(0.119419200166)*x[0]**o+(0.365663074152)*x[0]+(0.352455072404)*x[1]**o+(-0.0788953701144)*x[1]
        arg[0,3,2,0]=(-0.82075904747)*x[0]**o+(-0.224534176808)*x[0]+(-0.585094684368)*x[1]**o+(0.242958171035)*x[1]
        arg[0,3,2,1]=(-0.748061395205)*x[0]**o+(0.676995891547)*x[0]+(0.686644239408)*x[1]**o+(0.260350484065)*x[1]
        arg[0,4,0,0]=(0.128293903485)*x[0]**o+(0.226748407809)*x[0]+(0.129373873322)*x[1]**o+(0.988775321025)*x[1]
        arg[0,4,0,1]=(0.639638672694)*x[0]**o+(-0.817791454136)*x[0]+(-0.170677596319)*x[1]**o+(0.404161643274)*x[1]
        arg[0,4,1,0]=(0.43743097727)*x[0]**o+(-0.408230274495)*x[0]+(0.282544628333)*x[1]**o+(-0.542352697967)*x[1]
        arg[0,4,1,1]=(-0.995599965283)*x[0]**o+(-0.0512373236039)*x[0]+(0.218383977926)*x[1]**o+(0.929355335423)*x[1]
        arg[0,4,2,0]=(0.922609534479)*x[0]**o+(-0.166291574079)*x[0]+(-0.449331655188)*x[1]**o+(-0.0279347569009)*x[1]
        arg[0,4,2,1]=(0.950186933361)*x[0]**o+(-0.426615341822)*x[0]+(-0.167639663907)*x[1]**o+(0.219020502174)*x[1]
        arg[1,0,0,0]=(0.0573632292788)*x[0]**o+(0.543681035865)*x[0]+(-0.622510875807)*x[1]**o+(-0.171567467655)*x[1]
        arg[1,0,0,1]=(0.368975600444)*x[0]**o+(-0.754217532725)*x[0]+(0.469744047929)*x[1]**o+(-0.922445394777)*x[1]
        arg[1,0,1,0]=(-0.810194964715)*x[0]**o+(0.0861796117934)*x[0]+(0.625367069433)*x[1]**o+(-0.950906871025)*x[1]
        arg[1,0,1,1]=(0.744923275143)*x[0]**o+(0.465744045178)*x[0]+(0.359324627118)*x[1]**o+(0.273934205663)*x[1]
        arg[1,0,2,0]=(0.891959004941)*x[0]**o+(-0.498241249484)*x[0]+(0.318198109895)*x[1]**o+(0.0674564169104)*x[1]
        arg[1,0,2,1]=(0.842125439023)*x[0]**o+(0.731884782944)*x[0]+(-0.674805144805)*x[1]**o+(0.431782566319)*x[1]
        arg[1,1,0,0]=(-0.786719774805)*x[0]**o+(-0.55929144961)*x[0]+(-0.740263211102)*x[1]**o+(0.777914467998)*x[1]
        arg[1,1,0,1]=(0.831245977993)*x[0]**o+(-0.758991188073)*x[0]+(0.714032957488)*x[1]**o+(-0.879935556766)*x[1]
        arg[1,1,1,0]=(0.862796574462)*x[0]**o+(0.746875042565)*x[0]+(0.455949140326)*x[1]**o+(0.585491946591)*x[1]
        arg[1,1,1,1]=(-0.500409398892)*x[0]**o+(0.292753163538)*x[0]+(-0.62474682088)*x[1]**o+(0.583368504524)*x[1]
        arg[1,1,2,0]=(-0.402886961189)*x[0]**o+(-0.288733078165)*x[0]+(-0.770315341866)*x[1]**o+(-0.462764773299)*x[1]
        arg[1,1,2,1]=(0.358161520677)*x[0]**o+(-0.524052652951)*x[0]+(0.654162491323)*x[1]**o+(0.902382016998)*x[1]
        arg[1,2,0,0]=(-0.15539934232)*x[0]**o+(0.693162750731)*x[0]+(-0.0538737948754)*x[1]**o+(-0.025534990068)*x[1]
        arg[1,2,0,1]=(0.371159212282)*x[0]**o+(-0.751292369171)*x[0]+(-0.377617770006)*x[1]**o+(0.707647483842)*x[1]
        arg[1,2,1,0]=(0.117790556271)*x[0]**o+(0.848017682445)*x[0]+(-0.360936107893)*x[1]**o+(-0.564060873236)*x[1]
        arg[1,2,1,1]=(-0.524828128971)*x[0]**o+(0.737876348154)*x[0]+(0.183316445987)*x[1]**o+(-0.821207746237)*x[1]
        arg[1,2,2,0]=(0.300516177535)*x[0]**o+(-0.204187510383)*x[0]+(0.126601242737)*x[1]**o+(0.0794834423875)*x[1]
        arg[1,2,2,1]=(-0.29046926624)*x[0]**o+(-0.310057692431)*x[0]+(-0.810227948574)*x[1]**o+(-0.464128160277)*x[1]
        arg[1,3,0,0]=(-0.134275834453)*x[0]**o+(0.448577162559)*x[0]+(0.757723452446)*x[1]**o+(0.139202834725)*x[1]
        arg[1,3,0,1]=(0.882915728842)*x[0]**o+(-0.0750591947315)*x[0]+(0.503482692135)*x[1]**o+(-0.176350465386)*x[1]
        arg[1,3,1,0]=(-0.954792910945)*x[0]**o+(0.772215353622)*x[0]+(-0.861747803918)*x[1]**o+(-0.932727438775)*x[1]
        arg[1,3,1,1]=(0.747255393187)*x[0]**o+(0.0392661297355)*x[0]+(-0.322274864209)*x[1]**o+(-0.491462177972)*x[1]
        arg[1,3,2,0]=(0.139287057836)*x[0]**o+(0.312258185231)*x[0]+(-0.488288412308)*x[1]**o+(-0.185131700928)*x[1]
        arg[1,3,2,1]=(-0.861625516304)*x[0]**o+(0.681454010645)*x[0]+(0.206644380777)*x[1]**o+(-0.0344138929268)*x[1]
        arg[1,4,0,0]=(-0.714198122614)*x[0]**o+(-0.413113909739)*x[0]+(-0.971680855409)*x[1]**o+(0.602598611545)*x[1]
        arg[1,4,0,1]=(0.293578109913)*x[0]**o+(-0.495635685351)*x[0]+(0.347408878778)*x[1]**o+(-0.923814185786)*x[1]
        arg[1,4,1,0]=(0.843760126513)*x[0]**o+(-0.671075074679)*x[0]+(-0.284086603407)*x[1]**o+(0.437492233496)*x[1]
        arg[1,4,1,1]=(-0.342911357374)*x[0]**o+(-0.440232224635)*x[0]+(0.537701123084)*x[1]**o+(-0.525203729082)*x[1]
        arg[1,4,2,0]=(-0.71201360314)*x[0]**o+(-0.706332525171)*x[0]+(0.831438049852)*x[1]**o+(0.819141670055)*x[1]
        arg[1,4,2,1]=(-0.616403541928)*x[0]**o+(0.635973739636)*x[0]+(-0.0369972206543)*x[1]**o+(-0.08595692743)*x[1]
        arg[2,0,0,0]=(-0.959754423491)*x[0]**o+(-0.332473557912)*x[0]+(-0.464334304695)*x[1]**o+(0.597214020843)*x[1]
        arg[2,0,0,1]=(0.231051530608)*x[0]**o+(0.368524293555)*x[0]+(0.173044245392)*x[1]**o+(0.505867390994)*x[1]
        arg[2,0,1,0]=(0.376577055194)*x[0]**o+(-0.739990982509)*x[0]+(-0.816201949845)*x[1]**o+(-0.71233193641)*x[1]
        arg[2,0,1,1]=(-0.4891837531)*x[0]**o+(-0.121562226208)*x[0]+(-0.876294548464)*x[1]**o+(0.377763924093)*x[1]
        arg[2,0,2,0]=(-0.759548081841)*x[0]**o+(-0.118749434442)*x[0]+(0.602375475355)*x[1]**o+(-0.877213577079)*x[1]
        arg[2,0,2,1]=(0.114018259802)*x[0]**o+(-0.627149204501)*x[0]+(0.813648191424)*x[1]**o+(-0.612548616323)*x[1]
        arg[2,1,0,0]=(-0.572699800518)*x[0]**o+(0.507715829662)*x[0]+(-0.12122456959)*x[1]**o+(0.75522878228)*x[1]
        arg[2,1,0,1]=(0.0655268452799)*x[0]**o+(0.211191572823)*x[0]+(0.539760655497)*x[1]**o+(0.657397992057)*x[1]
        arg[2,1,1,0]=(0.358281119336)*x[0]**o+(0.119364032507)*x[0]+(-0.0242249415295)*x[1]**o+(-0.645986730609)*x[1]
        arg[2,1,1,1]=(0.523751704889)*x[0]**o+(-0.237670103222)*x[0]+(-0.995787126907)*x[1]**o+(-0.504129088442)*x[1]
        arg[2,1,2,0]=(-0.185380508646)*x[0]**o+(-0.218341933583)*x[0]+(0.149400154478)*x[1]**o+(0.00252616040715)*x[1]
        arg[2,1,2,1]=(0.81086932122)*x[0]**o+(0.387271527481)*x[0]+(0.908525646396)*x[1]**o+(-0.75383394204)*x[1]
        arg[2,2,0,0]=(-0.463416100256)*x[0]**o+(-0.891654021409)*x[0]+(-0.0495088159461)*x[1]**o+(0.398915277911)*x[1]
        arg[2,2,0,1]=(-0.200698149193)*x[0]**o+(0.127899540753)*x[0]+(-0.895280577411)*x[1]**o+(-0.197025422242)*x[1]
        arg[2,2,1,0]=(0.888683772867)*x[0]**o+(-0.84436563509)*x[0]+(-0.375709672588)*x[1]**o+(0.591336312818)*x[1]
        arg[2,2,1,1]=(-0.0167949020901)*x[0]**o+(0.145017501865)*x[0]+(0.868194479789)*x[1]**o+(-0.62439676328)*x[1]
        arg[2,2,2,0]=(0.441981698578)*x[0]**o+(-0.481156166596)*x[0]+(0.267937553127)*x[1]**o+(-0.37032246733)*x[1]
        arg[2,2,2,1]=(0.876701424579)*x[0]**o+(0.473798212378)*x[0]+(0.697645775876)*x[1]**o+(-0.798143376838)*x[1]
        arg[2,3,0,0]=(0.111231509217)*x[0]**o+(0.188121806171)*x[0]+(-0.0706155476337)*x[1]**o+(0.779423336221)*x[1]
        arg[2,3,0,1]=(-0.858913402812)*x[0]**o+(0.655459361097)*x[0]+(0.326282512878)*x[1]**o+(0.863386952614)*x[1]
        arg[2,3,1,0]=(-0.524551221953)*x[0]**o+(-0.807242415277)*x[0]+(0.575866251831)*x[1]**o+(-0.74503615846)*x[1]
        arg[2,3,1,1]=(0.391514075129)*x[0]**o+(0.477257809825)*x[0]+(-0.488263373008)*x[1]**o+(-0.29101442481)*x[1]
        arg[2,3,2,0]=(0.26532129161)*x[0]**o+(0.22794872765)*x[0]+(0.85094548724)*x[1]**o+(-0.369690624817)*x[1]
        arg[2,3,2,1]=(0.2339994493)*x[0]**o+(-0.901432708956)*x[0]+(0.309349741876)*x[1]**o+(0.132010344317)*x[1]
        arg[2,4,0,0]=(-0.713553783036)*x[0]**o+(0.791484993301)*x[0]+(0.339247480459)*x[1]**o+(0.158842307171)*x[1]
        arg[2,4,0,1]=(-0.733490022564)*x[0]**o+(0.631284830844)*x[0]+(0.340009298488)*x[1]**o+(-0.483162310574)*x[1]
        arg[2,4,1,0]=(0.709413437683)*x[0]**o+(0.896590124659)*x[0]+(0.680862759763)*x[1]**o+(-0.23579845663)*x[1]
        arg[2,4,1,1]=(-0.130761551034)*x[0]**o+(0.269095282394)*x[0]+(0.687201138134)*x[1]**o+(0.905183436376)*x[1]
        arg[2,4,2,0]=(0.256894911335)*x[0]**o+(0.706744915172)*x[0]+(0.910504833397)*x[1]**o+(-0.285088297895)*x[1]
        arg[2,4,2,1]=(0.95137247728)*x[0]**o+(-0.924669436646)*x[0]+(-0.146356272875)*x[1]**o+(-0.673326194162)*x[1]
        arg[3,0,0,0]=(0.663180444534)*x[0]**o+(0.894728781279)*x[0]+(0.894383747285)*x[1]**o+(0.969779862148)*x[1]
        arg[3,0,0,1]=(0.424556283891)*x[0]**o+(-0.97849198275)*x[0]+(0.367908872519)*x[1]**o+(0.624098355956)*x[1]
        arg[3,0,1,0]=(0.908698580251)*x[0]**o+(-0.899102004477)*x[0]+(0.509310001763)*x[1]**o+(-0.8399697646)*x[1]
        arg[3,0,1,1]=(0.523444225814)*x[0]**o+(0.410650511976)*x[0]+(-0.986699126405)*x[1]**o+(0.081494682537)*x[1]
        arg[3,0,2,0]=(-0.163604985314)*x[0]**o+(0.959338619715)*x[0]+(-0.843900271478)*x[1]**o+(-0.0218386544947)*x[1]
        arg[3,0,2,1]=(-0.490804271432)*x[0]**o+(0.123332672938)*x[0]+(0.902072835174)*x[1]**o+(-0.202340709957)*x[1]
        arg[3,1,0,0]=(-0.949731382685)*x[0]**o+(0.533695608969)*x[0]+(0.430512336347)*x[1]**o+(0.977434372776)*x[1]
        arg[3,1,0,1]=(0.945102298088)*x[0]**o+(-0.117807763531)*x[0]+(-0.203921504027)*x[1]**o+(-0.261266079346)*x[1]
        arg[3,1,1,0]=(0.762442770795)*x[0]**o+(-0.318456087863)*x[0]+(-0.365021977395)*x[1]**o+(0.0516932816311)*x[1]
        arg[3,1,1,1]=(-0.829061607399)*x[0]**o+(0.854900781365)*x[0]+(-0.397139314214)*x[1]**o+(-0.549679351555)*x[1]
        arg[3,1,2,0]=(-0.780040060687)*x[0]**o+(0.719393460042)*x[0]+(-0.435377419129)*x[1]**o+(0.655962063649)*x[1]
        arg[3,1,2,1]=(-0.853144390228)*x[0]**o+(0.0940222442927)*x[0]+(-0.642258121264)*x[1]**o+(-0.615386923437)*x[1]
        arg[3,2,0,0]=(0.377161814597)*x[0]**o+(0.349238572634)*x[0]+(0.361300611554)*x[1]**o+(0.554382639639)*x[1]
        arg[3,2,0,1]=(-0.51174671413)*x[0]**o+(-0.0586248632463)*x[0]+(0.083224329707)*x[1]**o+(0.506614699443)*x[1]
        arg[3,2,1,0]=(0.612695079311)*x[0]**o+(0.82053876378)*x[0]+(-0.132608876197)*x[1]**o+(0.701720563108)*x[1]
        arg[3,2,1,1]=(0.0928734170244)*x[0]**o+(-0.127564289866)*x[0]+(0.68468196965)*x[1]**o+(0.0168958305852)*x[1]
        arg[3,2,2,0]=(-0.863542893122)*x[0]**o+(0.0533007397457)*x[0]+(-0.0602194449612)*x[1]**o+(0.391122261927)*x[1]
        arg[3,2,2,1]=(-0.0942537498633)*x[0]**o+(-0.421741120475)*x[0]+(-0.0746938765947)*x[1]**o+(0.988111802567)*x[1]
        arg[3,3,0,0]=(-0.0588913579205)*x[0]**o+(0.439330145374)*x[0]+(-0.541714868844)*x[1]**o+(0.845344272279)*x[1]
        arg[3,3,0,1]=(0.221784155922)*x[0]**o+(0.983539923382)*x[0]+(-0.32332836636)*x[1]**o+(-0.892579743718)*x[1]
        arg[3,3,1,0]=(0.804097639203)*x[0]**o+(-0.342072604164)*x[0]+(0.755681096012)*x[1]**o+(0.432897246925)*x[1]
        arg[3,3,1,1]=(0.825696602707)*x[0]**o+(0.536999112965)*x[0]+(0.745033643035)*x[1]**o+(0.156806358776)*x[1]
        arg[3,3,2,0]=(-0.372114391352)*x[0]**o+(0.670417556114)*x[0]+(-0.569395510546)*x[1]**o+(-0.523784408534)*x[1]
        arg[3,3,2,1]=(0.439723359721)*x[0]**o+(0.809733189174)*x[0]+(0.781753556415)*x[1]**o+(0.375991547044)*x[1]
        arg[3,4,0,0]=(-0.429896412297)*x[0]**o+(-0.908820042346)*x[0]+(0.328787484151)*x[1]**o+(-0.610488251755)*x[1]
        arg[3,4,0,1]=(0.803567111353)*x[0]**o+(-0.487219379817)*x[0]+(-0.541538468561)*x[1]**o+(-0.692964506777)*x[1]
        arg[3,4,1,0]=(-0.963266835039)*x[0]**o+(0.667510333677)*x[0]+(0.308312799035)*x[1]**o+(0.686605377458)*x[1]
        arg[3,4,1,1]=(0.651198604373)*x[0]**o+(0.306465142929)*x[0]+(-0.923691546947)*x[1]**o+(-0.376240035765)*x[1]
        arg[3,4,2,0]=(-0.339406539705)*x[0]**o+(0.524709759622)*x[0]+(-0.0326564953763)*x[1]**o+(-0.658296780229)*x[1]
        arg[3,4,2,1]=(0.471355209305)*x[0]**o+(-0.473756698531)*x[0]+(0.193890658409)*x[1]**o+(-0.266632530307)*x[1]
        ref[0,0,0,0]=(0.765391656872)/(o+1.)+(-0.322334975868)+(0.415003066171)*0.5**o
        ref[0,0,0,1]=(-0.152730990885)/(o+1.)+(0.0623697786981)+(-0.928539602254)*0.5**o
        ref[0,0,1,0]=(0.505558859525)/(o+1.)+(0.0211473671443)+(-0.937831078575)*0.5**o
        ref[0,0,1,1]=(0.194087349442)/(o+1.)+(0.537213126576)+(0.909415074451)*0.5**o
        ref[0,0,2,0]=(0.518460161897)/(o+1.)+(-0.353834462413)+(0.935190448625)*0.5**o
        ref[0,0,2,1]=(0.578391007321)/(o+1.)+(0.277770233356)+(-0.94062714999)*0.5**o
        ref[0,1,0,0]=(-0.603956105825)/(o+1.)+(-0.612808261345)+(-0.197219678026)*0.5**o
        ref[0,1,0,1]=(-0.270612249495)/(o+1.)+(-0.0917787464331)+(-0.590715606611)*0.5**o
        ref[0,1,1,0]=(0.0875989587033)/(o+1.)+(0.333940978312)+(-0.365516638755)*0.5**o
        ref[0,1,1,1]=(0.625272530385)/(o+1.)+(-0.273562720598)+(0.315218558548)*0.5**o
        ref[0,1,2,0]=(0.115500534905)/(o+1.)+(-0.899119749149)+(-0.502497884169)*0.5**o
        ref[0,1,2,1]=(-0.0715650027129)/(o+1.)+(0.0286821705227)+(0.615837581258)*0.5**o
        ref[0,2,0,0]=(-0.532213103734)/(o+1.)+(0.270406536408)+(-0.898904577889)*0.5**o
        ref[0,2,0,1]=(0.751663691868)/(o+1.)+(0.0556640538013)+(-0.0056483770747)*0.5**o
        ref[0,2,1,0]=(-0.932579716748)/(o+1.)+(-0.364716190281)+(0.271196923039)*0.5**o
        ref[0,2,1,1]=(0.747625244884)/(o+1.)+(0.420842084614)+(-0.788206773125)*0.5**o
        ref[0,2,2,0]=(0.197684766271)/(o+1.)+(-0.889094904468)+(-0.787603701957)*0.5**o
        ref[0,2,2,1]=(-0.525306115268)/(o+1.)+(0.439959527228)+(0.417538087399)*0.5**o
        ref[0,3,0,0]=(-0.149386157207)/(o+1.)+(-0.289926844941)+(-0.230658480436)*0.5**o
        ref[0,3,0,1]=(-0.542257139282)/(o+1.)+(-0.0133255401565)+(-0.815859549425)*0.5**o
        ref[0,3,1,0]=(0.714019527678)/(o+1.)+(-0.00632729199566)+(0.0508974791413)*0.5**o
        ref[0,3,1,1]=(0.352455072404)/(o+1.)+(0.143383852019)+(0.119419200166)*0.5**o
        ref[0,3,2,0]=(-0.585094684368)/(o+1.)+(0.00921199711359)+(-0.82075904747)*0.5**o
        ref[0,3,2,1]=(0.686644239408)/(o+1.)+(0.468673187806)+(-0.748061395205)*0.5**o
        ref[0,4,0,0]=(0.129373873322)/(o+1.)+(0.607761864417)+(0.128293903485)*0.5**o
        ref[0,4,0,1]=(-0.170677596319)/(o+1.)+(-0.206814905431)+(0.639638672694)*0.5**o
        ref[0,4,1,0]=(0.282544628333)/(o+1.)+(-0.475291486231)+(0.43743097727)*0.5**o
        ref[0,4,1,1]=(0.218383977926)/(o+1.)+(0.43905900591)+(-0.995599965283)*0.5**o
        ref[0,4,2,0]=(-0.449331655188)/(o+1.)+(-0.0971131654902)+(0.922609534479)*0.5**o
        ref[0,4,2,1]=(-0.167639663907)/(o+1.)+(-0.103797419824)+(0.950186933361)*0.5**o
        ref[1,0,0,0]=(-0.622510875807)/(o+1.)+(0.186056784105)+(0.0573632292788)*0.5**o
        ref[1,0,0,1]=(0.469744047929)/(o+1.)+(-0.838331463751)+(0.368975600444)*0.5**o
        ref[1,0,1,0]=(0.625367069433)/(o+1.)+(-0.432363629616)+(-0.810194964715)*0.5**o
        ref[1,0,1,1]=(0.359324627118)/(o+1.)+(0.36983912542)+(0.744923275143)*0.5**o
        ref[1,0,2,0]=(0.318198109895)/(o+1.)+(-0.215392416287)+(0.891959004941)*0.5**o
        ref[1,0,2,1]=(-0.674805144805)/(o+1.)+(0.581833674632)+(0.842125439023)*0.5**o
        ref[1,1,0,0]=(-0.740263211102)/(o+1.)+(0.109311509194)+(-0.786719774805)*0.5**o
        ref[1,1,0,1]=(0.714032957488)/(o+1.)+(-0.81946337242)+(0.831245977993)*0.5**o
        ref[1,1,1,0]=(0.455949140326)/(o+1.)+(0.666183494578)+(0.862796574462)*0.5**o
        ref[1,1,1,1]=(-0.62474682088)/(o+1.)+(0.438060834031)+(-0.500409398892)*0.5**o
        ref[1,1,2,0]=(-0.770315341866)/(o+1.)+(-0.375748925732)+(-0.402886961189)*0.5**o
        ref[1,1,2,1]=(0.654162491323)/(o+1.)+(0.189164682023)+(0.358161520677)*0.5**o
        ref[1,2,0,0]=(-0.0538737948754)/(o+1.)+(0.333813880332)+(-0.15539934232)*0.5**o
        ref[1,2,0,1]=(-0.377617770006)/(o+1.)+(-0.0218224426646)+(0.371159212282)*0.5**o
        ref[1,2,1,0]=(-0.360936107893)/(o+1.)+(0.141978404604)+(0.117790556271)*0.5**o
        ref[1,2,1,1]=(0.183316445987)/(o+1.)+(-0.0416656990415)+(-0.524828128971)*0.5**o
        ref[1,2,2,0]=(0.126601242737)/(o+1.)+(-0.0623520339976)+(0.300516177535)*0.5**o
        ref[1,2,2,1]=(-0.810227948574)/(o+1.)+(-0.387092926354)+(-0.29046926624)*0.5**o
        ref[1,3,0,0]=(0.757723452446)/(o+1.)+(0.293889998642)+(-0.134275834453)*0.5**o
        ref[1,3,0,1]=(0.503482692135)/(o+1.)+(-0.125704830059)+(0.882915728842)*0.5**o
        ref[1,3,1,0]=(-0.861747803918)/(o+1.)+(-0.0802560425765)+(-0.954792910945)*0.5**o
        ref[1,3,1,1]=(-0.322274864209)/(o+1.)+(-0.226098024118)+(0.747255393187)*0.5**o
        ref[1,3,2,0]=(-0.488288412308)/(o+1.)+(0.0635632421511)+(0.139287057836)*0.5**o
        ref[1,3,2,1]=(0.206644380777)/(o+1.)+(0.323520058859)+(-0.861625516304)*0.5**o
        ref[1,4,0,0]=(-0.971680855409)/(o+1.)+(0.0947423509032)+(-0.714198122614)*0.5**o
        ref[1,4,0,1]=(0.347408878778)/(o+1.)+(-0.709724935569)+(0.293578109913)*0.5**o
        ref[1,4,1,0]=(-0.284086603407)/(o+1.)+(-0.116791420591)+(0.843760126513)*0.5**o
        ref[1,4,1,1]=(0.537701123084)/(o+1.)+(-0.482717976859)+(-0.342911357374)*0.5**o
        ref[1,4,2,0]=(0.831438049852)/(o+1.)+(0.056404572442)+(-0.71201360314)*0.5**o
        ref[1,4,2,1]=(-0.0369972206543)/(o+1.)+(0.275008406103)+(-0.616403541928)*0.5**o
        ref[2,0,0,0]=(-0.464334304695)/(o+1.)+(0.132370231466)+(-0.959754423491)*0.5**o
        ref[2,0,0,1]=(0.173044245392)/(o+1.)+(0.437195842274)+(0.231051530608)*0.5**o
        ref[2,0,1,0]=(-0.816201949845)/(o+1.)+(-0.72616145946)+(0.376577055194)*0.5**o
        ref[2,0,1,1]=(-0.876294548464)/(o+1.)+(0.128100848943)+(-0.4891837531)*0.5**o
        ref[2,0,2,0]=(0.602375475355)/(o+1.)+(-0.497981505761)+(-0.759548081841)*0.5**o
        ref[2,0,2,1]=(0.813648191424)/(o+1.)+(-0.619848910412)+(0.114018259802)*0.5**o
        ref[2,1,0,0]=(-0.12122456959)/(o+1.)+(0.631472305971)+(-0.572699800518)*0.5**o
        ref[2,1,0,1]=(0.539760655497)/(o+1.)+(0.43429478244)+(0.0655268452799)*0.5**o
        ref[2,1,1,0]=(-0.0242249415295)/(o+1.)+(-0.263311349051)+(0.358281119336)*0.5**o
        ref[2,1,1,1]=(-0.995787126907)/(o+1.)+(-0.370899595832)+(0.523751704889)*0.5**o
        ref[2,1,2,0]=(0.149400154478)/(o+1.)+(-0.107907886588)+(-0.185380508646)*0.5**o
        ref[2,1,2,1]=(0.908525646396)/(o+1.)+(-0.18328120728)+(0.81086932122)*0.5**o
        ref[2,2,0,0]=(-0.0495088159461)/(o+1.)+(-0.246369371749)+(-0.463416100256)*0.5**o
        ref[2,2,0,1]=(-0.895280577411)/(o+1.)+(-0.0345629407446)+(-0.200698149193)*0.5**o
        ref[2,2,1,0]=(-0.375709672588)/(o+1.)+(-0.126514661136)+(0.888683772867)*0.5**o
        ref[2,2,1,1]=(0.868194479789)/(o+1.)+(-0.239689630708)+(-0.0167949020901)*0.5**o
        ref[2,2,2,0]=(0.267937553127)/(o+1.)+(-0.425739316963)+(0.441981698578)*0.5**o
        ref[2,2,2,1]=(0.697645775876)/(o+1.)+(-0.16217258223)+(0.876701424579)*0.5**o
        ref[2,3,0,0]=(-0.0706155476337)/(o+1.)+(0.483772571196)+(0.111231509217)*0.5**o
        ref[2,3,0,1]=(0.326282512878)/(o+1.)+(0.759423156855)+(-0.858913402812)*0.5**o
        ref[2,3,1,0]=(0.575866251831)/(o+1.)+(-0.776139286869)+(-0.524551221953)*0.5**o
        ref[2,3,1,1]=(-0.488263373008)/(o+1.)+(0.0931216925076)+(0.391514075129)*0.5**o
        ref[2,3,2,0]=(0.85094548724)/(o+1.)+(-0.0708709485834)+(0.26532129161)*0.5**o
        ref[2,3,2,1]=(0.309349741876)/(o+1.)+(-0.384711182319)+(0.2339994493)*0.5**o
        ref[2,4,0,0]=(0.339247480459)/(o+1.)+(0.475163650236)+(-0.713553783036)*0.5**o
        ref[2,4,0,1]=(0.340009298488)/(o+1.)+(0.0740612601349)+(-0.733490022564)*0.5**o
        ref[2,4,1,0]=(0.680862759763)/(o+1.)+(0.330395834014)+(0.709413437683)*0.5**o
        ref[2,4,1,1]=(0.687201138134)/(o+1.)+(0.587139359385)+(-0.130761551034)*0.5**o
        ref[2,4,2,0]=(0.910504833397)/(o+1.)+(0.210828308638)+(0.256894911335)*0.5**o
        ref[2,4,2,1]=(-0.146356272875)/(o+1.)+(-0.798997815404)+(0.95137247728)*0.5**o
        ref[3,0,0,0]=(0.894383747285)/(o+1.)+(0.932254321713)+(0.663180444534)*0.5**o
        ref[3,0,0,1]=(0.367908872519)/(o+1.)+(-0.177196813397)+(0.424556283891)*0.5**o
        ref[3,0,1,0]=(0.509310001763)/(o+1.)+(-0.869535884539)+(0.908698580251)*0.5**o
        ref[3,0,1,1]=(-0.986699126405)/(o+1.)+(0.246072597257)+(0.523444225814)*0.5**o
        ref[3,0,2,0]=(-0.843900271478)/(o+1.)+(0.46874998261)+(-0.163604985314)*0.5**o
        ref[3,0,2,1]=(0.902072835174)/(o+1.)+(-0.0395040185096)+(-0.490804271432)*0.5**o
        ref[3,1,0,0]=(0.430512336347)/(o+1.)+(0.755564990872)+(-0.949731382685)*0.5**o
        ref[3,1,0,1]=(-0.203921504027)/(o+1.)+(-0.189536921439)+(0.945102298088)*0.5**o
        ref[3,1,1,0]=(-0.365021977395)/(o+1.)+(-0.133381403116)+(0.762442770795)*0.5**o
        ref[3,1,1,1]=(-0.397139314214)/(o+1.)+(0.152610714905)+(-0.829061607399)*0.5**o
        ref[3,1,2,0]=(-0.435377419129)/(o+1.)+(0.687677761846)+(-0.780040060687)*0.5**o
        ref[3,1,2,1]=(-0.642258121264)/(o+1.)+(-0.260682339572)+(-0.853144390228)*0.5**o
        ref[3,2,0,0]=(0.361300611554)/(o+1.)+(0.451810606136)+(0.377161814597)*0.5**o
        ref[3,2,0,1]=(0.083224329707)/(o+1.)+(0.223994918098)+(-0.51174671413)*0.5**o
        ref[3,2,1,0]=(-0.132608876197)/(o+1.)+(0.761129663444)+(0.612695079311)*0.5**o
        ref[3,2,1,1]=(0.68468196965)/(o+1.)+(-0.0553342296404)+(0.0928734170244)*0.5**o
        ref[3,2,2,0]=(-0.0602194449612)/(o+1.)+(0.222211500837)+(-0.863542893122)*0.5**o
        ref[3,2,2,1]=(-0.0746938765947)/(o+1.)+(0.283185341046)+(-0.0942537498633)*0.5**o
        ref[3,3,0,0]=(-0.541714868844)/(o+1.)+(0.642337208826)+(-0.0588913579205)*0.5**o
        ref[3,3,0,1]=(-0.32332836636)/(o+1.)+(0.0454800898319)+(0.221784155922)*0.5**o
        ref[3,3,1,0]=(0.755681096012)/(o+1.)+(0.0454123213802)+(0.804097639203)*0.5**o
        ref[3,3,1,1]=(0.745033643035)/(o+1.)+(0.34690273587)+(0.825696602707)*0.5**o
        ref[3,3,2,0]=(-0.569395510546)/(o+1.)+(0.0733165737898)+(-0.372114391352)*0.5**o
        ref[3,3,2,1]=(0.781753556415)/(o+1.)+(0.592862368109)+(0.439723359721)*0.5**o
        ref[3,4,0,0]=(0.328787484151)/(o+1.)+(-0.75965414705)+(-0.429896412297)*0.5**o
        ref[3,4,0,1]=(-0.541538468561)/(o+1.)+(-0.590091943297)+(0.803567111353)*0.5**o
        ref[3,4,1,0]=(0.308312799035)/(o+1.)+(0.677057855567)+(-0.963266835039)*0.5**o
        ref[3,4,1,1]=(-0.923691546947)/(o+1.)+(-0.0348874464182)+(0.651198604373)*0.5**o
        ref[3,4,2,0]=(-0.0326564953763)/(o+1.)+(-0.0667935103036)+(-0.339406539705)*0.5**o
        ref[3,4,2,1]=(0.193890658409)/(o+1.)+(-0.370194614419)+(0.471355209305)*0.5**o
      else:
        arg[0,0,0,0]=(0.0737627367843)*x[0]**o+(-0.897508234376)*x[0]+(0.499361502035)*x[1]**o+(-0.830793706142)*x[1]+(0.967672023686)*x[2]**o+(-0.94209897356)*x[2]
        arg[0,0,0,1]=(0.353968533247)*x[0]**o+(-0.828575104081)*x[0]+(0.111197799977)*x[1]**o+(0.475627052455)*x[1]+(0.10233739786)*x[2]**o+(-0.297090205985)*x[2]
        arg[0,0,1,0]=(0.315704665816)*x[0]**o+(0.450577419574)*x[0]+(-0.0225047225996)*x[1]**o+(0.221436098936)*x[1]+(-0.210708820992)*x[2]**o+(0.22813686037)*x[2]
        arg[0,0,1,1]=(-0.453912049362)*x[0]**o+(0.443266972459)*x[0]+(-0.166003491301)*x[1]**o+(0.325157034345)*x[1]+(0.975435329063)*x[2]**o+(0.516477372955)*x[2]
        arg[0,0,2,0]=(0.705225432895)*x[0]**o+(0.406870556161)*x[0]+(0.967131048725)*x[1]**o+(-0.31111693058)*x[1]+(0.915125425876)*x[2]**o+(-0.417407852906)*x[2]
        arg[0,0,2,1]=(-0.397162258761)*x[0]**o+(-0.49079016751)*x[0]+(-0.444905720899)*x[1]**o+(0.258998080457)*x[1]+(-0.406172174923)*x[2]**o+(0.166620074269)*x[2]
        arg[0,1,0,0]=(0.00444300957662)*x[0]**o+(0.59448685984)*x[0]+(-0.755596203939)*x[1]**o+(-0.587866578467)*x[1]+(-0.994620629569)*x[2]**o+(0.558539124849)*x[2]
        arg[0,1,0,1]=(0.726535641072)*x[0]**o+(-0.515254556008)*x[0]+(-0.737095044797)*x[1]**o+(0.396218854768)*x[1]+(-0.873340093434)*x[2]**o+(0.538194586759)*x[2]
        arg[0,1,1,0]=(-0.554579794515)*x[0]**o+(-0.754117891877)*x[0]+(0.242579027102)*x[1]**o+(0.368307663104)*x[1]+(0.308372932825)*x[2]**o+(-0.521440130826)*x[2]
        arg[0,1,1,1]=(0.294744137189)*x[0]**o+(0.44996935134)*x[0]+(0.118263237453)*x[1]**o+(-0.143566244715)*x[1]+(-0.927385432821)*x[2]**o+(0.912735237305)*x[2]
        arg[0,1,2,0]=(-0.30261020867)*x[0]**o+(-0.994883662031)*x[0]+(0.514662053738)*x[1]**o+(0.327770859964)*x[1]+(0.862452368458)*x[2]**o+(-0.739026842429)*x[2]
        arg[0,1,2,1]=(0.494855964972)*x[0]**o+(-0.358461686787)*x[0]+(0.406489619959)*x[1]**o+(0.417722618206)*x[1]+(0.679808195501)*x[2]**o+(-0.00502769522311)*x[2]
        arg[0,2,0,0]=(-0.914585215232)*x[0]**o+(0.305890296017)*x[0]+(-0.867826544668)*x[1]**o+(0.304580882671)*x[1]+(-0.401401956053)*x[2]**o+(0.842276423411)*x[2]
        arg[0,2,0,1]=(-0.260590397572)*x[0]**o+(0.541867335986)*x[0]+(0.0279475511153)*x[1]**o+(0.0255320376447)*x[1]+(-0.584584718877)*x[2]**o+(-0.640388032783)*x[2]
        arg[0,2,1,0]=(0.0455797513404)*x[0]**o+(-0.764984318339)*x[0]+(-0.257174640639)*x[1]**o+(0.745888292962)*x[1]+(0.964524202534)*x[2]**o+(0.435680191282)*x[2]
        arg[0,2,1,1]=(-0.300900136798)*x[0]**o+(-0.899164082043)*x[0]+(-0.63273853331)*x[1]**o+(0.56522562781)*x[1]+(-0.846482371222)*x[2]**o+(-0.909985752269)*x[2]
        arg[0,2,2,0]=(0.745711773093)*x[0]**o+(0.0627291150681)*x[0]+(-0.090340386706)*x[1]**o+(-0.0912293084958)*x[1]+(0.108613205226)*x[2]**o+(-0.28985444593)*x[2]
        arg[0,2,2,1]=(0.291340077334)*x[0]**o+(0.592856577974)*x[0]+(0.141857446535)*x[1]**o+(0.830526360843)*x[1]+(0.956840406828)*x[2]**o+(-0.717465405888)*x[2]
        arg[0,3,0,0]=(0.89242099254)*x[0]**o+(0.529348654364)*x[0]+(-0.074382350183)*x[1]**o+(0.874670648345)*x[1]+(-0.728104560168)*x[2]**o+(0.0781153963606)*x[2]
        arg[0,3,0,1]=(0.095390666498)*x[0]**o+(0.0124610436496)*x[0]+(0.278014506594)*x[1]**o+(0.923928359216)*x[1]+(-0.0905667357972)*x[2]**o+(-0.894031439498)*x[2]
        arg[0,3,1,0]=(-0.317750910722)*x[0]**o+(0.677406387747)*x[0]+(0.550094997541)*x[1]**o+(-0.30098732949)*x[1]+(0.501753868155)*x[2]**o+(-0.42859127752)*x[2]
        arg[0,3,1,1]=(-0.204455448392)*x[0]**o+(0.445237113438)*x[0]+(0.853596204887)*x[1]**o+(0.315701645178)*x[1]+(0.307192864939)*x[2]**o+(-0.708325809173)*x[2]
        arg[0,3,2,0]=(0.38581656579)*x[0]**o+(-0.79289714243)*x[0]+(-0.412226202311)*x[1]**o+(0.655534112291)*x[1]+(0.31352728014)*x[2]**o+(0.502089385396)*x[2]
        arg[0,3,2,1]=(-0.225571343976)*x[0]**o+(0.687765563016)*x[0]+(0.0421378888728)*x[1]**o+(0.54466301835)*x[1]+(-0.430288255786)*x[2]**o+(-0.117571440224)*x[2]
        arg[0,4,0,0]=(0.834083057621)*x[0]**o+(0.0292441566815)*x[0]+(-0.947154369416)*x[1]**o+(0.597153808995)*x[1]+(0.323706009194)*x[2]**o+(0.664093182644)*x[2]
        arg[0,4,0,1]=(-0.613152578007)*x[0]**o+(0.253155875339)*x[0]+(0.297138228345)*x[1]**o+(-0.604393247773)*x[1]+(-0.936718722281)*x[2]**o+(-0.410140848312)*x[2]
        arg[0,4,1,0]=(-0.373668614153)*x[0]**o+(-0.401240526097)*x[0]+(-0.417458139032)*x[1]**o+(0.456033864594)*x[1]+(-0.158434012225)*x[2]**o+(-0.33938136039)*x[2]
        arg[0,4,1,1]=(0.00943802199128)*x[0]**o+(-0.800756064115)*x[0]+(0.899199671858)*x[1]**o+(0.376830611387)*x[1]+(-0.930822199046)*x[2]**o+(0.192235807799)*x[2]
        arg[0,4,2,0]=(0.157099024354)*x[0]**o+(0.195942080381)*x[0]+(-0.554519054021)*x[1]**o+(0.871823312361)*x[1]+(-0.34803033039)*x[2]**o+(-0.646933331859)*x[2]
        arg[0,4,2,1]=(0.933762543076)*x[0]**o+(-0.169843985744)*x[0]+(0.971089176971)*x[1]**o+(-0.605176382286)*x[1]+(0.254369209427)*x[2]**o+(0.0314820745541)*x[2]
        arg[1,0,0,0]=(0.0417476479889)*x[0]**o+(-0.144293810851)*x[0]+(-0.677039420744)*x[1]**o+(-0.137097132377)*x[1]+(0.721082639859)*x[2]**o+(-0.32510577691)*x[2]
        arg[1,0,0,1]=(0.450986977446)*x[0]**o+(-0.921371106219)*x[0]+(0.536898979198)*x[1]**o+(-0.814951108186)*x[1]+(-0.733774506956)*x[2]**o+(0.938941747764)*x[2]
        arg[1,0,1,0]=(-0.23712795738)*x[0]**o+(0.450883240541)*x[0]+(0.101102533189)*x[1]**o+(-0.998253664232)*x[1]+(-0.843977488601)*x[2]**o+(-0.807025450448)*x[2]
        arg[1,0,1,1]=(-0.308721170682)*x[0]**o+(-0.546268734305)*x[0]+(0.643628083343)*x[1]**o+(0.206934468523)*x[1]+(-0.257667927436)*x[2]**o+(0.468860780891)*x[2]
        arg[1,0,2,0]=(-0.267232319593)*x[0]**o+(0.221737693042)*x[0]+(0.202218228468)*x[1]**o+(-0.560941556406)*x[1]+(0.910244160614)*x[2]**o+(0.361584784536)*x[2]
        arg[1,0,2,1]=(0.405254374432)*x[0]**o+(-0.102574362257)*x[0]+(0.527874164745)*x[1]**o+(0.0318395137256)*x[1]+(-0.448050293943)*x[2]**o+(-0.6401160036)*x[2]
        arg[1,1,0,0]=(-0.877224972777)*x[0]**o+(-0.544175806369)*x[0]+(0.528037913494)*x[1]**o+(-0.477854554671)*x[1]+(-0.394875438861)*x[2]**o+(-0.704656012962)*x[2]
        arg[1,1,0,1]=(-0.873928172379)*x[0]**o+(0.273455449944)*x[0]+(0.681090480547)*x[1]**o+(0.883243766093)*x[1]+(0.557357114631)*x[2]**o+(0.586148966774)*x[2]
        arg[1,1,1,0]=(0.552826305913)*x[0]**o+(-0.277269605193)*x[0]+(-0.681872908424)*x[1]**o+(-0.84578699459)*x[1]+(0.6543624385)*x[2]**o+(-0.355051513153)*x[2]
        arg[1,1,1,1]=(-0.71233607947)*x[0]**o+(-0.779861555179)*x[0]+(0.11496674995)*x[1]**o+(0.634901982779)*x[1]+(0.833044137166)*x[2]**o+(0.0443514408009)*x[2]
        arg[1,1,2,0]=(-0.411667274338)*x[0]**o+(0.218709061898)*x[0]+(0.132573928273)*x[1]**o+(-0.450677430962)*x[1]+(0.88420873514)*x[2]**o+(-0.618513305411)*x[2]
        arg[1,1,2,1]=(0.533005373575)*x[0]**o+(0.846343252936)*x[0]+(0.697076801609)*x[1]**o+(0.342260840742)*x[1]+(0.923380136498)*x[2]**o+(-0.727450493778)*x[2]
        arg[1,2,0,0]=(-0.637366803484)*x[0]**o+(-0.654649539562)*x[0]+(-0.443035194144)*x[1]**o+(0.4928304386)*x[1]+(0.889290653473)*x[2]**o+(0.208513517091)*x[2]
        arg[1,2,0,1]=(-0.404870452835)*x[0]**o+(-0.556431486625)*x[0]+(-0.428885935216)*x[1]**o+(0.208771213187)*x[1]+(0.706109349268)*x[2]**o+(0.726168991522)*x[2]
        arg[1,2,1,0]=(0.85574176569)*x[0]**o+(-0.371989246254)*x[0]+(0.0376283091444)*x[1]**o+(-0.164811219077)*x[1]+(0.842513532253)*x[2]**o+(-0.963454790343)*x[2]
        arg[1,2,1,1]=(0.576825533647)*x[0]**o+(0.866570909412)*x[0]+(-0.286228055595)*x[1]**o+(-0.509258889664)*x[1]+(0.358087406384)*x[2]**o+(0.2558870556)*x[2]
        arg[1,2,2,0]=(0.741411326194)*x[0]**o+(0.887444210472)*x[0]+(0.176659455323)*x[1]**o+(-0.292731470152)*x[1]+(0.179836617557)*x[2]**o+(-0.10894424483)*x[2]
        arg[1,2,2,1]=(0.386515069261)*x[0]**o+(-0.423389396097)*x[0]+(0.300384913804)*x[1]**o+(-0.569308908813)*x[1]+(0.845849686169)*x[2]**o+(-0.802483942058)*x[2]
        arg[1,3,0,0]=(0.848952965744)*x[0]**o+(0.113555873929)*x[0]+(0.132168122705)*x[1]**o+(0.464116031047)*x[1]+(-0.928746352905)*x[2]**o+(-0.176374657603)*x[2]
        arg[1,3,0,1]=(0.357870838328)*x[0]**o+(-0.294290586373)*x[0]+(0.395859596543)*x[1]**o+(-0.265635871788)*x[1]+(0.283887401402)*x[2]**o+(0.500378090749)*x[2]
        arg[1,3,1,0]=(-0.509333127149)*x[0]**o+(0.328856435768)*x[0]+(-0.377387784476)*x[1]**o+(-0.878505533565)*x[1]+(0.448760304667)*x[2]**o+(0.996692047048)*x[2]
        arg[1,3,1,1]=(-0.837885550386)*x[0]**o+(0.0608579436757)*x[0]+(-0.627132023609)*x[1]**o+(0.902538475491)*x[1]+(-0.449596037062)*x[2]**o+(-0.158786724549)*x[2]
        arg[1,3,2,0]=(0.205889276236)*x[0]**o+(0.576561726109)*x[0]+(-0.00281354319689)*x[1]**o+(0.214968600999)*x[1]+(0.811758703832)*x[2]**o+(0.260164242506)*x[2]
        arg[1,3,2,1]=(-0.331660060588)*x[0]**o+(0.713879758554)*x[0]+(0.804549392751)*x[1]**o+(-0.346397636822)*x[1]+(-0.364073503513)*x[2]**o+(0.338612348994)*x[2]
        arg[1,4,0,0]=(0.394201404247)*x[0]**o+(-0.416853395415)*x[0]+(-0.0581465874206)*x[1]**o+(0.105267182588)*x[1]+(0.791799718009)*x[2]**o+(0.0840172853234)*x[2]
        arg[1,4,0,1]=(-0.346144268453)*x[0]**o+(-0.661228785728)*x[0]+(0.326588395606)*x[1]**o+(0.766137046896)*x[1]+(0.566377202267)*x[2]**o+(0.613565461425)*x[2]
        arg[1,4,1,0]=(-0.907566140749)*x[0]**o+(-0.391612881392)*x[0]+(0.263322351365)*x[1]**o+(-0.4373853914)*x[1]+(0.248855925063)*x[2]**o+(0.973515668505)*x[2]
        arg[1,4,1,1]=(0.726194147182)*x[0]**o+(0.497593423927)*x[0]+(-0.951344923018)*x[1]**o+(0.55327321528)*x[1]+(-0.266195110749)*x[2]**o+(-0.310893832826)*x[2]
        arg[1,4,2,0]=(-0.940824546553)*x[0]**o+(-0.47838524024)*x[0]+(0.408728941942)*x[1]**o+(0.627300786998)*x[1]+(-0.0547545650101)*x[2]**o+(0.433290388525)*x[2]
        arg[1,4,2,1]=(0.728499572603)*x[0]**o+(-0.232088507059)*x[0]+(0.0550057703801)*x[1]**o+(0.159356057226)*x[1]+(0.111191390097)*x[2]**o+(-0.64498018454)*x[2]
        arg[2,0,0,0]=(0.467025642879)*x[0]**o+(-0.695379603222)*x[0]+(0.591732172092)*x[1]**o+(0.84325648142)*x[1]+(0.965696472916)*x[2]**o+(0.626196181241)*x[2]
        arg[2,0,0,1]=(-0.211129222294)*x[0]**o+(-0.501208172412)*x[0]+(-0.0162146409175)*x[1]**o+(-0.672364205306)*x[1]+(-0.851073479662)*x[2]**o+(-0.817568385627)*x[2]
        arg[2,0,1,0]=(-0.668065295797)*x[0]**o+(-0.90359857855)*x[0]+(0.040519772083)*x[1]**o+(-0.295766307198)*x[1]+(0.0491234828524)*x[2]**o+(-0.589557178985)*x[2]
        arg[2,0,1,1]=(0.928751918298)*x[0]**o+(-0.773787731767)*x[0]+(-0.744325160449)*x[1]**o+(-0.426987905682)*x[1]+(0.503339956851)*x[2]**o+(0.386756648382)*x[2]
        arg[2,0,2,0]=(-0.531818105049)*x[0]**o+(-0.586811490681)*x[0]+(0.780379352762)*x[1]**o+(-0.5575328003)*x[1]+(0.0354867101462)*x[2]**o+(0.581226069105)*x[2]
        arg[2,0,2,1]=(0.0150988200159)*x[0]**o+(0.56415563975)*x[0]+(0.540723773613)*x[1]**o+(0.0653377679637)*x[1]+(-0.52633696371)*x[2]**o+(-0.122811009405)*x[2]
        arg[2,1,0,0]=(-0.0829154151121)*x[0]**o+(-0.416192198737)*x[0]+(0.944232067474)*x[1]**o+(-0.159852455491)*x[1]+(-0.198397797188)*x[2]**o+(-0.684211447305)*x[2]
        arg[2,1,0,1]=(-0.834502772339)*x[0]**o+(-0.856817328109)*x[0]+(0.213303574315)*x[1]**o+(-0.283140791695)*x[1]+(-0.727061494352)*x[2]**o+(0.12871372765)*x[2]
        arg[2,1,1,0]=(0.247439924718)*x[0]**o+(-0.912464825903)*x[0]+(0.791177764168)*x[1]**o+(0.160047921728)*x[1]+(0.591086305304)*x[2]**o+(-0.774025941232)*x[2]
        arg[2,1,1,1]=(-0.120169534728)*x[0]**o+(-0.747254680906)*x[0]+(-0.394003570933)*x[1]**o+(-0.938198960695)*x[1]+(-0.0142776288517)*x[2]**o+(0.0627959740849)*x[2]
        arg[2,1,2,0]=(0.703841647274)*x[0]**o+(-0.946039767174)*x[0]+(-0.836717609846)*x[1]**o+(0.776573515772)*x[1]+(0.250502751027)*x[2]**o+(0.403896552902)*x[2]
        arg[2,1,2,1]=(-0.19760048131)*x[0]**o+(-0.250991189554)*x[0]+(0.278810924634)*x[1]**o+(-0.580148097047)*x[1]+(-0.197518781697)*x[2]**o+(0.116835806412)*x[2]
        arg[2,2,0,0]=(-0.507872937196)*x[0]**o+(-0.771841217512)*x[0]+(0.852666314762)*x[1]**o+(-0.482947444136)*x[1]+(0.805824662389)*x[2]**o+(-0.299255374349)*x[2]
        arg[2,2,0,1]=(-0.481403986338)*x[0]**o+(0.904322169038)*x[0]+(0.937553651477)*x[1]**o+(0.727763542606)*x[1]+(-0.69485539356)*x[2]**o+(0.547755372933)*x[2]
        arg[2,2,1,0]=(0.241675645258)*x[0]**o+(0.957681326492)*x[0]+(-0.305747218513)*x[1]**o+(-0.0393749799535)*x[1]+(-0.448892234759)*x[2]**o+(0.707429367761)*x[2]
        arg[2,2,1,1]=(0.114772627763)*x[0]**o+(-0.411147710203)*x[0]+(0.13389320737)*x[1]**o+(0.806309117051)*x[1]+(-0.572620769665)*x[2]**o+(0.0389925355751)*x[2]
        arg[2,2,2,0]=(0.887023209688)*x[0]**o+(0.158261816385)*x[0]+(0.687399231822)*x[1]**o+(-0.858491519094)*x[1]+(-0.00348150506922)*x[2]**o+(-0.863496570874)*x[2]
        arg[2,2,2,1]=(0.836731324893)*x[0]**o+(0.975472534225)*x[0]+(0.407275572226)*x[1]**o+(0.565132026147)*x[1]+(0.79110912675)*x[2]**o+(-0.0816802575667)*x[2]
        arg[2,3,0,0]=(0.196161020545)*x[0]**o+(0.254301467318)*x[0]+(0.759363127106)*x[1]**o+(0.536320544038)*x[1]+(0.522705219849)*x[2]**o+(0.376531450811)*x[2]
        arg[2,3,0,1]=(-0.862326385376)*x[0]**o+(0.766714365146)*x[0]+(0.279327620722)*x[1]**o+(-0.44416461578)*x[1]+(0.115982232563)*x[2]**o+(0.824064889151)*x[2]
        arg[2,3,1,0]=(0.571263839862)*x[0]**o+(-0.679998793927)*x[0]+(0.63548515376)*x[1]**o+(0.464171242163)*x[1]+(0.712124752567)*x[2]**o+(0.913131755622)*x[2]
        arg[2,3,1,1]=(0.272559456412)*x[0]**o+(0.0577748172306)*x[0]+(-0.610464803438)*x[1]**o+(0.233987628538)*x[1]+(0.904147555826)*x[2]**o+(-0.131245442847)*x[2]
        arg[2,3,2,0]=(0.3409363126)*x[0]**o+(0.40060221588)*x[0]+(-0.603551487822)*x[1]**o+(-0.697430353793)*x[1]+(-0.867009193249)*x[2]**o+(-0.178991284979)*x[2]
        arg[2,3,2,1]=(0.949170673623)*x[0]**o+(-0.0217541208215)*x[0]+(-0.775307373317)*x[1]**o+(0.421565612874)*x[1]+(0.419254478365)*x[2]**o+(-0.732203964657)*x[2]
        arg[2,4,0,0]=(0.876774885889)*x[0]**o+(0.441155536177)*x[0]+(-0.292113312365)*x[1]**o+(0.880590008772)*x[1]+(0.0732363738743)*x[2]**o+(-0.590855067487)*x[2]
        arg[2,4,0,1]=(0.347441329383)*x[0]**o+(-0.96566344519)*x[0]+(-0.0870932691958)*x[1]**o+(-0.793898415692)*x[1]+(0.911757004621)*x[2]**o+(0.623070769788)*x[2]
        arg[2,4,1,0]=(0.732123350436)*x[0]**o+(-0.707615856771)*x[0]+(-0.57969408421)*x[1]**o+(0.0643464469617)*x[1]+(-0.648380091597)*x[2]**o+(0.567100447405)*x[2]
        arg[2,4,1,1]=(-0.128872747083)*x[0]**o+(-0.463336399393)*x[0]+(-0.1679086419)*x[1]**o+(0.602218708561)*x[1]+(0.62788576637)*x[2]**o+(0.223548351914)*x[2]
        arg[2,4,2,0]=(0.619034818105)*x[0]**o+(0.167994824318)*x[0]+(-0.244892750219)*x[1]**o+(0.471338499791)*x[1]+(0.526950200066)*x[2]**o+(0.345611623572)*x[2]
        arg[2,4,2,1]=(0.645979236759)*x[0]**o+(0.79896884484)*x[0]+(-0.587486435644)*x[1]**o+(-0.467838428827)*x[1]+(0.672707985612)*x[2]**o+(-0.169680966884)*x[2]
        arg[3,0,0,0]=(-0.265836015694)*x[0]**o+(-0.383762854285)*x[0]+(0.420375720992)*x[1]**o+(0.739662661302)*x[1]+(-0.593559713248)*x[2]**o+(0.0743208592009)*x[2]
        arg[3,0,0,1]=(0.306246512933)*x[0]**o+(0.0357065903179)*x[0]+(0.213209244461)*x[1]**o+(-0.069836605971)*x[1]+(0.668535341179)*x[2]**o+(0.59870885392)*x[2]
        arg[3,0,1,0]=(0.509420593773)*x[0]**o+(0.995859415429)*x[0]+(-0.07734119959)*x[1]**o+(0.269006992126)*x[1]+(-0.465576902019)*x[2]**o+(0.318766487761)*x[2]
        arg[3,0,1,1]=(-0.609626065482)*x[0]**o+(0.970976299205)*x[0]+(-0.314734168176)*x[1]**o+(-0.536438958228)*x[1]+(0.900096595598)*x[2]**o+(-0.893816431641)*x[2]
        arg[3,0,2,0]=(0.269212290711)*x[0]**o+(-0.119897130167)*x[0]+(-0.937350065698)*x[1]**o+(0.913313769348)*x[1]+(0.311695335026)*x[2]**o+(0.58822688023)*x[2]
        arg[3,0,2,1]=(-0.445116541697)*x[0]**o+(-0.64630078598)*x[0]+(0.811173945666)*x[1]**o+(-0.910503175962)*x[1]+(0.160819266078)*x[2]**o+(-0.807050548571)*x[2]
        arg[3,1,0,0]=(0.25585673506)*x[0]**o+(-0.174293490388)*x[0]+(-0.510750951271)*x[1]**o+(-0.215780338673)*x[1]+(-0.531739311594)*x[2]**o+(0.479569840128)*x[2]
        arg[3,1,0,1]=(-0.159301665223)*x[0]**o+(0.88558738006)*x[0]+(-0.933255168276)*x[1]**o+(0.739621529924)*x[1]+(-0.371501556668)*x[2]**o+(-0.224043643673)*x[2]
        arg[3,1,1,0]=(0.673375237931)*x[0]**o+(0.29315421593)*x[0]+(0.378983835599)*x[1]**o+(-0.187220470447)*x[1]+(-0.122366431969)*x[2]**o+(-0.178182488821)*x[2]
        arg[3,1,1,1]=(0.273992606955)*x[0]**o+(-0.42302685869)*x[0]+(0.490650718142)*x[1]**o+(0.067822229282)*x[1]+(0.758676637186)*x[2]**o+(-0.840062442278)*x[2]
        arg[3,1,2,0]=(-0.533129977534)*x[0]**o+(0.743362501638)*x[0]+(0.0397932337285)*x[1]**o+(0.134261172942)*x[1]+(0.076467475088)*x[2]**o+(0.750124646154)*x[2]
        arg[3,1,2,1]=(0.977824314677)*x[0]**o+(0.322104772509)*x[0]+(0.478366489119)*x[1]**o+(0.139966919282)*x[1]+(0.823631645543)*x[2]**o+(0.29259439301)*x[2]
        arg[3,2,0,0]=(0.969349250234)*x[0]**o+(0.158604629065)*x[0]+(0.517280925739)*x[1]**o+(-0.32151445591)*x[1]+(0.516829269337)*x[2]**o+(0.935139450142)*x[2]
        arg[3,2,0,1]=(-0.673721580725)*x[0]**o+(0.1932869215)*x[0]+(0.47375679317)*x[1]**o+(0.377941367578)*x[1]+(-0.359306451576)*x[2]**o+(0.00498822119998)*x[2]
        arg[3,2,1,0]=(-0.432873699676)*x[0]**o+(0.191782167044)*x[0]+(0.191395777927)*x[1]**o+(-0.540088847941)*x[1]+(0.608451872364)*x[2]**o+(0.151589512705)*x[2]
        arg[3,2,1,1]=(0.633726614581)*x[0]**o+(-0.824133266602)*x[0]+(-0.380289061608)*x[1]**o+(-0.722136850408)*x[1]+(-0.807853184729)*x[2]**o+(0.152552828168)*x[2]
        arg[3,2,2,0]=(-0.182700878265)*x[0]**o+(0.105776705469)*x[0]+(0.837527799957)*x[1]**o+(0.120438395343)*x[1]+(-0.211142277471)*x[2]**o+(-0.0499206664041)*x[2]
        arg[3,2,2,1]=(0.999790796581)*x[0]**o+(0.523238189614)*x[0]+(0.108118557801)*x[1]**o+(-0.899694790145)*x[1]+(0.165089392648)*x[2]**o+(0.38599870475)*x[2]
        arg[3,3,0,0]=(-0.959143897162)*x[0]**o+(-0.968197835338)*x[0]+(-0.521866033091)*x[1]**o+(-0.560799919169)*x[1]+(0.604562365858)*x[2]**o+(0.195593261393)*x[2]
        arg[3,3,0,1]=(0.385513188376)*x[0]**o+(0.918485762628)*x[0]+(0.581126961809)*x[1]**o+(-0.365882492483)*x[1]+(-0.260563773993)*x[2]**o+(0.997016165521)*x[2]
        arg[3,3,1,0]=(0.809269330969)*x[0]**o+(0.472424328796)*x[0]+(-0.920680159926)*x[1]**o+(0.756205962059)*x[1]+(-0.886275742364)*x[2]**o+(-0.800132407145)*x[2]
        arg[3,3,1,1]=(-0.983474390608)*x[0]**o+(-0.436192301851)*x[0]+(0.653210945793)*x[1]**o+(-0.465357783291)*x[1]+(-0.799261076035)*x[2]**o+(-0.349990963152)*x[2]
        arg[3,3,2,0]=(0.710820041642)*x[0]**o+(-0.973768204693)*x[0]+(0.0408403957292)*x[1]**o+(0.61546181161)*x[1]+(0.915939389579)*x[2]**o+(0.634659938301)*x[2]
        arg[3,3,2,1]=(0.633364771689)*x[0]**o+(0.887553239682)*x[0]+(-0.64913373177)*x[1]**o+(0.557465560353)*x[1]+(-0.744072153243)*x[2]**o+(-0.301528624747)*x[2]
        arg[3,4,0,0]=(0.44039719913)*x[0]**o+(-0.0339525806656)*x[0]+(0.574166395893)*x[1]**o+(-0.810441051899)*x[1]+(0.254523432152)*x[2]**o+(-0.798403721953)*x[2]
        arg[3,4,0,1]=(0.253149910502)*x[0]**o+(-0.909846068796)*x[0]+(0.855107984234)*x[1]**o+(-0.973196168972)*x[1]+(0.646832394105)*x[2]**o+(0.811892995874)*x[2]
        arg[3,4,1,0]=(0.13027869782)*x[0]**o+(-0.424094152568)*x[0]+(-0.197473699021)*x[1]**o+(0.0795845331856)*x[1]+(-0.511301398362)*x[2]**o+(-0.331330168767)*x[2]
        arg[3,4,1,1]=(0.476554240918)*x[0]**o+(0.18800431479)*x[0]+(0.259031626216)*x[1]**o+(-0.0532533771273)*x[1]+(0.914847196458)*x[2]**o+(0.333851370508)*x[2]
        arg[3,4,2,0]=(0.800058524963)*x[0]**o+(-0.402028928896)*x[0]+(0.913881556051)*x[1]**o+(0.895516442873)*x[1]+(-0.0227700921994)*x[2]**o+(-0.0502319834692)*x[2]
        arg[3,4,2,1]=(-0.533834741079)*x[0]**o+(0.557954226232)*x[0]+(-0.121676666641)*x[1]**o+(0.781782385789)*x[1]+(0.0682870158781)*x[2]**o+(-0.236432552686)*x[2]
        ref[0,0,0,0]=(1.46703352572)/(o+1.)+(-1.33520045704)+(0.0737627367843)*0.5**o
        ref[0,0,0,1]=(0.213535197837)/(o+1.)+(-0.325019128805)+(0.353968533247)*0.5**o
        ref[0,0,1,0]=(-0.233213543591)/(o+1.)+(0.450075189439)+(0.315704665816)*0.5**o
        ref[0,0,1,1]=(0.809431837761)/(o+1.)+(0.642450689879)+(-0.453912049362)*0.5**o
        ref[0,0,2,0]=(1.8822564746)/(o+1.)+(-0.160827113662)+(0.705225432895)*0.5**o
        ref[0,0,2,1]=(-0.851077895823)/(o+1.)+(-0.032586006392)+(-0.397162258761)*0.5**o
        ref[0,1,0,0]=(-1.75021683351)/(o+1.)+(0.282579703111)+(0.00444300957662)*0.5**o
        ref[0,1,0,1]=(-1.61043513823)/(o+1.)+(0.209579442759)+(0.726535641072)*0.5**o
        ref[0,1,1,0]=(0.550951959927)/(o+1.)+(-0.453625179799)+(-0.554579794515)*0.5**o
        ref[0,1,1,1]=(-0.809122195368)/(o+1.)+(0.609569171964)+(0.294744137189)*0.5**o
        ref[0,1,2,0]=(1.3771144222)/(o+1.)+(-0.703069822248)+(-0.30261020867)*0.5**o
        ref[0,1,2,1]=(1.08629781546)/(o+1.)+(0.0271166180983)+(0.494855964972)*0.5**o
        ref[0,2,0,0]=(-1.26922850072)/(o+1.)+(0.72637380105)+(-0.914585215232)*0.5**o
        ref[0,2,0,1]=(-0.556637167762)/(o+1.)+(-0.036494329576)+(-0.260590397572)*0.5**o
        ref[0,2,1,0]=(0.707349561895)/(o+1.)+(0.208292082952)+(0.0455797513404)*0.5**o
        ref[0,2,1,1]=(-1.47922090453)/(o+1.)+(-0.621962103251)+(-0.300900136798)*0.5**o
        ref[0,2,2,0]=(0.01827281852)/(o+1.)+(-0.159177319679)+(0.745711773093)*0.5**o
        ref[0,2,2,1]=(1.09869785336)/(o+1.)+(0.352958766464)+(0.291340077334)*0.5**o
        ref[0,3,0,0]=(-0.802486910351)/(o+1.)+(0.741067349535)+(0.89242099254)*0.5**o
        ref[0,3,0,1]=(0.187447770797)/(o+1.)+(0.0211789816838)+(0.095390666498)*0.5**o
        ref[0,3,1,0]=(1.0518488657)/(o+1.)+(-0.0260861096315)+(-0.317750910722)*0.5**o
        ref[0,3,1,1]=(1.16078906983)/(o+1.)+(0.0263064747213)+(-0.204455448392)*0.5**o
        ref[0,3,2,0]=(-0.0986989221708)/(o+1.)+(0.182363177629)+(0.38581656579)*0.5**o
        ref[0,3,2,1]=(-0.388150366913)/(o+1.)+(0.557428570571)+(-0.225571343976)*0.5**o
        ref[0,4,0,0]=(-0.623448360222)/(o+1.)+(0.64524557416)+(0.834083057621)*0.5**o
        ref[0,4,0,1]=(-0.639580493935)/(o+1.)+(-0.380689110373)+(-0.613152578007)*0.5**o
        ref[0,4,1,0]=(-0.575892151257)/(o+1.)+(-0.142294010946)+(-0.373668614153)*0.5**o
        ref[0,4,1,1]=(-0.0316225271882)/(o+1.)+(-0.115844822465)+(0.00943802199128)*0.5**o
        ref[0,4,2,0]=(-0.902549384411)/(o+1.)+(0.210416030442)+(0.157099024354)*0.5**o
        ref[0,4,2,1]=(1.2254583864)/(o+1.)+(-0.371769146738)+(0.933762543076)*0.5**o
        ref[1,0,0,0]=(0.0440432191149)/(o+1.)+(-0.303248360069)+(0.0417476479889)*0.5**o
        ref[1,0,0,1]=(-0.196875527758)/(o+1.)+(-0.39869023332)+(0.450986977446)*0.5**o
        ref[1,0,1,0]=(-0.742874955412)/(o+1.)+(-0.677197937069)+(-0.23712795738)*0.5**o
        ref[1,0,1,1]=(0.385960155907)/(o+1.)+(0.0647632575546)+(-0.308721170682)*0.5**o
        ref[1,0,2,0]=(1.11246238908)/(o+1.)+(0.0111904605859)+(-0.267232319593)*0.5**o
        ref[1,0,2,1]=(0.0798238708014)/(o+1.)+(-0.355425426066)+(0.405254374432)*0.5**o
        ref[1,1,0,0]=(0.133162474633)/(o+1.)+(-0.863343187001)+(-0.877224972777)*0.5**o
        ref[1,1,0,1]=(1.23844759518)/(o+1.)+(0.871424091405)+(-0.873928172379)*0.5**o
        ref[1,1,1,0]=(-0.0275104699246)/(o+1.)+(-0.739054056468)+(0.552826305913)*0.5**o
        ref[1,1,1,1]=(0.948010887116)/(o+1.)+(-0.0503040657995)+(-0.71233607947)*0.5**o
        ref[1,1,2,0]=(1.01678266341)/(o+1.)+(-0.425240837237)+(-0.411667274338)*0.5**o
        ref[1,1,2,1]=(1.62045693811)/(o+1.)+(0.23057679995)+(0.533005373575)*0.5**o
        ref[1,2,0,0]=(0.44625545933)/(o+1.)+(0.0233472080645)+(-0.637366803484)*0.5**o
        ref[1,2,0,1]=(0.277223414052)/(o+1.)+(0.189254359042)+(-0.404870452835)*0.5**o
        ref[1,2,1,0]=(0.880141841398)/(o+1.)+(-0.750127627837)+(0.85574176569)*0.5**o
        ref[1,2,1,1]=(0.0718593507895)/(o+1.)+(0.306599537674)+(0.576825533647)*0.5**o
        ref[1,2,2,0]=(0.35649607288)/(o+1.)+(0.242884247745)+(0.741411326194)*0.5**o
        ref[1,2,2,1]=(1.14623459997)/(o+1.)+(-0.897591123484)+(0.386515069261)*0.5**o
        ref[1,3,0,0]=(-0.796578230199)/(o+1.)+(0.200648623687)+(0.848952965744)*0.5**o
        ref[1,3,0,1]=(0.679746997945)/(o+1.)+(-0.0297741837061)+(0.357870838328)*0.5**o
        ref[1,3,1,0]=(0.0713725201916)/(o+1.)+(0.223521474626)+(-0.509333127149)*0.5**o
        ref[1,3,1,1]=(-1.07672806067)/(o+1.)+(0.402304847309)+(-0.837885550386)*0.5**o
        ref[1,3,2,0]=(0.808945160636)/(o+1.)+(0.525847284807)+(0.205889276236)*0.5**o
        ref[1,3,2,1]=(0.440475889238)/(o+1.)+(0.353047235363)+(-0.331660060588)*0.5**o
        ref[1,4,0,0]=(0.733653130588)/(o+1.)+(-0.113784463751)+(0.394201404247)*0.5**o
        ref[1,4,0,1]=(0.892965597874)/(o+1.)+(0.359236861297)+(-0.346144268453)*0.5**o
        ref[1,4,1,0]=(0.512178276429)/(o+1.)+(0.0722586978563)+(-0.907566140749)*0.5**o
        ref[1,4,1,1]=(-1.21754003377)/(o+1.)+(0.369986403191)+(0.726194147182)*0.5**o
        ref[1,4,2,0]=(0.353974376932)/(o+1.)+(0.291102967642)+(-0.940824546553)*0.5**o
        ref[1,4,2,1]=(0.166197160477)/(o+1.)+(-0.358856317187)+(0.728499572603)*0.5**o
        ref[2,0,0,0]=(1.55742864501)/(o+1.)+(0.38703652972)+(0.467025642879)*0.5**o
        ref[2,0,0,1]=(-0.867288120579)/(o+1.)+(-0.995570381672)+(-0.211129222294)*0.5**o
        ref[2,0,1,0]=(0.0896432549354)/(o+1.)+(-0.894461032367)+(-0.668065295797)*0.5**o
        ref[2,0,1,1]=(-0.240985203598)/(o+1.)+(-0.407009494534)+(0.928751918298)*0.5**o
        ref[2,0,2,0]=(0.815866062908)/(o+1.)+(-0.281559110938)+(-0.531818105049)*0.5**o
        ref[2,0,2,1]=(0.0143868099026)/(o+1.)+(0.253341199154)+(0.0150988200159)*0.5**o
        ref[2,1,0,0]=(0.745834270286)/(o+1.)+(-0.630128050767)+(-0.0829154151121)*0.5**o
        ref[2,1,0,1]=(-0.513757920038)/(o+1.)+(-0.505622196077)+(-0.834502772339)*0.5**o
        ref[2,1,1,0]=(1.38226406947)/(o+1.)+(-0.763221422703)+(0.247439924718)*0.5**o
        ref[2,1,1,1]=(-0.408281199784)/(o+1.)+(-0.811328833758)+(-0.120169534728)*0.5**o
        ref[2,1,2,0]=(-0.586214858819)/(o+1.)+(0.11721515075)+(0.703841647274)*0.5**o
        ref[2,1,2,1]=(0.0812921429368)/(o+1.)+(-0.357151740094)+(-0.19760048131)*0.5**o
        ref[2,2,0,0]=(1.65849097715)/(o+1.)+(-0.777022017999)+(-0.507872937196)*0.5**o
        ref[2,2,0,1]=(0.242698257917)/(o+1.)+(1.08992054229)+(-0.481403986338)*0.5**o
        ref[2,2,1,0]=(-0.754639453272)/(o+1.)+(0.81286785715)+(0.241675645258)*0.5**o
        ref[2,2,1,1]=(-0.438727562295)/(o+1.)+(0.217076971211)+(0.114772627763)*0.5**o
        ref[2,2,2,0]=(0.683917726753)/(o+1.)+(-0.781863136791)+(0.887023209688)*0.5**o
        ref[2,2,2,1]=(1.19838469898)/(o+1.)+(0.729462151403)+(0.836731324893)*0.5**o
        ref[2,3,0,0]=(1.28206834695)/(o+1.)+(0.583576731084)+(0.196161020545)*0.5**o
        ref[2,3,0,1]=(0.395309853285)/(o+1.)+(0.573307319258)+(-0.862326385376)*0.5**o
        ref[2,3,1,0]=(1.34760990633)/(o+1.)+(0.348652101929)+(0.571263839862)*0.5**o
        ref[2,3,1,1]=(0.293682752388)/(o+1.)+(0.080258501461)+(0.272559456412)*0.5**o
        ref[2,3,2,0]=(-1.47056068107)/(o+1.)+(-0.237909711446)+(0.3409363126)*0.5**o
        ref[2,3,2,1]=(-0.356052894952)/(o+1.)+(-0.166196236302)+(0.949170673623)*0.5**o
        ref[2,4,0,0]=(-0.21887693849)/(o+1.)+(0.365445238731)+(0.876774885889)*0.5**o
        ref[2,4,0,1]=(0.824663735425)/(o+1.)+(-0.568245545547)+(0.347441329383)*0.5**o
        ref[2,4,1,0]=(-1.22807417581)/(o+1.)+(-0.0380844812022)+(0.732123350436)*0.5**o
        ref[2,4,1,1]=(0.45997712447)/(o+1.)+(0.181215330541)+(-0.128872747083)*0.5**o
        ref[2,4,2,0]=(0.282057449847)/(o+1.)+(0.49247247384)+(0.619034818105)*0.5**o
        ref[2,4,2,1]=(0.0852215499679)/(o+1.)+(0.0807247245641)+(0.645979236759)*0.5**o
        ref[3,0,0,0]=(-0.173183992256)/(o+1.)+(0.215110333109)+(-0.265836015694)*0.5**o
        ref[3,0,0,1]=(0.88174458564)/(o+1.)+(0.282289419133)+(0.306246512933)*0.5**o
        ref[3,0,1,0]=(-0.542918101609)/(o+1.)+(0.791816447658)+(0.509420593773)*0.5**o
        ref[3,0,1,1]=(0.585362427423)/(o+1.)+(-0.229639545331)+(-0.609626065482)*0.5**o
        ref[3,0,2,0]=(-0.625654730672)/(o+1.)+(0.690821759706)+(0.269212290711)*0.5**o
        ref[3,0,2,1]=(0.971993211744)/(o+1.)+(-1.18192725526)+(-0.445116541697)*0.5**o
        ref[3,1,0,0]=(-1.04249026287)/(o+1.)+(0.0447480055334)+(0.25585673506)*0.5**o
        ref[3,1,0,1]=(-1.30475672494)/(o+1.)+(0.700582633156)+(-0.159301665223)*0.5**o
        ref[3,1,1,0]=(0.25661740363)/(o+1.)+(-0.036124371669)+(0.673375237931)*0.5**o
        ref[3,1,1,1]=(1.24932735533)/(o+1.)+(-0.597633535843)+(0.273992606955)*0.5**o
        ref[3,1,2,0]=(0.116260708817)/(o+1.)+(0.813874160367)+(-0.533129977534)*0.5**o
        ref[3,1,2,1]=(1.30199813466)/(o+1.)+(0.377333042401)+(0.977824314677)*0.5**o
        ref[3,2,0,0]=(1.03411019508)/(o+1.)+(0.386114811649)+(0.969349250234)*0.5**o
        ref[3,2,0,1]=(0.114450341594)/(o+1.)+(0.288108255139)+(-0.673721580725)*0.5**o
        ref[3,2,1,0]=(0.799847650292)/(o+1.)+(-0.0983585840958)+(-0.432873699676)*0.5**o
        ref[3,2,1,1]=(-1.18814224634)/(o+1.)+(-0.696858644421)+(0.633726614581)*0.5**o
        ref[3,2,2,0]=(0.626385522485)/(o+1.)+(0.0881472172038)+(-0.182700878265)*0.5**o
        ref[3,2,2,1]=(0.27320795045)/(o+1.)+(0.0047710521094)+(0.999790796581)*0.5**o
        ref[3,3,0,0]=(0.0826963327674)/(o+1.)+(-0.666702246557)+(-0.959143897162)*0.5**o
        ref[3,3,0,1]=(0.320563187816)/(o+1.)+(0.774809717833)+(0.385513188376)*0.5**o
        ref[3,3,1,0]=(-1.80695590229)/(o+1.)+(0.214248941855)+(0.809269330969)*0.5**o
        ref[3,3,1,1]=(-0.146050130242)/(o+1.)+(-0.625770524147)+(-0.983474390608)*0.5**o
        ref[3,3,2,0]=(0.956779785308)/(o+1.)+(0.138176772609)+(0.710820041642)*0.5**o
        ref[3,3,2,1]=(-1.39320588501)/(o+1.)+(0.571745087644)+(0.633364771689)*0.5**o
        ref[3,4,0,0]=(0.828689828045)/(o+1.)+(-0.821398677259)+(0.44039719913)*0.5**o
        ref[3,4,0,1]=(1.50194037834)/(o+1.)+(-0.535574620947)+(0.253149910502)*0.5**o
        ref[3,4,1,0]=(-0.708775097383)/(o+1.)+(-0.337919894074)+(0.13027869782)*0.5**o
        ref[3,4,1,1]=(1.17387882267)/(o+1.)+(0.234301154085)+(0.476554240918)*0.5**o
        ref[3,4,2,0]=(0.891111463851)/(o+1.)+(0.221627765254)+(0.800058524963)*0.5**o
        ref[3,4,2,1]=(-0.0533896507631)/(o+1.)+(0.551652029667)+(-0.533834741079)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedSolution_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.570707549273)*x[0]+(-0.726264032186)*x[1]
        ref=(-0.0423765208873)/(o+1.)+(-0.791542656673)+(0.328490252775)*0.5**o
      else:
        arg=(1.28222892931)*x[0]+(-0.798170675576)*x[1]+(1.5701208405)*x[2]
        ref=(0.721975372868)/(o+1.)+(0.344720905832)+(0.642761909705)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedSolution_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.575166906798)*x[0]+(-1.63698773418)*x[1]
        arg[1]=(-0.0535946584636)*x[0]+(0.105969101775)*x[1]
        ref[0]=(-0.751926426344)/(o+1.)+(-0.395800138888)+(0.481705876734)*0.5**o
        ref[1]=(0.698496252242)/(o+1.)+(-0.265927161866)+(-0.114267485199)*0.5**o
      else:
        arg[0]=(-0.17879254138)*x[0]+(-1.12752123792)*x[1]+(0.921297311605)*x[2]
        arg[1]=(1.33131824618)*x[0]+(-0.392175059176)*x[1]+(0.545754281465)*x[2]
        ref[0]=(-0.494870303061)/(o+1.)+(0.0697636237654)+(-0.0296734121631)*0.5**o
        ref[1]=(-0.394300983704)/(o+1.)+(0.722261312754)+(0.434675826665)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedSolution_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.338658191734)*x[0]+(1.46127609683)*x[1]
        arg[0,1]=(-0.629116867491)*x[0]+(1.40141569066)*x[1]
        arg[0,2]=(0.105569779347)*x[0]+(0.239936324967)*x[1]
        arg[0,3]=(-0.281718331718)*x[0]+(0.280323404866)*x[1]
        arg[0,4]=(0.467341113661)*x[0]+(-0.50452062075)*x[1]
        arg[1,0]=(-0.841295219186)*x[0]+(0.524038163723)*x[1]
        arg[1,1]=(1.40143704236)*x[0]+(-1.50467961637)*x[1]
        arg[1,2]=(1.21078790505)*x[0]+(-0.731453470292)*x[1]
        arg[1,3]=(0.469556922076)*x[0]+(0.0686293204391)*x[1]
        arg[1,4]=(-1.26394637585)*x[0]+(-0.0361395276585)*x[1]
        arg[2,0]=(-1.12850883148)*x[0]+(0.284296673326)*x[1]
        arg[2,1]=(-0.626362828813)*x[0]+(1.19459827573)*x[1]
        arg[2,2]=(-0.545457293083)*x[0]+(1.547337019)*x[1]
        arg[2,3]=(0.664461986629)*x[0]+(-1.69082198266)*x[1]
        arg[2,4]=(-0.100004763305)*x[0]+(0.152159044423)*x[1]
        arg[3,0]=(0.307574843061)*x[0]+(1.20209181301)*x[1]
        arg[3,1]=(-0.0663203356627)*x[0]+(-1.26669863527)*x[1]
        arg[3,2]=(-1.20436798894)*x[0]+(-0.0375693379661)*x[1]
        arg[3,3]=(0.753649087747)*x[0]+(0.69376223486)*x[1]
        arg[3,4]=(-0.0512732264834)*x[0]+(-0.658097606436)*x[1]
        ref[0,0]=(0.996821439952)/(o+1.)+(0.015088160779)+(0.0956201435879)*0.5**o
        ref[0,1]=(0.643092963726)/(o+1.)+(0.239390724844)+(-0.349575590244)*0.5**o
        ref[0,2]=(-0.198468008977)/(o+1.)+(0.00411201743342)+(0.535750078425)*0.5**o
        ref[0,3]=(0.642491201137)/(o+1.)+(-0.623413199652)+(0.602940271315)*0.5**o
        ref[0,4]=(-0.105706120549)/(o+1.)+(-0.353560054939)+(0.775646723338)*0.5**o
        ref[1,0]=(0.144540669221)/(o+1.)+(0.0228268215803)+(-0.507451367845)*0.5**o
        ref[1,1]=(-0.820379339897)/(o+1.)+(0.0951825678509)+(0.526771630185)*0.5**o
        ref[1,2]=(-0.21214585824)/(o+1.)+(-0.0889183062985)+(0.869316905591)*0.5**o
        ref[1,3]=(0.291228940674)/(o+1.)+(0.192623928703)+(-0.138290555565)*0.5**o
        ref[1,4]=(-0.0663718956434)/(o+1.)+(-0.309484335614)+(-0.614745336642)*0.5**o
        ref[2,0]=(0.484892962201)/(o+1.)+(-0.367129859793)+(-0.594845400763)*0.5**o
        ref[2,1]=(0.768457634672)/(o+1.)+(0.154447657562)+(-0.509117502876)*0.5**o
        ref[2,2]=(0.707320338363)/(o+1.)+(-0.0592149232183)+(0.412989233992)*0.5**o
        ref[2,3]=(-0.983684339171)/(o+1.)+(-0.495377645598)+(0.948079634337)*0.5**o
        ref[2,4]=(-0.0315947220465)/(o+1.)+(0.506833986759)+(-0.929918970353)*0.5**o
        ref[3,0]=(0.54141067098)/(o+1.)+(0.489049131493)+(-0.00984227789859)*0.5**o
        ref[3,1]=(-0.552335053051)/(o+1.)+(-0.383325638803)+(-0.0140326402734)*0.5**o
        ref[3,2]=(0.0821472412575)/(o+1.)+(-0.335827719718)+(-0.652429128733)*0.5**o
        ref[3,3]=(0.112917208924)/(o+1.)+(0.667772807338)+(-0.00105150099312)*0.5**o
        ref[3,4]=(-0.19331508157)/(o+1.)+(-0.102296385708)+(-0.311462979932)*0.5**o
      else:
        arg[0,0]=(0.069245963142)*x[0]+(-0.185727379494)*x[1]+(-0.887410889699)*x[2]
        arg[0,1]=(-0.816745668949)*x[0]+(-0.250435680662)*x[1]+(0.701607830582)*x[2]
        arg[0,2]=(-0.862433954308)*x[0]+(0.731874372527)*x[1]+(-0.115384982779)*x[2]
        arg[0,3]=(-0.265247631604)*x[0]+(-0.157059741958)*x[1]+(-0.368419335078)*x[2]
        arg[0,4]=(1.66720203567)*x[0]+(-0.759393708719)*x[1]+(1.36360850704)*x[2]
        arg[1,0]=(-0.192822813795)*x[0]+(0.0358776152527)*x[1]+(-0.722864520788)*x[2]
        arg[1,1]=(-0.480606215471)*x[0]+(1.13682774573)*x[1]+(0.17900467623)*x[2]
        arg[1,2]=(0.346760852584)*x[0]+(0.994833030281)*x[1]+(0.507055933913)*x[2]
        arg[1,3]=(-0.509961853478)*x[0]+(-0.852997575505)*x[1]+(0.586527030253)*x[2]
        arg[1,4]=(0.93957975846)*x[0]+(0.28878335152)*x[1]+(0.128070931738)*x[2]
        arg[2,0]=(0.338283306313)*x[0]+(0.0244500761608)*x[1]+(-0.606875554888)*x[2]
        arg[2,1]=(0.850782624552)*x[0]+(-0.836382038189)*x[1]+(-1.41808459947)*x[2]
        arg[2,2]=(-1.06176229613)*x[0]+(-0.259247046061)*x[1]+(-0.583455484089)*x[2]
        arg[2,3]=(1.17437923282)*x[0]+(1.31485601495)*x[1]+(1.3139743231)*x[2]
        arg[2,4]=(0.362851636275)*x[0]+(0.804049783156)*x[1]+(1.09205000214)*x[2]
        arg[3,0]=(0.343603837195)*x[0]+(-1.49082257981)*x[1]+(-0.628668674545)*x[2]
        arg[3,1]=(-0.676847711765)*x[0]+(1.32350560453)*x[1]+(1.07275185686)*x[2]
        arg[3,2]=(0.570784985469)*x[0]+(-1.01212360602)*x[1]+(-0.0808776667606)*x[2]
        arg[3,3]=(0.506392444236)*x[0]+(-1.00848318041)*x[1]+(0.558777901541)*x[2]
        arg[3,4]=(-0.60925738291)*x[0]+(-0.386755957809)*x[1]+(0.60930115434)*x[2]
        ref[0,0]=(-0.695230136419)/(o+1.)+(-0.388400729083)+(0.468139288533)*0.5**o
        ref[0,1]=(0.462098132187)/(o+1.)+(-0.215041893287)+(-0.397587864642)*0.5**o
        ref[0,2]=(-0.262411385363)/(o+1.)+(0.275729226874)+(-0.534991632946)*0.5**o
        ref[0,3]=(0.117822633378)/(o+1.)+(0.0286827897592)+(-0.965914921536)*0.5**o
        ref[0,4]=(0.293575681039)/(o+1.)+(0.584039692827)+(0.809761767299)*0.5**o
        ref[1,0]=(0.627951922463)/(o+1.)+(-0.994889162557)+(0.482016683321)*0.5**o
        ref[1,1]=(1.83482238662)/(o+1.)+(-0.205719558739)+(-0.588157062647)*0.5**o
        ref[1,2]=(1.13612020398)/(o+1.)+(0.0262824938134)+(0.659964625174)*0.5**o
        ref[1,3]=(0.325264487741)/(o+1.)+(-0.0752154813413)+(-0.951265923787)*0.5**o
        ref[1,4]=(-0.149919283302)/(o+1.)+(0.77701502585)+(-0.0476767266789)*0.5**o
        ref[2,0]=(0.491504686171)/(o+1.)+(-0.562534706926)+(0.389422555265)*0.5**o
        ref[2,1]=(-1.52704217417)/(o+1.)+(0.120525042098)+(-0.117691923134)*0.5**o
        ref[2,2]=(-0.779796772543)/(o+1.)+(-0.318864997267)+(-0.486938059201)*0.5**o
        ref[2,3]=(1.53343318682)/(o+1.)+(0.731149467308)+(0.807477449443)*0.5**o
        ref[2,4]=(1.15919207694)/(o+1.)+(0.113921448815)+(0.871916446996)*0.5**o
        ref[3,0]=(-1.75312920944)/(o+1.)+(-0.153798982308)+(0.284839756895)*0.5**o
        ref[3,1]=(1.30985814569)/(o+1.)+(0.438430874237)+(-0.467310144534)*0.5**o
        ref[3,2]=(-0.737480320972)/(o+1.)+(0.0604403094397)+(0.094383414777)*0.5**o
        ref[3,3]=(-0.688723266524)/(o+1.)+(0.43486027018)+(-0.124310108473)*0.5**o
        ref[3,4]=(0.128399262674)/(o+1.)+(-0.372445844122)+(0.229780239192)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedSolution_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.993339371954)*x[0]+(-1.35308128985)*x[1]
        arg[0,0,1]=(-1.07886149134)*x[0]+(-1.38999283206)*x[1]
        arg[0,1,0]=(0.246468026972)*x[0]+(-1.37217464882)*x[1]
        arg[0,1,1]=(0.645672648599)*x[0]+(1.8070624103)*x[1]
        arg[1,0,0]=(-1.02278989388)*x[0]+(0.151634023103)*x[1]
        arg[1,0,1]=(-0.667332860541)*x[0]+(0.312742149193)*x[1]
        arg[1,1,0]=(-0.848647385832)*x[0]+(0.466438038516)*x[1]
        arg[1,1,1]=(-0.185495911372)*x[0]+(-0.137118786891)*x[1]
        arg[2,0,0]=(-0.0250555870731)*x[0]+(0.905553573283)*x[1]
        arg[2,0,1]=(-1.03940852939)*x[0]+(-1.57326016394)*x[1]
        arg[2,1,0]=(1.48834718811)*x[0]+(-0.340889253246)*x[1]
        arg[2,1,1]=(0.875721514739)*x[0]+(-0.310857412212)*x[1]
        arg[3,0,0]=(1.25129508831)*x[0]+(-1.69312043525)*x[1]
        arg[3,0,1]=(1.50124675312)*x[0]+(0.244195748654)*x[1]
        arg[3,1,0]=(0.292060478909)*x[0]+(-0.660005422696)*x[1]
        arg[3,1,1]=(0.791335231308)*x[0]+(0.466481573591)*x[1]
        arg[4,0,0]=(-0.361241947236)*x[0]+(-1.43591648587)*x[1]
        arg[4,0,1]=(-1.10385456695)*x[0]+(-0.325314195825)*x[1]
        arg[4,1,0]=(1.01004173033)*x[0]+(0.426824552857)*x[1]
        arg[4,1,1]=(-0.631382479612)*x[0]+(0.0603792190966)*x[1]
        arg[5,0,0]=(-0.30404368786)*x[0]+(-0.0102498522932)*x[1]
        arg[5,0,1]=(0.109188848746)*x[0]+(-1.16846160105)*x[1]
        arg[5,1,0]=(-0.351916879547)*x[0]+(-0.320714795448)*x[1]
        arg[5,1,1]=(-0.534138562652)*x[0]+(-0.256013589844)*x[1]
        ref[0,0,0]=(-0.506159375456)/(o+1.)+(-0.522895228783)+(-0.794470828783)*0.5**o
        ref[0,0,1]=(-0.930984367068)/(o+1.)+(-0.636851258685)+(-0.264167438968)*0.5**o
        ref[0,1,0]=(-0.674222792172)/(o+1.)+(-0.52131379048)+(0.591143751282)*0.5**o
        ref[0,1,1]=(0.953370751283)/(o+1.)+(0.377773224645)+(0.743817858326)*0.5**o
        ref[1,0,0]=(0.700508215428)/(o+1.)+(-0.303343530867)+(-0.964977024471)*0.5**o
        ref[1,0,1]=(-0.408185116452)/(o+1.)+(0.0668983569743)+(-0.0802023088445)*0.5**o
        ref[1,1,0]=(0.80742345527)/(o+1.)+(-0.405780705683)+(-0.37807139122)*0.5**o
        ref[1,1,1]=(0.0184615568095)/(o+1.)+(0.221921562266)+(-0.784919379605)*0.5**o
        ref[2,0,0]=(0.571906256883)/(o+1.)+(-0.0415588382133)+(0.391709405753)*0.5**o
        ref[2,0,1]=(-0.778176009451)/(o+1.)+(-0.761680990566)+(-0.311130702752)*0.5**o
        ref[2,1,0]=(0.498214686321)/(o+1.)+(-0.146173705483)+(0.941590659509)*0.5**o
        ref[2,1,1]=(-0.273747071818)/(o+1.)+(-0.0319184747153)+(0.902448123776)*0.5**o
        ref[3,0,0]=(-0.730029471178)/(o+1.)+(-0.288672747933)+(0.865549620109)*0.5**o
        ref[3,0,1]=(-0.371409186475)/(o+1.)+(0.623169375401)+(0.870512937445)*0.5**o
        ref[3,1,0]=(0.0611454391867)/(o+1.)+(-0.627723874214)+(0.826357365454)*0.5**o
        ref[3,1,1]=(0.785553447935)/(o+1.)+(0.246940213405)+(-0.021617069845)*0.5**o
        ref[4,0,0]=(-0.936201955907)/(o+1.)+(-0.484823720021)+(0.108690962842)*0.5**o
        ref[4,0,1]=(-0.277830390815)/(o+1.)+(-0.151786547071)+(-0.847765277819)*0.5**o
        ref[4,1,0]=(0.12293099533)/(o+1.)+(0.295620382054)+(0.722694523749)*0.5**o
        ref[4,1,1]=(-0.5517019185)/(o+1.)+(0.0850748937088)+(-0.189451129433)*0.5**o
        ref[5,0,0]=(-0.942533148805)/(o+1.)+(0.425680504872)+(-0.223121401092)*0.5**o
        ref[5,0,1]=(-0.377134228332)/(o+1.)+(-0.721019409569)+(0.759900295162)*0.5**o
        ref[5,1,0]=(-0.643828391942)/(o+1.)+(0.399974456311)+(-0.828752195674)*0.5**o
        ref[5,1,1]=(-0.118293731439)/(o+1.)+(-0.355568544431)+(0.0392786678054)*0.5**o
      else:
        arg[0,0,0]=(0.205446621454)*x[0]+(0.10254878667)*x[1]+(1.27673728979)*x[2]
        arg[0,0,1]=(0.758846313193)*x[0]+(0.63320336713)*x[1]+(-0.42638505807)*x[2]
        arg[0,1,0]=(-0.0127116561603)*x[0]+(-0.360359420027)*x[1]+(0.540508591493)*x[2]
        arg[0,1,1]=(-0.278918629538)*x[0]+(-0.0291534835772)*x[1]+(0.708655731565)*x[2]
        arg[1,0,0]=(-0.612965210483)*x[0]+(0.787787556912)*x[1]+(0.222384448462)*x[2]
        arg[1,0,1]=(1.07409083742)*x[0]+(1.19429173706)*x[1]+(-0.484979976779)*x[2]
        arg[1,1,0]=(-1.25714999157)*x[0]+(-0.321346019622)*x[1]+(0.278329320837)*x[2]
        arg[1,1,1]=(-0.512262689707)*x[0]+(1.6891992053)*x[1]+(-1.02351830068)*x[2]
        arg[2,0,0]=(-1.57911907773)*x[0]+(0.096124258122)*x[1]+(0.336619253521)*x[2]
        arg[2,0,1]=(0.0901144254199)*x[0]+(-1.36818186255)*x[1]+(1.2969609239)*x[2]
        arg[2,1,0]=(0.158329511499)*x[0]+(0.696997185217)*x[1]+(-0.400535235927)*x[2]
        arg[2,1,1]=(-1.44570853201)*x[0]+(0.237359587195)*x[1]+(-1.17323024801)*x[2]
        arg[3,0,0]=(-1.4325812824)*x[0]+(-0.733085265902)*x[1]+(0.125883186484)*x[2]
        arg[3,0,1]=(-0.0529889021435)*x[0]+(0.335154600942)*x[1]+(0.839337375209)*x[2]
        arg[3,1,0]=(-0.423552182963)*x[0]+(1.31813289315)*x[1]+(0.13235854915)*x[2]
        arg[3,1,1]=(0.53278572217)*x[0]+(0.159434073071)*x[1]+(-0.313621199136)*x[2]
        arg[4,0,0]=(0.747779887673)*x[0]+(0.30977483911)*x[1]+(-0.0760295896739)*x[2]
        arg[4,0,1]=(0.00295118909194)*x[0]+(-0.687914867842)*x[1]+(0.0386268422915)*x[2]
        arg[4,1,0]=(-0.276363532754)*x[0]+(0.52102466785)*x[1]+(-0.684840191917)*x[2]
        arg[4,1,1]=(0.0283138898538)*x[0]+(0.116042458294)*x[1]+(0.485578401528)*x[2]
        arg[5,0,0]=(-0.118402060192)*x[0]+(-1.18409809944)*x[1]+(-0.304048852342)*x[2]
        arg[5,0,1]=(0.174393790186)*x[0]+(-0.524912689263)*x[1]+(-0.648708392355)*x[2]
        arg[5,1,0]=(-0.254958756344)*x[0]+(-0.772071348728)*x[1]+(-0.0572494064027)*x[2]
        arg[5,1,1]=(-0.74777072018)*x[0]+(0.19109939364)*x[1]+(0.498973987814)*x[2]
        ref[0,0,0]=(0.986457444152)/(o+1.)+(0.258994989412)+(0.0802852749346)*0.5**o
        ref[0,0,1]=(-0.0286717143415)/(o+1.)+(0.554867138611)+(-0.115397940627)*0.5**o
        ref[0,1,0]=(-0.886482131138)/(o+1.)+(0.464293129749)+(0.125333386946)*0.5**o
        ref[0,1,1]=(0.199214164457)/(o+1.)+(0.336882900218)+(-0.472396346443)*0.5**o
        ref[1,0,0]=(0.36021680954)/(o+1.)+(0.408903924703)+(-0.780817864055)*0.5**o
        ref[1,0,1]=(0.44795581074)/(o+1.)+(0.472159267742)+(0.391128251477)*0.5**o
        ref[1,1,0]=(1.02665734893)/(o+1.)+(-0.71418625728)+(-0.898451524719)*0.5**o
        ref[1,1,1]=(0.0322676607777)/(o+1.)+(0.105378958385)+(-0.0896073626306)*0.5**o
        ref[2,0,0]=(1.15723247637)/(o+1.)+(-0.719604538702)+(-0.86439896505)*0.5**o
        ref[2,0,1]=(-0.254100181576)/(o+1.)+(-0.0387997401585)+(0.35059314866)*0.5**o
        ref[2,1,0]=(-0.636439167982)/(o+1.)+(0.563390681981)+(-0.0355507351918)*0.5**o
        ref[2,1,1]=(-0.265265225142)/(o+1.)+(-0.635620798949)+(-0.845072369786)*0.5**o
        ref[3,0,0]=(-0.062162800164)/(o+1.)+(-0.554499928282)+(-0.868620705087)*0.5**o
        ref[3,0,1]=(0.784681961813)/(o+1.)+(0.355202655131)+(-0.373584198067)*0.5**o
        ref[3,1,0]=(1.20272043429)/(o+1.)+(0.352677694103)+(-0.881136563162)*0.5**o
        ref[3,1,1]=(-0.311152797156)/(o+1.)+(0.313736563953)+(0.0622782653549)*0.5**o
        ref[4,0,0]=(-0.365517620238)/(o+1.)+(0.733502174253)+(-0.119961591158)*0.5**o
        ref[4,0,1]=(0.0155845896619)/(o+1.)+(-0.654963543839)+(0.648005661558)*0.5**o
        ref[4,1,0]=(0.405697975915)/(o+1.)+(-0.637805133954)+(0.429733235172)*0.5**o
        ref[4,1,1]=(-0.0469738326054)/(o+1.)+(0.0590721074184)+(0.558764367445)*0.5**o
        ref[5,0,0]=(-0.333825007954)/(o+1.)+(-0.46448520882)+(-0.343753586377)*0.5**o
        ref[5,0,1]=(-0.469125954795)/(o+1.)+(-0.147192309255)+(-0.235716718127)*0.5**o
        ref[5,1,0]=(-0.573275891777)/(o+1.)+(-0.478328163121)+(0.445652706545)*0.5**o
        ref[5,1,1]=(0.0545515234726)/(o+1.)+(-0.153919681346)+(0.195590500493)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedSolution_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.226142137728)*x[0]+(0.85681895183)*x[1]
        arg[0,0,0,1]=(-0.824407181762)*x[0]+(-0.290023144892)*x[1]
        arg[0,0,1,0]=(-1.25987816017)*x[0]+(0.824340990595)*x[1]
        arg[0,0,1,1]=(-0.0917811606765)*x[0]+(0.660211932964)*x[1]
        arg[0,0,2,0]=(-1.31073149178)*x[0]+(0.482324984918)*x[1]
        arg[0,0,2,1]=(0.256995458227)*x[0]+(0.79244254852)*x[1]
        arg[0,1,0,0]=(-0.0682147475407)*x[0]+(0.243620052061)*x[1]
        arg[0,1,0,1]=(0.071099982687)*x[0]+(-0.62986754897)*x[1]
        arg[0,1,1,0]=(-0.0716793733131)*x[0]+(-1.15235525814)*x[1]
        arg[0,1,1,1]=(0.193107575216)*x[0]+(-0.596149957543)*x[1]
        arg[0,1,2,0]=(-0.160142629886)*x[0]+(1.36581707067)*x[1]
        arg[0,1,2,1]=(-1.21477648376)*x[0]+(-0.732000298369)*x[1]
        arg[0,2,0,0]=(-1.02468971059)*x[0]+(0.330721595217)*x[1]
        arg[0,2,0,1]=(1.11414907391)*x[0]+(1.29386127812)*x[1]
        arg[0,2,1,0]=(1.1485850154)*x[0]+(0.387585977495)*x[1]
        arg[0,2,1,1]=(0.237382583072)*x[0]+(1.56248221279)*x[1]
        arg[0,2,2,0]=(-0.949129049229)*x[0]+(0.578823379063)*x[1]
        arg[0,2,2,1]=(-1.0667182537)*x[0]+(0.462227485959)*x[1]
        arg[0,3,0,0]=(1.37954377554)*x[0]+(0.541856880139)*x[1]
        arg[0,3,0,1]=(0.401584700319)*x[0]+(0.577102327147)*x[1]
        arg[0,3,1,0]=(-0.289796408108)*x[0]+(0.807853785325)*x[1]
        arg[0,3,1,1]=(0.0038467861822)*x[0]+(-0.931311510771)*x[1]
        arg[0,3,2,0]=(0.859036254821)*x[0]+(0.198884218516)*x[1]
        arg[0,3,2,1]=(-0.00310256070968)*x[0]+(0.511838475968)*x[1]
        arg[0,4,0,0]=(0.539686886802)*x[0]+(-0.324612924267)*x[1]
        arg[0,4,0,1]=(-0.0458655898565)*x[0]+(0.211810380966)*x[1]
        arg[0,4,1,0]=(0.302333533003)*x[0]+(0.0663033316831)*x[1]
        arg[0,4,1,1]=(-0.981519161639)*x[0]+(-0.00207756909914)*x[1]
        arg[0,4,2,0]=(0.283659447004)*x[0]+(-0.231526592979)*x[1]
        arg[0,4,2,1]=(-1.17679167875)*x[0]+(0.285698985756)*x[1]
        arg[1,0,0,0]=(-0.162900059582)*x[0]+(-0.877164340783)*x[1]
        arg[1,0,0,1]=(-0.50468605065)*x[0]+(-0.160440073709)*x[1]
        arg[1,0,1,0]=(-0.100901676644)*x[0]+(-0.296111526029)*x[1]
        arg[1,0,1,1]=(0.176107421869)*x[0]+(1.74393708745)*x[1]
        arg[1,0,2,0]=(-0.979807721703)*x[0]+(-0.229400841067)*x[1]
        arg[1,0,2,1]=(0.217472795645)*x[0]+(0.0881503243323)*x[1]
        arg[1,1,0,0]=(-0.127980123625)*x[0]+(0.0932095456541)*x[1]
        arg[1,1,0,1]=(0.829674410931)*x[0]+(-0.796196041601)*x[1]
        arg[1,1,1,0]=(1.28656110233)*x[0]+(-1.27850471392)*x[1]
        arg[1,1,1,1]=(1.159269884)*x[0]+(-0.973135433631)*x[1]
        arg[1,1,2,0]=(1.46274837856)*x[0]+(-1.81554270863)*x[1]
        arg[1,1,2,1]=(0.0973472511688)*x[0]+(0.551451261495)*x[1]
        arg[1,2,0,0]=(0.510667671293)*x[0]+(-0.51075982099)*x[1]
        arg[1,2,0,1]=(-1.41612093451)*x[0]+(0.614417669629)*x[1]
        arg[1,2,1,0]=(1.46595125796)*x[0]+(0.60374279122)*x[1]
        arg[1,2,1,1]=(-0.317185218187)*x[0]+(-0.0785532882696)*x[1]
        arg[1,2,2,0]=(-0.388504041993)*x[0]+(-0.838003398775)*x[1]
        arg[1,2,2,1]=(0.0497207129012)*x[0]+(-0.622155692247)*x[1]
        arg[1,3,0,0]=(-0.0471400179563)*x[0]+(1.3378948207)*x[1]
        arg[1,3,0,1]=(-1.22250822195)*x[0]+(0.629714783138)*x[1]
        arg[1,3,1,0]=(-1.28801080408)*x[0]+(-1.54079710038)*x[1]
        arg[1,3,1,1]=(-0.426412448469)*x[0]+(-1.09410474021)*x[1]
        arg[1,3,2,0]=(0.022834167351)*x[0]+(-0.497423389788)*x[1]
        arg[1,3,2,1]=(-0.787492124301)*x[0]+(-0.245621168419)*x[1]
        arg[1,4,0,0]=(-1.37814880919)*x[0]+(-0.704805848451)*x[1]
        arg[1,4,0,1]=(-1.12494770055)*x[0]+(1.44920024305)*x[1]
        arg[1,4,1,0]=(-0.809532572761)*x[0]+(1.16638587657)*x[1]
        arg[1,4,1,1]=(0.931784486816)*x[0]+(1.34629889453)*x[1]
        arg[1,4,2,0]=(-0.172430942793)*x[0]+(-0.584736998321)*x[1]
        arg[1,4,2,1]=(0.00542948736389)*x[0]+(0.43917597284)*x[1]
        arg[2,0,0,0]=(-0.946122337406)*x[0]+(-0.161940168137)*x[1]
        arg[2,0,0,1]=(-0.7856653779)*x[0]+(-0.75287701788)*x[1]
        arg[2,0,1,0]=(-1.08858367592)*x[0]+(-0.530380818694)*x[1]
        arg[2,0,1,1]=(-1.0573555095)*x[0]+(-0.854148765785)*x[1]
        arg[2,0,2,0]=(0.723412825328)*x[0]+(0.326761950437)*x[1]
        arg[2,0,2,1]=(-0.0143142161225)*x[0]+(-0.752724311876)*x[1]
        arg[2,1,0,0]=(-0.942548092355)*x[0]+(0.700861094897)*x[1]
        arg[2,1,0,1]=(-0.712286567957)*x[0]+(0.230406545738)*x[1]
        arg[2,1,1,0]=(1.27414839774)*x[0]+(0.724784649037)*x[1]
        arg[2,1,1,1]=(0.637494163002)*x[0]+(0.442489599192)*x[1]
        arg[2,1,2,0]=(1.81617631726)*x[0]+(1.56870218809)*x[1]
        arg[2,1,2,1]=(1.16419900666)*x[0]+(-0.044314729141)*x[1]
        arg[2,2,0,0]=(0.880267125196)*x[0]+(0.77124406259)*x[1]
        arg[2,2,0,1]=(0.482402867542)*x[0]+(0.0746924305017)*x[1]
        arg[2,2,1,0]=(0.105861259696)*x[0]+(0.484292207267)*x[1]
        arg[2,2,1,1]=(-0.107785838196)*x[0]+(0.701132243209)*x[1]
        arg[2,2,2,0]=(0.559717405844)*x[0]+(-1.5585942717)*x[1]
        arg[2,2,2,1]=(-0.989130712605)*x[0]+(0.221361137556)*x[1]
        arg[2,3,0,0]=(-0.045471818095)*x[0]+(-0.437941959792)*x[1]
        arg[2,3,0,1]=(1.25840468379)*x[0]+(-1.18872011057)*x[1]
        arg[2,3,1,0]=(1.70665829408)*x[0]+(0.705273921003)*x[1]
        arg[2,3,1,1]=(-0.0928709253845)*x[0]+(0.785115449756)*x[1]
        arg[2,3,2,0]=(-0.887243418838)*x[0]+(1.06790660642)*x[1]
        arg[2,3,2,1]=(-0.572644686872)*x[0]+(-0.672831948394)*x[1]
        arg[2,4,0,0]=(1.28282566801)*x[0]+(0.651985199447)*x[1]
        arg[2,4,0,1]=(0.0965663229335)*x[0]+(1.33819998052)*x[1]
        arg[2,4,1,0]=(0.205569041098)*x[0]+(-0.873095358733)*x[1]
        arg[2,4,1,1]=(-1.04452025745)*x[0]+(-0.204917033907)*x[1]
        arg[2,4,2,0]=(-0.181163938972)*x[0]+(0.0341993409272)*x[1]
        arg[2,4,2,1]=(1.68327027059)*x[0]+(-0.289436355642)*x[1]
        arg[3,0,0,0]=(1.05712810731)*x[0]+(0.915711819735)*x[1]
        arg[3,0,0,1]=(-0.171072039482)*x[0]+(-0.160656943623)*x[1]
        arg[3,0,1,0]=(-1.27063799869)*x[0]+(-0.605131642813)*x[1]
        arg[3,0,1,1]=(0.940512990835)*x[0]+(-0.116675577531)*x[1]
        arg[3,0,2,0]=(-0.194213869952)*x[0]+(0.364066752025)*x[1]
        arg[3,0,2,1]=(-0.328780465397)*x[0]+(-0.114689413673)*x[1]
        arg[3,1,0,0]=(-0.825914305895)*x[0]+(-0.167010056731)*x[1]
        arg[3,1,0,1]=(0.657779056207)*x[0]+(0.0666252451638)*x[1]
        arg[3,1,1,0]=(-0.464004426036)*x[0]+(-0.340932918409)*x[1]
        arg[3,1,1,1]=(-1.6698652667)*x[0]+(0.731870373141)*x[1]
        arg[3,1,2,0]=(1.39626250779)*x[0]+(-0.879152855557)*x[1]
        arg[3,1,2,1]=(1.34973919508)*x[0]+(-0.129753332761)*x[1]
        arg[3,2,0,0]=(1.03443450706)*x[0]+(0.807333477811)*x[1]
        arg[3,2,0,1]=(0.366011302596)*x[0]+(0.543804800926)*x[1]
        arg[3,2,1,0]=(0.262446846036)*x[0]+(-0.247786850991)*x[1]
        arg[3,2,1,1]=(-1.04657091196)*x[0]+(-0.485443426155)*x[1]
        arg[3,2,2,0]=(0.342434327538)*x[0]+(-0.148503585588)*x[1]
        arg[3,2,2,1]=(-0.189833429716)*x[0]+(0.424391874819)*x[1]
        arg[3,3,0,0]=(-0.118348838118)*x[0]+(0.397628264341)*x[1]
        arg[3,3,0,1]=(0.481440510084)*x[0]+(-0.393576221424)*x[1]
        arg[3,3,1,0]=(0.907533552208)*x[0]+(-0.668646549376)*x[1]
        arg[3,3,1,1]=(-1.49731714213)*x[0]+(-1.18710678264)*x[1]
        arg[3,3,2,0]=(-0.460660079207)*x[0]+(1.48078266444)*x[1]
        arg[3,3,2,1]=(-0.161073556988)*x[0]+(-0.486827042754)*x[1]
        arg[3,4,0,0]=(-0.668755455623)*x[0]+(0.345804059434)*x[1]
        arg[3,4,0,1]=(1.00528889193)*x[0]+(0.978507928191)*x[1]
        arg[3,4,1,0]=(0.98955856331)*x[0]+(1.0421852144)*x[1]
        arg[3,4,1,1]=(0.711688260873)*x[0]+(1.39962222322)*x[1]
        arg[3,4,2,0]=(1.08512429209)*x[0]+(-0.555249097743)*x[1]
        arg[3,4,2,1]=(-1.10443784007)*x[0]+(-0.782274429169)*x[1]
        ref[0,0,0,0]=(0.151153165564)/(o+1.)+(0.112609656739)+(0.254304335059)*0.5**o
        ref[0,0,0,1]=(-0.530602583511)/(o+1.)+(0.1817855378)+(-0.947398818743)*0.5**o
        ref[0,0,1,0]=(0.308747437738)/(o+1.)+(-0.0338129255643)+(-0.676658756184)*0.5**o
        ref[0,0,1,1]=(0.321026553427)/(o+1.)+(0.0692705367618)+(0.108863145337)*0.5**o
        ref[0,0,2,0]=(-0.0908034666655)/(o+1.)+(-0.1100205155)+(-0.517562009199)*0.5**o
        ref[0,0,2,1]=(0.303456040903)/(o+1.)+(0.500090875336)+(-0.254199784829)*0.5**o
        ref[0,1,0,0]=(-0.440641221831)/(o+1.)+(0.456218946873)+(-0.296391367396)*0.5**o
        ref[0,1,0,1]=(0.222628561762)/(o+1.)+(-0.162064412854)+(-0.457267302337)*0.5**o
        ref[0,1,1,0]=(-0.913361887408)/(o+1.)+(-0.356168674239)+(0.401664604431)*0.5**o
        ref[0,1,1,1]=(-0.543116639248)/(o+1.)+(-0.120902439422)+(0.381879135767)*0.5**o
        ref[0,1,2,0]=(0.993649262831)/(o+1.)+(0.384695858206)+(-0.55736653846)*0.5**o
        ref[0,1,2,1]=(-0.968077797451)/(o+1.)+(-0.187765942637)+(-0.603167099406)*0.5**o
        ref[0,2,0,0]=(0.873662019918)/(o+1.)+(-0.710635360033)+(-0.146359415223)*0.5**o
        ref[0,2,0,1]=(0.466408230693)/(o+1.)+(0.77626565285)+(0.389070815631)*0.5**o
        ref[0,2,1,0]=(-0.475010813579)/(o+1.)+(0.525435880006)+(0.960310046461)*0.5**o
        ref[0,2,1,1]=(0.697127520484)/(o+1.)+(0.357204276552)+(0.388328722275)*0.5**o
        ref[0,2,2,0]=(-0.108434738487)/(o+1.)+(0.356443300466)+(-0.97475753261)*0.5**o
        ref[0,2,2,1]=(0.952274964304)/(o+1.)+(-0.497379508976)+(-0.562006714089)*0.5**o
        ref[0,3,0,0]=(-0.0368012761671)/(o+1.)+(0.594000625518)+(0.770200680808)*0.5**o
        ref[0,3,0,1]=(0.450361861836)/(o+1.)+(0.182031052848)+(0.164263059934)*0.5**o
        ref[0,3,1,0]=(0.500714741956)/(o+1.)+(0.0479711343168)+(-0.0785996333735)*0.5**o
        ref[0,3,1,1]=(-0.949280938645)/(o+1.)+(0.452441874504)+(-0.88306753495)*0.5**o
        ref[0,3,2,0]=(0.186931712837)/(o+1.)+(0.222335175903)+(0.426318408694)*0.5**o
        ref[0,3,2,1]=(-0.288015428918)/(o+1.)+(0.80948657963)+(-0.822221815085)*0.5**o
        ref[0,4,0,0]=(0.670151184695)/(o+1.)+(-0.415883847013)+(0.376690471866)*0.5**o
        ref[0,4,0,1]=(-0.107106843992)/(o+1.)+(0.555375298435)+(-0.837698961768)*0.5**o
        ref[0,4,1,0]=(-0.299999096802)/(o+1.)+(0.668996782774)+(-0.669357604061)*0.5**o
        ref[0,4,1,1]=(0.778006183401)/(o+1.)+(-0.614473529032)+(-0.532655856076)*0.5**o
        ref[0,4,2,0]=(-0.250068389818)/(o+1.)+(0.156274314197)+(-0.0103473845504)*0.5**o
        ref[0,4,2,1]=(0.902125322216)/(o+1.)+(-0.616424567683)+(-0.560368879846)*0.5**o
        ref[1,0,0,0]=(-0.133690258295)/(o+1.)+(-0.715949360249)+(0.525524578427)*0.5**o
        ref[1,0,0,1]=(-0.788097689216)/(o+1.)+(0.0998405659256)+(-0.076709566994)*0.5**o
        ref[1,0,1,0]=(0.144280886187)/(o+1.)+(0.0775377290314)+(-0.696369546924)*0.5**o
        ref[1,0,1,1]=(0.824172196605)/(o+1.)+(0.561804029444)+(-0.0277357461773)*0.5**o
        ref[1,0,2,0]=(0.422564421802)/(o+1.)+(-0.60719307099)+(-0.417386842591)*0.5**o
        ref[1,0,2,1]=(0.778908672762)/(o+1.)+(0.0163446340765)+(-0.505974820938)*0.5**o
        ref[1,1,0,0]=(0.0574071047387)/(o+1.)+(-0.350514214742)+(0.608850746775)*0.5**o
        ref[1,1,0,1]=(-0.991660961792)/(o+1.)+(0.322084279455)+(0.380970772212)*0.5**o
        ref[1,1,1,0]=(-0.823761120268)/(o+1.)+(-0.0495419092898)+(0.930901327253)*0.5**o
        ref[1,1,1,1]=(-0.384729980569)/(o+1.)+(-0.0392583789163)+(0.649381188769)*0.5**o
        ref[1,1,2,0]=(-0.840365304069)/(o+1.)+(-0.0150512662777)+(0.517673506558)*0.5**o
        ref[1,1,2,1]=(0.555976660846)/(o+1.)+(-0.307265886898)+(0.707353625614)*0.5**o
        ref[1,2,0,0]=(-0.423056705491)/(o+1.)+(0.0559404393793)+(0.311083677036)*0.5**o
        ref[1,2,0,1]=(0.15409797664)/(o+1.)+(-0.0906637067519)+(-0.774473828018)*0.5**o
        ref[1,2,1,0]=(0.796718644088)/(o+1.)+(0.248012077098)+(0.776951250898)*0.5**o
        ref[1,2,1,1]=(-0.342830889263)/(o+1.)+(-0.282162737841)+(0.511417858487)*0.5**o
        ref[1,2,2,0]=(-0.722403016671)/(o+1.)+(-0.367059457954)+(0.230014491812)*0.5**o
        ref[1,2,2,1]=(-0.773685058881)/(o+1.)+(0.412807900451)+(-0.624365721366)*0.5**o
        ref[1,3,0,0]=(0.527481434758)/(o+1.)+(0.36602961117)+(0.031214145649)*0.5**o
        ref[1,3,0,1]=(0.0630927531585)/(o+1.)+(-0.164989701995)+(-0.325906787975)*0.5**o
        ref[1,3,1,0]=(-0.612390590593)/(o+1.)+(-0.78855996274)+(-0.63929738839)*0.5**o
        ref[1,3,1,1]=(-0.962656044265)/(o+1.)+(-0.00576535037253)+(-0.546330443673)*0.5**o
        ref[1,3,2,0]=(-0.947280922008)/(o+1.)+(-0.137449579996)+(0.747590859563)*0.5**o
        ref[1,3,2,1]=(-0.946623026202)/(o+1.)+(0.0787536550213)+(-0.24399757656)*0.5**o
        ref[1,4,0,0]=(-0.174471016943)/(o+1.)+(-0.636500556039)+(-0.635482528623)*0.5**o
        ref[1,4,0,1]=(0.806379391595)/(o+1.)+(-0.00304106577093)+(-0.476044717545)*0.5**o
        ref[1,4,1,0]=(0.729649125134)/(o+1.)+(0.0151477051616)+(-0.403091231646)*0.5**o
        ref[1,4,1,1]=(0.559976454745)/(o+1.)+(0.479295896421)+(0.75951513376)*0.5**o
        ref[1,4,2,0]=(0.272199270818)/(o+1.)+(-0.0929072274332)+(-0.843552757065)*0.5**o
        ref[1,4,2,1]=(-0.552663678377)/(o+1.)+(0.613784745778)+(-0.230300352975)*0.5**o
        ref[2,0,0,0]=(-0.174247908098)/(o+1.)+(-0.108873783282)+(-0.716067030882)*0.5**o
        ref[2,0,0,1]=(-0.300584841532)/(o+1.)+(-0.485363665493)+(-0.267230223262)*0.5**o
        ref[2,0,1,0]=(0.316261805154)/(o+1.)+(-0.523474106451)+(-0.888278086869)*0.5**o
        ref[2,0,1,1]=(-0.699622562861)/(o+1.)+(-0.151089538248)+(-0.909702635929)*0.5**o
        ref[2,0,2,0]=(0.728545639829)/(o+1.)+(0.192083148877)+(-0.0625371618174)*0.5**o
        ref[2,0,2,1]=(-0.22042267239)/(o+1.)+(-0.611974599324)+(0.677333343041)*0.5**o
        ref[2,1,0,0]=(-0.0734883811113)/(o+1.)+(0.36100831022)+(-0.890215236787)*0.5**o
        ref[2,1,0,1]=(0.121656829057)/(o+1.)+(-0.130801635799)+(-0.341933579678)*0.5**o
        ref[2,1,1,0]=(0.221353168629)/(o+1.)+(0.618635673483)+(0.540308531179)*0.5**o
        ref[2,1,1,1]=(0.0152580531273)/(o+1.)+(0.252173415542)+(0.560378877984)*0.5**o
        ref[2,1,2,0]=(0.580319777731)/(o+1.)+(0.90566612137)+(0.993226484879)*0.5**o
        ref[2,1,2,1]=(-0.710250320949)/(o+1.)+(0.742177021871)+(0.345780554724)*0.5**o
        ref[2,2,0,0]=(0.770701563507)/(o+1.)+(0.240823471881)+(0.399162680517)*0.5**o
        ref[2,2,0,1]=(0.76121761761)/(o+1.)+(0.128382656738)+(-0.460887633042)*0.5**o
        ref[2,2,1,0]=(-0.269854962057)/(o+1.)+(0.690854898294)+(-0.521701367567)*0.5**o
        ref[2,2,1,1]=(0.719518595712)/(o+1.)+(0.029596430244)+(-0.185365051187)*0.5**o
        ref[2,2,2,0]=(-0.671157761165)/(o+1.)+(-0.142688285012)+(-0.0423425346644)*0.5**o
        ref[2,2,2,1]=(-0.541156097112)/(o+1.)+(0.0765180918929)+(-0.379649661722)*0.5**o
        ref[2,3,0,0]=(-0.0604347109212)/(o+1.)+(-0.19403702073)+(-0.034905025507)*0.5**o
        ref[2,3,0,1]=(-0.793398507071)/(o+1.)+(0.255681280626)+(0.351720519037)*0.5**o
        ref[2,3,1,0]=(-0.267058413731)/(o+1.)+(0.852211426049)+(0.974567776717)*0.5**o
        ref[2,3,1,1]=(0.228396641207)/(o+1.)+(0.322922896735)+(-0.181997910305)*0.5**o
        ref[2,3,2,0]=(0.480987773937)/(o+1.)+(-0.197709028569)+(0.0950934707835)*0.5**o
        ref[2,3,2,1]=(-0.648934475027)/(o+1.)+(-0.418383297875)+(0.240224435511)*0.5**o
        ref[2,4,0,0]=(0.430020261528)/(o+1.)+(0.53421018733)+(0.436370231269)*0.5**o
        ref[2,4,0,1]=(0.906390660223)/(o+1.)+(-0.056057545856)+(0.640490734948)*0.5**o
        ref[2,4,1,0]=(-0.732099770943)/(o+1.)+(-0.0485027617732)+(0.161578976854)*0.5**o
        ref[2,4,1,1]=(-0.827495680848)/(o+1.)+(0.285044728936)+(-0.992031068384)*0.5**o
        ref[2,4,2,0]=(-0.919905471787)/(o+1.)+(0.656971386264)+(-0.541001898785)*0.5**o
        ref[2,4,2,1]=(0.161975875302)/(o+1.)+(0.272665206557)+(0.686527626533)*0.5**o
        ref[3,0,0,0]=(0.584120963259)/(o+1.)+(0.207930729804)+(0.972857504177)*0.5**o
        ref[3,0,0,1]=(0.315823141491)/(o+1.)+(-0.217561208386)+(-0.212429707823)*0.5**o
        ref[3,0,1,0]=(-0.732294996139)/(o+1.)+(-0.278653143596)+(-0.586168358175)*0.5**o
        ref[3,0,1,1]=(0.481321847225)/(o+1.)+(-0.070080648792)+(0.482676863662)*0.5**o
        ref[3,0,2,0]=(0.616679859076)/(o+1.)+(-0.418784546677)+(0.390742116352)*0.5**o
        ref[3,0,2,1]=(0.866778417448)/(o+1.)+(-0.686585656643)+(0.0629230167692)*0.5**o
        ref[3,1,0,0]=(0.531890054167)/(o+1.)+(-0.516187607633)+(-0.492439201527)*0.5**o
        ref[3,1,0,1]=(-0.512361639413)/(o+1.)+(0.499557182545)+(0.237651575694)*0.5**o
        ref[3,1,1,0]=(-0.67542965405)/(o+1.)+(0.274414249559)+(-0.678336189514)*0.5**o
        ref[3,1,1,1]=(0.371936242507)/(o+1.)+(-0.1636482466)+(-0.982634642862)*0.5**o
        ref[3,1,2,0]=(-0.799408831501)/(o+1.)+(0.458633399549)+(0.399251684639)*0.5**o
        ref[3,1,2,1]=(-0.0145253903686)/(o+1.)+(0.238472345905)+(0.757566560882)*0.5**o
        ref[3,2,0,0]=(-0.130519473112)/(o+1.)+(0.695277604837)+(0.581732248306)*0.5**o
        ref[3,2,0,1]=(0.632010536438)/(o+1.)+(0.123066205529)+(0.0316731560259)*0.5**o
        ref[3,2,1,0]=(-0.709926210356)/(o+1.)+(0.0926699972302)+(0.53924621094)*0.5**o
        ref[3,2,1,1]=(0.428094002247)/(o+1.)+(-0.841997926131)+(-0.276112488096)*0.5**o
        ref[3,2,2,0]=(0.708481148086)/(o+1.)+(-0.50498161534)+(0.495412824545)*0.5**o
        ref[3,2,2,1]=(0.782824622703)/(o+1.)+(-0.151509698687)+(-0.245246780227)*0.5**o
        ref[3,3,0,0]=(0.848460868418)/(o+1.)+(-0.104639248461)+(-0.359902945273)*0.5**o
        ref[3,3,0,1]=(0.0569151307298)/(o+1.)+(-0.0674224535469)+(0.165794065024)*0.5**o
        ref[3,3,1,0]=(-0.165683739908)/(o+1.)+(-0.183388699236)+(0.771348141212)*0.5**o
        ref[3,3,1,1]=(-0.932550058069)/(o+1.)+(-0.47828109001)+(-0.795311686688)*0.5**o
        ref[3,3,2,0]=(0.821542940923)/(o+1.)+(0.166853194487)+(-0.13512674466)*0.5**o
        ref[3,3,2,1]=(-0.46375871628)/(o+1.)+(0.132701267732)+(-0.449544418926)*0.5**o
        ref[3,4,0,0]=(-0.074530542168)/(o+1.)+(-0.0567877963797)+(-0.134845261261)*0.5**o
        ref[3,4,0,1]=(0.238671520479)/(o+1.)+(0.50689253572)+(0.731340228204)*0.5**o
        ref[3,4,1,0]=(0.981601513335)/(o+1.)+(0.400717209822)+(0.248707844726)*0.5**o
        ref[3,4,1,1]=(0.592304390107)/(o+1.)+(0.342408637328)+(0.834188819333)*0.5**o
        ref[3,4,2,0]=(-0.339991330909)/(o+1.)+(-0.0145922376156)+(0.899051000485)*0.5**o
        ref[3,4,2,1]=(-0.626610688164)/(o+1.)+(-0.417008165115)+(-0.426085250846)*0.5**o
      else:
        arg[0,0,0,0]=(0.697666818785)*x[0]+(1.00202295791)*x[1]+(0.617700157184)*x[2]
        arg[0,0,0,1]=(-0.465577819457)*x[0]+(-1.40054693355)*x[1]+(1.16912653295)*x[2]
        arg[0,0,1,0]=(-1.26696500634)*x[0]+(-0.0469756169975)*x[1]+(-0.445591738991)*x[2]
        arg[0,0,1,1]=(-1.61654321709)*x[0]+(-1.69522962467)*x[1]+(-1.06089094851)*x[2]
        arg[0,0,2,0]=(0.229468378506)*x[0]+(0.716813596991)*x[1]+(-0.237703937029)*x[2]
        arg[0,0,2,1]=(-1.12052051905)*x[0]+(0.835000004618)*x[1]+(-0.853004812404)*x[2]
        arg[0,1,0,0]=(-1.75500745705)*x[0]+(0.655055836882)*x[1]+(1.05207750735)*x[2]
        arg[0,1,0,1]=(-0.0746024824253)*x[0]+(-0.421596816958)*x[1]+(-0.375715855989)*x[2]
        arg[0,1,1,0]=(-0.963269684396)*x[0]+(0.431993248836)*x[1]+(-1.63061928649)*x[2]
        arg[0,1,1,1]=(-0.418757265018)*x[0]+(-0.665626168583)*x[1]+(-0.335734432507)*x[2]
        arg[0,1,2,0]=(0.980346404307)*x[0]+(1.45442049802)*x[1]+(-0.833420308612)*x[2]
        arg[0,1,2,1]=(-1.52536017847)*x[0]+(0.99484028681)*x[1]+(0.109754321663)*x[2]
        arg[0,2,0,0]=(1.697179262)*x[0]+(-0.928732802147)*x[1]+(-0.653379513626)*x[2]
        arg[0,2,0,1]=(0.682698902948)*x[0]+(0.829307959583)*x[1]+(-1.0300112276)*x[2]
        arg[0,2,1,0]=(0.026084323266)*x[0]+(1.28865784608)*x[1]+(0.830446580645)*x[2]
        arg[0,2,1,1]=(0.486291506032)*x[0]+(-0.519637742913)*x[1]+(0.81671330629)*x[2]
        arg[0,2,2,0]=(-0.761591624634)*x[0]+(0.370951131011)*x[1]+(-0.256748479312)*x[2]
        arg[0,2,2,1]=(-0.249974326513)*x[0]+(0.267323484635)*x[1]+(-1.09642130905)*x[2]
        arg[0,3,0,0]=(0.752057563255)*x[0]+(-1.2772262175)*x[1]+(-0.264569668983)*x[2]
        arg[0,3,0,1]=(-0.173167796447)*x[0]+(0.31873208356)*x[1]+(-0.599780701112)*x[2]
        arg[0,3,1,0]=(-0.912464014941)*x[0]+(-1.56452517166)*x[1]+(1.37651660499)*x[2]
        arg[0,3,1,1]=(0.302009667854)*x[0]+(0.524213064769)*x[1]+(0.316301522683)*x[2]
        arg[0,3,2,0]=(-1.2940154673)*x[0]+(-0.900061396992)*x[1]+(0.3860029847)*x[2]
        arg[0,3,2,1]=(-0.97354131341)*x[0]+(1.08221883964)*x[1]+(-0.739323076416)*x[2]
        arg[0,4,0,0]=(1.86535317192)*x[0]+(-0.478359589546)*x[1]+(-0.798014592302)*x[2]
        arg[0,4,0,1]=(-1.08518693851)*x[0]+(-0.186048654536)*x[1]+(1.04915573267)*x[2]
        arg[0,4,1,0]=(-1.09692811055)*x[0]+(1.01042920935)*x[1]+(-0.73978089555)*x[2]
        arg[0,4,1,1]=(0.654656964663)*x[0]+(0.279204596126)*x[1]+(0.886217781361)*x[2]
        arg[0,4,2,0]=(-0.54009886291)*x[0]+(0.378037248739)*x[1]+(1.09785568886)*x[2]
        arg[0,4,2,1]=(-1.43995958913)*x[0]+(-0.720157090994)*x[1]+(0.0676362028136)*x[2]
        arg[1,0,0,0]=(1.47735412608)*x[0]+(0.342355546228)*x[1]+(-0.0134502415508)*x[2]
        arg[1,0,0,1]=(-0.419536590369)*x[0]+(-1.61108470511)*x[1]+(1.51281979355)*x[2]
        arg[1,0,1,0]=(1.31765698245)*x[0]+(-0.7499566294)*x[1]+(0.934537503478)*x[2]
        arg[1,0,1,1]=(0.688467914883)*x[0]+(-1.32681280673)*x[1]+(-0.387405705058)*x[2]
        arg[1,0,2,0]=(1.37069587556)*x[0]+(0.540830111773)*x[1]+(0.477311068263)*x[2]
        arg[1,0,2,1]=(-0.572066213761)*x[0]+(-1.11950925494)*x[1]+(1.07939217872)*x[2]
        arg[1,1,0,0]=(0.564997105326)*x[0]+(-0.73130546389)*x[1]+(0.435910937304)*x[2]
        arg[1,1,0,1]=(0.513931739674)*x[0]+(-0.944043404826)*x[1]+(0.014114227468)*x[2]
        arg[1,1,1,0]=(-0.77850597256)*x[0]+(-0.445835136484)*x[1]+(0.157478357479)*x[2]
        arg[1,1,1,1]=(0.752431176866)*x[0]+(0.659759664199)*x[1]+(-0.859515433521)*x[2]
        arg[1,1,2,0]=(0.0693158730911)*x[0]+(-0.820997827204)*x[1]+(-0.714694436829)*x[2]
        arg[1,1,2,1]=(0.604674152847)*x[0]+(0.647261183132)*x[1]+(-1.21981352028)*x[2]
        arg[1,2,0,0]=(0.0997504695984)*x[0]+(-0.271549839083)*x[1]+(-0.573467835942)*x[2]
        arg[1,2,0,1]=(0.01861845581)*x[0]+(0.157410502406)*x[1]+(0.939257868776)*x[2]
        arg[1,2,1,0]=(1.44909894631)*x[0]+(0.116038517547)*x[1]+(-0.851940489233)*x[2]
        arg[1,2,1,1]=(1.34598841007)*x[0]+(1.79163125841)*x[1]+(0.719171044245)*x[2]
        arg[1,2,2,0]=(-0.214737791448)*x[0]+(-1.85810839095)*x[1]+(-0.240776599927)*x[2]
        arg[1,2,2,1]=(-0.574707645928)*x[0]+(-0.664173831956)*x[1]+(0.310262106696)*x[2]
        arg[1,3,0,0]=(0.296213453592)*x[0]+(-0.831112482555)*x[1]+(0.130075739128)*x[2]
        arg[1,3,0,1]=(-0.369510604457)*x[0]+(0.0373909697998)*x[1]+(1.6689129783)*x[2]
        arg[1,3,1,0]=(0.0922435881419)*x[0]+(-0.280519827004)*x[1]+(0.285933839232)*x[2]
        arg[1,3,1,1]=(-0.649253282272)*x[0]+(-0.460065044123)*x[1]+(-0.0711164948063)*x[2]
        arg[1,3,2,0]=(0.793313307581)*x[0]+(-0.0587956351149)*x[1]+(0.810487259739)*x[2]
        arg[1,3,2,1]=(-1.38668597055)*x[0]+(1.51396934198)*x[1]+(-1.13918550255)*x[2]
        arg[1,4,0,0]=(1.40078167737)*x[0]+(-1.61915436345)*x[1]+(0.421710091214)*x[2]
        arg[1,4,0,1]=(-1.37982015877)*x[0]+(1.30607371738)*x[1]+(-0.608888848419)*x[2]
        arg[1,4,1,0]=(-0.558087802805)*x[0]+(-0.214169499512)*x[1]+(-0.766242546503)*x[2]
        arg[1,4,1,1]=(-1.75503436584)*x[0]+(0.183604705451)*x[1]+(-0.148071484576)*x[2]
        arg[1,4,2,0]=(0.537942069142)*x[0]+(-0.00640749928252)*x[1]+(-0.277544629657)*x[2]
        arg[1,4,2,1]=(-0.652645833371)*x[0]+(-1.33439077026)*x[1]+(-0.226308197201)*x[2]
        arg[2,0,0,0]=(1.45646289279)*x[0]+(1.40842127675)*x[1]+(1.75078281495)*x[2]
        arg[2,0,0,1]=(0.867988651995)*x[0]+(1.35061812241)*x[1]+(-1.56520776323)*x[2]
        arg[2,0,1,0]=(0.549686523522)*x[0]+(-0.528085465813)*x[1]+(-0.994978820267)*x[2]
        arg[2,0,1,1]=(0.383595996552)*x[0]+(0.612989859087)*x[1]+(-0.560401247231)*x[2]
        arg[2,0,2,0]=(-0.341259919499)*x[0]+(-0.125491675042)*x[1]+(0.545572340757)*x[2]
        arg[2,0,2,1]=(-0.414269395434)*x[0]+(-0.433969881898)*x[1]+(-1.01820199328)*x[2]
        arg[2,1,0,0]=(-0.113575280389)*x[0]+(0.326916039942)*x[1]+(-0.576072890249)*x[2]
        arg[2,1,0,1]=(-0.296473969661)*x[0]+(0.195675099431)*x[1]+(-1.72378150831)*x[2]
        arg[2,1,1,0]=(-1.31575932758)*x[0]+(-0.122557423522)*x[1]+(0.53435027695)*x[2]
        arg[2,1,1,1]=(0.299435754608)*x[0]+(0.727140413706)*x[1]+(0.848931358475)*x[2]
        arg[2,1,2,0]=(0.2618965198)*x[0]+(-0.461672573339)*x[1]+(0.923592803172)*x[2]
        arg[2,1,2,1]=(-0.326817258523)*x[0]+(-0.0363300805442)*x[1]+(1.82049875918)*x[2]
        arg[2,2,0,0]=(-0.435113320888)*x[0]+(0.0257984353786)*x[1]+(0.287846577657)*x[2]
        arg[2,2,0,1]=(1.39178068279)*x[0]+(0.136781967539)*x[1]+(-1.38161590351)*x[2]
        arg[2,2,1,0]=(-0.887874199356)*x[0]+(-0.476769153654)*x[1]+(-0.618760485829)*x[2]
        arg[2,2,1,1]=(1.58883355098)*x[0]+(-0.889469047555)*x[1]+(-0.678486755843)*x[2]
        arg[2,2,2,0]=(0.237147029365)*x[0]+(-0.509883462401)*x[1]+(0.533163151551)*x[2]
        arg[2,2,2,1]=(0.66513913749)*x[0]+(1.30317030657)*x[1]+(0.804083365617)*x[2]
        arg[2,3,0,0]=(1.01531518403)*x[0]+(1.3753053143)*x[1]+(0.000877284235964)*x[2]
        arg[2,3,0,1]=(-0.40853642543)*x[0]+(0.154794268783)*x[1]+(-1.34306671117)*x[2]
        arg[2,3,1,0]=(0.43065284359)*x[0]+(-0.28965146009)*x[1]+(0.660248828789)*x[2]
        arg[2,3,1,1]=(-1.4334922681)*x[0]+(-0.0348802858609)*x[1]+(0.502051304974)*x[2]
        arg[2,3,2,0]=(-1.17399537487)*x[0]+(0.865787043946)*x[1]+(-0.0105949677478)*x[2]
        arg[2,3,2,1]=(0.63104158794)*x[0]+(0.436196079951)*x[1]+(0.498606623061)*x[2]
        arg[2,4,0,0]=(-1.44410177354)*x[0]+(0.513406326011)*x[1]+(0.699127199447)*x[2]
        arg[2,4,0,1]=(-0.981727023077)*x[0]+(0.484182120695)*x[1]+(0.667175832045)*x[2]
        arg[2,4,1,0]=(-1.3352602588)*x[0]+(0.626269023364)*x[1]+(0.813537646695)*x[2]
        arg[2,4,1,1]=(-0.932486930114)*x[0]+(0.299448807849)*x[1]+(0.195862014797)*x[2]
        arg[2,4,2,0]=(-0.0621489644806)*x[0]+(0.131979664158)*x[1]+(0.442029269778)*x[2]
        arg[2,4,2,1]=(-0.507508608633)*x[0]+(-1.28613226533)*x[1]+(-0.697920444023)*x[2]
        arg[3,0,0,0]=(-0.984942677585)*x[0]+(-1.58913299616)*x[1]+(-0.373026784822)*x[2]
        arg[3,0,0,1]=(0.616666427113)*x[0]+(-0.625162800471)*x[1]+(-0.796307290839)*x[2]
        arg[3,0,1,0]=(-0.48469203683)*x[0]+(-0.0572105166553)*x[1]+(0.809572383581)*x[2]
        arg[3,0,1,1]=(-0.487160309892)*x[0]+(0.77646454341)*x[1]+(-0.41947544573)*x[2]
        arg[3,0,2,0]=(-1.17902121554)*x[0]+(-0.244939532033)*x[1]+(-0.282994359476)*x[2]
        arg[3,0,2,1]=(0.33515909881)*x[0]+(0.146866561705)*x[1]+(1.81486098364)*x[2]
        arg[3,1,0,0]=(0.379977901341)*x[0]+(0.834531701517)*x[1]+(-0.0242027062166)*x[2]
        arg[3,1,0,1]=(0.184318292766)*x[0]+(1.08367911805)*x[1]+(-0.164531741594)*x[2]
        arg[3,1,1,0]=(-1.00967115996)*x[0]+(0.542699640149)*x[1]+(0.707616701259)*x[2]
        arg[3,1,1,1]=(0.856948233811)*x[0]+(-1.2186550297)*x[1]+(-0.034268798159)*x[2]
        arg[3,1,2,0]=(0.259470684477)*x[0]+(-1.03586983732)*x[1]+(-0.157275979341)*x[2]
        arg[3,1,2,1]=(-0.161954890658)*x[0]+(1.65853209885)*x[1]+(0.381113456719)*x[2]
        arg[3,2,0,0]=(-0.437413004681)*x[0]+(-0.730340273929)*x[1]+(-0.687689272494)*x[2]
        arg[3,2,0,1]=(-0.336618164118)*x[0]+(-1.24388666895)*x[1]+(-0.2198689254)*x[2]
        arg[3,2,1,0]=(0.0872793087887)*x[0]+(-0.336554573215)*x[1]+(-0.0597803523262)*x[2]
        arg[3,2,1,1]=(0.192805432018)*x[0]+(0.0540495582064)*x[1]+(0.846658042071)*x[2]
        arg[3,2,2,0]=(0.00794192604414)*x[0]+(-0.33504559053)*x[1]+(0.615701149787)*x[2]
        arg[3,2,2,1]=(0.210870961503)*x[0]+(1.14164095846)*x[1]+(-0.192785742791)*x[2]
        arg[3,3,0,0]=(-0.25895162041)*x[0]+(-0.225800436488)*x[1]+(-0.198007079838)*x[2]
        arg[3,3,0,1]=(-0.150363703231)*x[0]+(1.23870429535)*x[1]+(0.28635881247)*x[2]
        arg[3,3,1,0]=(-0.865980718216)*x[0]+(-0.0849853244335)*x[1]+(-0.292164217189)*x[2]
        arg[3,3,1,1]=(-0.350465968722)*x[0]+(0.09853736936)*x[1]+(-0.476711524665)*x[2]
        arg[3,3,2,0]=(0.311396300263)*x[0]+(-0.92437612034)*x[1]+(-0.757139962928)*x[2]
        arg[3,3,2,1]=(-0.912311919837)*x[0]+(0.267793631099)*x[1]+(0.4917345401)*x[2]
        arg[3,4,0,0]=(0.613662541753)*x[0]+(0.782757341361)*x[1]+(1.04209808158)*x[2]
        arg[3,4,0,1]=(0.492828613997)*x[0]+(1.11979095664)*x[1]+(-0.843545072354)*x[2]
        arg[3,4,1,0]=(-0.51939120556)*x[0]+(0.476004484286)*x[1]+(-0.516311785432)*x[2]
        arg[3,4,1,1]=(1.00423313567)*x[0]+(0.90776890388)*x[1]+(1.34423309497)*x[2]
        arg[3,4,2,0]=(-1.44069600826)*x[0]+(-1.39515290909)*x[1]+(1.19851590247)*x[2]
        arg[3,4,2,1]=(0.391851619877)*x[0]+(0.265529120553)*x[1]+(0.996610410913)*x[2]
        ref[0,0,0,0]=(1.19330263935)/(o+1.)+(0.55394275251)+(0.0162017895041)*0.5**o
        ref[0,0,0,1]=(-0.672351783684)/(o+1.)+(-0.187889622039)+(0.351132807711)*0.5**o
        ref[0,0,1,0]=(-1.27021996264)/(o+1.)+(0.236286530101)+(-0.961885459894)*0.5**o
        ref[0,0,1,1]=(-1.43442517107)/(o+1.)+(-1.00113374859)+(-0.935971122022)*0.5**o
        ref[0,0,2,0]=(-0.434813695127)/(o+1.)+(0.606673209583)+(-0.0699546855712)*0.5**o
        ref[0,0,2,1]=(-0.0434901247948)/(o+1.)+(-0.180911877983)+(-0.733211446073)*0.5**o
        ref[0,1,0,0]=(1.41514302182)/(o+1.)+(-0.247977413325)+(-0.967062307989)*0.5**o
        ref[0,1,0,1]=(0.69128100142)/(o+1.)+(-0.621609882923)+(-0.319976390946)*0.5**o
        ref[0,1,1,0]=(-0.881870429191)/(o+1.)+(-0.654075774082)+(0.0281262553057)*0.5**o
        ref[0,1,1,1]=(-1.27239116086)/(o+1.)+(-0.0854066429492)+(0.0230865806541)*0.5**o
        ref[0,1,2,0]=(-0.222663945351)/(o+1.)+(0.685312598747)+(0.453385341573)*0.5**o
        ref[0,1,2,1]=(0.101409038706)/(o+1.)+(0.151367946744)+(-0.824910502191)*0.5**o
        ref[0,2,0,0]=(-1.19509443892)/(o+1.)+(0.225437856866)+(0.85928567141)*0.5**o
        ref[0,2,0,1]=(0.285764347754)/(o+1.)+(-0.383432127133)+(0.963095541446)*0.5**o
        ref[0,2,1,0]=(0.728010227911)/(o+1.)+(0.983790919896)+(-0.550403317717)*0.5**o
        ref[0,2,1,1]=(0.563748259036)/(o+1.)+(0.165164845927)+(-0.110710881481)*0.5**o
        ref[0,2,2,0]=(-0.361502606429)/(o+1.)+(0.0276145482611)+(-0.341115463028)*0.5**o
        ref[0,2,2,1]=(-0.293320207933)/(o+1.)+(0.0336757304385)+(-0.853103403872)*0.5**o
        ref[0,3,0,0]=(-0.410233956527)/(o+1.)+(-0.474281126245)+(0.569057885785)*0.5**o
        ref[0,3,0,1]=(-0.835861970115)/(o+1.)+(0.0173715483175)+(0.346902459481)*0.5**o
        ref[0,3,1,0]=(0.0830466456187)/(o+1.)+(-0.288204681495)+(-0.607109864241)*0.5**o
        ref[0,3,1,1]=(1.07074130823)/(o+1.)+(-0.0437983012867)+(0.159379549652)*0.5**o
        ref[0,3,2,0]=(0.172301049609)/(o+1.)+(-0.491724874152)+(-0.996925180895)*0.5**o
        ref[0,3,2,1]=(-0.24891941221)/(o+1.)+(-0.03935856269)+(-0.3030090126)*0.5**o
        ref[0,4,0,0]=(-0.778575122429)/(o+1.)+(0.213085222078)+(0.941383668349)*0.5**o
        ref[0,4,0,1]=(0.880801627174)/(o+1.)+(-0.454451723401)+(-0.193978040747)*0.5**o
        ref[0,4,1,0]=(-0.835604724882)/(o+1.)+(0.127240834083)+(-0.245156740036)*0.5**o
        ref[0,4,1,1]=(1.62336716324)/(o+1.)+(-0.101085918464)+(0.398884015839)*0.5**o
        ref[0,4,2,0]=(1.16805827702)/(o+1.)+(0.211232591688)+(-0.654729385702)*0.5**o
        ref[0,4,2,1]=(-0.305424138957)/(o+1.)+(-0.431076572693)+(-0.924903192969)*0.5**o
        ref[1,0,0,0]=(-0.481596247529)/(o+1.)+(0.769902632559)+(0.748050413171)*0.5**o
        ref[1,0,0,1]=(0.177388661252)/(o+1.)+(-0.118766442613)+(-0.457657277952)*0.5**o
        ref[1,0,1,0]=(0.709890860372)/(o+1.)+(0.0668010230727)+(0.658744950012)*0.5**o
        ref[1,0,1,1]=(-1.45685781381)/(o+1.)+(-0.178674426089)+(0.788456069079)*0.5**o
        ref[1,0,2,0]=(0.194388185612)/(o+1.)+(0.649177543063)+(0.896093783855)*0.5**o
        ref[1,0,2,1]=(0.181561279507)/(o+1.)+(-0.200618291902)+(-0.392507985678)*0.5**o
        ref[1,1,0,0]=(-0.884376814027)/(o+1.)+(0.280382736159)+(0.593213920451)*0.5**o
        ref[1,1,0,1]=(-0.250709602218)/(o+1.)+(-0.492104211211)+(0.818920586956)*0.5**o
        ref[1,1,1,0]=(0.0529723694625)/(o+1.)+(-0.502047650638)+(-0.115739819752)*0.5**o
        ref[1,1,1,1]=(0.215812202957)/(o+1.)+(-0.308796450292)+(0.954456105171)*0.5**o
        ref[1,1,2,0]=(-0.991481050902)/(o+1.)+(0.162626252838)+(-0.800147845717)*0.5**o
        ref[1,1,2,1]=(-0.433989756951)/(o+1.)+(0.362776336712)+(-0.259441100777)*0.5**o
        ref[1,2,0,0]=(-0.930287052624)/(o+1.)+(-0.347752763108)+(0.880525373413)*0.5**o
        ref[1,2,0,1]=(0.698674852861)/(o+1.)+(-0.15119613196)+(0.719004238051)*0.5**o
        ref[1,2,1,0]=(-0.777216821365)/(o+1.)+(0.436336076297)+(0.617741643399)*0.5**o
        ref[1,2,1,1]=(1.09595223135)/(o+1.)+(0.978251130983)+(0.804336219412)*0.5**o
        ref[1,2,2,0]=(-0.651012630849)/(o+1.)+(-1.04894606664)+(0.43528198181)*0.5**o
        ref[1,2,2,1]=(-1.5227353883)/(o+1.)+(0.670019307557)+(-0.745922598001)*0.5**o
        ref[1,3,0,0]=(-0.750195637997)/(o+1.)+(0.231864245348)+(-0.118356142533)*0.5**o
        ref[1,3,0,1]=(1.17136226596)/(o+1.)+(-0.114976421389)+(0.395383920468)*0.5**o
        ref[1,3,1,0]=(-0.108070636769)/(o+1.)+(0.279145429234)+(-0.352562621327)*0.5**o
        ref[1,3,1,1]=(-0.144223481714)/(o+1.)+(-0.472954352493)+(-0.0903026345013)*0.5**o
        ref[1,3,2,0]=(-0.487564307786)/(o+1.)+(0.636973351449)+(0.758622537094)*0.5**o
        ref[1,3,2,1]=(0.390860965954)/(o+1.)+(-0.36647678165)+(-0.66980953377)*0.5**o
        ref[1,4,0,0]=(-1.04824984837)/(o+1.)+(0.213932231728)+(0.823722790054)*0.5**o
        ref[1,4,0,1]=(-0.0984653601336)/(o+1.)+(0.07867877705)+(-0.741527483773)*0.5**o
        ref[1,4,1,0]=(-0.146287867383)/(o+1.)+(-0.908951868158)+(0.425691754878)*0.5**o
        ref[1,4,1,1]=(-0.94160416554)/(o+1.)+(0.0459877010016)+(-0.869872381431)*0.5**o
        ref[1,4,2,0]=(0.0115742858952)/(o+1.)+(-0.222488399041)+(0.687392452389)*0.5**o
        ref[1,4,2,1]=(-0.956561936575)/(o+1.)+(-0.59770879042)+(-0.0613652834159)*0.5**o
        ref[2,0,0,0]=(1.83987381729)/(o+1.)+(0.956579317963)+(0.862634531274)*0.5**o
        ref[2,0,0,1]=(0.105446127421)/(o+1.)+(0.180661881191)+(0.186629121375)*0.5**o
        ref[2,0,1,0]=(0.0212167440766)/(o+1.)+(-0.337825588888)+(-0.318943328859)*0.5**o
        ref[2,0,1,1]=(0.744834715631)/(o+1.)+(-0.350058416663)+(0.391466726104)*0.5**o
        ref[2,0,2,0]=(-0.788776613856)/(o+1.)+(0.233776806014)+(0.400043748043)*0.5**o
        ref[2,0,2,1]=(-0.95346976307)/(o+1.)+(0.0274998652324)+(-0.967971238003)*0.5**o
        ref[2,1,0,0]=(-0.536051408942)/(o+1.)+(-0.208498016994)+(0.590315312233)*0.5**o
        ref[2,1,0,1]=(-1.04158508788)/(o+1.)+(-0.397885793101)+(0.0127762955396)*0.5**o
        ref[2,1,1,0]=(0.529914207253)/(o+1.)+(-0.547503443987)+(-0.338873793431)*0.5**o
        ref[2,1,1,1]=(0.62298670274)/(o+1.)+(0.449860048786)+(0.352800726478)*0.5**o
        ref[2,1,2,0]=(0.965292650595)/(o+1.)+(0.00980498639)+(-0.261085873742)*0.5**o
        ref[2,1,2,1]=(0.403285360803)/(o+1.)+(0.369160592269)+(0.315744874774)*0.5**o
        ref[2,2,0,0]=(-0.457764884724)/(o+1.)+(0.563657776706)+(-0.791018976541)*0.5**o
        ref[2,2,0,1]=(0.188151274044)/(o+1.)+(-0.476470545911)+(0.911736564588)*0.5**o
        ref[2,2,1,0]=(-0.558515085739)/(o+1.)+(-0.445583796553)+(-0.533721159994)*0.5**o
        ref[2,2,1,1]=(-0.431992337265)/(o+1.)+(-0.0957726221961)+(0.64441532924)*0.5**o
        ref[2,2,2,0]=(-0.159056461947)/(o+1.)+(-0.0326864536156)+(0.484856087693)*0.5**o
        ref[2,2,2,1]=(0.533116456037)/(o+1.)+(1.10197257725)+(0.0353311991352)*0.5**o
        ref[2,3,0,0]=(0.0570625147023)/(o+1.)+(0.74170779854)+(0.851019670777)*0.5**o
        ref[2,3,0,1]=(-0.495059904832)/(o+1.)+(-0.169954912181)+(-0.761839138623)*0.5**o
        ref[2,3,1,0]=(0.482375173314)/(o+1.)+(-0.19374838651)+(0.706371811996)*0.5**o
        ref[2,3,1,1]=(0.12928100721)/(o+1.)+(-0.317327862062)+(-0.460946532077)*0.5**o
        ref[2,3,2,0]=(-0.234362084763)/(o+1.)+(0.423486279181)+(-0.93141377227)*0.5**o
        ref[2,3,2,1]=(1.25586693002)/(o+1.)+(-0.127551329993)+(0.56508002092)*0.5**o
        ref[2,4,0,0]=(0.45235297749)/(o+1.)+(-0.0217537364788)+(-0.640413752615)*0.5**o
        ref[2,4,0,1]=(0.936809207384)/(o+1.)+(-0.151519610695)+(-0.464139056331)*0.5**o
        ref[2,4,1,0]=(0.867693208863)/(o+1.)+(0.105683574314)+(-0.974513946233)*0.5**o
        ref[2,4,1,1]=(-1.27821575101)/(o+1.)+(0.463163847311)+(-0.0852880510814)*0.5**o
        ref[2,4,2,0]=(-0.846224501366)/(o+1.)+(0.432619461409)+(0.492845548002)*0.5**o
        ref[2,4,2,1]=(-1.48923229995)/(o+1.)+(-0.505747396375)+(0.00916577472118)*0.5**o
        ref[3,0,0,0]=(-0.959215731121)/(o+1.)+(-0.904609108587)+(-0.178668510268)*0.5**o
        ref[3,0,0,1]=(-1.53181231865)/(o+1.)+(-0.0758543610408)+(0.878717376536)*0.5**o
        ref[3,0,1,0]=(1.18643238831)/(o+1.)+(-0.288825159097)+(-0.341112240015)*0.5**o
        ref[3,0,1,1]=(-0.0675694917031)/(o+1.)+(-0.168955703616)+(0.275309686724)*0.5**o
        ref[3,0,2,0]=(-0.663269860133)/(o+1.)+(-0.235102421885)+(-0.573480403148)*0.5**o
        ref[3,0,2,1]=(1.0969289778)/(o+1.)+(0.46603172821)+(0.267894209932)*0.5**o
        ref[3,1,0,0]=(-0.716870513559)/(o+1.)+(0.712039409283)+(0.483098591635)*0.5**o
        ref[3,1,0,1]=(0.0328531999411)/(o+1.)+(0.379319450988)+(0.311973567308)*0.5**o
        ref[3,1,1,0]=(1.08648208064)/(o+1.)+(-0.196831959338)+(-0.452172980519)*0.5**o
        ref[3,1,1,1]=(0.0813181502612)/(o+1.)+(-0.227364064527)+(-0.0225656152548)*0.5**o
        ref[3,1,2,0]=(-0.271000454408)/(o+1.)+(-0.765711687155)+(0.868748696531)*0.5**o
        ref[3,1,2,1]=(1.60569646253)/(o+1.)+(0.137437739736)+(-0.00288127709197)*0.5**o
        ref[3,2,0,0]=(-1.25887089059)/(o+1.)+(-0.525842290945)+(0.455112921376)*0.5**o
        ref[3,2,0,1]=(-1.63787880725)/(o+1.)+(0.00214706725405)+(-0.166789085722)*0.5**o
        ref[3,2,1,0]=(-1.82111307894)/(o+1.)+(0.561806270366)+(0.388444921454)*0.5**o
        ref[3,2,1,1]=(-0.489259906414)/(o+1.)+(0.582548398381)+(0.417676141947)*0.5**o
        ref[3,2,2,0]=(-0.0674375340391)/(o+1.)+(-0.00345549227349)+(0.362946003887)*0.5**o
        ref[3,2,2,1]=(0.662780682001)/(o+1.)+(-0.15453848218)+(0.806022459534)*0.5**o
        ref[3,3,0,0]=(-1.15795406515)/(o+1.)+(0.0926725746467)+(0.289849779118)*0.5**o
        ref[3,3,0,1]=(1.27520269547)/(o+1.)+(-0.286529818361)+(0.672556345847)*0.5**o
        ref[3,3,1,0]=(-0.343290651469)/(o+1.)+(-0.0269962100904)+(-0.845847188189)*0.5**o
        ref[3,3,1,1]=(0.497715053371)/(o+1.)+(-0.158577177766)+(-0.909200821866)*0.5**o
        ref[3,3,2,0]=(-0.952431538662)/(o+1.)+(0.0816482063045)+(-0.580984656952)*0.5**o
        ref[3,3,2,1]=(-0.176227469799)/(o+1.)+(0.0456157853688)+(-0.0677878495767)*0.5**o
        ref[3,4,0,0]=(0.976460483802)/(o+1.)+(0.347595506011)+(0.766866468871)*0.5**o
        ref[3,4,0,1]=(0.0358170327662)/(o+1.)+(-0.0991456259778)+(0.931548717473)*0.5**o
        ref[3,4,1,0]=(0.114544318514)/(o+1.)+(-0.201014865792)+(-0.272213093635)*0.5**o
        ref[3,4,1,1]=(0.554509925385)/(o+1.)+(1.23219332557)+(0.237338557994)*0.5**o
        ref[3,4,2,0]=(0.0803465444931)/(o+1.)+(-0.518257569997)+(-0.681164419371)*0.5**o
        ref[3,4,2,1]=(-0.0443609570522)/(o+1.)+(0.572813368096)+(0.552725372202)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.570707549273)*x[0]+(-0.726264032186)*x[1]
        ref=(-0.0423765208873)/(o+1.)+(-0.791542656673)+(0.328490252775)*0.5**o
      else:
        arg=(1.28222892931)*x[0]+(-0.798170675576)*x[1]+(1.5701208405)*x[2]
        ref=(0.721975372868)/(o+1.)+(0.344720905832)+(0.642761909705)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.575166906798)*x[0]+(-1.63698773418)*x[1]
        arg[1]=(-0.0535946584636)*x[0]+(0.105969101775)*x[1]
        ref[0]=(-0.751926426344)/(o+1.)+(-0.395800138888)+(0.481705876734)*0.5**o
        ref[1]=(0.698496252242)/(o+1.)+(-0.265927161866)+(-0.114267485199)*0.5**o
      else:
        arg[0]=(-0.17879254138)*x[0]+(-1.12752123792)*x[1]+(0.921297311605)*x[2]
        arg[1]=(1.33131824618)*x[0]+(-0.392175059176)*x[1]+(0.545754281465)*x[2]
        ref[0]=(-0.494870303061)/(o+1.)+(0.0697636237654)+(-0.0296734121631)*0.5**o
        ref[1]=(-0.394300983704)/(o+1.)+(0.722261312754)+(0.434675826665)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.338658191734)*x[0]+(1.46127609683)*x[1]
        arg[0,1]=(-0.629116867491)*x[0]+(1.40141569066)*x[1]
        arg[0,2]=(0.105569779347)*x[0]+(0.239936324967)*x[1]
        arg[0,3]=(-0.281718331718)*x[0]+(0.280323404866)*x[1]
        arg[0,4]=(0.467341113661)*x[0]+(-0.50452062075)*x[1]
        arg[1,0]=(-0.841295219186)*x[0]+(0.524038163723)*x[1]
        arg[1,1]=(1.40143704236)*x[0]+(-1.50467961637)*x[1]
        arg[1,2]=(1.21078790505)*x[0]+(-0.731453470292)*x[1]
        arg[1,3]=(0.469556922076)*x[0]+(0.0686293204391)*x[1]
        arg[1,4]=(-1.26394637585)*x[0]+(-0.0361395276585)*x[1]
        arg[2,0]=(-1.12850883148)*x[0]+(0.284296673326)*x[1]
        arg[2,1]=(-0.626362828813)*x[0]+(1.19459827573)*x[1]
        arg[2,2]=(-0.545457293083)*x[0]+(1.547337019)*x[1]
        arg[2,3]=(0.664461986629)*x[0]+(-1.69082198266)*x[1]
        arg[2,4]=(-0.100004763305)*x[0]+(0.152159044423)*x[1]
        arg[3,0]=(0.307574843061)*x[0]+(1.20209181301)*x[1]
        arg[3,1]=(-0.0663203356627)*x[0]+(-1.26669863527)*x[1]
        arg[3,2]=(-1.20436798894)*x[0]+(-0.0375693379661)*x[1]
        arg[3,3]=(0.753649087747)*x[0]+(0.69376223486)*x[1]
        arg[3,4]=(-0.0512732264834)*x[0]+(-0.658097606436)*x[1]
        ref[0,0]=(0.996821439952)/(o+1.)+(0.015088160779)+(0.0956201435879)*0.5**o
        ref[0,1]=(0.643092963726)/(o+1.)+(0.239390724844)+(-0.349575590244)*0.5**o
        ref[0,2]=(-0.198468008977)/(o+1.)+(0.00411201743342)+(0.535750078425)*0.5**o
        ref[0,3]=(0.642491201137)/(o+1.)+(-0.623413199652)+(0.602940271315)*0.5**o
        ref[0,4]=(-0.105706120549)/(o+1.)+(-0.353560054939)+(0.775646723338)*0.5**o
        ref[1,0]=(0.144540669221)/(o+1.)+(0.0228268215803)+(-0.507451367845)*0.5**o
        ref[1,1]=(-0.820379339897)/(o+1.)+(0.0951825678509)+(0.526771630185)*0.5**o
        ref[1,2]=(-0.21214585824)/(o+1.)+(-0.0889183062985)+(0.869316905591)*0.5**o
        ref[1,3]=(0.291228940674)/(o+1.)+(0.192623928703)+(-0.138290555565)*0.5**o
        ref[1,4]=(-0.0663718956434)/(o+1.)+(-0.309484335614)+(-0.614745336642)*0.5**o
        ref[2,0]=(0.484892962201)/(o+1.)+(-0.367129859793)+(-0.594845400763)*0.5**o
        ref[2,1]=(0.768457634672)/(o+1.)+(0.154447657562)+(-0.509117502876)*0.5**o
        ref[2,2]=(0.707320338363)/(o+1.)+(-0.0592149232183)+(0.412989233992)*0.5**o
        ref[2,3]=(-0.983684339171)/(o+1.)+(-0.495377645598)+(0.948079634337)*0.5**o
        ref[2,4]=(-0.0315947220465)/(o+1.)+(0.506833986759)+(-0.929918970353)*0.5**o
        ref[3,0]=(0.54141067098)/(o+1.)+(0.489049131493)+(-0.00984227789859)*0.5**o
        ref[3,1]=(-0.552335053051)/(o+1.)+(-0.383325638803)+(-0.0140326402734)*0.5**o
        ref[3,2]=(0.0821472412575)/(o+1.)+(-0.335827719718)+(-0.652429128733)*0.5**o
        ref[3,3]=(0.112917208924)/(o+1.)+(0.667772807338)+(-0.00105150099312)*0.5**o
        ref[3,4]=(-0.19331508157)/(o+1.)+(-0.102296385708)+(-0.311462979932)*0.5**o
      else:
        arg[0,0]=(0.069245963142)*x[0]+(-0.185727379494)*x[1]+(-0.887410889699)*x[2]
        arg[0,1]=(-0.816745668949)*x[0]+(-0.250435680662)*x[1]+(0.701607830582)*x[2]
        arg[0,2]=(-0.862433954308)*x[0]+(0.731874372527)*x[1]+(-0.115384982779)*x[2]
        arg[0,3]=(-0.265247631604)*x[0]+(-0.157059741958)*x[1]+(-0.368419335078)*x[2]
        arg[0,4]=(1.66720203567)*x[0]+(-0.759393708719)*x[1]+(1.36360850704)*x[2]
        arg[1,0]=(-0.192822813795)*x[0]+(0.0358776152527)*x[1]+(-0.722864520788)*x[2]
        arg[1,1]=(-0.480606215471)*x[0]+(1.13682774573)*x[1]+(0.17900467623)*x[2]
        arg[1,2]=(0.346760852584)*x[0]+(0.994833030281)*x[1]+(0.507055933913)*x[2]
        arg[1,3]=(-0.509961853478)*x[0]+(-0.852997575505)*x[1]+(0.586527030253)*x[2]
        arg[1,4]=(0.93957975846)*x[0]+(0.28878335152)*x[1]+(0.128070931738)*x[2]
        arg[2,0]=(0.338283306313)*x[0]+(0.0244500761608)*x[1]+(-0.606875554888)*x[2]
        arg[2,1]=(0.850782624552)*x[0]+(-0.836382038189)*x[1]+(-1.41808459947)*x[2]
        arg[2,2]=(-1.06176229613)*x[0]+(-0.259247046061)*x[1]+(-0.583455484089)*x[2]
        arg[2,3]=(1.17437923282)*x[0]+(1.31485601495)*x[1]+(1.3139743231)*x[2]
        arg[2,4]=(0.362851636275)*x[0]+(0.804049783156)*x[1]+(1.09205000214)*x[2]
        arg[3,0]=(0.343603837195)*x[0]+(-1.49082257981)*x[1]+(-0.628668674545)*x[2]
        arg[3,1]=(-0.676847711765)*x[0]+(1.32350560453)*x[1]+(1.07275185686)*x[2]
        arg[3,2]=(0.570784985469)*x[0]+(-1.01212360602)*x[1]+(-0.0808776667606)*x[2]
        arg[3,3]=(0.506392444236)*x[0]+(-1.00848318041)*x[1]+(0.558777901541)*x[2]
        arg[3,4]=(-0.60925738291)*x[0]+(-0.386755957809)*x[1]+(0.60930115434)*x[2]
        ref[0,0]=(-0.695230136419)/(o+1.)+(-0.388400729083)+(0.468139288533)*0.5**o
        ref[0,1]=(0.462098132187)/(o+1.)+(-0.215041893287)+(-0.397587864642)*0.5**o
        ref[0,2]=(-0.262411385363)/(o+1.)+(0.275729226874)+(-0.534991632946)*0.5**o
        ref[0,3]=(0.117822633378)/(o+1.)+(0.0286827897592)+(-0.965914921536)*0.5**o
        ref[0,4]=(0.293575681039)/(o+1.)+(0.584039692827)+(0.809761767299)*0.5**o
        ref[1,0]=(0.627951922463)/(o+1.)+(-0.994889162557)+(0.482016683321)*0.5**o
        ref[1,1]=(1.83482238662)/(o+1.)+(-0.205719558739)+(-0.588157062647)*0.5**o
        ref[1,2]=(1.13612020398)/(o+1.)+(0.0262824938134)+(0.659964625174)*0.5**o
        ref[1,3]=(0.325264487741)/(o+1.)+(-0.0752154813413)+(-0.951265923787)*0.5**o
        ref[1,4]=(-0.149919283302)/(o+1.)+(0.77701502585)+(-0.0476767266789)*0.5**o
        ref[2,0]=(0.491504686171)/(o+1.)+(-0.562534706926)+(0.389422555265)*0.5**o
        ref[2,1]=(-1.52704217417)/(o+1.)+(0.120525042098)+(-0.117691923134)*0.5**o
        ref[2,2]=(-0.779796772543)/(o+1.)+(-0.318864997267)+(-0.486938059201)*0.5**o
        ref[2,3]=(1.53343318682)/(o+1.)+(0.731149467308)+(0.807477449443)*0.5**o
        ref[2,4]=(1.15919207694)/(o+1.)+(0.113921448815)+(0.871916446996)*0.5**o
        ref[3,0]=(-1.75312920944)/(o+1.)+(-0.153798982308)+(0.284839756895)*0.5**o
        ref[3,1]=(1.30985814569)/(o+1.)+(0.438430874237)+(-0.467310144534)*0.5**o
        ref[3,2]=(-0.737480320972)/(o+1.)+(0.0604403094397)+(0.094383414777)*0.5**o
        ref[3,3]=(-0.688723266524)/(o+1.)+(0.43486027018)+(-0.124310108473)*0.5**o
        ref[3,4]=(0.128399262674)/(o+1.)+(-0.372445844122)+(0.229780239192)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.993339371954)*x[0]+(-1.35308128985)*x[1]
        arg[0,0,1]=(-1.07886149134)*x[0]+(-1.38999283206)*x[1]
        arg[0,1,0]=(0.246468026972)*x[0]+(-1.37217464882)*x[1]
        arg[0,1,1]=(0.645672648599)*x[0]+(1.8070624103)*x[1]
        arg[1,0,0]=(-1.02278989388)*x[0]+(0.151634023103)*x[1]
        arg[1,0,1]=(-0.667332860541)*x[0]+(0.312742149193)*x[1]
        arg[1,1,0]=(-0.848647385832)*x[0]+(0.466438038516)*x[1]
        arg[1,1,1]=(-0.185495911372)*x[0]+(-0.137118786891)*x[1]
        arg[2,0,0]=(-0.0250555870731)*x[0]+(0.905553573283)*x[1]
        arg[2,0,1]=(-1.03940852939)*x[0]+(-1.57326016394)*x[1]
        arg[2,1,0]=(1.48834718811)*x[0]+(-0.340889253246)*x[1]
        arg[2,1,1]=(0.875721514739)*x[0]+(-0.310857412212)*x[1]
        arg[3,0,0]=(1.25129508831)*x[0]+(-1.69312043525)*x[1]
        arg[3,0,1]=(1.50124675312)*x[0]+(0.244195748654)*x[1]
        arg[3,1,0]=(0.292060478909)*x[0]+(-0.660005422696)*x[1]
        arg[3,1,1]=(0.791335231308)*x[0]+(0.466481573591)*x[1]
        arg[4,0,0]=(-0.361241947236)*x[0]+(-1.43591648587)*x[1]
        arg[4,0,1]=(-1.10385456695)*x[0]+(-0.325314195825)*x[1]
        arg[4,1,0]=(1.01004173033)*x[0]+(0.426824552857)*x[1]
        arg[4,1,1]=(-0.631382479612)*x[0]+(0.0603792190966)*x[1]
        arg[5,0,0]=(-0.30404368786)*x[0]+(-0.0102498522932)*x[1]
        arg[5,0,1]=(0.109188848746)*x[0]+(-1.16846160105)*x[1]
        arg[5,1,0]=(-0.351916879547)*x[0]+(-0.320714795448)*x[1]
        arg[5,1,1]=(-0.534138562652)*x[0]+(-0.256013589844)*x[1]
        ref[0,0,0]=(-0.506159375456)/(o+1.)+(-0.522895228783)+(-0.794470828783)*0.5**o
        ref[0,0,1]=(-0.930984367068)/(o+1.)+(-0.636851258685)+(-0.264167438968)*0.5**o
        ref[0,1,0]=(-0.674222792172)/(o+1.)+(-0.52131379048)+(0.591143751282)*0.5**o
        ref[0,1,1]=(0.953370751283)/(o+1.)+(0.377773224645)+(0.743817858326)*0.5**o
        ref[1,0,0]=(0.700508215428)/(o+1.)+(-0.303343530867)+(-0.964977024471)*0.5**o
        ref[1,0,1]=(-0.408185116452)/(o+1.)+(0.0668983569743)+(-0.0802023088445)*0.5**o
        ref[1,1,0]=(0.80742345527)/(o+1.)+(-0.405780705683)+(-0.37807139122)*0.5**o
        ref[1,1,1]=(0.0184615568095)/(o+1.)+(0.221921562266)+(-0.784919379605)*0.5**o
        ref[2,0,0]=(0.571906256883)/(o+1.)+(-0.0415588382133)+(0.391709405753)*0.5**o
        ref[2,0,1]=(-0.778176009451)/(o+1.)+(-0.761680990566)+(-0.311130702752)*0.5**o
        ref[2,1,0]=(0.498214686321)/(o+1.)+(-0.146173705483)+(0.941590659509)*0.5**o
        ref[2,1,1]=(-0.273747071818)/(o+1.)+(-0.0319184747153)+(0.902448123776)*0.5**o
        ref[3,0,0]=(-0.730029471178)/(o+1.)+(-0.288672747933)+(0.865549620109)*0.5**o
        ref[3,0,1]=(-0.371409186475)/(o+1.)+(0.623169375401)+(0.870512937445)*0.5**o
        ref[3,1,0]=(0.0611454391867)/(o+1.)+(-0.627723874214)+(0.826357365454)*0.5**o
        ref[3,1,1]=(0.785553447935)/(o+1.)+(0.246940213405)+(-0.021617069845)*0.5**o
        ref[4,0,0]=(-0.936201955907)/(o+1.)+(-0.484823720021)+(0.108690962842)*0.5**o
        ref[4,0,1]=(-0.277830390815)/(o+1.)+(-0.151786547071)+(-0.847765277819)*0.5**o
        ref[4,1,0]=(0.12293099533)/(o+1.)+(0.295620382054)+(0.722694523749)*0.5**o
        ref[4,1,1]=(-0.5517019185)/(o+1.)+(0.0850748937088)+(-0.189451129433)*0.5**o
        ref[5,0,0]=(-0.942533148805)/(o+1.)+(0.425680504872)+(-0.223121401092)*0.5**o
        ref[5,0,1]=(-0.377134228332)/(o+1.)+(-0.721019409569)+(0.759900295162)*0.5**o
        ref[5,1,0]=(-0.643828391942)/(o+1.)+(0.399974456311)+(-0.828752195674)*0.5**o
        ref[5,1,1]=(-0.118293731439)/(o+1.)+(-0.355568544431)+(0.0392786678054)*0.5**o
      else:
        arg[0,0,0]=(0.205446621454)*x[0]+(0.10254878667)*x[1]+(1.27673728979)*x[2]
        arg[0,0,1]=(0.758846313193)*x[0]+(0.63320336713)*x[1]+(-0.42638505807)*x[2]
        arg[0,1,0]=(-0.0127116561603)*x[0]+(-0.360359420027)*x[1]+(0.540508591493)*x[2]
        arg[0,1,1]=(-0.278918629538)*x[0]+(-0.0291534835772)*x[1]+(0.708655731565)*x[2]
        arg[1,0,0]=(-0.612965210483)*x[0]+(0.787787556912)*x[1]+(0.222384448462)*x[2]
        arg[1,0,1]=(1.07409083742)*x[0]+(1.19429173706)*x[1]+(-0.484979976779)*x[2]
        arg[1,1,0]=(-1.25714999157)*x[0]+(-0.321346019622)*x[1]+(0.278329320837)*x[2]
        arg[1,1,1]=(-0.512262689707)*x[0]+(1.6891992053)*x[1]+(-1.02351830068)*x[2]
        arg[2,0,0]=(-1.57911907773)*x[0]+(0.096124258122)*x[1]+(0.336619253521)*x[2]
        arg[2,0,1]=(0.0901144254199)*x[0]+(-1.36818186255)*x[1]+(1.2969609239)*x[2]
        arg[2,1,0]=(0.158329511499)*x[0]+(0.696997185217)*x[1]+(-0.400535235927)*x[2]
        arg[2,1,1]=(-1.44570853201)*x[0]+(0.237359587195)*x[1]+(-1.17323024801)*x[2]
        arg[3,0,0]=(-1.4325812824)*x[0]+(-0.733085265902)*x[1]+(0.125883186484)*x[2]
        arg[3,0,1]=(-0.0529889021435)*x[0]+(0.335154600942)*x[1]+(0.839337375209)*x[2]
        arg[3,1,0]=(-0.423552182963)*x[0]+(1.31813289315)*x[1]+(0.13235854915)*x[2]
        arg[3,1,1]=(0.53278572217)*x[0]+(0.159434073071)*x[1]+(-0.313621199136)*x[2]
        arg[4,0,0]=(0.747779887673)*x[0]+(0.30977483911)*x[1]+(-0.0760295896739)*x[2]
        arg[4,0,1]=(0.00295118909194)*x[0]+(-0.687914867842)*x[1]+(0.0386268422915)*x[2]
        arg[4,1,0]=(-0.276363532754)*x[0]+(0.52102466785)*x[1]+(-0.684840191917)*x[2]
        arg[4,1,1]=(0.0283138898538)*x[0]+(0.116042458294)*x[1]+(0.485578401528)*x[2]
        arg[5,0,0]=(-0.118402060192)*x[0]+(-1.18409809944)*x[1]+(-0.304048852342)*x[2]
        arg[5,0,1]=(0.174393790186)*x[0]+(-0.524912689263)*x[1]+(-0.648708392355)*x[2]
        arg[5,1,0]=(-0.254958756344)*x[0]+(-0.772071348728)*x[1]+(-0.0572494064027)*x[2]
        arg[5,1,1]=(-0.74777072018)*x[0]+(0.19109939364)*x[1]+(0.498973987814)*x[2]
        ref[0,0,0]=(0.986457444152)/(o+1.)+(0.258994989412)+(0.0802852749346)*0.5**o
        ref[0,0,1]=(-0.0286717143415)/(o+1.)+(0.554867138611)+(-0.115397940627)*0.5**o
        ref[0,1,0]=(-0.886482131138)/(o+1.)+(0.464293129749)+(0.125333386946)*0.5**o
        ref[0,1,1]=(0.199214164457)/(o+1.)+(0.336882900218)+(-0.472396346443)*0.5**o
        ref[1,0,0]=(0.36021680954)/(o+1.)+(0.408903924703)+(-0.780817864055)*0.5**o
        ref[1,0,1]=(0.44795581074)/(o+1.)+(0.472159267742)+(0.391128251477)*0.5**o
        ref[1,1,0]=(1.02665734893)/(o+1.)+(-0.71418625728)+(-0.898451524719)*0.5**o
        ref[1,1,1]=(0.0322676607777)/(o+1.)+(0.105378958385)+(-0.0896073626306)*0.5**o
        ref[2,0,0]=(1.15723247637)/(o+1.)+(-0.719604538702)+(-0.86439896505)*0.5**o
        ref[2,0,1]=(-0.254100181576)/(o+1.)+(-0.0387997401585)+(0.35059314866)*0.5**o
        ref[2,1,0]=(-0.636439167982)/(o+1.)+(0.563390681981)+(-0.0355507351918)*0.5**o
        ref[2,1,1]=(-0.265265225142)/(o+1.)+(-0.635620798949)+(-0.845072369786)*0.5**o
        ref[3,0,0]=(-0.062162800164)/(o+1.)+(-0.554499928282)+(-0.868620705087)*0.5**o
        ref[3,0,1]=(0.784681961813)/(o+1.)+(0.355202655131)+(-0.373584198067)*0.5**o
        ref[3,1,0]=(1.20272043429)/(o+1.)+(0.352677694103)+(-0.881136563162)*0.5**o
        ref[3,1,1]=(-0.311152797156)/(o+1.)+(0.313736563953)+(0.0622782653549)*0.5**o
        ref[4,0,0]=(-0.365517620238)/(o+1.)+(0.733502174253)+(-0.119961591158)*0.5**o
        ref[4,0,1]=(0.0155845896619)/(o+1.)+(-0.654963543839)+(0.648005661558)*0.5**o
        ref[4,1,0]=(0.405697975915)/(o+1.)+(-0.637805133954)+(0.429733235172)*0.5**o
        ref[4,1,1]=(-0.0469738326054)/(o+1.)+(0.0590721074184)+(0.558764367445)*0.5**o
        ref[5,0,0]=(-0.333825007954)/(o+1.)+(-0.46448520882)+(-0.343753586377)*0.5**o
        ref[5,0,1]=(-0.469125954795)/(o+1.)+(-0.147192309255)+(-0.235716718127)*0.5**o
        ref[5,1,0]=(-0.573275891777)/(o+1.)+(-0.478328163121)+(0.445652706545)*0.5**o
        ref[5,1,1]=(0.0545515234726)/(o+1.)+(-0.153919681346)+(0.195590500493)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=1
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.226142137728)*x[0]+(0.85681895183)*x[1]
        arg[0,0,0,1]=(-0.824407181762)*x[0]+(-0.290023144892)*x[1]
        arg[0,0,1,0]=(-1.25987816017)*x[0]+(0.824340990595)*x[1]
        arg[0,0,1,1]=(-0.0917811606765)*x[0]+(0.660211932964)*x[1]
        arg[0,0,2,0]=(-1.31073149178)*x[0]+(0.482324984918)*x[1]
        arg[0,0,2,1]=(0.256995458227)*x[0]+(0.79244254852)*x[1]
        arg[0,1,0,0]=(-0.0682147475407)*x[0]+(0.243620052061)*x[1]
        arg[0,1,0,1]=(0.071099982687)*x[0]+(-0.62986754897)*x[1]
        arg[0,1,1,0]=(-0.0716793733131)*x[0]+(-1.15235525814)*x[1]
        arg[0,1,1,1]=(0.193107575216)*x[0]+(-0.596149957543)*x[1]
        arg[0,1,2,0]=(-0.160142629886)*x[0]+(1.36581707067)*x[1]
        arg[0,1,2,1]=(-1.21477648376)*x[0]+(-0.732000298369)*x[1]
        arg[0,2,0,0]=(-1.02468971059)*x[0]+(0.330721595217)*x[1]
        arg[0,2,0,1]=(1.11414907391)*x[0]+(1.29386127812)*x[1]
        arg[0,2,1,0]=(1.1485850154)*x[0]+(0.387585977495)*x[1]
        arg[0,2,1,1]=(0.237382583072)*x[0]+(1.56248221279)*x[1]
        arg[0,2,2,0]=(-0.949129049229)*x[0]+(0.578823379063)*x[1]
        arg[0,2,2,1]=(-1.0667182537)*x[0]+(0.462227485959)*x[1]
        arg[0,3,0,0]=(1.37954377554)*x[0]+(0.541856880139)*x[1]
        arg[0,3,0,1]=(0.401584700319)*x[0]+(0.577102327147)*x[1]
        arg[0,3,1,0]=(-0.289796408108)*x[0]+(0.807853785325)*x[1]
        arg[0,3,1,1]=(0.0038467861822)*x[0]+(-0.931311510771)*x[1]
        arg[0,3,2,0]=(0.859036254821)*x[0]+(0.198884218516)*x[1]
        arg[0,3,2,1]=(-0.00310256070968)*x[0]+(0.511838475968)*x[1]
        arg[0,4,0,0]=(0.539686886802)*x[0]+(-0.324612924267)*x[1]
        arg[0,4,0,1]=(-0.0458655898565)*x[0]+(0.211810380966)*x[1]
        arg[0,4,1,0]=(0.302333533003)*x[0]+(0.0663033316831)*x[1]
        arg[0,4,1,1]=(-0.981519161639)*x[0]+(-0.00207756909914)*x[1]
        arg[0,4,2,0]=(0.283659447004)*x[0]+(-0.231526592979)*x[1]
        arg[0,4,2,1]=(-1.17679167875)*x[0]+(0.285698985756)*x[1]
        arg[1,0,0,0]=(-0.162900059582)*x[0]+(-0.877164340783)*x[1]
        arg[1,0,0,1]=(-0.50468605065)*x[0]+(-0.160440073709)*x[1]
        arg[1,0,1,0]=(-0.100901676644)*x[0]+(-0.296111526029)*x[1]
        arg[1,0,1,1]=(0.176107421869)*x[0]+(1.74393708745)*x[1]
        arg[1,0,2,0]=(-0.979807721703)*x[0]+(-0.229400841067)*x[1]
        arg[1,0,2,1]=(0.217472795645)*x[0]+(0.0881503243323)*x[1]
        arg[1,1,0,0]=(-0.127980123625)*x[0]+(0.0932095456541)*x[1]
        arg[1,1,0,1]=(0.829674410931)*x[0]+(-0.796196041601)*x[1]
        arg[1,1,1,0]=(1.28656110233)*x[0]+(-1.27850471392)*x[1]
        arg[1,1,1,1]=(1.159269884)*x[0]+(-0.973135433631)*x[1]
        arg[1,1,2,0]=(1.46274837856)*x[0]+(-1.81554270863)*x[1]
        arg[1,1,2,1]=(0.0973472511688)*x[0]+(0.551451261495)*x[1]
        arg[1,2,0,0]=(0.510667671293)*x[0]+(-0.51075982099)*x[1]
        arg[1,2,0,1]=(-1.41612093451)*x[0]+(0.614417669629)*x[1]
        arg[1,2,1,0]=(1.46595125796)*x[0]+(0.60374279122)*x[1]
        arg[1,2,1,1]=(-0.317185218187)*x[0]+(-0.0785532882696)*x[1]
        arg[1,2,2,0]=(-0.388504041993)*x[0]+(-0.838003398775)*x[1]
        arg[1,2,2,1]=(0.0497207129012)*x[0]+(-0.622155692247)*x[1]
        arg[1,3,0,0]=(-0.0471400179563)*x[0]+(1.3378948207)*x[1]
        arg[1,3,0,1]=(-1.22250822195)*x[0]+(0.629714783138)*x[1]
        arg[1,3,1,0]=(-1.28801080408)*x[0]+(-1.54079710038)*x[1]
        arg[1,3,1,1]=(-0.426412448469)*x[0]+(-1.09410474021)*x[1]
        arg[1,3,2,0]=(0.022834167351)*x[0]+(-0.497423389788)*x[1]
        arg[1,3,2,1]=(-0.787492124301)*x[0]+(-0.245621168419)*x[1]
        arg[1,4,0,0]=(-1.37814880919)*x[0]+(-0.704805848451)*x[1]
        arg[1,4,0,1]=(-1.12494770055)*x[0]+(1.44920024305)*x[1]
        arg[1,4,1,0]=(-0.809532572761)*x[0]+(1.16638587657)*x[1]
        arg[1,4,1,1]=(0.931784486816)*x[0]+(1.34629889453)*x[1]
        arg[1,4,2,0]=(-0.172430942793)*x[0]+(-0.584736998321)*x[1]
        arg[1,4,2,1]=(0.00542948736389)*x[0]+(0.43917597284)*x[1]
        arg[2,0,0,0]=(-0.946122337406)*x[0]+(-0.161940168137)*x[1]
        arg[2,0,0,1]=(-0.7856653779)*x[0]+(-0.75287701788)*x[1]
        arg[2,0,1,0]=(-1.08858367592)*x[0]+(-0.530380818694)*x[1]
        arg[2,0,1,1]=(-1.0573555095)*x[0]+(-0.854148765785)*x[1]
        arg[2,0,2,0]=(0.723412825328)*x[0]+(0.326761950437)*x[1]
        arg[2,0,2,1]=(-0.0143142161225)*x[0]+(-0.752724311876)*x[1]
        arg[2,1,0,0]=(-0.942548092355)*x[0]+(0.700861094897)*x[1]
        arg[2,1,0,1]=(-0.712286567957)*x[0]+(0.230406545738)*x[1]
        arg[2,1,1,0]=(1.27414839774)*x[0]+(0.724784649037)*x[1]
        arg[2,1,1,1]=(0.637494163002)*x[0]+(0.442489599192)*x[1]
        arg[2,1,2,0]=(1.81617631726)*x[0]+(1.56870218809)*x[1]
        arg[2,1,2,1]=(1.16419900666)*x[0]+(-0.044314729141)*x[1]
        arg[2,2,0,0]=(0.880267125196)*x[0]+(0.77124406259)*x[1]
        arg[2,2,0,1]=(0.482402867542)*x[0]+(0.0746924305017)*x[1]
        arg[2,2,1,0]=(0.105861259696)*x[0]+(0.484292207267)*x[1]
        arg[2,2,1,1]=(-0.107785838196)*x[0]+(0.701132243209)*x[1]
        arg[2,2,2,0]=(0.559717405844)*x[0]+(-1.5585942717)*x[1]
        arg[2,2,2,1]=(-0.989130712605)*x[0]+(0.221361137556)*x[1]
        arg[2,3,0,0]=(-0.045471818095)*x[0]+(-0.437941959792)*x[1]
        arg[2,3,0,1]=(1.25840468379)*x[0]+(-1.18872011057)*x[1]
        arg[2,3,1,0]=(1.70665829408)*x[0]+(0.705273921003)*x[1]
        arg[2,3,1,1]=(-0.0928709253845)*x[0]+(0.785115449756)*x[1]
        arg[2,3,2,0]=(-0.887243418838)*x[0]+(1.06790660642)*x[1]
        arg[2,3,2,1]=(-0.572644686872)*x[0]+(-0.672831948394)*x[1]
        arg[2,4,0,0]=(1.28282566801)*x[0]+(0.651985199447)*x[1]
        arg[2,4,0,1]=(0.0965663229335)*x[0]+(1.33819998052)*x[1]
        arg[2,4,1,0]=(0.205569041098)*x[0]+(-0.873095358733)*x[1]
        arg[2,4,1,1]=(-1.04452025745)*x[0]+(-0.204917033907)*x[1]
        arg[2,4,2,0]=(-0.181163938972)*x[0]+(0.0341993409272)*x[1]
        arg[2,4,2,1]=(1.68327027059)*x[0]+(-0.289436355642)*x[1]
        arg[3,0,0,0]=(1.05712810731)*x[0]+(0.915711819735)*x[1]
        arg[3,0,0,1]=(-0.171072039482)*x[0]+(-0.160656943623)*x[1]
        arg[3,0,1,0]=(-1.27063799869)*x[0]+(-0.605131642813)*x[1]
        arg[3,0,1,1]=(0.940512990835)*x[0]+(-0.116675577531)*x[1]
        arg[3,0,2,0]=(-0.194213869952)*x[0]+(0.364066752025)*x[1]
        arg[3,0,2,1]=(-0.328780465397)*x[0]+(-0.114689413673)*x[1]
        arg[3,1,0,0]=(-0.825914305895)*x[0]+(-0.167010056731)*x[1]
        arg[3,1,0,1]=(0.657779056207)*x[0]+(0.0666252451638)*x[1]
        arg[3,1,1,0]=(-0.464004426036)*x[0]+(-0.340932918409)*x[1]
        arg[3,1,1,1]=(-1.6698652667)*x[0]+(0.731870373141)*x[1]
        arg[3,1,2,0]=(1.39626250779)*x[0]+(-0.879152855557)*x[1]
        arg[3,1,2,1]=(1.34973919508)*x[0]+(-0.129753332761)*x[1]
        arg[3,2,0,0]=(1.03443450706)*x[0]+(0.807333477811)*x[1]
        arg[3,2,0,1]=(0.366011302596)*x[0]+(0.543804800926)*x[1]
        arg[3,2,1,0]=(0.262446846036)*x[0]+(-0.247786850991)*x[1]
        arg[3,2,1,1]=(-1.04657091196)*x[0]+(-0.485443426155)*x[1]
        arg[3,2,2,0]=(0.342434327538)*x[0]+(-0.148503585588)*x[1]
        arg[3,2,2,1]=(-0.189833429716)*x[0]+(0.424391874819)*x[1]
        arg[3,3,0,0]=(-0.118348838118)*x[0]+(0.397628264341)*x[1]
        arg[3,3,0,1]=(0.481440510084)*x[0]+(-0.393576221424)*x[1]
        arg[3,3,1,0]=(0.907533552208)*x[0]+(-0.668646549376)*x[1]
        arg[3,3,1,1]=(-1.49731714213)*x[0]+(-1.18710678264)*x[1]
        arg[3,3,2,0]=(-0.460660079207)*x[0]+(1.48078266444)*x[1]
        arg[3,3,2,1]=(-0.161073556988)*x[0]+(-0.486827042754)*x[1]
        arg[3,4,0,0]=(-0.668755455623)*x[0]+(0.345804059434)*x[1]
        arg[3,4,0,1]=(1.00528889193)*x[0]+(0.978507928191)*x[1]
        arg[3,4,1,0]=(0.98955856331)*x[0]+(1.0421852144)*x[1]
        arg[3,4,1,1]=(0.711688260873)*x[0]+(1.39962222322)*x[1]
        arg[3,4,2,0]=(1.08512429209)*x[0]+(-0.555249097743)*x[1]
        arg[3,4,2,1]=(-1.10443784007)*x[0]+(-0.782274429169)*x[1]
        ref[0,0,0,0]=(0.151153165564)/(o+1.)+(0.112609656739)+(0.254304335059)*0.5**o
        ref[0,0,0,1]=(-0.530602583511)/(o+1.)+(0.1817855378)+(-0.947398818743)*0.5**o
        ref[0,0,1,0]=(0.308747437738)/(o+1.)+(-0.0338129255643)+(-0.676658756184)*0.5**o
        ref[0,0,1,1]=(0.321026553427)/(o+1.)+(0.0692705367618)+(0.108863145337)*0.5**o
        ref[0,0,2,0]=(-0.0908034666655)/(o+1.)+(-0.1100205155)+(-0.517562009199)*0.5**o
        ref[0,0,2,1]=(0.303456040903)/(o+1.)+(0.500090875336)+(-0.254199784829)*0.5**o
        ref[0,1,0,0]=(-0.440641221831)/(o+1.)+(0.456218946873)+(-0.296391367396)*0.5**o
        ref[0,1,0,1]=(0.222628561762)/(o+1.)+(-0.162064412854)+(-0.457267302337)*0.5**o
        ref[0,1,1,0]=(-0.913361887408)/(o+1.)+(-0.356168674239)+(0.401664604431)*0.5**o
        ref[0,1,1,1]=(-0.543116639248)/(o+1.)+(-0.120902439422)+(0.381879135767)*0.5**o
        ref[0,1,2,0]=(0.993649262831)/(o+1.)+(0.384695858206)+(-0.55736653846)*0.5**o
        ref[0,1,2,1]=(-0.968077797451)/(o+1.)+(-0.187765942637)+(-0.603167099406)*0.5**o
        ref[0,2,0,0]=(0.873662019918)/(o+1.)+(-0.710635360033)+(-0.146359415223)*0.5**o
        ref[0,2,0,1]=(0.466408230693)/(o+1.)+(0.77626565285)+(0.389070815631)*0.5**o
        ref[0,2,1,0]=(-0.475010813579)/(o+1.)+(0.525435880006)+(0.960310046461)*0.5**o
        ref[0,2,1,1]=(0.697127520484)/(o+1.)+(0.357204276552)+(0.388328722275)*0.5**o
        ref[0,2,2,0]=(-0.108434738487)/(o+1.)+(0.356443300466)+(-0.97475753261)*0.5**o
        ref[0,2,2,1]=(0.952274964304)/(o+1.)+(-0.497379508976)+(-0.562006714089)*0.5**o
        ref[0,3,0,0]=(-0.0368012761671)/(o+1.)+(0.594000625518)+(0.770200680808)*0.5**o
        ref[0,3,0,1]=(0.450361861836)/(o+1.)+(0.182031052848)+(0.164263059934)*0.5**o
        ref[0,3,1,0]=(0.500714741956)/(o+1.)+(0.0479711343168)+(-0.0785996333735)*0.5**o
        ref[0,3,1,1]=(-0.949280938645)/(o+1.)+(0.452441874504)+(-0.88306753495)*0.5**o
        ref[0,3,2,0]=(0.186931712837)/(o+1.)+(0.222335175903)+(0.426318408694)*0.5**o
        ref[0,3,2,1]=(-0.288015428918)/(o+1.)+(0.80948657963)+(-0.822221815085)*0.5**o
        ref[0,4,0,0]=(0.670151184695)/(o+1.)+(-0.415883847013)+(0.376690471866)*0.5**o
        ref[0,4,0,1]=(-0.107106843992)/(o+1.)+(0.555375298435)+(-0.837698961768)*0.5**o
        ref[0,4,1,0]=(-0.299999096802)/(o+1.)+(0.668996782774)+(-0.669357604061)*0.5**o
        ref[0,4,1,1]=(0.778006183401)/(o+1.)+(-0.614473529032)+(-0.532655856076)*0.5**o
        ref[0,4,2,0]=(-0.250068389818)/(o+1.)+(0.156274314197)+(-0.0103473845504)*0.5**o
        ref[0,4,2,1]=(0.902125322216)/(o+1.)+(-0.616424567683)+(-0.560368879846)*0.5**o
        ref[1,0,0,0]=(-0.133690258295)/(o+1.)+(-0.715949360249)+(0.525524578427)*0.5**o
        ref[1,0,0,1]=(-0.788097689216)/(o+1.)+(0.0998405659256)+(-0.076709566994)*0.5**o
        ref[1,0,1,0]=(0.144280886187)/(o+1.)+(0.0775377290314)+(-0.696369546924)*0.5**o
        ref[1,0,1,1]=(0.824172196605)/(o+1.)+(0.561804029444)+(-0.0277357461773)*0.5**o
        ref[1,0,2,0]=(0.422564421802)/(o+1.)+(-0.60719307099)+(-0.417386842591)*0.5**o
        ref[1,0,2,1]=(0.778908672762)/(o+1.)+(0.0163446340765)+(-0.505974820938)*0.5**o
        ref[1,1,0,0]=(0.0574071047387)/(o+1.)+(-0.350514214742)+(0.608850746775)*0.5**o
        ref[1,1,0,1]=(-0.991660961792)/(o+1.)+(0.322084279455)+(0.380970772212)*0.5**o
        ref[1,1,1,0]=(-0.823761120268)/(o+1.)+(-0.0495419092898)+(0.930901327253)*0.5**o
        ref[1,1,1,1]=(-0.384729980569)/(o+1.)+(-0.0392583789163)+(0.649381188769)*0.5**o
        ref[1,1,2,0]=(-0.840365304069)/(o+1.)+(-0.0150512662777)+(0.517673506558)*0.5**o
        ref[1,1,2,1]=(0.555976660846)/(o+1.)+(-0.307265886898)+(0.707353625614)*0.5**o
        ref[1,2,0,0]=(-0.423056705491)/(o+1.)+(0.0559404393793)+(0.311083677036)*0.5**o
        ref[1,2,0,1]=(0.15409797664)/(o+1.)+(-0.0906637067519)+(-0.774473828018)*0.5**o
        ref[1,2,1,0]=(0.796718644088)/(o+1.)+(0.248012077098)+(0.776951250898)*0.5**o
        ref[1,2,1,1]=(-0.342830889263)/(o+1.)+(-0.282162737841)+(0.511417858487)*0.5**o
        ref[1,2,2,0]=(-0.722403016671)/(o+1.)+(-0.367059457954)+(0.230014491812)*0.5**o
        ref[1,2,2,1]=(-0.773685058881)/(o+1.)+(0.412807900451)+(-0.624365721366)*0.5**o
        ref[1,3,0,0]=(0.527481434758)/(o+1.)+(0.36602961117)+(0.031214145649)*0.5**o
        ref[1,3,0,1]=(0.0630927531585)/(o+1.)+(-0.164989701995)+(-0.325906787975)*0.5**o
        ref[1,3,1,0]=(-0.612390590593)/(o+1.)+(-0.78855996274)+(-0.63929738839)*0.5**o
        ref[1,3,1,1]=(-0.962656044265)/(o+1.)+(-0.00576535037253)+(-0.546330443673)*0.5**o
        ref[1,3,2,0]=(-0.947280922008)/(o+1.)+(-0.137449579996)+(0.747590859563)*0.5**o
        ref[1,3,2,1]=(-0.946623026202)/(o+1.)+(0.0787536550213)+(-0.24399757656)*0.5**o
        ref[1,4,0,0]=(-0.174471016943)/(o+1.)+(-0.636500556039)+(-0.635482528623)*0.5**o
        ref[1,4,0,1]=(0.806379391595)/(o+1.)+(-0.00304106577093)+(-0.476044717545)*0.5**o
        ref[1,4,1,0]=(0.729649125134)/(o+1.)+(0.0151477051616)+(-0.403091231646)*0.5**o
        ref[1,4,1,1]=(0.559976454745)/(o+1.)+(0.479295896421)+(0.75951513376)*0.5**o
        ref[1,4,2,0]=(0.272199270818)/(o+1.)+(-0.0929072274332)+(-0.843552757065)*0.5**o
        ref[1,4,2,1]=(-0.552663678377)/(o+1.)+(0.613784745778)+(-0.230300352975)*0.5**o
        ref[2,0,0,0]=(-0.174247908098)/(o+1.)+(-0.108873783282)+(-0.716067030882)*0.5**o
        ref[2,0,0,1]=(-0.300584841532)/(o+1.)+(-0.485363665493)+(-0.267230223262)*0.5**o
        ref[2,0,1,0]=(0.316261805154)/(o+1.)+(-0.523474106451)+(-0.888278086869)*0.5**o
        ref[2,0,1,1]=(-0.699622562861)/(o+1.)+(-0.151089538248)+(-0.909702635929)*0.5**o
        ref[2,0,2,0]=(0.728545639829)/(o+1.)+(0.192083148877)+(-0.0625371618174)*0.5**o
        ref[2,0,2,1]=(-0.22042267239)/(o+1.)+(-0.611974599324)+(0.677333343041)*0.5**o
        ref[2,1,0,0]=(-0.0734883811113)/(o+1.)+(0.36100831022)+(-0.890215236787)*0.5**o
        ref[2,1,0,1]=(0.121656829057)/(o+1.)+(-0.130801635799)+(-0.341933579678)*0.5**o
        ref[2,1,1,0]=(0.221353168629)/(o+1.)+(0.618635673483)+(0.540308531179)*0.5**o
        ref[2,1,1,1]=(0.0152580531273)/(o+1.)+(0.252173415542)+(0.560378877984)*0.5**o
        ref[2,1,2,0]=(0.580319777731)/(o+1.)+(0.90566612137)+(0.993226484879)*0.5**o
        ref[2,1,2,1]=(-0.710250320949)/(o+1.)+(0.742177021871)+(0.345780554724)*0.5**o
        ref[2,2,0,0]=(0.770701563507)/(o+1.)+(0.240823471881)+(0.399162680517)*0.5**o
        ref[2,2,0,1]=(0.76121761761)/(o+1.)+(0.128382656738)+(-0.460887633042)*0.5**o
        ref[2,2,1,0]=(-0.269854962057)/(o+1.)+(0.690854898294)+(-0.521701367567)*0.5**o
        ref[2,2,1,1]=(0.719518595712)/(o+1.)+(0.029596430244)+(-0.185365051187)*0.5**o
        ref[2,2,2,0]=(-0.671157761165)/(o+1.)+(-0.142688285012)+(-0.0423425346644)*0.5**o
        ref[2,2,2,1]=(-0.541156097112)/(o+1.)+(0.0765180918929)+(-0.379649661722)*0.5**o
        ref[2,3,0,0]=(-0.0604347109212)/(o+1.)+(-0.19403702073)+(-0.034905025507)*0.5**o
        ref[2,3,0,1]=(-0.793398507071)/(o+1.)+(0.255681280626)+(0.351720519037)*0.5**o
        ref[2,3,1,0]=(-0.267058413731)/(o+1.)+(0.852211426049)+(0.974567776717)*0.5**o
        ref[2,3,1,1]=(0.228396641207)/(o+1.)+(0.322922896735)+(-0.181997910305)*0.5**o
        ref[2,3,2,0]=(0.480987773937)/(o+1.)+(-0.197709028569)+(0.0950934707835)*0.5**o
        ref[2,3,2,1]=(-0.648934475027)/(o+1.)+(-0.418383297875)+(0.240224435511)*0.5**o
        ref[2,4,0,0]=(0.430020261528)/(o+1.)+(0.53421018733)+(0.436370231269)*0.5**o
        ref[2,4,0,1]=(0.906390660223)/(o+1.)+(-0.056057545856)+(0.640490734948)*0.5**o
        ref[2,4,1,0]=(-0.732099770943)/(o+1.)+(-0.0485027617732)+(0.161578976854)*0.5**o
        ref[2,4,1,1]=(-0.827495680848)/(o+1.)+(0.285044728936)+(-0.992031068384)*0.5**o
        ref[2,4,2,0]=(-0.919905471787)/(o+1.)+(0.656971386264)+(-0.541001898785)*0.5**o
        ref[2,4,2,1]=(0.161975875302)/(o+1.)+(0.272665206557)+(0.686527626533)*0.5**o
        ref[3,0,0,0]=(0.584120963259)/(o+1.)+(0.207930729804)+(0.972857504177)*0.5**o
        ref[3,0,0,1]=(0.315823141491)/(o+1.)+(-0.217561208386)+(-0.212429707823)*0.5**o
        ref[3,0,1,0]=(-0.732294996139)/(o+1.)+(-0.278653143596)+(-0.586168358175)*0.5**o
        ref[3,0,1,1]=(0.481321847225)/(o+1.)+(-0.070080648792)+(0.482676863662)*0.5**o
        ref[3,0,2,0]=(0.616679859076)/(o+1.)+(-0.418784546677)+(0.390742116352)*0.5**o
        ref[3,0,2,1]=(0.866778417448)/(o+1.)+(-0.686585656643)+(0.0629230167692)*0.5**o
        ref[3,1,0,0]=(0.531890054167)/(o+1.)+(-0.516187607633)+(-0.492439201527)*0.5**o
        ref[3,1,0,1]=(-0.512361639413)/(o+1.)+(0.499557182545)+(0.237651575694)*0.5**o
        ref[3,1,1,0]=(-0.67542965405)/(o+1.)+(0.274414249559)+(-0.678336189514)*0.5**o
        ref[3,1,1,1]=(0.371936242507)/(o+1.)+(-0.1636482466)+(-0.982634642862)*0.5**o
        ref[3,1,2,0]=(-0.799408831501)/(o+1.)+(0.458633399549)+(0.399251684639)*0.5**o
        ref[3,1,2,1]=(-0.0145253903686)/(o+1.)+(0.238472345905)+(0.757566560882)*0.5**o
        ref[3,2,0,0]=(-0.130519473112)/(o+1.)+(0.695277604837)+(0.581732248306)*0.5**o
        ref[3,2,0,1]=(0.632010536438)/(o+1.)+(0.123066205529)+(0.0316731560259)*0.5**o
        ref[3,2,1,0]=(-0.709926210356)/(o+1.)+(0.0926699972302)+(0.53924621094)*0.5**o
        ref[3,2,1,1]=(0.428094002247)/(o+1.)+(-0.841997926131)+(-0.276112488096)*0.5**o
        ref[3,2,2,0]=(0.708481148086)/(o+1.)+(-0.50498161534)+(0.495412824545)*0.5**o
        ref[3,2,2,1]=(0.782824622703)/(o+1.)+(-0.151509698687)+(-0.245246780227)*0.5**o
        ref[3,3,0,0]=(0.848460868418)/(o+1.)+(-0.104639248461)+(-0.359902945273)*0.5**o
        ref[3,3,0,1]=(0.0569151307298)/(o+1.)+(-0.0674224535469)+(0.165794065024)*0.5**o
        ref[3,3,1,0]=(-0.165683739908)/(o+1.)+(-0.183388699236)+(0.771348141212)*0.5**o
        ref[3,3,1,1]=(-0.932550058069)/(o+1.)+(-0.47828109001)+(-0.795311686688)*0.5**o
        ref[3,3,2,0]=(0.821542940923)/(o+1.)+(0.166853194487)+(-0.13512674466)*0.5**o
        ref[3,3,2,1]=(-0.46375871628)/(o+1.)+(0.132701267732)+(-0.449544418926)*0.5**o
        ref[3,4,0,0]=(-0.074530542168)/(o+1.)+(-0.0567877963797)+(-0.134845261261)*0.5**o
        ref[3,4,0,1]=(0.238671520479)/(o+1.)+(0.50689253572)+(0.731340228204)*0.5**o
        ref[3,4,1,0]=(0.981601513335)/(o+1.)+(0.400717209822)+(0.248707844726)*0.5**o
        ref[3,4,1,1]=(0.592304390107)/(o+1.)+(0.342408637328)+(0.834188819333)*0.5**o
        ref[3,4,2,0]=(-0.339991330909)/(o+1.)+(-0.0145922376156)+(0.899051000485)*0.5**o
        ref[3,4,2,1]=(-0.626610688164)/(o+1.)+(-0.417008165115)+(-0.426085250846)*0.5**o
      else:
        arg[0,0,0,0]=(0.697666818785)*x[0]+(1.00202295791)*x[1]+(0.617700157184)*x[2]
        arg[0,0,0,1]=(-0.465577819457)*x[0]+(-1.40054693355)*x[1]+(1.16912653295)*x[2]
        arg[0,0,1,0]=(-1.26696500634)*x[0]+(-0.0469756169975)*x[1]+(-0.445591738991)*x[2]
        arg[0,0,1,1]=(-1.61654321709)*x[0]+(-1.69522962467)*x[1]+(-1.06089094851)*x[2]
        arg[0,0,2,0]=(0.229468378506)*x[0]+(0.716813596991)*x[1]+(-0.237703937029)*x[2]
        arg[0,0,2,1]=(-1.12052051905)*x[0]+(0.835000004618)*x[1]+(-0.853004812404)*x[2]
        arg[0,1,0,0]=(-1.75500745705)*x[0]+(0.655055836882)*x[1]+(1.05207750735)*x[2]
        arg[0,1,0,1]=(-0.0746024824253)*x[0]+(-0.421596816958)*x[1]+(-0.375715855989)*x[2]
        arg[0,1,1,0]=(-0.963269684396)*x[0]+(0.431993248836)*x[1]+(-1.63061928649)*x[2]
        arg[0,1,1,1]=(-0.418757265018)*x[0]+(-0.665626168583)*x[1]+(-0.335734432507)*x[2]
        arg[0,1,2,0]=(0.980346404307)*x[0]+(1.45442049802)*x[1]+(-0.833420308612)*x[2]
        arg[0,1,2,1]=(-1.52536017847)*x[0]+(0.99484028681)*x[1]+(0.109754321663)*x[2]
        arg[0,2,0,0]=(1.697179262)*x[0]+(-0.928732802147)*x[1]+(-0.653379513626)*x[2]
        arg[0,2,0,1]=(0.682698902948)*x[0]+(0.829307959583)*x[1]+(-1.0300112276)*x[2]
        arg[0,2,1,0]=(0.026084323266)*x[0]+(1.28865784608)*x[1]+(0.830446580645)*x[2]
        arg[0,2,1,1]=(0.486291506032)*x[0]+(-0.519637742913)*x[1]+(0.81671330629)*x[2]
        arg[0,2,2,0]=(-0.761591624634)*x[0]+(0.370951131011)*x[1]+(-0.256748479312)*x[2]
        arg[0,2,2,1]=(-0.249974326513)*x[0]+(0.267323484635)*x[1]+(-1.09642130905)*x[2]
        arg[0,3,0,0]=(0.752057563255)*x[0]+(-1.2772262175)*x[1]+(-0.264569668983)*x[2]
        arg[0,3,0,1]=(-0.173167796447)*x[0]+(0.31873208356)*x[1]+(-0.599780701112)*x[2]
        arg[0,3,1,0]=(-0.912464014941)*x[0]+(-1.56452517166)*x[1]+(1.37651660499)*x[2]
        arg[0,3,1,1]=(0.302009667854)*x[0]+(0.524213064769)*x[1]+(0.316301522683)*x[2]
        arg[0,3,2,0]=(-1.2940154673)*x[0]+(-0.900061396992)*x[1]+(0.3860029847)*x[2]
        arg[0,3,2,1]=(-0.97354131341)*x[0]+(1.08221883964)*x[1]+(-0.739323076416)*x[2]
        arg[0,4,0,0]=(1.86535317192)*x[0]+(-0.478359589546)*x[1]+(-0.798014592302)*x[2]
        arg[0,4,0,1]=(-1.08518693851)*x[0]+(-0.186048654536)*x[1]+(1.04915573267)*x[2]
        arg[0,4,1,0]=(-1.09692811055)*x[0]+(1.01042920935)*x[1]+(-0.73978089555)*x[2]
        arg[0,4,1,1]=(0.654656964663)*x[0]+(0.279204596126)*x[1]+(0.886217781361)*x[2]
        arg[0,4,2,0]=(-0.54009886291)*x[0]+(0.378037248739)*x[1]+(1.09785568886)*x[2]
        arg[0,4,2,1]=(-1.43995958913)*x[0]+(-0.720157090994)*x[1]+(0.0676362028136)*x[2]
        arg[1,0,0,0]=(1.47735412608)*x[0]+(0.342355546228)*x[1]+(-0.0134502415508)*x[2]
        arg[1,0,0,1]=(-0.419536590369)*x[0]+(-1.61108470511)*x[1]+(1.51281979355)*x[2]
        arg[1,0,1,0]=(1.31765698245)*x[0]+(-0.7499566294)*x[1]+(0.934537503478)*x[2]
        arg[1,0,1,1]=(0.688467914883)*x[0]+(-1.32681280673)*x[1]+(-0.387405705058)*x[2]
        arg[1,0,2,0]=(1.37069587556)*x[0]+(0.540830111773)*x[1]+(0.477311068263)*x[2]
        arg[1,0,2,1]=(-0.572066213761)*x[0]+(-1.11950925494)*x[1]+(1.07939217872)*x[2]
        arg[1,1,0,0]=(0.564997105326)*x[0]+(-0.73130546389)*x[1]+(0.435910937304)*x[2]
        arg[1,1,0,1]=(0.513931739674)*x[0]+(-0.944043404826)*x[1]+(0.014114227468)*x[2]
        arg[1,1,1,0]=(-0.77850597256)*x[0]+(-0.445835136484)*x[1]+(0.157478357479)*x[2]
        arg[1,1,1,1]=(0.752431176866)*x[0]+(0.659759664199)*x[1]+(-0.859515433521)*x[2]
        arg[1,1,2,0]=(0.0693158730911)*x[0]+(-0.820997827204)*x[1]+(-0.714694436829)*x[2]
        arg[1,1,2,1]=(0.604674152847)*x[0]+(0.647261183132)*x[1]+(-1.21981352028)*x[2]
        arg[1,2,0,0]=(0.0997504695984)*x[0]+(-0.271549839083)*x[1]+(-0.573467835942)*x[2]
        arg[1,2,0,1]=(0.01861845581)*x[0]+(0.157410502406)*x[1]+(0.939257868776)*x[2]
        arg[1,2,1,0]=(1.44909894631)*x[0]+(0.116038517547)*x[1]+(-0.851940489233)*x[2]
        arg[1,2,1,1]=(1.34598841007)*x[0]+(1.79163125841)*x[1]+(0.719171044245)*x[2]
        arg[1,2,2,0]=(-0.214737791448)*x[0]+(-1.85810839095)*x[1]+(-0.240776599927)*x[2]
        arg[1,2,2,1]=(-0.574707645928)*x[0]+(-0.664173831956)*x[1]+(0.310262106696)*x[2]
        arg[1,3,0,0]=(0.296213453592)*x[0]+(-0.831112482555)*x[1]+(0.130075739128)*x[2]
        arg[1,3,0,1]=(-0.369510604457)*x[0]+(0.0373909697998)*x[1]+(1.6689129783)*x[2]
        arg[1,3,1,0]=(0.0922435881419)*x[0]+(-0.280519827004)*x[1]+(0.285933839232)*x[2]
        arg[1,3,1,1]=(-0.649253282272)*x[0]+(-0.460065044123)*x[1]+(-0.0711164948063)*x[2]
        arg[1,3,2,0]=(0.793313307581)*x[0]+(-0.0587956351149)*x[1]+(0.810487259739)*x[2]
        arg[1,3,2,1]=(-1.38668597055)*x[0]+(1.51396934198)*x[1]+(-1.13918550255)*x[2]
        arg[1,4,0,0]=(1.40078167737)*x[0]+(-1.61915436345)*x[1]+(0.421710091214)*x[2]
        arg[1,4,0,1]=(-1.37982015877)*x[0]+(1.30607371738)*x[1]+(-0.608888848419)*x[2]
        arg[1,4,1,0]=(-0.558087802805)*x[0]+(-0.214169499512)*x[1]+(-0.766242546503)*x[2]
        arg[1,4,1,1]=(-1.75503436584)*x[0]+(0.183604705451)*x[1]+(-0.148071484576)*x[2]
        arg[1,4,2,0]=(0.537942069142)*x[0]+(-0.00640749928252)*x[1]+(-0.277544629657)*x[2]
        arg[1,4,2,1]=(-0.652645833371)*x[0]+(-1.33439077026)*x[1]+(-0.226308197201)*x[2]
        arg[2,0,0,0]=(1.45646289279)*x[0]+(1.40842127675)*x[1]+(1.75078281495)*x[2]
        arg[2,0,0,1]=(0.867988651995)*x[0]+(1.35061812241)*x[1]+(-1.56520776323)*x[2]
        arg[2,0,1,0]=(0.549686523522)*x[0]+(-0.528085465813)*x[1]+(-0.994978820267)*x[2]
        arg[2,0,1,1]=(0.383595996552)*x[0]+(0.612989859087)*x[1]+(-0.560401247231)*x[2]
        arg[2,0,2,0]=(-0.341259919499)*x[0]+(-0.125491675042)*x[1]+(0.545572340757)*x[2]
        arg[2,0,2,1]=(-0.414269395434)*x[0]+(-0.433969881898)*x[1]+(-1.01820199328)*x[2]
        arg[2,1,0,0]=(-0.113575280389)*x[0]+(0.326916039942)*x[1]+(-0.576072890249)*x[2]
        arg[2,1,0,1]=(-0.296473969661)*x[0]+(0.195675099431)*x[1]+(-1.72378150831)*x[2]
        arg[2,1,1,0]=(-1.31575932758)*x[0]+(-0.122557423522)*x[1]+(0.53435027695)*x[2]
        arg[2,1,1,1]=(0.299435754608)*x[0]+(0.727140413706)*x[1]+(0.848931358475)*x[2]
        arg[2,1,2,0]=(0.2618965198)*x[0]+(-0.461672573339)*x[1]+(0.923592803172)*x[2]
        arg[2,1,2,1]=(-0.326817258523)*x[0]+(-0.0363300805442)*x[1]+(1.82049875918)*x[2]
        arg[2,2,0,0]=(-0.435113320888)*x[0]+(0.0257984353786)*x[1]+(0.287846577657)*x[2]
        arg[2,2,0,1]=(1.39178068279)*x[0]+(0.136781967539)*x[1]+(-1.38161590351)*x[2]
        arg[2,2,1,0]=(-0.887874199356)*x[0]+(-0.476769153654)*x[1]+(-0.618760485829)*x[2]
        arg[2,2,1,1]=(1.58883355098)*x[0]+(-0.889469047555)*x[1]+(-0.678486755843)*x[2]
        arg[2,2,2,0]=(0.237147029365)*x[0]+(-0.509883462401)*x[1]+(0.533163151551)*x[2]
        arg[2,2,2,1]=(0.66513913749)*x[0]+(1.30317030657)*x[1]+(0.804083365617)*x[2]
        arg[2,3,0,0]=(1.01531518403)*x[0]+(1.3753053143)*x[1]+(0.000877284235964)*x[2]
        arg[2,3,0,1]=(-0.40853642543)*x[0]+(0.154794268783)*x[1]+(-1.34306671117)*x[2]
        arg[2,3,1,0]=(0.43065284359)*x[0]+(-0.28965146009)*x[1]+(0.660248828789)*x[2]
        arg[2,3,1,1]=(-1.4334922681)*x[0]+(-0.0348802858609)*x[1]+(0.502051304974)*x[2]
        arg[2,3,2,0]=(-1.17399537487)*x[0]+(0.865787043946)*x[1]+(-0.0105949677478)*x[2]
        arg[2,3,2,1]=(0.63104158794)*x[0]+(0.436196079951)*x[1]+(0.498606623061)*x[2]
        arg[2,4,0,0]=(-1.44410177354)*x[0]+(0.513406326011)*x[1]+(0.699127199447)*x[2]
        arg[2,4,0,1]=(-0.981727023077)*x[0]+(0.484182120695)*x[1]+(0.667175832045)*x[2]
        arg[2,4,1,0]=(-1.3352602588)*x[0]+(0.626269023364)*x[1]+(0.813537646695)*x[2]
        arg[2,4,1,1]=(-0.932486930114)*x[0]+(0.299448807849)*x[1]+(0.195862014797)*x[2]
        arg[2,4,2,0]=(-0.0621489644806)*x[0]+(0.131979664158)*x[1]+(0.442029269778)*x[2]
        arg[2,4,2,1]=(-0.507508608633)*x[0]+(-1.28613226533)*x[1]+(-0.697920444023)*x[2]
        arg[3,0,0,0]=(-0.984942677585)*x[0]+(-1.58913299616)*x[1]+(-0.373026784822)*x[2]
        arg[3,0,0,1]=(0.616666427113)*x[0]+(-0.625162800471)*x[1]+(-0.796307290839)*x[2]
        arg[3,0,1,0]=(-0.48469203683)*x[0]+(-0.0572105166553)*x[1]+(0.809572383581)*x[2]
        arg[3,0,1,1]=(-0.487160309892)*x[0]+(0.77646454341)*x[1]+(-0.41947544573)*x[2]
        arg[3,0,2,0]=(-1.17902121554)*x[0]+(-0.244939532033)*x[1]+(-0.282994359476)*x[2]
        arg[3,0,2,1]=(0.33515909881)*x[0]+(0.146866561705)*x[1]+(1.81486098364)*x[2]
        arg[3,1,0,0]=(0.379977901341)*x[0]+(0.834531701517)*x[1]+(-0.0242027062166)*x[2]
        arg[3,1,0,1]=(0.184318292766)*x[0]+(1.08367911805)*x[1]+(-0.164531741594)*x[2]
        arg[3,1,1,0]=(-1.00967115996)*x[0]+(0.542699640149)*x[1]+(0.707616701259)*x[2]
        arg[3,1,1,1]=(0.856948233811)*x[0]+(-1.2186550297)*x[1]+(-0.034268798159)*x[2]
        arg[3,1,2,0]=(0.259470684477)*x[0]+(-1.03586983732)*x[1]+(-0.157275979341)*x[2]
        arg[3,1,2,1]=(-0.161954890658)*x[0]+(1.65853209885)*x[1]+(0.381113456719)*x[2]
        arg[3,2,0,0]=(-0.437413004681)*x[0]+(-0.730340273929)*x[1]+(-0.687689272494)*x[2]
        arg[3,2,0,1]=(-0.336618164118)*x[0]+(-1.24388666895)*x[1]+(-0.2198689254)*x[2]
        arg[3,2,1,0]=(0.0872793087887)*x[0]+(-0.336554573215)*x[1]+(-0.0597803523262)*x[2]
        arg[3,2,1,1]=(0.192805432018)*x[0]+(0.0540495582064)*x[1]+(0.846658042071)*x[2]
        arg[3,2,2,0]=(0.00794192604414)*x[0]+(-0.33504559053)*x[1]+(0.615701149787)*x[2]
        arg[3,2,2,1]=(0.210870961503)*x[0]+(1.14164095846)*x[1]+(-0.192785742791)*x[2]
        arg[3,3,0,0]=(-0.25895162041)*x[0]+(-0.225800436488)*x[1]+(-0.198007079838)*x[2]
        arg[3,3,0,1]=(-0.150363703231)*x[0]+(1.23870429535)*x[1]+(0.28635881247)*x[2]
        arg[3,3,1,0]=(-0.865980718216)*x[0]+(-0.0849853244335)*x[1]+(-0.292164217189)*x[2]
        arg[3,3,1,1]=(-0.350465968722)*x[0]+(0.09853736936)*x[1]+(-0.476711524665)*x[2]
        arg[3,3,2,0]=(0.311396300263)*x[0]+(-0.92437612034)*x[1]+(-0.757139962928)*x[2]
        arg[3,3,2,1]=(-0.912311919837)*x[0]+(0.267793631099)*x[1]+(0.4917345401)*x[2]
        arg[3,4,0,0]=(0.613662541753)*x[0]+(0.782757341361)*x[1]+(1.04209808158)*x[2]
        arg[3,4,0,1]=(0.492828613997)*x[0]+(1.11979095664)*x[1]+(-0.843545072354)*x[2]
        arg[3,4,1,0]=(-0.51939120556)*x[0]+(0.476004484286)*x[1]+(-0.516311785432)*x[2]
        arg[3,4,1,1]=(1.00423313567)*x[0]+(0.90776890388)*x[1]+(1.34423309497)*x[2]
        arg[3,4,2,0]=(-1.44069600826)*x[0]+(-1.39515290909)*x[1]+(1.19851590247)*x[2]
        arg[3,4,2,1]=(0.391851619877)*x[0]+(0.265529120553)*x[1]+(0.996610410913)*x[2]
        ref[0,0,0,0]=(1.19330263935)/(o+1.)+(0.55394275251)+(0.0162017895041)*0.5**o
        ref[0,0,0,1]=(-0.672351783684)/(o+1.)+(-0.187889622039)+(0.351132807711)*0.5**o
        ref[0,0,1,0]=(-1.27021996264)/(o+1.)+(0.236286530101)+(-0.961885459894)*0.5**o
        ref[0,0,1,1]=(-1.43442517107)/(o+1.)+(-1.00113374859)+(-0.935971122022)*0.5**o
        ref[0,0,2,0]=(-0.434813695127)/(o+1.)+(0.606673209583)+(-0.0699546855712)*0.5**o
        ref[0,0,2,1]=(-0.0434901247948)/(o+1.)+(-0.180911877983)+(-0.733211446073)*0.5**o
        ref[0,1,0,0]=(1.41514302182)/(o+1.)+(-0.247977413325)+(-0.967062307989)*0.5**o
        ref[0,1,0,1]=(0.69128100142)/(o+1.)+(-0.621609882923)+(-0.319976390946)*0.5**o
        ref[0,1,1,0]=(-0.881870429191)/(o+1.)+(-0.654075774082)+(0.0281262553057)*0.5**o
        ref[0,1,1,1]=(-1.27239116086)/(o+1.)+(-0.0854066429492)+(0.0230865806541)*0.5**o
        ref[0,1,2,0]=(-0.222663945351)/(o+1.)+(0.685312598747)+(0.453385341573)*0.5**o
        ref[0,1,2,1]=(0.101409038706)/(o+1.)+(0.151367946744)+(-0.824910502191)*0.5**o
        ref[0,2,0,0]=(-1.19509443892)/(o+1.)+(0.225437856866)+(0.85928567141)*0.5**o
        ref[0,2,0,1]=(0.285764347754)/(o+1.)+(-0.383432127133)+(0.963095541446)*0.5**o
        ref[0,2,1,0]=(0.728010227911)/(o+1.)+(0.983790919896)+(-0.550403317717)*0.5**o
        ref[0,2,1,1]=(0.563748259036)/(o+1.)+(0.165164845927)+(-0.110710881481)*0.5**o
        ref[0,2,2,0]=(-0.361502606429)/(o+1.)+(0.0276145482611)+(-0.341115463028)*0.5**o
        ref[0,2,2,1]=(-0.293320207933)/(o+1.)+(0.0336757304385)+(-0.853103403872)*0.5**o
        ref[0,3,0,0]=(-0.410233956527)/(o+1.)+(-0.474281126245)+(0.569057885785)*0.5**o
        ref[0,3,0,1]=(-0.835861970115)/(o+1.)+(0.0173715483175)+(0.346902459481)*0.5**o
        ref[0,3,1,0]=(0.0830466456187)/(o+1.)+(-0.288204681495)+(-0.607109864241)*0.5**o
        ref[0,3,1,1]=(1.07074130823)/(o+1.)+(-0.0437983012867)+(0.159379549652)*0.5**o
        ref[0,3,2,0]=(0.172301049609)/(o+1.)+(-0.491724874152)+(-0.996925180895)*0.5**o
        ref[0,3,2,1]=(-0.24891941221)/(o+1.)+(-0.03935856269)+(-0.3030090126)*0.5**o
        ref[0,4,0,0]=(-0.778575122429)/(o+1.)+(0.213085222078)+(0.941383668349)*0.5**o
        ref[0,4,0,1]=(0.880801627174)/(o+1.)+(-0.454451723401)+(-0.193978040747)*0.5**o
        ref[0,4,1,0]=(-0.835604724882)/(o+1.)+(0.127240834083)+(-0.245156740036)*0.5**o
        ref[0,4,1,1]=(1.62336716324)/(o+1.)+(-0.101085918464)+(0.398884015839)*0.5**o
        ref[0,4,2,0]=(1.16805827702)/(o+1.)+(0.211232591688)+(-0.654729385702)*0.5**o
        ref[0,4,2,1]=(-0.305424138957)/(o+1.)+(-0.431076572693)+(-0.924903192969)*0.5**o
        ref[1,0,0,0]=(-0.481596247529)/(o+1.)+(0.769902632559)+(0.748050413171)*0.5**o
        ref[1,0,0,1]=(0.177388661252)/(o+1.)+(-0.118766442613)+(-0.457657277952)*0.5**o
        ref[1,0,1,0]=(0.709890860372)/(o+1.)+(0.0668010230727)+(0.658744950012)*0.5**o
        ref[1,0,1,1]=(-1.45685781381)/(o+1.)+(-0.178674426089)+(0.788456069079)*0.5**o
        ref[1,0,2,0]=(0.194388185612)/(o+1.)+(0.649177543063)+(0.896093783855)*0.5**o
        ref[1,0,2,1]=(0.181561279507)/(o+1.)+(-0.200618291902)+(-0.392507985678)*0.5**o
        ref[1,1,0,0]=(-0.884376814027)/(o+1.)+(0.280382736159)+(0.593213920451)*0.5**o
        ref[1,1,0,1]=(-0.250709602218)/(o+1.)+(-0.492104211211)+(0.818920586956)*0.5**o
        ref[1,1,1,0]=(0.0529723694625)/(o+1.)+(-0.502047650638)+(-0.115739819752)*0.5**o
        ref[1,1,1,1]=(0.215812202957)/(o+1.)+(-0.308796450292)+(0.954456105171)*0.5**o
        ref[1,1,2,0]=(-0.991481050902)/(o+1.)+(0.162626252838)+(-0.800147845717)*0.5**o
        ref[1,1,2,1]=(-0.433989756951)/(o+1.)+(0.362776336712)+(-0.259441100777)*0.5**o
        ref[1,2,0,0]=(-0.930287052624)/(o+1.)+(-0.347752763108)+(0.880525373413)*0.5**o
        ref[1,2,0,1]=(0.698674852861)/(o+1.)+(-0.15119613196)+(0.719004238051)*0.5**o
        ref[1,2,1,0]=(-0.777216821365)/(o+1.)+(0.436336076297)+(0.617741643399)*0.5**o
        ref[1,2,1,1]=(1.09595223135)/(o+1.)+(0.978251130983)+(0.804336219412)*0.5**o
        ref[1,2,2,0]=(-0.651012630849)/(o+1.)+(-1.04894606664)+(0.43528198181)*0.5**o
        ref[1,2,2,1]=(-1.5227353883)/(o+1.)+(0.670019307557)+(-0.745922598001)*0.5**o
        ref[1,3,0,0]=(-0.750195637997)/(o+1.)+(0.231864245348)+(-0.118356142533)*0.5**o
        ref[1,3,0,1]=(1.17136226596)/(o+1.)+(-0.114976421389)+(0.395383920468)*0.5**o
        ref[1,3,1,0]=(-0.108070636769)/(o+1.)+(0.279145429234)+(-0.352562621327)*0.5**o
        ref[1,3,1,1]=(-0.144223481714)/(o+1.)+(-0.472954352493)+(-0.0903026345013)*0.5**o
        ref[1,3,2,0]=(-0.487564307786)/(o+1.)+(0.636973351449)+(0.758622537094)*0.5**o
        ref[1,3,2,1]=(0.390860965954)/(o+1.)+(-0.36647678165)+(-0.66980953377)*0.5**o
        ref[1,4,0,0]=(-1.04824984837)/(o+1.)+(0.213932231728)+(0.823722790054)*0.5**o
        ref[1,4,0,1]=(-0.0984653601336)/(o+1.)+(0.07867877705)+(-0.741527483773)*0.5**o
        ref[1,4,1,0]=(-0.146287867383)/(o+1.)+(-0.908951868158)+(0.425691754878)*0.5**o
        ref[1,4,1,1]=(-0.94160416554)/(o+1.)+(0.0459877010016)+(-0.869872381431)*0.5**o
        ref[1,4,2,0]=(0.0115742858952)/(o+1.)+(-0.222488399041)+(0.687392452389)*0.5**o
        ref[1,4,2,1]=(-0.956561936575)/(o+1.)+(-0.59770879042)+(-0.0613652834159)*0.5**o
        ref[2,0,0,0]=(1.83987381729)/(o+1.)+(0.956579317963)+(0.862634531274)*0.5**o
        ref[2,0,0,1]=(0.105446127421)/(o+1.)+(0.180661881191)+(0.186629121375)*0.5**o
        ref[2,0,1,0]=(0.0212167440766)/(o+1.)+(-0.337825588888)+(-0.318943328859)*0.5**o
        ref[2,0,1,1]=(0.744834715631)/(o+1.)+(-0.350058416663)+(0.391466726104)*0.5**o
        ref[2,0,2,0]=(-0.788776613856)/(o+1.)+(0.233776806014)+(0.400043748043)*0.5**o
        ref[2,0,2,1]=(-0.95346976307)/(o+1.)+(0.0274998652324)+(-0.967971238003)*0.5**o
        ref[2,1,0,0]=(-0.536051408942)/(o+1.)+(-0.208498016994)+(0.590315312233)*0.5**o
        ref[2,1,0,1]=(-1.04158508788)/(o+1.)+(-0.397885793101)+(0.0127762955396)*0.5**o
        ref[2,1,1,0]=(0.529914207253)/(o+1.)+(-0.547503443987)+(-0.338873793431)*0.5**o
        ref[2,1,1,1]=(0.62298670274)/(o+1.)+(0.449860048786)+(0.352800726478)*0.5**o
        ref[2,1,2,0]=(0.965292650595)/(o+1.)+(0.00980498639)+(-0.261085873742)*0.5**o
        ref[2,1,2,1]=(0.403285360803)/(o+1.)+(0.369160592269)+(0.315744874774)*0.5**o
        ref[2,2,0,0]=(-0.457764884724)/(o+1.)+(0.563657776706)+(-0.791018976541)*0.5**o
        ref[2,2,0,1]=(0.188151274044)/(o+1.)+(-0.476470545911)+(0.911736564588)*0.5**o
        ref[2,2,1,0]=(-0.558515085739)/(o+1.)+(-0.445583796553)+(-0.533721159994)*0.5**o
        ref[2,2,1,1]=(-0.431992337265)/(o+1.)+(-0.0957726221961)+(0.64441532924)*0.5**o
        ref[2,2,2,0]=(-0.159056461947)/(o+1.)+(-0.0326864536156)+(0.484856087693)*0.5**o
        ref[2,2,2,1]=(0.533116456037)/(o+1.)+(1.10197257725)+(0.0353311991352)*0.5**o
        ref[2,3,0,0]=(0.0570625147023)/(o+1.)+(0.74170779854)+(0.851019670777)*0.5**o
        ref[2,3,0,1]=(-0.495059904832)/(o+1.)+(-0.169954912181)+(-0.761839138623)*0.5**o
        ref[2,3,1,0]=(0.482375173314)/(o+1.)+(-0.19374838651)+(0.706371811996)*0.5**o
        ref[2,3,1,1]=(0.12928100721)/(o+1.)+(-0.317327862062)+(-0.460946532077)*0.5**o
        ref[2,3,2,0]=(-0.234362084763)/(o+1.)+(0.423486279181)+(-0.93141377227)*0.5**o
        ref[2,3,2,1]=(1.25586693002)/(o+1.)+(-0.127551329993)+(0.56508002092)*0.5**o
        ref[2,4,0,0]=(0.45235297749)/(o+1.)+(-0.0217537364788)+(-0.640413752615)*0.5**o
        ref[2,4,0,1]=(0.936809207384)/(o+1.)+(-0.151519610695)+(-0.464139056331)*0.5**o
        ref[2,4,1,0]=(0.867693208863)/(o+1.)+(0.105683574314)+(-0.974513946233)*0.5**o
        ref[2,4,1,1]=(-1.27821575101)/(o+1.)+(0.463163847311)+(-0.0852880510814)*0.5**o
        ref[2,4,2,0]=(-0.846224501366)/(o+1.)+(0.432619461409)+(0.492845548002)*0.5**o
        ref[2,4,2,1]=(-1.48923229995)/(o+1.)+(-0.505747396375)+(0.00916577472118)*0.5**o
        ref[3,0,0,0]=(-0.959215731121)/(o+1.)+(-0.904609108587)+(-0.178668510268)*0.5**o
        ref[3,0,0,1]=(-1.53181231865)/(o+1.)+(-0.0758543610408)+(0.878717376536)*0.5**o
        ref[3,0,1,0]=(1.18643238831)/(o+1.)+(-0.288825159097)+(-0.341112240015)*0.5**o
        ref[3,0,1,1]=(-0.0675694917031)/(o+1.)+(-0.168955703616)+(0.275309686724)*0.5**o
        ref[3,0,2,0]=(-0.663269860133)/(o+1.)+(-0.235102421885)+(-0.573480403148)*0.5**o
        ref[3,0,2,1]=(1.0969289778)/(o+1.)+(0.46603172821)+(0.267894209932)*0.5**o
        ref[3,1,0,0]=(-0.716870513559)/(o+1.)+(0.712039409283)+(0.483098591635)*0.5**o
        ref[3,1,0,1]=(0.0328531999411)/(o+1.)+(0.379319450988)+(0.311973567308)*0.5**o
        ref[3,1,1,0]=(1.08648208064)/(o+1.)+(-0.196831959338)+(-0.452172980519)*0.5**o
        ref[3,1,1,1]=(0.0813181502612)/(o+1.)+(-0.227364064527)+(-0.0225656152548)*0.5**o
        ref[3,1,2,0]=(-0.271000454408)/(o+1.)+(-0.765711687155)+(0.868748696531)*0.5**o
        ref[3,1,2,1]=(1.60569646253)/(o+1.)+(0.137437739736)+(-0.00288127709197)*0.5**o
        ref[3,2,0,0]=(-1.25887089059)/(o+1.)+(-0.525842290945)+(0.455112921376)*0.5**o
        ref[3,2,0,1]=(-1.63787880725)/(o+1.)+(0.00214706725405)+(-0.166789085722)*0.5**o
        ref[3,2,1,0]=(-1.82111307894)/(o+1.)+(0.561806270366)+(0.388444921454)*0.5**o
        ref[3,2,1,1]=(-0.489259906414)/(o+1.)+(0.582548398381)+(0.417676141947)*0.5**o
        ref[3,2,2,0]=(-0.0674375340391)/(o+1.)+(-0.00345549227349)+(0.362946003887)*0.5**o
        ref[3,2,2,1]=(0.662780682001)/(o+1.)+(-0.15453848218)+(0.806022459534)*0.5**o
        ref[3,3,0,0]=(-1.15795406515)/(o+1.)+(0.0926725746467)+(0.289849779118)*0.5**o
        ref[3,3,0,1]=(1.27520269547)/(o+1.)+(-0.286529818361)+(0.672556345847)*0.5**o
        ref[3,3,1,0]=(-0.343290651469)/(o+1.)+(-0.0269962100904)+(-0.845847188189)*0.5**o
        ref[3,3,1,1]=(0.497715053371)/(o+1.)+(-0.158577177766)+(-0.909200821866)*0.5**o
        ref[3,3,2,0]=(-0.952431538662)/(o+1.)+(0.0816482063045)+(-0.580984656952)*0.5**o
        ref[3,3,2,1]=(-0.176227469799)/(o+1.)+(0.0456157853688)+(-0.0677878495767)*0.5**o
        ref[3,4,0,0]=(0.976460483802)/(o+1.)+(0.347595506011)+(0.766866468871)*0.5**o
        ref[3,4,0,1]=(0.0358170327662)/(o+1.)+(-0.0991456259778)+(0.931548717473)*0.5**o
        ref[3,4,1,0]=(0.114544318514)/(o+1.)+(-0.201014865792)+(-0.272213093635)*0.5**o
        ref[3,4,1,1]=(0.554509925385)/(o+1.)+(1.23219332557)+(0.237338557994)*0.5**o
        ref[3,4,2,0]=(0.0803465444931)/(o+1.)+(-0.518257569997)+(-0.681164419371)*0.5**o
        ref[3,4,2,1]=(-0.0443609570522)/(o+1.)+(0.572813368096)+(0.552725372202)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedFunctionOnContactZero_rank0(self):
      """
      tests integral of rank 0 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedFunctionOnContactZero(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedFunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.114918917602)*x[0]**o+(0.304834778607)*x[0]+(0.704057176139)*x[1]**o+(-0.0664624375434)*x[1]
        ref=(0.704057176139)/(o+1.)+(0.119186170532)+(-0.114918917602)*0.5**o
      else:
        arg=(-0.134961573162)*x[0]**o+(0.585433861814)*x[0]+(0.209596676649)*x[1]**o+(0.870867416311)*x[1]+(-0.902854033973)*x[2]**o+(-0.812135783993)*x[2]
        ref=(-0.693257357324)/(o+1.)+(0.322082747066)+(-0.134961573162)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedFunctionOnContactZero_rank1(self):
      """
      tests integral of rank 1 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedFunctionOnContactZero(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedFunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.441677901994)*x[0]**o+(0.0234076994082)*x[0]+(-0.520465368981)*x[1]**o+(-0.668717137452)*x[1]
        arg[1]=(0.0637451453664)*x[0]**o+(0.610543665309)*x[0]+(0.639008465664)*x[1]**o+(-0.460500756968)*x[1]
        ref[0]=(-0.520465368981)/(o+1.)+(-0.322654719022)+(0.441677901994)*0.5**o
        ref[1]=(0.639008465664)/(o+1.)+(0.0750214541708)+(0.0637451453664)*0.5**o
      else:
        arg[0]=(0.207042421638)*x[0]**o+(0.586067682859)*x[0]+(-0.731189752251)*x[1]**o+(0.439755444977)*x[1]+(-0.215027146515)*x[2]**o+(-0.22109699198)*x[2]
        arg[1]=(-0.482897746923)*x[0]**o+(-0.262848758384)*x[0]+(-0.130231246709)*x[1]**o+(0.494143899685)*x[1]+(0.871226664446)*x[2]**o+(-0.45136577521)*x[2]
        ref[0]=(-0.946216898767)/(o+1.)+(0.402363067928)+(0.207042421638)*0.5**o
        ref[1]=(0.740995417737)/(o+1.)+(-0.110035316955)+(-0.482897746923)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedFunctionOnContactZero_rank2(self):
      """
      tests integral of rank 2 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedFunctionOnContactZero(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedFunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(0.418217706031)*x[0]**o+(0.817101274679)*x[0]+(0.359227724634)*x[1]**o+(0.106952446872)*x[1]
        arg[0,1]=(0.735477258623)*x[0]**o+(-0.181692404492)*x[0]+(0.118248218763)*x[1]**o+(0.910033225874)*x[1]
        arg[0,2]=(-0.370301174151)*x[0]**o+(0.421589410528)*x[0]+(-0.91974297037)*x[1]**o+(0.676441845707)*x[1]
        arg[0,3]=(0.170535646217)*x[0]**o+(0.650458506701)*x[0]+(-0.431898202846)*x[1]**o+(-0.89133664224)*x[1]
        arg[0,4]=(-0.288793192482)*x[0]**o+(-0.752066662422)*x[0]+(0.279565184651)*x[1]**o+(-0.20569858505)*x[1]
        arg[1,0]=(-0.831478902729)*x[0]**o+(-0.545347589281)*x[0]+(-0.315738709444)*x[1]**o+(-0.407156023524)*x[1]
        arg[1,1]=(0.119396512736)*x[0]**o+(0.56655186098)*x[0]+(0.798206239231)*x[1]**o+(0.0342467197178)*x[1]
        arg[1,2]=(-0.264721761214)*x[0]**o+(-0.7583989175)*x[0]+(-0.350411882147)*x[1]**o+(-0.556879396863)*x[1]
        arg[1,3]=(0.976029513102)*x[0]**o+(-0.117830794452)*x[0]+(0.318067419468)*x[1]**o+(-0.496129688948)*x[1]
        arg[1,4]=(0.462797018845)*x[0]**o+(-0.214502973992)*x[0]+(0.155052684044)*x[1]**o+(0.264794267973)*x[1]
        arg[2,0]=(0.193892132778)*x[0]**o+(-0.440406597181)*x[0]+(0.527330868842)*x[1]**o+(0.856602716155)*x[1]
        arg[2,1]=(0.0418968976051)*x[0]**o+(0.944283628708)*x[0]+(0.923548919008)*x[1]**o+(0.958011594347)*x[1]
        arg[2,2]=(0.744026460715)*x[0]**o+(0.238896690247)*x[0]+(0.968891312128)*x[1]**o+(0.0636831438113)*x[1]
        arg[2,3]=(-0.289048983574)*x[0]**o+(0.897469203226)*x[0]+(0.948131139828)*x[1]**o+(0.821749060686)*x[1]
        arg[2,4]=(-0.903766144138)*x[0]**o+(-0.453139974489)*x[0]+(-0.715818671181)*x[1]**o+(-0.712367492971)*x[1]
        arg[3,0]=(0.315277263598)*x[0]**o+(-0.676815102073)*x[0]+(0.428704415911)*x[1]**o+(0.54703485039)*x[1]
        arg[3,1]=(-0.301882901745)*x[0]**o+(0.888822293672)*x[0]+(-0.496204235393)*x[1]**o+(0.210598259748)*x[1]
        arg[3,2]=(0.568071178492)*x[0]**o+(-0.552863936415)*x[0]+(0.758619958752)*x[1]**o+(0.520428540448)*x[1]
        arg[3,3]=(0.477425351692)*x[0]**o+(-0.427405588029)*x[0]+(0.197959279245)*x[1]**o+(0.328047998649)*x[1]
        arg[3,4]=(-0.869704203328)*x[0]**o+(-0.0825164966999)*x[0]+(0.779325592334)*x[1]**o+(0.419916607608)*x[1]
        ref[0,0]=(0.359227724634)/(o+1.)+(0.462026860776)+(0.418217706031)*0.5**o
        ref[0,1]=(0.118248218763)/(o+1.)+(0.364170410691)+(0.735477258623)*0.5**o
        ref[0,2]=(-0.91974297037)/(o+1.)+(0.549015628118)+(-0.370301174151)*0.5**o
        ref[0,3]=(-0.431898202846)/(o+1.)+(-0.120439067769)+(0.170535646217)*0.5**o
        ref[0,4]=(0.279565184651)/(o+1.)+(-0.478882623736)+(-0.288793192482)*0.5**o
        ref[1,0]=(-0.315738709444)/(o+1.)+(-0.476251806402)+(-0.831478902729)*0.5**o
        ref[1,1]=(0.798206239231)/(o+1.)+(0.300399290349)+(0.119396512736)*0.5**o
        ref[1,2]=(-0.350411882147)/(o+1.)+(-0.657639157182)+(-0.264721761214)*0.5**o
        ref[1,3]=(0.318067419468)/(o+1.)+(-0.3069802417)+(0.976029513102)*0.5**o
        ref[1,4]=(0.155052684044)/(o+1.)+(0.02514564699)+(0.462797018845)*0.5**o
        ref[2,0]=(0.527330868842)/(o+1.)+(0.208098059487)+(0.193892132778)*0.5**o
        ref[2,1]=(0.923548919008)/(o+1.)+(0.951147611528)+(0.0418968976051)*0.5**o
        ref[2,2]=(0.968891312128)/(o+1.)+(0.151289917029)+(0.744026460715)*0.5**o
        ref[2,3]=(0.948131139828)/(o+1.)+(0.859609131956)+(-0.289048983574)*0.5**o
        ref[2,4]=(-0.715818671181)/(o+1.)+(-0.58275373373)+(-0.903766144138)*0.5**o
        ref[3,0]=(0.428704415911)/(o+1.)+(-0.0648901258416)+(0.315277263598)*0.5**o
        ref[3,1]=(-0.496204235393)/(o+1.)+(0.54971027671)+(-0.301882901745)*0.5**o
        ref[3,2]=(0.758619958752)/(o+1.)+(-0.0162176979831)+(0.568071178492)*0.5**o
        ref[3,3]=(0.197959279245)/(o+1.)+(-0.0496787946897)+(0.477425351692)*0.5**o
        ref[3,4]=(0.779325592334)/(o+1.)+(0.168700055454)+(-0.869704203328)*0.5**o
      else:
        arg[0,0]=(0.204452789147)*x[0]**o+(0.411843111144)*x[0]+(-0.0502866367412)*x[1]**o+(0.957356261522)*x[1]+(0.173522462943)*x[2]**o+(-0.0194908382525)*x[2]
        arg[0,1]=(-0.49641528149)*x[0]**o+(0.158625922243)*x[0]+(0.670932733725)*x[1]**o+(-0.611829064862)*x[1]+(0.758373881782)*x[2]**o+(0.151236368375)*x[2]
        arg[0,2]=(0.577504765611)*x[0]**o+(0.872996499874)*x[0]+(-0.734000014722)*x[1]**o+(-0.890824538872)*x[1]+(-0.796786080271)*x[2]**o+(-0.389005562901)*x[2]
        arg[0,3]=(-0.933166619447)*x[0]**o+(-0.528627639194)*x[0]+(-0.828111326322)*x[1]**o+(0.528345547301)*x[1]+(0.99894455165)*x[2]**o+(0.472982583662)*x[2]
        arg[0,4]=(-0.495503045054)*x[0]**o+(0.629909217292)*x[0]+(0.901515784751)*x[1]**o+(-0.863668288904)*x[1]+(0.76892679355)*x[2]**o+(0.34730747389)*x[2]
        arg[1,0]=(0.303189803506)*x[0]**o+(-0.17543401741)*x[0]+(-0.283157585975)*x[1]**o+(0.950171529024)*x[1]+(0.0483247572497)*x[2]**o+(-0.604332298954)*x[2]
        arg[1,1]=(-0.225480407643)*x[0]**o+(-0.0521067696425)*x[0]+(-0.10259234894)*x[1]**o+(0.373551917005)*x[1]+(0.89047195372)*x[2]**o+(-0.292740890417)*x[2]
        arg[1,2]=(-0.476041156832)*x[0]**o+(0.916078681628)*x[0]+(-0.393887610676)*x[1]**o+(-0.651852504525)*x[1]+(0.611404175606)*x[2]**o+(0.336594647903)*x[2]
        arg[1,3]=(-0.415178048882)*x[0]**o+(0.145394549559)*x[0]+(0.680115054186)*x[1]**o+(0.261517386893)*x[1]+(0.781566117396)*x[2]**o+(-0.659634539391)*x[2]
        arg[1,4]=(0.874821341857)*x[0]**o+(0.317418381126)*x[0]+(0.051455923305)*x[1]**o+(0.670170143071)*x[1]+(-0.211902435336)*x[2]**o+(-0.175299130233)*x[2]
        arg[2,0]=(-0.372961546517)*x[0]**o+(0.199493150648)*x[0]+(-0.630239499285)*x[1]**o+(0.657412283442)*x[1]+(-0.82211915324)*x[2]**o+(-0.468612977158)*x[2]
        arg[2,1]=(-0.503168527887)*x[0]**o+(0.904406483201)*x[0]+(-0.206179628633)*x[1]**o+(0.0787026584997)*x[1]+(0.899590466467)*x[2]**o+(-0.893401768413)*x[2]
        arg[2,2]=(-0.860556408954)*x[0]**o+(0.39021140432)*x[0]+(-0.66425383352)*x[1]**o+(0.287711780417)*x[1]+(0.776884760126)*x[2]**o+(-0.115905192334)*x[2]
        arg[2,3]=(0.461359695993)*x[0]**o+(0.997877835019)*x[0]+(0.0513514112685)*x[1]**o+(-0.904812988908)*x[1]+(-0.903222331838)*x[2]**o+(-0.470185679041)*x[2]
        arg[2,4]=(-0.646560730528)*x[0]**o+(0.10038324067)*x[0]+(0.426753245517)*x[1]**o+(-0.109889624344)*x[1]+(-0.20158193753)*x[2]**o+(0.014258531224)*x[2]
        arg[3,0]=(0.364407704271)*x[0]**o+(0.840519040103)*x[0]+(0.528721642544)*x[1]**o+(-0.853679853614)*x[1]+(0.079836122575)*x[2]**o+(-0.439039178308)*x[2]
        arg[3,1]=(-0.0726491927955)*x[0]**o+(0.217804431494)*x[0]+(-0.454354285545)*x[1]**o+(0.390539936003)*x[1]+(-0.989667912688)*x[2]**o+(-0.0674079606774)*x[2]
        arg[3,2]=(0.62239422825)*x[0]**o+(0.998733716844)*x[0]+(0.796258161536)*x[1]**o+(0.618545466026)*x[1]+(-0.243188896911)*x[2]**o+(-0.677687584224)*x[2]
        arg[3,3]=(0.444073180258)*x[0]**o+(-0.764729326792)*x[0]+(-0.823491453567)*x[1]**o+(0.985393963131)*x[1]+(0.623800470977)*x[2]**o+(0.947328422513)*x[2]
        arg[3,4]=(0.686438920325)*x[0]**o+(0.789760134176)*x[0]+(-0.26011082782)*x[1]**o+(0.0692377043339)*x[1]+(-0.3712894236)*x[2]**o+(-0.542974191103)*x[2]
        ref[0,0]=(0.123235826202)/(o+1.)+(0.674854267207)+(0.204452789147)*0.5**o
        ref[0,1]=(1.42930661551)/(o+1.)+(-0.150983387122)+(-0.49641528149)*0.5**o
        ref[0,2]=(-1.53078609499)/(o+1.)+(-0.203416800949)+(0.577504765611)*0.5**o
        ref[0,3]=(0.170833225328)/(o+1.)+(0.236350245885)+(-0.933166619447)*0.5**o
        ref[0,4]=(1.6704425783)/(o+1.)+(0.0567742011389)+(-0.495503045054)*0.5**o
        ref[1,0]=(-0.234832828725)/(o+1.)+(0.0852026063299)+(0.303189803506)*0.5**o
        ref[1,1]=(0.787879604779)/(o+1.)+(0.0143521284725)+(-0.225480407643)*0.5**o
        ref[1,2]=(0.21751656493)/(o+1.)+(0.300410412503)+(-0.476041156832)*0.5**o
        ref[1,3]=(1.46168117158)/(o+1.)+(-0.126361301469)+(-0.415178048882)*0.5**o
        ref[1,4]=(-0.160446512031)/(o+1.)+(0.406144696982)+(0.874821341857)*0.5**o
        ref[2,0]=(-1.45235865253)/(o+1.)+(0.194146228466)+(-0.372961546517)*0.5**o
        ref[2,1]=(0.693410837835)/(o+1.)+(0.0448536866438)+(-0.503168527887)*0.5**o
        ref[2,2]=(0.112630926606)/(o+1.)+(0.281008996202)+(-0.860556408954)*0.5**o
        ref[2,3]=(-0.85187092057)/(o+1.)+(-0.188560416464)+(0.461359695993)*0.5**o
        ref[2,4]=(0.225171307987)/(o+1.)+(0.00237607377473)+(-0.646560730528)*0.5**o
        ref[3,0]=(0.608557765119)/(o+1.)+(-0.22609999591)+(0.364407704271)*0.5**o
        ref[3,1]=(-1.44402219823)/(o+1.)+(0.27046820341)+(-0.0726491927955)*0.5**o
        ref[3,2]=(0.553069264625)/(o+1.)+(0.469795799323)+(0.62239422825)*0.5**o
        ref[3,3]=(-0.19969098259)/(o+1.)+(0.583996529426)+(0.444073180258)*0.5**o
        ref[3,4]=(-0.631400251421)/(o+1.)+(0.158011823704)+(0.686438920325)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedFunctionOnContactZero_rank3(self):
      """
      tests integral of rank 3 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedFunctionOnContactZero(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedFunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.963014866001)*x[0]**o+(-0.592091539519)*x[0]+(0.397897510985)*x[1]**o+(0.485612885034)*x[1]
        arg[0,0,1]=(-0.764849027936)*x[0]**o+(0.936217923087)*x[0]+(-0.6256860991)*x[1]**o+(0.162359755936)*x[1]
        arg[0,1,0]=(-0.269607090458)*x[0]**o+(0.197100067821)*x[0]+(-0.216513043667)*x[1]**o+(0.143038940436)*x[1]
        arg[0,1,1]=(-0.432206914977)*x[0]**o+(0.177851208963)*x[0]+(0.0477652910843)*x[1]**o+(0.278333389583)*x[1]
        arg[1,0,0]=(-0.123940629628)*x[0]**o+(0.818553665062)*x[0]+(-0.168322144522)*x[1]**o+(0.485324994762)*x[1]
        arg[1,0,1]=(0.593609516922)*x[0]**o+(0.530508390905)*x[0]+(-0.938421636915)*x[1]**o+(-0.686459217917)*x[1]
        arg[1,1,0]=(-0.874022523162)*x[0]**o+(0.694500749056)*x[0]+(-0.650951516798)*x[1]**o+(0.45826621855)*x[1]
        arg[1,1,1]=(-0.259058106634)*x[0]**o+(0.0837282545694)*x[0]+(0.465734884341)*x[1]**o+(0.185558060279)*x[1]
        arg[2,0,0]=(0.346292866419)*x[0]**o+(0.982902010078)*x[0]+(0.0297606445334)*x[1]**o+(0.339043518987)*x[1]
        arg[2,0,1]=(0.749684718966)*x[0]**o+(-0.855799436867)*x[0]+(-0.186333913876)*x[1]**o+(-0.548968316245)*x[1]
        arg[2,1,0]=(-0.255759466052)*x[0]**o+(0.441882703647)*x[0]+(-0.0239807362322)*x[1]**o+(-0.236417762222)*x[1]
        arg[2,1,1]=(-0.0586694299212)*x[0]**o+(-0.912208422332)*x[0]+(-0.11610871626)*x[1]**o+(-0.512805786403)*x[1]
        arg[3,0,0]=(0.826027179564)*x[0]**o+(-0.589876767386)*x[0]+(-0.654858828463)*x[1]**o+(0.601656609023)*x[1]
        arg[3,0,1]=(-0.148714999203)*x[0]**o+(-0.409138795316)*x[0]+(0.290726299307)*x[1]**o+(-0.94009130345)*x[1]
        arg[3,1,0]=(0.63431415979)*x[0]**o+(-0.184553980231)*x[0]+(0.4087395755)*x[1]**o+(-0.694376406449)*x[1]
        arg[3,1,1]=(-0.000928514830108)*x[0]**o+(-0.782814501236)*x[0]+(0.177390244103)*x[1]**o+(0.698562703841)*x[1]
        arg[4,0,0]=(-0.939264907756)*x[0]**o+(0.191360117367)*x[0]+(-0.318750748728)*x[1]**o+(0.910713477162)*x[1]
        arg[4,0,1]=(-0.105186863713)*x[0]**o+(0.0319652211881)*x[0]+(0.118896230389)*x[1]**o+(-0.212677289177)*x[1]
        arg[4,1,0]=(-0.702838100428)*x[0]**o+(0.881695294345)*x[0]+(0.728231539571)*x[1]**o+(-0.192311604244)*x[1]
        arg[4,1,1]=(0.25197450886)*x[0]**o+(0.464857988125)*x[0]+(0.851996888259)*x[1]**o+(-0.408525083481)*x[1]
        arg[5,0,0]=(0.665627182446)*x[0]**o+(0.5367740452)*x[0]+(0.295800616406)*x[1]**o+(0.0620589956995)*x[1]
        arg[5,0,1]=(-0.37320168697)*x[0]**o+(0.072855481498)*x[0]+(0.539806161326)*x[1]**o+(-0.999091813027)*x[1]
        arg[5,1,0]=(-0.060845772709)*x[0]**o+(-0.931869215952)*x[0]+(-0.393477853475)*x[1]**o+(0.968288081996)*x[1]
        arg[5,1,1]=(-0.0768045781739)*x[0]**o+(-0.257318730887)*x[0]+(0.781473167319)*x[1]**o+(-0.458870794544)*x[1]
        ref[0,0,0]=(0.397897510985)/(o+1.)+(-0.0532393272425)+(-0.963014866001)*0.5**o
        ref[0,0,1]=(-0.6256860991)/(o+1.)+(0.549288839511)+(-0.764849027936)*0.5**o
        ref[0,1,0]=(-0.216513043667)/(o+1.)+(0.170069504128)+(-0.269607090458)*0.5**o
        ref[0,1,1]=(0.0477652910843)/(o+1.)+(0.228092299273)+(-0.432206914977)*0.5**o
        ref[1,0,0]=(-0.168322144522)/(o+1.)+(0.651939329912)+(-0.123940629628)*0.5**o
        ref[1,0,1]=(-0.938421636915)/(o+1.)+(-0.0779754135063)+(0.593609516922)*0.5**o
        ref[1,1,0]=(-0.650951516798)/(o+1.)+(0.576383483803)+(-0.874022523162)*0.5**o
        ref[1,1,1]=(0.465734884341)/(o+1.)+(0.134643157424)+(-0.259058106634)*0.5**o
        ref[2,0,0]=(0.0297606445334)/(o+1.)+(0.660972764533)+(0.346292866419)*0.5**o
        ref[2,0,1]=(-0.186333913876)/(o+1.)+(-0.702383876556)+(0.749684718966)*0.5**o
        ref[2,1,0]=(-0.0239807362322)/(o+1.)+(0.102732470713)+(-0.255759466052)*0.5**o
        ref[2,1,1]=(-0.11610871626)/(o+1.)+(-0.712507104367)+(-0.0586694299212)*0.5**o
        ref[3,0,0]=(-0.654858828463)/(o+1.)+(0.0058899208184)+(0.826027179564)*0.5**o
        ref[3,0,1]=(0.290726299307)/(o+1.)+(-0.674615049383)+(-0.148714999203)*0.5**o
        ref[3,1,0]=(0.4087395755)/(o+1.)+(-0.43946519334)+(0.63431415979)*0.5**o
        ref[3,1,1]=(0.177390244103)/(o+1.)+(-0.0421258986973)+(-0.000928514830108)*0.5**o
        ref[4,0,0]=(-0.318750748728)/(o+1.)+(0.551036797264)+(-0.939264907756)*0.5**o
        ref[4,0,1]=(0.118896230389)/(o+1.)+(-0.0903560339946)+(-0.105186863713)*0.5**o
        ref[4,1,0]=(0.728231539571)/(o+1.)+(0.344691845051)+(-0.702838100428)*0.5**o
        ref[4,1,1]=(0.851996888259)/(o+1.)+(0.0281664523221)+(0.25197450886)*0.5**o
        ref[5,0,0]=(0.295800616406)/(o+1.)+(0.29941652045)+(0.665627182446)*0.5**o
        ref[5,0,1]=(0.539806161326)/(o+1.)+(-0.463118165764)+(-0.37320168697)*0.5**o
        ref[5,1,0]=(-0.393477853475)/(o+1.)+(0.0182094330217)+(-0.060845772709)*0.5**o
        ref[5,1,1]=(0.781473167319)/(o+1.)+(-0.358094762716)+(-0.0768045781739)*0.5**o
      else:
        arg[0,0,0]=(0.638389367885)*x[0]**o+(-0.77703387527)*x[0]+(0.951496212227)*x[1]**o+(-0.484305156823)*x[1]+(-0.582437996275)*x[2]**o+(-0.460130065965)*x[2]
        arg[0,0,1]=(0.859011160063)*x[0]**o+(0.999372374168)*x[0]+(-0.946049724069)*x[1]**o+(0.522184535496)*x[1]+(0.00820405492191)*x[2]**o+(0.769305042211)*x[2]
        arg[0,1,0]=(-0.998681640062)*x[0]**o+(0.916063974594)*x[0]+(0.0801858148899)*x[1]**o+(0.475476401258)*x[1]+(-0.805247247843)*x[2]**o+(-0.809331278141)*x[2]
        arg[0,1,1]=(0.1340261052)*x[0]**o+(0.843204972218)*x[0]+(-0.809433597166)*x[1]**o+(-0.947798215446)*x[1]+(0.268796347717)*x[2]**o+(-0.896620567629)*x[2]
        arg[1,0,0]=(0.251201824008)*x[0]**o+(0.762438451805)*x[0]+(0.892722429031)*x[1]**o+(0.146944927124)*x[1]+(0.00953973207152)*x[2]**o+(0.423262196414)*x[2]
        arg[1,0,1]=(-0.258843593921)*x[0]**o+(-0.950574260099)*x[0]+(-0.49096256134)*x[1]**o+(0.446238902975)*x[1]+(-0.880706204083)*x[2]**o+(-0.400075002454)*x[2]
        arg[1,1,0]=(-0.854529145157)*x[0]**o+(0.118165533114)*x[0]+(-0.913218271084)*x[1]**o+(0.991977570414)*x[1]+(0.626241160161)*x[2]**o+(0.667843914102)*x[2]
        arg[1,1,1]=(0.636818650335)*x[0]**o+(-0.174996100829)*x[0]+(0.77317621688)*x[1]**o+(0.369280208456)*x[1]+(-0.0130124322187)*x[2]**o+(0.857098751007)*x[2]
        arg[2,0,0]=(-0.415847479141)*x[0]**o+(0.675986357495)*x[0]+(0.704926614111)*x[1]**o+(-0.166544295134)*x[1]+(-0.00917226654826)*x[2]**o+(-0.896432190739)*x[2]
        arg[2,0,1]=(-0.882383192316)*x[0]**o+(0.11122730635)*x[0]+(-0.693587089789)*x[1]**o+(-0.0905457091132)*x[1]+(-0.610526000157)*x[2]**o+(-0.992542379681)*x[2]
        arg[2,1,0]=(-0.476987072283)*x[0]**o+(-0.450296923812)*x[0]+(0.905270434608)*x[1]**o+(0.976270820516)*x[1]+(-0.233422717373)*x[2]**o+(-0.641253735386)*x[2]
        arg[2,1,1]=(0.873832514958)*x[0]**o+(-0.478059419217)*x[0]+(-0.941650455606)*x[1]**o+(0.307250152442)*x[1]+(0.865160620819)*x[2]**o+(0.961340749451)*x[2]
        arg[3,0,0]=(-0.443837945036)*x[0]**o+(-0.688773058425)*x[0]+(-0.654988618952)*x[1]**o+(-0.259511611372)*x[1]+(-0.931332326494)*x[2]**o+(0.939379631125)*x[2]
        arg[3,0,1]=(-0.181860312567)*x[0]**o+(-0.229066064127)*x[0]+(-0.359408642857)*x[1]**o+(-0.47945907585)*x[1]+(0.312345402782)*x[2]**o+(0.559298232285)*x[2]
        arg[3,1,0]=(-0.372379819015)*x[0]**o+(0.308179383085)*x[0]+(0.123613160006)*x[1]**o+(0.191944822984)*x[1]+(0.950869988808)*x[2]**o+(0.229360419984)*x[2]
        arg[3,1,1]=(-0.803126271269)*x[0]**o+(0.833324260042)*x[0]+(-0.917001746509)*x[1]**o+(-0.649776906898)*x[1]+(0.49624769362)*x[2]**o+(0.414080802594)*x[2]
        arg[4,0,0]=(-0.951747333891)*x[0]**o+(0.5919442178)*x[0]+(0.121310623341)*x[1]**o+(-0.321939845724)*x[1]+(0.358051737083)*x[2]**o+(-0.222801347833)*x[2]
        arg[4,0,1]=(-0.59462534105)*x[0]**o+(0.943860617698)*x[0]+(0.787822231831)*x[1]**o+(-0.0913435772425)*x[1]+(0.831651946729)*x[2]**o+(-0.903604465879)*x[2]
        arg[4,1,0]=(0.68655463207)*x[0]**o+(-0.98918689213)*x[0]+(0.251300089098)*x[1]**o+(-0.317828131349)*x[1]+(-0.105934216764)*x[2]**o+(-0.874506361945)*x[2]
        arg[4,1,1]=(-0.80544616293)*x[0]**o+(-0.321523684723)*x[0]+(0.874455533992)*x[1]**o+(0.200912811442)*x[1]+(-0.646776981862)*x[2]**o+(-0.988728118812)*x[2]
        arg[5,0,0]=(0.413387141328)*x[0]**o+(0.349210032211)*x[0]+(0.80870689028)*x[1]**o+(-0.633296311752)*x[1]+(0.24621029112)*x[2]**o+(0.853398479355)*x[2]
        arg[5,0,1]=(0.398787817372)*x[0]**o+(0.662875239272)*x[0]+(-0.719962467558)*x[1]**o+(0.49562860294)*x[1]+(-0.694859272211)*x[2]**o+(-0.792616661463)*x[2]
        arg[5,1,0]=(-0.298838035128)*x[0]**o+(0.121578680468)*x[0]+(-0.931062588633)*x[1]**o+(-0.0575832984853)*x[1]+(-0.696140085453)*x[2]**o+(0.836738246058)*x[2]
        arg[5,1,1]=(-0.949626960081)*x[0]**o+(-0.449843145228)*x[0]+(-0.304260073311)*x[1]**o+(-0.258880519466)*x[1]+(0.880466679751)*x[2]**o+(0.97505579364)*x[2]
        ref[0,0,0]=(0.369058215952)/(o+1.)+(-0.860734549029)+(0.638389367885)*0.5**o
        ref[0,0,1]=(-0.937845669147)/(o+1.)+(1.14543097594)+(0.859011160063)*0.5**o
        ref[0,1,0]=(-0.725061432954)/(o+1.)+(0.291104548855)+(-0.998681640062)*0.5**o
        ref[0,1,1]=(-0.540637249448)/(o+1.)+(-0.500606905428)+(0.1340261052)*0.5**o
        ref[1,0,0]=(0.902262161102)/(o+1.)+(0.666322787672)+(0.251201824008)*0.5**o
        ref[1,0,1]=(-1.37166876542)/(o+1.)+(-0.452205179789)+(-0.258843593921)*0.5**o
        ref[1,1,0]=(-0.286977110923)/(o+1.)+(0.888993508815)+(-0.854529145157)*0.5**o
        ref[1,1,1]=(0.760163784662)/(o+1.)+(0.525691429317)+(0.636818650335)*0.5**o
        ref[2,0,0]=(0.695754347563)/(o+1.)+(-0.193495064189)+(-0.415847479141)*0.5**o
        ref[2,0,1]=(-1.30411308995)/(o+1.)+(-0.485930391222)+(-0.882383192316)*0.5**o
        ref[2,1,0]=(0.671847717235)/(o+1.)+(-0.0576399193408)+(-0.476987072283)*0.5**o
        ref[2,1,1]=(-0.0764898347868)/(o+1.)+(0.395265741338)+(0.873832514958)*0.5**o
        ref[3,0,0]=(-1.58632094545)/(o+1.)+(-0.00445251933618)+(-0.443837945036)*0.5**o
        ref[3,0,1]=(-0.0470632400747)/(o+1.)+(-0.0746134538458)+(-0.181860312567)*0.5**o
        ref[3,1,0]=(1.07448314881)/(o+1.)+(0.364742313027)+(-0.372379819015)*0.5**o
        ref[3,1,1]=(-0.420754052889)/(o+1.)+(0.298814077869)+(-0.803126271269)*0.5**o
        ref[4,0,0]=(0.479362360424)/(o+1.)+(0.0236015121217)+(-0.951747333891)*0.5**o
        ref[4,0,1]=(1.61947417856)/(o+1.)+(-0.0255437127116)+(-0.59462534105)*0.5**o
        ref[4,1,0]=(0.145365872334)/(o+1.)+(-1.09076069271)+(0.68655463207)*0.5**o
        ref[4,1,1]=(0.22767855213)/(o+1.)+(-0.554669496046)+(-0.80544616293)*0.5**o
        ref[5,0,0]=(1.0549171814)/(o+1.)+(0.284656099907)+(0.413387141328)*0.5**o
        ref[5,0,1]=(-1.41482173977)/(o+1.)+(0.182943590374)+(0.398787817372)*0.5**o
        ref[5,1,0]=(-1.62720267409)/(o+1.)+(0.450366814021)+(-0.298838035128)*0.5**o
        ref[5,1,1]=(0.576206606439)/(o+1.)+(0.133166064473)+(-0.949626960081)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onReducedFunctionOnContactZero_fromData_ReducedFunctionOnContactZero_rank4(self):
      """
      tests integral of rank 4 Data on the ReducedFunctionOnContactZero

      assumptions: ReducedFunctionOnContactZero(self.domain) exists
                   self.domain supports integral on ReducedFunctionOnContactZero
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=ReducedFunctionOnContactZero(self.domain)
      w=ReducedFunctionOnContactZero(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.392628958003)*x[0]**o+(0.186084293395)*x[0]+(0.965300679335)*x[1]**o+(0.0986282244747)*x[1]
        arg[0,0,0,1]=(-0.503312425492)*x[0]**o+(-0.205791466802)*x[0]+(0.384782163488)*x[1]**o+(-0.608371097114)*x[1]
        arg[0,0,1,0]=(0.660162511699)*x[0]**o+(-0.263141686561)*x[0]+(-0.765235012795)*x[1]**o+(-0.371233144611)*x[1]
        arg[0,0,1,1]=(-0.986935412732)*x[0]**o+(0.645433351743)*x[0]+(-0.451665346257)*x[1]**o+(-0.713815759872)*x[1]
        arg[0,0,2,0]=(0.632875178502)*x[0]**o+(-0.257542208821)*x[0]+(-0.695553902133)*x[1]**o+(-0.677920876645)*x[1]
        arg[0,0,2,1]=(0.102625648731)*x[0]**o+(0.511896902175)*x[0]+(0.345909088853)*x[1]**o+(-0.0609413722414)*x[1]
        arg[0,1,0,0]=(-0.0108106931425)*x[0]**o+(-0.595497517075)*x[0]+(0.332480141972)*x[1]**o+(-0.18131109924)*x[1]
        arg[0,1,0,1]=(0.220826593491)*x[0]**o+(-0.202422451222)*x[0]+(0.328480565578)*x[1]**o+(0.851432750557)*x[1]
        arg[0,1,1,0]=(0.22735402466)*x[0]**o+(0.51437836535)*x[0]+(0.292927451477)*x[1]**o+(-0.148121856023)*x[1]
        arg[0,1,1,1]=(0.495079875966)*x[0]**o+(0.582755117761)*x[0]+(-0.322061131008)*x[1]**o+(0.953437449986)*x[1]
        arg[0,1,2,0]=(0.968583889554)*x[0]**o+(-0.593253483731)*x[0]+(0.286037351205)*x[1]**o+(0.997314411263)*x[1]
        arg[0,1,2,1]=(0.361498191707)*x[0]**o+(0.633275777697)*x[0]+(0.891305263219)*x[1]**o+(-0.194468605638)*x[1]
        arg[0,2,0,0]=(0.185647566516)*x[0]**o+(0.101448861247)*x[0]+(0.0870608077911)*x[1]**o+(-0.86477908948)*x[1]
        arg[0,2,0,1]=(0.227262327826)*x[0]**o+(0.812586702875)*x[0]+(-0.0229924234496)*x[1]**o+(-0.323581477982)*x[1]
        arg[0,2,1,0]=(0.0405753322301)*x[0]**o+(0.891289727227)*x[0]+(0.957377329048)*x[1]**o+(-0.982119724379)*x[1]
        arg[0,2,1,1]=(-0.546046301701)*x[0]**o+(0.628511534929)*x[0]+(-0.628262263897)*x[1]**o+(-0.454315708999)*x[1]
        arg[0,2,2,0]=(-0.828455793175)*x[0]**o+(0.190147126828)*x[0]+(0.499595322279)*x[1]**o+(-0.0882297087937)*x[1]
        arg[0,2,2,1]=(0.926138085222)*x[0]**o+(0.402659001583)*x[0]+(-0.678432418122)*x[1]**o+(0.236054704896)*x[1]
        arg[0,3,0,0]=(-0.473066058658)*x[0]**o+(0.57622727181)*x[0]+(0.62665109546)*x[1]**o+(0.948409659533)*x[1]
        arg[0,3,0,1]=(0.776790978673)*x[0]**o+(-0.0244752180934)*x[0]+(-0.459252843816)*x[1]**o+(0.616203849528)*x[1]
        arg[0,3,1,0]=(0.34783214571)*x[0]**o+(0.806528120142)*x[0]+(0.445864469556)*x[1]**o+(-0.221961922099)*x[1]
        arg[0,3,1,1]=(0.0717966037463)*x[0]**o+(-0.434117388366)*x[0]+(0.577708844669)*x[1]**o+(-0.548519157145)*x[1]
        arg[0,3,2,0]=(-0.548573971799)*x[0]**o+(-0.926704595492)*x[0]+(-0.33936340707)*x[1]**o+(0.0687290934591)*x[1]
        arg[0,3,2,1]=(-0.28309240461)*x[0]**o+(0.12914675768)*x[0]+(-0.833323497891)*x[1]**o+(-0.577727985559)*x[1]
        arg[0,4,0,0]=(-0.105416428922)*x[0]**o+(0.41564239907)*x[0]+(0.146657173319)*x[1]**o+(-0.0141417271845)*x[1]
        arg[0,4,0,1]=(-0.876111094069)*x[0]**o+(0.491957669628)*x[0]+(0.534941970665)*x[1]**o+(-0.995507289274)*x[1]
        arg[0,4,1,0]=(-0.678996394303)*x[0]**o+(0.183159947613)*x[0]+(-0.739368558349)*x[1]**o+(0.274319401135)*x[1]
        arg[0,4,1,1]=(0.0472145581152)*x[0]**o+(-0.669737608207)*x[0]+(-0.651647023142)*x[1]**o+(0.351208329111)*x[1]
        arg[0,4,2,0]=(0.114936857967)*x[0]**o+(0.564688790898)*x[0]+(-0.524234284672)*x[1]**o+(-0.127615425724)*x[1]
        arg[0,4,2,1]=(0.568834117773)*x[0]**o+(-0.936513147599)*x[0]+(0.133122220811)*x[1]**o+(0.31510090512)*x[1]
        arg[1,0,0,0]=(-0.821241886235)*x[0]**o+(0.779734235239)*x[0]+(-0.132449009841)*x[1]**o+(-0.0179700618848)*x[1]
        arg[1,0,0,1]=(0.624372422235)*x[0]**o+(-0.752831611955)*x[0]+(-0.573597121344)*x[1]**o+(0.376463665377)*x[1]
        arg[1,0,1,0]=(-0.568749808355)*x[0]**o+(-0.697521100346)*x[0]+(-0.249786553024)*x[1]**o+(-0.624593568423)*x[1]
        arg[1,0,1,1]=(0.0765557360835)*x[0]**o+(-0.744005823966)*x[0]+(0.59246455477)*x[1]**o+(0.231310687774)*x[1]
        arg[1,0,2,0]=(0.909614864187)*x[0]**o+(0.350543076144)*x[0]+(0.0689432766231)*x[1]**o+(-0.88465698758)*x[1]
        arg[1,0,2,1]=(0.0610206203468)*x[0]**o+(0.620401328094)*x[0]+(0.378273922837)*x[1]**o+(0.91988744934)*x[1]
        arg[1,1,0,0]=(0.917249169051)*x[0]**o+(0.756656473206)*x[0]+(0.728786835113)*x[1]**o+(0.116994159781)*x[1]
        arg[1,1,0,1]=(0.331129708654)*x[0]**o+(-0.14342302905)*x[0]+(-0.418402258918)*x[1]**o+(-0.808676850609)*x[1]
        arg[1,1,1,0]=(-0.527408405279)*x[0]**o+(-0.671440239997)*x[0]+(0.989603613532)*x[1]**o+(0.364280392731)*x[1]
        arg[1,1,1,1]=(-0.189543145217)*x[0]**o+(0.14762392216)*x[0]+(0.11459047938)*x[1]**o+(-0.760477267267)*x[1]
        arg[1,1,2,0]=(0.385274935809)*x[0]**o+(0.527744200995)*x[0]+(-0.862654310338)*x[1]**o+(0.904005467598)*x[1]
        arg[1,1,2,1]=(-0.380421555378)*x[0]**o+(0.598300323394)*x[0]+(-0.541819816686)*x[1]**o+(0.446934572528)*x[1]
        arg[1,2,0,0]=(0.446781075767)*x[0]**o+(0.906830344178)*x[0]+(0.158734254777)*x[1]**o+(-0.722982586936)*x[1]
        arg[1,2,0,1]=(0.927811104869)*x[0]**o+(-0.446795135388)*x[0]+(-0.90301726295)*x[1]**o+(0.289530807979)*x[1]
        arg[1,2,1,0]=(0.741527109588)*x[0]**o+(-0.370029728344)*x[0]+(0.157695458386)*x[1]**o+(-0.620804465293)*x[1]
        arg[1,2,1,1]=(0.193242816889)*x[0]**o+(0.455789913444)*x[0]+(0.0491001091271)*x[1]**o+(-0.942127264151)*x[1]
        arg[1,2,2,0]=(0.220709813709)*x[0]**o+(-0.713472965425)*x[0]+(-0.736262418061)*x[1]**o+(0.341347605023)*x[1]
        arg[1,2,2,1]=(0.0034506429721)*x[0]**o+(0.397289753453)*x[0]+(-0.167013881353)*x[1]**o+(0.390364445974)*x[1]
        arg[1,3,0,0]=(0.321471168172)*x[0]**o+(-0.237896433618)*x[0]+(-0.199843421009)*x[1]**o+(0.811319091471)*x[1]
        arg[1,3,0,1]=(-0.0799622908037)*x[0]**o+(-0.66155883346)*x[0]+(0.0303112997369)*x[1]**o+(-0.199256497975)*x[1]
        arg[1,3,1,0]=(-0.12051943489)*x[0]**o+(0.674374603779)*x[0]+(0.335933608833)*x[1]**o+(0.366671959553)*x[1]
        arg[1,3,1,1]=(-0.587242845986)*x[0]**o+(-0.495652656487)*x[0]+(0.10609614346)*x[1]**o+(0.177210588935)*x[1]
        arg[1,3,2,0]=(0.947782336638)*x[0]**o+(-0.327283099142)*x[0]+(-0.0280648836948)*x[1]**o+(0.133334543825)*x[1]
        arg[1,3,2,1]=(-0.0887414134508)*x[0]**o+(0.640694455701)*x[0]+(0.266916366237)*x[1]**o+(0.345886606493)*x[1]
        arg[1,4,0,0]=(-0.138184872938)*x[0]**o+(0.658635897645)*x[0]+(-0.00526240442626)*x[1]**o+(0.800443113371)*x[1]
        arg[1,4,0,1]=(-0.549831917928)*x[0]**o+(-0.666733703837)*x[0]+(0.400982699808)*x[1]**o+(0.739099662817)*x[1]
        arg[1,4,1,0]=(-0.137046722563)*x[0]**o+(-0.818450480933)*x[0]+(-0.827665726293)*x[1]**o+(-0.286585607624)*x[1]
        arg[1,4,1,1]=(-0.474156202572)*x[0]**o+(-0.855381707793)*x[0]+(0.113318564035)*x[1]**o+(0.218410511108)*x[1]
        arg[1,4,2,0]=(0.18476112439)*x[0]**o+(-0.964533442641)*x[0]+(-0.968611931598)*x[1]**o+(0.0973026144203)*x[1]
        arg[1,4,2,1]=(-0.372995418552)*x[0]**o+(0.998322146553)*x[0]+(-0.384672070744)*x[1]**o+(0.47536977268)*x[1]
        arg[2,0,0,0]=(0.882400857222)*x[0]**o+(0.455324640128)*x[0]+(0.764297272234)*x[1]**o+(0.498669000791)*x[1]
        arg[2,0,0,1]=(0.894017590794)*x[0]**o+(0.217984119634)*x[0]+(-0.0934000967188)*x[1]**o+(-0.886960376026)*x[1]
        arg[2,0,1,0]=(-0.382158057826)*x[0]**o+(-0.897639065996)*x[0]+(0.819002513335)*x[1]**o+(-0.0739374877518)*x[1]
        arg[2,0,1,1]=(-0.259771272131)*x[0]**o+(-0.533540086977)*x[0]+(0.366917477923)*x[1]**o+(-0.751904525722)*x[1]
        arg[2,0,2,0]=(-0.33946164111)*x[0]**o+(-0.317441178182)*x[0]+(-0.857625945147)*x[1]**o+(-0.84647892719)*x[1]
        arg[2,0,2,1]=(0.153353093148)*x[0]**o+(0.832883645984)*x[0]+(-0.359476461654)*x[1]**o+(0.332390899654)*x[1]
        arg[2,1,0,0]=(0.780980225775)*x[0]**o+(0.619330950566)*x[0]+(0.194312746335)*x[1]**o+(0.595311976655)*x[1]
        arg[2,1,0,1]=(-0.334149973283)*x[0]**o+(-0.670178805858)*x[0]+(-0.313038325156)*x[1]**o+(0.793149618246)*x[1]
        arg[2,1,1,0]=(-0.832852528577)*x[0]**o+(-0.606634692926)*x[0]+(-0.0643816133251)*x[1]**o+(-0.663765987285)*x[1]
        arg[2,1,1,1]=(0.300975732963)*x[0]**o+(0.965431406497)*x[0]+(-0.618456119184)*x[1]**o+(0.459530421885)*x[1]
        arg[2,1,2,0]=(0.122326105776)*x[0]**o+(0.0940802696378)*x[0]+(-0.986852501828)*x[1]**o+(0.370428043236)*x[1]
        arg[2,1,2,1]=(0.447916985436)*x[0]**o+(-0.161926311978)*x[0]+(-0.0933563424166)*x[1]**o+(0.836290263421)*x[1]
        arg[2,2,0,0]=(0.032076347277)*x[0]**o+(-0.447517143643)*x[0]+(-0.053769157756)*x[1]**o+(-0.0242981207801)*x[1]
        arg[2,2,0,1]=(-0.403935703163)*x[0]**o+(-0.960750093184)*x[0]+(-0.287956277998)*x[1]**o+(-0.168614701082)*x[1]
        arg[2,2,1,0]=(-0.205077009492)*x[0]**o+(-0.409536289697)*x[0]+(0.359071620541)*x[1]**o+(-0.466455759025)*x[1]
        arg[2,2,1,1]=(0.443658057141)*x[0]**o+(-0.748681939527)*x[0]+(0.2153167916)*x[1]**o+(-0.475899799738)*x[1]
        arg[2,2,2,0]=(-0.463879681251)*x[0]**o+(0.758317222041)*x[0]+(0.608221916958)*x[1]**o+(-0.237798399085)*x[1]
        arg[2,2,2,1]=(0.533981754168)*x[0]**o+(0.590125912931)*x[0]+(0.8957267808)*x[1]**o+(0.294612478767)*x[1]
        arg[2,3,0,0]=(0.499387701403)*x[0]**o+(0.0379132293822)*x[0]+(0.717384215206)*x[1]**o+(0.754497698503)*x[1]
        arg[2,3,0,1]=(0.866500398882)*x[0]**o+(0.23678788012)*x[0]+(0.395132533716)*x[1]**o+(-0.451619930593)*x[1]
        arg[2,3,1,0]=(-0.879663354176)*x[0]**o+(-0.328937922978)*x[0]+(0.688771917581)*x[1]**o+(-0.87392051489)*x[1]
        arg[2,3,1,1]=(0.291214284356)*x[0]**o+(-0.783159133488)*x[0]+(0.240960084233)*x[1]**o+(0.339043900275)*x[1]
        arg[2,3,2,0]=(0.47064755192)*x[0]**o+(-0.42618815064)*x[0]+(-0.505342655776)*x[1]**o+(0.408218280069)*x[1]
        arg[2,3,2,1]=(0.775731405735)*x[0]**o+(-0.768750178591)*x[0]+(-0.896333275063)*x[1]**o+(0.286558848957)*x[1]
        arg[2,4,0,0]=(-0.7221248688)*x[0]**o+(0.612509823967)*x[0]+(0.436744528143)*x[1]**o+(-0.761502758764)*x[1]
        arg[2,4,0,1]=(-0.742652222385)*x[0]**o+(0.214811997988)*x[0]+(-0.031320962337)*x[1]**o+(-0.320134664789)*x[1]
        arg[2,4,1,0]=(-0.529575527815)*x[0]**o+(0.623770482091)*x[0]+(-0.50790148262)*x[1]**o+(0.629874274271)*x[1]
        arg[2,4,1,1]=(0.535631957883)*x[0]**o+(0.439578618993)*x[0]+(-0.770747718564)*x[1]**o+(0.0798696116058)*x[1]
        arg[2,4,2,0]=(0.81688202721)*x[0]**o+(-0.0301054197201)*x[0]+(0.613495031968)*x[1]**o+(-0.153619455295)*x[1]
        arg[2,4,2,1]=(0.493471708612)*x[0]**o+(0.423626757568)*x[0]+(0.093174141001)*x[1]**o+(0.307452108722)*x[1]
        arg[3,0,0,0]=(-0.0668630243284)*x[0]**o+(-0.577709692979)*x[0]+(0.984307685082)*x[1]**o+(-0.171487030152)*x[1]
        arg[3,0,0,1]=(-0.210452484293)*x[0]**o+(-0.130476372055)*x[0]+(-0.693596086089)*x[1]**o+(-0.692143724329)*x[1]
        arg[3,0,1,0]=(-0.34853285523)*x[0]**o+(-0.369649709966)*x[0]+(0.824648362105)*x[1]**o+(0.733046240399)*x[1]
        arg[3,0,1,1]=(0.861903045965)*x[0]**o+(0.609535692452)*x[0]+(0.681514895836)*x[1]**o+(0.66773275418)*x[1]
        arg[3,0,2,0]=(0.0197670342844)*x[0]**o+(-0.251645539228)*x[0]+(-0.0390293468405)*x[1]**o+(-0.00955605638373)*x[1]
        arg[3,0,2,1]=(-0.499093940395)*x[0]**o+(0.660356623112)*x[0]+(0.969057635866)*x[1]**o+(-0.821377317197)*x[1]
        arg[3,1,0,0]=(-0.955568800095)*x[0]**o+(0.557877698917)*x[0]+(0.829599262959)*x[1]**o+(0.933736802562)*x[1]
        arg[3,1,0,1]=(-0.910307638058)*x[0]**o+(-0.3287646946)*x[0]+(-0.777548579252)*x[1]**o+(-0.971744353969)*x[1]
        arg[3,1,1,0]=(0.22353820834)*x[0]**o+(0.904479243377)*x[0]+(-0.108768166204)*x[1]**o+(0.823767232517)*x[1]
        arg[3,1,1,1]=(0.371995768061)*x[0]**o+(0.747630377273)*x[0]+(0.931152588498)*x[1]**o+(-0.0154330268188)*x[1]
        arg[3,1,2,0]=(0.4567417369)*x[0]**o+(0.424951293892)*x[0]+(-0.14032259215)*x[1]**o+(0.603329614924)*x[1]
        arg[3,1,2,1]=(-0.855100213731)*x[0]**o+(-0.667832882495)*x[0]+(0.942530922378)*x[1]**o+(0.55861156259)*x[1]
        arg[3,2,0,0]=(0.17190936572)*x[0]**o+(-0.904098615762)*x[0]+(0.991497679624)*x[1]**o+(0.373020358381)*x[1]
        arg[3,2,0,1]=(-0.683587206983)*x[0]**o+(0.718257814718)*x[0]+(-0.730243933765)*x[1]**o+(-0.583892189979)*x[1]
        arg[3,2,1,0]=(0.051672736629)*x[0]**o+(-0.991172563807)*x[0]+(0.36605487466)*x[1]**o+(0.680719877794)*x[1]
        arg[3,2,1,1]=(0.438014662973)*x[0]**o+(0.172969450125)*x[0]+(-0.424515284177)*x[1]**o+(0.615315132547)*x[1]
        arg[3,2,2,0]=(-0.709499532674)*x[0]**o+(0.985097212621)*x[0]+(0.326610298255)*x[1]**o+(0.994701669258)*x[1]
        arg[3,2,2,1]=(-0.657647707844)*x[0]**o+(-0.683954483768)*x[0]+(-0.846657945161)*x[1]**o+(0.852308904063)*x[1]
        arg[3,3,0,0]=(0.21346320453)*x[0]**o+(-0.181665119768)*x[0]+(0.549866100727)*x[1]**o+(-0.0858045316871)*x[1]
        arg[3,3,0,1]=(0.249957470685)*x[0]**o+(0.537464773638)*x[0]+(0.0224330511636)*x[1]**o+(0.114623595638)*x[1]
        arg[3,3,1,0]=(0.166507029533)*x[0]**o+(0.232150195873)*x[0]+(0.119453667263)*x[1]**o+(-0.261110960642)*x[1]
        arg[3,3,1,1]=(0.358248720077)*x[0]**o+(-0.871911261059)*x[0]+(0.57743453064)*x[1]**o+(0.950322066335)*x[1]
        arg[3,3,2,0]=(-0.896899724447)*x[0]**o+(-0.898884767716)*x[0]+(0.63832012742)*x[1]**o+(-0.833800375634)*x[1]
        arg[3,3,2,1]=(0.425194254583)*x[0]**o+(0.161328467295)*x[0]+(0.496219405916)*x[1]**o+(0.380935896093)*x[1]
        arg[3,4,0,0]=(0.793902581592)*x[0]**o+(-0.33428511164)*x[0]+(-0.283510438468)*x[1]**o+(0.228913046074)*x[1]
        arg[3,4,0,1]=(0.372028980539)*x[0]**o+(-0.388909011671)*x[0]+(-0.259885631206)*x[1]**o+(-0.950910172897)*x[1]
        arg[3,4,1,0]=(-0.80489673786)*x[0]**o+(-0.828175420038)*x[0]+(0.895070489116)*x[1]**o+(-0.742272132103)*x[1]
        arg[3,4,1,1]=(0.272535126101)*x[0]**o+(0.902668974891)*x[0]+(0.399816667015)*x[1]**o+(-0.953186908789)*x[1]
        arg[3,4,2,0]=(0.0341775288582)*x[0]**o+(0.962241492639)*x[0]+(0.906020184876)*x[1]**o+(-0.400536255726)*x[1]
        arg[3,4,2,1]=(-0.745016464485)*x[0]**o+(-0.85241829747)*x[0]+(-0.648936544722)*x[1]**o+(-0.997030074218)*x[1]
        ref[0,0,0,0]=(0.965300679335)/(o+1.)+(0.142356258935)+(0.392628958003)*0.5**o
        ref[0,0,0,1]=(0.384782163488)/(o+1.)+(-0.407081281958)+(-0.503312425492)*0.5**o
        ref[0,0,1,0]=(-0.765235012795)/(o+1.)+(-0.317187415586)+(0.660162511699)*0.5**o
        ref[0,0,1,1]=(-0.451665346257)/(o+1.)+(-0.0341912040643)+(-0.986935412732)*0.5**o
        ref[0,0,2,0]=(-0.695553902133)/(o+1.)+(-0.467731542733)+(0.632875178502)*0.5**o
        ref[0,0,2,1]=(0.345909088853)/(o+1.)+(0.225477764967)+(0.102625648731)*0.5**o
        ref[0,1,0,0]=(0.332480141972)/(o+1.)+(-0.388404308158)+(-0.0108106931425)*0.5**o
        ref[0,1,0,1]=(0.328480565578)/(o+1.)+(0.324505149668)+(0.220826593491)*0.5**o
        ref[0,1,1,0]=(0.292927451477)/(o+1.)+(0.183128254664)+(0.22735402466)*0.5**o
        ref[0,1,1,1]=(-0.322061131008)/(o+1.)+(0.768096283874)+(0.495079875966)*0.5**o
        ref[0,1,2,0]=(0.286037351205)/(o+1.)+(0.202030463766)+(0.968583889554)*0.5**o
        ref[0,1,2,1]=(0.891305263219)/(o+1.)+(0.219403586029)+(0.361498191707)*0.5**o
        ref[0,2,0,0]=(0.0870608077911)/(o+1.)+(-0.381665114117)+(0.185647566516)*0.5**o
        ref[0,2,0,1]=(-0.0229924234496)/(o+1.)+(0.244502612446)+(0.227262327826)*0.5**o
        ref[0,2,1,0]=(0.957377329048)/(o+1.)+(-0.0454149985758)+(0.0405753322301)*0.5**o
        ref[0,2,1,1]=(-0.628262263897)/(o+1.)+(0.0870979129647)+(-0.546046301701)*0.5**o
        ref[0,2,2,0]=(0.499595322279)/(o+1.)+(0.0509587090171)+(-0.828455793175)*0.5**o
        ref[0,2,2,1]=(-0.678432418122)/(o+1.)+(0.319356853239)+(0.926138085222)*0.5**o
        ref[0,3,0,0]=(0.62665109546)/(o+1.)+(0.762318465671)+(-0.473066058658)*0.5**o
        ref[0,3,0,1]=(-0.459252843816)/(o+1.)+(0.295864315717)+(0.776790978673)*0.5**o
        ref[0,3,1,0]=(0.445864469556)/(o+1.)+(0.292283099022)+(0.34783214571)*0.5**o
        ref[0,3,1,1]=(0.577708844669)/(o+1.)+(-0.491318272755)+(0.0717966037463)*0.5**o
        ref[0,3,2,0]=(-0.33936340707)/(o+1.)+(-0.428987751016)+(-0.548573971799)*0.5**o
        ref[0,3,2,1]=(-0.833323497891)/(o+1.)+(-0.224290613939)+(-0.28309240461)*0.5**o
        ref[0,4,0,0]=(0.146657173319)/(o+1.)+(0.200750335943)+(-0.105416428922)*0.5**o
        ref[0,4,0,1]=(0.534941970665)/(o+1.)+(-0.251774809823)+(-0.876111094069)*0.5**o
        ref[0,4,1,0]=(-0.739368558349)/(o+1.)+(0.228739674374)+(-0.678996394303)*0.5**o
        ref[0,4,1,1]=(-0.651647023142)/(o+1.)+(-0.159264639548)+(0.0472145581152)*0.5**o
        ref[0,4,2,0]=(-0.524234284672)/(o+1.)+(0.218536682587)+(0.114936857967)*0.5**o
        ref[0,4,2,1]=(0.133122220811)/(o+1.)+(-0.31070612124)+(0.568834117773)*0.5**o
        ref[1,0,0,0]=(-0.132449009841)/(o+1.)+(0.380882086677)+(-0.821241886235)*0.5**o
        ref[1,0,0,1]=(-0.573597121344)/(o+1.)+(-0.188183973289)+(0.624372422235)*0.5**o
        ref[1,0,1,0]=(-0.249786553024)/(o+1.)+(-0.661057334385)+(-0.568749808355)*0.5**o
        ref[1,0,1,1]=(0.59246455477)/(o+1.)+(-0.256347568096)+(0.0765557360835)*0.5**o
        ref[1,0,2,0]=(0.0689432766231)/(o+1.)+(-0.267056955718)+(0.909614864187)*0.5**o
        ref[1,0,2,1]=(0.378273922837)/(o+1.)+(0.770144388717)+(0.0610206203468)*0.5**o
        ref[1,1,0,0]=(0.728786835113)/(o+1.)+(0.436825316493)+(0.917249169051)*0.5**o
        ref[1,1,0,1]=(-0.418402258918)/(o+1.)+(-0.47604993983)+(0.331129708654)*0.5**o
        ref[1,1,1,0]=(0.989603613532)/(o+1.)+(-0.153579923633)+(-0.527408405279)*0.5**o
        ref[1,1,1,1]=(0.11459047938)/(o+1.)+(-0.306426672553)+(-0.189543145217)*0.5**o
        ref[1,1,2,0]=(-0.862654310338)/(o+1.)+(0.715874834296)+(0.385274935809)*0.5**o
        ref[1,1,2,1]=(-0.541819816686)/(o+1.)+(0.522617447961)+(-0.380421555378)*0.5**o
        ref[1,2,0,0]=(0.158734254777)/(o+1.)+(0.0919238786215)+(0.446781075767)*0.5**o
        ref[1,2,0,1]=(-0.90301726295)/(o+1.)+(-0.0786321637044)+(0.927811104869)*0.5**o
        ref[1,2,1,0]=(0.157695458386)/(o+1.)+(-0.495417096819)+(0.741527109588)*0.5**o
        ref[1,2,1,1]=(0.0491001091271)/(o+1.)+(-0.243168675353)+(0.193242816889)*0.5**o
        ref[1,2,2,0]=(-0.736262418061)/(o+1.)+(-0.186062680201)+(0.220709813709)*0.5**o
        ref[1,2,2,1]=(-0.167013881353)/(o+1.)+(0.393827099713)+(0.0034506429721)*0.5**o
        ref[1,3,0,0]=(-0.199843421009)/(o+1.)+(0.286711328927)+(0.321471168172)*0.5**o
        ref[1,3,0,1]=(0.0303112997369)/(o+1.)+(-0.430407665717)+(-0.0799622908037)*0.5**o
        ref[1,3,1,0]=(0.335933608833)/(o+1.)+(0.520523281666)+(-0.12051943489)*0.5**o
        ref[1,3,1,1]=(0.10609614346)/(o+1.)+(-0.159221033776)+(-0.587242845986)*0.5**o
        ref[1,3,2,0]=(-0.0280648836948)/(o+1.)+(-0.0969742776586)+(0.947782336638)*0.5**o
        ref[1,3,2,1]=(0.266916366237)/(o+1.)+(0.493290531097)+(-0.0887414134508)*0.5**o
        ref[1,4,0,0]=(-0.00526240442626)/(o+1.)+(0.729539505508)+(-0.138184872938)*0.5**o
        ref[1,4,0,1]=(0.400982699808)/(o+1.)+(0.0361829794896)+(-0.549831917928)*0.5**o
        ref[1,4,1,0]=(-0.827665726293)/(o+1.)+(-0.552518044279)+(-0.137046722563)*0.5**o
        ref[1,4,1,1]=(0.113318564035)/(o+1.)+(-0.318485598343)+(-0.474156202572)*0.5**o
        ref[1,4,2,0]=(-0.968611931598)/(o+1.)+(-0.433615414111)+(0.18476112439)*0.5**o
        ref[1,4,2,1]=(-0.384672070744)/(o+1.)+(0.736845959617)+(-0.372995418552)*0.5**o
        ref[2,0,0,0]=(0.764297272234)/(o+1.)+(0.476996820459)+(0.882400857222)*0.5**o
        ref[2,0,0,1]=(-0.0934000967188)/(o+1.)+(-0.334488128196)+(0.894017590794)*0.5**o
        ref[2,0,1,0]=(0.819002513335)/(o+1.)+(-0.485788276874)+(-0.382158057826)*0.5**o
        ref[2,0,1,1]=(0.366917477923)/(o+1.)+(-0.64272230635)+(-0.259771272131)*0.5**o
        ref[2,0,2,0]=(-0.857625945147)/(o+1.)+(-0.581960052686)+(-0.33946164111)*0.5**o
        ref[2,0,2,1]=(-0.359476461654)/(o+1.)+(0.582637272819)+(0.153353093148)*0.5**o
        ref[2,1,0,0]=(0.194312746335)/(o+1.)+(0.607321463611)+(0.780980225775)*0.5**o
        ref[2,1,0,1]=(-0.313038325156)/(o+1.)+(0.0614854061938)+(-0.334149973283)*0.5**o
        ref[2,1,1,0]=(-0.0643816133251)/(o+1.)+(-0.635200340106)+(-0.832852528577)*0.5**o
        ref[2,1,1,1]=(-0.618456119184)/(o+1.)+(0.712480914191)+(0.300975732963)*0.5**o
        ref[2,1,2,0]=(-0.986852501828)/(o+1.)+(0.232254156437)+(0.122326105776)*0.5**o
        ref[2,1,2,1]=(-0.0933563424166)/(o+1.)+(0.337181975722)+(0.447916985436)*0.5**o
        ref[2,2,0,0]=(-0.053769157756)/(o+1.)+(-0.235907632211)+(0.032076347277)*0.5**o
        ref[2,2,0,1]=(-0.287956277998)/(o+1.)+(-0.564682397133)+(-0.403935703163)*0.5**o
        ref[2,2,1,0]=(0.359071620541)/(o+1.)+(-0.437996024361)+(-0.205077009492)*0.5**o
        ref[2,2,1,1]=(0.2153167916)/(o+1.)+(-0.612290869632)+(0.443658057141)*0.5**o
        ref[2,2,2,0]=(0.608221916958)/(o+1.)+(0.260259411478)+(-0.463879681251)*0.5**o
        ref[2,2,2,1]=(0.8957267808)/(o+1.)+(0.442369195849)+(0.533981754168)*0.5**o
        ref[2,3,0,0]=(0.717384215206)/(o+1.)+(0.396205463942)+(0.499387701403)*0.5**o
        ref[2,3,0,1]=(0.395132533716)/(o+1.)+(-0.107416025237)+(0.866500398882)*0.5**o
        ref[2,3,1,0]=(0.688771917581)/(o+1.)+(-0.601429218934)+(-0.879663354176)*0.5**o
        ref[2,3,1,1]=(0.240960084233)/(o+1.)+(-0.222057616607)+(0.291214284356)*0.5**o
        ref[2,3,2,0]=(-0.505342655776)/(o+1.)+(-0.00898493528542)+(0.47064755192)*0.5**o
        ref[2,3,2,1]=(-0.896333275063)/(o+1.)+(-0.241095664817)+(0.775731405735)*0.5**o
        ref[2,4,0,0]=(0.436744528143)/(o+1.)+(-0.0744964673986)+(-0.7221248688)*0.5**o
        ref[2,4,0,1]=(-0.031320962337)/(o+1.)+(-0.0526613334003)+(-0.742652222385)*0.5**o
        ref[2,4,1,0]=(-0.50790148262)/(o+1.)+(0.626822378181)+(-0.529575527815)*0.5**o
        ref[2,4,1,1]=(-0.770747718564)/(o+1.)+(0.2597241153)+(0.535631957883)*0.5**o
        ref[2,4,2,0]=(0.613495031968)/(o+1.)+(-0.0918624375077)+(0.81688202721)*0.5**o
        ref[2,4,2,1]=(0.093174141001)/(o+1.)+(0.365539433145)+(0.493471708612)*0.5**o
        ref[3,0,0,0]=(0.984307685082)/(o+1.)+(-0.374598361566)+(-0.0668630243284)*0.5**o
        ref[3,0,0,1]=(-0.693596086089)/(o+1.)+(-0.411310048192)+(-0.210452484293)*0.5**o
        ref[3,0,1,0]=(0.824648362105)/(o+1.)+(0.181698265217)+(-0.34853285523)*0.5**o
        ref[3,0,1,1]=(0.681514895836)/(o+1.)+(0.638634223316)+(0.861903045965)*0.5**o
        ref[3,0,2,0]=(-0.0390293468405)/(o+1.)+(-0.130600797806)+(0.0197670342844)*0.5**o
        ref[3,0,2,1]=(0.969057635866)/(o+1.)+(-0.0805103470423)+(-0.499093940395)*0.5**o
        ref[3,1,0,0]=(0.829599262959)/(o+1.)+(0.745807250739)+(-0.955568800095)*0.5**o
        ref[3,1,0,1]=(-0.777548579252)/(o+1.)+(-0.650254524284)+(-0.910307638058)*0.5**o
        ref[3,1,1,0]=(-0.108768166204)/(o+1.)+(0.864123237947)+(0.22353820834)*0.5**o
        ref[3,1,1,1]=(0.931152588498)/(o+1.)+(0.366098675227)+(0.371995768061)*0.5**o
        ref[3,1,2,0]=(-0.14032259215)/(o+1.)+(0.514140454408)+(0.4567417369)*0.5**o
        ref[3,1,2,1]=(0.942530922378)/(o+1.)+(-0.0546106599524)+(-0.855100213731)*0.5**o
        ref[3,2,0,0]=(0.991497679624)/(o+1.)+(-0.26553912869)+(0.17190936572)*0.5**o
        ref[3,2,0,1]=(-0.730243933765)/(o+1.)+(0.0671828123695)+(-0.683587206983)*0.5**o
        ref[3,2,1,0]=(0.36605487466)/(o+1.)+(-0.155226343006)+(0.051672736629)*0.5**o
        ref[3,2,1,1]=(-0.424515284177)/(o+1.)+(0.394142291336)+(0.438014662973)*0.5**o
        ref[3,2,2,0]=(0.326610298255)/(o+1.)+(0.98989944094)+(-0.709499532674)*0.5**o
        ref[3,2,2,1]=(-0.846657945161)/(o+1.)+(0.0841772101477)+(-0.657647707844)*0.5**o
        ref[3,3,0,0]=(0.549866100727)/(o+1.)+(-0.133734825728)+(0.21346320453)*0.5**o
        ref[3,3,0,1]=(0.0224330511636)/(o+1.)+(0.326044184638)+(0.249957470685)*0.5**o
        ref[3,3,1,0]=(0.119453667263)/(o+1.)+(-0.0144803823844)+(0.166507029533)*0.5**o
        ref[3,3,1,1]=(0.57743453064)/(o+1.)+(0.0392054026384)+(0.358248720077)*0.5**o
        ref[3,3,2,0]=(0.63832012742)/(o+1.)+(-0.866342571675)+(-0.896899724447)*0.5**o
        ref[3,3,2,1]=(0.496219405916)/(o+1.)+(0.271132181694)+(0.425194254583)*0.5**o
        ref[3,4,0,0]=(-0.283510438468)/(o+1.)+(-0.0526860327828)+(0.793902581592)*0.5**o
        ref[3,4,0,1]=(-0.259885631206)/(o+1.)+(-0.669909592284)+(0.372028980539)*0.5**o
        ref[3,4,1,0]=(0.895070489116)/(o+1.)+(-0.78522377607)+(-0.80489673786)*0.5**o
        ref[3,4,1,1]=(0.399816667015)/(o+1.)+(-0.0252589669489)+(0.272535126101)*0.5**o
        ref[3,4,2,0]=(0.906020184876)/(o+1.)+(0.280852618457)+(0.0341775288582)*0.5**o
        ref[3,4,2,1]=(-0.648936544722)/(o+1.)+(-0.924724185844)+(-0.745016464485)*0.5**o
      else:
        arg[0,0,0,0]=(-0.891975745754)*x[0]**o+(0.678331209591)*x[0]+(-0.450022305538)*x[1]**o+(0.521272586024)*x[1]+(-0.28868708769)*x[2]**o+(-0.631617544339)*x[2]
        arg[0,0,0,1]=(0.428698930349)*x[0]**o+(-0.856251438077)*x[0]+(0.915766367729)*x[1]**o+(-0.35019463199)*x[1]+(-0.0029340113102)*x[2]**o+(-0.394458505835)*x[2]
        arg[0,0,1,0]=(0.974069118653)*x[0]**o+(-0.674028781962)*x[0]+(0.489718068973)*x[1]**o+(-0.471323892144)*x[1]+(0.11469271678)*x[2]**o+(-0.264584383354)*x[2]
        arg[0,0,1,1]=(0.813412823502)*x[0]**o+(0.739276532652)*x[0]+(-0.863216628203)*x[1]**o+(0.486539683681)*x[1]+(-0.771448803712)*x[2]**o+(-0.621526889056)*x[2]
        arg[0,0,2,0]=(-0.874502916672)*x[0]**o+(0.0708047915752)*x[0]+(0.917576817333)*x[1]**o+(-0.257626031306)*x[1]+(-0.161822453369)*x[2]**o+(0.488866912786)*x[2]
        arg[0,0,2,1]=(-0.258524075807)*x[0]**o+(0.206589504672)*x[0]+(0.232971263871)*x[1]**o+(-0.917991566808)*x[1]+(0.725943193794)*x[2]**o+(-0.170341108992)*x[2]
        arg[0,1,0,0]=(0.489505903018)*x[0]**o+(0.386376068728)*x[0]+(0.510797276159)*x[1]**o+(0.45356046467)*x[1]+(-0.664963144251)*x[2]**o+(0.128579947316)*x[2]
        arg[0,1,0,1]=(0.441567332096)*x[0]**o+(0.658686083904)*x[0]+(0.46601056348)*x[1]**o+(-0.130759928108)*x[1]+(0.324576589)*x[2]**o+(0.844700852997)*x[2]
        arg[0,1,1,0]=(0.388576628492)*x[0]**o+(0.41731399682)*x[0]+(-0.37037879729)*x[1]**o+(-0.00306376860454)*x[1]+(-0.464420131682)*x[2]**o+(0.570470055678)*x[2]
        arg[0,1,1,1]=(-0.0537780037583)*x[0]**o+(0.641854502644)*x[0]+(-0.475355608842)*x[1]**o+(-0.155501886306)*x[1]+(0.159833591997)*x[2]**o+(0.889951997355)*x[2]
        arg[0,1,2,0]=(0.58262140787)*x[0]**o+(0.487380368384)*x[0]+(-0.783228845976)*x[1]**o+(-0.683221249176)*x[1]+(0.98628901689)*x[2]**o+(0.591884616904)*x[2]
        arg[0,1,2,1]=(0.438472233911)*x[0]**o+(0.989088416794)*x[0]+(0.221688966923)*x[1]**o+(-0.119266918098)*x[1]+(-0.887421255415)*x[2]**o+(0.728082459156)*x[2]
        arg[0,2,0,0]=(-0.334110476968)*x[0]**o+(-0.877094545216)*x[0]+(-0.293135209996)*x[1]**o+(0.578110480224)*x[1]+(-0.898686340104)*x[2]**o+(0.528072783107)*x[2]
        arg[0,2,0,1]=(0.33041258354)*x[0]**o+(-0.70349630629)*x[0]+(0.349551161792)*x[1]**o+(-0.296687014146)*x[1]+(-0.272377440466)*x[2]**o+(-0.42221458274)*x[2]
        arg[0,2,1,0]=(-0.80021736912)*x[0]**o+(0.271727608951)*x[0]+(0.689351606243)*x[1]**o+(-0.95063104158)*x[1]+(0.447293221661)*x[2]**o+(0.962435109303)*x[2]
        arg[0,2,1,1]=(-0.773331228218)*x[0]**o+(0.569225340051)*x[0]+(-0.0330042729183)*x[1]**o+(-0.732867386826)*x[1]+(-0.399631128084)*x[2]**o+(0.235907094874)*x[2]
        arg[0,2,2,0]=(-0.943072868965)*x[0]**o+(-0.959364749946)*x[0]+(0.922934300445)*x[1]**o+(0.27257203243)*x[1]+(0.450453864748)*x[2]**o+(0.521284310775)*x[2]
        arg[0,2,2,1]=(-0.821201079286)*x[0]**o+(-0.801011107627)*x[0]+(0.554575277011)*x[1]**o+(-0.411361733513)*x[1]+(-0.858195570474)*x[2]**o+(0.420660103079)*x[2]
        arg[0,3,0,0]=(-0.0564164524098)*x[0]**o+(-0.865516143298)*x[0]+(-0.85754022189)*x[1]**o+(-0.0159800361584)*x[1]+(-0.891158469429)*x[2]**o+(-0.809909518115)*x[2]
        arg[0,3,0,1]=(-0.553858251068)*x[0]**o+(0.827558838148)*x[0]+(0.690081127613)*x[1]**o+(0.3475090336)*x[1]+(-0.190426828891)*x[2]**o+(-0.99589316535)*x[2]
        arg[0,3,1,0]=(-0.992317453163)*x[0]**o+(-0.992812531491)*x[0]+(-0.859683310869)*x[1]**o+(0.441779587739)*x[1]+(0.582457020078)*x[2]**o+(-0.959111461424)*x[2]
        arg[0,3,1,1]=(-0.713753922213)*x[0]**o+(0.504885148776)*x[0]+(-0.076988545634)*x[1]**o+(0.530828889101)*x[1]+(-0.335616242708)*x[2]**o+(-0.0421293420747)*x[2]
        arg[0,3,2,0]=(0.536024118289)*x[0]**o+(-0.527038052847)*x[0]+(-0.558392588656)*x[1]**o+(0.928618057856)*x[1]+(0.427507510033)*x[2]**o+(0.223570914141)*x[2]
        arg[0,3,2,1]=(-0.197968756117)*x[0]**o+(-0.58157606173)*x[0]+(-0.878630012825)*x[1]**o+(-0.380635552726)*x[1]+(-0.477570149125)*x[2]**o+(-0.642904015434)*x[2]
        arg[0,4,0,0]=(0.688763655127)*x[0]**o+(0.286011523933)*x[0]+(0.0933014250929)*x[1]**o+(0.355550563233)*x[1]+(0.121445291305)*x[2]**o+(-0.814538082974)*x[2]
        arg[0,4,0,1]=(-0.402717742839)*x[0]**o+(0.403946649835)*x[0]+(0.286978800579)*x[1]**o+(-0.363358056216)*x[1]+(-0.185233594102)*x[2]**o+(-0.902660250807)*x[2]
        arg[0,4,1,0]=(0.39352838659)*x[0]**o+(-0.430755011337)*x[0]+(-0.912518631162)*x[1]**o+(-0.634118795867)*x[1]+(-0.802445843416)*x[2]**o+(0.643395158048)*x[2]
        arg[0,4,1,1]=(-0.375507250175)*x[0]**o+(-0.609576942939)*x[0]+(0.35060243693)*x[1]**o+(0.384630130603)*x[1]+(0.165576820334)*x[2]**o+(-0.319012357341)*x[2]
        arg[0,4,2,0]=(0.0603590263818)*x[0]**o+(-0.435954775883)*x[0]+(-0.895329092913)*x[1]**o+(0.364809539103)*x[1]+(0.912864788565)*x[2]**o+(0.582044419963)*x[2]
        arg[0,4,2,1]=(-0.111089454685)*x[0]**o+(-0.247527778755)*x[0]+(0.897263901657)*x[1]**o+(0.168097655548)*x[1]+(-0.532273075067)*x[2]**o+(-0.158529268933)*x[2]
        arg[1,0,0,0]=(-0.940865155622)*x[0]**o+(0.639652431549)*x[0]+(0.204433172026)*x[1]**o+(-0.333435100662)*x[1]+(0.922286591304)*x[2]**o+(0.245091036189)*x[2]
        arg[1,0,0,1]=(-0.843207880309)*x[0]**o+(0.496567621809)*x[0]+(-0.747925907677)*x[1]**o+(-0.666195656311)*x[1]+(-0.836848401917)*x[2]**o+(-0.77301298659)*x[2]
        arg[1,0,1,0]=(-0.484987528212)*x[0]**o+(-0.208798022576)*x[0]+(-0.516295195217)*x[1]**o+(0.273733676249)*x[1]+(0.642865886081)*x[2]**o+(0.94556566928)*x[2]
        arg[1,0,1,1]=(0.285548625366)*x[0]**o+(-0.316870037241)*x[0]+(0.166055429845)*x[1]**o+(-0.907240066063)*x[1]+(-0.973602611383)*x[2]**o+(-0.232655535454)*x[2]
        arg[1,0,2,0]=(0.96797757731)*x[0]**o+(-0.0920831342923)*x[0]+(0.0367163548581)*x[1]**o+(0.372503137376)*x[1]+(0.114902128151)*x[2]**o+(0.147390259838)*x[2]
        arg[1,0,2,1]=(-0.486190283922)*x[0]**o+(-0.713417720041)*x[0]+(0.0212905465522)*x[1]**o+(-0.358636168795)*x[1]+(0.719370965272)*x[2]**o+(-0.506502687359)*x[2]
        arg[1,1,0,0]=(-0.216544472879)*x[0]**o+(-0.690540499698)*x[0]+(0.190155427044)*x[1]**o+(0.832206912182)*x[1]+(-0.108713936252)*x[2]**o+(-0.221373087337)*x[2]
        arg[1,1,0,1]=(0.982934604971)*x[0]**o+(0.604369826835)*x[0]+(-0.34365978148)*x[1]**o+(-0.167486779981)*x[1]+(-0.753609370582)*x[2]**o+(-0.220160888977)*x[2]
        arg[1,1,1,0]=(0.516177509446)*x[0]**o+(-0.474445723273)*x[0]+(0.813647087566)*x[1]**o+(-0.281201093802)*x[1]+(0.941465944108)*x[2]**o+(0.352728360059)*x[2]
        arg[1,1,1,1]=(-0.914732890633)*x[0]**o+(0.214962090901)*x[0]+(-0.710485058274)*x[1]**o+(-0.070756754056)*x[1]+(-0.449571590211)*x[2]**o+(0.353655711451)*x[2]
        arg[1,1,2,0]=(-0.96655555619)*x[0]**o+(-0.47460534732)*x[0]+(0.0610695053327)*x[1]**o+(0.815390116709)*x[1]+(-0.844916163552)*x[2]**o+(-0.433015304307)*x[2]
        arg[1,1,2,1]=(-0.611879453709)*x[0]**o+(0.849092848509)*x[0]+(0.0181322361566)*x[1]**o+(0.264173017827)*x[1]+(0.0452154806937)*x[2]**o+(0.0395300648465)*x[2]
        arg[1,2,0,0]=(0.558820946357)*x[0]**o+(-0.471519705598)*x[0]+(-0.200404864145)*x[1]**o+(0.452044160691)*x[1]+(-0.507559485963)*x[2]**o+(-0.993437841726)*x[2]
        arg[1,2,0,1]=(-0.944738664192)*x[0]**o+(-0.497536772117)*x[0]+(0.111399525675)*x[1]**o+(-0.93450294271)*x[1]+(-0.463702615737)*x[2]**o+(-0.330112672884)*x[2]
        arg[1,2,1,0]=(0.238225961058)*x[0]**o+(-0.471520173419)*x[0]+(0.483767084892)*x[1]**o+(-0.0359407284537)*x[1]+(-0.767379409753)*x[2]**o+(-0.283596371915)*x[2]
        arg[1,2,1,1]=(0.188699181238)*x[0]**o+(0.829593121989)*x[0]+(0.603377767593)*x[1]**o+(0.163334775468)*x[1]+(-0.695141594131)*x[2]**o+(-0.335208640181)*x[2]
        arg[1,2,2,0]=(-0.0953160757881)*x[0]**o+(0.179594592176)*x[0]+(0.95983902191)*x[1]**o+(-0.689622683527)*x[1]+(0.418442085936)*x[2]**o+(-0.271399861129)*x[2]
        arg[1,2,2,1]=(0.598691886555)*x[0]**o+(0.334120340646)*x[0]+(0.580763872443)*x[1]**o+(0.769016736836)*x[1]+(-0.442121483951)*x[2]**o+(0.0685264980499)*x[2]
        arg[1,3,0,0]=(-0.482944276372)*x[0]**o+(-0.0518057205519)*x[0]+(-0.878100108766)*x[1]**o+(-0.130957239599)*x[1]+(0.71270341547)*x[2]**o+(-0.313822116756)*x[2]
        arg[1,3,0,1]=(0.565128725562)*x[0]**o+(0.706149004661)*x[0]+(0.837747652328)*x[1]**o+(-0.429417376176)*x[1]+(0.179437627997)*x[2]**o+(-0.841831183522)*x[2]
        arg[1,3,1,0]=(0.608648887224)*x[0]**o+(0.118220851451)*x[0]+(0.492890967377)*x[1]**o+(-0.162344084552)*x[1]+(0.141203210671)*x[2]**o+(0.883457779687)*x[2]
        arg[1,3,1,1]=(0.0322521291544)*x[0]**o+(0.839748977403)*x[0]+(0.75094786544)*x[1]**o+(0.954833145672)*x[1]+(0.461457599013)*x[2]**o+(-0.761973385747)*x[2]
        arg[1,3,2,0]=(-0.757022609619)*x[0]**o+(-0.250893790099)*x[0]+(-0.973631035961)*x[1]**o+(0.506574565217)*x[1]+(0.103557430896)*x[2]**o+(0.881026337134)*x[2]
        arg[1,3,2,1]=(-0.395945302352)*x[0]**o+(-0.125428842065)*x[0]+(-0.339663004568)*x[1]**o+(-0.0272044592179)*x[1]+(0.75200506467)*x[2]**o+(-0.632922582454)*x[2]
        arg[1,4,0,0]=(0.0462127928202)*x[0]**o+(-0.745613488751)*x[0]+(0.357338297422)*x[1]**o+(0.739034094804)*x[1]+(-0.451247272253)*x[2]**o+(0.418030110847)*x[2]
        arg[1,4,0,1]=(0.914131238526)*x[0]**o+(-0.0280436066624)*x[0]+(0.861048515118)*x[1]**o+(0.148206807819)*x[1]+(-0.990856990354)*x[2]**o+(0.903615279172)*x[2]
        arg[1,4,1,0]=(0.189809824038)*x[0]**o+(-0.749477832258)*x[0]+(-0.802713794942)*x[1]**o+(0.0610727255572)*x[1]+(0.292940815327)*x[2]**o+(0.213443422205)*x[2]
        arg[1,4,1,1]=(0.792313292544)*x[0]**o+(0.321344675677)*x[0]+(-0.417516596097)*x[1]**o+(-0.63797359819)*x[1]+(-0.426112319872)*x[2]**o+(0.69281310288)*x[2]
        arg[1,4,2,0]=(-0.11017764073)*x[0]**o+(0.703037562344)*x[0]+(0.739591950873)*x[1]**o+(0.445612268331)*x[1]+(-0.071251825024)*x[2]**o+(0.00577350654185)*x[2]
        arg[1,4,2,1]=(-0.793584849832)*x[0]**o+(-0.546439256231)*x[0]+(-0.947194030803)*x[1]**o+(-0.0455700937677)*x[1]+(0.535368301918)*x[2]**o+(0.814435535521)*x[2]
        arg[2,0,0,0]=(0.249661949438)*x[0]**o+(-0.581533619092)*x[0]+(-0.674373505747)*x[1]**o+(0.464175602196)*x[1]+(-0.528086662834)*x[2]**o+(-0.212416176351)*x[2]
        arg[2,0,0,1]=(-0.588282108798)*x[0]**o+(0.761293399494)*x[0]+(0.99918874903)*x[1]**o+(0.542499992928)*x[1]+(0.433925746483)*x[2]**o+(0.972864491889)*x[2]
        arg[2,0,1,0]=(0.0572335623442)*x[0]**o+(0.444906443533)*x[0]+(-0.305538151235)*x[1]**o+(0.244530068532)*x[1]+(-0.20198975858)*x[2]**o+(-0.00490928477351)*x[2]
        arg[2,0,1,1]=(0.832496085695)*x[0]**o+(-0.797246004595)*x[0]+(0.873816520001)*x[1]**o+(0.628293733929)*x[1]+(-0.110922710624)*x[2]**o+(-0.0718221914062)*x[2]
        arg[2,0,2,0]=(0.140970192196)*x[0]**o+(0.582230323368)*x[0]+(-0.757764751267)*x[1]**o+(0.206061262231)*x[1]+(0.657237474788)*x[2]**o+(0.598632779121)*x[2]
        arg[2,0,2,1]=(-0.686218790734)*x[0]**o+(-0.157012470162)*x[0]+(0.448565721848)*x[1]**o+(0.0539087954076)*x[1]+(0.619821805007)*x[2]**o+(0.289144379283)*x[2]
        arg[2,1,0,0]=(0.787423884494)*x[0]**o+(-0.985959420171)*x[0]+(0.155584360983)*x[1]**o+(0.223292293778)*x[1]+(-0.437349719833)*x[2]**o+(0.989077189275)*x[2]
        arg[2,1,0,1]=(-0.597301631508)*x[0]**o+(-0.0414813766671)*x[0]+(-0.64883048443)*x[1]**o+(-0.18731615522)*x[1]+(0.735189823623)*x[2]**o+(-0.746633725819)*x[2]
        arg[2,1,1,0]=(-0.393486075452)*x[0]**o+(-0.480110060647)*x[0]+(0.559811870081)*x[1]**o+(0.0814282056993)*x[1]+(0.244899906416)*x[2]**o+(-0.884478480652)*x[2]
        arg[2,1,1,1]=(0.547747813175)*x[0]**o+(0.317890585903)*x[0]+(-0.70925099068)*x[1]**o+(-0.543227445702)*x[1]+(0.749693240223)*x[2]**o+(0.79984501038)*x[2]
        arg[2,1,2,0]=(-0.289583636666)*x[0]**o+(0.513926622265)*x[0]+(-0.00283544918865)*x[1]**o+(0.530724065811)*x[1]+(0.900479623243)*x[2]**o+(-0.146332301083)*x[2]
        arg[2,1,2,1]=(-0.487223003118)*x[0]**o+(0.161546483594)*x[0]+(-0.685211654772)*x[1]**o+(0.4803179816)*x[1]+(0.676120396917)*x[2]**o+(-0.174953085068)*x[2]
        arg[2,2,0,0]=(0.31746495834)*x[0]**o+(-0.513225182059)*x[0]+(0.33220677322)*x[1]**o+(-0.302371334854)*x[1]+(0.0964988251907)*x[2]**o+(-0.259532745762)*x[2]
        arg[2,2,0,1]=(0.313462846078)*x[0]**o+(0.079720902627)*x[0]+(-0.487932457563)*x[1]**o+(-0.786522786773)*x[1]+(0.357868214554)*x[2]**o+(0.251856050362)*x[2]
        arg[2,2,1,0]=(-0.180472378556)*x[0]**o+(0.345084802362)*x[0]+(0.641108557953)*x[1]**o+(0.198717742483)*x[1]+(0.0374114682326)*x[2]**o+(-0.915249907773)*x[2]
        arg[2,2,1,1]=(0.303787742688)*x[0]**o+(-0.941822487745)*x[0]+(-0.262567025842)*x[1]**o+(-0.197051445307)*x[1]+(0.381561965565)*x[2]**o+(-0.198549777232)*x[2]
        arg[2,2,2,0]=(-0.850284190461)*x[0]**o+(0.985224868544)*x[0]+(0.633868153422)*x[1]**o+(-0.57559927346)*x[1]+(0.68753726951)*x[2]**o+(-0.923696997656)*x[2]
        arg[2,2,2,1]=(0.559527478503)*x[0]**o+(-0.619764636475)*x[0]+(-0.0352163693945)*x[1]**o+(-0.37351931995)*x[1]+(0.0772333960938)*x[2]**o+(-0.353057755509)*x[2]
        arg[2,3,0,0]=(-0.274260374984)*x[0]**o+(0.365180349506)*x[0]+(-0.879564869545)*x[1]**o+(-0.0908162765131)*x[1]+(0.813491263836)*x[2]**o+(-0.396739585204)*x[2]
        arg[2,3,0,1]=(-0.793218219979)*x[0]**o+(0.227474218201)*x[0]+(0.97716610661)*x[1]**o+(0.317074032722)*x[1]+(0.076776446926)*x[2]**o+(0.069448617786)*x[2]
        arg[2,3,1,0]=(0.411400569812)*x[0]**o+(0.517743454926)*x[0]+(0.646697523033)*x[1]**o+(-0.973369313024)*x[1]+(-0.599741793926)*x[2]**o+(-0.743687020283)*x[2]
        arg[2,3,1,1]=(-0.491445354697)*x[0]**o+(-0.549840110396)*x[0]+(-0.260012915614)*x[1]**o+(0.671054626537)*x[1]+(0.0394966796969)*x[2]**o+(-0.701243931411)*x[2]
        arg[2,3,2,0]=(-0.806859723517)*x[0]**o+(-0.753940066715)*x[0]+(-0.664229274865)*x[1]**o+(0.0595484609931)*x[1]+(-0.921892262667)*x[2]**o+(-0.701770672395)*x[2]
        arg[2,3,2,1]=(-0.313979805844)*x[0]**o+(-0.542583864222)*x[0]+(-0.985990886553)*x[1]**o+(0.664926443979)*x[1]+(-0.99778334615)*x[2]**o+(0.268046810979)*x[2]
        arg[2,4,0,0]=(-0.41348240167)*x[0]**o+(0.473517720534)*x[0]+(-0.290880345559)*x[1]**o+(-0.607611624107)*x[1]+(-0.25537697471)*x[2]**o+(-0.472402477362)*x[2]
        arg[2,4,0,1]=(0.0222534371058)*x[0]**o+(-0.0739354104963)*x[0]+(-0.960057765718)*x[1]**o+(-0.772161403037)*x[1]+(-0.0356127352146)*x[2]**o+(-0.623605628308)*x[2]
        arg[2,4,1,0]=(-0.750995516053)*x[0]**o+(-0.289601799215)*x[0]+(0.294370180152)*x[1]**o+(0.956012147239)*x[1]+(-0.27958662106)*x[2]**o+(-0.370609593537)*x[2]
        arg[2,4,1,1]=(0.321767371741)*x[0]**o+(0.999223434136)*x[0]+(0.665865428241)*x[1]**o+(0.427942082269)*x[1]+(-0.135020552387)*x[2]**o+(0.692025479514)*x[2]
        arg[2,4,2,0]=(-0.519508249649)*x[0]**o+(0.0779688946471)*x[0]+(0.493489242872)*x[1]**o+(-0.0961136530093)*x[1]+(-0.0750243559261)*x[2]**o+(0.860970740104)*x[2]
        arg[2,4,2,1]=(0.807909049346)*x[0]**o+(0.388180877227)*x[0]+(-0.9044578433)*x[1]**o+(0.86765389355)*x[1]+(-0.196105779343)*x[2]**o+(0.448661151455)*x[2]
        arg[3,0,0,0]=(-0.780895567041)*x[0]**o+(-0.571253266808)*x[0]+(-0.904183399932)*x[1]**o+(-0.883296586722)*x[1]+(0.855945559487)*x[2]**o+(0.618533553098)*x[2]
        arg[3,0,0,1]=(-0.831076704037)*x[0]**o+(0.898347446798)*x[0]+(-0.716442620396)*x[1]**o+(0.761028369421)*x[1]+(-0.486505604444)*x[2]**o+(0.25474220145)*x[2]
        arg[3,0,1,0]=(-0.0127541044953)*x[0]**o+(0.759482315308)*x[0]+(-0.725655006866)*x[1]**o+(0.667310781285)*x[1]+(0.266075927439)*x[2]**o+(0.423300182252)*x[2]
        arg[3,0,1,1]=(-0.594972266882)*x[0]**o+(-0.842501149639)*x[0]+(-0.347686790781)*x[1]**o+(0.686403640474)*x[1]+(0.3452971966)*x[2]**o+(0.722258585706)*x[2]
        arg[3,0,2,0]=(-0.715711146058)*x[0]**o+(-0.835739963371)*x[0]+(0.966177361301)*x[1]**o+(-0.909111754995)*x[1]+(-0.175184576557)*x[2]**o+(0.695058239921)*x[2]
        arg[3,0,2,1]=(0.306251544915)*x[0]**o+(0.944775715803)*x[0]+(-0.740601754617)*x[1]**o+(0.434394890125)*x[1]+(-0.876527474971)*x[2]**o+(0.209321129689)*x[2]
        arg[3,1,0,0]=(-0.86788221871)*x[0]**o+(-0.833030657309)*x[0]+(0.84576621947)*x[1]**o+(-0.469186887277)*x[1]+(-0.527611740694)*x[2]**o+(-0.312054431001)*x[2]
        arg[3,1,0,1]=(-0.353678394652)*x[0]**o+(0.591287444611)*x[0]+(-0.213012375931)*x[1]**o+(-0.917259099251)*x[1]+(0.739016360622)*x[2]**o+(0.718754459378)*x[2]
        arg[3,1,1,0]=(-0.92646825221)*x[0]**o+(-0.870564973175)*x[0]+(-0.781805219404)*x[1]**o+(0.078530434642)*x[1]+(0.386018501087)*x[2]**o+(-0.120490492288)*x[2]
        arg[3,1,1,1]=(-0.824273737615)*x[0]**o+(0.401211543421)*x[0]+(0.360077381624)*x[1]**o+(-0.0599234883152)*x[1]+(0.246820480884)*x[2]**o+(-0.175313542134)*x[2]
        arg[3,1,2,0]=(0.807756928769)*x[0]**o+(0.889534807508)*x[0]+(0.633296189954)*x[1]**o+(-0.471821951252)*x[1]+(0.396011249874)*x[2]**o+(0.386800809575)*x[2]
        arg[3,1,2,1]=(-0.494145142543)*x[0]**o+(-0.514793985721)*x[0]+(0.825379848403)*x[1]**o+(-0.457860259786)*x[1]+(-0.599557868275)*x[2]**o+(-0.952704784672)*x[2]
        arg[3,2,0,0]=(0.970345295013)*x[0]**o+(-0.838468036663)*x[0]+(0.265634348738)*x[1]**o+(-0.873847124748)*x[1]+(0.416242648913)*x[2]**o+(0.59908438222)*x[2]
        arg[3,2,0,1]=(0.692377432429)*x[0]**o+(-0.338749041227)*x[0]+(-0.449383750398)*x[1]**o+(0.841669527559)*x[1]+(0.902482059741)*x[2]**o+(-0.655719647704)*x[2]
        arg[3,2,1,0]=(0.152417917522)*x[0]**o+(-0.78860144)*x[0]+(-0.881276170743)*x[1]**o+(0.692901550061)*x[1]+(-0.985377819322)*x[2]**o+(0.362747973601)*x[2]
        arg[3,2,1,1]=(0.238845792119)*x[0]**o+(-0.65838373309)*x[0]+(-0.659676514296)*x[1]**o+(0.870742876778)*x[1]+(-0.367345415947)*x[2]**o+(0.322353666541)*x[2]
        arg[3,2,2,0]=(0.456386394533)*x[0]**o+(-0.135575574979)*x[0]+(-0.731714323922)*x[1]**o+(0.936807859479)*x[1]+(0.511595594967)*x[2]**o+(0.570282464669)*x[2]
        arg[3,2,2,1]=(0.53299212624)*x[0]**o+(0.800998166434)*x[0]+(-0.431740087552)*x[1]**o+(-0.880092532661)*x[1]+(-0.418657988674)*x[2]**o+(-0.181008149338)*x[2]
        arg[3,3,0,0]=(0.60015312936)*x[0]**o+(-0.163125583081)*x[0]+(0.299668981953)*x[1]**o+(-0.761494422747)*x[1]+(0.404839953231)*x[2]**o+(-0.114039836538)*x[2]
        arg[3,3,0,1]=(-0.597481072133)*x[0]**o+(-0.506468582896)*x[0]+(0.406822348574)*x[1]**o+(0.247431348859)*x[1]+(-0.137941563908)*x[2]**o+(0.0503979259383)*x[2]
        arg[3,3,1,0]=(0.0921069092744)*x[0]**o+(-0.113960490794)*x[0]+(0.750059773909)*x[1]**o+(0.894945048248)*x[1]+(-0.154873449786)*x[2]**o+(-0.520229467775)*x[2]
        arg[3,3,1,1]=(0.349078696154)*x[0]**o+(0.141298430827)*x[0]+(0.249161007832)*x[1]**o+(0.0240310278591)*x[1]+(-0.18294527838)*x[2]**o+(-0.666698832632)*x[2]
        arg[3,3,2,0]=(-0.476684936147)*x[0]**o+(0.334093402615)*x[0]+(-0.326434684009)*x[1]**o+(0.91336431975)*x[1]+(0.483080537399)*x[2]**o+(0.439096732335)*x[2]
        arg[3,3,2,1]=(0.214713426286)*x[0]**o+(-0.973239849984)*x[0]+(0.984290637567)*x[1]**o+(-0.644668525299)*x[1]+(0.00985524319943)*x[2]**o+(0.68777403533)*x[2]
        arg[3,4,0,0]=(-0.88715423933)*x[0]**o+(0.642412222188)*x[0]+(-0.119667464132)*x[1]**o+(-0.201583116481)*x[1]+(-0.870921042381)*x[2]**o+(-0.00379039429884)*x[2]
        arg[3,4,0,1]=(0.511114986801)*x[0]**o+(-0.497617928997)*x[0]+(-0.189435233543)*x[1]**o+(-0.29288141421)*x[1]+(-0.476297664539)*x[2]**o+(-0.202915859137)*x[2]
        arg[3,4,1,0]=(0.0739878219028)*x[0]**o+(0.675050675951)*x[0]+(-0.45950101521)*x[1]**o+(0.306538598103)*x[1]+(-0.726069135031)*x[2]**o+(0.193290422951)*x[2]
        arg[3,4,1,1]=(-0.141324267565)*x[0]**o+(0.377343882109)*x[0]+(-0.16075604872)*x[1]**o+(0.263332274591)*x[1]+(0.886416182992)*x[2]**o+(0.931905996631)*x[2]
        arg[3,4,2,0]=(-0.76912732001)*x[0]**o+(-0.255011103121)*x[0]+(-0.372810348364)*x[1]**o+(0.108137766687)*x[1]+(-0.0761382180171)*x[2]**o+(0.46825331282)*x[2]
        arg[3,4,2,1]=(-0.515945940217)*x[0]**o+(-0.630017609554)*x[0]+(-0.815236764508)*x[1]**o+(0.636572471005)*x[1]+(0.244112465789)*x[2]**o+(-0.471661017589)*x[2]
        ref[0,0,0,0]=(-0.738709393227)/(o+1.)+(0.283993125638)+(-0.891975745754)*0.5**o
        ref[0,0,0,1]=(0.912832356419)/(o+1.)+(-0.800452287951)+(0.428698930349)*0.5**o
        ref[0,0,1,0]=(0.604410785754)/(o+1.)+(-0.70496852873)+(0.974069118653)*0.5**o
        ref[0,0,1,1]=(-1.63466543191)/(o+1.)+(0.302144663639)+(0.813412823502)*0.5**o
        ref[0,0,2,0]=(0.755754363963)/(o+1.)+(0.151022836528)+(-0.874502916672)*0.5**o
        ref[0,0,2,1]=(0.958914457666)/(o+1.)+(-0.440871585564)+(-0.258524075807)*0.5**o
        ref[0,1,0,0]=(-0.154165868092)/(o+1.)+(0.484258240357)+(0.489505903018)*0.5**o
        ref[0,1,0,1]=(0.79058715248)/(o+1.)+(0.686313504396)+(0.441567332096)*0.5**o
        ref[0,1,1,0]=(-0.834798928973)/(o+1.)+(0.492360141946)+(0.388576628492)*0.5**o
        ref[0,1,1,1]=(-0.315522016845)/(o+1.)+(0.688152306846)+(-0.0537780037583)*0.5**o
        ref[0,1,2,0]=(0.203060170914)/(o+1.)+(0.198021868056)+(0.58262140787)*0.5**o
        ref[0,1,2,1]=(-0.665732288492)/(o+1.)+(0.798951978927)+(0.438472233911)*0.5**o
        ref[0,2,0,0]=(-1.1918215501)/(o+1.)+(0.114544359058)+(-0.334110476968)*0.5**o
        ref[0,2,0,1]=(0.0771737213265)/(o+1.)+(-0.711198951588)+(0.33041258354)*0.5**o
        ref[0,2,1,0]=(1.1366448279)/(o+1.)+(0.141765838337)+(-0.80021736912)*0.5**o
        ref[0,2,1,1]=(-0.432635401002)/(o+1.)+(0.0361325240497)+(-0.773331228218)*0.5**o
        ref[0,2,2,0]=(1.37338816519)/(o+1.)+(-0.0827542033711)+(-0.943072868965)*0.5**o
        ref[0,2,2,1]=(-0.303620293463)/(o+1.)+(-0.39585636903)+(-0.821201079286)*0.5**o
        ref[0,3,0,0]=(-1.74869869132)/(o+1.)+(-0.845702848785)+(-0.0564164524098)*0.5**o
        ref[0,3,0,1]=(0.499654298722)/(o+1.)+(0.0895873531986)+(-0.553858251068)*0.5**o
        ref[0,3,1,0]=(-0.277226290791)/(o+1.)+(-0.755072202588)+(-0.992317453163)*0.5**o
        ref[0,3,1,1]=(-0.412604788342)/(o+1.)+(0.496792347901)+(-0.713753922213)*0.5**o
        ref[0,3,2,0]=(-0.130885078624)/(o+1.)+(0.312575459575)+(0.536024118289)*0.5**o
        ref[0,3,2,1]=(-1.35620016195)/(o+1.)+(-0.802557814945)+(-0.197968756117)*0.5**o
        ref[0,4,0,0]=(0.214746716397)/(o+1.)+(-0.0864879979042)+(0.688763655127)*0.5**o
        ref[0,4,0,1]=(0.101745206477)/(o+1.)+(-0.431035828594)+(-0.402717742839)*0.5**o
        ref[0,4,1,0]=(-1.71496447458)/(o+1.)+(-0.210739324578)+(0.39352838659)*0.5**o
        ref[0,4,1,1]=(0.516179257264)/(o+1.)+(-0.271979584839)+(-0.375507250175)*0.5**o
        ref[0,4,2,0]=(0.0175356956523)/(o+1.)+(0.255449591592)+(0.0603590263818)*0.5**o
        ref[0,4,2,1]=(0.36499082659)/(o+1.)+(-0.11897969607)+(-0.111089454685)*0.5**o
        ref[1,0,0,0]=(1.12671976333)/(o+1.)+(0.275654183538)+(-0.940865155622)*0.5**o
        ref[1,0,0,1]=(-1.58477430959)/(o+1.)+(-0.471320510546)+(-0.843207880309)*0.5**o
        ref[1,0,1,0]=(0.126570690864)/(o+1.)+(0.505250661477)+(-0.484987528212)*0.5**o
        ref[1,0,1,1]=(-0.807547181539)/(o+1.)+(-0.728382819379)+(0.285548625366)*0.5**o
        ref[1,0,2,0]=(0.151618483009)/(o+1.)+(0.213905131461)+(0.96797757731)*0.5**o
        ref[1,0,2,1]=(0.740661511824)/(o+1.)+(-0.789278288097)+(-0.486190283922)*0.5**o
        ref[1,1,0,0]=(0.0814414907926)/(o+1.)+(-0.0398533374263)+(-0.216544472879)*0.5**o
        ref[1,1,0,1]=(-1.09726915206)/(o+1.)+(0.108361078938)+(0.982934604971)*0.5**o
        ref[1,1,1,0]=(1.75511303167)/(o+1.)+(-0.201459228508)+(0.516177509446)*0.5**o
        ref[1,1,1,1]=(-1.16005664849)/(o+1.)+(0.248930524148)+(-0.914732890633)*0.5**o
        ref[1,1,2,0]=(-0.78384665822)/(o+1.)+(-0.0461152674588)+(-0.96655555619)*0.5**o
        ref[1,1,2,1]=(0.0633477168503)/(o+1.)+(0.576397965591)+(-0.611879453709)*0.5**o
        ref[1,2,0,0]=(-0.707964350108)/(o+1.)+(-0.506456693317)+(0.558820946357)*0.5**o
        ref[1,2,0,1]=(-0.352303090063)/(o+1.)+(-0.881076193856)+(-0.944738664192)*0.5**o
        ref[1,2,1,0]=(-0.283612324861)/(o+1.)+(-0.395528636894)+(0.238225961058)*0.5**o
        ref[1,2,1,1]=(-0.0917638265383)/(o+1.)+(0.328859628638)+(0.188699181238)*0.5**o
        ref[1,2,2,0]=(1.37828110785)/(o+1.)+(-0.39071397624)+(-0.0953160757881)*0.5**o
        ref[1,2,2,1]=(0.138642388492)/(o+1.)+(0.585831787766)+(0.598691886555)*0.5**o
        ref[1,3,0,0]=(-0.165396693296)/(o+1.)+(-0.248292538454)+(-0.482944276372)*0.5**o
        ref[1,3,0,1]=(1.01718528032)/(o+1.)+(-0.282549777519)+(0.565128725562)*0.5**o
        ref[1,3,1,0]=(0.634094178049)/(o+1.)+(0.419667273293)+(0.608648887224)*0.5**o
        ref[1,3,1,1]=(1.21240546445)/(o+1.)+(0.516304368664)+(0.0322521291544)*0.5**o
        ref[1,3,2,0]=(-0.870073605065)/(o+1.)+(0.568353556126)+(-0.757022609619)*0.5**o
        ref[1,3,2,1]=(0.412342060102)/(o+1.)+(-0.392777941868)+(-0.395945302352)*0.5**o
        ref[1,4,0,0]=(-0.0939089748304)/(o+1.)+(0.20572535845)+(0.0462127928202)*0.5**o
        ref[1,4,0,1]=(-0.129808475236)/(o+1.)+(0.511889240164)+(0.914131238526)*0.5**o
        ref[1,4,1,0]=(-0.509772979615)/(o+1.)+(-0.237480842248)+(0.189809824038)*0.5**o
        ref[1,4,1,1]=(-0.843628915969)/(o+1.)+(0.188092090183)+(0.792313292544)*0.5**o
        ref[1,4,2,0]=(0.668340125849)/(o+1.)+(0.577211668608)+(-0.11017764073)*0.5**o
        ref[1,4,2,1]=(-0.411825728886)/(o+1.)+(0.111213092761)+(-0.793584849832)*0.5**o
        ref[2,0,0,0]=(-1.20246016858)/(o+1.)+(-0.164887096623)+(0.249661949438)*0.5**o
        ref[2,0,0,1]=(1.43311449551)/(o+1.)+(1.13832894216)+(-0.588282108798)*0.5**o
        ref[2,0,1,0]=(-0.507527909815)/(o+1.)+(0.342263613646)+(0.0572335623442)*0.5**o
        ref[2,0,1,1]=(0.762893809377)/(o+1.)+(-0.120387231036)+(0.832496085695)*0.5**o
        ref[2,0,2,0]=(-0.100527276479)/(o+1.)+(0.69346218236)+(0.140970192196)*0.5**o
        ref[2,0,2,1]=(1.06838752686)/(o+1.)+(0.0930203522642)+(-0.686218790734)*0.5**o
        ref[2,1,0,0]=(-0.28176535885)/(o+1.)+(0.113205031441)+(0.787423884494)*0.5**o
        ref[2,1,0,1]=(0.0863593391922)/(o+1.)+(-0.487715628853)+(-0.597301631508)*0.5**o
        ref[2,1,1,0]=(0.804711776497)/(o+1.)+(-0.6415801678)+(-0.393486075452)*0.5**o
        ref[2,1,1,1]=(0.0404422495423)/(o+1.)+(0.28725407529)+(0.547747813175)*0.5**o
        ref[2,1,2,0]=(0.897644174054)/(o+1.)+(0.449159193497)+(-0.289583636666)*0.5**o
        ref[2,1,2,1]=(-0.00909125785473)/(o+1.)+(0.233455690063)+(-0.487223003118)*0.5**o
        ref[2,2,0,0]=(0.428705598411)/(o+1.)+(-0.537564631337)+(0.31746495834)*0.5**o
        ref[2,2,0,1]=(-0.13006424301)/(o+1.)+(-0.227472916892)+(0.313462846078)*0.5**o
        ref[2,2,1,0]=(0.678520026185)/(o+1.)+(-0.185723681464)+(-0.180472378556)*0.5**o
        ref[2,2,1,1]=(0.118994939723)/(o+1.)+(-0.668711855142)+(0.303787742688)*0.5**o
        ref[2,2,2,0]=(1.32140542293)/(o+1.)+(-0.257035701286)+(-0.850284190461)*0.5**o
        ref[2,2,2,1]=(0.0420170266993)/(o+1.)+(-0.673170855967)+(0.559527478503)*0.5**o
        ref[2,3,0,0]=(-0.0660736057092)/(o+1.)+(-0.0611877561056)+(-0.274260374984)*0.5**o
        ref[2,3,0,1]=(1.05394255354)/(o+1.)+(0.306998434354)+(-0.793218219979)*0.5**o
        ref[2,3,1,0]=(0.046955729107)/(o+1.)+(-0.599656439191)+(0.411400569812)*0.5**o
        ref[2,3,1,1]=(-0.220516235917)/(o+1.)+(-0.290014707635)+(-0.491445354697)*0.5**o
        ref[2,3,2,0]=(-1.58612153753)/(o+1.)+(-0.698081139059)+(-0.806859723517)*0.5**o
        ref[2,3,2,1]=(-1.9837742327)/(o+1.)+(0.195194695368)+(-0.313979805844)*0.5**o
        ref[2,4,0,0]=(-0.546257320269)/(o+1.)+(-0.303248190467)+(-0.41348240167)*0.5**o
        ref[2,4,0,1]=(-0.995670500932)/(o+1.)+(-0.734851220921)+(0.0222534371058)*0.5**o
        ref[2,4,1,0]=(0.0147835590923)/(o+1.)+(0.147900377243)+(-0.750995516053)*0.5**o
        ref[2,4,1,1]=(0.530844875854)/(o+1.)+(1.05959549796)+(0.321767371741)*0.5**o
        ref[2,4,2,0]=(0.418464886946)/(o+1.)+(0.421412990871)+(-0.519508249649)*0.5**o
        ref[2,4,2,1]=(-1.10056362264)/(o+1.)+(0.852247961116)+(0.807909049346)*0.5**o
        ref[3,0,0,0]=(-0.0482378404454)/(o+1.)+(-0.418008150216)+(-0.780895567041)*0.5**o
        ref[3,0,0,1]=(-1.20294822484)/(o+1.)+(0.957059008835)+(-0.831076704037)*0.5**o
        ref[3,0,1,0]=(-0.459579079427)/(o+1.)+(0.925046639422)+(-0.0127541044953)*0.5**o
        ref[3,0,1,1]=(-0.00238959418047)/(o+1.)+(0.28308053827)+(-0.594972266882)*0.5**o
        ref[3,0,2,0]=(0.790992784744)/(o+1.)+(-0.524896739222)+(-0.715711146058)*0.5**o
        ref[3,0,2,1]=(-1.61712922959)/(o+1.)+(0.794245867809)+(0.306251544915)*0.5**o
        ref[3,1,0,0]=(0.318154478777)/(o+1.)+(-0.807135987793)+(-0.86788221871)*0.5**o
        ref[3,1,0,1]=(0.526003984691)/(o+1.)+(0.196391402369)+(-0.353678394652)*0.5**o
        ref[3,1,1,0]=(-0.395786718317)/(o+1.)+(-0.456262515411)+(-0.92646825221)*0.5**o
        ref[3,1,1,1]=(0.606897862508)/(o+1.)+(0.082987256486)+(-0.824273737615)*0.5**o
        ref[3,1,2,0]=(1.02930743983)/(o+1.)+(0.402256832916)+(0.807756928769)*0.5**o
        ref[3,1,2,1]=(0.225821980128)/(o+1.)+(-0.96267951509)+(-0.494145142543)*0.5**o
        ref[3,2,0,0]=(0.681876997651)/(o+1.)+(-0.556615389596)+(0.970345295013)*0.5**o
        ref[3,2,0,1]=(0.453098309344)/(o+1.)+(-0.0763995806862)+(0.692377432429)*0.5**o
        ref[3,2,1,0]=(-1.86665399006)/(o+1.)+(0.133524041831)+(0.152417917522)*0.5**o
        ref[3,2,1,1]=(-1.02702193024)/(o+1.)+(0.267356405114)+(0.238845792119)*0.5**o
        ref[3,2,2,0]=(-0.220118728954)/(o+1.)+(0.685757374584)+(0.456386394533)*0.5**o
        ref[3,2,2,1]=(-0.850398076225)/(o+1.)+(-0.130051257782)+(0.53299212624)*0.5**o
        ref[3,3,0,0]=(0.704508935183)/(o+1.)+(-0.519329921183)+(0.60015312936)*0.5**o
        ref[3,3,0,1]=(0.268880784666)/(o+1.)+(-0.10431965405)+(-0.597481072133)*0.5**o
        ref[3,3,1,0]=(0.595186324123)/(o+1.)+(0.13037754484)+(0.0921069092744)*0.5**o
        ref[3,3,1,1]=(0.0662157294518)/(o+1.)+(-0.250684686973)+(0.349078696154)*0.5**o
        ref[3,3,2,0]=(0.156645853389)/(o+1.)+(0.84327722735)+(-0.476684936147)*0.5**o
        ref[3,3,2,1]=(0.994145880767)/(o+1.)+(-0.465067169976)+(0.214713426286)*0.5**o
        ref[3,4,0,0]=(-0.990588506513)/(o+1.)+(0.218519355704)+(-0.88715423933)*0.5**o
        ref[3,4,0,1]=(-0.665732898082)/(o+1.)+(-0.496707601172)+(0.511114986801)*0.5**o
        ref[3,4,1,0]=(-1.18557015024)/(o+1.)+(0.587439848502)+(0.0739878219028)*0.5**o
        ref[3,4,1,1]=(0.725660134271)/(o+1.)+(0.786291076666)+(-0.141324267565)*0.5**o
        ref[3,4,2,0]=(-0.448948566381)/(o+1.)+(0.160689988193)+(-0.76912732001)*0.5**o
        ref[3,4,2,1]=(-0.571124298719)/(o+1.)+(-0.232553078069)+(-0.515945940217)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactOne

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.0333010052777)*x[0]**o+(-0.242011298555)*x[0]+(0.686457578633)*x[1]**o+(0.438506741453)*x[1]
        ref=(0.686457578633)/(o+1.)+(0.0982477214491)+(-0.0333010052777)*0.5**o
      else:
        arg=(-0.365813216932)*x[0]**o+(0.969816151161)*x[0]+(0.349995764077)*x[1]**o+(0.867492181692)*x[1]+(-0.866177355796)*x[2]**o+(-0.203219031102)*x[2]
        ref=(-0.516181591719)/(o+1.)+(0.817044650875)+(-0.365813216932)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactOne

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.038446371199)*x[0]**o+(0.323075821067)*x[0]+(0.770528210463)*x[1]**o+(0.0434528043023)*x[1]
        arg[1]=(-0.461876402566)*x[0]**o+(0.746965376214)*x[0]+(0.861234840159)*x[1]**o+(-0.568165009958)*x[1]
        ref[0]=(0.770528210463)/(o+1.)+(0.183264312685)+(0.038446371199)*0.5**o
        ref[1]=(0.861234840159)/(o+1.)+(0.0894001831281)+(-0.461876402566)*0.5**o
      else:
        arg[0]=(-0.472858663642)*x[0]**o+(-0.680210745369)*x[0]+(0.175425838082)*x[1]**o+(0.0189427039667)*x[1]+(0.0365198456146)*x[2]**o+(0.7840527457)*x[2]
        arg[1]=(-0.0191775972693)*x[0]**o+(0.19303923194)*x[0]+(0.775528295333)*x[1]**o+(-0.765500708496)*x[1]+(0.351369471606)*x[2]**o+(-0.351587727906)*x[2]
        ref[0]=(0.211945683696)/(o+1.)+(0.0613923521485)+(-0.472858663642)*0.5**o
        ref[1]=(1.12689776694)/(o+1.)+(-0.462024602231)+(-0.0191775972693)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactOne

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.994443255559)*x[0]**o+(-0.403427184912)*x[0]+(-0.889656446862)*x[1]**o+(-0.411934106034)*x[1]
        arg[0,1]=(-0.864086447061)*x[0]**o+(-0.277489357308)*x[0]+(0.977910864864)*x[1]**o+(-0.0505775488074)*x[1]
        arg[0,2]=(-0.397179114534)*x[0]**o+(-0.448046821439)*x[0]+(-0.340253609736)*x[1]**o+(0.624988837716)*x[1]
        arg[0,3]=(-0.9155191116)*x[0]**o+(0.133777450278)*x[0]+(-0.809567026065)*x[1]**o+(-0.815374059364)*x[1]
        arg[0,4]=(-0.416764684181)*x[0]**o+(0.0965164570638)*x[0]+(0.190191799833)*x[1]**o+(0.0970709424519)*x[1]
        arg[1,0]=(0.646527294535)*x[0]**o+(-0.337119506636)*x[0]+(0.42669563245)*x[1]**o+(-0.185228568706)*x[1]
        arg[1,1]=(-0.411532647934)*x[0]**o+(0.178830973832)*x[0]+(-0.904886020892)*x[1]**o+(-0.740623071275)*x[1]
        arg[1,2]=(0.523024622808)*x[0]**o+(-0.226298848077)*x[0]+(-0.707740568617)*x[1]**o+(0.998182796571)*x[1]
        arg[1,3]=(0.840649917357)*x[0]**o+(-0.736279151571)*x[0]+(0.996017554392)*x[1]**o+(-0.397833319931)*x[1]
        arg[1,4]=(0.76711764419)*x[0]**o+(-0.905452991298)*x[0]+(-0.925675237951)*x[1]**o+(-0.982712645237)*x[1]
        arg[2,0]=(0.198547103682)*x[0]**o+(0.674537682234)*x[0]+(-0.891195618193)*x[1]**o+(0.979109691481)*x[1]
        arg[2,1]=(0.171629921737)*x[0]**o+(0.0102727001751)*x[0]+(0.467578393374)*x[1]**o+(0.583960984645)*x[1]
        arg[2,2]=(-0.637526271497)*x[0]**o+(-0.900821073531)*x[0]+(0.61637733965)*x[1]**o+(0.43717505341)*x[1]
        arg[2,3]=(-0.205861963967)*x[0]**o+(-0.104621062003)*x[0]+(0.582584591946)*x[1]**o+(0.351739662419)*x[1]
        arg[2,4]=(-0.179355911316)*x[0]**o+(-0.188415282678)*x[0]+(0.598342895007)*x[1]**o+(0.364638908577)*x[1]
        arg[3,0]=(0.150471526119)*x[0]**o+(-0.940789873202)*x[0]+(-0.524275825864)*x[1]**o+(0.729004683023)*x[1]
        arg[3,1]=(0.123933417087)*x[0]**o+(0.192413160498)*x[0]+(0.203730760507)*x[1]**o+(0.852010172966)*x[1]
        arg[3,2]=(-0.297766950847)*x[0]**o+(0.269905131596)*x[0]+(-0.0830931256847)*x[1]**o+(0.26964712569)*x[1]
        arg[3,3]=(0.96961517265)*x[0]**o+(0.363312967302)*x[0]+(-0.811142762589)*x[1]**o+(-0.420442983537)*x[1]
        arg[3,4]=(-0.314185602206)*x[0]**o+(-0.0446515646101)*x[0]+(-0.884651038732)*x[1]**o+(-0.166386477611)*x[1]
        ref[0,0]=(-0.889656446862)/(o+1.)+(-0.407680645473)+(-0.994443255559)*0.5**o
        ref[0,1]=(0.977910864864)/(o+1.)+(-0.164033453058)+(-0.864086447061)*0.5**o
        ref[0,2]=(-0.340253609736)/(o+1.)+(0.0884710081389)+(-0.397179114534)*0.5**o
        ref[0,3]=(-0.809567026065)/(o+1.)+(-0.340798304543)+(-0.9155191116)*0.5**o
        ref[0,4]=(0.190191799833)/(o+1.)+(0.0967936997578)+(-0.416764684181)*0.5**o
        ref[1,0]=(0.42669563245)/(o+1.)+(-0.261174037671)+(0.646527294535)*0.5**o
        ref[1,1]=(-0.904886020892)/(o+1.)+(-0.280896048722)+(-0.411532647934)*0.5**o
        ref[1,2]=(-0.707740568617)/(o+1.)+(0.385941974247)+(0.523024622808)*0.5**o
        ref[1,3]=(0.996017554392)/(o+1.)+(-0.567056235751)+(0.840649917357)*0.5**o
        ref[1,4]=(-0.925675237951)/(o+1.)+(-0.944082818268)+(0.76711764419)*0.5**o
        ref[2,0]=(-0.891195618193)/(o+1.)+(0.826823686857)+(0.198547103682)*0.5**o
        ref[2,1]=(0.467578393374)/(o+1.)+(0.29711684241)+(0.171629921737)*0.5**o
        ref[2,2]=(0.61637733965)/(o+1.)+(-0.231823010061)+(-0.637526271497)*0.5**o
        ref[2,3]=(0.582584591946)/(o+1.)+(0.123559300208)+(-0.205861963967)*0.5**o
        ref[2,4]=(0.598342895007)/(o+1.)+(0.0881118129495)+(-0.179355911316)*0.5**o
        ref[3,0]=(-0.524275825864)/(o+1.)+(-0.105892595089)+(0.150471526119)*0.5**o
        ref[3,1]=(0.203730760507)/(o+1.)+(0.522211666732)+(0.123933417087)*0.5**o
        ref[3,2]=(-0.0830931256847)/(o+1.)+(0.269776128643)+(-0.297766950847)*0.5**o
        ref[3,3]=(-0.811142762589)/(o+1.)+(-0.0285650081177)+(0.96961517265)*0.5**o
        ref[3,4]=(-0.884651038732)/(o+1.)+(-0.105519021111)+(-0.314185602206)*0.5**o
      else:
        arg[0,0]=(0.264691789613)*x[0]**o+(0.816541340997)*x[0]+(-0.402990464667)*x[1]**o+(-0.57479619687)*x[1]+(0.336313666995)*x[2]**o+(-0.353125285536)*x[2]
        arg[0,1]=(-0.0526559162494)*x[0]**o+(-0.901336706583)*x[0]+(0.237482536727)*x[1]**o+(-0.458564275773)*x[1]+(-0.90715992057)*x[2]**o+(-0.352062856077)*x[2]
        arg[0,2]=(0.786349112412)*x[0]**o+(-0.395569856472)*x[0]+(-0.996593023157)*x[1]**o+(-0.277001350627)*x[1]+(0.435375913289)*x[2]**o+(0.00380742297736)*x[2]
        arg[0,3]=(0.3336751197)*x[0]**o+(0.592185657129)*x[0]+(0.148956266814)*x[1]**o+(-0.852264651722)*x[1]+(-0.680249245511)*x[2]**o+(0.238843544413)*x[2]
        arg[0,4]=(0.875206676786)*x[0]**o+(0.0420423017124)*x[0]+(-0.690488811412)*x[1]**o+(-0.191328696331)*x[1]+(-0.861423624177)*x[2]**o+(0.483925405411)*x[2]
        arg[1,0]=(-0.573216715158)*x[0]**o+(-0.0668114726198)*x[0]+(0.382097918101)*x[1]**o+(-0.126485626438)*x[1]+(0.37521336616)*x[2]**o+(0.42402066058)*x[2]
        arg[1,1]=(-0.0919326514467)*x[0]**o+(-0.791351880042)*x[0]+(0.772694542694)*x[1]**o+(-0.537902003787)*x[1]+(0.424413342503)*x[2]**o+(-0.285553230731)*x[2]
        arg[1,2]=(0.855490362389)*x[0]**o+(0.60428104628)*x[0]+(-0.846565232506)*x[1]**o+(-0.243727717758)*x[1]+(0.900077399935)*x[2]**o+(-0.974305489047)*x[2]
        arg[1,3]=(-0.90802343526)*x[0]**o+(0.00950587842259)*x[0]+(0.821234711321)*x[1]**o+(-0.0628912085092)*x[1]+(-0.773563396868)*x[2]**o+(-0.403955952419)*x[2]
        arg[1,4]=(-0.33139358787)*x[0]**o+(-0.901382943687)*x[0]+(-0.981921375707)*x[1]**o+(-0.0654519807133)*x[1]+(0.880069861158)*x[2]**o+(-0.0796172270076)*x[2]
        arg[2,0]=(-0.202818692518)*x[0]**o+(-0.155436611971)*x[0]+(0.739467305515)*x[1]**o+(-0.12096877475)*x[1]+(0.479644796804)*x[2]**o+(0.0580715322389)*x[2]
        arg[2,1]=(0.0461326006936)*x[0]**o+(0.122168224674)*x[0]+(0.986788073931)*x[1]**o+(-0.435933938764)*x[1]+(-0.312527130215)*x[2]**o+(0.726805187165)*x[2]
        arg[2,2]=(-0.539659153805)*x[0]**o+(-0.26082353681)*x[0]+(0.367739719819)*x[1]**o+(0.943584828354)*x[1]+(0.688225221556)*x[2]**o+(-0.121563525696)*x[2]
        arg[2,3]=(0.861717128203)*x[0]**o+(0.0666983176426)*x[0]+(0.984510848927)*x[1]**o+(-0.321865378029)*x[1]+(0.522584734)*x[2]**o+(0.589527116625)*x[2]
        arg[2,4]=(0.778575699703)*x[0]**o+(0.515802361664)*x[0]+(0.817655905624)*x[1]**o+(0.883824717911)*x[1]+(0.656833761714)*x[2]**o+(-0.991585145611)*x[2]
        arg[3,0]=(0.752875513075)*x[0]**o+(0.615136567082)*x[0]+(-0.205265870704)*x[1]**o+(0.483918142728)*x[1]+(-0.0495322147557)*x[2]**o+(-0.132474571954)*x[2]
        arg[3,1]=(0.0563875061408)*x[0]**o+(-0.603033440119)*x[0]+(-0.0944809376206)*x[1]**o+(0.725330686731)*x[1]+(0.00527671533343)*x[2]**o+(-0.434519675586)*x[2]
        arg[3,2]=(0.100078550382)*x[0]**o+(-0.340988639989)*x[0]+(-0.392591649097)*x[1]**o+(-0.924705734588)*x[1]+(0.420410192075)*x[2]**o+(-0.94741221593)*x[2]
        arg[3,3]=(0.536414145801)*x[0]**o+(0.142421640429)*x[0]+(0.213162255423)*x[1]**o+(-0.695193008147)*x[1]+(0.9625091599)*x[2]**o+(0.534409542153)*x[2]
        arg[3,4]=(0.305683606235)*x[0]**o+(0.340964210305)*x[0]+(0.757807973246)*x[1]**o+(0.11459551493)*x[1]+(-0.81395047506)*x[2]**o+(0.636392281491)*x[2]
        ref[0,0]=(-0.0666767976714)/(o+1.)+(-0.0556900707042)+(0.264691789613)*0.5**o
        ref[0,1]=(-0.669677383843)/(o+1.)+(-0.855981919217)+(-0.0526559162494)*0.5**o
        ref[0,2]=(-0.561217109869)/(o+1.)+(-0.334381892061)+(0.786349112412)*0.5**o
        ref[0,3]=(-0.531292978697)/(o+1.)+(-0.0106177250896)+(0.3336751197)*0.5**o
        ref[0,4]=(-1.55191243559)/(o+1.)+(0.167319505396)+(0.875206676786)*0.5**o
        ref[1,0]=(0.757311284261)/(o+1.)+(0.115361780761)+(-0.573216715158)*0.5**o
        ref[1,1]=(1.1971078852)/(o+1.)+(-0.80740355728)+(-0.0919326514467)*0.5**o
        ref[1,2]=(0.0535121674292)/(o+1.)+(-0.306876080263)+(0.855490362389)*0.5**o
        ref[1,3]=(0.0476713144526)/(o+1.)+(-0.228670641253)+(-0.90802343526)*0.5**o
        ref[1,4]=(-0.101851514548)/(o+1.)+(-0.523226075704)+(-0.33139358787)*0.5**o
        ref[2,0]=(1.21911210232)/(o+1.)+(-0.109166927241)+(-0.202818692518)*0.5**o
        ref[2,1]=(0.674260943716)/(o+1.)+(0.206519736538)+(0.0461326006936)*0.5**o
        ref[2,2]=(1.05596494137)/(o+1.)+(0.280598882924)+(-0.539659153805)*0.5**o
        ref[2,3]=(1.50709558293)/(o+1.)+(0.167180028119)+(0.861717128203)*0.5**o
        ref[2,4]=(1.47448966734)/(o+1.)+(0.204020966982)+(0.778575699703)*0.5**o
        ref[3,0]=(-0.25479808546)/(o+1.)+(0.483290068928)+(0.752875513075)*0.5**o
        ref[3,1]=(-0.0892042222872)/(o+1.)+(-0.156111214487)+(0.0563875061408)*0.5**o
        ref[3,2]=(0.0278185429781)/(o+1.)+(-1.10655329525)+(0.100078550382)*0.5**o
        ref[3,3]=(1.17567141532)/(o+1.)+(-0.00918091278228)+(0.536414145801)*0.5**o
        ref[3,4]=(-0.0561425018138)/(o+1.)+(0.545976003363)+(0.305683606235)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnContactOne

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.448877659671)*x[0]**o+(0.130599173692)*x[0]+(-0.264185555629)*x[1]**o+(-0.722726040492)*x[1]
        arg[0,0,1]=(-0.543806308372)*x[0]**o+(-0.685242399166)*x[0]+(0.325336801501)*x[1]**o+(0.572960380228)*x[1]
        arg[0,1,0]=(-0.893080563297)*x[0]**o+(-0.251553981066)*x[0]+(0.256078388483)*x[1]**o+(-0.926701685259)*x[1]
        arg[0,1,1]=(0.346088914447)*x[0]**o+(0.166180418696)*x[0]+(0.353024178635)*x[1]**o+(0.731811935721)*x[1]
        arg[1,0,0]=(0.553083243483)*x[0]**o+(0.976153803522)*x[0]+(-0.260777317634)*x[1]**o+(-0.641936915628)*x[1]
        arg[1,0,1]=(-0.561076901068)*x[0]**o+(0.837832618302)*x[0]+(-0.638970151828)*x[1]**o+(-0.495169195543)*x[1]
        arg[1,1,0]=(-0.292805925132)*x[0]**o+(0.670472035586)*x[0]+(-0.752026521307)*x[1]**o+(-0.473087472861)*x[1]
        arg[1,1,1]=(-0.432857805275)*x[0]**o+(0.495387797249)*x[0]+(0.0579881469257)*x[1]**o+(0.831644440274)*x[1]
        arg[2,0,0]=(0.413433231384)*x[0]**o+(0.919831571878)*x[0]+(-0.373581626416)*x[1]**o+(0.16423318333)*x[1]
        arg[2,0,1]=(0.857566619293)*x[0]**o+(0.340454525569)*x[0]+(0.139298007219)*x[1]**o+(0.220542978265)*x[1]
        arg[2,1,0]=(-0.914184896115)*x[0]**o+(-0.0279186159835)*x[0]+(0.893417478351)*x[1]**o+(-0.108178714604)*x[1]
        arg[2,1,1]=(-0.749629612712)*x[0]**o+(-0.649957092299)*x[0]+(0.181944194719)*x[1]**o+(-0.988782927329)*x[1]
        arg[3,0,0]=(0.292013067678)*x[0]**o+(0.491680364317)*x[0]+(0.951144487798)*x[1]**o+(0.980949630108)*x[1]
        arg[3,0,1]=(0.89779245885)*x[0]**o+(-0.147546711846)*x[0]+(0.0902450198362)*x[1]**o+(-0.669984062969)*x[1]
        arg[3,1,0]=(0.62590631314)*x[0]**o+(0.329384454978)*x[0]+(-0.676917185409)*x[1]**o+(-0.0903173226074)*x[1]
        arg[3,1,1]=(-0.668268661271)*x[0]**o+(-0.847167879584)*x[0]+(0.329431843319)*x[1]**o+(-0.989253343115)*x[1]
        arg[4,0,0]=(-0.675111374291)*x[0]**o+(0.738030423939)*x[0]+(-0.413299452027)*x[1]**o+(0.36859580301)*x[1]
        arg[4,0,1]=(-0.207625666422)*x[0]**o+(-0.455639256266)*x[0]+(0.554118743878)*x[1]**o+(0.14044849621)*x[1]
        arg[4,1,0]=(-0.313337039275)*x[0]**o+(-0.451797308521)*x[0]+(0.201296275104)*x[1]**o+(-0.779869953962)*x[1]
        arg[4,1,1]=(-0.0308697706768)*x[0]**o+(0.214664850788)*x[0]+(-0.824157101269)*x[1]**o+(0.156012836486)*x[1]
        arg[5,0,0]=(0.277885566309)*x[0]**o+(-0.387820938776)*x[0]+(0.820602896312)*x[1]**o+(0.162715513355)*x[1]
        arg[5,0,1]=(-0.216951421349)*x[0]**o+(0.906019329549)*x[0]+(-0.330132336808)*x[1]**o+(0.416109194357)*x[1]
        arg[5,1,0]=(0.923496627826)*x[0]**o+(-0.177644464371)*x[0]+(0.482278717995)*x[1]**o+(-0.362003106154)*x[1]
        arg[5,1,1]=(0.9109111591)*x[0]**o+(-0.0919782899769)*x[0]+(0.103950387446)*x[1]**o+(0.447452648868)*x[1]
        ref[0,0,0]=(-0.264185555629)/(o+1.)+(-0.2960634334)+(0.448877659671)*0.5**o
        ref[0,0,1]=(0.325336801501)/(o+1.)+(-0.0561410094689)+(-0.543806308372)*0.5**o
        ref[0,1,0]=(0.256078388483)/(o+1.)+(-0.589127833162)+(-0.893080563297)*0.5**o
        ref[0,1,1]=(0.353024178635)/(o+1.)+(0.448996177208)+(0.346088914447)*0.5**o
        ref[1,0,0]=(-0.260777317634)/(o+1.)+(0.167108443947)+(0.553083243483)*0.5**o
        ref[1,0,1]=(-0.638970151828)/(o+1.)+(0.17133171138)+(-0.561076901068)*0.5**o
        ref[1,1,0]=(-0.752026521307)/(o+1.)+(0.0986922813626)+(-0.292805925132)*0.5**o
        ref[1,1,1]=(0.0579881469257)/(o+1.)+(0.663516118762)+(-0.432857805275)*0.5**o
        ref[2,0,0]=(-0.373581626416)/(o+1.)+(0.542032377604)+(0.413433231384)*0.5**o
        ref[2,0,1]=(0.139298007219)/(o+1.)+(0.280498751917)+(0.857566619293)*0.5**o
        ref[2,1,0]=(0.893417478351)/(o+1.)+(-0.0680486652939)+(-0.914184896115)*0.5**o
        ref[2,1,1]=(0.181944194719)/(o+1.)+(-0.819370009814)+(-0.749629612712)*0.5**o
        ref[3,0,0]=(0.951144487798)/(o+1.)+(0.736314997212)+(0.292013067678)*0.5**o
        ref[3,0,1]=(0.0902450198362)/(o+1.)+(-0.408765387408)+(0.89779245885)*0.5**o
        ref[3,1,0]=(-0.676917185409)/(o+1.)+(0.119533566185)+(0.62590631314)*0.5**o
        ref[3,1,1]=(0.329431843319)/(o+1.)+(-0.91821061135)+(-0.668268661271)*0.5**o
        ref[4,0,0]=(-0.413299452027)/(o+1.)+(0.553313113475)+(-0.675111374291)*0.5**o
        ref[4,0,1]=(0.554118743878)/(o+1.)+(-0.157595380028)+(-0.207625666422)*0.5**o
        ref[4,1,0]=(0.201296275104)/(o+1.)+(-0.615833631241)+(-0.313337039275)*0.5**o
        ref[4,1,1]=(-0.824157101269)/(o+1.)+(0.185338843637)+(-0.0308697706768)*0.5**o
        ref[5,0,0]=(0.820602896312)/(o+1.)+(-0.11255271271)+(0.277885566309)*0.5**o
        ref[5,0,1]=(-0.330132336808)/(o+1.)+(0.661064261953)+(-0.216951421349)*0.5**o
        ref[5,1,0]=(0.482278717995)/(o+1.)+(-0.269823785262)+(0.923496627826)*0.5**o
        ref[5,1,1]=(0.103950387446)/(o+1.)+(0.177737179446)+(0.9109111591)*0.5**o
      else:
        arg[0,0,0]=(0.843169691966)*x[0]**o+(0.252560741792)*x[0]+(0.0428740803857)*x[1]**o+(-0.702885246223)*x[1]+(0.175406156382)*x[2]**o+(-0.819206174419)*x[2]
        arg[0,0,1]=(-0.780678250187)*x[0]**o+(-0.628781977796)*x[0]+(0.345241549754)*x[1]**o+(-0.686783366275)*x[1]+(0.298501249797)*x[2]**o+(-0.592222734778)*x[2]
        arg[0,1,0]=(0.680331984783)*x[0]**o+(0.20945731105)*x[0]+(0.582220641498)*x[1]**o+(0.159144772508)*x[1]+(0.197544685245)*x[2]**o+(-0.542847499692)*x[2]
        arg[0,1,1]=(0.561649107089)*x[0]**o+(0.409538404439)*x[0]+(0.687461001448)*x[1]**o+(0.327986995905)*x[1]+(0.701007758697)*x[2]**o+(0.95919995454)*x[2]
        arg[1,0,0]=(0.0620287124894)*x[0]**o+(-0.946536581396)*x[0]+(-0.813637608115)*x[1]**o+(0.0808967031579)*x[1]+(0.545408819059)*x[2]**o+(0.250558134093)*x[2]
        arg[1,0,1]=(-0.0778300410758)*x[0]**o+(-0.852201314682)*x[0]+(0.712528858815)*x[1]**o+(-0.69009644973)*x[1]+(-0.0836140529567)*x[2]**o+(0.687579338803)*x[2]
        arg[1,1,0]=(-0.37648103863)*x[0]**o+(-0.264461846666)*x[0]+(-0.45921598716)*x[1]**o+(-0.516551658367)*x[1]+(0.512446750786)*x[2]**o+(0.850266522322)*x[2]
        arg[1,1,1]=(-0.951741356481)*x[0]**o+(-0.989023540683)*x[0]+(0.338787640816)*x[1]**o+(0.764043925298)*x[1]+(0.524009019713)*x[2]**o+(-0.912507794534)*x[2]
        arg[2,0,0]=(0.613468181683)*x[0]**o+(-0.357325179573)*x[0]+(0.153221256731)*x[1]**o+(-0.748686771579)*x[1]+(0.364085012368)*x[2]**o+(-0.134535056195)*x[2]
        arg[2,0,1]=(-0.728948376511)*x[0]**o+(-0.0657162843876)*x[0]+(-0.336013775703)*x[1]**o+(0.299044747397)*x[1]+(-0.712877779871)*x[2]**o+(0.229974926734)*x[2]
        arg[2,1,0]=(-0.213428159952)*x[0]**o+(0.680250111549)*x[0]+(-0.986942572737)*x[1]**o+(0.549006049647)*x[1]+(0.572921637402)*x[2]**o+(-0.241851127933)*x[2]
        arg[2,1,1]=(0.0367155018841)*x[0]**o+(-0.768111161527)*x[0]+(-0.748085216726)*x[1]**o+(-0.752536272228)*x[1]+(0.555945063877)*x[2]**o+(0.012687074005)*x[2]
        arg[3,0,0]=(-0.945391902075)*x[0]**o+(-0.581330133808)*x[0]+(-0.253390460088)*x[1]**o+(-0.885589795648)*x[1]+(-0.555900062171)*x[2]**o+(-0.241006607054)*x[2]
        arg[3,0,1]=(0.924110231188)*x[0]**o+(0.428520753545)*x[0]+(0.187773835144)*x[1]**o+(-0.53787305121)*x[1]+(0.0970284978901)*x[2]**o+(0.0580212350951)*x[2]
        arg[3,1,0]=(0.426389283701)*x[0]**o+(-0.0976148877161)*x[0]+(-0.364828466078)*x[1]**o+(-0.536398529197)*x[1]+(0.648605043498)*x[2]**o+(-0.234629706559)*x[2]
        arg[3,1,1]=(-0.736098543054)*x[0]**o+(0.985014182905)*x[0]+(-0.563513361433)*x[1]**o+(0.589434310221)*x[1]+(-0.360296462891)*x[2]**o+(-0.586591636182)*x[2]
        arg[4,0,0]=(0.953916732391)*x[0]**o+(-0.835620914249)*x[0]+(0.562803727423)*x[1]**o+(-0.904263689551)*x[1]+(-0.349815820142)*x[2]**o+(-0.156410988081)*x[2]
        arg[4,0,1]=(-0.236710267891)*x[0]**o+(0.958992060128)*x[0]+(0.711565908363)*x[1]**o+(-0.639771085045)*x[1]+(-0.919042661652)*x[2]**o+(-0.198907436333)*x[2]
        arg[4,1,0]=(0.275058935551)*x[0]**o+(0.563461500878)*x[0]+(0.438833998471)*x[1]**o+(0.686375712335)*x[1]+(-0.930322605963)*x[2]**o+(0.710777122765)*x[2]
        arg[4,1,1]=(-0.507845102241)*x[0]**o+(-0.390574086909)*x[0]+(-0.170020669232)*x[1]**o+(-0.107455242464)*x[1]+(-0.97056652533)*x[2]**o+(-0.720180192255)*x[2]
        arg[5,0,0]=(-0.922597172214)*x[0]**o+(0.229197305733)*x[0]+(-0.943214499543)*x[1]**o+(0.269256599046)*x[1]+(-0.11540537404)*x[2]**o+(0.956940487934)*x[2]
        arg[5,0,1]=(-0.0350868975578)*x[0]**o+(-0.547620412319)*x[0]+(-0.818344866429)*x[1]**o+(-0.0632082353078)*x[1]+(-0.490536679501)*x[2]**o+(-0.79641747407)*x[2]
        arg[5,1,0]=(0.877279477171)*x[0]**o+(-0.408752853807)*x[0]+(0.143432191997)*x[1]**o+(0.230679627667)*x[1]+(0.0346230515425)*x[2]**o+(0.357377660033)*x[2]
        arg[5,1,1]=(0.294375780312)*x[0]**o+(-0.688447065878)*x[0]+(0.213334642918)*x[1]**o+(0.664021137697)*x[1]+(-0.532996065961)*x[2]**o+(0.794315136407)*x[2]
        ref[0,0,0]=(0.218280236768)/(o+1.)+(-0.634765339425)+(0.843169691966)*0.5**o
        ref[0,0,1]=(0.64374279955)/(o+1.)+(-0.953894039424)+(-0.780678250187)*0.5**o
        ref[0,1,0]=(0.779765326742)/(o+1.)+(-0.0871227080671)+(0.680331984783)*0.5**o
        ref[0,1,1]=(1.38846876015)/(o+1.)+(0.848362677442)+(0.561649107089)*0.5**o
        ref[1,0,0]=(-0.268228789055)/(o+1.)+(-0.307540872072)+(0.0620287124894)*0.5**o
        ref[1,0,1]=(0.628914805859)/(o+1.)+(-0.427359212804)+(-0.0778300410758)*0.5**o
        ref[1,1,0]=(0.0532307636262)/(o+1.)+(0.0346265086445)+(-0.37648103863)*0.5**o
        ref[1,1,1]=(0.862796660529)/(o+1.)+(-0.56874370496)+(-0.951741356481)*0.5**o
        ref[2,0,0]=(0.5173062691)/(o+1.)+(-0.620273503673)+(0.613468181683)*0.5**o
        ref[2,0,1]=(-1.04889155557)/(o+1.)+(0.231651694871)+(-0.728948376511)*0.5**o
        ref[2,1,0]=(-0.414020935335)/(o+1.)+(0.493702516631)+(-0.213428159952)*0.5**o
        ref[2,1,1]=(-0.192140152849)/(o+1.)+(-0.753980179875)+(0.0367155018841)*0.5**o
        ref[3,0,0]=(-0.809290522259)/(o+1.)+(-0.853963268255)+(-0.945391902075)*0.5**o
        ref[3,0,1]=(0.284802333034)/(o+1.)+(-0.0256655312851)+(0.924110231188)*0.5**o
        ref[3,1,0]=(0.28377657742)/(o+1.)+(-0.434321561736)+(0.426389283701)*0.5**o
        ref[3,1,1]=(-0.923809824324)/(o+1.)+(0.493928428472)+(-0.736098543054)*0.5**o
        ref[4,0,0]=(0.212987907281)/(o+1.)+(-0.948147795941)+(0.953916732391)*0.5**o
        ref[4,0,1]=(-0.207476753289)/(o+1.)+(0.0601567693744)+(-0.236710267891)*0.5**o
        ref[4,1,0]=(-0.491488607492)/(o+1.)+(0.980307167989)+(0.275058935551)*0.5**o
        ref[4,1,1]=(-1.14058719456)/(o+1.)+(-0.609104760814)+(-0.507845102241)*0.5**o
        ref[5,0,0]=(-1.05861987358)/(o+1.)+(0.727697196357)+(-0.922597172214)*0.5**o
        ref[5,0,1]=(-1.30888154593)/(o+1.)+(-0.703623060848)+(-0.0350868975578)*0.5**o
        ref[5,1,0]=(0.17805524354)/(o+1.)+(0.0896522169463)+(0.877279477171)*0.5**o
        ref[5,1,1]=(-0.319661423044)/(o+1.)+(0.384944604113)+(0.294375780312)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnContactOne

      assumptions: ContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.182956096072)*x[0]**o+(0.857065806542)*x[0]+(-0.0352052809673)*x[1]**o+(0.950661773214)*x[1]
        arg[0,0,0,1]=(-0.81342189211)*x[0]**o+(0.36137453013)*x[0]+(0.0610373691872)*x[1]**o+(-0.974859653337)*x[1]
        arg[0,0,1,0]=(-0.149103531408)*x[0]**o+(0.853517194849)*x[0]+(0.623943596759)*x[1]**o+(0.441477983782)*x[1]
        arg[0,0,1,1]=(0.74912087771)*x[0]**o+(0.935896347068)*x[0]+(0.136600247183)*x[1]**o+(0.915043476739)*x[1]
        arg[0,0,2,0]=(-0.289042694587)*x[0]**o+(-0.0324580134489)*x[0]+(0.966454801911)*x[1]**o+(0.158747002668)*x[1]
        arg[0,0,2,1]=(0.0809737838902)*x[0]**o+(0.483125829124)*x[0]+(-0.156074498516)*x[1]**o+(-0.387215102608)*x[1]
        arg[0,1,0,0]=(0.638822193145)*x[0]**o+(-0.211942445721)*x[0]+(-0.016865109839)*x[1]**o+(-0.596486509057)*x[1]
        arg[0,1,0,1]=(0.611604473356)*x[0]**o+(-0.330484367791)*x[0]+(-0.897448372147)*x[1]**o+(-0.0188556077191)*x[1]
        arg[0,1,1,0]=(-0.745407006471)*x[0]**o+(0.547985829362)*x[0]+(-0.229098133178)*x[1]**o+(0.339357994536)*x[1]
        arg[0,1,1,1]=(0.756528549281)*x[0]**o+(0.0968167016962)*x[0]+(0.481206893907)*x[1]**o+(-0.320299781291)*x[1]
        arg[0,1,2,0]=(0.335544368565)*x[0]**o+(0.143490409353)*x[0]+(0.367303506039)*x[1]**o+(-0.459001464112)*x[1]
        arg[0,1,2,1]=(0.669933676654)*x[0]**o+(0.0679909847194)*x[0]+(0.182483888026)*x[1]**o+(-0.382138842505)*x[1]
        arg[0,2,0,0]=(-0.929163949925)*x[0]**o+(0.670533855076)*x[0]+(0.587675819052)*x[1]**o+(0.221939557678)*x[1]
        arg[0,2,0,1]=(0.639896773528)*x[0]**o+(-0.166901547286)*x[0]+(-0.83934259298)*x[1]**o+(-0.522536841927)*x[1]
        arg[0,2,1,0]=(-0.399556357179)*x[0]**o+(0.860582177909)*x[0]+(0.675592435624)*x[1]**o+(0.77480796903)*x[1]
        arg[0,2,1,1]=(-0.281969675471)*x[0]**o+(-0.250831251067)*x[0]+(0.14396936899)*x[1]**o+(-0.65775670156)*x[1]
        arg[0,2,2,0]=(0.339204154215)*x[0]**o+(0.125319656279)*x[0]+(0.417557775604)*x[1]**o+(-0.985431777777)*x[1]
        arg[0,2,2,1]=(-0.655500794874)*x[0]**o+(-0.419889131296)*x[0]+(-0.0763188160723)*x[1]**o+(0.995600932757)*x[1]
        arg[0,3,0,0]=(-0.483746115201)*x[0]**o+(-0.256632816498)*x[0]+(-0.0533973517656)*x[1]**o+(-0.982541041474)*x[1]
        arg[0,3,0,1]=(0.446810992304)*x[0]**o+(-0.823523374302)*x[0]+(-0.607238533539)*x[1]**o+(-0.97562384609)*x[1]
        arg[0,3,1,0]=(-0.0235907627093)*x[0]**o+(-0.0720587626585)*x[0]+(0.930589323947)*x[1]**o+(0.327916365569)*x[1]
        arg[0,3,1,1]=(0.486342062258)*x[0]**o+(0.430279082913)*x[0]+(0.237445053125)*x[1]**o+(0.577650421809)*x[1]
        arg[0,3,2,0]=(-0.581025309937)*x[0]**o+(0.97026755344)*x[0]+(0.180505264557)*x[1]**o+(0.289585805796)*x[1]
        arg[0,3,2,1]=(-0.523356041281)*x[0]**o+(-0.502230606793)*x[0]+(0.915270822508)*x[1]**o+(0.622076448072)*x[1]
        arg[0,4,0,0]=(-0.584782530121)*x[0]**o+(-0.433761163729)*x[0]+(-0.449731522195)*x[1]**o+(0.392234342377)*x[1]
        arg[0,4,0,1]=(-0.0724106581167)*x[0]**o+(0.76896002907)*x[0]+(0.0374933582543)*x[1]**o+(-0.417790382045)*x[1]
        arg[0,4,1,0]=(0.0460360663465)*x[0]**o+(-0.0343116732267)*x[0]+(0.815864788848)*x[1]**o+(-0.922236546574)*x[1]
        arg[0,4,1,1]=(-0.978584425712)*x[0]**o+(0.249337867126)*x[0]+(-0.822961904166)*x[1]**o+(0.627473480274)*x[1]
        arg[0,4,2,0]=(-0.267227111945)*x[0]**o+(0.791320996051)*x[0]+(0.721009228711)*x[1]**o+(0.00199546728243)*x[1]
        arg[0,4,2,1]=(-0.141372777283)*x[0]**o+(-0.996912487736)*x[0]+(0.94281427253)*x[1]**o+(-0.780685413978)*x[1]
        arg[1,0,0,0]=(-0.721911758033)*x[0]**o+(0.0119756829508)*x[0]+(0.583083015744)*x[1]**o+(0.80826073507)*x[1]
        arg[1,0,0,1]=(0.929636966638)*x[0]**o+(-0.35107696413)*x[0]+(-0.656523378462)*x[1]**o+(0.610681274197)*x[1]
        arg[1,0,1,0]=(0.594847872723)*x[0]**o+(-0.15013091781)*x[0]+(-0.562311637909)*x[1]**o+(0.871283992407)*x[1]
        arg[1,0,1,1]=(0.529274819592)*x[0]**o+(0.998035152453)*x[0]+(-0.265309908189)*x[1]**o+(-0.703391843389)*x[1]
        arg[1,0,2,0]=(0.2321959905)*x[0]**o+(-0.781602927232)*x[0]+(-0.0509405210126)*x[1]**o+(0.839298348419)*x[1]
        arg[1,0,2,1]=(0.0811824999287)*x[0]**o+(0.916373714851)*x[0]+(0.817375877796)*x[1]**o+(-0.926274023089)*x[1]
        arg[1,1,0,0]=(0.418525096774)*x[0]**o+(0.064653235493)*x[0]+(0.15912570069)*x[1]**o+(0.47570489259)*x[1]
        arg[1,1,0,1]=(0.784138248952)*x[0]**o+(-0.715130508682)*x[0]+(-0.299421222048)*x[1]**o+(-0.816706522916)*x[1]
        arg[1,1,1,0]=(-0.911449050332)*x[0]**o+(-0.472182975049)*x[0]+(0.935180253062)*x[1]**o+(0.527208406385)*x[1]
        arg[1,1,1,1]=(-0.378301315644)*x[0]**o+(0.430065353544)*x[0]+(0.827581865806)*x[1]**o+(-0.355263817375)*x[1]
        arg[1,1,2,0]=(-0.882543004053)*x[0]**o+(0.503794963637)*x[0]+(-0.580086073626)*x[1]**o+(0.545676378283)*x[1]
        arg[1,1,2,1]=(-0.116197074531)*x[0]**o+(-0.0629871602139)*x[0]+(0.320343551712)*x[1]**o+(0.0552550885696)*x[1]
        arg[1,2,0,0]=(-0.870347384636)*x[0]**o+(-0.611439594673)*x[0]+(0.707766015783)*x[1]**o+(0.0653036268139)*x[1]
        arg[1,2,0,1]=(-0.219872666014)*x[0]**o+(-0.435867669656)*x[0]+(0.825489444837)*x[1]**o+(-0.737253859382)*x[1]
        arg[1,2,1,0]=(-0.963149287055)*x[0]**o+(0.964306345088)*x[0]+(0.502736178959)*x[1]**o+(0.998017404878)*x[1]
        arg[1,2,1,1]=(-0.586350249194)*x[0]**o+(-0.437455605038)*x[0]+(-0.182719660538)*x[1]**o+(-0.50797820305)*x[1]
        arg[1,2,2,0]=(-0.423818475166)*x[0]**o+(0.290902418459)*x[0]+(-0.304129806418)*x[1]**o+(-0.67571878331)*x[1]
        arg[1,2,2,1]=(0.901384945886)*x[0]**o+(0.0526906910209)*x[0]+(0.438755721196)*x[1]**o+(-0.774352941544)*x[1]
        arg[1,3,0,0]=(-0.797968602257)*x[0]**o+(0.856637297517)*x[0]+(0.427243495756)*x[1]**o+(0.212942988457)*x[1]
        arg[1,3,0,1]=(-0.696126104381)*x[0]**o+(0.616445534254)*x[0]+(-0.943037028688)*x[1]**o+(0.807265721375)*x[1]
        arg[1,3,1,0]=(-0.69028368426)*x[0]**o+(-0.652017674488)*x[0]+(-0.772965740874)*x[1]**o+(-0.726830598291)*x[1]
        arg[1,3,1,1]=(-0.534347181294)*x[0]**o+(-0.861020241321)*x[0]+(-0.250549859157)*x[1]**o+(-0.638289396201)*x[1]
        arg[1,3,2,0]=(-0.827981962115)*x[0]**o+(-0.427544131008)*x[0]+(-0.516514308221)*x[1]**o+(-0.101906062547)*x[1]
        arg[1,3,2,1]=(0.864701869194)*x[0]**o+(0.876327129524)*x[0]+(0.806201346091)*x[1]**o+(0.845259832272)*x[1]
        arg[1,4,0,0]=(-0.589044559231)*x[0]**o+(0.697104039773)*x[0]+(-0.288389617381)*x[1]**o+(0.422618969092)*x[1]
        arg[1,4,0,1]=(0.814971550967)*x[0]**o+(-0.665151459593)*x[0]+(-0.696145763418)*x[1]**o+(0.906949165457)*x[1]
        arg[1,4,1,0]=(0.760190450826)*x[0]**o+(-0.883964913626)*x[0]+(-0.507159320061)*x[1]**o+(-0.15707026975)*x[1]
        arg[1,4,1,1]=(-0.0416603697257)*x[0]**o+(0.560561779164)*x[0]+(0.0153574982122)*x[1]**o+(0.75228872946)*x[1]
        arg[1,4,2,0]=(0.105929120872)*x[0]**o+(0.130738968719)*x[0]+(-0.0991159921294)*x[1]**o+(0.972627650896)*x[1]
        arg[1,4,2,1]=(-0.503911541225)*x[0]**o+(-0.00357063650385)*x[0]+(0.377083955928)*x[1]**o+(0.343010153434)*x[1]
        arg[2,0,0,0]=(0.953733725947)*x[0]**o+(-0.988629842686)*x[0]+(0.196766730386)*x[1]**o+(0.0427993038279)*x[1]
        arg[2,0,0,1]=(0.491469820597)*x[0]**o+(0.708248604732)*x[0]+(-0.600347310418)*x[1]**o+(-0.0380188651191)*x[1]
        arg[2,0,1,0]=(0.615118477478)*x[0]**o+(0.832030863133)*x[0]+(0.783037278839)*x[1]**o+(-0.331824170309)*x[1]
        arg[2,0,1,1]=(-0.806125471265)*x[0]**o+(0.763233458333)*x[0]+(-0.381389710969)*x[1]**o+(0.652974431609)*x[1]
        arg[2,0,2,0]=(-0.831727679795)*x[0]**o+(0.239296785135)*x[0]+(-0.13737609858)*x[1]**o+(0.583210422723)*x[1]
        arg[2,0,2,1]=(-0.805843206194)*x[0]**o+(-0.523576669019)*x[0]+(0.159758297984)*x[1]**o+(-0.298308874886)*x[1]
        arg[2,1,0,0]=(0.589135530105)*x[0]**o+(0.272145879514)*x[0]+(0.325741074892)*x[1]**o+(-0.579782654647)*x[1]
        arg[2,1,0,1]=(0.106049062774)*x[0]**o+(0.790280056075)*x[0]+(0.0171600098883)*x[1]**o+(-0.419540445094)*x[1]
        arg[2,1,1,0]=(-0.181511670026)*x[0]**o+(-0.814531462508)*x[0]+(-0.576165158737)*x[1]**o+(0.679678310717)*x[1]
        arg[2,1,1,1]=(-0.709695453728)*x[0]**o+(-0.489487606257)*x[0]+(-0.424093531654)*x[1]**o+(0.535427169862)*x[1]
        arg[2,1,2,0]=(0.60069280023)*x[0]**o+(-0.909882731214)*x[0]+(0.640141428017)*x[1]**o+(-0.697396064954)*x[1]
        arg[2,1,2,1]=(0.896964594562)*x[0]**o+(0.976387328697)*x[0]+(0.144061490066)*x[1]**o+(0.378042869796)*x[1]
        arg[2,2,0,0]=(0.447633246258)*x[0]**o+(-0.531416367062)*x[0]+(0.199304002742)*x[1]**o+(0.85195706213)*x[1]
        arg[2,2,0,1]=(-0.683614706554)*x[0]**o+(0.933255565209)*x[0]+(-0.328047392082)*x[1]**o+(-0.0865769137182)*x[1]
        arg[2,2,1,0]=(0.22474720984)*x[0]**o+(0.0910500178874)*x[0]+(0.449220001922)*x[1]**o+(-0.0406214557829)*x[1]
        arg[2,2,1,1]=(-0.266419887395)*x[0]**o+(-0.48883285131)*x[0]+(-0.794933934805)*x[1]**o+(0.182439649969)*x[1]
        arg[2,2,2,0]=(0.949852186215)*x[0]**o+(-0.116121704259)*x[0]+(-0.977131144423)*x[1]**o+(0.698402625747)*x[1]
        arg[2,2,2,1]=(-0.0446180768654)*x[0]**o+(-0.742263614438)*x[0]+(-0.898675367333)*x[1]**o+(-0.260891146746)*x[1]
        arg[2,3,0,0]=(0.819774310005)*x[0]**o+(-0.0677371406664)*x[0]+(0.276747165681)*x[1]**o+(-0.42828783)*x[1]
        arg[2,3,0,1]=(-0.391001671385)*x[0]**o+(0.0558814994728)*x[0]+(-0.624786978129)*x[1]**o+(0.178742280566)*x[1]
        arg[2,3,1,0]=(0.904597103646)*x[0]**o+(-0.825472271707)*x[0]+(0.112221476422)*x[1]**o+(0.269349199527)*x[1]
        arg[2,3,1,1]=(-0.190955697638)*x[0]**o+(0.441698994606)*x[0]+(-0.26229840557)*x[1]**o+(0.148974540187)*x[1]
        arg[2,3,2,0]=(-0.873098622934)*x[0]**o+(-0.624915584376)*x[0]+(-0.151646578512)*x[1]**o+(-0.914878228283)*x[1]
        arg[2,3,2,1]=(-0.712569947477)*x[0]**o+(-0.72679361801)*x[0]+(-0.0946869166491)*x[1]**o+(-0.580512407933)*x[1]
        arg[2,4,0,0]=(-0.655752680768)*x[0]**o+(-0.409544975855)*x[0]+(0.865325438086)*x[1]**o+(-0.227397026937)*x[1]
        arg[2,4,0,1]=(0.670566711209)*x[0]**o+(-0.794343533883)*x[0]+(-0.127116396219)*x[1]**o+(-0.422419866647)*x[1]
        arg[2,4,1,0]=(0.46903441444)*x[0]**o+(0.0689129759715)*x[0]+(-0.549325647826)*x[1]**o+(0.893700020399)*x[1]
        arg[2,4,1,1]=(-0.0099246750652)*x[0]**o+(0.446742615509)*x[0]+(0.900251202513)*x[1]**o+(0.221599931055)*x[1]
        arg[2,4,2,0]=(0.218854380066)*x[0]**o+(0.877849336909)*x[0]+(-0.338211083089)*x[1]**o+(0.746950973063)*x[1]
        arg[2,4,2,1]=(-0.890762146523)*x[0]**o+(0.439653904008)*x[0]+(0.0802890292215)*x[1]**o+(-0.305233739845)*x[1]
        arg[3,0,0,0]=(0.284622209957)*x[0]**o+(0.983284338924)*x[0]+(-0.732588048234)*x[1]**o+(0.654066388109)*x[1]
        arg[3,0,0,1]=(0.379628076244)*x[0]**o+(-0.75362875047)*x[0]+(0.288422119968)*x[1]**o+(0.139197401407)*x[1]
        arg[3,0,1,0]=(0.0594472092467)*x[0]**o+(-0.391581973307)*x[0]+(-0.898996153568)*x[1]**o+(-2.75165483707e-05)*x[1]
        arg[3,0,1,1]=(-0.241252797172)*x[0]**o+(-0.87384982932)*x[0]+(0.811628957112)*x[1]**o+(0.0170539509618)*x[1]
        arg[3,0,2,0]=(0.255617459837)*x[0]**o+(-0.744071491889)*x[0]+(0.745552769828)*x[1]**o+(0.128612266243)*x[1]
        arg[3,0,2,1]=(0.0725082197254)*x[0]**o+(0.779602587374)*x[0]+(-0.751879287668)*x[1]**o+(0.838360641095)*x[1]
        arg[3,1,0,0]=(-0.444520674459)*x[0]**o+(-0.811442143542)*x[0]+(-0.485010478999)*x[1]**o+(0.727600464745)*x[1]
        arg[3,1,0,1]=(0.260061065042)*x[0]**o+(0.932320346965)*x[0]+(0.556377882844)*x[1]**o+(0.284348284894)*x[1]
        arg[3,1,1,0]=(0.561755002522)*x[0]**o+(-0.842983435746)*x[0]+(0.729209244377)*x[1]**o+(-0.254683846974)*x[1]
        arg[3,1,1,1]=(-0.949556399673)*x[0]**o+(-0.0804522142379)*x[0]+(0.066141483426)*x[1]**o+(0.78626145966)*x[1]
        arg[3,1,2,0]=(0.833198651881)*x[0]**o+(-0.297955946153)*x[0]+(-0.561247934789)*x[1]**o+(-0.645124432611)*x[1]
        arg[3,1,2,1]=(-0.641727656043)*x[0]**o+(0.32981201062)*x[0]+(-0.152588766563)*x[1]**o+(-0.289611460761)*x[1]
        arg[3,2,0,0]=(0.825643539598)*x[0]**o+(0.924286336436)*x[0]+(0.413089850944)*x[1]**o+(-0.640198886209)*x[1]
        arg[3,2,0,1]=(0.0855693847128)*x[0]**o+(-0.363881253576)*x[0]+(0.0397854550832)*x[1]**o+(-0.844391996875)*x[1]
        arg[3,2,1,0]=(-0.8248537795)*x[0]**o+(0.153070857979)*x[0]+(0.970009460355)*x[1]**o+(0.431000028243)*x[1]
        arg[3,2,1,1]=(-0.978024209918)*x[0]**o+(-0.527494257814)*x[0]+(-0.814691105217)*x[1]**o+(0.458334868681)*x[1]
        arg[3,2,2,0]=(0.944359157869)*x[0]**o+(-0.868016282191)*x[0]+(0.410849160429)*x[1]**o+(0.480177162607)*x[1]
        arg[3,2,2,1]=(0.109166287134)*x[0]**o+(0.940100154901)*x[0]+(0.630946061805)*x[1]**o+(0.347160020469)*x[1]
        arg[3,3,0,0]=(0.364585502975)*x[0]**o+(0.717282598047)*x[0]+(0.517805310585)*x[1]**o+(0.963452728824)*x[1]
        arg[3,3,0,1]=(0.0373453558524)*x[0]**o+(0.943887402198)*x[0]+(0.983624425071)*x[1]**o+(-0.396242794059)*x[1]
        arg[3,3,1,0]=(-0.563724031351)*x[0]**o+(0.959753988518)*x[0]+(0.118506885299)*x[1]**o+(-0.587748202342)*x[1]
        arg[3,3,1,1]=(0.460419807013)*x[0]**o+(0.237753399499)*x[0]+(0.585589833251)*x[1]**o+(0.935177937667)*x[1]
        arg[3,3,2,0]=(-0.750248086829)*x[0]**o+(-0.561629848994)*x[0]+(-0.919919077536)*x[1]**o+(0.239984179086)*x[1]
        arg[3,3,2,1]=(-0.0969897163659)*x[0]**o+(0.185433022889)*x[0]+(0.733111631837)*x[1]**o+(-0.508147081928)*x[1]
        arg[3,4,0,0]=(0.247054568876)*x[0]**o+(0.721751593201)*x[0]+(-0.684740924498)*x[1]**o+(0.28395202476)*x[1]
        arg[3,4,0,1]=(-0.0949248638736)*x[0]**o+(0.270285273316)*x[0]+(0.850836559464)*x[1]**o+(-0.771085947382)*x[1]
        arg[3,4,1,0]=(-0.965410502362)*x[0]**o+(-0.582448742001)*x[0]+(-0.90307507827)*x[1]**o+(0.914051445745)*x[1]
        arg[3,4,1,1]=(0.443340058617)*x[0]**o+(0.434462120769)*x[0]+(0.522015964301)*x[1]**o+(-0.520184389865)*x[1]
        arg[3,4,2,0]=(-0.961375562593)*x[0]**o+(-0.474630747481)*x[0]+(0.59574636341)*x[1]**o+(0.434190359663)*x[1]
        arg[3,4,2,1]=(0.174154210501)*x[0]**o+(-0.510858596799)*x[0]+(-0.0279700806065)*x[1]**o+(-0.0401612371912)*x[1]
        ref[0,0,0,0]=(-0.0352052809673)/(o+1.)+(0.903863789878)+(0.182956096072)*0.5**o
        ref[0,0,0,1]=(0.0610373691872)/(o+1.)+(-0.306742561603)+(-0.81342189211)*0.5**o
        ref[0,0,1,0]=(0.623943596759)/(o+1.)+(0.647497589315)+(-0.149103531408)*0.5**o
        ref[0,0,1,1]=(0.136600247183)/(o+1.)+(0.925469911903)+(0.74912087771)*0.5**o
        ref[0,0,2,0]=(0.966454801911)/(o+1.)+(0.0631444946098)+(-0.289042694587)*0.5**o
        ref[0,0,2,1]=(-0.156074498516)/(o+1.)+(0.047955363258)+(0.0809737838902)*0.5**o
        ref[0,1,0,0]=(-0.016865109839)/(o+1.)+(-0.404214477389)+(0.638822193145)*0.5**o
        ref[0,1,0,1]=(-0.897448372147)/(o+1.)+(-0.174669987755)+(0.611604473356)*0.5**o
        ref[0,1,1,0]=(-0.229098133178)/(o+1.)+(0.443671911949)+(-0.745407006471)*0.5**o
        ref[0,1,1,1]=(0.481206893907)/(o+1.)+(-0.111741539797)+(0.756528549281)*0.5**o
        ref[0,1,2,0]=(0.367303506039)/(o+1.)+(-0.157755527379)+(0.335544368565)*0.5**o
        ref[0,1,2,1]=(0.182483888026)/(o+1.)+(-0.157073928893)+(0.669933676654)*0.5**o
        ref[0,2,0,0]=(0.587675819052)/(o+1.)+(0.446236706377)+(-0.929163949925)*0.5**o
        ref[0,2,0,1]=(-0.83934259298)/(o+1.)+(-0.344719194607)+(0.639896773528)*0.5**o
        ref[0,2,1,0]=(0.675592435624)/(o+1.)+(0.81769507347)+(-0.399556357179)*0.5**o
        ref[0,2,1,1]=(0.14396936899)/(o+1.)+(-0.454293976313)+(-0.281969675471)*0.5**o
        ref[0,2,2,0]=(0.417557775604)/(o+1.)+(-0.430056060749)+(0.339204154215)*0.5**o
        ref[0,2,2,1]=(-0.0763188160723)/(o+1.)+(0.28785590073)+(-0.655500794874)*0.5**o
        ref[0,3,0,0]=(-0.0533973517656)/(o+1.)+(-0.619586928986)+(-0.483746115201)*0.5**o
        ref[0,3,0,1]=(-0.607238533539)/(o+1.)+(-0.899573610196)+(0.446810992304)*0.5**o
        ref[0,3,1,0]=(0.930589323947)/(o+1.)+(0.127928801455)+(-0.0235907627093)*0.5**o
        ref[0,3,1,1]=(0.237445053125)/(o+1.)+(0.503964752361)+(0.486342062258)*0.5**o
        ref[0,3,2,0]=(0.180505264557)/(o+1.)+(0.629926679618)+(-0.581025309937)*0.5**o
        ref[0,3,2,1]=(0.915270822508)/(o+1.)+(0.0599229206396)+(-0.523356041281)*0.5**o
        ref[0,4,0,0]=(-0.449731522195)/(o+1.)+(-0.0207634106763)+(-0.584782530121)*0.5**o
        ref[0,4,0,1]=(0.0374933582543)/(o+1.)+(0.175584823513)+(-0.0724106581167)*0.5**o
        ref[0,4,1,0]=(0.815864788848)/(o+1.)+(-0.4782741099)+(0.0460360663465)*0.5**o
        ref[0,4,1,1]=(-0.822961904166)/(o+1.)+(0.4384056737)+(-0.978584425712)*0.5**o
        ref[0,4,2,0]=(0.721009228711)/(o+1.)+(0.396658231667)+(-0.267227111945)*0.5**o
        ref[0,4,2,1]=(0.94281427253)/(o+1.)+(-0.888798950857)+(-0.141372777283)*0.5**o
        ref[1,0,0,0]=(0.583083015744)/(o+1.)+(0.41011820901)+(-0.721911758033)*0.5**o
        ref[1,0,0,1]=(-0.656523378462)/(o+1.)+(0.129802155034)+(0.929636966638)*0.5**o
        ref[1,0,1,0]=(-0.562311637909)/(o+1.)+(0.360576537299)+(0.594847872723)*0.5**o
        ref[1,0,1,1]=(-0.265309908189)/(o+1.)+(0.147321654532)+(0.529274819592)*0.5**o
        ref[1,0,2,0]=(-0.0509405210126)/(o+1.)+(0.0288477105933)+(0.2321959905)*0.5**o
        ref[1,0,2,1]=(0.817375877796)/(o+1.)+(-0.00495015411903)+(0.0811824999287)*0.5**o
        ref[1,1,0,0]=(0.15912570069)/(o+1.)+(0.270179064041)+(0.418525096774)*0.5**o
        ref[1,1,0,1]=(-0.299421222048)/(o+1.)+(-0.765918515799)+(0.784138248952)*0.5**o
        ref[1,1,1,0]=(0.935180253062)/(o+1.)+(0.0275127156677)+(-0.911449050332)*0.5**o
        ref[1,1,1,1]=(0.827581865806)/(o+1.)+(0.0374007680845)+(-0.378301315644)*0.5**o
        ref[1,1,2,0]=(-0.580086073626)/(o+1.)+(0.52473567096)+(-0.882543004053)*0.5**o
        ref[1,1,2,1]=(0.320343551712)/(o+1.)+(-0.00386603582212)+(-0.116197074531)*0.5**o
        ref[1,2,0,0]=(0.707766015783)/(o+1.)+(-0.273067983929)+(-0.870347384636)*0.5**o
        ref[1,2,0,1]=(0.825489444837)/(o+1.)+(-0.586560764519)+(-0.219872666014)*0.5**o
        ref[1,2,1,0]=(0.502736178959)/(o+1.)+(0.981161874983)+(-0.963149287055)*0.5**o
        ref[1,2,1,1]=(-0.182719660538)/(o+1.)+(-0.472716904044)+(-0.586350249194)*0.5**o
        ref[1,2,2,0]=(-0.304129806418)/(o+1.)+(-0.192408182425)+(-0.423818475166)*0.5**o
        ref[1,2,2,1]=(0.438755721196)/(o+1.)+(-0.360831125262)+(0.901384945886)*0.5**o
        ref[1,3,0,0]=(0.427243495756)/(o+1.)+(0.534790142987)+(-0.797968602257)*0.5**o
        ref[1,3,0,1]=(-0.943037028688)/(o+1.)+(0.711855627815)+(-0.696126104381)*0.5**o
        ref[1,3,1,0]=(-0.772965740874)/(o+1.)+(-0.689424136389)+(-0.69028368426)*0.5**o
        ref[1,3,1,1]=(-0.250549859157)/(o+1.)+(-0.749654818761)+(-0.534347181294)*0.5**o
        ref[1,3,2,0]=(-0.516514308221)/(o+1.)+(-0.264725096778)+(-0.827981962115)*0.5**o
        ref[1,3,2,1]=(0.806201346091)/(o+1.)+(0.860793480898)+(0.864701869194)*0.5**o
        ref[1,4,0,0]=(-0.288389617381)/(o+1.)+(0.559861504432)+(-0.589044559231)*0.5**o
        ref[1,4,0,1]=(-0.696145763418)/(o+1.)+(0.120898852932)+(0.814971550967)*0.5**o
        ref[1,4,1,0]=(-0.507159320061)/(o+1.)+(-0.520517591688)+(0.760190450826)*0.5**o
        ref[1,4,1,1]=(0.0153574982122)/(o+1.)+(0.656425254312)+(-0.0416603697257)*0.5**o
        ref[1,4,2,0]=(-0.0991159921294)/(o+1.)+(0.551683309807)+(0.105929120872)*0.5**o
        ref[1,4,2,1]=(0.377083955928)/(o+1.)+(0.169719758465)+(-0.503911541225)*0.5**o
        ref[2,0,0,0]=(0.196766730386)/(o+1.)+(-0.472915269429)+(0.953733725947)*0.5**o
        ref[2,0,0,1]=(-0.600347310418)/(o+1.)+(0.335114869806)+(0.491469820597)*0.5**o
        ref[2,0,1,0]=(0.783037278839)/(o+1.)+(0.250103346412)+(0.615118477478)*0.5**o
        ref[2,0,1,1]=(-0.381389710969)/(o+1.)+(0.708103944971)+(-0.806125471265)*0.5**o
        ref[2,0,2,0]=(-0.13737609858)/(o+1.)+(0.411253603929)+(-0.831727679795)*0.5**o
        ref[2,0,2,1]=(0.159758297984)/(o+1.)+(-0.410942771953)+(-0.805843206194)*0.5**o
        ref[2,1,0,0]=(0.325741074892)/(o+1.)+(-0.153818387566)+(0.589135530105)*0.5**o
        ref[2,1,0,1]=(0.0171600098883)/(o+1.)+(0.185369805491)+(0.106049062774)*0.5**o
        ref[2,1,1,0]=(-0.576165158737)/(o+1.)+(-0.0674265758955)+(-0.181511670026)*0.5**o
        ref[2,1,1,1]=(-0.424093531654)/(o+1.)+(0.0229697818029)+(-0.709695453728)*0.5**o
        ref[2,1,2,0]=(0.640141428017)/(o+1.)+(-0.803639398084)+(0.60069280023)*0.5**o
        ref[2,1,2,1]=(0.144061490066)/(o+1.)+(0.677215099246)+(0.896964594562)*0.5**o
        ref[2,2,0,0]=(0.199304002742)/(o+1.)+(0.160270347534)+(0.447633246258)*0.5**o
        ref[2,2,0,1]=(-0.328047392082)/(o+1.)+(0.423339325745)+(-0.683614706554)*0.5**o
        ref[2,2,1,0]=(0.449220001922)/(o+1.)+(0.0252142810523)+(0.22474720984)*0.5**o
        ref[2,2,1,1]=(-0.794933934805)/(o+1.)+(-0.153196600671)+(-0.266419887395)*0.5**o
        ref[2,2,2,0]=(-0.977131144423)/(o+1.)+(0.291140460744)+(0.949852186215)*0.5**o
        ref[2,2,2,1]=(-0.898675367333)/(o+1.)+(-0.501577380592)+(-0.0446180768654)*0.5**o
        ref[2,3,0,0]=(0.276747165681)/(o+1.)+(-0.248012485333)+(0.819774310005)*0.5**o
        ref[2,3,0,1]=(-0.624786978129)/(o+1.)+(0.11731189002)+(-0.391001671385)*0.5**o
        ref[2,3,1,0]=(0.112221476422)/(o+1.)+(-0.27806153609)+(0.904597103646)*0.5**o
        ref[2,3,1,1]=(-0.26229840557)/(o+1.)+(0.295336767397)+(-0.190955697638)*0.5**o
        ref[2,3,2,0]=(-0.151646578512)/(o+1.)+(-0.76989690633)+(-0.873098622934)*0.5**o
        ref[2,3,2,1]=(-0.0946869166491)/(o+1.)+(-0.653653012971)+(-0.712569947477)*0.5**o
        ref[2,4,0,0]=(0.865325438086)/(o+1.)+(-0.318471001396)+(-0.655752680768)*0.5**o
        ref[2,4,0,1]=(-0.127116396219)/(o+1.)+(-0.608381700265)+(0.670566711209)*0.5**o
        ref[2,4,1,0]=(-0.549325647826)/(o+1.)+(0.481306498185)+(0.46903441444)*0.5**o
        ref[2,4,1,1]=(0.900251202513)/(o+1.)+(0.334171273282)+(-0.0099246750652)*0.5**o
        ref[2,4,2,0]=(-0.338211083089)/(o+1.)+(0.812400154986)+(0.218854380066)*0.5**o
        ref[2,4,2,1]=(0.0802890292215)/(o+1.)+(0.0672100820814)+(-0.890762146523)*0.5**o
        ref[3,0,0,0]=(-0.732588048234)/(o+1.)+(0.818675363516)+(0.284622209957)*0.5**o
        ref[3,0,0,1]=(0.288422119968)/(o+1.)+(-0.307215674532)+(0.379628076244)*0.5**o
        ref[3,0,1,0]=(-0.898996153568)/(o+1.)+(-0.195804744928)+(0.0594472092467)*0.5**o
        ref[3,0,1,1]=(0.811628957112)/(o+1.)+(-0.428397939179)+(-0.241252797172)*0.5**o
        ref[3,0,2,0]=(0.745552769828)/(o+1.)+(-0.307729612823)+(0.255617459837)*0.5**o
        ref[3,0,2,1]=(-0.751879287668)/(o+1.)+(0.808981614234)+(0.0725082197254)*0.5**o
        ref[3,1,0,0]=(-0.485010478999)/(o+1.)+(-0.0419208393982)+(-0.444520674459)*0.5**o
        ref[3,1,0,1]=(0.556377882844)/(o+1.)+(0.60833431593)+(0.260061065042)*0.5**o
        ref[3,1,1,0]=(0.729209244377)/(o+1.)+(-0.54883364136)+(0.561755002522)*0.5**o
        ref[3,1,1,1]=(0.066141483426)/(o+1.)+(0.352904622711)+(-0.949556399673)*0.5**o
        ref[3,1,2,0]=(-0.561247934789)/(o+1.)+(-0.471540189382)+(0.833198651881)*0.5**o
        ref[3,1,2,1]=(-0.152588766563)/(o+1.)+(0.0201002749295)+(-0.641727656043)*0.5**o
        ref[3,2,0,0]=(0.413089850944)/(o+1.)+(0.142043725114)+(0.825643539598)*0.5**o
        ref[3,2,0,1]=(0.0397854550832)/(o+1.)+(-0.604136625225)+(0.0855693847128)*0.5**o
        ref[3,2,1,0]=(0.970009460355)/(o+1.)+(0.292035443111)+(-0.8248537795)*0.5**o
        ref[3,2,1,1]=(-0.814691105217)/(o+1.)+(-0.0345796945668)+(-0.978024209918)*0.5**o
        ref[3,2,2,0]=(0.410849160429)/(o+1.)+(-0.193919559792)+(0.944359157869)*0.5**o
        ref[3,2,2,1]=(0.630946061805)/(o+1.)+(0.643630087685)+(0.109166287134)*0.5**o
        ref[3,3,0,0]=(0.517805310585)/(o+1.)+(0.840367663435)+(0.364585502975)*0.5**o
        ref[3,3,0,1]=(0.983624425071)/(o+1.)+(0.273822304069)+(0.0373453558524)*0.5**o
        ref[3,3,1,0]=(0.118506885299)/(o+1.)+(0.186002893088)+(-0.563724031351)*0.5**o
        ref[3,3,1,1]=(0.585589833251)/(o+1.)+(0.586465668583)+(0.460419807013)*0.5**o
        ref[3,3,2,0]=(-0.919919077536)/(o+1.)+(-0.160822834954)+(-0.750248086829)*0.5**o
        ref[3,3,2,1]=(0.733111631837)/(o+1.)+(-0.161357029519)+(-0.0969897163659)*0.5**o
        ref[3,4,0,0]=(-0.684740924498)/(o+1.)+(0.50285180898)+(0.247054568876)*0.5**o
        ref[3,4,0,1]=(0.850836559464)/(o+1.)+(-0.250400337033)+(-0.0949248638736)*0.5**o
        ref[3,4,1,0]=(-0.90307507827)/(o+1.)+(0.165801351872)+(-0.965410502362)*0.5**o
        ref[3,4,1,1]=(0.522015964301)/(o+1.)+(-0.0428611345479)+(0.443340058617)*0.5**o
        ref[3,4,2,0]=(0.59574636341)/(o+1.)+(-0.0202201939092)+(-0.961375562593)*0.5**o
        ref[3,4,2,1]=(-0.0279700806065)/(o+1.)+(-0.275509916995)+(0.174154210501)*0.5**o
      else:
        arg[0,0,0,0]=(-0.793251694335)*x[0]**o+(0.616970271582)*x[0]+(-0.805609243189)*x[1]**o+(0.949515692474)*x[1]+(0.484069738431)*x[2]**o+(-0.0407533911672)*x[2]
        arg[0,0,0,1]=(-0.624206604764)*x[0]**o+(-0.992961148702)*x[0]+(-0.737741763879)*x[1]**o+(0.161544434398)*x[1]+(0.47292947772)*x[2]**o+(0.525538846945)*x[2]
        arg[0,0,1,0]=(0.115999561481)*x[0]**o+(-0.612694096044)*x[0]+(0.625539443386)*x[1]**o+(0.857353715882)*x[1]+(-0.0811256709155)*x[2]**o+(0.183474706394)*x[2]
        arg[0,0,1,1]=(-0.972466831671)*x[0]**o+(0.262980887655)*x[0]+(0.920924576059)*x[1]**o+(0.983517658667)*x[1]+(0.579443973505)*x[2]**o+(0.101139612823)*x[2]
        arg[0,0,2,0]=(0.708180260094)*x[0]**o+(-0.0950913766285)*x[0]+(0.361602191798)*x[1]**o+(-0.880757848201)*x[1]+(0.326272205058)*x[2]**o+(-0.997155694972)*x[2]
        arg[0,0,2,1]=(0.0282615573045)*x[0]**o+(-0.317851994728)*x[0]+(0.449330147388)*x[1]**o+(-0.156946318079)*x[1]+(0.344196894245)*x[2]**o+(-0.045082290904)*x[2]
        arg[0,1,0,0]=(0.231621820235)*x[0]**o+(0.0985804817846)*x[0]+(-0.343623333022)*x[1]**o+(0.115043475441)*x[1]+(-0.910742772023)*x[2]**o+(-0.663592068553)*x[2]
        arg[0,1,0,1]=(0.114056801155)*x[0]**o+(-0.463711315494)*x[0]+(0.380521147429)*x[1]**o+(0.294823684238)*x[1]+(0.510501672453)*x[2]**o+(-0.161234251875)*x[2]
        arg[0,1,1,0]=(-0.973115195017)*x[0]**o+(-0.0133290715609)*x[0]+(-0.642072931588)*x[1]**o+(-0.580655605735)*x[1]+(-0.191371539061)*x[2]**o+(-0.0513051439064)*x[2]
        arg[0,1,1,1]=(-0.304758579369)*x[0]**o+(-0.321904612871)*x[0]+(-0.389525877547)*x[1]**o+(-0.0602638359252)*x[1]+(0.199726168523)*x[2]**o+(-0.10313327614)*x[2]
        arg[0,1,2,0]=(0.0923844967157)*x[0]**o+(-0.369319307184)*x[0]+(-0.614761123101)*x[1]**o+(0.78817430449)*x[1]+(-0.783809995839)*x[2]**o+(-0.701522582246)*x[2]
        arg[0,1,2,1]=(0.793568921835)*x[0]**o+(0.352244892726)*x[0]+(0.185546719529)*x[1]**o+(0.145969792058)*x[1]+(0.85147546253)*x[2]**o+(-0.862069843156)*x[2]
        arg[0,2,0,0]=(0.570673248048)*x[0]**o+(0.607924942767)*x[0]+(-0.353952625913)*x[1]**o+(0.111796842318)*x[1]+(-0.765670542263)*x[2]**o+(0.589532724922)*x[2]
        arg[0,2,0,1]=(0.299061155215)*x[0]**o+(-0.542978954882)*x[0]+(0.0946151623892)*x[1]**o+(0.310266302534)*x[1]+(-0.00880229885823)*x[2]**o+(0.585508874849)*x[2]
        arg[0,2,1,0]=(0.694725244159)*x[0]**o+(0.530004132186)*x[0]+(-0.32588478629)*x[1]**o+(0.357278329148)*x[1]+(-0.93877612203)*x[2]**o+(0.34973013935)*x[2]
        arg[0,2,1,1]=(-0.83896527178)*x[0]**o+(0.487231460025)*x[0]+(0.493952010418)*x[1]**o+(0.247442509692)*x[1]+(-0.312456515075)*x[2]**o+(0.90248200924)*x[2]
        arg[0,2,2,0]=(0.2228052704)*x[0]**o+(0.901373696962)*x[0]+(0.149826948427)*x[1]**o+(0.234013159005)*x[1]+(0.717685535862)*x[2]**o+(-0.42865545543)*x[2]
        arg[0,2,2,1]=(0.63141216243)*x[0]**o+(0.448446314659)*x[0]+(0.379625538779)*x[1]**o+(-0.807448154272)*x[1]+(0.238724835223)*x[2]**o+(-0.478216911196)*x[2]
        arg[0,3,0,0]=(0.00698107166168)*x[0]**o+(-0.727207190747)*x[0]+(-0.78731647832)*x[1]**o+(0.885371753974)*x[1]+(-0.0146821479126)*x[2]**o+(-0.763310617205)*x[2]
        arg[0,3,0,1]=(0.243423742325)*x[0]**o+(0.0587322175837)*x[0]+(-0.453213790216)*x[1]**o+(0.861270481703)*x[1]+(0.867332526083)*x[2]**o+(-0.713187214416)*x[2]
        arg[0,3,1,0]=(-0.106629554393)*x[0]**o+(0.723693136654)*x[0]+(-0.0276604293602)*x[1]**o+(0.339456904686)*x[1]+(-0.102075459215)*x[2]**o+(-0.0348969356596)*x[2]
        arg[0,3,1,1]=(0.375143733111)*x[0]**o+(-0.490781234413)*x[0]+(0.168903339622)*x[1]**o+(0.594080859808)*x[1]+(-0.0802336915361)*x[2]**o+(0.472348979787)*x[2]
        arg[0,3,2,0]=(0.955685455438)*x[0]**o+(0.105142628533)*x[0]+(-0.446384800971)*x[1]**o+(-0.905998305229)*x[1]+(-0.30011637186)*x[2]**o+(0.0218195326037)*x[2]
        arg[0,3,2,1]=(0.353654445481)*x[0]**o+(-0.0477024594709)*x[0]+(-0.495075817497)*x[1]**o+(-0.425091398447)*x[1]+(-0.571207600446)*x[2]**o+(0.177935012163)*x[2]
        arg[0,4,0,0]=(0.974759684151)*x[0]**o+(-0.876975976179)*x[0]+(-0.472336406836)*x[1]**o+(0.991410874408)*x[1]+(0.297823444758)*x[2]**o+(-0.161436760659)*x[2]
        arg[0,4,0,1]=(-0.0799922241539)*x[0]**o+(0.975369691404)*x[0]+(0.302361880712)*x[1]**o+(0.171779641966)*x[1]+(0.878428630413)*x[2]**o+(0.814497247319)*x[2]
        arg[0,4,1,0]=(0.532633280629)*x[0]**o+(0.902436316468)*x[0]+(-0.676910177358)*x[1]**o+(-0.986029752926)*x[1]+(-0.519990783496)*x[2]**o+(0.430468215617)*x[2]
        arg[0,4,1,1]=(-0.353243658293)*x[0]**o+(-0.908938062877)*x[0]+(0.922933758125)*x[1]**o+(0.00500161721147)*x[1]+(0.189861558609)*x[2]**o+(-0.623307494462)*x[2]
        arg[0,4,2,0]=(-0.720840869801)*x[0]**o+(0.0349736577597)*x[0]+(-0.516608060915)*x[1]**o+(0.688756843577)*x[1]+(-0.302494303699)*x[2]**o+(-0.0703783111937)*x[2]
        arg[0,4,2,1]=(-0.179073400214)*x[0]**o+(0.128485402427)*x[0]+(-0.726789308786)*x[1]**o+(-0.691163482057)*x[1]+(0.329009412577)*x[2]**o+(-0.79293394396)*x[2]
        arg[1,0,0,0]=(-0.384011553333)*x[0]**o+(0.285271264922)*x[0]+(0.512105930181)*x[1]**o+(0.190321101872)*x[1]+(-0.509334982029)*x[2]**o+(0.128097742502)*x[2]
        arg[1,0,0,1]=(-0.808211570649)*x[0]**o+(-0.730450956462)*x[0]+(0.92309757237)*x[1]**o+(-0.703749565272)*x[1]+(0.301129716544)*x[2]**o+(0.244379489105)*x[2]
        arg[1,0,1,0]=(0.0404274840294)*x[0]**o+(-0.483903392253)*x[0]+(0.73680959705)*x[1]**o+(0.190850758522)*x[1]+(0.133136432605)*x[2]**o+(0.910055186498)*x[2]
        arg[1,0,1,1]=(0.302394139901)*x[0]**o+(0.802524687002)*x[0]+(0.970871728408)*x[1]**o+(0.207348092291)*x[1]+(0.666310912781)*x[2]**o+(0.510392805767)*x[2]
        arg[1,0,2,0]=(-0.981981997649)*x[0]**o+(-0.465269233707)*x[0]+(0.916146418639)*x[1]**o+(0.143602438774)*x[1]+(-0.947874263512)*x[2]**o+(0.0840394012229)*x[2]
        arg[1,0,2,1]=(-0.237479805943)*x[0]**o+(-0.89993757244)*x[0]+(-0.759650487895)*x[1]**o+(-0.807139164373)*x[1]+(-0.377055204014)*x[2]**o+(-0.57656201026)*x[2]
        arg[1,1,0,0]=(-0.543719871098)*x[0]**o+(-0.726308160246)*x[0]+(0.821244616742)*x[1]**o+(0.718112961487)*x[1]+(-0.655445553192)*x[2]**o+(-0.599104608292)*x[2]
        arg[1,1,0,1]=(-0.2624625471)*x[0]**o+(0.85250071151)*x[0]+(-0.665066081561)*x[1]**o+(-0.566502138196)*x[1]+(-0.572858653628)*x[2]**o+(-0.862890633323)*x[2]
        arg[1,1,1,0]=(0.88003513115)*x[0]**o+(-0.0150959972442)*x[0]+(-0.698880737368)*x[1]**o+(-0.236909810427)*x[1]+(-0.721315930706)*x[2]**o+(-0.219894530829)*x[2]
        arg[1,1,1,1]=(0.177746615088)*x[0]**o+(0.205790814183)*x[0]+(-0.216021190604)*x[1]**o+(0.384079010991)*x[1]+(0.61634008905)*x[2]**o+(0.940900776108)*x[2]
        arg[1,1,2,0]=(0.889070121473)*x[0]**o+(0.68131533527)*x[0]+(0.709207182081)*x[1]**o+(-0.765184991252)*x[1]+(0.506328018952)*x[2]**o+(-0.46041174473)*x[2]
        arg[1,1,2,1]=(0.951736215954)*x[0]**o+(-0.147499702642)*x[0]+(0.60348358384)*x[1]**o+(0.777404433567)*x[1]+(-0.482441452679)*x[2]**o+(0.714835256434)*x[2]
        arg[1,2,0,0]=(-0.954701036263)*x[0]**o+(-0.360901887475)*x[0]+(-0.644285028623)*x[1]**o+(0.169271524632)*x[1]+(-0.687047203668)*x[2]**o+(0.205115312108)*x[2]
        arg[1,2,0,1]=(0.256489282842)*x[0]**o+(0.17406448362)*x[0]+(0.97500243816)*x[1]**o+(0.614577997602)*x[1]+(0.0537536804583)*x[2]**o+(-0.211963274032)*x[2]
        arg[1,2,1,0]=(-0.94015914517)*x[0]**o+(0.169982490736)*x[0]+(0.482085495724)*x[1]**o+(-0.556994404153)*x[1]+(-0.809311296883)*x[2]**o+(0.719907503024)*x[2]
        arg[1,2,1,1]=(0.367156421931)*x[0]**o+(0.489648825153)*x[0]+(-0.0398225517816)*x[1]**o+(-0.495726296032)*x[1]+(0.359374061053)*x[2]**o+(0.231991952482)*x[2]
        arg[1,2,2,0]=(-0.812516694781)*x[0]**o+(-0.631181257391)*x[0]+(-0.723784471687)*x[1]**o+(0.178976877342)*x[1]+(-0.629293758013)*x[2]**o+(-0.313897724608)*x[2]
        arg[1,2,2,1]=(-0.497461944186)*x[0]**o+(0.268998525103)*x[0]+(-0.478017492026)*x[1]**o+(0.026395792841)*x[1]+(0.581256386137)*x[2]**o+(-0.476961644324)*x[2]
        arg[1,3,0,0]=(-0.227077550639)*x[0]**o+(0.400407424161)*x[0]+(-0.53529812303)*x[1]**o+(-0.418594299853)*x[1]+(-0.195190916346)*x[2]**o+(-0.624284483774)*x[2]
        arg[1,3,0,1]=(0.822091021645)*x[0]**o+(-0.31133140026)*x[0]+(0.0665130965049)*x[1]**o+(0.596697216047)*x[1]+(0.466195828646)*x[2]**o+(-0.477988391299)*x[2]
        arg[1,3,1,0]=(-0.0341577310045)*x[0]**o+(-0.77936644392)*x[0]+(0.954811382784)*x[1]**o+(0.837374830631)*x[1]+(0.738992234768)*x[2]**o+(-0.501187788985)*x[2]
        arg[1,3,1,1]=(-0.102867224088)*x[0]**o+(0.00829793194765)*x[0]+(0.081790556957)*x[1]**o+(-0.779351210472)*x[1]+(-0.735321115716)*x[2]**o+(0.126919542418)*x[2]
        arg[1,3,2,0]=(0.556301076488)*x[0]**o+(0.0923434629749)*x[0]+(-0.191048735713)*x[1]**o+(-0.218833028687)*x[1]+(-0.088867516146)*x[2]**o+(-0.247214200389)*x[2]
        arg[1,3,2,1]=(0.888436531887)*x[0]**o+(-0.79225495914)*x[0]+(0.0154788262219)*x[1]**o+(-0.210622890864)*x[1]+(0.782629134632)*x[2]**o+(0.350772185791)*x[2]
        arg[1,4,0,0]=(0.651898073685)*x[0]**o+(-0.84098616689)*x[0]+(0.674088959443)*x[1]**o+(-0.658824952898)*x[1]+(-0.423551541558)*x[2]**o+(0.985098530356)*x[2]
        arg[1,4,0,1]=(-0.149187282052)*x[0]**o+(0.0880370482988)*x[0]+(0.780809276595)*x[1]**o+(-0.610833366472)*x[1]+(0.794946384304)*x[2]**o+(-0.817031796429)*x[2]
        arg[1,4,1,0]=(-0.285938055108)*x[0]**o+(-0.106165154734)*x[0]+(-0.948376845189)*x[1]**o+(0.874607682056)*x[1]+(-0.100051825909)*x[2]**o+(-0.483640835026)*x[2]
        arg[1,4,1,1]=(-0.34015136472)*x[0]**o+(0.991331180352)*x[0]+(-0.925796518928)*x[1]**o+(-0.872775691287)*x[1]+(0.953898056819)*x[2]**o+(0.985025365955)*x[2]
        arg[1,4,2,0]=(-0.464687865356)*x[0]**o+(-0.439271850879)*x[0]+(0.302716649876)*x[1]**o+(0.235186289965)*x[1]+(0.672062761925)*x[2]**o+(-0.793689262392)*x[2]
        arg[1,4,2,1]=(-0.777967458564)*x[0]**o+(0.0565891925223)*x[0]+(-0.27539733623)*x[1]**o+(0.252329868046)*x[1]+(-0.269529792944)*x[2]**o+(-0.953051723483)*x[2]
        arg[2,0,0,0]=(0.665321754689)*x[0]**o+(0.00639681939987)*x[0]+(0.262193891905)*x[1]**o+(-0.312800735151)*x[1]+(0.8403244871)*x[2]**o+(-0.607084328204)*x[2]
        arg[2,0,0,1]=(0.876435018569)*x[0]**o+(-0.388250784543)*x[0]+(-0.799866349878)*x[1]**o+(0.114896594902)*x[1]+(-0.438064877186)*x[2]**o+(-0.458871344685)*x[2]
        arg[2,0,1,0]=(-0.344552466761)*x[0]**o+(0.967217640605)*x[0]+(-0.00183230845245)*x[1]**o+(-0.861206591247)*x[1]+(0.795588257851)*x[2]**o+(-0.333607767932)*x[2]
        arg[2,0,1,1]=(0.841069648591)*x[0]**o+(0.580162460873)*x[0]+(-0.0277035983747)*x[1]**o+(-0.404616536792)*x[1]+(-0.403195438454)*x[2]**o+(-0.202877947947)*x[2]
        arg[2,0,2,0]=(0.192691357169)*x[0]**o+(-0.519968251714)*x[0]+(-0.575397578091)*x[1]**o+(-0.029378989829)*x[1]+(-0.958861195684)*x[2]**o+(0.848705509543)*x[2]
        arg[2,0,2,1]=(0.556616365284)*x[0]**o+(0.679993071506)*x[0]+(-0.800652717274)*x[1]**o+(0.979182847258)*x[1]+(-0.501953582042)*x[2]**o+(0.127003985335)*x[2]
        arg[2,1,0,0]=(0.947750123445)*x[0]**o+(-0.0350691553359)*x[0]+(0.0650485999092)*x[1]**o+(-0.131216537555)*x[1]+(0.888782559831)*x[2]**o+(0.0338703818654)*x[2]
        arg[2,1,0,1]=(0.962918564559)*x[0]**o+(-0.119367421389)*x[0]+(0.43003549616)*x[1]**o+(-0.097883763913)*x[1]+(0.230422206305)*x[2]**o+(0.187273479711)*x[2]
        arg[2,1,1,0]=(-0.549909211973)*x[0]**o+(0.263443098451)*x[0]+(-0.476703606205)*x[1]**o+(0.216159301182)*x[1]+(-0.561829938323)*x[2]**o+(0.943016557203)*x[2]
        arg[2,1,1,1]=(0.382867749635)*x[0]**o+(0.404317006935)*x[0]+(0.786117847428)*x[1]**o+(-0.80719258612)*x[1]+(0.471144889777)*x[2]**o+(0.6478813349)*x[2]
        arg[2,1,2,0]=(0.355742024596)*x[0]**o+(-0.728812391059)*x[0]+(0.766039604734)*x[1]**o+(-0.532629598363)*x[1]+(-0.518284303077)*x[2]**o+(-0.972348067947)*x[2]
        arg[2,1,2,1]=(-0.882334243081)*x[0]**o+(-0.0940633993231)*x[0]+(-0.00166643832426)*x[1]**o+(-0.971243725495)*x[1]+(-0.0422614083707)*x[2]**o+(-0.547881520447)*x[2]
        arg[2,2,0,0]=(0.372164678051)*x[0]**o+(-0.0644361811234)*x[0]+(0.886078780084)*x[1]**o+(-0.459054628604)*x[1]+(-0.983880547097)*x[2]**o+(-0.669749873181)*x[2]
        arg[2,2,0,1]=(0.126650330827)*x[0]**o+(0.375111974063)*x[0]+(-0.759797654917)*x[1]**o+(-0.663216091322)*x[1]+(-0.683191373782)*x[2]**o+(-0.607048449469)*x[2]
        arg[2,2,1,0]=(-0.805920040628)*x[0]**o+(-0.0144685877057)*x[0]+(0.00875222056291)*x[1]**o+(-0.110000267929)*x[1]+(-0.478201664476)*x[2]**o+(-0.0537439708493)*x[2]
        arg[2,2,1,1]=(-0.550624517803)*x[0]**o+(0.570407090431)*x[0]+(0.661026826148)*x[1]**o+(0.905189813242)*x[1]+(0.00707531661352)*x[2]**o+(0.711072343064)*x[2]
        arg[2,2,2,0]=(-0.644276976064)*x[0]**o+(-0.151665972531)*x[0]+(-0.645369927259)*x[1]**o+(0.915967227715)*x[1]+(-0.211750817665)*x[2]**o+(-0.596646212118)*x[2]
        arg[2,2,2,1]=(0.564688999107)*x[0]**o+(0.635094410913)*x[0]+(0.281863287848)*x[1]**o+(-0.927529659408)*x[1]+(0.581071087484)*x[2]**o+(0.164399277305)*x[2]
        arg[2,3,0,0]=(0.64754765253)*x[0]**o+(0.176693707964)*x[0]+(0.172801737347)*x[1]**o+(-0.769408746261)*x[1]+(-0.176026486601)*x[2]**o+(0.187304474822)*x[2]
        arg[2,3,0,1]=(0.365315558093)*x[0]**o+(0.888853264239)*x[0]+(0.0541759434362)*x[1]**o+(0.241489226711)*x[1]+(0.441822201832)*x[2]**o+(0.075541840076)*x[2]
        arg[2,3,1,0]=(-0.984940830356)*x[0]**o+(0.521596714942)*x[0]+(-0.933671223761)*x[1]**o+(-0.132907654205)*x[1]+(-0.225804727932)*x[2]**o+(0.987334445626)*x[2]
        arg[2,3,1,1]=(-0.785516613303)*x[0]**o+(0.0324126413814)*x[0]+(-0.084233004699)*x[1]**o+(0.45131442059)*x[1]+(0.36498759129)*x[2]**o+(0.417036982561)*x[2]
        arg[2,3,2,0]=(-0.310575473358)*x[0]**o+(-0.769219454218)*x[0]+(-0.263431454577)*x[1]**o+(0.474143448264)*x[1]+(-0.490633240671)*x[2]**o+(0.435609988849)*x[2]
        arg[2,3,2,1]=(0.230960030802)*x[0]**o+(-0.480593443732)*x[0]+(0.547888534735)*x[1]**o+(-0.818842709599)*x[1]+(-0.857141410253)*x[2]**o+(-0.287898487847)*x[2]
        arg[2,4,0,0]=(-0.811008715357)*x[0]**o+(-0.608505768579)*x[0]+(-0.117642307757)*x[1]**o+(-0.658229045784)*x[1]+(0.246542958193)*x[2]**o+(-0.42783248265)*x[2]
        arg[2,4,0,1]=(0.817994599948)*x[0]**o+(0.394931907893)*x[0]+(-0.273993145403)*x[1]**o+(0.9572397013)*x[1]+(0.101025833331)*x[2]**o+(-0.925874548613)*x[2]
        arg[2,4,1,0]=(-0.626152000269)*x[0]**o+(0.821322294557)*x[0]+(0.543761841392)*x[1]**o+(0.112341361443)*x[1]+(-0.395657376418)*x[2]**o+(0.815481187683)*x[2]
        arg[2,4,1,1]=(0.97134483493)*x[0]**o+(0.303341541402)*x[0]+(-0.270102237344)*x[1]**o+(-0.982452271352)*x[1]+(0.435568243011)*x[2]**o+(-0.200051496445)*x[2]
        arg[2,4,2,0]=(-0.827528679064)*x[0]**o+(-0.559447627475)*x[0]+(-0.867397899368)*x[1]**o+(0.915539157819)*x[1]+(0.370118757042)*x[2]**o+(0.578366179408)*x[2]
        arg[2,4,2,1]=(0.685882920875)*x[0]**o+(0.830016468585)*x[0]+(-0.603607199824)*x[1]**o+(0.888619939961)*x[1]+(-0.264173896988)*x[2]**o+(-0.588065135532)*x[2]
        arg[3,0,0,0]=(0.480000662344)*x[0]**o+(-0.240072007989)*x[0]+(0.252669294205)*x[1]**o+(-0.657466732916)*x[1]+(0.945149322572)*x[2]**o+(0.340592627149)*x[2]
        arg[3,0,0,1]=(-0.819956304014)*x[0]**o+(0.910408628621)*x[0]+(-0.875714560314)*x[1]**o+(0.653323198674)*x[1]+(0.910127896386)*x[2]**o+(0.21382952806)*x[2]
        arg[3,0,1,0]=(0.736655857249)*x[0]**o+(-0.982636511328)*x[0]+(0.747965385081)*x[1]**o+(-0.952056699676)*x[1]+(0.143168283422)*x[2]**o+(0.891386576579)*x[2]
        arg[3,0,1,1]=(0.615723453617)*x[0]**o+(0.57376968582)*x[0]+(0.766802940979)*x[1]**o+(-0.294849095727)*x[1]+(-0.26718123792)*x[2]**o+(-0.979175148343)*x[2]
        arg[3,0,2,0]=(-0.299975789693)*x[0]**o+(-0.941879795797)*x[0]+(-0.332633604206)*x[1]**o+(0.887589360228)*x[1]+(0.570321438438)*x[2]**o+(0.107180499249)*x[2]
        arg[3,0,2,1]=(-0.77479605873)*x[0]**o+(0.928359807986)*x[0]+(-0.144170026638)*x[1]**o+(0.533835316696)*x[1]+(0.183938136052)*x[2]**o+(-0.87336340483)*x[2]
        arg[3,1,0,0]=(-0.121858163593)*x[0]**o+(-0.971122306623)*x[0]+(-0.351891554168)*x[1]**o+(-0.989754547278)*x[1]+(0.48106592456)*x[2]**o+(0.750009716044)*x[2]
        arg[3,1,0,1]=(-0.332320017719)*x[0]**o+(0.229814969323)*x[0]+(0.950038106614)*x[1]**o+(-0.364221636516)*x[1]+(-0.842684764717)*x[2]**o+(-0.430003876831)*x[2]
        arg[3,1,1,0]=(0.1036461168)*x[0]**o+(0.243047542195)*x[0]+(-0.189881740385)*x[1]**o+(-0.646948186795)*x[1]+(-0.246523431838)*x[2]**o+(0.0363222140239)*x[2]
        arg[3,1,1,1]=(0.629747493968)*x[0]**o+(0.257110968013)*x[0]+(-0.594332699815)*x[1]**o+(-0.74041867705)*x[1]+(0.656583120031)*x[2]**o+(0.392776534802)*x[2]
        arg[3,1,2,0]=(0.997368307137)*x[0]**o+(-0.709004475665)*x[0]+(-0.379447862403)*x[1]**o+(-0.368700793823)*x[1]+(0.293673648471)*x[2]**o+(-0.555545691082)*x[2]
        arg[3,1,2,1]=(-0.323027033928)*x[0]**o+(0.804640557762)*x[0]+(-0.199434780533)*x[1]**o+(0.604194682736)*x[1]+(0.945516011113)*x[2]**o+(-0.831838977373)*x[2]
        arg[3,2,0,0]=(0.255071922458)*x[0]**o+(0.261075290259)*x[0]+(0.198792431961)*x[1]**o+(0.987681568369)*x[1]+(-0.257626079249)*x[2]**o+(0.817801578613)*x[2]
        arg[3,2,0,1]=(-0.732615916833)*x[0]**o+(0.8114347023)*x[0]+(0.308896981216)*x[1]**o+(-0.164981244979)*x[1]+(0.724411853266)*x[2]**o+(-0.647969863039)*x[2]
        arg[3,2,1,0]=(0.126755528121)*x[0]**o+(0.0203968097627)*x[0]+(0.350888385037)*x[1]**o+(0.765584878761)*x[1]+(0.43521215637)*x[2]**o+(0.5821821143)*x[2]
        arg[3,2,1,1]=(0.795956472753)*x[0]**o+(0.363631111688)*x[0]+(-0.296417527594)*x[1]**o+(-0.0927953540926)*x[1]+(0.396041397741)*x[2]**o+(0.43932000066)*x[2]
        arg[3,2,2,0]=(-0.926754341344)*x[0]**o+(-0.688414966522)*x[0]+(-0.660984101932)*x[1]**o+(-0.627883344982)*x[1]+(0.403036784646)*x[2]**o+(0.818277068314)*x[2]
        arg[3,2,2,1]=(-0.358921307079)*x[0]**o+(0.636847962491)*x[0]+(-0.510050460541)*x[1]**o+(-0.758190944228)*x[1]+(-0.0319326931965)*x[2]**o+(-0.643422203077)*x[2]
        arg[3,3,0,0]=(-0.930730482483)*x[0]**o+(-0.883991363505)*x[0]+(0.156751809769)*x[1]**o+(0.669949596594)*x[1]+(0.697840698407)*x[2]**o+(0.721622061073)*x[2]
        arg[3,3,0,1]=(0.108814498453)*x[0]**o+(-0.164024398834)*x[0]+(-0.172084798233)*x[1]**o+(0.031530303081)*x[1]+(-0.985663868487)*x[2]**o+(-0.290224642955)*x[2]
        arg[3,3,1,0]=(0.690575802875)*x[0]**o+(-0.720428203968)*x[0]+(-0.837111899789)*x[1]**o+(0.140256279088)*x[1]+(0.532813462778)*x[2]**o+(-0.557885762558)*x[2]
        arg[3,3,1,1]=(-0.0785388159589)*x[0]**o+(-0.40595445804)*x[0]+(-0.691255332383)*x[1]**o+(0.54108326284)*x[1]+(0.737684932212)*x[2]**o+(0.423508847197)*x[2]
        arg[3,3,2,0]=(-0.589838770269)*x[0]**o+(-0.557133995504)*x[0]+(-0.297626023793)*x[1]**o+(-0.511809000987)*x[1]+(-0.933566600332)*x[2]**o+(-0.21434993446)*x[2]
        arg[3,3,2,1]=(0.19892263441)*x[0]**o+(0.633952342142)*x[0]+(0.0179400729177)*x[1]**o+(0.154020947355)*x[1]+(-0.620690264897)*x[2]**o+(0.524853027077)*x[2]
        arg[3,4,0,0]=(0.475231682303)*x[0]**o+(-0.48174419167)*x[0]+(-0.100854189221)*x[1]**o+(0.818166518054)*x[1]+(-0.922566309657)*x[2]**o+(-0.498745068755)*x[2]
        arg[3,4,0,1]=(0.613982633884)*x[0]**o+(-0.947323459945)*x[0]+(0.17395204013)*x[1]**o+(0.929669087853)*x[1]+(-0.617278089466)*x[2]**o+(0.900537470392)*x[2]
        arg[3,4,1,0]=(-0.403633510324)*x[0]**o+(0.954097943555)*x[0]+(-0.463422043607)*x[1]**o+(0.843916487223)*x[1]+(-0.5955912138)*x[2]**o+(-0.765444657727)*x[2]
        arg[3,4,1,1]=(-0.663412886802)*x[0]**o+(-0.547440885077)*x[0]+(-0.479993570508)*x[1]**o+(0.0620303034445)*x[1]+(-0.653759917895)*x[2]**o+(-0.681004441743)*x[2]
        arg[3,4,2,0]=(-0.994529375243)*x[0]**o+(0.546983479745)*x[0]+(0.17913912435)*x[1]**o+(0.660366383627)*x[1]+(0.0747943517387)*x[2]**o+(-0.321063660452)*x[2]
        arg[3,4,2,1]=(-0.538074654323)*x[0]**o+(-0.81330034965)*x[0]+(0.660614896398)*x[1]**o+(-0.918192909648)*x[1]+(0.553635166581)*x[2]**o+(-0.918131802189)*x[2]
        ref[0,0,0,0]=(-0.321539504757)/(o+1.)+(0.762866286444)+(-0.793251694335)*0.5**o
        ref[0,0,0,1]=(-0.264812286159)/(o+1.)+(-0.15293893368)+(-0.624206604764)*0.5**o
        ref[0,0,1,0]=(0.54441377247)/(o+1.)+(0.214067163116)+(0.115999561481)*0.5**o
        ref[0,0,1,1]=(1.50036854956)/(o+1.)+(0.673819079573)+(-0.972466831671)*0.5**o
        ref[0,0,2,0]=(0.687874396856)/(o+1.)+(-0.986502459901)+(0.708180260094)*0.5**o
        ref[0,0,2,1]=(0.793527041633)/(o+1.)+(-0.259940301856)+(0.0282615573045)*0.5**o
        ref[0,1,0,0]=(-1.25436610505)/(o+1.)+(-0.224984055664)+(0.231621820235)*0.5**o
        ref[0,1,0,1]=(0.891022819882)/(o+1.)+(-0.165060941565)+(0.114056801155)*0.5**o
        ref[0,1,1,0]=(-0.833444470649)/(o+1.)+(-0.322644910601)+(-0.973115195017)*0.5**o
        ref[0,1,1,1]=(-0.189799709023)/(o+1.)+(-0.242650862468)+(-0.304758579369)*0.5**o
        ref[0,1,2,0]=(-1.39857111894)/(o+1.)+(-0.14133379247)+(0.0923844967157)*0.5**o
        ref[0,1,2,1]=(1.03702218206)/(o+1.)+(-0.181927579186)+(0.793568921835)*0.5**o
        ref[0,2,0,0]=(-1.11962316818)/(o+1.)+(0.654627255004)+(0.570673248048)*0.5**o
        ref[0,2,0,1]=(0.085812863531)/(o+1.)+(0.17639811125)+(0.299061155215)*0.5**o
        ref[0,2,1,0]=(-1.26466090832)/(o+1.)+(0.618506300342)+(0.694725244159)*0.5**o
        ref[0,2,1,1]=(0.181495495342)/(o+1.)+(0.818577989479)+(-0.83896527178)*0.5**o
        ref[0,2,2,0]=(0.867512484289)/(o+1.)+(0.353365700269)+(0.2228052704)*0.5**o
        ref[0,2,2,1]=(0.618350374002)/(o+1.)+(-0.418609375405)+(0.63141216243)*0.5**o
        ref[0,3,0,0]=(-0.801998626232)/(o+1.)+(-0.302573026989)+(0.00698107166168)*0.5**o
        ref[0,3,0,1]=(0.414118735868)/(o+1.)+(0.103407742435)+(0.243423742325)*0.5**o
        ref[0,3,1,0]=(-0.129735888575)/(o+1.)+(0.51412655284)+(-0.106629554393)*0.5**o
        ref[0,3,1,1]=(0.0886696480864)/(o+1.)+(0.287824302591)+(0.375143733111)*0.5**o
        ref[0,3,2,0]=(-0.746501172831)/(o+1.)+(-0.389518072046)+(0.955685455438)*0.5**o
        ref[0,3,2,1]=(-1.06628341794)/(o+1.)+(-0.147429422878)+(0.353654445481)*0.5**o
        ref[0,4,0,0]=(-0.174512962078)/(o+1.)+(-0.023500931215)+(0.974759684151)*0.5**o
        ref[0,4,0,1]=(1.18079051113)/(o+1.)+(0.980823290345)+(-0.0799922241539)*0.5**o
        ref[0,4,1,0]=(-1.19690096085)/(o+1.)+(0.173437389579)+(0.532633280629)*0.5**o
        ref[0,4,1,1]=(1.11279531673)/(o+1.)+(-0.763621970064)+(-0.353243658293)*0.5**o
        ref[0,4,2,0]=(-0.819102364613)/(o+1.)+(0.326676095071)+(-0.720840869801)*0.5**o
        ref[0,4,2,1]=(-0.397779896209)/(o+1.)+(-0.677806011795)+(-0.179073400214)*0.5**o
        ref[1,0,0,0]=(0.00277094815158)/(o+1.)+(0.301845054648)+(-0.384011553333)*0.5**o
        ref[1,0,0,1]=(1.22422728891)/(o+1.)+(-0.594910516315)+(-0.808211570649)*0.5**o
        ref[1,0,1,0]=(0.869946029655)/(o+1.)+(0.308501276384)+(0.0404274840294)*0.5**o
        ref[1,0,1,1]=(1.63718264119)/(o+1.)+(0.76013279253)+(0.302394139901)*0.5**o
        ref[1,0,2,0]=(-0.0317278448724)/(o+1.)+(-0.118813696855)+(-0.981981997649)*0.5**o
        ref[1,0,2,1]=(-1.13670569191)/(o+1.)+(-1.14181937354)+(-0.237479805943)*0.5**o
        ref[1,1,0,0]=(0.16579906355)/(o+1.)+(-0.303649903525)+(-0.543719871098)*0.5**o
        ref[1,1,0,1]=(-1.23792473519)/(o+1.)+(-0.288446030005)+(-0.2624625471)*0.5**o
        ref[1,1,1,0]=(-1.42019666807)/(o+1.)+(-0.23595016925)+(0.88003513115)*0.5**o
        ref[1,1,1,1]=(0.400318898446)/(o+1.)+(0.765385300641)+(0.177746615088)*0.5**o
        ref[1,1,2,0]=(1.21553520103)/(o+1.)+(-0.272140700356)+(0.889070121473)*0.5**o
        ref[1,1,2,1]=(0.121042131161)/(o+1.)+(0.672369993679)+(0.951736215954)*0.5**o
        ref[1,2,0,0]=(-1.33133223229)/(o+1.)+(0.0067424746325)+(-0.954701036263)*0.5**o
        ref[1,2,0,1]=(1.02875611862)/(o+1.)+(0.288339603595)+(0.256489282842)*0.5**o
        ref[1,2,1,0]=(-0.32722580116)/(o+1.)+(0.166447794803)+(-0.94015914517)*0.5**o
        ref[1,2,1,1]=(0.319551509271)/(o+1.)+(0.112957240802)+(0.367156421931)*0.5**o
        ref[1,2,2,0]=(-1.3530782297)/(o+1.)+(-0.383051052329)+(-0.812516694781)*0.5**o
        ref[1,2,2,1]=(0.103238894111)/(o+1.)+(-0.09078366319)+(-0.497461944186)*0.5**o
        ref[1,3,0,0]=(-0.730489039376)/(o+1.)+(-0.321235679733)+(-0.227077550639)*0.5**o
        ref[1,3,0,1]=(0.532708925151)/(o+1.)+(-0.0963112877562)+(0.822091021645)*0.5**o
        ref[1,3,1,0]=(1.69380361755)/(o+1.)+(-0.221589701137)+(-0.0341577310045)*0.5**o
        ref[1,3,1,1]=(-0.653530558759)/(o+1.)+(-0.322066868053)+(-0.102867224088)*0.5**o
        ref[1,3,2,0]=(-0.279916251859)/(o+1.)+(-0.18685188305)+(0.556301076488)*0.5**o
        ref[1,3,2,1]=(0.798107960854)/(o+1.)+(-0.326052832107)+(0.888436531887)*0.5**o
        ref[1,4,0,0]=(0.250537417885)/(o+1.)+(-0.257356294716)+(0.651898073685)*0.5**o
        ref[1,4,0,1]=(1.5757556609)/(o+1.)+(-0.669914057301)+(-0.149187282052)*0.5**o
        ref[1,4,1,0]=(-1.0484286711)/(o+1.)+(0.142400846148)+(-0.285938055108)*0.5**o
        ref[1,4,1,1]=(0.028101537891)/(o+1.)+(0.55179042751)+(-0.34015136472)*0.5**o
        ref[1,4,2,0]=(0.974779411801)/(o+1.)+(-0.498887411653)+(-0.464687865356)*0.5**o
        ref[1,4,2,1]=(-0.544927129174)/(o+1.)+(-0.322066331457)+(-0.777967458564)*0.5**o
        ref[2,0,0,0]=(1.102518379)/(o+1.)+(-0.456744121978)+(0.665321754689)*0.5**o
        ref[2,0,0,1]=(-1.23793122706)/(o+1.)+(-0.366112767163)+(0.876435018569)*0.5**o
        ref[2,0,1,0]=(0.793755949398)/(o+1.)+(-0.113798359287)+(-0.344552466761)*0.5**o
        ref[2,0,1,1]=(-0.430899036829)/(o+1.)+(-0.0136660119329)+(0.841069648591)*0.5**o
        ref[2,0,2,0]=(-1.53425877377)/(o+1.)+(0.149679134)+(0.192691357169)*0.5**o
        ref[2,0,2,1]=(-1.30260629932)/(o+1.)+(0.89308995205)+(0.556616365284)*0.5**o
        ref[2,1,0,0]=(0.95383115974)/(o+1.)+(-0.0662076555126)+(0.947750123445)*0.5**o
        ref[2,1,0,1]=(0.660457702465)/(o+1.)+(-0.0149888527952)+(0.962918564559)*0.5**o
        ref[2,1,1,0]=(-1.03853354453)/(o+1.)+(0.711309478418)+(-0.549909211973)*0.5**o
        ref[2,1,1,1]=(1.25726273721)/(o+1.)+(0.122502877857)+(0.382867749635)*0.5**o
        ref[2,1,2,0]=(0.247755301657)/(o+1.)+(-1.11689502868)+(0.355742024596)*0.5**o
        ref[2,1,2,1]=(-0.043927846695)/(o+1.)+(-0.806594322632)+(-0.882334243081)*0.5**o
        ref[2,2,0,0]=(-0.0978017670124)/(o+1.)+(-0.596620341454)+(0.372164678051)*0.5**o
        ref[2,2,0,1]=(-1.4429890287)/(o+1.)+(-0.447576283364)+(0.126650330827)*0.5**o
        ref[2,2,1,0]=(-0.469449443913)/(o+1.)+(-0.0891064132419)+(-0.805920040628)*0.5**o
        ref[2,2,1,1]=(0.668102142762)/(o+1.)+(1.09333462337)+(-0.550624517803)*0.5**o
        ref[2,2,2,0]=(-0.857120744924)/(o+1.)+(0.0838275215328)+(-0.644276976064)*0.5**o
        ref[2,2,2,1]=(0.862934375332)/(o+1.)+(-0.0640179855954)+(0.564688999107)*0.5**o
        ref[2,3,0,0]=(-0.00322474925369)/(o+1.)+(-0.202705281737)+(0.64754765253)*0.5**o
        ref[2,3,0,1]=(0.495998145268)/(o+1.)+(0.602942165513)+(0.365315558093)*0.5**o
        ref[2,3,1,0]=(-1.15947595169)/(o+1.)+(0.688011753181)+(-0.984940830356)*0.5**o
        ref[2,3,1,1]=(0.280754586591)/(o+1.)+(0.450382022266)+(-0.785516613303)*0.5**o
        ref[2,3,2,0]=(-0.754064695248)/(o+1.)+(0.0702669914474)+(-0.310575473358)*0.5**o
        ref[2,3,2,1]=(-0.309252875518)/(o+1.)+(-0.793667320589)+(0.230960030802)*0.5**o
        ref[2,4,0,0]=(0.128900650436)/(o+1.)+(-0.847283648506)+(-0.811008715357)*0.5**o
        ref[2,4,0,1]=(-0.172967312072)/(o+1.)+(0.21314853029)+(0.817994599948)*0.5**o
        ref[2,4,1,0]=(0.148104464974)/(o+1.)+(0.874572421841)+(-0.626152000269)*0.5**o
        ref[2,4,1,1]=(0.165466005667)/(o+1.)+(-0.439581113198)+(0.97134483493)*0.5**o
        ref[2,4,2,0]=(-0.497279142326)/(o+1.)+(0.467228854876)+(-0.827528679064)*0.5**o
        ref[2,4,2,1]=(-0.867781096812)/(o+1.)+(0.565285636507)+(0.685882920875)*0.5**o
        ref[3,0,0,0]=(1.19781861678)/(o+1.)+(-0.278473056878)+(0.480000662344)*0.5**o
        ref[3,0,0,1]=(0.0344133360711)/(o+1.)+(0.888780677678)+(-0.819956304014)*0.5**o
        ref[3,0,1,0]=(0.891133668503)/(o+1.)+(-0.521653317213)+(0.736655857249)*0.5**o
        ref[3,0,1,1]=(0.499621703059)/(o+1.)+(-0.350127279125)+(0.615723453617)*0.5**o
        ref[3,0,2,0]=(0.237687834232)/(o+1.)+(0.0264450318399)+(-0.299975789693)*0.5**o
        ref[3,0,2,1]=(0.0397681094138)/(o+1.)+(0.294415859926)+(-0.77479605873)*0.5**o
        ref[3,1,0,0]=(0.129174370391)/(o+1.)+(-0.605433568929)+(-0.121858163593)*0.5**o
        ref[3,1,0,1]=(0.107353341897)/(o+1.)+(-0.282205272012)+(-0.332320017719)*0.5**o
        ref[3,1,1,0]=(-0.436405172223)/(o+1.)+(-0.183789215288)+(0.1036461168)*0.5**o
        ref[3,1,1,1]=(0.0622504202157)/(o+1.)+(-0.0452655871179)+(0.629747493968)*0.5**o
        ref[3,1,2,0]=(-0.0857742139322)/(o+1.)+(-0.816625480285)+(0.997368307137)*0.5**o
        ref[3,1,2,1]=(0.746081230581)/(o+1.)+(0.288498131562)+(-0.323027033928)*0.5**o
        ref[3,2,0,0]=(-0.0588336472879)/(o+1.)+(1.03327921862)+(0.255071922458)*0.5**o
        ref[3,2,0,1]=(1.03330883448)/(o+1.)+(-0.000758202858916)+(-0.732615916833)*0.5**o
        ref[3,2,1,0]=(0.786100541406)/(o+1.)+(0.684081901412)+(0.126755528121)*0.5**o
        ref[3,2,1,1]=(0.099623870147)/(o+1.)+(0.355077879128)+(0.795956472753)*0.5**o
        ref[3,2,2,0]=(-0.257947317287)/(o+1.)+(-0.249010621595)+(-0.926754341344)*0.5**o
        ref[3,2,2,1]=(-0.541983153738)/(o+1.)+(-0.382382592407)+(-0.358921307079)*0.5**o
        ref[3,3,0,0]=(0.854592508176)/(o+1.)+(0.253790147081)+(-0.930730482483)*0.5**o
        ref[3,3,0,1]=(-1.15774866672)/(o+1.)+(-0.211359369354)+(0.108814498453)*0.5**o
        ref[3,3,1,0]=(-0.304298437011)/(o+1.)+(-0.569028843719)+(0.690575802875)*0.5**o
        ref[3,3,1,1]=(0.0464295998283)/(o+1.)+(0.279318825998)+(-0.0785388159589)*0.5**o
        ref[3,3,2,0]=(-1.23119262412)/(o+1.)+(-0.641646465476)+(-0.589838770269)*0.5**o
        ref[3,3,2,1]=(-0.60275019198)/(o+1.)+(0.656413158287)+(0.19892263441)*0.5**o
        ref[3,4,0,0]=(-1.02342049888)/(o+1.)+(-0.0811613711857)+(0.475231682303)*0.5**o
        ref[3,4,0,1]=(-0.443326049337)/(o+1.)+(0.44144154915)+(0.613982633884)*0.5**o
        ref[3,4,1,0]=(-1.05901325741)/(o+1.)+(0.516284886526)+(-0.403633510324)*0.5**o
        ref[3,4,1,1]=(-1.1337534884)/(o+1.)+(-0.583207511688)+(-0.663412886802)*0.5**o
        ref[3,4,2,0]=(0.253933476088)/(o+1.)+(0.44314310146)+(-0.994529375243)*0.5**o
        ref[3,4,2,1]=(1.21425006298)/(o+1.)+(-1.32481253074)+(-0.538074654323)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_Solution_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactOne

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(-0.441098788712)*x[0]**o+(0.082953252724)*x[0]+(-0.97005044508)*x[1]**o+(0.928239302707)*x[1]
        ref=(-0.97005044508)/(o+1.)+(0.505596277715)+(-0.441098788712)*0.5**o
      else:
        arg=(0.384995424392)*x[0]**o+(0.999298118235)*x[0]+(0.144745460948)*x[1]**o+(0.0545198589076)*x[1]+(-0.773375797864)*x[2]**o+(0.145524911495)*x[2]
        ref=(-0.628630336917)/(o+1.)+(0.599671444318)+(0.384995424392)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_Solution_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactOne

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.0745889108767)*x[0]**o+(-0.694812761318)*x[0]+(-0.357791726239)*x[1]**o+(-0.785448350489)*x[1]
        arg[1]=(0.436006633056)*x[0]**o+(-0.797943632034)*x[0]+(-0.903321835716)*x[1]**o+(0.299527545727)*x[1]
        ref[0]=(-0.357791726239)/(o+1.)+(-0.740130555904)+(0.0745889108767)*0.5**o
        ref[1]=(-0.903321835716)/(o+1.)+(-0.249208043154)+(0.436006633056)*0.5**o
      else:
        arg[0]=(-0.292048194246)*x[0]**o+(-0.612402340241)*x[0]+(0.830253896674)*x[1]**o+(-0.0173245051245)*x[1]+(-0.105062444571)*x[2]**o+(0.968232377167)*x[2]
        arg[1]=(0.153118126553)*x[0]**o+(0.958628221601)*x[0]+(-0.326368437133)*x[1]**o+(0.878861879269)*x[1]+(-0.297398071379)*x[2]**o+(0.155202299471)*x[2]
        ref[0]=(0.725191452103)/(o+1.)+(0.169252765901)+(-0.292048194246)*0.5**o
        ref[1]=(-0.623766508512)/(o+1.)+(0.99634620017)+(0.153118126553)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_Solution_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactOne

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(0.410346849722)*x[0]**o+(0.86102012198)*x[0]+(-0.0143604505833)*x[1]**o+(-0.333268586255)*x[1]
        arg[0,1]=(-0.104991947387)*x[0]**o+(0.51540973662)*x[0]+(0.449773354197)*x[1]**o+(0.0397236625379)*x[1]
        arg[0,2]=(0.0010671797269)*x[0]**o+(0.165202831822)*x[0]+(-0.536444395178)*x[1]**o+(0.142112903729)*x[1]
        arg[0,3]=(0.0655247081801)*x[0]**o+(-0.59093644572)*x[0]+(-0.0135713515051)*x[1]**o+(0.665578431202)*x[1]
        arg[0,4]=(0.902382714765)*x[0]**o+(-0.358716290301)*x[0]+(-0.504792876433)*x[1]**o+(-0.760982991404)*x[1]
        arg[1,0]=(0.663815334483)*x[0]**o+(-0.967978720711)*x[0]+(-0.308953852146)*x[1]**o+(-0.250315651366)*x[1]
        arg[1,1]=(0.602976841368)*x[0]**o+(0.407346455361)*x[0]+(0.820076160142)*x[1]**o+(0.720380314572)*x[1]
        arg[1,2]=(0.0899286747557)*x[0]**o+(-0.636748133402)*x[0]+(0.615853620374)*x[1]**o+(0.389136375831)*x[1]
        arg[1,3]=(-0.695859848989)*x[0]**o+(-0.160892907203)*x[0]+(0.972252211687)*x[1]**o+(0.971653054313)*x[1]
        arg[1,4]=(-0.229141368314)*x[0]**o+(0.434757135281)*x[0]+(0.508751791725)*x[1]**o+(-0.593022987916)*x[1]
        arg[2,0]=(-0.732297666821)*x[0]**o+(-0.885623150041)*x[0]+(0.355264312146)*x[1]**o+(-0.203582600498)*x[1]
        arg[2,1]=(-0.0683706126862)*x[0]**o+(0.390953181924)*x[0]+(-0.104115827873)*x[1]**o+(-0.0586349400277)*x[1]
        arg[2,2]=(0.213052201915)*x[0]**o+(0.813758731408)*x[0]+(-0.927085393446)*x[1]**o+(-0.70884993627)*x[1]
        arg[2,3]=(-0.11792311156)*x[0]**o+(0.705053056049)*x[0]+(-0.988858028527)*x[1]**o+(-0.532493233644)*x[1]
        arg[2,4]=(0.643258167584)*x[0]**o+(0.991059864292)*x[0]+(0.466734669253)*x[1]**o+(-0.763209421816)*x[1]
        arg[3,0]=(-0.686036752646)*x[0]**o+(-0.0152278415398)*x[0]+(0.572818298467)*x[1]**o+(0.0883817883272)*x[1]
        arg[3,1]=(-0.498088836117)*x[0]**o+(0.413958831603)*x[0]+(-0.788338642889)*x[1]**o+(-0.219412111863)*x[1]
        arg[3,2]=(0.895103303091)*x[0]**o+(0.967224972282)*x[0]+(0.778129303031)*x[1]**o+(-0.239067231978)*x[1]
        arg[3,3]=(-0.715385920659)*x[0]**o+(-0.928446013585)*x[0]+(-0.919578735328)*x[1]**o+(0.735438220506)*x[1]
        arg[3,4]=(0.273825880641)*x[0]**o+(0.602304312788)*x[0]+(-0.788415866411)*x[1]**o+(0.841776933557)*x[1]
        ref[0,0]=(-0.0143604505833)/(o+1.)+(0.263875767862)+(0.410346849722)*0.5**o
        ref[0,1]=(0.449773354197)/(o+1.)+(0.277566699579)+(-0.104991947387)*0.5**o
        ref[0,2]=(-0.536444395178)/(o+1.)+(0.153657867776)+(0.0010671797269)*0.5**o
        ref[0,3]=(-0.0135713515051)/(o+1.)+(0.0373209927411)+(0.0655247081801)*0.5**o
        ref[0,4]=(-0.504792876433)/(o+1.)+(-0.559849640852)+(0.902382714765)*0.5**o
        ref[1,0]=(-0.308953852146)/(o+1.)+(-0.609147186038)+(0.663815334483)*0.5**o
        ref[1,1]=(0.820076160142)/(o+1.)+(0.563863384966)+(0.602976841368)*0.5**o
        ref[1,2]=(0.615853620374)/(o+1.)+(-0.123805878786)+(0.0899286747557)*0.5**o
        ref[1,3]=(0.972252211687)/(o+1.)+(0.405380073555)+(-0.695859848989)*0.5**o
        ref[1,4]=(0.508751791725)/(o+1.)+(-0.0791329263176)+(-0.229141368314)*0.5**o
        ref[2,0]=(0.355264312146)/(o+1.)+(-0.544602875269)+(-0.732297666821)*0.5**o
        ref[2,1]=(-0.104115827873)/(o+1.)+(0.166159120948)+(-0.0683706126862)*0.5**o
        ref[2,2]=(-0.927085393446)/(o+1.)+(0.0524543975689)+(0.213052201915)*0.5**o
        ref[2,3]=(-0.988858028527)/(o+1.)+(0.0862799112023)+(-0.11792311156)*0.5**o
        ref[2,4]=(0.466734669253)/(o+1.)+(0.113925221238)+(0.643258167584)*0.5**o
        ref[3,0]=(0.572818298467)/(o+1.)+(0.0365769733937)+(-0.686036752646)*0.5**o
        ref[3,1]=(-0.788338642889)/(o+1.)+(0.0972733598697)+(-0.498088836117)*0.5**o
        ref[3,2]=(0.778129303031)/(o+1.)+(0.364078870152)+(0.895103303091)*0.5**o
        ref[3,3]=(-0.919578735328)/(o+1.)+(-0.0965038965398)+(-0.715385920659)*0.5**o
        ref[3,4]=(-0.788415866411)/(o+1.)+(0.722040623173)+(0.273825880641)*0.5**o
      else:
        arg[0,0]=(-0.903197979191)*x[0]**o+(-0.497252911882)*x[0]+(0.5689188153)*x[1]**o+(-0.569434759407)*x[1]+(-0.204533481399)*x[2]**o+(-0.0255207088467)*x[2]
        arg[0,1]=(-0.0296606374337)*x[0]**o+(0.917552577919)*x[0]+(0.746132694062)*x[1]**o+(0.285568252958)*x[1]+(0.175571603104)*x[2]**o+(0.5701931124)*x[2]
        arg[0,2]=(0.657981464039)*x[0]**o+(-0.105816675859)*x[0]+(-0.210726712386)*x[1]**o+(-0.457303699577)*x[1]+(-0.225517453595)*x[2]**o+(0.61593257413)*x[2]
        arg[0,3]=(-0.652766009035)*x[0]**o+(0.538908603782)*x[0]+(0.226594150058)*x[1]**o+(0.691428293374)*x[1]+(-0.210896072508)*x[2]**o+(0.538061905948)*x[2]
        arg[0,4]=(0.175024193446)*x[0]**o+(0.433285241007)*x[0]+(0.928411521986)*x[1]**o+(-0.241175862661)*x[1]+(-0.779192552542)*x[2]**o+(0.428100531631)*x[2]
        arg[1,0]=(-0.094230207815)*x[0]**o+(-0.113255272088)*x[0]+(-0.880816596679)*x[1]**o+(0.56261116141)*x[1]+(-0.418606336511)*x[2]**o+(0.858058623759)*x[2]
        arg[1,1]=(0.08556853498)*x[0]**o+(0.95491440456)*x[0]+(-0.435275492231)*x[1]**o+(0.339105115812)*x[1]+(-0.634101556975)*x[2]**o+(0.357557194572)*x[2]
        arg[1,2]=(-0.692883812524)*x[0]**o+(0.673540357384)*x[0]+(0.240219465837)*x[1]**o+(0.139957925862)*x[1]+(0.135892430596)*x[2]**o+(0.383465818668)*x[2]
        arg[1,3]=(-0.892921935206)*x[0]**o+(0.0694647682287)*x[0]+(0.0530888770821)*x[1]**o+(-0.0735398003251)*x[1]+(-0.40702392212)*x[2]**o+(-0.223494881098)*x[2]
        arg[1,4]=(0.743112585337)*x[0]**o+(-0.12393012942)*x[0]+(-0.660616664197)*x[1]**o+(-0.0334489536139)*x[1]+(-0.0566394013558)*x[2]**o+(-0.0281273872158)*x[2]
        arg[2,0]=(-0.500500871517)*x[0]**o+(-0.326237506809)*x[0]+(-0.960480146386)*x[1]**o+(0.0901654083154)*x[1]+(0.466124845857)*x[2]**o+(0.6908848619)*x[2]
        arg[2,1]=(0.120233846989)*x[0]**o+(0.520609850629)*x[0]+(-0.407423400749)*x[1]**o+(0.59145151128)*x[1]+(0.981171407459)*x[2]**o+(-0.703987165108)*x[2]
        arg[2,2]=(0.811775583161)*x[0]**o+(-0.209650983138)*x[0]+(-0.286223486261)*x[1]**o+(-0.864122954179)*x[1]+(0.609725467363)*x[2]**o+(0.0311464898903)*x[2]
        arg[2,3]=(-0.325647787579)*x[0]**o+(0.956001955119)*x[0]+(0.637272546129)*x[1]**o+(-0.951109798103)*x[1]+(0.406445558466)*x[2]**o+(-0.930854597943)*x[2]
        arg[2,4]=(0.0621315789778)*x[0]**o+(-0.917033227802)*x[0]+(-0.179584904974)*x[1]**o+(0.283357518809)*x[1]+(-0.778863909832)*x[2]**o+(-0.524497331838)*x[2]
        arg[3,0]=(0.140605486894)*x[0]**o+(0.172882849759)*x[0]+(-0.883586756712)*x[1]**o+(-0.160461884399)*x[1]+(-0.718343713337)*x[2]**o+(0.196937721256)*x[2]
        arg[3,1]=(-0.660232764173)*x[0]**o+(-0.92091042631)*x[0]+(-0.568852134661)*x[1]**o+(-0.331441330509)*x[1]+(-0.220286922375)*x[2]**o+(-0.256407439105)*x[2]
        arg[3,2]=(0.593940468197)*x[0]**o+(0.527756437117)*x[0]+(0.943980933807)*x[1]**o+(0.749415167795)*x[1]+(0.484043332105)*x[2]**o+(0.40135030381)*x[2]
        arg[3,3]=(0.870508483705)*x[0]**o+(-0.234209640456)*x[0]+(-0.62692310882)*x[1]**o+(0.352628981854)*x[1]+(-0.24937844453)*x[2]**o+(0.73438735864)*x[2]
        arg[3,4]=(-0.638437449353)*x[0]**o+(0.308007229542)*x[0]+(0.537590844528)*x[1]**o+(-0.755595388361)*x[1]+(-0.152781228745)*x[2]**o+(-0.450122095362)*x[2]
        ref[0,0]=(0.364385333901)/(o+1.)+(-0.546104190068)+(-0.903197979191)*0.5**o
        ref[0,1]=(0.921704297167)/(o+1.)+(0.886656971638)+(-0.0296606374337)*0.5**o
        ref[0,2]=(-0.436244165981)/(o+1.)+(0.0264060993469)+(0.657981464039)*0.5**o
        ref[0,3]=(0.0156980775503)/(o+1.)+(0.884199401552)+(-0.652766009035)*0.5**o
        ref[0,4]=(0.149218969444)/(o+1.)+(0.310104954989)+(0.175024193446)*0.5**o
        ref[1,0]=(-1.29942293319)/(o+1.)+(0.653707256541)+(-0.094230207815)*0.5**o
        ref[1,1]=(-1.06937704921)/(o+1.)+(0.825788357473)+(0.08556853498)*0.5**o
        ref[1,2]=(0.376111896433)/(o+1.)+(0.598482050957)+(-0.692883812524)*0.5**o
        ref[1,3]=(-0.353935045038)/(o+1.)+(-0.113784956597)+(-0.892921935206)*0.5**o
        ref[1,4]=(-0.717256065552)/(o+1.)+(-0.0927532351247)+(0.743112585337)*0.5**o
        ref[2,0]=(-0.494355300528)/(o+1.)+(0.227406381703)+(-0.500500871517)*0.5**o
        ref[2,1]=(0.57374800671)/(o+1.)+(0.2040370984)+(0.120233846989)*0.5**o
        ref[2,2]=(0.323501981101)/(o+1.)+(-0.521313723714)+(0.811775583161)*0.5**o
        ref[2,3]=(1.04371810459)/(o+1.)+(-0.462981220464)+(-0.325647787579)*0.5**o
        ref[2,4]=(-0.958448814807)/(o+1.)+(-0.579086520416)+(0.0621315789778)*0.5**o
        ref[3,0]=(-1.60193047005)/(o+1.)+(0.104679343308)+(0.140605486894)*0.5**o
        ref[3,1]=(-0.789139057035)/(o+1.)+(-0.754379597962)+(-0.660232764173)*0.5**o
        ref[3,2]=(1.42802426591)/(o+1.)+(0.839260954361)+(0.593940468197)*0.5**o
        ref[3,3]=(-0.87630155335)/(o+1.)+(0.426403350019)+(0.870508483705)*0.5**o
        ref[3,4]=(0.384809615783)/(o+1.)+(-0.448855127091)+(-0.638437449353)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_Solution_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnContactOne

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(-0.315756069525)*x[0]**o+(0.253240552004)*x[0]+(0.86540335823)*x[1]**o+(-0.616203780065)*x[1]
        arg[0,0,1]=(-0.948116617977)*x[0]**o+(0.892154336439)*x[0]+(0.136605864823)*x[1]**o+(-0.135599413854)*x[1]
        arg[0,1,0]=(0.249388669107)*x[0]**o+(-0.299043026617)*x[0]+(0.335963868826)*x[1]**o+(-0.147202749474)*x[1]
        arg[0,1,1]=(0.649003124445)*x[0]**o+(0.102344972673)*x[0]+(-0.795038258171)*x[1]**o+(0.149716983842)*x[1]
        arg[1,0,0]=(0.814606374941)*x[0]**o+(-0.949251236145)*x[0]+(0.723727126173)*x[1]**o+(-0.75419822708)*x[1]
        arg[1,0,1]=(0.043251358201)*x[0]**o+(-0.227748374318)*x[0]+(-0.231556019421)*x[1]**o+(-0.152524439909)*x[1]
        arg[1,1,0]=(0.565989211326)*x[0]**o+(-0.488248650798)*x[0]+(-0.353077780419)*x[1]**o+(-0.0139123365766)*x[1]
        arg[1,1,1]=(-0.106085432662)*x[0]**o+(0.627311916563)*x[0]+(-0.496301561324)*x[1]**o+(-0.648448849796)*x[1]
        arg[2,0,0]=(0.741868098354)*x[0]**o+(-0.486898566862)*x[0]+(0.0461464851414)*x[1]**o+(0.0133916949979)*x[1]
        arg[2,0,1]=(0.969108066213)*x[0]**o+(-0.688797899381)*x[0]+(0.897017408844)*x[1]**o+(0.807714585761)*x[1]
        arg[2,1,0]=(-0.138159486947)*x[0]**o+(-0.377759480876)*x[0]+(-0.764484355987)*x[1]**o+(-0.723912265361)*x[1]
        arg[2,1,1]=(-0.781167132076)*x[0]**o+(0.119512137054)*x[0]+(-0.535230596139)*x[1]**o+(0.441358968417)*x[1]
        arg[3,0,0]=(-0.663820836909)*x[0]**o+(0.907668749761)*x[0]+(-0.345114213135)*x[1]**o+(0.0580058218243)*x[1]
        arg[3,0,1]=(-0.359054701663)*x[0]**o+(0.0389188839076)*x[0]+(0.168050801441)*x[1]**o+(-0.695770024565)*x[1]
        arg[3,1,0]=(0.70118269628)*x[0]**o+(0.87976625188)*x[0]+(0.154503286758)*x[1]**o+(0.807878896951)*x[1]
        arg[3,1,1]=(-0.143516320927)*x[0]**o+(-0.369452129865)*x[0]+(-0.0392268449697)*x[1]**o+(-0.046104227955)*x[1]
        arg[4,0,0]=(0.197244254754)*x[0]**o+(-0.580444991466)*x[0]+(0.334036744845)*x[1]**o+(0.000791083620504)*x[1]
        arg[4,0,1]=(-0.892626226865)*x[0]**o+(0.140042741589)*x[0]+(0.158775259845)*x[1]**o+(-0.308266331028)*x[1]
        arg[4,1,0]=(0.650374542633)*x[0]**o+(0.586152174865)*x[0]+(0.0201324360301)*x[1]**o+(-0.813221777119)*x[1]
        arg[4,1,1]=(0.149656968314)*x[0]**o+(0.700641349434)*x[0]+(-0.529780404433)*x[1]**o+(0.512159137234)*x[1]
        arg[5,0,0]=(0.288664621864)*x[0]**o+(0.119652133593)*x[0]+(-0.599113573241)*x[1]**o+(0.879966724309)*x[1]
        arg[5,0,1]=(0.861443356689)*x[0]**o+(0.530607121787)*x[0]+(0.240306513701)*x[1]**o+(-0.575772515225)*x[1]
        arg[5,1,0]=(0.721635890008)*x[0]**o+(-0.109537583449)*x[0]+(0.162041613112)*x[1]**o+(0.229143033467)*x[1]
        arg[5,1,1]=(-0.876930820494)*x[0]**o+(0.749378811252)*x[0]+(0.886213554632)*x[1]**o+(0.433327797349)*x[1]
        ref[0,0,0]=(0.86540335823)/(o+1.)+(-0.18148161403)+(-0.315756069525)*0.5**o
        ref[0,0,1]=(0.136605864823)/(o+1.)+(0.378277461292)+(-0.948116617977)*0.5**o
        ref[0,1,0]=(0.335963868826)/(o+1.)+(-0.223122888045)+(0.249388669107)*0.5**o
        ref[0,1,1]=(-0.795038258171)/(o+1.)+(0.126030978258)+(0.649003124445)*0.5**o
        ref[1,0,0]=(0.723727126173)/(o+1.)+(-0.851724731613)+(0.814606374941)*0.5**o
        ref[1,0,1]=(-0.231556019421)/(o+1.)+(-0.190136407114)+(0.043251358201)*0.5**o
        ref[1,1,0]=(-0.353077780419)/(o+1.)+(-0.251080493687)+(0.565989211326)*0.5**o
        ref[1,1,1]=(-0.496301561324)/(o+1.)+(-0.0105684666165)+(-0.106085432662)*0.5**o
        ref[2,0,0]=(0.0461464851414)/(o+1.)+(-0.236753435932)+(0.741868098354)*0.5**o
        ref[2,0,1]=(0.897017408844)/(o+1.)+(0.05945834319)+(0.969108066213)*0.5**o
        ref[2,1,0]=(-0.764484355987)/(o+1.)+(-0.550835873118)+(-0.138159486947)*0.5**o
        ref[2,1,1]=(-0.535230596139)/(o+1.)+(0.280435552736)+(-0.781167132076)*0.5**o
        ref[3,0,0]=(-0.345114213135)/(o+1.)+(0.482837285793)+(-0.663820836909)*0.5**o
        ref[3,0,1]=(0.168050801441)/(o+1.)+(-0.328425570329)+(-0.359054701663)*0.5**o
        ref[3,1,0]=(0.154503286758)/(o+1.)+(0.843822574415)+(0.70118269628)*0.5**o
        ref[3,1,1]=(-0.0392268449697)/(o+1.)+(-0.20777817891)+(-0.143516320927)*0.5**o
        ref[4,0,0]=(0.334036744845)/(o+1.)+(-0.289826953923)+(0.197244254754)*0.5**o
        ref[4,0,1]=(0.158775259845)/(o+1.)+(-0.0841117947196)+(-0.892626226865)*0.5**o
        ref[4,1,0]=(0.0201324360301)/(o+1.)+(-0.113534801127)+(0.650374542633)*0.5**o
        ref[4,1,1]=(-0.529780404433)/(o+1.)+(0.606400243334)+(0.149656968314)*0.5**o
        ref[5,0,0]=(-0.599113573241)/(o+1.)+(0.499809428951)+(0.288664621864)*0.5**o
        ref[5,0,1]=(0.240306513701)/(o+1.)+(-0.022582696719)+(0.861443356689)*0.5**o
        ref[5,1,0]=(0.162041613112)/(o+1.)+(0.0598027250089)+(0.721635890008)*0.5**o
        ref[5,1,1]=(0.886213554632)/(o+1.)+(0.5913533043)+(-0.876930820494)*0.5**o
      else:
        arg[0,0,0]=(0.804252705823)*x[0]**o+(-0.215722921442)*x[0]+(-0.806316349529)*x[1]**o+(-0.984002993222)*x[1]+(-0.411946014731)*x[2]**o+(-0.526356191855)*x[2]
        arg[0,0,1]=(-0.845902068304)*x[0]**o+(0.0773568471048)*x[0]+(-0.746877301439)*x[1]**o+(-0.581914187059)*x[1]+(-0.0299873869697)*x[2]**o+(-0.133702230427)*x[2]
        arg[0,1,0]=(0.0113663840954)*x[0]**o+(-0.155702215771)*x[0]+(-0.134490626112)*x[1]**o+(0.973329917011)*x[1]+(-0.71585624135)*x[2]**o+(-0.840113198085)*x[2]
        arg[0,1,1]=(0.708589516593)*x[0]**o+(0.40282025998)*x[0]+(0.114498353117)*x[1]**o+(0.117244230433)*x[1]+(0.110135318853)*x[2]**o+(-0.0227879044898)*x[2]
        arg[1,0,0]=(0.836102144172)*x[0]**o+(0.212564417583)*x[0]+(0.744354739679)*x[1]**o+(-0.265633978841)*x[1]+(0.843108381264)*x[2]**o+(-0.288428427501)*x[2]
        arg[1,0,1]=(-0.0712209756059)*x[0]**o+(-0.186158140814)*x[0]+(0.184457099888)*x[1]**o+(0.381365494077)*x[1]+(-0.547874738232)*x[2]**o+(0.645124013254)*x[2]
        arg[1,1,0]=(-0.982992052158)*x[0]**o+(0.259627201342)*x[0]+(-0.885159598031)*x[1]**o+(-0.720754022982)*x[1]+(-0.762444117798)*x[2]**o+(0.737661938668)*x[2]
        arg[1,1,1]=(0.518426662725)*x[0]**o+(0.718331452733)*x[0]+(0.523870992899)*x[1]**o+(0.527831420124)*x[1]+(0.791634534316)*x[2]**o+(0.17320870773)*x[2]
        arg[2,0,0]=(-0.0368697940364)*x[0]**o+(-0.101718169515)*x[0]+(0.933712377315)*x[1]**o+(-0.465193151914)*x[1]+(-0.113637506449)*x[2]**o+(0.606619949802)*x[2]
        arg[2,0,1]=(-0.785633726304)*x[0]**o+(0.526518355588)*x[0]+(0.151859749675)*x[1]**o+(0.805094094642)*x[1]+(-0.881122119524)*x[2]**o+(0.85143600659)*x[2]
        arg[2,1,0]=(0.969244001518)*x[0]**o+(0.22600349917)*x[0]+(0.404826692149)*x[1]**o+(0.633039564539)*x[1]+(0.470567669552)*x[2]**o+(0.378348877016)*x[2]
        arg[2,1,1]=(-0.310035929524)*x[0]**o+(0.42632271287)*x[0]+(0.866930472029)*x[1]**o+(-0.124483457934)*x[1]+(0.291920406068)*x[2]**o+(0.573984377154)*x[2]
        arg[3,0,0]=(0.245553971829)*x[0]**o+(-0.567563766526)*x[0]+(0.326654705159)*x[1]**o+(0.642370867777)*x[1]+(-0.538450732704)*x[2]**o+(0.773349470647)*x[2]
        arg[3,0,1]=(0.427017913395)*x[0]**o+(0.885132021286)*x[0]+(0.0338747045378)*x[1]**o+(-0.978508839789)*x[1]+(-0.734559666249)*x[2]**o+(0.379991690191)*x[2]
        arg[3,1,0]=(-0.29533334088)*x[0]**o+(-0.894548834889)*x[0]+(0.537043269081)*x[1]**o+(-0.559173506199)*x[1]+(0.359216600837)*x[2]**o+(0.269563745611)*x[2]
        arg[3,1,1]=(-0.146152544907)*x[0]**o+(-0.95681567795)*x[0]+(-0.0323978120064)*x[1]**o+(-0.5351716012)*x[1]+(-0.548536561192)*x[2]**o+(0.489675460511)*x[2]
        arg[4,0,0]=(0.0552991386013)*x[0]**o+(0.772417012152)*x[0]+(-0.298212559744)*x[1]**o+(-0.189934984078)*x[1]+(0.74843296731)*x[2]**o+(-0.00412448772016)*x[2]
        arg[4,0,1]=(0.577046858044)*x[0]**o+(-0.804142676861)*x[0]+(-0.0703211154568)*x[1]**o+(-0.84092953544)*x[1]+(0.254119640557)*x[2]**o+(-0.765048199333)*x[2]
        arg[4,1,0]=(-0.358558179011)*x[0]**o+(-0.998072158283)*x[0]+(0.508152998513)*x[1]**o+(-0.0813296609683)*x[1]+(-0.534332816004)*x[2]**o+(-0.189623317277)*x[2]
        arg[4,1,1]=(0.840098089823)*x[0]**o+(0.966726062679)*x[0]+(0.920079609258)*x[1]**o+(-0.242532131681)*x[1]+(0.429713768025)*x[2]**o+(0.171223902751)*x[2]
        arg[5,0,0]=(0.695331559401)*x[0]**o+(0.819482211381)*x[0]+(0.550452400412)*x[1]**o+(0.518496496336)*x[1]+(-0.24727850431)*x[2]**o+(-0.796406493488)*x[2]
        arg[5,0,1]=(0.469488028534)*x[0]**o+(0.457530124732)*x[0]+(0.922206279456)*x[1]**o+(0.678758573006)*x[1]+(0.34102098746)*x[2]**o+(0.108616278675)*x[2]
        arg[5,1,0]=(0.444799933589)*x[0]**o+(-0.361568786934)*x[0]+(0.470174557338)*x[1]**o+(-0.344557214342)*x[1]+(0.894860467988)*x[2]**o+(-0.347027743579)*x[2]
        arg[5,1,1]=(0.0304781332872)*x[0]**o+(-0.69046475182)*x[0]+(0.873260975232)*x[1]**o+(-0.705100555993)*x[1]+(0.660057279013)*x[2]**o+(-0.910002146105)*x[2]
        ref[0,0,0]=(-1.21826236426)/(o+1.)+(-0.863041053259)+(0.804252705823)*0.5**o
        ref[0,0,1]=(-0.776864688409)/(o+1.)+(-0.31912978519)+(-0.845902068304)*0.5**o
        ref[0,1,0]=(-0.850346867462)/(o+1.)+(-0.0112427484226)+(0.0113663840954)*0.5**o
        ref[0,1,1]=(0.22463367197)/(o+1.)+(0.248638292962)+(0.708589516593)*0.5**o
        ref[1,0,0]=(1.58746312094)/(o+1.)+(-0.17074899438)+(0.836102144172)*0.5**o
        ref[1,0,1]=(-0.363417638344)/(o+1.)+(0.420165683258)+(-0.0712209756059)*0.5**o
        ref[1,1,0]=(-1.64760371583)/(o+1.)+(0.138267558514)+(-0.982992052158)*0.5**o
        ref[1,1,1]=(1.31550552722)/(o+1.)+(0.709685790293)+(0.518426662725)*0.5**o
        ref[2,0,0]=(0.820074870865)/(o+1.)+(0.0198543141864)+(-0.0368697940364)*0.5**o
        ref[2,0,1]=(-0.729262369848)/(o+1.)+(1.09152422841)+(-0.785633726304)*0.5**o
        ref[2,1,0]=(0.875394361701)/(o+1.)+(0.618695970363)+(0.969244001518)*0.5**o
        ref[2,1,1]=(1.1588508781)/(o+1.)+(0.437911816045)+(-0.310035929524)*0.5**o
        ref[3,0,0]=(-0.211796027546)/(o+1.)+(0.424078285949)+(0.245553971829)*0.5**o
        ref[3,0,1]=(-0.700684961711)/(o+1.)+(0.143307435844)+(0.427017913395)*0.5**o
        ref[3,1,0]=(0.896259869918)/(o+1.)+(-0.592079297738)+(-0.29533334088)*0.5**o
        ref[3,1,1]=(-0.580934373198)/(o+1.)+(-0.50115590932)+(-0.146152544907)*0.5**o
        ref[4,0,0]=(0.450220407566)/(o+1.)+(0.289178770177)+(0.0552991386013)*0.5**o
        ref[4,0,1]=(0.1837985251)/(o+1.)+(-1.20506020582)+(0.577046858044)*0.5**o
        ref[4,1,0]=(-0.0261798174917)/(o+1.)+(-0.634512568264)+(-0.358558179011)*0.5**o
        ref[4,1,1]=(1.34979337728)/(o+1.)+(0.447708916874)+(0.840098089823)*0.5**o
        ref[5,0,0]=(0.303173896103)/(o+1.)+(0.270786107114)+(0.695331559401)*0.5**o
        ref[5,0,1]=(1.26322726692)/(o+1.)+(0.622452488207)+(0.469488028534)*0.5**o
        ref[5,1,0]=(1.36503502533)/(o+1.)+(-0.526576872427)+(0.444799933589)*0.5**o
        ref[5,1,1]=(1.53331825424)/(o+1.)+(-1.15278372696)+(0.0304781332872)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_Solution_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnContactOne

      assumptions: Solution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=Solution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(-0.348972036919)*x[0]**o+(0.123542094911)*x[0]+(-0.596353540657)*x[1]**o+(0.717656729405)*x[1]
        arg[0,0,0,1]=(0.187733014896)*x[0]**o+(0.76244697815)*x[0]+(0.261409664458)*x[1]**o+(0.0213714031642)*x[1]
        arg[0,0,1,0]=(0.951655197513)*x[0]**o+(-0.865274104487)*x[0]+(0.59127040848)*x[1]**o+(-0.261622300136)*x[1]
        arg[0,0,1,1]=(-0.188127934304)*x[0]**o+(-0.513768204254)*x[0]+(0.77162116582)*x[1]**o+(-0.12621176914)*x[1]
        arg[0,0,2,0]=(0.770717242337)*x[0]**o+(-0.991038362841)*x[0]+(0.908566805373)*x[1]**o+(-0.24890064672)*x[1]
        arg[0,0,2,1]=(-0.543253359807)*x[0]**o+(-0.0504511921259)*x[0]+(-0.724584621389)*x[1]**o+(0.999965312141)*x[1]
        arg[0,1,0,0]=(0.986939140609)*x[0]**o+(0.236894127675)*x[0]+(0.520162431169)*x[1]**o+(0.306413415759)*x[1]
        arg[0,1,0,1]=(-0.171773212929)*x[0]**o+(-0.248436968201)*x[0]+(0.43818378458)*x[1]**o+(-0.495734982435)*x[1]
        arg[0,1,1,0]=(-0.369350989684)*x[0]**o+(0.59047357584)*x[0]+(0.422340945702)*x[1]**o+(0.326270009775)*x[1]
        arg[0,1,1,1]=(0.559606389467)*x[0]**o+(-0.0507064379501)*x[0]+(-0.0464190048363)*x[1]**o+(0.18964879235)*x[1]
        arg[0,1,2,0]=(-0.989408371198)*x[0]**o+(0.108750148268)*x[0]+(-0.347984444559)*x[1]**o+(-0.819590699683)*x[1]
        arg[0,1,2,1]=(-0.289295900777)*x[0]**o+(-0.579698633174)*x[0]+(-0.124928198101)*x[1]**o+(0.640901698875)*x[1]
        arg[0,2,0,0]=(0.132599667548)*x[0]**o+(-0.102095433147)*x[0]+(0.446169236302)*x[1]**o+(-0.844062525718)*x[1]
        arg[0,2,0,1]=(-0.452662917924)*x[0]**o+(-0.0497770418153)*x[0]+(-0.258525519083)*x[1]**o+(-0.0365197858914)*x[1]
        arg[0,2,1,0]=(0.318740871592)*x[0]**o+(0.562764515639)*x[0]+(-0.209089266173)*x[1]**o+(0.922024900771)*x[1]
        arg[0,2,1,1]=(-0.825757440563)*x[0]**o+(0.634009249217)*x[0]+(0.954836398565)*x[1]**o+(-0.910652766097)*x[1]
        arg[0,2,2,0]=(-0.640337007803)*x[0]**o+(0.862625623816)*x[0]+(-0.0327529254483)*x[1]**o+(-0.327305638603)*x[1]
        arg[0,2,2,1]=(-0.956300293885)*x[0]**o+(0.456116376134)*x[0]+(0.312023159058)*x[1]**o+(0.768546792276)*x[1]
        arg[0,3,0,0]=(-0.927040285049)*x[0]**o+(-0.614994799603)*x[0]+(-0.431859510779)*x[1]**o+(-0.088774748513)*x[1]
        arg[0,3,0,1]=(-0.807009561768)*x[0]**o+(0.00323409463398)*x[0]+(0.274814268019)*x[1]**o+(-0.132755038246)*x[1]
        arg[0,3,1,0]=(0.931774139846)*x[0]**o+(0.665327108449)*x[0]+(0.130437344004)*x[1]**o+(-0.00859025546518)*x[1]
        arg[0,3,1,1]=(-0.314188008746)*x[0]**o+(0.405251729437)*x[0]+(-0.313484462262)*x[1]**o+(-0.670765905491)*x[1]
        arg[0,3,2,0]=(0.748691505108)*x[0]**o+(0.717110963353)*x[0]+(-0.451892315269)*x[1]**o+(-0.103930621637)*x[1]
        arg[0,3,2,1]=(-0.124290384657)*x[0]**o+(0.0761550925153)*x[0]+(0.864166710803)*x[1]**o+(0.050297196468)*x[1]
        arg[0,4,0,0]=(-0.726606985714)*x[0]**o+(-0.304862755503)*x[0]+(-0.364735884875)*x[1]**o+(0.769999145518)*x[1]
        arg[0,4,0,1]=(-0.716457484159)*x[0]**o+(0.850185561874)*x[0]+(0.924535166786)*x[1]**o+(0.414539526778)*x[1]
        arg[0,4,1,0]=(-0.67154700238)*x[0]**o+(-0.0990079744678)*x[0]+(-0.738258048173)*x[1]**o+(-0.105333364414)*x[1]
        arg[0,4,1,1]=(-0.568569022142)*x[0]**o+(-0.865340476096)*x[0]+(0.592223896873)*x[1]**o+(0.545216501911)*x[1]
        arg[0,4,2,0]=(0.959619955514)*x[0]**o+(-0.625097978453)*x[0]+(0.447944032533)*x[1]**o+(-0.792670959618)*x[1]
        arg[0,4,2,1]=(-0.446578795272)*x[0]**o+(0.516341492325)*x[0]+(0.383143400712)*x[1]**o+(-0.339255681885)*x[1]
        arg[1,0,0,0]=(0.130600745253)*x[0]**o+(-0.225084866063)*x[0]+(-0.871963625443)*x[1]**o+(-0.0741251944735)*x[1]
        arg[1,0,0,1]=(-0.684630134211)*x[0]**o+(-0.298097256903)*x[0]+(0.511206122216)*x[1]**o+(-0.784104136535)*x[1]
        arg[1,0,1,0]=(0.624178362847)*x[0]**o+(0.161787148072)*x[0]+(-0.0906465041298)*x[1]**o+(-0.0302149132362)*x[1]
        arg[1,0,1,1]=(-0.403988173215)*x[0]**o+(-0.423895624882)*x[0]+(-0.0594261878161)*x[1]**o+(0.71753539369)*x[1]
        arg[1,0,2,0]=(0.411689797339)*x[0]**o+(0.0555768292842)*x[0]+(0.786614145981)*x[1]**o+(-0.701418967065)*x[1]
        arg[1,0,2,1]=(0.971933572797)*x[0]**o+(0.203165120814)*x[0]+(0.50340688009)*x[1]**o+(-0.443869607992)*x[1]
        arg[1,1,0,0]=(0.517782412737)*x[0]**o+(-0.233071827503)*x[0]+(0.310115588175)*x[1]**o+(-0.970604318783)*x[1]
        arg[1,1,0,1]=(0.250379154253)*x[0]**o+(-0.0914513321258)*x[0]+(0.466403827523)*x[1]**o+(-0.454369386353)*x[1]
        arg[1,1,1,0]=(-0.235520035004)*x[0]**o+(0.0932661387959)*x[0]+(0.0964933173918)*x[1]**o+(0.585245824454)*x[1]
        arg[1,1,1,1]=(-0.466145341257)*x[0]**o+(0.532910678544)*x[0]+(0.757545184326)*x[1]**o+(-0.488241176428)*x[1]
        arg[1,1,2,0]=(0.802070127079)*x[0]**o+(0.30227826497)*x[0]+(0.76818718588)*x[1]**o+(-0.966380392422)*x[1]
        arg[1,1,2,1]=(0.764411747219)*x[0]**o+(0.62151105703)*x[0]+(0.310875960956)*x[1]**o+(0.218417169467)*x[1]
        arg[1,2,0,0]=(0.627865841486)*x[0]**o+(0.328844171432)*x[0]+(0.47405718918)*x[1]**o+(0.238471462812)*x[1]
        arg[1,2,0,1]=(-0.890402471541)*x[0]**o+(0.0654874441479)*x[0]+(0.411199516291)*x[1]**o+(0.0669990208336)*x[1]
        arg[1,2,1,0]=(-0.789401901979)*x[0]**o+(-0.379725885915)*x[0]+(-0.103817842125)*x[1]**o+(0.358055283478)*x[1]
        arg[1,2,1,1]=(-0.619688531158)*x[0]**o+(0.314022911152)*x[0]+(-0.515963904961)*x[1]**o+(0.807912272743)*x[1]
        arg[1,2,2,0]=(0.696584946679)*x[0]**o+(-0.156405028983)*x[0]+(0.99374085906)*x[1]**o+(0.692362331558)*x[1]
        arg[1,2,2,1]=(0.463393460998)*x[0]**o+(-0.0302433372264)*x[0]+(0.243006695756)*x[1]**o+(0.934578685307)*x[1]
        arg[1,3,0,0]=(-0.760149691585)*x[0]**o+(0.305312987392)*x[0]+(0.478665319824)*x[1]**o+(-0.395734016727)*x[1]
        arg[1,3,0,1]=(-0.495638457166)*x[0]**o+(-0.352868042523)*x[0]+(0.0577132713041)*x[1]**o+(0.856507718332)*x[1]
        arg[1,3,1,0]=(-0.588160041725)*x[0]**o+(-0.941242199161)*x[0]+(-0.883563055172)*x[1]**o+(-0.474349426459)*x[1]
        arg[1,3,1,1]=(-0.822752260982)*x[0]**o+(0.284881595545)*x[0]+(-0.914198363943)*x[1]**o+(0.421107500666)*x[1]
        arg[1,3,2,0]=(-0.125099230328)*x[0]**o+(-0.0167977306674)*x[0]+(-0.880147366581)*x[1]**o+(0.947650728684)*x[1]
        arg[1,3,2,1]=(-0.770895899992)*x[0]**o+(-0.121705997003)*x[0]+(-0.0486858273347)*x[1]**o+(0.666179109672)*x[1]
        arg[1,4,0,0]=(-0.767476534476)*x[0]**o+(0.720013119195)*x[0]+(-0.540220631624)*x[1]**o+(-0.363702385125)*x[1]
        arg[1,4,0,1]=(0.90195509915)*x[0]**o+(0.104569298687)*x[0]+(0.901816813435)*x[1]**o+(0.421060929159)*x[1]
        arg[1,4,1,0]=(0.0901631681262)*x[0]**o+(-0.330570875574)*x[0]+(0.892147778885)*x[1]**o+(-0.102976747188)*x[1]
        arg[1,4,1,1]=(-0.336016437275)*x[0]**o+(-0.00913798206221)*x[0]+(-0.20211674106)*x[1]**o+(-0.0470039081275)*x[1]
        arg[1,4,2,0]=(0.194383418491)*x[0]**o+(-0.913993330791)*x[0]+(-0.0207457431142)*x[1]**o+(-0.728130242724)*x[1]
        arg[1,4,2,1]=(-0.285983741049)*x[0]**o+(-0.359824200302)*x[0]+(-0.286913656927)*x[1]**o+(-0.801477464894)*x[1]
        arg[2,0,0,0]=(-0.266085322677)*x[0]**o+(-0.477036323188)*x[0]+(0.967304544659)*x[1]**o+(-0.843721203165)*x[1]
        arg[2,0,0,1]=(-0.431692878995)*x[0]**o+(-0.326793076488)*x[0]+(0.478220893643)*x[1]**o+(-0.953607426852)*x[1]
        arg[2,0,1,0]=(0.800398295672)*x[0]**o+(-0.726923243756)*x[0]+(0.641194491232)*x[1]**o+(0.933755071699)*x[1]
        arg[2,0,1,1]=(0.778158607269)*x[0]**o+(0.192895772287)*x[0]+(0.160540751687)*x[1]**o+(0.132780133987)*x[1]
        arg[2,0,2,0]=(0.0700487728725)*x[0]**o+(0.366571342769)*x[0]+(0.213123464402)*x[1]**o+(0.0224970803568)*x[1]
        arg[2,0,2,1]=(-0.608321696505)*x[0]**o+(0.416761589908)*x[0]+(0.0499280394381)*x[1]**o+(-0.678150075911)*x[1]
        arg[2,1,0,0]=(-0.580987690899)*x[0]**o+(-0.625951863984)*x[0]+(0.0064094419355)*x[1]**o+(0.646045286912)*x[1]
        arg[2,1,0,1]=(-0.881255703221)*x[0]**o+(-0.153796152409)*x[0]+(0.320566497467)*x[1]**o+(-0.217827916224)*x[1]
        arg[2,1,1,0]=(0.981593086009)*x[0]**o+(0.659643450249)*x[0]+(0.824530802391)*x[1]**o+(0.109125926037)*x[1]
        arg[2,1,1,1]=(-0.79695617371)*x[0]**o+(-0.340030091407)*x[0]+(0.476648191338)*x[1]**o+(-0.751961375342)*x[1]
        arg[2,1,2,0]=(0.00785807621726)*x[0]**o+(-0.177037931946)*x[0]+(-0.322685883101)*x[1]**o+(0.639485466195)*x[1]
        arg[2,1,2,1]=(0.386667953213)*x[0]**o+(0.805115373435)*x[0]+(0.489529148574)*x[1]**o+(0.380650996708)*x[1]
        arg[2,2,0,0]=(-0.205678281473)*x[0]**o+(0.725798252716)*x[0]+(-0.884620261631)*x[1]**o+(0.737678769923)*x[1]
        arg[2,2,0,1]=(-0.859841848792)*x[0]**o+(0.215497396508)*x[0]+(-0.9231710768)*x[1]**o+(-0.504914651252)*x[1]
        arg[2,2,1,0]=(-0.0161889255216)*x[0]**o+(0.218150095097)*x[0]+(0.508505799895)*x[1]**o+(0.441270271663)*x[1]
        arg[2,2,1,1]=(0.645494826531)*x[0]**o+(0.812462584734)*x[0]+(0.498639261012)*x[1]**o+(0.695415642432)*x[1]
        arg[2,2,2,0]=(-0.855342734108)*x[0]**o+(0.174928383231)*x[0]+(-0.550636347213)*x[1]**o+(0.814295216919)*x[1]
        arg[2,2,2,1]=(-0.146152468476)*x[0]**o+(-0.810876776267)*x[0]+(0.77444775812)*x[1]**o+(0.758171250375)*x[1]
        arg[2,3,0,0]=(0.914771099535)*x[0]**o+(-0.801103658802)*x[0]+(-0.339761119169)*x[1]**o+(-0.0697409372498)*x[1]
        arg[2,3,0,1]=(-0.278017621273)*x[0]**o+(-0.916896227964)*x[0]+(0.0151205004002)*x[1]**o+(-0.36424189479)*x[1]
        arg[2,3,1,0]=(0.635845368235)*x[0]**o+(-0.286420165946)*x[0]+(-0.547314386502)*x[1]**o+(-0.297882680065)*x[1]
        arg[2,3,1,1]=(0.236860892332)*x[0]**o+(0.670289851551)*x[0]+(-0.437288046493)*x[1]**o+(0.0610684872034)*x[1]
        arg[2,3,2,0]=(0.096479176127)*x[0]**o+(-0.992608241155)*x[0]+(0.622650933171)*x[1]**o+(-0.171594052732)*x[1]
        arg[2,3,2,1]=(0.259913661654)*x[0]**o+(0.521353257934)*x[0]+(0.141212812345)*x[1]**o+(0.0311971708742)*x[1]
        arg[2,4,0,0]=(0.208498140469)*x[0]**o+(0.914285328223)*x[0]+(-0.368860713612)*x[1]**o+(0.744057640483)*x[1]
        arg[2,4,0,1]=(-0.0760701135919)*x[0]**o+(-0.0666410245297)*x[0]+(-0.442858585142)*x[1]**o+(-0.869273907698)*x[1]
        arg[2,4,1,0]=(0.694291134331)*x[0]**o+(0.153487646161)*x[0]+(-0.454726570824)*x[1]**o+(0.12533511793)*x[1]
        arg[2,4,1,1]=(0.505643124198)*x[0]**o+(0.935208927517)*x[0]+(0.120271053976)*x[1]**o+(0.472499209985)*x[1]
        arg[2,4,2,0]=(0.458890639035)*x[0]**o+(-0.38125264637)*x[0]+(-0.742525095759)*x[1]**o+(0.0727341692272)*x[1]
        arg[2,4,2,1]=(-0.228856512525)*x[0]**o+(0.215540806099)*x[0]+(-0.492076254398)*x[1]**o+(0.325121281149)*x[1]
        arg[3,0,0,0]=(0.791320435019)*x[0]**o+(0.264208069532)*x[0]+(0.105218141263)*x[1]**o+(-0.947073121631)*x[1]
        arg[3,0,0,1]=(0.384081306848)*x[0]**o+(-0.735420806741)*x[0]+(0.695451719819)*x[1]**o+(-0.104071285853)*x[1]
        arg[3,0,1,0]=(0.176280431402)*x[0]**o+(0.766248516714)*x[0]+(0.620920827635)*x[1]**o+(0.571778731591)*x[1]
        arg[3,0,1,1]=(0.895148060637)*x[0]**o+(0.120401904357)*x[0]+(0.500131556768)*x[1]**o+(-0.432523916076)*x[1]
        arg[3,0,2,0]=(-0.171375065793)*x[0]**o+(0.731777853738)*x[0]+(0.914774635339)*x[1]**o+(-0.329504614878)*x[1]
        arg[3,0,2,1]=(-0.65523706909)*x[0]**o+(-0.309487164889)*x[0]+(-0.244763521442)*x[1]**o+(0.790744116827)*x[1]
        arg[3,1,0,0]=(-0.515742288)*x[0]**o+(0.688021032329)*x[0]+(0.744558730681)*x[1]**o+(0.663410706435)*x[1]
        arg[3,1,0,1]=(0.299920493853)*x[0]**o+(0.221554378219)*x[0]+(-0.262111371211)*x[1]**o+(0.597219370531)*x[1]
        arg[3,1,1,0]=(-0.491648532317)*x[0]**o+(-0.0763921400417)*x[0]+(0.517994656474)*x[1]**o+(0.690649210498)*x[1]
        arg[3,1,1,1]=(0.291881729319)*x[0]**o+(0.388296683726)*x[0]+(0.304223657331)*x[1]**o+(-0.17249216188)*x[1]
        arg[3,1,2,0]=(0.309288981861)*x[0]**o+(-0.638172982762)*x[0]+(-0.566768084047)*x[1]**o+(-0.694439437371)*x[1]
        arg[3,1,2,1]=(-0.680391436031)*x[0]**o+(0.876096306554)*x[0]+(0.676979754874)*x[1]**o+(0.376507786465)*x[1]
        arg[3,2,0,0]=(-0.688464200232)*x[0]**o+(0.404140014906)*x[0]+(-0.389359624725)*x[1]**o+(0.218257423404)*x[1]
        arg[3,2,0,1]=(-0.321661287857)*x[0]**o+(0.305707842777)*x[0]+(0.0488067596906)*x[1]**o+(0.475071042874)*x[1]
        arg[3,2,1,0]=(-0.172172832131)*x[0]**o+(-0.0589598637728)*x[0]+(0.997578239775)*x[1]**o+(0.0919614884141)*x[1]
        arg[3,2,1,1]=(0.500590025777)*x[0]**o+(0.258724052781)*x[0]+(0.233819575609)*x[1]**o+(0.787949345995)*x[1]
        arg[3,2,2,0]=(-0.464931198743)*x[0]**o+(0.689246395117)*x[0]+(0.0701366334357)*x[1]**o+(-0.206391031385)*x[1]
        arg[3,2,2,1]=(-0.352492545609)*x[0]**o+(-0.160385273073)*x[0]+(0.876791526094)*x[1]**o+(0.748257988817)*x[1]
        arg[3,3,0,0]=(0.70346530112)*x[0]**o+(-0.351192123239)*x[0]+(-0.985298342079)*x[1]**o+(-0.286817451723)*x[1]
        arg[3,3,0,1]=(0.841285056577)*x[0]**o+(0.741986440026)*x[0]+(-0.758645627646)*x[1]**o+(-0.914494381783)*x[1]
        arg[3,3,1,0]=(-0.614293213484)*x[0]**o+(-0.746555667296)*x[0]+(-0.301781252786)*x[1]**o+(-0.721633195357)*x[1]
        arg[3,3,1,1]=(-0.287510519978)*x[0]**o+(-0.764960517935)*x[0]+(-0.948032248565)*x[1]**o+(-0.548676002552)*x[1]
        arg[3,3,2,0]=(0.145506943729)*x[0]**o+(-0.353925293653)*x[0]+(0.786904887352)*x[1]**o+(-0.36548133551)*x[1]
        arg[3,3,2,1]=(0.60782215735)*x[0]**o+(0.872444227513)*x[0]+(-0.237571902274)*x[1]**o+(-0.0559745367747)*x[1]
        arg[3,4,0,0]=(0.691107698337)*x[0]**o+(0.0800532750053)*x[0]+(-0.847456183316)*x[1]**o+(0.911068576288)*x[1]
        arg[3,4,0,1]=(-0.00167606578449)*x[0]**o+(-0.697603805697)*x[0]+(-0.362440886816)*x[1]**o+(-0.0604379772766)*x[1]
        arg[3,4,1,0]=(-0.74026325403)*x[0]**o+(-0.539076677004)*x[0]+(-0.317176913637)*x[1]**o+(0.759776288772)*x[1]
        arg[3,4,1,1]=(-0.230005454291)*x[0]**o+(-0.929557352119)*x[0]+(-0.435125210116)*x[1]**o+(0.106892135893)*x[1]
        arg[3,4,2,0]=(-0.638149620463)*x[0]**o+(-0.0511424128986)*x[0]+(0.460436795546)*x[1]**o+(-0.919535518181)*x[1]
        arg[3,4,2,1]=(-0.793537245531)*x[0]**o+(0.409768940347)*x[0]+(0.632560678557)*x[1]**o+(0.700596370851)*x[1]
        ref[0,0,0,0]=(-0.596353540657)/(o+1.)+(0.420599412158)+(-0.348972036919)*0.5**o
        ref[0,0,0,1]=(0.261409664458)/(o+1.)+(0.391909190657)+(0.187733014896)*0.5**o
        ref[0,0,1,0]=(0.59127040848)/(o+1.)+(-0.563448202311)+(0.951655197513)*0.5**o
        ref[0,0,1,1]=(0.77162116582)/(o+1.)+(-0.319989986697)+(-0.188127934304)*0.5**o
        ref[0,0,2,0]=(0.908566805373)/(o+1.)+(-0.619969504781)+(0.770717242337)*0.5**o
        ref[0,0,2,1]=(-0.724584621389)/(o+1.)+(0.474757060008)+(-0.543253359807)*0.5**o
        ref[0,1,0,0]=(0.520162431169)/(o+1.)+(0.271653771717)+(0.986939140609)*0.5**o
        ref[0,1,0,1]=(0.43818378458)/(o+1.)+(-0.372085975318)+(-0.171773212929)*0.5**o
        ref[0,1,1,0]=(0.422340945702)/(o+1.)+(0.458371792807)+(-0.369350989684)*0.5**o
        ref[0,1,1,1]=(-0.0464190048363)/(o+1.)+(0.0694711771999)+(0.559606389467)*0.5**o
        ref[0,1,2,0]=(-0.347984444559)/(o+1.)+(-0.355420275708)+(-0.989408371198)*0.5**o
        ref[0,1,2,1]=(-0.124928198101)/(o+1.)+(0.0306015328506)+(-0.289295900777)*0.5**o
        ref[0,2,0,0]=(0.446169236302)/(o+1.)+(-0.473078979432)+(0.132599667548)*0.5**o
        ref[0,2,0,1]=(-0.258525519083)/(o+1.)+(-0.0431484138534)+(-0.452662917924)*0.5**o
        ref[0,2,1,0]=(-0.209089266173)/(o+1.)+(0.742394708205)+(0.318740871592)*0.5**o
        ref[0,2,1,1]=(0.954836398565)/(o+1.)+(-0.13832175844)+(-0.825757440563)*0.5**o
        ref[0,2,2,0]=(-0.0327529254483)/(o+1.)+(0.267659992607)+(-0.640337007803)*0.5**o
        ref[0,2,2,1]=(0.312023159058)/(o+1.)+(0.612331584205)+(-0.956300293885)*0.5**o
        ref[0,3,0,0]=(-0.431859510779)/(o+1.)+(-0.351884774058)+(-0.927040285049)*0.5**o
        ref[0,3,0,1]=(0.274814268019)/(o+1.)+(-0.064760471806)+(-0.807009561768)*0.5**o
        ref[0,3,1,0]=(0.130437344004)/(o+1.)+(0.328368426492)+(0.931774139846)*0.5**o
        ref[0,3,1,1]=(-0.313484462262)/(o+1.)+(-0.132757088027)+(-0.314188008746)*0.5**o
        ref[0,3,2,0]=(-0.451892315269)/(o+1.)+(0.306590170858)+(0.748691505108)*0.5**o
        ref[0,3,2,1]=(0.864166710803)/(o+1.)+(0.0632261444916)+(-0.124290384657)*0.5**o
        ref[0,4,0,0]=(-0.364735884875)/(o+1.)+(0.232568195007)+(-0.726606985714)*0.5**o
        ref[0,4,0,1]=(0.924535166786)/(o+1.)+(0.632362544326)+(-0.716457484159)*0.5**o
        ref[0,4,1,0]=(-0.738258048173)/(o+1.)+(-0.102170669441)+(-0.67154700238)*0.5**o
        ref[0,4,1,1]=(0.592223896873)/(o+1.)+(-0.160061987092)+(-0.568569022142)*0.5**o
        ref[0,4,2,0]=(0.447944032533)/(o+1.)+(-0.708884469036)+(0.959619955514)*0.5**o
        ref[0,4,2,1]=(0.383143400712)/(o+1.)+(0.0885429052198)+(-0.446578795272)*0.5**o
        ref[1,0,0,0]=(-0.871963625443)/(o+1.)+(-0.149605030268)+(0.130600745253)*0.5**o
        ref[1,0,0,1]=(0.511206122216)/(o+1.)+(-0.541100696719)+(-0.684630134211)*0.5**o
        ref[1,0,1,0]=(-0.0906465041298)/(o+1.)+(0.065786117418)+(0.624178362847)*0.5**o
        ref[1,0,1,1]=(-0.0594261878161)/(o+1.)+(0.146819884404)+(-0.403988173215)*0.5**o
        ref[1,0,2,0]=(0.786614145981)/(o+1.)+(-0.32292106889)+(0.411689797339)*0.5**o
        ref[1,0,2,1]=(0.50340688009)/(o+1.)+(-0.120352243589)+(0.971933572797)*0.5**o
        ref[1,1,0,0]=(0.310115588175)/(o+1.)+(-0.601838073143)+(0.517782412737)*0.5**o
        ref[1,1,0,1]=(0.466403827523)/(o+1.)+(-0.27291035924)+(0.250379154253)*0.5**o
        ref[1,1,1,0]=(0.0964933173918)/(o+1.)+(0.339255981625)+(-0.235520035004)*0.5**o
        ref[1,1,1,1]=(0.757545184326)/(o+1.)+(0.0223347510579)+(-0.466145341257)*0.5**o
        ref[1,1,2,0]=(0.76818718588)/(o+1.)+(-0.332051063726)+(0.802070127079)*0.5**o
        ref[1,1,2,1]=(0.310875960956)/(o+1.)+(0.419964113249)+(0.764411747219)*0.5**o
        ref[1,2,0,0]=(0.47405718918)/(o+1.)+(0.283657817122)+(0.627865841486)*0.5**o
        ref[1,2,0,1]=(0.411199516291)/(o+1.)+(0.0662432324908)+(-0.890402471541)*0.5**o
        ref[1,2,1,0]=(-0.103817842125)/(o+1.)+(-0.0108353012186)+(-0.789401901979)*0.5**o
        ref[1,2,1,1]=(-0.515963904961)/(o+1.)+(0.560967591948)+(-0.619688531158)*0.5**o
        ref[1,2,2,0]=(0.99374085906)/(o+1.)+(0.267978651288)+(0.696584946679)*0.5**o
        ref[1,2,2,1]=(0.243006695756)/(o+1.)+(0.45216767404)+(0.463393460998)*0.5**o
        ref[1,3,0,0]=(0.478665319824)/(o+1.)+(-0.0452105146678)+(-0.760149691585)*0.5**o
        ref[1,3,0,1]=(0.0577132713041)/(o+1.)+(0.251819837904)+(-0.495638457166)*0.5**o
        ref[1,3,1,0]=(-0.883563055172)/(o+1.)+(-0.70779581281)+(-0.588160041725)*0.5**o
        ref[1,3,1,1]=(-0.914198363943)/(o+1.)+(0.352994548105)+(-0.822752260982)*0.5**o
        ref[1,3,2,0]=(-0.880147366581)/(o+1.)+(0.465426499008)+(-0.125099230328)*0.5**o
        ref[1,3,2,1]=(-0.0486858273347)/(o+1.)+(0.272236556335)+(-0.770895899992)*0.5**o
        ref[1,4,0,0]=(-0.540220631624)/(o+1.)+(0.178155367035)+(-0.767476534476)*0.5**o
        ref[1,4,0,1]=(0.901816813435)/(o+1.)+(0.262815113923)+(0.90195509915)*0.5**o
        ref[1,4,1,0]=(0.892147778885)/(o+1.)+(-0.216773811381)+(0.0901631681262)*0.5**o
        ref[1,4,1,1]=(-0.20211674106)/(o+1.)+(-0.0280709450949)+(-0.336016437275)*0.5**o
        ref[1,4,2,0]=(-0.0207457431142)/(o+1.)+(-0.821061786757)+(0.194383418491)*0.5**o
        ref[1,4,2,1]=(-0.286913656927)/(o+1.)+(-0.580650832598)+(-0.285983741049)*0.5**o
        ref[2,0,0,0]=(0.967304544659)/(o+1.)+(-0.660378763176)+(-0.266085322677)*0.5**o
        ref[2,0,0,1]=(0.478220893643)/(o+1.)+(-0.64020025167)+(-0.431692878995)*0.5**o
        ref[2,0,1,0]=(0.641194491232)/(o+1.)+(0.103415913972)+(0.800398295672)*0.5**o
        ref[2,0,1,1]=(0.160540751687)/(o+1.)+(0.162837953137)+(0.778158607269)*0.5**o
        ref[2,0,2,0]=(0.213123464402)/(o+1.)+(0.194534211563)+(0.0700487728725)*0.5**o
        ref[2,0,2,1]=(0.0499280394381)/(o+1.)+(-0.130694243001)+(-0.608321696505)*0.5**o
        ref[2,1,0,0]=(0.0064094419355)/(o+1.)+(0.0100467114639)+(-0.580987690899)*0.5**o
        ref[2,1,0,1]=(0.320566497467)/(o+1.)+(-0.185812034316)+(-0.881255703221)*0.5**o
        ref[2,1,1,0]=(0.824530802391)/(o+1.)+(0.384384688143)+(0.981593086009)*0.5**o
        ref[2,1,1,1]=(0.476648191338)/(o+1.)+(-0.545995733375)+(-0.79695617371)*0.5**o
        ref[2,1,2,0]=(-0.322685883101)/(o+1.)+(0.231223767125)+(0.00785807621726)*0.5**o
        ref[2,1,2,1]=(0.489529148574)/(o+1.)+(0.592883185071)+(0.386667953213)*0.5**o
        ref[2,2,0,0]=(-0.884620261631)/(o+1.)+(0.73173851132)+(-0.205678281473)*0.5**o
        ref[2,2,0,1]=(-0.9231710768)/(o+1.)+(-0.144708627372)+(-0.859841848792)*0.5**o
        ref[2,2,1,0]=(0.508505799895)/(o+1.)+(0.32971018338)+(-0.0161889255216)*0.5**o
        ref[2,2,1,1]=(0.498639261012)/(o+1.)+(0.753939113583)+(0.645494826531)*0.5**o
        ref[2,2,2,0]=(-0.550636347213)/(o+1.)+(0.494611800075)+(-0.855342734108)*0.5**o
        ref[2,2,2,1]=(0.77444775812)/(o+1.)+(-0.0263527629459)+(-0.146152468476)*0.5**o
        ref[2,3,0,0]=(-0.339761119169)/(o+1.)+(-0.435422298026)+(0.914771099535)*0.5**o
        ref[2,3,0,1]=(0.0151205004002)/(o+1.)+(-0.640569061377)+(-0.278017621273)*0.5**o
        ref[2,3,1,0]=(-0.547314386502)/(o+1.)+(-0.292151423006)+(0.635845368235)*0.5**o
        ref[2,3,1,1]=(-0.437288046493)/(o+1.)+(0.365679169377)+(0.236860892332)*0.5**o
        ref[2,3,2,0]=(0.622650933171)/(o+1.)+(-0.582101146943)+(0.096479176127)*0.5**o
        ref[2,3,2,1]=(0.141212812345)/(o+1.)+(0.276275214404)+(0.259913661654)*0.5**o
        ref[2,4,0,0]=(-0.368860713612)/(o+1.)+(0.829171484353)+(0.208498140469)*0.5**o
        ref[2,4,0,1]=(-0.442858585142)/(o+1.)+(-0.467957466114)+(-0.0760701135919)*0.5**o
        ref[2,4,1,0]=(-0.454726570824)/(o+1.)+(0.139411382046)+(0.694291134331)*0.5**o
        ref[2,4,1,1]=(0.120271053976)/(o+1.)+(0.703854068751)+(0.505643124198)*0.5**o
        ref[2,4,2,0]=(-0.742525095759)/(o+1.)+(-0.154259238572)+(0.458890639035)*0.5**o
        ref[2,4,2,1]=(-0.492076254398)/(o+1.)+(0.270331043624)+(-0.228856512525)*0.5**o
        ref[3,0,0,0]=(0.105218141263)/(o+1.)+(-0.34143252605)+(0.791320435019)*0.5**o
        ref[3,0,0,1]=(0.695451719819)/(o+1.)+(-0.419746046297)+(0.384081306848)*0.5**o
        ref[3,0,1,0]=(0.620920827635)/(o+1.)+(0.669013624153)+(0.176280431402)*0.5**o
        ref[3,0,1,1]=(0.500131556768)/(o+1.)+(-0.15606100586)+(0.895148060637)*0.5**o
        ref[3,0,2,0]=(0.914774635339)/(o+1.)+(0.20113661943)+(-0.171375065793)*0.5**o
        ref[3,0,2,1]=(-0.244763521442)/(o+1.)+(0.240628475969)+(-0.65523706909)*0.5**o
        ref[3,1,0,0]=(0.744558730681)/(o+1.)+(0.675715869382)+(-0.515742288)*0.5**o
        ref[3,1,0,1]=(-0.262111371211)/(o+1.)+(0.409386874375)+(0.299920493853)*0.5**o
        ref[3,1,1,0]=(0.517994656474)/(o+1.)+(0.307128535228)+(-0.491648532317)*0.5**o
        ref[3,1,1,1]=(0.304223657331)/(o+1.)+(0.107902260923)+(0.291881729319)*0.5**o
        ref[3,1,2,0]=(-0.566768084047)/(o+1.)+(-0.666306210066)+(0.309288981861)*0.5**o
        ref[3,1,2,1]=(0.676979754874)/(o+1.)+(0.62630204651)+(-0.680391436031)*0.5**o
        ref[3,2,0,0]=(-0.389359624725)/(o+1.)+(0.311198719155)+(-0.688464200232)*0.5**o
        ref[3,2,0,1]=(0.0488067596906)/(o+1.)+(0.390389442826)+(-0.321661287857)*0.5**o
        ref[3,2,1,0]=(0.997578239775)/(o+1.)+(0.0165008123207)+(-0.172172832131)*0.5**o
        ref[3,2,1,1]=(0.233819575609)/(o+1.)+(0.523336699388)+(0.500590025777)*0.5**o
        ref[3,2,2,0]=(0.0701366334357)/(o+1.)+(0.241427681866)+(-0.464931198743)*0.5**o
        ref[3,2,2,1]=(0.876791526094)/(o+1.)+(0.293936357872)+(-0.352492545609)*0.5**o
        ref[3,3,0,0]=(-0.985298342079)/(o+1.)+(-0.319004787481)+(0.70346530112)*0.5**o
        ref[3,3,0,1]=(-0.758645627646)/(o+1.)+(-0.0862539708788)+(0.841285056577)*0.5**o
        ref[3,3,1,0]=(-0.301781252786)/(o+1.)+(-0.734094431327)+(-0.614293213484)*0.5**o
        ref[3,3,1,1]=(-0.948032248565)/(o+1.)+(-0.656818260243)+(-0.287510519978)*0.5**o
        ref[3,3,2,0]=(0.786904887352)/(o+1.)+(-0.359703314582)+(0.145506943729)*0.5**o
        ref[3,3,2,1]=(-0.237571902274)/(o+1.)+(0.408234845369)+(0.60782215735)*0.5**o
        ref[3,4,0,0]=(-0.847456183316)/(o+1.)+(0.495560925646)+(0.691107698337)*0.5**o
        ref[3,4,0,1]=(-0.362440886816)/(o+1.)+(-0.379020891487)+(-0.00167606578449)*0.5**o
        ref[3,4,1,0]=(-0.317176913637)/(o+1.)+(0.110349805884)+(-0.74026325403)*0.5**o
        ref[3,4,1,1]=(-0.435125210116)/(o+1.)+(-0.411332608113)+(-0.230005454291)*0.5**o
        ref[3,4,2,0]=(0.460436795546)/(o+1.)+(-0.48533896554)+(-0.638149620463)*0.5**o
        ref[3,4,2,1]=(0.632560678557)/(o+1.)+(0.555182655599)+(-0.793537245531)*0.5**o
      else:
        arg[0,0,0,0]=(0.816458625566)*x[0]**o+(-0.438180707327)*x[0]+(-0.39836196112)*x[1]**o+(0.641701267855)*x[1]+(0.919064096208)*x[2]**o+(0.0155653082454)*x[2]
        arg[0,0,0,1]=(-0.709065020122)*x[0]**o+(-0.787938005558)*x[0]+(-0.459829890247)*x[1]**o+(-0.306587811189)*x[1]+(0.648348776519)*x[2]**o+(-0.782857657922)*x[2]
        arg[0,0,1,0]=(-0.09165636265)*x[0]**o+(-0.215464577259)*x[0]+(-0.445060581145)*x[1]**o+(0.541797069021)*x[1]+(-0.312096860473)*x[2]**o+(0.307061070632)*x[2]
        arg[0,0,1,1]=(0.900670566988)*x[0]**o+(-0.157226841228)*x[0]+(-0.00514019601556)*x[1]**o+(0.167616993369)*x[1]+(0.331077209212)*x[2]**o+(0.0797784869412)*x[2]
        arg[0,0,2,0]=(-0.883617789072)*x[0]**o+(-0.233507045302)*x[0]+(0.540311388635)*x[1]**o+(-0.0698351272775)*x[1]+(0.598059668701)*x[2]**o+(-0.842961707358)*x[2]
        arg[0,0,2,1]=(-0.770041366552)*x[0]**o+(-0.683157969301)*x[0]+(0.235580399076)*x[1]**o+(-0.994177047985)*x[1]+(-0.90074656484)*x[2]**o+(-0.0133600820046)*x[2]
        arg[0,1,0,0]=(0.90396926296)*x[0]**o+(0.0492056402145)*x[0]+(0.32744155595)*x[1]**o+(0.0162505824727)*x[1]+(0.386607661085)*x[2]**o+(0.759639829913)*x[2]
        arg[0,1,0,1]=(0.47956310963)*x[0]**o+(0.638067028609)*x[0]+(-0.520596788355)*x[1]**o+(0.476897948019)*x[1]+(0.950320232053)*x[2]**o+(-0.603034001297)*x[2]
        arg[0,1,1,0]=(-0.75177304495)*x[0]**o+(0.369014786745)*x[0]+(-0.332220011735)*x[1]**o+(-0.835059332186)*x[1]+(0.0494323092557)*x[2]**o+(-0.796200422375)*x[2]
        arg[0,1,1,1]=(-0.109937058453)*x[0]**o+(0.607637390544)*x[0]+(0.464147925564)*x[1]**o+(0.650325883743)*x[1]+(0.698196425121)*x[2]**o+(0.933002044297)*x[2]
        arg[0,1,2,0]=(-0.13801307581)*x[0]**o+(0.75768744715)*x[0]+(-0.35990643116)*x[1]**o+(-0.787975707318)*x[1]+(-0.259281382264)*x[2]**o+(0.971003238549)*x[2]
        arg[0,1,2,1]=(-0.518540484981)*x[0]**o+(0.197912267942)*x[0]+(0.481058074022)*x[1]**o+(0.918189335061)*x[1]+(0.160239271801)*x[2]**o+(0.75950986792)*x[2]
        arg[0,2,0,0]=(0.286922773164)*x[0]**o+(-0.874592150595)*x[0]+(0.416014232771)*x[1]**o+(-0.492379239759)*x[1]+(0.950640982197)*x[2]**o+(-0.10625158241)*x[2]
        arg[0,2,0,1]=(-0.116447648923)*x[0]**o+(-0.499481854319)*x[0]+(-0.556602044084)*x[1]**o+(0.753256358343)*x[1]+(0.280780022694)*x[2]**o+(-0.879728755507)*x[2]
        arg[0,2,1,0]=(-0.322803067618)*x[0]**o+(0.0974878233964)*x[0]+(0.420547949274)*x[1]**o+(0.304839957099)*x[1]+(-0.12605355567)*x[2]**o+(-0.166970509739)*x[2]
        arg[0,2,1,1]=(0.997823909059)*x[0]**o+(0.76749010767)*x[0]+(0.376351544673)*x[1]**o+(-0.316937252621)*x[1]+(-0.627670639733)*x[2]**o+(0.584121037987)*x[2]
        arg[0,2,2,0]=(0.516866019209)*x[0]**o+(0.413564799736)*x[0]+(-0.0730736747833)*x[1]**o+(-0.500984764568)*x[1]+(-0.989323894645)*x[2]**o+(-0.261458643035)*x[2]
        arg[0,2,2,1]=(0.777538237337)*x[0]**o+(-0.0623276952959)*x[0]+(0.167068351133)*x[1]**o+(-0.345550070545)*x[1]+(-0.28166577069)*x[2]**o+(-0.568650037569)*x[2]
        arg[0,3,0,0]=(-0.622067362782)*x[0]**o+(0.971384323624)*x[0]+(-0.099597286042)*x[1]**o+(-0.265680407306)*x[1]+(-0.446275631545)*x[2]**o+(-0.0637591060995)*x[2]
        arg[0,3,0,1]=(-0.49698112241)*x[0]**o+(0.766181263321)*x[0]+(-0.977014162391)*x[1]**o+(0.806444170647)*x[1]+(-0.441023882645)*x[2]**o+(-0.919279365683)*x[2]
        arg[0,3,1,0]=(0.917091542097)*x[0]**o+(-0.67016353228)*x[0]+(0.0468346020734)*x[1]**o+(0.81092419675)*x[1]+(0.174037096652)*x[2]**o+(0.564190407348)*x[2]
        arg[0,3,1,1]=(-0.437188100982)*x[0]**o+(-0.0857174204982)*x[0]+(-0.971592105442)*x[1]**o+(-0.122352019844)*x[1]+(-0.853225566811)*x[2]**o+(-0.861128476903)*x[2]
        arg[0,3,2,0]=(-0.841265485658)*x[0]**o+(0.553186811235)*x[0]+(0.414489259007)*x[1]**o+(-0.800997281522)*x[1]+(0.630269543743)*x[2]**o+(0.263584335528)*x[2]
        arg[0,3,2,1]=(0.383282548921)*x[0]**o+(0.260041562865)*x[0]+(-0.95180999378)*x[1]**o+(0.566954668715)*x[1]+(0.688843907846)*x[2]**o+(-0.166837115553)*x[2]
        arg[0,4,0,0]=(0.0385891845805)*x[0]**o+(0.324932244772)*x[0]+(-0.946997425504)*x[1]**o+(-0.745439090316)*x[1]+(-0.721657519976)*x[2]**o+(-0.714366519961)*x[2]
        arg[0,4,0,1]=(-0.387739686761)*x[0]**o+(-0.860915144676)*x[0]+(0.270056169204)*x[1]**o+(-0.914861208322)*x[1]+(-0.453514265988)*x[2]**o+(-0.495768314604)*x[2]
        arg[0,4,1,0]=(-0.329464483421)*x[0]**o+(0.313424566493)*x[0]+(0.185677682455)*x[1]**o+(-0.720200750426)*x[1]+(0.425742092834)*x[2]**o+(0.505209526425)*x[2]
        arg[0,4,1,1]=(0.63272559561)*x[0]**o+(-0.026429919604)*x[0]+(-0.14023723263)*x[1]**o+(-0.580805084228)*x[1]+(0.141232832837)*x[2]**o+(0.241327430534)*x[2]
        arg[0,4,2,0]=(-0.847582850604)*x[0]**o+(0.207270297356)*x[0]+(0.851958035148)*x[1]**o+(0.336594484304)*x[1]+(-0.221434386211)*x[2]**o+(-0.607602400875)*x[2]
        arg[0,4,2,1]=(-0.722990535113)*x[0]**o+(-0.379964881881)*x[0]+(0.825779173038)*x[1]**o+(-0.285509543358)*x[1]+(-0.518809494335)*x[2]**o+(-0.788507606636)*x[2]
        arg[1,0,0,0]=(0.0725951752952)*x[0]**o+(-0.282800153416)*x[0]+(-0.0228303688498)*x[1]**o+(-0.863127549259)*x[1]+(-0.383553228278)*x[2]**o+(-0.0151737809185)*x[2]
        arg[1,0,0,1]=(-0.254920010173)*x[0]**o+(0.160195189848)*x[0]+(0.216407190109)*x[1]**o+(-0.814599655672)*x[1]+(0.771746025302)*x[2]**o+(0.0636221760766)*x[2]
        arg[1,0,1,0]=(0.486342223622)*x[0]**o+(0.736242651359)*x[0]+(0.720722697611)*x[1]**o+(0.974853194395)*x[1]+(-0.834707965096)*x[2]**o+(-0.947169299038)*x[2]
        arg[1,0,1,1]=(0.777015150638)*x[0]**o+(-0.228082772046)*x[0]+(-0.51095285588)*x[1]**o+(-0.835057692508)*x[1]+(-0.564012944223)*x[2]**o+(-0.898040787549)*x[2]
        arg[1,0,2,0]=(0.240813275278)*x[0]**o+(-0.509566588932)*x[0]+(0.247803237451)*x[1]**o+(0.156442781059)*x[1]+(0.618367580757)*x[2]**o+(0.943435734207)*x[2]
        arg[1,0,2,1]=(-0.670257601629)*x[0]**o+(-0.470716006203)*x[0]+(-0.559207336128)*x[1]**o+(-0.81577896974)*x[1]+(0.0571093198914)*x[2]**o+(0.477103420898)*x[2]
        arg[1,1,0,0]=(0.985895581872)*x[0]**o+(-0.00359465828149)*x[0]+(0.00739312124399)*x[1]**o+(-0.931299968425)*x[1]+(-0.10047347666)*x[2]**o+(0.410589913002)*x[2]
        arg[1,1,0,1]=(-0.954132399275)*x[0]**o+(0.159222338169)*x[0]+(-0.865370419503)*x[1]**o+(-0.744696059283)*x[1]+(-0.794993976252)*x[2]**o+(0.961664562839)*x[2]
        arg[1,1,1,0]=(0.462810965625)*x[0]**o+(0.927378551017)*x[0]+(-0.105005418179)*x[1]**o+(0.282340667975)*x[1]+(-0.443931241323)*x[2]**o+(0.740071617715)*x[2]
        arg[1,1,1,1]=(0.645629353541)*x[0]**o+(0.848804271896)*x[0]+(-0.882497356826)*x[1]**o+(-0.74661619875)*x[1]+(-0.927133238344)*x[2]**o+(-0.502758459229)*x[2]
        arg[1,1,2,0]=(-0.554993653726)*x[0]**o+(-0.473286752336)*x[0]+(0.759624773306)*x[1]**o+(0.809169103932)*x[1]+(0.657766085675)*x[2]**o+(0.763913478843)*x[2]
        arg[1,1,2,1]=(0.1476587556)*x[0]**o+(0.385711162597)*x[0]+(-0.0767252754622)*x[1]**o+(-0.544968629318)*x[1]+(0.16751959089)*x[2]**o+(-0.742697412247)*x[2]
        arg[1,2,0,0]=(-0.715887407694)*x[0]**o+(0.273818352078)*x[0]+(0.0426629688432)*x[1]**o+(-0.808531265186)*x[1]+(-0.699839650334)*x[2]**o+(-0.332093073009)*x[2]
        arg[1,2,0,1]=(0.48440337855)*x[0]**o+(0.765443731546)*x[0]+(-0.0761968055499)*x[1]**o+(0.0516389293584)*x[1]+(-0.486224925751)*x[2]**o+(-0.796105709176)*x[2]
        arg[1,2,1,0]=(0.621562227153)*x[0]**o+(0.907213416636)*x[0]+(0.591392192921)*x[1]**o+(-0.825911160831)*x[1]+(0.690773959314)*x[2]**o+(-0.744144678185)*x[2]
        arg[1,2,1,1]=(0.472846565983)*x[0]**o+(-0.837395663533)*x[0]+(0.526438435448)*x[1]**o+(0.310053110281)*x[1]+(0.634301291566)*x[2]**o+(0.608270679412)*x[2]
        arg[1,2,2,0]=(-0.212000572333)*x[0]**o+(0.644218219058)*x[0]+(0.827779140409)*x[1]**o+(0.563968121281)*x[1]+(-0.0556319977635)*x[2]**o+(0.98126338415)*x[2]
        arg[1,2,2,1]=(-0.632752637148)*x[0]**o+(-0.999490611532)*x[0]+(0.0717950178432)*x[1]**o+(-0.0611949850049)*x[1]+(0.740177288895)*x[2]**o+(0.241010363236)*x[2]
        arg[1,3,0,0]=(0.390081216336)*x[0]**o+(-0.389460065845)*x[0]+(0.0812808695521)*x[1]**o+(0.0843698990866)*x[1]+(0.672844227186)*x[2]**o+(0.306345015188)*x[2]
        arg[1,3,0,1]=(-0.402887151891)*x[0]**o+(-0.908615473265)*x[0]+(-0.296307155209)*x[1]**o+(0.228858413043)*x[1]+(0.509401868055)*x[2]**o+(-0.793787805038)*x[2]
        arg[1,3,1,0]=(0.253947492129)*x[0]**o+(0.401465417831)*x[0]+(-0.942329176703)*x[1]**o+(-0.518371272083)*x[1]+(0.90142806242)*x[2]**o+(0.694084281869)*x[2]
        arg[1,3,1,1]=(-0.435656267291)*x[0]**o+(-0.743058541536)*x[0]+(-0.22434079396)*x[1]**o+(-0.414125965702)*x[1]+(0.743546448275)*x[2]**o+(0.323188772386)*x[2]
        arg[1,3,2,0]=(-0.069319863146)*x[0]**o+(-0.0749969565014)*x[0]+(0.29700943253)*x[1]**o+(0.972639470201)*x[1]+(-0.270592469279)*x[2]**o+(-0.355098058901)*x[2]
        arg[1,3,2,1]=(0.741825647399)*x[0]**o+(0.542055833087)*x[0]+(-0.0684409109262)*x[1]**o+(0.842094761143)*x[1]+(-0.777320219447)*x[2]**o+(0.758570478904)*x[2]
        arg[1,4,0,0]=(-0.66003338591)*x[0]**o+(-0.393755284573)*x[0]+(0.617154869065)*x[1]**o+(-0.480999198967)*x[1]+(0.350625972066)*x[2]**o+(0.733472524967)*x[2]
        arg[1,4,0,1]=(-0.0001759210396)*x[0]**o+(-0.0900574950432)*x[0]+(-0.592936627579)*x[1]**o+(-0.647214933825)*x[1]+(-0.43249743908)*x[2]**o+(-0.901387692551)*x[2]
        arg[1,4,1,0]=(0.052435899043)*x[0]**o+(0.163326392489)*x[0]+(0.433169876656)*x[1]**o+(0.977133611649)*x[1]+(-0.311959404527)*x[2]**o+(0.297218570651)*x[2]
        arg[1,4,1,1]=(0.00806522196579)*x[0]**o+(-0.121807466189)*x[0]+(-0.176557734776)*x[1]**o+(0.865146379005)*x[1]+(-0.304648040177)*x[2]**o+(0.269150163824)*x[2]
        arg[1,4,2,0]=(0.440212388104)*x[0]**o+(-0.590292628428)*x[0]+(-0.595359323633)*x[1]**o+(0.606734888384)*x[1]+(-0.951693103773)*x[2]**o+(-0.539741612207)*x[2]
        arg[1,4,2,1]=(0.346121727918)*x[0]**o+(0.188961570963)*x[0]+(0.402818736255)*x[1]**o+(-0.474029523974)*x[1]+(-0.95972237672)*x[2]**o+(0.983543070357)*x[2]
        arg[2,0,0,0]=(-0.801941188348)*x[0]**o+(-0.318215868754)*x[0]+(0.322031073259)*x[1]**o+(-0.00368402006147)*x[1]+(0.473846239338)*x[2]**o+(0.512752561753)*x[2]
        arg[2,0,0,1]=(0.786459471841)*x[0]**o+(0.941049841845)*x[0]+(0.064717660139)*x[1]**o+(-0.891518891268)*x[1]+(-0.317681565962)*x[2]**o+(-0.672310725168)*x[2]
        arg[2,0,1,0]=(0.813571093948)*x[0]**o+(-0.146551267883)*x[0]+(0.942106359835)*x[1]**o+(0.165543949281)*x[1]+(-0.0301736488709)*x[2]**o+(0.936915469225)*x[2]
        arg[2,0,1,1]=(-0.0298664722355)*x[0]**o+(-0.596644819792)*x[0]+(-0.306728572634)*x[1]**o+(0.770671782936)*x[1]+(-0.334606536003)*x[2]**o+(0.888940190159)*x[2]
        arg[2,0,2,0]=(0.688553055025)*x[0]**o+(-0.241541868482)*x[0]+(-0.694609526571)*x[1]**o+(-0.509896457784)*x[1]+(-0.0610272843414)*x[2]**o+(-0.251290389152)*x[2]
        arg[2,0,2,1]=(-0.959007707329)*x[0]**o+(-0.193856482901)*x[0]+(-0.138697126551)*x[1]**o+(0.768166305153)*x[1]+(0.951006645306)*x[2]**o+(-0.533986863953)*x[2]
        arg[2,1,0,0]=(0.648021743651)*x[0]**o+(-0.839074631201)*x[0]+(-0.445337118707)*x[1]**o+(0.997475278653)*x[1]+(-0.154475323496)*x[2]**o+(-0.45292992537)*x[2]
        arg[2,1,0,1]=(-0.134785884326)*x[0]**o+(-0.906738442209)*x[0]+(0.00509900741927)*x[1]**o+(0.489865232888)*x[1]+(0.620823018353)*x[2]**o+(-0.480127263607)*x[2]
        arg[2,1,1,0]=(-0.210371458935)*x[0]**o+(0.308453655681)*x[0]+(-0.510473100104)*x[1]**o+(-0.0776488834445)*x[1]+(0.58103002434)*x[2]**o+(0.039267622777)*x[2]
        arg[2,1,1,1]=(-0.124204683462)*x[0]**o+(0.429525935007)*x[0]+(0.183174600008)*x[1]**o+(-0.665914708196)*x[1]+(-0.576713549033)*x[2]**o+(-0.3788567811)*x[2]
        arg[2,1,2,0]=(-0.864065825608)*x[0]**o+(0.644046059464)*x[0]+(-0.265959280151)*x[1]**o+(0.989465634246)*x[1]+(0.536466956842)*x[2]**o+(0.689999246325)*x[2]
        arg[2,1,2,1]=(0.37622234855)*x[0]**o+(-0.905554688955)*x[0]+(-0.0258037833143)*x[1]**o+(-0.23415538846)*x[1]+(-0.64461795706)*x[2]**o+(0.262259040646)*x[2]
        arg[2,2,0,0]=(0.0541619627989)*x[0]**o+(-0.978918782799)*x[0]+(0.478333900479)*x[1]**o+(0.958652455667)*x[1]+(-0.746701804693)*x[2]**o+(-0.557405653804)*x[2]
        arg[2,2,0,1]=(0.611472347101)*x[0]**o+(-0.409754132769)*x[0]+(0.291274496706)*x[1]**o+(0.901983238042)*x[1]+(-0.102917678824)*x[2]**o+(-0.668205224505)*x[2]
        arg[2,2,1,0]=(-0.926062469523)*x[0]**o+(0.101032437783)*x[0]+(-0.385485330853)*x[1]**o+(0.0521756546126)*x[1]+(0.132320977758)*x[2]**o+(0.471619572767)*x[2]
        arg[2,2,1,1]=(0.0689111652346)*x[0]**o+(-0.517070173331)*x[0]+(-0.599184868344)*x[1]**o+(-0.814239996638)*x[1]+(0.97403158037)*x[2]**o+(-0.546843890524)*x[2]
        arg[2,2,2,0]=(-0.133251706696)*x[0]**o+(-0.0954736405212)*x[0]+(0.780584398494)*x[1]**o+(-0.505305860913)*x[1]+(-0.11518705364)*x[2]**o+(0.0855698333433)*x[2]
        arg[2,2,2,1]=(-0.567309977436)*x[0]**o+(-0.284182307402)*x[0]+(0.220302202269)*x[1]**o+(-0.252628588826)*x[1]+(-0.368390646272)*x[2]**o+(-0.577499260947)*x[2]
        arg[2,3,0,0]=(0.477968626266)*x[0]**o+(0.522764690241)*x[0]+(-0.0339165683557)*x[1]**o+(0.623897338329)*x[1]+(-0.169558925831)*x[2]**o+(-0.426663511213)*x[2]
        arg[2,3,0,1]=(0.185482762361)*x[0]**o+(0.496017413966)*x[0]+(-0.217189284009)*x[1]**o+(0.488205372581)*x[1]+(0.445762454794)*x[2]**o+(-0.423587587222)*x[2]
        arg[2,3,1,0]=(-0.489549524298)*x[0]**o+(-0.482299084307)*x[0]+(0.567729727254)*x[1]**o+(-0.341933025591)*x[1]+(-0.325429025641)*x[2]**o+(0.476030984261)*x[2]
        arg[2,3,1,1]=(-0.390754226646)*x[0]**o+(-0.980255652462)*x[0]+(0.109328516217)*x[1]**o+(-0.0141509875662)*x[1]+(0.486458644215)*x[2]**o+(-0.668018419371)*x[2]
        arg[2,3,2,0]=(0.352912006363)*x[0]**o+(-0.542969561227)*x[0]+(0.512067200296)*x[1]**o+(0.661110389562)*x[1]+(0.289744832231)*x[2]**o+(-0.176943484459)*x[2]
        arg[2,3,2,1]=(0.535262244127)*x[0]**o+(0.528325019892)*x[0]+(-0.106312276613)*x[1]**o+(-0.484575045858)*x[1]+(-0.407717587021)*x[2]**o+(-0.786366604025)*x[2]
        arg[2,4,0,0]=(-0.511771630262)*x[0]**o+(-0.590485876141)*x[0]+(0.493685168985)*x[1]**o+(-0.726700517739)*x[1]+(-0.213137223607)*x[2]**o+(-0.566923814015)*x[2]
        arg[2,4,0,1]=(0.421559088561)*x[0]**o+(0.838784944841)*x[0]+(-0.693654869402)*x[1]**o+(-0.0755493272975)*x[1]+(-0.174245799979)*x[2]**o+(0.242245946157)*x[2]
        arg[2,4,1,0]=(0.681235817735)*x[0]**o+(-0.313564199551)*x[0]+(0.663598935418)*x[1]**o+(-0.604327253895)*x[1]+(0.698985821444)*x[2]**o+(0.298899431146)*x[2]
        arg[2,4,1,1]=(0.432866305507)*x[0]**o+(-0.891501121013)*x[0]+(-0.290106178906)*x[1]**o+(0.268152357012)*x[1]+(0.75279498948)*x[2]**o+(0.211312203371)*x[2]
        arg[2,4,2,0]=(-0.956901622446)*x[0]**o+(0.742548241435)*x[0]+(-0.956209816095)*x[1]**o+(-0.539017080821)*x[1]+(-0.08062797007)*x[2]**o+(0.383843198102)*x[2]
        arg[2,4,2,1]=(-0.16700370288)*x[0]**o+(0.388837619298)*x[0]+(-0.190586726252)*x[1]**o+(0.916004535916)*x[1]+(0.310254387059)*x[2]**o+(-0.380371301717)*x[2]
        arg[3,0,0,0]=(0.64046380712)*x[0]**o+(0.254058760972)*x[0]+(-0.829050222873)*x[1]**o+(0.289942284927)*x[1]+(0.196268101069)*x[2]**o+(0.748880970446)*x[2]
        arg[3,0,0,1]=(-0.226057776579)*x[0]**o+(0.982685008224)*x[0]+(0.819680514589)*x[1]**o+(0.48005222662)*x[1]+(-0.615248069776)*x[2]**o+(-0.0923489923012)*x[2]
        arg[3,0,1,0]=(-0.358031084019)*x[0]**o+(-0.83941518843)*x[0]+(-0.0687218422525)*x[1]**o+(-0.648976147029)*x[1]+(0.711009285701)*x[2]**o+(0.360758898243)*x[2]
        arg[3,0,1,1]=(-0.312175299678)*x[0]**o+(0.360877864123)*x[0]+(0.653283960509)*x[1]**o+(0.323527948404)*x[1]+(-0.278454805659)*x[2]**o+(0.0585194127844)*x[2]
        arg[3,0,2,0]=(0.367207247645)*x[0]**o+(-0.0743547693027)*x[0]+(-0.473358544608)*x[1]**o+(0.332985041128)*x[1]+(-0.899612971909)*x[2]**o+(0.69409768473)*x[2]
        arg[3,0,2,1]=(-0.806787228991)*x[0]**o+(-0.110344204889)*x[0]+(-0.631610779885)*x[1]**o+(0.0239853435419)*x[1]+(0.83036063388)*x[2]**o+(0.623825256849)*x[2]
        arg[3,1,0,0]=(-0.268344201624)*x[0]**o+(-0.59230799177)*x[0]+(0.493719194706)*x[1]**o+(-0.128351291806)*x[1]+(0.0584636422077)*x[2]**o+(0.0859725996454)*x[2]
        arg[3,1,0,1]=(-0.870410307785)*x[0]**o+(-0.917370176071)*x[0]+(0.395701966434)*x[1]**o+(0.694856510628)*x[1]+(-0.748440730513)*x[2]**o+(0.748465450614)*x[2]
        arg[3,1,1,0]=(-0.178465888488)*x[0]**o+(-0.956884701156)*x[0]+(0.0472790389286)*x[1]**o+(0.946702274722)*x[1]+(-0.200668750878)*x[2]**o+(-0.0104023894447)*x[2]
        arg[3,1,1,1]=(0.113559334778)*x[0]**o+(0.270505514561)*x[0]+(-0.238008799138)*x[1]**o+(0.533067625871)*x[1]+(-0.586814102301)*x[2]**o+(0.045657074859)*x[2]
        arg[3,1,2,0]=(-0.574716699633)*x[0]**o+(-0.659499274639)*x[0]+(0.892605757655)*x[1]**o+(-0.39697940457)*x[1]+(0.363011145793)*x[2]**o+(-0.364722526091)*x[2]
        arg[3,1,2,1]=(-0.0193167048813)*x[0]**o+(0.761104649925)*x[0]+(0.490164183635)*x[1]**o+(0.10693267057)*x[1]+(0.635328989408)*x[2]**o+(-0.353114187925)*x[2]
        arg[3,2,0,0]=(0.0777423524941)*x[0]**o+(-0.431011355257)*x[0]+(-0.356762042134)*x[1]**o+(-0.94361679144)*x[1]+(-0.716191700456)*x[2]**o+(0.49906825221)*x[2]
        arg[3,2,0,1]=(0.80859948321)*x[0]**o+(0.542680788872)*x[0]+(-0.626897369633)*x[1]**o+(-0.821877255135)*x[1]+(-0.455575534023)*x[2]**o+(0.317480554025)*x[2]
        arg[3,2,1,0]=(-0.0144328272818)*x[0]**o+(-0.395367848628)*x[0]+(-0.601042089925)*x[1]**o+(-0.0822921879892)*x[1]+(0.896199761801)*x[2]**o+(-0.215055215166)*x[2]
        arg[3,2,1,1]=(0.377945289524)*x[0]**o+(-0.580064012583)*x[0]+(0.993926713349)*x[1]**o+(-0.736307101943)*x[1]+(-0.939628422411)*x[2]**o+(-0.923219473393)*x[2]
        arg[3,2,2,0]=(-0.733150287394)*x[0]**o+(0.692501827859)*x[0]+(0.904655347343)*x[1]**o+(0.310386512564)*x[1]+(-0.93795078156)*x[2]**o+(0.772534565721)*x[2]
        arg[3,2,2,1]=(-0.155229801255)*x[0]**o+(0.580356480314)*x[0]+(0.0673352927499)*x[1]**o+(-0.274975805243)*x[1]+(0.553818209037)*x[2]**o+(-0.835142674988)*x[2]
        arg[3,3,0,0]=(0.60752130978)*x[0]**o+(0.847750395856)*x[0]+(0.8635279539)*x[1]**o+(0.16341072973)*x[1]+(0.470011948821)*x[2]**o+(0.41529971393)*x[2]
        arg[3,3,0,1]=(0.755049219363)*x[0]**o+(0.66080573214)*x[0]+(-0.0256982576552)*x[1]**o+(-0.0311937109251)*x[1]+(-0.028843413794)*x[2]**o+(-0.19345503082)*x[2]
        arg[3,3,1,0]=(-0.39765237082)*x[0]**o+(-0.746359385218)*x[0]+(0.275396324626)*x[1]**o+(0.064571659107)*x[1]+(0.771289010752)*x[2]**o+(-6.69891594269e-05)*x[2]
        arg[3,3,1,1]=(0.350497253938)*x[0]**o+(-0.236419516561)*x[0]+(0.673483791847)*x[1]**o+(-0.934644138115)*x[1]+(0.994408125332)*x[2]**o+(-0.398423596854)*x[2]
        arg[3,3,2,0]=(-0.952544750261)*x[0]**o+(0.24256752245)*x[0]+(0.376782843601)*x[1]**o+(-0.349840168156)*x[1]+(-0.260325141903)*x[2]**o+(-0.452866344447)*x[2]
        arg[3,3,2,1]=(-0.0232941915849)*x[0]**o+(0.191625715695)*x[0]+(-0.490187753329)*x[1]**o+(-0.823976612466)*x[1]+(0.476804203752)*x[2]**o+(0.61057521184)*x[2]
        arg[3,4,0,0]=(0.0279130167137)*x[0]**o+(-0.134953133648)*x[0]+(0.0821614836141)*x[1]**o+(-0.894046763697)*x[1]+(-0.497197221983)*x[2]**o+(0.586297244251)*x[2]
        arg[3,4,0,1]=(0.0423746259802)*x[0]**o+(0.458075512165)*x[0]+(0.855157549678)*x[1]**o+(-0.607654212682)*x[1]+(-0.672693159955)*x[2]**o+(-0.300116892313)*x[2]
        arg[3,4,1,0]=(-0.695685046398)*x[0]**o+(-0.709807910289)*x[0]+(-0.167879145281)*x[1]**o+(0.524084329764)*x[1]+(0.763706690019)*x[2]**o+(-0.219502671318)*x[2]
        arg[3,4,1,1]=(0.864599640696)*x[0]**o+(-0.341125612414)*x[0]+(-0.548888211648)*x[1]**o+(0.918497762429)*x[1]+(-0.445600024463)*x[2]**o+(0.221658300325)*x[2]
        arg[3,4,2,0]=(-0.893419673448)*x[0]**o+(0.653718653341)*x[0]+(0.323868832441)*x[1]**o+(0.905933530129)*x[1]+(0.62045953406)*x[2]**o+(0.145474797235)*x[2]
        arg[3,4,2,1]=(0.480643160777)*x[0]**o+(-0.15896921521)*x[0]+(-0.329564714008)*x[1]**o+(-0.613303857496)*x[1]+(-0.483827450625)*x[2]**o+(-0.91023336048)*x[2]
        ref[0,0,0,0]=(0.520702135089)/(o+1.)+(0.109542934387)+(0.816458625566)*0.5**o
        ref[0,0,0,1]=(0.188518886272)/(o+1.)+(-0.938691737334)+(-0.709065020122)*0.5**o
        ref[0,0,1,0]=(-0.757157441618)/(o+1.)+(0.316696781197)+(-0.09165636265)*0.5**o
        ref[0,0,1,1]=(0.325937013197)/(o+1.)+(0.0450843195409)+(0.900670566988)*0.5**o
        ref[0,0,2,0]=(1.13837105734)/(o+1.)+(-0.573151939969)+(-0.883617789072)*0.5**o
        ref[0,0,2,1]=(-0.665166165764)/(o+1.)+(-0.845347549645)+(-0.770041366552)*0.5**o
        ref[0,1,0,0]=(0.714049217035)/(o+1.)+(0.4125480263)+(0.90396926296)*0.5**o
        ref[0,1,0,1]=(0.429723443697)/(o+1.)+(0.255965487666)+(0.47956310963)*0.5**o
        ref[0,1,1,0]=(-0.282787702479)/(o+1.)+(-0.631122483908)+(-0.75177304495)*0.5**o
        ref[0,1,1,1]=(1.16234435068)/(o+1.)+(1.09548265929)+(-0.109937058453)*0.5**o
        ref[0,1,2,0]=(-0.619187813423)/(o+1.)+(0.47035748919)+(-0.13801307581)*0.5**o
        ref[0,1,2,1]=(0.641297345823)/(o+1.)+(0.937805735461)+(-0.518540484981)*0.5**o
        ref[0,2,0,0]=(1.36665521497)/(o+1.)+(-0.736611486382)+(0.286922773164)*0.5**o
        ref[0,2,0,1]=(-0.27582202139)/(o+1.)+(-0.312977125741)+(-0.116447648923)*0.5**o
        ref[0,2,1,0]=(0.294494393603)/(o+1.)+(0.117678635379)+(-0.322803067618)*0.5**o
        ref[0,2,1,1]=(-0.251319095061)/(o+1.)+(0.517336946518)+(0.997823909059)*0.5**o
        ref[0,2,2,0]=(-1.06239756943)/(o+1.)+(-0.174439303934)+(0.516866019209)*0.5**o
        ref[0,2,2,1]=(-0.114597419557)/(o+1.)+(-0.488263901705)+(0.777538237337)*0.5**o
        ref[0,3,0,0]=(-0.545872917587)/(o+1.)+(0.320972405109)+(-0.622067362782)*0.5**o
        ref[0,3,0,1]=(-1.41803804504)/(o+1.)+(0.326673034142)+(-0.49698112241)*0.5**o
        ref[0,3,1,0]=(0.220871698725)/(o+1.)+(0.352475535909)+(0.917091542097)*0.5**o
        ref[0,3,1,1]=(-1.82481767225)/(o+1.)+(-0.534598958622)+(-0.437188100982)*0.5**o
        ref[0,3,2,0]=(1.04475880275)/(o+1.)+(0.00788693262039)+(-0.841265485658)*0.5**o
        ref[0,3,2,1]=(-0.262966085934)/(o+1.)+(0.330079558014)+(0.383282548921)*0.5**o
        ref[0,4,0,0]=(-1.66865494548)/(o+1.)+(-0.567436682752)+(0.0385891845805)*0.5**o
        ref[0,4,0,1]=(-0.183458096784)/(o+1.)+(-1.1357723338)+(-0.387739686761)*0.5**o
        ref[0,4,1,0]=(0.611419775289)/(o+1.)+(0.0492166712458)+(-0.329464483421)*0.5**o
        ref[0,4,1,1]=(0.000995600207189)/(o+1.)+(-0.182953786649)+(0.63272559561)*0.5**o
        ref[0,4,2,0]=(0.630523648936)/(o+1.)+(-0.0318688096074)+(-0.847582850604)*0.5**o
        ref[0,4,2,1]=(0.306969678703)/(o+1.)+(-0.726991015938)+(-0.722990535113)*0.5**o
        ref[1,0,0,0]=(-0.406383597128)/(o+1.)+(-0.580550741796)+(0.0725951752952)*0.5**o
        ref[1,0,0,1]=(0.988153215411)/(o+1.)+(-0.295391144874)+(-0.254920010173)*0.5**o
        ref[1,0,1,0]=(-0.113985267485)/(o+1.)+(0.381963273358)+(0.486342223622)*0.5**o
        ref[1,0,1,1]=(-1.0749658001)/(o+1.)+(-0.980590626052)+(0.777015150638)*0.5**o
        ref[1,0,2,0]=(0.866170818208)/(o+1.)+(0.295155963167)+(0.240813275278)*0.5**o
        ref[1,0,2,1]=(-0.502098016236)/(o+1.)+(-0.404695777523)+(-0.670257601629)*0.5**o
        ref[1,1,0,0]=(-0.0930803554155)/(o+1.)+(-0.262152356852)+(0.985895581872)*0.5**o
        ref[1,1,0,1]=(-1.66036439576)/(o+1.)+(0.188095420862)+(-0.954132399275)*0.5**o
        ref[1,1,1,0]=(-0.548936659502)/(o+1.)+(0.974895418354)+(0.462810965625)*0.5**o
        ref[1,1,1,1]=(-1.80963059517)/(o+1.)+(-0.200285193041)+(0.645629353541)*0.5**o
        ref[1,1,2,0]=(1.41739085898)/(o+1.)+(0.549897915219)+(-0.554993653726)*0.5**o
        ref[1,1,2,1]=(0.0907943154278)/(o+1.)+(-0.450977439484)+(0.1476587556)*0.5**o
        ref[1,2,0,0]=(-0.657176681491)/(o+1.)+(-0.433402993059)+(-0.715887407694)*0.5**o
        ref[1,2,0,1]=(-0.562421731301)/(o+1.)+(0.0104884758643)+(0.48440337855)*0.5**o
        ref[1,2,1,0]=(1.28216615223)/(o+1.)+(-0.33142121119)+(0.621562227153)*0.5**o
        ref[1,2,1,1]=(1.16073972701)/(o+1.)+(0.0404640630801)+(0.472846565983)*0.5**o
        ref[1,2,2,0]=(0.772147142645)/(o+1.)+(1.09472486224)+(-0.212000572333)*0.5**o
        ref[1,2,2,1]=(0.811972306738)/(o+1.)+(-0.409837616651)+(-0.632752637148)*0.5**o
        ref[1,3,0,0]=(0.754125096738)/(o+1.)+(0.000627424214716)+(0.390081216336)*0.5**o
        ref[1,3,0,1]=(0.213094712846)/(o+1.)+(-0.73677243263)+(-0.402887151891)*0.5**o
        ref[1,3,1,0]=(-0.0409011142831)/(o+1.)+(0.288589213809)+(0.253947492129)*0.5**o
        ref[1,3,1,1]=(0.519205654314)/(o+1.)+(-0.416997867426)+(-0.435656267291)*0.5**o
        ref[1,3,2,0]=(0.0264169632513)/(o+1.)+(0.271272227399)+(-0.069319863146)*0.5**o
        ref[1,3,2,1]=(-0.845761130374)/(o+1.)+(1.07136053657)+(0.741825647399)*0.5**o
        ref[1,4,0,0]=(0.967780841131)/(o+1.)+(-0.0706409792865)+(-0.66003338591)*0.5**o
        ref[1,4,0,1]=(-1.02543406666)/(o+1.)+(-0.81933006071)+(-0.0001759210396)*0.5**o
        ref[1,4,1,0]=(0.121210472128)/(o+1.)+(0.718839287394)+(0.052435899043)*0.5**o
        ref[1,4,1,1]=(-0.481205774953)/(o+1.)+(0.50624453832)+(0.00806522196579)*0.5**o
        ref[1,4,2,0]=(-1.54705242741)/(o+1.)+(-0.261649676126)+(0.440212388104)*0.5**o
        ref[1,4,2,1]=(-0.556903640465)/(o+1.)+(0.349237558673)+(0.346121727918)*0.5**o
        ref[2,0,0,0]=(0.795877312597)/(o+1.)+(0.0954263364687)+(-0.801941188348)*0.5**o
        ref[2,0,0,1]=(-0.252963905823)/(o+1.)+(-0.311389887295)+(0.786459471841)*0.5**o
        ref[2,0,1,0]=(0.911932710964)/(o+1.)+(0.477954075311)+(0.813571093948)*0.5**o
        ref[2,0,1,1]=(-0.641335108636)/(o+1.)+(0.531483576652)+(-0.0298664722355)*0.5**o
        ref[2,0,2,0]=(-0.755636810913)/(o+1.)+(-0.501364357709)+(0.688553055025)*0.5**o
        ref[2,0,2,1]=(0.812309518755)/(o+1.)+(0.0201614791497)+(-0.959007707329)*0.5**o
        ref[2,1,0,0]=(-0.599812442203)/(o+1.)+(-0.147264638959)+(0.648021743651)*0.5**o
        ref[2,1,0,1]=(0.625922025772)/(o+1.)+(-0.448500236464)+(-0.134785884326)*0.5**o
        ref[2,1,1,0]=(0.0705569242367)/(o+1.)+(0.135036197507)+(-0.210371458935)*0.5**o
        ref[2,1,1,1]=(-0.393538949024)/(o+1.)+(-0.307622777145)+(-0.124204683462)*0.5**o
        ref[2,1,2,0]=(0.270507676691)/(o+1.)+(1.16175547002)+(-0.864065825608)*0.5**o
        ref[2,1,2,1]=(-0.670421740374)/(o+1.)+(-0.438725518385)+(0.37622234855)*0.5**o
        ref[2,2,0,0]=(-0.268367904213)/(o+1.)+(-0.288835990468)+(0.0541619627989)*0.5**o
        ref[2,2,0,1]=(0.188356817882)/(o+1.)+(-0.0879880596161)+(0.611472347101)*0.5**o
        ref[2,2,1,0]=(-0.253164353095)/(o+1.)+(0.312413832582)+(-0.926062469523)*0.5**o
        ref[2,2,1,1]=(0.374846712027)/(o+1.)+(-0.939077030246)+(0.0689111652346)*0.5**o
        ref[2,2,2,0]=(0.665397344854)/(o+1.)+(-0.257604834045)+(-0.133251706696)*0.5**o
        ref[2,2,2,1]=(-0.148088444003)/(o+1.)+(-0.557155078588)+(-0.567309977436)*0.5**o
        ref[2,3,0,0]=(-0.203475494187)/(o+1.)+(0.359999258678)+(0.477968626266)*0.5**o
        ref[2,3,0,1]=(0.228573170785)/(o+1.)+(0.280317599662)+(0.185482762361)*0.5**o
        ref[2,3,1,0]=(0.242300701614)/(o+1.)+(-0.174100562819)+(-0.489549524298)*0.5**o
        ref[2,3,1,1]=(0.595787160431)/(o+1.)+(-0.8312125297)+(-0.390754226646)*0.5**o
        ref[2,3,2,0]=(0.801812032527)/(o+1.)+(-0.0294013280621)+(0.352912006363)*0.5**o
        ref[2,3,2,1]=(-0.514029863634)/(o+1.)+(-0.371308314996)+(0.535262244127)*0.5**o
        ref[2,4,0,0]=(0.280547945378)/(o+1.)+(-0.942055103948)+(-0.511771630262)*0.5**o
        ref[2,4,0,1]=(-0.867900669381)/(o+1.)+(0.50274078185)+(0.421559088561)*0.5**o
        ref[2,4,1,0]=(1.36258475686)/(o+1.)+(-0.30949601115)+(0.681235817735)*0.5**o
        ref[2,4,1,1]=(0.462688810574)/(o+1.)+(-0.206018280315)+(0.432866305507)*0.5**o
        ref[2,4,2,0]=(-1.03683778617)/(o+1.)+(0.293687179358)+(-0.956901622446)*0.5**o
        ref[2,4,2,1]=(0.119667660807)/(o+1.)+(0.462235426749)+(-0.16700370288)*0.5**o
        ref[3,0,0,0]=(-0.632782121804)/(o+1.)+(0.646441008172)+(0.64046380712)*0.5**o
        ref[3,0,0,1]=(0.204432444813)/(o+1.)+(0.685194121272)+(-0.226057776579)*0.5**o
        ref[3,0,1,0]=(0.642287443449)/(o+1.)+(-0.563816218608)+(-0.358031084019)*0.5**o
        ref[3,0,1,1]=(0.37482915485)/(o+1.)+(0.371462612656)+(-0.312175299678)*0.5**o
        ref[3,0,2,0]=(-1.37297151652)/(o+1.)+(0.476363978278)+(0.367207247645)*0.5**o
        ref[3,0,2,1]=(0.198749853995)/(o+1.)+(0.268733197751)+(-0.806787228991)*0.5**o
        ref[3,1,0,0]=(0.552182836914)/(o+1.)+(-0.317343341965)+(-0.268344201624)*0.5**o
        ref[3,1,0,1]=(-0.352738764079)/(o+1.)+(0.262975892585)+(-0.870410307785)*0.5**o
        ref[3,1,1,0]=(-0.153389711949)/(o+1.)+(-0.0102924079393)+(-0.178465888488)*0.5**o
        ref[3,1,1,1]=(-0.824822901439)/(o+1.)+(0.424615107645)+(0.113559334778)*0.5**o
        ref[3,1,2,0]=(1.25561690345)/(o+1.)+(-0.71060060265)+(-0.574716699633)*0.5**o
        ref[3,1,2,1]=(1.12549317304)/(o+1.)+(0.257461566285)+(-0.0193167048813)*0.5**o
        ref[3,2,0,0]=(-1.07295374259)/(o+1.)+(-0.437779947243)+(0.0777423524941)*0.5**o
        ref[3,2,0,1]=(-1.08247290366)/(o+1.)+(0.019142043881)+(0.80859948321)*0.5**o
        ref[3,2,1,0]=(0.295157671875)/(o+1.)+(-0.346357625892)+(-0.0144328272818)*0.5**o
        ref[3,2,1,1]=(0.054298290938)/(o+1.)+(-1.11979529396)+(0.377945289524)*0.5**o
        ref[3,2,2,0]=(-0.0332954342171)/(o+1.)+(0.887711453071)+(-0.733150287394)*0.5**o
        ref[3,2,2,1]=(0.621153501787)/(o+1.)+(-0.264880999958)+(-0.155229801255)*0.5**o
        ref[3,3,0,0]=(1.33353990272)/(o+1.)+(0.713230419758)+(0.60752130978)*0.5**o
        ref[3,3,0,1]=(-0.0545416714491)/(o+1.)+(0.218078495198)+(0.755049219363)*0.5**o
        ref[3,3,1,0]=(1.04668533538)/(o+1.)+(-0.340927357635)+(-0.39765237082)*0.5**o
        ref[3,3,1,1]=(1.66789191718)/(o+1.)+(-0.784743625765)+(0.350497253938)*0.5**o
        ref[3,3,2,0]=(0.116457701698)/(o+1.)+(-0.280069495076)+(-0.952544750261)*0.5**o
        ref[3,3,2,1]=(-0.0133835495768)/(o+1.)+(-0.0108878424651)+(-0.0232941915849)*0.5**o
        ref[3,4,0,0]=(-0.415035738369)/(o+1.)+(-0.221351326547)+(0.0279130167137)*0.5**o
        ref[3,4,0,1]=(0.182464389723)/(o+1.)+(-0.224847796415)+(0.0423746259802)*0.5**o
        ref[3,4,1,0]=(0.595827544738)/(o+1.)+(-0.202613125922)+(-0.695685046398)*0.5**o
        ref[3,4,1,1]=(-0.99448823611)/(o+1.)+(0.39951522517)+(0.864599640696)*0.5**o
        ref[3,4,2,0]=(0.944328366501)/(o+1.)+(0.852563490353)+(-0.893419673448)*0.5**o
        ref[3,4,2,1]=(-0.813392164634)/(o+1.)+(-0.841253216593)+(0.480643160777)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedSolution_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactOne

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.241972460279)*x[0]+(0.539729925673)*x[1]
        ref=(-0.11832904353)/(o+1.)+(0.649911364492)+(-0.399791299503)*0.5**o
      else:
        arg=(0.53919797656)*x[0]+(0.379010664884)*x[1]+(0.274416064529)*x[2]
        ref=(0.307947301472)/(o+1.)+(0.120674157748)+(0.643329089005)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedSolution_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactOne

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.84849017056)*x[0]+(-0.161962845202)*x[1]
        arg[1]=(-0.794717919135)*x[0]+(1.16003773358)*x[1]
        ref[0]=(0.780514448202)/(o+1.)+(-0.34628958021)+(0.598592037576)*0.5**o
        ref[1]=(0.570158156278)/(o+1.)+(0.241192316704)+(-0.687222975237)*0.5**o
      else:
        arg[0]=(-0.637352163634)*x[0]+(1.49477462643)*x[1]+(-0.284018740238)*x[2]
        arg[1]=(0.125937459904)*x[0]+(-0.206499869891)*x[1]+(-0.342856240345)*x[2]
        ref[0]=(-0.274486314451)/(o+1.)+(0.379737553853)+(0.0884149293071)*0.5**o
        ref[1]=(-0.78452814021)/(o+1.)+(-0.122017410137)+(0.605144310151)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedSolution_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactOne

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.440868762581)*x[0]+(-0.216223666616)*x[1]
        arg[0,1]=(-1.78807355591)*x[0]+(-0.208278123838)*x[1]
        arg[0,2]=(-0.448066998103)*x[0]+(-0.819815507826)*x[1]
        arg[0,3]=(1.27247881689)*x[0]+(0.0653439567221)*x[1]
        arg[0,4]=(-1.00344990223)*x[0]+(-0.291263790583)*x[1]
        arg[1,0]=(0.440080423838)*x[0]+(0.0356836558113)*x[1]
        arg[1,1]=(-1.49545340091)*x[0]+(-0.133615590211)*x[1]
        arg[1,2]=(1.05947638385)*x[0]+(0.408685904238)*x[1]
        arg[1,3]=(-0.170309086605)*x[0]+(0.660523970285)*x[1]
        arg[1,4]=(0.764323678475)*x[0]+(-0.529415080303)*x[1]
        arg[2,0]=(0.55270081325)*x[0]+(-1.60630146231)*x[1]
        arg[2,1]=(1.57419330248)*x[0]+(-1.24644288706)*x[1]
        arg[2,2]=(0.0416757238363)*x[0]+(0.0247379738465)*x[1]
        arg[2,3]=(1.0200370393)*x[0]+(0.364209137693)*x[1]
        arg[2,4]=(-1.21858490389)*x[0]+(0.523056254546)*x[1]
        arg[3,0]=(0.0524205820842)*x[0]+(1.65953217854)*x[1]
        arg[3,1]=(1.21824883395)*x[0]+(0.905190834998)*x[1]
        arg[3,2]=(0.226344434525)*x[0]+(-0.735349531462)*x[1]
        arg[3,3]=(-1.00295245717)*x[0]+(-0.719605376499)*x[1]
        arg[3,4]=(-0.446228759693)*x[0]+(1.23441974277)*x[1]
        ref[0,0]=(0.347328582602)/(o+1.)+(-0.520484463519)+(0.0365479152384)*0.5**o
        ref[0,1]=(-0.274564442352)/(o+1.)+(-0.413557210678)+(-0.894672816039)*0.5**o
        ref[0,2]=(0.0720644688954)/(o+1.)+(-0.906636966188)+(0.473326957552)*0.5**o
        ref[0,3]=(-0.747749341777)/(o+1.)+(0.730249972378)+(0.625072170633)*0.5**o
        ref[0,4]=(-0.194031116169)/(o+1.)+(-0.432063325829)+(-0.236555924983)*0.5**o
        ref[1,0]=(0.807081538033)/(o+1.)+(-0.612266650007)+(0.893215841629)*0.5**o
        ref[1,1]=(-0.662516518957)/(o+1.)+(-0.176318925886)+(-0.61391462039)*0.5**o
        ref[1,2]=(-0.570345636156)/(o+1.)+(0.791877954963)+(0.454752014319)*0.5**o
        ref[1,3]=(-0.312400910795)/(o+1.)+(0.212935137466)+(0.376745519544)*0.5**o
        ref[1,4]=(-0.764680414264)/(o+1.)+(0.174215486579)+(0.651158039278)*0.5**o
        ref[2,0]=(-0.837942771642)/(o+1.)+(-0.187938486702)+(0.160219095985)*0.5**o
        ref[2,1]=(-0.672421039761)/(o+1.)+(0.17384002882)+(0.652491397549)*0.5**o
        ref[2,2]=(-0.656187949922)/(o+1.)+(0.559888736121)+(-0.397175824637)*0.5**o
        ref[2,3]=(0.259558368195)/(o+1.)+(0.55148644978)+(0.0217149092406)*0.5**o
        ref[2,4]=(-0.188397537901)/(o+1.)+(-0.0561706170512)+(-0.394789877345)*0.5**o
        ref[3,0]=(0.91055628617)/(o+1.)+(0.574211274098)+(-0.347026073739)*0.5**o
        ref[3,1]=(0.29225509237)/(o+1.)+(0.64447705482)+(0.542230466941)*0.5**o
        ref[3,2]=(0.121065624384)/(o+1.)+(0.0316495277836)+(-0.693369776888)*0.5**o
        ref[3,3]=(-0.943523334976)/(o+1.)+(-0.266302644031)+(-0.246429210629)*0.5**o
        ref[3,4]=(0.714915399455)/(o+1.)+(0.459963131994)+(-0.846650680369)*0.5**o
      else:
        arg[0,0]=(-0.493869713454)*x[0]+(-0.381969413515)*x[1]+(1.59862368848)*x[2]
        arg[0,1]=(-0.581245543547)*x[0]+(0.961117271889)*x[1]+(-0.158073515343)*x[2]
        arg[0,2]=(-0.635283143641)*x[0]+(0.695573236987)*x[1]+(-0.719180832278)*x[2]
        arg[0,3]=(-0.737780596733)*x[0]+(-1.11171283352)*x[1]+(0.459954346315)*x[2]
        arg[0,4]=(1.12522065074)*x[0]+(-1.08427904854)*x[1]+(-0.715679673556)*x[2]
        arg[1,0]=(0.972468186181)*x[0]+(-0.211014530435)*x[1]+(-0.734414099002)*x[2]
        arg[1,1]=(-1.67337704748)*x[0]+(0.871096213089)*x[1]+(1.51063313223)*x[2]
        arg[1,2]=(-0.222013007834)*x[0]+(1.46474508326)*x[1]+(-0.979243908862)*x[2]
        arg[1,3]=(1.21195281087)*x[0]+(0.274082389329)*x[1]+(1.56144436465)*x[2]
        arg[1,4]=(1.35812865884)*x[0]+(1.25819798853)*x[1]+(0.895334048395)*x[2]
        arg[2,0]=(-0.167404344437)*x[0]+(0.0710591934315)*x[1]+(-0.124469683665)*x[2]
        arg[2,1]=(-0.492240632996)*x[0]+(-0.317251117382)*x[1]+(0.0195461703529)*x[2]
        arg[2,2]=(-0.277590266225)*x[0]+(-1.65859476899)*x[1]+(-0.0100216405668)*x[2]
        arg[2,3]=(-0.0701281298455)*x[0]+(0.502137254703)*x[1]+(0.701220923443)*x[2]
        arg[2,4]=(-0.72186096407)*x[0]+(-0.900487568179)*x[1]+(-0.735573922826)*x[2]
        arg[3,0]=(-0.0677324474129)*x[0]+(-0.649963209718)*x[1]+(-0.770861099726)*x[2]
        arg[3,1]=(0.563588189481)*x[0]+(0.714614932251)*x[1]+(-0.0957764632886)*x[2]
        arg[3,2]=(1.13337395062)*x[0]+(0.507556316073)*x[1]+(-0.196759085949)*x[2]
        arg[3,3]=(-1.2636883168)*x[0]+(-0.811660344076)*x[1]+(0.645462052598)*x[2]
        arg[3,4]=(-0.104175186987)*x[0]+(-0.929176038155)*x[1]+(1.07530379856)*x[2]
        ref[0,0]=(0.84538674973)/(o+1.)+(0.0569713556632)+(-0.236544899547)*0.5**o
        ref[0,1]=(-0.307689589699)/(o+1.)+(0.245040337642)+(0.0394071274131)*0.5**o
        ref[0,2]=(0.955244736357)/(o+1.)+(-0.341771136504)+(-0.930593202281)*0.5**o
        ref[0,3]=(-0.353490461722)/(o+1.)+(-0.476228433901)+(-0.083591754415)*0.5**o
        ref[0,4]=(-1.31220485482)/(o+1.)+(0.0466426656676)+(0.544181452128)*0.5**o
        ref[1,0]=(0.22773475414)/(o+1.)+(-0.488454801654)+(0.776214405914)*0.5**o
        ref[1,1]=(1.50688678707)/(o+1.)+(0.0851415404695)+(-0.96881757017)*0.5**o
        ref[1,2]=(0.313471097102)/(o+1.)+(-0.0875594518156)+(0.125135973096)*0.5**o
        ref[1,3]=(0.269521220179)/(o+1.)+(1.1192007195)+(0.539556905661)*0.5**o
        ref[1,4]=(1.91499876064)/(o+1.)+(0.509063797893)+(0.578534339343)*0.5**o
        ref[2,0]=(0.941352651562)/(o+1.)+(-0.966199402186)+(0.770231318139)*0.5**o
        ref[2,1]=(-0.141490823562)/(o+1.)+(-0.20919987022)+(-0.230055016023)*0.5**o
        ref[2,2]=(-0.916148047971)/(o+1.)+(-0.371740600192)+(-0.286577427426)*0.5**o
        ref[2,3]=(0.229516219331)/(o+1.)+(0.274790532736)+(0.354132763498)*0.5**o
        ref[2,4]=(-1.19120612619)/(o+1.)+(-0.139859859385)+(-0.88699661011)*0.5**o
        ref[3,0]=(-1.48630121465)/(o+1.)+(0.0744166251063)+(-0.151088792417)*0.5**o
        ref[3,1]=(-0.335583560513)/(o+1.)+(0.277436410841)+(0.963137397276)*0.5**o
        ref[3,2]=(-0.673942688644)/(o+1.)+(0.70077509905)+(0.716563671283)*0.5**o
        ref[3,3]=(-0.148394092795)/(o+1.)+(-0.237190239072)+(-0.807112037336)*0.5**o
        ref[3,4]=(0.529983886211)/(o+1.)+(-0.371055083588)+(0.254078854382)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedSolution_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnContactOne

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.661725715165)*x[0]+(0.219720717581)*x[1]
        arg[0,0,1]=(-0.642240082304)*x[0]+(-0.279652756776)*x[1]
        arg[0,1,0]=(0.0204235813151)*x[0]+(-0.374873249933)*x[1]
        arg[0,1,1]=(1.4418481252)*x[0]+(0.205656834772)*x[1]
        arg[1,0,0]=(0.65992179951)*x[0]+(-1.04887875458)*x[1]
        arg[1,0,1]=(-0.195480216199)*x[0]+(-0.373905275039)*x[1]
        arg[1,1,0]=(0.781251543352)*x[0]+(0.0416248109536)*x[1]
        arg[1,1,1]=(-0.925751319189)*x[0]+(-0.113992910046)*x[1]
        arg[2,0,0]=(-0.0767414065227)*x[0]+(-0.156682840973)*x[1]
        arg[2,0,1]=(0.688457272529)*x[0]+(0.524916336046)*x[1]
        arg[2,1,0]=(0.753722830084)*x[0]+(-0.890325839466)*x[1]
        arg[2,1,1]=(0.730789290145)*x[0]+(-0.278986703716)*x[1]
        arg[3,0,0]=(-0.287376796439)*x[0]+(-0.153987233224)*x[1]
        arg[3,0,1]=(-1.15687435255)*x[0]+(0.250005890238)*x[1]
        arg[3,1,0]=(-0.845312830629)*x[0]+(0.256211962352)*x[1]
        arg[3,1,1]=(-0.574311820519)*x[0]+(-0.440297035998)*x[1]
        arg[4,0,0]=(-0.360607647308)*x[0]+(-1.29380437557)*x[1]
        arg[4,0,1]=(1.32019433485)*x[0]+(0.295255956618)*x[1]
        arg[4,1,0]=(-1.64162941972)*x[0]+(0.117482372162)*x[1]
        arg[4,1,1]=(-0.303103646619)*x[0]+(-0.88458982681)*x[1]
        arg[5,0,0]=(-0.666761879038)*x[0]+(-0.735786278877)*x[1]
        arg[5,0,1]=(0.0892896382617)*x[0]+(-0.45478872773)*x[1]
        arg[5,1,0]=(-0.50985624415)*x[0]+(-0.400022939567)*x[1]
        arg[5,1,1]=(1.39240706368)*x[0]+(0.0149740822664)*x[1]
        ref[0,0,0]=(-0.114834538488)/(o+1.)+(0.240621006484)+(0.515038958264)*0.5**o
        ref[0,0,1]=(0.588698337077)/(o+1.)+(-0.702256377233)+(-0.10607842169)*0.5**o
        ref[0,1,0]=(0.13849818224)/(o+1.)+(-0.277555277991)+(0.0621627051247)*0.5**o
        ref[0,1,1]=(-0.0878009521163)/(o+1.)+(0.630168348947)+(0.474969214192)*0.5**o
        ref[1,0,0]=(-0.696600538111)/(o+1.)+(-0.343318260912)+(0.994280104861)*0.5**o
        ref[1,0,1]=(-0.0887026688527)/(o+1.)+(-0.385753227979)+(0.290823633572)*0.5**o
        ref[1,1,0]=(0.831408608956)/(o+1.)+(-0.242985671853)+(0.477439089057)*0.5**o
        ref[1,1,1]=(-0.621059053191)/(o+1.)+(-0.0700203481241)+(-0.278644479797)*0.5**o
        ref[2,0,0]=(0.7890720399)/(o+1.)+(-0.539721196393)+(0.0569461053905)*0.5**o
        ref[2,0,1]=(-0.228628151401)/(o+1.)+(0.826946074171)+(-0.211890388366)*0.5**o
        ref[2,1,0]=(-0.837617325493)/(o+1.)+(0.289656653751)+(0.121701008608)*0.5**o
        ref[2,1,1]=(0.246577988625)/(o+1.)+(0.0512227634376)+(0.102779070928)*0.5**o
        ref[3,0,0]=(-0.0109584684212)/(o+1.)+(-0.334888666171)+(0.239371771102)*0.5**o
        ref[3,0,1]=(-0.125987676419)/(o+1.)+(-0.170179004032)+(-0.440522777827)*0.5**o
        ref[3,1,0]=(-0.354093141636)/(o+1.)+(-0.177294275919)+(0.119580825198)*0.5**o
        ref[3,1,1]=(-0.746761075412)/(o+1.)+(0.192068760731)+(-0.651985302566)*0.5**o
        ref[4,0,0]=(-0.42411673219)/(o+1.)+(-0.579162944937)+(-0.0719694008158)*0.5**o
        ref[4,0,1]=(0.169881169584)/(o+1.)+(0.483614029132)+(0.478341063616)*0.5**o
        ref[4,1,0]=(0.943567836264)/(o+1.)+(-0.898782941264)+(-0.670149001297)*0.5**o
        ref[4,1,1]=(-0.703610807936)/(o+1.)+(0.225342628657)+(-0.934767922806)*0.5**o
        ref[5,0,0]=(-0.788792225213)/(o+1.)+(-0.1768857093)+(-0.259984514102)*0.5**o
        ref[5,0,1]=(-0.788708879899)/(o+1.)+(-0.202288491141)+(0.827786772711)*0.5**o
        ref[5,1,0]=(-0.823081809707)/(o+1.)+(0.274720963093)+(-0.636239300197)*0.5**o
        ref[5,1,1]=(0.498922549807)/(o+1.)+(0.0129388257746)+(0.882580944588)*0.5**o
      else:
        arg[0,0,0]=(-0.426613654844)*x[0]+(-1.01433980801)*x[1]+(0.564621874916)*x[2]
        arg[0,0,1]=(-0.438521968882)*x[0]+(1.6033775267)*x[1]+(0.691349995832)*x[2]
        arg[0,1,0]=(-0.04337513731)*x[0]+(0.142927668234)*x[1]+(0.506581723199)*x[2]
        arg[0,1,1]=(-0.120598868162)*x[0]+(-0.213666787813)*x[1]+(-1.04221679332)*x[2]
        arg[1,0,0]=(0.153696490712)*x[0]+(-0.122516529815)*x[1]+(-1.08837272712)*x[2]
        arg[1,0,1]=(0.410706464237)*x[0]+(0.720079363315)*x[1]+(-0.767790357539)*x[2]
        arg[1,1,0]=(-1.03726428173)*x[0]+(-0.316809853003)*x[1]+(-0.780405868082)*x[2]
        arg[1,1,1]=(-0.427547967216)*x[0]+(-0.897766643609)*x[1]+(-1.56913552404)*x[2]
        arg[2,0,0]=(-0.282789248898)*x[0]+(-0.110846936031)*x[1]+(-1.4329391906)*x[2]
        arg[2,0,1]=(1.46866588617)*x[0]+(-1.23022501959)*x[1]+(-0.3123390018)*x[2]
        arg[2,1,0]=(-0.314115718621)*x[0]+(1.7333934438)*x[1]+(0.833572700726)*x[2]
        arg[2,1,1]=(0.464246462638)*x[0]+(0.267484068646)*x[1]+(0.342009204787)*x[2]
        arg[3,0,0]=(0.578853099163)*x[0]+(-0.0453855508465)*x[1]+(-0.0751578475516)*x[2]
        arg[3,0,1]=(-0.526734485752)*x[0]+(-1.03286310016)*x[1]+(-0.398986583917)*x[2]
        arg[3,1,0]=(0.0239088277798)*x[0]+(0.205105470192)*x[1]+(-1.04986349166)*x[2]
        arg[3,1,1]=(0.130893744001)*x[0]+(-0.273702140669)*x[1]+(-0.0804222415128)*x[2]
        arg[4,0,0]=(-0.822661618584)*x[0]+(-0.267189204519)*x[1]+(-0.290013766971)*x[2]
        arg[4,0,1]=(-1.46759823734)*x[0]+(0.317465945316)*x[1]+(0.108284317752)*x[2]
        arg[4,1,0]=(-1.3920990386)*x[0]+(-1.17823778694)*x[1]+(-0.18366434307)*x[2]
        arg[4,1,1]=(-0.0915496209516)*x[0]+(-0.367624609832)*x[1]+(1.30630257323)*x[2]
        arg[5,0,0]=(0.117840830455)*x[0]+(-0.667875433194)*x[1]+(-1.59716688455)*x[2]
        arg[5,0,1]=(-0.818693227605)*x[0]+(0.483842075248)*x[1]+(-0.408373205535)*x[2]
        arg[5,1,0]=(1.36906770301)*x[0]+(0.494926179999)*x[1]+(0.717041707262)*x[2]
        arg[5,1,1]=(0.819757116081)*x[0]+(0.0982458749015)*x[1]+(0.275080391599)*x[2]
        ref[0,0,0]=(0.236560699949)/(o+1.)+(-0.392784388084)+(-0.327323511722)*0.5**o
        ref[0,0,1]=(1.32491522045)/(o+1.)+(0.525146863711)+(-0.519003394222)*0.5**o
        ref[0,1,0]=(0.433749865563)/(o+1.)+(0.23007126008)+(-0.287758131599)*0.5**o
        ref[0,1,1]=(-0.0563021060686)/(o+1.)+(-0.403921160703)+(-0.512338021825)*0.5**o
        ref[1,0,0]=(-0.170496965963)/(o+1.)+(-0.914421777259)+(0.942147754258)*0.5**o
        ref[1,0,1]=(-0.372707483726)/(o+1.)+(0.636041158872)+(-0.536379364005)*0.5**o
        ref[1,1,0]=(-0.211529878364)/(o+1.)+(-0.912151985776)+(-0.0986461529033)*0.5**o
        ref[1,1,1]=(-1.32533998437)/(o+1.)+(-0.685030198386)+(-0.19904975372)*0.5**o
        ref[2,0,0]=(0.152048902469)/(o+1.)+(-0.88880590209)+(-0.201012473813)*0.5**o
        ref[2,0,1]=(-0.748900052246)/(o+1.)+(-0.0226120471998)+(0.720226011425)*0.5**o
        ref[2,1,0]=(0.77797707941)/(o+1.)+(1.01635055442)+(-0.557827762351)*0.5**o
        ref[2,1,1]=(1.08969334716)/(o+1.)+(-0.404644728844)+(0.793335846595)*0.5**o
        ref[3,0,0]=(1.56498448669)/(o+1.)+(-0.699967188685)+(0.293259591449)*0.5**o
        ref[3,0,1]=(-0.620133427128)/(o+1.)+(-0.550037413688)+(-0.238375915329)*0.5**o
        ref[3,1,0]=(-0.156380586288)/(o+1.)+(-0.0154369973565)+(-0.633594612687)*0.5**o
        ref[3,1,1]=(-0.352989559326)/(o+1.)+(-0.290547031627)+(0.710852984398)*0.5**o
        ref[4,0,0]=(-0.147579361355)/(o+1.)+(-0.615173424765)+(-0.00193837918766)*0.5**o
        ref[4,0,1]=(0.598152345277)/(o+1.)+(-0.540248606426)+(-0.559503106696)*0.5**o
        ref[4,1,0]=(-1.17182448825)/(o+1.)+(-0.590551681576)+(-0.401073317208)*0.5**o
        ref[4,1,1]=(0.837504141173)/(o+1.)+(0.0807414267186)+(-0.15185865216)*0.5**o
        ref[5,0,0]=(-0.646601115533)/(o+1.)+(-1.20895265273)+(0.9173049337)*0.5**o
        ref[5,0,1]=(1.26626110052)/(o+1.)+(-0.56471923973)+(-0.880046978955)*0.5**o
        ref[5,1,0]=(-0.069063169298)/(o+1.)+(0.883536796964)+(0.883025165644)*0.5**o
        ref[5,1,1]=(1.05999338993)/(o+1.)+(-0.392519102893)+(0.918128198439)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedSolution_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnContactOne

      assumptions: ReducedSolution(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedSolution(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.257898226293)*x[0]+(-0.01230083314)*x[1]
        arg[0,0,0,1]=(0.0531183070647)*x[0]+(-0.170192287836)*x[1]
        arg[0,0,1,0]=(-0.364644940864)*x[0]+(1.04227651446)*x[1]
        arg[0,0,1,1]=(-0.894334108203)*x[0]+(0.151390331211)*x[1]
        arg[0,0,2,0]=(1.03825401708)*x[0]+(-0.882136097149)*x[1]
        arg[0,0,2,1]=(-1.05304910155)*x[0]+(1.06866311708)*x[1]
        arg[0,1,0,0]=(1.54742542159)*x[0]+(0.941778348482)*x[1]
        arg[0,1,0,1]=(1.1797380956)*x[0]+(-0.296417138234)*x[1]
        arg[0,1,1,0]=(-1.09196028027)*x[0]+(-0.783377624006)*x[1]
        arg[0,1,1,1]=(0.370952191443)*x[0]+(-0.0854141858991)*x[1]
        arg[0,1,2,0]=(-1.23552368139)*x[0]+(1.23848088087)*x[1]
        arg[0,1,2,1]=(0.216345350114)*x[0]+(-0.136524468427)*x[1]
        arg[0,2,0,0]=(-1.21206901263)*x[0]+(0.804219582144)*x[1]
        arg[0,2,0,1]=(0.192956526208)*x[0]+(0.588417678858)*x[1]
        arg[0,2,1,0]=(-0.940175603742)*x[0]+(0.329047260784)*x[1]
        arg[0,2,1,1]=(-0.138597387792)*x[0]+(0.398298326223)*x[1]
        arg[0,2,2,0]=(0.10089766824)*x[0]+(1.25459493298)*x[1]
        arg[0,2,2,1]=(-0.401163525883)*x[0]+(0.276542765282)*x[1]
        arg[0,3,0,0]=(0.252156795268)*x[0]+(-0.50223063539)*x[1]
        arg[0,3,0,1]=(0.466244610905)*x[0]+(-0.587025224301)*x[1]
        arg[0,3,1,0]=(-0.268082012713)*x[0]+(-1.28874675764)*x[1]
        arg[0,3,1,1]=(1.10867463519)*x[0]+(-0.165500912864)*x[1]
        arg[0,3,2,0]=(-1.94681908266)*x[0]+(0.166854108476)*x[1]
        arg[0,3,2,1]=(-0.537150417922)*x[0]+(0.0872851457131)*x[1]
        arg[0,4,0,0]=(-0.376548581807)*x[0]+(-0.245201133388)*x[1]
        arg[0,4,0,1]=(-0.135287690767)*x[0]+(0.750376238942)*x[1]
        arg[0,4,1,0]=(-0.0240491596907)*x[0]+(-1.21337088424)*x[1]
        arg[0,4,1,1]=(0.699215733397)*x[0]+(0.0596356032853)*x[1]
        arg[0,4,2,0]=(-0.521316010702)*x[0]+(-1.75309802615)*x[1]
        arg[0,4,2,1]=(0.403640172305)*x[0]+(1.41465827077)*x[1]
        arg[1,0,0,0]=(-1.01538089315)*x[0]+(-1.14785578909)*x[1]
        arg[1,0,0,1]=(1.10045201832)*x[0]+(0.630761728795)*x[1]
        arg[1,0,1,0]=(0.137673646147)*x[0]+(-0.387129090253)*x[1]
        arg[1,0,1,1]=(-1.04588287863)*x[0]+(-1.22433029859)*x[1]
        arg[1,0,2,0]=(-0.567830263535)*x[0]+(-0.405808616295)*x[1]
        arg[1,0,2,1]=(0.139326102495)*x[0]+(-1.93153384363)*x[1]
        arg[1,1,0,0]=(-0.638524762921)*x[0]+(0.506405751653)*x[1]
        arg[1,1,0,1]=(0.130775153144)*x[0]+(0.531317202208)*x[1]
        arg[1,1,1,0]=(0.557476206038)*x[0]+(0.447588676184)*x[1]
        arg[1,1,1,1]=(-0.538795916875)*x[0]+(-0.593869493531)*x[1]
        arg[1,1,2,0]=(-1.09827483942)*x[0]+(0.376431061876)*x[1]
        arg[1,1,2,1]=(-0.0074505620387)*x[0]+(-0.798645151486)*x[1]
        arg[1,2,0,0]=(0.251283911459)*x[0]+(0.194595071813)*x[1]
        arg[1,2,0,1]=(1.00482763799)*x[0]+(-1.00059863394)*x[1]
        arg[1,2,1,0]=(-0.453095434684)*x[0]+(-0.513922011418)*x[1]
        arg[1,2,1,1]=(0.558548837484)*x[0]+(0.0696328406825)*x[1]
        arg[1,2,2,0]=(-1.20625552545)*x[0]+(-0.151371710692)*x[1]
        arg[1,2,2,1]=(0.100262209656)*x[0]+(-0.314628804179)*x[1]
        arg[1,3,0,0]=(0.947063141882)*x[0]+(-0.297271200562)*x[1]
        arg[1,3,0,1]=(0.0760277251261)*x[0]+(-1.17982146387)*x[1]
        arg[1,3,1,0]=(-1.46946063969)*x[0]+(-0.615302663567)*x[1]
        arg[1,3,1,1]=(0.419747293729)*x[0]+(-0.629842531949)*x[1]
        arg[1,3,2,0]=(-1.44607954917)*x[0]+(0.520024093924)*x[1]
        arg[1,3,2,1]=(-1.01556392596)*x[0]+(-0.307173811354)*x[1]
        arg[1,4,0,0]=(-0.164339335244)*x[0]+(0.390736414982)*x[1]
        arg[1,4,0,1]=(-0.266932380288)*x[0]+(-0.731533848078)*x[1]
        arg[1,4,1,0]=(-0.903825489055)*x[0]+(-1.81564784421)*x[1]
        arg[1,4,1,1]=(-0.426008882665)*x[0]+(-1.09792948908)*x[1]
        arg[1,4,2,0]=(-1.32984927589)*x[0]+(1.58196302541)*x[1]
        arg[1,4,2,1]=(-0.0581032806145)*x[0]+(-0.323073355302)*x[1]
        arg[2,0,0,0]=(-0.744014251373)*x[0]+(-0.818253254872)*x[1]
        arg[2,0,0,1]=(0.368656965396)*x[0]+(-0.0864646380237)*x[1]
        arg[2,0,1,0]=(-0.112428845773)*x[0]+(0.109170104421)*x[1]
        arg[2,0,1,1]=(0.0682241548911)*x[0]+(-1.65999482283)*x[1]
        arg[2,0,2,0]=(0.212777058993)*x[0]+(0.127163480945)*x[1]
        arg[2,0,2,1]=(-0.256989443356)*x[0]+(0.183206687414)*x[1]
        arg[2,1,0,0]=(0.578997994373)*x[0]+(0.413087453689)*x[1]
        arg[2,1,0,1]=(0.886532090807)*x[0]+(0.330645525532)*x[1]
        arg[2,1,1,0]=(0.18224742814)*x[0]+(-0.713166591262)*x[1]
        arg[2,1,1,1]=(-0.25218296706)*x[0]+(0.736797897014)*x[1]
        arg[2,1,2,0]=(0.885270371408)*x[0]+(-1.35641397127)*x[1]
        arg[2,1,2,1]=(-0.963214290879)*x[0]+(0.348421598509)*x[1]
        arg[2,2,0,0]=(0.884120745351)*x[0]+(0.956857507884)*x[1]
        arg[2,2,0,1]=(-0.00795298517561)*x[0]+(0.51243740322)*x[1]
        arg[2,2,1,0]=(-0.563655252975)*x[0]+(-0.716305131721)*x[1]
        arg[2,2,1,1]=(0.648243491623)*x[0]+(-0.142239786485)*x[1]
        arg[2,2,2,0]=(1.50677848542)*x[0]+(-0.359722006818)*x[1]
        arg[2,2,2,1]=(-0.522592086302)*x[0]+(-0.18544927497)*x[1]
        arg[2,3,0,0]=(0.0822787762966)*x[0]+(0.196309662733)*x[1]
        arg[2,3,0,1]=(0.232573258627)*x[0]+(-0.0597976775432)*x[1]
        arg[2,3,1,0]=(1.50895737052)*x[0]+(0.546625934312)*x[1]
        arg[2,3,1,1]=(-1.64023373286)*x[0]+(0.220053261777)*x[1]
        arg[2,3,2,0]=(1.5897208842)*x[0]+(-0.0355885594992)*x[1]
        arg[2,3,2,1]=(-1.52822349982)*x[0]+(-0.0906934717656)*x[1]
        arg[2,4,0,0]=(0.496696582601)*x[0]+(-0.303888675059)*x[1]
        arg[2,4,0,1]=(-0.0819586112177)*x[0]+(-0.0449047294378)*x[1]
        arg[2,4,1,0]=(-0.678530758513)*x[0]+(-1.2173963447)*x[1]
        arg[2,4,1,1]=(-0.645450475268)*x[0]+(0.356771113426)*x[1]
        arg[2,4,2,0]=(-0.486643021867)*x[0]+(0.977763869689)*x[1]
        arg[2,4,2,1]=(-0.458991807397)*x[0]+(1.80038432923)*x[1]
        arg[3,0,0,0]=(0.270795948755)*x[0]+(-0.691512096338)*x[1]
        arg[3,0,0,1]=(-0.756717130099)*x[0]+(1.16189640534)*x[1]
        arg[3,0,1,0]=(0.967554346699)*x[0]+(-0.272119008963)*x[1]
        arg[3,0,1,1]=(1.14296491195)*x[0]+(-0.707175662379)*x[1]
        arg[3,0,2,0]=(1.15665030692)*x[0]+(0.414803389409)*x[1]
        arg[3,0,2,1]=(1.38997274955)*x[0]+(1.07065568192)*x[1]
        arg[3,1,0,0]=(-0.623998925511)*x[0]+(0.375545806421)*x[1]
        arg[3,1,0,1]=(-0.363431948398)*x[0]+(-0.183100417297)*x[1]
        arg[3,1,1,0]=(0.0435866301729)*x[0]+(-0.557137572068)*x[1]
        arg[3,1,1,1]=(-0.253225217468)*x[0]+(-0.0657932188329)*x[1]
        arg[3,1,2,0]=(0.147700921064)*x[0]+(0.681624563015)*x[1]
        arg[3,1,2,1]=(-0.075394297616)*x[0]+(0.271741245789)*x[1]
        arg[3,2,0,0]=(1.34803044812)*x[0]+(0.87991065609)*x[1]
        arg[3,2,0,1]=(0.0529487135353)*x[0]+(0.806480092352)*x[1]
        arg[3,2,1,0]=(-1.52038624166)*x[0]+(-0.992718138685)*x[1]
        arg[3,2,1,1]=(0.0859852188749)*x[0]+(1.63512596202)*x[1]
        arg[3,2,2,0]=(1.46925148575)*x[0]+(-1.60383564005)*x[1]
        arg[3,2,2,1]=(0.114463395473)*x[0]+(-1.0186377376)*x[1]
        arg[3,3,0,0]=(-1.02962502658)*x[0]+(-1.06788543146)*x[1]
        arg[3,3,0,1]=(-0.184391597652)*x[0]+(0.462603936483)*x[1]
        arg[3,3,1,0]=(0.611113335782)*x[0]+(0.282417397043)*x[1]
        arg[3,3,1,1]=(-0.193423873796)*x[0]+(-0.56108614964)*x[1]
        arg[3,3,2,0]=(-0.0290687598224)*x[0]+(-0.460221158417)*x[1]
        arg[3,3,2,1]=(0.784513522959)*x[0]+(-0.484813129206)*x[1]
        arg[3,4,0,0]=(0.423780748664)*x[0]+(-0.344020787894)*x[1]
        arg[3,4,0,1]=(1.10623024451)*x[0]+(-0.775008153301)*x[1]
        arg[3,4,1,0]=(0.554847534291)*x[0]+(-0.854749107999)*x[1]
        arg[3,4,1,1]=(-0.639595693597)*x[0]+(-0.966152813212)*x[1]
        arg[3,4,2,0]=(-0.0172621996023)*x[0]+(0.321815074269)*x[1]
        arg[3,4,2,1]=(0.39605156824)*x[0]+(-0.499138231749)*x[1]
        ref[0,0,0,0]=(0.169892838337)/(o+1.)+(-0.323261281472)+(0.722227117759)*0.5**o
        ref[0,0,0,1]=(-0.899797330597)/(o+1.)+(0.731242549464)+(-0.679761749104)*0.5**o
        ref[0,0,1,0]=(0.948969615551)/(o+1.)+(-0.173478478707)+(0.0756189154575)*0.5**o
        ref[0,0,1,1]=(-0.484419525942)/(o+1.)+(0.287761319651)+(-0.834046890353)*0.5**o
        ref[0,0,2,0]=(-0.275366085384)/(o+1.)+(-0.051636507926)+(0.534757021171)*0.5**o
        ref[0,0,2,1]=(0.338862631582)/(o+1.)+(0.076489529483)+(-0.47622767501)*0.5**o
        ref[0,1,0,0]=(0.960885993642)/(o+1.)+(0.36959166751)+(0.789134441414)*0.5**o
        ref[0,1,0,1]=(-0.101908900511)/(o+1.)+(0.343409131566)+(0.298411594744)*0.5**o
        ref[0,1,1,0]=(-0.50227382988)/(o+1.)+(-0.431549057432)+(-0.509965959533)*0.5**o
        ref[0,1,1,1]=(-0.294216689224)/(o+1.)+(0.293114907564)+(-0.00647512035965)*0.5**o
        ref[0,1,2,0]=(0.330313139948)/(o+1.)+(0.0601559902451)+(-0.447667920962)*0.5**o
        ref[0,1,2,1]=(-0.855325117375)/(o+1.)+(0.455706690618)+(0.0237326178252)*0.5**o
        ref[0,2,0,0]=(0.958948557218)/(o+1.)+(-0.307015973897)+(-0.752766039908)*0.5**o
        ref[0,2,0,1]=(0.622666086232)/(o+1.)+(0.417782845058)+(-0.676857571283)*0.5**o
        ref[0,2,1,0]=(-0.476269525636)/(o+1.)+(0.203518519787)+(-0.541895856898)*0.5**o
        ref[0,2,1,1]=(0.396862410812)/(o+1.)+(-0.318442063423)+(0.499722654464)*0.5**o
        ref[0,2,2,0]=(0.291546173382)/(o+1.)+(0.345688404218)+(0.372569619405)*0.5**o
        ref[0,2,2,1]=(-0.0863432016899)/(o+1.)+(0.296494638577)+(-0.631266836064)*0.5**o
        ref[0,3,0,0]=(0.437179077921)/(o+1.)+(-0.340515851088)+(-0.00622121586734)*0.5**o
        ref[0,3,0,1]=(-0.0670787919003)/(o+1.)+(-0.492735838895)+(0.931769856293)*0.5**o
        ref[0,3,1,0]=(-0.455620785864)/(o+1.)+(-0.866886545624)+(0.632565106759)*0.5**o
        ref[0,3,1,1]=(0.769395985412)/(o+1.)+(-0.303642587154)+(0.781062911223)*0.5**o
        ref[0,3,2,0]=(-0.338160377272)/(o+1.)+(-0.237545833658)+(-0.966712929597)*0.5**o
        ref[0,3,2,1]=(-0.796435534783)/(o+1.)+(0.544922318125)+(-0.743274373676)*0.5**o
        ref[0,4,0,0]=(0.452932120968)/(o+1.)+(-0.512874769603)+(-0.048932296957)*0.5**o
        ref[0,4,0,1]=(0.361579034112)/(o+1.)+(-0.0561290568189)+(0.365767627701)*0.5**o
        ref[0,4,1,0]=(-0.626053161593)/(o+1.)+(-0.761324946295)+(0.911283010254)*0.5**o
        ref[0,4,1,1]=(-0.0455977379835)/(o+1.)+(0.292122388548)+(0.22020429757)*0.5**o
        ref[0,4,2,0]=(-0.881052488675)/(o+1.)+(-0.210171400955)+(-0.973018746263)*0.5**o
        ref[0,4,2,1]=(0.955475719603)/(o+1.)+(0.367959093676)+(0.126904536122)*0.5**o
        ref[1,0,0,0]=(-0.314523133291)/(o+1.)+(-0.649561146707)+(-0.54959125554)*0.5**o
        ref[1,0,0,1]=(0.776726613012)/(o+1.)+(0.0747896120381)+(0.804907910029)*0.5**o
        ref[1,0,1,0]=(0.416309554773)/(o+1.)+(0.0650606132082)+(-0.795886225295)*0.5**o
        ref[1,0,1,1]=(-0.642083920221)/(o+1.)+(-0.518812997818)+(-0.590503261366)*0.5**o
        ref[1,0,2,0]=(0.209400082886)/(o+1.)+(-0.592256267485)+(0.00147357225359)*0.5**o
        ref[1,0,2,1]=(-0.969945221754)/(o+1.)+(-0.364559871358)+(-0.0931427766681)*0.5**o
        ref[1,1,0,0]=(0.941138659497)/(o+1.)+(-0.103485523138)+(-0.866286624489)*0.5**o
        ref[1,1,0,1]=(0.527395399401)/(o+1.)+(0.373358049173)+(-0.612019142395)*0.5**o
        ref[1,1,1,0]=(-0.276942103917)/(o+1.)+(0.624509599487)+(0.0329877871648)*0.5**o
        ref[1,1,1,1]=(-0.153477829221)/(o+1.)+(-0.0616268433725)+(-0.85593389444)*0.5**o
        ref[1,1,2,0]=(0.2685285297)/(o+1.)+(-0.301824551915)+(-0.386723203413)*0.5**o
        ref[1,1,2,1]=(-0.530960074652)/(o+1.)+(0.248621312801)+(-0.772378264474)*0.5**o
        ref[1,2,0,0]=(0.745071931955)/(o+1.)+(-0.149268301427)+(-0.000656345828701)*0.5**o
        ref[1,2,0,1]=(-0.581822428087)/(o+1.)+(0.2315691847)+(0.122913062738)*0.5**o
        ref[1,2,1,0]=(-0.325165099021)/(o+1.)+(-0.414758139139)+(0.187663931198)*0.5**o
        ref[1,2,1,1]=(-0.389683593964)/(o+1.)+(0.265985790556)+(0.485893691018)*0.5**o
        ref[1,2,2,0]=(0.661969003098)/(o+1.)+(-0.514584521701)+(-0.990427195841)*0.5**o
        ref[1,2,2,1]=(-0.0828384828188)/(o+1.)+(-0.482166570368)+(0.832805029032)*0.5**o
        ref[1,3,0,0]=(-0.756020671713)/(o+1.)+(0.627704058228)+(0.150404496577)*0.5**o
        ref[1,3,0,1]=(-0.93648010541)/(o+1.)+(0.11022703478)+(-0.387767702896)*0.5**o
        ref[1,3,1,0]=(-0.995823513997)/(o+1.)+(-0.242072805569)+(-0.604794178119)*0.5**o
        ref[1,3,1,1]=(-0.543071878541)/(o+1.)+(-0.160890620667)+(0.654757881656)*0.5**o
        ref[1,3,2,0]=(0.803184441579)/(o+1.)+(-0.612293875048)+(-0.504652146723)*0.5**o
        ref[1,3,2,1]=(-0.786714439322)/(o+1.)+(-0.175556007517)+(-0.18491128296)*0.5**o
        ref[1,4,0,0]=(-0.302928744298)/(o+1.)+(0.476360987754)+(-0.423396151472)*0.5**o
        ref[1,4,0,1]=(-0.410240602476)/(o+1.)+(-0.49551630458)+(0.40280698327)*0.5**o
        ref[1,4,1,0]=(-0.820677400346)/(o+1.)+(-0.573037933363)+(-0.752720066188)*0.5**o
        ref[1,4,1,1]=(-0.492748923658)/(o+1.)+(-0.600739806799)+(0.170290165507)*0.5**o
        ref[1,4,2,0]=(0.926654770824)/(o+1.)+(0.011282908181)+(-0.697106837662)*0.5**o
        ref[1,4,2,1]=(-0.958477382611)/(o+1.)+(0.618515858902)+(-0.65973097111)*0.5**o
        ref[2,0,0,0]=(-0.15636950624)/(o+1.)+(-0.520670547262)+(-0.36455690548)*0.5**o
        ref[2,0,0,1]=(-0.726424878286)/(o+1.)+(0.697757734621)+(-0.386898263585)*0.5**o
        ref[2,0,1,0]=(-0.055162393508)/(o+1.)+(0.247015646426)+(-0.442127640696)*0.5**o
        ref[2,0,1,1]=(-0.769332288875)/(o+1.)+(-0.0917175733456)+(-0.639003232373)*0.5**o
        ref[2,0,2,0]=(-0.196390038297)/(o+1.)+(0.0977907270548)+(0.340749124126)*0.5**o
        ref[2,0,2,1]=(0.841927481859)/(o+1.)+(-0.489185426146)+(0.0626606144906)*0.5**o
        ref[2,1,0,0]=(0.590018249148)/(o+1.)+(0.378476967182)+(-0.354886735451)*0.5**o
        ref[2,1,0,1]=(0.178495604742)/(o+1.)+(0.233073081069)+(0.572535849459)*0.5**o
        ref[2,1,1,0]=(-0.762671064919)/(o+1.)+(-0.108539686142)+(0.448831274081)*0.5**o
        ref[2,1,1,1]=(0.927591406535)/(o+1.)+(0.0441635703243)+(-0.53130361723)*0.5**o
        ref[2,1,2,0]=(-0.604360706259)/(o+1.)+(-0.134848102332)+(0.402913311063)*0.5**o
        ref[2,1,2,1]=(0.229727627441)/(o+1.)+(-0.323296604293)+(-0.197927111225)*0.5**o
        ref[2,2,0,0]=(0.287478356112)/(o+1.)+(0.762635506201)+(0.0282288847209)*0.5**o
        ref[2,2,0,1]=(0.243518182621)/(o+1.)+(-0.241502069199)+(0.743970373822)*0.5**o
        ref[2,2,1,0]=(-0.369049760378)/(o+1.)+(-0.21941778965)+(-0.472075045017)*0.5**o
        ref[2,2,1,1]=(-0.745343658666)/(o+1.)+(0.193375557775)+(0.864596248255)*0.5**o
        ref[2,2,2,0]=(-0.491769251494)/(o+1.)+(0.442742387618)+(0.753340954862)*0.5**o
        ref[2,2,2,1]=(0.551196377158)/(o+1.)+(-0.523535406613)+(-0.212166925203)*0.5**o
        ref[2,3,0,0]=(0.192271522168)/(o+1.)+(-0.0181953944687)+(0.122707705798)*0.5**o
        ref[2,3,0,1]=(-0.593768738594)/(o+1.)+(0.199088910545)+(0.368366498588)*0.5**o
        ref[2,3,1,0]=(0.0866211808983)/(o+1.)+(0.723512694127)+(0.521936735685)*0.5**o
        ref[2,3,1,1]=(-0.453288268254)/(o+1.)+(-0.15580970281)+(-0.655272797205)*0.5**o
        ref[2,3,2,0]=(0.614105802669)/(o+1.)+(0.0411427694898)+(0.85774098305)*0.5**o
        ref[2,3,2,1]=(0.263133533654)/(o+1.)+(-0.628776355867)+(-0.624497793507)*0.5**o
        ref[2,4,0,0]=(-0.809347914722)/(o+1.)+(0.367719412237)+(0.26671699779)*0.5**o
        ref[2,4,0,1]=(0.773124137566)/(o+1.)+(-0.442496395542)+(-0.0149946871367)*0.5**o
        ref[2,4,1,0]=(-0.927311525126)/(o+1.)+(-0.0125177466568)+(-0.943580084777)*0.5**o
        ref[2,4,1,1]=(0.306409665483)/(o+1.)+(0.128569553604)+(-0.852228134533)*0.5**o
        ref[2,4,2,0]=(0.926185522145)/(o+1.)+(0.0750538146738)+(-0.58517230367)*0.5**o
        ref[2,4,2,1]=(0.858245470922)/(o+1.)+(0.0125876600893)+(0.457971730734)*0.5**o
        ref[3,0,0,0]=(-0.74501947953)/(o+1.)+(0.343942243752)+(-0.363581155556)*0.5**o
        ref[3,0,0,1]=(0.850890737948)/(o+1.)+(0.157997002133)+(-0.761705466968)*0.5**o
        ref[3,0,1,0]=(-0.0468962248591)/(o+1.)+(0.0297980532763)+(0.682735456043)*0.5**o
        ref[3,0,1,1]=(0.0126326425456)/(o+1.)+(-0.076115296096)+(0.575387199213)*0.5**o
        ref[3,0,2,0]=(-0.410549752333)/(o+1.)+(0.85556811978)+(0.270867209104)*0.5**o
        ref[3,0,2,1]=(0.580487848545)/(o+1.)+(0.59069171729)+(0.698757148346)*0.5**o
        ref[3,1,0,0]=(0.139052775786)/(o+1.)+(-0.334747475889)+(0.281989056901)*0.5**o
        ref[3,1,0,1]=(0.591083610998)/(o+1.)+(-0.322680889668)+(-0.492254197357)*0.5**o
        ref[3,1,1,0]=(0.384713965427)/(o+1.)+(-0.522411606102)+(0.146558304882)*0.5**o
        ref[3,1,1,1]=(-0.827615582009)/(o+1.)+(-0.00295218575854)+(0.514501517225)*0.5**o
        ref[3,1,2,0]=(0.84762578765)/(o+1.)+(0.208361555685)+(-0.43502341494)*0.5**o
        ref[3,1,2,1]=(-0.554668615664)/(o+1.)+(0.825347835483)+(-0.899680107129)*0.5**o
        ref[3,2,0,0]=(0.652802712143)/(o+1.)+(0.397197582731)+(0.7807432266)*0.5**o
        ref[3,2,0,1]=(0.421495251937)/(o+1.)+(-0.119571710512)+(0.677076974975)*0.5**o
        ref[3,2,1,0]=(-0.230439630375)/(o+1.)+(-0.733165339017)+(-0.816334071941)*0.5**o
        ref[3,2,1,1]=(0.64153308098)/(o+1.)+(0.778374176432)+(-0.477170252948)*0.5**o
        ref[3,2,2,0]=(-0.987848717403)/(o+1.)+(-0.0606435888479)+(0.974551740798)*0.5**o
        ref[3,2,2,1]=(-0.318593711859)/(o+1.)+(-0.459904454409)+(0.334228278552)*0.5**o
        ref[3,3,0,0]=(-0.418115413351)/(o+1.)+(-0.348616063735)+(-0.982162917221)*0.5**o
        ref[3,3,0,1]=(-0.437684435015)/(o+1.)+(0.804145379606)+(-0.892393985366)*0.5**o
        ref[3,3,1,0]=(0.944100110135)/(o+1.)+(-0.059994562245)+(0.0694197471801)*0.5**o
        ref[3,3,1,1]=(-0.860656048613)/(o+1.)+(0.178136109535)+(-0.250126193893)*0.5**o
        ref[3,3,2,0]=(-0.363631991908)/(o+1.)+(-0.298384287282)+(0.471110648232)*0.5**o
        ref[3,3,2,1]=(-0.578021736479)/(o+1.)+(0.320913094365)+(0.235895941501)*0.5**o
        ref[3,4,0,0]=(-0.946995649946)/(o+1.)+(0.0451978887455)+(0.936359833225)*0.5**o
        ref[3,4,0,1]=(-0.636305818291)/(o+1.)+(0.258932088634)+(0.449663732234)*0.5**o
        ref[3,4,1,0]=(-0.823421937366)/(o+1.)+(0.112980674356)+(0.297559014947)*0.5**o
        ref[3,4,1,1]=(-0.441768022626)/(o+1.)+(-0.240840962617)+(-0.682298558949)*0.5**o
        ref[3,4,2,0]=(-0.258808936822)/(o+1.)+(0.510306577882)+(-0.457251344276)*0.5**o
        ref[3,4,2,1]=(-0.489365749016)/(o+1.)+(-0.296617385332)+(0.979513856171)*0.5**o
      else:
        arg[0,0,0,0]=(-0.474247989728)*x[0]+(-0.768689352235)*x[1]+(-1.08527900853)*x[2]
        arg[0,0,0,1]=(-0.237113797789)*x[0]+(0.0842735741018)*x[1]+(0.203858551029)*x[2]
        arg[0,0,1,0]=(0.24237127665)*x[0]+(0.421022955803)*x[1]+(-1.52818540927)*x[2]
        arg[0,0,1,1]=(1.34624865817)*x[0]+(-0.798943272818)*x[1]+(-0.823168386315)*x[2]
        arg[0,0,2,0]=(-0.906723771267)*x[0]+(0.328893123771)*x[1]+(-0.942770202463)*x[2]
        arg[0,0,2,1]=(1.54305197694)*x[0]+(0.813063941984)*x[1]+(0.205532727168)*x[2]
        arg[0,1,0,0]=(0.399159979872)*x[0]+(-1.52089837934)*x[1]+(-1.4064251406)*x[2]
        arg[0,1,0,1]=(-0.121139354731)*x[0]+(1.12768246422)*x[1]+(0.234299409411)*x[2]
        arg[0,1,1,0]=(-1.62513899388)*x[0]+(-0.173131700583)*x[1]+(1.24199996799)*x[2]
        arg[0,1,1,1]=(-0.371929434177)*x[0]+(-0.606278610409)*x[1]+(-0.00600790063066)*x[2]
        arg[0,1,2,0]=(-0.804531294423)*x[0]+(0.869392248002)*x[1]+(0.840417359943)*x[2]
        arg[0,1,2,1]=(0.329452946275)*x[0]+(1.1072495446)*x[1]+(-0.213517932323)*x[2]
        arg[0,2,0,0]=(1.02784887224)*x[0]+(0.435517042337)*x[1]+(1.58767206677)*x[2]
        arg[0,2,0,1]=(-0.0120997177)*x[0]+(0.430135803314)*x[1]+(-0.717473068956)*x[2]
        arg[0,2,1,0]=(0.955455517302)*x[0]+(0.712976918968)*x[1]+(-0.624804508641)*x[2]
        arg[0,2,1,1]=(-1.03293986594)*x[0]+(-0.539062834482)*x[1]+(-1.61172701428)*x[2]
        arg[0,2,2,0]=(-0.177662810138)*x[0]+(0.553352366127)*x[1]+(0.695417121111)*x[2]
        arg[0,2,2,1]=(0.318133474679)*x[0]+(-0.452528947926)*x[1]+(0.468602505091)*x[2]
        arg[0,3,0,0]=(-0.294467725916)*x[0]+(1.67314409894)*x[1]+(0.100736440007)*x[2]
        arg[0,3,0,1]=(-0.517162349887)*x[0]+(-1.57430191254)*x[1]+(1.24045877402)*x[2]
        arg[0,3,1,0]=(0.667393640368)*x[0]+(0.340993632212)*x[1]+(-1.09920535559)*x[2]
        arg[0,3,1,1]=(0.37451406727)*x[0]+(0.37350649326)*x[1]+(0.357440871322)*x[2]
        arg[0,3,2,0]=(-0.989366398194)*x[0]+(0.596179154759)*x[1]+(-0.573280308087)*x[2]
        arg[0,3,2,1]=(-0.25561190141)*x[0]+(1.42296905238)*x[1]+(-1.21032031399)*x[2]
        arg[0,4,0,0]=(-0.41394085285)*x[0]+(-0.751213929646)*x[1]+(-0.68470718293)*x[2]
        arg[0,4,0,1]=(0.702985204167)*x[0]+(-0.0375821582587)*x[1]+(0.290630803819)*x[2]
        arg[0,4,1,0]=(0.0558204653441)*x[0]+(1.61940057302)*x[1]+(-0.518863528354)*x[2]
        arg[0,4,1,1]=(1.25876303604)*x[0]+(-1.40923202908)*x[1]+(1.05202333762)*x[2]
        arg[0,4,2,0]=(0.0715295483294)*x[0]+(0.883869702529)*x[1]+(-0.463253474408)*x[2]
        arg[0,4,2,1]=(0.564563789615)*x[0]+(-0.135049742988)*x[1]+(-1.3118878567)*x[2]
        arg[1,0,0,0]=(0.707267335291)*x[0]+(-0.197244449872)*x[1]+(-0.114362773781)*x[2]
        arg[1,0,0,1]=(-0.144762815659)*x[0]+(0.434177706452)*x[1]+(0.0754859099691)*x[2]
        arg[1,0,1,0]=(-0.906934991359)*x[0]+(0.179011995627)*x[1]+(0.823936036469)*x[2]
        arg[1,0,1,1]=(1.11165724605)*x[0]+(0.48023223731)*x[1]+(0.981353808181)*x[2]
        arg[1,0,2,0]=(0.499460583481)*x[0]+(0.171586243561)*x[1]+(-0.252474775657)*x[2]
        arg[1,0,2,1]=(-0.604289370884)*x[0]+(1.11610914086)*x[1]+(0.747439062711)*x[2]
        arg[1,1,0,0]=(0.696191463882)*x[0]+(0.349547714404)*x[1]+(0.651361019042)*x[2]
        arg[1,1,0,1]=(0.152679923426)*x[0]+(-0.448442911425)*x[1]+(1.33964371501)*x[2]
        arg[1,1,1,0]=(-0.74612592015)*x[0]+(-1.27089239786)*x[1]+(-0.261725631736)*x[2]
        arg[1,1,1,1]=(1.50839493929)*x[0]+(1.93229724987)*x[1]+(-0.963996845831)*x[2]
        arg[1,1,2,0]=(0.867984447375)*x[0]+(-0.777493811854)*x[1]+(0.847847553133)*x[2]
        arg[1,1,2,1]=(1.34249389498)*x[0]+(0.776160804062)*x[1]+(-0.295393972199)*x[2]
        arg[1,2,0,0]=(-0.655771613243)*x[0]+(1.58220214455)*x[1]+(-0.406292477629)*x[2]
        arg[1,2,0,1]=(0.59294895986)*x[0]+(0.544961164723)*x[1]+(0.634279670948)*x[2]
        arg[1,2,1,0]=(1.15324179862)*x[0]+(0.645764835217)*x[1]+(0.670364725309)*x[2]
        arg[1,2,1,1]=(-0.819459106271)*x[0]+(0.024815625404)*x[1]+(0.769753650676)*x[2]
        arg[1,2,2,0]=(1.19042366161)*x[0]+(0.982020548972)*x[1]+(0.92730609738)*x[2]
        arg[1,2,2,1]=(-0.742441488349)*x[0]+(-1.3101429812)*x[1]+(-0.198104246324)*x[2]
        arg[1,3,0,0]=(0.349205030437)*x[0]+(-0.745240339596)*x[1]+(-0.744742092259)*x[2]
        arg[1,3,0,1]=(0.993101296711)*x[0]+(-1.48331355036)*x[1]+(-1.00616778677)*x[2]
        arg[1,3,1,0]=(-1.10586442824)*x[0]+(1.71922449779)*x[1]+(-0.943451153259)*x[2]
        arg[1,3,1,1]=(-1.72393763035)*x[0]+(-1.14113495956)*x[1]+(-0.211909615809)*x[2]
        arg[1,3,2,0]=(-0.378302963108)*x[0]+(-0.494705322002)*x[1]+(0.940164363332)*x[2]
        arg[1,3,2,1]=(0.850541148743)*x[0]+(-1.45461945468)*x[1]+(1.20025821328)*x[2]
        arg[1,4,0,0]=(1.33882463795)*x[0]+(0.475208133119)*x[1]+(0.722686059253)*x[2]
        arg[1,4,0,1]=(0.0750422155969)*x[0]+(1.29758288845)*x[1]+(-1.41431668738)*x[2]
        arg[1,4,1,0]=(1.36072344575)*x[0]+(-0.345743547563)*x[1]+(-0.63248018754)*x[2]
        arg[1,4,1,1]=(0.579049229913)*x[0]+(0.954245074078)*x[1]+(-0.170538926965)*x[2]
        arg[1,4,2,0]=(1.67615484634)*x[0]+(0.561067019023)*x[1]+(-1.26952326914)*x[2]
        arg[1,4,2,1]=(0.700910547934)*x[0]+(0.58717328764)*x[1]+(0.795138196965)*x[2]
        arg[2,0,0,0]=(0.763193592359)*x[0]+(-0.397549328381)*x[1]+(-0.230593559545)*x[2]
        arg[2,0,0,1]=(0.454026153468)*x[0]+(-1.12464443717)*x[1]+(-0.147462098246)*x[2]
        arg[2,0,1,0]=(-0.276457034935)*x[0]+(0.118173156654)*x[1]+(0.57296366275)*x[2]
        arg[2,0,1,1]=(1.31887812253)*x[0]+(-0.544644483254)*x[1]+(0.0013728350582)*x[2]
        arg[2,0,2,0]=(-0.257589384265)*x[0]+(-0.016005453203)*x[1]+(-0.508836430965)*x[2]
        arg[2,0,2,1]=(-0.259420301773)*x[0]+(0.122645293168)*x[1]+(0.381613885289)*x[2]
        arg[2,1,0,0]=(-1.64168512418)*x[0]+(0.025182006192)*x[1]+(0.0268780458975)*x[2]
        arg[2,1,0,1]=(-0.0885344200898)*x[0]+(0.408795219114)*x[1]+(0.635798801606)*x[2]
        arg[2,1,1,0]=(-0.959658503283)*x[0]+(1.034415556)*x[1]+(0.725681105748)*x[2]
        arg[2,1,1,1]=(1.59312707789)*x[0]+(-0.116514952101)*x[1]+(-0.86121633507)*x[2]
        arg[2,1,2,0]=(-1.44859572067)*x[0]+(-1.28344247275)*x[1]+(0.299296401869)*x[2]
        arg[2,1,2,1]=(-0.488424372501)*x[0]+(1.23481402288)*x[1]+(0.01702902463)*x[2]
        arg[2,2,0,0]=(-1.29986951244)*x[0]+(-1.20190912751)*x[1]+(-0.755848291051)*x[2]
        arg[2,2,0,1]=(-0.52421656312)*x[0]+(-0.335471509689)*x[1]+(-1.01264312499)*x[2]
        arg[2,2,1,0]=(-1.12066642609)*x[0]+(-1.02138558053)*x[1]+(-1.00777755552)*x[2]
        arg[2,2,1,1]=(-1.42127024744)*x[0]+(1.63480022479)*x[1]+(0.542244906643)*x[2]
        arg[2,2,2,0]=(-0.418852880297)*x[0]+(-0.983110436453)*x[1]+(-0.622603673805)*x[2]
        arg[2,2,2,1]=(0.727366446825)*x[0]+(-0.256324856361)*x[1]+(-0.20608740758)*x[2]
        arg[2,3,0,0]=(0.548268302883)*x[0]+(0.0102173127665)*x[1]+(0.869833576217)*x[2]
        arg[2,3,0,1]=(-1.70159091061)*x[0]+(1.15876875083)*x[1]+(-0.234757706773)*x[2]
        arg[2,3,1,0]=(-1.72817024817)*x[0]+(0.690055132218)*x[1]+(-0.149398998825)*x[2]
        arg[2,3,1,1]=(1.80829412387)*x[0]+(-0.217899654855)*x[1]+(-0.922672217822)*x[2]
        arg[2,3,2,0]=(1.23855200328)*x[0]+(0.412795411508)*x[1]+(0.960476191755)*x[2]
        arg[2,3,2,1]=(-1.43061904621)*x[0]+(-0.766039000745)*x[1]+(-1.25765397361)*x[2]
        arg[2,4,0,0]=(-0.660872016687)*x[0]+(-0.000405233718708)*x[1]+(-0.178507968219)*x[2]
        arg[2,4,0,1]=(-0.168526882871)*x[0]+(-1.06936802685)*x[1]+(1.54083857104)*x[2]
        arg[2,4,1,0]=(0.565807359529)*x[0]+(-1.41401297939)*x[1]+(-1.23254260537)*x[2]
        arg[2,4,1,1]=(-0.373920075802)*x[0]+(0.407330812903)*x[1]+(-0.204667001328)*x[2]
        arg[2,4,2,0]=(0.874100879885)*x[0]+(-0.191238094267)*x[1]+(1.54890969328)*x[2]
        arg[2,4,2,1]=(0.566428125313)*x[0]+(-1.60899595601)*x[1]+(-1.05112529768)*x[2]
        arg[3,0,0,0]=(-0.688037248111)*x[0]+(-1.56483484784)*x[1]+(0.26782504619)*x[2]
        arg[3,0,0,1]=(0.418788954459)*x[0]+(-0.670455226092)*x[1]+(-0.0562760591375)*x[2]
        arg[3,0,1,0]=(-0.2614362754)*x[0]+(0.271050519855)*x[1]+(0.227811982923)*x[2]
        arg[3,0,1,1]=(-0.873198928647)*x[0]+(-0.483763141395)*x[1]+(0.33532383979)*x[2]
        arg[3,0,2,0]=(-0.864161938176)*x[0]+(-1.4682754386)*x[1]+(-0.38126411162)*x[2]
        arg[3,0,2,1]=(0.142827521381)*x[0]+(0.464483940188)*x[1]+(0.214297576606)*x[2]
        arg[3,1,0,0]=(0.188747079113)*x[0]+(0.590821253713)*x[1]+(0.908850367108)*x[2]
        arg[3,1,0,1]=(0.0587541582624)*x[0]+(0.955494266151)*x[1]+(1.70822209849)*x[2]
        arg[3,1,1,0]=(-0.420356861523)*x[0]+(-0.551445918262)*x[1]+(-0.0187114754269)*x[2]
        arg[3,1,1,1]=(1.07071658401)*x[0]+(-0.589035602194)*x[1]+(0.133725074027)*x[2]
        arg[3,1,2,0]=(0.370477043767)*x[0]+(0.226262353618)*x[1]+(0.566087784489)*x[2]
        arg[3,1,2,1]=(1.77828660075)*x[0]+(0.607419742685)*x[1]+(0.0297054744582)*x[2]
        arg[3,2,0,0]=(-1.19326939474)*x[0]+(1.20875153887)*x[1]+(-0.124756222091)*x[2]
        arg[3,2,0,1]=(0.0645190914271)*x[0]+(0.357580475716)*x[1]+(-1.59461469718)*x[2]
        arg[3,2,1,0]=(1.4853619905)*x[0]+(0.751582301648)*x[1]+(0.859363177167)*x[2]
        arg[3,2,1,1]=(1.36658655075)*x[0]+(-0.776650996613)*x[1]+(0.654743994261)*x[2]
        arg[3,2,2,0]=(0.406696005677)*x[0]+(-0.528546876384)*x[1]+(-0.439775325396)*x[2]
        arg[3,2,2,1]=(-0.725198159461)*x[0]+(-0.0933136564416)*x[1]+(-0.720365141393)*x[2]
        arg[3,3,0,0]=(-0.428669112669)*x[0]+(0.626162511175)*x[1]+(-0.382776233923)*x[2]
        arg[3,3,0,1]=(0.340474939218)*x[0]+(-0.140550383925)*x[1]+(0.353836983746)*x[2]
        arg[3,3,1,0]=(0.851646816711)*x[0]+(0.0171684195949)*x[1]+(0.38173044322)*x[2]
        arg[3,3,1,1]=(0.359647722225)*x[0]+(0.425799941165)*x[1]+(-0.80174052212)*x[2]
        arg[3,3,2,0]=(0.13012568736)*x[0]+(-0.573735693651)*x[1]+(1.08843730088)*x[2]
        arg[3,3,2,1]=(-1.01372953294)*x[0]+(0.335282956856)*x[1]+(1.24570512628)*x[2]
        arg[3,4,0,0]=(-0.05674609039)*x[0]+(-0.814879575064)*x[1]+(-0.169789408264)*x[2]
        arg[3,4,0,1]=(1.5968230938)*x[0]+(0.0679819277023)*x[1]+(0.892284314512)*x[2]
        arg[3,4,1,0]=(1.22220298478)*x[0]+(-0.0258417926472)*x[1]+(0.440670138568)*x[2]
        arg[3,4,1,1]=(0.647072768716)*x[0]+(-0.177651087906)*x[1]+(-0.115553644646)*x[2]
        arg[3,4,2,0]=(1.29816363306)*x[0]+(-1.18656044167)*x[1]+(-0.465776533157)*x[2]
        arg[3,4,2,1]=(0.328907496091)*x[0]+(-0.179857937151)*x[1]+(-0.476306938306)*x[2]
        ref[0,0,0,0]=(-0.530378359482)/(o+1.)+(-0.935143327523)+(0.0724486640358)*0.5**o
        ref[0,0,0,1]=(1.22837847871)/(o+1.)+(-0.113350954728)+(-0.950658241915)*0.5**o
        ref[0,0,1,0]=(-0.000414683535449)/(o+1.)+(-0.183158731552)+(-0.49805903018)*0.5**o
        ref[0,0,1,1]=(-0.882730389132)/(o+1.)+(-0.112174958816)+(0.831217305802)*0.5**o
        ref[0,0,2,0]=(-1.1913557966)/(o+1.)+(-0.0615750891074)+(-0.206094875146)*0.5**o
        ref[0,0,2,1]=(0.634536492589)/(o+1.)+(0.649890059464)+(0.627332034578)*0.5**o
        ref[0,1,0,0]=(-1.38665625775)/(o+1.)+(-1.0210653524)+(0.900623422471)*0.5**o
        ref[0,1,0,1]=(1.20633678024)/(o+1.)+(0.0888193121856)+(-0.143132885707)*0.5**o
        ref[0,1,1,0]=(0.452225278035)/(o+1.)+(-0.148926943801)+(-0.710642116901)*0.5**o
        ref[0,1,1,1]=(-0.244223430204)/(o+1.)+(-0.619607191996)+(0.499221868978)*0.5**o
        ref[0,1,2,0]=(0.0383849036606)/(o+1.)+(0.759484971644)+(-0.652076533427)*0.5**o
        ref[0,1,2,1]=(0.207558055759)/(o+1.)+(0.502305382384)+(0.0110157380215)*0.5**o
        ref[0,2,0,0]=(0.761374280144)/(o+1.)+(0.777616611515)+(0.734430478174)*0.5**o
        ref[0,2,0,1]=(0.554502862028)/(o+1.)+(-0.409274602016)+(-0.0353906413371)*0.5**o
        ref[0,2,1,0]=(-0.252543841709)/(o+1.)+(0.386640377894)+(0.522891013551)*0.5**o
        ref[0,2,1,1]=(-1.52497235887)/(o+1.)+(-0.807603254401)+(-0.0435508470367)*0.5**o
        ref[0,2,2,0]=(0.172485606257)/(o+1.)+(0.350297526187)+(0.19802601847)*0.5**o
        ref[0,2,2,1]=(-0.141271359379)/(o+1.)+(0.0509198134622)+(0.373638764298)*0.5**o
        ref[0,3,0,0]=(1.31278836809)/(o+1.)+(-0.0918739323256)+(0.350372309595)*0.5**o
        ref[0,3,0,1]=(-0.396933928109)/(o+1.)+(-0.436137805999)+(0.418204051696)*0.5**o
        ref[0,3,1,0]=(-0.206619840099)/(o+1.)+(-0.118703289798)+(0.353208336689)*0.5**o
        ref[0,3,1,1]=(-0.595153371293)/(o+1.)+(0.615722581562)+(0.469169640021)*0.5**o
        ref[0,3,2,0]=(-0.680858997986)/(o+1.)+(-0.0393223937964)+(-0.206963765944)*0.5**o
        ref[0,3,2,1]=(0.012753264598)/(o+1.)+(0.246188741489)+(-0.548093910595)*0.5**o
        ref[0,4,0,0]=(-0.40478987122)/(o+1.)+(-0.890399681609)+(0.33572726901)*0.5**o
        ref[0,4,0,1]=(0.0280062557264)/(o+1.)+(0.0358049413228)+(0.856417711355)*0.5**o
        ref[0,4,1,0]=(0.181560004565)/(o+1.)+(0.329885369854)+(0.31502676574)*0.5**o
        ref[0,4,1,1]=(0.0552681276841)/(o+1.)+(0.00372919426319)+(0.838827828375)*0.5**o
        ref[0,4,2,0]=(0.656057162724)/(o+1.)+(0.179315527684)+(-0.522542441641)*0.5**o
        ref[0,4,2,1]=(0.126189389317)/(o+1.)+(-0.429452001643)+(-0.149659196108)*0.5**o
        ref[1,0,0,0]=(1.43443485527)/(o+1.)+(-0.55091754491)+(0.0630603461901)*0.5**o
        ref[1,0,0,1]=(-0.612191779306)/(o+1.)+(0.604025235381)+(-0.230957890695)*0.5**o
        ref[1,0,1,0]=(0.373085423681)/(o+1.)+(-0.0817391530697)+(-0.113594076805)*0.5**o
        ref[1,0,1,1]=(0.761862677657)/(o+1.)+(0.819524484428)+(0.172331645033)*0.5**o
        ref[1,0,2,0]=(0.929265450746)/(o+1.)+(-0.382585784938)+(0.254478170515)*0.5**o
        ref[1,0,2,1]=(1.58236203463)/(o+1.)+(-0.280641188811)+(0.238179175679)*0.5**o
        ref[1,1,0,0]=(-0.117055726312)/(o+1.)+(0.957914090571)+(-0.101672257501)*0.5**o
        ref[1,1,0,1]=(0.0849062553796)/(o+1.)+(0.0703564566846)+(0.818261558262)*0.5**o
        ref[1,1,1,0]=(-1.72219113622)/(o+1.)+(-0.357454724939)+(0.15835663635)*0.5**o
        ref[1,1,1,1]=(0.739078400468)/(o+1.)+(0.532292543169)+(0.673031856531)*0.5**o
        ref[1,1,2,0]=(0.255023909903)/(o+1.)+(0.229656917655)+(0.22400044344)*0.5**o
        ref[1,1,2,1]=(0.518856765199)/(o+1.)+(0.380552790125)+(0.543298381396)*0.5**o
        ref[1,2,0,0]=(0.0321563884831)/(o+1.)+(0.25981567762)+(-0.0316496900482)*0.5**o
        ref[1,2,0,1]=(1.09318176771)/(o+1.)+(0.0572442907603)+(0.564519446305)*0.5**o
        ref[1,2,1,0]=(0.756962791387)/(o+1.)+(0.69419429624)+(0.32401997528)*0.5**o
        ref[1,2,1,1]=(1.28480299241)/(o+1.)+(-0.201990593541)+(-0.905711635518)*0.5**o
        ref[1,2,2,0]=(1.04969566442)/(o+1.)+(0.72854097372)+(0.592972696101)*0.5**o
        ref[1,2,2,1]=(-1.77086264775)/(o+1.)+(-0.261914180361)+(0.0440022926066)*0.5**o
        ref[1,3,0,0]=(-0.477780531124)/(o+1.)+(-0.181354085163)+(-0.300288699967)*0.5**o
        ref[1,3,0,1]=(-0.797627013456)/(o+1.)+(-0.523163955197)+(0.347574883435)*0.5**o
        ref[1,3,1,0]=(0.428806106821)/(o+1.)+(-0.0514584302394)+(-0.655980330052)*0.5**o
        ref[1,3,1,1]=(-0.141740459457)/(o+1.)+(-1.05699960931)+(-0.821242527644)*0.5**o
        ref[1,3,2,0]=(0.0181810985873)/(o+1.)+(-0.180697139553)+(0.41036925874)*0.5**o
        ref[1,3,2,1]=(-0.061077367)/(o+1.)+(0.250077446763)+(0.157102380812)*0.5**o
        ref[1,4,0,0]=(1.14087212913)/(o+1.)+(0.368059029864)+(0.659728641463)*0.5**o
        ref[1,4,0,1]=(0.365658713316)/(o+1.)+(0.088533821044)+(-0.584417938734)*0.5**o
        ref[1,4,1,0]=(-1.11681308321)/(o+1.)+(0.438275560627)+(0.622761672604)*0.5**o
        ref[1,4,1,1]=(1.18893372018)/(o+1.)+(0.172934240923)+(-0.172046824999)*0.5**o
        ref[1,4,2,0]=(0.16240489493)/(o+1.)+(-0.0145923852256)+(0.834478471749)*0.5**o
        ref[1,4,2,1]=(0.571355829378)/(o+1.)+(0.526896417749)+(0.458073367663)*0.5**o
        ref[2,0,0,0]=(-0.489234572937)/(o+1.)+(-0.0135589093901)+(0.65140309615)*0.5**o
        ref[2,0,0,1]=(-1.55383491931)/(o+1.)+(0.0161644619972)+(0.703425613366)*0.5**o
        ref[2,0,1,0]=(-0.258604855723)/(o+1.)+(0.0827812678671)+(0.507722104457)*0.5**o
        ref[2,0,1,1]=(0.0504178529481)/(o+1.)+(0.182991850332)+(0.359204920726)*0.5**o
        ref[2,0,2,0]=(0.652043479841)/(o+1.)+(-0.443015450013)+(-0.548443848248)*0.5**o
        ref[2,0,2,1]=(0.340317839857)/(o+1.)+(-0.379565594545)+(0.663652225917)*0.5**o
        ref[2,1,0,0]=(-0.587863196667)/(o+1.)+(-0.0137797674284)+(-0.974202340562)*0.5**o
        ref[2,1,0,1]=(1.00030747039)/(o+1.)+(-0.399997754768)+(0.755747639771)*0.5**o
        ref[2,1,1,0]=(0.703413281036)/(o+1.)+(0.487016208736)+(-0.877007540039)*0.5**o
        ref[2,1,1,1]=(0.263166830572)/(o+1.)+(-0.157259452849)+(0.666747865841)*0.5**o
        ref[2,1,2,0]=(0.104532343004)/(o+1.)+(-1.02548001493)+(-0.486314104701)*0.5**o
        ref[2,1,2,1]=(0.933518655478)/(o+1.)+(0.0369928229139)+(-0.244085626298)*0.5**o
        ref[2,2,0,0]=(-1.50550384519)/(o+1.)+(-0.590996227594)+(-0.570130630622)*0.5**o
        ref[2,2,0,1]=(-0.60318683839)/(o+1.)+(-0.38308245528)+(-0.502979448853)*0.5**o
        ref[2,2,1,0]=(-1.11258910879)/(o+1.)+(-0.518777277969)+(-0.999685897413)*0.5**o
        ref[2,2,1,1]=(1.84366617107)/(o+1.)+(-0.232866738261)+(-0.622157810558)*0.5**o
        ref[2,2,2,0]=(-0.620181988585)/(o+1.)+(-0.836162444183)+(0.267939886396)*0.5**o
        ref[2,2,2,1]=(-0.826647389099)/(o+1.)+(0.0929536518109)+(0.905694268362)*0.5**o
        ref[2,3,0,0]=(0.184067674443)/(o+1.)+(0.259745858757)+(0.724759799909)*0.5**o
        ref[2,3,0,1]=(0.0175595303049)/(o+1.)+(0.0560117358912)+(-0.907162868639)*0.5**o
        ref[2,3,1,0]=(-0.609443154617)/(o+1.)+(0.186540819401)+(-0.951152598958)*0.5**o
        ref[2,3,1,1]=(-1.61858082094)/(o+1.)+(0.679439858578)+(0.927423354971)*0.5**o
        ref[2,3,2,0]=(0.44350652486)/(o+1.)+(0.81466177481)+(0.53899353207)*0.5**o
        ref[2,3,2,1]=(-0.877289862245)/(o+1.)+(-1.05742260849)+(-0.462176941342)*0.5**o
        ref[2,4,0,0]=(-0.736381416539)/(o+1.)+(-0.0139374259612)+(-0.0755289501641)*0.5**o
        ref[2,4,0,1]=(0.690429326999)/(o+1.)+(-0.0293644018251)+(-0.328756862036)*0.5**o
        ref[2,4,1,0]=(-1.57117034002)/(o+1.)+(-0.531463765387)+(0.553349645561)*0.5**o
        ref[2,4,1,1]=(1.24898175282)/(o+1.)+(-0.452273024959)+(-0.515691967125)*0.5**o
        ref[2,4,2,0]=(0.239126191643)/(o+1.)+(0.722504563932)+(0.547637159392)*0.5**o
        ref[2,4,2,1]=(-1.40763254209)/(o+1.)+(-0.809456441012)+(0.932852295739)*0.5**o
        ref[3,0,0,0]=(-0.416554297686)/(o+1.)+(-0.632193610911)+(-0.304105530258)*0.5**o
        ref[3,0,0,1]=(-1.02983448036)/(o+1.)+(0.579212231778)+(-0.436532313966)*0.5**o
        ref[3,0,1,0]=(0.42124133216)/(o+1.)+(0.368290418827)+(-0.920395942436)*0.5**o
        ref[3,0,1,1]=(0.158195474854)/(o+1.)+(-0.17459474514)+(-0.830644214826)*0.5**o
        ref[3,0,2,0]=(-1.18749647329)/(o+1.)+(-0.506173996178)+(-0.513857022748)*0.5**o
        ref[3,0,2,1]=(0.233437753018)/(o+1.)+(0.455705145371)+(-0.323239005586)*0.5**o
        ref[3,1,0,0]=(1.42991958566)/(o+1.)+(0.053769921022)+(0.150959272234)*0.5**o
        ref[3,1,0,1]=(1.48423731111)/(o+1.)+(0.964048315997)+(-0.689863420196)*0.5**o
        ref[3,1,1,0]=(0.868837676765)/(o+1.)+(-1.01261543254)+(0.165878933098)*0.5**o
        ref[3,1,1,1]=(0.820750307831)/(o+1.)+(-0.300930223821)+(0.396516195659)*0.5**o
        ref[3,1,2,0]=(0.477077782202)/(o+1.)+(-0.0416336989494)+(0.769016797571)*0.5**o
        ref[3,1,2,1]=(0.359895440587)/(o+1.)+(0.555322395811)+(0.944871585683)*0.5**o
        ref[3,2,0,0]=(0.657668315832)/(o+1.)+(-0.240321895174)+(-0.286298603451)*0.5**o
        ref[3,2,0,1]=(-0.941360568282)/(o+1.)+(-0.0246544136593)+(-0.181845734439)*0.5**o
        ref[3,2,1,0]=(1.60316834971)/(o+1.)+(0.252291602298)+(0.988555915005)*0.5**o
        ref[3,2,1,1]=(0.833811957678)/(o+1.)+(-0.111970621871)+(0.634808834458)*0.5**o
        ref[3,2,2,0]=(-1.38877073784)/(o+1.)+(0.538356636307)+(-0.249568730877)*0.5**o
        ref[3,2,2,1]=(-0.311839184097)/(o+1.)+(-0.698114103264)+(0.169190433329)*0.5**o
        ref[3,3,0,0]=(0.743170989696)/(o+1.)+(-0.499224746788)+(0.0699956684628)*0.5**o
        ref[3,3,0,1]=(-0.293538786051)/(o+1.)+(0.26667992866)+(0.313940467768)*0.5**o
        ref[3,3,1,0]=(0.951483461345)/(o+1.)+(0.148317904561)+(0.00242640905887)*0.5**o
        ref[3,3,1,1]=(-0.110753912776)/(o+1.)+(-0.44308689138)+(0.980634836805)*0.5**o
        ref[3,3,2,0]=(-0.337895223544)/(o+1.)+(0.358941646037)+(0.264839226057)*0.5**o
        ref[3,3,2,1]=(0.857823611315)/(o+1.)+(0.30404611512)+(-0.898657291359)*0.5**o
        ref[3,4,0,0]=(0.037944509376)/(o+1.)+(-0.364801700413)+(-0.349756182268)*0.5**o
        ref[3,4,0,1]=(1.01114399396)/(o+1.)+(0.457103279485)+(0.631738783076)*0.5**o
        ref[3,4,1,0]=(0.969931913888)/(o+1.)+(-0.165922085756)+(0.998943588321)*0.5**o
        ref[3,4,1,1]=(1.35441114248)/(o+1.)+(-0.502422727199)+(0.00430234808496)*0.5**o
        ref[3,4,2,0]=(-0.367201097566)/(o+1.)+(-0.337780780531)+(0.688589316866)*0.5**o
        ref[3,4,2,1]=(-0.0547905846061)/(o+1.)+(-0.293870420685)+(0.31527404661)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedContinuousFunction_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactOne

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.241972460279)*x[0]+(0.539729925673)*x[1]
        ref=(-0.11832904353)/(o+1.)+(0.649911364492)+(-0.399791299503)*0.5**o
      else:
        arg=(0.53919797656)*x[0]+(0.379010664884)*x[1]+(0.274416064529)*x[2]
        ref=(0.307947301472)/(o+1.)+(0.120674157748)+(0.643329089005)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedContinuousFunction_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactOne

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.84849017056)*x[0]+(-0.161962845202)*x[1]
        arg[1]=(-0.794717919135)*x[0]+(1.16003773358)*x[1]
        ref[0]=(0.780514448202)/(o+1.)+(-0.34628958021)+(0.598592037576)*0.5**o
        ref[1]=(0.570158156278)/(o+1.)+(0.241192316704)+(-0.687222975237)*0.5**o
      else:
        arg[0]=(-0.637352163634)*x[0]+(1.49477462643)*x[1]+(-0.284018740238)*x[2]
        arg[1]=(0.125937459904)*x[0]+(-0.206499869891)*x[1]+(-0.342856240345)*x[2]
        ref[0]=(-0.274486314451)/(o+1.)+(0.379737553853)+(0.0884149293071)*0.5**o
        ref[1]=(-0.78452814021)/(o+1.)+(-0.122017410137)+(0.605144310151)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedContinuousFunction_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactOne

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(-0.440868762581)*x[0]+(-0.216223666616)*x[1]
        arg[0,1]=(-1.78807355591)*x[0]+(-0.208278123838)*x[1]
        arg[0,2]=(-0.448066998103)*x[0]+(-0.819815507826)*x[1]
        arg[0,3]=(1.27247881689)*x[0]+(0.0653439567221)*x[1]
        arg[0,4]=(-1.00344990223)*x[0]+(-0.291263790583)*x[1]
        arg[1,0]=(0.440080423838)*x[0]+(0.0356836558113)*x[1]
        arg[1,1]=(-1.49545340091)*x[0]+(-0.133615590211)*x[1]
        arg[1,2]=(1.05947638385)*x[0]+(0.408685904238)*x[1]
        arg[1,3]=(-0.170309086605)*x[0]+(0.660523970285)*x[1]
        arg[1,4]=(0.764323678475)*x[0]+(-0.529415080303)*x[1]
        arg[2,0]=(0.55270081325)*x[0]+(-1.60630146231)*x[1]
        arg[2,1]=(1.57419330248)*x[0]+(-1.24644288706)*x[1]
        arg[2,2]=(0.0416757238363)*x[0]+(0.0247379738465)*x[1]
        arg[2,3]=(1.0200370393)*x[0]+(0.364209137693)*x[1]
        arg[2,4]=(-1.21858490389)*x[0]+(0.523056254546)*x[1]
        arg[3,0]=(0.0524205820842)*x[0]+(1.65953217854)*x[1]
        arg[3,1]=(1.21824883395)*x[0]+(0.905190834998)*x[1]
        arg[3,2]=(0.226344434525)*x[0]+(-0.735349531462)*x[1]
        arg[3,3]=(-1.00295245717)*x[0]+(-0.719605376499)*x[1]
        arg[3,4]=(-0.446228759693)*x[0]+(1.23441974277)*x[1]
        ref[0,0]=(0.347328582602)/(o+1.)+(-0.520484463519)+(0.0365479152384)*0.5**o
        ref[0,1]=(-0.274564442352)/(o+1.)+(-0.413557210678)+(-0.894672816039)*0.5**o
        ref[0,2]=(0.0720644688954)/(o+1.)+(-0.906636966188)+(0.473326957552)*0.5**o
        ref[0,3]=(-0.747749341777)/(o+1.)+(0.730249972378)+(0.625072170633)*0.5**o
        ref[0,4]=(-0.194031116169)/(o+1.)+(-0.432063325829)+(-0.236555924983)*0.5**o
        ref[1,0]=(0.807081538033)/(o+1.)+(-0.612266650007)+(0.893215841629)*0.5**o
        ref[1,1]=(-0.662516518957)/(o+1.)+(-0.176318925886)+(-0.61391462039)*0.5**o
        ref[1,2]=(-0.570345636156)/(o+1.)+(0.791877954963)+(0.454752014319)*0.5**o
        ref[1,3]=(-0.312400910795)/(o+1.)+(0.212935137466)+(0.376745519544)*0.5**o
        ref[1,4]=(-0.764680414264)/(o+1.)+(0.174215486579)+(0.651158039278)*0.5**o
        ref[2,0]=(-0.837942771642)/(o+1.)+(-0.187938486702)+(0.160219095985)*0.5**o
        ref[2,1]=(-0.672421039761)/(o+1.)+(0.17384002882)+(0.652491397549)*0.5**o
        ref[2,2]=(-0.656187949922)/(o+1.)+(0.559888736121)+(-0.397175824637)*0.5**o
        ref[2,3]=(0.259558368195)/(o+1.)+(0.55148644978)+(0.0217149092406)*0.5**o
        ref[2,4]=(-0.188397537901)/(o+1.)+(-0.0561706170512)+(-0.394789877345)*0.5**o
        ref[3,0]=(0.91055628617)/(o+1.)+(0.574211274098)+(-0.347026073739)*0.5**o
        ref[3,1]=(0.29225509237)/(o+1.)+(0.64447705482)+(0.542230466941)*0.5**o
        ref[3,2]=(0.121065624384)/(o+1.)+(0.0316495277836)+(-0.693369776888)*0.5**o
        ref[3,3]=(-0.943523334976)/(o+1.)+(-0.266302644031)+(-0.246429210629)*0.5**o
        ref[3,4]=(0.714915399455)/(o+1.)+(0.459963131994)+(-0.846650680369)*0.5**o
      else:
        arg[0,0]=(-0.493869713454)*x[0]+(-0.381969413515)*x[1]+(1.59862368848)*x[2]
        arg[0,1]=(-0.581245543547)*x[0]+(0.961117271889)*x[1]+(-0.158073515343)*x[2]
        arg[0,2]=(-0.635283143641)*x[0]+(0.695573236987)*x[1]+(-0.719180832278)*x[2]
        arg[0,3]=(-0.737780596733)*x[0]+(-1.11171283352)*x[1]+(0.459954346315)*x[2]
        arg[0,4]=(1.12522065074)*x[0]+(-1.08427904854)*x[1]+(-0.715679673556)*x[2]
        arg[1,0]=(0.972468186181)*x[0]+(-0.211014530435)*x[1]+(-0.734414099002)*x[2]
        arg[1,1]=(-1.67337704748)*x[0]+(0.871096213089)*x[1]+(1.51063313223)*x[2]
        arg[1,2]=(-0.222013007834)*x[0]+(1.46474508326)*x[1]+(-0.979243908862)*x[2]
        arg[1,3]=(1.21195281087)*x[0]+(0.274082389329)*x[1]+(1.56144436465)*x[2]
        arg[1,4]=(1.35812865884)*x[0]+(1.25819798853)*x[1]+(0.895334048395)*x[2]
        arg[2,0]=(-0.167404344437)*x[0]+(0.0710591934315)*x[1]+(-0.124469683665)*x[2]
        arg[2,1]=(-0.492240632996)*x[0]+(-0.317251117382)*x[1]+(0.0195461703529)*x[2]
        arg[2,2]=(-0.277590266225)*x[0]+(-1.65859476899)*x[1]+(-0.0100216405668)*x[2]
        arg[2,3]=(-0.0701281298455)*x[0]+(0.502137254703)*x[1]+(0.701220923443)*x[2]
        arg[2,4]=(-0.72186096407)*x[0]+(-0.900487568179)*x[1]+(-0.735573922826)*x[2]
        arg[3,0]=(-0.0677324474129)*x[0]+(-0.649963209718)*x[1]+(-0.770861099726)*x[2]
        arg[3,1]=(0.563588189481)*x[0]+(0.714614932251)*x[1]+(-0.0957764632886)*x[2]
        arg[3,2]=(1.13337395062)*x[0]+(0.507556316073)*x[1]+(-0.196759085949)*x[2]
        arg[3,3]=(-1.2636883168)*x[0]+(-0.811660344076)*x[1]+(0.645462052598)*x[2]
        arg[3,4]=(-0.104175186987)*x[0]+(-0.929176038155)*x[1]+(1.07530379856)*x[2]
        ref[0,0]=(0.84538674973)/(o+1.)+(0.0569713556632)+(-0.236544899547)*0.5**o
        ref[0,1]=(-0.307689589699)/(o+1.)+(0.245040337642)+(0.0394071274131)*0.5**o
        ref[0,2]=(0.955244736357)/(o+1.)+(-0.341771136504)+(-0.930593202281)*0.5**o
        ref[0,3]=(-0.353490461722)/(o+1.)+(-0.476228433901)+(-0.083591754415)*0.5**o
        ref[0,4]=(-1.31220485482)/(o+1.)+(0.0466426656676)+(0.544181452128)*0.5**o
        ref[1,0]=(0.22773475414)/(o+1.)+(-0.488454801654)+(0.776214405914)*0.5**o
        ref[1,1]=(1.50688678707)/(o+1.)+(0.0851415404695)+(-0.96881757017)*0.5**o
        ref[1,2]=(0.313471097102)/(o+1.)+(-0.0875594518156)+(0.125135973096)*0.5**o
        ref[1,3]=(0.269521220179)/(o+1.)+(1.1192007195)+(0.539556905661)*0.5**o
        ref[1,4]=(1.91499876064)/(o+1.)+(0.509063797893)+(0.578534339343)*0.5**o
        ref[2,0]=(0.941352651562)/(o+1.)+(-0.966199402186)+(0.770231318139)*0.5**o
        ref[2,1]=(-0.141490823562)/(o+1.)+(-0.20919987022)+(-0.230055016023)*0.5**o
        ref[2,2]=(-0.916148047971)/(o+1.)+(-0.371740600192)+(-0.286577427426)*0.5**o
        ref[2,3]=(0.229516219331)/(o+1.)+(0.274790532736)+(0.354132763498)*0.5**o
        ref[2,4]=(-1.19120612619)/(o+1.)+(-0.139859859385)+(-0.88699661011)*0.5**o
        ref[3,0]=(-1.48630121465)/(o+1.)+(0.0744166251063)+(-0.151088792417)*0.5**o
        ref[3,1]=(-0.335583560513)/(o+1.)+(0.277436410841)+(0.963137397276)*0.5**o
        ref[3,2]=(-0.673942688644)/(o+1.)+(0.70077509905)+(0.716563671283)*0.5**o
        ref[3,3]=(-0.148394092795)/(o+1.)+(-0.237190239072)+(-0.807112037336)*0.5**o
        ref[3,4]=(0.529983886211)/(o+1.)+(-0.371055083588)+(0.254078854382)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedContinuousFunction_rank3(self):
      """
      tests integral of rank 3 Data on the FunctionOnContactOne

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(6, 2, 2),w)
      ref=numpy.zeros((6, 2, 2),numpy.float_)
      if dim==2:
        arg[0,0,0]=(0.661725715165)*x[0]+(0.219720717581)*x[1]
        arg[0,0,1]=(-0.642240082304)*x[0]+(-0.279652756776)*x[1]
        arg[0,1,0]=(0.0204235813151)*x[0]+(-0.374873249933)*x[1]
        arg[0,1,1]=(1.4418481252)*x[0]+(0.205656834772)*x[1]
        arg[1,0,0]=(0.65992179951)*x[0]+(-1.04887875458)*x[1]
        arg[1,0,1]=(-0.195480216199)*x[0]+(-0.373905275039)*x[1]
        arg[1,1,0]=(0.781251543352)*x[0]+(0.0416248109536)*x[1]
        arg[1,1,1]=(-0.925751319189)*x[0]+(-0.113992910046)*x[1]
        arg[2,0,0]=(-0.0767414065227)*x[0]+(-0.156682840973)*x[1]
        arg[2,0,1]=(0.688457272529)*x[0]+(0.524916336046)*x[1]
        arg[2,1,0]=(0.753722830084)*x[0]+(-0.890325839466)*x[1]
        arg[2,1,1]=(0.730789290145)*x[0]+(-0.278986703716)*x[1]
        arg[3,0,0]=(-0.287376796439)*x[0]+(-0.153987233224)*x[1]
        arg[3,0,1]=(-1.15687435255)*x[0]+(0.250005890238)*x[1]
        arg[3,1,0]=(-0.845312830629)*x[0]+(0.256211962352)*x[1]
        arg[3,1,1]=(-0.574311820519)*x[0]+(-0.440297035998)*x[1]
        arg[4,0,0]=(-0.360607647308)*x[0]+(-1.29380437557)*x[1]
        arg[4,0,1]=(1.32019433485)*x[0]+(0.295255956618)*x[1]
        arg[4,1,0]=(-1.64162941972)*x[0]+(0.117482372162)*x[1]
        arg[4,1,1]=(-0.303103646619)*x[0]+(-0.88458982681)*x[1]
        arg[5,0,0]=(-0.666761879038)*x[0]+(-0.735786278877)*x[1]
        arg[5,0,1]=(0.0892896382617)*x[0]+(-0.45478872773)*x[1]
        arg[5,1,0]=(-0.50985624415)*x[0]+(-0.400022939567)*x[1]
        arg[5,1,1]=(1.39240706368)*x[0]+(0.0149740822664)*x[1]
        ref[0,0,0]=(-0.114834538488)/(o+1.)+(0.240621006484)+(0.515038958264)*0.5**o
        ref[0,0,1]=(0.588698337077)/(o+1.)+(-0.702256377233)+(-0.10607842169)*0.5**o
        ref[0,1,0]=(0.13849818224)/(o+1.)+(-0.277555277991)+(0.0621627051247)*0.5**o
        ref[0,1,1]=(-0.0878009521163)/(o+1.)+(0.630168348947)+(0.474969214192)*0.5**o
        ref[1,0,0]=(-0.696600538111)/(o+1.)+(-0.343318260912)+(0.994280104861)*0.5**o
        ref[1,0,1]=(-0.0887026688527)/(o+1.)+(-0.385753227979)+(0.290823633572)*0.5**o
        ref[1,1,0]=(0.831408608956)/(o+1.)+(-0.242985671853)+(0.477439089057)*0.5**o
        ref[1,1,1]=(-0.621059053191)/(o+1.)+(-0.0700203481241)+(-0.278644479797)*0.5**o
        ref[2,0,0]=(0.7890720399)/(o+1.)+(-0.539721196393)+(0.0569461053905)*0.5**o
        ref[2,0,1]=(-0.228628151401)/(o+1.)+(0.826946074171)+(-0.211890388366)*0.5**o
        ref[2,1,0]=(-0.837617325493)/(o+1.)+(0.289656653751)+(0.121701008608)*0.5**o
        ref[2,1,1]=(0.246577988625)/(o+1.)+(0.0512227634376)+(0.102779070928)*0.5**o
        ref[3,0,0]=(-0.0109584684212)/(o+1.)+(-0.334888666171)+(0.239371771102)*0.5**o
        ref[3,0,1]=(-0.125987676419)/(o+1.)+(-0.170179004032)+(-0.440522777827)*0.5**o
        ref[3,1,0]=(-0.354093141636)/(o+1.)+(-0.177294275919)+(0.119580825198)*0.5**o
        ref[3,1,1]=(-0.746761075412)/(o+1.)+(0.192068760731)+(-0.651985302566)*0.5**o
        ref[4,0,0]=(-0.42411673219)/(o+1.)+(-0.579162944937)+(-0.0719694008158)*0.5**o
        ref[4,0,1]=(0.169881169584)/(o+1.)+(0.483614029132)+(0.478341063616)*0.5**o
        ref[4,1,0]=(0.943567836264)/(o+1.)+(-0.898782941264)+(-0.670149001297)*0.5**o
        ref[4,1,1]=(-0.703610807936)/(o+1.)+(0.225342628657)+(-0.934767922806)*0.5**o
        ref[5,0,0]=(-0.788792225213)/(o+1.)+(-0.1768857093)+(-0.259984514102)*0.5**o
        ref[5,0,1]=(-0.788708879899)/(o+1.)+(-0.202288491141)+(0.827786772711)*0.5**o
        ref[5,1,0]=(-0.823081809707)/(o+1.)+(0.274720963093)+(-0.636239300197)*0.5**o
        ref[5,1,1]=(0.498922549807)/(o+1.)+(0.0129388257746)+(0.882580944588)*0.5**o
      else:
        arg[0,0,0]=(-0.426613654844)*x[0]+(-1.01433980801)*x[1]+(0.564621874916)*x[2]
        arg[0,0,1]=(-0.438521968882)*x[0]+(1.6033775267)*x[1]+(0.691349995832)*x[2]
        arg[0,1,0]=(-0.04337513731)*x[0]+(0.142927668234)*x[1]+(0.506581723199)*x[2]
        arg[0,1,1]=(-0.120598868162)*x[0]+(-0.213666787813)*x[1]+(-1.04221679332)*x[2]
        arg[1,0,0]=(0.153696490712)*x[0]+(-0.122516529815)*x[1]+(-1.08837272712)*x[2]
        arg[1,0,1]=(0.410706464237)*x[0]+(0.720079363315)*x[1]+(-0.767790357539)*x[2]
        arg[1,1,0]=(-1.03726428173)*x[0]+(-0.316809853003)*x[1]+(-0.780405868082)*x[2]
        arg[1,1,1]=(-0.427547967216)*x[0]+(-0.897766643609)*x[1]+(-1.56913552404)*x[2]
        arg[2,0,0]=(-0.282789248898)*x[0]+(-0.110846936031)*x[1]+(-1.4329391906)*x[2]
        arg[2,0,1]=(1.46866588617)*x[0]+(-1.23022501959)*x[1]+(-0.3123390018)*x[2]
        arg[2,1,0]=(-0.314115718621)*x[0]+(1.7333934438)*x[1]+(0.833572700726)*x[2]
        arg[2,1,1]=(0.464246462638)*x[0]+(0.267484068646)*x[1]+(0.342009204787)*x[2]
        arg[3,0,0]=(0.578853099163)*x[0]+(-0.0453855508465)*x[1]+(-0.0751578475516)*x[2]
        arg[3,0,1]=(-0.526734485752)*x[0]+(-1.03286310016)*x[1]+(-0.398986583917)*x[2]
        arg[3,1,0]=(0.0239088277798)*x[0]+(0.205105470192)*x[1]+(-1.04986349166)*x[2]
        arg[3,1,1]=(0.130893744001)*x[0]+(-0.273702140669)*x[1]+(-0.0804222415128)*x[2]
        arg[4,0,0]=(-0.822661618584)*x[0]+(-0.267189204519)*x[1]+(-0.290013766971)*x[2]
        arg[4,0,1]=(-1.46759823734)*x[0]+(0.317465945316)*x[1]+(0.108284317752)*x[2]
        arg[4,1,0]=(-1.3920990386)*x[0]+(-1.17823778694)*x[1]+(-0.18366434307)*x[2]
        arg[4,1,1]=(-0.0915496209516)*x[0]+(-0.367624609832)*x[1]+(1.30630257323)*x[2]
        arg[5,0,0]=(0.117840830455)*x[0]+(-0.667875433194)*x[1]+(-1.59716688455)*x[2]
        arg[5,0,1]=(-0.818693227605)*x[0]+(0.483842075248)*x[1]+(-0.408373205535)*x[2]
        arg[5,1,0]=(1.36906770301)*x[0]+(0.494926179999)*x[1]+(0.717041707262)*x[2]
        arg[5,1,1]=(0.819757116081)*x[0]+(0.0982458749015)*x[1]+(0.275080391599)*x[2]
        ref[0,0,0]=(0.236560699949)/(o+1.)+(-0.392784388084)+(-0.327323511722)*0.5**o
        ref[0,0,1]=(1.32491522045)/(o+1.)+(0.525146863711)+(-0.519003394222)*0.5**o
        ref[0,1,0]=(0.433749865563)/(o+1.)+(0.23007126008)+(-0.287758131599)*0.5**o
        ref[0,1,1]=(-0.0563021060686)/(o+1.)+(-0.403921160703)+(-0.512338021825)*0.5**o
        ref[1,0,0]=(-0.170496965963)/(o+1.)+(-0.914421777259)+(0.942147754258)*0.5**o
        ref[1,0,1]=(-0.372707483726)/(o+1.)+(0.636041158872)+(-0.536379364005)*0.5**o
        ref[1,1,0]=(-0.211529878364)/(o+1.)+(-0.912151985776)+(-0.0986461529033)*0.5**o
        ref[1,1,1]=(-1.32533998437)/(o+1.)+(-0.685030198386)+(-0.19904975372)*0.5**o
        ref[2,0,0]=(0.152048902469)/(o+1.)+(-0.88880590209)+(-0.201012473813)*0.5**o
        ref[2,0,1]=(-0.748900052246)/(o+1.)+(-0.0226120471998)+(0.720226011425)*0.5**o
        ref[2,1,0]=(0.77797707941)/(o+1.)+(1.01635055442)+(-0.557827762351)*0.5**o
        ref[2,1,1]=(1.08969334716)/(o+1.)+(-0.404644728844)+(0.793335846595)*0.5**o
        ref[3,0,0]=(1.56498448669)/(o+1.)+(-0.699967188685)+(0.293259591449)*0.5**o
        ref[3,0,1]=(-0.620133427128)/(o+1.)+(-0.550037413688)+(-0.238375915329)*0.5**o
        ref[3,1,0]=(-0.156380586288)/(o+1.)+(-0.0154369973565)+(-0.633594612687)*0.5**o
        ref[3,1,1]=(-0.352989559326)/(o+1.)+(-0.290547031627)+(0.710852984398)*0.5**o
        ref[4,0,0]=(-0.147579361355)/(o+1.)+(-0.615173424765)+(-0.00193837918766)*0.5**o
        ref[4,0,1]=(0.598152345277)/(o+1.)+(-0.540248606426)+(-0.559503106696)*0.5**o
        ref[4,1,0]=(-1.17182448825)/(o+1.)+(-0.590551681576)+(-0.401073317208)*0.5**o
        ref[4,1,1]=(0.837504141173)/(o+1.)+(0.0807414267186)+(-0.15185865216)*0.5**o
        ref[5,0,0]=(-0.646601115533)/(o+1.)+(-1.20895265273)+(0.9173049337)*0.5**o
        ref[5,0,1]=(1.26626110052)/(o+1.)+(-0.56471923973)+(-0.880046978955)*0.5**o
        ref[5,1,0]=(-0.069063169298)/(o+1.)+(0.883536796964)+(0.883025165644)*0.5**o
        ref[5,1,1]=(1.05999338993)/(o+1.)+(-0.392519102893)+(0.918128198439)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_ReducedContinuousFunction_rank4(self):
      """
      tests integral of rank 4 Data on the FunctionOnContactOne

      assumptions: ReducedContinuousFunction(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=1
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=ReducedContinuousFunction(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5, 3, 2),w)
      ref=numpy.zeros((4, 5, 3, 2),numpy.float_)
      if dim==2:
        arg[0,0,0,0]=(0.257898226293)*x[0]+(-0.01230083314)*x[1]
        arg[0,0,0,1]=(0.0531183070647)*x[0]+(-0.170192287836)*x[1]
        arg[0,0,1,0]=(-0.364644940864)*x[0]+(1.04227651446)*x[1]
        arg[0,0,1,1]=(-0.894334108203)*x[0]+(0.151390331211)*x[1]
        arg[0,0,2,0]=(1.03825401708)*x[0]+(-0.882136097149)*x[1]
        arg[0,0,2,1]=(-1.05304910155)*x[0]+(1.06866311708)*x[1]
        arg[0,1,0,0]=(1.54742542159)*x[0]+(0.941778348482)*x[1]
        arg[0,1,0,1]=(1.1797380956)*x[0]+(-0.296417138234)*x[1]
        arg[0,1,1,0]=(-1.09196028027)*x[0]+(-0.783377624006)*x[1]
        arg[0,1,1,1]=(0.370952191443)*x[0]+(-0.0854141858991)*x[1]
        arg[0,1,2,0]=(-1.23552368139)*x[0]+(1.23848088087)*x[1]
        arg[0,1,2,1]=(0.216345350114)*x[0]+(-0.136524468427)*x[1]
        arg[0,2,0,0]=(-1.21206901263)*x[0]+(0.804219582144)*x[1]
        arg[0,2,0,1]=(0.192956526208)*x[0]+(0.588417678858)*x[1]
        arg[0,2,1,0]=(-0.940175603742)*x[0]+(0.329047260784)*x[1]
        arg[0,2,1,1]=(-0.138597387792)*x[0]+(0.398298326223)*x[1]
        arg[0,2,2,0]=(0.10089766824)*x[0]+(1.25459493298)*x[1]
        arg[0,2,2,1]=(-0.401163525883)*x[0]+(0.276542765282)*x[1]
        arg[0,3,0,0]=(0.252156795268)*x[0]+(-0.50223063539)*x[1]
        arg[0,3,0,1]=(0.466244610905)*x[0]+(-0.587025224301)*x[1]
        arg[0,3,1,0]=(-0.268082012713)*x[0]+(-1.28874675764)*x[1]
        arg[0,3,1,1]=(1.10867463519)*x[0]+(-0.165500912864)*x[1]
        arg[0,3,2,0]=(-1.94681908266)*x[0]+(0.166854108476)*x[1]
        arg[0,3,2,1]=(-0.537150417922)*x[0]+(0.0872851457131)*x[1]
        arg[0,4,0,0]=(-0.376548581807)*x[0]+(-0.245201133388)*x[1]
        arg[0,4,0,1]=(-0.135287690767)*x[0]+(0.750376238942)*x[1]
        arg[0,4,1,0]=(-0.0240491596907)*x[0]+(-1.21337088424)*x[1]
        arg[0,4,1,1]=(0.699215733397)*x[0]+(0.0596356032853)*x[1]
        arg[0,4,2,0]=(-0.521316010702)*x[0]+(-1.75309802615)*x[1]
        arg[0,4,2,1]=(0.403640172305)*x[0]+(1.41465827077)*x[1]
        arg[1,0,0,0]=(-1.01538089315)*x[0]+(-1.14785578909)*x[1]
        arg[1,0,0,1]=(1.10045201832)*x[0]+(0.630761728795)*x[1]
        arg[1,0,1,0]=(0.137673646147)*x[0]+(-0.387129090253)*x[1]
        arg[1,0,1,1]=(-1.04588287863)*x[0]+(-1.22433029859)*x[1]
        arg[1,0,2,0]=(-0.567830263535)*x[0]+(-0.405808616295)*x[1]
        arg[1,0,2,1]=(0.139326102495)*x[0]+(-1.93153384363)*x[1]
        arg[1,1,0,0]=(-0.638524762921)*x[0]+(0.506405751653)*x[1]
        arg[1,1,0,1]=(0.130775153144)*x[0]+(0.531317202208)*x[1]
        arg[1,1,1,0]=(0.557476206038)*x[0]+(0.447588676184)*x[1]
        arg[1,1,1,1]=(-0.538795916875)*x[0]+(-0.593869493531)*x[1]
        arg[1,1,2,0]=(-1.09827483942)*x[0]+(0.376431061876)*x[1]
        arg[1,1,2,1]=(-0.0074505620387)*x[0]+(-0.798645151486)*x[1]
        arg[1,2,0,0]=(0.251283911459)*x[0]+(0.194595071813)*x[1]
        arg[1,2,0,1]=(1.00482763799)*x[0]+(-1.00059863394)*x[1]
        arg[1,2,1,0]=(-0.453095434684)*x[0]+(-0.513922011418)*x[1]
        arg[1,2,1,1]=(0.558548837484)*x[0]+(0.0696328406825)*x[1]
        arg[1,2,2,0]=(-1.20625552545)*x[0]+(-0.151371710692)*x[1]
        arg[1,2,2,1]=(0.100262209656)*x[0]+(-0.314628804179)*x[1]
        arg[1,3,0,0]=(0.947063141882)*x[0]+(-0.297271200562)*x[1]
        arg[1,3,0,1]=(0.0760277251261)*x[0]+(-1.17982146387)*x[1]
        arg[1,3,1,0]=(-1.46946063969)*x[0]+(-0.615302663567)*x[1]
        arg[1,3,1,1]=(0.419747293729)*x[0]+(-0.629842531949)*x[1]
        arg[1,3,2,0]=(-1.44607954917)*x[0]+(0.520024093924)*x[1]
        arg[1,3,2,1]=(-1.01556392596)*x[0]+(-0.307173811354)*x[1]
        arg[1,4,0,0]=(-0.164339335244)*x[0]+(0.390736414982)*x[1]
        arg[1,4,0,1]=(-0.266932380288)*x[0]+(-0.731533848078)*x[1]
        arg[1,4,1,0]=(-0.903825489055)*x[0]+(-1.81564784421)*x[1]
        arg[1,4,1,1]=(-0.426008882665)*x[0]+(-1.09792948908)*x[1]
        arg[1,4,2,0]=(-1.32984927589)*x[0]+(1.58196302541)*x[1]
        arg[1,4,2,1]=(-0.0581032806145)*x[0]+(-0.323073355302)*x[1]
        arg[2,0,0,0]=(-0.744014251373)*x[0]+(-0.818253254872)*x[1]
        arg[2,0,0,1]=(0.368656965396)*x[0]+(-0.0864646380237)*x[1]
        arg[2,0,1,0]=(-0.112428845773)*x[0]+(0.109170104421)*x[1]
        arg[2,0,1,1]=(0.0682241548911)*x[0]+(-1.65999482283)*x[1]
        arg[2,0,2,0]=(0.212777058993)*x[0]+(0.127163480945)*x[1]
        arg[2,0,2,1]=(-0.256989443356)*x[0]+(0.183206687414)*x[1]
        arg[2,1,0,0]=(0.578997994373)*x[0]+(0.413087453689)*x[1]
        arg[2,1,0,1]=(0.886532090807)*x[0]+(0.330645525532)*x[1]
        arg[2,1,1,0]=(0.18224742814)*x[0]+(-0.713166591262)*x[1]
        arg[2,1,1,1]=(-0.25218296706)*x[0]+(0.736797897014)*x[1]
        arg[2,1,2,0]=(0.885270371408)*x[0]+(-1.35641397127)*x[1]
        arg[2,1,2,1]=(-0.963214290879)*x[0]+(0.348421598509)*x[1]
        arg[2,2,0,0]=(0.884120745351)*x[0]+(0.956857507884)*x[1]
        arg[2,2,0,1]=(-0.00795298517561)*x[0]+(0.51243740322)*x[1]
        arg[2,2,1,0]=(-0.563655252975)*x[0]+(-0.716305131721)*x[1]
        arg[2,2,1,1]=(0.648243491623)*x[0]+(-0.142239786485)*x[1]
        arg[2,2,2,0]=(1.50677848542)*x[0]+(-0.359722006818)*x[1]
        arg[2,2,2,1]=(-0.522592086302)*x[0]+(-0.18544927497)*x[1]
        arg[2,3,0,0]=(0.0822787762966)*x[0]+(0.196309662733)*x[1]
        arg[2,3,0,1]=(0.232573258627)*x[0]+(-0.0597976775432)*x[1]
        arg[2,3,1,0]=(1.50895737052)*x[0]+(0.546625934312)*x[1]
        arg[2,3,1,1]=(-1.64023373286)*x[0]+(0.220053261777)*x[1]
        arg[2,3,2,0]=(1.5897208842)*x[0]+(-0.0355885594992)*x[1]
        arg[2,3,2,1]=(-1.52822349982)*x[0]+(-0.0906934717656)*x[1]
        arg[2,4,0,0]=(0.496696582601)*x[0]+(-0.303888675059)*x[1]
        arg[2,4,0,1]=(-0.0819586112177)*x[0]+(-0.0449047294378)*x[1]
        arg[2,4,1,0]=(-0.678530758513)*x[0]+(-1.2173963447)*x[1]
        arg[2,4,1,1]=(-0.645450475268)*x[0]+(0.356771113426)*x[1]
        arg[2,4,2,0]=(-0.486643021867)*x[0]+(0.977763869689)*x[1]
        arg[2,4,2,1]=(-0.458991807397)*x[0]+(1.80038432923)*x[1]
        arg[3,0,0,0]=(0.270795948755)*x[0]+(-0.691512096338)*x[1]
        arg[3,0,0,1]=(-0.756717130099)*x[0]+(1.16189640534)*x[1]
        arg[3,0,1,0]=(0.967554346699)*x[0]+(-0.272119008963)*x[1]
        arg[3,0,1,1]=(1.14296491195)*x[0]+(-0.707175662379)*x[1]
        arg[3,0,2,0]=(1.15665030692)*x[0]+(0.414803389409)*x[1]
        arg[3,0,2,1]=(1.38997274955)*x[0]+(1.07065568192)*x[1]
        arg[3,1,0,0]=(-0.623998925511)*x[0]+(0.375545806421)*x[1]
        arg[3,1,0,1]=(-0.363431948398)*x[0]+(-0.183100417297)*x[1]
        arg[3,1,1,0]=(0.0435866301729)*x[0]+(-0.557137572068)*x[1]
        arg[3,1,1,1]=(-0.253225217468)*x[0]+(-0.0657932188329)*x[1]
        arg[3,1,2,0]=(0.147700921064)*x[0]+(0.681624563015)*x[1]
        arg[3,1,2,1]=(-0.075394297616)*x[0]+(0.271741245789)*x[1]
        arg[3,2,0,0]=(1.34803044812)*x[0]+(0.87991065609)*x[1]
        arg[3,2,0,1]=(0.0529487135353)*x[0]+(0.806480092352)*x[1]
        arg[3,2,1,0]=(-1.52038624166)*x[0]+(-0.992718138685)*x[1]
        arg[3,2,1,1]=(0.0859852188749)*x[0]+(1.63512596202)*x[1]
        arg[3,2,2,0]=(1.46925148575)*x[0]+(-1.60383564005)*x[1]
        arg[3,2,2,1]=(0.114463395473)*x[0]+(-1.0186377376)*x[1]
        arg[3,3,0,0]=(-1.02962502658)*x[0]+(-1.06788543146)*x[1]
        arg[3,3,0,1]=(-0.184391597652)*x[0]+(0.462603936483)*x[1]
        arg[3,3,1,0]=(0.611113335782)*x[0]+(0.282417397043)*x[1]
        arg[3,3,1,1]=(-0.193423873796)*x[0]+(-0.56108614964)*x[1]
        arg[3,3,2,0]=(-0.0290687598224)*x[0]+(-0.460221158417)*x[1]
        arg[3,3,2,1]=(0.784513522959)*x[0]+(-0.484813129206)*x[1]
        arg[3,4,0,0]=(0.423780748664)*x[0]+(-0.344020787894)*x[1]
        arg[3,4,0,1]=(1.10623024451)*x[0]+(-0.775008153301)*x[1]
        arg[3,4,1,0]=(0.554847534291)*x[0]+(-0.854749107999)*x[1]
        arg[3,4,1,1]=(-0.639595693597)*x[0]+(-0.966152813212)*x[1]
        arg[3,4,2,0]=(-0.0172621996023)*x[0]+(0.321815074269)*x[1]
        arg[3,4,2,1]=(0.39605156824)*x[0]+(-0.499138231749)*x[1]
        ref[0,0,0,0]=(0.169892838337)/(o+1.)+(-0.323261281472)+(0.722227117759)*0.5**o
        ref[0,0,0,1]=(-0.899797330597)/(o+1.)+(0.731242549464)+(-0.679761749104)*0.5**o
        ref[0,0,1,0]=(0.948969615551)/(o+1.)+(-0.173478478707)+(0.0756189154575)*0.5**o
        ref[0,0,1,1]=(-0.484419525942)/(o+1.)+(0.287761319651)+(-0.834046890353)*0.5**o
        ref[0,0,2,0]=(-0.275366085384)/(o+1.)+(-0.051636507926)+(0.534757021171)*0.5**o
        ref[0,0,2,1]=(0.338862631582)/(o+1.)+(0.076489529483)+(-0.47622767501)*0.5**o
        ref[0,1,0,0]=(0.960885993642)/(o+1.)+(0.36959166751)+(0.789134441414)*0.5**o
        ref[0,1,0,1]=(-0.101908900511)/(o+1.)+(0.343409131566)+(0.298411594744)*0.5**o
        ref[0,1,1,0]=(-0.50227382988)/(o+1.)+(-0.431549057432)+(-0.509965959533)*0.5**o
        ref[0,1,1,1]=(-0.294216689224)/(o+1.)+(0.293114907564)+(-0.00647512035965)*0.5**o
        ref[0,1,2,0]=(0.330313139948)/(o+1.)+(0.0601559902451)+(-0.447667920962)*0.5**o
        ref[0,1,2,1]=(-0.855325117375)/(o+1.)+(0.455706690618)+(0.0237326178252)*0.5**o
        ref[0,2,0,0]=(0.958948557218)/(o+1.)+(-0.307015973897)+(-0.752766039908)*0.5**o
        ref[0,2,0,1]=(0.622666086232)/(o+1.)+(0.417782845058)+(-0.676857571283)*0.5**o
        ref[0,2,1,0]=(-0.476269525636)/(o+1.)+(0.203518519787)+(-0.541895856898)*0.5**o
        ref[0,2,1,1]=(0.396862410812)/(o+1.)+(-0.318442063423)+(0.499722654464)*0.5**o
        ref[0,2,2,0]=(0.291546173382)/(o+1.)+(0.345688404218)+(0.372569619405)*0.5**o
        ref[0,2,2,1]=(-0.0863432016899)/(o+1.)+(0.296494638577)+(-0.631266836064)*0.5**o
        ref[0,3,0,0]=(0.437179077921)/(o+1.)+(-0.340515851088)+(-0.00622121586734)*0.5**o
        ref[0,3,0,1]=(-0.0670787919003)/(o+1.)+(-0.492735838895)+(0.931769856293)*0.5**o
        ref[0,3,1,0]=(-0.455620785864)/(o+1.)+(-0.866886545624)+(0.632565106759)*0.5**o
        ref[0,3,1,1]=(0.769395985412)/(o+1.)+(-0.303642587154)+(0.781062911223)*0.5**o
        ref[0,3,2,0]=(-0.338160377272)/(o+1.)+(-0.237545833658)+(-0.966712929597)*0.5**o
        ref[0,3,2,1]=(-0.796435534783)/(o+1.)+(0.544922318125)+(-0.743274373676)*0.5**o
        ref[0,4,0,0]=(0.452932120968)/(o+1.)+(-0.512874769603)+(-0.048932296957)*0.5**o
        ref[0,4,0,1]=(0.361579034112)/(o+1.)+(-0.0561290568189)+(0.365767627701)*0.5**o
        ref[0,4,1,0]=(-0.626053161593)/(o+1.)+(-0.761324946295)+(0.911283010254)*0.5**o
        ref[0,4,1,1]=(-0.0455977379835)/(o+1.)+(0.292122388548)+(0.22020429757)*0.5**o
        ref[0,4,2,0]=(-0.881052488675)/(o+1.)+(-0.210171400955)+(-0.973018746263)*0.5**o
        ref[0,4,2,1]=(0.955475719603)/(o+1.)+(0.367959093676)+(0.126904536122)*0.5**o
        ref[1,0,0,0]=(-0.314523133291)/(o+1.)+(-0.649561146707)+(-0.54959125554)*0.5**o
        ref[1,0,0,1]=(0.776726613012)/(o+1.)+(0.0747896120381)+(0.804907910029)*0.5**o
        ref[1,0,1,0]=(0.416309554773)/(o+1.)+(0.0650606132082)+(-0.795886225295)*0.5**o
        ref[1,0,1,1]=(-0.642083920221)/(o+1.)+(-0.518812997818)+(-0.590503261366)*0.5**o
        ref[1,0,2,0]=(0.209400082886)/(o+1.)+(-0.592256267485)+(0.00147357225359)*0.5**o
        ref[1,0,2,1]=(-0.969945221754)/(o+1.)+(-0.364559871358)+(-0.0931427766681)*0.5**o
        ref[1,1,0,0]=(0.941138659497)/(o+1.)+(-0.103485523138)+(-0.866286624489)*0.5**o
        ref[1,1,0,1]=(0.527395399401)/(o+1.)+(0.373358049173)+(-0.612019142395)*0.5**o
        ref[1,1,1,0]=(-0.276942103917)/(o+1.)+(0.624509599487)+(0.0329877871648)*0.5**o
        ref[1,1,1,1]=(-0.153477829221)/(o+1.)+(-0.0616268433725)+(-0.85593389444)*0.5**o
        ref[1,1,2,0]=(0.2685285297)/(o+1.)+(-0.301824551915)+(-0.386723203413)*0.5**o
        ref[1,1,2,1]=(-0.530960074652)/(o+1.)+(0.248621312801)+(-0.772378264474)*0.5**o
        ref[1,2,0,0]=(0.745071931955)/(o+1.)+(-0.149268301427)+(-0.000656345828701)*0.5**o
        ref[1,2,0,1]=(-0.581822428087)/(o+1.)+(0.2315691847)+(0.122913062738)*0.5**o
        ref[1,2,1,0]=(-0.325165099021)/(o+1.)+(-0.414758139139)+(0.187663931198)*0.5**o
        ref[1,2,1,1]=(-0.389683593964)/(o+1.)+(0.265985790556)+(0.485893691018)*0.5**o
        ref[1,2,2,0]=(0.661969003098)/(o+1.)+(-0.514584521701)+(-0.990427195841)*0.5**o
        ref[1,2,2,1]=(-0.0828384828188)/(o+1.)+(-0.482166570368)+(0.832805029032)*0.5**o
        ref[1,3,0,0]=(-0.756020671713)/(o+1.)+(0.627704058228)+(0.150404496577)*0.5**o
        ref[1,3,0,1]=(-0.93648010541)/(o+1.)+(0.11022703478)+(-0.387767702896)*0.5**o
        ref[1,3,1,0]=(-0.995823513997)/(o+1.)+(-0.242072805569)+(-0.604794178119)*0.5**o
        ref[1,3,1,1]=(-0.543071878541)/(o+1.)+(-0.160890620667)+(0.654757881656)*0.5**o
        ref[1,3,2,0]=(0.803184441579)/(o+1.)+(-0.612293875048)+(-0.504652146723)*0.5**o
        ref[1,3,2,1]=(-0.786714439322)/(o+1.)+(-0.175556007517)+(-0.18491128296)*0.5**o
        ref[1,4,0,0]=(-0.302928744298)/(o+1.)+(0.476360987754)+(-0.423396151472)*0.5**o
        ref[1,4,0,1]=(-0.410240602476)/(o+1.)+(-0.49551630458)+(0.40280698327)*0.5**o
        ref[1,4,1,0]=(-0.820677400346)/(o+1.)+(-0.573037933363)+(-0.752720066188)*0.5**o
        ref[1,4,1,1]=(-0.492748923658)/(o+1.)+(-0.600739806799)+(0.170290165507)*0.5**o
        ref[1,4,2,0]=(0.926654770824)/(o+1.)+(0.011282908181)+(-0.697106837662)*0.5**o
        ref[1,4,2,1]=(-0.958477382611)/(o+1.)+(0.618515858902)+(-0.65973097111)*0.5**o
        ref[2,0,0,0]=(-0.15636950624)/(o+1.)+(-0.520670547262)+(-0.36455690548)*0.5**o
        ref[2,0,0,1]=(-0.726424878286)/(o+1.)+(0.697757734621)+(-0.386898263585)*0.5**o
        ref[2,0,1,0]=(-0.055162393508)/(o+1.)+(0.247015646426)+(-0.442127640696)*0.5**o
        ref[2,0,1,1]=(-0.769332288875)/(o+1.)+(-0.0917175733456)+(-0.639003232373)*0.5**o
        ref[2,0,2,0]=(-0.196390038297)/(o+1.)+(0.0977907270548)+(0.340749124126)*0.5**o
        ref[2,0,2,1]=(0.841927481859)/(o+1.)+(-0.489185426146)+(0.0626606144906)*0.5**o
        ref[2,1,0,0]=(0.590018249148)/(o+1.)+(0.378476967182)+(-0.354886735451)*0.5**o
        ref[2,1,0,1]=(0.178495604742)/(o+1.)+(0.233073081069)+(0.572535849459)*0.5**o
        ref[2,1,1,0]=(-0.762671064919)/(o+1.)+(-0.108539686142)+(0.448831274081)*0.5**o
        ref[2,1,1,1]=(0.927591406535)/(o+1.)+(0.0441635703243)+(-0.53130361723)*0.5**o
        ref[2,1,2,0]=(-0.604360706259)/(o+1.)+(-0.134848102332)+(0.402913311063)*0.5**o
        ref[2,1,2,1]=(0.229727627441)/(o+1.)+(-0.323296604293)+(-0.197927111225)*0.5**o
        ref[2,2,0,0]=(0.287478356112)/(o+1.)+(0.762635506201)+(0.0282288847209)*0.5**o
        ref[2,2,0,1]=(0.243518182621)/(o+1.)+(-0.241502069199)+(0.743970373822)*0.5**o
        ref[2,2,1,0]=(-0.369049760378)/(o+1.)+(-0.21941778965)+(-0.472075045017)*0.5**o
        ref[2,2,1,1]=(-0.745343658666)/(o+1.)+(0.193375557775)+(0.864596248255)*0.5**o
        ref[2,2,2,0]=(-0.491769251494)/(o+1.)+(0.442742387618)+(0.753340954862)*0.5**o
        ref[2,2,2,1]=(0.551196377158)/(o+1.)+(-0.523535406613)+(-0.212166925203)*0.5**o
        ref[2,3,0,0]=(0.192271522168)/(o+1.)+(-0.0181953944687)+(0.122707705798)*0.5**o
        ref[2,3,0,1]=(-0.593768738594)/(o+1.)+(0.199088910545)+(0.368366498588)*0.5**o
        ref[2,3,1,0]=(0.0866211808983)/(o+1.)+(0.723512694127)+(0.521936735685)*0.5**o
        ref[2,3,1,1]=(-0.453288268254)/(o+1.)+(-0.15580970281)+(-0.655272797205)*0.5**o
        ref[2,3,2,0]=(0.614105802669)/(o+1.)+(0.0411427694898)+(0.85774098305)*0.5**o
        ref[2,3,2,1]=(0.263133533654)/(o+1.)+(-0.628776355867)+(-0.624497793507)*0.5**o
        ref[2,4,0,0]=(-0.809347914722)/(o+1.)+(0.367719412237)+(0.26671699779)*0.5**o
        ref[2,4,0,1]=(0.773124137566)/(o+1.)+(-0.442496395542)+(-0.0149946871367)*0.5**o
        ref[2,4,1,0]=(-0.927311525126)/(o+1.)+(-0.0125177466568)+(-0.943580084777)*0.5**o
        ref[2,4,1,1]=(0.306409665483)/(o+1.)+(0.128569553604)+(-0.852228134533)*0.5**o
        ref[2,4,2,0]=(0.926185522145)/(o+1.)+(0.0750538146738)+(-0.58517230367)*0.5**o
        ref[2,4,2,1]=(0.858245470922)/(o+1.)+(0.0125876600893)+(0.457971730734)*0.5**o
        ref[3,0,0,0]=(-0.74501947953)/(o+1.)+(0.343942243752)+(-0.363581155556)*0.5**o
        ref[3,0,0,1]=(0.850890737948)/(o+1.)+(0.157997002133)+(-0.761705466968)*0.5**o
        ref[3,0,1,0]=(-0.0468962248591)/(o+1.)+(0.0297980532763)+(0.682735456043)*0.5**o
        ref[3,0,1,1]=(0.0126326425456)/(o+1.)+(-0.076115296096)+(0.575387199213)*0.5**o
        ref[3,0,2,0]=(-0.410549752333)/(o+1.)+(0.85556811978)+(0.270867209104)*0.5**o
        ref[3,0,2,1]=(0.580487848545)/(o+1.)+(0.59069171729)+(0.698757148346)*0.5**o
        ref[3,1,0,0]=(0.139052775786)/(o+1.)+(-0.334747475889)+(0.281989056901)*0.5**o
        ref[3,1,0,1]=(0.591083610998)/(o+1.)+(-0.322680889668)+(-0.492254197357)*0.5**o
        ref[3,1,1,0]=(0.384713965427)/(o+1.)+(-0.522411606102)+(0.146558304882)*0.5**o
        ref[3,1,1,1]=(-0.827615582009)/(o+1.)+(-0.00295218575854)+(0.514501517225)*0.5**o
        ref[3,1,2,0]=(0.84762578765)/(o+1.)+(0.208361555685)+(-0.43502341494)*0.5**o
        ref[3,1,2,1]=(-0.554668615664)/(o+1.)+(0.825347835483)+(-0.899680107129)*0.5**o
        ref[3,2,0,0]=(0.652802712143)/(o+1.)+(0.397197582731)+(0.7807432266)*0.5**o
        ref[3,2,0,1]=(0.421495251937)/(o+1.)+(-0.119571710512)+(0.677076974975)*0.5**o
        ref[3,2,1,0]=(-0.230439630375)/(o+1.)+(-0.733165339017)+(-0.816334071941)*0.5**o
        ref[3,2,1,1]=(0.64153308098)/(o+1.)+(0.778374176432)+(-0.477170252948)*0.5**o
        ref[3,2,2,0]=(-0.987848717403)/(o+1.)+(-0.0606435888479)+(0.974551740798)*0.5**o
        ref[3,2,2,1]=(-0.318593711859)/(o+1.)+(-0.459904454409)+(0.334228278552)*0.5**o
        ref[3,3,0,0]=(-0.418115413351)/(o+1.)+(-0.348616063735)+(-0.982162917221)*0.5**o
        ref[3,3,0,1]=(-0.437684435015)/(o+1.)+(0.804145379606)+(-0.892393985366)*0.5**o
        ref[3,3,1,0]=(0.944100110135)/(o+1.)+(-0.059994562245)+(0.0694197471801)*0.5**o
        ref[3,3,1,1]=(-0.860656048613)/(o+1.)+(0.178136109535)+(-0.250126193893)*0.5**o
        ref[3,3,2,0]=(-0.363631991908)/(o+1.)+(-0.298384287282)+(0.471110648232)*0.5**o
        ref[3,3,2,1]=(-0.578021736479)/(o+1.)+(0.320913094365)+(0.235895941501)*0.5**o
        ref[3,4,0,0]=(-0.946995649946)/(o+1.)+(0.0451978887455)+(0.936359833225)*0.5**o
        ref[3,4,0,1]=(-0.636305818291)/(o+1.)+(0.258932088634)+(0.449663732234)*0.5**o
        ref[3,4,1,0]=(-0.823421937366)/(o+1.)+(0.112980674356)+(0.297559014947)*0.5**o
        ref[3,4,1,1]=(-0.441768022626)/(o+1.)+(-0.240840962617)+(-0.682298558949)*0.5**o
        ref[3,4,2,0]=(-0.258808936822)/(o+1.)+(0.510306577882)+(-0.457251344276)*0.5**o
        ref[3,4,2,1]=(-0.489365749016)/(o+1.)+(-0.296617385332)+(0.979513856171)*0.5**o
      else:
        arg[0,0,0,0]=(-0.474247989728)*x[0]+(-0.768689352235)*x[1]+(-1.08527900853)*x[2]
        arg[0,0,0,1]=(-0.237113797789)*x[0]+(0.0842735741018)*x[1]+(0.203858551029)*x[2]
        arg[0,0,1,0]=(0.24237127665)*x[0]+(0.421022955803)*x[1]+(-1.52818540927)*x[2]
        arg[0,0,1,1]=(1.34624865817)*x[0]+(-0.798943272818)*x[1]+(-0.823168386315)*x[2]
        arg[0,0,2,0]=(-0.906723771267)*x[0]+(0.328893123771)*x[1]+(-0.942770202463)*x[2]
        arg[0,0,2,1]=(1.54305197694)*x[0]+(0.813063941984)*x[1]+(0.205532727168)*x[2]
        arg[0,1,0,0]=(0.399159979872)*x[0]+(-1.52089837934)*x[1]+(-1.4064251406)*x[2]
        arg[0,1,0,1]=(-0.121139354731)*x[0]+(1.12768246422)*x[1]+(0.234299409411)*x[2]
        arg[0,1,1,0]=(-1.62513899388)*x[0]+(-0.173131700583)*x[1]+(1.24199996799)*x[2]
        arg[0,1,1,1]=(-0.371929434177)*x[0]+(-0.606278610409)*x[1]+(-0.00600790063066)*x[2]
        arg[0,1,2,0]=(-0.804531294423)*x[0]+(0.869392248002)*x[1]+(0.840417359943)*x[2]
        arg[0,1,2,1]=(0.329452946275)*x[0]+(1.1072495446)*x[1]+(-0.213517932323)*x[2]
        arg[0,2,0,0]=(1.02784887224)*x[0]+(0.435517042337)*x[1]+(1.58767206677)*x[2]
        arg[0,2,0,1]=(-0.0120997177)*x[0]+(0.430135803314)*x[1]+(-0.717473068956)*x[2]
        arg[0,2,1,0]=(0.955455517302)*x[0]+(0.712976918968)*x[1]+(-0.624804508641)*x[2]
        arg[0,2,1,1]=(-1.03293986594)*x[0]+(-0.539062834482)*x[1]+(-1.61172701428)*x[2]
        arg[0,2,2,0]=(-0.177662810138)*x[0]+(0.553352366127)*x[1]+(0.695417121111)*x[2]
        arg[0,2,2,1]=(0.318133474679)*x[0]+(-0.452528947926)*x[1]+(0.468602505091)*x[2]
        arg[0,3,0,0]=(-0.294467725916)*x[0]+(1.67314409894)*x[1]+(0.100736440007)*x[2]
        arg[0,3,0,1]=(-0.517162349887)*x[0]+(-1.57430191254)*x[1]+(1.24045877402)*x[2]
        arg[0,3,1,0]=(0.667393640368)*x[0]+(0.340993632212)*x[1]+(-1.09920535559)*x[2]
        arg[0,3,1,1]=(0.37451406727)*x[0]+(0.37350649326)*x[1]+(0.357440871322)*x[2]
        arg[0,3,2,0]=(-0.989366398194)*x[0]+(0.596179154759)*x[1]+(-0.573280308087)*x[2]
        arg[0,3,2,1]=(-0.25561190141)*x[0]+(1.42296905238)*x[1]+(-1.21032031399)*x[2]
        arg[0,4,0,0]=(-0.41394085285)*x[0]+(-0.751213929646)*x[1]+(-0.68470718293)*x[2]
        arg[0,4,0,1]=(0.702985204167)*x[0]+(-0.0375821582587)*x[1]+(0.290630803819)*x[2]
        arg[0,4,1,0]=(0.0558204653441)*x[0]+(1.61940057302)*x[1]+(-0.518863528354)*x[2]
        arg[0,4,1,1]=(1.25876303604)*x[0]+(-1.40923202908)*x[1]+(1.05202333762)*x[2]
        arg[0,4,2,0]=(0.0715295483294)*x[0]+(0.883869702529)*x[1]+(-0.463253474408)*x[2]
        arg[0,4,2,1]=(0.564563789615)*x[0]+(-0.135049742988)*x[1]+(-1.3118878567)*x[2]
        arg[1,0,0,0]=(0.707267335291)*x[0]+(-0.197244449872)*x[1]+(-0.114362773781)*x[2]
        arg[1,0,0,1]=(-0.144762815659)*x[0]+(0.434177706452)*x[1]+(0.0754859099691)*x[2]
        arg[1,0,1,0]=(-0.906934991359)*x[0]+(0.179011995627)*x[1]+(0.823936036469)*x[2]
        arg[1,0,1,1]=(1.11165724605)*x[0]+(0.48023223731)*x[1]+(0.981353808181)*x[2]
        arg[1,0,2,0]=(0.499460583481)*x[0]+(0.171586243561)*x[1]+(-0.252474775657)*x[2]
        arg[1,0,2,1]=(-0.604289370884)*x[0]+(1.11610914086)*x[1]+(0.747439062711)*x[2]
        arg[1,1,0,0]=(0.696191463882)*x[0]+(0.349547714404)*x[1]+(0.651361019042)*x[2]
        arg[1,1,0,1]=(0.152679923426)*x[0]+(-0.448442911425)*x[1]+(1.33964371501)*x[2]
        arg[1,1,1,0]=(-0.74612592015)*x[0]+(-1.27089239786)*x[1]+(-0.261725631736)*x[2]
        arg[1,1,1,1]=(1.50839493929)*x[0]+(1.93229724987)*x[1]+(-0.963996845831)*x[2]
        arg[1,1,2,0]=(0.867984447375)*x[0]+(-0.777493811854)*x[1]+(0.847847553133)*x[2]
        arg[1,1,2,1]=(1.34249389498)*x[0]+(0.776160804062)*x[1]+(-0.295393972199)*x[2]
        arg[1,2,0,0]=(-0.655771613243)*x[0]+(1.58220214455)*x[1]+(-0.406292477629)*x[2]
        arg[1,2,0,1]=(0.59294895986)*x[0]+(0.544961164723)*x[1]+(0.634279670948)*x[2]
        arg[1,2,1,0]=(1.15324179862)*x[0]+(0.645764835217)*x[1]+(0.670364725309)*x[2]
        arg[1,2,1,1]=(-0.819459106271)*x[0]+(0.024815625404)*x[1]+(0.769753650676)*x[2]
        arg[1,2,2,0]=(1.19042366161)*x[0]+(0.982020548972)*x[1]+(0.92730609738)*x[2]
        arg[1,2,2,1]=(-0.742441488349)*x[0]+(-1.3101429812)*x[1]+(-0.198104246324)*x[2]
        arg[1,3,0,0]=(0.349205030437)*x[0]+(-0.745240339596)*x[1]+(-0.744742092259)*x[2]
        arg[1,3,0,1]=(0.993101296711)*x[0]+(-1.48331355036)*x[1]+(-1.00616778677)*x[2]
        arg[1,3,1,0]=(-1.10586442824)*x[0]+(1.71922449779)*x[1]+(-0.943451153259)*x[2]
        arg[1,3,1,1]=(-1.72393763035)*x[0]+(-1.14113495956)*x[1]+(-0.211909615809)*x[2]
        arg[1,3,2,0]=(-0.378302963108)*x[0]+(-0.494705322002)*x[1]+(0.940164363332)*x[2]
        arg[1,3,2,1]=(0.850541148743)*x[0]+(-1.45461945468)*x[1]+(1.20025821328)*x[2]
        arg[1,4,0,0]=(1.33882463795)*x[0]+(0.475208133119)*x[1]+(0.722686059253)*x[2]
        arg[1,4,0,1]=(0.0750422155969)*x[0]+(1.29758288845)*x[1]+(-1.41431668738)*x[2]
        arg[1,4,1,0]=(1.36072344575)*x[0]+(-0.345743547563)*x[1]+(-0.63248018754)*x[2]
        arg[1,4,1,1]=(0.579049229913)*x[0]+(0.954245074078)*x[1]+(-0.170538926965)*x[2]
        arg[1,4,2,0]=(1.67615484634)*x[0]+(0.561067019023)*x[1]+(-1.26952326914)*x[2]
        arg[1,4,2,1]=(0.700910547934)*x[0]+(0.58717328764)*x[1]+(0.795138196965)*x[2]
        arg[2,0,0,0]=(0.763193592359)*x[0]+(-0.397549328381)*x[1]+(-0.230593559545)*x[2]
        arg[2,0,0,1]=(0.454026153468)*x[0]+(-1.12464443717)*x[1]+(-0.147462098246)*x[2]
        arg[2,0,1,0]=(-0.276457034935)*x[0]+(0.118173156654)*x[1]+(0.57296366275)*x[2]
        arg[2,0,1,1]=(1.31887812253)*x[0]+(-0.544644483254)*x[1]+(0.0013728350582)*x[2]
        arg[2,0,2,0]=(-0.257589384265)*x[0]+(-0.016005453203)*x[1]+(-0.508836430965)*x[2]
        arg[2,0,2,1]=(-0.259420301773)*x[0]+(0.122645293168)*x[1]+(0.381613885289)*x[2]
        arg[2,1,0,0]=(-1.64168512418)*x[0]+(0.025182006192)*x[1]+(0.0268780458975)*x[2]
        arg[2,1,0,1]=(-0.0885344200898)*x[0]+(0.408795219114)*x[1]+(0.635798801606)*x[2]
        arg[2,1,1,0]=(-0.959658503283)*x[0]+(1.034415556)*x[1]+(0.725681105748)*x[2]
        arg[2,1,1,1]=(1.59312707789)*x[0]+(-0.116514952101)*x[1]+(-0.86121633507)*x[2]
        arg[2,1,2,0]=(-1.44859572067)*x[0]+(-1.28344247275)*x[1]+(0.299296401869)*x[2]
        arg[2,1,2,1]=(-0.488424372501)*x[0]+(1.23481402288)*x[1]+(0.01702902463)*x[2]
        arg[2,2,0,0]=(-1.29986951244)*x[0]+(-1.20190912751)*x[1]+(-0.755848291051)*x[2]
        arg[2,2,0,1]=(-0.52421656312)*x[0]+(-0.335471509689)*x[1]+(-1.01264312499)*x[2]
        arg[2,2,1,0]=(-1.12066642609)*x[0]+(-1.02138558053)*x[1]+(-1.00777755552)*x[2]
        arg[2,2,1,1]=(-1.42127024744)*x[0]+(1.63480022479)*x[1]+(0.542244906643)*x[2]
        arg[2,2,2,0]=(-0.418852880297)*x[0]+(-0.983110436453)*x[1]+(-0.622603673805)*x[2]
        arg[2,2,2,1]=(0.727366446825)*x[0]+(-0.256324856361)*x[1]+(-0.20608740758)*x[2]
        arg[2,3,0,0]=(0.548268302883)*x[0]+(0.0102173127665)*x[1]+(0.869833576217)*x[2]
        arg[2,3,0,1]=(-1.70159091061)*x[0]+(1.15876875083)*x[1]+(-0.234757706773)*x[2]
        arg[2,3,1,0]=(-1.72817024817)*x[0]+(0.690055132218)*x[1]+(-0.149398998825)*x[2]
        arg[2,3,1,1]=(1.80829412387)*x[0]+(-0.217899654855)*x[1]+(-0.922672217822)*x[2]
        arg[2,3,2,0]=(1.23855200328)*x[0]+(0.412795411508)*x[1]+(0.960476191755)*x[2]
        arg[2,3,2,1]=(-1.43061904621)*x[0]+(-0.766039000745)*x[1]+(-1.25765397361)*x[2]
        arg[2,4,0,0]=(-0.660872016687)*x[0]+(-0.000405233718708)*x[1]+(-0.178507968219)*x[2]
        arg[2,4,0,1]=(-0.168526882871)*x[0]+(-1.06936802685)*x[1]+(1.54083857104)*x[2]
        arg[2,4,1,0]=(0.565807359529)*x[0]+(-1.41401297939)*x[1]+(-1.23254260537)*x[2]
        arg[2,4,1,1]=(-0.373920075802)*x[0]+(0.407330812903)*x[1]+(-0.204667001328)*x[2]
        arg[2,4,2,0]=(0.874100879885)*x[0]+(-0.191238094267)*x[1]+(1.54890969328)*x[2]
        arg[2,4,2,1]=(0.566428125313)*x[0]+(-1.60899595601)*x[1]+(-1.05112529768)*x[2]
        arg[3,0,0,0]=(-0.688037248111)*x[0]+(-1.56483484784)*x[1]+(0.26782504619)*x[2]
        arg[3,0,0,1]=(0.418788954459)*x[0]+(-0.670455226092)*x[1]+(-0.0562760591375)*x[2]
        arg[3,0,1,0]=(-0.2614362754)*x[0]+(0.271050519855)*x[1]+(0.227811982923)*x[2]
        arg[3,0,1,1]=(-0.873198928647)*x[0]+(-0.483763141395)*x[1]+(0.33532383979)*x[2]
        arg[3,0,2,0]=(-0.864161938176)*x[0]+(-1.4682754386)*x[1]+(-0.38126411162)*x[2]
        arg[3,0,2,1]=(0.142827521381)*x[0]+(0.464483940188)*x[1]+(0.214297576606)*x[2]
        arg[3,1,0,0]=(0.188747079113)*x[0]+(0.590821253713)*x[1]+(0.908850367108)*x[2]
        arg[3,1,0,1]=(0.0587541582624)*x[0]+(0.955494266151)*x[1]+(1.70822209849)*x[2]
        arg[3,1,1,0]=(-0.420356861523)*x[0]+(-0.551445918262)*x[1]+(-0.0187114754269)*x[2]
        arg[3,1,1,1]=(1.07071658401)*x[0]+(-0.589035602194)*x[1]+(0.133725074027)*x[2]
        arg[3,1,2,0]=(0.370477043767)*x[0]+(0.226262353618)*x[1]+(0.566087784489)*x[2]
        arg[3,1,2,1]=(1.77828660075)*x[0]+(0.607419742685)*x[1]+(0.0297054744582)*x[2]
        arg[3,2,0,0]=(-1.19326939474)*x[0]+(1.20875153887)*x[1]+(-0.124756222091)*x[2]
        arg[3,2,0,1]=(0.0645190914271)*x[0]+(0.357580475716)*x[1]+(-1.59461469718)*x[2]
        arg[3,2,1,0]=(1.4853619905)*x[0]+(0.751582301648)*x[1]+(0.859363177167)*x[2]
        arg[3,2,1,1]=(1.36658655075)*x[0]+(-0.776650996613)*x[1]+(0.654743994261)*x[2]
        arg[3,2,2,0]=(0.406696005677)*x[0]+(-0.528546876384)*x[1]+(-0.439775325396)*x[2]
        arg[3,2,2,1]=(-0.725198159461)*x[0]+(-0.0933136564416)*x[1]+(-0.720365141393)*x[2]
        arg[3,3,0,0]=(-0.428669112669)*x[0]+(0.626162511175)*x[1]+(-0.382776233923)*x[2]
        arg[3,3,0,1]=(0.340474939218)*x[0]+(-0.140550383925)*x[1]+(0.353836983746)*x[2]
        arg[3,3,1,0]=(0.851646816711)*x[0]+(0.0171684195949)*x[1]+(0.38173044322)*x[2]
        arg[3,3,1,1]=(0.359647722225)*x[0]+(0.425799941165)*x[1]+(-0.80174052212)*x[2]
        arg[3,3,2,0]=(0.13012568736)*x[0]+(-0.573735693651)*x[1]+(1.08843730088)*x[2]
        arg[3,3,2,1]=(-1.01372953294)*x[0]+(0.335282956856)*x[1]+(1.24570512628)*x[2]
        arg[3,4,0,0]=(-0.05674609039)*x[0]+(-0.814879575064)*x[1]+(-0.169789408264)*x[2]
        arg[3,4,0,1]=(1.5968230938)*x[0]+(0.0679819277023)*x[1]+(0.892284314512)*x[2]
        arg[3,4,1,0]=(1.22220298478)*x[0]+(-0.0258417926472)*x[1]+(0.440670138568)*x[2]
        arg[3,4,1,1]=(0.647072768716)*x[0]+(-0.177651087906)*x[1]+(-0.115553644646)*x[2]
        arg[3,4,2,0]=(1.29816363306)*x[0]+(-1.18656044167)*x[1]+(-0.465776533157)*x[2]
        arg[3,4,2,1]=(0.328907496091)*x[0]+(-0.179857937151)*x[1]+(-0.476306938306)*x[2]
        ref[0,0,0,0]=(-0.530378359482)/(o+1.)+(-0.935143327523)+(0.0724486640358)*0.5**o
        ref[0,0,0,1]=(1.22837847871)/(o+1.)+(-0.113350954728)+(-0.950658241915)*0.5**o
        ref[0,0,1,0]=(-0.000414683535449)/(o+1.)+(-0.183158731552)+(-0.49805903018)*0.5**o
        ref[0,0,1,1]=(-0.882730389132)/(o+1.)+(-0.112174958816)+(0.831217305802)*0.5**o
        ref[0,0,2,0]=(-1.1913557966)/(o+1.)+(-0.0615750891074)+(-0.206094875146)*0.5**o
        ref[0,0,2,1]=(0.634536492589)/(o+1.)+(0.649890059464)+(0.627332034578)*0.5**o
        ref[0,1,0,0]=(-1.38665625775)/(o+1.)+(-1.0210653524)+(0.900623422471)*0.5**o
        ref[0,1,0,1]=(1.20633678024)/(o+1.)+(0.0888193121856)+(-0.143132885707)*0.5**o
        ref[0,1,1,0]=(0.452225278035)/(o+1.)+(-0.148926943801)+(-0.710642116901)*0.5**o
        ref[0,1,1,1]=(-0.244223430204)/(o+1.)+(-0.619607191996)+(0.499221868978)*0.5**o
        ref[0,1,2,0]=(0.0383849036606)/(o+1.)+(0.759484971644)+(-0.652076533427)*0.5**o
        ref[0,1,2,1]=(0.207558055759)/(o+1.)+(0.502305382384)+(0.0110157380215)*0.5**o
        ref[0,2,0,0]=(0.761374280144)/(o+1.)+(0.777616611515)+(0.734430478174)*0.5**o
        ref[0,2,0,1]=(0.554502862028)/(o+1.)+(-0.409274602016)+(-0.0353906413371)*0.5**o
        ref[0,2,1,0]=(-0.252543841709)/(o+1.)+(0.386640377894)+(0.522891013551)*0.5**o
        ref[0,2,1,1]=(-1.52497235887)/(o+1.)+(-0.807603254401)+(-0.0435508470367)*0.5**o
        ref[0,2,2,0]=(0.172485606257)/(o+1.)+(0.350297526187)+(0.19802601847)*0.5**o
        ref[0,2,2,1]=(-0.141271359379)/(o+1.)+(0.0509198134622)+(0.373638764298)*0.5**o
        ref[0,3,0,0]=(1.31278836809)/(o+1.)+(-0.0918739323256)+(0.350372309595)*0.5**o
        ref[0,3,0,1]=(-0.396933928109)/(o+1.)+(-0.436137805999)+(0.418204051696)*0.5**o
        ref[0,3,1,0]=(-0.206619840099)/(o+1.)+(-0.118703289798)+(0.353208336689)*0.5**o
        ref[0,3,1,1]=(-0.595153371293)/(o+1.)+(0.615722581562)+(0.469169640021)*0.5**o
        ref[0,3,2,0]=(-0.680858997986)/(o+1.)+(-0.0393223937964)+(-0.206963765944)*0.5**o
        ref[0,3,2,1]=(0.012753264598)/(o+1.)+(0.246188741489)+(-0.548093910595)*0.5**o
        ref[0,4,0,0]=(-0.40478987122)/(o+1.)+(-0.890399681609)+(0.33572726901)*0.5**o
        ref[0,4,0,1]=(0.0280062557264)/(o+1.)+(0.0358049413228)+(0.856417711355)*0.5**o
        ref[0,4,1,0]=(0.181560004565)/(o+1.)+(0.329885369854)+(0.31502676574)*0.5**o
        ref[0,4,1,1]=(0.0552681276841)/(o+1.)+(0.00372919426319)+(0.838827828375)*0.5**o
        ref[0,4,2,0]=(0.656057162724)/(o+1.)+(0.179315527684)+(-0.522542441641)*0.5**o
        ref[0,4,2,1]=(0.126189389317)/(o+1.)+(-0.429452001643)+(-0.149659196108)*0.5**o
        ref[1,0,0,0]=(1.43443485527)/(o+1.)+(-0.55091754491)+(0.0630603461901)*0.5**o
        ref[1,0,0,1]=(-0.612191779306)/(o+1.)+(0.604025235381)+(-0.230957890695)*0.5**o
        ref[1,0,1,0]=(0.373085423681)/(o+1.)+(-0.0817391530697)+(-0.113594076805)*0.5**o
        ref[1,0,1,1]=(0.761862677657)/(o+1.)+(0.819524484428)+(0.172331645033)*0.5**o
        ref[1,0,2,0]=(0.929265450746)/(o+1.)+(-0.382585784938)+(0.254478170515)*0.5**o
        ref[1,0,2,1]=(1.58236203463)/(o+1.)+(-0.280641188811)+(0.238179175679)*0.5**o
        ref[1,1,0,0]=(-0.117055726312)/(o+1.)+(0.957914090571)+(-0.101672257501)*0.5**o
        ref[1,1,0,1]=(0.0849062553796)/(o+1.)+(0.0703564566846)+(0.818261558262)*0.5**o
        ref[1,1,1,0]=(-1.72219113622)/(o+1.)+(-0.357454724939)+(0.15835663635)*0.5**o
        ref[1,1,1,1]=(0.739078400468)/(o+1.)+(0.532292543169)+(0.673031856531)*0.5**o
        ref[1,1,2,0]=(0.255023909903)/(o+1.)+(0.229656917655)+(0.22400044344)*0.5**o
        ref[1,1,2,1]=(0.518856765199)/(o+1.)+(0.380552790125)+(0.543298381396)*0.5**o
        ref[1,2,0,0]=(0.0321563884831)/(o+1.)+(0.25981567762)+(-0.0316496900482)*0.5**o
        ref[1,2,0,1]=(1.09318176771)/(o+1.)+(0.0572442907603)+(0.564519446305)*0.5**o
        ref[1,2,1,0]=(0.756962791387)/(o+1.)+(0.69419429624)+(0.32401997528)*0.5**o
        ref[1,2,1,1]=(1.28480299241)/(o+1.)+(-0.201990593541)+(-0.905711635518)*0.5**o
        ref[1,2,2,0]=(1.04969566442)/(o+1.)+(0.72854097372)+(0.592972696101)*0.5**o
        ref[1,2,2,1]=(-1.77086264775)/(o+1.)+(-0.261914180361)+(0.0440022926066)*0.5**o
        ref[1,3,0,0]=(-0.477780531124)/(o+1.)+(-0.181354085163)+(-0.300288699967)*0.5**o
        ref[1,3,0,1]=(-0.797627013456)/(o+1.)+(-0.523163955197)+(0.347574883435)*0.5**o
        ref[1,3,1,0]=(0.428806106821)/(o+1.)+(-0.0514584302394)+(-0.655980330052)*0.5**o
        ref[1,3,1,1]=(-0.141740459457)/(o+1.)+(-1.05699960931)+(-0.821242527644)*0.5**o
        ref[1,3,2,0]=(0.0181810985873)/(o+1.)+(-0.180697139553)+(0.41036925874)*0.5**o
        ref[1,3,2,1]=(-0.061077367)/(o+1.)+(0.250077446763)+(0.157102380812)*0.5**o
        ref[1,4,0,0]=(1.14087212913)/(o+1.)+(0.368059029864)+(0.659728641463)*0.5**o
        ref[1,4,0,1]=(0.365658713316)/(o+1.)+(0.088533821044)+(-0.584417938734)*0.5**o
        ref[1,4,1,0]=(-1.11681308321)/(o+1.)+(0.438275560627)+(0.622761672604)*0.5**o
        ref[1,4,1,1]=(1.18893372018)/(o+1.)+(0.172934240923)+(-0.172046824999)*0.5**o
        ref[1,4,2,0]=(0.16240489493)/(o+1.)+(-0.0145923852256)+(0.834478471749)*0.5**o
        ref[1,4,2,1]=(0.571355829378)/(o+1.)+(0.526896417749)+(0.458073367663)*0.5**o
        ref[2,0,0,0]=(-0.489234572937)/(o+1.)+(-0.0135589093901)+(0.65140309615)*0.5**o
        ref[2,0,0,1]=(-1.55383491931)/(o+1.)+(0.0161644619972)+(0.703425613366)*0.5**o
        ref[2,0,1,0]=(-0.258604855723)/(o+1.)+(0.0827812678671)+(0.507722104457)*0.5**o
        ref[2,0,1,1]=(0.0504178529481)/(o+1.)+(0.182991850332)+(0.359204920726)*0.5**o
        ref[2,0,2,0]=(0.652043479841)/(o+1.)+(-0.443015450013)+(-0.548443848248)*0.5**o
        ref[2,0,2,1]=(0.340317839857)/(o+1.)+(-0.379565594545)+(0.663652225917)*0.5**o
        ref[2,1,0,0]=(-0.587863196667)/(o+1.)+(-0.0137797674284)+(-0.974202340562)*0.5**o
        ref[2,1,0,1]=(1.00030747039)/(o+1.)+(-0.399997754768)+(0.755747639771)*0.5**o
        ref[2,1,1,0]=(0.703413281036)/(o+1.)+(0.487016208736)+(-0.877007540039)*0.5**o
        ref[2,1,1,1]=(0.263166830572)/(o+1.)+(-0.157259452849)+(0.666747865841)*0.5**o
        ref[2,1,2,0]=(0.104532343004)/(o+1.)+(-1.02548001493)+(-0.486314104701)*0.5**o
        ref[2,1,2,1]=(0.933518655478)/(o+1.)+(0.0369928229139)+(-0.244085626298)*0.5**o
        ref[2,2,0,0]=(-1.50550384519)/(o+1.)+(-0.590996227594)+(-0.570130630622)*0.5**o
        ref[2,2,0,1]=(-0.60318683839)/(o+1.)+(-0.38308245528)+(-0.502979448853)*0.5**o
        ref[2,2,1,0]=(-1.11258910879)/(o+1.)+(-0.518777277969)+(-0.999685897413)*0.5**o
        ref[2,2,1,1]=(1.84366617107)/(o+1.)+(-0.232866738261)+(-0.622157810558)*0.5**o
        ref[2,2,2,0]=(-0.620181988585)/(o+1.)+(-0.836162444183)+(0.267939886396)*0.5**o
        ref[2,2,2,1]=(-0.826647389099)/(o+1.)+(0.0929536518109)+(0.905694268362)*0.5**o
        ref[2,3,0,0]=(0.184067674443)/(o+1.)+(0.259745858757)+(0.724759799909)*0.5**o
        ref[2,3,0,1]=(0.0175595303049)/(o+1.)+(0.0560117358912)+(-0.907162868639)*0.5**o
        ref[2,3,1,0]=(-0.609443154617)/(o+1.)+(0.186540819401)+(-0.951152598958)*0.5**o
        ref[2,3,1,1]=(-1.61858082094)/(o+1.)+(0.679439858578)+(0.927423354971)*0.5**o
        ref[2,3,2,0]=(0.44350652486)/(o+1.)+(0.81466177481)+(0.53899353207)*0.5**o
        ref[2,3,2,1]=(-0.877289862245)/(o+1.)+(-1.05742260849)+(-0.462176941342)*0.5**o
        ref[2,4,0,0]=(-0.736381416539)/(o+1.)+(-0.0139374259612)+(-0.0755289501641)*0.5**o
        ref[2,4,0,1]=(0.690429326999)/(o+1.)+(-0.0293644018251)+(-0.328756862036)*0.5**o
        ref[2,4,1,0]=(-1.57117034002)/(o+1.)+(-0.531463765387)+(0.553349645561)*0.5**o
        ref[2,4,1,1]=(1.24898175282)/(o+1.)+(-0.452273024959)+(-0.515691967125)*0.5**o
        ref[2,4,2,0]=(0.239126191643)/(o+1.)+(0.722504563932)+(0.547637159392)*0.5**o
        ref[2,4,2,1]=(-1.40763254209)/(o+1.)+(-0.809456441012)+(0.932852295739)*0.5**o
        ref[3,0,0,0]=(-0.416554297686)/(o+1.)+(-0.632193610911)+(-0.304105530258)*0.5**o
        ref[3,0,0,1]=(-1.02983448036)/(o+1.)+(0.579212231778)+(-0.436532313966)*0.5**o
        ref[3,0,1,0]=(0.42124133216)/(o+1.)+(0.368290418827)+(-0.920395942436)*0.5**o
        ref[3,0,1,1]=(0.158195474854)/(o+1.)+(-0.17459474514)+(-0.830644214826)*0.5**o
        ref[3,0,2,0]=(-1.18749647329)/(o+1.)+(-0.506173996178)+(-0.513857022748)*0.5**o
        ref[3,0,2,1]=(0.233437753018)/(o+1.)+(0.455705145371)+(-0.323239005586)*0.5**o
        ref[3,1,0,0]=(1.42991958566)/(o+1.)+(0.053769921022)+(0.150959272234)*0.5**o
        ref[3,1,0,1]=(1.48423731111)/(o+1.)+(0.964048315997)+(-0.689863420196)*0.5**o
        ref[3,1,1,0]=(0.868837676765)/(o+1.)+(-1.01261543254)+(0.165878933098)*0.5**o
        ref[3,1,1,1]=(0.820750307831)/(o+1.)+(-0.300930223821)+(0.396516195659)*0.5**o
        ref[3,1,2,0]=(0.477077782202)/(o+1.)+(-0.0416336989494)+(0.769016797571)*0.5**o
        ref[3,1,2,1]=(0.359895440587)/(o+1.)+(0.555322395811)+(0.944871585683)*0.5**o
        ref[3,2,0,0]=(0.657668315832)/(o+1.)+(-0.240321895174)+(-0.286298603451)*0.5**o
        ref[3,2,0,1]=(-0.941360568282)/(o+1.)+(-0.0246544136593)+(-0.181845734439)*0.5**o
        ref[3,2,1,0]=(1.60316834971)/(o+1.)+(0.252291602298)+(0.988555915005)*0.5**o
        ref[3,2,1,1]=(0.833811957678)/(o+1.)+(-0.111970621871)+(0.634808834458)*0.5**o
        ref[3,2,2,0]=(-1.38877073784)/(o+1.)+(0.538356636307)+(-0.249568730877)*0.5**o
        ref[3,2,2,1]=(-0.311839184097)/(o+1.)+(-0.698114103264)+(0.169190433329)*0.5**o
        ref[3,3,0,0]=(0.743170989696)/(o+1.)+(-0.499224746788)+(0.0699956684628)*0.5**o
        ref[3,3,0,1]=(-0.293538786051)/(o+1.)+(0.26667992866)+(0.313940467768)*0.5**o
        ref[3,3,1,0]=(0.951483461345)/(o+1.)+(0.148317904561)+(0.00242640905887)*0.5**o
        ref[3,3,1,1]=(-0.110753912776)/(o+1.)+(-0.44308689138)+(0.980634836805)*0.5**o
        ref[3,3,2,0]=(-0.337895223544)/(o+1.)+(0.358941646037)+(0.264839226057)*0.5**o
        ref[3,3,2,1]=(0.857823611315)/(o+1.)+(0.30404611512)+(-0.898657291359)*0.5**o
        ref[3,4,0,0]=(0.037944509376)/(o+1.)+(-0.364801700413)+(-0.349756182268)*0.5**o
        ref[3,4,0,1]=(1.01114399396)/(o+1.)+(0.457103279485)+(0.631738783076)*0.5**o
        ref[3,4,1,0]=(0.969931913888)/(o+1.)+(-0.165922085756)+(0.998943588321)*0.5**o
        ref[3,4,1,1]=(1.35441114248)/(o+1.)+(-0.502422727199)+(0.00430234808496)*0.5**o
        ref[3,4,2,0]=(-0.367201097566)/(o+1.)+(-0.337780780531)+(0.688589316866)*0.5**o
        ref[3,4,2,1]=(-0.0547905846061)/(o+1.)+(-0.293870420685)+(0.31527404661)*0.5**o
      res=integrate(arg,where=w_ref)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 3, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_FunctionOnContactOne_rank0(self):
      """
      tests integral of rank 0 Data on the FunctionOnContactOne

      assumptions: FunctionOnContactOne(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=FunctionOnContactOne(self.domain)
      x=w.getX()
      arg=Data(0,(),w)
      if dim==2:
        arg=(0.311787641998)*x[0]**o+(0.161673282875)*x[0]+(0.32570909237)*x[1]**o+(-0.249404742764)*x[1]
        ref=(0.32570909237)/(o+1.)+(-0.0438657299447)+(0.311787641998)*0.5**o
      else:
        arg=(0.0675036998003)*x[0]**o+(-0.247751392136)*x[0]+(0.791676220029)*x[1]**o+(-0.269015236163)*x[1]+(0.265735156875)*x[2]**o+(0.820319192519)*x[2]
        ref=(1.0574113769)/(o+1.)+(0.15177628211)+(0.0675036998003)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_FunctionOnContactOne_rank1(self):
      """
      tests integral of rank 1 Data on the FunctionOnContactOne

      assumptions: FunctionOnContactOne(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=FunctionOnContactOne(self.domain)
      x=w.getX()
      arg=Data(0,(2,),w)
      ref=numpy.zeros((2,),numpy.float_)
      if dim==2:
        arg[0]=(0.99235950851)*x[0]**o+(0.00867392831518)*x[0]+(-0.598024390869)*x[1]**o+(0.00945232777857)*x[1]
        arg[1]=(-0.140261338996)*x[0]**o+(-0.166524016889)*x[0]+(0.59850076881)*x[1]**o+(-0.805756215354)*x[1]
        ref[0]=(-0.598024390869)/(o+1.)+(0.00906312804687)+(0.99235950851)*0.5**o
        ref[1]=(0.59850076881)/(o+1.)+(-0.486140116122)+(-0.140261338996)*0.5**o
      else:
        arg[0]=(-0.158686556523)*x[0]**o+(0.249193164275)*x[0]+(-0.801383648208)*x[1]**o+(-0.439021533504)*x[1]+(-0.112035191366)*x[2]**o+(-0.323385862625)*x[2]
        arg[1]=(0.392997873484)*x[0]**o+(-0.867468854104)*x[0]+(0.804459243476)*x[1]**o+(0.129028860314)*x[1]+(-0.252715502862)*x[2]**o+(0.457355923469)*x[2]
        ref[0]=(-0.913418839574)/(o+1.)+(-0.256607115927)+(-0.158686556523)*0.5**o
        ref[1]=(0.551743740614)/(o+1.)+(-0.140542035161)+(0.392997873484)*0.5**o
      res=integrate(arg)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_integrate_onFunctionOnContactOne_fromData_FunctionOnContactOne_rank2(self):
      """
      tests integral of rank 2 Data on the FunctionOnContactOne

      assumptions: FunctionOnContactOne(self.domain) exists
                   self.domain supports integral on FunctionOnContactOne
      """
      o=self.order
      dim=self.domain.getDim()
      w_ref=FunctionOnContactOne(self.domain)
      w=FunctionOnContactOne(self.domain)
      x=w.getX()
      arg=Data(0,(4, 5),w)
      ref=numpy.zeros((4, 5),numpy.float_)
      if dim==2:
        arg[0,0]=(0.330520725992)*x[0]**o+(0.160118740246)*x[0]+(-0.178670895269)*x[1]**o+(-0.281660040648)*x[1]
        arg[0,1]=(-0.871801084852)*x[0]**o+(0.69254903