#include "Python.h"
#include "domlette.h"

/*Internal Functions*/

void Element_ReleaseNode(PyElementObject *elem)
{
  PyObject *values;
  int ctr;

  values = PyDict_Values(elem->elementData.attributes);
  for(ctr = 0;ctr < PyList_GET_SIZE(values);ctr ++) {
    Node_ReleaseNode((PyNodeObject *)PyList_GET_ITEM(values,ctr));
  }

  Py_XDECREF(values);

  Py_XDECREF(elem->elementData.attributes);
  elem->elementData.attributes = NULL;

  /*Recursively call on childNodes*/
  for(ctr = 0; ctr < PyList_GET_SIZE(elem->elementData.childNodes); ctr ++) {
    Node_ReleaseNode((PyNodeObject *)PyList_GET_ITEM(elem->elementData.childNodes,ctr));
  }
  Py_XDECREF(elem->elementData.childNodes);
  elem->elementData.childNodes = NULL;


  Py_XDECREF(elem->nodeData.namespaceURI);
  Py_XDECREF(elem->nodeData.prefix);
  Py_XDECREF(elem->nodeData.localName);
  Py_XDECREF(elem->nodeData.nodeName);

}




PyNodeObject *Element_AppendChild(PyElementObject * self,PyNodeObject *child) {
  /*Assumes ownership*/
  PyList_Append(self->elementData.childNodes,(PyObject *)child);
  Py_XDECREF(child);
  Py_XDECREF(child->nodeData.parentNode);
  child->nodeData.parentNode = (PyObject *)self;
  Py_XINCREF(self);
  return child;
}

void Element_SetAttributeNS(PyElementObject *self,
                              const char *namespaceURI,
                              const char *localName,
                              const char *prefix,
                              const char *value, unsigned long *docIx)
{
  PyObject *key;
  char *nodeName;
  PyDocumentObject *doc = (PyDocumentObject*)self->nodeData.ownerDocument;

  register PyAttrObject *attr =
  (PyAttrObject *) malloc(sizeof(PyAttrObject));

  /*if (attr == NULL)
    return PyErr_NoMemory();
  */

  Node_INIT((PyNodeObject*)attr, doc);

  attr->ob_type = &PyDomletteAttr_Type;
  attr->nodeData.parentNode = (PyObject *)self;
  Py_XINCREF(self);

  /*The from_pool does an incref*/
  attr->nodeData.namespaceURI = pystring_from_pool(doc->documentData.stringPool,
                                                  namespaceURI);
  attr->nodeData.prefix = pystring_from_pool(doc->documentData.stringPool,
                                            prefix);
  attr->nodeData.localName = pystring_from_pool(doc->documentData.stringPool,
                                               localName);
  nodeName = nodename_from_parts(prefix, localName);
  attr->nodeData.nodeName = pystring_from_pool(doc->documentData.stringPool,
                                              nodeName);
  free(nodeName);

  attr->attrData.nodeValue = PyString_FromString(value);
  attr->nodeData.docIndex = (*docIx)++;

  _Py_NewReference(attr);

  key = PyTuple_New(2);
  if (!strcmp(localName, "xmlns")) {
    /* a default namespace decl */
    PyTuple_SET_ITEM(key, 0, PyString_FromString(XMLNS_NAMESPACE));
    PyTuple_SET_ITEM(key, 1, PyString_FromString(""));
  } else if (prefix && !strcmp(prefix, "xmlns")) {
    /* a namespace decl */
    PyTuple_SET_ITEM(key, 0, PyString_FromString(XMLNS_NAMESPACE));
    PyTuple_SET_ITEM(key, 1, PyString_FromString(localName));
  } else if (namespaceURI && *namespaceURI) {
    /* with namespace */
    PyTuple_SET_ITEM(key, 0, PyString_FromString(namespaceURI));
    PyTuple_SET_ITEM(key, 1, PyString_FromString(localName));
  } else {
    /* default namespace */
    PyTuple_SET_ITEM(key, 0, PyString_FromString(""));
    PyTuple_SET_ITEM(key, 1, PyString_FromString(localName));
  }

  PyDict_SetItem(self->elementData.attributes,key,(PyObject *)attr);

  Py_XDECREF(attr);
  Py_XDECREF(key);


}


PyObject *Element_getAttributeNodeNSHelper(PyObject * self, PyObject * args) {
  /*return self.attributes.get((namespaceURI, localName), None)*/
  PyObject *namespaceUri;
  PyObject *localName;
  PyObject *key;
  PyObject *res;


  if(!PyArg_ParseTuple(args,"OO:getAttributeNodeNS",&namespaceUri,&localName))
    return NULL;

  if (!PyString_Check(namespaceUri) && ! PyString_Check(localName)) {
    PyErr_SetString(ErrorObject, "namespaceUri and localName must be a string");
    return NULL;
  }

  key = PyTuple_New(2);
  PyTuple_SetItem(key,0,(PyObject *)namespaceUri);
  PyTuple_SetItem(key,1,(PyObject *)localName);
  Py_XINCREF(namespaceUri);
  Py_XINCREF(localName);

  res = PyDict_GetItem(((PyElementObject *)self)->elementData.attributes,key);
  if (res == NULL) {
    res = Py_None;
  }
  Py_DECREF(key);
  return res;
}



/*
  Our External Interface

*/

static char PyElement_getAttributeNodeNS__doc__[] =
"Get an attribute from a namespace";
PyObject *PyElement_getAttributeNodeNS(PyObject * self, PyObject * args) {
  /*return self.attributes.get((namespaceURI, localName), None)*/
  PyObject *res;
  res = Element_getAttributeNodeNSHelper(self,args);
  Py_INCREF(res);

  return res;
}

static char PyElement_getAttributeNode__doc__[] =
"Get an attribute with out namespace";
PyObject *PyElement_getAttributeNode(PyObject * self, PyObject * args) {
  /*return self.attributes.get((None, localName), None)*/
  PyObject *res = Py_None;
  Py_INCREF(res);
  return res;
}



static char PyElement_getAttributeNS__doc__[] =
"Get an attribute's value from a namespace";
PyObject *PyElement_getAttributeNS(PyObject * self, PyObject * args) {
  /*return self.attributes.get((namespaceURI, localName), None).value*/
  PyObject *res;

  res = Element_getAttributeNodeNSHelper(self,args);
  if (res == Py_None) {
    return PyString_FromString("");
  }
  Py_INCREF(((PyAttrObject *)res)->attrData.nodeValue);
  return ((PyAttrObject *)res)->attrData.nodeValue;
}

static char PyElement_getAttribute__doc__[] =
"Get an attribute's value ";
PyObject *PyElement_getAttribute(PyObject * self, PyObject * args) {
  /*return self.attributes.get((None, localName), None).value*/
  PyObject *res = PyString_FromString("");
  Py_INCREF(res);
  return res;
}


static char PyElement_hasAttributeNS__doc__[] =
"Query if an element has an attribute from a namespace";
PyObject *PyElement_hasAttributeNS(PyObject * self, PyObject * args) {
  /*return self.attributes.get((namespaceURI, localName), None).value*/
  PyObject *res;

  res = Element_getAttributeNodeNSHelper(self,args);

  if (res == NULL) {
    Py_INCREF(Py_None);
    return Py_None;
  }

  if (res == Py_None) {
    return PyInt_FromLong(0);
  }
  return PyInt_FromLong(1);

}

static char PyElement_hasAttribute__doc__[] =
"Query if an element has an attribute";
PyObject *PyElement_hasAttribute(PyObject * self, PyObject * args) {
  return PyInt_FromLong(0);
}




static struct PyMethodDef Element_methods[] = {
  NODE_METHODS,
  {"getAttributeNodeNS",   (PyCFunction)PyElement_getAttributeNodeNS,  METH_VARARGS, PyElement_getAttributeNodeNS__doc__},
  {"getAttributeNS",   (PyCFunction)PyElement_getAttributeNS,  METH_VARARGS, PyElement_getAttributeNS__doc__},
  {"hasAttributeNS",   (PyCFunction)PyElement_hasAttributeNS,  METH_VARARGS, PyElement_hasAttributeNS__doc__},
  {"getAttributeNode",   (PyCFunction)PyElement_getAttributeNode,  METH_VARARGS, PyElement_getAttributeNode__doc__},
  {"getAttribute",   (PyCFunction)PyElement_getAttribute,  METH_VARARGS, PyElement_getAttribute__doc__},
  {"hasAttribute",   (PyCFunction)PyElement_hasAttribute,  METH_VARARGS, PyElement_hasAttribute__doc__},
  {NULL,     NULL}      /* sentinel */
};




/*
  Our Type interface
*/


static PyObject *element_getattr(PyElementObject *self, char *name)
{
  int temp;
  PyObject *rt = NULL;

  if (!strcmp(name,"tagName")) {
    rt = self->nodeData.nodeName;
  }
  else if (!strcmp(name,"nodeType")) {
    rt = g_elementNodeType;
  }
  else if (!strcmp(name,"nodeName")) {
    rt = self->nodeData.nodeName;
  }
  else if (!strcmp(name,"localName")) {
    rt = self->nodeData.localName;
  }
  else if (!strcmp(name,"prefix")) {
    rt = self->nodeData.prefix;
  }
  else if (!strcmp(name,"namespaceURI")) {
    rt = self->nodeData.namespaceURI;
  }
  else if (!strcmp(name,"attributes")) {
    rt = self->elementData.attributes;
  }
  else if (!strcmp(name,"childNodes")) {
    rt = self->elementData.childNodes;
  }
  else if (!strcmp(name,"lastChild")) {
    temp = PyList_GET_SIZE(self->elementData.childNodes);
    if (temp) {
      rt = PyList_GET_ITEM(self->elementData.childNodes,temp-1);
    } else {
      rt = Py_None;
    }
  }
  else if (!strcmp(name,"firstChild")) {
    if (PyList_GET_SIZE(self->elementData.childNodes)) {
      rt = PyList_GET_ITEM(self->elementData.childNodes,0);
    } else {
      rt = Py_None;
    }
  }
  if (rt) {
    Py_INCREF(rt);
    return rt;
  }
  return node_getattr((PyNodeObject*)self,name,Element_methods);
}


static void element_dealloc(PyElementObject *node)
{
  node_dealloc((PyNodeObject*)node);
}


static void element_build_display_string(char *buf, PyElementObject *v)
{

  sprintf(buf, "<cDomlette Element Node at %p: name='%s' with %d attributes and %d children>",
      v,
      PyString_AS_STRING(v->nodeData.nodeName),
      PyDict_Size(v->elementData.attributes),
      PyList_GET_SIZE(v->elementData.childNodes)
      );
}


static int element_print(PyElementObject *v, FILE *fp, int flags)
/* flags not used but required by interface */
{
  char buf[256];
  element_build_display_string(buf, v);
  fputs(buf, fp);
  return 0;
}

static PyObject *element_repr(PyElementObject *mp)
{

  char buf[256];
  element_build_display_string(buf, mp);
  return PyString_FromString(buf);
}


PyTypeObject PyDomletteElement_Type = {
    PyObject_HEAD_INIT(0)
    0,                              /*ob_size*/
    "cDomlette.Element",            /*tp_name*/
    sizeof(PyElementObject),        /*tp_basicsize*/
    0,                              /*tp_itemsize*/
    /**** methods ****/
    (destructor)element_dealloc,    /*tp_dealloc*/
    (printfunc)element_print,       /*tp_print*/
    (getattrfunc)element_getattr,   /*tp_getattr*/
    (setattrfunc)0,                 /*tp_setattr*/
    (cmpfunc)0,                     /*tp_compare*/
    (reprfunc)element_repr,         /*tp_repr*/
    0,                              /*tp_as_number*/
    0,                              /*tp_as_sequence*/
    0,                              /*tp_as_mapping*/
    (hashfunc)0,                    /*tp_hash*/
    (ternaryfunc)0,                 /*tp_call*/
    (reprfunc)0,                    /*tp_str*/
    0L, 0L, 0L, 0L,
};

