#!/usr/bin/env python
# arch-tag: 9982e83a-ad99-4992-9d92-80ff4d3119e8
# Copyright (C) 2005 Canonical Ltd.
#                    Author: David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test parsing of back-end version strings.
"""

import sys

import pybaz as arch
from pybaz import compat
from pybaz.tests import framework


class ExpectedVersionParser(object):

    def __init__(self, test, line):
        self.test = test
        self.line = line
        self.version = None
        self.release = None
        self.cotm = None

    def check(self):
        parse_version = compat.BazaarCommandVersion._parse
        version, release, cotm = parse_version(self.line)
        self.test.assertEqual(version, self.version)
        self.test.assertEqual(release, self.release)
        self.test.assertEqual(cotm, self.cotm)


class VersionParser(framework.NewTestCase):

    def expected(self, line):
        return ExpectedVersionParser(self, line)

    def test_baz_1_0(self):
        """Bazaar 1.0 release string is parsed correctly"""
        expect = self.expected(
            "baz Bazaar version 1.0 (thelove@canonical.com/dists"
            "--bazaar--0[releases/bazaar-1.0.cfg]) from regexps.com\n")
        expect.version = '1.0'
        expect.release = (1, 0, 0)
        expect.check()

    def test_baz_1_0_1(self):
        """Bazaar 1.0.1 release string is parsed correctly"""
        expect = self.expected(
            "baz Bazaar version 1.0.1 (thelove@canonical.com/dists"
            "--bazaar--0[releases/bazaar-1.0.cfg]) from regexps.com\n")
        expect.version = '1.0.1'
        expect.release = (1, 0, 1)
        expect.check()

    def test_baz_1_1_cotm(self):
        """Bazaar 1.1 cotm string is parsed correctly."""
        expect = self.expected(
            "baz Bazaar version 1.1-~200501101321 (thelove@"
            "canonical.com/dists--bazaar--1.1[bazaar--devo.cfg])")
        expect.version = '1.1-~200501101321'
        expect.release = (1, 1, 0)
        expect.cotm = 200501101321L
        expect.check()

    def test_baz_1_2_cotm(self):
        """Bazaar 1.2 cotm string is parsed correctly."""
        expect = self.expected(
            "baz Bazaar version 1.2~200502031254 (thelove@"
            "canonical.com/dists--bazaar--1.2[bazaar--devo.cfg])")
        expect.version = '1.2~200502031254'
        expect.release = (1, 2, 0)
        expect.cotm = 200502031254L
        expect.check()


class VersionComparison(framework.NewTestCase):

    def version(self, string):
        return compat.BazaarCommandVersion('baz Bazaar version %s' % string)

    def assertLessThan(self, a, b):
        if not (a < b):
            self.fail('%r !< %r' % (a, b))
        if not (b > a):
            self.fail('%r !> %r' % (b, a))

    def test_release_release(self):
        """BazaarCommandVersion compares releases correctly."""
        v1 = self.version('1.0')
        v2 = self.version('1.0')
        self.assertEqual(v1, v2)
        v2 = self.version('1.0.1')
        self.assertLessThan(v1, v2)
        v2 = self.version('1.1')
        self.assertLessThan(v1, v2)

    def test_release_cotm(self):
        """BazaarCommandVersion comparse release and cotm correctly."""
        v1 = self.version('1.0')
        v2 = self.version('1.1-~200501101321')
        self.assertLessThan(v1, v2)
        v1 = self.version('1.0.1')
        self.assertLessThan(v1, v2)
        v1 = v2
        v2 = self.version('1.1')
        self.assertLessThan(v1, v2)
        v2 = self.version('1.1.1')
        self.assertLessThan(v1, v2)
        v2 = self.version('1.2')
        self.assertLessThan(v1, v2)

    def test_cotm_cotm(self):
        """BazaarCommandVersion compares cotms correctly."""
        v1 = self.version('1.1-~200501101321')
        v2 = self.version('1.1~200501101321')
        self.assertEqual(v1, v2)
        v2 = self.version('1.2~200502031254')
        self.assertLessThan(v1, v2)
        v1 = self.version('1.2~200502031242')
        self.assertLessThan(v1, v2)


class CompatStatus(framework.NewTestCase):

    def version(self, string):
        return compat.BazaarCommandVersion('baz Bazaar version %s' % string)

    def set_baz(self, ver):
        self.baz = self.version(ver)

    def set_pybaz(self, ver):
        self.pybaz = self.version(ver)

    def check(self, status):
        self.assertEqual(compat._get_status(self.pybaz, self.baz),
                         getattr(compat.status, status))

    def test_same_minor(self):
        self.set_pybaz('1.1')
        self.set_baz('1.1')
        self.check('OK')
        self.set_baz('1.1.1')
        self.check('OK')

    def test_baz_more_recent(self):
        self.set_pybaz('1.1')
        self.set_baz('1.2')
        self.check('BAZ_TOO_NEW')
        self.set_baz('1.2~200502031254')
        self.check('BAZ_TOO_NEW')

    def test_baz_older(self):
        self.set_pybaz('1.1')
        self.set_baz('1.0')
        self.check('OK')
        self.set_baz('1.1-~200501101321')
        self.check('UNSUPPORTED')

    def test_cotm_pybaz(self):
        self.set_pybaz('1.2~200502141856')
        self.set_baz('1.1')
        self.check('OK')
        self.set_baz('1.1-~200501101321')
        self.check('UNSUPPORTED')
        self.set_baz('1.2~200502031254')
        self.check('COTM')
        self.set_baz('1.2')
        self.check('BAZ_TOO_NEW')
        self.set_baz('1.3~200502031254')
        self.check('BAZ_TOO_NEW')


class BackendCompatibility(framework.NewTestCase):

    def test(self):
        """Can get pybaz.backend.compatibility."""
        unused = arch.backend.compatibility


framework.register(__name__)
