/****************************************************************************
** servicesdlg.cpp - a dialog for browsing Jabber services/agents/transports
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include<qlayout.h>
#include<qpushbutton.h>
#include<qframe.h>

#include"servicesdlg.h"
#include"common.h"


static ServicesDlg *services_ptr = 0;

ServicesDlg::ServicesDlg(int _localStatus, QWidget *parent, const char *name)
:ServicesUI(parent, name, FALSE, WDestructiveClose)
{
	setCaption(CAP(tr("Manage Services")));
	setIcon(transport2icon("", STATUS_ONLINE));

	services_ptr = this;

	isBusy = FALSE;
	localStatus = _localStatus;

	busy = new BusyWidget(this);
	vb_side->addWidget(busy);

	pb_register->setEnabled(FALSE);
	pb_search->setEnabled(FALSE);
	pb_refresh->setEnabled(FALSE);

	connect(pb_refresh, SIGNAL(clicked()), SLOT(doRefresh()));
	connect(pb_close, SIGNAL(clicked()), SLOT(close()));
	connect(pb_register, SIGNAL(clicked()), SLOT(serviceRegister()));
	connect(pb_search, SIGNAL(clicked()), SLOT(serviceSearch()));
	connect(li_services, SIGNAL(highlighted(int)), SLOT(serviceSelected(int)));
	connect(li_services, SIGNAL(doubleClicked(QListBoxItem *)), SLOT(doubleClicked(QListBoxItem *)));

	resize(400,260);
}

ServicesDlg::~ServicesDlg()
{
	services_ptr = 0;
}

/* static */ ServicesDlg *ServicesDlg::find()
{
	return services_ptr;
}

void ServicesDlg::closeEvent(QCloseEvent *e)
{
	// cancel active transaction (refresh or getform)
	if(isBusy) {
		signalCancelTransaction(id);
	}

	// accept the close event
	e->accept();
}

void ServicesDlg::restoreWidgets()
{
	if(!isBusy) {
		pb_refresh->setEnabled(TRUE);
		li_services->setEnabled(TRUE);

		serviceSelected(li_services->currentItem());
	}
}

void ServicesDlg::doRefresh()
{
	if(localStatus == STATUS_OFFLINE) {
		QMessageBox::warning(this, tr("Warning"), tr("You must be connected to the server to manage services.  Please login first."));
		return;
	}

	li_services->clear();

	isBusy = TRUE;
	busy->start();
	pb_refresh->setEnabled(FALSE);
	pb_register->setEnabled(FALSE);
	pb_search->setEnabled(FALSE);

	signalRefresh(&id);
}

void ServicesDlg::loadSuccess(JabRoster *list)
{
	servicesList = *list;
	li_services->clear();

	QPtrListIterator<JabRosterEntry> it(servicesList.list);
	JabRosterEntry *entry;
	for(; (entry = it.current()); ++it) {
		li_services->insertItem(transport2icon(entry->jid), entry->nick);
	}

	if(isBusy) {
		busy->stop();
		isBusy = FALSE;
		restoreWidgets();
	}

	if(li_services->count() > 0)
		li_services->setSelected(0, TRUE);
}

void ServicesDlg::loadFail()
{
	if(isBusy) {
		busy->stop();
		isBusy = FALSE;
		restoreWidgets();

		QMessageBox::critical(this, tr("Error"), tr("There was an error retrieving the list of services."));
	}
}

void ServicesDlg::loadFormSuccess()
{
	if(isBusy) {
		busy->stop();
		isBusy = FALSE;
		restoreWidgets();
	}
}

void ServicesDlg::loadFormFail()
{
	if(isBusy) {
		busy->stop();
		isBusy = FALSE;
		restoreWidgets();

		if(actionType == SERVICESDLG_ACT_REGISTER)
			QMessageBox::critical(this, tr("Error"), tr("There was an error retrieving the registration form for the %1 service.").arg(regname));
		else
			QMessageBox::critical(this, tr("Error"), tr("There was an error retrieving the search form for the %1 service.").arg(regname));
	}
}

void ServicesDlg::localUpdate(const JabRosterEntry &e)
{
	int oldstate = localStatus;
	localStatus = e.localStatus();

	if(localStatus == STATUS_OFFLINE) {
		if(isBusy) {
			busy->stop();
			isBusy = FALSE;
		}

		pb_register->setEnabled(FALSE);
		pb_search->setEnabled(FALSE);
		pb_refresh->setEnabled(FALSE);
		li_services->clear();
		servicesList.clear();
	}
	else {
		// coming online?
		if(oldstate == STATUS_OFFLINE) {
			doRefresh(); // this will also turn the refresh button on when finished
		}
	}
}

void ServicesDlg::doubleClicked(QListBoxItem *)
{
	serviceRegister();
}

void ServicesDlg::serviceSelected(int x)
{
	if(x == -1)
		return;
	JabRosterEntry *entry = servicesList.list.at(x);
	if(!entry)
		return;

	if(localStatus == STATUS_OFFLINE)
		return;

	pb_register->setEnabled(entry->isRegisterable);
	pb_search->setEnabled(entry->isSearchable);
}

void ServicesDlg::serviceRegister()
{
	if(!pb_register->isEnabled())
		return;

	JabRosterEntry *entry = servicesList.list.at(li_services->currentItem());
	if(!entry)
		return;

	//printf("ServicesDlg: getting registration form: %s\n", entry->nick.latin1());
	regname = entry->jid;
	actionType = SERVICESDLG_ACT_REGISTER;

	// block user input
	li_services->setEnabled(FALSE);
	pb_refresh->setEnabled(FALSE);
	pb_register->setEnabled(FALSE);
	pb_search->setEnabled(FALSE);

	isBusy = TRUE;
	busy->start();

	signalGetRegForm(entry->jid, &id);
}

void ServicesDlg::serviceSearch()
{
	JabRosterEntry *entry = servicesList.list.at(li_services->currentItem());
	if(!entry)
		return;

	//printf("ServicesDlg: getting search form: %s\n", entry->nick.latin1());
	regname = entry->jid;
	actionType = SERVICESDLG_ACT_SEARCH;

	// block user input
	li_services->setEnabled(FALSE);
	pb_refresh->setEnabled(FALSE);
	pb_register->setEnabled(FALSE);
	pb_search->setEnabled(FALSE);

	isBusy = TRUE;
	busy->start();

	signalSearch(entry->jid, &id);
}


RegistrationDlg::RegistrationDlg(const JabForm &_form, QWidget *parent, const char *name)
:QDialog(parent, name, FALSE, WDestructiveClose), UniqueWindow(TRUE, "RegistrationDlg", this, cleanJid(_form.jid.s()))
{
	form = _form;
	isBusy = FALSE;

	setCaption(CAP(tr("Service Registration")));
	resize(16, 16);

	QVBoxLayout *vb1 = new QVBoxLayout(this, 4);
	QLabel *l;
	l = new QLabel(QString(tr("Registration for \"%1\"")).arg(form.jid.s()), this);
	l->setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed));
	vb1->addWidget(l);
	l = new QLabel(QString("<qt>%1</qt>").arg(form.instructions), this);
	l->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	vb1->addWidget(l);

	QGridLayout *gr = new QGridLayout(vb1, 2);
	int at = 0;
	lb_field.setAutoDelete(TRUE);
	le_field.setAutoDelete(TRUE);
	QPtrListIterator<JabFormField> it(form);
	JabFormField *f;
	for(; (f = it.current()); ++it) {
		QLabel *lb = new QLabel(f->fieldName(), this);
		QLineEdit *le = new QLineEdit(this);
		if(f->type() == JabFormField::password)
			le->setEchoMode(QLineEdit::Password);
		le->setText(f->value());

		lb_field.append(lb);
		le_field.append(le);

		gr->addWidget(lb, at, 0);
		gr->addWidget(le, at, 1);
		++at;
	}

	QHBoxLayout *hb1 = new QHBoxLayout(vb1);
	busy = new BusyWidget(this);
	hb1->addStretch(1);
	hb1->addWidget(busy);

	QFrame *line = new QFrame(this);
	line->setFixedHeight(2);
	line->setFrameStyle(QFrame::HLine | QFrame::Sunken);
	vb1->addWidget(line);

	QHBoxLayout *hb2 = new QHBoxLayout(vb1);
	pb_register = new QPushButton(tr("&Register"), this);
	connect(pb_register, SIGNAL(clicked()), SLOT(doRegSubmit()));
	hb2->addWidget(pb_register);
	hb2->addStretch(1);
	pb_close = new QPushButton(tr("&Close"), this);
	connect(pb_close, SIGNAL(clicked()), SLOT(close()));
	hb2->addWidget(pb_close);

	vb1->activate();

	resize(320, height() + 80);
}

void RegistrationDlg::closeEvent(QCloseEvent *e)
{
	if(isBusy) {
		//QMessageBox::information(this, tr("Busy"), tr("Registration has already been submitted, it cannot be canceled.  Please wait for the server to respond."));
		e->ignore();
		return;
	}

	e->accept();
}

void RegistrationDlg::putRegFormResponse(bool ok, const QString &err)
{
	busy->stop();
	isBusy = FALSE;

	if(ok) {
		QMessageBox::information(this, tr("Success"), tr("Registration was successful."));
	}
	else {
		QMessageBox::critical(this, tr("Failed"), QString(tr("Registration failed.  Reason:\n\"%1\"")).arg(err));
	}

	close();
}

void RegistrationDlg::doRegSubmit()
{
	JabForm submitform = form;

	// import the changes back into the form.
	// the QPtrList of QLineEdits should be in the same order
	QPtrListIterator<JabFormField> it(submitform);
	QPtrListIterator<QLineEdit> lit(le_field);
	JabFormField *f;
	for(; (f = it.current());) {
		QLineEdit *le = lit.current();
		f->setValue(le->text());
		++it;
		++lit;
	}

	pb_register->setEnabled(FALSE);
	pb_close->setEnabled(FALSE);
	isBusy = TRUE;
	busy->start();

	signalSubmitForm(submitform, &id);
}

void RegistrationDlg::localUpdate(const JabRosterEntry &e)
{
	int localStatus = e.localStatus();

	// if the user goes offline then cancel the form
	if(localStatus == STATUS_OFFLINE) {
		if(isBusy) {
			putRegFormResponse(FALSE, "Disconnected");  // <-- this will turn off busy also
		}
		close();
	}
}

/*static*/ RegistrationDlg * RegistrationDlg::find(const QString &xjid)
{
        return (RegistrationDlg *)UniqueWindowBank::find("RegistrationDlg", cleanJid(xjid));
}
