/****************************************************************************
** filetrans.cpp - classes for handling file transfers
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"filetrans.h"
#include"common.h"
#include<qtimer.h>
#include<qdir.h>
#include<qstringlist.h>

#include<stdlib.h>
#include<time.h>

#ifndef Q_WS_WIN
#include<signal.h>
#endif


static QString urlEncode(const QString &in);
static QString urlDecode(const QString &in);
static QString mimeGuess(const QString &in);
static QString generateHTMLDirectory(bool base, const QString &dir, const QString &title);


/****************************************************************************
  FileTransfer
****************************************************************************/
FileTransfer::FileTransfer(FileServer *_par)
{
	par = _par;
	v_isValid = FALSE;
	finished = FALSE;
	at = 0;
	bytesSent = 0;
	type = FT_FILE;
}

FileTransfer::~FileTransfer()
{
}

void FileTransfer::discard()
{
	/*if(v_isValid) {
		sock.close();
		if(!finished) {
			file.close();
		}
	}*/

	discarding();
	deleteLater();
}

void FileTransfer::start()
{
	connect(&sock, SIGNAL(readyRead()), SLOT(sock_readyRead()));
	connect(&sock, SIGNAL(connectionClosed()), SLOT(sock_connectionClosed()));
	connect(&sock, SIGNAL(bytesWritten(int)), SLOT(sock_bytesWritten(int)));

	printf("%04d: new transfer: ip=[%s]\n", id, sock.peerAddress().toString().latin1());
}

void FileTransfer::processChunk()
{
	if(finished)
		return;

	QByteArray data;
	data.resize(1024);
	int readSize;

	if(type == FT_FILE) {
		// read a chunk from the file
		readSize = file.readBlock(data.data(), 1024);

		if(file.atEnd()) {
			file.close();
			finished = TRUE;
		}
	}
	else {
		int left = stringData.size() - bytesSent;
		if(left > 1024)
			readSize = 1024;
		else {
			readSize = left;
			finished = TRUE;
		}

		for(int n = 0; n < readSize; ++n)
			data[n] = stringData[bytesSent + n];
	}

	bytesSent += readSize;
	printf("\r%04d: sent %d bytes                    ", id, bytesSent);
	fflush(stdout);
	sock.writeBlock(data.data(), readSize);
}

void FileTransfer::sock_readyRead()
{
	printf("%04d: data ready\n", id);

	if(!isValid()) {
		int size = sock.readBlock(recvBuf + at, 1024 - at);
		if(size == -1)
			return;
		at += size;

		//if(size > 1023)
		//	size = 1023;
		//recvBuf[size] = 0;
		//printf("got: [%s]\n", recvBuf);
		//return;

		bool ok = FALSE;

		if(at > 1023) {
			at = 1023;
			recvBuf[at] = 0;
			ok = TRUE;
		}
		else {
			for(int n = 0; n < at; ++n) {
				if(recvBuf[n] == '\n' || recvBuf[n] == '\r') {
					recvBuf[n] = 0;
					ok = TRUE;
					break;
				}
			}
		}

		if(ok) {
			// see if it is a valid request
			QString str = recvBuf;
			printf("%04d: request: [%s]\n", id, str.latin1());

			QString res, http_ver;
			if(!getGETInfo(str, &res, &http_ver)) {
				discard();
				return;
			}

			printf("%04d: res: [%s], http_ver: [%s]\n", id, res.latin1(), http_ver.latin1());

			char *useImage_data = 0;
			int useImage_len = 0;
			if(res == "/image/ft_back.png") {
				useImage_data = pixdat_ft_back;
				useImage_len = pixlen_ft_back;
			}
			else if(res == "/image/ft_file.png") {
				useImage_data = pixdat_ft_file;
				useImage_len = pixlen_ft_file;
			}
			else if(res == "/image/ft_folder.png") {
				useImage_data = pixdat_ft_folder;
				useImage_len = pixlen_ft_folder;
			}


			if(!useImage_data) {
				QString resource, key;
				if(!extractResourceInfo(res, &resource, &key)) {
					discard();
					return;
				}

				// does the key pass?
				FileServerItem *fi = par->getFSI(key);
				if(!fi) {
					discard();
					return;
				}

				printf("%04d: validated: key=[%s]\n", id, key.latin1());

				// how about the resource?
				QString fname;
				QFileInfo info;
				int transType = FT_FILE;

				if(fi->type == 0) {
					if(resource != fi->file.fileName()) {
						discard();
						return;
					}
					fname = fi->file.filePath();
					info.setFile(fname);
				}
				else {
					QString base, rest;
					int n = resource.find('/');
					if(n == -1) {
						base = resource;
						rest = "/";
					}
					else {
						base = resource.mid(0, n);
						rest = resource.mid(n);
					}

					if(base != fi->file.fileName()) {
						discard();
						return;
					}

					fname = fi->file.filePath() + rest;
					info.setFile(fname);
					QDir d;
					if(info.isDir())
						d.setPath(fname);
					else
						d = info.dir();

					// simplify
					QString str = d.canonicalPath();
					if(str.isNull()) {
						discard();
						return;
					}
					fname = str;
					if(!info.isDir())
						fname += QString("/") + info.fileName();

					info.setFile(fname);

					QString basePath = fi->file.filePath();
					int len = basePath.length();
					for(n = 0; n < len; ++n) {
						if(basePath.at(n) != fname.at(n))
							break;
					}
					if(n < len) {
						printf("%04d: illegal path.  hacking attempt?\n", id);
						discard();
						return;
					}

					if(info.isDir()) {
						transType = FT_STRING;
					}
				}


				if(transType == FT_FILE) {
					printf("%04d: serving: [%s]\n", id, info.filePath().latin1());

					type = FT_FILE;

					// now try to open it
					file.setName(fname);
					if(!file.open(IO_ReadOnly)) {
						discard();
						return;
					}

					v_isValid = TRUE;
					valid();

					// write header
					QString header;
					header.sprintf(
						"HTTP/1.0 200 OK\n"
						"Content-Type: %s\n"
						"Content-Length: %d\n"
						"Connection: close\n"
						"\n", mimeGuess(info.fileName()).latin1(), info.size());
					printf("Sending header:\n%s", header.latin1());
					sock.writeBlock(header, header.length());
				}
				else {
					printf("%04d: listing: [%s]\n", id, info.filePath().latin1());

					bool base = FALSE;
					if(fi->file.filePath() == info.filePath())
						base = TRUE;

					type = FT_STRING;
					stringData = generateHTMLDirectory(base, info.filePath(), QString("Index of %1").arg(resource)).utf8();

					v_isValid = TRUE;
					valid();

					// write header
					QString header = QString(
						"HTTP/1.0 200 OK\n"
						"Content-Type: text/html\n"
						"Content-Length: %1\n"
						"Connection: close\n"
						"\n").arg(stringData.size());
					printf("Sending header:\n%s", header.latin1());
					sock.writeBlock(header, header.length());
				}
			}
			else {
				printf("%04d: serving: [%s]\n", id, res.latin1());

				type = FT_STRING;
				stringData.resize(useImage_len);
				memcpy(stringData.data(), useImage_data, useImage_len);

				v_isValid = TRUE;
				valid();

				// write header
				QString header = QString(
					"HTTP/1.0 200 OK\n"
					"Content-Type: image/png\n"
					"Content-Length: %1\n"
					"Connection: close\n"
					"\n").arg(stringData.size());
				printf("Sending header:\n%s", header.latin1());
				sock.writeBlock(header, header.length());

			}

			// write the first chunk
			processChunk();
		}
	}
}

void FileTransfer::sock_connectionClosed()
{
	printf("\n%04d: connection closed\n", id);
	discard();
}

void FileTransfer::sock_bytesWritten(int)
{
	if(sock.bytesToWrite() <= 0) {
		if(finished) {
			printf("\n%04d: transfer finished.\n", id);
			sock.close();
			discard();
			return;
		}

		QTimer::singleShot(0, this, SLOT(processChunk()));
	}
}

bool FileTransfer::getGETInfo(const QString &getReq, QString *resource, QString *http_ver)
{
	QString req = getReq.simplifyWhiteSpace();

	// these must be the first 4 chars
	if(req.left(4) != "GET ")
		return FALSE;

	QString str = getReq.mid(4);

	// is there a space later?  if so, grab what is after it (assuming HTTP version)
	*http_ver = "";
	int n = str.find(' ');
	if(n != -1) {
		if((int)str.length() > n + 1)
			*http_ver = str.mid(n + 1);
		str = str.mid(0, n);
	}
	*resource = str;

	return TRUE;
}

bool FileTransfer::extractResourceInfo(const QString &resource, QString *fname, QString *key)
{
	QString str = resource;

	// make sure we begin with a slash
	if(str.at(0) != '/')
		return FALSE;

	// remove it
	str.remove(0,1);

	// make sure there is a slash to separate key and filename
	int n = str.find('/');
	if(n == -1)
		return FALSE;

	*key = str.mid(0, n);

	*fname = str.mid(n);
	fname->remove(0,1); // remove the beginning slash
	*fname = urlDecode(*fname);

	return TRUE;
}

QString urlEncode(const QString &in)
{
	QString out;

	for(unsigned int n = 0; n < in.length(); ++n) {
		if(in.at(n) == '.') {
			out.append('.');
		}
		else if(in.at(n) == ' ') {
			out.append('+');
		}
		else if(!in.at(n).isLetterOrNumber()) {
			// hex encode
			QString hex;
			hex.sprintf("%%%02X", in.at(n).latin1());
			out.append(hex);
		}
		else {
			out.append(in.at(n));
		}
	}

	return out;
}

QString urlDecode(const QString &in)
{
	QString out;

	for(unsigned int n = 0; n < in.length(); ++n) {
		if(in.at(n) == '%' && (in.length() - n - 1) >= 2) {
			QString str = in.mid(n+1,2);
			bool ok;
			char c = str.toInt(&ok, 16);
			if(!ok)
				continue;

			QChar a(c);
			out.append(a);
			n += 2;
		}
		else if(in.at(n) == '+')
			out.append(' ');
		else
			out.append(in.at(n));
	}

	return out;
}

QString mimeGuess(const QString &in)
{
	QString ext;
	int n;
	for(n = in.length()-1; n > 0; --n) {
		if(in.at(n) == '.')
			break;
	}
	if(n == 0)
		ext = "";
	else
		ext = in.mid(n+1).lower();

	if(ext == "txt" || ext == "c" || ext == "h" || ext == "cpp")
		return "text/plain";
	else if(ext == "gif")
		return "image/gif";
	else if(ext == "jpeg" || ext == "jpg" || ext == "jpe")
		return "image/jpeg";
	else if(ext == "png")
		return "image/png";

	return "application/octet-stream";
}

QString generateHTMLDirectory(bool base, const QString &dir, const QString &title)
{
	QDir d(dir);
	QString text;

	// blatant sourceforge ripoff
	text = QString(
		"<html>\n"
		"<head>\n"
		"  <title>%1</title>\n"
		"  <style type=\"text/css\">\n"
		"    body {\n"
		//"      background-image: url(img/sf-stipple.png);\n"
		"      background-color: #aaa;\n"
		"      margin: 0;\n"
		"    }\n"
		"    td { font: small sans-serif; padding-left: 0.5em; padding-right: 0.5em;}\n"
		"    .footer {text-align: center;}\n"
		"    .directory {color: #595;}\n"
		"    .sort {color: #000;}\n"
		"    .label-date {font-family: monospace; color: #555; text-align: left;}\n"
		"    .label-size {font-family: monospace; color: #555; text-align: center;}\n"
		"  </style>\n"
		"</head>\n"
		"\n"
		"<body>\n"
		"\n"
		"<table align=\"center\" width=\"90%\" bgcolor=\"#FFFFFF\" cellpadding=\"0\" cellspacing=\"0\" border=\"0\">\n"
		"  <tr><td align=\"center\">\n"
		"    <table width=\"100%\" cellpadding=\"3\" cellspacing=\"1\" bgcolor=\"#FFFFFF\">\n"
		"      <tr bgcolor=\"#FFFFFF\">\n<td>Psi Folder Share</td></tr>\n"
		//"      <tr bgcolor=\"#FFFFFF\">\n"
		//"        <td colspan=\"3\">Current Directory: <b>http://telia.dl.sourceforge.net/ mirrors/ </b></td>\n"
		//"      </tr>\n"
		"      <tr bgcolor=\"#FFFFFF\">\n"
		"        <th width=\"80%\">File Name</th>\n"
		"        <th width=\"10%\">Size</th>\n"
		"        <th width=\"10%\">Date</th>\n"
		"      </tr>\n"
		).arg(title);

	QStringList entries = d.entryList();
	entries.sort();
	bool colorOn = FALSE;
	for(QStringList::Iterator it = entries.begin(); it != entries.end(); ++it) {
		QString fname = *it;
		if(fname == ".")
			continue;

		if(fname == ".." && base)
			continue;

		QString str = "";
		if(colorOn) {
			str += "<tr bgcolor=\"#DDDDDD\">\n";
			colorOn = FALSE;
		}
		else {
			str += "<tr bgcolor=\"#EEEEEE\">\n";
			colorOn = TRUE;
		}

		QFileInfo info(dir + '/' + fname);
		if(info.isDir())
			fname += '/';

		if(fname == "../")
			str += QString("<td nowrap align=\"left\"><img src=\"%1\" align=\"middle\" width=\"20\" height=\"22\">&nbsp;<a href=\"../\"><font class=\"directory\"><b>-- Parent Directory --</b></font></a></td>\n").arg("/image/ft_back.png");
		else
			str += QString("<td nowrap align=\"left\"><img src=\"%1\" align=\"middle\" width=\"20\" height=\"22\">&nbsp;").arg(info.isDir() ? "/image/ft_folder.png" : "/image/ft_file.png") + QString("<a href=\"%1\">").arg(fname) + QString("<font class=\"directory\"><b>%1</b></font></a></td>\n").arg(fname);

		QString size, date;
		if(info.isDir())
			size = "-";
		else
			size.sprintf("%d", info.size());

		date = info.lastModified().toString();

		str += QString(
			"  <td nowrap align=\"center\" class=\"label-size\">%1</td>\n").arg(size);
		str += QString(
			"  <td nowrap align=\"left\" class=\"label-date\">%1</td>\n"
			"</tr>\n").arg(date);

		text += str;
	}

	text += QString(
		"        <tr>\n"
		"          <td colspan=\"3\">\n"
		"            <pre></pre>\n"
		"          </td>\n"
		"        </tr>\n"
		"      </table>\n"
		"      <p class=\"footer\">%1</p>\n"
		"    </td>\n"
		"  </tr>\n"
		"</table>\n"
		"\n"
		"</body>\n"
		"</html>\n\n").arg(QDateTime::currentDateTime().toString());

	return text;
}


/****************************************************************************
  FileServer
****************************************************************************/
FileServer::FileServer()
{
#ifndef Q_WS_WIN
	// block SIGPIPE
	old_sigpipe = signal(SIGPIPE, SIG_IGN);
#endif

	// seed random number generator
	srand(time(NULL));

	current_id = 0;
	sigmap[0] = new QSignalMapper(this);
	sigmap[1] = new QSignalMapper(this);
	connect(sigmap[0], SIGNAL(mapped(int)), this, SLOT(transferDiscarded(int)));
	filelist.setAutoDelete(TRUE);

	host = "127.0.0.1";
	port = 12345;
	serv = 0;
}

FileServer::~FileServer()
{
	delete sigmap[0];
	delete sigmap[1];

	ftlist.setAutoDelete(TRUE);
	ftlist.clear();

#ifndef Q_WS_WIN
	signal(SIGPIPE, old_sigpipe);
#endif
}

void FileServer::setServer(const QString &_host, int _port)
{
	host = _host;
	port = _port;

	listen();
}

FileServerItem *FileServer::addFile(const QString &_fname, const QString &desc, const QString &who)
{
	QString fname = _fname;

	// hack off a trailing backslash
	if(fname.at(fname.length()-1) == '/')
		fname.remove(fname.length()-1, 1);

	// get unused key
	QString key;
	bool ok;
	do {
		key = genKey();

		ok = TRUE;
		QPtrListIterator<FileServerItem> it(filelist);
		FileServerItem *fi;
		for(; (fi = it.current()); ++it) {
			if(fi->key == key) {
				ok = FALSE;
				break;
			}
		}
	} while(!ok);

	QFileInfo info(fname);
	info.convertToAbs();
	if(!info.exists())
		return 0;

	FileServerItem *i;
	i = new FileServerItem;
	i->file = info;
	i->type = info.isDir() ? 1: 0;
	i->key = key;
	i->who = who;
	i->desc = desc;
	i->url = QString("http://") + host + QString(":%1").arg(port) + "/" + key + "/" + urlEncode(i->file.fileName());
	if(i->type == 1)
		i->url += '/';
	filelist.append(i);

	return i;
}

QPtrList<FileServerItem> FileServer::items() const
{
	QPtrList<FileServerItem> list;
	list.setAutoDelete(FALSE);

	QPtrListIterator<FileServerItem> it(filelist);
	FileServerItem *fi;
	for(; (fi = it.current()); ++it)
		list.append(fi);

	return list;
}

void FileServer::listen()
{
	if(serv) {
		delete serv;
		serv = 0;
	}

	serv = new ServSock(port);
	connect(serv, SIGNAL(connectionReady(int)), SLOT(connectionReady(int)));
}

FileTransfer *FileServer::findByID(int id)
{
	QPtrListIterator<FileTransfer> it(ftlist);
	for(FileTransfer *f; (f = it.current()); ++it) {
		if(f->id == id)
			return f;
	}

	return 0;
}

void FileServer::connectionReady(int sock)
{
	FileTransfer *f = new FileTransfer(this);
	f->sock.setSocket(sock);
	f->id = current_id;
	ftlist.append(f);
	++current_id;

	sigmap[0]->setMapping(f, f->id);
	connect(f, SIGNAL(discarding()), sigmap[0], SLOT(map()));
	f->start();
}

QString FileServer::genKey()
{
	QString key;

	for(int i = 0; i < 4; ++i) {
		int word = rand() & 0xffff;
		for(int n = 0; n < 4; ++n) {
			QString s;
			s.sprintf("%x", (word >> (n * 4)) & 0xf);
			key.append(s);
		}
	}

	return key;
}

FileServerItem *FileServer::getFSI(const QString &key)
{
	QPtrListIterator<FileServerItem> it(filelist);
	FileServerItem *fi;
	for(; (fi = it.current()); ++it) {
		if(fi->key == key)
			return fi;
	}

	return 0;
}

void FileServer::transferDiscarded(int x)
{
	FileTransfer *f = findByID(x);
	if(!f)
		return;

	printf("%04d: transfer object discarded.\n", f->id);

	ftlist.remove(f);
}

//			QGuardedPtr<FileTransfer> gf = f;
//			newDownload(gf);

/****************************************************************************
  ServSock
****************************************************************************/
ServSock::ServSock(int port)
:QServerSocket(port, 16)
{
}

void ServSock::newConnection(int x)
{
	connectionReady(x);
}
