/****************************************************************************
** accountdlg.cpp - dialogs for manipulating Jabber accounts
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"accountdlg.h"
#include"common.h"

#include<qpushbutton.h>
#include<qlistview.h>
#include<qlineedit.h>
#include<qcheckbox.h>
#include<qlabel.h>
#include<qlayout.h>


static AccountManageDlg *accountmanage_ptr = 0;

AccountManageDlg::AccountManageDlg(JabSessionManager *_jsm, QWidget *parent, const char *name)
:AccountManageUI(parent, name, FALSE, WDestructiveClose)
{
	setCaption(CAP(tr("Jabber Accounts")));

	accountmanage_ptr = this;
	jsm = _jsm;

	// setup signals
	connect(pb_modify, SIGNAL(clicked()), SLOT(modify()));
	connect(pb_close, SIGNAL(clicked()), SLOT(reject()));
	connect(lv_accs, SIGNAL(doubleClicked(QListViewItem *)), SLOT(modify(QListViewItem *)));

	connect(jsm, SIGNAL(sessionUpdate(JabSession *)), SLOT(sessionUpdate(JabSession *)));

	lv_accs->setAllColumnsShowFocus(TRUE);

	QPtrList<JabSession> list = jsm->getList();
	QPtrListIterator<JabSession> it(list);
	for(JabSession *s; (s = it.current()); ++it)
		new AccountViewItem(lv_accs, s);

	pb_add->setEnabled(FALSE);
	pb_remove->setEnabled(FALSE);

	if(lv_accs->childCount() > 0)
		lv_accs->setSelected(lv_accs->firstChild(), TRUE);
}

AccountManageDlg::~AccountManageDlg()
{
	accountmanage_ptr = 0;
}

/* static */ AccountManageDlg *AccountManageDlg::find()
{
	return accountmanage_ptr;
}

void AccountManageDlg::modify()
{
	QListViewItem *lvi = lv_accs->currentItem();
	if(!lvi)
		return;

	modify(lvi);
}

void AccountManageDlg::modify(QListViewItem *lvi)
{
	signalModify(lvi->text(0));
}

void AccountManageDlg::updateEntry(JabSession *s)
{
	AccountViewItem *i;
	QListViewItemIterator it(lv_accs);
	for(; (i = (AccountViewItem *)it.current()) ; ++it) {
		if(i->s == s)
			break;
	}
	if(!i)
		return;

	i->updateInfo();
}

void AccountManageDlg::sessionUpdate(JabSession *s)
{
	updateEntry(s);
}


AccountViewItem::AccountViewItem(QListView *par, JabSession *_s)
:QListViewItem(par)
{
	s = _s;

	updateInfo();
}

void AccountViewItem::updateInfo()
{
	setText(0, s->name());
	setPixmap(0, *pix_account);
	setText(1, s->acc()->host);
	setText(2, s->serv()->isActive() ? "Active" : "Not active");
}


AccountModifyDlg::AccountModifyDlg(JabSession *_s, bool _haveSSL, QWidget *parent, const char *name)
:AccountModifyUI(parent, name, FALSE, WDestructiveClose), UniqueWindow(TRUE, "AccountModifyDlg", this, _s->acc()->name)
{
	setCaption(tr("Account Properties"));
	setIcon(*pix_account);

	resize(width(), sizeHint().height());

	s = _s;
	haveSSL = _haveSSL;
	UserAccount &acc = *s->acc();

	connect(pb_close, SIGNAL(clicked()), SLOT(reject()));
	connect(ck_pass, SIGNAL(toggled(bool)), le_pass, SLOT(setEnabled(bool)));
	connect(pb_save, SIGNAL(clicked()), SLOT(save()));
	connect(pb_reg, SIGNAL(clicked()), SLOT(doRegister()));

	le_pass->setEnabled(FALSE);

	pb_unreg->setEnabled(FALSE);

	if(!acc.name.isEmpty()) {
		le_name->setText(acc.name);
		le_host->setText(acc.host);
		le_port->setText(QString::number(acc.port));
		if(acc.opt_ssl && haveSSL)
			ck_ssl->setChecked(acc.opt_ssl);
		connect(ck_ssl, SIGNAL(toggled(bool)), SLOT(sslToggled(bool)));

		le_user->setText(acc.user);
		if(acc.opt_pass) {
			ck_pass->setChecked(TRUE);
			le_pass->setText(acc.pass);
		}
		le_resource->setText(acc.resource);
		le_priority->setText(QString::number(acc.priority));

		ck_plain->setChecked(acc.opt_plain);
		ck_auto->setChecked(acc.opt_auto);
		ck_log->setChecked(acc.opt_log);
		ck_keepAlive->setChecked(acc.opt_keepAlive);

		pb_save->setFocus();
	}
	else
		le_name->setFocus();
}

AccountModifyDlg::~AccountModifyDlg()
{
}

/*static*/ AccountModifyDlg * AccountModifyDlg::find(const QString &accname)
{
        return (AccountModifyDlg *)UniqueWindowBank::find("AccountModifyDlg", accname.lower());
}

void AccountModifyDlg::sslToggled(bool on)
{
	if(on && !haveSSL) {
		QMessageBox::information(this, tr("OpenSSL error"), QString("<qt>") + tr("Sorry, but you must have <b>OpenSSL v0.9.6b</b> or compatible installed to use this feature.") + "</qt>");
		ck_ssl->setChecked(FALSE);
		return;
	}

	le_port->setText(on ? "5223": "5222");
}

void AccountModifyDlg::save()
{
	if(s->serv()->isActive()) {
		QMessageBox::information(this, tr("Error"), tr("Please disconnect from this account before trying to modify it."));
		return;
	}

	UserAccount &acc = *s->acc();

	QString before = acc.name.lower();
	QString after = le_name->text().lower();
	if(after.length() == 0) {
		QMessageBox::information(this, tr("Error"), tr("You must specify a name for the account before you may save it."));
		return;
	}
	if(before != after) {
		JabSessionManager *jsm = (JabSessionManager *)s->parent();
		if(jsm->find(after)) {
			QMessageBox::information(this, tr("Error"), tr("There is already an account with this name.  Please select a different name."));
			return;
		}
	}

	bool newUser = FALSE;
	if(acc.user != le_user->text() || acc.host != le_host->text() || acc.port != le_port->text().toInt()) {
		newUser = TRUE;
	}

	acc.name = le_name->text();
	acc.host = le_host->text();
	acc.port = le_port->text().toInt();
	acc.opt_ssl = ck_ssl->isChecked();
	acc.user = le_user->text();
	acc.opt_pass = ck_pass->isChecked();
	if(acc.opt_pass)
		acc.pass = le_pass->text();
	else
		acc.pass = "";
	acc.resource = le_resource->text();
	acc.priority = le_priority->text().toInt();

	acc.opt_plain = ck_plain->isChecked();
	acc.opt_auto = ck_auto->isChecked();
	acc.opt_log = ck_log->isChecked();
	acc.opt_keepAlive = ck_keepAlive->isChecked();

	s->update(newUser);

	accept();
}

void AccountModifyDlg::doRegister()
{
	if(le_pass->text().isEmpty()) {
		QMessageBox::warning(this, tr("Error"), tr("You must specify a password for the new account."));
		return;
	}

	int n = QMessageBox::information(this, tr("Warning"),tr(
		"You have elected to create a new account on the specified server.\n"
		"Please note that this may not work on all servers, which means you may\n"
		"have to sign up with the host some other way instead (like through the web).\n"
		"\n"
		"Think about it like this:  if you were running a mail server, would you\n"
		"allow anonymous users to sign up for accounts on your system?  Probably not.\n"
		"For now though (Spring 2002), most hosts (like jabber.com and jabber.org) allow\n"
		"anonymous signups.  Check jabberview.com for a nice list of free hosts.\n"
		"\n"
		"Click on \"Yes\" to attempt to register an account.\n"),
		tr("&Yes"), tr("&No"));

	if(n != 0)
		return;

	// get the form opts that we need
	QString host = le_host->text();
	int port = le_port->text().toInt();
	bool opt_ssl = ck_ssl->isChecked();
	QString user = le_user->text();
	QString pass = le_pass->text();

	// we should probably verify they are ok

	// perform a registration
	AccountRegisterDlg::reg(this, host, port, opt_ssl, user, pass);
}


AccountRegisterDlg::AccountRegisterDlg(QWidget *par, const char *name)
:QDialog(par, name, TRUE)
{
	setCaption(tr("Account Registration"));

	QVBoxLayout *vb = new QVBoxLayout(this, 4);
	QHBoxLayout *hb = new QHBoxLayout(vb);
	QLabel *l = new QLabel(tr("Registering account..."), this);
	hb->addWidget(l);
	busy = new BusyWidget(this);
	hb->addWidget(busy);
	QPushButton *pb_cancel = new QPushButton(tr("&Cancel"), this);
	hb->addWidget(pb_cancel);
	connect(pb_cancel, SIGNAL(clicked()), SLOT(cancel()));
	pb_cancel->setFocus();

	jab = new Jabber;
	connect(jab, SIGNAL(accRegisterFinished(bool, const QString &)), SLOT(accRegisterFinished(bool, const QString &)));
	connect(jab, SIGNAL(error(JabError *)), SLOT(jab_error(JabError *)));
}

AccountRegisterDlg::~AccountRegisterDlg()
{
	delete jab;
}

void AccountRegisterDlg::closeEvent(QCloseEvent *e)
{
	stop();

	QDialog::closeEvent(e);
}

void AccountRegisterDlg::start(const QString &host, int port, bool opt_ssl, const QString &user, const QString &pass)
{
	busy->start();

	jab->setSSLEnabled(opt_ssl);
	jab->setHost(host, port);
	jab->setAccount(user, pass, "");
	jab->accRegister();
}

void AccountRegisterDlg::stop()
{
	jab->disc();
}

void AccountRegisterDlg::cancel()
{
	stop();
	reject();
}

bool AccountRegisterDlg::reg(QWidget *par, const QString &host, int port, bool opt_ssl, const QString &user, const QString &pass)
{
	AccountRegisterDlg *w = new AccountRegisterDlg(par);
	w->start(host, port, opt_ssl, user, pass);
	w->exec();
	delete w;

	return TRUE;
}

void AccountRegisterDlg::accRegisterFinished(bool ok, const QString &errMsg)
{
	jab->disc();
	busy->stop();

	if(ok)
		QMessageBox::information(this, tr("Success"), tr("The account was registered successfully."));
	else
		QMessageBox::critical(this, tr("Error"), QString(tr("There was an error registering the account.\nReason: %1")).arg(errMsg));

	accept();
}

void AccountRegisterDlg::jab_error(JabError *err)
{
	jab->disc();
	busy->stop();

	QMessageBox::critical(this, tr("Error"), QString(tr("There was an error communicating with the Jabber server.\nReason: %1")).arg(err->msg));

	accept();
}
