/*
 **************************************************************************
 *
 * Boot-ROM-Code to load an operating system across a TCP/IP network.
 *
 * Module:  udp.c
 * Purpose: UDP and IP handling
 * Entries: init_udp, udp_open, udp_read, udp_write
 *
 **************************************************************************
 *
 * Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
 * Copyright (C) 2001 Pete Zaitcev <zaitcev@yahoo.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <general.h>
#include <net.h>
#include <romlib.h>
#include "./netpriv.h"
#include "./arp.h"
#include "./ip.h"
#include "./udp.h"

/*
 * An arbitrary constant. The only constraint is to let
 * the packet and UDP pseudo-header to fit into an Ethernet frame.
 */
#define UDP_SEND_SIZE   577

/*
 * The checksum pseudo-header.
 */
struct udp_pseudo {
  unsigned long ip_saddr;	/* set to iphdr.ip_saddr */
  unsigned long ip_daddr;	/* set to iphdr.ip_daddr */
  unsigned char filler;		/* set to 0 */
  unsigned char ip_protocol;	/* set to iphdr.ip_protocol */
  unsigned short udp_len;	/* set to udphdr.udp_len */
};

#define UDP_PSEUDO_LEN   (sizeof(struct udp_pseudo))

/*
 **************************************************************************
 * 
 * Global variables:
 */
static unsigned short usource;		/* Source port number		*/
static unsigned short udest;		/* Destination port number	*/
static unsigned char  udhw[ETH_ALEN];	/* Destination hardware addr	*/
static t_ipaddr       udaddr;		/* Destination IP address	*/
static int            packetid;		/* packet ID counter		*/
static int            buflen;		/* length of contents in buffer	*/
static unsigned char *readbuf;		/* pointer to receive buffer	*/
static unsigned char *writebuf;		/* pointer to output buffer	*/



/*
 **************************************************************************
 * 
 * Open a new UDP socket.
 */
int udp_open(daddr, source, dest)
t_ipaddr daddr;
int      source;
int      dest;
{
  register unsigned char *addr;

  /* Set global variables */
  usource = source;
  udest = dest;
  udaddr = daddr;

  /* Find a routing to the destination host */
  if ((addr = ip_resolve(udaddr)) == NULL)
	return(FALSE);
  memcpy(udhw, addr, ETH_ALEN);
  return(TRUE);
}


/*
 **************************************************************************
 * 
 * IP receiver routine
 */
static int ip_recv(buf, bufsize, addr)
unsigned char *buf;
int            bufsize;
unsigned char *addr;
{
  struct iphdr *ipp = ((struct iphdr *)buf);
  struct udphdr *udpp = ((struct udphdr *)(buf + IP_MIN_HSIZE));
  struct udp_pseudo psehdr;

  int size;
  t_ipaddr dadr;

#ifdef DEBUG
  printf("UDP: got %d bytes from %x:%x:%x:%x:%x:%x\n",
	bufsize, addr[0], addr[1], addr[2], addr[3], addr[4], addr[5]);
#endif

  /*
   * Do some checks on the received packet. It must be an IP packet
   * with a valid version number and header size. Also the checksum
   * of the header must be correct, and the packet should not be frag-
   * mented. The packet should also not be too large, and it has to be
   * a UDP packet, and be destined to the correct port number.
   * XXX The code does not check the port despite the comment --zaitcev
   */
  size = ntohs(ipp->ip_tot_len) - IP_MIN_HSIZE;
  if (bufsize <= (IP_MIN_HSIZE + sizeof(struct udphdr)) ||
    size < sizeof(struct udphdr) || size > (IP_RECV_SIZE - IP_MIN_HSIZE) ||
    ipp->ip_ihl_version != ((IP_VERSION << 4) | (IP_MIN_HSIZE >> 2)) ||
    ipchksum((unsigned char *)ipp, IP_MIN_HSIZE) != 0 ||
    (ntohs(ipp->ip_frag_off) & 0x3fff) || ipp->ip_protocol != IPPROTO_UDP ||
    usource != ntohs(udpp->udp_dest) || ntohs(udpp->udp_len) > size)
	return(FALSE);

  /*
   * The destination address of the packet must be valid. It must either
   * be 0.0.0.0 (which means "any"), 255.255.255.255 for broadcast packets,
   * or our own IP number.
   */
  dadr = ntohl(ld4((char *)&ipp->ip_daddr));
  if (dadr != IP_ANY && dadr != IP_BROADCAST && dadr != myipaddr &&
      myipaddr != IP_ANY)
	return(FALSE);

#ifdef DEBUG
  printf("UDP: got packet from %lx.%d for local port %d\n",
			dadr, ntohs(udpp->udp_source), ntohs(udpp->udp_dest));
#endif

  /* Update the ARP table using the received packet */
  addcache(addr, ntohl(ld4((char *)&ipp->ip_saddr)));

  /*
   * Update the sockets destination port number and get the packet from
   * the receive buffer.
   */
  udest = ntohs(udpp->udp_source);
  buflen = ntohs(udpp->udp_len) - sizeof(struct udphdr);
  readbuf = (unsigned char *)udpp + sizeof(struct udphdr);

  /*
   * Compute the checksum.
   */
  if (udpp->udp_check != 0) {
    int csum, cslen;
    unsigned char *p;

    psehdr.ip_saddr = ld4((char *)&ipp->ip_saddr);
    psehdr.ip_daddr = ld4((char *)&ipp->ip_daddr);
    psehdr.filler = 0;
    psehdr.ip_protocol = ipp->ip_protocol;
    psehdr.udp_len = udpp->udp_len;
    p = (unsigned char *)udpp - UDP_PSEUDO_LEN;
    memcpy((char *)p, (char *)&psehdr, UDP_PSEUDO_LEN);
    cslen = UDP_PSEUDO_LEN + ntohs(udpp->udp_len);
    if ((csum = ipchksum(p, cslen)) != 0) {
      unsigned sum0 = ntohs(udpp->udp_check);
      udpp->udp_check = 0;
      csum = ipchksum(p, cslen);
      printf("UDP: bad checksum: host 0x%x our 0x%x\n", sum0, csum);
    }
  }

  return(TRUE);
}



/*
 **************************************************************************
 * 
 * Read one packet from a UDP socket
 */
int udp_read(buf, bufsize, timeout, abortch)
char *buf;
int   bufsize;
int   timeout;
char  abortch;
{
  int len;

  /* Wait until we get something */
  set_timeout(timeout);
  while (!chk_timeout() && buflen == 0) { }
  if ((len = buflen) > bufsize)
	len = bufsize;
  if (len > 0) {
	memcpy(buf, readbuf, len);
	empty_buf();
	buflen = 0;
  }
  return(len);
}



/*
 **************************************************************************
 * 
 * Write one packet to a UDP socket
 */
int udp_write(buf, len)
char *buf;
int   len;
{
  struct iphdr *ipp = (struct iphdr *)writebuf;
  struct udphdr *udpp = (struct udphdr *)(writebuf + IP_MIN_HSIZE);
  struct udp_pseudo *psep;
  int len1;

#ifdef DEBUG
  printf("UDP: sending %d bytes from %ld.%d to %ld.%d (%x:%x:%x:%x:%x:%x)\n",
			len, myipaddr, usource, udaddr, udest,
			udhw[0], udhw[1], udhw[2], udhw[3], udhw[4], udhw[5]);
#endif

  /*
   * Copy the packet into the output buffer. It's size should not be larger
   * than maximum UDP packet
   */
  if (len > UDP_SEND_SIZE) {
    printf("udp_write: too long (%d)\n", len);
    return 0;
  }
  memcpy(((char *)udpp) + sizeof(struct udphdr), buf, len);
  len += sizeof(struct udphdr);

  /* Prepare UDP datagram */
  udpp->udp_source = htons(usource);
  udpp->udp_dest = htons(udest);
  udpp->udp_len = htons(len);
  udpp->udp_check = 0;

  len1 = len;
  if (len & 1) {
    ((char *)udpp)[len1] = 0;
    len1++;
  }
  psep = (struct udp_pseudo *) (((char *)udpp) + len1);
  st4((char *)&psep->ip_saddr, htonl(myipaddr));
  st4((char *)&psep->ip_daddr, htonl(udaddr));
  psep->filler = 0;
  psep->ip_protocol = IPPROTO_UDP;
  psep->udp_len = htons(len);
  len1 += UDP_PSEUDO_LEN;

  /* Yep, no htons here: checksup is counted over bytes. */
  if ((udpp->udp_check = ipchksum((unsigned char *)udpp, len1)) == 0) {
    udpp->udp_check = 0xffff;
  }

  /* Prepare IP header */
  ipp->ip_ihl_version = ((IP_VERSION << 4) | ((IP_MIN_HSIZE/4) & 0x0f)) & 0xff;
  ipp->ip_tos         = IP_STD_TOS;
  ipp->ip_tot_len     = htons(IP_MIN_HSIZE + len);
  ipp->ip_id          = htons(++packetid);
  ipp->ip_frag_off    = 0;
  ipp->ip_check       = 0;
  ipp->ip_ttl         = IP_TTL;
  ipp->ip_protocol    = IPPROTO_UDP;
  st4((char *)&ipp->ip_daddr, htonl(udaddr));
  st4((char *)&ipp->ip_saddr, htonl(myipaddr));
  ipp->ip_check       = ipchksum(writebuf, IP_MIN_HSIZE);

  /* Finally send out the packet */
  (void)write_packet(IP_MIN_HSIZE + len, htons(ETH_P_IP), udhw);

  return(len);
}


/*
 **************************************************************************
 * 
 * Network intialization:
 */
int init_udp()
{
  /* Set module name for error handling */
  net_module_name = "udp";

  /* Register IP packet type and set write buffer pointer */
  if ((writebuf = reg_type(htons(ETH_P_IP), ip_recv)) == NULL)
	return(FALSE);

  return(TRUE);
}
