/***************************************************************************
                          tagger_mp3.cpp  -  description
                             -------------------
    begin                : Wed May 9 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#if HAVE_ID3

#include "tagger_mp3.h"

#include "configuration.h"

#include <ctype.h>

#include <qfile.h>

#include <id3/tag.h>
#include <id3/misc_support.h>

#include <fstream>
#include <iostream>
#include <stdexcept>
using namespace std;

int layer_table[4]= {0, 3, 2, 1};

int frequency_table[4][4] = {
                                {11025,12000, 8000, -1}, /* MPEG 2.5 */
                                {   -1    -1,   -1, -1},
                                {22050,24000,16000, -1}, /* MPEG 2.0 */
                                {44100,48000,32000, -1}  /* MPEG 1.0 */
                            };

int bitrate_table[4][4][16] = {
                                  {   /* MPEG 2.5 */
                                      { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
                                      {  0,  8, 16, 24, 32, 40, 48, 56, 64, 80, 96,112,128,144,160, -1}, /* layer 3 */
                                      {  0,  8, 16, 24, 32, 40, 48, 56, 64, 80, 96,112,128,144,160, -1}, /* layer 2 */
                                      {  0, 32, 48, 56, 64, 80, 96,112,128,144,160,176,192,224,256, -1}  /* layer 1  */
                                  },{
                                      { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
                                      { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
                                      { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
                                      { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}
                                  },{ /* MPEG 2.0 */
                                      { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
                                      {  0,  8, 16, 24, 32, 40, 48, 56, 64, 80, 96,112,128,144,160, -1}, /* layer 3 */
                                      {  0,  8, 16, 24, 32, 40, 48, 56, 64, 80, 96,112,128,144,160, -1}, /* layer 2 */
                                      {  0, 32, 48, 56, 64, 80, 96,112,128,144,160,176,192,224,256, -1}  /* layer 1 */
                                  },{ /* MPEG 1.0 */
                                      { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
                                      {  0, 32, 40, 48, 56, 64, 80, 96,112,128,160,192,224,256,320, -1}, /* layer 3 */
                                      {  0, 32, 48, 56, 64, 80, 96,112,128,160,192,224,256,320,384, -1}, /* layer 2 */
                                      {  0, 32, 64, 96,128,160,192,224,256,288,320,352,384,416,448, -1}  /* layer 1 */
                                  }
                              };

int frameSizeIndex[] = {0, 72000, 72000, 24000};


Tagger_MP3::Tagger_MP3() {}
Tagger_MP3::~Tagger_MP3() {}

bool Tagger_MP3::getInfo(TRACK *track, bool lazyRead) {
    MP3Info mp3;

    mp3.filename = QString(track->path + "/" + track->filename);

    getMP3Info(&mp3, false, !lazyRead);

    if(!mp3.header_isValid)
        return false;

    track->mimetype   = MIMETYPE_MP3;
    track->version    = mp3.header.version;
    track->layer      = mp3.layer;
    track->mode       = mp3.header.mode;
    track->bitrate    = mp3.isVBR ? -(int)mp3.vbr_average : mp3.bitrate;
    track->samplerate = mp3.frequency;
    track->length     = mp3.seconds;

    if((track->title       = mp3.id3v2.title)   == "")
        track->title = mp3.id3.title;
    if((track->artist      = mp3.id3v2.artist)  == "")
        track->artist = mp3.id3.artist;
    if((track->album       = mp3.id3v2.album)   == "")
        track->album = mp3.id3.album;
    if((track->year        = mp3.id3v2.year)    == "")
        track->year = mp3.id3.year;
    if((track->comment     = mp3.id3v2.comment) == "")
        track->comment = mp3.id3.comment;
    if((track->tracknumber = mp3.id3v2.track)   == 0)
        track->tracknumber = (int)mp3.id3.track[0];
    if((track->genre       = mp3.id3v2.genre)   == 255)
        track->genre = (int)mp3.id3.genre[0];

    track->artist = track->artist.simplifyWhiteSpace();

    /*
      track->title      = mp3.id3.title;
      track->artist     = mp3.id3.artist;
      track->album      = mp3.id3.album;
      track->year       = mp3.id3.year;
      track->tracknumber= (int)mp3.id3.track[0];
      track->genre      = (int)mp3.id3.genre[0];
      track->comment    = mp3.id3.comment;
    */
    return true;
}

bool Tagger_MP3::getTagOnly(TRACK *track) {
    MP3Info mp3;
    mp3.filename = QString(track->path + "/" + track->filename);
    mp3.file.setName(mp3.filename);
    if(!mp3.file.open(IO_ReadOnly))
        return false;
    mp3.dataSize = mp3.file.size();
    getID3(&mp3);
    mp3.file.close();

    config->lock()
    ;
    bool readID3v2 = config->readID3v2();
    config->unlock();

    if(readID3v2) {
        if((track->title       = mp3.id3v2.title)   == "")
            track->title = mp3.id3.title;
        if((track->artist      = mp3.id3v2.artist)  == "")
            track->artist = mp3.id3.artist;
        if((track->album       = mp3.id3v2.album)   == "")
            track->album = mp3.id3.album;
        if((track->year        = mp3.id3v2.year)    == "")
            track->year = mp3.id3.year;
        if((track->comment     = mp3.id3v2.comment) == "")
            track->comment = mp3.id3.comment;
        if((track->tracknumber = mp3.id3v2.track)   == 0)
            track->tracknumber = (int)mp3.id3.track[0];
        if((track->genre       = mp3.id3v2.genre)   == 255)
            track->genre = (int)mp3.id3.genre[0];
    } else {

        track->tracknumber = (int)mp3.id3.track[0];
        track->genre       = (int)mp3.id3.genre[0];

        QString readTitle = mp3.id3.title;
        if((readTitle.length() > track->title.length()) || !track->title.startsWith(readTitle))
            track->title = readTitle;

        QString readArtist = mp3.id3.artist;
        if((readArtist.length() > track->artist.length()) || !track->artist.startsWith(readArtist))
            track->artist = readArtist;

        QString readAlbum = mp3.id3.album;
        if((readAlbum.length() > track->album.length()) || !track->album.startsWith(readAlbum))
            track->artist = readAlbum;

        QString readYear = mp3.id3.year;
        if((readYear.length() > track->year.length()) || !track->year.startsWith(readYear))
            track->year = readYear;

        QString readComment = mp3.id3.comment;
        if((readComment.length() > track->comment.length()) || !track->comment.startsWith(readComment))
            track->comment = readComment;
    }

    track->artist = track->artist.simplifyWhiteSpace();

    return true;
}

bool Tagger_MP3::writeInfo(TRACK *track) {
    MP3Info mp3;

    mp3.filename = QString(track->path + "/" + track->filename);

    int c;
    c = track->title.length();
    if(c > 30)
        c = 30;
    for(int i=0; i < c; i++)
        mp3.id3.title[i] = track->title[i].latin1();
    mp3.id3.title[c] = '\0';
    c = track->artist.length();
    if(c > 30)
        c = 30;
    for(int i=0; i < c; i++)
        mp3.id3.artist[i] = track->artist[i].latin1();
    mp3.id3.artist[c] = '\0';
    c = track->album.length();
    if(c > 30)
        c = 30;
    for(int i=0; i < c; i++)
        mp3.id3.album[i] = track->album[i].latin1();
    mp3.id3.album[c] = '\0';
    c = track->year.length();
    if(c > 5)
        c = 5;
    for(int i=0; i < c; i++)
        mp3.id3.year[i] = track->year[i].latin1();
    mp3.id3.year[c] = '\0';
    c = track->comment.length();
    if(c > 30)
        c = 30;
    for(int i=0; i < c; i++)
        mp3.id3.comment[i] = track->comment[i].latin1();
    mp3.id3.comment[c] = '\0';
    mp3.id3.genre[0] = (unsigned char)track->genre;
    if(track->tracknumber > 0 && track->tracknumber <= 255)
        mp3.id3.track[0] = (unsigned char)track->tracknumber;
    else
        mp3.id3.track[0] = '\0';

    mp3.id3v2.title = track->title;
    mp3.id3v2.artist = track->artist;
    mp3.id3v2.album = track->album;
    mp3.id3v2.year = track->year;
    mp3.id3v2.comment = track->comment;

    return putID3(&mp3);
}

#define TEXT_FIELD_LEN  30
#define INT_FIELD_LEN   4

#define MIN_CONSEC_GOOD_FRAMES 4
#define FRAME_HEADER_SIZE 4
#define MIN_FRAME_SIZE 21
#define NUM_SAMPLES 4


int Tagger_MP3::getMP3Info(MP3Info *mp3, bool scanFull, bool vbrScanFull) {
    int frame_type[15] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
    int frames = 0, frame_types = 0, frames_so_far = 0;

    float seconds = 0, total_rate = 0;
    int vbr_median = 0;

    long samplePos, dataStart = 0;
    int lastBitrate, currBitrate;
    int counter = 0;

    mp3->isVBR          = 0;
    mp3->badFrames      = 0;
    mp3->header_isValid = false;

    mp3->file.setName(mp3->filename);
    if(!mp3->file.open(IO_ReadOnly))
        return 0;

    mp3->dataSize = mp3->file.size();

    getID3(mp3);

    if(!scanFull) {
        if(getFirstHeader(mp3, 0L)) {
            dataStart = mp3->file.at();
            lastBitrate = 15 - mp3->header.bitrate;
            while((counter < NUM_SAMPLES) && lastBitrate) {
                samplePos = (counter * (mp3->dataSize/NUM_SAMPLES + 1)) + dataStart;
                if(getFirstHeader(mp3, samplePos))
                    currBitrate = 15 - mp3->header.bitrate;
                else
                    currBitrate = -1;

                if(currBitrate != lastBitrate) {
                    mp3->isVBR = 1;
                    if(vbrScanFull) {
                        counter = NUM_SAMPLES;
                        scanFull = true;
                        ;
                    }
                }
                lastBitrate = currBitrate;
                counter++;
            }
        }
        if(!scanFull && mp3->header_isValid) {
            mp3->bitrate     = bitrate_table[mp3->header.version][mp3->header.layer][mp3->header.bitrate];
            mp3->frames      = (mp3->dataSize - dataStart) / frameLength(&mp3->header);
            mp3->seconds     = (int)((float)(frameLength(&mp3->header) * mp3->frames) / (float)(mp3->bitrate * 125) + 0.5);
            mp3->vbr_average = (float)mp3->bitrate;
        }
    }

    if(scanFull) {
        if(getFirstHeader(mp3, 0L)) {
            dataStart = mp3->file.at();
            while((currBitrate = getNextHeader(mp3))) {
                frame_type[15 - currBitrate]++;
                frames++;
            }
            for(counter = 0; counter < 15; counter++) {
                if(frame_type[counter]) {
                    frame_types++;
                    mp3->header.bitrate = counter;
                    currBitrate = bitrate_table[mp3->header.version][mp3->header.layer][mp3->header.bitrate];
                    frames_so_far += frame_type[counter];
                    seconds += (float)(frameLength(&mp3->header) * frame_type[counter]) / (float)(currBitrate * 125);
                    total_rate += (float)(currBitrate * frame_type[counter]);
                    if((vbr_median == 0) && (frames_so_far >= frames/2))
                        vbr_median = counter;
                }
            }
            mp3->seconds     = (int)(seconds + 0.5);
            mp3->bitrate     = vbr_median;
            mp3->vbr_average = total_rate / (float)frames;
            mp3->frames      = frames;
            if(frame_types > 1)
                mp3->isVBR = 1;
        }
    }

    if(mp3->header_isValid) {
        mp3->frequency = frequency_table[mp3->header.version][mp3->header.frequency];
        mp3->layer     = layer_table[mp3->header.layer];
    }

    mp3->file.close();

    return mp3->header_isValid;
}


int Tagger_MP3::getFirstHeader(MP3Info *mp3, long startPos) {
    MP3Header h;
    long validStart = 0;

    int k, l = 0, c;

    mp3->file.at(startPos);
    while(1) {
        while(((c = mp3->file.getch()) != 255) && (c != -1))
            ;
        if(c == 255) {
            mp3->file.ungetch(c);
            validStart = mp3->file.at();
            if((l = getHeader(mp3, &mp3->header))) {
                mp3->file.at(mp3->file.at() + l - FRAME_HEADER_SIZE);
                for(k = 1; (k < MIN_CONSEC_GOOD_FRAMES) && (mp3->dataSize - mp3->file.at() >= FRAME_HEADER_SIZE); k++) {
                    if(!(l = getHeader(mp3, &h)))
                        break;
                    if(!sameConstant(&mp3->header, &h))
                        break;
                    mp3->file.at(mp3->file.at() + l - FRAME_HEADER_SIZE);
                }
                if(k == MIN_CONSEC_GOOD_FRAMES) {
                    mp3->file.at(validStart);
                    mp3->header_isValid = true;
                    return 1;
                }
            }
        } else {
            qWarning( " Skipping %s: not a valid mp3 file....", mp3->file.name().latin1() );
            return 0;
        }
    }

    return 0;
}

int Tagger_MP3::getNextHeader(MP3Info *mp3) {
    int l=0,c,skipBytes=0;
    MP3Header h;

    while(1) {
        while((c = mp3->file.getch()) != 255 && (c != -1) && (mp3->file.at() < mp3->dataSize))
            skipBytes++;
        if(c == 255) {
            mp3->file.ungetch(c);
            if((l = getHeader(mp3, &h))) {
                if(skipBytes)
                    mp3->badFrames++;
                mp3->file.at(mp3->file.at() + l - FRAME_HEADER_SIZE);
                return 15 - h.bitrate;
            } else
                skipBytes += FRAME_HEADER_SIZE;
        } else {
            if(skipBytes)
                mp3->badFrames++;
            return 0;
        }
    }
}


int Tagger_MP3::getHeader(MP3Info *mp3, MP3Header *header) {
    char cb[FRAME_HEADER_SIZE];
    unsigned char *buffer;
    buffer = (unsigned char*)&cb;

    //  unsigned char buffer[FRAME_HEADER_SIZE];

    int fl;

    //  if(mp3->file.readBlock(&(char)buffer[0], FRAME_HEADER_SIZE) == -1) {
    //    header->sync=0;
    //    return 0;
    //  }

    if(mp3->file.readBlock(cb, FRAME_HEADER_SIZE) == -1) {
        header->sync=0;
        return 0;
    }

    header->sync          = ((unsigned int)buffer[0] + (256*(buffer[1] >> 5)));
    header->version       = (buffer[1] >> 3) & 3;
    header->layer         = (buffer[1] >> 1) & 3;
    header->isProtected   =  buffer[1] & 1;
    header->bitrate       = (buffer[2] >> 4) & 0x0F;
    header->frequency     = (buffer[2] >> 2) & 0x3;
    header->isPadded      = (buffer[2] >>1) & 0x1;
    header->isPrivate     = buffer[2] & 0x1;
    header->mode          = (buffer[3] >> 6) & 0x3;
    header->mode_ext      = (buffer[3] >> 4) & 0x3;
    header->isCopyrighted = (buffer[3] >> 3) & 0x1;
    header->isOriginal    = (buffer[3] >> 2) & 0x1;
    header->emphasis      = buffer[3] & 0x3;

    if((header->sync != 0x7FF) || (header->version == 1) || (header->layer == 0)
            || (header->bitrate == 15) || (header->frequency == 3) || (header->emphasis == 2)) {
        header->sync=0;
        return 0;
    }

    return ((fl = frameLength(header)) >= MIN_FRAME_SIZE ? fl : 0);
}

int Tagger_MP3::frameLength(MP3Header *header) {
    return header->sync == 0x7FF ?
           (frameSizeIndex[header->layer] * ((header->version == 3) ? 2 : 1) *
            bitrate_table[header->version][header->layer][header->bitrate]) /
           frequency_table[header->version][header->frequency] + header->isPadded
       : 1;
}

int Tagger_MP3::sameConstant(MP3Header *h1, MP3Header *h2) {
    if(h1 == h2)
        return 1;

    if((h1->version       == h2->version         ) &&
            (h1->layer         == h2->layer           ) &&
            (h1->isProtected   == h2->isProtected     ) &&
            (h1->frequency     == h2->frequency       ) &&
            (h1->mode          == h2->mode            ) &&
            (h1->isCopyrighted == h2->isCopyrighted   ) &&
            (h1->isOriginal    == h2->isOriginal      ) &&
            (h1->emphasis      == h2->emphasis        ))
        return 1;
    else
        return 0;
}

int Tagger_MP3::getID3(MP3Info *mp3) {
    // get ID3v1

    char tag[4];

    mp3->id3.title[0] = '\0';
    mp3->id3.artist[0] = '\0';
    mp3->id3.album[0] = '\0';
    mp3->id3.year[0] = '\0';
    mp3->id3.comment[0] = '\0';
    mp3->id3.genre[0] = '\0';
    mp3->id3.track[0] = '\0';

    if(mp3->file.size() >= 128) {
        if(!mp3->file.at(mp3->file.size() - 128))
            return 1;
        else {
            mp3->file.readBlock(&tag[0], 3);
            tag[3] = '\0';
            mp3->id3.genre[0] = 255;

            if((tag[0] == 'T') && (tag[1] == 'A') && (tag[2] == 'G')) {

                mp3->dataSize -= 128;
                mp3->file.at(mp3->file.size() - 125);
                mp3->file.readBlock(&mp3->id3.title[0], 30);
                mp3->id3.title[30] = '\0';
                mp3->file.readBlock(&mp3->id3.artist[0],30);
                mp3->id3.artist[30] = '\0';
                mp3->file.readBlock(&mp3->id3.album[0], 30);
                mp3->id3.album[30] = '\0';
                mp3->file.readBlock(&mp3->id3.year[0], 4);
                mp3->id3.year[4] = '\0';
                mp3->file.readBlock(&mp3->id3.comment[0],30);
                mp3->id3.comment[30] = '\0';
                if(mp3->id3.comment[28] == '\0') {
                    mp3->id3.track[0] = mp3->id3.comment[29];
                }
                mp3->file.readBlock((char*)&mp3->id3.genre[0], 1);
                unpad(mp3->id3.title);
                unpad(mp3->id3.artist);
                unpad(mp3->id3.album);
                unpad(mp3->id3.year);
                unpad(mp3->id3.comment);
            }
        }
    }

    // get ID3v2

    config->lock()
    ;
    bool readID3v2 = config->readID3v2();
    config->unlock();

    mp3->id3v2.title = "";
    mp3->id3v2.artist = "";
    mp3->id3v2.album = "";
    mp3->id3v2.year = "";
    mp3->id3v2.track = 0;
    mp3->id3v2.comment = "";
    mp3->id3v2.genre = 255;


    if(!readID3v2)
        return 0;

    ID3_Tag myTag;

    myTag.Link(mp3->filename, ID3TT_ID3V2);

    //  ID3_Frame *myFrame;

    // Some (ill-formed?) mp3 file generate id3lib exception when reading comment tags

    try {
        if((mp3->id3v2.artist  = ID3_GetArtist(&myTag))  == 0)
            mp3->id3v2.artist = "";
    } catch (logic_error e) {
        mp3->id3v2.artist = "";
        if ( verbose )
            qWarning(" **** Exception: %s reading artist from: %s", e.what(), mp3->filename.latin1());
    }

    try {
        if((mp3->id3v2.album   = ID3_GetAlbum(&myTag))   == 0)
            mp3->id3v2.album = "";
    } catch (logic_error e) {
        mp3->id3v2.album = "";
        if ( verbose )
            qWarning(" **** Exception: %s reading album from: %s", e.what(), mp3->filename.latin1());
    }

    try {
        if((mp3->id3v2.title   = ID3_GetTitle(&myTag))   == 0)
            mp3->id3v2.title = "";
    } catch (logic_error e) {
        mp3->id3v2.title = "";
        if ( verbose )
            qWarning(" **** Exception: %s reading title from: %s", e.what(), mp3->filename.latin1());
    }

    try {
        if((mp3->id3v2.year    = ID3_GetYear(&myTag))    == 0)
            mp3->id3v2.year = "";
    } catch (logic_error e) {
        mp3->id3v2.year = "";
        if ( verbose )
            qWarning(" **** Exception: %s reading year from: %s", e.what(), mp3->filename.latin1());
    }

    try {
        if((mp3->id3v2.comment = ID3_GetComment(&myTag)) == 0)
            mp3->id3v2.comment = "";
    } catch (logic_error e) {
        mp3->id3v2.comment = "";
        if ( verbose )
            qWarning(" **** Exception: %s reading comment from: %s", e.what(), mp3->filename.latin1());
    }

    try {
        mp3->id3v2.track   = ID3_GetTrackNum(&myTag);
        if(mp3->id3v2.track < 0 || mp3->id3v2.track > 255)
            mp3->id3v2.track = 0;
    } catch (logic_error e) {
        mp3->id3v2.track = 0;
        if ( verbose )
            qWarning(" **** Exception: %s when reading track number from: %s", e.what(), mp3->filename.latin1());
    }

    try {
        mp3->id3v2.genre   = ID3_GetGenreNum(&myTag);
        if(mp3->id3v2.genre < 0 || mp3->id3v2.genre > 255)
            mp3->id3v2.genre = 0;
    } catch (logic_error e) {
        mp3->id3v2.genre = 0;
        if ( verbose )
            qWarning(" **** Exception: %s reading genre from: %s", e.what(), mp3->filename.latin1());
    }

    if (verbose == 6 ) {
        cout << "ID3v2 Tags: (" << mp3->filename << ")" << endl;
        cout << "  Title   : " << mp3->id3v2.title << endl;
        cout << "  Artist  : " << mp3->id3v2.artist << endl;
        cout << "  Album   : " << mp3->id3v2.album << endl;
        cout << "  Year    : " << mp3->id3v2.year << endl;
        cout << "  Track   : " << mp3->id3v2.track << endl;
        cout << "  Comment : " << mp3->id3v2.comment << endl;
        cout << "  Genre   : " << mp3->id3v2.genre << endl << endl;

        cout << "ID3 Tags: " << endl;
        cout << "  Title   : " << mp3->id3.title << endl;
        cout << "  Artist  : " << mp3->id3.artist << endl;
        cout << "  Album   : " << mp3->id3.album << endl;
        cout << "  Year    : " << mp3->id3.year << endl;
        cout << "  Track   : " << (int)mp3->id3.track[0]<< endl;
        cout << "  Comment : " << mp3->id3.comment << endl;
        cout << "  Genre   : " << (int)mp3->id3.genre[0] << endl << endl;
    }
    return 0;
}

int Tagger_MP3::putID3(MP3Info *mp3) {
    char tag[4];
    char buf[129];

    int tagPosition;

    strcpy(buf,"TAG");
    pad(mp3->id3.title,TEXT_FIELD_LEN);
    strncat(buf,mp3->id3.title,TEXT_FIELD_LEN);
    pad(mp3->id3.artist,TEXT_FIELD_LEN);
    strncat(buf,mp3->id3.artist,TEXT_FIELD_LEN);
    pad(mp3->id3.album,TEXT_FIELD_LEN);
    strncat(buf,mp3->id3.album,TEXT_FIELD_LEN);
    pad(mp3->id3.year,INT_FIELD_LEN);
    strncat(buf,mp3->id3.year,INT_FIELD_LEN);
    pad(mp3->id3.comment,TEXT_FIELD_LEN);
    strncat(buf,mp3->id3.comment,TEXT_FIELD_LEN);
    strncat(buf,(char*)mp3->id3.genre,1);
    if(mp3->id3.track[0] != '\0') {
        buf[125] = '\0';
        buf[126] = mp3->id3.track[0];
    }

    mp3->file.setName(mp3->filename);
    if(!mp3->file.open(IO_ReadWrite))
        return 0;

    if(mp3->file.size() < 128)
        return 0;
    if(!mp3->file.at(mp3->file.size() - 128))
        return 0;

    mp3->file.readBlock(&tag[0], 3);
    tag[3] = '\0';

    if((tag[0] == 'T') && (tag[1] == 'A') && (tag[2] == 'G'))
        tagPosition = mp3->file.size() - 128;
    else
        tagPosition = mp3->file.size();

    mp3->file.at(tagPosition);
    int ret = mp3->file.writeBlock(&buf[0], 128);
    mp3->file.close();

    config->lock()
    ;
    bool writeID3v2 = config->writeID3v2();
    config->unlock();

    if(!writeID3v2)
        return ret;

    ID3_Tag myTag;

    myTag.Link(mp3->filename, ID3TT_ID3V2);

    ID3_AddArtist(&myTag, mp3->id3v2.artist, true);
    ID3_AddAlbum(&myTag, mp3->id3v2.album, true);
    ID3_AddTitle(&myTag, mp3->id3v2.title, true);
    ID3_AddYear(&myTag, mp3->id3v2.year, true);
    ID3_AddComment(&myTag, mp3->id3v2.comment, true);
    ID3_AddTrack(&myTag, mp3->id3.track[0], 0, true);
    ID3_AddGenre(&myTag, mp3->id3.genre[0], true);

    myTag.SetUnsync(false);
    myTag.SetExtendedHeader(true);
    myTag.SetCompression(true);
    myTag.SetPadding(true);

    myTag.Update();

    return ret;
}

char *Tagger_MP3::pad(char *s, int length) {
    int l = strlen(s);
    while(l < length) {
        s[l] = ' ';
        l++;
    }
    s[l]='\0';
    return s;
}

char *Tagger_MP3::unpad(char *s) {
    char *pos = s + strlen(s) - 1;
    while(isspace(pos[0]))
        (pos--)[0] = 0;
    return s;
}
#endif /* HAVE_ID3 */
